/* valarealliteral.vala
 *
 * Copyright (C) 2006-2009  Jürg Billeter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 */

#include <vala/valarealliteral.h>
#include <vala/valasourcereference.h>
#include <vala/valacodevisitor.h>
#include <vala/valaexpression.h>
#include <vala/valasemanticanalyzer.h>
#include <vala/valafloatingtype.h>
#include <vala/valasymbol.h>
#include <vala/valascope.h>
#include <vala/valastruct.h>




struct _ValaRealLiteralPrivate {
	char* _value;
};

#define VALA_REAL_LITERAL_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_REAL_LITERAL, ValaRealLiteralPrivate))
enum  {
	VALA_REAL_LITERAL_DUMMY_PROPERTY
};
static void vala_real_literal_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor);
static gboolean vala_real_literal_real_is_pure (ValaExpression* base);
static char* vala_real_literal_real_to_string (ValaCodeNode* base);
static gboolean vala_real_literal_real_check (ValaCodeNode* base, ValaSemanticAnalyzer* analyzer);
static gpointer vala_real_literal_parent_class = NULL;
static void vala_real_literal_finalize (ValaCodeNode* obj);



/**
 * Creates a new real literal.
 *
 * @param r      literal value
 * @param source reference to source code
 * @return       newly created real literal
 */
ValaRealLiteral* vala_real_literal_construct (GType object_type, const char* r, ValaSourceReference* source) {
	ValaRealLiteral* self;
	g_return_val_if_fail (r != NULL, NULL);
	g_return_val_if_fail (source != NULL, NULL);
	self = (ValaRealLiteral*) g_type_create_instance (object_type);
	vala_real_literal_set_value (self, r);
	vala_code_node_set_source_reference ((ValaCodeNode*) self, source);
	return self;
}


ValaRealLiteral* vala_real_literal_new (const char* r, ValaSourceReference* source) {
	return vala_real_literal_construct (VALA_TYPE_REAL_LITERAL, r, source);
}


static void vala_real_literal_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor) {
	ValaRealLiteral * self;
	self = (ValaRealLiteral*) base;
	g_return_if_fail (visitor != NULL);
	vala_code_visitor_visit_real_literal (visitor, self);
	vala_code_visitor_visit_expression (visitor, (ValaExpression*) self);
}


/**
 * Returns the type name of the value this literal represents.
 *
 * @return the name of literal type
 */
char* vala_real_literal_get_type_name (ValaRealLiteral* self) {
	gboolean _tmp0;
	g_return_val_if_fail (self != NULL, NULL);
	_tmp0 = FALSE;
	if (g_str_has_suffix (self->priv->_value, "f")) {
		_tmp0 = TRUE;
	} else {
		_tmp0 = g_str_has_suffix (self->priv->_value, "F");
	}
	if (_tmp0) {
		return g_strdup ("float");
	}
	return g_strdup ("double");
}


static gboolean vala_real_literal_real_is_pure (ValaExpression* base) {
	ValaRealLiteral * self;
	self = (ValaRealLiteral*) base;
	return TRUE;
}


static char* vala_real_literal_real_to_string (ValaCodeNode* base) {
	ValaRealLiteral * self;
	const char* _tmp0;
	self = (ValaRealLiteral*) base;
	_tmp0 = NULL;
	return (_tmp0 = self->priv->_value, (_tmp0 == NULL) ? NULL : g_strdup (_tmp0));
}


static gboolean vala_real_literal_real_check (ValaCodeNode* base, ValaSemanticAnalyzer* analyzer) {
	ValaRealLiteral * self;
	ValaFloatingType* _tmp3;
	ValaStruct* _tmp2;
	char* _tmp1;
	self = (ValaRealLiteral*) base;
	g_return_val_if_fail (analyzer != NULL, FALSE);
	if (vala_code_node_get_checked ((ValaCodeNode*) self)) {
		return !vala_code_node_get_error ((ValaCodeNode*) self);
	}
	vala_code_node_set_checked ((ValaCodeNode*) self, TRUE);
	_tmp3 = NULL;
	_tmp2 = NULL;
	_tmp1 = NULL;
	vala_expression_set_value_type ((ValaExpression*) self, (ValaDataType*) (_tmp3 = vala_floating_type_new (_tmp2 = VALA_STRUCT (vala_scope_lookup (vala_symbol_get_scope (analyzer->root_symbol), _tmp1 = vala_real_literal_get_type_name (self))))));
	(_tmp3 == NULL) ? NULL : (_tmp3 = (vala_code_node_unref (_tmp3), NULL));
	(_tmp2 == NULL) ? NULL : (_tmp2 = (vala_code_node_unref (_tmp2), NULL));
	_tmp1 = (g_free (_tmp1), NULL);
	return !vala_code_node_get_error ((ValaCodeNode*) self);
}


const char* vala_real_literal_get_value (ValaRealLiteral* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return self->priv->_value;
}


void vala_real_literal_set_value (ValaRealLiteral* self, const char* value) {
	char* _tmp2;
	const char* _tmp1;
	g_return_if_fail (self != NULL);
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_value = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL) ? NULL : g_strdup (_tmp1)), self->priv->_value = (g_free (self->priv->_value), NULL), _tmp2);
}


static void vala_real_literal_class_init (ValaRealLiteralClass * klass) {
	vala_real_literal_parent_class = g_type_class_peek_parent (klass);
	VALA_CODE_NODE_CLASS (klass)->finalize = vala_real_literal_finalize;
	g_type_class_add_private (klass, sizeof (ValaRealLiteralPrivate));
	VALA_CODE_NODE_CLASS (klass)->accept = vala_real_literal_real_accept;
	VALA_EXPRESSION_CLASS (klass)->is_pure = vala_real_literal_real_is_pure;
	VALA_CODE_NODE_CLASS (klass)->to_string = vala_real_literal_real_to_string;
	VALA_CODE_NODE_CLASS (klass)->check = vala_real_literal_real_check;
}


static void vala_real_literal_instance_init (ValaRealLiteral * self) {
	self->priv = VALA_REAL_LITERAL_GET_PRIVATE (self);
}


static void vala_real_literal_finalize (ValaCodeNode* obj) {
	ValaRealLiteral * self;
	self = VALA_REAL_LITERAL (obj);
	self->priv->_value = (g_free (self->priv->_value), NULL);
	VALA_CODE_NODE_CLASS (vala_real_literal_parent_class)->finalize (obj);
}


GType vala_real_literal_get_type (void) {
	static GType vala_real_literal_type_id = 0;
	if (vala_real_literal_type_id == 0) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaRealLiteralClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_real_literal_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaRealLiteral), 0, (GInstanceInitFunc) vala_real_literal_instance_init, NULL };
		vala_real_literal_type_id = g_type_register_static (VALA_TYPE_LITERAL, "ValaRealLiteral", &g_define_type_info, 0);
	}
	return vala_real_literal_type_id;
}




