/* valathrowstatement.vala
 *
 * Copyright (C) 2007-2009  Jürg Billeter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 */

#include <glib.h>
#include <glib-object.h>
#include <stdlib.h>
#include <string.h>
#include <gee.h>


#define VALA_TYPE_CODE_NODE (vala_code_node_get_type ())
#define VALA_CODE_NODE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_CODE_NODE, ValaCodeNode))
#define VALA_CODE_NODE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_CODE_NODE, ValaCodeNodeClass))
#define VALA_IS_CODE_NODE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_CODE_NODE))
#define VALA_IS_CODE_NODE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_CODE_NODE))
#define VALA_CODE_NODE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_CODE_NODE, ValaCodeNodeClass))

typedef struct _ValaCodeNode ValaCodeNode;
typedef struct _ValaCodeNodeClass ValaCodeNodeClass;
typedef struct _ValaCodeNodePrivate ValaCodeNodePrivate;

#define VALA_TYPE_CODE_VISITOR (vala_code_visitor_get_type ())
#define VALA_CODE_VISITOR(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_CODE_VISITOR, ValaCodeVisitor))
#define VALA_CODE_VISITOR_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_CODE_VISITOR, ValaCodeVisitorClass))
#define VALA_IS_CODE_VISITOR(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_CODE_VISITOR))
#define VALA_IS_CODE_VISITOR_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_CODE_VISITOR))
#define VALA_CODE_VISITOR_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_CODE_VISITOR, ValaCodeVisitorClass))

typedef struct _ValaCodeVisitor ValaCodeVisitor;
typedef struct _ValaCodeVisitorClass ValaCodeVisitorClass;

#define VALA_TYPE_SEMANTIC_ANALYZER (vala_semantic_analyzer_get_type ())
#define VALA_SEMANTIC_ANALYZER(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_SEMANTIC_ANALYZER, ValaSemanticAnalyzer))
#define VALA_SEMANTIC_ANALYZER_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_SEMANTIC_ANALYZER, ValaSemanticAnalyzerClass))
#define VALA_IS_SEMANTIC_ANALYZER(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_SEMANTIC_ANALYZER))
#define VALA_IS_SEMANTIC_ANALYZER_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_SEMANTIC_ANALYZER))
#define VALA_SEMANTIC_ANALYZER_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_SEMANTIC_ANALYZER, ValaSemanticAnalyzerClass))

typedef struct _ValaSemanticAnalyzer ValaSemanticAnalyzer;
typedef struct _ValaSemanticAnalyzerClass ValaSemanticAnalyzerClass;

#define VALA_TYPE_DATA_TYPE (vala_data_type_get_type ())
#define VALA_DATA_TYPE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_DATA_TYPE, ValaDataType))
#define VALA_DATA_TYPE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_DATA_TYPE, ValaDataTypeClass))
#define VALA_IS_DATA_TYPE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_DATA_TYPE))
#define VALA_IS_DATA_TYPE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_DATA_TYPE))
#define VALA_DATA_TYPE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_DATA_TYPE, ValaDataTypeClass))

typedef struct _ValaDataType ValaDataType;
typedef struct _ValaDataTypeClass ValaDataTypeClass;

#define VALA_TYPE_EXPRESSION (vala_expression_get_type ())
#define VALA_EXPRESSION(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_EXPRESSION, ValaExpression))
#define VALA_EXPRESSION_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_EXPRESSION, ValaExpressionClass))
#define VALA_IS_EXPRESSION(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_EXPRESSION))
#define VALA_IS_EXPRESSION_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_EXPRESSION))
#define VALA_EXPRESSION_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_EXPRESSION, ValaExpressionClass))

typedef struct _ValaExpression ValaExpression;
typedef struct _ValaExpressionClass ValaExpressionClass;

#define VALA_TYPE_SYMBOL (vala_symbol_get_type ())
#define VALA_SYMBOL(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_SYMBOL, ValaSymbol))
#define VALA_SYMBOL_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_SYMBOL, ValaSymbolClass))
#define VALA_IS_SYMBOL(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_SYMBOL))
#define VALA_IS_SYMBOL_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_SYMBOL))
#define VALA_SYMBOL_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_SYMBOL, ValaSymbolClass))

typedef struct _ValaSymbol ValaSymbol;
typedef struct _ValaSymbolClass ValaSymbolClass;

#define VALA_TYPE_LOCAL_VARIABLE (vala_local_variable_get_type ())
#define VALA_LOCAL_VARIABLE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_LOCAL_VARIABLE, ValaLocalVariable))
#define VALA_LOCAL_VARIABLE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_LOCAL_VARIABLE, ValaLocalVariableClass))
#define VALA_IS_LOCAL_VARIABLE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_LOCAL_VARIABLE))
#define VALA_IS_LOCAL_VARIABLE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_LOCAL_VARIABLE))
#define VALA_LOCAL_VARIABLE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_LOCAL_VARIABLE, ValaLocalVariableClass))

typedef struct _ValaLocalVariable ValaLocalVariable;
typedef struct _ValaLocalVariableClass ValaLocalVariableClass;

#define VALA_TYPE_ATTRIBUTE (vala_attribute_get_type ())
#define VALA_ATTRIBUTE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_ATTRIBUTE, ValaAttribute))
#define VALA_ATTRIBUTE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_ATTRIBUTE, ValaAttributeClass))
#define VALA_IS_ATTRIBUTE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_ATTRIBUTE))
#define VALA_IS_ATTRIBUTE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_ATTRIBUTE))
#define VALA_ATTRIBUTE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_ATTRIBUTE, ValaAttributeClass))

typedef struct _ValaAttribute ValaAttribute;
typedef struct _ValaAttributeClass ValaAttributeClass;

#define VALA_TYPE_STATEMENT (vala_statement_get_type ())
#define VALA_STATEMENT(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_STATEMENT, ValaStatement))
#define VALA_IS_STATEMENT(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_STATEMENT))
#define VALA_STATEMENT_GET_INTERFACE(obj) (G_TYPE_INSTANCE_GET_INTERFACE ((obj), VALA_TYPE_STATEMENT, ValaStatementIface))

typedef struct _ValaStatement ValaStatement;
typedef struct _ValaStatementIface ValaStatementIface;

#define VALA_TYPE_THROW_STATEMENT (vala_throw_statement_get_type ())
#define VALA_THROW_STATEMENT(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_THROW_STATEMENT, ValaThrowStatement))
#define VALA_THROW_STATEMENT_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_THROW_STATEMENT, ValaThrowStatementClass))
#define VALA_IS_THROW_STATEMENT(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_THROW_STATEMENT))
#define VALA_IS_THROW_STATEMENT_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_THROW_STATEMENT))
#define VALA_THROW_STATEMENT_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_THROW_STATEMENT, ValaThrowStatementClass))

typedef struct _ValaThrowStatement ValaThrowStatement;
typedef struct _ValaThrowStatementClass ValaThrowStatementClass;
typedef struct _ValaThrowStatementPrivate ValaThrowStatementPrivate;

#define VALA_TYPE_SOURCE_REFERENCE (vala_source_reference_get_type ())
#define VALA_SOURCE_REFERENCE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_SOURCE_REFERENCE, ValaSourceReference))
#define VALA_SOURCE_REFERENCE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_SOURCE_REFERENCE, ValaSourceReferenceClass))
#define VALA_IS_SOURCE_REFERENCE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_SOURCE_REFERENCE))
#define VALA_IS_SOURCE_REFERENCE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_SOURCE_REFERENCE))
#define VALA_SOURCE_REFERENCE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_SOURCE_REFERENCE, ValaSourceReferenceClass))

typedef struct _ValaSourceReference ValaSourceReference;
typedef struct _ValaSourceReferenceClass ValaSourceReferenceClass;

#define VALA_TYPE_TYPESYMBOL (vala_typesymbol_get_type ())
#define VALA_TYPESYMBOL(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_TYPESYMBOL, ValaTypeSymbol))
#define VALA_TYPESYMBOL_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_TYPESYMBOL, ValaTypeSymbolClass))
#define VALA_IS_TYPESYMBOL(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_TYPESYMBOL))
#define VALA_IS_TYPESYMBOL_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_TYPESYMBOL))
#define VALA_TYPESYMBOL_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_TYPESYMBOL, ValaTypeSymbolClass))

typedef struct _ValaTypeSymbol ValaTypeSymbol;
typedef struct _ValaTypeSymbolClass ValaTypeSymbolClass;

#define VALA_TYPE_ERROR_DOMAIN (vala_error_domain_get_type ())
#define VALA_ERROR_DOMAIN(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_ERROR_DOMAIN, ValaErrorDomain))
#define VALA_ERROR_DOMAIN_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_ERROR_DOMAIN, ValaErrorDomainClass))
#define VALA_IS_ERROR_DOMAIN(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_ERROR_DOMAIN))
#define VALA_IS_ERROR_DOMAIN_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_ERROR_DOMAIN))
#define VALA_ERROR_DOMAIN_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_ERROR_DOMAIN, ValaErrorDomainClass))

typedef struct _ValaErrorDomain ValaErrorDomain;
typedef struct _ValaErrorDomainClass ValaErrorDomainClass;

#define VALA_TYPE_ERROR_CODE (vala_error_code_get_type ())
#define VALA_ERROR_CODE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_ERROR_CODE, ValaErrorCode))
#define VALA_ERROR_CODE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_ERROR_CODE, ValaErrorCodeClass))
#define VALA_IS_ERROR_CODE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_ERROR_CODE))
#define VALA_IS_ERROR_CODE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_ERROR_CODE))
#define VALA_ERROR_CODE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_ERROR_CODE, ValaErrorCodeClass))

typedef struct _ValaErrorCode ValaErrorCode;
typedef struct _ValaErrorCodeClass ValaErrorCodeClass;

#define VALA_TYPE_REFERENCE_TYPE (vala_reference_type_get_type ())
#define VALA_REFERENCE_TYPE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_REFERENCE_TYPE, ValaReferenceType))
#define VALA_REFERENCE_TYPE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_REFERENCE_TYPE, ValaReferenceTypeClass))
#define VALA_IS_REFERENCE_TYPE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_REFERENCE_TYPE))
#define VALA_IS_REFERENCE_TYPE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_REFERENCE_TYPE))
#define VALA_REFERENCE_TYPE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_REFERENCE_TYPE, ValaReferenceTypeClass))

typedef struct _ValaReferenceType ValaReferenceType;
typedef struct _ValaReferenceTypeClass ValaReferenceTypeClass;

#define VALA_TYPE_ERROR_TYPE (vala_error_type_get_type ())
#define VALA_ERROR_TYPE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_ERROR_TYPE, ValaErrorType))
#define VALA_ERROR_TYPE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_ERROR_TYPE, ValaErrorTypeClass))
#define VALA_IS_ERROR_TYPE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_ERROR_TYPE))
#define VALA_IS_ERROR_TYPE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_ERROR_TYPE))
#define VALA_ERROR_TYPE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_ERROR_TYPE, ValaErrorTypeClass))

typedef struct _ValaErrorType ValaErrorType;
typedef struct _ValaErrorTypeClass ValaErrorTypeClass;

/**
 * Represents a part of the parsed source code.
 *
 * Code nodes get created by the parser and are used throughout the whole
 * compilation process.
 */
struct _ValaCodeNode {
	GTypeInstance parent_instance;
	volatile int ref_count;
	ValaCodeNodePrivate * priv;
	GList* attributes;
};

struct _ValaCodeNodeClass {
	GTypeClass parent_class;
	void (*finalize) (ValaCodeNode *self);
	void (*accept) (ValaCodeNode* self, ValaCodeVisitor* visitor);
	void (*accept_children) (ValaCodeNode* self, ValaCodeVisitor* visitor);
	gboolean (*check) (ValaCodeNode* self, ValaSemanticAnalyzer* analyzer);
	void (*replace_type) (ValaCodeNode* self, ValaDataType* old_type, ValaDataType* new_type);
	void (*replace_expression) (ValaCodeNode* self, ValaExpression* old_node, ValaExpression* new_node);
	char* (*to_string) (ValaCodeNode* self);
	void (*get_defined_variables) (ValaCodeNode* self, GeeCollection* collection);
	void (*get_used_variables) (ValaCodeNode* self, GeeCollection* collection);
};

/**
 * Interface for all statement types.
 */
struct _ValaStatementIface {
	GTypeInterface parent_iface;
};

/**
 * Represents a throw statement in the source code.
 */
struct _ValaThrowStatement {
	ValaCodeNode parent_instance;
	ValaThrowStatementPrivate * priv;
};

struct _ValaThrowStatementClass {
	ValaCodeNodeClass parent_class;
};

struct _ValaThrowStatementPrivate {
	ValaExpression* _error_expression;
};



gpointer vala_code_node_ref (gpointer instance);
void vala_code_node_unref (gpointer instance);
GParamSpec* vala_param_spec_code_node (const gchar* name, const gchar* nick, const gchar* blurb, GType object_type, GParamFlags flags);
void vala_value_set_code_node (GValue* value, gpointer v_object);
gpointer vala_value_get_code_node (const GValue* value);
GType vala_code_node_get_type (void);
gpointer vala_code_visitor_ref (gpointer instance);
void vala_code_visitor_unref (gpointer instance);
GParamSpec* vala_param_spec_code_visitor (const gchar* name, const gchar* nick, const gchar* blurb, GType object_type, GParamFlags flags);
void vala_value_set_code_visitor (GValue* value, gpointer v_object);
gpointer vala_value_get_code_visitor (const GValue* value);
GType vala_code_visitor_get_type (void);
GType vala_semantic_analyzer_get_type (void);
GType vala_data_type_get_type (void);
GType vala_expression_get_type (void);
GType vala_symbol_get_type (void);
GType vala_local_variable_get_type (void);
GType vala_attribute_get_type (void);
GType vala_statement_get_type (void);
GType vala_throw_statement_get_type (void);
#define VALA_THROW_STATEMENT_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_THROW_STATEMENT, ValaThrowStatementPrivate))
enum  {
	VALA_THROW_STATEMENT_DUMMY_PROPERTY
};
gpointer vala_source_reference_ref (gpointer instance);
void vala_source_reference_unref (gpointer instance);
GParamSpec* vala_param_spec_source_reference (const gchar* name, const gchar* nick, const gchar* blurb, GType object_type, GParamFlags flags);
void vala_value_set_source_reference (GValue* value, gpointer v_object);
gpointer vala_value_get_source_reference (const GValue* value);
GType vala_source_reference_get_type (void);
void vala_code_node_set_source_reference (ValaCodeNode* self, ValaSourceReference* value);
void vala_throw_statement_set_error_expression (ValaThrowStatement* self, ValaExpression* value);
ValaThrowStatement* vala_throw_statement_new (ValaExpression* error_expression, ValaSourceReference* source_reference);
ValaThrowStatement* vala_throw_statement_construct (GType object_type, ValaExpression* error_expression, ValaSourceReference* source_reference);
ValaThrowStatement* vala_throw_statement_new (ValaExpression* error_expression, ValaSourceReference* source_reference);
void vala_code_visitor_visit_throw_statement (ValaCodeVisitor* self, ValaThrowStatement* stmt);
static void vala_throw_statement_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor);
ValaExpression* vala_throw_statement_get_error_expression (ValaThrowStatement* self);
void vala_code_node_accept (ValaCodeNode* self, ValaCodeVisitor* visitor);
void vala_code_visitor_visit_end_full_expression (ValaCodeVisitor* self, ValaExpression* expr);
static void vala_throw_statement_real_accept_children (ValaCodeNode* base, ValaCodeVisitor* visitor);
static void vala_throw_statement_real_replace_expression (ValaCodeNode* base, ValaExpression* old_node, ValaExpression* new_node);
gboolean vala_code_node_get_checked (ValaCodeNode* self);
gboolean vala_code_node_get_error (ValaCodeNode* self);
void vala_code_node_set_checked (ValaCodeNode* self, gboolean value);
ValaSourceReference* vala_code_node_get_source_reference (ValaCodeNode* self);
GType vala_typesymbol_get_type (void);
GType vala_error_domain_get_type (void);
GType vala_error_code_get_type (void);
ValaErrorType* vala_error_type_new (ValaErrorDomain* error_domain, ValaErrorCode* error_code, ValaSourceReference* source_reference);
ValaErrorType* vala_error_type_construct (GType object_type, ValaErrorDomain* error_domain, ValaErrorCode* error_code, ValaSourceReference* source_reference);
GType vala_reference_type_get_type (void);
GType vala_error_type_get_type (void);
void vala_expression_set_target_type (ValaExpression* self, ValaDataType* value);
void vala_data_type_set_value_owned (ValaDataType* self, gboolean value);
ValaDataType* vala_expression_get_target_type (ValaExpression* self);
gboolean vala_code_node_check (ValaCodeNode* self, ValaSemanticAnalyzer* analyzer);
void vala_code_node_set_error (ValaCodeNode* self, gboolean value);
ValaDataType* vala_expression_get_value_type (ValaExpression* self);
void vala_report_error (ValaSourceReference* source, const char* message);
char* vala_code_node_to_string (ValaCodeNode* self);
ValaDataType* vala_data_type_copy (ValaDataType* self);
void vala_code_node_add_error_type (ValaCodeNode* self, ValaDataType* error_type);
static gboolean vala_throw_statement_real_check (ValaCodeNode* base, ValaSemanticAnalyzer* analyzer);
void vala_code_node_get_defined_variables (ValaCodeNode* self, GeeCollection* collection);
static void vala_throw_statement_real_get_defined_variables (ValaCodeNode* base, GeeCollection* collection);
void vala_code_node_get_used_variables (ValaCodeNode* self, GeeCollection* collection);
static void vala_throw_statement_real_get_used_variables (ValaCodeNode* base, GeeCollection* collection);
void vala_code_node_set_parent_node (ValaCodeNode* self, ValaCodeNode* value);
static gpointer vala_throw_statement_parent_class = NULL;
static ValaStatementIface* vala_throw_statement_vala_statement_parent_iface = NULL;
static void vala_throw_statement_finalize (ValaCodeNode* obj);



/**
 * Creates a new throw statement.
 *
 * @param error_expression the error expression
 * @param source_reference reference to source code
 * @return                 newly created throw statement
 */
ValaThrowStatement* vala_throw_statement_construct (GType object_type, ValaExpression* error_expression, ValaSourceReference* source_reference) {
	ValaThrowStatement* self;
	g_return_val_if_fail (error_expression != NULL, NULL);
	self = (ValaThrowStatement*) g_type_create_instance (object_type);
	vala_code_node_set_source_reference ((ValaCodeNode*) self, source_reference);
	vala_throw_statement_set_error_expression (self, error_expression);
	return self;
}


ValaThrowStatement* vala_throw_statement_new (ValaExpression* error_expression, ValaSourceReference* source_reference) {
	return vala_throw_statement_construct (VALA_TYPE_THROW_STATEMENT, error_expression, source_reference);
}


static void vala_throw_statement_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor) {
	ValaThrowStatement * self;
	self = (ValaThrowStatement*) base;
	g_return_if_fail (visitor != NULL);
	vala_code_visitor_visit_throw_statement (visitor, self);
}


static void vala_throw_statement_real_accept_children (ValaCodeNode* base, ValaCodeVisitor* visitor) {
	ValaThrowStatement * self;
	self = (ValaThrowStatement*) base;
	g_return_if_fail (visitor != NULL);
	if (vala_throw_statement_get_error_expression (self) != NULL) {
		vala_code_node_accept ((ValaCodeNode*) vala_throw_statement_get_error_expression (self), visitor);
		vala_code_visitor_visit_end_full_expression (visitor, vala_throw_statement_get_error_expression (self));
	}
}


static void vala_throw_statement_real_replace_expression (ValaCodeNode* base, ValaExpression* old_node, ValaExpression* new_node) {
	ValaThrowStatement * self;
	self = (ValaThrowStatement*) base;
	g_return_if_fail (old_node != NULL);
	g_return_if_fail (new_node != NULL);
	if (vala_throw_statement_get_error_expression (self) == old_node) {
		vala_throw_statement_set_error_expression (self, new_node);
	}
}


static gboolean vala_throw_statement_real_check (ValaCodeNode* base, ValaSemanticAnalyzer* analyzer) {
	ValaThrowStatement * self;
	ValaErrorType* _tmp1;
	ValaDataType* error_type;
	gboolean _tmp7;
	self = (ValaThrowStatement*) base;
	g_return_val_if_fail (analyzer != NULL, FALSE);
	if (vala_code_node_get_checked ((ValaCodeNode*) self)) {
		return !vala_code_node_get_error ((ValaCodeNode*) self);
	}
	vala_code_node_set_checked ((ValaCodeNode*) self, TRUE);
	_tmp1 = NULL;
	vala_expression_set_target_type (vala_throw_statement_get_error_expression (self), (ValaDataType*) (_tmp1 = vala_error_type_new (NULL, NULL, vala_code_node_get_source_reference ((ValaCodeNode*) self))));
	(_tmp1 == NULL) ? NULL : (_tmp1 = (vala_code_node_unref (_tmp1), NULL));
	vala_data_type_set_value_owned (vala_expression_get_target_type (vala_throw_statement_get_error_expression (self)), TRUE);
	if (vala_throw_statement_get_error_expression (self) != NULL) {
		if (!vala_code_node_check ((ValaCodeNode*) vala_throw_statement_get_error_expression (self), analyzer)) {
			vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
			return FALSE;
		}
		if (vala_expression_get_value_type (vala_throw_statement_get_error_expression (self)) == NULL) {
			vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) vala_throw_statement_get_error_expression (self)), "invalid error expression");
			vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
			return FALSE;
		}
		if (!VALA_IS_ERROR_TYPE (vala_expression_get_value_type (vala_throw_statement_get_error_expression (self)))) {
			char* _tmp5;
			char* _tmp4;
			_tmp5 = NULL;
			_tmp4 = NULL;
			vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) vala_throw_statement_get_error_expression (self)), _tmp5 = g_strdup_printf ("`%s' is not an error type", _tmp4 = vala_code_node_to_string ((ValaCodeNode*) vala_expression_get_value_type (vala_throw_statement_get_error_expression (self)))));
			_tmp5 = (g_free (_tmp5), NULL);
			_tmp4 = (g_free (_tmp4), NULL);
			vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
			return FALSE;
		}
	}
	error_type = vala_data_type_copy (vala_expression_get_value_type (vala_throw_statement_get_error_expression (self)));
	vala_code_node_set_source_reference ((ValaCodeNode*) error_type, vala_code_node_get_source_reference ((ValaCodeNode*) self));
	vala_code_node_add_error_type ((ValaCodeNode*) self, error_type);
	return (_tmp7 = !vala_code_node_get_error ((ValaCodeNode*) self), (error_type == NULL) ? NULL : (error_type = (vala_code_node_unref (error_type), NULL)), _tmp7);
}


static void vala_throw_statement_real_get_defined_variables (ValaCodeNode* base, GeeCollection* collection) {
	ValaThrowStatement * self;
	self = (ValaThrowStatement*) base;
	g_return_if_fail (collection != NULL);
	vala_code_node_get_defined_variables ((ValaCodeNode*) vala_throw_statement_get_error_expression (self), collection);
}


static void vala_throw_statement_real_get_used_variables (ValaCodeNode* base, GeeCollection* collection) {
	ValaThrowStatement * self;
	self = (ValaThrowStatement*) base;
	g_return_if_fail (collection != NULL);
	vala_code_node_get_used_variables ((ValaCodeNode*) vala_throw_statement_get_error_expression (self), collection);
}


ValaExpression* vala_throw_statement_get_error_expression (ValaThrowStatement* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return self->priv->_error_expression;
}


void vala_throw_statement_set_error_expression (ValaThrowStatement* self, ValaExpression* value) {
	ValaExpression* _tmp2;
	ValaExpression* _tmp1;
	g_return_if_fail (self != NULL);
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_error_expression = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL) ? NULL : vala_code_node_ref (_tmp1)), (self->priv->_error_expression == NULL) ? NULL : (self->priv->_error_expression = (vala_code_node_unref (self->priv->_error_expression), NULL)), _tmp2);
	if (self->priv->_error_expression != NULL) {
		vala_code_node_set_parent_node ((ValaCodeNode*) self->priv->_error_expression, (ValaCodeNode*) self);
	}
}


static void vala_throw_statement_class_init (ValaThrowStatementClass * klass) {
	vala_throw_statement_parent_class = g_type_class_peek_parent (klass);
	VALA_CODE_NODE_CLASS (klass)->finalize = vala_throw_statement_finalize;
	g_type_class_add_private (klass, sizeof (ValaThrowStatementPrivate));
	VALA_CODE_NODE_CLASS (klass)->accept = vala_throw_statement_real_accept;
	VALA_CODE_NODE_CLASS (klass)->accept_children = vala_throw_statement_real_accept_children;
	VALA_CODE_NODE_CLASS (klass)->replace_expression = vala_throw_statement_real_replace_expression;
	VALA_CODE_NODE_CLASS (klass)->check = vala_throw_statement_real_check;
	VALA_CODE_NODE_CLASS (klass)->get_defined_variables = vala_throw_statement_real_get_defined_variables;
	VALA_CODE_NODE_CLASS (klass)->get_used_variables = vala_throw_statement_real_get_used_variables;
}


static void vala_throw_statement_vala_statement_interface_init (ValaStatementIface * iface) {
	vala_throw_statement_vala_statement_parent_iface = g_type_interface_peek_parent (iface);
}


static void vala_throw_statement_instance_init (ValaThrowStatement * self) {
	self->priv = VALA_THROW_STATEMENT_GET_PRIVATE (self);
}


static void vala_throw_statement_finalize (ValaCodeNode* obj) {
	ValaThrowStatement * self;
	self = VALA_THROW_STATEMENT (obj);
	(self->priv->_error_expression == NULL) ? NULL : (self->priv->_error_expression = (vala_code_node_unref (self->priv->_error_expression), NULL));
	VALA_CODE_NODE_CLASS (vala_throw_statement_parent_class)->finalize (obj);
}


GType vala_throw_statement_get_type (void) {
	static GType vala_throw_statement_type_id = 0;
	if (vala_throw_statement_type_id == 0) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaThrowStatementClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_throw_statement_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaThrowStatement), 0, (GInstanceInitFunc) vala_throw_statement_instance_init, NULL };
		static const GInterfaceInfo vala_statement_info = { (GInterfaceInitFunc) vala_throw_statement_vala_statement_interface_init, (GInterfaceFinalizeFunc) NULL, NULL};
		vala_throw_statement_type_id = g_type_register_static (VALA_TYPE_CODE_NODE, "ValaThrowStatement", &g_define_type_info, 0);
		g_type_add_interface_static (vala_throw_statement_type_id, VALA_TYPE_STATEMENT, &vala_statement_info);
	}
	return vala_throw_statement_type_id;
}




