/* valadatatype.vala
 *
 * Copyright (C) 2006-2008  Jürg Billeter, Raffaele Sandrini
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 *	Raffaele Sandrini <raffaele@sandrini.ch>
 */

#include <glib.h>
#include <glib-object.h>
#include <stdlib.h>
#include <string.h>
#include <gee.h>


#define VALA_TYPE_CODE_NODE (vala_code_node_get_type ())
#define VALA_CODE_NODE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_CODE_NODE, ValaCodeNode))
#define VALA_CODE_NODE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_CODE_NODE, ValaCodeNodeClass))
#define VALA_IS_CODE_NODE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_CODE_NODE))
#define VALA_IS_CODE_NODE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_CODE_NODE))
#define VALA_CODE_NODE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_CODE_NODE, ValaCodeNodeClass))

typedef struct _ValaCodeNode ValaCodeNode;
typedef struct _ValaCodeNodeClass ValaCodeNodeClass;
typedef struct _ValaCodeNodePrivate ValaCodeNodePrivate;

#define VALA_TYPE_CODE_VISITOR (vala_code_visitor_get_type ())
#define VALA_CODE_VISITOR(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_CODE_VISITOR, ValaCodeVisitor))
#define VALA_CODE_VISITOR_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_CODE_VISITOR, ValaCodeVisitorClass))
#define VALA_IS_CODE_VISITOR(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_CODE_VISITOR))
#define VALA_IS_CODE_VISITOR_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_CODE_VISITOR))
#define VALA_CODE_VISITOR_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_CODE_VISITOR, ValaCodeVisitorClass))

typedef struct _ValaCodeVisitor ValaCodeVisitor;
typedef struct _ValaCodeVisitorClass ValaCodeVisitorClass;

#define VALA_TYPE_SEMANTIC_ANALYZER (vala_semantic_analyzer_get_type ())
#define VALA_SEMANTIC_ANALYZER(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_SEMANTIC_ANALYZER, ValaSemanticAnalyzer))
#define VALA_SEMANTIC_ANALYZER_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_SEMANTIC_ANALYZER, ValaSemanticAnalyzerClass))
#define VALA_IS_SEMANTIC_ANALYZER(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_SEMANTIC_ANALYZER))
#define VALA_IS_SEMANTIC_ANALYZER_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_SEMANTIC_ANALYZER))
#define VALA_SEMANTIC_ANALYZER_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_SEMANTIC_ANALYZER, ValaSemanticAnalyzerClass))

typedef struct _ValaSemanticAnalyzer ValaSemanticAnalyzer;
typedef struct _ValaSemanticAnalyzerClass ValaSemanticAnalyzerClass;

#define VALA_TYPE_DATA_TYPE (vala_data_type_get_type ())
#define VALA_DATA_TYPE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_DATA_TYPE, ValaDataType))
#define VALA_DATA_TYPE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_DATA_TYPE, ValaDataTypeClass))
#define VALA_IS_DATA_TYPE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_DATA_TYPE))
#define VALA_IS_DATA_TYPE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_DATA_TYPE))
#define VALA_DATA_TYPE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_DATA_TYPE, ValaDataTypeClass))

typedef struct _ValaDataType ValaDataType;
typedef struct _ValaDataTypeClass ValaDataTypeClass;

#define VALA_TYPE_EXPRESSION (vala_expression_get_type ())
#define VALA_EXPRESSION(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_EXPRESSION, ValaExpression))
#define VALA_EXPRESSION_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_EXPRESSION, ValaExpressionClass))
#define VALA_IS_EXPRESSION(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_EXPRESSION))
#define VALA_IS_EXPRESSION_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_EXPRESSION))
#define VALA_EXPRESSION_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_EXPRESSION, ValaExpressionClass))

typedef struct _ValaExpression ValaExpression;
typedef struct _ValaExpressionClass ValaExpressionClass;

#define VALA_TYPE_SYMBOL (vala_symbol_get_type ())
#define VALA_SYMBOL(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_SYMBOL, ValaSymbol))
#define VALA_SYMBOL_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_SYMBOL, ValaSymbolClass))
#define VALA_IS_SYMBOL(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_SYMBOL))
#define VALA_IS_SYMBOL_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_SYMBOL))
#define VALA_SYMBOL_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_SYMBOL, ValaSymbolClass))

typedef struct _ValaSymbol ValaSymbol;
typedef struct _ValaSymbolClass ValaSymbolClass;

#define VALA_TYPE_LOCAL_VARIABLE (vala_local_variable_get_type ())
#define VALA_LOCAL_VARIABLE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_LOCAL_VARIABLE, ValaLocalVariable))
#define VALA_LOCAL_VARIABLE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_LOCAL_VARIABLE, ValaLocalVariableClass))
#define VALA_IS_LOCAL_VARIABLE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_LOCAL_VARIABLE))
#define VALA_IS_LOCAL_VARIABLE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_LOCAL_VARIABLE))
#define VALA_LOCAL_VARIABLE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_LOCAL_VARIABLE, ValaLocalVariableClass))

typedef struct _ValaLocalVariable ValaLocalVariable;
typedef struct _ValaLocalVariableClass ValaLocalVariableClass;

#define VALA_TYPE_ATTRIBUTE (vala_attribute_get_type ())
#define VALA_ATTRIBUTE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_ATTRIBUTE, ValaAttribute))
#define VALA_ATTRIBUTE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_ATTRIBUTE, ValaAttributeClass))
#define VALA_IS_ATTRIBUTE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_ATTRIBUTE))
#define VALA_IS_ATTRIBUTE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_ATTRIBUTE))
#define VALA_ATTRIBUTE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_ATTRIBUTE, ValaAttributeClass))

typedef struct _ValaAttribute ValaAttribute;
typedef struct _ValaAttributeClass ValaAttributeClass;
typedef struct _ValaDataTypePrivate ValaDataTypePrivate;

#define VALA_TYPE_SCOPE (vala_scope_get_type ())
#define VALA_SCOPE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_SCOPE, ValaScope))
#define VALA_SCOPE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_SCOPE, ValaScopeClass))
#define VALA_IS_SCOPE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_SCOPE))
#define VALA_IS_SCOPE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_SCOPE))
#define VALA_SCOPE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_SCOPE, ValaScopeClass))

typedef struct _ValaScope ValaScope;
typedef struct _ValaScopeClass ValaScopeClass;

#define VALA_TYPE_FORMAL_PARAMETER (vala_formal_parameter_get_type ())
#define VALA_FORMAL_PARAMETER(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_FORMAL_PARAMETER, ValaFormalParameter))
#define VALA_FORMAL_PARAMETER_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_FORMAL_PARAMETER, ValaFormalParameterClass))
#define VALA_IS_FORMAL_PARAMETER(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_FORMAL_PARAMETER))
#define VALA_IS_FORMAL_PARAMETER_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_FORMAL_PARAMETER))
#define VALA_FORMAL_PARAMETER_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_FORMAL_PARAMETER, ValaFormalParameterClass))

typedef struct _ValaFormalParameter ValaFormalParameter;
typedef struct _ValaFormalParameterClass ValaFormalParameterClass;

#define VALA_TYPE_TYPESYMBOL (vala_typesymbol_get_type ())
#define VALA_TYPESYMBOL(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_TYPESYMBOL, ValaTypeSymbol))
#define VALA_TYPESYMBOL_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_TYPESYMBOL, ValaTypeSymbolClass))
#define VALA_IS_TYPESYMBOL(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_TYPESYMBOL))
#define VALA_IS_TYPESYMBOL_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_TYPESYMBOL))
#define VALA_TYPESYMBOL_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_TYPESYMBOL, ValaTypeSymbolClass))

typedef struct _ValaTypeSymbol ValaTypeSymbol;
typedef struct _ValaTypeSymbolClass ValaTypeSymbolClass;

#define VALA_TYPE_TYPEPARAMETER (vala_typeparameter_get_type ())
#define VALA_TYPEPARAMETER(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_TYPEPARAMETER, ValaTypeParameter))
#define VALA_TYPEPARAMETER_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_TYPEPARAMETER, ValaTypeParameterClass))
#define VALA_IS_TYPEPARAMETER(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_TYPEPARAMETER))
#define VALA_IS_TYPEPARAMETER_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_TYPEPARAMETER))
#define VALA_TYPEPARAMETER_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_TYPEPARAMETER, ValaTypeParameterClass))

typedef struct _ValaTypeParameter ValaTypeParameter;
typedef struct _ValaTypeParameterClass ValaTypeParameterClass;

#define VALA_TYPE_SOURCE_REFERENCE (vala_source_reference_get_type ())
#define VALA_SOURCE_REFERENCE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_SOURCE_REFERENCE, ValaSourceReference))
#define VALA_SOURCE_REFERENCE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_SOURCE_REFERENCE, ValaSourceReferenceClass))
#define VALA_IS_SOURCE_REFERENCE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_SOURCE_REFERENCE))
#define VALA_IS_SOURCE_REFERENCE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_SOURCE_REFERENCE))
#define VALA_SOURCE_REFERENCE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_SOURCE_REFERENCE, ValaSourceReferenceClass))

typedef struct _ValaSourceReference ValaSourceReference;
typedef struct _ValaSourceReferenceClass ValaSourceReferenceClass;

#define VALA_TYPE_REFERENCE_TYPE (vala_reference_type_get_type ())
#define VALA_REFERENCE_TYPE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_REFERENCE_TYPE, ValaReferenceType))
#define VALA_REFERENCE_TYPE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_REFERENCE_TYPE, ValaReferenceTypeClass))
#define VALA_IS_REFERENCE_TYPE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_REFERENCE_TYPE))
#define VALA_IS_REFERENCE_TYPE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_REFERENCE_TYPE))
#define VALA_REFERENCE_TYPE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_REFERENCE_TYPE, ValaReferenceTypeClass))

typedef struct _ValaReferenceType ValaReferenceType;
typedef struct _ValaReferenceTypeClass ValaReferenceTypeClass;

#define VALA_TYPE_ARRAY_TYPE (vala_array_type_get_type ())
#define VALA_ARRAY_TYPE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_ARRAY_TYPE, ValaArrayType))
#define VALA_ARRAY_TYPE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_ARRAY_TYPE, ValaArrayTypeClass))
#define VALA_IS_ARRAY_TYPE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_ARRAY_TYPE))
#define VALA_IS_ARRAY_TYPE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_ARRAY_TYPE))
#define VALA_ARRAY_TYPE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_ARRAY_TYPE, ValaArrayTypeClass))

typedef struct _ValaArrayType ValaArrayType;
typedef struct _ValaArrayTypeClass ValaArrayTypeClass;

#define VALA_TYPE_DELEGATE_TYPE (vala_delegate_type_get_type ())
#define VALA_DELEGATE_TYPE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_DELEGATE_TYPE, ValaDelegateType))
#define VALA_DELEGATE_TYPE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_DELEGATE_TYPE, ValaDelegateTypeClass))
#define VALA_IS_DELEGATE_TYPE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_DELEGATE_TYPE))
#define VALA_IS_DELEGATE_TYPE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_DELEGATE_TYPE))
#define VALA_DELEGATE_TYPE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_DELEGATE_TYPE, ValaDelegateTypeClass))

typedef struct _ValaDelegateType ValaDelegateType;
typedef struct _ValaDelegateTypeClass ValaDelegateTypeClass;

#define VALA_TYPE_DELEGATE (vala_delegate_get_type ())
#define VALA_DELEGATE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_DELEGATE, ValaDelegate))
#define VALA_DELEGATE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_DELEGATE, ValaDelegateClass))
#define VALA_IS_DELEGATE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_DELEGATE))
#define VALA_IS_DELEGATE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_DELEGATE))
#define VALA_DELEGATE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_DELEGATE, ValaDelegateClass))

typedef struct _ValaDelegate ValaDelegate;
typedef struct _ValaDelegateClass ValaDelegateClass;

#define VALA_TYPE_POINTER_TYPE (vala_pointer_type_get_type ())
#define VALA_POINTER_TYPE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_POINTER_TYPE, ValaPointerType))
#define VALA_POINTER_TYPE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_POINTER_TYPE, ValaPointerTypeClass))
#define VALA_IS_POINTER_TYPE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_POINTER_TYPE))
#define VALA_IS_POINTER_TYPE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_POINTER_TYPE))
#define VALA_POINTER_TYPE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_POINTER_TYPE, ValaPointerTypeClass))

typedef struct _ValaPointerType ValaPointerType;
typedef struct _ValaPointerTypeClass ValaPointerTypeClass;

#define VALA_TYPE_ENUM (vala_enum_get_type ())
#define VALA_ENUM(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_ENUM, ValaEnum))
#define VALA_ENUM_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_ENUM, ValaEnumClass))
#define VALA_IS_ENUM(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_ENUM))
#define VALA_IS_ENUM_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_ENUM))
#define VALA_ENUM_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_ENUM, ValaEnumClass))

typedef struct _ValaEnum ValaEnum;
typedef struct _ValaEnumClass ValaEnumClass;

#define VALA_TYPE_STRUCT (vala_struct_get_type ())
#define VALA_STRUCT(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_STRUCT, ValaStruct))
#define VALA_STRUCT_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_STRUCT, ValaStructClass))
#define VALA_IS_STRUCT(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_STRUCT))
#define VALA_IS_STRUCT_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_STRUCT))
#define VALA_STRUCT_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_STRUCT, ValaStructClass))

typedef struct _ValaStruct ValaStruct;
typedef struct _ValaStructClass ValaStructClass;

#define VALA_TYPE_GENERIC_TYPE (vala_generic_type_get_type ())
#define VALA_GENERIC_TYPE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_GENERIC_TYPE, ValaGenericType))
#define VALA_GENERIC_TYPE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_GENERIC_TYPE, ValaGenericTypeClass))
#define VALA_IS_GENERIC_TYPE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_GENERIC_TYPE))
#define VALA_IS_GENERIC_TYPE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_GENERIC_TYPE))
#define VALA_GENERIC_TYPE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_GENERIC_TYPE, ValaGenericTypeClass))

typedef struct _ValaGenericType ValaGenericType;
typedef struct _ValaGenericTypeClass ValaGenericTypeClass;

/**
 * Represents a part of the parsed source code.
 *
 * Code nodes get created by the parser and are used throughout the whole
 * compilation process.
 */
struct _ValaCodeNode {
	GTypeInstance parent_instance;
	volatile int ref_count;
	ValaCodeNodePrivate * priv;
	GList* attributes;
};

struct _ValaCodeNodeClass {
	GTypeClass parent_class;
	void (*finalize) (ValaCodeNode *self);
	void (*accept) (ValaCodeNode* self, ValaCodeVisitor* visitor);
	void (*accept_children) (ValaCodeNode* self, ValaCodeVisitor* visitor);
	gboolean (*check) (ValaCodeNode* self, ValaSemanticAnalyzer* analyzer);
	void (*replace_type) (ValaCodeNode* self, ValaDataType* old_type, ValaDataType* new_type);
	void (*replace_expression) (ValaCodeNode* self, ValaExpression* old_node, ValaExpression* new_node);
	char* (*to_string) (ValaCodeNode* self);
	void (*get_defined_variables) (ValaCodeNode* self, GeeCollection* collection);
	void (*get_used_variables) (ValaCodeNode* self, GeeCollection* collection);
};

/**
 * A reference to a data type. This is used to specify static types of
 * expressions.
 */
struct _ValaDataType {
	ValaCodeNode parent_instance;
	ValaDataTypePrivate * priv;
};

struct _ValaDataTypeClass {
	ValaCodeNodeClass parent_class;
	char* (*get_cname) (ValaDataType* self);
	char* (*get_cdeclarator_suffix) (ValaDataType* self);
	char* (*get_lower_case_cname) (ValaDataType* self, const char* infix);
	char* (*to_qualified_string) (ValaDataType* self, ValaScope* scope);
	ValaDataType* (*copy) (ValaDataType* self);
	gboolean (*equals) (ValaDataType* self, ValaDataType* type2);
	gboolean (*stricter) (ValaDataType* self, ValaDataType* type2);
	gboolean (*compatible) (ValaDataType* self, ValaDataType* target_type);
	gboolean (*is_invokable) (ValaDataType* self);
	ValaDataType* (*get_return_type) (ValaDataType* self);
	GeeList* (*get_parameters) (ValaDataType* self);
	gboolean (*is_reference_type_or_type_parameter) (ValaDataType* self);
	gboolean (*is_array) (ValaDataType* self);
	GeeList* (*get_symbols) (ValaDataType* self);
	ValaSymbol* (*get_member) (ValaDataType* self, const char* member_name);
	ValaSymbol* (*get_pointer_member) (ValaDataType* self, const char* member_name);
	gboolean (*is_real_struct_type) (ValaDataType* self);
	char* (*get_type_id) (ValaDataType* self);
	char* (*get_type_signature) (ValaDataType* self);
	gboolean (*is_disposable) (ValaDataType* self);
};

struct _ValaDataTypePrivate {
	gboolean _value_owned;
	gboolean _nullable;
	ValaTypeSymbol* _data_type;
	ValaTypeParameter* _type_parameter;
	gboolean _floating_reference;
	gboolean _is_dynamic;
	GeeList* type_argument_list;
};



gpointer vala_code_node_ref (gpointer instance);
void vala_code_node_unref (gpointer instance);
GParamSpec* vala_param_spec_code_node (const gchar* name, const gchar* nick, const gchar* blurb, GType object_type, GParamFlags flags);
void vala_value_set_code_node (GValue* value, gpointer v_object);
gpointer vala_value_get_code_node (const GValue* value);
GType vala_code_node_get_type (void);
gpointer vala_code_visitor_ref (gpointer instance);
void vala_code_visitor_unref (gpointer instance);
GParamSpec* vala_param_spec_code_visitor (const gchar* name, const gchar* nick, const gchar* blurb, GType object_type, GParamFlags flags);
void vala_value_set_code_visitor (GValue* value, gpointer v_object);
gpointer vala_value_get_code_visitor (const GValue* value);
GType vala_code_visitor_get_type (void);
GType vala_semantic_analyzer_get_type (void);
GType vala_data_type_get_type (void);
GType vala_expression_get_type (void);
GType vala_symbol_get_type (void);
GType vala_local_variable_get_type (void);
GType vala_attribute_get_type (void);
gpointer vala_scope_ref (gpointer instance);
void vala_scope_unref (gpointer instance);
GParamSpec* vala_param_spec_scope (const gchar* name, const gchar* nick, const gchar* blurb, GType object_type, GParamFlags flags);
void vala_value_set_scope (GValue* value, gpointer v_object);
gpointer vala_value_get_scope (const GValue* value);
GType vala_scope_get_type (void);
GType vala_formal_parameter_get_type (void);
GType vala_typesymbol_get_type (void);
GType vala_typeparameter_get_type (void);
#define VALA_DATA_TYPE_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_DATA_TYPE, ValaDataTypePrivate))
enum  {
	VALA_DATA_TYPE_DUMMY_PROPERTY
};
static GeeList* vala_data_type__empty_type_list;
static GeeList* vala_data_type__empty_type_list = NULL;
void vala_code_node_set_parent_node (ValaCodeNode* self, ValaCodeNode* value);
void vala_data_type_add_type_argument (ValaDataType* self, ValaDataType* arg);
GeeList* vala_data_type_get_type_arguments (ValaDataType* self);
void vala_data_type_remove_all_type_arguments (ValaDataType* self);
void vala_code_node_accept (ValaCodeNode* self, ValaCodeVisitor* visitor);
void vala_code_visitor_visit_data_type (ValaCodeVisitor* self, ValaDataType* type);
static void vala_data_type_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor);
gpointer vala_source_reference_ref (gpointer instance);
void vala_source_reference_unref (gpointer instance);
GParamSpec* vala_param_spec_source_reference (const gchar* name, const gchar* nick, const gchar* blurb, GType object_type, GParamFlags flags);
void vala_value_set_source_reference (GValue* value, gpointer v_object);
gpointer vala_value_get_source_reference (const GValue* value);
GType vala_source_reference_get_type (void);
void vala_report_error (ValaSourceReference* source, const char* message);
ValaSourceReference* vala_code_node_get_source_reference (ValaCodeNode* self);
char* vala_data_type_get_cname (ValaDataType* self);
static char* vala_data_type_real_get_cname (ValaDataType* self);
char* vala_data_type_get_cdeclarator_suffix (ValaDataType* self);
static char* vala_data_type_real_get_cdeclarator_suffix (ValaDataType* self);
GType vala_reference_type_get_type (void);
GType vala_array_type_get_type (void);
ValaDataType* vala_array_type_get_element_type (ValaArrayType* self);
ValaTypeSymbol* vala_data_type_get_data_type (ValaDataType* self);
gboolean vala_typesymbol_is_reference_type (ValaTypeSymbol* self);
char* vala_typesymbol_get_cname (ValaTypeSymbol* self, gboolean const_type);
char* vala_data_type_get_const_cname (ValaDataType* self);
char* vala_symbol_get_lower_case_cname (ValaSymbol* self, const char* infix);
char* vala_data_type_get_lower_case_cname (ValaDataType* self, const char* infix);
static char* vala_data_type_real_get_lower_case_cname (ValaDataType* self, const char* infix);
char* vala_data_type_to_qualified_string (ValaDataType* self, ValaScope* scope);
static char* vala_data_type_real_to_string (ValaCodeNode* base);
ValaSymbol* vala_symbol_get_parent_symbol (ValaSymbol* self);
const char* vala_symbol_get_name (ValaSymbol* self);
ValaSymbol* vala_scope_lookup (ValaScope* self, const char* name);
ValaScope* vala_scope_get_parent_scope (ValaScope* self);
char* vala_symbol_get_full_name (ValaSymbol* self);
gboolean vala_data_type_get_value_owned (ValaDataType* self);
gboolean vala_data_type_get_nullable (ValaDataType* self);
static char* vala_data_type_real_to_qualified_string (ValaDataType* self, ValaScope* scope);
ValaDataType* vala_data_type_copy (ValaDataType* self);
static ValaDataType* vala_data_type_real_copy (ValaDataType* self);
ValaTypeParameter* vala_data_type_get_type_parameter (ValaDataType* self);
gboolean vala_typeparameter_equals (ValaTypeParameter* self, ValaTypeParameter* param2);
gboolean vala_data_type_get_floating_reference (ValaDataType* self);
gboolean vala_data_type_equals (ValaDataType* self, ValaDataType* type2);
static gboolean vala_data_type_real_equals (ValaDataType* self, ValaDataType* type2);
gboolean vala_data_type_stricter (ValaDataType* self, ValaDataType* type2);
static gboolean vala_data_type_real_stricter (ValaDataType* self, ValaDataType* type2);
static void vala_data_type_real_replace_type (ValaCodeNode* base, ValaDataType* old_type, ValaDataType* new_type);
GType vala_delegate_type_get_type (void);
GType vala_delegate_get_type (void);
ValaDelegate* vala_delegate_type_get_delegate_symbol (ValaDelegateType* self);
GType vala_pointer_type_get_type (void);
GType vala_enum_get_type (void);
GType vala_struct_get_type (void);
gboolean vala_struct_is_integer_type (ValaStruct* self);
gboolean vala_struct_is_floating_type (ValaStruct* self);
gint vala_struct_get_rank (ValaStruct* self);
gboolean vala_typesymbol_is_subtype_of (ValaTypeSymbol* self, ValaTypeSymbol* t);
gboolean vala_data_type_compatible (ValaDataType* self, ValaDataType* target_type);
static gboolean vala_data_type_real_compatible (ValaDataType* self, ValaDataType* target_type);
gboolean vala_data_type_is_invokable (ValaDataType* self);
static gboolean vala_data_type_real_is_invokable (ValaDataType* self);
ValaDataType* vala_data_type_get_return_type (ValaDataType* self);
static ValaDataType* vala_data_type_real_get_return_type (ValaDataType* self);
GeeList* vala_data_type_get_parameters (ValaDataType* self);
static GeeList* vala_data_type_real_get_parameters (ValaDataType* self);
gboolean vala_data_type_is_reference_type_or_type_parameter (ValaDataType* self);
static gboolean vala_data_type_real_is_reference_type_or_type_parameter (ValaDataType* self);
gboolean vala_data_type_is_array (ValaDataType* self);
static gboolean vala_data_type_real_is_array (ValaDataType* self);
GeeList* vala_data_type_get_symbols (ValaDataType* self);
static GeeList* vala_data_type_real_get_symbols (ValaDataType* self);
ValaSymbol* vala_semantic_analyzer_symbol_lookup_inherited (ValaSymbol* sym, const char* name);
ValaSymbol* vala_data_type_get_member (ValaDataType* self, const char* member_name);
static ValaSymbol* vala_data_type_real_get_member (ValaDataType* self, const char* member_name);
ValaSymbol* vala_data_type_get_pointer_member (ValaDataType* self, const char* member_name);
static ValaSymbol* vala_data_type_real_get_pointer_member (ValaDataType* self, const char* member_name);
gboolean vala_struct_is_simple_type (ValaStruct* self);
gboolean vala_data_type_is_real_struct_type (ValaDataType* self);
static gboolean vala_data_type_real_is_real_struct_type (ValaDataType* self);
char* vala_typesymbol_get_type_id (ValaTypeSymbol* self);
char* vala_data_type_get_type_id (ValaDataType* self);
static char* vala_data_type_real_get_type_id (ValaDataType* self);
char* vala_typesymbol_get_type_signature (ValaTypeSymbol* self);
char* vala_data_type_get_type_signature (ValaDataType* self);
static char* vala_data_type_real_get_type_signature (ValaDataType* self);
gboolean vala_data_type_is_disposable (ValaDataType* self);
static gboolean vala_data_type_real_is_disposable (ValaDataType* self);
GType vala_generic_type_get_type (void);
ValaDataType* vala_semantic_analyzer_get_actual_type (ValaDataType* derived_instance_type, ValaGenericType* generic_type, ValaCodeNode* node_reference);
ValaDataType* vala_data_type_get_actual_type (ValaDataType* self, ValaDataType* derived_instance_type, ValaCodeNode* node_reference);
void vala_data_type_set_value_owned (ValaDataType* self, gboolean value);
void vala_data_type_set_nullable (ValaDataType* self, gboolean value);
void vala_data_type_set_data_type (ValaDataType* self, ValaTypeSymbol* value);
void vala_data_type_set_type_parameter (ValaDataType* self, ValaTypeParameter* value);
void vala_data_type_set_floating_reference (ValaDataType* self, gboolean value);
gboolean vala_data_type_get_is_dynamic (ValaDataType* self);
void vala_data_type_set_is_dynamic (ValaDataType* self, gboolean value);
static gpointer vala_data_type_parent_class = NULL;
static void vala_data_type_finalize (ValaCodeNode* obj);



/**
 * Appends the specified type as generic type argument.
 *
 * @param arg a type reference
 */
void vala_data_type_add_type_argument (ValaDataType* self, ValaDataType* arg) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (arg != NULL);
	if (self->priv->type_argument_list == NULL) {
		GeeList* _tmp0_;
		_tmp0_ = NULL;
		self->priv->type_argument_list = (_tmp0_ = (GeeList*) gee_array_list_new (VALA_TYPE_DATA_TYPE, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, g_direct_equal), (self->priv->type_argument_list == NULL) ? NULL : (self->priv->type_argument_list = (gee_collection_object_unref (self->priv->type_argument_list), NULL)), _tmp0_);
	}
	gee_collection_add ((GeeCollection*) self->priv->type_argument_list, arg);
	vala_code_node_set_parent_node ((ValaCodeNode*) arg, (ValaCodeNode*) self);
}


/**
 * Returns a copy of the list of generic type arguments.
 *
 * @return type argument list
 */
GeeList* vala_data_type_get_type_arguments (ValaDataType* self) {
	GeeList* _tmp4_;
	g_return_val_if_fail (self != NULL, NULL);
	if (self->priv->type_argument_list != NULL) {
		GeeList* _tmp0_;
		_tmp0_ = NULL;
		return (_tmp0_ = self->priv->type_argument_list, (_tmp0_ == NULL) ? NULL : gee_collection_object_ref (_tmp0_));
	}
	if (vala_data_type__empty_type_list == NULL) {
		GeeList* _tmp3_;
		GeeArrayList* _tmp2_;
		_tmp3_ = NULL;
		_tmp2_ = NULL;
		vala_data_type__empty_type_list = (_tmp3_ = (GeeList*) gee_read_only_list_new (VALA_TYPE_DATA_TYPE, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, (GeeList*) (_tmp2_ = gee_array_list_new (VALA_TYPE_DATA_TYPE, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, g_direct_equal))), (vala_data_type__empty_type_list == NULL) ? NULL : (vala_data_type__empty_type_list = (gee_collection_object_unref (vala_data_type__empty_type_list), NULL)), _tmp3_);
		(_tmp2_ == NULL) ? NULL : (_tmp2_ = (gee_collection_object_unref (_tmp2_), NULL));
	}
	_tmp4_ = NULL;
	return (_tmp4_ = vala_data_type__empty_type_list, (_tmp4_ == NULL) ? NULL : gee_collection_object_ref (_tmp4_));
}


/**
 * Removes all generic type arguments.
 */
void vala_data_type_remove_all_type_arguments (ValaDataType* self) {
	GeeList* _tmp0_;
	g_return_if_fail (self != NULL);
	_tmp0_ = NULL;
	self->priv->type_argument_list = (_tmp0_ = NULL, (self->priv->type_argument_list == NULL) ? NULL : (self->priv->type_argument_list = (gee_collection_object_unref (self->priv->type_argument_list), NULL)), _tmp0_);
}


static void vala_data_type_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor) {
	ValaDataType * self;
	gboolean _tmp0_;
	self = (ValaDataType*) base;
	g_return_if_fail (visitor != NULL);
	_tmp0_ = FALSE;
	if (self->priv->type_argument_list != NULL) {
		_tmp0_ = gee_collection_get_size ((GeeCollection*) self->priv->type_argument_list) > 0;
	} else {
		_tmp0_ = FALSE;
	}
	if (_tmp0_) {
		{
			GeeIterator* _type_arg_it;
			_type_arg_it = gee_iterable_iterator ((GeeIterable*) self->priv->type_argument_list);
			while (gee_iterator_next (_type_arg_it)) {
				ValaDataType* type_arg;
				type_arg = (ValaDataType*) gee_iterator_get (_type_arg_it);
				vala_code_node_accept ((ValaCodeNode*) type_arg, visitor);
				(type_arg == NULL) ? NULL : (type_arg = (vala_code_node_unref (type_arg), NULL));
			}
			(_type_arg_it == NULL) ? NULL : (_type_arg_it = (gee_collection_object_unref (_type_arg_it), NULL));
		}
	}
	vala_code_visitor_visit_data_type (visitor, self);
}


/**
 * Returns the name and qualifiers of this type as it is used in C code.
 *
 * @return the type string to be used in C code
 */
static char* vala_data_type_real_get_cname (ValaDataType* self) {
	g_return_val_if_fail (self != NULL, NULL);
	/* raise error*/
	vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) self), "unresolved type reference");
	return NULL;
}


char* vala_data_type_get_cname (ValaDataType* self) {
	return VALA_DATA_TYPE_GET_CLASS (self)->get_cname (self);
}


static char* vala_data_type_real_get_cdeclarator_suffix (ValaDataType* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return g_strdup ("");
}


char* vala_data_type_get_cdeclarator_suffix (ValaDataType* self) {
	return VALA_DATA_TYPE_GET_CLASS (self)->get_cdeclarator_suffix (self);
}


/**
 * Returns the name and qualifiers of this type as it is used in C code
 * in a const declaration.
 *
 * @return the type string to be used in C code const declarations
 */
char* vala_data_type_get_const_cname (ValaDataType* self) {
	char* ptr;
	ValaTypeSymbol* t;
	char* _tmp6_;
	char* _tmp7_;
	char* _tmp8_;
	g_return_val_if_fail (self != NULL, NULL);
	ptr = NULL;
	t = NULL;
	/* FIXME: workaround to make constant arrays possible*/
	if (VALA_IS_ARRAY_TYPE (self)) {
		ValaTypeSymbol* _tmp1_;
		ValaTypeSymbol* _tmp0_;
		_tmp1_ = NULL;
		_tmp0_ = NULL;
		t = (_tmp1_ = (_tmp0_ = vala_array_type_get_element_type (VALA_ARRAY_TYPE (self))->priv->_data_type, (_tmp0_ == NULL) ? NULL : vala_code_node_ref (_tmp0_)), (t == NULL) ? NULL : (t = (vala_code_node_unref (t), NULL)), _tmp1_);
	} else {
		ValaTypeSymbol* _tmp3_;
		ValaTypeSymbol* _tmp2_;
		_tmp3_ = NULL;
		_tmp2_ = NULL;
		t = (_tmp3_ = (_tmp2_ = self->priv->_data_type, (_tmp2_ == NULL) ? NULL : vala_code_node_ref (_tmp2_)), (t == NULL) ? NULL : (t = (vala_code_node_unref (t), NULL)), _tmp3_);
	}
	if (!vala_typesymbol_is_reference_type (t)) {
		char* _tmp4_;
		_tmp4_ = NULL;
		ptr = (_tmp4_ = g_strdup (""), ptr = (g_free (ptr), NULL), _tmp4_);
	} else {
		char* _tmp5_;
		_tmp5_ = NULL;
		ptr = (_tmp5_ = g_strdup ("*"), ptr = (g_free (ptr), NULL), _tmp5_);
	}
	_tmp6_ = NULL;
	_tmp7_ = NULL;
	_tmp8_ = NULL;
	return (_tmp8_ = (_tmp7_ = g_strdup_printf ("const %s%s", _tmp6_ = vala_typesymbol_get_cname (t, FALSE), ptr), _tmp6_ = (g_free (_tmp6_), NULL), _tmp7_), ptr = (g_free (ptr), NULL), (t == NULL) ? NULL : (t = (vala_code_node_unref (t), NULL)), _tmp8_);
}


/**
 * Returns the C name of this data type in lower case. Words are
 * separated by underscores.
 *
 * @param infix a string to be placed between namespace and data type
 *              name or null
 * @return      the lower case name to be used in C code
 */
static char* vala_data_type_real_get_lower_case_cname (ValaDataType* self, const char* infix) {
	g_return_val_if_fail (self != NULL, NULL);
	return vala_symbol_get_lower_case_cname ((ValaSymbol*) self->priv->_data_type, infix);
}


char* vala_data_type_get_lower_case_cname (ValaDataType* self, const char* infix) {
	return VALA_DATA_TYPE_GET_CLASS (self)->get_lower_case_cname (self, infix);
}


static char* vala_data_type_real_to_string (ValaCodeNode* base) {
	ValaDataType * self;
	self = (ValaDataType*) base;
	return vala_data_type_to_qualified_string (self, NULL);
}


static char* vala_data_type_real_to_qualified_string (ValaDataType* self, ValaScope* scope) {
	char* s;
	GeeList* type_args;
	char* _tmp20_;
	g_return_val_if_fail (self != NULL, NULL);
	s = NULL;
	if (self->priv->_data_type != NULL) {
		ValaSymbol* _tmp0_;
		ValaSymbol* global_symbol;
		ValaSymbol* sym;
		ValaScope* _tmp3_;
		ValaScope* parent_scope;
		gboolean _tmp8_;
		_tmp0_ = NULL;
		global_symbol = (_tmp0_ = (ValaSymbol*) self->priv->_data_type, (_tmp0_ == NULL) ? NULL : vala_code_node_ref (_tmp0_));
		while (vala_symbol_get_name (vala_symbol_get_parent_symbol (global_symbol)) != NULL) {
			ValaSymbol* _tmp2_;
			ValaSymbol* _tmp1_;
			_tmp2_ = NULL;
			_tmp1_ = NULL;
			global_symbol = (_tmp2_ = (_tmp1_ = vala_symbol_get_parent_symbol (global_symbol), (_tmp1_ == NULL) ? NULL : vala_code_node_ref (_tmp1_)), (global_symbol == NULL) ? NULL : (global_symbol = (vala_code_node_unref (global_symbol), NULL)), _tmp2_);
		}
		sym = NULL;
		_tmp3_ = NULL;
		parent_scope = (_tmp3_ = scope, (_tmp3_ == NULL) ? NULL : vala_scope_ref (_tmp3_));
		while (TRUE) {
			gboolean _tmp4_;
			ValaSymbol* _tmp5_;
			ValaScope* _tmp7_;
			ValaScope* _tmp6_;
			_tmp4_ = FALSE;
			if (sym == NULL) {
				_tmp4_ = parent_scope != NULL;
			} else {
				_tmp4_ = FALSE;
			}
			if (!_tmp4_) {
				break;
			}
			_tmp5_ = NULL;
			sym = (_tmp5_ = vala_scope_lookup (parent_scope, vala_symbol_get_name (global_symbol)), (sym == NULL) ? NULL : (sym = (vala_code_node_unref (sym), NULL)), _tmp5_);
			_tmp7_ = NULL;
			_tmp6_ = NULL;
			parent_scope = (_tmp7_ = (_tmp6_ = vala_scope_get_parent_scope (parent_scope), (_tmp6_ == NULL) ? NULL : vala_scope_ref (_tmp6_)), (parent_scope == NULL) ? NULL : (parent_scope = (vala_scope_unref (parent_scope), NULL)), _tmp7_);
		}
		_tmp8_ = FALSE;
		if (sym != NULL) {
			_tmp8_ = global_symbol != sym;
		} else {
			_tmp8_ = FALSE;
		}
		if (_tmp8_) {
			char* _tmp10_;
			char* _tmp9_;
			_tmp10_ = NULL;
			_tmp9_ = NULL;
			s = (_tmp10_ = g_strconcat ("global::", _tmp9_ = vala_symbol_get_full_name ((ValaSymbol*) self->priv->_data_type), NULL), s = (g_free (s), NULL), _tmp10_);
			_tmp9_ = (g_free (_tmp9_), NULL);
			;
		} else {
			char* _tmp11_;
			_tmp11_ = NULL;
			s = (_tmp11_ = vala_symbol_get_full_name ((ValaSymbol*) self->priv->_data_type), s = (g_free (s), NULL), _tmp11_);
		}
		(global_symbol == NULL) ? NULL : (global_symbol = (vala_code_node_unref (global_symbol), NULL));
		(sym == NULL) ? NULL : (sym = (vala_code_node_unref (sym), NULL));
		(parent_scope == NULL) ? NULL : (parent_scope = (vala_scope_unref (parent_scope), NULL));
	} else {
		char* _tmp12_;
		_tmp12_ = NULL;
		s = (_tmp12_ = g_strdup ("null"), s = (g_free (s), NULL), _tmp12_);
	}
	type_args = vala_data_type_get_type_arguments (self);
	if (gee_collection_get_size ((GeeCollection*) type_args) > 0) {
		char* _tmp13_;
		gboolean first;
		char* _tmp18_;
		_tmp13_ = NULL;
		s = (_tmp13_ = g_strconcat (s, "<", NULL), s = (g_free (s), NULL), _tmp13_);
		first = TRUE;
		{
			GeeIterator* _type_arg_it;
			_type_arg_it = gee_iterable_iterator ((GeeIterable*) type_args);
			while (gee_iterator_next (_type_arg_it)) {
				ValaDataType* type_arg;
				char* _tmp17_;
				char* _tmp16_;
				type_arg = (ValaDataType*) gee_iterator_get (_type_arg_it);
				if (!first) {
					char* _tmp14_;
					_tmp14_ = NULL;
					s = (_tmp14_ = g_strconcat (s, ",", NULL), s = (g_free (s), NULL), _tmp14_);
				} else {
					first = FALSE;
				}
				if (!type_arg->priv->_value_owned) {
					char* _tmp15_;
					_tmp15_ = NULL;
					s = (_tmp15_ = g_strconcat (s, "weak ", NULL), s = (g_free (s), NULL), _tmp15_);
				}
				_tmp17_ = NULL;
				_tmp16_ = NULL;
				s = (_tmp17_ = g_strconcat (s, _tmp16_ = vala_data_type_to_qualified_string (type_arg, scope), NULL), s = (g_free (s), NULL), _tmp17_);
				_tmp16_ = (g_free (_tmp16_), NULL);
				(type_arg == NULL) ? NULL : (type_arg = (vala_code_node_unref (type_arg), NULL));
			}
			(_type_arg_it == NULL) ? NULL : (_type_arg_it = (gee_collection_object_unref (_type_arg_it), NULL));
		}
		_tmp18_ = NULL;
		s = (_tmp18_ = g_strconcat (s, ">", NULL), s = (g_free (s), NULL), _tmp18_);
	}
	if (self->priv->_nullable) {
		char* _tmp19_;
		_tmp19_ = NULL;
		s = (_tmp19_ = g_strconcat (s, "?", NULL), s = (g_free (s), NULL), _tmp19_);
	}
	_tmp20_ = NULL;
	return (_tmp20_ = s, (type_args == NULL) ? NULL : (type_args = (gee_collection_object_unref (type_args), NULL)), _tmp20_);
}


char* vala_data_type_to_qualified_string (ValaDataType* self, ValaScope* scope) {
	return VALA_DATA_TYPE_GET_CLASS (self)->to_qualified_string (self, scope);
}


static ValaDataType* vala_data_type_real_copy (ValaDataType* self) {
	g_return_val_if_fail (self != NULL, NULL);
	g_critical ("Type `%s' does not implement abstract method `vala_data_type_copy'", g_type_name (G_TYPE_FROM_INSTANCE (self)));
	return NULL;
}


/**
 * Creates a shallow copy of this type reference.
 *
 * @return copy of this type reference
 */
ValaDataType* vala_data_type_copy (ValaDataType* self) {
	return VALA_DATA_TYPE_GET_CLASS (self)->copy (self);
}


/**
 * Checks two type references for equality. May only be used with
 * resolved type references.
 *
 * @param type2 a type reference
 * @return      true if this type reference is equal to type2, false
 *              otherwise
 */
static gboolean vala_data_type_real_equals (ValaDataType* self, ValaDataType* type2) {
	gboolean _tmp3_;
	g_return_val_if_fail (self != NULL, FALSE);
	g_return_val_if_fail (type2 != NULL, FALSE);
	if (type2->priv->_value_owned != self->priv->_value_owned) {
		return FALSE;
	}
	if (type2->priv->_nullable != self->priv->_nullable) {
		return FALSE;
	}
	if (type2->priv->_data_type != self->priv->_data_type) {
		return FALSE;
	}
	_tmp3_ = FALSE;
	if (type2->priv->_type_parameter != NULL) {
		_tmp3_ = TRUE;
	} else {
		_tmp3_ = self->priv->_type_parameter != NULL;
	}
	if (_tmp3_) {
		gboolean _tmp4_;
		_tmp4_ = FALSE;
		if (type2->priv->_type_parameter == NULL) {
			_tmp4_ = TRUE;
		} else {
			_tmp4_ = self->priv->_type_parameter == NULL;
		}
		if (_tmp4_) {
			return FALSE;
		}
		if (!vala_typeparameter_equals (type2->priv->_type_parameter, self->priv->_type_parameter)) {
			return FALSE;
		}
	}
	if (type2->priv->_floating_reference != self->priv->_floating_reference) {
		return FALSE;
	}
	return TRUE;
}


gboolean vala_data_type_equals (ValaDataType* self, ValaDataType* type2) {
	return VALA_DATA_TYPE_GET_CLASS (self)->equals (self, type2);
}


/**
 * Checks whether this type reference is at least as strict as the
 * specified type reference type2.
 *
 * @param type2 a type reference
 * @return      true if this type reference is stricter or equal
 */
static gboolean vala_data_type_real_stricter (ValaDataType* self, ValaDataType* type2) {
	gboolean _tmp1_;
	gboolean _tmp3_;
	g_return_val_if_fail (self != NULL, FALSE);
	g_return_val_if_fail (type2 != NULL, FALSE);
	if (type2->priv->_value_owned != self->priv->_value_owned) {
		return FALSE;
	}
	_tmp1_ = FALSE;
	if (!type2->priv->_nullable) {
		_tmp1_ = self->priv->_nullable;
	} else {
		_tmp1_ = FALSE;
	}
	if (_tmp1_) {
		return FALSE;
	}
	_tmp3_ = FALSE;
	if (self->priv->_type_parameter != NULL) {
		_tmp3_ = TRUE;
	} else {
		_tmp3_ = type2->priv->_type_parameter != NULL;
	}
	/* temporarily ignore type parameters */
	if (_tmp3_) {
		return TRUE;
	}
	if (type2->priv->_data_type != self->priv->_data_type) {
		/* FIXME: allow this type reference to refer to a
		        subtype of the type type2 is referring to*/
		return FALSE;
	}
	if (type2->priv->_floating_reference != self->priv->_floating_reference) {
		return FALSE;
	}
	return TRUE;
}


gboolean vala_data_type_stricter (ValaDataType* self, ValaDataType* type2) {
	return VALA_DATA_TYPE_GET_CLASS (self)->stricter (self, type2);
}


static void vala_data_type_real_replace_type (ValaCodeNode* base, ValaDataType* old_type, ValaDataType* new_type) {
	ValaDataType * self;
	self = (ValaDataType*) base;
	g_return_if_fail (old_type != NULL);
	g_return_if_fail (new_type != NULL);
	if (self->priv->type_argument_list != NULL) {
		{
			gint i;
			i = 0;
			for (; i < gee_collection_get_size ((GeeCollection*) self->priv->type_argument_list); i++) {
				ValaDataType* _tmp0_;
				gboolean _tmp1_;
				_tmp0_ = NULL;
				if ((_tmp1_ = (_tmp0_ = (ValaDataType*) gee_list_get (self->priv->type_argument_list, i)) == old_type, (_tmp0_ == NULL) ? NULL : (_tmp0_ = (vala_code_node_unref (_tmp0_), NULL)), _tmp1_)) {
					gee_list_set (self->priv->type_argument_list, i, new_type);
					return;
				}
			}
		}
	}
}


static gboolean vala_data_type_real_compatible (ValaDataType* self, ValaDataType* target_type) {
	gboolean _tmp0_;
	gboolean _tmp9_;
	gboolean _tmp10_;
	gboolean _tmp13_;
	gboolean _tmp22_;
	gboolean _tmp23_;
	g_return_val_if_fail (self != NULL, FALSE);
	g_return_val_if_fail (target_type != NULL, FALSE);
	_tmp0_ = FALSE;
	if (VALA_IS_DELEGATE_TYPE (target_type)) {
		_tmp0_ = VALA_IS_DELEGATE_TYPE (self);
	} else {
		_tmp0_ = FALSE;
	}
	if (_tmp0_) {
		return vala_delegate_type_get_delegate_symbol (VALA_DELEGATE_TYPE (target_type)) == vala_delegate_type_get_delegate_symbol (VALA_DELEGATE_TYPE (self));
	}
	if (VALA_IS_POINTER_TYPE (target_type)) {
		gboolean _tmp2_;
		_tmp2_ = FALSE;
		if (self->priv->_type_parameter != NULL) {
			_tmp2_ = TRUE;
		} else {
			gboolean _tmp3_;
			_tmp3_ = FALSE;
			if (self->priv->_data_type != NULL) {
				gboolean _tmp4_;
				_tmp4_ = FALSE;
				if (vala_typesymbol_is_reference_type (self->priv->_data_type)) {
					_tmp4_ = TRUE;
				} else {
					_tmp4_ = VALA_IS_DELEGATE_TYPE (self);
				}
				_tmp3_ = _tmp4_;
			} else {
				_tmp3_ = FALSE;
			}
			_tmp2_ = _tmp3_;
		}
		/* any reference or array type or pointer type can be cast to a generic pointer */
		if (_tmp2_) {
			return TRUE;
		}
		return FALSE;
	}
	/* temporarily ignore type parameters */
	if (target_type->priv->_type_parameter != NULL) {
		return TRUE;
	}
	if (VALA_IS_ARRAY_TYPE (self) != VALA_IS_ARRAY_TYPE (target_type)) {
		return FALSE;
	}
	_tmp9_ = FALSE;
	_tmp10_ = FALSE;
	if (VALA_IS_ENUM (self->priv->_data_type)) {
		_tmp10_ = VALA_IS_STRUCT (target_type->priv->_data_type);
	} else {
		_tmp10_ = FALSE;
	}
	if (_tmp10_) {
		_tmp9_ = vala_struct_is_integer_type (VALA_STRUCT (target_type->priv->_data_type));
	} else {
		_tmp9_ = FALSE;
	}
	if (_tmp9_) {
		return TRUE;
	}
	if (self->priv->_data_type == target_type->priv->_data_type) {
		return TRUE;
	}
	_tmp13_ = FALSE;
	if (VALA_IS_STRUCT (self->priv->_data_type)) {
		_tmp13_ = VALA_IS_STRUCT (target_type->priv->_data_type);
	} else {
		_tmp13_ = FALSE;
	}
	if (_tmp13_) {
		ValaStruct* _tmp14_;
		ValaStruct* expr_struct;
		ValaStruct* _tmp15_;
		ValaStruct* expect_struct;
		gboolean _tmp16_;
		gboolean _tmp18_;
		gboolean _tmp19_;
		_tmp14_ = NULL;
		expr_struct = (_tmp14_ = VALA_STRUCT (self->priv->_data_type), (_tmp14_ == NULL) ? NULL : vala_code_node_ref (_tmp14_));
		_tmp15_ = NULL;
		expect_struct = (_tmp15_ = VALA_STRUCT (target_type->priv->_data_type), (_tmp15_ == NULL) ? NULL : vala_code_node_ref (_tmp15_));
		_tmp16_ = FALSE;
		if (vala_struct_is_integer_type (expr_struct)) {
			_tmp16_ = vala_struct_is_floating_type (expect_struct);
		} else {
			_tmp16_ = FALSE;
		}
		/* integer types may be implicitly cast to floating point types */
		if (_tmp16_) {
			gboolean _tmp17_;
			return (_tmp17_ = TRUE, (expr_struct == NULL) ? NULL : (expr_struct = (vala_code_node_unref (expr_struct), NULL)), (expect_struct == NULL) ? NULL : (expect_struct = (vala_code_node_unref (expect_struct), NULL)), _tmp17_);
		}
		_tmp18_ = FALSE;
		_tmp19_ = FALSE;
		if (vala_struct_is_integer_type (expr_struct)) {
			_tmp19_ = vala_struct_is_integer_type (expect_struct);
		} else {
			_tmp19_ = FALSE;
		}
		if (_tmp19_) {
			_tmp18_ = TRUE;
		} else {
			gboolean _tmp20_;
			_tmp20_ = FALSE;
			if (vala_struct_is_floating_type (expr_struct)) {
				_tmp20_ = vala_struct_is_floating_type (expect_struct);
			} else {
				_tmp20_ = FALSE;
			}
			_tmp18_ = _tmp20_;
		}
		if (_tmp18_) {
			if (vala_struct_get_rank (expr_struct) <= vala_struct_get_rank (expect_struct)) {
				gboolean _tmp21_;
				return (_tmp21_ = TRUE, (expr_struct == NULL) ? NULL : (expr_struct = (vala_code_node_unref (expr_struct), NULL)), (expect_struct == NULL) ? NULL : (expect_struct = (vala_code_node_unref (expect_struct), NULL)), _tmp21_);
			}
		}
		(expr_struct == NULL) ? NULL : (expr_struct = (vala_code_node_unref (expr_struct), NULL));
		(expect_struct == NULL) ? NULL : (expect_struct = (vala_code_node_unref (expect_struct), NULL));
	}
	_tmp22_ = FALSE;
	_tmp23_ = FALSE;
	if (self->priv->_data_type != NULL) {
		_tmp23_ = target_type->priv->_data_type != NULL;
	} else {
		_tmp23_ = FALSE;
	}
	if (_tmp23_) {
		_tmp22_ = vala_typesymbol_is_subtype_of (self->priv->_data_type, target_type->priv->_data_type);
	} else {
		_tmp22_ = FALSE;
	}
	if (_tmp22_) {
		return TRUE;
	}
	return FALSE;
}


gboolean vala_data_type_compatible (ValaDataType* self, ValaDataType* target_type) {
	return VALA_DATA_TYPE_GET_CLASS (self)->compatible (self, target_type);
}


/**
 * Returns whether instances of this type are invokable.
 *
 * @return true if invokable, false otherwise
 */
static gboolean vala_data_type_real_is_invokable (ValaDataType* self) {
	g_return_val_if_fail (self != NULL, FALSE);
	return FALSE;
}


gboolean vala_data_type_is_invokable (ValaDataType* self) {
	return VALA_DATA_TYPE_GET_CLASS (self)->is_invokable (self);
}


/**
 * Returns the return type of this invokable.
 *
 * @return return type
 */
static ValaDataType* vala_data_type_real_get_return_type (ValaDataType* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return NULL;
}


ValaDataType* vala_data_type_get_return_type (ValaDataType* self) {
	return VALA_DATA_TYPE_GET_CLASS (self)->get_return_type (self);
}


/**
 * Returns copy of the list of invocation parameters.
 *
 * @return parameter list
 */
static GeeList* vala_data_type_real_get_parameters (ValaDataType* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return NULL;
}


GeeList* vala_data_type_get_parameters (ValaDataType* self) {
	return VALA_DATA_TYPE_GET_CLASS (self)->get_parameters (self);
}


static gboolean vala_data_type_real_is_reference_type_or_type_parameter (ValaDataType* self) {
	gboolean _tmp0_;
	gboolean _tmp1_;
	g_return_val_if_fail (self != NULL, FALSE);
	_tmp0_ = FALSE;
	_tmp1_ = FALSE;
	if (self->priv->_data_type != NULL) {
		_tmp1_ = vala_typesymbol_is_reference_type (self->priv->_data_type);
	} else {
		_tmp1_ = FALSE;
	}
	if (_tmp1_) {
		_tmp0_ = TRUE;
	} else {
		_tmp0_ = self->priv->_type_parameter != NULL;
	}
	return _tmp0_;
}


gboolean vala_data_type_is_reference_type_or_type_parameter (ValaDataType* self) {
	return VALA_DATA_TYPE_GET_CLASS (self)->is_reference_type_or_type_parameter (self);
}


static gboolean vala_data_type_real_is_array (ValaDataType* self) {
	g_return_val_if_fail (self != NULL, FALSE);
	return FALSE;
}


gboolean vala_data_type_is_array (ValaDataType* self) {
	return VALA_DATA_TYPE_GET_CLASS (self)->is_array (self);
}


/**
 * Returns a list of symbols that define this type.
 *
 * @return symbol list
 */
static GeeList* vala_data_type_real_get_symbols (ValaDataType* self) {
	GeeArrayList* symbols;
	g_return_val_if_fail (self != NULL, NULL);
	symbols = gee_array_list_new (VALA_TYPE_SYMBOL, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, g_direct_equal);
	if (self->priv->_data_type != NULL) {
		gee_collection_add ((GeeCollection*) symbols, (ValaSymbol*) self->priv->_data_type);
	}
	return (GeeList*) symbols;
}


GeeList* vala_data_type_get_symbols (ValaDataType* self) {
	return VALA_DATA_TYPE_GET_CLASS (self)->get_symbols (self);
}


static ValaSymbol* vala_data_type_real_get_member (ValaDataType* self, const char* member_name) {
	g_return_val_if_fail (self != NULL, NULL);
	g_return_val_if_fail (member_name != NULL, NULL);
	if (self->priv->_data_type != NULL) {
		return vala_semantic_analyzer_symbol_lookup_inherited ((ValaSymbol*) self->priv->_data_type, member_name);
	}
	return NULL;
}


ValaSymbol* vala_data_type_get_member (ValaDataType* self, const char* member_name) {
	return VALA_DATA_TYPE_GET_CLASS (self)->get_member (self, member_name);
}


static ValaSymbol* vala_data_type_real_get_pointer_member (ValaDataType* self, const char* member_name) {
	g_return_val_if_fail (self != NULL, NULL);
	g_return_val_if_fail (member_name != NULL, NULL);
	return NULL;
}


ValaSymbol* vala_data_type_get_pointer_member (ValaDataType* self, const char* member_name) {
	return VALA_DATA_TYPE_GET_CLASS (self)->get_pointer_member (self, member_name);
}


/**
 * Checks whether this data type references a real struct. A real struct
 * is a struct which is not a simple (fundamental) type.
 */
static gboolean vala_data_type_real_is_real_struct_type (ValaDataType* self) {
	ValaStruct* _tmp1_;
	ValaTypeSymbol* _tmp0_;
	ValaStruct* s;
	gboolean _tmp2_;
	gboolean _tmp4_;
	g_return_val_if_fail (self != NULL, FALSE);
	_tmp1_ = NULL;
	_tmp0_ = NULL;
	s = (_tmp1_ = (_tmp0_ = self->priv->_data_type, VALA_IS_STRUCT (_tmp0_) ? ((ValaStruct*) _tmp0_) : NULL), (_tmp1_ == NULL) ? NULL : vala_code_node_ref (_tmp1_));
	_tmp2_ = FALSE;
	if (s != NULL) {
		_tmp2_ = !vala_struct_is_simple_type (s);
	} else {
		_tmp2_ = FALSE;
	}
	if (_tmp2_) {
		gboolean _tmp3_;
		return (_tmp3_ = TRUE, (s == NULL) ? NULL : (s = (vala_code_node_unref (s), NULL)), _tmp3_);
	}
	return (_tmp4_ = FALSE, (s == NULL) ? NULL : (s = (vala_code_node_unref (s), NULL)), _tmp4_);
}


gboolean vala_data_type_is_real_struct_type (ValaDataType* self) {
	return VALA_DATA_TYPE_GET_CLASS (self)->is_real_struct_type (self);
}


static char* vala_data_type_real_get_type_id (ValaDataType* self) {
	g_return_val_if_fail (self != NULL, NULL);
	if (self->priv->_data_type != NULL) {
		return vala_typesymbol_get_type_id (self->priv->_data_type);
	} else {
		return NULL;
	}
}


char* vala_data_type_get_type_id (ValaDataType* self) {
	return VALA_DATA_TYPE_GET_CLASS (self)->get_type_id (self);
}


/**
 * Returns type signature as used for GVariant and D-Bus.
 */
static char* vala_data_type_real_get_type_signature (ValaDataType* self) {
	g_return_val_if_fail (self != NULL, NULL);
	if (self->priv->_data_type != NULL) {
		char* sig;
		GeeList* type_args;
		gboolean _tmp0_;
		gboolean _tmp1_;
		char* _tmp4_;
		sig = vala_typesymbol_get_type_signature (self->priv->_data_type);
		type_args = vala_data_type_get_type_arguments (self);
		_tmp0_ = FALSE;
		_tmp1_ = FALSE;
		if (sig != NULL) {
			_tmp1_ = strstr (sig, "%s") != NULL;
		} else {
			_tmp1_ = FALSE;
		}
		if (_tmp1_) {
			_tmp0_ = gee_collection_get_size ((GeeCollection*) type_args) > 0;
		} else {
			_tmp0_ = FALSE;
		}
		if (_tmp0_) {
			char* element_sig;
			char* _tmp3_;
			element_sig = g_strdup ("");
			{
				GeeIterator* _type_arg_it;
				_type_arg_it = gee_iterable_iterator ((GeeIterable*) type_args);
				while (gee_iterator_next (_type_arg_it)) {
					ValaDataType* type_arg;
					char* s;
					type_arg = (ValaDataType*) gee_iterator_get (_type_arg_it);
					s = vala_data_type_get_type_signature (type_arg);
					if (s != NULL) {
						char* _tmp2_;
						_tmp2_ = NULL;
						element_sig = (_tmp2_ = g_strconcat (element_sig, s, NULL), element_sig = (g_free (element_sig), NULL), _tmp2_);
					}
					(type_arg == NULL) ? NULL : (type_arg = (vala_code_node_unref (type_arg), NULL));
					s = (g_free (s), NULL);
				}
				(_type_arg_it == NULL) ? NULL : (_type_arg_it = (gee_collection_object_unref (_type_arg_it), NULL));
			}
			_tmp3_ = NULL;
			sig = (_tmp3_ = g_strdup_printf (sig, element_sig), sig = (g_free (sig), NULL), _tmp3_);
			element_sig = (g_free (element_sig), NULL);
		}
		_tmp4_ = NULL;
		return (_tmp4_ = sig, (type_args == NULL) ? NULL : (type_args = (gee_collection_object_unref (type_args), NULL)), _tmp4_);
	} else {
		return NULL;
	}
}


char* vala_data_type_get_type_signature (ValaDataType* self) {
	return VALA_DATA_TYPE_GET_CLASS (self)->get_type_signature (self);
}


/**
 * Returns whether the value needs to be disposed, i.e. whether
 * allocated memory or other resources need to be released when
 * the value is no longer needed.
 */
static gboolean vala_data_type_real_is_disposable (ValaDataType* self) {
	g_return_val_if_fail (self != NULL, FALSE);
	if (!self->priv->_value_owned) {
		return FALSE;
	}
	if (vala_data_type_is_reference_type_or_type_parameter (self)) {
		return TRUE;
	}
	return FALSE;
}


gboolean vala_data_type_is_disposable (ValaDataType* self) {
	return VALA_DATA_TYPE_GET_CLASS (self)->is_disposable (self);
}


ValaDataType* vala_data_type_get_actual_type (ValaDataType* self, ValaDataType* derived_instance_type, ValaCodeNode* node_reference) {
	ValaDataType* _tmp2_;
	ValaDataType* _result_;
	g_return_val_if_fail (self != NULL, NULL);
	g_return_val_if_fail (node_reference != NULL, NULL);
	if (derived_instance_type == NULL) {
		ValaDataType* _tmp0_;
		_tmp0_ = NULL;
		return (_tmp0_ = self, (_tmp0_ == NULL) ? NULL : vala_code_node_ref (_tmp0_));
	}
	_tmp2_ = NULL;
	_result_ = (_tmp2_ = self, (_tmp2_ == NULL) ? NULL : vala_code_node_ref (_tmp2_));
	if (VALA_IS_GENERIC_TYPE (_result_)) {
		ValaDataType* _tmp3_;
		_tmp3_ = NULL;
		_result_ = (_tmp3_ = vala_semantic_analyzer_get_actual_type (derived_instance_type, VALA_GENERIC_TYPE (_result_), node_reference), (_result_ == NULL) ? NULL : (_result_ = (vala_code_node_unref (_result_), NULL)), _tmp3_);
	} else {
		if (_result_->priv->type_argument_list != NULL) {
			ValaDataType* _tmp4_;
			/* recursely get actual types for type arguments*/
			_tmp4_ = NULL;
			_result_ = (_tmp4_ = vala_data_type_copy (_result_), (_result_ == NULL) ? NULL : (_result_ = (vala_code_node_unref (_result_), NULL)), _tmp4_);
			{
				gint i;
				i = 0;
				for (; i < gee_collection_get_size ((GeeCollection*) _result_->priv->type_argument_list); i++) {
					ValaDataType* _tmp6_;
					ValaDataType* _tmp5_;
					_tmp6_ = NULL;
					_tmp5_ = NULL;
					gee_list_set (_result_->priv->type_argument_list, i, _tmp6_ = vala_data_type_get_actual_type (_tmp5_ = (ValaDataType*) gee_list_get (_result_->priv->type_argument_list, i), derived_instance_type, node_reference));
					(_tmp6_ == NULL) ? NULL : (_tmp6_ = (vala_code_node_unref (_tmp6_), NULL));
					(_tmp5_ == NULL) ? NULL : (_tmp5_ = (vala_code_node_unref (_tmp5_), NULL));
				}
			}
		}
	}
	return _result_;
}


gboolean vala_data_type_get_value_owned (ValaDataType* self) {
	g_return_val_if_fail (self != NULL, FALSE);
	return self->priv->_value_owned;
}


void vala_data_type_set_value_owned (ValaDataType* self, gboolean value) {
	g_return_if_fail (self != NULL);
	self->priv->_value_owned = value;
}


gboolean vala_data_type_get_nullable (ValaDataType* self) {
	g_return_val_if_fail (self != NULL, FALSE);
	return self->priv->_nullable;
}


void vala_data_type_set_nullable (ValaDataType* self, gboolean value) {
	g_return_if_fail (self != NULL);
	self->priv->_nullable = value;
}


ValaTypeSymbol* vala_data_type_get_data_type (ValaDataType* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return self->priv->_data_type;
}


void vala_data_type_set_data_type (ValaDataType* self, ValaTypeSymbol* value) {
	g_return_if_fail (self != NULL);
	self->priv->_data_type = value;
}


ValaTypeParameter* vala_data_type_get_type_parameter (ValaDataType* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return self->priv->_type_parameter;
}


void vala_data_type_set_type_parameter (ValaDataType* self, ValaTypeParameter* value) {
	ValaTypeParameter* _tmp2_;
	ValaTypeParameter* _tmp1_;
	g_return_if_fail (self != NULL);
	_tmp2_ = NULL;
	_tmp1_ = NULL;
	self->priv->_type_parameter = (_tmp2_ = (_tmp1_ = value, (_tmp1_ == NULL) ? NULL : vala_code_node_ref (_tmp1_)), (self->priv->_type_parameter == NULL) ? NULL : (self->priv->_type_parameter = (vala_code_node_unref (self->priv->_type_parameter), NULL)), _tmp2_);
}


gboolean vala_data_type_get_floating_reference (ValaDataType* self) {
	g_return_val_if_fail (self != NULL, FALSE);
	return self->priv->_floating_reference;
}


void vala_data_type_set_floating_reference (ValaDataType* self, gboolean value) {
	g_return_if_fail (self != NULL);
	self->priv->_floating_reference = value;
}


gboolean vala_data_type_get_is_dynamic (ValaDataType* self) {
	g_return_val_if_fail (self != NULL, FALSE);
	return self->priv->_is_dynamic;
}


void vala_data_type_set_is_dynamic (ValaDataType* self, gboolean value) {
	g_return_if_fail (self != NULL);
	self->priv->_is_dynamic = value;
}


static void vala_data_type_class_init (ValaDataTypeClass * klass) {
	vala_data_type_parent_class = g_type_class_peek_parent (klass);
	VALA_CODE_NODE_CLASS (klass)->finalize = vala_data_type_finalize;
	g_type_class_add_private (klass, sizeof (ValaDataTypePrivate));
	VALA_CODE_NODE_CLASS (klass)->accept = vala_data_type_real_accept;
	VALA_DATA_TYPE_CLASS (klass)->get_cname = vala_data_type_real_get_cname;
	VALA_DATA_TYPE_CLASS (klass)->get_cdeclarator_suffix = vala_data_type_real_get_cdeclarator_suffix;
	VALA_DATA_TYPE_CLASS (klass)->get_lower_case_cname = vala_data_type_real_get_lower_case_cname;
	VALA_CODE_NODE_CLASS (klass)->to_string = vala_data_type_real_to_string;
	VALA_DATA_TYPE_CLASS (klass)->to_qualified_string = vala_data_type_real_to_qualified_string;
	VALA_DATA_TYPE_CLASS (klass)->copy = vala_data_type_real_copy;
	VALA_DATA_TYPE_CLASS (klass)->equals = vala_data_type_real_equals;
	VALA_DATA_TYPE_CLASS (klass)->stricter = vala_data_type_real_stricter;
	VALA_CODE_NODE_CLASS (klass)->replace_type = vala_data_type_real_replace_type;
	VALA_DATA_TYPE_CLASS (klass)->compatible = vala_data_type_real_compatible;
	VALA_DATA_TYPE_CLASS (klass)->is_invokable = vala_data_type_real_is_invokable;
	VALA_DATA_TYPE_CLASS (klass)->get_return_type = vala_data_type_real_get_return_type;
	VALA_DATA_TYPE_CLASS (klass)->get_parameters = vala_data_type_real_get_parameters;
	VALA_DATA_TYPE_CLASS (klass)->is_reference_type_or_type_parameter = vala_data_type_real_is_reference_type_or_type_parameter;
	VALA_DATA_TYPE_CLASS (klass)->is_array = vala_data_type_real_is_array;
	VALA_DATA_TYPE_CLASS (klass)->get_symbols = vala_data_type_real_get_symbols;
	VALA_DATA_TYPE_CLASS (klass)->get_member = vala_data_type_real_get_member;
	VALA_DATA_TYPE_CLASS (klass)->get_pointer_member = vala_data_type_real_get_pointer_member;
	VALA_DATA_TYPE_CLASS (klass)->is_real_struct_type = vala_data_type_real_is_real_struct_type;
	VALA_DATA_TYPE_CLASS (klass)->get_type_id = vala_data_type_real_get_type_id;
	VALA_DATA_TYPE_CLASS (klass)->get_type_signature = vala_data_type_real_get_type_signature;
	VALA_DATA_TYPE_CLASS (klass)->is_disposable = vala_data_type_real_is_disposable;
}


static void vala_data_type_instance_init (ValaDataType * self) {
	self->priv = VALA_DATA_TYPE_GET_PRIVATE (self);
}


static void vala_data_type_finalize (ValaCodeNode* obj) {
	ValaDataType * self;
	self = VALA_DATA_TYPE (obj);
	(self->priv->_type_parameter == NULL) ? NULL : (self->priv->_type_parameter = (vala_code_node_unref (self->priv->_type_parameter), NULL));
	(self->priv->type_argument_list == NULL) ? NULL : (self->priv->type_argument_list = (gee_collection_object_unref (self->priv->type_argument_list), NULL));
	VALA_CODE_NODE_CLASS (vala_data_type_parent_class)->finalize (obj);
}


GType vala_data_type_get_type (void) {
	static GType vala_data_type_type_id = 0;
	if (vala_data_type_type_id == 0) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaDataTypeClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_data_type_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaDataType), 0, (GInstanceInitFunc) vala_data_type_instance_init, NULL };
		vala_data_type_type_id = g_type_register_static (VALA_TYPE_CODE_NODE, "ValaDataType", &g_define_type_info, G_TYPE_FLAG_ABSTRACT);
	}
	return vala_data_type_type_id;
}




