/* valadeclarationstatement.vala
 *
 * Copyright (C) 2006-2008  Jürg Billeter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 */

#include <glib.h>
#include <glib-object.h>
#include <stdlib.h>
#include <string.h>
#include <gee.h>


#define VALA_TYPE_CODE_NODE (vala_code_node_get_type ())
#define VALA_CODE_NODE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_CODE_NODE, ValaCodeNode))
#define VALA_CODE_NODE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_CODE_NODE, ValaCodeNodeClass))
#define VALA_IS_CODE_NODE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_CODE_NODE))
#define VALA_IS_CODE_NODE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_CODE_NODE))
#define VALA_CODE_NODE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_CODE_NODE, ValaCodeNodeClass))

typedef struct _ValaCodeNode ValaCodeNode;
typedef struct _ValaCodeNodeClass ValaCodeNodeClass;
typedef struct _ValaCodeNodePrivate ValaCodeNodePrivate;

#define VALA_TYPE_CODE_VISITOR (vala_code_visitor_get_type ())
#define VALA_CODE_VISITOR(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_CODE_VISITOR, ValaCodeVisitor))
#define VALA_CODE_VISITOR_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_CODE_VISITOR, ValaCodeVisitorClass))
#define VALA_IS_CODE_VISITOR(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_CODE_VISITOR))
#define VALA_IS_CODE_VISITOR_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_CODE_VISITOR))
#define VALA_CODE_VISITOR_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_CODE_VISITOR, ValaCodeVisitorClass))

typedef struct _ValaCodeVisitor ValaCodeVisitor;
typedef struct _ValaCodeVisitorClass ValaCodeVisitorClass;

#define VALA_TYPE_SEMANTIC_ANALYZER (vala_semantic_analyzer_get_type ())
#define VALA_SEMANTIC_ANALYZER(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_SEMANTIC_ANALYZER, ValaSemanticAnalyzer))
#define VALA_SEMANTIC_ANALYZER_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_SEMANTIC_ANALYZER, ValaSemanticAnalyzerClass))
#define VALA_IS_SEMANTIC_ANALYZER(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_SEMANTIC_ANALYZER))
#define VALA_IS_SEMANTIC_ANALYZER_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_SEMANTIC_ANALYZER))
#define VALA_SEMANTIC_ANALYZER_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_SEMANTIC_ANALYZER, ValaSemanticAnalyzerClass))

typedef struct _ValaSemanticAnalyzer ValaSemanticAnalyzer;
typedef struct _ValaSemanticAnalyzerClass ValaSemanticAnalyzerClass;

#define VALA_TYPE_DATA_TYPE (vala_data_type_get_type ())
#define VALA_DATA_TYPE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_DATA_TYPE, ValaDataType))
#define VALA_DATA_TYPE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_DATA_TYPE, ValaDataTypeClass))
#define VALA_IS_DATA_TYPE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_DATA_TYPE))
#define VALA_IS_DATA_TYPE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_DATA_TYPE))
#define VALA_DATA_TYPE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_DATA_TYPE, ValaDataTypeClass))

typedef struct _ValaDataType ValaDataType;
typedef struct _ValaDataTypeClass ValaDataTypeClass;

#define VALA_TYPE_EXPRESSION (vala_expression_get_type ())
#define VALA_EXPRESSION(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_EXPRESSION, ValaExpression))
#define VALA_EXPRESSION_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_EXPRESSION, ValaExpressionClass))
#define VALA_IS_EXPRESSION(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_EXPRESSION))
#define VALA_IS_EXPRESSION_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_EXPRESSION))
#define VALA_EXPRESSION_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_EXPRESSION, ValaExpressionClass))

typedef struct _ValaExpression ValaExpression;
typedef struct _ValaExpressionClass ValaExpressionClass;

#define VALA_TYPE_SYMBOL (vala_symbol_get_type ())
#define VALA_SYMBOL(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_SYMBOL, ValaSymbol))
#define VALA_SYMBOL_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_SYMBOL, ValaSymbolClass))
#define VALA_IS_SYMBOL(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_SYMBOL))
#define VALA_IS_SYMBOL_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_SYMBOL))
#define VALA_SYMBOL_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_SYMBOL, ValaSymbolClass))

typedef struct _ValaSymbol ValaSymbol;
typedef struct _ValaSymbolClass ValaSymbolClass;

#define VALA_TYPE_LOCAL_VARIABLE (vala_local_variable_get_type ())
#define VALA_LOCAL_VARIABLE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_LOCAL_VARIABLE, ValaLocalVariable))
#define VALA_LOCAL_VARIABLE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_LOCAL_VARIABLE, ValaLocalVariableClass))
#define VALA_IS_LOCAL_VARIABLE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_LOCAL_VARIABLE))
#define VALA_IS_LOCAL_VARIABLE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_LOCAL_VARIABLE))
#define VALA_LOCAL_VARIABLE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_LOCAL_VARIABLE, ValaLocalVariableClass))

typedef struct _ValaLocalVariable ValaLocalVariable;
typedef struct _ValaLocalVariableClass ValaLocalVariableClass;

#define VALA_TYPE_ATTRIBUTE (vala_attribute_get_type ())
#define VALA_ATTRIBUTE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_ATTRIBUTE, ValaAttribute))
#define VALA_ATTRIBUTE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_ATTRIBUTE, ValaAttributeClass))
#define VALA_IS_ATTRIBUTE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_ATTRIBUTE))
#define VALA_IS_ATTRIBUTE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_ATTRIBUTE))
#define VALA_ATTRIBUTE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_ATTRIBUTE, ValaAttributeClass))

typedef struct _ValaAttribute ValaAttribute;
typedef struct _ValaAttributeClass ValaAttributeClass;

#define VALA_TYPE_STATEMENT (vala_statement_get_type ())
#define VALA_STATEMENT(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_STATEMENT, ValaStatement))
#define VALA_IS_STATEMENT(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_STATEMENT))
#define VALA_STATEMENT_GET_INTERFACE(obj) (G_TYPE_INSTANCE_GET_INTERFACE ((obj), VALA_TYPE_STATEMENT, ValaStatementIface))

typedef struct _ValaStatement ValaStatement;
typedef struct _ValaStatementIface ValaStatementIface;

#define VALA_TYPE_DECLARATION_STATEMENT (vala_declaration_statement_get_type ())
#define VALA_DECLARATION_STATEMENT(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_DECLARATION_STATEMENT, ValaDeclarationStatement))
#define VALA_DECLARATION_STATEMENT_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_DECLARATION_STATEMENT, ValaDeclarationStatementClass))
#define VALA_IS_DECLARATION_STATEMENT(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_DECLARATION_STATEMENT))
#define VALA_IS_DECLARATION_STATEMENT_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_DECLARATION_STATEMENT))
#define VALA_DECLARATION_STATEMENT_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_DECLARATION_STATEMENT, ValaDeclarationStatementClass))

typedef struct _ValaDeclarationStatement ValaDeclarationStatement;
typedef struct _ValaDeclarationStatementClass ValaDeclarationStatementClass;
typedef struct _ValaDeclarationStatementPrivate ValaDeclarationStatementPrivate;

#define VALA_TYPE_SOURCE_REFERENCE (vala_source_reference_get_type ())
#define VALA_SOURCE_REFERENCE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_SOURCE_REFERENCE, ValaSourceReference))
#define VALA_SOURCE_REFERENCE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_SOURCE_REFERENCE, ValaSourceReferenceClass))
#define VALA_IS_SOURCE_REFERENCE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_SOURCE_REFERENCE))
#define VALA_IS_SOURCE_REFERENCE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_SOURCE_REFERENCE))
#define VALA_SOURCE_REFERENCE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_SOURCE_REFERENCE, ValaSourceReferenceClass))

typedef struct _ValaSourceReference ValaSourceReference;
typedef struct _ValaSourceReferenceClass ValaSourceReferenceClass;

/**
 * Represents a part of the parsed source code.
 *
 * Code nodes get created by the parser and are used throughout the whole
 * compilation process.
 */
struct _ValaCodeNode {
	GTypeInstance parent_instance;
	volatile int ref_count;
	ValaCodeNodePrivate * priv;
	GList* attributes;
};

struct _ValaCodeNodeClass {
	GTypeClass parent_class;
	void (*finalize) (ValaCodeNode *self);
	void (*accept) (ValaCodeNode* self, ValaCodeVisitor* visitor);
	void (*accept_children) (ValaCodeNode* self, ValaCodeVisitor* visitor);
	gboolean (*check) (ValaCodeNode* self, ValaSemanticAnalyzer* analyzer);
	void (*replace_type) (ValaCodeNode* self, ValaDataType* old_type, ValaDataType* new_type);
	void (*replace_expression) (ValaCodeNode* self, ValaExpression* old_node, ValaExpression* new_node);
	char* (*to_string) (ValaCodeNode* self);
	void (*get_defined_variables) (ValaCodeNode* self, GeeCollection* collection);
	void (*get_used_variables) (ValaCodeNode* self, GeeCollection* collection);
};

/**
 * Interface for all statement types.
 */
struct _ValaStatementIface {
	GTypeInterface parent_iface;
};

/**
 * Represents a local variable or constant declaration statement in the source code.
 */
struct _ValaDeclarationStatement {
	ValaCodeNode parent_instance;
	ValaDeclarationStatementPrivate * priv;
};

struct _ValaDeclarationStatementClass {
	ValaCodeNodeClass parent_class;
};

struct _ValaDeclarationStatementPrivate {
	ValaSymbol* _declaration;
};



gpointer vala_code_node_ref (gpointer instance);
void vala_code_node_unref (gpointer instance);
GParamSpec* vala_param_spec_code_node (const gchar* name, const gchar* nick, const gchar* blurb, GType object_type, GParamFlags flags);
void vala_value_set_code_node (GValue* value, gpointer v_object);
gpointer vala_value_get_code_node (const GValue* value);
GType vala_code_node_get_type (void);
gpointer vala_code_visitor_ref (gpointer instance);
void vala_code_visitor_unref (gpointer instance);
GParamSpec* vala_param_spec_code_visitor (const gchar* name, const gchar* nick, const gchar* blurb, GType object_type, GParamFlags flags);
void vala_value_set_code_visitor (GValue* value, gpointer v_object);
gpointer vala_value_get_code_visitor (const GValue* value);
GType vala_code_visitor_get_type (void);
GType vala_semantic_analyzer_get_type (void);
GType vala_data_type_get_type (void);
GType vala_expression_get_type (void);
GType vala_symbol_get_type (void);
GType vala_local_variable_get_type (void);
GType vala_attribute_get_type (void);
GType vala_statement_get_type (void);
GType vala_declaration_statement_get_type (void);
#define VALA_DECLARATION_STATEMENT_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_DECLARATION_STATEMENT, ValaDeclarationStatementPrivate))
enum  {
	VALA_DECLARATION_STATEMENT_DUMMY_PROPERTY
};
void vala_declaration_statement_set_declaration (ValaDeclarationStatement* self, ValaSymbol* value);
gpointer vala_source_reference_ref (gpointer instance);
void vala_source_reference_unref (gpointer instance);
GParamSpec* vala_param_spec_source_reference (const gchar* name, const gchar* nick, const gchar* blurb, GType object_type, GParamFlags flags);
void vala_value_set_source_reference (GValue* value, gpointer v_object);
gpointer vala_value_get_source_reference (const GValue* value);
GType vala_source_reference_get_type (void);
void vala_code_node_set_source_reference (ValaCodeNode* self, ValaSourceReference* value);
ValaDeclarationStatement* vala_declaration_statement_new (ValaSymbol* declaration, ValaSourceReference* source_reference);
ValaDeclarationStatement* vala_declaration_statement_construct (GType object_type, ValaSymbol* declaration, ValaSourceReference* source_reference);
ValaDeclarationStatement* vala_declaration_statement_new (ValaSymbol* declaration, ValaSourceReference* source_reference);
void vala_code_visitor_visit_declaration_statement (ValaCodeVisitor* self, ValaDeclarationStatement* stmt);
static void vala_declaration_statement_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor);
ValaSymbol* vala_declaration_statement_get_declaration (ValaDeclarationStatement* self);
void vala_code_node_accept (ValaCodeNode* self, ValaCodeVisitor* visitor);
static void vala_declaration_statement_real_accept_children (ValaCodeNode* base, ValaCodeVisitor* visitor);
gboolean vala_code_node_get_checked (ValaCodeNode* self);
gboolean vala_code_node_get_error (ValaCodeNode* self);
void vala_code_node_set_checked (ValaCodeNode* self, gboolean value);
gboolean vala_code_node_check (ValaCodeNode* self, ValaSemanticAnalyzer* analyzer);
ValaExpression* vala_local_variable_get_initializer (ValaLocalVariable* self);
GeeList* vala_code_node_get_error_types (ValaCodeNode* self);
ValaDataType* vala_data_type_copy (ValaDataType* self);
ValaSourceReference* vala_code_node_get_source_reference (ValaCodeNode* self);
void vala_code_node_add_error_type (ValaCodeNode* self, ValaDataType* error_type);
static gboolean vala_declaration_statement_real_check (ValaCodeNode* base, ValaSemanticAnalyzer* analyzer);
void vala_code_node_get_defined_variables (ValaCodeNode* self, GeeCollection* collection);
static void vala_declaration_statement_real_get_defined_variables (ValaCodeNode* base, GeeCollection* collection);
void vala_code_node_get_used_variables (ValaCodeNode* self, GeeCollection* collection);
static void vala_declaration_statement_real_get_used_variables (ValaCodeNode* base, GeeCollection* collection);
void vala_code_node_set_parent_node (ValaCodeNode* self, ValaCodeNode* value);
static gpointer vala_declaration_statement_parent_class = NULL;
static ValaStatementIface* vala_declaration_statement_vala_statement_parent_iface = NULL;
static void vala_declaration_statement_finalize (ValaCodeNode* obj);



/**
 * Creates a new declaration statement.
 *
 * @param decl   local variable declaration
 * @param source reference to source code
 * @return       newly created declaration statement
 */
ValaDeclarationStatement* vala_declaration_statement_construct (GType object_type, ValaSymbol* declaration, ValaSourceReference* source_reference) {
	ValaDeclarationStatement* self;
	g_return_val_if_fail (declaration != NULL, NULL);
	self = (ValaDeclarationStatement*) g_type_create_instance (object_type);
	vala_declaration_statement_set_declaration (self, declaration);
	vala_code_node_set_source_reference ((ValaCodeNode*) self, source_reference);
	return self;
}


ValaDeclarationStatement* vala_declaration_statement_new (ValaSymbol* declaration, ValaSourceReference* source_reference) {
	return vala_declaration_statement_construct (VALA_TYPE_DECLARATION_STATEMENT, declaration, source_reference);
}


static void vala_declaration_statement_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor) {
	ValaDeclarationStatement * self;
	self = (ValaDeclarationStatement*) base;
	g_return_if_fail (visitor != NULL);
	vala_code_visitor_visit_declaration_statement (visitor, self);
}


static void vala_declaration_statement_real_accept_children (ValaCodeNode* base, ValaCodeVisitor* visitor) {
	ValaDeclarationStatement * self;
	self = (ValaDeclarationStatement*) base;
	g_return_if_fail (visitor != NULL);
	vala_code_node_accept ((ValaCodeNode*) vala_declaration_statement_get_declaration (self), visitor);
}


static gboolean vala_declaration_statement_real_check (ValaCodeNode* base, ValaSemanticAnalyzer* analyzer) {
	ValaDeclarationStatement * self;
	ValaLocalVariable* _tmp2_;
	ValaSymbol* _tmp1_;
	ValaLocalVariable* local;
	gboolean _tmp3_;
	gboolean _tmp6_;
	self = (ValaDeclarationStatement*) base;
	g_return_val_if_fail (analyzer != NULL, FALSE);
	if (vala_code_node_get_checked ((ValaCodeNode*) self)) {
		return !vala_code_node_get_error ((ValaCodeNode*) self);
	}
	vala_code_node_set_checked ((ValaCodeNode*) self, TRUE);
	vala_code_node_check ((ValaCodeNode*) vala_declaration_statement_get_declaration (self), analyzer);
	_tmp2_ = NULL;
	_tmp1_ = NULL;
	local = (_tmp2_ = (_tmp1_ = vala_declaration_statement_get_declaration (self), VALA_IS_LOCAL_VARIABLE (_tmp1_) ? ((ValaLocalVariable*) _tmp1_) : NULL), (_tmp2_ == NULL) ? NULL : vala_code_node_ref (_tmp2_));
	_tmp3_ = FALSE;
	if (local != NULL) {
		_tmp3_ = vala_local_variable_get_initializer (local) != NULL;
	} else {
		_tmp3_ = FALSE;
	}
	if (_tmp3_) {
		{
			GeeList* _tmp4_;
			GeeIterator* _tmp5_;
			GeeIterator* _error_type_it;
			_tmp4_ = NULL;
			_tmp5_ = NULL;
			_error_type_it = (_tmp5_ = gee_iterable_iterator ((GeeIterable*) (_tmp4_ = vala_code_node_get_error_types ((ValaCodeNode*) vala_local_variable_get_initializer (local)))), (_tmp4_ == NULL) ? NULL : (_tmp4_ = (gee_collection_object_unref (_tmp4_), NULL)), _tmp5_);
			while (gee_iterator_next (_error_type_it)) {
				ValaDataType* error_type;
				ValaDataType* initializer_error_type;
				error_type = (ValaDataType*) gee_iterator_get (_error_type_it);
				/* ensure we can trace back which expression may throw errors of this type*/
				initializer_error_type = vala_data_type_copy (error_type);
				vala_code_node_set_source_reference ((ValaCodeNode*) initializer_error_type, vala_code_node_get_source_reference ((ValaCodeNode*) vala_local_variable_get_initializer (local)));
				vala_code_node_add_error_type ((ValaCodeNode*) self, initializer_error_type);
				(error_type == NULL) ? NULL : (error_type = (vala_code_node_unref (error_type), NULL));
				(initializer_error_type == NULL) ? NULL : (initializer_error_type = (vala_code_node_unref (initializer_error_type), NULL));
			}
			(_error_type_it == NULL) ? NULL : (_error_type_it = (gee_collection_object_unref (_error_type_it), NULL));
		}
	}
	return (_tmp6_ = !vala_code_node_get_error ((ValaCodeNode*) self), (local == NULL) ? NULL : (local = (vala_code_node_unref (local), NULL)), _tmp6_);
}


static void vala_declaration_statement_real_get_defined_variables (ValaCodeNode* base, GeeCollection* collection) {
	ValaDeclarationStatement * self;
	ValaLocalVariable* _tmp1_;
	ValaSymbol* _tmp0_;
	ValaLocalVariable* local;
	gboolean _tmp2_;
	self = (ValaDeclarationStatement*) base;
	g_return_if_fail (collection != NULL);
	_tmp1_ = NULL;
	_tmp0_ = NULL;
	local = (_tmp1_ = (_tmp0_ = vala_declaration_statement_get_declaration (self), VALA_IS_LOCAL_VARIABLE (_tmp0_) ? ((ValaLocalVariable*) _tmp0_) : NULL), (_tmp1_ == NULL) ? NULL : vala_code_node_ref (_tmp1_));
	_tmp2_ = FALSE;
	if (local != NULL) {
		_tmp2_ = vala_local_variable_get_initializer (local) != NULL;
	} else {
		_tmp2_ = FALSE;
	}
	if (_tmp2_) {
		vala_code_node_get_defined_variables ((ValaCodeNode*) vala_local_variable_get_initializer (local), collection);
		gee_collection_add (collection, local);
	}
	(local == NULL) ? NULL : (local = (vala_code_node_unref (local), NULL));
}


static void vala_declaration_statement_real_get_used_variables (ValaCodeNode* base, GeeCollection* collection) {
	ValaDeclarationStatement * self;
	ValaLocalVariable* _tmp1_;
	ValaSymbol* _tmp0_;
	ValaLocalVariable* local;
	gboolean _tmp2_;
	self = (ValaDeclarationStatement*) base;
	g_return_if_fail (collection != NULL);
	_tmp1_ = NULL;
	_tmp0_ = NULL;
	local = (_tmp1_ = (_tmp0_ = vala_declaration_statement_get_declaration (self), VALA_IS_LOCAL_VARIABLE (_tmp0_) ? ((ValaLocalVariable*) _tmp0_) : NULL), (_tmp1_ == NULL) ? NULL : vala_code_node_ref (_tmp1_));
	_tmp2_ = FALSE;
	if (local != NULL) {
		_tmp2_ = vala_local_variable_get_initializer (local) != NULL;
	} else {
		_tmp2_ = FALSE;
	}
	if (_tmp2_) {
		vala_code_node_get_used_variables ((ValaCodeNode*) vala_local_variable_get_initializer (local), collection);
	}
	(local == NULL) ? NULL : (local = (vala_code_node_unref (local), NULL));
}


ValaSymbol* vala_declaration_statement_get_declaration (ValaDeclarationStatement* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return self->priv->_declaration;
}


void vala_declaration_statement_set_declaration (ValaDeclarationStatement* self, ValaSymbol* value) {
	ValaSymbol* _tmp2_;
	ValaSymbol* _tmp1_;
	g_return_if_fail (self != NULL);
	_tmp2_ = NULL;
	_tmp1_ = NULL;
	self->priv->_declaration = (_tmp2_ = (_tmp1_ = value, (_tmp1_ == NULL) ? NULL : vala_code_node_ref (_tmp1_)), (self->priv->_declaration == NULL) ? NULL : (self->priv->_declaration = (vala_code_node_unref (self->priv->_declaration), NULL)), _tmp2_);
	if (self->priv->_declaration != NULL) {
		vala_code_node_set_parent_node ((ValaCodeNode*) self->priv->_declaration, (ValaCodeNode*) self);
	}
}


static void vala_declaration_statement_class_init (ValaDeclarationStatementClass * klass) {
	vala_declaration_statement_parent_class = g_type_class_peek_parent (klass);
	VALA_CODE_NODE_CLASS (klass)->finalize = vala_declaration_statement_finalize;
	g_type_class_add_private (klass, sizeof (ValaDeclarationStatementPrivate));
	VALA_CODE_NODE_CLASS (klass)->accept = vala_declaration_statement_real_accept;
	VALA_CODE_NODE_CLASS (klass)->accept_children = vala_declaration_statement_real_accept_children;
	VALA_CODE_NODE_CLASS (klass)->check = vala_declaration_statement_real_check;
	VALA_CODE_NODE_CLASS (klass)->get_defined_variables = vala_declaration_statement_real_get_defined_variables;
	VALA_CODE_NODE_CLASS (klass)->get_used_variables = vala_declaration_statement_real_get_used_variables;
}


static void vala_declaration_statement_vala_statement_interface_init (ValaStatementIface * iface) {
	vala_declaration_statement_vala_statement_parent_iface = g_type_interface_peek_parent (iface);
}


static void vala_declaration_statement_instance_init (ValaDeclarationStatement * self) {
	self->priv = VALA_DECLARATION_STATEMENT_GET_PRIVATE (self);
}


static void vala_declaration_statement_finalize (ValaCodeNode* obj) {
	ValaDeclarationStatement * self;
	self = VALA_DECLARATION_STATEMENT (obj);
	(self->priv->_declaration == NULL) ? NULL : (self->priv->_declaration = (vala_code_node_unref (self->priv->_declaration), NULL));
	VALA_CODE_NODE_CLASS (vala_declaration_statement_parent_class)->finalize (obj);
}


GType vala_declaration_statement_get_type (void) {
	static GType vala_declaration_statement_type_id = 0;
	if (vala_declaration_statement_type_id == 0) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaDeclarationStatementClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_declaration_statement_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaDeclarationStatement), 0, (GInstanceInitFunc) vala_declaration_statement_instance_init, NULL };
		static const GInterfaceInfo vala_statement_info = { (GInterfaceInitFunc) vala_declaration_statement_vala_statement_interface_init, (GInterfaceFinalizeFunc) NULL, NULL};
		vala_declaration_statement_type_id = g_type_register_static (VALA_TYPE_CODE_NODE, "ValaDeclarationStatement", &g_define_type_info, 0);
		g_type_add_interface_static (vala_declaration_statement_type_id, VALA_TYPE_STATEMENT, &vala_statement_info);
	}
	return vala_declaration_statement_type_id;
}




