/* valadynamicsignal.vala
 *
 * Copyright (C) 2007-2008  Jürg Billeter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 */

#include <glib.h>
#include <glib-object.h>
#include <stdlib.h>
#include <string.h>
#include <gee.h>


#define VALA_TYPE_CODE_NODE (vala_code_node_get_type ())
#define VALA_CODE_NODE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_CODE_NODE, ValaCodeNode))
#define VALA_CODE_NODE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_CODE_NODE, ValaCodeNodeClass))
#define VALA_IS_CODE_NODE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_CODE_NODE))
#define VALA_IS_CODE_NODE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_CODE_NODE))
#define VALA_CODE_NODE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_CODE_NODE, ValaCodeNodeClass))

typedef struct _ValaCodeNode ValaCodeNode;
typedef struct _ValaCodeNodeClass ValaCodeNodeClass;
typedef struct _ValaCodeNodePrivate ValaCodeNodePrivate;

#define VALA_TYPE_CODE_VISITOR (vala_code_visitor_get_type ())
#define VALA_CODE_VISITOR(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_CODE_VISITOR, ValaCodeVisitor))
#define VALA_CODE_VISITOR_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_CODE_VISITOR, ValaCodeVisitorClass))
#define VALA_IS_CODE_VISITOR(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_CODE_VISITOR))
#define VALA_IS_CODE_VISITOR_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_CODE_VISITOR))
#define VALA_CODE_VISITOR_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_CODE_VISITOR, ValaCodeVisitorClass))

typedef struct _ValaCodeVisitor ValaCodeVisitor;
typedef struct _ValaCodeVisitorClass ValaCodeVisitorClass;

#define VALA_TYPE_SEMANTIC_ANALYZER (vala_semantic_analyzer_get_type ())
#define VALA_SEMANTIC_ANALYZER(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_SEMANTIC_ANALYZER, ValaSemanticAnalyzer))
#define VALA_SEMANTIC_ANALYZER_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_SEMANTIC_ANALYZER, ValaSemanticAnalyzerClass))
#define VALA_IS_SEMANTIC_ANALYZER(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_SEMANTIC_ANALYZER))
#define VALA_IS_SEMANTIC_ANALYZER_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_SEMANTIC_ANALYZER))
#define VALA_SEMANTIC_ANALYZER_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_SEMANTIC_ANALYZER, ValaSemanticAnalyzerClass))

typedef struct _ValaSemanticAnalyzer ValaSemanticAnalyzer;
typedef struct _ValaSemanticAnalyzerClass ValaSemanticAnalyzerClass;

#define VALA_TYPE_DATA_TYPE (vala_data_type_get_type ())
#define VALA_DATA_TYPE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_DATA_TYPE, ValaDataType))
#define VALA_DATA_TYPE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_DATA_TYPE, ValaDataTypeClass))
#define VALA_IS_DATA_TYPE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_DATA_TYPE))
#define VALA_IS_DATA_TYPE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_DATA_TYPE))
#define VALA_DATA_TYPE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_DATA_TYPE, ValaDataTypeClass))

typedef struct _ValaDataType ValaDataType;
typedef struct _ValaDataTypeClass ValaDataTypeClass;

#define VALA_TYPE_EXPRESSION (vala_expression_get_type ())
#define VALA_EXPRESSION(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_EXPRESSION, ValaExpression))
#define VALA_EXPRESSION_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_EXPRESSION, ValaExpressionClass))
#define VALA_IS_EXPRESSION(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_EXPRESSION))
#define VALA_IS_EXPRESSION_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_EXPRESSION))
#define VALA_EXPRESSION_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_EXPRESSION, ValaExpressionClass))

typedef struct _ValaExpression ValaExpression;
typedef struct _ValaExpressionClass ValaExpressionClass;

#define VALA_TYPE_SYMBOL (vala_symbol_get_type ())
#define VALA_SYMBOL(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_SYMBOL, ValaSymbol))
#define VALA_SYMBOL_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_SYMBOL, ValaSymbolClass))
#define VALA_IS_SYMBOL(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_SYMBOL))
#define VALA_IS_SYMBOL_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_SYMBOL))
#define VALA_SYMBOL_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_SYMBOL, ValaSymbolClass))

typedef struct _ValaSymbol ValaSymbol;
typedef struct _ValaSymbolClass ValaSymbolClass;

#define VALA_TYPE_LOCAL_VARIABLE (vala_local_variable_get_type ())
#define VALA_LOCAL_VARIABLE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_LOCAL_VARIABLE, ValaLocalVariable))
#define VALA_LOCAL_VARIABLE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_LOCAL_VARIABLE, ValaLocalVariableClass))
#define VALA_IS_LOCAL_VARIABLE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_LOCAL_VARIABLE))
#define VALA_IS_LOCAL_VARIABLE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_LOCAL_VARIABLE))
#define VALA_LOCAL_VARIABLE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_LOCAL_VARIABLE, ValaLocalVariableClass))

typedef struct _ValaLocalVariable ValaLocalVariable;
typedef struct _ValaLocalVariableClass ValaLocalVariableClass;

#define VALA_TYPE_ATTRIBUTE (vala_attribute_get_type ())
#define VALA_ATTRIBUTE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_ATTRIBUTE, ValaAttribute))
#define VALA_ATTRIBUTE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_ATTRIBUTE, ValaAttributeClass))
#define VALA_IS_ATTRIBUTE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_ATTRIBUTE))
#define VALA_IS_ATTRIBUTE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_ATTRIBUTE))
#define VALA_ATTRIBUTE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_ATTRIBUTE, ValaAttributeClass))

typedef struct _ValaAttribute ValaAttribute;
typedef struct _ValaAttributeClass ValaAttributeClass;
typedef struct _ValaSymbolPrivate ValaSymbolPrivate;

#define VALA_TYPE_MEMBER (vala_member_get_type ())
#define VALA_MEMBER(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_MEMBER, ValaMember))
#define VALA_MEMBER_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_MEMBER, ValaMemberClass))
#define VALA_IS_MEMBER(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_MEMBER))
#define VALA_IS_MEMBER_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_MEMBER))
#define VALA_MEMBER_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_MEMBER, ValaMemberClass))

typedef struct _ValaMember ValaMember;
typedef struct _ValaMemberClass ValaMemberClass;
typedef struct _ValaMemberPrivate ValaMemberPrivate;

#define VALA_TYPE_LOCKABLE (vala_lockable_get_type ())
#define VALA_LOCKABLE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_LOCKABLE, ValaLockable))
#define VALA_IS_LOCKABLE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_LOCKABLE))
#define VALA_LOCKABLE_GET_INTERFACE(obj) (G_TYPE_INSTANCE_GET_INTERFACE ((obj), VALA_TYPE_LOCKABLE, ValaLockableIface))

typedef struct _ValaLockable ValaLockable;
typedef struct _ValaLockableIface ValaLockableIface;

#define VALA_TYPE_SIGNAL (vala_signal_get_type ())
#define VALA_SIGNAL(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_SIGNAL, ValaSignal))
#define VALA_SIGNAL_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_SIGNAL, ValaSignalClass))
#define VALA_IS_SIGNAL(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_SIGNAL))
#define VALA_IS_SIGNAL_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_SIGNAL))
#define VALA_SIGNAL_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_SIGNAL, ValaSignalClass))

typedef struct _ValaSignal ValaSignal;
typedef struct _ValaSignalClass ValaSignalClass;
typedef struct _ValaSignalPrivate ValaSignalPrivate;

#define VALA_TYPE_DYNAMIC_SIGNAL (vala_dynamic_signal_get_type ())
#define VALA_DYNAMIC_SIGNAL(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_DYNAMIC_SIGNAL, ValaDynamicSignal))
#define VALA_DYNAMIC_SIGNAL_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_DYNAMIC_SIGNAL, ValaDynamicSignalClass))
#define VALA_IS_DYNAMIC_SIGNAL(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_DYNAMIC_SIGNAL))
#define VALA_IS_DYNAMIC_SIGNAL_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_DYNAMIC_SIGNAL))
#define VALA_DYNAMIC_SIGNAL_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_DYNAMIC_SIGNAL, ValaDynamicSignalClass))

typedef struct _ValaDynamicSignal ValaDynamicSignal;
typedef struct _ValaDynamicSignalClass ValaDynamicSignalClass;
typedef struct _ValaDynamicSignalPrivate ValaDynamicSignalPrivate;

#define VALA_TYPE_SOURCE_REFERENCE (vala_source_reference_get_type ())
#define VALA_SOURCE_REFERENCE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_SOURCE_REFERENCE, ValaSourceReference))
#define VALA_SOURCE_REFERENCE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_SOURCE_REFERENCE, ValaSourceReferenceClass))
#define VALA_IS_SOURCE_REFERENCE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_SOURCE_REFERENCE))
#define VALA_IS_SOURCE_REFERENCE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_SOURCE_REFERENCE))
#define VALA_SOURCE_REFERENCE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_SOURCE_REFERENCE, ValaSourceReferenceClass))

typedef struct _ValaSourceReference ValaSourceReference;
typedef struct _ValaSourceReferenceClass ValaSourceReferenceClass;

/**
 * Represents a part of the parsed source code.
 *
 * Code nodes get created by the parser and are used throughout the whole
 * compilation process.
 */
struct _ValaCodeNode {
	GTypeInstance parent_instance;
	volatile int ref_count;
	ValaCodeNodePrivate * priv;
	GList* attributes;
};

struct _ValaCodeNodeClass {
	GTypeClass parent_class;
	void (*finalize) (ValaCodeNode *self);
	void (*accept) (ValaCodeNode* self, ValaCodeVisitor* visitor);
	void (*accept_children) (ValaCodeNode* self, ValaCodeVisitor* visitor);
	gboolean (*check) (ValaCodeNode* self, ValaSemanticAnalyzer* analyzer);
	void (*replace_type) (ValaCodeNode* self, ValaDataType* old_type, ValaDataType* new_type);
	void (*replace_expression) (ValaCodeNode* self, ValaExpression* old_node, ValaExpression* new_node);
	char* (*to_string) (ValaCodeNode* self);
	void (*get_defined_variables) (ValaCodeNode* self, GeeCollection* collection);
	void (*get_used_variables) (ValaCodeNode* self, GeeCollection* collection);
};

/**
 * Represents a node in the symbol tree.
 */
struct _ValaSymbol {
	ValaCodeNode parent_instance;
	ValaSymbolPrivate * priv;
};

struct _ValaSymbolClass {
	ValaCodeNodeClass parent_class;
	char* (*get_cprefix) (ValaSymbol* self);
	char* (*get_lower_case_cname) (ValaSymbol* self, const char* infix);
	char* (*get_lower_case_cprefix) (ValaSymbol* self);
	GeeList* (*get_cheader_filenames) (ValaSymbol* self);
};

/**
 * Represents a general class member.
 */
struct _ValaMember {
	ValaSymbol parent_instance;
	ValaMemberPrivate * priv;
};

struct _ValaMemberClass {
	ValaSymbolClass parent_class;
};

/**
 * Represents a lockable object.
 */
struct _ValaLockableIface {
	GTypeInterface parent_iface;
	gboolean (*get_lock_used) (ValaLockable* self);
	void (*set_lock_used) (ValaLockable* self, gboolean used);
};

/**
 * Represents an object signal. Signals enable objects to provide notifications.
 */
struct _ValaSignal {
	ValaMember parent_instance;
	ValaSignalPrivate * priv;
};

struct _ValaSignalClass {
	ValaMemberClass parent_class;
};

/**
 * Represents a late bound signal.
 */
struct _ValaDynamicSignal {
	ValaSignal parent_instance;
	ValaDynamicSignalPrivate * priv;
};

struct _ValaDynamicSignalClass {
	ValaSignalClass parent_class;
};

struct _ValaDynamicSignalPrivate {
	ValaDataType* _dynamic_type;
	ValaExpression* _handler;
};



gpointer vala_code_node_ref (gpointer instance);
void vala_code_node_unref (gpointer instance);
GParamSpec* vala_param_spec_code_node (const gchar* name, const gchar* nick, const gchar* blurb, GType object_type, GParamFlags flags);
void vala_value_set_code_node (GValue* value, gpointer v_object);
gpointer vala_value_get_code_node (const GValue* value);
GType vala_code_node_get_type (void);
gpointer vala_code_visitor_ref (gpointer instance);
void vala_code_visitor_unref (gpointer instance);
GParamSpec* vala_param_spec_code_visitor (const gchar* name, const gchar* nick, const gchar* blurb, GType object_type, GParamFlags flags);
void vala_value_set_code_visitor (GValue* value, gpointer v_object);
gpointer vala_value_get_code_visitor (const GValue* value);
GType vala_code_visitor_get_type (void);
GType vala_semantic_analyzer_get_type (void);
GType vala_data_type_get_type (void);
GType vala_expression_get_type (void);
GType vala_symbol_get_type (void);
GType vala_local_variable_get_type (void);
GType vala_attribute_get_type (void);
GType vala_member_get_type (void);
GType vala_lockable_get_type (void);
GType vala_signal_get_type (void);
GType vala_dynamic_signal_get_type (void);
#define VALA_DYNAMIC_SIGNAL_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_DYNAMIC_SIGNAL, ValaDynamicSignalPrivate))
enum  {
	VALA_DYNAMIC_SIGNAL_DUMMY_PROPERTY
};
gpointer vala_source_reference_ref (gpointer instance);
void vala_source_reference_unref (gpointer instance);
GParamSpec* vala_param_spec_source_reference (const gchar* name, const gchar* nick, const gchar* blurb, GType object_type, GParamFlags flags);
void vala_value_set_source_reference (GValue* value, gpointer v_object);
gpointer vala_value_get_source_reference (const GValue* value);
GType vala_source_reference_get_type (void);
ValaSignal* vala_signal_new (const char* name, ValaDataType* return_type, ValaSourceReference* source_reference);
ValaSignal* vala_signal_construct (GType object_type, const char* name, ValaDataType* return_type, ValaSourceReference* source_reference);
void vala_dynamic_signal_set_dynamic_type (ValaDynamicSignal* self, ValaDataType* value);
ValaDynamicSignal* vala_dynamic_signal_new (ValaDataType* dynamic_type, const char* name, ValaDataType* return_type, ValaSourceReference* source_reference);
ValaDynamicSignal* vala_dynamic_signal_construct (GType object_type, ValaDataType* dynamic_type, const char* name, ValaDataType* return_type, ValaSourceReference* source_reference);
ValaDynamicSignal* vala_dynamic_signal_new (ValaDataType* dynamic_type, const char* name, ValaDataType* return_type, ValaSourceReference* source_reference);
static gboolean vala_dynamic_signal_real_check (ValaCodeNode* base, ValaSemanticAnalyzer* analyzer);
ValaDataType* vala_dynamic_signal_get_dynamic_type (ValaDynamicSignal* self);
ValaExpression* vala_dynamic_signal_get_handler (ValaDynamicSignal* self);
void vala_dynamic_signal_set_handler (ValaDynamicSignal* self, ValaExpression* value);
static gpointer vala_dynamic_signal_parent_class = NULL;
static void vala_dynamic_signal_finalize (ValaCodeNode* obj);



ValaDynamicSignal* vala_dynamic_signal_construct (GType object_type, ValaDataType* dynamic_type, const char* name, ValaDataType* return_type, ValaSourceReference* source_reference) {
	ValaDynamicSignal* self;
	g_return_val_if_fail (dynamic_type != NULL, NULL);
	g_return_val_if_fail (name != NULL, NULL);
	g_return_val_if_fail (return_type != NULL, NULL);
	self = (ValaDynamicSignal*) vala_signal_construct (object_type, name, return_type, source_reference);
	vala_dynamic_signal_set_dynamic_type (self, dynamic_type);
	return self;
}


ValaDynamicSignal* vala_dynamic_signal_new (ValaDataType* dynamic_type, const char* name, ValaDataType* return_type, ValaSourceReference* source_reference) {
	return vala_dynamic_signal_construct (VALA_TYPE_DYNAMIC_SIGNAL, dynamic_type, name, return_type, source_reference);
}


static gboolean vala_dynamic_signal_real_check (ValaCodeNode* base, ValaSemanticAnalyzer* analyzer) {
	ValaDynamicSignal * self;
	self = (ValaDynamicSignal*) base;
	g_return_val_if_fail (analyzer != NULL, FALSE);
	return TRUE;
}


ValaDataType* vala_dynamic_signal_get_dynamic_type (ValaDynamicSignal* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return self->priv->_dynamic_type;
}


void vala_dynamic_signal_set_dynamic_type (ValaDynamicSignal* self, ValaDataType* value) {
	ValaDataType* _tmp2_;
	ValaDataType* _tmp1_;
	g_return_if_fail (self != NULL);
	_tmp2_ = NULL;
	_tmp1_ = NULL;
	self->priv->_dynamic_type = (_tmp2_ = (_tmp1_ = value, (_tmp1_ == NULL) ? NULL : vala_code_node_ref (_tmp1_)), (self->priv->_dynamic_type == NULL) ? NULL : (self->priv->_dynamic_type = (vala_code_node_unref (self->priv->_dynamic_type), NULL)), _tmp2_);
}


ValaExpression* vala_dynamic_signal_get_handler (ValaDynamicSignal* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return self->priv->_handler;
}


void vala_dynamic_signal_set_handler (ValaDynamicSignal* self, ValaExpression* value) {
	ValaExpression* _tmp2_;
	ValaExpression* _tmp1_;
	g_return_if_fail (self != NULL);
	_tmp2_ = NULL;
	_tmp1_ = NULL;
	self->priv->_handler = (_tmp2_ = (_tmp1_ = value, (_tmp1_ == NULL) ? NULL : vala_code_node_ref (_tmp1_)), (self->priv->_handler == NULL) ? NULL : (self->priv->_handler = (vala_code_node_unref (self->priv->_handler), NULL)), _tmp2_);
}


static void vala_dynamic_signal_class_init (ValaDynamicSignalClass * klass) {
	vala_dynamic_signal_parent_class = g_type_class_peek_parent (klass);
	VALA_CODE_NODE_CLASS (klass)->finalize = vala_dynamic_signal_finalize;
	g_type_class_add_private (klass, sizeof (ValaDynamicSignalPrivate));
	VALA_CODE_NODE_CLASS (klass)->check = vala_dynamic_signal_real_check;
}


static void vala_dynamic_signal_instance_init (ValaDynamicSignal * self) {
	self->priv = VALA_DYNAMIC_SIGNAL_GET_PRIVATE (self);
}


static void vala_dynamic_signal_finalize (ValaCodeNode* obj) {
	ValaDynamicSignal * self;
	self = VALA_DYNAMIC_SIGNAL (obj);
	(self->priv->_dynamic_type == NULL) ? NULL : (self->priv->_dynamic_type = (vala_code_node_unref (self->priv->_dynamic_type), NULL));
	(self->priv->_handler == NULL) ? NULL : (self->priv->_handler = (vala_code_node_unref (self->priv->_handler), NULL));
	VALA_CODE_NODE_CLASS (vala_dynamic_signal_parent_class)->finalize (obj);
}


GType vala_dynamic_signal_get_type (void) {
	static GType vala_dynamic_signal_type_id = 0;
	if (vala_dynamic_signal_type_id == 0) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaDynamicSignalClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_dynamic_signal_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaDynamicSignal), 0, (GInstanceInitFunc) vala_dynamic_signal_instance_init, NULL };
		vala_dynamic_signal_type_id = g_type_register_static (VALA_TYPE_SIGNAL, "ValaDynamicSignal", &g_define_type_info, 0);
	}
	return vala_dynamic_signal_type_id;
}




