/* valaccodeenum.vala
 *
 * Copyright (C) 2006-2008  Jürg Billeter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 */

#include <glib.h>
#include <glib-object.h>
#include <stdlib.h>
#include <string.h>
#include <gee.h>


#define VALA_TYPE_CCODE_NODE (vala_ccode_node_get_type ())
#define VALA_CCODE_NODE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_CCODE_NODE, ValaCCodeNode))
#define VALA_CCODE_NODE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_CCODE_NODE, ValaCCodeNodeClass))
#define VALA_IS_CCODE_NODE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_CCODE_NODE))
#define VALA_IS_CCODE_NODE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_CCODE_NODE))
#define VALA_CCODE_NODE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_CCODE_NODE, ValaCCodeNodeClass))

typedef struct _ValaCCodeNode ValaCCodeNode;
typedef struct _ValaCCodeNodeClass ValaCCodeNodeClass;
typedef struct _ValaCCodeNodePrivate ValaCCodeNodePrivate;

#define VALA_TYPE_CCODE_WRITER (vala_ccode_writer_get_type ())
#define VALA_CCODE_WRITER(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_CCODE_WRITER, ValaCCodeWriter))
#define VALA_CCODE_WRITER_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_CCODE_WRITER, ValaCCodeWriterClass))
#define VALA_IS_CCODE_WRITER(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_CCODE_WRITER))
#define VALA_IS_CCODE_WRITER_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_CCODE_WRITER))
#define VALA_CCODE_WRITER_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_CCODE_WRITER, ValaCCodeWriterClass))

typedef struct _ValaCCodeWriter ValaCCodeWriter;
typedef struct _ValaCCodeWriterClass ValaCCodeWriterClass;

#define VALA_TYPE_CCODE_ENUM (vala_ccode_enum_get_type ())
#define VALA_CCODE_ENUM(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_CCODE_ENUM, ValaCCodeEnum))
#define VALA_CCODE_ENUM_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_CCODE_ENUM, ValaCCodeEnumClass))
#define VALA_IS_CCODE_ENUM(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_CCODE_ENUM))
#define VALA_IS_CCODE_ENUM_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_CCODE_ENUM))
#define VALA_CCODE_ENUM_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_CCODE_ENUM, ValaCCodeEnumClass))

typedef struct _ValaCCodeEnum ValaCCodeEnum;
typedef struct _ValaCCodeEnumClass ValaCCodeEnumClass;
typedef struct _ValaCCodeEnumPrivate ValaCCodeEnumPrivate;

#define VALA_TYPE_CCODE_ENUM_VALUE (vala_ccode_enum_value_get_type ())
#define VALA_CCODE_ENUM_VALUE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_CCODE_ENUM_VALUE, ValaCCodeEnumValue))
#define VALA_CCODE_ENUM_VALUE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_CCODE_ENUM_VALUE, ValaCCodeEnumValueClass))
#define VALA_IS_CCODE_ENUM_VALUE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_CCODE_ENUM_VALUE))
#define VALA_IS_CCODE_ENUM_VALUE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_CCODE_ENUM_VALUE))
#define VALA_CCODE_ENUM_VALUE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_CCODE_ENUM_VALUE, ValaCCodeEnumValueClass))

typedef struct _ValaCCodeEnumValue ValaCCodeEnumValue;
typedef struct _ValaCCodeEnumValueClass ValaCCodeEnumValueClass;

#define VALA_TYPE_CCODE_LINE_DIRECTIVE (vala_ccode_line_directive_get_type ())
#define VALA_CCODE_LINE_DIRECTIVE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_CCODE_LINE_DIRECTIVE, ValaCCodeLineDirective))
#define VALA_CCODE_LINE_DIRECTIVE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_CCODE_LINE_DIRECTIVE, ValaCCodeLineDirectiveClass))
#define VALA_IS_CCODE_LINE_DIRECTIVE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_CCODE_LINE_DIRECTIVE))
#define VALA_IS_CCODE_LINE_DIRECTIVE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_CCODE_LINE_DIRECTIVE))
#define VALA_CCODE_LINE_DIRECTIVE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_CCODE_LINE_DIRECTIVE, ValaCCodeLineDirectiveClass))

typedef struct _ValaCCodeLineDirective ValaCCodeLineDirective;
typedef struct _ValaCCodeLineDirectiveClass ValaCCodeLineDirectiveClass;

/**
 * Represents a node in the C code tree.
 */
struct _ValaCCodeNode {
	GTypeInstance parent_instance;
	volatile int ref_count;
	ValaCCodeNodePrivate * priv;
};

struct _ValaCCodeNodeClass {
	GTypeClass parent_class;
	void (*finalize) (ValaCCodeNode *self);
	void (*write) (ValaCCodeNode* self, ValaCCodeWriter* writer);
	void (*write_declaration) (ValaCCodeNode* self, ValaCCodeWriter* writer);
	void (*write_combined) (ValaCCodeNode* self, ValaCCodeWriter* writer);
};

/**
 * Represents an enum in the C code.
 */
struct _ValaCCodeEnum {
	ValaCCodeNode parent_instance;
	ValaCCodeEnumPrivate * priv;
};

struct _ValaCCodeEnumClass {
	ValaCCodeNodeClass parent_class;
};

struct _ValaCCodeEnumPrivate {
	char* _name;
	GeeList* values;
};



gpointer vala_ccode_node_ref (gpointer instance);
void vala_ccode_node_unref (gpointer instance);
GParamSpec* vala_param_spec_ccode_node (const gchar* name, const gchar* nick, const gchar* blurb, GType object_type, GParamFlags flags);
void vala_value_set_ccode_node (GValue* value, gpointer v_object);
gpointer vala_value_get_ccode_node (const GValue* value);
GType vala_ccode_node_get_type (void);
gpointer vala_ccode_writer_ref (gpointer instance);
void vala_ccode_writer_unref (gpointer instance);
GParamSpec* vala_param_spec_ccode_writer (const gchar* name, const gchar* nick, const gchar* blurb, GType object_type, GParamFlags flags);
void vala_value_set_ccode_writer (GValue* value, gpointer v_object);
gpointer vala_value_get_ccode_writer (const GValue* value);
GType vala_ccode_writer_get_type (void);
GType vala_ccode_enum_get_type (void);
GType vala_ccode_enum_value_get_type (void);
#define VALA_CCODE_ENUM_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_CCODE_ENUM, ValaCCodeEnumPrivate))
enum  {
	VALA_CCODE_ENUM_DUMMY_PROPERTY
};
void vala_ccode_enum_set_name (ValaCCodeEnum* self, const char* value);
ValaCCodeEnum* vala_ccode_enum_new (const char* name);
ValaCCodeEnum* vala_ccode_enum_construct (GType object_type, const char* name);
ValaCCodeEnum* vala_ccode_enum_new (const char* name);
void vala_ccode_enum_add_value (ValaCCodeEnum* self, ValaCCodeEnumValue* value);
const char* vala_ccode_enum_get_name (ValaCCodeEnum* self);
void vala_ccode_writer_write_string (ValaCCodeWriter* self, const char* s);
void vala_ccode_writer_write_begin_block (ValaCCodeWriter* self);
void vala_ccode_writer_write_newline (ValaCCodeWriter* self);
GType vala_ccode_line_directive_get_type (void);
void vala_ccode_writer_write_indent (ValaCCodeWriter* self, ValaCCodeLineDirective* line);
void vala_ccode_node_write (ValaCCodeNode* self, ValaCCodeWriter* writer);
void vala_ccode_writer_write_end_block (ValaCCodeWriter* self);
static void vala_ccode_enum_real_write (ValaCCodeNode* base, ValaCCodeWriter* writer);
static gpointer vala_ccode_enum_parent_class = NULL;
static void vala_ccode_enum_finalize (ValaCCodeNode* obj);



ValaCCodeEnum* vala_ccode_enum_construct (GType object_type, const char* name) {
	ValaCCodeEnum* self;
	self = (ValaCCodeEnum*) g_type_create_instance (object_type);
	vala_ccode_enum_set_name (self, name);
	return self;
}


ValaCCodeEnum* vala_ccode_enum_new (const char* name) {
	return vala_ccode_enum_construct (VALA_TYPE_CCODE_ENUM, name);
}


/**
 * Adds the specified value to this enum.
 *
 * @param name  enum value name
 * @param value optional numerical value
 */
void vala_ccode_enum_add_value (ValaCCodeEnum* self, ValaCCodeEnumValue* value) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (value != NULL);
	gee_collection_add ((GeeCollection*) self->priv->values, value);
}


static void vala_ccode_enum_real_write (ValaCCodeNode* base, ValaCCodeWriter* writer) {
	ValaCCodeEnum * self;
	gboolean first;
	self = (ValaCCodeEnum*) base;
	g_return_if_fail (writer != NULL);
	if (self->priv->_name != NULL) {
		vala_ccode_writer_write_string (writer, "typedef ");
	}
	vala_ccode_writer_write_string (writer, "enum ");
	vala_ccode_writer_write_begin_block (writer);
	first = TRUE;
	{
		GeeIterator* _value_it;
		_value_it = gee_iterable_iterator ((GeeIterable*) self->priv->values);
		while (gee_iterator_next (_value_it)) {
			ValaCCodeEnumValue* value;
			value = (ValaCCodeEnumValue*) gee_iterator_get (_value_it);
			if (!first) {
				vala_ccode_writer_write_string (writer, ",");
				vala_ccode_writer_write_newline (writer);
			}
			vala_ccode_writer_write_indent (writer, NULL);
			vala_ccode_node_write ((ValaCCodeNode*) value, writer);
			first = FALSE;
			(value == NULL) ? NULL : (value = (vala_ccode_node_unref (value), NULL));
		}
		(_value_it == NULL) ? NULL : (_value_it = (gee_collection_object_unref (_value_it), NULL));
	}
	if (!first) {
		vala_ccode_writer_write_newline (writer);
	}
	vala_ccode_writer_write_end_block (writer);
	if (self->priv->_name != NULL) {
		vala_ccode_writer_write_string (writer, " ");
		vala_ccode_writer_write_string (writer, self->priv->_name);
	}
	vala_ccode_writer_write_string (writer, ";");
	vala_ccode_writer_write_newline (writer);
}


const char* vala_ccode_enum_get_name (ValaCCodeEnum* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return self->priv->_name;
}


void vala_ccode_enum_set_name (ValaCCodeEnum* self, const char* value) {
	char* _tmp2_;
	const char* _tmp1_;
	g_return_if_fail (self != NULL);
	_tmp2_ = NULL;
	_tmp1_ = NULL;
	self->priv->_name = (_tmp2_ = (_tmp1_ = value, (_tmp1_ == NULL) ? NULL : g_strdup (_tmp1_)), self->priv->_name = (g_free (self->priv->_name), NULL), _tmp2_);
}


static void vala_ccode_enum_class_init (ValaCCodeEnumClass * klass) {
	vala_ccode_enum_parent_class = g_type_class_peek_parent (klass);
	VALA_CCODE_NODE_CLASS (klass)->finalize = vala_ccode_enum_finalize;
	g_type_class_add_private (klass, sizeof (ValaCCodeEnumPrivate));
	VALA_CCODE_NODE_CLASS (klass)->write = vala_ccode_enum_real_write;
}


static void vala_ccode_enum_instance_init (ValaCCodeEnum * self) {
	self->priv = VALA_CCODE_ENUM_GET_PRIVATE (self);
	self->priv->values = (GeeList*) gee_array_list_new (VALA_TYPE_CCODE_ENUM_VALUE, (GBoxedCopyFunc) vala_ccode_node_ref, vala_ccode_node_unref, g_direct_equal);
}


static void vala_ccode_enum_finalize (ValaCCodeNode* obj) {
	ValaCCodeEnum * self;
	self = VALA_CCODE_ENUM (obj);
	self->priv->_name = (g_free (self->priv->_name), NULL);
	(self->priv->values == NULL) ? NULL : (self->priv->values = (gee_collection_object_unref (self->priv->values), NULL));
	VALA_CCODE_NODE_CLASS (vala_ccode_enum_parent_class)->finalize (obj);
}


GType vala_ccode_enum_get_type (void) {
	static GType vala_ccode_enum_type_id = 0;
	if (vala_ccode_enum_type_id == 0) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaCCodeEnumClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_ccode_enum_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaCCodeEnum), 0, (GInstanceInitFunc) vala_ccode_enum_instance_init, NULL };
		vala_ccode_enum_type_id = g_type_register_static (VALA_TYPE_CCODE_NODE, "ValaCCodeEnum", &g_define_type_info, 0);
	}
	return vala_ccode_enum_type_id;
}




