/* valavapicheck.vala
 *
 * Copyright (C) 2007  Mathias Hasselmann
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Mathias Hasselmann <mathias.hasselmann@gmx.de>
 */

#include <glib.h>
#include <glib-object.h>
#include <vala.h>
#include <gee.h>
#include <stdlib.h>
#include <string.h>
#include <gidlparser.h>
#include <gidlmodule.h>
#include <gidlnode.h>
#include <stdio.h>
#include <glib/gstdio.h>


#define VALA_TYPE_VAPI_CHECK (vala_vapi_check_get_type ())
#define VALA_VAPI_CHECK(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_VAPI_CHECK, ValaVAPICheck))
#define VALA_VAPI_CHECK_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_VAPI_CHECK, ValaVAPICheckClass))
#define VALA_IS_VAPI_CHECK(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_VAPI_CHECK))
#define VALA_IS_VAPI_CHECK_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_VAPI_CHECK))
#define VALA_VAPI_CHECK_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_VAPI_CHECK, ValaVAPICheckClass))

typedef struct _ValaVAPICheck ValaVAPICheck;
typedef struct _ValaVAPICheckClass ValaVAPICheckClass;
typedef struct _ValaVAPICheckPrivate ValaVAPICheckPrivate;

struct _ValaVAPICheck {
	GObject parent_instance;
	ValaVAPICheckPrivate * priv;
};

struct _ValaVAPICheckClass {
	GObjectClass parent_class;
};

struct _ValaVAPICheckPrivate {
	ValaCodeContext* _context;
	ValaSourceFile* _gidl;
	ValaSourceFile* _metadata;
	GeeList* _scope;
	GeeSet* _symbols;
};



GType vala_vapi_check_get_type (void);
#define VALA_VAPI_CHECK_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_VAPI_CHECK, ValaVAPICheckPrivate))
enum  {
	VALA_VAPI_CHECK_DUMMY_PROPERTY,
	VALA_VAPI_CHECK_CONTEXT,
	VALA_VAPI_CHECK_GIDL,
	VALA_VAPI_CHECK_METADATA
};
static void vala_vapi_check_set_gidl (ValaVAPICheck* self, ValaSourceFile* value);
static void vala_vapi_check_set_metadata (ValaVAPICheck* self, ValaSourceFile* value);
static void vala_vapi_check_set_context (ValaVAPICheck* self, ValaCodeContext* value);
ValaVAPICheck* vala_vapi_check_new (const char* gidlname, ValaCodeContext* context);
ValaVAPICheck* vala_vapi_check_construct (GType object_type, const char* gidlname, ValaCodeContext* context);
ValaVAPICheck* vala_vapi_check_new (const char* gidlname, ValaCodeContext* context);
ValaSourceFile* vala_vapi_check_get_gidl (ValaVAPICheck* self);
static void vala_vapi_check_parse_members (ValaVAPICheck* self, const char* name, GList* members);
static void _g_list_free_g_idl_module_free (GList* self);
static void vala_vapi_check_parse_gidl (ValaVAPICheck* self);
static char* vala_vapi_check_get_scope (ValaVAPICheck* self);
static void vala_vapi_check_add_symbol (ValaVAPICheck* self, const char* name, const char* separator);
static void vala_vapi_check_enter_scope (ValaVAPICheck* self, const char* name);
static void vala_vapi_check_leave_scope (ValaVAPICheck* self);
ValaSourceFile* vala_vapi_check_get_metadata (ValaVAPICheck* self);
static gint vala_vapi_check_check_metadata (ValaVAPICheck* self);
gint vala_vapi_check_run (ValaVAPICheck* self);
static gint vala_vapi_check_main (char** args, int args_length1);
ValaCodeContext* vala_vapi_check_get_context (ValaVAPICheck* self);
static gpointer vala_vapi_check_parent_class = NULL;
static void vala_vapi_check_finalize (GObject* obj);
static void vala_vapi_check_get_property (GObject * object, guint property_id, GValue * value, GParamSpec * pspec);
static void vala_vapi_check_set_property (GObject * object, guint property_id, const GValue * value, GParamSpec * pspec);
static void _vala_array_destroy (gpointer array, gint array_length, GDestroyNotify destroy_func);
static void _vala_array_free (gpointer array, gint array_length, GDestroyNotify destroy_func);
static gint _vala_array_length (gpointer array);



static char* string_substring (const char* self, glong offset, glong len) {
	glong string_length;
	const char* start;
	g_return_val_if_fail (self != NULL, NULL);
	string_length = g_utf8_strlen (self, -1);
	if (offset < 0) {
		offset = string_length + offset;
		g_return_val_if_fail (offset >= 0, NULL);
	} else {
		g_return_val_if_fail (offset <= string_length, NULL);
	}
	if (len < 0) {
		len = string_length - offset;
	}
	g_return_val_if_fail ((offset + len) <= string_length, NULL);
	start = g_utf8_offset_to_pointer (self, offset);
	return g_strndup (start, ((gchar*) g_utf8_offset_to_pointer (start, len)) - ((gchar*) start));
}


ValaVAPICheck* vala_vapi_check_construct (GType object_type, const char* gidlname, ValaCodeContext* context) {
	GParameter * __params;
	GParameter * __params_it;
	ValaSourceFile* _tmp1_;
	ValaCodeContext* _tmp0_;
	ValaSourceFile* _tmp4_;
	char* _tmp3_;
	char* _tmp2_;
	ValaVAPICheck * self;
	g_return_val_if_fail (gidlname != NULL, NULL);
	g_return_val_if_fail (context != NULL, NULL);
	__params = g_new0 (GParameter, 3);
	__params_it = __params;
	_tmp1_ = NULL;
	_tmp0_ = NULL;
	__params_it->name = "gidl";
	g_value_init (&__params_it->value, VALA_TYPE_SOURCE_FILE);
	vala_value_set_source_file (&__params_it->value, _tmp1_ = vala_source_file_new (context, gidlname, FALSE, NULL));
	__params_it++;
	(_tmp1_ == NULL) ? NULL : (_tmp1_ = (vala_source_file_unref (_tmp1_), NULL));
	(_tmp0_ == NULL) ? NULL : (_tmp0_ = (vala_code_context_unref (_tmp0_), NULL));
	_tmp4_ = NULL;
	_tmp3_ = NULL;
	_tmp2_ = NULL;
	vala_vapi_check_set_metadata (self, _tmp4_ = vala_source_file_new (context, _tmp3_ = g_strconcat (_tmp2_ = string_substring (gidlname, (glong) 0, g_utf8_strlen (gidlname, -1) - 5), ".metadata", NULL), FALSE, NULL));
	(_tmp4_ == NULL) ? NULL : (_tmp4_ = (vala_source_file_unref (_tmp4_), NULL));
	_tmp3_ = (g_free (_tmp3_), NULL);
	_tmp2_ = (g_free (_tmp2_), NULL);
	vala_vapi_check_set_context (self, context);
	self = g_object_newv (object_type, __params_it - __params, __params);
	while (__params_it > __params) {
		--__params_it;
		g_value_unset (&__params_it->value);
	}
	g_free (__params);
	return self;
}


ValaVAPICheck* vala_vapi_check_new (const char* gidlname, ValaCodeContext* context) {
	return vala_vapi_check_construct (VALA_TYPE_VAPI_CHECK, gidlname, context);
}


static void _g_list_free_g_idl_module_free (GList* self) {
	g_list_foreach (self, (GFunc) g_idl_module_free, NULL);
	g_list_free (self);
}


static void vala_vapi_check_parse_gidl (ValaVAPICheck* self) {
	GError * _inner_error_;
	GeeList* _tmp0_;
	GeeSet* _tmp1_;
	g_return_if_fail (self != NULL);
	_inner_error_ = NULL;
	_tmp0_ = NULL;
	self->priv->_scope = (_tmp0_ = (GeeList*) gee_array_list_new (G_TYPE_STRING, (GBoxedCopyFunc) g_strdup, g_free, g_direct_equal), (self->priv->_scope == NULL) ? NULL : (self->priv->_scope = (gee_collection_object_unref (self->priv->_scope), NULL)), _tmp0_);
	_tmp1_ = NULL;
	self->priv->_symbols = (_tmp1_ = (GeeSet*) gee_hash_set_new (G_TYPE_STRING, (GBoxedCopyFunc) g_strdup, g_free, g_str_hash, g_str_equal), (self->priv->_symbols == NULL) ? NULL : (self->priv->_symbols = (gee_collection_object_unref (self->priv->_symbols), NULL)), _tmp1_);
	{
		GList* _tmp2_;
		_tmp2_ = g_idl_parse_file (vala_source_file_get_filename (self->priv->_gidl), &_inner_error_);
		if (_inner_error_ != NULL) {
			if (_inner_error_->domain == G_MARKUP_ERROR) {
				goto __catch0_g_markup_error;
			}
			goto __finally0;
		}
		{
			GList* module_collection;
			GList* module_it;
			module_collection = _tmp2_;
			for (module_it = module_collection; module_it != NULL; module_it = module_it->next) {
				GIdlModule* module;
				module = (GIdlModule*) module_it->data;
				{
					vala_vapi_check_parse_members (self, module->name, module->entries);
				}
			}
			(module_collection == NULL) ? NULL : (module_collection = (_g_list_free_g_idl_module_free (module_collection), NULL));
		}
	}
	goto __finally0;
	__catch0_g_markup_error:
	{
		GError * e;
		e = _inner_error_;
		_inner_error_ = NULL;
		{
			fprintf (stderr, "%s: %s\n", vala_source_file_get_filename (self->priv->_gidl), e->message);
			(e == NULL) ? NULL : (e = (g_error_free (e), NULL));
		}
	}
	__finally0:
	if (_inner_error_ != NULL) {
		g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, _inner_error_->message);
		g_clear_error (&_inner_error_);
		return;
	}
}


static void vala_vapi_check_add_symbol (ValaVAPICheck* self, const char* name, const char* separator) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (name != NULL);
	if (NULL != separator) {
		char* _tmp1_;
		char* _tmp0_;
		char* _tmp2_;
		char* fullname;
		_tmp1_ = NULL;
		_tmp0_ = NULL;
		_tmp2_ = NULL;
		fullname = (_tmp2_ = g_strconcat (_tmp1_ = g_strconcat (_tmp0_ = vala_vapi_check_get_scope (self), separator, NULL), name, NULL), _tmp1_ = (g_free (_tmp1_), NULL), _tmp0_ = (g_free (_tmp0_), NULL), _tmp2_);
		gee_collection_add ((GeeCollection*) self->priv->_symbols, fullname);
		fullname = (g_free (fullname), NULL);
	} else {
		gee_collection_add ((GeeCollection*) self->priv->_symbols, name);
	}
}


static char* vala_vapi_check_get_scope (ValaVAPICheck* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return (char*) gee_list_get (self->priv->_scope, gee_collection_get_size ((GeeCollection*) self->priv->_scope) - 1);
}


static void vala_vapi_check_enter_scope (ValaVAPICheck* self, const char* name) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (name != NULL);
	gee_collection_add ((GeeCollection*) self->priv->_scope, name);
	vala_vapi_check_add_symbol (self, name, NULL);
}


static void vala_vapi_check_leave_scope (ValaVAPICheck* self) {
	g_return_if_fail (self != NULL);
	gee_list_remove_at (self->priv->_scope, gee_collection_get_size ((GeeCollection*) self->priv->_scope) - 1);
}


static void vala_vapi_check_parse_members (ValaVAPICheck* self, const char* name, GList* members) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (name != NULL);
	vala_vapi_check_enter_scope (self, name);
	{
		GList* node_collection;
		GList* node_it;
		node_collection = members;
		for (node_it = node_collection; node_it != NULL; node_it = node_it->next) {
			GIdlNode* node;
			node = (GIdlNode*) node_it->data;
			{
				switch (node->type) {
					case G_IDL_NODE_ENUM:
					{
						vala_vapi_check_parse_members (self, ((GIdlNodeEnum*) node)->gtype_name, ((GIdlNodeEnum*) node)->values);
						break;
					}
					case G_IDL_NODE_FUNCTION:
					{
						vala_vapi_check_parse_members (self, ((GIdlNodeFunction*) node)->symbol, ((GIdlNodeFunction*) node)->parameters);
						break;
					}
					case G_IDL_NODE_BOXED:
					{
						vala_vapi_check_parse_members (self, ((GIdlNodeBoxed*) node)->gtype_name, ((GIdlNodeBoxed*) node)->members);
						break;
					}
					case G_IDL_NODE_INTERFACE:
					case G_IDL_NODE_OBJECT:
					{
						vala_vapi_check_parse_members (self, ((GIdlNodeInterface*) node)->gtype_name, ((GIdlNodeInterface*) node)->members);
						break;
					}
					case G_IDL_NODE_FIELD:
					case G_IDL_NODE_PARAM:
					{
						vala_vapi_check_add_symbol (self, node->name, ".");
						break;
					}
					case G_IDL_NODE_PROPERTY:
					case G_IDL_NODE_SIGNAL:
					{
						vala_vapi_check_add_symbol (self, node->name, "::");
						break;
					}
					case G_IDL_NODE_STRUCT:
					{
						vala_vapi_check_parse_members (self, node->name, ((GIdlNodeStruct*) node)->members);
						break;
					}
					case G_IDL_NODE_VALUE:
					case G_IDL_NODE_VFUNC:
					{
						break;
					}
					default:
					{
						g_warning ("valavapicheck.vala:121: TODO: %s: Implement support for type %d nodes", node->name, (gint) node->type);
						break;
					}
				}
			}
		}
	}
	vala_vapi_check_leave_scope (self);
}


static gint vala_vapi_check_check_metadata (ValaVAPICheck* self) {
	GError * _inner_error_;
	g_return_val_if_fail (self != NULL, 0);
	_inner_error_ = NULL;
	{
		GIOChannel* metafile;
		char* line;
		gint lineno;
		gint _tmp9_;
		metafile = g_io_channel_new_file (vala_source_file_get_filename (self->priv->_metadata), "r", &_inner_error_);
		if (_inner_error_ != NULL) {
			if (_inner_error_->domain == G_FILE_ERROR) {
				goto __catch1_g_file_error;
			}
			goto __finally1;
		}
		line = NULL;
		lineno = 1;
		while (TRUE) {
			char* _tmp2_;
			GIOStatus _tmp1_;
			char* _tmp0_;
			GIOStatus _tmp3_;
			char** _tmp5_;
			gint tokens_size;
			gint tokens_length1;
			char** _tmp4_;
			char** tokens;
			const char* _tmp6_;
			char* symbol;
			gboolean _tmp7_;
			_tmp2_ = NULL;
			_tmp0_ = NULL;
			_tmp3_ = (_tmp1_ = g_io_channel_read_line (metafile, &_tmp0_, NULL, NULL, &_inner_error_), line = (_tmp2_ = _tmp0_, line = (g_free (line), NULL), _tmp2_), _tmp1_);
			if (_inner_error_ != NULL) {
				(metafile == NULL) ? NULL : (metafile = (g_io_channel_unref (metafile), NULL));
				line = (g_free (line), NULL);
				if (_inner_error_->domain == G_FILE_ERROR) {
					goto __catch1_g_file_error;
				}
				goto __finally1;
			}
			if (!(G_IO_STATUS_NORMAL == _tmp3_)) {
				break;
			}
			_tmp5_ = NULL;
			_tmp4_ = NULL;
			tokens = (_tmp5_ = _tmp4_ = g_strsplit (line, " ", 2), tokens_length1 = _vala_array_length (_tmp4_), tokens_size = tokens_length1, _tmp5_);
			_tmp6_ = NULL;
			symbol = (_tmp6_ = tokens[0], (_tmp6_ == NULL) ? NULL : g_strdup (_tmp6_));
			_tmp7_ = FALSE;
			if (strlen (symbol) > 0) {
				_tmp7_ = !gee_collection_contains ((GeeCollection*) self->priv->_symbols, symbol);
			} else {
				_tmp7_ = FALSE;
			}
			if (_tmp7_) {
				ValaSourceReference* src;
				char* _tmp8_;
				src = vala_source_reference_new (self->priv->_metadata, lineno, 1, lineno, (gint) g_utf8_strlen (symbol, -1));
				_tmp8_ = NULL;
				vala_report_error (src, _tmp8_ = g_strdup_printf ("Symbol `%s' not found", symbol));
				_tmp8_ = (g_free (_tmp8_), NULL);
				(src == NULL) ? NULL : (src = (vala_source_reference_unref (src), NULL));
			}
			lineno = lineno + 1;
			tokens = (_vala_array_free (tokens, tokens_length1, (GDestroyNotify) g_free), NULL);
			symbol = (g_free (symbol), NULL);
		}
		return (_tmp9_ = 0, (metafile == NULL) ? NULL : (metafile = (g_io_channel_unref (metafile), NULL)), line = (g_free (line), NULL), _tmp9_);
	}
	goto __finally1;
	__catch1_g_file_error:
	{
		GError * _error_;
		_error_ = _inner_error_;
		_inner_error_ = NULL;
		{
			char* _tmp10_;
			gint _tmp11_;
			_tmp10_ = NULL;
			vala_report_error (NULL, _tmp10_ = g_strdup_printf ("%s: %s", vala_source_file_get_filename (self->priv->_metadata), _error_->message));
			_tmp10_ = (g_free (_tmp10_), NULL);
			return (_tmp11_ = 1, (_error_ == NULL) ? NULL : (_error_ = (g_error_free (_error_), NULL)), _tmp11_);
		}
	}
	__finally1:
	if (_inner_error_ != NULL) {
		g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, _inner_error_->message);
		g_clear_error (&_inner_error_);
		return 0;
	}
}


gint vala_vapi_check_run (ValaVAPICheck* self) {
	g_return_val_if_fail (self != NULL, 0);
	if (!g_file_test (vala_source_file_get_filename (self->priv->_gidl), G_FILE_TEST_IS_REGULAR)) {
		char* _tmp0_;
		_tmp0_ = NULL;
		vala_report_error (NULL, _tmp0_ = g_strdup_printf ("%s not found", vala_source_file_get_filename (self->priv->_gidl)));
		_tmp0_ = (g_free (_tmp0_), NULL);
		return 2;
	}
	if (!g_file_test (vala_source_file_get_filename (self->priv->_metadata), G_FILE_TEST_IS_REGULAR)) {
		char* _tmp2_;
		_tmp2_ = NULL;
		vala_report_error (NULL, _tmp2_ = g_strdup_printf ("%s not found", vala_source_file_get_filename (self->priv->_metadata)));
		_tmp2_ = (g_free (_tmp2_), NULL);
		return 2;
	}
	vala_vapi_check_parse_gidl (self);
	return vala_vapi_check_check_metadata (self);
}


static gint vala_vapi_check_main (char** args, int args_length1) {
	gboolean _tmp0_;
	ValaCodeContext* _tmp3_;
	ValaVAPICheck* _tmp4_;
	ValaVAPICheck* vapicheck;
	gint _tmp5_;
	_tmp0_ = FALSE;
	if (2 != args_length1) {
		_tmp0_ = TRUE;
	} else {
		_tmp0_ = !g_str_has_suffix (args[1], ".gidl");
	}
	if (_tmp0_) {
		char* _tmp1_;
		_tmp1_ = NULL;
		fprintf (stdout, "Usage: %s library.gidl\n", _tmp1_ = g_path_get_basename (args[0]));
		_tmp1_ = (g_free (_tmp1_), NULL);
		return 2;
	}
	_tmp3_ = NULL;
	_tmp4_ = NULL;
	vapicheck = (_tmp4_ = vala_vapi_check_new (args[1], _tmp3_ = vala_code_context_new ()), (_tmp3_ == NULL) ? NULL : (_tmp3_ = (vala_code_context_unref (_tmp3_), NULL)), _tmp4_);
	return (_tmp5_ = vala_vapi_check_run (vapicheck), (vapicheck == NULL) ? NULL : (vapicheck = (g_object_unref (vapicheck), NULL)), _tmp5_);
}


int main (int argc, char ** argv) {
	g_type_init ();
	return vala_vapi_check_main (argv, argc);
}


ValaCodeContext* vala_vapi_check_get_context (ValaVAPICheck* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return self->priv->_context;
}


static void vala_vapi_check_set_context (ValaVAPICheck* self, ValaCodeContext* value) {
	ValaCodeContext* _tmp2_;
	ValaCodeContext* _tmp1_;
	g_return_if_fail (self != NULL);
	_tmp2_ = NULL;
	_tmp1_ = NULL;
	self->priv->_context = (_tmp2_ = (_tmp1_ = value, (_tmp1_ == NULL) ? NULL : vala_code_context_ref (_tmp1_)), (self->priv->_context == NULL) ? NULL : (self->priv->_context = (vala_code_context_unref (self->priv->_context), NULL)), _tmp2_);
	g_object_notify ((GObject *) self, "context");
}


ValaSourceFile* vala_vapi_check_get_gidl (ValaVAPICheck* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return self->priv->_gidl;
}


static void vala_vapi_check_set_gidl (ValaVAPICheck* self, ValaSourceFile* value) {
	ValaSourceFile* _tmp2_;
	ValaSourceFile* _tmp1_;
	g_return_if_fail (self != NULL);
	_tmp2_ = NULL;
	_tmp1_ = NULL;
	self->priv->_gidl = (_tmp2_ = (_tmp1_ = value, (_tmp1_ == NULL) ? NULL : vala_source_file_ref (_tmp1_)), (self->priv->_gidl == NULL) ? NULL : (self->priv->_gidl = (vala_source_file_unref (self->priv->_gidl), NULL)), _tmp2_);
	g_object_notify ((GObject *) self, "gidl");
}


ValaSourceFile* vala_vapi_check_get_metadata (ValaVAPICheck* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return self->priv->_metadata;
}


static void vala_vapi_check_set_metadata (ValaVAPICheck* self, ValaSourceFile* value) {
	ValaSourceFile* _tmp2_;
	ValaSourceFile* _tmp1_;
	g_return_if_fail (self != NULL);
	_tmp2_ = NULL;
	_tmp1_ = NULL;
	self->priv->_metadata = (_tmp2_ = (_tmp1_ = value, (_tmp1_ == NULL) ? NULL : vala_source_file_ref (_tmp1_)), (self->priv->_metadata == NULL) ? NULL : (self->priv->_metadata = (vala_source_file_unref (self->priv->_metadata), NULL)), _tmp2_);
	g_object_notify ((GObject *) self, "metadata");
}


static void vala_vapi_check_class_init (ValaVAPICheckClass * klass) {
	vala_vapi_check_parent_class = g_type_class_peek_parent (klass);
	g_type_class_add_private (klass, sizeof (ValaVAPICheckPrivate));
	G_OBJECT_CLASS (klass)->get_property = vala_vapi_check_get_property;
	G_OBJECT_CLASS (klass)->set_property = vala_vapi_check_set_property;
	G_OBJECT_CLASS (klass)->finalize = vala_vapi_check_finalize;
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_VAPI_CHECK_CONTEXT, vala_param_spec_code_context ("context", "context", "context", VALA_TYPE_CODE_CONTEXT, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_CONSTRUCT_ONLY));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_VAPI_CHECK_GIDL, vala_param_spec_source_file ("gidl", "gidl", "gidl", VALA_TYPE_SOURCE_FILE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_CONSTRUCT_ONLY));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_VAPI_CHECK_METADATA, vala_param_spec_source_file ("metadata", "metadata", "metadata", VALA_TYPE_SOURCE_FILE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_CONSTRUCT_ONLY));
}


static void vala_vapi_check_instance_init (ValaVAPICheck * self) {
	self->priv = VALA_VAPI_CHECK_GET_PRIVATE (self);
}


static void vala_vapi_check_finalize (GObject* obj) {
	ValaVAPICheck * self;
	self = VALA_VAPI_CHECK (obj);
	(self->priv->_context == NULL) ? NULL : (self->priv->_context = (vala_code_context_unref (self->priv->_context), NULL));
	(self->priv->_gidl == NULL) ? NULL : (self->priv->_gidl = (vala_source_file_unref (self->priv->_gidl), NULL));
	(self->priv->_metadata == NULL) ? NULL : (self->priv->_metadata = (vala_source_file_unref (self->priv->_metadata), NULL));
	(self->priv->_scope == NULL) ? NULL : (self->priv->_scope = (gee_collection_object_unref (self->priv->_scope), NULL));
	(self->priv->_symbols == NULL) ? NULL : (self->priv->_symbols = (gee_collection_object_unref (self->priv->_symbols), NULL));
	G_OBJECT_CLASS (vala_vapi_check_parent_class)->finalize (obj);
}


GType vala_vapi_check_get_type (void) {
	static GType vala_vapi_check_type_id = 0;
	if (vala_vapi_check_type_id == 0) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaVAPICheckClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_vapi_check_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaVAPICheck), 0, (GInstanceInitFunc) vala_vapi_check_instance_init, NULL };
		vala_vapi_check_type_id = g_type_register_static (G_TYPE_OBJECT, "ValaVAPICheck", &g_define_type_info, 0);
	}
	return vala_vapi_check_type_id;
}


static void vala_vapi_check_get_property (GObject * object, guint property_id, GValue * value, GParamSpec * pspec) {
	ValaVAPICheck * self;
	gpointer boxed;
	self = VALA_VAPI_CHECK (object);
	switch (property_id) {
		case VALA_VAPI_CHECK_CONTEXT:
		vala_value_set_code_context (value, vala_vapi_check_get_context (self));
		break;
		case VALA_VAPI_CHECK_GIDL:
		vala_value_set_source_file (value, vala_vapi_check_get_gidl (self));
		break;
		case VALA_VAPI_CHECK_METADATA:
		vala_value_set_source_file (value, vala_vapi_check_get_metadata (self));
		break;
		default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
		break;
	}
}


static void vala_vapi_check_set_property (GObject * object, guint property_id, const GValue * value, GParamSpec * pspec) {
	ValaVAPICheck * self;
	self = VALA_VAPI_CHECK (object);
	switch (property_id) {
		case VALA_VAPI_CHECK_CONTEXT:
		vala_vapi_check_set_context (self, vala_value_get_code_context (value));
		break;
		case VALA_VAPI_CHECK_GIDL:
		vala_vapi_check_set_gidl (self, vala_value_get_source_file (value));
		break;
		case VALA_VAPI_CHECK_METADATA:
		vala_vapi_check_set_metadata (self, vala_value_get_source_file (value));
		break;
		default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
		break;
	}
}


static void _vala_array_destroy (gpointer array, gint array_length, GDestroyNotify destroy_func) {
	if ((array != NULL) && (destroy_func != NULL)) {
		int i;
		for (i = 0; i < array_length; i = i + 1) {
			if (((gpointer*) array)[i] != NULL) {
				destroy_func (((gpointer*) array)[i]);
			}
		}
	}
}


static void _vala_array_free (gpointer array, gint array_length, GDestroyNotify destroy_func) {
	_vala_array_destroy (array, array_length, destroy_func);
	g_free (array);
}


static gint _vala_array_length (gpointer array) {
	int length;
	length = 0;
	if (array) {
		while (((gpointer*) array)[length]) {
			length++;
		}
	}
	return length;
}




