/* hashset.vala
 *
 * Copyright (C) 1995-1997  Peter Mattis, Spencer Kimball and Josh MacDonald
 * Copyright (C) 1997-2000  GLib Team and others
 * Copyright (C) 2007-2009  Jürg Billeter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 */

#include <glib.h>
#include <glib-object.h>


#define GEE_TYPE_COLLECTION_OBJECT (gee_collection_object_get_type ())
#define GEE_COLLECTION_OBJECT(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), GEE_TYPE_COLLECTION_OBJECT, GeeCollectionObject))
#define GEE_COLLECTION_OBJECT_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), GEE_TYPE_COLLECTION_OBJECT, GeeCollectionObjectClass))
#define GEE_IS_COLLECTION_OBJECT(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), GEE_TYPE_COLLECTION_OBJECT))
#define GEE_IS_COLLECTION_OBJECT_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), GEE_TYPE_COLLECTION_OBJECT))
#define GEE_COLLECTION_OBJECT_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), GEE_TYPE_COLLECTION_OBJECT, GeeCollectionObjectClass))

typedef struct _GeeCollectionObject GeeCollectionObject;
typedef struct _GeeCollectionObjectClass GeeCollectionObjectClass;
typedef struct _GeeCollectionObjectPrivate GeeCollectionObjectPrivate;

#define GEE_TYPE_ITERABLE (gee_iterable_get_type ())
#define GEE_ITERABLE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), GEE_TYPE_ITERABLE, GeeIterable))
#define GEE_IS_ITERABLE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), GEE_TYPE_ITERABLE))
#define GEE_ITERABLE_GET_INTERFACE(obj) (G_TYPE_INSTANCE_GET_INTERFACE ((obj), GEE_TYPE_ITERABLE, GeeIterableIface))

typedef struct _GeeIterable GeeIterable;
typedef struct _GeeIterableIface GeeIterableIface;

#define GEE_TYPE_ITERATOR (gee_iterator_get_type ())
#define GEE_ITERATOR(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), GEE_TYPE_ITERATOR, GeeIterator))
#define GEE_IS_ITERATOR(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), GEE_TYPE_ITERATOR))
#define GEE_ITERATOR_GET_INTERFACE(obj) (G_TYPE_INSTANCE_GET_INTERFACE ((obj), GEE_TYPE_ITERATOR, GeeIteratorIface))

typedef struct _GeeIterator GeeIterator;
typedef struct _GeeIteratorIface GeeIteratorIface;

#define GEE_TYPE_COLLECTION (gee_collection_get_type ())
#define GEE_COLLECTION(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), GEE_TYPE_COLLECTION, GeeCollection))
#define GEE_IS_COLLECTION(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), GEE_TYPE_COLLECTION))
#define GEE_COLLECTION_GET_INTERFACE(obj) (G_TYPE_INSTANCE_GET_INTERFACE ((obj), GEE_TYPE_COLLECTION, GeeCollectionIface))

typedef struct _GeeCollection GeeCollection;
typedef struct _GeeCollectionIface GeeCollectionIface;

#define GEE_TYPE_SET (gee_set_get_type ())
#define GEE_SET(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), GEE_TYPE_SET, GeeSet))
#define GEE_IS_SET(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), GEE_TYPE_SET))
#define GEE_SET_GET_INTERFACE(obj) (G_TYPE_INSTANCE_GET_INTERFACE ((obj), GEE_TYPE_SET, GeeSetIface))

typedef struct _GeeSet GeeSet;
typedef struct _GeeSetIface GeeSetIface;

#define GEE_TYPE_HASH_SET (gee_hash_set_get_type ())
#define GEE_HASH_SET(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), GEE_TYPE_HASH_SET, GeeHashSet))
#define GEE_HASH_SET_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), GEE_TYPE_HASH_SET, GeeHashSetClass))
#define GEE_IS_HASH_SET(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), GEE_TYPE_HASH_SET))
#define GEE_IS_HASH_SET_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), GEE_TYPE_HASH_SET))
#define GEE_HASH_SET_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), GEE_TYPE_HASH_SET, GeeHashSetClass))

typedef struct _GeeHashSet GeeHashSet;
typedef struct _GeeHashSetClass GeeHashSetClass;
typedef struct _GeeHashSetPrivate GeeHashSetPrivate;
typedef struct _GeeHashSetNode GeeHashSetNode;

#define GEE_HASH_SET_TYPE_ITERATOR (gee_hash_set_iterator_get_type ())
#define GEE_HASH_SET_ITERATOR(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), GEE_HASH_SET_TYPE_ITERATOR, GeeHashSetIterator))
#define GEE_HASH_SET_ITERATOR_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), GEE_HASH_SET_TYPE_ITERATOR, GeeHashSetIteratorClass))
#define GEE_HASH_SET_IS_ITERATOR(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), GEE_HASH_SET_TYPE_ITERATOR))
#define GEE_HASH_SET_IS_ITERATOR_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), GEE_HASH_SET_TYPE_ITERATOR))
#define GEE_HASH_SET_ITERATOR_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), GEE_HASH_SET_TYPE_ITERATOR, GeeHashSetIteratorClass))

typedef struct _GeeHashSetIterator GeeHashSetIterator;
typedef struct _GeeHashSetIteratorClass GeeHashSetIteratorClass;
typedef struct _GeeHashSetIteratorPrivate GeeHashSetIteratorPrivate;

/**
 * Base class for all collections.
 */
struct _GeeCollectionObject {
	GTypeInstance parent_instance;
	volatile int ref_count;
	GeeCollectionObjectPrivate * priv;
};

struct _GeeCollectionObjectClass {
	GTypeClass parent_class;
	void (*finalize) (GeeCollectionObject *self);
};

/**
 * Implemented by classes that support a simple iteration over instances of the
 * collection.
 */
struct _GeeIteratorIface {
	GTypeInterface parent_iface;
	gboolean (*next) (GeeIterator* self);
	gpointer (*get) (GeeIterator* self);
};

/**
 * Implemented by classes that support a simple iteration over instances of the
 * collection.
 */
struct _GeeIterableIface {
	GTypeInterface parent_iface;
	GType (*get_element_type) (GeeIterable* self);
	GeeIterator* (*iterator) (GeeIterable* self);
};

/**
 * Serves as the base interface for implementing collection classes. Defines
 * size, iteration, and modification methods.
 */
struct _GeeCollectionIface {
	GTypeInterface parent_iface;
	gboolean (*contains) (GeeCollection* self, gconstpointer item);
	gboolean (*add) (GeeCollection* self, gconstpointer item);
	gboolean (*remove) (GeeCollection* self, gconstpointer item);
	void (*clear) (GeeCollection* self);
	gint (*get_size) (GeeCollection* self);
};

/**
 * A set is a collection without duplicates.
 */
struct _GeeSetIface {
	GTypeInterface parent_iface;
};

/**
 * Hashtable implementation of the Set interface.
 */
struct _GeeHashSet {
	GeeCollectionObject parent_instance;
	GeeHashSetPrivate * priv;
};

struct _GeeHashSetClass {
	GeeCollectionObjectClass parent_class;
};

struct _GeeHashSetPrivate {
	GType g_type;
	GBoxedCopyFunc g_dup_func;
	GDestroyNotify g_destroy_func;
	gint _array_size;
	gint _nnodes;
	GeeHashSetNode** _nodes;
	gint _nodes_length1;
	gint _nodes_size;
	gint _stamp;
	GHashFunc _hash_func;
	GEqualFunc _equal_func;
};

struct _GeeHashSetNode {
	gpointer key;
	GeeHashSetNode* next;
	guint key_hash;
};

struct _GeeHashSetIterator {
	GeeCollectionObject parent_instance;
	GeeHashSetIteratorPrivate * priv;
};

struct _GeeHashSetIteratorClass {
	GeeCollectionObjectClass parent_class;
};

struct _GeeHashSetIteratorPrivate {
	GType g_type;
	GBoxedCopyFunc g_dup_func;
	GDestroyNotify g_destroy_func;
	GeeHashSet* _set;
	gint _index;
	GeeHashSetNode* _node;
	gint _stamp;
};


static gpointer gee_hash_set_iterator_parent_class = NULL;
static GeeIteratorIface* gee_hash_set_iterator_gee_iterator_parent_iface = NULL;
static gpointer gee_hash_set_parent_class = NULL;
static GeeIterableIface* gee_hash_set_gee_iterable_parent_iface = NULL;
static GeeCollectionIface* gee_hash_set_gee_collection_parent_iface = NULL;
static GeeSetIface* gee_hash_set_gee_set_parent_iface = NULL;

gpointer gee_collection_object_ref (gpointer instance);
void gee_collection_object_unref (gpointer instance);
GParamSpec* gee_param_spec_collection_object (const gchar* name, const gchar* nick, const gchar* blurb, GType object_type, GParamFlags flags);
void gee_value_set_collection_object (GValue* value, gpointer v_object);
gpointer gee_value_get_collection_object (const GValue* value);
GType gee_collection_object_get_type (void);
GType gee_iterator_get_type (void);
GType gee_iterable_get_type (void);
GType gee_collection_get_type (void);
GType gee_set_get_type (void);
GType gee_hash_set_get_type (void);
#define GEE_HASH_SET_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), GEE_TYPE_HASH_SET, GeeHashSetPrivate))
enum  {
	GEE_HASH_SET_DUMMY_PROPERTY
};
#define GEE_HASH_SET_MIN_SIZE 11
#define GEE_HASH_SET_MAX_SIZE 13845163
void gee_hash_set_set_hash_func (GeeHashSet* self, GHashFunc value);
void gee_hash_set_set_equal_func (GeeHashSet* self, GEqualFunc value);
GeeHashSet* gee_hash_set_new (GType g_type, GBoxedCopyFunc g_dup_func, GDestroyNotify g_destroy_func, GHashFunc hash_func, GEqualFunc equal_func);
GeeHashSet* gee_hash_set_construct (GType object_type, GType g_type, GBoxedCopyFunc g_dup_func, GDestroyNotify g_destroy_func, GHashFunc hash_func, GEqualFunc equal_func);
GeeHashSet* gee_hash_set_new (GType g_type, GBoxedCopyFunc g_dup_func, GDestroyNotify g_destroy_func, GHashFunc hash_func, GEqualFunc equal_func);
static GeeHashSetNode** gee_hash_set_lookup_node (GeeHashSet* self, gconstpointer key);
static gboolean gee_hash_set_real_contains (GeeCollection* base, gconstpointer key);
static GType gee_hash_set_real_get_element_type (GeeIterable* base);
static GeeHashSetIterator* gee_hash_set_iterator_new (GType g_type, GBoxedCopyFunc g_dup_func, GDestroyNotify g_destroy_func, GeeHashSet* set);
static GeeHashSetIterator* gee_hash_set_iterator_construct (GType object_type, GType g_type, GBoxedCopyFunc g_dup_func, GDestroyNotify g_destroy_func, GeeHashSet* set);
static GType gee_hash_set_iterator_get_type (void);
static GeeIterator* gee_hash_set_real_iterator (GeeIterable* base);
static GeeHashSetNode* gee_hash_set_node_new (gpointer k, guint hash);
static GeeHashSetNode* gee_hash_set_node_new (gpointer k, guint hash);
static void gee_hash_set_resize (GeeHashSet* self);
static gboolean gee_hash_set_real_add (GeeCollection* base, gconstpointer key);
static gboolean gee_hash_set_real_remove (GeeCollection* base, gconstpointer key);
static void gee_hash_set_real_clear (GeeCollection* base);
void gee_collection_clear (GeeCollection* self);
static void gee_hash_set_node_free (GeeHashSetNode* self);
static void gee_hash_set_node_instance_init (GeeHashSetNode * self);
#define GEE_HASH_SET_ITERATOR_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), GEE_HASH_SET_TYPE_ITERATOR, GeeHashSetIteratorPrivate))
enum  {
	GEE_HASH_SET_ITERATOR_DUMMY_PROPERTY
};
static void gee_hash_set_iterator_set_set (GeeHashSetIterator* self, GeeHashSet* value);
static GeeHashSetIterator* gee_hash_set_iterator_new (GType g_type, GBoxedCopyFunc g_dup_func, GDestroyNotify g_destroy_func, GeeHashSet* set);
static gboolean gee_hash_set_iterator_real_next (GeeIterator* base);
static gpointer gee_hash_set_iterator_real_get (GeeIterator* base);
static void gee_hash_set_iterator_finalize (GeeCollectionObject* obj);
static void gee_hash_set_finalize (GeeCollectionObject* obj);
static void _vala_array_destroy (gpointer array, gint array_length, GDestroyNotify destroy_func);
static void _vala_array_free (gpointer array, gint array_length, GDestroyNotify destroy_func);



GeeHashSet* gee_hash_set_construct (GType object_type, GType g_type, GBoxedCopyFunc g_dup_func, GDestroyNotify g_destroy_func, GHashFunc hash_func, GEqualFunc equal_func) {
	GeeHashSet* self;
	GeeHashSetNode** _tmp0_;
	self = (GeeHashSet*) g_type_create_instance (object_type);
	self->priv->g_type = g_type;
	self->priv->g_dup_func = g_dup_func;
	self->priv->g_destroy_func = g_destroy_func;
	gee_hash_set_set_hash_func (self, hash_func);
	gee_hash_set_set_equal_func (self, equal_func);
	self->priv->_array_size = GEE_HASH_SET_MIN_SIZE;
	_tmp0_ = NULL;
	self->priv->_nodes = (_tmp0_ = g_new0 (GeeHashSetNode*, self->priv->_array_size + 1), self->priv->_nodes = (_vala_array_free (self->priv->_nodes, self->priv->_nodes_length1, (GDestroyNotify) gee_hash_set_node_free), NULL), self->priv->_nodes_length1 = self->priv->_array_size, self->priv->_nodes_size = self->priv->_nodes_length1, _tmp0_);
	return self;
}


GeeHashSet* gee_hash_set_new (GType g_type, GBoxedCopyFunc g_dup_func, GDestroyNotify g_destroy_func, GHashFunc hash_func, GEqualFunc equal_func) {
	return gee_hash_set_construct (GEE_TYPE_HASH_SET, g_type, g_dup_func, g_destroy_func, hash_func, equal_func);
}


static GeeHashSetNode** gee_hash_set_lookup_node (GeeHashSet* self, gconstpointer key) {
	GeeHashSetNode** result;
	guint hash_value;
	GeeHashSetNode** node;
	g_return_val_if_fail (self != NULL, NULL);
	hash_value = self->priv->_hash_func (key);
	node = &self->priv->_nodes[hash_value % self->priv->_array_size];
	while (TRUE) {
		gboolean _tmp0_;
		_tmp0_ = FALSE;
		if ((*node) != NULL) {
			gboolean _tmp1_;
			_tmp1_ = FALSE;
			if (hash_value != (*node)->key_hash) {
				_tmp1_ = TRUE;
			} else {
				_tmp1_ = !self->priv->_equal_func ((*node)->key, key);
			}
			_tmp0_ = _tmp1_;
		} else {
			_tmp0_ = FALSE;
		}
		if (!_tmp0_) {
			break;
		}
		node = &(*node)->next;
	}
	result = node;
	return result;
}


static gboolean gee_hash_set_real_contains (GeeCollection* base, gconstpointer key) {
	GeeHashSet * self;
	gboolean result;
	GeeHashSetNode** node;
	self = (GeeHashSet*) base;
	node = gee_hash_set_lookup_node (self, key);
	result = (*node) != NULL;
	return result;
}


static GType gee_hash_set_real_get_element_type (GeeIterable* base) {
	GeeHashSet * self;
	GType result;
	self = (GeeHashSet*) base;
	result = self->priv->g_type;
	return result;
}


static GeeIterator* gee_hash_set_real_iterator (GeeIterable* base) {
	GeeHashSet * self;
	GeeIterator* result;
	self = (GeeHashSet*) base;
	result = (GeeIterator*) gee_hash_set_iterator_new (self->priv->g_type, (GBoxedCopyFunc) self->priv->g_dup_func, self->priv->g_destroy_func, self);
	return result;
}


static gboolean gee_hash_set_real_add (GeeCollection* base, gconstpointer key) {
	GeeHashSet * self;
	gboolean result;
	GeeHashSetNode** node;
	self = (GeeHashSet*) base;
	node = gee_hash_set_lookup_node (self, key);
	if ((*node) != NULL) {
		result = FALSE;
		return result;
	} else {
		guint hash_value;
		gconstpointer _tmp0_;
		hash_value = self->priv->_hash_func (key);
		_tmp0_ = NULL;
		*node = gee_hash_set_node_new ((_tmp0_ = key, ((_tmp0_ == NULL) || (self->priv->g_dup_func == NULL)) ? ((gpointer) _tmp0_) : self->priv->g_dup_func ((gpointer) _tmp0_)), hash_value);
		self->priv->_nnodes++;
		gee_hash_set_resize (self);
		self->priv->_stamp++;
		result = TRUE;
		return result;
	}
}


static gboolean gee_hash_set_real_remove (GeeCollection* base, gconstpointer key) {
	GeeHashSet * self;
	gboolean result;
	GeeHashSetNode** node;
	self = (GeeHashSet*) base;
	node = gee_hash_set_lookup_node (self, key);
	if ((*node) != NULL) {
		GeeHashSetNode* _tmp0_;
		GeeHashSetNode* next;
		gpointer _tmp1_;
		GeeHashSetNode* _tmp2_;
		_tmp0_ = NULL;
		next = (_tmp0_ = (*node)->next, (*node)->next = NULL, _tmp0_);
		_tmp1_ = NULL;
		(*node)->key = (_tmp1_ = NULL, (((*node)->key == NULL) || (self->priv->g_destroy_func == NULL)) ? NULL : ((*node)->key = (self->priv->g_destroy_func ((*node)->key), NULL)), _tmp1_);
		gee_hash_set_node_free (*node);
		_tmp2_ = NULL;
		*node = (_tmp2_ = next, next = NULL, _tmp2_);
		self->priv->_nnodes--;
		gee_hash_set_resize (self);
		self->priv->_stamp++;
		result = TRUE;
		(next == NULL) ? NULL : (next = (gee_hash_set_node_free (next), NULL));
		return result;
	}
	result = FALSE;
	return result;
}


static void gee_hash_set_real_clear (GeeCollection* base) {
	GeeHashSet * self;
	self = (GeeHashSet*) base;
	{
		gint i;
		i = 0;
		{
			gboolean _tmp0_;
			_tmp0_ = TRUE;
			while (TRUE) {
				GeeHashSetNode* _tmp1_;
				GeeHashSetNode* node;
				if (!_tmp0_) {
					i++;
				}
				_tmp0_ = FALSE;
				if (!(i < self->priv->_array_size)) {
					break;
				}
				_tmp1_ = NULL;
				node = (_tmp1_ = self->priv->_nodes[i], self->priv->_nodes[i] = NULL, _tmp1_);
				while (TRUE) {
					GeeHashSetNode* _tmp2_;
					GeeHashSetNode* next;
					gpointer _tmp3_;
					GeeHashSetNode* _tmp5_;
					GeeHashSetNode* _tmp4_;
					if (!(node != NULL)) {
						break;
					}
					_tmp2_ = NULL;
					next = (_tmp2_ = node->next, node->next = NULL, _tmp2_);
					_tmp3_ = NULL;
					node->key = (_tmp3_ = NULL, ((node->key == NULL) || (self->priv->g_destroy_func == NULL)) ? NULL : (node->key = (self->priv->g_destroy_func (node->key), NULL)), _tmp3_);
					_tmp5_ = NULL;
					_tmp4_ = NULL;
					node = (_tmp5_ = (_tmp4_ = next, next = NULL, _tmp4_), (node == NULL) ? NULL : (node = (gee_hash_set_node_free (node), NULL)), _tmp5_);
					(next == NULL) ? NULL : (next = (gee_hash_set_node_free (next), NULL));
				}
				(node == NULL) ? NULL : (node = (gee_hash_set_node_free (node), NULL));
			}
		}
	}
	self->priv->_nnodes = 0;
	gee_hash_set_resize (self);
}


static void gee_hash_set_resize (GeeHashSet* self) {
	gboolean _tmp0_;
	gboolean _tmp1_;
	g_return_if_fail (self != NULL);
	_tmp0_ = FALSE;
	_tmp1_ = FALSE;
	if (self->priv->_array_size >= (3 * self->priv->_nnodes)) {
		_tmp1_ = self->priv->_array_size >= GEE_HASH_SET_MIN_SIZE;
	} else {
		_tmp1_ = FALSE;
	}
	if (_tmp1_) {
		_tmp0_ = TRUE;
	} else {
		gboolean _tmp2_;
		_tmp2_ = FALSE;
		if ((3 * self->priv->_array_size) <= self->priv->_nnodes) {
			_tmp2_ = self->priv->_array_size < GEE_HASH_SET_MAX_SIZE;
		} else {
			_tmp2_ = FALSE;
		}
		_tmp0_ = _tmp2_;
	}
	if (_tmp0_) {
		gint new_array_size;
		GeeHashSetNode** _tmp3_;
		gint new_nodes_size;
		gint new_nodes_length1;
		GeeHashSetNode** new_nodes;
		GeeHashSetNode** _tmp17_;
		GeeHashSetNode** _tmp16_;
		new_array_size = (gint) g_spaced_primes_closest ((guint) self->priv->_nnodes);
		new_array_size = CLAMP (new_array_size, GEE_HASH_SET_MIN_SIZE, GEE_HASH_SET_MAX_SIZE);
		_tmp3_ = NULL;
		new_nodes = (_tmp3_ = g_new0 (GeeHashSetNode*, new_array_size + 1), new_nodes_length1 = new_array_size, new_nodes_size = new_nodes_length1, _tmp3_);
		{
			gint i;
			i = 0;
			{
				gboolean _tmp4_;
				_tmp4_ = TRUE;
				while (TRUE) {
					GeeHashSetNode* node;
					GeeHashSetNode* next;
					if (!_tmp4_) {
						i++;
					}
					_tmp4_ = FALSE;
					if (!(i < self->priv->_array_size)) {
						break;
					}
					node = NULL;
					next = NULL;
					{
						GeeHashSetNode* _tmp6_;
						GeeHashSetNode* _tmp5_;
						gboolean _tmp7_;
						_tmp6_ = NULL;
						_tmp5_ = NULL;
						node = (_tmp6_ = (_tmp5_ = self->priv->_nodes[i], self->priv->_nodes[i] = NULL, _tmp5_), (node == NULL) ? NULL : (node = (gee_hash_set_node_free (node), NULL)), _tmp6_);
						_tmp7_ = TRUE;
						while (TRUE) {
							GeeHashSetNode* _tmp11_;
							GeeHashSetNode* _tmp10_;
							guint hash_val;
							GeeHashSetNode* _tmp13_;
							GeeHashSetNode* _tmp12_;
							GeeHashSetNode* _tmp15_;
							GeeHashSetNode* _tmp14_;
							if (!_tmp7_) {
								GeeHashSetNode* _tmp9_;
								GeeHashSetNode* _tmp8_;
								_tmp9_ = NULL;
								_tmp8_ = NULL;
								node = (_tmp9_ = (_tmp8_ = next, next = NULL, _tmp8_), (node == NULL) ? NULL : (node = (gee_hash_set_node_free (node), NULL)), _tmp9_);
							}
							_tmp7_ = FALSE;
							if (!(node != NULL)) {
								break;
							}
							_tmp11_ = NULL;
							_tmp10_ = NULL;
							next = (_tmp11_ = (_tmp10_ = node->next, node->next = NULL, _tmp10_), (next == NULL) ? NULL : (next = (gee_hash_set_node_free (next), NULL)), _tmp11_);
							hash_val = node->key_hash % new_array_size;
							_tmp13_ = NULL;
							_tmp12_ = NULL;
							node->next = (_tmp13_ = (_tmp12_ = new_nodes[hash_val], new_nodes[hash_val] = NULL, _tmp12_), (node->next == NULL) ? NULL : (node->next = (gee_hash_set_node_free (node->next), NULL)), _tmp13_);
							_tmp15_ = NULL;
							_tmp14_ = NULL;
							new_nodes[hash_val] = (_tmp15_ = (_tmp14_ = node, node = NULL, _tmp14_), (new_nodes[hash_val] == NULL) ? NULL : (new_nodes[hash_val] = (gee_hash_set_node_free (new_nodes[hash_val]), NULL)), _tmp15_);
						}
					}
					(node == NULL) ? NULL : (node = (gee_hash_set_node_free (node), NULL));
					(next == NULL) ? NULL : (next = (gee_hash_set_node_free (next), NULL));
				}
			}
		}
		_tmp17_ = NULL;
		_tmp16_ = NULL;
		self->priv->_nodes = (_tmp17_ = (_tmp16_ = new_nodes, new_nodes = NULL, _tmp16_), self->priv->_nodes = (_vala_array_free (self->priv->_nodes, self->priv->_nodes_length1, (GDestroyNotify) gee_hash_set_node_free), NULL), self->priv->_nodes_length1 = new_nodes_length1, self->priv->_nodes_size = self->priv->_nodes_length1, _tmp17_);
		self->priv->_array_size = new_array_size;
		new_nodes = (_vala_array_free (new_nodes, new_nodes_length1, (GDestroyNotify) gee_hash_set_node_free), NULL);
	}
}


static gint gee_hash_set_real_get_size (GeeCollection* base) {
	gint result;
	GeeHashSet* self;
	self = (GeeHashSet*) base;
	result = self->priv->_nnodes;
	return result;
}


void gee_hash_set_set_hash_func (GeeHashSet* self, GHashFunc value) {
	g_return_if_fail (self != NULL);
	self->priv->_hash_func = value;
}


void gee_hash_set_set_equal_func (GeeHashSet* self, GEqualFunc value) {
	g_return_if_fail (self != NULL);
	self->priv->_equal_func = value;
}


static GeeHashSetNode* gee_hash_set_node_new (gpointer k, guint hash) {
	GeeHashSetNode* self;
	gpointer _tmp0_;
	self = g_slice_new0 (GeeHashSetNode);
	gee_hash_set_node_instance_init (self);
	_tmp0_ = NULL;
	self->key = (_tmp0_ = k, k = NULL, _tmp0_);
	self->key_hash = hash;
	return self;
}


static void gee_hash_set_node_instance_init (GeeHashSetNode * self) {
}


static void gee_hash_set_node_free (GeeHashSetNode* self) {
	(self->next == NULL) ? NULL : (self->next = (gee_hash_set_node_free (self->next), NULL));
	g_slice_free (GeeHashSetNode, self);
}


static GeeHashSetIterator* gee_hash_set_iterator_construct (GType object_type, GType g_type, GBoxedCopyFunc g_dup_func, GDestroyNotify g_destroy_func, GeeHashSet* set) {
	GeeHashSetIterator* self;
	g_return_val_if_fail (set != NULL, NULL);
	self = (GeeHashSetIterator*) g_type_create_instance (object_type);
	self->priv->g_type = g_type;
	self->priv->g_dup_func = g_dup_func;
	self->priv->g_destroy_func = g_destroy_func;
	gee_hash_set_iterator_set_set (self, set);
	return self;
}


static GeeHashSetIterator* gee_hash_set_iterator_new (GType g_type, GBoxedCopyFunc g_dup_func, GDestroyNotify g_destroy_func, GeeHashSet* set) {
	return gee_hash_set_iterator_construct (GEE_HASH_SET_TYPE_ITERATOR, g_type, g_dup_func, g_destroy_func, set);
}


static gboolean gee_hash_set_iterator_real_next (GeeIterator* base) {
	GeeHashSetIterator * self;
	gboolean result;
	self = (GeeHashSetIterator*) base;
	if (self->priv->_node != NULL) {
		self->priv->_node = self->priv->_node->next;
	}
	while (TRUE) {
		gboolean _tmp0_;
		_tmp0_ = FALSE;
		if (self->priv->_node == NULL) {
			_tmp0_ = (self->priv->_index + 1) < self->priv->_set->priv->_array_size;
		} else {
			_tmp0_ = FALSE;
		}
		if (!_tmp0_) {
			break;
		}
		self->priv->_index++;
		self->priv->_node = self->priv->_set->priv->_nodes[self->priv->_index];
	}
	result = self->priv->_node != NULL;
	return result;
}


static gpointer gee_hash_set_iterator_real_get (GeeIterator* base) {
	GeeHashSetIterator * self;
	gpointer result;
	gconstpointer _tmp0_;
	self = (GeeHashSetIterator*) base;
	g_assert (self->priv->_stamp == self->priv->_set->priv->_stamp);
	g_assert (self->priv->_node != NULL);
	_tmp0_ = NULL;
	result = (_tmp0_ = self->priv->_node->key, ((_tmp0_ == NULL) || (self->priv->g_dup_func == NULL)) ? ((gpointer) _tmp0_) : self->priv->g_dup_func ((gpointer) _tmp0_));
	return result;
}


static void gee_hash_set_iterator_set_set (GeeHashSetIterator* self, GeeHashSet* value) {
	GeeHashSet* _tmp1_;
	GeeHashSet* _tmp0_;
	g_return_if_fail (self != NULL);
	_tmp1_ = NULL;
	_tmp0_ = NULL;
	self->priv->_set = (_tmp1_ = (_tmp0_ = value, (_tmp0_ == NULL) ? NULL : gee_collection_object_ref (_tmp0_)), (self->priv->_set == NULL) ? NULL : (self->priv->_set = (gee_collection_object_unref (self->priv->_set), NULL)), _tmp1_);
	self->priv->_stamp = self->priv->_set->priv->_stamp;
}


static void gee_hash_set_iterator_class_init (GeeHashSetIteratorClass * klass) {
	gee_hash_set_iterator_parent_class = g_type_class_peek_parent (klass);
	GEE_COLLECTION_OBJECT_CLASS (klass)->finalize = gee_hash_set_iterator_finalize;
	g_type_class_add_private (klass, sizeof (GeeHashSetIteratorPrivate));
}


static void gee_hash_set_iterator_gee_iterator_interface_init (GeeIteratorIface * iface) {
	gee_hash_set_iterator_gee_iterator_parent_iface = g_type_interface_peek_parent (iface);
	iface->next = gee_hash_set_iterator_real_next;
	iface->get = gee_hash_set_iterator_real_get;
}


static void gee_hash_set_iterator_instance_init (GeeHashSetIterator * self) {
	self->priv = GEE_HASH_SET_ITERATOR_GET_PRIVATE (self);
	self->priv->_index = -1;
	self->priv->_stamp = 0;
}


static void gee_hash_set_iterator_finalize (GeeCollectionObject* obj) {
	GeeHashSetIterator * self;
	self = GEE_HASH_SET_ITERATOR (obj);
	(self->priv->_set == NULL) ? NULL : (self->priv->_set = (gee_collection_object_unref (self->priv->_set), NULL));
	GEE_COLLECTION_OBJECT_CLASS (gee_hash_set_iterator_parent_class)->finalize (obj);
}


static GType gee_hash_set_iterator_get_type (void) {
	static GType gee_hash_set_iterator_type_id = 0;
	if (gee_hash_set_iterator_type_id == 0) {
		static const GTypeInfo g_define_type_info = { sizeof (GeeHashSetIteratorClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) gee_hash_set_iterator_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (GeeHashSetIterator), 0, (GInstanceInitFunc) gee_hash_set_iterator_instance_init, NULL };
		static const GInterfaceInfo gee_iterator_info = { (GInterfaceInitFunc) gee_hash_set_iterator_gee_iterator_interface_init, (GInterfaceFinalizeFunc) NULL, NULL};
		gee_hash_set_iterator_type_id = g_type_register_static (GEE_TYPE_COLLECTION_OBJECT, "GeeHashSetIterator", &g_define_type_info, 0);
		g_type_add_interface_static (gee_hash_set_iterator_type_id, GEE_TYPE_ITERATOR, &gee_iterator_info);
	}
	return gee_hash_set_iterator_type_id;
}


static void gee_hash_set_class_init (GeeHashSetClass * klass) {
	gee_hash_set_parent_class = g_type_class_peek_parent (klass);
	GEE_COLLECTION_OBJECT_CLASS (klass)->finalize = gee_hash_set_finalize;
	g_type_class_add_private (klass, sizeof (GeeHashSetPrivate));
}


static void gee_hash_set_gee_iterable_interface_init (GeeIterableIface * iface) {
	gee_hash_set_gee_iterable_parent_iface = g_type_interface_peek_parent (iface);
	iface->get_element_type = gee_hash_set_real_get_element_type;
	iface->iterator = gee_hash_set_real_iterator;
}


static void gee_hash_set_gee_collection_interface_init (GeeCollectionIface * iface) {
	gee_hash_set_gee_collection_parent_iface = g_type_interface_peek_parent (iface);
	iface->contains = gee_hash_set_real_contains;
	iface->add = gee_hash_set_real_add;
	iface->remove = gee_hash_set_real_remove;
	iface->clear = gee_hash_set_real_clear;
	iface->get_size = gee_hash_set_real_get_size;
}


static void gee_hash_set_gee_set_interface_init (GeeSetIface * iface) {
	gee_hash_set_gee_set_parent_iface = g_type_interface_peek_parent (iface);
}


static void gee_hash_set_instance_init (GeeHashSet * self) {
	self->priv = GEE_HASH_SET_GET_PRIVATE (self);
	self->priv->_stamp = 0;
}


static void gee_hash_set_finalize (GeeCollectionObject* obj) {
	GeeHashSet * self;
	self = GEE_HASH_SET (obj);
	{
		gee_collection_clear ((GeeCollection*) self);
	}
	self->priv->_nodes = (_vala_array_free (self->priv->_nodes, self->priv->_nodes_length1, (GDestroyNotify) gee_hash_set_node_free), NULL);
	GEE_COLLECTION_OBJECT_CLASS (gee_hash_set_parent_class)->finalize (obj);
}


GType gee_hash_set_get_type (void) {
	static GType gee_hash_set_type_id = 0;
	if (gee_hash_set_type_id == 0) {
		static const GTypeInfo g_define_type_info = { sizeof (GeeHashSetClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) gee_hash_set_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (GeeHashSet), 0, (GInstanceInitFunc) gee_hash_set_instance_init, NULL };
		static const GInterfaceInfo gee_iterable_info = { (GInterfaceInitFunc) gee_hash_set_gee_iterable_interface_init, (GInterfaceFinalizeFunc) NULL, NULL};
		static const GInterfaceInfo gee_collection_info = { (GInterfaceInitFunc) gee_hash_set_gee_collection_interface_init, (GInterfaceFinalizeFunc) NULL, NULL};
		static const GInterfaceInfo gee_set_info = { (GInterfaceInitFunc) gee_hash_set_gee_set_interface_init, (GInterfaceFinalizeFunc) NULL, NULL};
		gee_hash_set_type_id = g_type_register_static (GEE_TYPE_COLLECTION_OBJECT, "GeeHashSet", &g_define_type_info, 0);
		g_type_add_interface_static (gee_hash_set_type_id, GEE_TYPE_ITERABLE, &gee_iterable_info);
		g_type_add_interface_static (gee_hash_set_type_id, GEE_TYPE_COLLECTION, &gee_collection_info);
		g_type_add_interface_static (gee_hash_set_type_id, GEE_TYPE_SET, &gee_set_info);
	}
	return gee_hash_set_type_id;
}


static void _vala_array_destroy (gpointer array, gint array_length, GDestroyNotify destroy_func) {
	if ((array != NULL) && (destroy_func != NULL)) {
		int i;
		for (i = 0; i < array_length; i = i + 1) {
			if (((gpointer*) array)[i] != NULL) {
				destroy_func (((gpointer*) array)[i]);
			}
		}
	}
}


static void _vala_array_free (gpointer array, gint array_length, GDestroyNotify destroy_func) {
	_vala_array_destroy (array, array_length, destroy_func);
	g_free (array);
}




