/* valaerrordomain.vala
 *
 * Copyright (C) 2008-2009  Jürg Billeter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 */

#include <glib.h>
#include <glib-object.h>
#include <stdlib.h>
#include <string.h>
#include <gee.h>


#define VALA_TYPE_CODE_NODE (vala_code_node_get_type ())
#define VALA_CODE_NODE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_CODE_NODE, ValaCodeNode))
#define VALA_CODE_NODE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_CODE_NODE, ValaCodeNodeClass))
#define VALA_IS_CODE_NODE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_CODE_NODE))
#define VALA_IS_CODE_NODE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_CODE_NODE))
#define VALA_CODE_NODE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_CODE_NODE, ValaCodeNodeClass))

typedef struct _ValaCodeNode ValaCodeNode;
typedef struct _ValaCodeNodeClass ValaCodeNodeClass;
typedef struct _ValaCodeNodePrivate ValaCodeNodePrivate;

#define VALA_TYPE_CODE_VISITOR (vala_code_visitor_get_type ())
#define VALA_CODE_VISITOR(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_CODE_VISITOR, ValaCodeVisitor))
#define VALA_CODE_VISITOR_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_CODE_VISITOR, ValaCodeVisitorClass))
#define VALA_IS_CODE_VISITOR(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_CODE_VISITOR))
#define VALA_IS_CODE_VISITOR_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_CODE_VISITOR))
#define VALA_CODE_VISITOR_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_CODE_VISITOR, ValaCodeVisitorClass))

typedef struct _ValaCodeVisitor ValaCodeVisitor;
typedef struct _ValaCodeVisitorClass ValaCodeVisitorClass;

#define VALA_TYPE_SEMANTIC_ANALYZER (vala_semantic_analyzer_get_type ())
#define VALA_SEMANTIC_ANALYZER(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_SEMANTIC_ANALYZER, ValaSemanticAnalyzer))
#define VALA_SEMANTIC_ANALYZER_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_SEMANTIC_ANALYZER, ValaSemanticAnalyzerClass))
#define VALA_IS_SEMANTIC_ANALYZER(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_SEMANTIC_ANALYZER))
#define VALA_IS_SEMANTIC_ANALYZER_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_SEMANTIC_ANALYZER))
#define VALA_SEMANTIC_ANALYZER_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_SEMANTIC_ANALYZER, ValaSemanticAnalyzerClass))

typedef struct _ValaSemanticAnalyzer ValaSemanticAnalyzer;
typedef struct _ValaSemanticAnalyzerClass ValaSemanticAnalyzerClass;

#define VALA_TYPE_DATA_TYPE (vala_data_type_get_type ())
#define VALA_DATA_TYPE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_DATA_TYPE, ValaDataType))
#define VALA_DATA_TYPE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_DATA_TYPE, ValaDataTypeClass))
#define VALA_IS_DATA_TYPE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_DATA_TYPE))
#define VALA_IS_DATA_TYPE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_DATA_TYPE))
#define VALA_DATA_TYPE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_DATA_TYPE, ValaDataTypeClass))

typedef struct _ValaDataType ValaDataType;
typedef struct _ValaDataTypeClass ValaDataTypeClass;

#define VALA_TYPE_EXPRESSION (vala_expression_get_type ())
#define VALA_EXPRESSION(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_EXPRESSION, ValaExpression))
#define VALA_EXPRESSION_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_EXPRESSION, ValaExpressionClass))
#define VALA_IS_EXPRESSION(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_EXPRESSION))
#define VALA_IS_EXPRESSION_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_EXPRESSION))
#define VALA_EXPRESSION_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_EXPRESSION, ValaExpressionClass))

typedef struct _ValaExpression ValaExpression;
typedef struct _ValaExpressionClass ValaExpressionClass;

#define VALA_TYPE_SYMBOL (vala_symbol_get_type ())
#define VALA_SYMBOL(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_SYMBOL, ValaSymbol))
#define VALA_SYMBOL_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_SYMBOL, ValaSymbolClass))
#define VALA_IS_SYMBOL(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_SYMBOL))
#define VALA_IS_SYMBOL_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_SYMBOL))
#define VALA_SYMBOL_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_SYMBOL, ValaSymbolClass))

typedef struct _ValaSymbol ValaSymbol;
typedef struct _ValaSymbolClass ValaSymbolClass;

#define VALA_TYPE_LOCAL_VARIABLE (vala_local_variable_get_type ())
#define VALA_LOCAL_VARIABLE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_LOCAL_VARIABLE, ValaLocalVariable))
#define VALA_LOCAL_VARIABLE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_LOCAL_VARIABLE, ValaLocalVariableClass))
#define VALA_IS_LOCAL_VARIABLE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_LOCAL_VARIABLE))
#define VALA_IS_LOCAL_VARIABLE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_LOCAL_VARIABLE))
#define VALA_LOCAL_VARIABLE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_LOCAL_VARIABLE, ValaLocalVariableClass))

typedef struct _ValaLocalVariable ValaLocalVariable;
typedef struct _ValaLocalVariableClass ValaLocalVariableClass;

#define VALA_TYPE_ATTRIBUTE (vala_attribute_get_type ())
#define VALA_ATTRIBUTE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_ATTRIBUTE, ValaAttribute))
#define VALA_ATTRIBUTE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_ATTRIBUTE, ValaAttributeClass))
#define VALA_IS_ATTRIBUTE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_ATTRIBUTE))
#define VALA_IS_ATTRIBUTE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_ATTRIBUTE))
#define VALA_ATTRIBUTE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_ATTRIBUTE, ValaAttributeClass))

typedef struct _ValaAttribute ValaAttribute;
typedef struct _ValaAttributeClass ValaAttributeClass;
typedef struct _ValaSymbolPrivate ValaSymbolPrivate;

#define VALA_TYPE_TYPESYMBOL (vala_typesymbol_get_type ())
#define VALA_TYPESYMBOL(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_TYPESYMBOL, ValaTypeSymbol))
#define VALA_TYPESYMBOL_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_TYPESYMBOL, ValaTypeSymbolClass))
#define VALA_IS_TYPESYMBOL(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_TYPESYMBOL))
#define VALA_IS_TYPESYMBOL_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_TYPESYMBOL))
#define VALA_TYPESYMBOL_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_TYPESYMBOL, ValaTypeSymbolClass))

typedef struct _ValaTypeSymbol ValaTypeSymbol;
typedef struct _ValaTypeSymbolClass ValaTypeSymbolClass;
typedef struct _ValaTypeSymbolPrivate ValaTypeSymbolPrivate;

#define VALA_TYPE_ERROR_DOMAIN (vala_error_domain_get_type ())
#define VALA_ERROR_DOMAIN(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_ERROR_DOMAIN, ValaErrorDomain))
#define VALA_ERROR_DOMAIN_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_ERROR_DOMAIN, ValaErrorDomainClass))
#define VALA_IS_ERROR_DOMAIN(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_ERROR_DOMAIN))
#define VALA_IS_ERROR_DOMAIN_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_ERROR_DOMAIN))
#define VALA_ERROR_DOMAIN_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_ERROR_DOMAIN, ValaErrorDomainClass))

typedef struct _ValaErrorDomain ValaErrorDomain;
typedef struct _ValaErrorDomainClass ValaErrorDomainClass;
typedef struct _ValaErrorDomainPrivate ValaErrorDomainPrivate;

#define VALA_TYPE_ERROR_CODE (vala_error_code_get_type ())
#define VALA_ERROR_CODE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_ERROR_CODE, ValaErrorCode))
#define VALA_ERROR_CODE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_ERROR_CODE, ValaErrorCodeClass))
#define VALA_IS_ERROR_CODE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_ERROR_CODE))
#define VALA_IS_ERROR_CODE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_ERROR_CODE))
#define VALA_ERROR_CODE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_ERROR_CODE, ValaErrorCodeClass))

typedef struct _ValaErrorCode ValaErrorCode;
typedef struct _ValaErrorCodeClass ValaErrorCodeClass;

#define VALA_TYPE_MEMBER (vala_member_get_type ())
#define VALA_MEMBER(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_MEMBER, ValaMember))
#define VALA_MEMBER_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_MEMBER, ValaMemberClass))
#define VALA_IS_MEMBER(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_MEMBER))
#define VALA_IS_MEMBER_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_MEMBER))
#define VALA_MEMBER_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_MEMBER, ValaMemberClass))

typedef struct _ValaMember ValaMember;
typedef struct _ValaMemberClass ValaMemberClass;

#define VALA_TYPE_METHOD (vala_method_get_type ())
#define VALA_METHOD(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_METHOD, ValaMethod))
#define VALA_METHOD_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_METHOD, ValaMethodClass))
#define VALA_IS_METHOD(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_METHOD))
#define VALA_IS_METHOD_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_METHOD))
#define VALA_METHOD_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_METHOD, ValaMethodClass))

typedef struct _ValaMethod ValaMethod;
typedef struct _ValaMethodClass ValaMethodClass;
#define _gee_collection_object_unref0(var) ((var == NULL) ? NULL : (var = (gee_collection_object_unref (var), NULL)))
#define _g_free0(var) (var = (g_free (var), NULL))

#define VALA_TYPE_SOURCE_REFERENCE (vala_source_reference_get_type ())
#define VALA_SOURCE_REFERENCE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_SOURCE_REFERENCE, ValaSourceReference))
#define VALA_SOURCE_REFERENCE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_SOURCE_REFERENCE, ValaSourceReferenceClass))
#define VALA_IS_SOURCE_REFERENCE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_SOURCE_REFERENCE))
#define VALA_IS_SOURCE_REFERENCE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_SOURCE_REFERENCE))
#define VALA_SOURCE_REFERENCE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_SOURCE_REFERENCE, ValaSourceReferenceClass))

typedef struct _ValaSourceReference ValaSourceReference;
typedef struct _ValaSourceReferenceClass ValaSourceReferenceClass;

#define VALA_TYPE_COMMENT (vala_comment_get_type ())
#define VALA_COMMENT(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_COMMENT, ValaComment))
#define VALA_COMMENT_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_COMMENT, ValaCommentClass))
#define VALA_IS_COMMENT(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_COMMENT))
#define VALA_IS_COMMENT_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_COMMENT))
#define VALA_COMMENT_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_COMMENT, ValaCommentClass))

typedef struct _ValaComment ValaComment;
typedef struct _ValaCommentClass ValaCommentClass;

#define VALA_TYPE_SCOPE (vala_scope_get_type ())
#define VALA_SCOPE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_SCOPE, ValaScope))
#define VALA_SCOPE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_SCOPE, ValaScopeClass))
#define VALA_IS_SCOPE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_SCOPE))
#define VALA_IS_SCOPE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_SCOPE))
#define VALA_SCOPE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_SCOPE, ValaScopeClass))

typedef struct _ValaScope ValaScope;
typedef struct _ValaScopeClass ValaScopeClass;

#define VALA_TYPE_CREATION_METHOD (vala_creation_method_get_type ())
#define VALA_CREATION_METHOD(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_CREATION_METHOD, ValaCreationMethod))
#define VALA_CREATION_METHOD_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_CREATION_METHOD, ValaCreationMethodClass))
#define VALA_IS_CREATION_METHOD(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_CREATION_METHOD))
#define VALA_IS_CREATION_METHOD_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_CREATION_METHOD))
#define VALA_CREATION_METHOD_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_CREATION_METHOD, ValaCreationMethodClass))

typedef struct _ValaCreationMethod ValaCreationMethod;
typedef struct _ValaCreationMethodClass ValaCreationMethodClass;

#define TYPE_MEMBER_BINDING (member_binding_get_type ())

#define VALA_TYPE_REFERENCE_TYPE (vala_reference_type_get_type ())
#define VALA_REFERENCE_TYPE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_REFERENCE_TYPE, ValaReferenceType))
#define VALA_REFERENCE_TYPE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_REFERENCE_TYPE, ValaReferenceTypeClass))
#define VALA_IS_REFERENCE_TYPE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_REFERENCE_TYPE))
#define VALA_IS_REFERENCE_TYPE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_REFERENCE_TYPE))
#define VALA_REFERENCE_TYPE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_REFERENCE_TYPE, ValaReferenceTypeClass))

typedef struct _ValaReferenceType ValaReferenceType;
typedef struct _ValaReferenceTypeClass ValaReferenceTypeClass;

#define VALA_TYPE_ERROR_TYPE (vala_error_type_get_type ())
#define VALA_ERROR_TYPE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_ERROR_TYPE, ValaErrorType))
#define VALA_ERROR_TYPE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_ERROR_TYPE, ValaErrorTypeClass))
#define VALA_IS_ERROR_TYPE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_ERROR_TYPE))
#define VALA_IS_ERROR_TYPE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_ERROR_TYPE))
#define VALA_ERROR_TYPE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_ERROR_TYPE, ValaErrorTypeClass))

typedef struct _ValaErrorType ValaErrorType;
typedef struct _ValaErrorTypeClass ValaErrorTypeClass;

#define VALA_TYPE_FORMAL_PARAMETER (vala_formal_parameter_get_type ())
#define VALA_FORMAL_PARAMETER(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_FORMAL_PARAMETER, ValaFormalParameter))
#define VALA_FORMAL_PARAMETER_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_FORMAL_PARAMETER, ValaFormalParameterClass))
#define VALA_IS_FORMAL_PARAMETER(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_FORMAL_PARAMETER))
#define VALA_IS_FORMAL_PARAMETER_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_FORMAL_PARAMETER))
#define VALA_FORMAL_PARAMETER_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_FORMAL_PARAMETER, ValaFormalParameterClass))

typedef struct _ValaFormalParameter ValaFormalParameter;
typedef struct _ValaFormalParameterClass ValaFormalParameterClass;
#define _vala_code_node_unref0(var) ((var == NULL) ? NULL : (var = (vala_code_node_unref (var), NULL)))

struct _ValaCodeNode {
	GTypeInstance parent_instance;
	volatile int ref_count;
	ValaCodeNodePrivate * priv;
	GList* attributes;
};

struct _ValaCodeNodeClass {
	GTypeClass parent_class;
	void (*finalize) (ValaCodeNode *self);
	void (*accept) (ValaCodeNode* self, ValaCodeVisitor* visitor);
	void (*accept_children) (ValaCodeNode* self, ValaCodeVisitor* visitor);
	gboolean (*check) (ValaCodeNode* self, ValaSemanticAnalyzer* analyzer);
	void (*replace_type) (ValaCodeNode* self, ValaDataType* old_type, ValaDataType* new_type);
	void (*replace_expression) (ValaCodeNode* self, ValaExpression* old_node, ValaExpression* new_node);
	char* (*to_string) (ValaCodeNode* self);
	void (*get_defined_variables) (ValaCodeNode* self, GeeCollection* collection);
	void (*get_used_variables) (ValaCodeNode* self, GeeCollection* collection);
};

struct _ValaSymbol {
	ValaCodeNode parent_instance;
	ValaSymbolPrivate * priv;
};

struct _ValaSymbolClass {
	ValaCodeNodeClass parent_class;
	char* (*get_cprefix) (ValaSymbol* self);
	char* (*get_lower_case_cname) (ValaSymbol* self, const char* infix);
	char* (*get_lower_case_cprefix) (ValaSymbol* self);
	GeeList* (*get_cheader_filenames) (ValaSymbol* self);
	gboolean (*is_instance_member) (ValaSymbol* self);
	gboolean (*is_class_member) (ValaSymbol* self);
};

struct _ValaTypeSymbol {
	ValaSymbol parent_instance;
	ValaTypeSymbolPrivate * priv;
};

struct _ValaTypeSymbolClass {
	ValaSymbolClass parent_class;
	char* (*get_cname) (ValaTypeSymbol* self, gboolean const_type);
	gboolean (*is_reference_type) (ValaTypeSymbol* self);
	char* (*get_dup_function) (ValaTypeSymbol* self);
	char* (*get_free_function) (ValaTypeSymbol* self);
	char* (*get_copy_function) (ValaTypeSymbol* self);
	char* (*get_destroy_function) (ValaTypeSymbol* self);
	gboolean (*is_reference_counting) (ValaTypeSymbol* self);
	char* (*get_ref_function) (ValaTypeSymbol* self);
	char* (*get_unref_function) (ValaTypeSymbol* self);
	char* (*get_ref_sink_function) (ValaTypeSymbol* self);
	char* (*get_type_id) (ValaTypeSymbol* self);
	char* (*get_marshaller_type_name) (ValaTypeSymbol* self);
	char* (*get_param_spec_function) (ValaTypeSymbol* self);
	char* (*get_get_value_function) (ValaTypeSymbol* self);
	char* (*get_set_value_function) (ValaTypeSymbol* self);
	char* (*get_upper_case_cname) (ValaTypeSymbol* self, const char* infix);
	char* (*get_default_value) (ValaTypeSymbol* self);
	gboolean (*is_subtype_of) (ValaTypeSymbol* self, ValaTypeSymbol* t);
	gint (*get_type_parameter_index) (ValaTypeSymbol* self, const char* name);
	char* (*get_type_signature) (ValaTypeSymbol* self);
};

struct _ValaErrorDomain {
	ValaTypeSymbol parent_instance;
	ValaErrorDomainPrivate * priv;
};

struct _ValaErrorDomainClass {
	ValaTypeSymbolClass parent_class;
};

struct _ValaErrorDomainPrivate {
	GeeList* codes;
	GeeList* methods;
	char* cname;
	char* cprefix;
	char* lower_case_cprefix;
	char* lower_case_csuffix;
};

typedef enum  {
	MEMBER_BINDING_INSTANCE,
	MEMBER_BINDING_CLASS,
	MEMBER_BINDING_STATIC
} MemberBinding;


static gpointer vala_error_domain_parent_class = NULL;

gpointer vala_code_node_ref (gpointer instance);
void vala_code_node_unref (gpointer instance);
GParamSpec* vala_param_spec_code_node (const gchar* name, const gchar* nick, const gchar* blurb, GType object_type, GParamFlags flags);
void vala_value_set_code_node (GValue* value, gpointer v_object);
gpointer vala_value_get_code_node (const GValue* value);
GType vala_code_node_get_type (void);
gpointer vala_code_visitor_ref (gpointer instance);
void vala_code_visitor_unref (gpointer instance);
GParamSpec* vala_param_spec_code_visitor (const gchar* name, const gchar* nick, const gchar* blurb, GType object_type, GParamFlags flags);
void vala_value_set_code_visitor (GValue* value, gpointer v_object);
gpointer vala_value_get_code_visitor (const GValue* value);
GType vala_code_visitor_get_type (void);
GType vala_semantic_analyzer_get_type (void);
GType vala_data_type_get_type (void);
GType vala_expression_get_type (void);
GType vala_symbol_get_type (void);
GType vala_local_variable_get_type (void);
GType vala_attribute_get_type (void);
GType vala_typesymbol_get_type (void);
GType vala_error_domain_get_type (void);
GType vala_error_code_get_type (void);
GType vala_member_get_type (void);
GType vala_method_get_type (void);
#define VALA_ERROR_DOMAIN_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_ERROR_DOMAIN, ValaErrorDomainPrivate))
enum  {
	VALA_ERROR_DOMAIN_DUMMY_PROPERTY
};
gpointer vala_source_reference_ref (gpointer instance);
void vala_source_reference_unref (gpointer instance);
GParamSpec* vala_param_spec_source_reference (const gchar* name, const gchar* nick, const gchar* blurb, GType object_type, GParamFlags flags);
void vala_value_set_source_reference (GValue* value, gpointer v_object);
gpointer vala_value_get_source_reference (const GValue* value);
GType vala_source_reference_get_type (void);
gpointer vala_comment_ref (gpointer instance);
void vala_comment_unref (gpointer instance);
GParamSpec* vala_param_spec_comment (const gchar* name, const gchar* nick, const gchar* blurb, GType object_type, GParamFlags flags);
void vala_value_set_comment (GValue* value, gpointer v_object);
gpointer vala_value_get_comment (const GValue* value);
GType vala_comment_get_type (void);
ValaTypeSymbol* vala_typesymbol_construct (GType object_type, const char* name, ValaSourceReference* source_reference, ValaComment* comment);
ValaErrorDomain* vala_error_domain_new (const char* name, ValaSourceReference* source_reference, ValaComment* comment);
ValaErrorDomain* vala_error_domain_construct (GType object_type, const char* name, ValaSourceReference* source_reference, ValaComment* comment);
gpointer vala_scope_ref (gpointer instance);
void vala_scope_unref (gpointer instance);
GParamSpec* vala_param_spec_scope (const gchar* name, const gchar* nick, const gchar* blurb, GType object_type, GParamFlags flags);
void vala_value_set_scope (GValue* value, gpointer v_object);
gpointer vala_value_get_scope (const GValue* value);
GType vala_scope_get_type (void);
ValaScope* vala_symbol_get_scope (ValaSymbol* self);
void vala_scope_add (ValaScope* self, const char* name, ValaSymbol* sym);
const char* vala_symbol_get_name (ValaSymbol* self);
void vala_error_domain_add_code (ValaErrorDomain* self, ValaErrorCode* ecode);
GType vala_creation_method_get_type (void);
void vala_report_error (ValaSourceReference* source, const char* message);
ValaSourceReference* vala_code_node_get_source_reference (ValaCodeNode* self);
void vala_code_node_set_error (ValaCodeNode* self, gboolean value);
GType member_binding_get_type (void);
MemberBinding vala_method_get_binding (ValaMethod* self);
ValaErrorType* vala_error_type_new (ValaErrorDomain* error_domain, ValaErrorCode* error_code, ValaSourceReference* source_reference);
ValaErrorType* vala_error_type_construct (GType object_type, ValaErrorDomain* error_domain, ValaErrorCode* error_code, ValaSourceReference* source_reference);
GType vala_reference_type_get_type (void);
GType vala_error_type_get_type (void);
ValaFormalParameter* vala_formal_parameter_new (const char* name, ValaDataType* parameter_type, ValaSourceReference* source_reference);
ValaFormalParameter* vala_formal_parameter_construct (GType object_type, const char* name, ValaDataType* parameter_type, ValaSourceReference* source_reference);
GType vala_formal_parameter_get_type (void);
void vala_method_set_this_parameter (ValaMethod* self, ValaFormalParameter* value);
ValaFormalParameter* vala_method_get_this_parameter (ValaMethod* self);
void vala_error_domain_add_method (ValaErrorDomain* self, ValaMethod* m);
GeeList* vala_error_domain_get_codes (ValaErrorDomain* self);
GeeList* vala_error_domain_get_methods (ValaErrorDomain* self);
void vala_code_visitor_visit_error_domain (ValaCodeVisitor* self, ValaErrorDomain* edomain);
static void vala_error_domain_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor);
void vala_code_node_accept (ValaCodeNode* self, ValaCodeVisitor* visitor);
static void vala_error_domain_real_accept_children (ValaCodeNode* base, ValaCodeVisitor* visitor);
ValaSymbol* vala_symbol_get_parent_symbol (ValaSymbol* self);
char* vala_symbol_get_cprefix (ValaSymbol* self);
static char* vala_error_domain_real_get_cname (ValaTypeSymbol* base, gboolean const_type);
char* vala_symbol_get_lower_case_cname (ValaSymbol* self, const char* infix);
static char* vala_error_domain_real_get_lower_case_cprefix (ValaSymbol* base);
char* vala_symbol_camel_case_to_lower_case (const char* camel_case);
static char* vala_error_domain_get_lower_case_csuffix (ValaErrorDomain* self);
char* vala_symbol_get_lower_case_cprefix (ValaSymbol* self);
static char* vala_error_domain_real_get_lower_case_cname (ValaSymbol* base, const char* infix);
static char* vala_error_domain_real_get_upper_case_cname (ValaTypeSymbol* base, const char* infix);
static gboolean vala_error_domain_real_is_reference_type (ValaTypeSymbol* base);
static void vala_error_domain_set_cname (ValaErrorDomain* self, const char* cname);
char* vala_typesymbol_get_upper_case_cname (ValaTypeSymbol* self, const char* infix);
static char* vala_error_domain_real_get_cprefix (ValaSymbol* base);
void vala_error_domain_set_cprefix (ValaErrorDomain* self, const char* cprefix);
gboolean vala_attribute_has_argument (ValaAttribute* self, const char* name);
char* vala_attribute_get_string (ValaAttribute* self, const char* name);
void vala_typesymbol_add_cheader_filename (ValaTypeSymbol* self, const char* filename);
static void vala_error_domain_process_ccode_attribute (ValaErrorDomain* self, ValaAttribute* a);
const char* vala_attribute_get_name (ValaAttribute* self);
void vala_error_domain_process_attributes (ValaErrorDomain* self);
static char* vala_error_domain_real_get_type_id (ValaTypeSymbol* base);
static char* vala_error_domain_real_get_marshaller_type_name (ValaTypeSymbol* base);
static char* vala_error_domain_real_get_get_value_function (ValaTypeSymbol* base);
static char* vala_error_domain_real_get_set_value_function (ValaTypeSymbol* base);
gboolean vala_code_node_get_checked (ValaCodeNode* self);
gboolean vala_code_node_get_error (ValaCodeNode* self);
void vala_code_node_set_checked (ValaCodeNode* self, gboolean value);
gboolean vala_code_node_check (ValaCodeNode* self, ValaSemanticAnalyzer* analyzer);
static gboolean vala_error_domain_real_check (ValaCodeNode* base, ValaSemanticAnalyzer* analyzer);
static void vala_error_domain_finalize (ValaCodeNode* obj);
static void _vala_array_destroy (gpointer array, gint array_length, GDestroyNotify destroy_func);
static void _vala_array_free (gpointer array, gint array_length, GDestroyNotify destroy_func);
static gint _vala_array_length (gpointer array);
static int _vala_strcmp0 (const char * str1, const char * str2);



ValaErrorDomain* vala_error_domain_construct (GType object_type, const char* name, ValaSourceReference* source_reference, ValaComment* comment) {
	ValaErrorDomain* self;
	g_return_val_if_fail (name != NULL, NULL);
	self = (ValaErrorDomain*) vala_typesymbol_construct (object_type, name, source_reference, comment);
	return self;
}


ValaErrorDomain* vala_error_domain_new (const char* name, ValaSourceReference* source_reference, ValaComment* comment) {
	return vala_error_domain_construct (VALA_TYPE_ERROR_DOMAIN, name, source_reference, comment);
}


void vala_error_domain_add_code (ValaErrorDomain* self, ValaErrorCode* ecode) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (ecode != NULL);
	gee_collection_add ((GeeCollection*) self->priv->codes, ecode);
	vala_scope_add (vala_symbol_get_scope ((ValaSymbol*) self), vala_symbol_get_name ((ValaSymbol*) ecode), (ValaSymbol*) ecode);
}


void vala_error_domain_add_method (ValaErrorDomain* self, ValaMethod* m) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (m != NULL);
	if (VALA_IS_CREATION_METHOD (m)) {
		vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) m), "construction methods may only be declared within classes and structs");
		vala_code_node_set_error ((ValaCodeNode*) m, TRUE);
		return;
	}
	if (vala_method_get_binding (m) == MEMBER_BINDING_INSTANCE) {
		ValaFormalParameter* _tmp1_;
		ValaErrorType* _tmp0_;
		vala_method_set_this_parameter (m, _tmp1_ = vala_formal_parameter_new ("this", (ValaDataType*) (_tmp0_ = vala_error_type_new (self, NULL, NULL)), NULL));
		_vala_code_node_unref0 (_tmp1_);
		_vala_code_node_unref0 (_tmp0_);
		vala_scope_add (vala_symbol_get_scope ((ValaSymbol*) m), vala_symbol_get_name ((ValaSymbol*) vala_method_get_this_parameter (m)), (ValaSymbol*) vala_method_get_this_parameter (m));
	}
	gee_collection_add ((GeeCollection*) self->priv->methods, m);
	vala_scope_add (vala_symbol_get_scope ((ValaSymbol*) self), vala_symbol_get_name ((ValaSymbol*) m), (ValaSymbol*) m);
}


GeeList* vala_error_domain_get_codes (ValaErrorDomain* self) {
	GeeList* result;
	g_return_val_if_fail (self != NULL, NULL);
	result = (GeeList*) gee_read_only_list_new (VALA_TYPE_ERROR_CODE, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, self->priv->codes);
	return result;
}


GeeList* vala_error_domain_get_methods (ValaErrorDomain* self) {
	GeeList* result;
	g_return_val_if_fail (self != NULL, NULL);
	result = (GeeList*) gee_read_only_list_new (VALA_TYPE_METHOD, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, self->priv->methods);
	return result;
}


static void vala_error_domain_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor) {
	ValaErrorDomain * self;
	self = (ValaErrorDomain*) base;
	g_return_if_fail (visitor != NULL);
	vala_code_visitor_visit_error_domain (visitor, self);
}


static void vala_error_domain_real_accept_children (ValaCodeNode* base, ValaCodeVisitor* visitor) {
	ValaErrorDomain * self;
	self = (ValaErrorDomain*) base;
	g_return_if_fail (visitor != NULL);
	{
		GeeIterator* _ecode_it;
		_ecode_it = gee_iterable_iterator ((GeeIterable*) self->priv->codes);
		while (TRUE) {
			ValaErrorCode* ecode;
			if (!gee_iterator_next (_ecode_it)) {
				break;
			}
			ecode = (ValaErrorCode*) gee_iterator_get (_ecode_it);
			vala_code_node_accept ((ValaCodeNode*) ecode, visitor);
			_vala_code_node_unref0 (ecode);
		}
		_gee_collection_object_unref0 (_ecode_it);
	}
	{
		GeeIterator* _m_it;
		_m_it = gee_iterable_iterator ((GeeIterable*) self->priv->methods);
		while (TRUE) {
			ValaMethod* m;
			if (!gee_iterator_next (_m_it)) {
				break;
			}
			m = (ValaMethod*) gee_iterator_get (_m_it);
			vala_code_node_accept ((ValaCodeNode*) m, visitor);
			_vala_code_node_unref0 (m);
		}
		_gee_collection_object_unref0 (_m_it);
	}
}


static char* vala_error_domain_real_get_cname (ValaTypeSymbol* base, gboolean const_type) {
	ValaErrorDomain * self;
	char* result;
	self = (ValaErrorDomain*) base;
	if (self->priv->cname == NULL) {
		char* _tmp1_;
		char* _tmp0_;
		self->priv->cname = (_tmp1_ = g_strdup_printf ("%s%s", _tmp0_ = vala_symbol_get_cprefix (vala_symbol_get_parent_symbol ((ValaSymbol*) self)), vala_symbol_get_name ((ValaSymbol*) self)), _g_free0 (self->priv->cname), _tmp1_);
		_g_free0 (_tmp0_);
	}
	result = g_strdup (self->priv->cname);
	return result;
}


static char* vala_error_domain_real_get_lower_case_cprefix (ValaSymbol* base) {
	ValaErrorDomain * self;
	char* result;
	self = (ValaErrorDomain*) base;
	if (self->priv->lower_case_cprefix == NULL) {
		char* _tmp1_;
		char* _tmp0_;
		self->priv->lower_case_cprefix = (_tmp1_ = g_strdup_printf ("%s_", _tmp0_ = vala_symbol_get_lower_case_cname ((ValaSymbol*) self, NULL)), _g_free0 (self->priv->lower_case_cprefix), _tmp1_);
		_g_free0 (_tmp0_);
	}
	result = g_strdup (self->priv->lower_case_cprefix);
	return result;
}


static char* vala_error_domain_get_lower_case_csuffix (ValaErrorDomain* self) {
	char* result;
	g_return_val_if_fail (self != NULL, NULL);
	if (self->priv->lower_case_csuffix == NULL) {
		char* _tmp0_;
		self->priv->lower_case_csuffix = (_tmp0_ = vala_symbol_camel_case_to_lower_case (vala_symbol_get_name ((ValaSymbol*) self)), _g_free0 (self->priv->lower_case_csuffix), _tmp0_);
	}
	result = g_strdup (self->priv->lower_case_csuffix);
	return result;
}


static char* vala_error_domain_real_get_lower_case_cname (ValaSymbol* base, const char* infix) {
	ValaErrorDomain * self;
	char* result;
	char* _tmp1_;
	char* _tmp0_;
	char* _tmp2_;
	self = (ValaErrorDomain*) base;
	if (infix == NULL) {
		infix = "";
	}
	result = (_tmp2_ = g_strdup_printf ("%s%s%s", _tmp0_ = vala_symbol_get_lower_case_cprefix (vala_symbol_get_parent_symbol ((ValaSymbol*) self)), infix, _tmp1_ = vala_error_domain_get_lower_case_csuffix (self)), _g_free0 (_tmp1_), _g_free0 (_tmp0_), _tmp2_);
	return result;
}


static char* vala_error_domain_real_get_upper_case_cname (ValaTypeSymbol* base, const char* infix) {
	ValaErrorDomain * self;
	char* result;
	char* _tmp0_;
	char* _tmp1_;
	self = (ValaErrorDomain*) base;
	result = (_tmp1_ = g_utf8_strup (_tmp0_ = vala_symbol_get_lower_case_cname ((ValaSymbol*) self, infix), -1), _g_free0 (_tmp0_), _tmp1_);
	return result;
}


static gboolean vala_error_domain_real_is_reference_type (ValaTypeSymbol* base) {
	ValaErrorDomain * self;
	gboolean result;
	self = (ValaErrorDomain*) base;
	result = FALSE;
	return result;
}


static void vala_error_domain_set_cname (ValaErrorDomain* self, const char* cname) {
	char* _tmp0_;
	g_return_if_fail (self != NULL);
	g_return_if_fail (cname != NULL);
	self->priv->cname = (_tmp0_ = g_strdup (cname), _g_free0 (self->priv->cname), _tmp0_);
}


static char* vala_error_domain_real_get_cprefix (ValaSymbol* base) {
	ValaErrorDomain * self;
	char* result;
	self = (ValaErrorDomain*) base;
	if (self->priv->cprefix == NULL) {
		char* _tmp1_;
		char* _tmp0_;
		self->priv->cprefix = (_tmp1_ = g_strdup_printf ("%s_", _tmp0_ = vala_typesymbol_get_upper_case_cname ((ValaTypeSymbol*) self, NULL)), _g_free0 (self->priv->cprefix), _tmp1_);
		_g_free0 (_tmp0_);
	}
	result = g_strdup (self->priv->cprefix);
	return result;
}


void vala_error_domain_set_cprefix (ValaErrorDomain* self, const char* cprefix) {
	char* _tmp0_;
	g_return_if_fail (self != NULL);
	g_return_if_fail (cprefix != NULL);
	self->priv->cprefix = (_tmp0_ = g_strdup (cprefix), _g_free0 (self->priv->cprefix), _tmp0_);
}


static void vala_error_domain_process_ccode_attribute (ValaErrorDomain* self, ValaAttribute* a) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (a != NULL);
	if (vala_attribute_has_argument (a, "cname")) {
		char* _tmp0_;
		vala_error_domain_set_cname (self, _tmp0_ = vala_attribute_get_string (a, "cname"));
		_g_free0 (_tmp0_);
	}
	if (vala_attribute_has_argument (a, "cprefix")) {
		char* _tmp1_;
		vala_error_domain_set_cprefix (self, _tmp1_ = vala_attribute_get_string (a, "cprefix"));
		_g_free0 (_tmp1_);
	}
	if (vala_attribute_has_argument (a, "lower_case_csuffix")) {
		char* _tmp2_;
		self->priv->lower_case_csuffix = (_tmp2_ = vala_attribute_get_string (a, "lower_case_csuffix"), _g_free0 (self->priv->lower_case_csuffix), _tmp2_);
	}
	if (vala_attribute_has_argument (a, "cheader_filename")) {
		char* val;
		val = vala_attribute_get_string (a, "cheader_filename");
		{
			char** _tmp3_;
			char** filename_collection;
			int filename_collection_length1;
			int filename_it;
			filename_collection = _tmp3_ = g_strsplit (val, ",", 0);
			filename_collection_length1 = _vala_array_length (_tmp3_);
			for (filename_it = 0; filename_it < _vala_array_length (_tmp3_); filename_it = filename_it + 1) {
				char* filename;
				filename = g_strdup (filename_collection[filename_it]);
				{
					vala_typesymbol_add_cheader_filename ((ValaTypeSymbol*) self, filename);
					_g_free0 (filename);
				}
			}
			filename_collection = (_vala_array_free (filename_collection, filename_collection_length1, (GDestroyNotify) g_free), NULL);
		}
		_g_free0 (val);
	}
}


static gpointer _vala_code_node_ref0 (gpointer self) {
	return self ? vala_code_node_ref (self) : NULL;
}


void vala_error_domain_process_attributes (ValaErrorDomain* self) {
	g_return_if_fail (self != NULL);
	{
		GList* a_collection;
		GList* a_it;
		a_collection = ((ValaCodeNode*) self)->attributes;
		for (a_it = a_collection; a_it != NULL; a_it = a_it->next) {
			ValaAttribute* a;
			a = _vala_code_node_ref0 ((ValaAttribute*) a_it->data);
			{
				if (_vala_strcmp0 (vala_attribute_get_name (a), "CCode") == 0) {
					vala_error_domain_process_ccode_attribute (self, a);
				}
				_vala_code_node_unref0 (a);
			}
		}
	}
}


static char* vala_error_domain_real_get_type_id (ValaTypeSymbol* base) {
	ValaErrorDomain * self;
	char* result;
	self = (ValaErrorDomain*) base;
	result = g_strdup ("G_TYPE_POINTER");
	return result;
}


static char* vala_error_domain_real_get_marshaller_type_name (ValaTypeSymbol* base) {
	ValaErrorDomain * self;
	char* result;
	self = (ValaErrorDomain*) base;
	result = g_strdup ("POINTER");
	return result;
}


static char* vala_error_domain_real_get_get_value_function (ValaTypeSymbol* base) {
	ValaErrorDomain * self;
	char* result;
	self = (ValaErrorDomain*) base;
	result = g_strdup ("g_value_get_pointer");
	return result;
}


static char* vala_error_domain_real_get_set_value_function (ValaTypeSymbol* base) {
	ValaErrorDomain * self;
	char* result;
	self = (ValaErrorDomain*) base;
	result = g_strdup ("g_value_set_pointer");
	return result;
}


static gboolean vala_error_domain_real_check (ValaCodeNode* base, ValaSemanticAnalyzer* analyzer) {
	ValaErrorDomain * self;
	gboolean result;
	self = (ValaErrorDomain*) base;
	g_return_val_if_fail (analyzer != NULL, FALSE);
	if (vala_code_node_get_checked ((ValaCodeNode*) self)) {
		result = !vala_code_node_get_error ((ValaCodeNode*) self);
		return result;
	}
	vala_code_node_set_checked ((ValaCodeNode*) self, TRUE);
	vala_error_domain_process_attributes (self);
	{
		GeeIterator* _ecode_it;
		_ecode_it = gee_iterable_iterator ((GeeIterable*) self->priv->codes);
		while (TRUE) {
			ValaErrorCode* ecode;
			if (!gee_iterator_next (_ecode_it)) {
				break;
			}
			ecode = (ValaErrorCode*) gee_iterator_get (_ecode_it);
			vala_code_node_check ((ValaCodeNode*) ecode, analyzer);
			_vala_code_node_unref0 (ecode);
		}
		_gee_collection_object_unref0 (_ecode_it);
	}
	{
		GeeIterator* _m_it;
		_m_it = gee_iterable_iterator ((GeeIterable*) self->priv->methods);
		while (TRUE) {
			ValaMethod* m;
			if (!gee_iterator_next (_m_it)) {
				break;
			}
			m = (ValaMethod*) gee_iterator_get (_m_it);
			vala_code_node_check ((ValaCodeNode*) m, analyzer);
			_vala_code_node_unref0 (m);
		}
		_gee_collection_object_unref0 (_m_it);
	}
	result = !vala_code_node_get_error ((ValaCodeNode*) self);
	return result;
}


static void vala_error_domain_class_init (ValaErrorDomainClass * klass) {
	vala_error_domain_parent_class = g_type_class_peek_parent (klass);
	VALA_CODE_NODE_CLASS (klass)->finalize = vala_error_domain_finalize;
	g_type_class_add_private (klass, sizeof (ValaErrorDomainPrivate));
	VALA_CODE_NODE_CLASS (klass)->accept = vala_error_domain_real_accept;
	VALA_CODE_NODE_CLASS (klass)->accept_children = vala_error_domain_real_accept_children;
	VALA_TYPESYMBOL_CLASS (klass)->get_cname = vala_error_domain_real_get_cname;
	VALA_SYMBOL_CLASS (klass)->get_lower_case_cprefix = vala_error_domain_real_get_lower_case_cprefix;
	VALA_SYMBOL_CLASS (klass)->get_lower_case_cname = vala_error_domain_real_get_lower_case_cname;
	VALA_TYPESYMBOL_CLASS (klass)->get_upper_case_cname = vala_error_domain_real_get_upper_case_cname;
	VALA_TYPESYMBOL_CLASS (klass)->is_reference_type = vala_error_domain_real_is_reference_type;
	VALA_SYMBOL_CLASS (klass)->get_cprefix = vala_error_domain_real_get_cprefix;
	VALA_TYPESYMBOL_CLASS (klass)->get_type_id = vala_error_domain_real_get_type_id;
	VALA_TYPESYMBOL_CLASS (klass)->get_marshaller_type_name = vala_error_domain_real_get_marshaller_type_name;
	VALA_TYPESYMBOL_CLASS (klass)->get_get_value_function = vala_error_domain_real_get_get_value_function;
	VALA_TYPESYMBOL_CLASS (klass)->get_set_value_function = vala_error_domain_real_get_set_value_function;
	VALA_CODE_NODE_CLASS (klass)->check = vala_error_domain_real_check;
}


static void vala_error_domain_instance_init (ValaErrorDomain * self) {
	self->priv = VALA_ERROR_DOMAIN_GET_PRIVATE (self);
	self->priv->codes = (GeeList*) gee_array_list_new (VALA_TYPE_ERROR_CODE, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, g_direct_equal);
	self->priv->methods = (GeeList*) gee_array_list_new (VALA_TYPE_METHOD, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, g_direct_equal);
}


static void vala_error_domain_finalize (ValaCodeNode* obj) {
	ValaErrorDomain * self;
	self = VALA_ERROR_DOMAIN (obj);
	_gee_collection_object_unref0 (self->priv->codes);
	_gee_collection_object_unref0 (self->priv->methods);
	_g_free0 (self->priv->cname);
	_g_free0 (self->priv->cprefix);
	_g_free0 (self->priv->lower_case_cprefix);
	_g_free0 (self->priv->lower_case_csuffix);
	VALA_CODE_NODE_CLASS (vala_error_domain_parent_class)->finalize (obj);
}


GType vala_error_domain_get_type (void) {
	static GType vala_error_domain_type_id = 0;
	if (vala_error_domain_type_id == 0) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaErrorDomainClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_error_domain_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaErrorDomain), 0, (GInstanceInitFunc) vala_error_domain_instance_init, NULL };
		vala_error_domain_type_id = g_type_register_static (VALA_TYPE_TYPESYMBOL, "ValaErrorDomain", &g_define_type_info, 0);
	}
	return vala_error_domain_type_id;
}


static void _vala_array_destroy (gpointer array, gint array_length, GDestroyNotify destroy_func) {
	if ((array != NULL) && (destroy_func != NULL)) {
		int i;
		for (i = 0; i < array_length; i = i + 1) {
			if (((gpointer*) array)[i] != NULL) {
				destroy_func (((gpointer*) array)[i]);
			}
		}
	}
}


static void _vala_array_free (gpointer array, gint array_length, GDestroyNotify destroy_func) {
	_vala_array_destroy (array, array_length, destroy_func);
	g_free (array);
}


static gint _vala_array_length (gpointer array) {
	int length;
	length = 0;
	if (array) {
		while (((gpointer*) array)[length]) {
			length++;
		}
	}
	return length;
}


static int _vala_strcmp0 (const char * str1, const char * str2) {
	if (str1 == NULL) {
		return -(str1 != str2);
	}
	if (str2 == NULL) {
		return str1 != str2;
	}
	return strcmp (str1, str2);
}




