/* X-Chat
 * Copyright (C) 1998 Peter Zelezny.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 */

#include <stdio.h>
#include <unistd.h>
#include <string.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/time.h>
#include <fcntl.h>
#include <errno.h>
#include <dirent.h>
#include "util.h"
#ifndef SKIPCONFIG
#include "../myconfig.h"
#endif
#ifdef SOCKS
#include <socks.h>
#endif

extern char *g_get_home_dir (void);


#ifdef MEMORY_DEBUG

#define malloc malloc
#define free free

int current_mem_usage;

struct mem_block
{
   void *buf;
   int size;
   struct mem_block *next;
};

struct mem_block *mroot = NULL;

void *
xchat_malloc (int size)
{
   void *ret;
   struct mem_block *new;

   current_mem_usage += size;
   ret = malloc (size);
   if (!ret)
   {
      printf ("Out of memory! (%d)\n", current_mem_usage);
      exit (255);
   }
   new = malloc (sizeof (struct mem_block));
   new->buf = ret;
   new->size = size;
   new->next = mroot;
   mroot = new;

   printf ("Malloc'ed %d bytes, now %d\n", size, current_mem_usage);

   return ret;
}

void 
xchat_free (void *buf)
{
   struct mem_block *cur, *last;

   last = NULL;
   cur = mroot;
   while (cur)
   {
      if (buf == cur->buf)
         break;
      last = cur;
      cur = cur->next;
   }
   if (cur == NULL)
   {
      printf ("Tried to free unknown block %x!\n", buf);
      /*      abort(); */
      free (buf);
      free (cur);
      return;
   }
   current_mem_usage -= cur->size;
   printf ("Free'ed %d bytes, usage now %d\n", cur->size, current_mem_usage);
   if (last)
      last->next = cur->next;
   else
      mroot = cur->next;
   free (cur);
}

#endif /* MEMORY_DEBUG */


char *
file_part (char *file)
{
   char *filepart = file;
   while (1)
   {
      switch (*file)
      {
      case 0:
         return (filepart);
      case '/':
         filepart = file + 1;
         break;
      }
      file++;
   }
}

void 
path_part (char *file, char *path)
{
   char *filepart = file_part (file);
   *filepart = 0;
   strcpy (path, file);
}

char *
nocasestrstr (char *text, char *tofind)  /* like strstr(), but nocase */
{
   char *ret = text, *find = tofind;

   while (1)
   {
      if (*find == 0)
         return ret;
      if (*text == 0)
         return 0;
      if (toupper (*find) != toupper (*text))
      {
         ret = text + 1;
         find = tofind;
      } else
         find++;
      text++;
   }
}

char *
errorstring (int err)
{
   static char tbuf[16];
   switch (err)
   {
   case ECONNREFUSED:
      return ("Connection refused");
   case ENETUNREACH:
   case EHOSTUNREACH:
      return ("No route to host");
   case ETIMEDOUT:
      return ("Connection timed out");
   }
   sprintf (tbuf, "%d", err);
   return (tbuf);
}

int 
waitline (int sok, char *buf, int bufsize)
{
   int i = 0;

   while (1)
   {
      if (read (sok, &buf[i], 1) < 1)
         return -1;
      if (buf[i] == '\n')
      {
         buf[i] = 0;
         return i;
      }
      i++;
      if (i == (bufsize - 1))
         return 0;
   }
}

/* checks for "~" in a file and expands */

void 
check_homedir (char *file)
{
   char temp[128];
   if (*file == '~')
   {
      strcpy (temp, file);
      sprintf (file, "%s%s", g_get_home_dir (), (char *) &temp[1]);
   }
}

int 
get_mhz (void)
{
   char *mhz;
   char buf[128];
   int fh;

   fh = open ("/proc/cpuinfo", O_RDONLY);  /* linux 2.2+ only */
   if (fh != -1)
   {
      while (1)
      {
         if (waitline (fh, buf, sizeof buf) < 0)
            break;
         if (!strncmp (buf, "cycle frequency [Hz", 19))  /* alpha */
         {
            close (fh);
            mhz = strchr (buf, ':');
            if (mhz)
               return atoi (mhz + 2) / 1048576;
            return 0;
         }
         if (!strncmp (buf, "cpu MHz", 7))  /* x86 */
         {
            close (fh);
            mhz = strchr (buf, ':');
            if (mhz)
               return (int) (atof (mhz + 2) + 0.5);
            return 0;
         }
      }
      close (fh);
   }
   return 0;
}

int
buf_get_line (char *ibuf, char **buf, int *position, int len)
{
   int pos = *position, spos = pos;

   if (pos == len)
     return 0;
   
   while (ibuf[pos++] != '\n')
   {
      if (pos == len)
         return 0;
   }
   pos--;
   ibuf[pos] = 0;
   *buf = &ibuf[spos];
   pos++;
   *position = pos;
   return 1;
}


/* borrowed from ircII */

int
match(mask, string)
     u_char *mask, *string;
{
  unsigned char *m = mask, *n = string, *ma = NULL, *na = NULL;
  int just = 0;
  unsigned char *pm = NULL, *pn = NULL;
  int lp_c = 0, count = 0, last_count = 0;

  while(1)
    {

      switch (*m)
	{
	case '*':
	  goto ultimate_lameness;
	  break;
	case '%':
	  goto ultimate_lameness2;
	  break;
	case '?':
	  m++;
	  if (!*n++) return 0;
	  break;
	case 0:
	  if (!*n)
	    return 1;
	case '\\':
	  if (((*m=='\\') && (m[1]=='*')) || (m[1]=='?')) m++;
	default:
	  if (tolower(*m) != tolower(*n))
	    return 0;
	  else
	    {
	      count++;
	      m++;
	      n++;
	    }
	}
    }

  while(1)
    {

      switch (*m)
	{
	case '*':
ultimate_lameness:
	  ma = ++m;
	  na = n;
	  just = 1;
	  last_count = count;
	  break;
	case '%':
ultimate_lameness2:
	  pm = ++m;
	  pn = n;
	  lp_c = count;
	  if (*n == ' ') pm = NULL;
	  break;
	case '?':
	  m++;
	  if (!*n++) return 0;
	  break;
	case 0:
	  if (!*n || just)
	    return 1;
	case '\\':
	  if (((*m=='\\') && (m[1]=='*')) || (m[1]=='?')) m++;
	default:
	  just = 0;
	  if (tolower(*m) != tolower(*n))
	    {
	      if (!*n) return 0;
	      if (pm)
		{
		  m = pm;
		  n = ++pn;
		  count = lp_c;
		  if (*n == ' ') pm = NULL;
		}
	      else
		if (ma)
		  {
		    m = ma;
		    n = ++na;
		    if (!*n) return 0;
		    count = last_count;
		  }
		else return 0;
	    }
	  else
	    {
	      count++;
	      m++;
	      n++;
	    }
	}
    }
}

void
for_files(char *dirname, char *mask, void callback(char *file))
{
   DIR *dir;
   struct dirent *ent;
   char buf[256];

   dir = opendir(dirname);
   if(dir)
   {
	while((ent = readdir(dir)))
	{
	   if(strcmp(ent->d_name, ".") && strcmp(ent->d_name, ".."))
	   {
		if(match(mask, ent->d_name))
		{
		   sprintf(buf, "%s/%s", dirname, ent->d_name);
		   callback(buf);
		}
	   }
	}
	closedir(dir);
   }
}


/************************************************************************
 *    This technique was borrowed in part from the source code to 
 *    ircd-hybrid-5.3 to implement case-insensitive string matches which
 *    are fully compliant with Section 2.2 of RFC 1459, the copyright
 *    of that code being (C) 1990 Jarkko Oikarinen and under the GPL.
 *    
 *    A special thanks goes to Mr. Okarinen for being the one person who
 *    seems to have ever noticed this section in the original RFC and
 *    written code for it.  Shame on all the rest of you (myself included).
 *    
 *        --+ Dagmar d'Surreal
 */


int 
rfc_casecmp (char *s1, char *s2)
{
   register unsigned char *str1 = (unsigned char *) s1;
   register unsigned char *str2 = (unsigned char *) s2;
   register int res;

   while ((res = toupper (*str1) - toupper (*str2)) == 0)
   {
      if (*str1 == '\0')
         return 0;
      str1++;
      str2++;
   }
   return (res);
}

int 
rfc_ncasecmp (char *str1, char *str2, int n)
{
   register unsigned char *s1 = (unsigned char *) str1;
   register unsigned char *s2 = (unsigned char *) str2;
   register int res;

   while ((res = toupper (*s1) - toupper (*s2)) == 0)
   {
      s1++;
      s2++;
      n--;
      if (n == 0 || (*s1 == '\0' && *s2 == '\0'))
         return 0;
   }
   return (res);
}

unsigned char tolowertab[] =
{0, 0x1, 0x2, 0x3, 0x4, 0x5, 0x6, 0x7, 0x8, 0x9, 0xa,
 0xb, 0xc, 0xd, 0xe, 0xf, 0x10, 0x11, 0x12, 0x13, 0x14,
 0x15, 0x16, 0x17, 0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d,
 0x1e, 0x1f,
 ' ', '!', '"', '#', '$', '%', '&', 0x27, '(', ')',
 '*', '+', ',', '-', '.', '/',
 '0', '1', '2', '3', '4', '5', '6', '7', '8', '9',
 ':', ';', '<', '=', '>', '?',
 '@', 'a', 'b', 'c', 'd', 'e', 'f', 'g', 'h', 'i',
 'j', 'k', 'l', 'm', 'n', 'o', 'p', 'q', 'r', 's',
 't', 'u', 'v', 'w', 'x', 'y', 'z', '{', '|', '}', '~',
 '_',
 '`', 'a', 'b', 'c', 'd', 'e', 'f', 'g', 'h', 'i',
 'j', 'k', 'l', 'm', 'n', 'o', 'p', 'q', 'r', 's',
 't', 'u', 'v', 'w', 'x', 'y', 'z', '{', '|', '}', '~',
 0x7f,
 0x80, 0x81, 0x82, 0x83, 0x84, 0x85, 0x86, 0x87, 0x88, 0x89,
 0x8a, 0x8b, 0x8c, 0x8d, 0x8e, 0x8f,
 0x90, 0x91, 0x92, 0x93, 0x94, 0x95, 0x96, 0x97, 0x98, 0x99,
 0x9a, 0x9b, 0x9c, 0x9d, 0x9e, 0x9f,
 0xa0, 0xa1, 0xa2, 0xa3, 0xa4, 0xa5, 0xa6, 0xa7, 0xa8, 0xa9,
 0xaa, 0xab, 0xac, 0xad, 0xae, 0xaf,
 0xb0, 0xb1, 0xb2, 0xb3, 0xb4, 0xb5, 0xb6, 0xb7, 0xb8, 0xb9,
 0xba, 0xbb, 0xbc, 0xbd, 0xbe, 0xbf,
 0xc0, 0xc1, 0xc2, 0xc3, 0xc4, 0xc5, 0xc6, 0xc7, 0xc8, 0xc9,
 0xca, 0xcb, 0xcc, 0xcd, 0xce, 0xcf,
 0xd0, 0xd1, 0xd2, 0xd3, 0xd4, 0xd5, 0xd6, 0xd7, 0xd8, 0xd9,
 0xda, 0xdb, 0xdc, 0xdd, 0xde, 0xdf,
 0xe0, 0xe1, 0xe2, 0xe3, 0xe4, 0xe5, 0xe6, 0xe7, 0xe8, 0xe9,
 0xea, 0xeb, 0xec, 0xed, 0xee, 0xef,
 0xf0, 0xf1, 0xf2, 0xf3, 0xf4, 0xf5, 0xf6, 0xf7, 0xf8, 0xf9,
 0xfa, 0xfb, 0xfc, 0xfd, 0xfe, 0xff};

unsigned char touppertab[] =
{0, 0x1, 0x2, 0x3, 0x4, 0x5, 0x6, 0x7, 0x8, 0x9, 0xa,
 0xb, 0xc, 0xd, 0xe, 0xf, 0x10, 0x11, 0x12, 0x13, 0x14,
 0x15, 0x16, 0x17, 0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d,
 0x1e, 0x1f,
 ' ', '!', '"', '#', '$', '%', '&', 0x27, '(', ')',
 '*', '+', ',', '-', '.', '/',
 '0', '1', '2', '3', '4', '5', '6', '7', '8', '9',
 ':', ';', '<', '=', '>', '?',
 '@', 'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I',
 'J', 'K', 'L', 'M', 'N', 'O', 'P', 'Q', 'R', 'S',
 'T', 'U', 'V', 'W', 'X', 'Y', 'Z', '[', '\\', ']', '^',
 0x5f,
 '`', 'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I',
 'J', 'K', 'L', 'M', 'N', 'O', 'P', 'Q', 'R', 'S',
 'T', 'U', 'V', 'W', 'X', 'Y', 'Z', '[', '\\', ']', '^',
 0x7f,
 0x80, 0x81, 0x82, 0x83, 0x84, 0x85, 0x86, 0x87, 0x88, 0x89,
 0x8a, 0x8b, 0x8c, 0x8d, 0x8e, 0x8f,
 0x90, 0x91, 0x92, 0x93, 0x94, 0x95, 0x96, 0x97, 0x98, 0x99,
 0x9a, 0x9b, 0x9c, 0x9d, 0x9e, 0x9f,
 0xa0, 0xa1, 0xa2, 0xa3, 0xa4, 0xa5, 0xa6, 0xa7, 0xa8, 0xa9,
 0xaa, 0xab, 0xac, 0xad, 0xae, 0xaf,
 0xb0, 0xb1, 0xb2, 0xb3, 0xb4, 0xb5, 0xb6, 0xb7, 0xb8, 0xb9,
 0xba, 0xbb, 0xbc, 0xbd, 0xbe, 0xbf,
 0xc0, 0xc1, 0xc2, 0xc3, 0xc4, 0xc5, 0xc6, 0xc7, 0xc8, 0xc9,
 0xca, 0xcb, 0xcc, 0xcd, 0xce, 0xcf,
 0xd0, 0xd1, 0xd2, 0xd3, 0xd4, 0xd5, 0xd6, 0xd7, 0xd8, 0xd9,
 0xda, 0xdb, 0xdc, 0xdd, 0xde, 0xdf,
 0xe0, 0xe1, 0xe2, 0xe3, 0xe4, 0xe5, 0xe6, 0xe7, 0xe8, 0xe9,
 0xea, 0xeb, 0xec, 0xed, 0xee, 0xef,
 0xf0, 0xf1, 0xf2, 0xf3, 0xf4, 0xf5, 0xf6, 0xf7, 0xf8, 0xf9,
 0xfa, 0xfb, 0xfc, 0xfd, 0xfe, 0xff};
