/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/* e-table.c: this file is part of printers-admin, a ximian-setup-tool frontend 
 * for print administration.
 * 
 * Copyright (C) 2000-2001 Ximian, Inc.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.
 *
 * Authors: Tambet Ingo <tambet@ximian.com> and Arturo Espinosa <arturo@ximian.com>.
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <gnome.h>
#include "xst.h"
#include <gal/e-table/e-table-scrolled.h>
#include <gal/e-table/e-table-memory.h>
#include <gal/e-table/e-table-memory-callbacks.h>
#include <gal/e-table/e-cell-text.h>
#include <gal/e-paned/e-hpaned.h>

#include "e-table.h"
#include "callbacks.h"

#define PRINT_TABLE_SPEC "print.etspec"

extern XstTool *tool;

GtkWidget *print_table = NULL;

gint active_table;

/* e-table states. */

const gchar *adv_print_state = "\
<ETableState> \
  <column source=\"0\"/> \
  <column source=\"1\"/> \
  <column source=\"2\"/> \
  <column source=\"3\"/> \
  <column source=\"4\"/> \
  <grouping> \
    <leaf column=\"0\" ascending=\"true\"/> \
  </grouping> \
</ETableState>";

const gchar *basic_print_state = "\
<ETableState> \
  <column source=\"0\"/> \
  <column source=\"1\"/> \
  <column source=\"2\"/> \
  <grouping> \
    <leaf column=\"0\" ascending=\"true\"/> \
  </grouping> \
</ETableState>";

static int
print_col_count (ETableModel *etm, void *data)
{
        return COL_PRINTER_LAST;
}

static void *
duplicate_value (ETableModel *etm, int col, const void *value, void *data)
{
        return g_strdup (value);
}

static void
free_value (ETableModel *etm, int col, void *value, void *data)
{
        g_free (value);
}

static void *
initialize_value (ETableModel *etm, int col, void *data)
{
        return g_strdup ("");
}

static gboolean
value_is_empty (ETableModel *etm, int col, const void *value, void *data)
{
        return !(value && *(char *)value);
}

static char *
value_to_string (ETableModel *etm, int col, const void *value, void *data)
{
	return (gchar *)value;
}

static struct {
	int   field_id;
	char *attr_name;
} field_mappings [] = {
	{ COL_PRINTER_NAME,      "name" },
	{ COL_PRINTER_TYPE,      "type" },
	{ COL_PRINTER_DEVICE,    "gsdevice" },
	{ COL_PRINTER_PAPERSIZE, "paper_size" },
	{ COL_PRINTER_RESOLUTION, "resolution" },
	{ COL_PRINTER_SHARE,      "share" },
	{ 0, NULL }
};

static void
print_set_value_at (ETableModel *etm, int col, int row, const void *val, void *data)
{
	xmlNodePtr node;

	g_return_if_fail (xst_tool_get_access (tool));

	node = e_table_memory_get_data (E_TABLE_MEMORY (etm), row);
	if (!node)
		return;

	switch (col)
	{
		case COL_PRINTER_NAME:
		case COL_PRINTER_TYPE:
		case COL_PRINTER_DEVICE:
		case COL_PRINTER_PAPERSIZE:
		case COL_PRINTER_RESOLUTION:
		case COL_PRINTER_SHARE:
	default:
		break;
	}

	g_warning ("Set value at stubbed");
}

static gboolean
is_editable (ETableModel *etm, int col, int row, void *model_data)
{
	return (xst_tool_get_access (tool) &&
		   (xst_dialog_get_complexity (tool->main_dialog) == XST_DIALOG_ADVANCED));
}

static void *
print_value_at (ETableModel *etm, int col, int row, void *model_data)
{
	xmlNodePtr node;
	int        i;

	node = e_table_memory_get_data (E_TABLE_MEMORY (etm), row);

	if (!node)
		return NULL;

	for (i = 0; field_mappings [i].attr_name; i++) {
		if (field_mappings [i].field_id == col) {
			node = xst_xml_element_find_first (
				node, field_mappings [i].attr_name);

			if (!node)
				return NULL;

			return xst_xml_element_get_content (node);
		}
	}

	g_warning ("print_value_at: wrong col nr");
	return NULL;

}

static gint
id_compare (gconstpointer id1, gconstpointer id2)
{
	g_return_val_if_fail (id1 != NULL, 1);
	g_return_val_if_fail (id2 != NULL, -1);

	if (atoi (id1) > atoi (id2))
		return 1;
	if (atoi (id1) < atoi (id2))
		return -1;
	return 0;
}

static ETableExtras *
create_extras (void)
{
	ETableExtras *extras;
/*	ECell *ec; */

	extras = e_table_extras_new ();
	e_table_extras_add_compare (extras, "id_compare", id_compare);

	return extras;
}

static void
cursor_change (ETable *et, gint row, gpointer print_data)
{
	gint table;

	table = GPOINTER_TO_INT (E_TABLE_MEMORY_CALLBACKS (et->model)->data);
}

static void
create_print_table (void)
{
	ETable *table;
	ETableModel *model;
	ETableExtras *extras;
	GtkWidget *container;
	gchar *spec;

	extras = create_extras ();

	model = e_table_memory_callbacks_new (print_col_count,
					      print_value_at,
					      print_set_value_at,
					      is_editable,
					      duplicate_value,
					      free_value,
					      initialize_value,
					      value_is_empty,
					      value_to_string,
					      GINT_TO_POINTER (TABLE_PRINTERS));

	spec = xst_conf_get_string (tool, "spec");
	if (!spec) {
		spec = xst_ui_load_etspec (tool->etspecs_common_path, PRINT_TABLE_SPEC);
		if (!spec)
			g_error ("create_print_table: Couldn't make table.");
		xst_conf_set_string (tool, "spec", spec);
	}
		
	print_table = e_table_scrolled_new (E_TABLE_MODEL (model), extras, spec,
					   basic_print_state);

	g_free(spec);

	table = e_table_scrolled_get_table (E_TABLE_SCROLLED (print_table));
	gtk_signal_connect (GTK_OBJECT (table), "cursor_change", cursor_change, NULL);
	gtk_signal_connect (GTK_OBJECT (table), "double_click", on_settings_clicked, NULL);

	container = xst_dialog_get_widget (tool->main_dialog, "printers_holder");
	gtk_container_add (GTK_CONTAINER (container), print_table);
	gtk_widget_show (print_table);
}

static void
clear_all_tables (void)
{
	e_table_memory_clear (E_TABLE_MEMORY (
		e_table_scrolled_get_table (
			E_TABLE_SCROLLED (print_table))->model));
}

void
populate_all_tables (void)
{
	ETable      *table;
	ETableModel *model;
	xmlNodePtr   node, root_node;

	table = e_table_scrolled_get_table (
		E_TABLE_SCROLLED (print_table));
	g_return_if_fail (table != NULL);

	model = table->model;
	g_return_if_fail (model != NULL);

	root_node = xst_xml_doc_get_root (tool->config);

	if (!root_node) {
		g_warning ("No data");
		return;
	}

	e_table_memory_freeze (E_TABLE_MEMORY (model));
	for (node = root_node->childs; node; node = node->next) {
		if (xst_xml_element_find_first (node, "name"))
			e_table_memory_insert (E_TABLE_MEMORY (model), -1, node);
	}

	e_table_memory_thaw (E_TABLE_MEMORY (model));
}

extern guint
create_tables (void)
{
	/* Main tables */
	create_print_table ();

	return FALSE;
}

void
tables_update_content (void)
{
	ETable *u_table;
	xmlNodePtr u_node, g_node;
	gint row;
	guint saved_table;

	saved_table = active_table;
	
	u_table = e_table_scrolled_get_table (E_TABLE_SCROLLED (print_table));

	u_node = g_node = NULL;
	
	/* Get selected print node */
	if ((row = e_table_get_cursor_row (u_table)) >= 0)
		u_node = e_table_memory_get_data (E_TABLE_MEMORY (u_table->model), row);
	
	clear_all_tables ();
	populate_all_tables ();
}

void
tables_set_state (gboolean state)
{
	ETable *u_table;
	
	u_table = e_table_scrolled_get_table (E_TABLE_SCROLLED (print_table));

	if (state)
		e_table_set_state (u_table, adv_print_state);
	else
		e_table_set_state (u_table, basic_print_state);

	tables_update_content ();
}

static ETable *
get_table (gint tbl)
{
	ETable *table = NULL;

	switch (tbl)
	{
	case TABLE_PRINTERS:
		table = e_table_scrolled_get_table (E_TABLE_SCROLLED (print_table));
		break;
		
	default:
		return NULL;
	}
	
	return table;
}

static ETable *
get_current_table (void)
{
	return get_table (active_table);
}

xmlNodePtr
get_selected_node (void)
{
	ETable *table;
	gint row;

	g_return_val_if_fail (table = get_current_table (), NULL);

	if ((row = e_table_get_cursor_row (table)) >= 0)
		return e_table_memory_get_data (E_TABLE_MEMORY (table->model), row);
	
	return NULL;
}

gboolean
delete_selected_node (gint tbl)
{
	ETable *table;
	gint row;

	g_return_val_if_fail (table = get_table (tbl), FALSE);

	if ((row = e_table_get_cursor_row (table)) >= 0) {
		e_table_memory_remove (E_TABLE_MEMORY (table->model), row);
		return TRUE;
	}

	return FALSE;
}

void
current_table_update_row (gint tbl)
{
	ETable *table;

	table = get_table (tbl);
	e_table_model_row_changed (table->model,
				   e_table_get_cursor_row (table));
}

void
current_table_new_row (xmlNodePtr node, gint tbl)
{
	ETable *table;

	g_return_if_fail (node != NULL);

	table = get_table (tbl);
	e_table_memory_insert (E_TABLE_MEMORY (table->model), -1, node);
}
