/* $NetBSD$ */

/*
 * Copyright  2005 Alistair Crooks.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. The name of the author may not be used to endorse or promote
 *    products derived from this software without specific prior written
 *    permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE
 * GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
#include "config.h"

#ifdef HAVE_SYS_CDEFS_H
#include <sys/cdefs.h>
#endif

#ifndef lint
__COPYRIGHT("@(#) Copyright  2005 \
	        Alistair Crooks.  All rights reserved.");
__RCSID("$NetBSD$");
#endif

#include <sys/types.h>
#include <sys/stat.h>

#ifdef HAVE_SYS_WAIT_H
#include <sys/wait.h>
#endif

#ifdef HAVE_SYS_PARAM_H
#include <sys/param.h>
#endif

#ifdef HAVE_SYS_ENDIAN_H
#include <sys/endian.h>
#endif


#include <ctype.h>

#ifdef HAVE_DIRENT_H
#include <dirent.h>
#endif

#ifdef HAVE_ERR_H
#include <err.h>
#endif

#ifdef HAVE_ERRNO_H
#include <errno.h>
#endif

#ifdef HAVE_FCNTL_H
#include <fcntl.h>
#endif

#ifdef HAVE_FNMATCH_H
#include <fnmatch.h>
#endif

#ifdef HAVE_GRP_H
#include <grp.h>
#endif

#ifdef HAVE_PWD_H
#include <pwd.h>
#endif

#ifdef HAVE_REGEX_H
#include "regex.h"
#endif

#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>

#ifdef HAVE_STRING_H
#include <string.h>
#endif

#include <unistd.h>

#ifdef HAVE_ZLIB_H
#include <zlib.h>
#endif

#ifdef HAVE_BZLIB_H
#include <bzlib.h>
#endif

#include "defs.h"
#include "aa.h"

enum {
	REGMATCHMAX = 10,

	ZLIB_SMALLFILE = 128,
	ZLIB_OVERHEAD = 12,

	BZLIB_BLOCKSIZE = 9,
	BZLIB_VERBOSITY = 0,
	BZLIB_WORKFACTOR = 30
};




#define BSWAP16(x)	((((x) & 0xff) << 8) | (((x) & 0xff00) >> 8))
#define BSWAP32(x)	((((x) & 0xff) << 24) | (((x) & 0xff00) << 8) | (((x) & 0xff0000) >> 8) | (((x) & 0xff000000) >> 24))
#define BSWAP64(x)	((((x) & 0xff) << 56) | (((x) & 0xff00) << 40) | (((x) & 0xff0000) << 24) | (((x) & 0xff000000) << 8) | \
		 (((x) & 0xff00000000) >> 8) | (((x) & 0xff0000000000) >> 24) | (((x) & 0xff000000000000) >> 40) | (((x) & 0xff00000000000000) >> 56))

static const char	*protections = " SEB";


uint16_t
archangel_htobe16(uint16_t in)
{
	int	indian = 1;

	return (*(char *) &indian) ? BSWAP16(in) : in;
}

uint16_t
archangel_be16toh(uint16_t in)
{
	return archangel_htobe16(in);
}

uint32_t
archangel_htobe32(uint32_t in)
{
	int	indian = 1;

	return (*(char *) &indian) ? BSWAP32(in) : in;
}

uint32_t
archangel_be32toh(uint32_t in)
{
	return archangel_htobe32(in);
}

uint64_t
archangel_htobe64(uint64_t in)
{
	int	indian = 1;

	return (*(char *) &indian) ? BSWAP64(in) : in;
}

uint64_t
archangel_be64toh(uint64_t in)
{
	return archangel_htobe64(in);
}

/* get the credentials */
static void
get_creds(aa_t *aa, char protection)
{
	if ((protection & ARCHANGEL_ENCRYPTED) && aa->id[0] == 0x0) {
		printf("gpg id? ");
		(void) fflush(stdout);
		if (fgets(aa->id, sizeof(aa->id), stdin) == NULL) {
			exit(EXIT_SUCCESS);
		}
		aa->id[MIN(sizeof(aa->id), strlen(aa->id)) - 1] = 0x0;
	}
	if ((protection & (ARCHANGEL_SIGNED | ARCHANGEL_ENCRYPTED)) && aa->phrase[0] == 0x0) {
		(void) strlcpy(aa->phrase, getpass("gpg passphrase? "), sizeof(aa->phrase));
	}
}

/* a replacement for system(3) */
static int
asystem(aa_t *aa, const char *file, int enc, char protection, char *name)
{
	char	desc[20];
	int	fdv[2];
	int	status;
	int	pid;

	pipe(fdv);
	get_creds(aa, protection);
	switch(pid = fork()) {
	case 0: /* child */
		(void) close(fdv[1]);
		(void) snprintf(desc, sizeof(desc), "%d", fdv[0]);
		if (enc) {
			switch(protection) {
			case ARCHANGEL_SIGNED:
				execl(PREFIX "/bin/gpg", "gpg", "--sign", "--passphrase-fd", desc, file, NULL);
				_exit(127);
			case ARCHANGEL_ENCRYPTED:
				execl(PREFIX "/bin/gpg", "gpg", "--encrypt", "-r", aa->id, "--passphrase-fd", desc, file, NULL);
				_exit(127);
			case (ARCHANGEL_SIGNED | ARCHANGEL_ENCRYPTED):
				execl(PREFIX "/bin/gpg", "gpg", "--sign", "--encrypt", "-r", aa->id, "--passphrase-fd", desc, file, NULL);
				_exit(127);
			}
		} else {
			execl(PREFIX "/bin/gpg", "gpg", "--decrypt", "--output", name, "--passphrase-fd", desc, file, NULL);
			_exit(127);
		}
		break;
	default: /* parent */
		(void) close(fdv[0]);
		(void) write(fdv[1], aa->phrase, strlen(aa->phrase));
		(void) write(fdv[1], "\n", 1);
		(void) wait(&status);
		(void) close(fdv[1]);
		break;
	}
	return 0;
}

/* gzip a file, putting the result in zname. If compressed is larger than original, then just use the original file */
static int
gzip_compress_file(aa_t *aa, int fd, char *name, char *zname, int znamelen, int64_t maxfsize, int64_t *zsize)
{
	struct stat	st;
	int64_t		i;
	gzFile		z;
	char		buf[KB(32)];
	int		ret;
	int		zfd;
	int		cc;

	ret = 1;
	(void) strlcpy(zname, "/tmp/aa.XXXXXX", znamelen);
	zfd = mkstemp(zname);
	z = gzdopen(zfd, "wb9");
	for (i = 0 ; ret && i < maxfsize ; i += cc) {
		if ((cc = read(fd, buf, MIN(sizeof(buf), maxfsize - i))) < 0) {
			warn("gzip_compress_file: short read \"%s\"", name);
			ret = 0;
		} else if (gzwrite(z, buf, cc) != cc) {
			warn("gzip_compress_file: short gzwrite \"%s\"", name);
			ret = 0;
		}
	}
	gzclose(z);
	(void) stat(zname, &st);
	*zsize = st.st_size;
	return ret;
}

/* bzip2 a file, putting the result in zname. If compressed is larger than original, then just use the original file */
static int
bzip2_compress_file(aa_t *aa, int fd, char *name, char *zname, int znamelen, int64_t maxfsize, int64_t *zsize)
{
	struct stat	 st;
	int64_t		 i;
	BZFILE		*z;
	char		 buf[KB(32)];
	int		 ret;
	int		 zfd;
	int		 cc;

	ret = 1;
	(void) strlcpy(zname, "/tmp/aa.XXXXXX", znamelen);
	zfd = mkstemp(zname);
	z = BZ2_bzdopen(zfd, "wb9");
	for (i = 0 ; ret && i < maxfsize ; i += cc) {
		if ((cc = read(fd, buf, MIN(sizeof(buf), maxfsize - i))) < 0) {
			warn("bzip2_compress_file: short read \"%s\"", name);
			ret = 0;
		} else if (BZ2_bzwrite(z, buf, cc) != cc) {
			warn("bzip2_compress_file: short BZ2_bzwrite \"%s\"", name);
			ret = 0;
		}
	}
	BZ2_bzclose(z);
	(void) stat(zname, &st);
	*zsize = st.st_size;
	return ret;
}

/* do not compress a file - copy of original goes in zname */
static int
no_compress_file(aa_t *aa, int infd, char *name, char *zname, int znamelen, int64_t maxfsize, int64_t *zsize)
{
	struct stat	st;
	int64_t		i;
	char		buf[KB(32)];
	int		ret;
	int		outfd;
	int		cc;

	ret = 1;
	(void) strlcpy(zname, "/tmp/aa.XXXXXX", znamelen);
	outfd = mkstemp(zname);
	for (i = 0 ; ret && i < maxfsize ; i += cc) {
		if ((cc = read(infd, buf, MIN(sizeof(buf), maxfsize - i))) < 0) {
			warn("no_compress_file: short read \"%s\"", name);
			ret = 0;
		} else if (write(outfd, buf, cc) != cc) {
			warn("no_compress_file: short gzwrite \"%s\"", name);
			ret = 0;
		}
	}
	(void) close(outfd);
	(void) stat(zname, &st);
	*zsize = st.st_size;
	return ret;
}

/* "sign" a file, putting the result in zname */
static int
sign_file(aa_t *aa, int infd, char *name, char *zname, int znamelen, int64_t maxfsize, int64_t *zsize, char protection)
{
	struct stat	st;
	int64_t		i;
	char		buf[KB(32)];
	int		ret;
	int		outfd;
	int		cc;

	ret = 1;
	(void) strlcpy(zname, "/tmp/aa.XXXXXX", znamelen);
	outfd = mkstemp(zname);
	for (i = 0 ; ret && i < maxfsize ; i += cc) {
		if ((cc = read(infd, buf, MIN(sizeof(buf), maxfsize - i))) < 0) {
			warn("sign_file: short read \"%s\"", name);
			ret = 0;
		} else if (write(outfd, buf, cc) != cc) {
			warn("sign_file: short gzwrite \"%s\"", name);
			ret = 0;
		}
	}
	(void) close(outfd);
	get_creds(aa, protection);
	(void) asystem(aa, zname, 1, protection, NULL);
	(void) strlcat(zname, ".gpg", znamelen);
	(void) stat(zname, &st);
	*zsize = st.st_size;
	return ret;
}

/* find the best compression algorithm */
static int
find_best(aa_t *aa, aaent_t *aep, char *name, int fd, int64_t from)
{
	unsigned long	gzc;
	unsigned	bzc;
	char		gzbuf[KB(2)];
	char		bzbuf[KB(2)];
	char		buf[KB(1)];
	int		cc;

	if (lseek(fd, from, SEEK_SET) < 0) {
		warn("find_best: can't lseek \"%s\" to %lld", name, (long long) from);
		(void) close(fd);
		return ARCHANGEL_LITERAL;
	}
	if ((cc = read(fd, buf, sizeof(buf))) < 0) {
		(void) close(fd);
		return ARCHANGEL_LITERAL;
	}
	if (lseek(fd, from, SEEK_SET) < 0) {
		warn("find_best: can't lseek \"%s\" to %lld", name, (long long) from);
		(void) close(fd);
		return ARCHANGEL_LITERAL;
	}
	BZ2_bzBuffToBuffCompress(bzbuf, &bzc, buf, cc, BZLIB_BLOCKSIZE, BZLIB_VERBOSITY, BZLIB_WORKFACTOR);
	compress(gzbuf, &gzc, buf, cc);
	if (aa->verbose > 2) {
		printf("find_best: file \"%s\", original %d, libz %ld, libbz2 %d\n", name, cc, gzc, bzc);
	}
	if (bzc < gzc || MIN(aep->sub.size - from, aa->maxfsize) >= KB(900)) {
		return ARCHANGEL_BZLIB;
	}
	if (cc >= ZLIB_SMALLFILE && gzc + ZLIB_OVERHEAD < cc) {
		/* small files gzip badly, as overhead is needed */
		return ARCHANGEL_ZLIB;
	}
	return ARCHANGEL_LITERAL;
}

/* copy stream from aa->fd into zname - if name is NULL, just read from aa->fd */
static int
copystream(aa_t *aa, char *zname, int znamelen, int64_t zsize, int64_t padding, char *name, char *buf, int32_t bufsize)
{
	int64_t		i;
	int		ret;
	int		zfd;
	int		cc;

	zfd = -1;
	if (name != NULL) {
		(void) strlcpy(zname, "/tmp/aa.XXXXXX", znamelen);
		zfd = mkstemp(zname);
	}
	for (ret = 1, i = 0 ; ret && i < zsize ; i += cc) {
		if ((cc = read(aa->fd, buf, MIN(bufsize, zsize - i))) < 0) {
			warn("copystream: short read \"%s\"", name);
			ret = 0;
		} else if (name != NULL && write(zfd, buf, cc) != cc) {
			warn("copystream: short write \"%s\"", name);
			ret = 0;
		}
	}
	if (padding > 0 && read(aa->fd, buf, padding) != padding) {
		warn("copystream: short padding read \"%s\"", name);
		ret = 0;
	}
	if (name != NULL) {
		(void) close(zfd);
	}
	return ret;
}

/* decompress `zname' into `fd' */
static int
literal_decompress_file(int infd, int64_t size, char *buf, int bufsize, int fd, char *name)
{
	int64_t	i;
	int	ret;
	int	cc;

	for (ret = 1, i = 0 ; ret && i < size ; i += cc) {
		if ((cc = read(infd, buf, MIN(bufsize, size - i))) < 0) {
			warn("literal_decompress_file: short read \"%s\"", name);
			ret = 0;
		} else if (write(fd, buf, cc) != cc) {
			warn("literal_decompress_file: short write \"%s\"", name);
			ret = 0;
		}
	}
	(void) close(infd);
	return ret;
}

/* decompress a file (which is in a temp file) */
static int
decompressfile(aa_t *aa, aaent_t *aep, int fd, char *zname, char *name, char *buf, int bufsize)
{
	int64_t	 i;
	BZFILE	*bz;
	gzFile	 z;
	int	 infd;
	int	 ret;
	int	 cc;

	ret = 1;
	switch (aep->protection) {
	case ARCHANGEL_SIGNED:
		infd = asystem(aa, zname, 0, aep->protection, name);
		break;
	case ARCHANGEL_ENCRYPTED:
	case (ARCHANGEL_SIGNED | ARCHANGEL_ENCRYPTED):
		infd = asystem(aa, zname, 0, aep->protection, name);
		break;
	case ARCHANGEL_CLEAR:
		switch(aep->sub.compression) {
		case ARCHANGEL_ZLIB:
			z = gzopen(zname, "rb9");
			for (i = 0 ; ret && i < aep->sub.size ; i += cc) {
				if ((cc = gzread(z, buf, MIN(bufsize, aep->sub.size - i))) < 0) {
					warn("decompressfile: short read \"%s\"", name);
					ret = 0;
				} else if (write(fd, buf, cc) != cc) {
					warn("decompressfile: short write \"%s\"", name);
					ret = 0;
				}
			}
			gzclose(z);
			break;
		case ARCHANGEL_BZLIB:
			bz = BZ2_bzopen(zname, "rb9");
			for (i = 0 ; ret && i < aep->sub.size ; i += cc) {
				if ((cc = BZ2_bzread(bz, buf, MIN(bufsize, aep->sub.size - i))) < 0) {
					warn("decompressfile: short read \"%s\"", name);
					ret = 0;
				} else if (write(fd, buf, cc) != cc) {
					warn("decompressfile: short write \"%s\"", name);
					ret = 0;
				}
			}
			BZ2_bzclose(bz);
			break;
		default:
			infd = open(zname, O_RDONLY);
			literal_decompress_file(infd, aep->sub.size, buf, bufsize, fd, name);
			break;
		}
		(void) unlink(zname);
		break;
	}
	return ret;
}

/* copy file into aa->fd */
static int
readfile(aa_t *aa, aaent_t *aep, char *name, int64_t size)
{
	int64_t	i;
	char	buf[KB(32)];
	int	ret;
	int	fd;
	int	cc;

	if ((fd = open(name, O_RDONLY)) < 0) {
		warn("readfile: can't read \"%s\"", name);
		return 0;
	}
	for (ret = 1, i = 0 ; ret && i < size ; i += cc) {
		if ((cc = read(fd, buf, MIN(sizeof(buf), size - i))) < 0) {
			warn("readfile: short read \"%s\"", name);
			ret = 0;
		} else if (write(aa->fd, buf, cc) != cc) {
			warn("readfile: short write \"%s\"", name);
			ret = 0;
		}
	}
	(void) close(fd);
	return ret;
}

#ifdef HAVE_SETXATTR            
/* set the external attributes */
static int
ext_attr_set(aa_t *aa, aaent_t *aep, char *name);
{
	char	*ap;
	char	*eq;
	char	*nl;
	int	 i;

	for (ap = aep->sub.xattr ; ap < aep->sub.xattr + aep->sub.xattrlen ; ap = nl + 1) {
		if ((nl = strchr(ap, '\n')) == NULL) {
			return 0;
		}
		if ((eq = strchr(ap, '=')) == NULL) {
			warn("WARNING: malformed attribute (%s) for \"%s\"", ap, name);
		} else {
			*eq = 0x0;
			if (setxattr(name, ap, eq + 1, (int)(nl - eq) - 1, XATTR_CREATE) < 0) {
				warn("WARNING: can't setxattr (%s) for \"%s\"", ap, name);
			}
			*eq = '=';
		}
	}
}
#endif

/* extract a file into 'name' - if name is NULL, just jump over the entry */
static int
extractfile(aa_t *aa, aaent_t *aep, char *name)
{
	struct timeval	tv[2];
	char		zname[ARCHANGEL_MAXPATHLEN];
	char		buf[KB(32)];
	int		mode;
	int		ret;
	int		fd;

	ret = 1;
	if (name != NULL) {
		if (strncmp(name, "../", 3) == 0) {
			warnx("extractfile: too dangerous to extract \"%s\"", name);
			return 0;
		}
		if (unlink(name) < 0 && errno != ENOENT) {
			warn("extractfile: can't unlink \"%s\"", name);
			return 0;
		}
		mode = (aep->sub.offset == 0) ? (O_RDWR | O_CREAT) : (O_RDWR | O_APPEND);
		if (aep->protection == ARCHANGEL_CLEAR && (fd = open(name, mode, aep->sub.mode)) < 0) {
			warn("extractfile: can't open \"%s\" mode %x", name, mode);
			return 0;
		}
	}
	if (!copystream(aa, zname, sizeof(zname), aep->sub.zsize, aep->sub.padding, name, buf, sizeof(buf))) {
		warnx("extractfile: can't open \"%s\"", name);
		return 0;
	}
	if (name != NULL) {
		if (!decompressfile(aa, aep, fd, zname, name, buf, sizeof(buf))) {
			warn("extractfile: short read \"%s\" at offset %lld", name, (long long) aep->sub.offset);
			ret = 0;
		}
		if (ret == 1) {
			tv[0].tv_sec = aep->sub.atime;
			tv[0].tv_usec = 0;
			tv[1].tv_sec = aep->sub.mtime;
			tv[1].tv_usec = 0;
			mode = (aep->sub.offset == 0) ? (O_RDWR | O_CREAT) : (O_RDWR | O_APPEND);
			if (aep->protection != ARCHANGEL_CLEAR && (fd = open(name, mode, aep->sub.mode)) < 0) {
				warn("extractfile: can't open \"%s\" mode %x", name, mode);
				return 0;
			}
			if (futimes(fd, tv) < 0) {
				warn("extractfile: can't set access/modification time for \"%s\"", name);
			}
			if (aep->sub.xattrlen > 0) {
#ifdef HAVE_SETXATTR
				(void) ext_attr_set(aa, aep, name);
#else
				warnx("WARNING: archive contains extended attributes for \"name\", no way of setting them. Skipping", name);
#endif
			}
		}
		(void) close(fd);
	}
	return ret;
}

/* dump an entry */
static void
dump_aaent(aaent_t *aap)
{
	printf("\nmagic1: %x\n", aap->magic1);
	printf("next: %lld\n", aap->next);
	printf("protection: %c\n", protections[aap->protection]);
	printf("size: %lld\n", aap->sub.size);
	printf("ino: %lld\n", aap->sub.ino);
	printf("mode: %o\n", aap->sub.mode);
	printf("dev: %o\n", aap->sub.dev);
	printf("nlink: %d\n", aap->sub.nlink);
	printf("user: %s\n", aap->sub.user);
	printf("group: %s\n", aap->sub.group);
	printf("mtime: %lld\n", aap->sub.mtime);
	printf("atime: %lld\n", aap->sub.atime);
	printf("ctime: %lld\n", aap->sub.ctime);
	printf("birth: %lld\n", aap->sub.birth);
	printf("blocks: %lld\n", aap->sub.blocks);
	printf("zsize: %lld\n", aap->sub.zsize);
	printf("padding: %lld\n", aap->sub.padding);
	printf("flags: %x\n", aap->sub.flags);
	printf("gen: %d\n", aap->sub.gen);
	printf("compression: %c\n", aap->sub.compression);
	printf("type: %c\n", aap->sub.type);
	printf("offset: %lld\n", aap->sub.offset);
	printf("part: %d\n", aap->sub.part);
	printf("lastpart: %d\n", aap->sub.lastpart);
	printf("namelen: %d\n", aap->sub.namelen);
	printf("magic2: %x\n", aap->magic2);
}

/* write the header to aa->fd */
static int
writeheader(aa_t *aa, aaent_t *ap)
{
	aaent_t	a;

	(void) memset(&a, 0x0, sizeof(a));
	a.magic1 = archangel_htobe32(ap->magic1);
	a.magic2 = archangel_htobe32(ap->magic2);
	a.protection = ap->protection;
	a.next = archangel_htobe64(ap->next);
	a.sub.size = archangel_htobe64(ap->sub.size);
	a.sub.offset = archangel_htobe64(ap->sub.offset);
	a.sub.part = archangel_htobe32(ap->sub.part);
	a.sub.ino = archangel_htobe64(ap->sub.ino);
	a.sub.mode = archangel_htobe32(ap->sub.mode);
	a.sub.dev = archangel_htobe32(ap->sub.dev);
	a.sub.nlink = archangel_htobe32(ap->sub.nlink);
	(void) memcpy(a.sub.user, ap->sub.user, sizeof(a.sub.user));
	(void) memcpy(a.sub.group, ap->sub.group, sizeof(a.sub.group));
	a.sub.mtime = archangel_htobe64(ap->sub.mtime);
	a.sub.atime = archangel_htobe64(ap->sub.atime);
	a.sub.ctime = archangel_htobe64(ap->sub.ctime);
	a.sub.birth = archangel_htobe64(ap->sub.birth);
	a.sub.blocks = archangel_htobe64(ap->sub.blocks);
	a.sub.zsize = archangel_htobe64(ap->sub.zsize);
	a.sub.padding = archangel_htobe64(ap->sub.padding);
	a.sub.flags = archangel_htobe32(ap->sub.flags);
	a.sub.gen = archangel_htobe32(ap->sub.gen);
	a.sub.compression = ap->sub.compression;
	a.sub.lastpart = ap->sub.lastpart;
	a.sub.type = ap->sub.type;
	a.sub.namelen = archangel_htobe32(ap->sub.namelen);
	a.sub.xattrlen = archangel_htobe32(ap->sub.xattrlen);
	if (write(aa->fd, &a, sizeof(a)) != sizeof(a)) {
		warn("writeheader: short write \"%s\"", aa->name);
		return 0;
	}
	return 1;
}

/* read the header from aa->fd */
static int
readheader(aa_t *aa, aaent_t *ap)
{
	aaent_t	a;
	int	cc;

	(void) memset(&a, 0x0, sizeof(a));
	if ((cc = read(aa->fd, &a, sizeof(a))) != sizeof(a)) {
		return 0;
	}
	ap->magic1 = archangel_be32toh(a.magic1);
	ap->magic2 = archangel_be32toh(a.magic2);
	ap->protection = a.protection;
	ap->next = archangel_be64toh(a.next);
	ap->sub.offset = archangel_be64toh(a.sub.offset);
	ap->sub.part = archangel_be32toh(a.sub.part);
	ap->sub.size = archangel_be64toh(a.sub.size);
	ap->sub.ino = archangel_be64toh(a.sub.ino);
	ap->sub.mode = archangel_be32toh(a.sub.mode);
	ap->sub.dev = archangel_be32toh(a.sub.dev);
	ap->sub.nlink = archangel_be32toh(a.sub.nlink);
	(void) memcpy(ap->sub.user, a.sub.user, sizeof(a.sub.user));
	(void) memcpy(ap->sub.group, a.sub.group, sizeof(a.sub.group));
	ap->sub.mtime = archangel_be64toh(a.sub.mtime);
	ap->sub.atime = archangel_be64toh(a.sub.atime);
	ap->sub.ctime = archangel_be64toh(a.sub.ctime);
	ap->sub.birth = archangel_be64toh(a.sub.birth);
	ap->sub.blocks = archangel_be64toh(a.sub.blocks);
	ap->sub.zsize = archangel_be64toh(a.sub.zsize);
	ap->sub.padding = archangel_be64toh(a.sub.padding);
	ap->sub.flags = archangel_be32toh(a.sub.flags);
	ap->sub.gen = archangel_be32toh(a.sub.gen);
	ap->sub.compression = a.sub.compression;
	ap->sub.lastpart = a.sub.lastpart;
	ap->sub.type = a.sub.type;
	ap->sub.namelen = archangel_be32toh(a.sub.namelen);
	ap->sub.xattrlen = archangel_be32toh(a.sub.xattrlen);
	if (aa->verbose > 2) {
		dump_aaent(ap);
	}
	return 1;
}

/* perform basic sanity checks on the header we've read */
static int
sanity_checks(aa_t *aa, aaent_t *aep)
{
	time_t	now;

	(void) time(&now);
	if (aep->sub.size >= MB(256)) {
		warnx("sanity check failure: size (%lld) out of range", aep->sub.size);
		return 0;
	}
	if (aep->sub.size >= MB(256)) {
		warnx("sanity check failure: zsize (%lld) out of range", aep->sub.zsize);
		return 0;
	}
	if (aep->sub.padding >= MB(1)) {
		warnx("sanity check failure: padding (%lld) out of range", aep->sub.padding);
		return 0;
	}
	if (aep->sub.offset >= aep->sub.size) {
		warnx("sanity check failure: offset (%lld) out of range", aep->sub.offset);
		return 0;
	}
	if (aep->sub.mtime < 0 || aep->sub.mtime > now + (24 * 60 * 60)) {
		warnx("sanity check warning: mtime (%lld) is out of range - continuing", aep->sub.mtime);
	}
	if (aep->sub.ctime < 0 || aep->sub.ctime > now + (24 * 60 * 60)) {
		warnx("sanity check warning: ctime (%lld) is out of range - continuing", aep->sub.ctime);
	}
	if (aep->sub.atime < 0 || aep->sub.atime > now + (24 * 60 * 60)) {
		warnx("sanity check warning: atime (%lld) is out of range - continuing", aep->sub.atime);
	}
	switch(aep->sub.compression) {
	case ARCHANGEL_LITERAL:
	case ARCHANGEL_ZLIB:
	case ARCHANGEL_BZLIB:
		break;
	default:
		warnx("sanity check failure: compression (%c,%d) out of range", aep->sub.compression, aep->sub.compression);
		return 0;
	}
	switch(aep->protection) {
	case ARCHANGEL_CLEAR:
	case ARCHANGEL_SIGNED:
	case ARCHANGEL_ENCRYPTED:
	case (ARCHANGEL_SIGNED | ARCHANGEL_ENCRYPTED):
		break;
	default:
		warnx("sanity check failure: protection (%c) out of range", aep->protection);
		return 0;
	}
	if (aep->sub.namelen >= ARCHANGEL_MAXPATHLEN) {
		warnx("sanity check failure: name length (%d) out of range", aep->sub.namelen);
		return 0;
	}
	if (aep->sub.xattrlen >= KB(4)) {
		warnx("sanity check failure: extended attribute length (%d) out of range", aep->sub.xattrlen);
		return 0;
	}
	return 1;
}

/* pad the buffer with 'padding' random bytes */
static void
pad(char *buf, int64_t padding)
{
	int	i;

	for (i = 0 ; i < padding ; i++) {
		buf[i] = random() & 0xff;
	}
}

#define DAYSPERNYEAR	365
#define SECSPERDAY	(24 * 60 * 60)
#define SIXMONTHS	((DAYSPERNYEAR / 2) * SECSPERDAY)

/* print the time of a file - from ls(1) */
static void
printtime(FILE *fp, time_t ftime)
{       
	time_t	 now;
	char	*longstring;
	int	 i;

	(void) time(&now);
	longstring = ctime(&ftime);
	for (i = 4 ; i < 11 ; i++) {
		(void) fputc(longstring[i], fp);
	}
	if (ftime + SIXMONTHS > now && ftime - SIXMONTHS < now) {
		for (i = 11; i < 16; ++i) {
			(void) fputc(longstring[i], fp);
		}
	} else {  
		(void) fputc(' ', fp);
		for (i = 20; i < 24; ++i) {
			(void) fputc(longstring[i], fp);
		}
	}
	(void) fputc(' ', fp);
}

/* print the details for the entry */
static int
print_details(FILE *fp, aa_t *aa, aaent_t *aep, int list, char *newname, uint64_t size, uint64_t zsize, const char *term)
{
	char	buf[40];
	int	cc;

	if (list) {
		if (aa->verbose) {
			(void) strmode(aep->sub.mode, buf);
			(void) fprintf(fp, "%s %*lu ", buf, 2, (unsigned long) aep->sub.nlink);
			(void) fprintf(fp, "%-*s ", 8, aep->sub.user);
			(void) fprintf(fp, "%-*s ", 8, aep->sub.group);
			if (aa->verbose > 1) {
				cc = snprintf(buf, sizeof(buf), "%*llu/%llu=%2lld%%[%c]%c ",
					10,
					(uint64_t)zsize,
					(uint64_t)size,
					((size == 0) ? 0 : (100 * zsize) / size),
					(aep->sub.compression == 0x0) ? ARCHANGEL_LITERAL : aep->sub.compression,
					protections[aep->protection]);
				(void) fprintf(fp, "%*s ", 30, buf);
			} else {
				(void) fprintf(fp, "%*llu ", 10, (uint64_t)size);
			}
			printtime(fp, aep->sub.mtime);
		}
		(void) fprintf(fp, "%s%s", (newname) ? newname : aep->sub.name, term);
	}
	return 1;
}

/* public function to list an archive */
static int
print_entry(FILE *fp, aa_t *aa, aaent_t *aep, int list, char *newname, uint64_t size, uint64_t zsize)
{
	char	lnk[ARCHANGEL_MAXPATHLEN];
	int	cc;

	print_details(fp, aa, aep, list, newname, size, zsize, "");
	switch(aep->sub.mode & S_IFMT) {
	case S_IFLNK:
		if (aep->sub.size < 0 || aep->sub.size >= ARCHANGEL_MAXPATHLEN) {
			warn("print_entry: sanity check failed - dubious length (%lld) for symbolic link \"%s\"", aep->sub.size, aa->name);
			return 0;
		}
		if ((cc = read(aa->fd, lnk, aep->sub.size + 1)) != aep->sub.size + 1) {
			warn("print_entry: short symlink read \"%s\"", aa->name);
			return 0;
		}
		lnk[cc] = 0x0;
		if (list && aa->verbose) {
			(void) fprintf(fp, " -> %s", lnk);
		}
		break;
	case S_IFREG:
		if (!extractfile(aa, aep, NULL)) {
			warnx("print_entry: short stdin read \"%s\"", aa->name);
		}
		break;
	case S_IFBLK:
	case S_IFCHR:
	case S_IFDIR:
		break;
	default:
		warnx("print_entry: unrecognised type %o", (aep->sub.mode & S_IFMT));
	}
	if (list) {
		(void) fputc('\n', fp);
	}
	return 1;
}

#ifdef HAVE_GETXATTR
static int
ext_attr_get(aa_t *aa, aaent_t *aep, char *path)
{
	char	 keys[KB(4)];
	char	 attrs[KB(4)];
	char	*key;
	char	 val[BUFSIZ];
	int	 ac;
	int	 cc;
	int	 vc;

	if ((cc = listxattr(path, keys, sizeof(keys))) > 0) {
		for (ac = 0, key = keys ; (int)(key - keys) < cc ; key += strlen(key)) {
			if ((vc = getxattr(path, key, val, sizeof(val))) < 0) {
				warnx("file \"%s\" attribute \"%s\" is too long, skipping...", path, key);
			} else {
				ac += snprintf(&attrs[ac], sizeof(attrs) - ac, "%s=%s\n", key, val);
			}
		}
		aep->sub.xattrlen = ac;
		aep->sub.xattr = strnsave(attrs, ac);
	}
	return 1;
}
#endif

/* add an entry to aa->fd */
static int
addent(aa_t *aa, char *name, int namelen, struct stat *sp, int64_t from)
{
	struct passwd	*pwp;
	struct group	*grp;
	aaent_t		 a;
	int64_t		 size;
	char		 zname[ARCHANGEL_MAXPATHLEN];
	char		 randy[ARCHANGEL_MAXPATHLEN];
	char		 sym[ARCHANGEL_MAXPATHLEN];
	int		 fd;
	int		 cc;

	if (sp->st_dev == aa->dev && sp->st_ino == aa->inode) {
		warnx("ignoring attempt to add \"%s\" recursively", name);
		return 1;
	}
	if ((a.sub.padding = (aa->maxpadding == 0) ? 0 : random() % MIN(sizeof(sym), aa->maxpadding)) > 0) {
		pad(randy, a.sub.padding);
	}
	(void) memset(&a, 0x0, sizeof(a));
	a.magic1 = ARCHANGEL_MAGIC1;
	a.magic2 = ARCHANGEL_MAGIC2;
	a.sub.compression = ARCHANGEL_LITERAL;
	a.sub.ino = sp->st_ino;
	a.sub.mode = sp->st_mode;
	a.sub.dev = sp->st_dev;
	a.sub.nlink = sp->st_nlink;
	(void) strlcpy(a.sub.user, (pwp = getpwuid(sp->st_uid)) ? pwp->pw_name : "nobody", sizeof(a.sub.user));
	(void) strlcpy(a.sub.group, (grp = getgrgid(sp->st_gid)) ? grp->gr_name : "nobody", sizeof(a.sub.group));
	a.sub.mtime = sp->st_mtime;
	a.sub.atime = sp->st_atime;
	a.sub.ctime = sp->st_ctime;
#ifdef HAVE_STRUCT_STAT_ST_BIRTHTIME
	a.sub.birth = sp->st_birthtime;
#endif
	a.sub.size = sp->st_size;
	a.sub.blocks = sp->st_blocks;
	a.sub.flags = sp->st_flags;
	a.sub.offset = from;
	a.sub.part = from / aa->maxfsize;
	a.sub.lastpart = (sp->st_size - from) < aa->maxfsize;
	a.sub.zsize = 0;
	a.sub.name = name;
	a.sub.namelen = namelen;
#ifdef HAVE_GETXATTR
	if (a.sub.lastpart) {
		ext_attr_get(aa, &a, name);
	}
#endif
	switch (sp->st_mode & S_IFMT) {
	case S_IFREG:
		a.sub.type = ARCHANGEL_FILE;
		if ((fd = open(name, O_RDONLY)) < 0) {
			warn("addent: can't read \"%s\"", name);
			return 0;
		}
		if (lseek(fd, from, SEEK_SET) < 0) {
			warn("addent: can't lseek \"%s\" to %lld", name, (long long) from);
			return 0;
		}
		size = MIN(a.sub.size - from, aa->maxfsize);
		switch (a.protection = aa->protection) {
		case ARCHANGEL_CLEAR:
			switch (a.sub.compression = find_best(aa, &a, name, fd, from)) {
			case ARCHANGEL_ZLIB:
				if (!gzip_compress_file(aa, fd, name, zname, sizeof(zname), size, &a.sub.zsize)) {
					warnx("addent: can't gzip \"%s\"", name);
					return 0;
				}
				break;
			case ARCHANGEL_BZLIB:
				if (!bzip2_compress_file(aa, fd, name, zname, sizeof(zname), size, &a.sub.zsize)) {
					warnx("addent: can't bzip2 \"%s\"", name);
					return 0;
				}
				break;
			case ARCHANGEL_LITERAL:
				if (!no_compress_file(aa, fd, name, zname, sizeof(zname), size, &a.sub.zsize)) {
					warnx("addent: can't gzip \"%s\"", name);
					return 0;
				}
				break;
			default:
				warnx("addent: weird compression chosen");
				return 0;
			}
			if (a.sub.zsize > size) {
				printf("WARNING: HEURISTIC FAILED for \"%s\" (%c, zsize %lld, size %lld)\n", name, a.sub.compression, a.sub.zsize, a.sub.size);
				/* compressed file is larger than uncompressed, so just store uncompressed */
				if (lseek(fd, from, SEEK_SET) < 0) {
					warn("addent: can't lseek \"%s\" to %lld", name, (long long) from);
					return 0;
				}
				a.sub.compression = ARCHANGEL_LITERAL;
				if (!no_compress_file(aa, fd, name, zname, sizeof(zname), size, &a.sub.zsize)) {
					warnx("addent: can't gzip \"%s\"", name);
					return 0;
				}
			}
			break;
		default:
			if (!sign_file(aa, fd, name, zname, sizeof(zname), size, &a.sub.zsize, a.protection)) {
				warnx("addent: can't sign \"%s\"", name);
				return 0;
			}
			break;
		}
		(void) close(fd);
		a.sub.size = MIN(sp->st_size - from, aa->maxfsize);
		a.next = sizeof(a) + (namelen + 1) + a.sub.zsize + a.sub.padding;
		break;
	case S_IFLNK:
		a.sub.type = ARCHANGEL_SYMLINK;
		if ((cc = readlink(name, sym, sizeof(sym))) < 0) {
			warn("addent: can't readlink \"%s\"", name);
			return 0;
		}
		sym[cc] = 0x0;
		a.next = sizeof(a) + (namelen + 1) + (cc + 1);
		break;
	case S_IFBLK:
		a.next = sizeof(a) + (namelen + 1);
		a.sub.type = ARCHANGEL_BLOCK_SPECIAL;
		break;
	case S_IFCHR:
		a.next = sizeof(a) + (namelen + 1);
		a.sub.type = ARCHANGEL_CHAR_SPECIAL;
		break;
	case S_IFDIR:
		a.next = sizeof(a) + (namelen + 1);
		a.sub.type = ARCHANGEL_DIR;
		break;
	default:
		a.next = sizeof(a) + (namelen + 1);
		break;
	}
	if (!writeheader(aa, &a)) {
		warnx("addent: short header write \"%s\"", name);
		return 0;
	}
	if (write(aa->fd, name, namelen + 1) != namelen + 1) {
		warn("addent: short name write \"%s\"", name);
		return 0;
	}
	if (aa->verbose) {
		printf("%s\n", name);
	}
	switch(sp->st_mode & S_IFMT) {
	case S_IFREG:
		if (!readfile(aa, &a, zname, a.sub.zsize)) {
			warnx("addent: short reg file write \"%s\"", name);
			return 0;
		}
		if (a.sub.padding > 0 && write(aa->fd, randy, a.sub.padding) != a.sub.padding) {
			warn("addent: short padding write \"%s\"", name);
			return 0;
		}
		(void) unlink(zname);
		aa->off += a.next;
		break;
	case S_IFBLK:
	case S_IFCHR:
	case S_IFDIR:
		break;
	case S_IFLNK:
		if (write(aa->fd, sym, cc + 1) != cc + 1) {
			warn("addent: short symlink write \"%s\"", name);
			return 0;
		}
		break;
	default:
		warnx("addent: type %o not recognised for \"%s\"", (sp->st_mode & S_IFMT), name);
		return 0;
	}
	if (aa->logfile) {
		print_details(aa->logfile, aa, &a, 1, name, a.sub.size, a.sub.zsize, "\n");
	}
	return 1;
}

/* return 1 if the file is not wanted */
static int
exclude(strv_t *exclv, char *fullname, char *d_name)
{
	int	i;

	for (i = 0 ; i < exclv->c ; i++) {
		if (fnmatch(exclv->v[i], d_name, FNM_PATHNAME) == 0 ||
		    fnmatch(exclv->v[i], fullname, FNM_PATHNAME) == 0) {
			return 1;
		}
	}
	return 0;
}

/* compile the xforms into regular expressions */
static int
compile_xforms(aa_t *aa)
{
	regex_t	*rv;
	int	 ret;
	int	 i;

	ret = 1;
	if (aa->xformfrom.c > 0) {
		NEWARRAY(regex_t, rv, aa->xformfrom.c, "compile_xforms", return 0);
		aa->cxform = rv;
		for (i = 0 ; i < aa->xformfrom.c ; i++) {
			if (regcomp(&rv[i], aa->xformfrom.v[i], REG_EXTENDED) != 0) {
				warn("copile_xforms: can't compile regular expression \"%s\"", aa->xformfrom.v[i]);
				ret = 0;
			}
		}
	}
	return ret;
}

/* substitute in a regular expression */
static void
regsub(regmatch_t *matchv, char *oldname, char *sub, char *newname, int size)
{
	char	*from;
	char	*to;
	int	 n;
	int	 i;

	for (from = sub, to = newname ; *from && (int)(to - newname) < size - 1 ; ) {
		switch(*from) {
		case '\\':
			if (isdigit((unsigned)*(from + 1))) {
				n = "0123456789"[(unsigned)*(from + 1)];
				for (i = matchv[n].rm_so ; i < matchv[n].rm_eo && (int)(to - newname) < size - 1 ; i++) {
					*to++ = oldname[i];
				}
			} else {
				*to++ = *(from + 1);
			}
			from += 2;
			break;
		case '&':
			for (i = matchv[0].rm_so ; i < matchv[0].rm_eo && (int)(to - newname) < size - 1 ; i++) {
				*to++ = oldname[i];
			}
			*from += 1;
			break;
		default:
			*to++ = *from++;
		}
	}
	*to = 0x0;
}

/* transform the name based on any substitutions */
static char *
transform_name(aa_t *aa, char *name, int size, const char *fmt, ...)
{
	regmatch_t	 matchv[REGMATCHMAX];
	regex_t		*rv;
	va_list		 vp;
	char		 old[ARCHANGEL_MAXPATHLEN];
	int	 	 i;

	va_start(vp, fmt);
	(void) vsnprintf(name, size, fmt, vp);
	va_end(vp);
	if (aa->xformfrom.c > 0 && aa->cxform == NULL) {
		compile_xforms(aa);
	}
	rv = aa->cxform;
	for (i = 0 ; i < aa->xformfrom.c ; i++) {
		if (regexec(&rv[i], name, REGMATCHMAX, matchv, 0) == 0) {
			(void) strlcpy(old, name, sizeof(old));
			regsub(matchv, old, aa->xformto.v[i], name, size);
		}
	}
	return name;
}

/* add a directory to aa->fd */
static int
adddir(aa_t *aa, char *d)
{
	struct dirent	*dp;
	struct stat	 st;
	int64_t		 n;
	char		 name[ARCHANGEL_MAXPATHLEN];
	DIR		*dirp;
	int		 done;
	int		 ret;

	if ((dirp = opendir(d)) == NULL) {
		warn("adddir: can't opendir \"%s\"", d);
		return 0;
	}
	for (ret = 1 ; (dp = readdir(dirp)) != NULL ; ) {
		if (strcmp(dp->d_name, ".") == 0 || strcmp(dp->d_name, "..") == 0) {
			continue;
		}
		if (transform_name(aa, name, sizeof(name), "%s/%s", d, dp->d_name) == NULL) {
			continue;
		}
		if (exclude(&aa->excludes, name, dp->d_name)) {
			continue;
		}
		if (lstat(name, &st) < 0) {
			warn("adddir: can't stat \"%s\"", name);
			ret = 0;
			continue;
		}
		for (done = 0, n = 0 ; !done && n < st.st_size ; n += aa->maxfsize) {
			if (!addent(aa, name, strlen(name), &st, n)) {
				warnx("adddir: can't add \"%s\"", name);
				ret = 0;
				done = 1;
				break;
			}
		}
		switch(st.st_mode & S_IFMT) {
		case S_IFDIR:
			if (!adddir(aa, name)) {
				ret = 0;
			}
			break;
		case S_IFREG:
		case S_IFCHR:
		case S_IFBLK:
		case S_IFLNK:
		case S_IFIFO:
			break;
		default:
			break;
		}
		if (aa->verbose) {
		}
	}
	(void) closedir(dirp);
	return ret;
}

/* return 1 if name is wanted (in the non-null arg list) */
static int
wanted(char *name, int argc, char **argv)
{
	int	i;

	if (argc == 0) {
		return 1;
	}
	for (i = 0 ; i < argc ; i++) {
		if (fnmatch(argv[i], name, FNM_PATHNAME) == 0) {
			return 1;
		}
	}
	return 0;
}

/* check to see if we have a -C argument rather than a filename */
static int
arg_is_directory(char *buf)
{
	char	*cp;

	for (cp = buf ; *cp && isspace((unsigned)*cp) ; cp++) {
	}
	if (strncmp(cp, "-C", 2) == 0) {
		for (cp += 2 ; *cp && isspace((unsigned)*cp) ; cp++) {
		}
		if (chdir(cp) < 0) {
			warn("can't chdir to \"%s\"", cp);
		}
		return 1;
	}
	return 0;
}

/* escaped strchr() - ignore any occurrences of 'c' preceded by '\\' */
static const char *
estrchr(const char *s, int c)
{
	const char	*cp;

	for (cp = s ; *cp && *cp != c ; cp++) {
		if (*cp == '\\') {
			cp += 1;
		}
	}
	return (*cp == c) ? cp : NULL;
}

/**************************************************************************/
/* public functions start here */
/**************************************************************************/

/* save `n' chars of `s' in malloc'd memory */
char *
strnsave(const char *s, ssize_t n)
{
	char	*cp;

	if (n < 0) {
		n = strlen(s);
	}
	NEWARRAY(char, cp, n + 1, "strnsave", exit(EXIT_FAILURE));
	(void) memcpy(cp, s, n);
	cp[n] = 0x0;
	return cp;
}

/* public function to add an entry to the archive */
int
archangel_add_entry(aa_t *aa, char *name, size_t namelen)
{
	struct stat	st;
	int64_t		n;

	if (lstat(name, &st) < 0) {
		warn("archangel_add_entry: can't find \"%s\"", name);
		return 0;
	}
	if ((st.st_mode & S_IFMT) == S_IFDIR) {
		if (!adddir(aa, name)) {
			return 0;
		}
	} else if (!exclude(&aa->excludes, name, name)) {
		for (n = 0 ; n < st.st_size ; n += aa->maxfsize) {
			if (!addent(aa, name, namelen, &st, n)) {
				return 0;
			}
		}
	}
	return 1;
}

/* public function to list an archive */
int
archangel_list_entry(aa_t *aa, aaent_t *aep, int list, char *newname)
{
	return print_entry(stdout, aa, aep, list, newname, aep->sub.size, aep->sub.zsize);
}

/* public function to list an entry in the archive */
int
archangel_list(aa_t *aa, int argc, char **argv)
{
	uint64_t	zsize;
	uint64_t	size;
	aaent_t		a;
	char		newname[ARCHANGEL_MAXPATHLEN];
	char		name[ARCHANGEL_MAXPATHLEN];
	int		doit;
	int		ret;
	int		cc;

	(void) memset(&a, 0x0, sizeof(a));
	aa->off = 0; 
	size = zsize = 0;
	ret = 1;
	for (;;) {
		if (!readheader(aa, &a)) {
			return ret;
		}
		if (!sanity_checks(aa, &a)) {
			warnx("sanity checks failed for \"%s\"", aa->name);
			return 0;
		}
		aa->off += sizeof(a);
		if (a.magic1 != ARCHANGEL_MAGIC1 || a.magic2 != ARCHANGEL_MAGIC2) {
			(void) warnx("archangel_list: bad magic \"%s\" archive has (%x,%x), expecting (%x,%x) at offset %lld\n",
					aa->name, a.magic1, a.magic2, ARCHANGEL_MAGIC1, ARCHANGEL_MAGIC2,
					aa->off);
			return 0;
		}
		if ((cc = read(aa->fd, a.sub.name = name, a.sub.namelen + 1)) != a.sub.namelen + 1) {
			warn("archangel_list: short name read \"%s\"\n", aa->name);
			return 0;
		}
		aa->off += a.sub.namelen + 1;
		doit = (transform_name(aa, newname, sizeof(newname), "%s", name) == NULL) ? 0 : wanted(newname, argc, argv);
		size += a.sub.size;
		zsize += a.sub.zsize;
		if (!print_entry(stdout, aa, &a, a.sub.lastpart, newname, size, zsize)) {
			ret = 0;
		}
		if (a.sub.lastpart) {
			size = zsize = 0;
		}
	}
}

/* public function to extract the next entry from the archive */
int
archangel_extract_entry(aa_t *aa, aaent_t *aep, int doit, char *newname)
{
	char	sym[ARCHANGEL_MAXPATHLEN];
	int	ret;
	int	cc;

	ret = 1;
	if (doit && aa->verbose) {
		printf("%s\n", (newname) ? newname : aep->sub.name);
	}
	switch(aep->sub.mode & S_IFMT) {
	case S_IFLNK:
		if ((cc = read(aa->fd, sym, aep->sub.size + 1)) != aep->sub.size + 1) {
			warn("archangel_extract_entry: short symlink read \"%s\"", aa->name);
			return 0;
		}
		if (doit && symlink(sym, (newname) ? newname : aep->sub.name) < 0) {
			warn("archangel_extract_entry: can't symlink \"%s\"", (newname) ? newname : aep->sub.name);
			ret = 0;
		}
		break;
	case S_IFDIR:
		if (doit && mkdir((newname) ? newname : aep->sub.name, aep->sub.mode) < 0) {
			warn("archangel_extract_entry: can't mkdir \"%s\"", (newname) ? newname : aep->sub.name);
			ret = 0;
		}
		break;
	case S_IFREG:
		if (!extractfile(aa, aep, (doit) ? (newname) ? newname : aep->sub.name : NULL)) {
			warnx("archangel_extract_entry: can't extract \"%s\"", (newname) ? newname : aep->sub.name);
			ret = 0;
		}
		break;
	case S_IFBLK:
	case S_IFCHR:
		if (doit && mknod((newname) ? newname : aep->sub.name, aep->sub.mode, aep->sub.dev) < 0) {
			warn("archangel_extract_entry: can't mknod \"%s\"", (newname) ? newname : aep->sub.name);
			ret = 0;
		}
		break;
	default:
		warnx("archangel_extract_entry: unrecognised type %o", (aep->sub.mode & S_IFMT));
	}
	aa->off += aep->sub.size;
	return ret;
}

/* public function to extract an entry from the archive */
int
archangel_extract(aa_t *aa, int argc, char **argv)
{
	aaent_t	a;
	char	newname[ARCHANGEL_MAXPATHLEN];
	char	name[ARCHANGEL_MAXPATHLEN];
	int	doit;
	int	ret;
	int	cc;

	(void) memset(&a, 0x0, sizeof(a));
	aa->off = 0; 
	ret = 1;
	for (;;) {
		if (!readheader(aa, &a)) {
			return ret;
		}
		aa->off += sizeof(a);
		if (a.magic1 != ARCHANGEL_MAGIC1 || a.magic2 != ARCHANGEL_MAGIC2) {
			warnx("archangel_extract: bad magic \"%s\"", aa->name);
			return 0;
		}
		if ((cc = read(aa->fd, a.sub.name = name, a.sub.namelen + 1)) != a.sub.namelen + 1) {
			warn("archangel_extract: short name read \"%s\"", aa->name);
			return 0;
		}
		aa->off += a.sub.namelen + 1;
		doit = (transform_name(aa, newname, sizeof(newname), "%s", name) == NULL) ? 0 : wanted(newname, argc, argv);
		if (!archangel_extract_entry(aa, &a, doit, newname)) {
			ret = 0;
		}
	}
}

/* public function to add to the archive */
int
archangel_add(aa_t *aa, int argc, char **argv)
{
	int	ret;
	int	i;

	for (ret = 1, i = 0 ; i < argc ; i++) {
		if (arg_is_directory(argv[i])) {
			continue;
		}
		if (!archangel_add_entry(aa, argv[i], strlen(argv[i]))) {
			ret = 0;
		}
	}
	return ret;
}

/* open the archive */
int
archangel_open(aa_t *aa, char *name, int action, uint64_t maxfsize, int verbose)
{
	struct stat	st;
	int		mode;

	aa->verbose = verbose;
	if (name == NULL || strcmp(name, "-") == 0) {
		aa->fd = (action == ARCHANGEL_ADD || action == ARCHANGEL_APPEND) ? STDOUT_FILENO : STDIN_FILENO;
	} else {
		if (action == ARCHANGEL_LIST || action == ARCHANGEL_EXTRACT) {
			mode = O_RDONLY;
			if (stat(name, &st) < 0) {
				warn("can't find \"%s\" archive", name);
				return 0;
			}
		} else if (action == ARCHANGEL_ADD || action == ARCHANGEL_APPEND) {
			mode = O_RDWR;
			if (stat(name, &st) < 0) {
				mode |= O_CREAT;
			} else if (action == ARCHANGEL_ADD) {
				mode |= O_TRUNC;
			}
		}
		if ((aa->fd = open(name, mode, 0666)) < 0) {
			warn("can't open \"%s\" archive", name);
			return 0;
		}
		(void) fstat(aa->fd, &st);
		aa->dev = st.st_dev;
		aa->inode = st.st_ino;
		if (action == ARCHANGEL_APPEND) {
			(void) lseek(aa->fd, 0, SEEK_END);
		}
	}
	aa->maxfsize = maxfsize;
	return 1;
}

/* close the archive's file descriptor */
void
archangel_close(aa_t *aa)
{
	(void) close(aa->fd);
}

/* add a pattern to the transform or exclude arrays */
int
archangel_add_pattern(aa_t *aa, const char *type, char *pat)
{
	const char	*term1;
	const char	*term2;

	if (strcmp(type, "exclude") == 0) {
		ALLOC(char *, aa->excludes.v, aa->excludes.size, aa->excludes.c, 30, 30, "archangel_add_pattern excludes", return 0);
		aa->excludes.v[aa->excludes.c++] = strdup(pat);
		return 1;
	}
	if (strcmp(type, "transform") == 0) {
		ALLOC(char *, aa->xformfrom.v, aa->xformfrom.size, aa->xformfrom.c, 30, 30, "archangel_add_pattern transforms", return 0);
		ALLOC(char *, aa->xformto.v, aa->xformto.size, aa->xformto.c, 30, 30, "archangel_add_pattern transforms", return 0);
		if ((term1 = estrchr(pat + 1, *pat)) == NULL) {
			warn("archangel_add_pattern: malformed tranformation pattern: \"%s\"", pat);
			return 0;
		}
		if ((term2 = estrchr(term1 + 1, *pat)) == NULL) {
			term2 = estrchr(term1 + 1, 0x0);
		}
		aa->xformfrom.v[aa->xformfrom.c++] = strnsave(pat + 1, (int)(term1 - pat) - 1);
		aa->xformto.v[aa->xformto.c++] = strnsave(term1 + 1, (int)(term2 - term1) - 1);
	}
	return 1;
}

/* initialise the archangel structure */
void
archangel_init(aa_t *aa)
{
	(void) memset(aa, 0x0, sizeof(*aa));
	aa->protection = ARCHANGEL_CLEAR;
}
