#include <sys/types.h> 
#include <sys/param.h> 

#include <stdint.h> 
#include <stdio.h> 
#include <string.h> 
#include <stdlib.h> 
#include <unistd.h> 

#include "critbit.h"

/* private structure for a node */
typedef struct node_t {
	void		*child[2];	/* ptr to child if node is internal; otherwise string */
	uint32_t	 critbitpos;	/* the crit-bit position */
	uint8_t		 otherbits;	/* other bits used in calculation - direction etc */
} node_t;

/* used locally here for searching */
typedef struct search_t {
	node_t		 *node;		/* node pointer */
	uint32_t	  direction;	/* direction - left or right */
	void		**wherep;	/* address of p */
	void		**parent;	/* address of parent of p */
	uint8_t		 *top;		/* new top of tree */
} search_t;

#define IS_INTERNAL_NODE(z)	((intptr_t)z & 1)
#define INTERNAL_NODE(z)	((void *)(z - 1))
#define TO_INTERNAL_NODE(z)	((void *)(1 + (char *)z))

/* recursive function to traverse all nodes */
static void
clearnode(void *top)
{
	uint8_t		*p;
	node_t		*node;

	p = top;
	if (IS_INTERNAL_NODE(p)) {
		node = INTERNAL_NODE(p);
		clearnode(node->child[0]);
		clearnode(node->child[1]);
		free(node);
	} else {
		free(p);
	}

}

/* traverse the tree, running func for every node which has the prefix */
static int
allprefixed_traverse(uint8_t *top, int (*func)(const char *, void *, size_t), void *arg, size_t namelen)
{
	unsigned	 direction;
	node_t		*node;

	if (IS_INTERNAL_NODE(top)) {
		node = INTERNAL_NODE(top);
		for (direction = 0 ; direction < 2 ; direction++) {
			switch(allprefixed_traverse(node->child[direction], func, arg, namelen)) {
			case 1:
				break;
			case 0:
				return 0;
			default:
				return -1;
			}
		}
		return 1;
	}
	return (*func)((const char *)top, arg, namelen);
}

/* find the entry */
static inline uint8_t *
findentry(search_t *search, uint8_t *p, const uint8_t *name, size_t namelen)
{
	uint8_t		 ch;

	memset(search, 0x0, sizeof(*search));
	search->wherep = (void **)&p;
	while (IS_INTERNAL_NODE(p)) {
		search->parent = search->wherep;
		search->node = INTERNAL_NODE(p);
		ch = (search->node->critbitpos < namelen) ? name[search->node->critbitpos] : 0x0;
		search->direction = (1U + (search->node->otherbits | ch)) >> 8;
		search->wherep = search->node->child + search->direction;
		p = *search->wherep;
		if (search->node->critbitpos < namelen) {
			search->top = p;
		}
	}
	return p;
}

/* allocate the string */
static inline char *
allocate(const void *str, size_t namelen)
{
	size_t	 size;
	char	*mem;

	size = sizeof(namelen) + namelen + 1;
	if (posix_memalign((void **)&mem, sizeof(void *), size) != 0) {
		return NULL;
	}
	memcpy(mem, &namelen, sizeof(namelen));
	memcpy(&mem[sizeof(namelen)], str, namelen);
	mem[sizeof(namelen) + namelen] = 0x0;
	return mem;
}

/* small function to total number of entries */
static inline int
total(const char *p, void *arg, size_t namelen)
{
	int	*count = (int *)arg;

	*count += 1;
	return 1;
}

/*************************************************************************/

/* return non-zero if string 'str' is present in the tree */
int
critbit_contains(critbit_t *t, const void *str, size_t namelen)
{
	search_t	 search;
	uint8_t		*p;

	if (t == NULL || str == NULL || t->root == NULL) {
		return 0;
	}
	p = findentry(&search, t->root, (const uint8_t *)str, namelen);
	if (memcmp(str, &p[sizeof(namelen)], namelen) != 0 || p[sizeof(namelen) + namelen] != 0) {
		return 0;
	}
	return 1;
}

/* insert string 'str' in the tree */
int
critbit_insert(critbit_t *t, const void *key, size_t keylen)
{
	const uint8_t	 *name;
	uint32_t	  newcritbitpos;
	uint32_t	  newotherbits;
	uint32_t	  newdirection;
	search_t	  search;
	uint32_t	  direction;
	uint8_t		 *p;
	uint8_t		  ch;
	node_t		 *newnode;
	node_t		 *node;
	void		**wherep;
	char		 *mem;

	if (t == NULL || key == NULL) {
		return 0;
	}
	name = (const uint8_t *)key;
	if ((p = t->root) == NULL) {
		if ((t->root = allocate(key, keylen)) == NULL) {
			return 0;
		}
		return 2;
	}
	newotherbits = 0;
	p = findentry(&search, t->root, (const uint8_t *)key, keylen);
	for (newcritbitpos = 0 ; newcritbitpos < keylen ; newcritbitpos++) {
		if (p[sizeof(keylen) + newcritbitpos] != name[newcritbitpos]) {
			newotherbits = p[sizeof(keylen) + newcritbitpos] ^ name[newcritbitpos];
			break;
		}
	}
	if (newcritbitpos == keylen && (newotherbits = p[sizeof(keylen) + newcritbitpos]) == 0) {
		return 1;
	}
	newotherbits |= newotherbits >> 1;
	newotherbits |= newotherbits >> 2;
	newotherbits |= newotherbits >> 4;
	newotherbits = (newotherbits & ~(newotherbits >> 1)) ^ 0xff;
	ch = p[sizeof(keylen) + newcritbitpos];
	newdirection = (1 + (newotherbits | ch)) >> 8;
	if (posix_memalign((void **)&newnode, sizeof(void *), sizeof(*newnode)) ||
	    (mem = allocate(key, keylen)) == NULL) {
		return 0;
	}
	newnode->critbitpos = newcritbitpos;
	newnode->otherbits = (uint8_t)newotherbits;
	newnode->child[1 - newdirection] = mem;
	wherep = &t->root;
	for (;;) {
		p = *wherep;
		if (!IS_INTERNAL_NODE(p)) {
			break;
		}
		node = INTERNAL_NODE(p);
		if (node->critbitpos > newcritbitpos ||
		    (node->critbitpos == newcritbitpos && node->otherbits > newotherbits)) {
			break;
		}
		ch = (node->critbitpos < keylen) ? name[node->critbitpos] : 0x0;
		direction = (1U + (node->otherbits | ch)) >> 8;
		wherep = node->child + direction;
	}
	newnode->child[newdirection] = *wherep;
	*wherep = TO_INTERNAL_NODE(newnode);
	return 2;
}

/* delete key 'str' from the tree */
int
critbit_delete(critbit_t *t, const void *str, size_t namelen)
{
	search_t	  search;
	uint8_t		 *p;

	if (t == NULL || str == NULL || (p = t->root) == NULL) {
		return 0;
	}
	if ((p = findentry(&search, t->root, (const uint8_t *)str, namelen)) == NULL ||
	    memcmp(str, &p[sizeof(namelen)], namelen) != 0) {
		return 0;
	}
	free(p);
	if (search.parent) {
		*search.parent = search.node->child[1 - search.direction];
		free(search.node);
	} else {
		t->root = NULL;
	}
	return 1;
}

/* clear the root of the tree */
void
critbit_clear(critbit_t *t)
{
	if (t) {
		if (t->root) {
			clearnode(t->root);
		}
		t->root = NULL;
	}
}

/* run the func 'func' for all nodes with the given prefix (passed as arg)*/
int
critbit_allprefixed(critbit_t *t, const void *prefix, size_t namelen, int (*func)(const char *, void *, size_t), void *arg)
{
	search_t	 search;
	uint8_t		*p;

	if (t == NULL || prefix == NULL || func == NULL || t->root == NULL) {
		return 1;
	}
	if ((p = findentry(&search, t->root, prefix, namelen)) == NULL ||
	    memcmp(&p[sizeof(namelen)], prefix, namelen) != 0) {
		return 1;
	}
	return allprefixed_traverse(search.top, func, arg, namelen);
}

/* return the number of elements in the tree */
int
critbit_length(critbit_t *tree)
{
	int	count;

	if (tree == NULL) {
		return 0;
	}
	count = 0;
	allprefixed_traverse(tree->root, total, &count, 0);
	return count;
}
