/*	$NetBSD: ww.h,v 1.17 2003/08/07 11:17:35 agc Exp $	*/

/*
 * Copyright (c) 1983, 1993
 *	The Regents of the University of California.  All rights reserved.
 *
 * This code is derived from software contributed to Berkeley by
 * Edward Wang at The University of California, Berkeley.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 *	@(#)ww.h	8.1 (Berkeley) 6/6/93
 */

#ifndef __WW_H__
#define __WW_H__

#include <sys/types.h>
#include <sys/time.h>

#ifdef OLD_TTY
#include <sgtty.h>
#else
#include <termios.h>
#endif
#include <inttypes.h>
#include <setjmp.h>
#include <stdarg.h>
#include <stdio.h>
#include <unistd.h>

#ifndef __UNCONST
#define __UNCONST(a) ((void *)(unsigned long)(const void *)(a))
#endif /* !__UNCONST */

#define MAX_NUM_WINDOWS	254		/* maximum number of windows */
#define MAX_SHELL_ARGC	128
#define TTY_NAME_LENGTH	15		/* /dev/pty* or /dev/pts/ * */

/* Macros to clear/set/test flags. */
#define	SET(t, f)	(t) |= (f)
#define	CLR(t, f)	(t) &= ~(f)
#define	ISSET(t, f)	((t) & (f))

	/* a rectangle */
typedef struct ww_dim_t {
	int nr;			/* number of rows */
	int nc;			/* number of columns */
	int t;			/* top, bottom */
	int b;			/* top, bottom */
	int l;			/* left, right */
	int r;			/* left, right */
} ww_dim_t;

	/* a coordinate */
typedef struct ww_pos_t {
	int r;			/* row */
	int c;			/* column */
} ww_pos_t;

	/* the window structure */
typedef struct ww_t {
	int ww_flags;

		/* general flags and states */
	int ww_state;		/* state of window */
#define WWS_INITIAL	0	/* just opened */
#define WWS_HASPROC	1	/* has process on pty */
#define WWS_DEAD	3	/* child died */
#define	ww_oflags	ww_flags
#define WWO_REVERSE	0x0001	/* make it all reverse video */
#define WWO_GLASS	0x0002	/* make it all glass */
#define WWO_FRAME	0x0004	/* this is a frame window */
#define	WWO_ALLFLAGS	0x0007

		/* information for overlap */
	struct ww_t *ww_forw;	/* doubly linked list, for overlapping info */
	struct ww_t *ww_back;
	unsigned char ww_index;	/* the window index, for wwindex[] */
#define WWX_NOBODY	MAX_NUM_WINDOWS
	int ww_order;		/* the overlapping order */

		/* sizes and positions */
	ww_dim_t	 ww_w;	/* window size and pos */
	ww_dim_t	 ww_b;	/* buffer size and pos */
	ww_dim_t	 ww_i;	/* the part inside the screen */
	ww_pos_t	 ww_cur;	/* the cursor position, relative to ww_w */

		/* arrays */
	char **ww_win;		/* the window */
	union ww_char **ww_buf;	/* the buffer */
	char **ww_fmap;		/* map for frame and box windows */
	short *ww_nvis;		/* how many ww_buf chars are visible per row */

		/* information for wwwrite() and company */
	int ww_wstate;		/* state for outputting characters */
	char ww_modes;		/* current display modes */
#define	ww_wflags	ww_flags
#define	WWW_INSERT	0x0008	/* insert mode */
#define	WWW_MAPNL	0x0010	/* map \n to \r\n */
#define	WWW_NOUPDATE	0x0020	/* don't do updates in wwwrite() */
#define	WWW_UNCTRL	0x0040	/* expand control characters */
#define	WWW_NOINTR	0x0080	/* wwwrite() not interruptable */
#define	WWW_HASCURSOR	0x0100	/* has fake cursor */

		/* things for the window process and io */
	int ww_type;
#define	WWT_PTY		0	/* pty */
#define	WWT_SOCKET	1	/* socket pair */
#define	WWT_INTERNAL	2
#define	ww_pflags	ww_flags
#define	WWP_STOPPED	0x0200	/* output stopped */
	int ww_pty;		/* file descriptor of pty or socket pair */
	int ww_socket;		/* other end of socket pair */
	int ww_pid;		/* pid of process, if WWS_HASPROC true */
	char ww_ttyname[TTY_NAME_LENGTH];	/* "/dev/ttyp?" */
	char *ww_ob;		/* output buffer */
	char *ww_obe;		/* end of ww_ob */
	char *ww_obp;		/* current read position in ww_ob */
	char *ww_obq;		/* current write position in ww_ob */

		/* things for the user, they really don't belong here */
	int ww_id;		/* the user window id */
#define	ww_uflags	ww_flags
#define	WWU_CENTER	0x0400	/* center the label */
#define	WWU_HASFRAME	0x0800	/* frame it */
#define	WWU_KEEPOPEN	0x1000	/* keep it open after the process dies */
#define	WWU_ALLFLAGS	0x1c00
	char *ww_label;		/* the user supplied label */
	ww_dim_t	 ww_alt;	/* alternate position and size */
} ww_t;

	/* state of a tty */
typedef struct ww_tty_t {
#ifdef OLD_TTY
	struct sgttyb ww_sgttyb;
	struct tchars ww_tchars;
	struct ltchars ww_ltchars;
	int ww_lmode;
	int ww_ldisc;
#else
	struct termios ww_termios;
#endif
} ww_tty_t;

union ww_char {
	uint16_t	c_w;		/* as a word */
	struct {
#if 0
#if BYTE_ORDER == LITTLE_ENDIAN || BYTE_ORDER == PDP_ENDIAN
		char C_c;	/* the character part */
		char C_m;	/* the mode part */
#endif
#if BYTE_ORDER == BIG_ENDIAN
		char C_m;	/* the mode part */
		char C_c;	/* the character part */
#endif
#else
		char C_c;	/* the character part */
		char C_m;	/* the mode part */
#endif
	} c_un;
};
char getcharpart(union ww_char *);
char setcharpart(union ww_char *, char);
char getmodepart(union ww_char *);
char setmodepart(union ww_char *, char);
short getwhole(union ww_char *);
short setwhole(union ww_char *, short);
short setparts(union ww_char *, char, char);

	/* for display update */
typedef struct ww_update_t {
	int best_gain;
	int best_col;
	int gain;
} ww_update_t;

	/* parts of ww_char */
#define WWC_CMASK	0x00ff
#define WWC_MMASK	0xff00
#define WWC_MSHIFT	8

	/* c_m bits */
#define WWM_REV		0x01	/* reverse video */
#define WWM_BLK		0x02	/* blinking */
#define WWM_UL		0x04	/* underlined */
#define WWM_GRP		0x08	/* graphics */
#define WWM_DIM		0x10	/* half intensity */
#define WWM_USR		0x20	/* user specified mode */
#define WWM_GLS		0x40	/* window only, glass, i.e., transparent */

	/* flags for ww_fmap */
#define WWF_U		0x01
#define WWF_R		0x02
#define WWF_D		0x04
#define WWF_L		0x08
#define WWF_MASK	(WWF_U|WWF_R|WWF_D|WWF_L)
#define WWF_LABEL	0x40
#define WWF_TOP		0x80

	/* error codes */
#define WWE_NOERR	0
#define WWE_SYS		1		/* system error */
#define WWE_NOMEM	2		/* out of memory */
#define WWE_TOOMANY	3		/* too many windows */
#define WWE_NOPTY	4		/* no more ptys */
#define WWE_SIZE	5		/* bad window size */
#define WWE_BADTERM	6		/* bad terminal type */
#define WWE_CANTDO	7		/* dumb terminal */

	/* wwtouched[] bits, there used to be more than one */
#define WWU_TOUCHED	0x01		/* touched */

typedef struct winvars_t {
	/* the window structures */
	ww_t	 wwhead;
	ww_t	 *wwindex[MAX_NUM_WINDOWS + 1];	/* last location is for wwnobody */
	ww_t	 wwnobody;

	/* tty things */
	ww_tty_t	wwoldtty;		/* the old (saved) terminal settings */
	ww_tty_t	wwnewtty;		/* the new (current) terminal settings */
	ww_tty_t	wwwintty;		/* the terminal settings for windows */
	char *wwterm;			/* the terminal name */
	char wwtermcap[1024];		/* place for the termcap */

	/* generally useful variables */
	int wwnrow;			/* the screen size */
	int wwncol;			/* the screen size */
	char wwavailmodes;		/* actually supported modes */
	char wwcursormodes;		/* the modes for the fake cursor */
	char wwwrap;			/* terminal has auto wrap around */
	int wwdtablesize;		/* result of getdtablesize() call */
	unsigned char **wwsmap;		/* the screen map */
	union ww_char **wwos;		/* the old (current) screen */
	union ww_char **wwns;		/* the new (desired) screen */
	union ww_char **wwcs;		/* the checkpointed screen */
	char *wwtouched;		/* wwns changed flags */
	ww_update_t	*wwupd;	/* for display update */
	int wwospeed;			/* output baud rate, copied from wwoldtty */
	int wwbaud;			/* wwospeed converted into actual number */
	int wwcursorrow;		/* where we want the cursor to be */
	int wwcursorcol;		/* where we want the cursor to be */
	int wwerrno;			/* error number */

	/* statistics */
	uint64_t wwnflush;
	uint64_t wwnwr;
	uint64_t wwnwre;
	uint64_t wwnwrz;
	uint64_t wwnwrc;
	uint64_t wwnwwr;
	uint64_t wwnwwra;
	uint64_t wwnwwrc;
	uint64_t wwntokdef;
	uint64_t wwntokuse;
	uint64_t wwntokbad;
	uint64_t wwntoksave;
	uint64_t wwntokc;
	uint64_t wwnupdate;
	uint64_t wwnupdline;
	uint64_t wwnupdmiss;
	uint64_t wwnupdscan;
	uint64_t wwnupdclreol;
	uint64_t wwnupdclreos;
	uint64_t wwnupdclreosmiss;
	uint64_t wwnupdclreosline;
	uint64_t wwnread;
	uint64_t wwnreade;
	uint64_t wwnreadz;
	uint64_t wwnreadc;
	uint64_t wwnreadack;
	uint64_t wwnreadnack;
	uint64_t wwnreadstat;
	uint64_t wwnreadec;
	uint64_t wwnwread;
	uint64_t wwnwreade;
	uint64_t wwnwreadz;
	uint64_t wwnwreadd;
	uint64_t wwnwreadc;
	uint64_t wwnwreadp;
	uint64_t wwnselect;
	uint64_t wwnselecte;
	uint64_t wwnselectz;

	/* things for handling input */
	ww_t	 *wwcurwin;	/* window to copy input into */
	char *wwib;		/* input (keyboard) buffer */
	char *wwibe;		/* wwib + sizeof buffer */
	char *wwibp;		/* current read position in buffer */
	char *wwibq;		/* current write position in buffer */

	/* things for short circuiting wwiomux() */
	char wwintr;		/* interrupting */
	char wwsetjmp;		/* want a longjmp() from wwrint() and wwchild() */
	jmp_buf wwjmpbuf;	/* jmpbuf for above */

	/* definitions */
	struct timeval starttime;

	ww_t	 *window[MAX_NUM_WINDOWS];	/* the windows */
	ww_t	 *selwin;		/* the selected window */
	ww_t	 *lastselwin;		/* the last selected window */
	ww_t	 *cmdwin;		/* the command window */
	ww_t	 *framewin;		/* the window for framing */
	ww_t	 *boxwin;		/* the window for the box */
	ww_t	 *fgwin;		/* the last foreground window */

	char *default_shell[MAX_SHELL_ARGC];	/* default shell argv */
	char *default_shellfile;		/* default shell program */
	int default_nline;		/* default buffer size for new windows */
	int default_smooth;		/* default "smooth" parameter */
	char escapec;			/* the escape character */
	char literalc;			/* the literal character */
	char	screen_compat;		/* non-zero if we're emulating */

	/* flags */
	char quit;			/* quit command issued */
	char terse;			/* terse mode */
	char debug;			/* debug mode */
	char incmd;			/* in command mode */

	/* checkpointing */
	int wwdocheckpoint;

	char wwwintermcap[1024];	/* terminal-specific but window-independent
				   part of the window termcap */
	/* where to put the temporary terminfo directory */
	char wwterminfopath[1024];
} winvars_t;

	/* quicky macros */
#define WWSETCURSOR(wv, r,c) ((wv)->wwcursorrow = (r), (wv)->wwcursorcol = (c))
#define WWCURTOWIN(wv, w)	WWSETCURSOR((wv), (w)->ww_cur.r, (w)->ww_cur.c)
#define WWUNBOX(wp,w)	wwunframe(wp, w)
#define WWCLREOL(wv,w,r,c)	wwclreol1((wv), (w), (r), (c), 0)
#define WWREDRAWWIN(wp,w)	wwredrawwin1((wp), (w), (w)->ww_i.t, (w)->ww_i.b, 0)
#define WWUPDATE(wv)	wwupdate1(wv, 0, (wv)->wwnrow);

	/* things for handling input */
#define WWMASKC(c)	((c) & 0x7f)
#define WWGETC(wv)	(((wv)->wwibp < (wv)->wwibq) ? WWMASKC(*(wv)->wwibp++) : -1)
#define WWPEEKC(wv)	(((wv)->wwibp < (wv)->wwibq) ? WWMASKC(*(wv)->wwibp) : -1)

	/* things for short circuiting wwiomux() */
#define WWINTERRUPT(wv)	(wv)->wwintr
#define WWSETINTR(wv)	do {						\
	(wv)->wwintr = 1;						\
	if ((wv)->wwsetjmp) {						\
		longjmp((wv)->wwjmpbuf, 1);				\
	}								\
} while (/* CONSTCOND */0)
#define WWCLRINTR(wv)	((wv)->wwintr = 0)

#define ISFG(wv, w)		((w)->ww_order <= (wv)->fgwin->ww_order)

	/* the window virtual terminal */
#define WWT_TERM	"window-v2"
#define WWT_TERMCAP	"WW|window-v2|window program version 2:\
	:am:bs:da:db:ms:pt:cr=^M:nl=^J:bl=^G:ta=^I:\
	:cm=\\EY%+ %+ :le=^H:nd=\\EC:up=\\EA:do=\\EB:ho=\\EH:\
	:cd=\\EJ:ce=\\EK:cl=\\EE:me=\\Er^?:"
#define WWT_REV		"se=\\ErA:so=\\EsA:mr=\\EsA:"
#define WWT_BLK		"BE=\\ErB:BS=\\EsB:mb=\\EsB:"
#define WWT_UL		"ue=\\ErD:us=\\EsD:"
#define WWT_GRP		"ae=\\ErH:as=\\EsH:"
#define WWT_DIM		"HE=\\ErP:HS=\\EsP:mh=\\EsP:"
#define WWT_USR		"XE=\\Er`:XS=\\Es`:"
#define WWT_ALDL	"al=\\EL:dl=\\EM:"
#define WWT_IMEI	"im=\\E@:ei=\\EO:ic=:mi:" /* XXX, ic for emacs bug */
#define WWT_IC		"ic=\\EP:"
#define WWT_DC		"dc=\\EN:"

ww_t	 *wwopen(winvars_t *, int, int, int, int, int, int, int);
void	wwadd(winvars_t *, ww_t *, ww_t *);
char  **wwalloc(winvars_t *, int, int, int, int, int);
void	wwbell(void);
void	wwbox(winvars_t *, ww_t *, int, int, int, int);
void	wwchild(int);
void	wwclreol1(winvars_t *, ww_t *, int, int, char);
void	wwclreos(winvars_t *, ww_t *, int, int);
void	wwcursor(winvars_t *, ww_t *, int);
void	wwdelete(winvars_t *, ww_t *);
void	wwdelline(winvars_t *, ww_t *, int);
void	wwdumpns(winvars_t *);
void	wwdumpnvis(ww_t *);
void	wwdumpos(winvars_t *);
void	wwdumpsmap(winvars_t *);
void	wwdumpwin(ww_t *);
void	wwend(int);
const char *wwerror(winvars_t *);
void	wwflush(winvars_t *);
void	wwframe(winvars_t *, ww_t *, ww_t *);
void	wwframec(winvars_t *, ww_t *, int, int, char);
void	wwfree(char **, int);
int	wwgetpty(winvars_t *, ww_t *);
int	wwgettty(winvars_t *, int, ww_tty_t *);
int	wwgetttysize(winvars_t *, int, int *, int *);
void	wwgets(winvars_t *, char *, int, ww_t *);
int	wwinit(winvars_t *);
void	wwinsline(winvars_t *, ww_t *, int);
void	wwiomux(winvars_t *);
void	wwlabel(winvars_t *, ww_t *, ww_t *, int, char *, int);
void	wwmove(winvars_t *, ww_t *, int, int);
void	wwprintf(ww_t *, const char *, ...);
void	wwputc(char, ww_t *);
void	wwputs(winvars_t *, const char *, ww_t *);
void	wwredraw(winvars_t *);
void	wwredrawwin1(winvars_t *, ww_t *,int, int, int);
void	wwquit(int);
void	wwrint(winvars_t *);
void	wwscroll(winvars_t *, ww_t *, int);
int	wwscroll1(winvars_t *, ww_t *, int, int, int, int);
void	wwsetcursormodes(winvars_t *, int);
int	wwsettty(winvars_t *, int, ww_tty_t *);
int	wwsize(winvars_t *, ww_t *, int, int);
int	wwspawn(winvars_t *, ww_t *, char *, char **);
void	wwstart(winvars_t *winvars);
int	wwstarttty(winvars_t *, int);
int	wwstoptty(winvars_t *, int);
void	wwsuspend(winvars_t *);
void	wwunframe(winvars_t *, ww_t *);
void	wwupdate1(winvars_t *, int, int);
int	wwvisible(winvars_t *, ww_t *);
void	wwvprintf(ww_t *, const char *, va_list);
int	wwwrite(winvars_t *, ww_t *, const char *, int);
#ifdef TERMINFO
int	wwterminfoinit(winvars_t *);
int	wwterminfoend(winvars_t *);
#endif

#undef MIN
#undef MAX
#define MIN(x, y)	((x) > (y) ? (y) : (x))
#define MAX(x, y)	((x) > (y) ? (x) : (y))

void save_winvars(winvars_t *);
winvars_t *get_winvars(void);


#endif /* __WW_H__ */
