/*	$NetBSD: wwdelchar.c,v 1.7 2003/08/07 11:17:38 agc Exp $	*/

/*
 * Copyright (c) 1983, 1993
 *	The Regents of the University of California.  All rights reserved.
 *
 * This code is derived from software contributed to Berkeley by
 * Edward Wang at The University of California, Berkeley.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#include <sys/cdefs.h>

#ifndef lint
__RCSID("$NetBSD: wwdelchar.c,v 1.7 2003/08/07 11:17:38 agc Exp $");
#endif /* not lint */

#include <fcntl.h>
#include <signal.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <termcap.h>
#include <unistd.h>

#include "ww.h"
#include "tt.h"
#include "xx.h"
#include "char.h"

static void
wwdelchar(winvars_t *winvars, ww_t *w, int row, int col)
{
	int i;
	int nvis;

	/*
	 * First, shift the line.
	 */
	{
		dispcell_t *p, *q;

		p = &w->ww_buf[row][col];
		q = p + 1;
		for (i = w->ww_b.r - col; --i > 0;) {
			*p++ = *q++;
		}
		setcharpart(p, ' ');
	}

	/*
	 * If can't see it, just return.
	 */
	if (row < w->ww_i.t || row >= w->ww_i.b
	    || w->ww_i.r <= 0 || w->ww_i.r <= col)
		return;

	if (col < w->ww_i.l)
		col = w->ww_i.l;

	/*
	 * Now find out how much is actually changed, and fix wwns.
	 */
	{
		dispcell_t *buf;
		char *win;
		dispcell_t *ns;
		unsigned char *smap;
		char touched;

		nvis = 0;
		smap = &winvars->wwsmap[row][col];
		for (i = col; i < w->ww_i.r && *smap++ != w->ww_index; i++)
			;
		if (i >= w->ww_i.r)
			return;
		col = i;
		buf = w->ww_buf[row];
		win = w->ww_win[row];
		ns = winvars->wwns[row];
		smap = &winvars->wwsmap[row][i];
		touched = winvars->wwtouched[row];
		for (; i < w->ww_i.r; i++) {
			if (*smap++ != w->ww_index)
				continue;
			touched |= WWU_TOUCHED;
			if (win[i]) {
				setparts(&ns[i], getcharpart(&buf[i]), win[i]);
			} else {
				nvis++;
				ns[i] = buf[i];
			}
		}
		winvars->wwtouched[row] = touched;
	}

	/*
	 * Can/Should we use delete character?
	 */
	if (tt.tt_delchar != 0 && nvis > (winvars->wwncol - col) / 2) {
		dispcell_t *p, *q;

		xxdelchar(row, col);
		p = &winvars->wwos[row][col];
		q = p + 1;
		for (i = winvars->wwncol - col; --i > 0;) {
			*p++ = *q++;
		}
		setcharpart(p, ' ');
	}
}

static void
wwcopyscreen(winvars_t *winvars, dispcell_t **s1, dispcell_t **s2)
{
	int i;
	int s;

	s = winvars->wwncol * sizeof **s1;
	for (i = winvars->wwnrow; --i >= 0;)
		memmove(*s2++, *s1++, (unsigned)s);
}

/* ARGSUSED0 */
static void
wwalarm(int dummy __unused)
{
	winvars_t	*winvars;

	winvars = get_winvars();
	winvars->wwdocheckpoint = 1;
}


/* ARGSUSED0 */
void
wwend(int done __unused)
{
	winvars_t	*winvars;

	winvars = get_winvars();
	if (tt.tt_checkpoint) {
		(void) alarm(0);
		winvars->wwdocheckpoint = 0;
	}
	xxend();
	(void) wwsettty(winvars, 0, &winvars->wwoldtty);
#ifdef TERMINFO
	if (done)
		wwterminfoend(winvars);
#endif
}

/* ARGSUSED0 */
void
wwquit(int dummy __unused)
{
	wwend(1);
	exit(EXIT_FAILURE);
}

/*
 * Reset data structures and terminal from an unknown state.
 * Restoring wwos has been taken care of elsewhere.
 */
static void
wwreset(winvars_t *winvars)
{
	int i;

	xxreset();
	for (i = 0; i < winvars->wwnrow; i++)
		winvars->wwtouched[i] = WWU_TOUCHED;
}

static void
wwcheckpoint(winvars_t *winvars)
{
	sigset_t nsigset, osigset;

	sigemptyset(&nsigset);
	sigaddset(&nsigset, SIGALRM);
	sigprocmask(SIG_BLOCK, &nsigset, &osigset);

	tt.tt_ack = 0;
	do {
		(*tt.tt_checkpoint)();
#ifndef OLD_TTY
		(void) tcdrain(1);
#endif
		(void) alarm(3);
		for (winvars->wwdocheckpoint = 0; !winvars->wwdocheckpoint && tt.tt_ack == 0;)
			sigsuspend(&osigset);
	} while (tt.tt_ack == 0);
	(void) alarm(0);
	winvars->wwdocheckpoint = 0;
	if (tt.tt_ack < 0) {
		wwcopyscreen(winvars, winvars->wwcs, winvars->wwos);
		(void) alarm(1);
		wwreset(winvars);
		WWUPDATE(winvars);
		wwflush(winvars);
	} else {
		wwcopyscreen(winvars, winvars->wwos, winvars->wwcs);
		(void) alarm(3);
	}

	sigprocmask(SIG_SETMASK, &osigset, NULL);
}

static void
wwaddcap(const char *cap, char **kp)
{
	char tbuf[512];
	char *tp = tbuf;
	char *str, *p;

	if ((str = tgetstr(cap, &tp)) != 0) {
		while ((*(*kp)++ = *cap++))
			;
		(*kp)[-1] = '=';
		while (*str) {
			for (p = __UNCONST(UNCTRL(*str++)); (*(*kp)++ = *p++);)
				;
			(*kp)--;
		}
		*(*kp)++ = ':';
		**kp = 0;
	}
}

static void
wwaddcap1(const char *cap, char **kp)
{
	while ((*(*kp)++ = *cap++))
		;
	(*kp)--;
}

static void
wwstart1(winvars_t *winvars)
{
	int i, j;

	for (i = 0; i < winvars->wwnrow; i++) {
		for (j = 0; j < winvars->wwncol; j++) {
			setcharpart(&winvars->wwos[i][j], ' ');
			if (tt.tt_checkpoint) {
				setcharpart(&winvars->wwcs[i][j], ' ');
			}
		}
	}
	xxstart();
	if (tt.tt_checkpoint) {
		winvars->wwdocheckpoint = 1;
	}
}

static void
wwinschar(winvars_t *winvars, ww_t *w, int row, int col, char c, char m)
{
	int i;
	int nvis;

	/*
	 * First, shift the line.
	 */
	{
		dispcell_t *p, *q;

		p = &w->ww_buf[row][w->ww_b.r];
		q = p - 1;
		for (i = w->ww_b.r - col; --i > 0;)
			*--p = *--q;
		setparts(q, c, m);
	}

	/*
	 * If can't see it, just return.
	 */
	if (row < w->ww_i.t || row >= w->ww_i.b ||
	    w->ww_i.r <= 0 || w->ww_i.r <= col)
		return;

	if (col < w->ww_i.l)
		col = w->ww_i.l;

	/*
	 * Now find out how much is actually changed, and fix wwns.
	 */
	{
		dispcell_t *buf;
		char *win;
		dispcell_t *ns;
		unsigned char *smap;
		char touched;

		nvis = 0;
		smap = &winvars->wwsmap[row][col];
		for (i = col; i < w->ww_i.r && *smap++ != w->ww_index; i++)
			;
		if (i >= w->ww_i.r)
			return;
		col = i;
		buf = w->ww_buf[row];
		win = w->ww_win[row];
		ns = winvars->wwns[row];
		smap = &winvars->wwsmap[row][i];
		touched = winvars->wwtouched[row];
		for (; i < w->ww_i.r; i++) {
			if (*smap++ != w->ww_index)
				continue;
			touched |= WWU_TOUCHED;
			if (win[i]) {
				setparts(&ns[i],
					getcharpart(&buf[i]),
					getmodepart(&buf[i]) ^ win[i]);
			} else {
				nvis++;
				ns[i] = buf[i];
			}
		}
		winvars->wwtouched[row] = touched;
	}

	/*
	 * Can/Should we use delete character?
	 */
	if ((tt.tt_inschar || tt.tt_insspace) && nvis > (winvars->wwncol - col) / 2) {
		dispcell_t *p, *q;

		if (tt.tt_inschar) {
			xxinschar(row, col, c, m);
		} else {
			xxinsspace(row, col);
			c = ' ';
			m = 0;
		}
		p = &winvars->wwos[row][winvars->wwncol];
		q = p - 1;
		for (i = winvars->wwncol - col; --i > 0;)
			*--p = *--q;
		setparts(q, c, m);
	}
}

/************ ========= *****************************/

void
wwflush(winvars_t *winvars)
{
	int row, col;

	if ((row = winvars->wwcursorrow) < 0)
		row = 0;
	else if (row >= winvars->wwnrow)
		row = winvars->wwnrow - 1;
	if ((col = winvars->wwcursorcol) < 0)
		col = 0;
	else if (col >= winvars->wwncol)
		col = winvars->wwncol - 1;
	xxmove(row, col);
	if (winvars->wwdocheckpoint) {
		xxflush(0);
		wwcheckpoint(winvars);
	} else
		xxflush(1);
}

int
wwinit(winvars_t *winvars)
{
	int i, j;
	char *kp;
	sigset_t nsigset, osigset;

	winvars->wwdtablesize = 3;
	winvars->wwhead.ww_forw = &winvars->wwhead;
	winvars->wwhead.ww_back = &winvars->wwhead;

	sigemptyset(&nsigset);
	sigaddset(&nsigset, SIGCHLD);
	sigaddset(&nsigset, SIGALRM);
	sigaddset(&nsigset, SIGHUP);
	sigaddset(&nsigset, SIGTERM);
	sigprocmask(SIG_BLOCK, &nsigset, &osigset);

	if (signal(SIGCHLD, wwchild) == SIG_ERR ||
	    signal(SIGHUP, wwquit) == SIG_ERR ||
	    signal(SIGTERM, wwquit) == SIG_ERR ||
	    signal(SIGPIPE, SIG_IGN) == SIG_ERR) {
		winvars->wwerrno = WWE_SYS;
		return -1;
	}

	if (wwgettty(winvars, 0, &winvars->wwoldtty) < 0)
		return -1;
	winvars->wwwintty = winvars->wwoldtty;
#ifdef OLD_TTY
	winvars->wwwintty.ww_sgttyb.sg_flags &= ~XTABS;
	winvars->wwnewtty.ww_sgttyb = winvars->wwoldtty.ww_sgttyb;
	winvars->wwnewtty.ww_sgttyb.sg_erase = -1;
	winvars->wwnewtty.ww_sgttyb.sg_kill = -1;
	winvars->wwnewtty.ww_sgttyb.sg_flags |= CBREAK;
	winvars->wwnewtty.ww_sgttyb.sg_flags &= ~(ECHO|CRMOD);
	winvars->wwnewtty.ww_tchars.t_intrc = -1;
	winvars->wwnewtty.ww_tchars.t_quitc = -1;
	winvars->wwnewtty.ww_tchars.t_startc = -1;
	winvars->wwnewtty.ww_tchars.t_stopc = -1;
	winvars->wwnewtty.ww_tchars.t_eofc = -1;
	winvars->wwnewtty.ww_tchars.t_brkc = -1;
	winvars->wwnewtty.ww_ltchars.t_suspc = -1;
	winvars->wwnewtty.ww_ltchars.t_dsuspc = -1;
	winvars->wwnewtty.ww_ltchars.t_rprntc = -1;
	winvars->wwnewtty.ww_ltchars.t_flushc = -1;
	winvars->wwnewtty.ww_ltchars.t_werasc = -1;
	winvars->wwnewtty.ww_ltchars.t_lnextc = -1;
	winvars->wwnewtty.ww_lmode = winvars->wwoldtty.ww_lmode | LLITOUT;
	winvars->wwnewtty.ww_ldisc = winvars->wwoldtty.ww_ldisc;
#else
#ifndef OXTABS
#define OXTABS XTABS
#endif
#ifndef _POSIX_VDISABLE
#define _POSIX_VDISABLE -1
#endif
	winvars->wwwintty.ww_termios.c_oflag &= ~OXTABS;
	winvars->wwwintty.ww_termios.c_cflag &= ~CLOCAL;
	winvars->wwwintty.ww_termios.c_cflag |= HUPCL;
	winvars->wwnewtty.ww_termios = winvars->wwoldtty.ww_termios;
	winvars->wwnewtty.ww_termios.c_iflag &=
		~(ISTRIP | INLCR | IGNCR | ICRNL | IXON | IXOFF | IMAXBEL);
	winvars->wwnewtty.ww_termios.c_oflag = 0;
	winvars->wwnewtty.ww_termios.c_cflag &= ~(CSIZE | PARENB);
	winvars->wwnewtty.ww_termios.c_cflag |= CS8;
	winvars->wwnewtty.ww_termios.c_lflag = 0;
	for (i = 0; i < NCCS; i++)
		winvars->wwnewtty.ww_termios.c_cc[i] = _POSIX_VDISABLE;
	winvars->wwnewtty.ww_termios.c_cc[VMIN] = 1;
	winvars->wwnewtty.ww_termios.c_cc[VTIME] = 0;
#endif
	if (wwsettty(winvars, 0, &winvars->wwnewtty) < 0)
		goto bad;

	if ((winvars->wwterm = getenv("TERM")) == 0) {
		winvars->wwerrno = WWE_BADTERM;
		goto bad;
	}
	if (tgetent(winvars->wwtermcap, winvars->wwterm) != 1) {
		winvars->wwerrno = WWE_BADTERM;
		goto bad;
	}
#ifdef OLD_TTY
	winvars->wwospeed = winvars->wwoldtty.ww_sgttyb.sg_ospeed;
#else
	winvars->wwospeed = cfgetospeed(&winvars->wwoldtty.ww_termios);
#endif

	if (xxinit() < 0)
		goto bad;
	winvars->wwnrow = tt.tt_nrow;
	winvars->wwncol = tt.tt_ncol;
	winvars->wwavailmodes = tt.tt_availmodes;
	winvars->wwwrap = tt.tt_wrap;

	if (winvars->wwavailmodes & WWM_REV)
		winvars->wwcursormodes = WWM_REV | (winvars->wwavailmodes & WWM_BLK);
	else if (winvars->wwavailmodes & WWM_UL)
		winvars->wwcursormodes = WWM_UL;

	if ((winvars->wwib = malloc((unsigned) 512)) == NULL)
		goto bad;
	winvars->wwibe = winvars->wwib + 512;
	winvars->wwibq = winvars->wwibp = winvars->wwib;

	winvars->wwsmap = (unsigned char **)
		wwalloc(winvars, 0, 0, winvars->wwnrow, winvars->wwncol, sizeof (unsigned char));
	if (winvars->wwsmap == 0)
		goto bad;
	for (i = 0; i < winvars->wwnrow; i++)
		for (j = 0; j < winvars->wwncol; j++)
			winvars->wwsmap[i][j] = WWX_NOBODY;

	winvars->wwos = (dispcell_t **)
		wwalloc(winvars, 0, 0, winvars->wwnrow, winvars->wwncol, sizeof (dispcell_t));
	if (winvars->wwos == 0)
		goto bad;
	/* wwos is cleared in wwstart1() */
	winvars->wwns = (dispcell_t **)
		wwalloc(winvars, 0, 0, winvars->wwnrow, winvars->wwncol, sizeof (dispcell_t));
	if (winvars->wwns == 0)
		goto bad;
	for (i = 0; i < winvars->wwnrow; i++)
		for (j = 0; j < winvars->wwncol; j++)
			setparts(&winvars->wwns[i][j], ' ', 0);
	if (tt.tt_checkpoint) {
		/* wwcs is also cleared in wwstart1() */
		winvars->wwcs = (dispcell_t **)
			wwalloc(winvars, 0, 0, winvars->wwnrow, winvars->wwncol, sizeof (dispcell_t));
		if (winvars->wwcs == NULL)
			goto bad;
	}

	winvars->wwtouched = malloc((unsigned) winvars->wwnrow);
	if (winvars->wwtouched == NULL) {
		winvars->wwerrno = WWE_NOMEM;
		goto bad;
	}
	for (i = 0; i < winvars->wwnrow; i++)
		winvars->wwtouched[i] = 0;

	winvars->wwupd = malloc((unsigned) winvars->wwnrow * sizeof *winvars->wwupd);
	if (winvars->wwupd == NULL) {
		winvars->wwerrno = WWE_NOMEM;
		goto bad;
	}

	winvars->wwindex[WWX_NOBODY] = &winvars->wwnobody;
	winvars->wwnobody.ww_order = MAX_NUM_WINDOWS;

	kp = winvars->wwwintermcap;
	if (winvars->wwavailmodes & WWM_REV)
		wwaddcap1(WWT_REV, &kp);
	if (winvars->wwavailmodes & WWM_BLK)
		wwaddcap1(WWT_BLK, &kp);
	if (winvars->wwavailmodes & WWM_UL)
		wwaddcap1(WWT_UL, &kp);
	if (winvars->wwavailmodes & WWM_GRP)
		wwaddcap1(WWT_GRP, &kp);
	if (winvars->wwavailmodes & WWM_DIM)
		wwaddcap1(WWT_DIM, &kp);
	if (winvars->wwavailmodes & WWM_USR)
		wwaddcap1(WWT_USR, &kp);
	if (winvars->wwavailmodes & WWM_BOLD)
		wwaddcap1(WWT_BOLD, &kp);
	if ((tt.tt_insline && tt.tt_delline) || tt.tt_setscroll)
		wwaddcap1(WWT_ALDL, &kp);
	if (tt.tt_inschar)
		wwaddcap1(WWT_IMEI, &kp);
	if (tt.tt_insspace)
		wwaddcap1(WWT_IC, &kp);
	if (tt.tt_delchar)
		wwaddcap1(WWT_DC, &kp);
	wwaddcap("kb", &kp);
	wwaddcap("ku", &kp);
	wwaddcap("kd", &kp);
	wwaddcap("kl", &kp);
	wwaddcap("kr", &kp);
	wwaddcap("kh", &kp);
	if ((j = tgetnum("kn")) >= 0) {
		char cap[32];

		(void) snprintf(kp, sizeof(winvars->wwwintermcap) - (int)(kp - winvars->wwwintermcap),
			"kn#%d:", j);
		for (; *kp; kp++)
			;
		for (i = 1; i <= j; i++) {
			(void) snprintf(cap, sizeof(cap), "k%d", i);
			wwaddcap(cap, &kp);
			cap[0] = 'l';
			wwaddcap(cap, &kp);
		}
	}
	/*
	 * It's ok to do this here even if setenv() is destructive
	 * since tt_init() has already made its own copy of it and
	 * wwterm now points to the copy.
	 */
	(void) setenv("TERM", WWT_TERM, 1);
#ifdef TERMINFO
	if (wwterminfoinit(winvars) < 0)
		goto bad;
#endif

	if (tt.tt_checkpoint)
		if (signal(SIGALRM, wwalarm) == SIG_ERR) {
			winvars->wwerrno = WWE_SYS;
			goto bad;
		}
	wwstart1(winvars);

	sigprocmask(SIG_SETMASK, &osigset, NULL);
	return 0;

bad:
	/*
	 * Don't bother to free storage.  We're supposed
	 * to exit when wwinit fails anyway.
	 */
	(void) wwsettty(winvars, 0, &winvars->wwoldtty);

	sigprocmask(SIG_SETMASK, &osigset, NULL);
	return -1;
}

void
wwstart(winvars_t *winvars)
{
	int i;

	(void) wwsettty(winvars, 0, &winvars->wwnewtty);
	for (i = 0; i < winvars->wwnrow; i++)
		winvars->wwtouched[i] = WWU_TOUCHED;
	wwstart1(winvars);
}

void
wwredraw(winvars_t *winvars)
{
	int i, j;
	dispcell_t *os;

	xxclear();
	for (i = 0; i < winvars->wwnrow; i++) {
		winvars->wwtouched[i] = WWU_TOUCHED;
		os = winvars->wwos[i];
		for (j = winvars->wwncol; --j >= 0;)
			setparts(os++, ' ', 0);
	}
}

void
wwscroll(winvars_t *winvars, ww_t *w, int n)
{
	int dir;
	int top;

	if (n == 0)
		return;
	dir = n < 0 ? -1 : 1;
	top = w->ww_b.t - n;
	if (top > w->ww_w.t)
		top = w->ww_w.t;
	else if (top + w->ww_b.nr < w->ww_w.b)
		top = w->ww_w.b - w->ww_b.nr;
	n = abs(top - w->ww_b.t);
	if (n < w->ww_i.nr) {
		while (--n >= 0) {
			(void) wwscroll1(winvars, w, w->ww_i.t, w->ww_i.b, dir, 0);
			w->ww_buf += dir;
			w->ww_b.t -= dir;
			w->ww_b.b -= dir;
		}
	} else {
		w->ww_buf -= top - w->ww_b.t;
		w->ww_b.t = top;
		w->ww_b.b = top + w->ww_b.nr;
		WWREDRAWWIN(winvars, w);
	}
}

/*
 * Scroll one line, between 'row1' and 'row2', in direction 'dir'.
 * Don't adjust ww_scroll.
 * And don't redraw 'leaveit' lines.
 */
int
wwscroll1(winvars_t *winvars, ww_t *w, int row1, int row2, int dir, int leaveit)
{
	int i;
	int row1x, row2x;
	int nvis;
	int nvismax;
	int scrolled = 0;

	/*
	 * See how many lines on the screen are affected.
	 * And calculate row1x, row2x, and left at the same time.
	 */
	for (i = row1; i < row2 && w->ww_nvis[i] == 0; i++)
		;
	if (i >= row2)			/* can't do any fancy stuff */
		goto out;
	row1x = i;
	for (i = row2 - 1; i >= row1 && w->ww_nvis[i] == 0; i--)
		;
	if (i <= row1x)
		goto out;		/* just one line is easy */
	row2x = i + 1;

	/*
	 * See how much of this window is visible.
	 */
	nvismax = winvars->wwncol * (row2x - row1x);
	nvis = 0;
	for (i = row1x; i < row2x; i++)
		nvis += w->ww_nvis[i];

	/*
	 * If it's a good idea to scroll and the terminal can, then do it.
	 */
	if (nvis < nvismax / 2)
		goto no_scroll;		/* not worth it */
	if ((dir > 0 ? tt.tt_scroll_down == 0 : tt.tt_scroll_up == 0) ||
	    ((tt.tt_scroll_top != row1x || tt.tt_scroll_bot != row2x - 1) &&
	    tt.tt_setscroll == 0))
		if (tt.tt_delline == 0 || tt.tt_insline == 0)
			goto no_scroll;
	xxscroll(dir, row1x, row2x);
	scrolled = 1;
	/*
	 * Fix up the old screen.
	 */
	{
		dispcell_t *tmp;
		dispcell_t **cpp, **cqq;

		if (dir > 0) {
			cpp = &winvars->wwos[row1x];
			cqq = cpp + 1;
			tmp = *cpp;
			for (i = row2x - row1x; --i > 0;)
				*cpp++ = *cqq++;
			*cpp = tmp;
		} else {
			cpp = &winvars->wwos[row2x];
			cqq = cpp - 1;
			tmp = *cqq;
			for (i = row2x - row1x; --i > 0;)
				*--cpp = *--cqq;
			*cqq = tmp;
		}
		for (i = winvars->wwncol; --i >= 0;)
			setparts(tmp++, ' ', 0);
	}

no_scroll:
	/*
	 * Fix the new screen.
	 */
	if (nvis == nvismax) {
		/*
		 * Can shift whole lines.
		 */
		if (dir > 0) {
			{
				dispcell_t *tmp;
				dispcell_t **cpp, **cqq;

				cpp = &winvars->wwns[row1x];
				cqq = cpp + 1;
				tmp = *cpp;
				for (i = row2x - row1x; --i > 0;)
					*cpp++ = *cqq++;
				*cpp = tmp;
			}
			if (scrolled) {
				char *p, *q;

				p = &winvars->wwtouched[row1x];
				q = p + 1;
				for (i = row2x - row1x; --i > 0;)
					*p++ = *q++;
				*p |= WWU_TOUCHED;
			} else {
				char *p;

				p = &winvars->wwtouched[row1x];
				for (i = row2x - row1x; --i >= 0;)
					*p++ |= WWU_TOUCHED;
			}
			wwredrawwin1(winvars, w, row1, row1x, dir);
			wwredrawwin1(winvars, w, row2x - 1, row2 - leaveit, dir);
		} else {
			{
				dispcell_t *tmp;
				dispcell_t **cpp, **cqq;

				cpp = &winvars->wwns[row2x];
				cqq = cpp - 1;
				tmp = *cqq;
				for (i = row2x - row1x; --i > 0;)
					*--cpp = *--cqq;
				*cqq = tmp;
			}
			if (scrolled) {
				char *p, *q;

				p = &winvars->wwtouched[row2x];
				q = p - 1;
				for (i = row2x - row1x; --i > 0;)
					*--p = *--q;
				*q |= WWU_TOUCHED;
			} else {
				char *p;

				p = &winvars->wwtouched[row1x];
				for (i = row2x - row1x; --i >= 0;)
					*p++ |= WWU_TOUCHED;
			}
			wwredrawwin1(winvars, w, row1 + leaveit, row1x + 1, dir);
			wwredrawwin1(winvars, w, row2x, row2, dir);
		}
	} else {
		if (scrolled) {
			char *p;

			p = &winvars->wwtouched[row1x];
			for (i = row2x - row1x; --i >= 0;)
				*p++ |= WWU_TOUCHED;
		}
out:
		if (dir > 0)
			wwredrawwin1(winvars, w, row1, row2 - leaveit, dir);
		else
			wwredrawwin1(winvars, w, row1 + leaveit, row2, dir);
	}
	return scrolled;
}

void
wwclreos(winvars_t *winvars, ww_t *w, int row, int col)
{
	int i;

	WWCLREOL(winvars, w, row, col);
	for (i = row + 1; i < w->ww_b.b; i++)
		WWCLREOL(winvars, w, i, w->ww_b.l);
	/* XXX */
	if (!ISSET(w->windowflags, WWW_NOUPDATE))
		wwupdate1(winvars, w->ww_i.t, w->ww_i.b);
}

void
wwupdate1(winvars_t *winvars, int top, int bot)
{
	int i;
	int j;
	char *touched;
	ww_update_t *upd;
	char check_clreos = 0;
	int scan_top, scan_bot;

	winvars->wwnupdate++;
	{
		char *t1 = winvars->wwtouched + top, *t2 = winvars->wwtouched + bot;
		int n;

		while (!*t1++)
			if (t1 == t2)
				return;
		while (!*--t2)
			;
		scan_top = top = t1 - winvars->wwtouched - 1;
		scan_bot = bot = t2 - winvars->wwtouched + 1;
		if (scan_bot - scan_top > 1 &&
		    (tt.tt_clreos != 0 || tt.tt_clear != 0)) {
			int st = tt.tt_clreos != 0 ? scan_top : 0;

			/*
			 * t1 is one past the first touched row,
			 * t2 is on the last touched row.
			 */
			for (t1--, n = 1; t1 < t2;)
				if (*t1++)
					n++;
			/*
			 * If we can't clreos then we try for clearing
			 * the whole screen.
			 */
			if ((check_clreos = n * 10 > (winvars->wwnrow - st) * 9)) {
				scan_top = st;
				scan_bot = winvars->wwnrow;
			}
		}
	}
	if (tt.tt_clreol == 0 && !check_clreos)
		goto simple;
	for (i = scan_top, touched = &winvars->wwtouched[i], upd = &winvars->wwupd[i];
	     i < scan_bot;
	     i++, touched++, upd++) {
		int gain = 0;
		int best_gain = 0;
		int best_col = 0;
		dispcell_t *ns, *os;

		if (WWINTERRUPT(winvars))
			return;
		if (!check_clreos && !*touched)
			continue;
		winvars->wwnupdscan++;
		j = winvars->wwncol;
		ns = &winvars->wwns[i][j];
		os = &winvars->wwos[i][j];
		while (--j >= 0) {
			/*
			 * The cost of clearing is:
			 *	ncol - nblank + X
			 * The cost of straight update is, more or less:
			 *	ncol - nsame
			 * We clear if  nblank - nsame > X
			 * X is the clreol overhead.
			 * So we make gain = nblank - nsame.
			 */
			if (getwhole(--ns) == getwhole(--os))
				gain--;
			else
				best_gain--;
			if (getwhole(ns) == ' ')
				gain++;
			if (gain > best_gain) {
				best_col = j;
				best_gain = gain;
			}
		}
		upd->best_gain = best_gain;
		upd->best_col = best_col;
		upd->gain = gain;
	}
	if (check_clreos) {
		ww_update_t *u;
		int gain = 0;
		int best_gain = 0;
		int best_row = 0;
		int simple_gain = 0;
		char didit = 0;

		/*
		 * gain is the advantage of clearing all the lines.
		 * best_gain is the advantage of clearing to eos
		 * at best_row and u->best_col.
		 * simple_gain is the advantage of using only clreol.
		 * We use g > best_gain because u->best_col can be
		 * undefined when u->best_gain is 0 so we can't use it.
		 */
		for (j = scan_bot - 1, u = winvars->wwupd + j; j >= top; j--, u--) {
			int g = gain + u->best_gain;

			if (g > best_gain) {
				best_gain = g;
				best_row = j;
			}
			gain += u->gain;
			if (tt.tt_clreol != 0 && u->best_gain > 4)
				simple_gain += u->best_gain - 4;
		}
		if (tt.tt_clreos == 0) {
			if (gain > simple_gain && gain > 4) {
				xxclear();
				i = top = scan_top;
				bot = scan_bot;
				j = 0;
				didit = 1;
			}
		} else
			if (best_gain > simple_gain && best_gain > 4) {
				i = best_row;
				xxclreos(i, j = winvars->wwupd[i].best_col);
				bot = scan_bot;
				didit = 1;
			}
		if (didit) {
			winvars->wwnupdclreos++;
			winvars->wwnupdclreosline += winvars->wwnrow - i;
			u = winvars->wwupd + i;
			while (i < scan_bot) {
				dispcell_t *os = &winvars->wwos[i][j];

				for (j = winvars->wwncol - j; --j >= 0;) {
					char	mode = getmodepart(os);

					setparts(os++, ' ', mode);
				}
				winvars->wwtouched[i++] |= WWU_TOUCHED;
				u++->best_gain = 0;
				j = 0;
			}
		} else
			winvars->wwnupdclreosmiss++;
	}
simple:
	for (i = top, touched = &winvars->wwtouched[i], upd = &winvars->wwupd[i]; i < bot;
	     i++, touched++, upd++) {
		dispcell_t *os, *ns;
		char didit;

		if (!*touched)
			continue;
		*touched = 0;
		winvars->wwnupdline++;
		didit = 0;
		if (tt.tt_clreol != 0 && upd->best_gain > 4) {
			winvars->wwnupdclreol++;
			xxclreol(i, j = upd->best_col);
			for (os = &winvars->wwos[i][j], j = winvars->wwncol - j; --j >= 0;) {
				setparts(os++, ' ', 0);
			}
			didit = 1;
		}
		ns = winvars->wwns[i];
		os = winvars->wwos[i];
		for (j = 0; j < winvars->wwncol;) {
			char *p, *q;
			char m;
			int c;
			int n;
			char buf[512];			/* > winvars->wwncol */
			dispcell_t lastc;

			for (; j++ < winvars->wwncol && getwhole(ns++) == getwhole(os++);)
				;
			if (j > winvars->wwncol)
				break;
			p = buf;
			m = getmodepart(&ns[-1]);
			c = j - 1;
			os[-1] = ns[-1];
			*p++ = getcharpart(&ns[-1]);
			n = 5;
			q = p;
			while (j < winvars->wwncol && getmodepart(ns) == m) {
				*p++ = getcharpart(ns);
				if (getwhole(ns) == getwhole(os)) {
					if (--n <= 0)
						break;
					os++;
					ns++;
				} else {
					n = 5;
					q = p;
					lastc = *os;
					*os++ = *ns++;
				}
				j++;
			}
			n = q - buf;
			if (!winvars->wwwrap || i != winvars->wwnrow - 1 || c + n != winvars->wwncol)
				xxwrite(i, c, buf, n, m);
			else if (tt.tt_inschar || tt.tt_insspace) {
				if (n > 1) {
					q[-2] = q[-1];
					n--;
				} else
					c--;
				xxwrite(i, c, buf, n, m);
				c += n - 1;
				if (tt.tt_inschar)
					xxinschar(i, c, getcharpart(&ns[-2]),
						getmodepart(&ns[-2]));
				else {
					char	ch;

					xxinsspace(i, c);
					ch = getcharpart(&ns[-2]);
					xxwrite(i, c, &ch, 1,
						getmodepart(&ns[-2]));
				}
			} else {
				if (--n)
					xxwrite(i, c, buf, n, m);
				os[-1] = lastc;
				*touched = WWU_TOUCHED;
			}
			didit = 1;
		}
		if (!didit)
			winvars->wwnupdmiss++;
	}
}

#define UPDATE() \
	if (!ISSET(w->windowflags, WWW_NOUPDATE) && w->ww_cur.r >= 0 && \
	    w->ww_cur.r < winvars->wwnrow && winvars->wwtouched[w->ww_cur.r]) \
		wwupdate1(winvars, w->ww_cur.r, w->ww_cur.r + 1)

#define TABSPACES(pos, left)	(8 - ((pos) - (left)) & 7)

/*
 * To support control character expansion, we save the old
 * p and q values in r and s, and point p at the beginning
 * of the expanded string, and q at some safe place beyond it
 * (p + 10).  At strategic points in the loops, we check
 * for (r && !*p) and restore the saved values back into
 * p and q.  Essentially, we implement a stack of depth 2,
 * to avoid recursion, which might be a better idea.
 */
int
wwwrite(winvars_t *winvars, ww_t *w, const char *p, int n)
{
	int hascursor;
	const char *savep = p;
	const char *q = p + n;
	const char *r = NULL;
	const char *s = NULL;

#ifdef lint
	s = NULL;		/* define it before possible use */
#endif
	hascursor = ISSET(w->windowflags, WWW_HASCURSOR);
	if (hascursor)
		wwcursor(winvars, w, 0);
	while (p < q && !ISSET(w->processflags, WWP_STOPPED) &&
	    (!WWINTERRUPT(winvars) || ISSET(w->windowflags, WWW_NOINTR))) {
		if (r && !*p) {
			p = r;
			q = s;
			r = 0;
			continue;
		}
		if (w->ww_wstate == 0 &&
		    (ISPRINT(*p) ||
		    (ISSET(w->windowflags, WWW_UNCTRL) && ISUNCTRL(*p)))) {
			int curc;
			dispcell_t *bp;
			int col, col1;

			if (ISSET(w->windowflags, WWW_INSERT)) {
				/* this is very slow */
				if (*p == '\t') {
					p++;
					w->ww_cur.c += TABSPACES(w->ww_cur.c, w->ww_w.l);
					goto chklf;
				}
				if (!ISPRINT(*p)) {
					r = p + 1;
					s = q;
					p = UNCTRL(*p);
					q = p + 10;
				}
				wwinschar(winvars, w, w->ww_cur.r, w->ww_cur.c,
					*p++, w->ww_modes);
				goto right;
			}

			bp = &w->ww_buf[w->ww_cur.r][w->ww_cur.c];
			curc = w->ww_cur.c;
			while (curc < w->ww_w.r && p < q) {
				if (!*p && r) {
					p = r;
					q = s;
					r = 0;
				} else if (*p == '\t') {
					int tmp = TABSPACES(curc, w->ww_w.l);

					p++;
					curc += tmp;
					bp += tmp;
				} else if (ISPRINT(*p)) {
					setparts(bp++, *p++, w->ww_modes);
					curc++;
				} else if (ISSET(w->windowflags, WWW_UNCTRL) &&
					   ISUNCTRL(*p)) {
					r = p + 1;
					s = q;
					p = UNCTRL(*p);
					q = p + 10;
				} else {
					break;
				}
			}
			col = MAX(w->ww_cur.c, w->ww_i.l);
			col1 = MIN(curc, w->ww_i.r);
			w->ww_cur.c = curc;
			if (w->ww_cur.r >= w->ww_i.t
			    && w->ww_cur.r < w->ww_i.b) {
				dispcell_t *ns = winvars->wwns[w->ww_cur.r];
				unsigned char *smap =
				    &winvars->wwsmap[w->ww_cur.r][col];
				char *win = w->ww_win[w->ww_cur.r];
				int nchanged = 0;

				bp = w->ww_buf[w->ww_cur.r];
				for (curc = col; curc < col1; curc++) {
					if (*smap++ == w->ww_index) {
						nchanged++;
						setparts(&ns[curc], getcharpart(&bp[curc]),
							getmodepart(&bp[curc]) ^ win[curc]);
					}
				}
				if (nchanged > 0)
					winvars->wwtouched[w->ww_cur.r] |= WWU_TOUCHED;
			}
chklf:
			if (w->ww_cur.c >= w->ww_w.r)
				goto crlf;
		} else switch (w->ww_wstate) {
		case 0:
			switch (*p++) {
			case '\n':
				if (ISSET(w->windowflags, WWW_MAPNL))
crlf:
					w->ww_cur.c = w->ww_w.l;
lf:
				UPDATE();
				if (++w->ww_cur.r >= w->ww_w.b) {
					w->ww_cur.r = w->ww_w.b - 1;
					if (w->ww_w.b < w->ww_b.b) {
						(void) wwscroll1(winvars, w, w->ww_i.t,
							w->ww_i.b, 1, 0);
						w->ww_buf++;
						w->ww_b.t--;
						w->ww_b.b--;
					} else
						wwdelline(winvars, w, w->ww_b.t);
				}
				break;
			case '\b':
				if (--w->ww_cur.c < w->ww_w.l) {
					w->ww_cur.c = w->ww_w.r - 1;
					goto up;
				}
				break;
			case '\r':
				w->ww_cur.c = w->ww_w.l;
				break;
			case CONTROL('g'):
				/* LINTED */
				TTPUTC(CONTROL('g'));
				break;
			case CONTROL('['):
				w->ww_wstate = 1;
				break;
			}
			break;
		case 1:
			w->ww_wstate = 0;
			switch (*p++) {
			case '@':
				SET(w->windowflags, WWW_INSERT);
				break;
			case 'A':
up:
				UPDATE();
				if (--w->ww_cur.r < w->ww_w.t) {
					w->ww_cur.r = w->ww_w.t;
					if (w->ww_w.t > w->ww_b.t) {
						(void) wwscroll1(winvars, w, w->ww_i.t,
							w->ww_i.b, -1, 0);
						w->ww_buf--;
						w->ww_b.t++;
						w->ww_b.b++;
					} else
						wwinsline(winvars, w, w->ww_b.t);
				}
				break;
			case 'B':
				goto lf;
			case 'C':
right:
				w->ww_cur.c++;
				goto chklf;
			case 'E':
				w->ww_buf -= w->ww_w.t - w->ww_b.t;
				w->ww_b.t = w->ww_w.t;
				w->ww_b.b = w->ww_b.t + w->ww_b.nr;
				w->ww_cur.r = w->ww_w.t;
				w->ww_cur.c = w->ww_w.l;
				wwclreos(winvars, w, w->ww_w.t, w->ww_w.l);
				break;
			case 'H':
				UPDATE();
				w->ww_cur.r = w->ww_w.t;
				w->ww_cur.c = w->ww_w.l;
				break;
			case 'J':
				wwclreos(winvars, w, w->ww_cur.r, w->ww_cur.c);
				break;
			case 'K':
				WWCLREOL(winvars, w, w->ww_cur.r, w->ww_cur.c);
				break;
			case 'L':
				UPDATE();
				wwinsline(winvars, w, w->ww_cur.r);
				break;
			case 'M':
				wwdelline(winvars, w, w->ww_cur.r);
				break;
			case 'N':
				wwdelchar(winvars, w, w->ww_cur.r, w->ww_cur.c);
				break;
			case 'O':
				CLR(w->windowflags, WWW_INSERT);
				break;
			case 'P':
				wwinschar(winvars, w, w->ww_cur.r, w->ww_cur.c, ' ', 0);
				break;
			case 'X':
				WWUPDATE(winvars);
				break;
			case 'Y':
				UPDATE();
				w->ww_wstate = 2;
				break;
			case 'Z':
				WWUPDATE(winvars);
				xxflush(0);
				break;
			case 's':
				w->ww_wstate = 4;
				break;
			case 'r':
				w->ww_wstate = 5;
				break;
			}
			break;
		case 2:
			w->ww_cur.r = w->ww_w.t +
				(unsigned)(*p++ - ' ') % w->ww_w.nr;
			w->ww_wstate = 3;
			break;
		case 3:
			w->ww_cur.c = w->ww_w.l +
				(unsigned)(*p++ - ' ') % w->ww_w.nc;
			w->ww_wstate = 0;
			break;
		case 4:
			w->ww_modes |= *p++ & winvars->wwavailmodes;
			w->ww_wstate = 0;
			break;
		case 5:
			w->ww_modes &= ~*p++;
			w->ww_wstate = 0;
			break;
		}
	}
	if (hascursor)
		wwcursor(winvars, w, 1);
	winvars->wwnwwr++;
	winvars->wwnwwra += n;
	n = p - savep;
	winvars->wwnwwrc += n;
	return n;
}
