/* $NetBSD: $ */
/*	$KAME: if_mip.c,v 1.12 2007/06/14 12:09:42 itojun Exp $	*/

/*
 * Copyright (C) 2004 WIDE Project.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the project nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE PROJECT AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE PROJECT OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#include <sys/cdefs.h>
__KERNEL_RCSID(0, "$NetBSD: $");

#ifdef _KERNEL_OPT
#include "opt_inet.h"
#include "opt_mip6.h"
#endif /* _KERNEL_OPT */

#include <sys/param.h>
#include <sys/systm.h>
#include <sys/kernel.h>
#include <sys/socket.h>
#include <sys/sockio.h>
#include <sys/malloc.h>
#include <sys/mbuf.h>

#include <net/if.h>
#include <net/if_types.h>
#include <net/route.h>
#include <net/bpf.h>

#include <netinet/in.h>
#include <netinet/ip6.h>

#include <netinet6/in6.h>
#include <netinet6/in6_var.h>
#include <netinet6/ip6_var.h>
#include <netinet6/nd6.h>
#include <netinet6/mip6_var.h>

#include <net/if_mip.h>

#include "mip.h"

#include <net/net_osdep.h>

#if NMIP > 0
#include <netinet6/in6_var.h>
#include <netinet6/mip6.h>

extern struct mip6stat mip6stat;

void mipattach(int);

LIST_HEAD(mip_softc_head, mip_softc) mip_softc_list;

static int	mip_clone_create(struct if_clone *, int);
static int	mip_clone_destroy(struct ifnet *);

static struct if_clone mip_cloner =
    IF_CLONE_INITIALIZER("mip", mip_clone_create, mip_clone_destroy);

void
mipattach(int count)
{
	LIST_INIT(&mip_softc_list);
	if_clone_attach(&mip_cloner);
}

static int
mip_clone_create(struct if_clone *ifc, int unit)
{
	struct mip_softc *sc;
	struct ifnet *ifp;

	sc = malloc(sizeof(struct mip_softc), M_DEVBUF, M_WAITOK|M_ZERO);
	ifp = &sc->mip_if;

	if_initname(ifp, ifc->ifc_name, unit);
	ifp->if_flags = IFF_MULTICAST | IFF_SIMPLEX;
	ifp->if_mtu = MIP_MTU;
	ifp->if_ioctl = mip_ioctl;
	ifp->if_output = mip_output;
	ifp->if_type = IFT_MOBILEIP;
	ifp->if_addrlen = 0;
	ifp->if_dlt = DLT_NULL;
	ifp->if_softc = sc;
	IFQ_SET_READY(ifp->if_snd);
	if_attach(ifp);
	if_alloc_sadl(ifp);

	bpf_attach(&sc->mip_if, DLT_NULL, sizeof(u_int));

		/* XXX
		 * various mip_softc initialization should be here.
		 */

	/* create mip_softc list */
	LIST_INSERT_HEAD(&mip_softc_list, sc, mip_entry);
	return 0;
}

static int
mip_clone_destroy(struct ifnet *ifp)
{
	struct mip_softc * sc = ifp->if_softc;

	LIST_REMOVE(sc, mip_entry);
	bpf_detach(ifp);
	if_detach(ifp);
	free(sc, M_DEVBUF);

	return (0);
}

int
mip_output(struct ifnet *ifp, struct mbuf *m, const struct sockaddr *dst,
	const struct rtentry *rt)
{
	int error = 0;
	struct mip6_bul_internal *mbul;
	struct ip6_hdr *ip6;

	/* This function is copyed from looutput */

	if ((m->m_flags & M_PKTHDR) == 0)
		panic("mip_output no HDR");

	if (rt && rt->rt_flags & (RTF_REJECT|RTF_BLACKHOLE)) {
		m_freem(m);
		return (rt->rt_flags & RTF_BLACKHOLE ? 0 :
		    rt->rt_flags & RTF_HOST ? EHOSTUNREACH : ENETUNREACH);
	}

#ifndef PULLDOWN_TEST
	/*
	 * KAME requires that the packet to be contiguous on the
	 * mbuf.  We need to make that sure.
	 * this kind of code should be avoided.
	 * XXX: fails to join if interface MTU > MCLBYTES.  jumbogram?
	 */
	if (m->m_len != m->m_pkthdr.len) {
		struct mbuf *n = NULL;
		int maxlen;

		MGETHDR(n, M_DONTWAIT, MT_HEADER);
		maxlen = MHLEN;
		if (n)
			M_COPY_PKTHDR(n, m);
		if (n && m->m_pkthdr.len > maxlen) {
			MCLGET(n, M_DONTWAIT);
			maxlen = MCLBYTES;
			if ((n->m_flags & M_EXT) == 0) {
				m_free(n);
				n = NULL;
			}
		}
		if (!n) {
			printf("mip_output: mbuf allocation failed\n");
			m_freem(m);
			return ENOBUFS;
		}

		if (m->m_pkthdr.len <= maxlen) {
			m_copydata(m, 0, m->m_pkthdr.len, mtod(n, void *));
			n->m_len = m->m_pkthdr.len;
			n->m_next = NULL;
			m_freem(m);
		} else {
			m_copydata(m, 0, maxlen, mtod(n, void *));
			m_adj(m, maxlen);
			n->m_len = maxlen;
			n->m_next = m;
		}
		m = n;
	}
#endif

	ifp->if_opackets++;
	ifp->if_obytes += m->m_pkthdr.len;

	switch (dst->sa_family) {
	case AF_INET6:
		/*
		 * if ! link-local, prepend an outer ip header and
		 * send it.  if link-local, discard it.
		 */
		ip6 = mtod(m, struct ip6_hdr *);
		if (IN6_IS_ADDR_MULTICAST(&ip6->ip6_src)
		    || IN6_IS_ADDR_LINKLOCAL(&ip6->ip6_src)
		    || IN6_IS_ADDR_V4MAPPED(&ip6->ip6_src)
		    || IN6_IS_ADDR_LOOPBACK(&ip6->ip6_src)
		    || IN6_IS_ADDR_LOOPBACK(&ip6->ip6_src))
			goto done;
		if(IN6_IS_ADDR_LINKLOCAL(&ip6->ip6_dst)
		    || IN6_IS_ADDR_V4MAPPED(&ip6->ip6_dst)
		    || IN6_IS_ADDR_LOOPBACK(&ip6->ip6_dst)
		    || IN6_IS_ADDR_LOOPBACK(&ip6->ip6_dst))
			goto done;

		/*
		 * find the home registration entry for this source
		 * address.
		 */
		mbul = mip6_bul_get_home_agent(&ip6->ip6_src);
		if (!mbul)
			goto done;

		if (IN6_IS_ADDR_UNSPECIFIED(&mbul->mbul_peeraddr))
			goto done;

		M_PREPEND(m, sizeof(struct ip6_hdr), M_DONTWAIT);
		if (m && m->m_len < sizeof(struct ip6_hdr))
			m = m_pullup(m, sizeof(struct ip6_hdr));
		if (m == NULL)
			return (0);

		ip6 = mtod(m, struct ip6_hdr *);
		ip6->ip6_flow = 0;
		ip6->ip6_vfc &= ~IPV6_VERSION_MASK;
		ip6->ip6_vfc |= IPV6_VERSION;
		ip6->ip6_plen = htons((u_short)m->m_pkthdr.len - sizeof(*ip6));
		ip6->ip6_nxt = IPPROTO_IPV6;
		ip6->ip6_hlim = ip6_defhlim;
		ip6->ip6_src = mbul->mbul_coa;
		ip6->ip6_dst = mbul->mbul_peeraddr;
		mip6stat.mip6s_orevtunnel++;
		/* XXX */
		error = ip6_output(m, NULL, NULL, IPV6_MINMTU, NULL,
		    NULL, &ifp);
		return (error);
 done:
		break;
	default:
		printf("mip_output: af=%d unexpected\n", dst->sa_family);
		m_freem(m);
		return (EAFNOSUPPORT);
	}

	m_freem(m);
	return (0);
	/*
	 * This function would be called when pre_output() function was kicked.
	 * Returning with EJUSTRETURN is necessarry to avoid call output()
	 * function.
	 */
	return (EJUSTRETURN);
}

int
mip_ioctl(struct ifnet *ifp, unsigned long cmd, void *data)
{
	int error;
	struct ifreq *ifr = (struct ifreq *)data;
#if NMIP > 0
	struct if_bulreq *bulreq;
	struct mip6_bul_internal *mbul = NULL, *nmbul = NULL;
	struct bul6info *bul6;
	int addlen = sizeof(struct if_bulreq) + sizeof(struct bul6info);

        register struct ifaddr *ifa;
        struct in6_ifaddr *ia6;
	char ip6buf[INET6_ADDRSTRLEN];
#endif

	error = 0;

	switch(cmd) {
	case SIOCINITIFADDR:
		ifp->if_flags |= IFF_UP | IFF_RUNNING;
		/*
		 * Everything else is done at a higher level.
		 */
		break;

	case SIOCADDMULTI:
	case SIOCDELMULTI:
		if (ifr == 0) {
			error = EAFNOSUPPORT;		/* XXX */
			break;
		}
		switch (ifr->ifr_addr.sa_family) {
#ifdef INET6
		case AF_INET6:
			break;
#endif
		default:
			error = EAFNOSUPPORT;
			break;
		}
		break;
#if NMIP > 0
	case SIOCGBULIST:
		if (!MIP6_IS_MN)
			return EOPNOTSUPP;

		bulreq = (struct if_bulreq *)data;
		bul6 = bulreq->ifbu_info;
		bulreq->ifbu_count = 0;
		
		TAILQ_FOREACH(ifa, &ifp->if_addrlist, ifa_list) 
		{
			if (ifa->ifa_addr->sa_family != AF_INET6)
				continue;

			ia6 = (struct in6_ifaddr *)ifa;
			if (LIST_EMPTY(MBUL_LIST(ia6)))
				continue;
			
			for (mbul = LIST_FIRST(MBUL_LIST(ia6)); mbul;
			     mbul = nmbul) {
				nmbul = LIST_NEXT(mbul, mbul_entry);
				
				if (addlen > bulreq->ifbu_len) 
					break; /* no buffer space to add BUL */
				
				memcpy(&bul6->bul_peeraddr,
				    &mbul->mbul_peeraddr,
				    sizeof(mbul->mbul_peeraddr));
				printf("adding entry %s\n", IN6_PRINT(ip6buf, &mbul->mbul_peeraddr));
				memcpy(&bul6->bul_hoa, &mbul->mbul_hoa,
				      sizeof(mbul->mbul_hoa));
				memcpy(&bul6->bul_coa, &mbul->mbul_coa,
				      sizeof(mbul->mbul_coa));
				bul6->bul_flags = mbul->mbul_flags;
				bul6->bul_ifindex = mbul->mbul_mip->mip_if.if_index;
				
				bul6 += sizeof(struct bul6info);
				addlen += sizeof(struct bul6info);
				bulreq->ifbu_count ++;
			}
		}
		break;
#endif
	default:
		error = ifioctl_common(ifp, cmd, data);
		break;
	}

	return (error);
}

int
mip_is_mip_softc(struct ifnet *ifp)
{
	struct mip_softc *mipsc;

	for (mipsc = LIST_FIRST(&mip_softc_list); mipsc;
	     mipsc = LIST_NEXT(mipsc, mip_entry)) {
		if ((void *)ifp == (void *)mipsc)
			return (1);
	}
	return (0);
}

#endif /* NMIP > 0 */
