/*	$NetBSD: $ */

/*-
 * Copyright (c) 2008 The NetBSD Foundation, Inc.
 * All rights reserved.
 *
 * This code is derived from software contributed to The NetBSD Foundation
 * by Robert Swindells
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE NETBSD FOUNDATION, INC. AND CONTRIBUTORS
 * ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
 * TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE FOUNDATION OR CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <sys/cdefs.h>
__KERNEL_RCSID(0, "$NetBSD: $");

#include <sys/param.h>
#include <sys/systm.h>
#include <sys/conf.h>
#include <sys/device.h>
#include <sys/malloc.h>
#include <sys/kernel.h>

#include <dev/ofw/openfirm.h>

#include <machine/autoconf.h>
#include <machine/bus.h>

#include <dev/ata/atavar.h>
#include <dev/ic/wdcvar.h>

#include <arch/powerpc/mpc5200/mpc5200reg.h>
#include <arch/powerpc/mpc5200/mpc5200var.h>

static int ide_match(device_t, cfdata_t, void *);
static void ide_attach(device_t, device_t, void *);

struct ide_softc {
	struct wdc_softc	sc_wdcdev;	/* common wdc definitions */
	struct ata_channel	*sc_chanarray[1]; /* channels definition */
	struct ata_channel	sc_channel; /* generic part */
	struct ata_queue	sc_chqueue; /* channel queue */
	void			*sc_ih;	/* irq handler */
	struct wdc_regs		sc_wdc_regs;
	int sc_irq;
};

CFATTACH_DECL_NEW(ide, sizeof(struct ide_softc),
	ide_match, ide_attach, NULL, NULL);

static int
ide_match(device_t parent, cfdata_t cf, void *aux)
{
	struct confargs *ca = aux;
	char name[32];

	if (strcmp(ca->ca_name, "ata") != 0)
		return 0;

	OF_getprop(ca->ca_node, "compatible", name, sizeof(name));
	aprint_normal("ide_match name: %s\n", name);
#if 0
	if (strcmp(name, "mpc5200-ata") != 0)
		return 0;
#endif

	return 1;
}

static void
ide_attach(device_t parent, device_t self, void *aux)
{
	struct ide_softc *sc = device_private(self);
	struct confargs *ca = aux;
	struct ata_channel *cp;
	struct wdc_regs *wdr;
	struct reg {
		uint32_t addr;
		uint32_t size;
	} range, *rp = &range;
	u_int32_t interrupts[3];
	int i;

	aprint_normal("\n");

	sc->sc_wdcdev.sc_atac.atac_dev = self;
	sc->sc_wdcdev.regs = &sc->sc_wdc_regs;

	if (OF_getprop(ca->ca_node, "reg", rp, sizeof(range))
	    != sizeof(range))
		return;

	/* Fill in wdc and channel infos */
	sc->sc_wdcdev.sc_atac.atac_cap |= ATAC_CAP_DATA16;
	sc->sc_wdcdev.sc_atac.atac_pio_cap = 0;
	sc->sc_wdcdev.sc_atac.atac_channels = sc->sc_chanarray;
	sc->sc_wdcdev.sc_atac.atac_nchannels = 1;
	
	sc->sc_chanarray[0] = &sc->sc_channel;
	cp = &sc->sc_channel;
	wdr = &sc->sc_wdc_regs;

	cp->ch_channel = 0;
	cp->ch_atac = &sc->sc_wdcdev.sc_atac;
	cp->ch_queue = &sc->sc_chqueue;
	cp->ch_ndrive = 1;
	wdr->cmd_iot = wdr->ctl_iot = ca->ca_tag;
	if (bus_space_map(wdr->cmd_iot, rp->addr, rp->size, 0, &wdr->cmd_baseioh) != 0) {
		aprint_normal_dev(self, "map failed\n");
		return;
	}

	for (i = 0; i < WDC_NREG; i++) {
		if (bus_space_subregion(wdr->cmd_iot, wdr->cmd_baseioh,
				(i * 4) + 0x60, 4, &wdr->cmd_iohs[i]) != 0) {
			bus_space_unmap(wdr->cmd_iot, wdr->cmd_baseioh, rp->size);
			aprint_normal_dev(self, "subregion failed\n");
			return;
		}
	}
	wdc_init_shadow_regs(cp);

	if (bus_space_subregion(wdr->ctl_iot, wdr->cmd_baseioh, 0x5c, 4,
				&wdr->ctl_ioh) != 0) {
		bus_space_unmap(wdr->cmd_iot, wdr->cmd_baseioh, rp->size);
		aprint_normal_dev(self, "subregion failed\n");
		return;
	}

	if (OF_getprop(ca->ca_node, "interrupts", interrupts, sizeof(interrupts))
	    != sizeof(interrupts))
		return;

	sc->sc_irq = (interrupts[0] * 32) + interrupts[1];
	sc->sc_ih = intr_establish(sc->sc_irq, IST_EDGE, IPL_BIO, wdcintr, sc);

	wdcattach(cp);
}
