$NetBSD$
--- wcfxs/wcfxs.c.orig	2008-09-04 08:14:04.000000000 +0100
+++ wcfxs/wcfxs.c	2008-10-30 09:35:14.000000000 +0000
@@ -1,5 +1,5 @@
 /*
- * Wilcard TDM400P TDM FXS/FXO Interface Driver for Zapata Telephony interface
+ * Wildcard TDM400P TDM FXS/FXO Interface Driver for Zapata Telephony interface
  *
  * Ported on FreeBSD by
  *      Alexander Timoshenko <gonzo@portaone.com>, PortaOne software
@@ -30,7 +30,6 @@
  */
 
 #include <sys/param.h>  /* defines used in kernel.h */
-#include <sys/module.h>
 #include <sys/systm.h>
 #include <sys/errno.h>
 #include <sys/kernel.h> /* types used in module initialization */
@@ -39,8 +38,10 @@
 #include <sys/malloc.h>
 #include <sys/lock.h>
 #include <sys/endian.h>
-#include <sys/bus.h>  /* structs, prototypes for pci bus stuff */
 
+#if defined(__FreeBSD__)
+#include <sys/bus.h>  /* structs, prototypes for pci bus stuff */
+#include <sys/module.h>
 #include <machine/bus.h>
 #include <sys/rman.h>
 #include <machine/resource.h>
@@ -57,18 +58,23 @@
 #include <dev/pci/pcireg.h>
 #include <dev/pci/pcivar.h>
 #endif
-
 #if __FreeBSD_version < 500000
 #include <sys/kobj.h>
 #endif
+#elif defined(__NetBSD__)
+#include <sys/device.h>
+#include <uvm/uvm_extern.h>     /* for PAGE_SIZE */
+
+#include <dev/pci/pcireg.h>
+#include <dev/pci/pcivar.h>
+#include <dev/pci/pcidevs.h>
+#endif
 
 #ifdef USE_SWI
 #include <sys/kthread.h>
 #include <sys/interrupt.h>
 #endif
 
-
-
 #include "proslic.h"
 #include "wcfxs.h"
 #include "wcfxsreg.h"
@@ -333,12 +339,29 @@
 #define MIN(a,b) (((a)<(b))?(a):(b))
 #endif
 
+#if defined(__NetBSD__)
+typedef struct device * device_t;
+#endif
+
 struct wcfxs_softc {
+#if defined(__FreeBSD__)
 	struct pci_dev *dev;
+	struct cdev * d; /* need to keep it for destroy_dev() */
+#elif defined(__NetBSD__)
+	struct device		sc_dv;		/* generic device */
+	bus_space_tag_t		sc_iot;
+	bus_space_handle_t	sc_ioh;
+	bus_addr_t		sc_iob;
+	bus_size_t		sc_iosize;
+
+	void			*sc_ih;
+	bus_dma_tag_t		sc_dmat;
+	pci_chipset_tag_t	sc_pct;
+#endif
+
 	char *variety;
 	struct zt_span span;
 	unsigned char ios;
-	struct cdev * d; /* need to keep it for destroy_dev() */
 	int usecount;
 	int intcount;
 	int opermode;
@@ -399,7 +422,12 @@
 	bus_space_handle_t 		wc_base_addr_handle;
 	int						wc_ioport_rid;
 
+#ifdef __FreeBSD__
 	struct wcfxs_data		wc_ldata;
+#elif defined(__NetBSD__)
+	struct wcfxs_dma	sc_read;
+	struct wcfxs_dma	sc_write;
+#endif
 
 	struct resource			* wc_irq;
 	void *					irq_handler;
@@ -461,19 +489,32 @@
 static void wcfxs_restart_dma(struct wcfxs_softc *sc);
 
 static int wcfxs_init_proslic(struct wcfxs_softc *sc, int card, int fast , int manual, int sane);
-static int wcfxs_detach(device_t dev);
+static int wcfxs_detach(device_t dev
+#ifdef __NetBSD__
+, int
+#endif
+);
 
 /* sysctl stuff */
+#ifdef __FreeBSD__
 extern struct sysctl_oid * zt_root;
 static struct sysctl_oid * wcfxs_root;
 static struct sysctl_ctx_list wcfxs_clist;
+#endif
 
 /* syscall handlers */
+#ifdef __FreeBSD__
 d_open_t wcfxs_sys_open;
 d_close_t wcfxs_sys_close;
 d_read_t wcfxs_sys_read;
+#elif defined(__NetBSD__)
+dev_type_open(wcfxs_sys_open);
+dev_type_close(wcfxs_sys_close);
+dev_type_read(wcfxs_sys_read);
+#endif
 
 /* Character device entry points */
+#if defined __FreeBSD__
 static struct cdevsw wcfxs_cdevsw = {
 #if __FreeBSD_version < 502103
 #ifdef MAJOR_AUTO
@@ -489,7 +530,86 @@
 	.d_close =   wcfxs_sys_close,
 	.d_read  =   wcfxs_sys_read
 };
+#elif defined __NetBSD__
+int wcfxs_match(struct device *, struct cfdata *, void *);
+static void wcfxs_attach(struct device *, struct device *, void *);
+
+CFATTACH_DECL(wcfxs, sizeof(struct wcfxs_softc),
+        wcfxs_match, wcfxs_attach, wcfxs_detach, NULL);
+/* no cdevsw, because there's no read/write interface as of yet */
+static void wcfxs_dma_free(struct wcfxs_softc *sc, struct wcfxs_dma *dma);
+static int wcfxs_dma_malloc(struct wcfxs_softc *sc, struct wcfxs_dma *dma);
+#endif
 
+#ifndef __FreeBSD__
+/*
+ * Compute number of ticks in the specified amount of time.
+ * (code from FreeBSD)
+ */
+int tvtohz(struct timeval *);
+int
+tvtohz(tv)
+	struct timeval *tv;
+{
+	register unsigned long ticks;
+	register long sec, usec;
+	static struct timeval firstcall;
+
+	/*
+	 * If the number of usecs in the whole seconds part of the time
+	 * difference fits in a long, then the total number of usecs will
+	 * fit in an unsigned long.  Compute the total and convert it to
+	 * ticks, rounding up and adding 1 to allow for the current tick
+	 * to expire.  Rounding also depends on unsigned long arithmetic
+	 * to avoid overflow.
+	 *
+	 * Otherwise, if the number of ticks in the whole seconds part of
+	 * the time difference fits in a long, then convert the parts to
+	 * ticks separately and add, using similar rounding methods and
+	 * overflow avoidance.  This method would work in the previous
+	 * case but it is slightly slower and assumes that hz is integral.
+	 *
+	 * Otherwise, round the time difference down to the maximum
+	 * representable value.
+	 *
+	 * If ints have 32 bits, then the maximum value for any timeout in
+	 * 10ms ticks is 248 days.
+	 */
+	if (firstcall.tv_sec == 0) {
+		firstcall.tv_sec = tv->tv_sec;
+		firstcall.tv_usec = tv->tv_usec;
+	}
+
+	sec = tv->tv_sec - firstcall.tv_sec;
+	usec = tv->tv_usec - firstcall.tv_usec;
+	if (usec < 0) {
+		sec--;
+		usec += 1000000;
+	}
+	if (sec < 0) {
+#ifdef DIAGNOSTIC
+		if (usec > 0) {
+			sec++;
+			usec -= 1000000;
+		}
+		printf("tvotohz: negative time difference %ld sec %ld usec\n",
+		       sec, usec);
+#endif
+		ticks = 1;
+	} else if (sec <= (LONG_MAX / hz)) {
+		ticks = sec * hz + ((unsigned long)usec * hz) / 1000000;
+	} else {
+		ticks = LONG_MAX;
+	}
+	if (ticks > INT_MAX)
+		ticks = INT_MAX;
+	return ((int)ticks);
+}
+
+#endif
+
+/* remove inlining for debugging */
+#define __inline 
 
 #ifdef USE_SWI
 static __inline void wcfxs_swi_transmitprep(struct wcfxs_softc *sc, unsigned char ints)
@@ -499,9 +619,9 @@
 	int x, idx;
 	if (ints & 0x01) 
 		/* Write is at interrupt address.  Start writing from normal offset */
-		writechunk = sc->wc_ldata.wc_writechunk;
+		writechunk = WRITECHUNK(sc);
 	else 
-		writechunk = sc->wc_ldata.wc_writechunk + ZT_CHUNKSIZE;
+		writechunk = WRITECHUNK(sc) + ZT_CHUNKSIZE;
 
     /* 
      * Write part get  the chunk and put it 
@@ -543,10 +663,10 @@
 	unsigned char c_readchunk[4][ZT_CHUNKSIZE];
 
 	if (ints & 0x08)
-		readchunk = sc->wc_ldata.wc_readchunk + ZT_CHUNKSIZE;
+		readchunk = READCHUNK(sc) + ZT_CHUNKSIZE;
 	else
 		/* Read is at interrupt address.  Valid data is available at normal offset */
-		readchunk = sc->wc_ldata.wc_readchunk;
+		readchunk = READCHUNK(sc);
 
 	for (x=0;x<ZT_CHUNKSIZE;x++) {
 		if (sc->cardflag & (1 << 3))
@@ -591,7 +711,7 @@
 	struct wcfxs_softc * sc = arg;
 	int x, y, idx;
 
-	FXS_LOCK(sc->swi_lock);
+	ZAP_LOCK(sc->swi_lock);
     for(y=0;y<NUM_CARDS;y++)
     {
         /*
@@ -612,7 +732,7 @@
         }
     }
 
-	FXS_UNLOCK(sc->swi_lock);
+	ZAP_UNLOCK(sc->swi_lock);
 
 
     /* do proper stuff */
@@ -632,7 +752,7 @@
 	/* Calculate Transmission */
 	zt_transmit(&sc->span);
     
-	FXS_LOCK(sc->swi_lock);
+	ZAP_LOCK(sc->swi_lock);
     for(y=0;y<NUM_CARDS;y++)
     {
         int idx;
@@ -661,7 +781,7 @@
 
         sc->kt_writebuf_inidx[y] = idx;
 	}
-	FXS_UNLOCK(sc->swi_lock);
+	ZAP_UNLOCK(sc->swi_lock);
 }
 #endif
 
@@ -674,9 +794,9 @@
 	int x;
 	if (ints & 0x01) 
 		/* Write is at interrupt address.  Start writing from normal offset */
-		writechunk = sc->wc_ldata.wc_writechunk;
+		writechunk = WRITECHUNK(sc);
 	else 
-		writechunk = sc->wc_ldata.wc_writechunk + ZT_CHUNKSIZE;
+		writechunk = WRITECHUNK(sc) + ZT_CHUNKSIZE;
 	/* Calculate Transmission */
 	zt_transmit(&sc->span);
 
@@ -691,9 +811,7 @@
 			writechunk[x] |= (sc->chans[1].writechunk[x] << 8);
 		if (sc->cardflag & (1 << 0))
 			writechunk[x] |= (sc->chans[0].writechunk[x]);
-		
 	}
-
 }
 
 static __inline void wcfxs_receiveprep(struct wcfxs_softc *sc, unsigned char ints)
@@ -702,10 +820,10 @@
 	int x;
 
 	if (ints & 0x08)
-		readchunk = sc->wc_ldata.wc_readchunk + ZT_CHUNKSIZE;
+		readchunk = READCHUNK(sc) + ZT_CHUNKSIZE;
 	else
 		/* Read is at interrupt address.  Valid data is available at normal offset */
-		readchunk = sc->wc_ldata.wc_readchunk;
+		readchunk = READCHUNK(sc);
 	for (x=0;x<ZT_CHUNKSIZE;x++) {
 		if (sc->cardflag & (1 << 3))
 			sc->chans[3].readchunk[x] = (readchunk[x] >> 24) & 0xff;
@@ -944,9 +1062,9 @@
 
 static void wcfxs_setreg(struct wcfxs_softc *sc, int card, unsigned char reg, unsigned char value)
 {
-	FXS_LOCK(sc->lock);
+	ZAP_LOCK(sc->lock);
 	__wcfxs_setreg(sc, card, reg, value);
-	FXS_UNLOCK(sc->lock);
+	ZAP_UNLOCK(sc->lock);
 }
 
 static unsigned char __wcfxs_getreg(struct wcfxs_softc *sc, int card, unsigned char reg)
@@ -964,19 +1082,19 @@
 
 static __inline void reset_spi(struct wcfxs_softc *sc, int card)
 {
-	FXS_LOCK(sc->lock);
+	ZAP_LOCK(sc->lock);
 	__wcfxs_setcard(sc, card);
 	__reset_spi(sc);
 	__reset_spi(sc);
-	FXS_UNLOCK(sc->lock);
+	ZAP_UNLOCK(sc->lock);
 }
 
 static unsigned char wcfxs_getreg(struct wcfxs_softc *sc, int card, unsigned char reg)
 {
 	unsigned char res;
-	FXS_LOCK(sc->lock);
+	ZAP_LOCK(sc->lock);
 	res = __wcfxs_getreg(sc, card, reg);
-	FXS_UNLOCK(sc->lock);
+	ZAP_UNLOCK(sc->lock);
 	return res;
 }
 
@@ -1024,14 +1142,14 @@
 		if (address == 255)
 			return 0;
 	}
-	FXS_LOCK(sc->lock);
+	ZAP_LOCK(sc->lock);
 	if(!__wait_access(sc, card)) {
 		__wcfxs_setreg(sc, card, IDA_LO,(unsigned char)(data & 0xFF));
 		__wcfxs_setreg(sc, card, IDA_HI,(unsigned char)((data & 0xFF00)>>8));
 		__wcfxs_setreg(sc, card, IAA,address);
 		res = 0;
 	};
-	FXS_UNLOCK(sc->lock);
+	ZAP_UNLOCK(sc->lock);
 	return res;
 }
 
@@ -1045,7 +1163,7 @@
 		if (address == 255)
 			return 0;
 	}
-	FXS_LOCK(sc->lock);
+	ZAP_LOCK(sc->lock);
 	if (!__wait_access(sc, card)) {
 		__wcfxs_setreg(sc, card, IAA, address);
 		if (!__wait_access(sc, card)) {
@@ -1057,7 +1175,7 @@
 			p = "Failed to wait inside\n";
 	} else
 		p = "failed to wait\n";
-	FXS_UNLOCK(sc->lock);
+	ZAP_UNLOCK(sc->lock);
 	if (p)
 		printf(p);
 	return res;
@@ -1108,12 +1226,12 @@
     return 0;
 }
 
-#if __FreeBSD_version < 700031
+#ifdef INTERRUPT_VOID
 static void
 #else
 static int
 #endif
-wcfxs_interrupt(void * arg)
+wcfxs_interrupt(void *arg)
 {
 	struct wcfxs_softc *sc = arg;
 	unsigned char ints;
@@ -1123,7 +1241,7 @@
 	WRITE1(WC_INTSTAT, ints);
 
 	if (!ints)
-#if __FreeBSD_version < 700031
+#ifdef INTERRUPT_VOID
 		return;
 #else
 		return FILTER_HANDLED;
@@ -1133,7 +1251,7 @@
 		/* Stop DMA, wait for watchdog */
 		printf("TDM PCI Master abort\n");
 		wcfxs_stop_dma(sc);
-#if __FreeBSD_version < 700031
+#ifdef INTERRUPT_VOID
 		return;
 #else
 		return FILTER_HANDLED;
@@ -1142,7 +1260,7 @@
 	
 	if (ints & 0x20) {
 		printf("PCI Target abort\n");
-#if __FreeBSD_version < 700031
+#if INTERRUPT_VOID
 		return;
 #else
 		return FILTER_HANDLED;
@@ -1206,10 +1324,10 @@
 		wcfxs_receiveprep(sc, ints);
 		wcfxs_transmitprep(sc, ints);
 #else
-	    FXS_LOCK(sc->swi_lock);
+	    ZAP_LOCK(sc->swi_lock);
         wcfxs_swi_receiveprep(sc, ints);
 		wcfxs_swi_transmitprep(sc, ints);
-	    FXS_UNLOCK(sc->swi_lock);
+	    ZAP_UNLOCK(sc->swi_lock);
         /* 
          * Copy readchunk/writechunk to buffers 
          *      and fire up swi 
@@ -1220,7 +1338,7 @@
 #endif
 	}
 
-#if __FreeBSD_version >= 700031
+#ifndef INTERRUPT_VOID
 	return FILTER_HANDLED;
 #endif
 }
@@ -1316,12 +1434,20 @@
 	wcfxs_setreg(sc, card, 14, 0x10);
 
 	/* Wait for one second */
+#ifdef __FreeBSD__
 	microuptime(&tv);
+#elif defined(__NetBSD__)
+	microtime(&tv);
+#endif
 	origjiffies = tvtohz(&tv);
 	jiffies = origjiffies;
 
 	while((vbat = wcfxs_getreg(sc, card, 82)) > 0x6) {
+#ifdef __FreeBSD__
 		microuptime(&tv);
+#elif defined(__NetBSD__)
+		microtime(&tv);
+#endif
 		jiffies = tvtohz(&tv);
 		if ((jiffies - origjiffies) >= (hz/2))
 			break;;
@@ -1352,7 +1478,11 @@
 	/* wcfxs_setreg(sc, card, 93, 0x99); */
 
 	/* Wait for VBat to powerup */
+#ifdef __FreeBSD__
 	microuptime(&tv);
+#elif defined(__NetBSD__)
+	microtime(&tv);
+#endif
 	jiffies = tvtohz(&tv);
 	origjiffies = jiffies;
 
@@ -1365,7 +1495,11 @@
 
 	while((vbat = wcfxs_getreg(sc, card, 82)) < 0xc0) {
 		/* Wait no more than 500ms */
+#ifdef __FreeBSD__
 		microuptime(&tv);
+#elif defined(__NetBSD__)
+		microtime(&tv);
+#endif
 		jiffies = tvtohz(&tv);
 		if ((jiffies - origjiffies) > hz/2) {
 			break;
@@ -1434,11 +1568,19 @@
 	wcfxs_setreg(sc, card, 97, 0x18); /* (0x18)Calibrations without the ADC and DAC offset and without common mode calibration. */
 	wcfxs_setreg(sc, card, 96, 0x47); /* (0x47)	Calibrate common mode and differential DAC mode DAC + ILIM */
 
+#ifdef __FreeBSD__
 	microuptime(&tv);
+#elif defined(__NetBSD__)
+	microtime(&tv);
+#endif
 	jiffies = tvtohz(&tv);
 	origjiffies=jiffies;
 	while( wcfxs_getreg(sc,card,96)!=0 ){
+#ifdef __FreeBSD__
 		microuptime(&tv);
+#elif defined(__NetBSD__)
+		microtime(&tv);
+#endif
 		jiffies = tvtohz(&tv);
 		if((jiffies-origjiffies)>80*hz/100)
 			return -1;
@@ -1452,9 +1594,13 @@
 /*******************************This is also available as a function *******************************************/
 	/* Delay 10ms */
 	origjiffies=jiffies; 
-	while((jiffies-origjiffies)<1*hz/100)
+	while((jiffies-origjiffies) < (hz/100) + 1)
 	{
+#ifdef __FreeBSD__
 		microuptime(&tv);
+#elif defined(__NetBSD__)
+		microtime(&tv);
+#endif
 		jiffies = tvtohz(&tv);
 	}
 
@@ -1472,13 +1618,21 @@
 	for ( i=0x1f; i>0; i--)
 	{
 		wcfxs_setreg(sc, card, 98,i);
+#ifdef __FreeBSD__
 		microuptime(&tv);
+#elif defined(__NetBSD__)
+		microtime(&tv);
+#endif
 		jiffies = tvtohz(&tv);
 		origjiffies=jiffies; 
 
 		while((jiffies-origjiffies)<4*hz/100)
 		{
+#ifdef __FreeBSD__
 			microuptime(&tv);
+#elif defined(__NetBSD__)
+			microtime(&tv);
+#endif
 			jiffies = tvtohz(&tv);
 		}
 
@@ -1489,12 +1643,20 @@
 	for ( i=0x1f; i>0; i--)
 	{
 		wcfxs_setreg(sc, card, 99,i);
+#ifdef __FreeBSD__
 		microuptime(&tv);
+#elif defined(__NetBSD__)
+		microtime(&tv);
+#endif
 		jiffies = tvtohz(&tv);
 		origjiffies=jiffies; 
 		while((jiffies-origjiffies)<4*hz/100)
 		{
+#ifdef __FreeBSD__
 			microuptime(&tv);
+#elif defined(__NetBSD__)
+			microtime(&tv);
+#endif
 			jiffies = tvtohz(&tv);
 		};
 		if((wcfxs_getreg(sc, card, 89)) == 0)
@@ -1504,13 +1666,21 @@
 /*******************************The preceding is the manual gain mismatch calibration****************************/
 /**********************************The following is the longitudinal Balance Cal***********************************/
 	wcfxs_setreg(sc, card, 64, 1);
+#ifdef __FreeBSD__
 	microuptime(&tv);
+#elif defined(__NetBSD__)
+	microtime(&tv);
+#endif
 	jiffies = tvtohz(&tv);
 	origjiffies=jiffies; 
 	/*  Sleep 100? */
 	while((jiffies-origjiffies)<10*hz/100)
 	{
+#ifdef __FreeBSD__
 		microuptime(&tv);
+#elif defined(__NetBSD__)
+		microtime(&tv);
+#endif
 		jiffies = tvtohz(&tv);
 	}	
 
@@ -1543,7 +1713,11 @@
 	wcfxs_setreg(sc, card, 96, 0x5f);
 
 	/* Wait for it to finish */
+#ifdef __FreeBSD__
 	microuptime(&tv);
+#elif defined(__NetBSD__)
+	microtime(&tv);
+#endif
 	jiffies = tvtohz(&tv);
 	origjiffies = jiffies;
 
@@ -1552,7 +1726,11 @@
 			printf("Timeout waiting for calibration of module %d\n", card);
 			return -1;
 		}
+#ifdef __FreeBSD__
 		microuptime(&tv);
+#elif defined(__NetBSD__)
+		microtime(&tv);
+#endif
 		jiffies = tvtohz(&tv);
 	}
 	
@@ -1574,13 +1752,14 @@
 	long jiffies;
 	struct timeval tv;
 	unsigned char reg16=0, reg26=0, reg30=0, reg31=0;
+	int ret;
 
 	sc->modtype[card] = MOD_TYPE_FXO;
 	/* Sanity check the ProSLIC */
 	reset_spi(sc, card);
 	if (!sane && wcfxs_voicedaa_insane(sc, card))
 	{
-		printf("ProSLIC sanity check failed\n");
+		printf("ProSLIC sanity check failed for card %d\n",card);
 		return -2;
 	}
 
@@ -1634,19 +1813,29 @@
 
 	/* Wait 1000ms for ISO-cap to come up */
 
+#ifdef __FreeBSD__
 	microuptime(&tv);
+#elif defined(__NetBSD__)
+	microtime(&tv);
+#endif
 	jiffies = tvtohz(&tv);
 	newjiffies = jiffies;
 	newjiffies += 2 * hz;
+	printf("jiffies: %ld -> %ld\n", jiffies, newjiffies);
 
 	while((jiffies < newjiffies) && !(wcfxs_getreg(sc, card, 11) & 0xf0))
 	{
+#ifdef __FreeBSD__
 		microuptime(&tv);
+#elif defined(__NetBSD__)
+		microtime(&tv);
+#endif
 		jiffies = tvtohz(&tv);
 	}
 
-	if (!(wcfxs_getreg(sc, card, 11) & 0xf0)) {
-		printf("VoiceDAA did not bring up ISO link properly!\n");
+	if (!((ret = wcfxs_getreg(sc, card, 11)) & 0xf0)) {
+		printf("VoiceDAA did not bring up ISO link properly [reg %d]!\n",
+			ret);
 		return -1;
 	}
 	if (debug)
@@ -2130,10 +2319,14 @@
 static int wcfxs_open(struct zt_chan *chan)
 {
 	struct wcfxs_softc *sc = chan->pvt;
-	if (!(sc->cardflag & (1 << (chan->chanpos - 1))))
+	if (!(sc->cardflag & (1 << (chan->chanpos - 1)))) {
+		printf("chanpos %d, no cardflag.\n", chan->chanpos);
 		return ENODEV;
-	if (sc->dead)
+	}
+	if (sc->dead) {
+		printf("we are the dead\n");
 		return ENODEV;
+	}
 	sc->usecount++;
 	return 0;
 }
@@ -2228,6 +2421,7 @@
 static int wcfxs_initialize(struct wcfxs_softc *sc)
 {
 	int x;
+	char buffer[256];
 	/* Zapata stuff */
 	sprintf(sc->span.name, "WCTDM/%d", sc->pos);
 	sprintf(sc->span.desc, "%s Board %d", sc->variety, sc->pos + 1);
@@ -2349,7 +2543,7 @@
 	WRITE1(WC_FSCDELAY, 0x0);
 
 	/* Setup DMA Addresses */
-	write_dma_addr = vtophys(sc->wc_ldata.wc_writechunk);
+	write_dma_addr = vtophys((vaddr_t) WRITECHUNK(sc));
 	
 	WRITE4(WC_DMAWS, (unsigned int)write_dma_addr); /* Write start */
 	WRITE4(WC_DMAWI, (unsigned int)write_dma_addr +
@@ -2357,8 +2551,8 @@
 	WRITE4(WC_DMAWE, (unsigned int)write_dma_addr +
 				ZT_CHUNKSIZE * 8 - 4); /* End */
 
-	/* read_dma_addr = vtophys(sc->wc_ldata.wc_readchunk); */
-	read_dma_addr =  vtophys(sc->wc_ldata.wc_readchunk);
+	/* read_dma_addr = vtophys(READCHUNK(sc)); */
+	read_dma_addr =  vtophys((vaddr_t) READCHUNK(sc));
 
 	WRITE4(WC_DMARS, (unsigned int)read_dma_addr); /* Read start */
 	WRITE4(WC_DMARI, (unsigned int)read_dma_addr +
@@ -2460,7 +2654,7 @@
 }
 
 static struct wcfxs_ident wcfxs_ident_table[] = {
-	/* {vendor, device, subverndor, card info }, */
+	/* {vendor, device, subvendor, card info }, */
 	{ 0xe159, 0x0001, 0xa159,	&wcfxs },
 	{ 0xe159, 0x0001, 0xe159,	&wcfxs },
 	{ 0xe159, 0x0001, 0xb100,	&wcfxse },
@@ -2493,7 +2687,7 @@
 }
 
 
-
+#ifdef __FreeBSD__
 static 
 int wcfxs_dma_allocate(struct wcfxs_softc * sc)
 {
@@ -2518,7 +2712,7 @@
 	}
 
 	error = bus_dmamem_alloc(sc->wc_ldata.wc_write_tag,
-		(void **)&sc->wc_ldata.wc_writechunk, BUS_DMA_NOWAIT,
+		(void **)&WRITECHUNK(sc), BUS_DMA_NOWAIT,
 		&sc->wc_ldata.wc_write_dmamap);
 
 	if (error) {
@@ -2530,13 +2724,13 @@
 	}
 
 	error = bus_dmamap_load(sc->wc_ldata.wc_write_tag,
-		sc->wc_ldata.wc_write_dmamap, sc->wc_ldata.wc_writechunk,
+		sc->wc_ldata.wc_write_dmamap, WRITECHUNK(sc),
 		ZT_MAX_CHUNKSIZE * 2 * 2 * 2 * 4, wcfxs_dma_map_addr,
 		&sc->wc_ldata.wc_write_dmaaddr, 0);
 
 	if (error) {
 			printf("wcfxs: can't load dma mapping");
-			bus_dmamem_free(sc->wc_ldata.wc_write_tag, sc->wc_ldata.wc_writechunk, 
+			bus_dmamem_free(sc->wc_ldata.wc_write_tag, WRITECHUNK(sc), 
 								sc->wc_ldata.wc_write_dmamap);
 			/* destroy map */
 			bus_dmamap_destroy(sc->wc_ldata.wc_write_tag, sc->wc_ldata.wc_write_dmamap);
@@ -2567,7 +2761,7 @@
 	}
 
 	error = bus_dmamem_alloc(sc->wc_ldata.wc_read_tag,
-		(void **)&sc->wc_ldata.wc_readchunk, BUS_DMA_NOWAIT,
+		(void **)&READCHUNK(sc), BUS_DMA_NOWAIT,
 		&sc->wc_ldata.wc_read_dmamap);
 
 	if (error) {
@@ -2579,13 +2773,13 @@
 	}
 
 	error = bus_dmamap_load(sc->wc_ldata.wc_read_tag,
-		sc->wc_ldata.wc_read_dmamap, sc->wc_ldata.wc_readchunk,
+		sc->wc_ldata.wc_read_dmamap, READCHUNK(sc),
 		ZT_MAX_CHUNKSIZE * 2 * 2 * 2 * 4, wcfxs_dma_map_addr,
 		&sc->wc_ldata.wc_read_dmaaddr, BUS_DMA_NOWAIT);
 
 	if (error) {
 			printf("wcfxs: can't load dma mapping");
-			bus_dmamem_free(sc->wc_ldata.wc_read_tag, sc->wc_ldata.wc_readchunk, 
+			bus_dmamem_free(sc->wc_ldata.wc_read_tag, READCHUNK(sc), 
 								sc->wc_ldata.wc_read_dmamap);
 			/* destroy map */
 			bus_dmamap_destroy(sc->wc_ldata.wc_read_tag, sc->wc_ldata.wc_read_dmamap);
@@ -2615,6 +2809,7 @@
 {
 	sysctl_ctx_free(&wcfxs_clist);
 }
+#endif /* FreeBSD */
 
 
 
@@ -2622,36 +2817,47 @@
 /*
  * Return identification string if this is device is ours.
  */
+#ifdef __FreeBSD__
 static int
 wcfxs_probe(device_t dev)
+#elif defined(__NetBSD__)
+static int  wcfxs_probe(struct device *dev, struct cfdata *cf, void *aux)
+#endif
 {
 	uint16_t			vid;
 	uint16_t			devid;
 	uint16_t			svid;
 	struct wcfxs_ident		*ident;
-
+#ifdef __NetBSD__
+	struct pci_attach_args *pa = aux;
+	pcireg_t subdev;
+        subdev = pci_conf_read(pa->pa_pc, pa->pa_tag, PCI_SUBSYS_ID_REG);
+
+	vid	= PCI_VENDOR(pa->pa_id);
+	devid 	= PCI_PRODUCT(pa->pa_id);
+	svid 	= PCI_VENDOR(subdev);
+#elif defined(__FreeBSD)
 	vid	= pci_get_vendor(dev);
-	devid 	= pci_get_device(dev);
-	svid 	= pci_get_subvendor(dev);
-
+	devid	= pci_get_device(dev);
+	svid	= pci_get_subvendor(dev);                        
+#endif
 	for (ident = wcfxs_ident_table; ident->device_info != NULL; ident++) {
 		/* printf("a)%x?%x %x?%x\n", ident->vid,vid, ident->did,devid);
 		printf("b)%x?%x %x?%x\n", ident->rid, revid, ident->rid, -1); */
 		if ((ident->vid == vid) && (ident->did == devid) &&
 			(ident->svid == svid || ident->svid == -1)) {
 
-			printf("FXS device: vendor=%x device=%x subvendor=%x\n", 
+			printf("ZapTel device: vendor=%x device=%x subvendor=%x\n", 
 				vid, devid, svid);
-			device_set_desc(dev, ident->device_info->name);
-			return (0);
+			return (1);
 		}
 	}
-	return (ENXIO);
+	return (0);
 }
 
 /* Attach function is only called if the probe is successful */
-static int
-wcfxs_attach(device_t dev)
+#ifdef __FreeBSD__
+static int wcfxs_attach(device_t dev)
 {
 	u_int16_t 				error;
 	u_int32_t 				command;
@@ -2698,10 +2904,10 @@
 	unit = device_get_unit(dev);
 	/* not realy necessary  */
 	memset(sc, 0, sizeof(struct wcfxs_softc));
-	FXS_LOCK_INIT(sc->lock, "TDM card lock");
+	ZAP_LOCK_INIT(sc->lock, "TDM card lock");
 
 #ifdef USE_SWI
-	FXS_LOCK_INIT(sc->swi_lock, "TDM SWI lock");
+	ZAP_LOCK_INIT(sc->swi_lock, "TDM SWI lock");
 #endif
 
 	printf("FXS Attach for wcfxs%d: deviceID : 0x%x\n",
@@ -2752,7 +2958,7 @@
 
 	/* Alocate memory for two zt chunks - receive and transmit */
 
-	if(wcfxs_dma_allocate(sc))
+	if (wcfxs_dma_allocate(sc))
 	{
 		printf("wcfxs: Unable to intialize DMA memory\n");
 		wcfxs_detach(dev);
@@ -2785,7 +2991,7 @@
 
 	/* Now we should set up the interrupt handler */
 
-#if __FreeBSD_version < 700031
+#if __FreeBSD_version < 700031 || defined(__NetBSD__)
 	error = bus_setup_intr(dev, sc->wc_irq, INTR_TYPE_CLK | INTR_FAST,
 		wcfxs_interrupt, sc, &(sc->irq_handler));
 	if (error) {
@@ -2828,10 +3034,10 @@
 
 	wcfxs_generate_description(sc);
 	/* Initialize Write/Buffers to all blank data */
-	memset((void *)(sc->wc_ldata.wc_writechunk), 0, 
+	memset((void *)(WRITECHUNK(sc)), 0, 
 			ZT_MAX_CHUNKSIZE * 2 * 2 * 2 * 4);
 
-	memset((void *)(sc->wc_ldata.wc_readchunk), 0, 
+	memset((void *)(READCHUNK(sc)), 0, 
 			ZT_MAX_CHUNKSIZE * 2 * 2 * 2 * 4);
 
 #ifdef USE_SWI
@@ -2865,9 +3071,240 @@
 	printf("We failed: %d\n", error);
 	return error;
 }
+#elif defined(__NetBSD__)
+static void wcfxs_attach(struct device *parent, struct device *dev, void *aux)
+{
+	struct wcfxs_softc *sc = (struct wcfxs_softc *) dev;
+	struct pci_attach_args *pa = aux;
+	pci_chipset_tag_t pc = pa->pa_pc;
+	pcireg_t subdev;
+	pci_intr_handle_t ih;
+	const char *intrstr;
+	struct wcfxs_ident *id;
+	int i;
+	int error;
+	int opermode;
+
+#ifdef _LKM
+	malloc_type_attach(M_TDM);  /* XXX this  belongs elsewhere */
+#endif
+
+	/* read the subdevice information - XXX code duplication */
+	subdev = pci_conf_read(pc, pa->pa_tag, PCI_SUBSYS_ID_REG);
+
+	for (id = wcfxs_ident_table ; id->device_info != NULL; id++)
+		if (id->vid == PCI_VENDOR(pa->pa_id) &&
+			id->did == PCI_PRODUCT(pa->pa_id) &&
+			(id->svid == PCI_VENDOR(subdev) ||
+			id->svid == (pci_vendor_id_t) -1))
+			break;
+
+	if (id->device_info == NULL) {
+		printf("Card info not found\n");
+		return;
+	}
+
+	printf(": %s\n",id->device_info->name);
+
+#ifdef OPERMODE
+	opermode=OPERMODE;
+#else
+	opermode=0;
+#endif
+	if (opermode < 0 ||
+	    opermode >= sizeof(fxo_modes) / sizeof(fxo_modes[0])) {
+		int i;
+
+		printf("Invalid/unknown operating mode %d specified.  Please choose one of:\n", opermode);
+		for (i = 0; i < sizeof(fxo_modes) / sizeof(fxo_modes[0]); i++)
+			printf("  %d (%s)\n", i, fxo_modes[i].name);
+		error = ENODEV;
+		goto fail;
+	}
+	aprint_normal("%s: operating mode: %s\n", 
+		sc->sc_dv.dv_xname, fxo_modes[opermode].name);
+	sc->opermode = opermode;
+
+	/*
+	 * Make sure bus mastering, ioports, and memory mapping are enabled.
+	 */
+	pci_conf_write(pc, pa->pa_tag, PCI_COMMAND_STATUS_REG,
+		pci_conf_read(pc, pa->pa_tag, PCI_COMMAND_STATUS_REG) |
+		PCI_COMMAND_MASTER_ENABLE | PCI_COMMAND_IO_ENABLE |
+		PCI_COMMAND_MEM_ENABLE);
+
+	if (!(pci_conf_read(pc, pa->pa_tag, PCI_COMMAND_STATUS_REG) &
+		PCI_COMMAND_IO_ENABLE)) {
+		printf("%s: failed to enable I/O ports\n",
+			sc->sc_dv.dv_xname);
+		goto fail;
+	}
+	if (!(pci_conf_read(pc, pa->pa_tag, PCI_COMMAND_STATUS_REG) &
+		PCI_COMMAND_MEM_ENABLE)) {
+		printf("%s: failed to enable memory mapping\n",
+			sc->sc_dv.dv_xname);
+		goto fail;
+	}
+
+	if ((error = pci_mapreg_map(pa, PCIR_BARS, PCI_MAPREG_TYPE_IO, 0,
+		&sc->sc_iot, &sc->sc_ioh, &sc->sc_iob, &sc->sc_iosize)) != 0) {
+	    aprint_error("%s: can't map i/o space, error = %d\n",
+		sc->sc_dv.dv_xname, error);
+	    goto fail;
+	}
+	sc->sc_pct = pc;
+	sc->sc_dmat = pa->pa_dmat;
+	sc->curcard = -1;
+	sc->cards = 4;   /* XXX magic number? */
+	sc->pos = sc->sc_dv.dv_unit;
+	sc->variety = id->device_info->name;
+	for (i = 0; i < NUM_CARDS; i++)
+		sc->flags[i] = id->device_info->flags;
+	sc->usecount = 0;
+
+	/*
+	 * so far as I can tell, this is just wrong; we don't need to
+	 * scan for anything, because zaptel's already loaded; all we
+	 * need is zt_register
+	 */
+	// zaptel_attach_mi(&sc->sc_dv);
+
+	/*
+	 * Set up DMA mappings
+	 */
+	if (!wcfxs_dma_malloc(sc, &sc->sc_read))
+	    goto fail;
+	if (!wcfxs_dma_malloc(sc, &sc->sc_write))
+	    goto fail_1;
+
+	if (wcfxs_initialize(sc)) {
+	    unsigned char x;
+
+	    /* set reset low */
+	    x = READ1(WC_CNTL);
+	    WRITE1(WC_CNTL, (~0x1) & x);
+	    aprint_error("%s: unable to initialize FXS\n",
+		sc->sc_dv.dv_xname);
+	    goto fail_2;
+	}
+
+	/* Map and establish the interrupt */
+	if (pci_intr_map(pa, &ih)) {
+	    aprint_error("%s: can't map interrupt\n", sc->sc_dv.dv_xname);
+	    goto fail;
+	}
+
+	intrstr = pci_intr_string(pc, ih);
+	sc->sc_ih = pci_intr_establish(pc, ih, IPL_NET, wcfxs_interrupt, sc);
+	if (sc->sc_ih == NULL) {
+	    aprint_error("%s: unable to establish interrupt",
+		sc->sc_dv.dv_xname);
+	    if (intrstr != NULL)
+		aprint_normal(" at %s", intrstr);
+	    aprint_normal("\n");
+	    goto fail;
+	}
+	aprint_normal("%s: interrupting at %s\n",sc->sc_dv.dv_xname, intrstr);
+
+	if (wcfxs_hardware_init(sc)) {
+		unsigned char x;
+
+		/* set reset low */
+		x = READ1(WC_CNTL);
+		WRITE1(WC_CNTL, (~0x1) & x);
+
+		goto fail_2;
+	}
+
+	wcfxs_post_initialize(sc);
+	wcfxs_generate_description(sc);
+
+	/* Initialize buffers to 0 */
+	memset((void *) sc->sc_read.chunk, 0, WCFXS_SIZE);
+	memset((void *) sc->sc_write.chunk, 0, WCFXS_SIZE);
+
+	wcfxs_enable_interrupts(sc);
+	wcfxs_start_dma(sc);
+printf("wcfxs started\n");
+
+	return;
+
+ fail_2:
+	wcfxs_dma_free(sc, &sc->sc_write);
+ fail_1:
+	wcfxs_dma_free(sc, &sc->sc_read);
+ fail:
+	return;
+}
+
+/* wcfxs_dma_malloc() returns 0 on failure, 1 on success */
+static int
+wcfxs_dma_malloc(struct wcfxs_softc *sc, struct wcfxs_dma *dma)
+{
+	int error;
+
+	/* set up read buffer first */
+	if ((error = bus_dmamem_alloc(sc->sc_dmat, WCFXS_SIZE, PAGE_SIZE, 0,
+	    &dma->dma_seg, 1, &dma->dma_nseg, BUS_DMA_NOWAIT)) != 0) {
+		aprint_error(
+		    "%s: unable to allocate chunk, error = %d\n",
+		    sc->sc_dv.dv_xname, error);
+		goto fail_0;
+	}
+
+	if ((error = bus_dmamem_map(sc->sc_dmat, &dma->dma_seg,
+	    dma->dma_nseg, WCFXS_SIZE, (caddr_t *)&dma->chunk,
+	    BUS_DMA_NOWAIT)) != 0) {
+		aprint_error(
+		    "%s: unable to map chunk, error = %d\n",
+		    sc->sc_dv.dv_xname, error);
+		goto fail_1;
+	}
+
+	if ((error = bus_dmamap_create(sc->sc_dmat, WCFXS_SIZE, 1, WCFXS_SIZE,
+								   0, 0, &dma->dma_map)) != 0) {
+		aprint_error("%s: unable to create DMA map, "
+		    "error = %d\n",sc->sc_dv.dv_xname, error);
+		goto fail_2;
+	}
+
+	if ((error = bus_dmamap_load(sc->sc_dmat, dma->dma_map,
+	     dma->chunk, WCFXS_SIZE, NULL, BUS_DMA_NOWAIT)) != 0) {
+		aprint_error(
+		    "%s: unable to load DMA map, error = %d\n",
+		    sc->sc_dv.dv_xname, error);
+		goto fail_3;
+	}
+
+	/* success! */
+	return 1;
+
+ fail_3:
+	bus_dmamap_destroy(sc->sc_dmat, dma->dma_map);
+ fail_2:
+	bus_dmamem_unmap(sc->sc_dmat, (caddr_t)dma->chunk, WCFXS_SIZE);
+ fail_1:
+	bus_dmamem_free(sc->sc_dmat, &dma->dma_seg,
+	    dma->dma_nseg);
+ fail_0:
+	return 0;
+}
+
+static void
+wcfxs_dma_free(struct wcfxs_softc *sc, struct wcfxs_dma *dma)
+{
+	int size = ZT_MAX_CHUNKSIZE * 2 * 2 * 2 * 4; /* XXX magic */
+
+	bus_dmamap_destroy(sc->sc_dmat, dma->dma_map);
+	bus_dmamem_unmap(sc->sc_dmat, (caddr_t)dma->chunk, size);
+	bus_dmamem_free(sc->sc_dmat, &dma->dma_seg,
+	    dma->dma_nseg);
+}
+#endif
 
 
 /* Detach device. */
+#ifdef __FreeBSD__
 static int
 wcfxs_detach(device_t dev)
 {
@@ -2905,7 +3342,7 @@
 		if (sc->wc_ldata.wc_read_tag) {
 			bus_dmamap_unload(sc->wc_ldata.wc_read_tag,
 				sc->wc_ldata.wc_read_dmamap);
-			bus_dmamem_free(sc->wc_ldata.wc_read_tag, sc->wc_ldata.wc_readchunk,
+			bus_dmamem_free(sc->wc_ldata.wc_read_tag, READCHUNK(sc),
 				sc->wc_ldata.wc_read_dmamap);
 			bus_dma_tag_destroy(sc->wc_ldata.wc_read_tag);
 		}
@@ -2913,7 +3350,7 @@
 		if (sc->wc_ldata.wc_write_tag) {
 			bus_dmamap_unload(sc->wc_ldata.wc_write_tag,
 				sc->wc_ldata.wc_write_dmamap);
-			bus_dmamem_free(sc->wc_ldata.wc_write_tag, sc->wc_ldata.wc_writechunk,
+			bus_dmamem_free(sc->wc_ldata.wc_write_tag, WRITECHUNK(sc),
 				sc->wc_ldata.wc_write_dmamap);
 			bus_dma_tag_destroy(sc->wc_ldata.wc_write_tag);
 		}
@@ -2927,9 +3364,9 @@
 		else
 			sc->dead = 1;
 
-		FXS_LOCK_DESTROY(sc->lock);
+		ZAP_LOCK_DESTROY(sc->lock);
 #ifdef USE_SWI
-	    FXS_LOCK_DESTROY(sc->swi_lock);
+	    ZAP_LOCK_DESTROY(sc->swi_lock);
 #endif
 		if(sc->d)
 			destroy_dev(sc->d);
@@ -2945,6 +3382,45 @@
 	return (0);
 }
 
+#elif defined(__NetBSD__)
+static int
+wcfxs_detach(struct device* self, int flags)
+{
+	struct wcfxs_softc *sc = (struct wcfxs_softc *) self;
+
+	if (sc) {
+		/* don't unload if module is in use */
+		if (sc->usecount)
+			return (EBUSY);
+	}
+
+	wcfxs_disable_interrupts(sc);
+	if (sc->sc_ih != NULL)
+	    pci_intr_disestablish(sc->sc_pct, sc->sc_ih);
+	wcfxs_stop_dma(sc);
+	wcfxs_reset_tdm(sc);
+
+	/* free read/write dma buffers */
+	wcfxs_dma_free(sc, &sc->sc_read);
+	wcfxs_dma_free(sc, &sc->sc_write);
+
+	/* Reset PCI chip and registers */
+	WRITE1(WC_CNTL, 0x0e);
+
+	wcfxs_release(sc);
+
+	if (sc->sc_iosize)
+	    bus_space_unmap(sc->sc_iot, sc->sc_ioh, sc->sc_iosize);
+
+#ifdef _LKM
+	malloc_type_detach(M_TDM);  /* XXX this  belongs elsewhere */
+#endif
+	aprint_naive("FXS detach!\n");
+
+	return (0);
+}
+#endif
+
 /* Called during system shutdown after sync. */
 static int
 wcfxs_shutdown(device_t dev)
@@ -2976,6 +3452,35 @@
 	zt_unregister(&sc->span);
 }
 
+#ifdef __NetBSD__
+int
+wcfxs_match(struct device *parent, struct cfdata *match, void *aux)
+{
+        struct pci_attach_args *pa = aux;
+        pcireg_t subdev;
+        struct wcfxs_ident *id;
+
+        /* read the subdevice information */
+        subdev = pci_conf_read(pa->pa_pc, pa->pa_tag, PCI_SUBSYS_ID_REG);
+
+        for (id = wcfxs_ident_table; id->device_info != NULL; id++) {
+                if (id->vid == PCI_VENDOR(pa->pa_id) &&
+                        id->did == PCI_PRODUCT(pa->pa_id) &&
+                        (id->svid == PCI_VENDOR(subdev) ||
+                        id->svid == (pci_vendor_id_t) -1)) {
+
+                        printf("FXS device: ");
+                        printf("vendor=%x ",PCI_VENDOR(pa->pa_id));
+                        printf("product=%x ",PCI_PRODUCT(pa->pa_id));
+                        printf("subvendor=%x\n",PCI_VENDOR(subdev));
+                        return (1);
+                }
+        }
+        return (0);
+}
+#endif
+
+#ifdef __FreeBSD__
 /* 
  * syscall routines 
  * We will use them to emulate Linux /proc behaviour --
@@ -3025,6 +3530,9 @@
 	return uiomove(sc->card_configuration +  uio->uio_offset, size, uio);
 }
 
+#endif /* __FreeBSD__ */
+
+#ifdef __FreeBSD__
 static int
 wcfxs_load(module_t mod, int cmd, void *arg)
 {
@@ -3049,8 +3557,6 @@
 	return err;
 }
 
-
-
 static device_method_t wcfxs_methods[] = {
 	/* Device interface */
 	DEVMETHOD(device_probe,	 	wcfxs_probe),
@@ -3095,3 +3601,4 @@
 /* DRIVER_MODULE(wcfxs, pci, wcfxs_driver, wcfxs_devclass, 0, 0); */
 DECLARE_MODULE(wcfxs, wcfxs_pci_mod, SI_SUB_DRIVERS, SI_ORDER_MIDDLE);
 #endif
+#endif /* __FreeBSD__ */
