/* **********************************************************
 * Copyright (C) 1998-2000 VMware, Inc.
 * All Rights Reserved
 * $Id: vmx86.h,v 1.3 2003/02/16 15:32:33 bad Exp $
 * **********************************************************/

/*
 *  vmx86.h - Platform independent data and interface for supporting 
 *            the vmx86 device driver. 
 */

#ifndef VMX86_H
#define VMX86_H

#define INCLUDE_ALLOW_MODULE
#include "includeCheck.h"


/*
 * The ifdefs we use in the VMDriver structure are not really what I call
 * "platform independent"... --hpreg
 */
#ifdef __linux__
#   include "linux-version.h"
#endif

#include "x86.h"
#include "modulecall.h"
#include "vm_types.h"
#include "initblock.h"
#include "memmgmt.h"
#include "wslimits.h"


/*
 * VMDriver - the main data structure for the driver side of a 
 *            virtual machine.
 */ 

#define NUM_PASSTHROUGH_IRQS 10

typedef struct VMDriver {
   int                 count;        /* reference count */
   int                 id;           /* unique VM ID */
   struct VMDriver    *nextDriver;   /* Next on list of all VMDrivers */
   void               *processID;    /* User level process handle. */
   void               *uniqHandle;   /* Unique handle of driver */

   int                 logFD;        /* File descriptor for log messages */
   struct VMCrossPage *crosspage;    /* Pointer to the crossover page 
				      * shared with the monitor.  */
   struct page        *crosspagePage;
#ifdef WINNT_DDK
   void               *crossPageMDL;
#endif
#ifdef __FreeBSD__
  struct KernelAddr *driverPtrs;
#endif
#ifdef __NetBSD__
  void *		crossvaddr;
  void *		crossuaddr;
#endif
   volatile uint32   (*hostAPIC)[4]; /* kseg pointer to host APIC */

   void               *memtracker;   /* Memory tracker pointer */
   struct PhysTracker *physTracker;
   int                 releaseCount;
   Bool                checkFuncFailed;
   struct PerfCounter *perfCounter;
   VMX86Stats	       stats;
   VMMemMgmtInfo       memMgmtInfo;
   WslimitsInfo        wslimitsInfo;

#ifdef SUPPORT_PASSTHROUGH
   /* When Passthrough IRQs fire, we buffer them here until queried */
   int                 numPendingPassthroughIRQs;
   unsigned char       pendingPassthroughIRQs[NUM_PASSTHROUGH_IRQS];
#endif
} VMDriver;


extern VMDriver *Vmx86_CreateVM(void *uniqHandle, void *processId, 
				WslimitsInfo *minfo);
extern VMDriver *Vmx86_BindVM(int id);
extern int Vmx86_ReleaseVM(VMDriver *vm);
extern int Vmx86_InitVM(VMDriver *vm, InitBlock *initParams);
extern int Vmx86_LateInitVM(VMDriver *vm);
extern int Vmx86_RunVM(VMDriver *vm);
extern void Vmx86_SetStartTime(VmTimeStart *st);
extern uint32 Vmx86_GetMHzEstimate(VmTimeStart *st);
extern VMDriver *Vmx86_GetVMforProcess(void *processId);
extern int Vmx86_LockPage(VMDriver *vm, void *addr, Bool userPage);
extern int Vmx86_UnlockPage(VMDriver *vm, void *addr, Bool userPage);
extern int32 Vmx86_GetNumVMs(void);
extern int32 Vmx86_GetTotalMemUsage(void);
extern int32 Vmx86_GetLockedPagesLimit(void);
extern Bool Vmx86_SetLockedPagesLimit(int32 limit);
extern void Vmx86_SetLockedPageLimitMonitorStatus(Bool status);
extern int32 Vmx86_ChangeLockedPagesLimit(int32 limit);
extern Bool Vmx86_GetMemInfo(VMDriver *curVM,
                             int32 curVMOnly,
                             VMGetMemInfoArgs *args,
                             int outArgsLength);
extern Bool Vmx86_GetMemInfoCopy(VMDriver *curVM,
				 VMGetMemInfoArgs *args);
extern void Vmx86_SetMemInfo(VMDriver *curVM, VMSetMemInfoArgs *args);
extern void Vmx86_Add2MonPageTable(VMDriver *vm, VPN vpn, MPN mpn,
				   Bool readOnly);
extern Bool Vmx86_PAEEnabled(void);

#endif 
