/*
 * Copyright (c) 2004, 2005, id Quantique SA, Switzerland
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * Redistributions of source code must retain the above copyright notice, this
 * list of conditions and the following disclaimer.
 * Redistributions in binary form must reproduce the above copyright notice,
 * this list of conditions and the following disclaimer in the documentation
 * and/or other materials provided with the distribution.
 * Neither the name of id Quantique nor the names of its contributors may be
 * used to endorse or promote products derived from this software without
 * specific prior written permission.
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>

#include <unistd.h>

#include <string.h>
#include <errno.h>

#include <iostream>
#include <sstream>

#include "QuantisCpp.h"

namespace Quantis {
  FilePtr::FilePtr(string filename, int mode) {
    openFile(filename, mode);
  }

  FilePtr::FilePtr(string baseName, int cardNumber, int mode) {
    std::ostringstream filename;
    filename << baseName << cardNumber;
    openFile(filename.str(), mode);
  }

  void FilePtr::openFile(string filename, int mode) {
    fd = open(filename.c_str(), mode);
    if (fd < 0)
      throw QuantisError("Could not open " + filename + ": "
			     + strerror(errno));
  }

  unsigned int FilePtr::readIoctl(int request) {
    unsigned int value;
    
    if (ioctl(fd, request, &value) < 0) {
      std::ostringstream error;
      error << "Error while doing ioctl " << request << ": " << strerror(errno);
      throw QuantisError(error.str());
    }

    return value;
  }

  void FilePtr::writeIoctl(int request, unsigned int value) {
    if (ioctl(fd, request, &value) < 0) {
      std::ostringstream error;
      error << "Error while doing ioctl " << request << ": " << strerror(errno);
      throw QuantisError(error.str());
    }
  }

  unsigned int FilePtr::read(unsigned char *buffer, unsigned int length) {
    int ret;
    if ((ret = ::read(fd, buffer, length)) < 0) {
      std::ostringstream error;
      error << "Error while reading " << length << " bytes : " << strerror(errno);
      throw QuantisError(error.str());
    }

    return ret;
  }

  void FilePtr::readBlocking(unsigned char *buffer, unsigned int length) {
    unsigned char *ptr = buffer;
    do {
      int readBytes = read(ptr, length);
      if (readBytes < 0) {
	std::ostringstream error;
	error << "Error while reading " << length << " bytes: " << strerror(errno);
	throw QuantisError(error.str());
      }
      length -= readBytes;
      ptr += readBytes;
    } while (length > 0);
  }

  int Quantis::getCardCount(string baseName) {
    for (unsigned int i = 0; i < Quantis::MaxCards; i++) {
      try {
	FilePtr fd(baseName, i, O_RDONLY);
	return fd.readIoctl(QUANTIS_IOCTL_GET_CARD_COUNT);
      } catch (QuantisError e) {
      }
    }
    return 0;
  }

  int Quantis::getDriverVersion(string baseName) {
    for (unsigned int i = 0; i < Quantis::MaxCards; i++) {
      try {
	FilePtr fd(baseName, i, O_RDONLY);
	return fd.readIoctl(QUANTIS_IOCTL_GET_CARD_COUNT);
      } catch (QuantisError e) {
      }
    }
    throw QuantisError("Could find no card device to get the driver version");
  }

  unsigned char Quantis::readByte() {
    unsigned char buf[1];
    fd.readBlocking(buf, 1);
    return buf[0];
  }

  unsigned short Quantis::readShort() {
    unsigned char buf[2];
    fd.readBlocking(buf, 2);
    return buf[0] | (buf[1] << 8);
  }

  unsigned long Quantis::readLong() {
    unsigned char buf[4];
    fd.readBlocking(buf, 4);
    return buf[0] | (buf[1] << 8) | (buf[2] << 16) | (buf[3] << 24);
  }

  unsigned int Quantis::read(unsigned char *buffer, unsigned int length) {
    return fd.read(buffer, length);
  }

  void Quantis::readBlocking(unsigned char *buffer, unsigned int length) {
    fd.readBlocking(buffer, length);
  }
}
