/*
 * Copyright (c) 2004, 2005, id Quantique SA, Switzerland
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * Redistributions of source code must retain the above copyright notice, this
 * list of conditions and the following disclaimer.
 * Redistributions in binary form must reproduce the above copyright notice,
 * this list of conditions and the following disclaimer in the documentation
 * and/or other materials provided with the distribution.
 * Neither the name of id Quantique nor the names of its contributors may be
 * used to endorse or promote products derived from this software without
 * specific prior written permission.
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

/**# Examples for the Quantis PCI library
 **
 ** This part shows examples of applications using the Quantis
 ** library. It is automatically generated from the `examples.c' file
 ** and contains the complete sourcecode.
 **/

#include <stdio.h>
#include "quantis.h"

/**## Getting information about Quantis boards
 **
 ** This example shows how the Quantis library can be used to get
 ** information about the present Quantis PCI boards.
 **
 ** First, we define a function to print module bitmasks. If the bit
 ** corresponding to a module is set in the bitmask, its name is
 ** printed on the standard output.
 **/
void quantis_example_print_bitmask(int bitmask) {
  int i;
  for (i = 0; i < 4; i++) {
    if (bitmask & (1 << i)) {
      printf("Bit for module %d is set\n", i);
    }
  }
}

/**
 ** The main function of this example first prints some information
 ** about the library version and the driver version. It then gets the
 ** number of installed Quantis PCI boards, and cycles through the
 ** boards, printing their board version, the names of the enabled
 ** modules and the status of the enabled modules.
 **/
void quantis_example_information(void) {
  int count;
  int i;
  int lib_version, driver_version;

  /** Print the version information. If we can't get the driver
   ** version, we print an error message and quit.
   **/
  lib_version = quantisLibVersion();
  printf("Quantis Library Version: %d.%d\n", lib_version / 10, lib_version % 10);
  driver_version = quantisDriverVersion();
  if (driver_version == 0) {
    printf("Could not get the Quantis driver version.\n");
    printf("This could mean that no card is installed.\n");
    return;
  } else {
    printf("Quantis Driver Version: %d.%d\n", driver_version / 10,
	   driver_version % 10);
  }
  
  count = quantisCount();
  printf("Quantis information example: found %d Quantis PCI boards\n", count);

  /** Now we print information about each Quantis PCI board. First we
   ** get the board version, and jump to the next board if we get an
   ** error. Then we get the modules mask, which shows us which modules
   ** are present on the PCI board. We then print the current status
   ** of each enabled module if some modules are present.
   **/
  for (i = 0; i < count; i++) {
    int modules_mask;
    int status_mask;
    int board_version;
    
    board_version = quantisBoardVersion(i);
    if (board_version == 0) {
      printf("Could not get the board version of the Quantis PCI board nr. %d.\n",
	     i);
      continue;
    } else {
      printf("Quantis PCI board nr %d, board version: %.4x\n", i, board_version);
    }

    modules_mask = quantisGetModules(i);
    if (modules_mask == 0) {
      printf("Quantis PCI board nr %d: no modules are installed\n", i);
      continue;
    } else {
      printf("Quantis PCI board nr %d: showing present modules\n", i);
      quantis_example_print_bitmask(modules_mask);
    }

    status_mask = quantisModulesStatus(i);
    if (status_mask < 0) {
      printf("Quantis PCI board nr %d: could not get the modules status\n", i);
      continue;
    } else {
      printf("Quantis PCI board nr %d: showing the status of enabled modules\n", i);
      quantis_example_print_bitmask(status_mask);
    }
  }
}

/**## Reading random data from a Quantis PCI board
 **
 ** In this example, we read random data from the first Quantis PCI
 ** board present in the system.
 **/
void quantis_example_read(void) {
  unsigned char buf[4096];
  int bytes;

  printf("Quantis read example: trying to read 4096 bytes from the first PCI board\n");
  bytes = quantisRead(0, buf, 4096);
  if (bytes < 0) {
    printf("Quantis read example: got an error while trying to read 4096 bytes\n");
  } else {
    printf("Quantis read example: read %d bytes from the first PCI board\n", bytes);
  }
}

/**## Test main routine
 **
 ** This is the main routine for the test program, which calls all the
 ** test routines.
 **/
int main(void) {
  quantis_example_information();
  quantis_example_read();
  return 0;
}
