/*
 * qrng gets true random numbers via Quantis
 * Copyright (c) 2004, 2005, id Quantique SA, Switzerland
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * Redistributions of source code must retain the above copyright notice, this
 * list of conditions and the following disclaimer.
 * Redistributions in binary form must reproduce the above copyright notice,
 * this list of conditions and the following disclaimer in the documentation
 * and/or other materials provided with the distribution.
 * Neither the name of id Quantique nor the names of its contributors may be
 * used to endorse or promote products derived from this software without
 * specific prior written permission.
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#define VERSION "0.1"

#ifdef WIN32  
#include "getopt.h"
#else
#include <unistd.h>
#endif

#include <iostream>
#include <fstream>
#include "QuantisCpp.h"

using std::cerr;
using std::endl;

static void usage(void) {
  cerr << "qrng v" + (std::string)VERSION + " Copyright (C) 2004 id Quantique SA" << endl
       << "Usage: ./qrng -n bytes [-b|-d max] [-o output] [-c] [-p cardnumber]" << endl
       << "-n : amount of bytes" << endl
       << "-b : dump in binary (default in ASCII hex) " << endl
       << "-d : dump in decimal with max the max number (must be below 65535)" << endl
       << "-o : output file (default to stdout)" << endl
       << "-c : order data in one column" << endl
       << "-p : PCI card number (default 0)" << endl;
}

typedef enum {
  BINARY,
  DECIMAL,
  ASCII
} outputMode_e;

int qrng(Quantis::Quantis &quantis, unsigned int count, outputMode_e outputMode,
	 char col, unsigned long max, std::ostream &out) {
  if (!out) {
    return EXIT_FAILURE;
  }

  switch (outputMode) {
  case BINARY: {
    unsigned char *buf = new unsigned char[count];
    quantis.readBlocking(buf, count); 
    for (unsigned int i = 0;  i < count; i++)
      out.put(buf[i]);
    delete buf;
    break;
  }
    
  case ASCII: {
    unsigned char *buf = new unsigned char[count];
    quantis.readBlocking(buf, count);
    
    out << std::hex;
    for (unsigned int i = 0;  i < count; i++)
      out << (unsigned int)buf[i] << col;
    if (col != '\n')
      out << endl;
    delete buf;
    break;
  }
    
  case DECIMAL:
    for (unsigned int i = 0;  i < count; i++)
      out << quantis.readBoundedLong(max) << col;
    if (col != '\n')
      out << endl;
    break;
    
  default:
    cerr << "Unknown outputMode" << endl;
    return EXIT_FAILURE;
  }
  
  out << std::flush;

  return EXIT_SUCCESS;
}

int main(int argc, char *argv[]) {
  unsigned int count      = 0,
    cardNumber = 0,
    max = 0;
  outputMode_e outputMode = ASCII;
  std::string  output = "";
  char         col = ' ';

  char c;
  while ((c = getopt(argc, argv, "n:bo:p:d:c")) >= 0) {
    switch (c) {
    case 'n':
      count = atoi(optarg);
      break;
      
    case 'b':
      outputMode = BINARY;
      break;
      
    case 'o':
      output = optarg;
      break;
      
    case 'p':
      cardNumber = atoi(optarg);
      break;
      
    case 'd':
      max = atoi(optarg);
      outputMode = DECIMAL;
      break;

    case 'c':
      col = '\n';
      break;
      
    default:
      usage();
      return EXIT_FAILURE;
    }
  }

  if ((argc == 1) || (count == 0) ||
      ((outputMode == DECIMAL) &&
       ((max == 0) || (max > 65535)))) {
    usage();
    return EXIT_FAILURE;
  }

  try {
    Quantis::Quantis quantis(cardNumber);

    quantis.reset();
    std::ostream *ostr = &std::cout;
    if (output != "") {
      ostr = new std::ofstream(output.c_str());
    }
    int ret = qrng(quantis, count, outputMode, col, max, *ostr);
    if (output != "") {
      delete ostr;
    }
  } catch (Quantis::QuantisError& e) {
    cerr << "Got an error while executing qrng: " << e.getErrorMessage() << endl;
    return EXIT_FAILURE;
  }

  return 0;
}
