/*-
 * Copyright (c) 2010 The NetBSD Foundation, Inc.
 * All rights reserved.
 *
 * This code is derived from software contributed to The NetBSD Foundation
 * by Yorick Hardy
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE NETBSD FOUNDATION, INC. AND CONTRIBUTORS
 * ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
 * TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE FOUNDATION OR CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

/*
 * Documentation:
 *   src/sys/dev/microcode/uticom/uticom.h
 *   http://focus.ti.com/docs/prod/folders/print/tusb3410.html
 *   http://www.ti.com/lit/gpn/tusb3410
 *   http://www.ti.com/litv/zip/sllc139
 */

#include <sys/param.h>
#include <sys/systm.h>
#include <sys/kernel.h>
#include <sys/device.h>
#include <sys/malloc.h>
#include <sys/conf.h>
#include <sys/tty.h>

#include <machine/bus.h>

#include <dev/firmload.h>

#include <dev/usb/usb.h>
#include <dev/usb/usbhid.h>

#include <dev/usb/usbdi.h>
#include <dev/usb/usbdivar.h>
#include <dev/usb/usbdi_util.h>
#include <dev/usb/usbdevs.h>
#include <dev/usb/usbcdc.h>

#include <dev/usb/ucomvar.h>

/* use the firmware definitions, endpoints and requests */
#include <dev/microcode/uticom/uticom.h>

/*
 * define UTICOM_TUSB3410FW_H to include the firmware in the
 * driver, instead of loading the firmware from disk
 */
#define UTICOM_TUSB3410FW_H
#ifdef UTICOM_TUSB3410FW_H
#include <dev/microcode/uticom/tusb3410fw.h>
#endif

#ifdef USB_DEBUG
static int	uticomdebug = 1;

#define DPRINTFN(n, x)	do { if (uticomdebug > (n)) printf x; } while (0)
#else
#define DPRINTFN(n, x)
#endif

#define DPRINTF(x) DPRINTFN(0, x)

#define UTICOM_INTR_INTERVAL	100	/* ms */

struct	uticom_softc {
	USBBASEDEVICE		sc_dev;		/* base device */
	usbd_device_handle	sc_udev;	/* device */
	usbd_interface_handle	sc_iface;	/* interface */

	int			sc_iface_number; /* interface number */

	usbd_interface_handle	sc_intr_iface;	/* interrupt interface */
	int			sc_intr_number;	/* interrupt number */
	usbd_pipe_handle	sc_intr_pipe;	/* interrupt pipe */
	u_char			*sc_intr_buf;	/* interrupt buffer */
	int			sc_isize;

	u_char			sc_lsr;		/* line status register */
	u_char			sc_msr;		/* modem status register */

	device_t		sc_subdev;
	u_char			sc_dying;
	u_char			sc_open;
};

static	void uticom_intr(usbd_xfer_handle, usbd_private_handle, usbd_status);

static	void uticom_set(void *, int, int, int);
static	void uticom_get_status(void *, int, u_char *, u_char *);
static	int  uticom_param(void *, int, struct termios *);
static	int  uticom_open(void *, int);
static	void uticom_close(void *, int);

static int uticom_load_fw(struct uticom_softc *sc);

struct ucom_methods uticom_methods = {
	uticom_get_status,
	uticom_set,
	uticom_param,
	NULL, /* uticom_ioctl, TODO */
	uticom_open,
	uticom_close,
	NULL,
	NULL
};

int	uticom_match(device_t, cfdata_t, void *);
void	uticom_attach(device_t, device_t, void *);
int	uticom_detach(device_t, int);
int	uticom_activate(device_t, enum devact);

CFATTACH_DECL_NEW(uticom, sizeof(struct uticom_softc), uticom_match,
			uticom_attach, uticom_detach, uticom_activate);

USB_MATCH(uticom)
{
	USB_MATCH_START(uticom, uaa);

	if (uaa->vendor == USB_VENDOR_TI &&
	    uaa->product == USB_PRODUCT_TI_TUSB3410)
		return (UMATCH_VENDOR_PRODUCT);
	return (0);
}

USB_ATTACH(uticom)
{
	USB_ATTACH_START(uticom, sc, uaa);
	usbd_device_handle dev = uaa->device;
	usb_config_descriptor_t *cdesc;
	usb_interface_descriptor_t *id;
	usb_endpoint_descriptor_t *ed;
	usbd_status err;
	int status, i;
	usb_device_descriptor_t *dd;
	struct ucom_attach_args uca;

	sc->sc_dev = self;
	sc->sc_udev = dev;
	sc->sc_subdev = NULL;

	sc->sc_dying = 0;
	sc->sc_open = 0;

	aprint_naive("\n");
	aprint_normal("\n");

	/* Initialize endpoints. */
	uca.bulkin = uca.bulkout = -1;
	sc->sc_intr_number = -1;
	sc->sc_intr_pipe = NULL;

	sc->sc_lsr = 0;
	sc->sc_msr = 0;

	dd = usbd_get_device_descriptor(sc->sc_udev);
	aprint_debug_dev(self, "uticom_attach: num of configurations %d\n",
		dd->bNumConfigurations);

	/* The device without firmware has single configuration with single
	 * bulk out interface. */
	if (dd->bNumConfigurations > 1)
		goto fwload_done;

	/* Loading firmware. */
	aprint_debug_dev(self, "uticom_attach: preparing to load firmware\n");
	aprint_debug_dev(self, "uticom_attach: setting config\n");
	err = usbd_set_config_index(sc->sc_udev, TI3410_BOOT_CONFIGURATION, 1);
	if (err) {
		aprint_error_dev(self, "failed to set configuration: %s\n",
			usbd_errstr(err));
		sc->sc_dying = 1;
		return;
	}

	aprint_debug_dev(self, "uticom_attach: getting descriptor\n");
	/* Get the config descriptor. */
	cdesc = usbd_get_config_descriptor(sc->sc_udev);

	if (cdesc == NULL) {
		aprint_error_dev(self, "failed to get configuration "
				"descriptor\n");
		sc->sc_dying = 1;
		return;
	}

	err = usbd_device2interface_handle(dev, TI3410_BOOT_INTERFACE,
	    &sc->sc_iface);
	if (err) {
		aprint_error_dev(self, "failed to get interface: %s\n",
			usbd_errstr(err));
		sc->sc_dying = 1;
		return;
	}

	aprint_debug_dev(self, "uticom_attach: attempting to load firmware\n");

	status = uticom_load_fw(sc);

	if (status) {
		aprint_error_dev(self, "firmware load failed\n");
		sc->sc_dying = 1;
		return;
	} else
		aprint_debug_dev(self, "firmware load succeeded\n");

	/*
	 * reload the device descriptor
	 * try a few times in case the firmware has not finished
	 * initializing yet.
	 */
	for (status=USBD_STALLED, i=0; i<3 && status; ++i) {
		status = usbd_reload_device_desc(sc->sc_udev);
		DPRINTF(("uticom_attach: reload device desc: %s\n",
			usbd_errstr(status)));
		if (status) DELAY(10);
	}

	if (status) {
		aprint_error_dev(self, "error reloading device "
				"descriptor: %s\n",
			usbd_errstr(status));
		sc->sc_dying = 1;
		return;
	}

fwload_done:
	dd = usbd_get_device_descriptor(sc->sc_udev);
	aprint_debug_dev(self, "uticom_attach: num of configurations %d\n",
		dd->bNumConfigurations);

	if (dd->bcdDevice[0] != TI3410_FIRMWARE_VERSION_MAJOR ||
		dd->bcdDevice[1] != TI3410_FIRMWARE_VERSION_MINOR) {
		aprint_error_dev(self, "firmware version mismatch: "
			"expected %d.%02d, got %d.%02d.\n",
			TI3410_FIRMWARE_VERSION_MAJOR,
			TI3410_FIRMWARE_VERSION_MINOR,
			dd->bcdDevice[0], dd->bcdDevice[1]); 
		sc->sc_dying = 1;
		return;
	}

	err = usbd_set_config_index(sc->sc_udev,
				TI3410_ACTIVE_CONFIGURATION, 1);
	if (err) {
		aprint_error_dev(self, "failed to set configuration: %s\n",
			usbd_errstr(err));
		sc->sc_dying = 1;
		return;
	}

	/* Get the config descriptor. */
	cdesc = usbd_get_config_descriptor(sc->sc_udev);
	if (cdesc == NULL) {
		aprint_error_dev(self, "failed to get configuration "
				"descriptor\n");
		sc->sc_dying = 1;
		return;
	}

	err = usbd_device2interface_handle(sc->sc_udev, TI3410_ACTIVE_INTERFACE,
		&sc->sc_iface);
	if (err) {
		aprint_error_dev(self, "failed to get interface: %s\n",
			usbd_errstr(err));
		sc->sc_dying = 1;
		return;
	}

	/* Find the interrupt endpoints. */
	id = usbd_get_interface_descriptor(sc->sc_iface);
	sc->sc_iface_number = id->bInterfaceNumber;

	for (i = 0; i < id->bNumEndpoints; i++) {
		ed = usbd_interface2endpoint_descriptor(sc->sc_iface, i);
		if (ed == NULL) {
			aprint_error_dev(self,
				"no endpoint descriptor for %d\n", i);
			sc->sc_dying = 1;
			return;
		}

		if (UE_GET_DIR(ed->bEndpointAddress) == UE_DIR_IN &&
		    UE_GET_XFERTYPE(ed->bmAttributes) == UE_BULK) {
			uca.bulkin = ed->bEndpointAddress;
			uca.ibufsize = UGETW(ed->wMaxPacketSize);
			uca.ibufsizepad = UGETW(ed->wMaxPacketSize);
			aprint_debug_dev(self, "uticom_attach: ibufsize=%d\n",
				(int)uca.ibufsize);
		}
		if (UE_GET_DIR(ed->bEndpointAddress) == UE_DIR_OUT &&
		    UE_GET_XFERTYPE(ed->bmAttributes) == UE_BULK) {
			uca.bulkout = ed->bEndpointAddress;
			uca.obufsize = UGETW(ed->wMaxPacketSize);
			aprint_debug_dev(self, "uticom_attach: obufsize=%d\n",
				(int)uca.obufsize);
		}
		if (UE_GET_DIR(ed->bEndpointAddress) == UE_DIR_IN &&
		    UE_GET_XFERTYPE(ed->bmAttributes) == UE_INTERRUPT) {
			sc->sc_intr_number = ed->bEndpointAddress;
			sc->sc_isize = UGETW(ed->wMaxPacketSize);
			aprint_debug_dev(self, "uticom_attach: intrsize=%d\n",
				(int)sc->sc_isize);
		}
	}

	if (sc->sc_intr_number == -1) {
		aprint_error_dev(self, "could not find interrupt in\n");
		sc->sc_dying = 1;
		return;
	}

	/* Keep interface for interrupt. */
	sc->sc_intr_iface = sc->sc_iface;

	if (uca.bulkin == -1) {
		aprint_error_dev(self, "could not find data bulk in\n");
		sc->sc_dying = 1;
		return;
	}

	if (uca.bulkout == -1) {
		aprint_error_dev(self, "could not find data bulk out\n");
		sc->sc_dying = 1;
		return;
	}

	uca.portno = UCOM_UNK_PORTNO;
	uca.device = sc->sc_udev;
	uca.iface = sc->sc_iface;
	uca.opkthdrlen = 0;
	uca.methods = &uticom_methods;
	uca.arg = sc;
	uca.info = NULL;

	aprint_debug_dev(self, "uticom_attach: in = 0x%x, out = 0x%x, "
			"intr = 0x%x\n", uca.bulkin, uca.bulkout,
			sc->sc_intr_number);

	sc->sc_subdev = config_found_sm_loc(self, "ucombus", NULL, &uca,
					    ucomprint, ucomsubmatch);

	if (!pmf_device_register(self, NULL, NULL))
		aprint_error_dev(self, "couldn't establish power handler\n");
	USB_ATTACH_SUCCESS_RETURN;
}

int
uticom_activate(device_t self, enum devact act)
{
	struct uticom_softc *sc = device_private(self);

	switch (act) {
	case DVACT_DEACTIVATE:
		sc->sc_dying = 1;
		return 0;
	default:
		return EOPNOTSUPP;
	}
}

USB_DETACH(uticom)
{
	USB_DETACH_START(uticom, sc);
	usbd_status err;

	aprint_debug_dev(self, "uticom_detach\n");
	sc->sc_dying = 1;
	pmf_device_deregister(self);

	if (sc->sc_subdev != NULL) {
		config_detach(sc->sc_subdev, flags);
		sc->sc_subdev = NULL;
	}

	if (sc->sc_intr_pipe != NULL) {
		usbd_abort_pipe(sc->sc_intr_pipe);
		usbd_close_pipe(sc->sc_intr_pipe);
		free(sc->sc_intr_buf, M_USBDEV);
		sc->sc_intr_pipe = NULL;
	}

	usbd_add_drv_event(USB_EVENT_DRIVER_DETACH, sc->sc_udev,
			   USBDEV(sc->sc_dev));

	err = usbd_set_config_index(sc->sc_udev, USB_UNCONFIG_INDEX, 1);
#if 0
	if (err)
		printf("%s: failed to set configuration: %s\n",
			devname, usbd_errstr(err));
#endif

	return (0);
}

static void
uticom_set(void *addr, int portno, int reg, int onoff)
{
	usbd_status status;
	struct uticom_softc *sc = addr;
	usb_device_request_t request;

	request.bmRequestType = UT_WRITE_VENDOR_INTERFACE;

	switch (reg) {
	case UCOM_SET_DTR:
		DPRINTF(("uticom_set: DTR %s\n", (onoff)?"on":"off"));
		request.bRequest = TI3410_RQ_SET_DTR;
		break;
	case UCOM_SET_RTS:
		DPRINTF(("uticom_set: DTR %s\n", (onoff)?"on":"off"));
		request.bRequest = TI3410_RQ_SET_RTS;
		break;
	case UCOM_SET_BREAK:
		DPRINTF(("uticom_set: DTR %s\n", (onoff)?"on":"off"));
		request.bRequest = TI3410_RQ_SET_BREAK;
		break;
	default:
		return;
	}

	USETW(request.wValue, (onoff)?1:0);
	USETW(request.wIndex, TI3410_ACTIVE_INTERFACE);
	USETW(request.wLength, 0);

	status = usbd_do_request(sc->sc_udev, &request, NULL);
	if (status)
		printf("uticom_set: %s\n", usbd_errstr(status));
}

static int
uticom_param(void *vsc, int portno, struct termios *t)
{
	struct uticom_softc *sc = (struct uticom_softc *)vsc;
	usb_device_request_t request;
	usbd_status err;
	uint8_t dll, dlh;
	uint32_t divisor;
	const char *devname = device_xname(sc->sc_dev);

	DPRINTF(("%s: uticom_param\n", devname));

	/*
	 * From tusb3410 documentation:
	 *  DLH DLL = (96/6.5 MHz)/(baud rate * 16)
	 *          = 12000000/(13 * baud rate)
	 *          = 24000000/(26 * baud rate)
	 *
	 * For rounding we use
	 *  DLH DLL = (24000000 + 13*baud rate) / (26 * baud rate)
	 *
	 * This formula generates the table provided in the tusb3410
	 * documentation. The calculation is done here instead of in
	 * firmware because we have the benefit of 32 bit integers,
	 * whereas the tusb3410 8052 MCU is 8 bit.
	 */

	divisor = (24000000 + 13*t->c_ospeed) / (26 * t->c_ospeed);
	dll = divisor & 0x00ff;
	dlh = (divisor & 0xff00) >> 8;
	DPRINTF(("%s: DLL:DLH=%x:%x\n", devname, (int)dll, (int)dlh));
	request.bmRequestType = UT_WRITE_VENDOR_INTERFACE;
	request.bRequest = TI3410_RQ_SET_DLLDLH;
	request.wValue[0] = dll;
	request.wValue[1] = dlh;
	USETW(request.wIndex, TI3410_ACTIVE_INTERFACE);
	USETW(request.wLength, 0);

	err = usbd_do_request(sc->sc_udev, &request, NULL);
	if (err) {
		printf("%s: uticom_param set dll/dlh: %s\n",
			devname, usbd_errstr(err));
		return (EIO);
	}

	request.bmRequestType = UT_WRITE_VENDOR_INTERFACE;
	request.bRequest = TI3410_RQ_SET_PARAM;
	USETW(request.wValue, 0);
	USETW(request.wIndex, TI3410_ACTIVE_INTERFACE);
	USETW(request.wLength, 0);

	switch (ISSET(t->c_cflag, CSIZE)) {
	case CS5: request.wValue[0] |= TI3410_PARAM_DATABITS_5; break;
	case CS6: request.wValue[0] |= TI3410_PARAM_DATABITS_6; break;
	case CS7: request.wValue[0] |= TI3410_PARAM_DATABITS_7; break;
	case CS8: request.wValue[0] |= TI3410_PARAM_DATABITS_8; break;
	default:
		return (EIO);
	}

	if (ISSET(t->c_cflag, CSTOPB))
		request.wValue[0] |= TI3410_PARAM_STOPBITS_2;
	else
		request.wValue[0] |= TI3410_PARAM_STOPBITS_1;

	if (ISSET(t->c_cflag, PARENB))
		request.wValue[0] |= TI3410_PARAM_PARITY_ENABLE;

	if (ISSET(t->c_cflag, PARODD))
		request.wValue[0] |= TI3410_PARAM_PARITY_ODD;
	else
		request.wValue[0] |= TI3410_PARAM_PARITY_EVEN;

	if (ISSET(t->c_cflag, MDMBUF))
		request.wValue[0] |= TI3410_PARAM_FIFO;

	if (ISSET(t->c_cflag, CRTSCTS))
		request.wValue[1] |= TI3410_PARAM_CRTSCTS;

	if (ISSET(t->c_cflag, CDTRCTS))
		request.wValue[1] |= TI3410_PARAM_CDTRCTS;

	DPRINTF(("uticom_param: param request %x %x\n",
		request.wValue[0], request.wValue[1]));

	err = usbd_do_request(sc->sc_udev, &request, NULL);
	if (err) {
		printf("%s: uticom_param: %s\n", devname, usbd_errstr(err));
		return (EIO);
	}
		
	return (0);
}

static int
uticom_open(void *addr, int portno)
{
	struct uticom_softc *sc = addr;
	const char *devname = device_xname(sc->sc_dev);
	usb_device_request_t request;
	usbd_status err;

	if (sc->sc_dying)
		return (ENXIO);

	DPRINTF(("%s: uticom_open\n", devname));

	if (sc->sc_intr_number != -1 && sc->sc_intr_pipe == NULL) {
		sc->sc_intr_buf = malloc(sc->sc_isize, M_USBDEV, M_WAITOK);
		err = usbd_open_pipe_intr(sc->sc_intr_iface, sc->sc_intr_number,
		    USBD_SHORT_XFER_OK, &sc->sc_intr_pipe, sc, sc->sc_intr_buf,
		    sc->sc_isize, uticom_intr, UTICOM_INTR_INTERVAL);
		if (err) {
			printf("%s: cannot open interrupt pipe (addr %d)\n",
				devname, sc->sc_intr_number);
			sc->sc_dying = 1;
			return (ENXIO);
		}
	}

	if (!sc->sc_open) {
#if 0
		request.bmRequestType = UT_WRITE_ENDPOINT;
		request.bRequest = UR_CLEAR_FEATURE;
		USETW(request.wIndex, TI3410_ACTIVE_BULK_IN_ENDPOINT);
		USETW(request.wValue, UF_ENDPOINT_HALT);
		USETW(request.wLength, 0);

		err = usbd_do_request(sc->sc_udev, &request, NULL);
		if (err) {
			printf("%s: uticom_open reset bulk: %s\n",
				devname, usbd_errstr(err));
			return (ENXIO);
		}

		request.bmRequestType = UT_WRITE_ENDPOINT;
		request.bRequest = UR_CLEAR_FEATURE;
		USETW(request.wIndex, TI3410_ACTIVE_BULK_OUT_ENDPOINT);
		USETW(request.wValue, UF_ENDPOINT_HALT);
		USETW(request.wLength, 0);

		err = usbd_do_request(sc->sc_udev, &request, NULL);
		if (err) {
			printf("%s: uticom_open reset bulk: %s\n",
				devname, usbd_errstr(err));
			return (ENXIO);
		}
#endif
		/* clear the DMA buffers and reset the UART */
		request.bmRequestType = UT_WRITE_VENDOR_INTERFACE;
		request.bRequest = TI3410_RQ_RESET_BULK;
		USETW(request.wIndex, TI3410_ACTIVE_INTERFACE);
		USETW(request.wValue, 0);
		USETW(request.wLength, 0);

		err = usbd_do_request(sc->sc_udev, &request, NULL);
		if (err) {
			printf("%s: uticom_open reset bulk: %s\n",
				devname, usbd_errstr(err));
		}

		sc->sc_open = 1;
	}

	DPRINTF(("%s: uticom_open: port opened\n", devname));
	return (0);
}

static void
uticom_close(void *addr, int portno)
{
	struct uticom_softc *sc = addr;
	const char *devname = device_xname(sc->sc_dev);
	usbd_status err;

	if (sc->sc_dying)
		return;

	DPRINTF(("%s: uticom_close: close\n", devname));

	if (sc->sc_intr_pipe != NULL) {
		err = usbd_abort_pipe(sc->sc_intr_pipe);
		if (err)
			printf("%s: abort interrupt pipe failed: %s\n",
				devname, usbd_errstr(err));
		err = usbd_close_pipe(sc->sc_intr_pipe);
		if (err)
			printf("%s: close interrupt pipe failed: %s\n",
				devname, usbd_errstr(err));
		free(sc->sc_intr_buf, M_USBDEV);
		sc->sc_intr_pipe = NULL;
	}

	sc->sc_open = 0;
}

static void
uticom_intr(usbd_xfer_handle xfer, usbd_private_handle priv, usbd_status status)
{
	struct uticom_softc *sc = priv;
	u_char *buf = sc->sc_intr_buf;
	const char *devname = device_xname(sc->sc_dev);
	struct ucom_softc *usc = device_private(sc->sc_subdev);

	if (sc->sc_dying)
		return;

	if (status != USBD_NORMAL_COMPLETION) {
		if (status == USBD_NOT_STARTED || status == USBD_CANCELLED) {
			DPRINTF(("%s: uticom_intr: int status: %s\n",
				devname, usbd_errstr(status)));
			return;
		}

		DPRINTF(("%s: uticom_intr: abnormal status: %s\n",
			devname, usbd_errstr(status)));
		usbd_clear_endpoint_stall_async(sc->sc_intr_pipe);
		return;
	}

	if (xfer->actlen == UTICOM_INTR_LEN && buf != NULL) {
		DPRINTF(("uticom_intr: lsr = %02x msr = %02x status = %02x\n",
				(int)buf[UTICOM_INTR_LSR],
				(int)buf[UTICOM_INTR_MSR],
				(int)buf[UTICOM_INTR_STATUS]));
		if (buf[UTICOM_INTR_LSR] & TI3410_LSR_PTE)
			sc->sc_lsr |= ULSR_PE;
		else
			sc->sc_lsr &= ~ULSR_PE;
		if (buf[UTICOM_INTR_LSR] & TI3410_LSR_FRE)
			sc->sc_lsr |= ULSR_FE;
		else
			sc->sc_lsr &= ~ULSR_FE;
		if (buf[UTICOM_INTR_LSR] & TI3410_LSR_BRK)
			sc->sc_lsr |= ULSR_BI;
		else
			sc->sc_lsr &= ~ULSR_BI;
/* These conditions are handled by the TUSB3410 DMA controller */
#if 0
		if (buf[UTICOM_INTR_LSR] & TI3410_LSR_RxF)
			sc->sc_lsr |= ULSR_RXRDY;
		else
			sc->sc_lsr &= ~ULSR_RXRDY;
		if (buf[UTICOM_INTR_LSR] & TI3410_LSR_TxE)
			sc->sc_lsr |= ULSR_TSRE;
		else
			sc->sc_lsr &= ~ULSR_TSRE;
		if (buf[UTICOM_INTR_LSR] & TI3410_LSR_TEMT)
			sc->sc_lsr |= ULSR_TXRDY;
		else
			sc->sc_lsr &= ~ULSR_TXRDY;
#endif
		if (buf[UTICOM_INTR_MSR] & TI3410_MSR_DCTS)
			sc->sc_msr |= UMSR_DCTS;
		else
			sc->sc_msr &= ~UMSR_DCTS;
		if (buf[UTICOM_INTR_MSR] & TI3410_MSR_DDSR)
			sc->sc_msr |= UMSR_DDSR;
		else
			sc->sc_msr &= ~UMSR_DDSR;
		if (buf[UTICOM_INTR_MSR] & TI3410_MSR_TRI)
			sc->sc_msr |= UMSR_TERI;
		else
			sc->sc_msr &= ~UMSR_TERI;
		if (buf[UTICOM_INTR_MSR] & TI3410_MSR_DDCD)
			sc->sc_msr |= UMSR_DDCD;
		else
			sc->sc_msr &= ~UMSR_DDCD;
		if (buf[UTICOM_INTR_MSR] & TI3410_MSR_CTS)
			sc->sc_msr |= UMSR_CTS;
		else
			sc->sc_msr &= ~UMSR_CTS;
		if (buf[UTICOM_INTR_MSR] & TI3410_MSR_DSR)
			sc->sc_msr |= UMSR_DSR;
		else
			sc->sc_msr &= ~UMSR_DSR;
		if (buf[UTICOM_INTR_MSR] & TI3410_MSR_RI)
			sc->sc_msr |= UMSR_RI;
		else
			sc->sc_msr &= ~UMSR_RI;
		if (buf[UTICOM_INTR_MSR] & TI3410_MSR_DCD)
			sc->sc_msr |= UMSR_DCD;
		else
			sc->sc_msr &= ~UMSR_DCD;

		ucom_status_change(usc);

		if (buf[UTICOM_INTR_STATUS] & UTICOM_INTR_STATUS_OVERRUN)
			printf("%s: overrun\n", devname);
	}
}

static void
uticom_get_status(void *addr, int portno, u_char *lsr, u_char *msr)
{
	struct uticom_softc *sc = addr;

	DPRINTF(("uticom_get_status: %02x %02x\n", sc->sc_lsr, sc->sc_msr));

	if (lsr != NULL) *lsr = sc->sc_lsr;
	if (msr != NULL) *msr = sc->sc_msr;

	return;
}

static int
uticom_load_fw(struct uticom_softc *sc)
{
	u_char *obuf;
	int buffer_size, firmware_size, pos;
	uint8_t cs = 0, *buffer;
	usbd_status err;
	usbd_xfer_handle oxfer = 0;
	usbd_status error = 0;
	usbd_pipe_handle pipe;
	const char *devname = device_xname(sc->sc_dev);
#ifdef UTICOM_TUSB3410FW_H
	firmware_size = sizeof(tusb3410fw);
	DPRINTF(("%s: buffer size: %d\n", devname, (int)firmware_size));
	/* Add 3 bytes for the header */
	buffer_size = firmware_size + 3;
	buffer = malloc(buffer_size, M_USBDEV, M_WAITOK);
	memcpy(buffer+3, tusb3410fw, firmware_size);
#else
	int fwerr;
	firmware_handle_t fh;

	if((fwerr = firmware_open("uticom", "tusb3410fw", &fh))) {
		printf("%s: uticom_load_fw: firmware_open: %d\n",
		    devname, fwerr);
		return fwerr;
	}

	firmware_size = firmware_get_size(fh);
	DPRINTF(("%s: buffer size: %d\n", devname, (int)firmware_size));
	/* Add 3 bytes for the header */
	buffer_size = firmware_size + 3;
	buffer = malloc(buffer_size, M_USBDEV, M_WAITOK);

	if (!buffer) {
		printf("%s: uticom_load_fw: out of memory\n",
		    devname);
		return ENOMEM;
	}
	DPRINTF(("%s: buffer allocated.\n", devname));

	/* 3 bytes for the header */
	fwerr = firmware_read(fh, 0, buffer+3, firmware_size);
	firmware_close(fh);

	if(fwerr) {
		printf("%s: uticom_load_fw: firmware_read: %d\n",
			devname, fwerr);
		return fwerr;
	}
#endif

	for (pos = 3; pos < buffer_size; ++pos)
		cs = cs + buffer[pos];

	buffer[0] = (uint8_t)(firmware_size & 0xff);
	buffer[1] = (uint8_t)((firmware_size & 0xff00) >> 8);
	buffer[2] = cs;

	DPRINTF(("%s: firmware length %d with checksum %x\n",
		devname, buffer[0] + (((int)buffer[1]) << 8), (int)buffer[2]));

	DPRINTF(("%s: loading firmware ...\n", devname));

	/* from the documentation we load the firmware using endpoint 1 */
	err = usbd_open_pipe(sc->sc_iface, TI3410_BOOT_FWUPLOAD_ENDPOINT,
		USBD_EXCLUSIVE_USE, &pipe);
	if (err) {
		printf("%s: open bulk out error (firmware): %s\n",
			devname, usbd_errstr(err));
		error = EIO;
		goto finish;
	}

	oxfer = usbd_alloc_xfer(sc->sc_udev);
	if (oxfer == NULL) {
		error = ENOMEM;
		goto finish;
	}

	obuf = usbd_alloc_buffer(oxfer, buffer_size);
	if (obuf == NULL) {
		error = ENOMEM;
		goto finish;
	}

	/* copy the 3 byte header and firmware */
	memcpy(obuf, buffer, buffer_size);

	usbd_setup_xfer(oxfer, pipe, (usbd_private_handle)sc, obuf, buffer_size,
	    USBD_NO_COPY, USBD_NO_TIMEOUT, 0);
	err = usbd_sync_transfer(oxfer);

	if (err != USBD_NORMAL_COMPLETION)
		printf("%s: uticom_load_fw: error: %s\n",
		    devname, usbd_errstr(err));

finish:
	usbd_free_buffer(oxfer);
	usbd_free_xfer(oxfer);
	oxfer = NULL;
	usbd_abort_pipe(pipe);
	usbd_close_pipe(pipe);
	free(buffer, M_USBDEV);
	return err;
}
