/*
 * farsight-transmitter.h - Base class for network transmitter plugins
 *
 * Farsight Core
 * Copyright (C) 2006 Collabora Ltd.
 * Copyright (C) 2006 Nokia Corporation
 *   @author Philippe Kalaf <philippe.kalaf@collabora.co.uk>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifndef __FARSIGHT_TRANSMITTER_H__
#define __FARSIGHT_TRANSMITTER_H__

#include <glib.h>
#include <glib-object.h>
#include <gst/gstelement.h>

#include "farsight-plugin.h"
#include "farsight-codec.h"
#include "farsight-transport.h"

G_BEGIN_DECLS

/* TYPE MACROS */
#define FARSIGHT_TYPE_TRANSMITTER \
  (farsight_transmitter_get_type())
#define FARSIGHT_TRANSMITTER(obj) \
  (G_TYPE_CHECK_INSTANCE_CAST((obj), FARSIGHT_TYPE_TRANSMITTER, FarsightTransmitter))
#define FARSIGHT_TRANSMITTER_CLASS(klass) \
  (G_TYPE_CHECK_CLASS_CAST((klass), FARSIGHT_TYPE_TRANSMITTER, FarsightTransmitterClass))
#define FARSIGHT_IS_TRANSMITTER(obj) \
  (G_TYPE_CHECK_INSTANCE_TYPE((obj), FARSIGHT_TYPE_TRANSMITTER))
#define FARSIGHT_IS_TRANSMITTER_CLASS(klass) \
  (G_TYPE_CHECK_CLASS_TYPE((klass), FARSIGHT_TYPE_TRANSMITTER))
#define FARSIGHT_TRANSMITTER_GET_CLASS(obj) \
  (G_TYPE_INSTANCE_GET_CLASS ((obj), FARSIGHT_TYPE_TRANSMITTER, FarsightTransmitterClass))

typedef struct _FarsightTransmitter FarsightTransmitter;
typedef struct _FarsightTransmitterClass FarsightTransmitterClass;
typedef struct _FarsightTransmitterPrivate FarsightTransmitterPrivate;

/**
 * SECTION:FarsightTransmitter
 * @short_description: A network transmitter (i.e. libjingle-p2p)
 */

/**
 * FarsightTransmitterError:
 * @ERROR_UNKNOWN: An unknown error occured
 *
 * Enum used to indicate an error condition.
 */
typedef enum
{
  FARSIGHT_TRANSMITTER_ERROR_UNKNOWN
} FarsightTransmitterError;

/**
 * FarsightTransmitterState:
 * @FARSIGHT_TRANSMITTER_STATE_DISCONNECTED: The transmitter is not connected
 * @FARSIGHT_TRANSMITTER_STATE_CONNECTING: The transmitter is trying to establish a
 * connection
 * @FARSIGHT_TRANSMITTER_STATE_CONNECTED: The transmitter is connected and can start
 * streaming.
 *
 * An enum used to signal the state of a #FarsightTransmitter
 */
typedef enum
{
  FARSIGHT_TRANSMITTER_STATE_DISCONNECTED,
  FARSIGHT_TRANSMITTER_STATE_CONNECTING,
  FARSIGHT_TRANSMITTER_STATE_CONNECTED
} FarsightTransmitterState;

struct _FarsightTransmitterClass
{
  GObjectClass parent_class;

  /*virtual methods */
  gboolean (* prepare) (FarsightTransmitter *self);
  gboolean (* stop) (FarsightTransmitter *self);

  void (* add_remote_candidates) (FarsightTransmitter *self, const GList *remote_candidate_list);

  GstElement * (* get_gst_src) (FarsightTransmitter *self);
  GstElement * (* get_gst_sink) (FarsightTransmitter *self);
};

/**
 * FarsightTransmitter:
 * 
 * A network transmitter that farsight plugins can use to communicate on the
 * network. These transmitter can contain advanced connectivity features such as
 * STUN, TURN or can be plain UDP / TCP sockets.
 */
struct _FarsightTransmitter
{
  GObject parent;

  FarsightPlugin *plugin;

  /*< private >*/
  FarsightTransmitterPrivate *priv;
};

GType farsight_transmitter_get_type (void);

FarsightTransmitter *farsight_transmitter_factory_make (
    const gchar *transmitter_id);
void farsight_transmitter_destroy (FarsightTransmitter *self);

gboolean farsight_transmitter_prepare (FarsightTransmitter *self);
gboolean farsight_transmitter_stop (FarsightTransmitter *self);

void farsight_transmitter_add_remote_candidates (FarsightTransmitter *self,
    const GList *remote_candidate_list);

GstElement * farsight_transmitter_get_gst_src (FarsightTransmitter *self);
GstElement * farsight_transmitter_get_gst_sink (FarsightTransmitter *self);

/* Signals */
void farsight_transmitter_signal_new_native_candidate (
    FarsightTransmitter *self,
    const FarsightTransportInfo *candidate);

void farsight_transmitter_signal_native_candidates_prepared (
    FarsightTransmitter *self);

void farsight_transmitter_signal_new_active_candidate_pair (
    FarsightTransmitter *self,
    const gchar *native_candidate_id,
    const gchar *remote_candidate_id);

void farsight_transmitter_signal_connection_state_changed (
    FarsightTransmitter *self,
    FarsightTransmitterState state);

void farsight_transmitter_signal_error (
    FarsightTransmitter *self,
    FarsightTransmitterError type,
    gchar *message);

G_END_DECLS

#endif
