/*
 * Copyright 2006 Sun Microsystems, Inc.  All rights reserved.
 * You may not modify, use, reproduce, or distribute this
 * software except in compliance with the terms of the License at:
 *
 *   http://developer.sun.com/berkeley_license.html
 *
 * $Id: script.js,v 1.18 2006/05/02 11:01:05 mattbohm Exp $
 */

dojo.require("dojo.io.*");

/**
 * Define our component container object (if necessary).
 */
if (bpui_undefined("progressbar", bpui)) {

  var componentsObject = new Object();
  bpui.progressbar = {
    /**
     * Localizable messages in the default language.  Treat this variable
     * as an associative array keyed by a message identifier.  Robust
     * applications will dynamically replace these messages with a localized
     * version based on the locale of the current user.
     */
    messages : new Object(),


    /**
     * Holder for state information for the various component instances.
     * Treat this variable as an associative array with clientId as key
     * and bpui.progressbar.Instance objects as values.
     */
    state : componentsObject,


    /**
     * An alias for the state property
     */
    components : componentsObject,

    /**
     * The version number of this script.
     */
    version : {
      major: 0,
      minor: 1,
      patch: 0,
      flag: "",
      toString: function() {
        with (bpui.progressbar.version) {
          return major + "." + minor + "." + patch + flag;
        }
      }
    },

    /**
    * The unparameterized URL to send AJAX requests to
    */
    unparameterizedUrl : null,

    /**
    * The value of window.onload before we modify it.
    */
    windowOnload : null

  }
}

// Initialize the localizable messages for this component
bpui.progressbar.messages["bindError"] = "An error occurred performing an asynchronous request";
bpui.progressbar.messages["unboundPercentage"] = "Warning: the percentage property of a server-side progress bar component was not bound";

bpui.progressbar.ajaxOnloadHandler = function() {
    var clientId;
    for (clientId in bpui.progressbar.state) {
        var progressBarInstance = bpui.progressbar.state[clientId];
        if (progressBarInstance.autoStart) {
            var functionRef = function(){progressBarInstance.start(null);};
            setTimeout(functionRef, progressBarInstance.interval);
        }
    }
    if (null != bpui.progressbar.windowOnload) {
	bpui.progressbar.windowOnload;
    }
}

bpui.progressbar.getBindArgs = function(parameterizedUrl) {
    var bindArgs = {
        url:      parameterizedUrl,
        mimetype: "text/xml",
        error:    bpui.progressbar.processError,
        load:     bpui.progressbar.processLoad
    };
    return bindArgs;
}

/**
 * Process an asynchronous response when it is ready.
 *
 * @param type Event type ("load")
 * @param data Data returned with the response
 * @param event Low-level event object
 */
bpui.progressbar.processLoad = function(type, data, event) {
    if (event.readyState == 4) {
        if (event.status == 200) {
          bpui.progressbar.processResponse(data);
        }
    }
}

/**
 * Process the failure of an asynchronous response.
 *
 * @param type Event type ("error")
 * @param errorObject Low-level error object
 */
bpui.progressbar.processError = function(type, errorObject) {
    window.status = bpui.progressbar.messages["bindError"] + ":  " + errorObject;
}

bpui.progressbar.processResponse = function(data) {

    var clientIdNode = data.getElementsByTagName("clientId")[0];
    var percentageNode = data.getElementsByTagName("percentage")[0];
    var percentageTextNodes = data.getElementsByTagName("percentageText");
    var statusTextNodes = data.getElementsByTagName("status");
    var clientId = clientIdNode.firstChild.nodeValue;
    var percentage = percentageNode.firstChild.nodeValue;
    var progressBarInstance = bpui.progressbar.state[clientId];

    if (!progressBarInstance.useAjax) {
        return;
    }

    if (progressBarInstance.status == -1 || progressBarInstance.status == 5) {
        return; //already failed or complete -- don't update unless re-started
    }

    var failed = false;
    if (statusTextNodes != null && statusTextNodes.length > 0 && statusTextNodes[0].firstChild != null && statusTextNodes[0].firstChild.nodeValue == -1) {
        failed = true;
    }

    var percentageText = null;
    if (percentageTextNodes != null && percentageTextNodes.length > 0) {
        if (percentageTextNodes[0].firstChild == null) {
            percentageText = "&nbsp;";
        }
        else {
            percentageText = percentageTextNodes[0].firstChild.nodeValue;
        }
    }

    progressBarInstance.processResponseData(percentage, percentageText, failed);
}

bpui.progressbar.getForm = function(clientId) {
    var tag = dojo.byId(clientId);
    while (tag != null) {
        if ("form" == tag.tagName.toLowerCase()) {
          return tag;
        }
        tag = tag.parentNode;
    }
    return null;
}

bpui.progressbar.Instance = function(clientId, interval, percentage, percentageBinding) {
    var thisref = this;
    bpui.progressbar.state[clientId] = this;
    this.clientId = clientId;
    this.percentage = percentage;
    this.percentageText = null;
    //-1 = failed, 0 = not started, 1 = started, 2 = paused, 3 = resumed, 4 = stopped, 5 = complete
    this.status = 0;
    this.interval = interval;
    this.bindings = new Array();
    this.bindings["percentage"] = percentageBinding;
    this.bindings["percentageText"] = null;
    this.bindings["startOperation"] = null;
    this.bindings["pauseOperation"] = null;
    this.bindings["resumeOperation"] = null;
    this.bindings["stopOperation"] = null;
    this.bindings["failed"] = null;
    this.onComplete = null;
    this.onFail = null;
    this.onUpdate = null;
    this.onPoll = null;
    this.autoStart = false;
    this.invokeStartOperation = true;
    this.invokePauseOperation = true;
    this.invokeResumeOperation = true;
    this.invokeStopOperation = true;
    this.completionDelay = 50;
    this.failureDelay = 50;
    this.useAjax = true;
    this.indeterminateSection = 10;  //constant of the percentage to show as "bpui_progressbar_portionComplete" for indeterminate case
    this.indeterminateInterval = 20;       //call modify display repeatedly with this interval
    this.indeterminateIncrement = 1;        //how much to increment the indeterminatePercentage each time
    this.indeterminatePercentage = 0;       //sum of the "bpui_progressbar_portionIndeterminate" and "bpui_progressbar_portionComplete" percentages for indeterminate case
    this.indeterminateIntervalId = null;    //id returned from the setInterval function

    this.submitForm = function() {
        var form = bpui.progressbar.getForm(thisref.clientId);
        //append a new hidden field to this form
        //with id the same as this progress bar instance's clientId
        var newElement = document.createElement('input');
        newElement.type = 'hidden';
        newElement.id = thisref.clientId;
        newElement.name = thisref.clientId;
        newElement.value = "value";
        form.appendChild(newElement);
        form.submit();
        return false;
    }

    this.canStart = function() {
        var status = thisref.status;
        if (status == -1 || status == 0 || status == 4 || status == 5) {
            return true;
        }
        return false;
    }

    //paramArray is an associative array of name-value pairs to be used in the operation, if applicable. It can be null.
    this.start = function(paramArray) {
        if (!thisref.canStart()) {
            return; //progress bar can't be started based on current status
        }
        thisref.status = 1;
        thisref._poll("startOperation", paramArray);
    }

    this.canPause = function() {
        var status = thisref.status;
        if (status == 1 || status == 3) {
            return true;
        }
        return false;
    }

    //paramArray is an associative array of name-value pairs to be used in the operation, if applicable. It can be null.
    this.pause = function(paramArray) {
        if (!thisref.canPause()) {
            return; //progress bar can't be paused based on current status
        }
        thisref.status = 2;
        thisref._halt("pauseOperation", paramArray);
    }

    this.canResume = function() {
        var status = thisref.status;
        if (status == 2) {
            return true;
        }
        return false;
    }

    //paramArray is an associative array of name-value pairs to be used in the operation, if applicable. It can be null.
    this.resume = function(paramArray) {
        if (!thisref.canResume()) {
            return; //progress bar can't be resumed based on current status
        }
        thisref.status = 3;
        thisref._poll("resumeOperation", paramArray);
    }

    this.canStop = function() {
        var status = thisref.status;
        if (status == 1 || status == 2 || status == 3) {
            return true;
        }
        return false;
    }

    //paramArray is an associative array of name-value pairs to be used in the operation, if applicable. It can be null.
    this.stop = function(paramArray) {
        if (!thisref.canStop()) {
            return; //progress bar can't be stopped based on current status
        }
        thisref.status = 4;
        thisref._halt("stopOperation", paramArray);
    }

    //this is an internally used function.
    //bindingKey can be null, "startOperation", "resumeOperation"
    //paramArray is an associative array of name-value pairs to be used in the operation, if applicable. It can be null.
    this._poll = function(bindingKey, paramArray) {
        var canAppendParamArrayToRequest = false;
        if (bindingKey == null && thisref.onPoll != null) {
            var event = new Object();
            event.component = thisref;
            event.params = null;
            thisref.onPoll(event);
            paramArray = event.params;
            canAppendParamArrayToRequest = true;
        }
        if (!thisref.useAjax) {
            return;
        }
        if (thisref.bindings["percentage"] == null) {
            var msg = bpui.progressbar.messages["unboundPercentage"];
            window.status = msg;
            //alert(msg);
            return;
        }
        var paramPrefix = "bpui_progressbar_";
        var parameterizedUrl = bpui.progressbar.unparameterizedUrl + "?" + paramPrefix + "percentageBinding=" + escape(thisref.bindings["percentage"]) + "&" + paramPrefix + "clientId=" + thisref.clientId;
        if (thisref.bindings["percentageText"] != null) {
            parameterizedUrl += "&" + paramPrefix + "percentageTextBinding=" + escape(thisref.bindings["percentageText"]);
        }
        if (thisref.bindings["failed"] != null) {
            parameterizedUrl += "&" + paramPrefix + "failedBinding=" + escape(thisref.bindings["failed"]);
        }
        if ( bindingKey != null && thisref.bindings[bindingKey] != null && 
            (bindingKey == "startOperation" && thisref.invokeStartOperation) ||
            (bindingKey == "resumeOperation" && thisref.invokeResumeOperation) ) {

            parameterizedUrl += "&" + paramPrefix + bindingKey + "Binding=" + escape(thisref.bindings[bindingKey]);
            canAppendParamArrayToRequest = true;
        }
        if (canAppendParamArrayToRequest && paramArray != null) {
            for (var key in paramArray) {
                parameterizedUrl += "&" + key + "=" + escape(paramArray[key]);
            }
        }
        var bindArgs = bpui.progressbar.getBindArgs(parameterizedUrl);
        dojo.io.bind(bindArgs);
    }

    //this is an internally used function.
    //bindingKey must be "pauseOperation" or "stopOperation". it can't be null.
    //paramArray is an associative array of name-value pairs to be used in the operation, if applicable. It can be null.
    this._halt = function(bindingKey, paramArray) {
        if (!thisref.useAjax) {
            return;
        }
        if ((bindingKey == "pauseOperation" && thisref.invokePauseOperation) ||
            (bindingKey == "stopOperation" && thisref.invokeStopOperation)) {

            if (thisref.bindings["percentage"] == null) {
                var msg = bpui.progressbar.messages["unboundPercentage"];
                window.status = msg;
                //alert(msg);
                return;
            }
            var paramPrefix = "bpui_progressbar_";
            var parameterizedUrl = bpui.progressbar.unparameterizedUrl + "?" + paramPrefix + "percentageBinding=" + escape(thisref.bindings["percentage"]) + "&" + paramPrefix + "clientId=" + thisref.clientId + "&" + paramPrefix + bindingKey + "Binding=" + escape(thisref.bindings[bindingKey]);
            if (thisref.bindings["percentageText"] != null) {
                parameterizedUrl += "&" + paramPrefix + "percentageTextBinding=" + escape(thisref.bindings["percentageText"]);
            }
            if (thisref.bindings["failed"] != null) {
                parameterizedUrl += "&" + paramPrefix + "failedBinding=" + escape(thisref.bindings["failed"]);
            }
            if (paramArray != null) {
                for (name in paramArray) {
                    parameterizedUrl += "&" + name + "=" + escape(paramArray[name]);
                }
            }
            var bindArgs = bpui.progressbar.getBindArgs(parameterizedUrl);
            dojo.io.bind(bindArgs);
        }
    }

    this._modifyIndeterminateDisplay = function() {
        if (!thisref._canModifyIndeterminateDisplay()) {
            if (thisref.indeterminateIntervalId != null) {
                clearInterval(thisref.indeterminateIntervalId);
                thisref.indeterminateIntervalId = null;
            }
            return;
        }
        thisref._modifyBarDisplay(thisref.percentage);
    }

    this._canModifyIndeterminateDisplay = function() {
        return (thisref.percentage == -1 && (thisref.status == 1 || thisref.status == 3));
    }

    this._modifyTextDisplay = function(text) {
        var percentageCell = dojo.byId(thisref.clientId + "_percentageText");
        percentageCell.innerHTML = text;
    }

    // show the current percentage
    this._modifyBarDisplay = function(percent) {

        if (percent == -1 && thisref.status == -1) {
            return;	//don't update indeterminate if progress has failed
        }

        var indeterminatePortion = 0;
        var completePortion = percent;
        var remainingPortion = 100 - completePortion;

        if (completePortion == -1) {
            var indetPercentage = thisref.indeterminatePercentage;
            if (indetPercentage > (100 + thisref.indeterminateSection)) {
                indetPercentage = 100 + thisref.indeterminateSection;  //rare, defensive
            }
            if (indetPercentage < thisref.indeterminateSection) {
                indeterminatePortion = 0;
                completePortion = indetPercentage;
                remainingPortion = 100 - indetPercentage;
            }
            else if (indetPercentage > 100) {
                indeterminatePortion = indetPercentage - thisref.indeterminateSection;
                completePortion = 100 - indeterminatePortion;
                remainingPortion = 0;
            }
            else {
                indeterminatePortion = indetPercentage - thisref.indeterminateSection;
                completePortion = thisref.indeterminateSection;
                remainingPortion = 100 - indetPercentage;
            }

            //increment indeterminatePercentage
            thisref.indeterminatePercentage += thisref.indeterminateIncrement;
            if (thisref.indeterminatePercentage > (100 + thisref.indeterminateSection)) {
                thisref.indeterminatePercentage = 0;
            }
        }

        var portionIndetCell = dojo.byId(thisref.clientId + "_portionIndeterminate");
        var portionComplCell = dojo.byId(thisref.clientId + "_portionComplete");
        var portionRemCell = dojo.byId(thisref.clientId + "_portionRemaining");

        portionIndetCell.style.width = "" + indeterminatePortion + "%"
        portionComplCell.style.width = "" + completePortion + "%";
        portionRemCell.style.width = "" + remainingPortion + "%";
    }

    this.processResponseData = function(percent, text, failed) {
        thisref.percentage = percent;
        thisref.percentageText = text;

        if (failed) {
            thisref.status = -1;    //failed;
        }
        else if (thisref.status != -1 && thisref.percentage >= 100) {
            thisref.status = 5;    //complete;
        }

        if (text == null) {
            text = "&nbsp;";
            if (thisref.percentage >= 0) {
                text = "" + thisref.percentage + "%";
            }
        }

        var event = null;
        var skipUpdate = false;
        if (thisref.onUpdate != null) {
            event = new Object();
            event.component = thisref;
            event.text = text;
            event.skipUpdate = false;
            thisref.onUpdate(event);
        }

        if (event != null) {
            text = event.text;
            skipUpdate = event.skipUpdate;
        }

        if (!skipUpdate) {
            thisref._modifyTextDisplay(text);

            //now to modify bar display

            if (thisref.percentage == -1) {
                if (thisref.indeterminateIntervalId == null && thisref._canModifyIndeterminateDisplay()) {
                    var functionref = function() {thisref._modifyIndeterminateDisplay();};
                    thisref.indeterminateIntervalId = setInterval(functionref, thisref.indeterminateInterval);
                }
            }
            else {
                thisref._modifyBarDisplay(thisref.percentage);
            }
        }

        if (failed) {
            var functionRef = function(){thisref._fail();};
            setTimeout(functionRef, thisref.failureDelay);
            return;
        }

        if (thisref.percentage < 100) {
            if (thisref.status == 1 || thisref.status == 3) {
                var functionRef = function(){thisref._poll(null, null);};
                setTimeout(functionRef, thisref.interval);
            }
        } else {
            //invoke complete function even if status is paused (2) or stopped (4)
            //check percent to avoid calling _complete() more than once 
            if (thisref.status != -1 && percent >= 100) {
                var functionRef = function(){thisref._complete();};
                setTimeout(functionRef, thisref.completionDelay);
            }
        }
    }

    this._complete = function() {
        if (thisref.onComplete != null) {
            thisref.onComplete(thisref);
        }
    }

    this._fail = function() {
        if (thisref.onFail != null) {
            thisref.onFail(thisref);
        }
    }

    this.isVisible = function() {
        var element = dojo.byId(thisref.clientId);
        return element.style.display != "none";
    }

    this.setVisible = function(visible) {
        var element = dojo.byId(thisref.clientId);
        var display = visible ? "block" : "none";
        element.style.display = display;
    }
}

//this code calls bpui.progressbar.ajaxOnloadHandler when the page has 
//finished loading.
bpui.progressbar.windowOnload = window.onload;
window.onload = bpui.progressbar.ajaxOnloadHandler;