require 'capistrano/command'

module Capistrano
  class Configuration
    module Actions
      module Invocation
        def self.included(base) #:nodoc:
          base.extend(ClassMethods)

          base.send :alias_method, :initialize_without_invocation, :initialize
          base.send :alias_method, :initialize, :initialize_with_invocation

          base.default_io_proc = Proc.new do |ch, stream, out|
            level = stream == :err ? :important : :info
            ch[:options][:logger].send(level, out, "#{stream} :: #{ch[:server]}")
          end
        end

        module ClassMethods
          attr_accessor :default_io_proc
        end

        def initialize_with_invocation(*args) #:nodoc:
          initialize_without_invocation(*args)
          set :default_environment, {}
          set :default_run_options, {}
        end

        def parallel(options={})
          raise ArgumentError, "parallel() requires a block" unless block_given?
          tree = Command::Tree.new(self) { |t| yield t }
          run_tree(tree)
        end

        # Invokes the given command. If a +via+ key is given, it will be used
        # to determine what method to use to invoke the command. It defaults
        # to :run, but may be :sudo, or any other method that conforms to the
        # same interface as run and sudo.
        def invoke_command(cmd, options={}, &block)
          options = options.dup
          via = options.delete(:via) || :run
          send(via, cmd, options, &block)
        end

        # Execute the given command on all servers that are the target of the
        # current task. If a block is given, it is invoked for all output
        # generated by the command, and should accept three parameters: the SSH
        # channel (which may be used to send data back to the remote process),
        # the stream identifier (<tt>:err</tt> for stderr, and <tt>:out</tt> for
        # stdout), and the data that was received.
        def run(cmd, options={}, &block)
          block ||= self.class.default_io_proc
          tree = Command::Tree.new(self) { |t| t.else(cmd, &block) }
          run_tree(tree, options)
        end

        def run_tree(tree, options={})
          if tree.branches.empty? && tree.fallback
            logger.debug "executing #{tree.fallback}"
          elsif tree.branches.any?
            logger.debug "executing multiple commands in parallel"
            tree.each do |branch|
              logger.trace "-> #{branch}"
            end
          else
            raise ArgumentError, "attempt to execute without specifying a command"
          end

          return if dry_run || (debug && continue_execution(tree) == false)

          options = add_default_command_options(options)

          tree.each do |branch|
            if branch.command.include?(sudo)
              branch.callback = sudo_behavior_callback(branch.callback)
            end
          end

          execute_on_servers(options) do |servers|
            targets = servers.map { |s| sessions[s] }
            Command.process(tree, targets, options.merge(:logger => logger))
          end
        end

        # Returns the command string used by capistrano to invoke a comamnd via
        # sudo.
        #
        #   run "#{sudo :as => 'bob'} mkdir /path/to/dir"
        #
        # It can also be invoked like #run, but executing the command via sudo.
        # This assumes that the sudo password (if required) is the same as the
        # password for logging in to the server.
        #
        #   sudo "mkdir /path/to/dir"
        #
        # Also, this method understands a <tt>:sudo</tt> configuration variable,
        # which (if specified) will be used as the full path to the sudo
        # executable on the remote machine:
        #
        #   set :sudo, "/opt/local/bin/sudo"
        def sudo(*parameters, &block)
          options = parameters.last.is_a?(Hash) ? parameters.pop.dup : {}
          command = parameters.first
          user = options[:as] && "-u #{options.delete(:as)}"

          sudo_prompt_option = "-p '#{sudo_prompt}'" unless sudo_prompt.empty?
          sudo_command = [fetch(:sudo, "sudo"), sudo_prompt_option, user].compact.join(" ")

          if command
            command = sudo_command + " " + command
            run(command, options, &block)
          else
            return sudo_command
          end
        end

        # Returns a Proc object that defines the behavior of the sudo
        # callback. The returned Proc will defer to the +fallback+ argument
        # (which should also be a Proc) for any output it does not
        # explicitly handle.
        def sudo_behavior_callback(fallback) #:nodoc:
          # in order to prevent _each host_ from prompting when the password
          # was wrong, let's track which host prompted first and only allow
          # subsequent prompts from that host.
          prompt_host = nil

          Proc.new do |ch, stream, out|
            if out =~ /^#{Regexp.escape(sudo_prompt)}/
              ch.send_data "#{self[:password]}\n"
            elsif out =~ /^Sorry, try again/
              if prompt_host.nil? || prompt_host == ch[:server]
                prompt_host = ch[:server]
                logger.important out, "#{stream} :: #{ch[:server]}"
                reset! :password
              end
            elsif fallback
              fallback.call(ch, stream, out)
            end
          end
        end

        # Merges the various default command options into the options hash and
        # returns the result. The default command options that are understand
        # are:
        #
        # * :default_environment: If the :env key already exists, the :env
        #   key is merged into default_environment and then added back into
        #   options.
        # * :default_shell: if the :shell key already exists, it will be used.
        #   Otherwise, if the :default_shell key exists in the configuration,
        #   it will be used. Otherwise, no :shell key is added.
        def add_default_command_options(options)
          defaults = self[:default_run_options]
          options = defaults.merge(options)

          env = self[:default_environment]
          env = env.merge(options[:env]) if options[:env]
          options[:env] = env unless env.empty?

          shell = options[:shell] || self[:default_shell]
          options[:shell] = shell unless shell.nil?

          options
        end

        # Returns the prompt text to use with sudo
        def sudo_prompt
          fetch(:sudo_prompt, "sudo password: ")
        end
        
        def continue_execution(tree)
          if tree.branches.length == 1
            continue_execution_for_branch(tree.branches.first)
          else
            tree.each { |branch| branch.skip! unless continue_execution_for_branch(branch) }
            tree.any? { |branch| !branch.skip? }
          end
        end

        def continue_execution_for_branch(branch)
          case Capistrano::CLI.debug_prompt(branch)
            when "y"
              true
            when "n"
              false
            when "a"
              exit(-1)
          end
        end
      end
    end
  end
end
