/*

Copyright (C) 2000 - 2007 Christian Kreibich <christian@whoop.org>.

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to
deal in the Software without restriction, including without limitation the
rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
sell copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies of the Software and its documentation and acknowledgment shall be
given in the documentation and software packages that this Software was
used.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
THE AUTHORS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

*/
#ifndef __nd_protocol_plugin_h
#define __nd_protocol_plugin_h

#include <ltdl.h>
#include <nd_protocol.h>

/* GUI protocol visualization plugins -- Netdude itself doesn't know how
 * to display any protocols, all visualizations are implemented as plugins
 * to keep things modularized.
 */
typedef struct nd_protoplugin
{
  const char     *(*name) (void);
  const char     *(*description) (void);
  const char     *(*author) (void);
  const char     *(*version) (void);

  /* Generic initialization/cleanup. Use as needed. */

  LND_Protocol   *(*init) (void);
  int            (*cleanup) (void);

  /* full path to .o filename */
  char          *filename;
  lt_dlhandle    lt;

  LND_Protocol  *proto;

} ND_ProtoPlugin;


/**
 * ND_ProtoPluginFunc - signature of plugin iteration callbacks.
 * @plugin: iterated plugin.
 * @user_data: arbitrary user data.
 *
 * This is the signature of functions that can be passed to
 * nd_proto_plugin_foreach().
 */
typedef void (*ND_ProtoPluginFunc)(ND_ProtoPlugin *plugin, void *user_data);


/**
 * nd_proto_plugin_init - initializes protocol plugins.
 *
 * The function scans the plugin directories, reads in the
 * plugins it finds and hooks them into the main window
 * and the plugin registry.
 */
void          nd_proto_plugin_init(void);

/**
 * nd_proto_plugin_get_description - returns plugin description string.
 * @pp: plugin to query.
 *
 * The function returns a pointer to statically allocated memory
 * containing a string describing the plugin.
 *
 * Returns: description string.
 */
const char   *nd_proto_plugin_get_description(ND_ProtoPlugin *pp);

/**
 * nd_proto_plugin_get_author - returns plugin author.
 * @pp: plugin to query.
 *
 * The function returns a pointer to statically allocated memory
 * containing a string with the author's name.
 *
 * Returns: author name string.
 */
const char   *nd_proto_plugin_get_author(ND_ProtoPlugin *pp);

/**
 * nd_proto_plugin_get_version - returns plugin version.
 * @pp: plugin to query.
 *
 * The function returns a pointer to statically allocated memory
 * containing the plugin's version string.
 *
 * Returns: version string.
 */
const char   *nd_proto_plugin_get_version(ND_ProtoPlugin *pp);

/**
 * nd_proto_plugin_get_name - returns plugin name.
 * @pp: plugin to query.
 *
 * The function returns a pointer to statically allocated memory
 * containing the plugin name.
 *
 * Returns: plugin name.
 */
const char   *nd_proto_plugin_get_name(ND_ProtoPlugin *pp);

/**
 * nd_proto_plugin_show_about - shows About dialog for plugin.
 * @pp: plugin whose About info is shown.
 *
 * The function shows the About dialog for @pp, containing
 * version and author info, a description etc.
 */
void          nd_proto_plugin_show_about(ND_ProtoPlugin *pp);


/**
 * nd_proto_plugin_foreach - plugin iteration.
 * @callback: callback to call for each plugin.
 * @user_data: arbitrary user data passed through to @callback.
 *
 * The function iterates over the registered protocol plugins
 * and applies @callback to each of them.
 */
void          nd_proto_plugin_foreach(ND_ProtoPluginFunc callback,
				      void *user_data);

#endif
