# -*- coding: utf-8 -*-
# Moovida - Home multimedia server
# Copyright (C) 2006-2009 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 3.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Moovida with Fluendo's plugins.
#
# The GPL part of Moovida is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Moovida" in the root directory of this distribution package
# for details on that license.

"""
Locale helper functions
"""

import locale, platform, sys

# "Sets the locale for all categories to the user's default setting
# (typically specified in the LANG environment variable)."
# quoted from Python's documentation:
# http://docs.python.org/lib/module-locale.html
#
# This is not thread-safe and does not need to be done more than once.
if locale.getlocale() == (None, None):
    locale.setlocale(locale.LC_ALL, "")

def system_encoding():
    """
    Return the encoding used by the system.

    @rtype: C{str}
    """
    # setlocale is done at import time; not doing it here allows
    # system_encoding to be thread-safe
    return locale.getpreferredencoding(do_setlocale=False)

def filesystem_encoding():
    """
    Return the encoding used by the file system.

    @rtype: C{str}
    """
    return sys.getfilesystemencoding()

def glib_file_encoding():
    """
    Return a guess of the encoding expected by the file manipulation functions
    of glib.

    @rtype: C{str}
    """
    # See
    # http://library.gnome.org/devel/glib/2.20/glib-File-Utilities.html#glib-File-Utilities.description

    if platform.system() == 'Windows':
        return 'UTF-8'

    return filesystem_encoding()

def gst_file_encoding():
    """
    Return a guess of the encoding expected by gstreamer for file
    names/paths/uris

    @rtype: C{str}
    """
    # gstreamer's filesrc uses a copy of glib's g_open()
    return glib_file_encoding()

def gdk_pixbuf_file_encoding():
    """
    Return the encoding expected by gdk_pixbuf functions for file paths.
    Please note you can also pass unicode objects to gdk_pixbuf.

    @rtype: C{str}
    """
    # Experimentally and according to the source code, gdk_pixbuf expects
    # something in file system encoding, even though its documentation says
    # it expects glib encoding.
    return filesystem_encoding()

def pgm_file_encoding():
    """
    Return the encoding expected by pigment functions for file paths.
    Please note you can also pass unicode objects to pigment.

    @rtype: C{str}
    """
    # pigment opens all its files through gdk_pixbuf
    return gdk_pixbuf_file_encoding()

