/*
 *
 * (C) Copyright IBM Corp. 1998-2011 - All Rights Reserved
 * with additions by Sun Microsystems 2002-2006
 *
 */

#ifndef __LESWAPS_H
#define __LESWAPS_H

#include "LETypes.h"

/**
 * \file 
 * \brief C++ API: Endian independent access to data for LayoutEngine
 */

U_NAMESPACE_BEGIN

// There exist popular font file which contain unaligned tables
// (e.g. "Watanabe Gothic"'s "mort" table)
// On some platforms unaligned memory accesses cause a crash.
// The ALLOW_UNALIGNED hack prevents these crashes by assuming that
// every use of the SWAPx macros in ICU's layout engine is intended
// for loading a big endian value and replaces them appropriately.
#define ALLOW_UNALIGNED_HACK

/**
 * A convenience macro which invokes the swapWord member function
 * from a concise call.
 *
 * @stable ICU 2.8
 */
#ifndef ALLOW_UNALIGNED_HACK
#define SWAPW(value) LESwaps::swapWord((le_uint16)(value))

/**
 * A convenience macro which invokes the swapLong member function
 * from a concise call.
 *
 * @stable ICU 2.8
 */
#define SWAPL(value) LESwaps::swapLong((le_uint32)(value))

#else // ALLOW_UNALIGNED_HACK

#define SWAPW(rValue) loadBigEndianWord(reinterpret_cast<const le_uint16&>(rValue))
#define SWAPL(rValue) loadBigEndianLong(reinterpret_cast<const le_uint32&>(rValue))

inline le_uint16 loadBigEndianWord( const le_uint16& rValue )
{
    const le_uint8* p = reinterpret_cast<const le_uint8*>(&rValue);
    return ((p[0] << 8) + p[1]);
}

inline le_uint32 loadBigEndianLong( const le_uint32& rValue )
{
    const le_uint8* p = reinterpret_cast<const le_uint8*>(&rValue);
    return ((p[0]<<24) + (p[1]<<16) + (p[2]<<8) +p[3]);
}

#endif // ALLOW_UNALIGNED_HACK


/**
 * This class is used to access data which stored in big endian order
 * regardless of the conventions of the platform.
 *
 * All methods are static and inline in an attempt to induce the compiler
 * to do most of the calculations at compile time.
 *
 * @stable ICU 2.8
 */
class U_LAYOUT_API LESwaps /* not : public UObject because all methods are static */ {
public:

    /**
     * Reads a big-endian 16-bit word and returns a native-endian value.
     * No-op on a big-endian platform, byte-swaps on a little-endian platform.
     *
     * @param value - the word to be byte swapped
     *
     * @return the byte swapped word
     *
     * @stable ICU 2.8
     */
    static le_uint16 swapWord(le_uint16 value)
    {
#if (defined(U_IS_BIG_ENDIAN) && U_IS_BIG_ENDIAN) || \
    (defined(BYTE_ORDER) && defined(BIG_ENDIAN) && (BYTE_ORDER == BIG_ENDIAN)) || \
    defined(__BIG_ENDIAN__)
        // Fastpath when we know that the platform is big-endian.
        return value;
#else
        // Reads a big-endian value on any platform.
        const le_uint8 *p = reinterpret_cast<const le_uint8 *>(&value);
        return (le_uint16)((p[0] << 8) | p[1]);
#endif
    };

    /**
     * Reads a big-endian 32-bit word and returns a native-endian value.
     * No-op on a big-endian platform, byte-swaps on a little-endian platform.
     *
     * @param value - the long to be byte swapped
     *
     * @return the byte swapped long
     *
     * @stable ICU 2.8
     */
    static le_uint32 swapLong(le_uint32 value)
    {
#if (defined(U_IS_BIG_ENDIAN) && U_IS_BIG_ENDIAN) || \
    (defined(BYTE_ORDER) && defined(BIG_ENDIAN) && (BYTE_ORDER == BIG_ENDIAN)) || \
    defined(__BIG_ENDIAN__)
        // Fastpath when we know that the platform is big-endian.
        return value;
#else
        // Reads a big-endian value on any platform.
        const le_uint8 *p = reinterpret_cast<const le_uint8 *>(&value);
        return (le_uint32)((p[0] << 24) | (p[1] << 16) | (p[2] << 8) | p[3]);
#endif
    };

private:
    LESwaps() {} // private - forbid instantiation
};

U_NAMESPACE_END
#endif
