<?php
/* Icinga Web 2 | (c) 2013-2015 Icinga Development Team | GPLv2+ */

namespace Icinga\Module\Monitoring\Forms\Command\Object;

use Icinga\Module\Monitoring\Command\Object\ToggleObjectFeatureCommand;
use Icinga\Module\Monitoring\Object\MonitoredObject;
use Icinga\Web\Notification;

/**
 * Form for enabling or disabling features of Icinga objects, i.e. hosts or services
 */
class ToggleObjectFeaturesCommandForm extends ObjectsCommandForm
{
    /**
     * (non-PHPDoc)
     * @see \Zend_Form::init() For the method documentation.
     */
    public function init()
    {
        $this->setUseFormAutosubmit();
        $this->setAttrib('class', 'inline object-features');
    }

    /**
     * (non-PHPDoc)
     * @see \Icinga\Web\Form::createElements() For the method documentation.
     */
    public function createElements(array $formData = array())
    {
        $toggleDisabled = $this->hasPermission('monitoring/command/feature/object')  ? null : '';

        $this->addElement(
            'checkbox',
            ToggleObjectFeatureCommand::FEATURE_ACTIVE_CHECKS,
            array(
                'label'         => $this->translate('Active Checks'),
                'autosubmit'    => true,
                'disabled'      => $toggleDisabled
            )
        );
        $this->addElement(
            'checkbox',
            ToggleObjectFeatureCommand::FEATURE_PASSIVE_CHECKS,
            array(
                'label'         => $this->translate('Passive Checks'),
                'autosubmit'    => true,
                'disabled'      => $toggleDisabled
            )
        );

        if (! preg_match('~^v2\.\d+\.\d+.*$~', $this->getIcingaVersion())) {
            $this->addElement(
                'checkbox',
                ToggleObjectFeatureCommand::FEATURE_OBSESSING,
                array(
                    'label'         => $this->translate('Obsessing'),
                    'autosubmit'    => true,
                    'disabled'      => $toggleDisabled
                )
            );
        }

        $this->addElement(
            'checkbox',
            ToggleObjectFeatureCommand::FEATURE_NOTIFICATIONS,
            array(
                'label'         => $this->translate('Notifications'),
                'autosubmit'    => true,
                'disabled'      => $toggleDisabled
            )
        );
        $this->addElement(
            'checkbox',
            ToggleObjectFeatureCommand::FEATURE_EVENT_HANDLER,
            array(
                'label'         => $this->translate('Event Handler'),
                'autosubmit'    => true,
                'disabled'      => $toggleDisabled
            )
        );
        $this->addElement(
            'checkbox',
            ToggleObjectFeatureCommand::FEATURE_FLAP_DETECTION,
            array(
                'label'         => $this->translate('Flap Detection'),
                'autosubmit'    => true,
                'disabled'      => $toggleDisabled
            )
        );
    }

    /**
     * Load feature status
     *
     * @param   MonitoredObject $object
     *
     * @return  $this
     */
    public function load(MonitoredObject $object)
    {
        $this->create();
        foreach ($this->getValues() as $feature => $enabled) {
            $element = $this->getElement($feature);
            $element->setChecked($object->{$feature});
            if ((bool) $object->{$feature . '_changed'} === true) {
                $element->setDescription($this->translate('changed'));
            }
        }

        return $this;
    }

    /**
     * (non-PHPDoc)
     * @see \Icinga\Web\Form::onSuccess() For the method documentation.
     */
    public function onSuccess()
    {
        $this->assertPermission('monitoring/command/feature/object');

        $notifications = array(
            ToggleObjectFeatureCommand::FEATURE_ACTIVE_CHECKS => array(
                $this->translate('Enabling active checks..'),
                $this->translate('Disabling active checks..')
            ),
            ToggleObjectFeatureCommand::FEATURE_PASSIVE_CHECKS => array(
                $this->translate('Enabling passive checks..'),
                $this->translate('Disabling passive checks..')
            ),
            ToggleObjectFeatureCommand::FEATURE_OBSESSING => array(
                $this->translate('Enabling obsessing..'),
                $this->translate('Disabling obsessing..')
            ),
            ToggleObjectFeatureCommand::FEATURE_NOTIFICATIONS => array(
                $this->translate('Enabling notifications..'),
                $this->translate('Disabling notifications..')
            ),
            ToggleObjectFeatureCommand::FEATURE_EVENT_HANDLER => array(
                $this->translate('Enabling event handler..'),
                $this->translate('Disabling event handler..')
            ),
            ToggleObjectFeatureCommand::FEATURE_FLAP_DETECTION => array(
                $this->translate('Enabling flap detection..'),
                $this->translate('Disabling flap detection..')
            )
        );

        foreach ($this->objects as $object) {
            /** @var \Icinga\Module\Monitoring\Object\MonitoredObject $object */
            foreach ($this->getValues() as $feature => $enabled) {
                if ((bool) $object->{$feature} !== (bool) $enabled) {
                    $toggleFeature = new ToggleObjectFeatureCommand();
                    $toggleFeature
                        ->setFeature($feature)
                        ->setObject($object)
                        ->setEnabled($enabled);
                    $this->getTransport($this->request)->send($toggleFeature);

                    Notification::success(
                        $notifications[$feature][$enabled ? 0 : 1]
                    );
                }
            }
        }

        return true;
    }

    /**
     * Fetch and return the program version of the current instance
     *
     * @return  string
     */
    protected function getIcingaVersion()
    {
        return $this->getBackend()->select()->from('programstatus', array('program_version'))->fetchOne();
    }
}
