/*******************************************************************************
 * Copyright (c) 2001, 2004 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.wst.rdb.connection.internal.ui.wizards;

import java.util.Arrays;
import java.util.List;
import java.util.ResourceBundle;

import org.eclipse.jface.wizard.IWizardPage;
import org.eclipse.jface.wizard.WizardPage;
import org.eclipse.swt.widgets.Combo;
import org.eclipse.wst.rdb.connection.internal.ui.util.TString;
import org.eclipse.wst.rdb.connection.internal.ui.util.resource.ResourceLoader;


public abstract class RDBWizardPage extends WizardPage
{
	// constants
	protected static final int SIZING_TEXT_FIELD_WIDTH = 150;
	protected static final int COMBO_HISTORY_LENGTH = 5;
	
   /** A static reference to our resource bundle. */
   protected static ResourceBundle bundle;
    
	/**
	 * The page to be shown next. This is set using setNextPage
	 * when a connection type radio button is selected.
	 */
	protected IWizardPage nextPage = null;
	
	/**
	 * Creates a new wizard page.
	 *
	 * @param pageName the name of the page
	 */
	protected RDBWizardPage(String pageName)
	{
		super(pageName);
	}
	
	/**
	 * Determines if the page as a whole is complete and correct.
	 * Calls this class and all child classes.
	 * @return True if the page as a whole is complete and correct.
	 */
	protected abstract boolean determinePageCompletion();
	
	/**
	 * Sets the page to be shown after this page.
	 * @param page The next wizard page
	 */
	public void setNextPage(IWizardPage page) {
		nextPage = page;
	}
	
	/**
	 * Returns the wizard page that would to be shown if the user was to
	 * press the Next button.
	 *
	 * @return the next wizard page, or <code>null</code> if none
	 */
	public IWizardPage getNextPage() {
		if (nextPage != null)
			return nextPage;
		else
			return super.getNextPage();
	}
	
	/**
	 * Adds an entry to a history, while taking care of duplicate history items
	 * and excessively long histories.  The assumption is made that all histories
	 * should be of length <code>WizardDataTransferPage.COMBO_HISTORY_LENGTH</code>.
	 *
	 * @param history the current history
	 * @param newEntry the entry to add to the history
	 */
	protected String[] addToHistory(String[] history, String newEntry)
	{
		if (history == null)
			history = new String[0];
		java.util.ArrayList l = new java.util.ArrayList(Arrays.asList(history));
		addToHistory(l,newEntry);
		String[] r = new String[l.size()];
		l.toArray(r);
		
		return r;
	}
	
	/**
	 * Adds an entry to a history, while taking care of duplicate history items
	 * and excessively long histories.  The assumption is made that all histories
	 * should be of length <code>WizardDataTransferPage.COMBO_HISTORY_LENGTH</code>.
	 *
	 * @param history the current history
	 * @param newEntry the entry to add to the history
	 */
	protected void addToHistory(List history, String newEntry)
	{
		history.remove(newEntry);
		history.add(0,newEntry);
		
		// since only one new item was added, we can be over the limit
		// by at most one item
		if (history.size() > COMBO_HISTORY_LENGTH)
			history.remove(COMBO_HISTORY_LENGTH);
	}
	
	/**
	 * Loads a combo with values from the dialog settings.
	 * @param A Combo control to be loaded with values from the dialog settings.
	 * @param Previous values from the dialog settings.
    * @param set True if we want to set the initial value to the first element.
	 */
	protected void loadComboSettings(Combo combo, String[] values, boolean set)
	{
      if (set)
		   combo.setText(values[0].trim());
		for(int i = 0; i < values.length; i++)
			combo.add(values[i]);
	}
	
   /** Gets a static reference to our resource bundle. */
   public static ResourceBundle getBundle()
   {
      if (bundle == null)
        bundle = ResourceLoader.INSTANCE.getResourceBundle();
      return bundle;
   }
    
	/**
	 * Gets translated text from the resource bundle.
	 * @param key The key specified in plugin.properties for the string.
	 * @return The string value for the given key.
	 */
	public String getString(String key)
	{
		return getBundle().getString(key);
	}
	
	/**
	 * Gets translated text with substitutions from the resource bundle.
	 * @param key The key specified in plugin.properties for the string.
	 * @param substitutions An Object array containing substitions for {0}, {1}, etc.
	 * @return The string value for the given key, with substitutions.
	 */
	public String getString(String key, Object[] substitutions)
	{
		String str = getBundle().getString(key);
		int s = 0;
		if (substitutions != null)
			s = substitutions.length;
		if (s > 0 && str.indexOf("%1") > -1) //$NON-NLS-1$
			str = TString.change(str, "%1", substitutions[0].toString()); //$NON-NLS-1$
		if (s > 1 && str.indexOf("%2") > -1) //$NON-NLS-1$
			str = TString.change(str, "%2", substitutions[1].toString()); //$NON-NLS-1$
		if (s > 2 && str.indexOf("%3") > -1) //$NON-NLS-1$
			str = TString.change(str, "%3", substitutions[2].toString()); //$NON-NLS-1$
		return str;
	}
	
   /**
	 * The <code>WizardPage</code> implementation of this method
	 * declared on <code>DialogPage</code> updates the container
	 * if this is the current page.
	 * This implementation first checks that the message has
	 * already been set.
	 */
	public void setErrorMessage(String newMessage) {
		String oldMessage = getErrorMessage();
		if ((newMessage == null && oldMessage != null)
		|| (newMessage != null && oldMessage == null)
		|| (newMessage != null && !newMessage.equals(oldMessage))
		|| (oldMessage != null && !oldMessage.equals(newMessage)))
		{
			super.setErrorMessage(newMessage);
		}
	}
	
}
