# Copyright 2013 Google Inc. All Rights Reserved.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#    http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

"""config command group."""

import argparse

from googlecloudsdk.calliope import base
from googlecloudsdk.calliope import exceptions as c_exc
from googlecloudsdk.core import config
from googlecloudsdk.core import properties


class Config(base.Group):
  """View and edit Google Cloud SDK properties.

  Configuration properties are used to modify the behavior of gcloud and other
  Cloud SDK commands.  Most behavior can be controlled via the use of flags,
  but setting properties provides a way to maintain the same settings across
  command executions.
  """

  detailed_help = {
      'DESCRIPTION': '{description}',
      '+AVAILABLE PROPERTIES': properties.VALUES.GetHelpString(),
  }

  INSTALLATION_FLAG = base.Argument(
      '--installation',
      required=False,
      action='store_true',
      help='Update the property in the gcloud installation.',
      detailed_help="""\
          Typically properties are updated only in the currently active
          configuration, but when `--installation` is given the property is
          updated for the entire gcloud installation."""
      )

  DEPRECATED_SCOPE_FLAG = base.Argument(
      '--scope',
      required=False,
      choices=properties.Scope.AllScopeNames(),
      help=argparse.SUPPRESS)

  @staticmethod
  def RequestedScope(args):
    # This hackiness will go away when we rip out args.scope everywhere
    install = 'installation' if getattr(args, 'installation', False) else None
    scope_arg = getattr(args, 'scope', None)

    return properties.Scope.FromId(scope_arg or install)

  @staticmethod
  def PropertiesCompleter(prefix, parsed_args, **unused_kwargs):
    """Properties commmand completion helper."""
    all_sections = properties.VALUES.AllSections()
    options = []

    if '/' in prefix:
      # Section has been specified, only return properties under that section.
      parts = prefix.split('/', 1)
      section = parts[0]
      prefix = parts[1]
      if section in all_sections:
        section_str = section + '/'
        props = properties.VALUES.Section(section).AllProperties()
        options.extend([section_str + p for p in props if p.startswith(prefix)])
    else:
      # No section.  Return matching sections and properties in the default
      # group.
      options.extend([s + '/' for s in all_sections if s.startswith(prefix)])
      section = properties.VALUES.default_section.name
      props = properties.VALUES.Section(section).AllProperties()
      options.extend([p for p in props if p.startswith(prefix)])

    return options
