/**
 * @class Array
 * #core
 */
Ext.apply(Array.prototype, {
	/**
	 * Creates a shallow clone of the array. 
	 * @return {Array} a clone of the array.
	 */
	clone : function()
	{
		return this.slice(0);
	},

	/**
	 * Equality comparison for the Array. Only when both
	 * arrays contain the exact same elements the Arrays
	 * are equal.
	 */
	equals : function(arr)
	{
		// We are comparing the exact same references.
		if (this === arr) {
			return true;
		}

		// If the lengths are not equal, then the arrays
		// cannot be equal.
		if (this.length !== arr.length) {
			return false;
		}

		// Compare each element for equality.
		for (var i = 0, len = arr.length; i < len; i++) {
			if (this[i] !== arr[i]) {
				return false;
			}
		}
		return true;
	},

	/**
	 * Removes all elements from the array.
	 */
	clear : function()
	{
		this.splice(0, this.length);
	}
});
/**
 * @class Date
 *
 * The date parsing and formatting syntax contains a subset of
 * <a href="http://www.php.net/date">PHP's date() function</a>, and the formats that are
 * supported will provide results equivalent to their PHP versions.
 *
 * The following is a list of all currently supported formats:
 * <pre>
Format  Description                                                               Example returned values
------  -----------------------------------------------------------------------   -----------------------
  d     Day of the month, 2 digits with leading zeros                             01 to 31
  D     A short textual representation of the day of the week                     Mon to Sun
  j     Day of the month without leading zeros                                    1 to 31
  l     A full textual representation of the day of the week                      Sunday to Saturday
  N     ISO-8601 numeric representation of the day of the week                    1 (for Monday) through 7 (for Sunday)
  S     English ordinal suffix for the day of the month, 2 characters             st, nd, rd or th. Works well with j
  w     Numeric representation of the day of the week                             0 (for Sunday) to 6 (for Saturday)
  z     The day of the year (starting from 0)                                     0 to 364 (365 in leap years)
  W     ISO-8601 week number of year, weeks starting on Monday                    01 to 53
  F     A full textual representation of a month, such as January or March        January to December
  m     Numeric representation of a month, with leading zeros                     01 to 12
  M     A short textual representation of a month                                 Jan to Dec
  n     Numeric representation of a month, without leading zeros                  1 to 12
  t     Number of days in the given month                                         28 to 31
  L     Whether it's a leap year                                                  1 if it is a leap year, 0 otherwise.
  o     ISO-8601 year number (identical to (Y), but if the ISO week number (W)    Examples: 1998 or 2004
  belongs to the previous or next year, that year is used instead)
  Y     A full numeric representation of a year, 4 digits                         Examples: 1999 or 2003
  y     A two digit representation of a year                                      Examples: 99 or 03
  a     Lowercase Ante meridiem and Post meridiem                                 am or pm
  A     Uppercase Ante meridiem and Post meridiem                                 AM or PM
  g     12-hour format of an hour without leading zeros                           1 to 12
  G     24-hour format of an hour without leading zeros                           0 to 23
  h     12-hour format of an hour with leading zeros                              01 to 12
  H     24-hour format of an hour with leading zeros                              00 to 23
  i     Minutes, with leading zeros                                               00 to 59
  s     Seconds, with leading zeros                                               00 to 59
  u     Decimal fraction of a second                                              Examples:
        (minimum 1 digit, arbitrary number of digits allowed)                     001 (i.e. 0.001s) or
                                                                                  100 (i.e. 0.100s) or
                                                                                  999 (i.e. 0.999s) or
                                                                                  999876543210 (i.e. 0.999876543210s)
  O     Difference to Greenwich time (GMT) in hours and minutes                   Example: +1030
  P     Difference to Greenwich time (GMT) with colon between hours and minutes   Example: -08:00
  T     Timezone abbreviation of the machine running the code                     Examples: EST, MDT, PDT ...
  Z     Timezone offset in seconds (negative if west of UTC, positive if east)    -43200 to 50400
  c     ISO 8601 date
        Notes:                                                                    Examples:
        1) If unspecified, the month / day defaults to the current month / day,   1991 or
           the time defaults to midnight, while the timezone defaults to the      1992-10 or
           browser's timezone. If a time is specified, it must include both hours 1993-09-20 or
           and minutes. The "T" delimiter, seconds, milliseconds and timezone     1994-08-19T16:20+01:00 or
           are optional.                                                          1995-07-18T17:21:28-02:00 or
        2) The decimal fraction of a second, if specified, must contain at        1996-06-17T18:22:29.98765+03:00 or
           least 1 digit (there is no limit to the maximum number                 1997-05-16T19:23:30,12345-0400 or
           of digits allowed), and may be delimited by either a '.' or a ','      1998-04-15T20:24:31.2468Z or
        Refer to the examples on the right for the various levels of              1999-03-14T20:24:32Z or
        date-time granularity which are supported, or see                         2000-02-13T21:25:33
        http://www.w3.org/TR/NOTE-datetime for more info.                         2001-01-12 22:26:34
  U     Seconds since the Unix Epoch (January 1 1970 00:00:00 GMT)                1193432466 or -2138434463
  M$    Microsoft AJAX serialized dates                                           \/Date(1238606590509)\/ (i.e. UTC milliseconds since epoch) or
                                                                                  \/Date(1238606590509+0800)\/
</pre>
 *
 * Example usage (note that you must escape format specifiers with '\\' to render them as character literals):
 * <pre><code>
// Sample date:
// 'Wed Jan 10 2007 15:05:01 GMT-0600 (Central Standard Time)'

var dt = new Date('1/10/2007 03:05:01 PM GMT-0600');
document.write(dt.format('Y-m-d'));                           // 2007-01-10
document.write(dt.format('F j, Y, g:i a'));                   // January 10, 2007, 3:05 pm
document.write(dt.format('l, \\t\\he jS \\of F Y h:i:s A'));  // Wednesday, the 10th of January 2007 03:05:01 PM
</code></pre>
 *
 * Here are some standard date/time patterns that you might find helpful.  They
 * are not part of the source of Date.js, but to use them you can simply copy this
 * block of code into any script that is included after Date.js and they will also become
 * globally available on the Date object.  Feel free to add or remove patterns as needed in your code.
 * <pre><code>
Date.patterns = {
    ISO8601Long:"Y-m-d H:i:s",
    ISO8601Short:"Y-m-d",
    ShortDate: "d/m/Y",
    LongDate: "l jS F Y",
    FullDateTime: "l jS F Y G:i:s",
    MonthDay: "jS F",
    ShortTime: "G:i",
    LongTime: "G:i:s",
    SortableDateTime: "Y-m-d\\TH:i:s",
    UniversalSortableDateTime: "Y-m-d H:i:sO",
    YearMonth: "F, Y"
};
</code></pre>
 *
 * Example usage:
 * <pre><code>
var dt = new Date();
document.write(dt.format(Date.patterns.ShortDate));
</code></pre>
 * <p>Developer-written, custom formats may be used by supplying both a formatting and a parsing function
 * which perform to specialized requirements. The functions are stored in {@link #parseFunctions} and {@link #formatFunctions}.</p>
 * #core
 */
Ext.apply(Date.prototype, {
	/**
	 * Provides a convenient method for performing basic date arithmetic. This method
	 * does not modify the Date instance being called - it creates and returns
	 * a new Date instance containing the resulting date value.
	 *
	 * Examples:
	 * <pre><code>
		// Basic usage:
		var dt = new Date('10/29/2006').add(Date.DAY, 5);
		document.write(dt); //returns 'Fri Nov 03 2006 00:00:00'

		// Negative values will be subtracted:
		var dt2 = new Date('10/1/2006').add(Date.DAY, -5);
		document.write(dt2); //returns 'Tue Sep 26 2006 00:00:00'

		// You can even chain several calls together in one line:
		var dt3 = new Date('10/1/2006').add(Date.DAY, 5).add(Date.HOUR, 8).add(Date.MINUTE, -30);
		document.write(dt3); //returns 'Fri Oct 06 2006 07:30:00'
	 * </code></pre>
	 *
	 * Furthermore, changes to {@link Date#HOUR hours}, {@link Date#MINUTE minutes},
	 * {@link Date#SECOND seconds} and {@link Date#MILLI milliseconds} are treated more accurately
	 * regarding DST changes then the {@link Date#DAY days}, {@link Date#MONTH months} and {@link Date#YEAR years}
	 * changes. When changing the time the standard is applied, which means that if the DST kicks in at 2AM,
	 * and the time becomes 3AM. Doing new Date('Mar 25 2012 01:00').add(Date.HOUR, 1) will be 'Mar 25 2012 03:00'.
	 * However when changing the date, we will use the JS behavior, which means that
	 * new Date('Mar 24 2012 02:00').add(Date.DAY, 1) could become 'Mar 25 2012 01:00' as JS will not correctly
	 * move the time correctly passed the DST switch.
	 *
	 * @param {String} interval A valid date interval enum value.
	 * @param {Number} value The amount to add to the current date.
	 * @return {Date} The new Date instance.
	 */
	add : function(interval, value)
	{
		var d = this.clone();
		if (!interval || value === 0) {
			return d;
		}

		switch(interval.toLowerCase()) {
			// Changing the time is done more accuretely then
			// changing the date. This is because we have to work
			// around DST issues (which we don't care for when
			// changing the day). In JS, we have the following
			// scenario at the following date: Mar 25 2012.
			// At 2:00:00 the DST kicks in and the time will be
			//     Mar 25 2012 03:00:00
			// However, when using setMilliseconds, setSeconds,
			// setMinutes or setHours, JS decides to wrap back
			// to:
			// 	Mar 25 2012 01:00:00
			// How can this go wrong, take the following date:
			//      a = new Date('Mar 25 2012 01:45:00')
			// add 30 minutes to it
			//      a.setMinutes(a.getMinutes() + 30)
			// we expect the time to be 03:15:00 however JS
			// decides to fall back to 01:15:00.
			// To fix this correctly, we have to work using timestamps
			// as JS is able to correctly step over the DST switch.
			case Date.HOUR:
				// Convert value to minutes
				value *= 60;
			case Date.MINUTE:
				// Convert value to seconds
				value *= 60;
			case Date.SECOND:
				// Convert value to milliseconds
				value *= 1000;
			case Date.MILLI:
				d = new Date(d.getTime() + value);
				break;
			// Changing the date is done with less accuracy,
			// basically we don't care if we come at exactly
			// the same time as before. If the JS decides to
			// perform weird tricks, then so be it.
			case Date.DAY:
				d.setDate(this.getDate() + value);
				break;
			case Date.MONTH:
				var day = this.getDate();
				if (day > 28) {
					day = Math.min(day, this.getFirstDateOfMonth().add(Date.MONTH, value).getLastDateOfMonth().getDate());
				}
				d.setDate(day);
				d.setMonth(this.getMonth() + value);
				break;
			case Date.YEAR:
				d.setFullYear(this.getFullYear() + value);
				break;
		}
		return d;
	},

	/**
	 * This should be called if the Date() object represents a UTC date, and we want to obtain
	 * the time it represents in UTC shown as if it was the localtime. This implies that:
	 *
	 * 00:00 UTC (01:00 GMT+0100) will be converted to 00:00 GMT+0100 (23:00 UTC)
	 *
	 * @return {Date} The UTC date
	 */
	toUTC : function()
	{
		var utc = new Date(this.getTime() + (this.getTimezoneOffset() * 60000));

		// Obtain the DST difference which might have occured during conversion,
		// if there was a difference it must be applied to the utc date accordingly.
		utc.setMilliseconds(utc.getMilliseconds() + Date.getDSTDiff(utc, this));

		return utc;
	},

	/**
	 * This should be called if the Date() was obtained using {@link #toUTC}, and we want
	 * to convert the date back to the local representation.
	 *
	 * 00:00 GMT+0100 (23:00 UTC) with be converted to 00:00 UTC (01:00 GMT+0100) 
	 *
	 * @return {Date} The local-time date
	 */
	fromUTC : function()
	{
		return new Date(this.getTime() - (this.getTimezoneOffset() * 60000));
	},

	/**
	 * Get the next given weekday starting from this date. If the current date is this weekday,
	 * then the current day will be returned.
	 * @param {Number} weekday The day in the week to skip to (0: Sunday -  6: Saturday). If
	 * not given, tomorrow will be returned.
	 * @return {Date} this or the clone
	 */
	getNextWeekDay : function(weekday)
	{
		var currentday = this.getDay();

		if (!Ext.isDefined(weekday)) {
			return this.add(Date.DAY, 1);
		} else if (weekday < currentday) {
			return this.add(Date.DAY, 7 - (currentday - weekday));
		} else {
			return this.add(Date.DAY, weekday - currentday);
		}
	},

	/**
	 * Get the previous given weekday starting from this date. If the current date is this weekday,
	 * then the current day will be returned.
	 * @param {Number} weekday The day in the week to skip to (0: Sunday -  6: Saturday). If
	 * not given, yesterday will be returned.
	 * @return {Date} this or the clone
	 */
	getPreviousWeekDay : function(weekday)
	{
		var currentday = this.getDay();

		if (!Ext.isDefined(weekday)) {
			return this.add(Date.DAY, -1);
		} else if (weekday <= currentday) {
			return this.add(Date.DAY, weekday - currentday);
		} else {
			return this.add(Date.DAY, -7 + (weekday - currentday));
		}
	},

	/**
	 * Attempts to clear all Second and millisecond time information from this Date by rounding the time down
	 * to the current minute.
	 * @param {Boolean} clone true to create a clone of this date, clear the time and return it (defaults to false).
	 * @return {Date} this or the clone.
	 */
	clearSeconds : function(clone) 
	{
		if (clone) {
			return this.clone().clearSeconds();
		}

		// clear seconds
		this.setSeconds(0);
		this.setMilliseconds(0);

		return this;
	},

	/**
	 * Round the number of milliseconds/seconds/minutes or hours depending on the given value.
	 * Note that days, months and years cannot be rounded.
	 *
	 * Example of rounding:
	 *  9:12   round to 30	-> 9:00
	 *  9:17   round to 30	-> 9:30
	 *
	 * @param {String} field The field to round (e.g. {@link Date.MINUTE}, {@link Date.SECOND}, etc.)
	 * @param {Number} roundTimeValue The number of minutes to round the time to.
	 * @return {Date} this date
	 */
	round : function(field, roundTimeValue)
	{
		// For each field we have a slightly different approach.
		// In all cases, if the field-value is already rounded,
		// then we don't need to do anything.
		// The calculation for the rounded value looks a bit weird, but
		// it is a bit more optimal then calling this.floor() or this.ceiling().
		// For seconds or higher units, we set all smaller units to 0,
		// to correctly round of the entire time.
		switch (field) {
			case Date.MILLI:
				var value = this.getMilliseconds();
				if (value % roundTimeValue > 0) {
					this.setMilliseconds(value - (value % roundTimeValue) + (((value % roundTimeValue) >= (roundTimeValue / 2)) * roundTimeValue));
				}
				break;
			case Date.SECOND:
				var value = this.getSeconds();
				if (value % roundTimeValue > 0) {
					this.setSeconds(value - (value % roundTimeValue) + (((value % roundTimeValue) >= (roundTimeValue / 2)) * roundTimeValue));
				}
				this.setMilliseconds(0);
				break;
			case Date.MINUTE:
				var value = this.getMinutes();
				if (value % roundTimeValue > 0) {
					this.setMinutes(value - (value % roundTimeValue) + (((value % roundTimeValue) >= (roundTimeValue / 2)) * roundTimeValue));
				}
				this.setSeconds(0);
				this.setMilliseconds(0);
				break;
			case Date.HOUR:
				var value = this.getHours();
				if (value % roundTimeValue > 0) {
					this.setHours(value - (value % roundTimeValue) + (((value % roundTimeValue) >= (roundTimeValue / 2)) * roundTimeValue));
				}
				this.setMinutes(0);
				this.setSeconds(0);
				this.setMilliseconds(0);
				break;
		}

		return this;
	},

	/**
	 * Function to ceil timings according to the passed ceil milliseconds, seconds, minutes or hours.
	 * Note that days, months and years cannot be ceiled.
	 * @param {String} field The field to ceil (e.g. {@link Date.MINUTE}, {@link Date.SECOND}, etc.)
	 * @param date ceilTimeValue date time which needs to be ceil (5/10/15/30/60 or so on)
	 * @return number Time number which is unixtimestamp of time.
	 *
	 * Example to understand what the code is actually suppose to do.
	 *	9:12	5min		ceil-9:15
	 *			10min		ceil-9.20
	 *			15min		ceil-9.15
	 *			30min		ceil-9.30
	 *			1hr/60min	ceil-10.00
	 *
	 */ 
	ceil : function(field, ceilTimeValue)
	{
		// For each field we have a slightly different approach.
		// In all cases, if the field-value is already rounded to the
		// given ceiling then we don't need to do anything.
		// For seconds or higher units, we set all smaller units to 0,
		// to correctly round of the entire time.
		switch (field) {
			case Date.MILLI:
				var value = this.getMilliseconds();
				if (value % ceilTimeValue > 0) {
					this.setMilliseconds(value - (value % ceilTimeValue) + ceilTimeValue);
				}
				break;
			case Date.SECOND:
				var value = this.getSeconds();
				if (value % ceilTimeValue > 0) {
					this.setSeconds(value - (value % ceilTimeValue) + ceilTimeValue);
				}
				this.setMilliseconds(0);
				break;
			case Date.MINUTE:
				var value = this.getMinutes();
				if (value % ceilTimeValue > 0) {
					this.setMinutes(value - (value % ceilTimeValue) + ceilTimeValue);
				}
				this.setSeconds(0);
				this.setMilliseconds(0);
				break;
			case Date.HOUR:
				var value = this.getHours();
				if (value % ceilTimeValue > 0) {
					this.setHours(value - (value % ceilTimeValue) + ceilTimeValue);
				}
				this.setMinutes(0);
				this.setSeconds(0);
				this.setMilliseconds(0);
				break;
		}

		return this;
	},

	/**
	 * Function to floor timings according to the passed floor milliseconds, seconds, minutes or hours.
	 * Note that days, months and years cannot be floored.
	 * @param {String} field The field to floor (e.g. {@link Date.MINUTE}, {@link Date.SECOND}, etc.)
	 * @param {Number} floorTimeValue date time which needs to be floor (5/10/15/30/60 or so on)
	 * @return {Date} This Date object
	 *
	 * Example to understand what the code is actually suppose to do.
	 *	9:12	5min		floor-9.10
	 *			10min		floor-9.10
	 *			15min		floor-9.00
	 *			30min		floor-9.00
	 *			1hr/60min	floor-9.00
	 *
	 */ 
	floor : function(field, floorTimeValue)
	{
		// For each field we have a slightly different approach.
		// In all cases, if the field-value is already rounded to the
		// given floor then we don't need to do anything.
		// For seconds or higher units, we set all smaller units to 0,
		// to correctly round of the entire time.
		switch (field) {
			case Date.MILLI:
				var value = this.getMilliseconds();
				if (value % floorTimeValue > 0) {
					this.setMilliseconds(value - (value % floorTimeValue));
				}
				break;
			case Date.SECOND:
				var value = this.getSeconds();
				if (value % floorTimeValue > 0) {
					this.setSeconds(value - (value % floorTimeValue));
				}
				this.setMilliseconds(0);
				break;
			case Date.MINUTE:
				var value = this.getMinutes();
				if (value % floorTimeValue > 0) {
					this.setMinutes(value - (value % floorTimeValue));
				}
				this.setSeconds(0);
				this.setMilliseconds(0);
				break;
			case Date.HOUR:
				var value = this.getHours();
				if (value % floorTimeValue > 0) {
					this.setHours(value - (value % floorTimeValue));
				}
				this.setMinutes(0);
				this.setSeconds(0);
				this.setMilliseconds(0);
				break;
		}

		return this;
	},

	/**
	 * Get the week number of the month (1 to 5)
	*/
	getWeekOfMonth : function()
	{
		// get current week number in year
		var currentWeek = this.getWeekOfYear();

		// get month's first week number in year
		var monthStartDate = this.add(Date.DAY, -(this.getDate() - 1));
		var monthStartWeek = monthStartDate.getWeekOfYear();

		return currentWeek - monthStartWeek + 1; 
	}
});

Ext.apply(Date, {
	/**
	 * The number milliseconds per day
	 *
	 * @property
	 * @type Number
	 */
	dayInMillis : 24 * 60 * 60 * 1000,

	/**
	 * Calculate the DST difference between the 2 given dates.
	 * The first date serves as base, so when 'date' is not DST, but
	 * the second date is DST then a negative offset is returned. A
	 * positive value is returned when it is the other way around.
	 * When both dates have the same DST offset then this returns 0.
	 * @param {Date} date The base date from where the DST is calculated.
	 * @return {Number} milliseconds The DST difference in milliseconds
	 */
	getDSTDiff : function(a, b)
	{
		return (a.getTimezoneOffset() - b.getTimezoneOffset()) * 60 * 1000;
	},

	/**
	 * Calculates the difference between the 2 given dates.
	 * This applies the {@link #getDSTDiff} if needed to ensure that
	 * it always calculates the correct difference regardless of the DST changes
	 * which might have been made.
	 *
	 * In its absolute basic this function is equal to 'a.getTime() - b.getTime()'.
	 *
	 * @param {String} field The field which indicates the accuracy of the diff (e.g. {@link Date.MINUTE}, {@link Date.SECOND}, etc.)
	 * @param {Date} a The date object
	 * @param {Date} b The date object
	 * @return {Number} The difference between the 2 given dates
	 */
	diff : function(field, a, b)
	{
		var ta = a.getTime();
		var tb = b.getTime();
		var difference = ta-tb;

		switch (field) {
			case Date.DAY:
				// For calculating days we apply the same
				// inaccuracy as Date::add() we are not 100%
				// sure a day lasts 24 hour when DST is in play.
				difference -= Date.getDSTDiff(a, b);
				difference /= 24;
			case Date.HOUR:
				difference /= 60;
			case Date.MINUTE:
				difference /= 60;
			case Date.SECOND:
				difference /= 1000;
			case Date.MILLI:
			default:
				break;
		}

		return difference;
	},

	/**
	 * Function to getTimezone and all dst props
	 * This is a hard one. To create a recurring appointment, we need to save
	 * the start and end time of the appointment in local time. So if I'm in 
	 * GMT+8, and I want the appointment at 9:00, I will simply save 9*60 = 540
	 * in the startDate. To make this usable for other users in other timezones,
	 * we have to tell the server in which timezone this is. The timezone is normally
	 * defined as a startdate and enddate for DST, the offset in minutes (so GMT+2 is 120)
	 * plus the extra DST offset when DST is in effect. 
	 *
	 * We can't retrieve this directly from the browser, so we assume that the DST change
	 * will occure on a Sunday at 2:00 or 3:00 AM, and simply scan all the sundays in a
	 * year, looking for changes. We then have to guess which bit is DST and which is 'normal'
	 * by assuming that the DST offset will be less than the normal offset. From this we
	 * calculate the start and end dates of DST and the actuall offset in minutes.
	 *
	 * Unfortunately we can't detect the difference between 'the last week of october' and
	 * 'the fourth week of october'. This can cause subtle problems, so we assume 'last week'
	 * because this is most prevalent.
	 * 
	 * Note that this doesn't work for many strange DST changes, see 
	 * http://webexhibits.org/daylightsaving/g.html
	 * @static
	 */
	getTimezoneStruct : function()
	{
		var tzswitch = [],
			switchCount = 0,
			testDate = new Date(),
			tzStruct = {};

		// Clear the time
		testDate.setMonth(0);
		testDate.setDate(1);
		testDate.setMinutes(0);
		testDate.setSeconds(0);
		testDate.setMilliseconds(0);

		// Move to the next sunday
		testDate = testDate.getNextWeekDay(0);

		// Use 5:00 am because any change should have happened by then
		testDate.setHours(5);

		var lastoffset = testDate.getTimezoneOffset();

		for(var weekNr = 0; weekNr < 52; weekNr++) {
			if(testDate.getTimezoneOffset() != lastoffset) {
				// Found a switch
				tzswitch[switchCount] = {
					switchweek : testDate.getWeekOfMonth(),
					switchmonth : testDate.getMonth(),
					offset : testDate.getTimezoneOffset()
				};

				switchCount++;

				// We assume DST is only set or removed once per year
				if(switchCount == 2)
					break;
					
				lastoffset = testDate.getTimezoneOffset();
			}
			
			// advance one week
			testDate = testDate.add(Date.DAY, 7);
		}
		
		if(switchCount == 0) {
			// No DST in this timezone
			tzStruct = {
				timezone : testDate.getTimezoneOffset(),
				timezonedst : 0,
				dststartday : 0,
				dststartweek : 0,
				dststartmonth : 0,
				dststarthour : 0,
				dstendday : 0,
				dstendweek : 0,
				dstendmonth : 0,
				dstendhour : 0
			};

			return tzStruct;
		} else if(switchCount == 1) {
			// This should be impossible unless DST started somewhere in the year 2000
			// and ended more than a year later. This is an error.
			return tzStruct;
		} else if(switchCount == 2) {
			if(tzswitch[0].offset < tzswitch[1].offset) {
				// Northern hemisphere, eg DST is during Mar-Oct
				tzStruct = {
					timezone : tzswitch[1].offset,
					timezonedst : tzswitch[0].offset - tzswitch[1].offset,
					dststartday : 0, // assume sunday
					dststartweek : tzswitch[0].switchweek == 4 ? 5 : tzswitch[0].switchweek, // assume 'last' week if week = 4
					dststartmonth : tzswitch[0].switchmonth + 1, // javascript months are zero index based
					dststarthour : 2, // Start at 02:00 AM
					dstendday : 0,
					dstendweek : tzswitch[1].switchweek == 4 ? 5 : tzswitch[1].switchweek,
					dstendmonth : tzswitch[1].switchmonth + 1,
					dstendhour : 3
				};

				return tzStruct;
				
			} else {
				// Southern hemisphere, eg DST is during Oct-Mar
				tzStruct = {
					timezone : tzswitch[0].offset,
					timezonedst : tzswitch[1].offset - tzswitch[0].offset,
					dststartday : 0, // assume sunday
					dststartweek : tzswitch[1].switchweek == 4 ? 5 : tzswitch[1].switchweek, // assume 'last' week if week = 4
					dststartmonth : tzswitch[1].switchmonth + 1,
					dststarthour : 2, // Start at 02:00 AM
					dstendday : 0,
					dstendweek : tzswitch[0].switchweek == 4 ? 5 : tzswitch[0].switchweek,
					dstendmonth : tzswitch[0].switchmonth + 1,
					dstendhour : 3
				};
				
				return tzStruct;
			}
		} else {
			// Multi-DST timezone ? This is also an error.
			return tzStruct;
		}
	}
});
/**
 * @class Object
 * #core
 */
// Backwards compatibility with old browsers which you should not use!
// Reference: https://developer.mozilla.org/en/JavaScript/Reference/Global_Objects/Object/keys 
if (!Object.keys) {
	Object.keys = (function () {
		var hasOwnProperty = Object.prototype.hasOwnProperty,
		hasDontEnumBug = !({toString: null}).propertyIsEnumerable('toString'),
		dontEnums = [
			'toString',
			'toLocaleString',
			'valueOf',
			'hasOwnProperty',
			'isPrototypeOf',
			'propertyIsEnumerable',
			'constructor'
		],
		dontEnumsLength = dontEnums.length

		return function (obj) {
			if (typeof obj !== 'object' && typeof obj !== 'function' || obj === null)
				throw new TypeError('Object.keys called on non-object')

			var result = []

			for (var prop in obj) {
				if (hasOwnProperty.call(obj, prop)) result.push(prop)
			}

			if (hasDontEnumBug) {
				for (var i=0; i < dontEnumsLength; i++) {
					if (hasOwnProperty.call(obj, dontEnums[i])) result.push(dontEnums[i])
				}
			}
			return result
		}
	})()
}
/**
 * @class String
 * #core
 */
Ext.apply(String.prototype, {
	/*
	 * Trims whitespace from either end of a string, leaving spaces within the string intact.
	 * This function in ExtJS was only trimming SP (\x20) but not NBSP (\xA0) so changed regular expression
	 * to trim SP and NBSP both.
	 * acording to ECMA-262 NBSP is a whitespace character so it should be handled by \s class in regular
	 * expressions but probably IE doesn't know that.
	 * Example:
	 * <pre><code>
	 var s = '  foo bar  ';
	 alert('-' + s + '-');         //alerts "- foo bar -"
	 alert('-' + s.trim() + '-');  //alerts "-foo bar-"
	 </code></pre>
	 * @return {String} The trimmed string
	 */
	trim : function()
	{
		var re = /^[\s\xA0]+|[\s\xA0]+$/g;
		return function() { return this.replace(re, ""); };
	}()
});

Ext.applyIf(String, {
	/**
	 * Pads the right side of a string with a specified character.  This is especially useful
	 * for normalizing number and date strings.  Example usage:
	 * <pre><code>
	 var s = String.rightPad('123', 5, '0');
	 // s now contains the string: '12300'
	 * </code></pre>
	 * @param {String} value The original string
	 * @param {Number} padSize The total length of the output string
	 * @param {String} padChar (optional) The character with which to pad the original string (defaults to empty string " ")
	 * @return {String} The padded string
	 * @static
	 */
	rightPad : function(value, padSize, padChar)
	{
		var result = String(value);
		if(!padChar) {
			padChar = ' ';
		}

		while (result.length < padSize) {
			result += padChar;
		}

		return result;
	}
});
Ext.namespace('Zarafa');

/**
 * @class Zarafa
 * Global convenience methods.
 * @singleton
 * #core
 */
Ext.apply(Zarafa, {
	/**
	 * Ready flag which indicates that Zarafa has been loaded.
	 * (See {@link #onReady}).
	 * @property
	 * @type Boolean
	 */
	isReady : false,

	/**
	 * Registration object for {@link #onReady} onto which all event
	 * handlers are being registered which want to be notified when
	 * Zarafa has been intialized and ready for plugin interaction.
	 *
	 * @property
	 * @type Ext.util.Event
	 * @private
	 */
	readyEvent : new Ext.util.Event(),
	
	/**
	 * The time that the user has not done any action 
	 * (like mousemove, click, or keypress) in the WebApp.
	 * 
	 * @property
	 * @type Integer
	 * @private
	 */
	idleTime : 0,

	/**
	 * Adds a listener to be notified when Zarafa is ready. This will be somewhere during {@link Ext.onReady}, when
	 * Zarafa has initialized the bare essentials. When the event is fired, the {@link Zarafa.core.Container} will
	 * be available, and plugins are allowed to register.
	 *
	 * @param {Function} fn The method the event invokes.
	 * @param {Object} scope (optional) The scope (<code>this</code> reference) in which the handler function executes. Defaults to the browser window.
	 * @param {Boolean} options (optional) Options object as passed to {@link Ext.Element#addListener}. It is recommended that the options
	 * <code>{single: true}</code> be used so that the handler is removed on first invocation.
	 */
	onReady : function(fn, scope, options)
	{
		this.readyEvent.addListener(fn, scope, options);

		// If the environment is already ready, can
		// should call fireReady again to fire the
		// just registered event.
		if (this.isReady) {
			this.fireReady();
		}
	},

	/**
	 * Called when {@link Ext.onReady} has been invoked, and Zarafa has been initialized.
	 * All handlers registered through {@link #onReady} will now be fired and {@link #isReady}
	 * will be set.
	 *
	 * @private
	 */
	fireReady : function()
	{
		this.isReady = true;
		this.readyEvent.fire();
		this.readyEvent.clearListeners();
	},

	/**
	 * Initialize all Global variables as used by the WebApp.
	 *
	 * This will utilize some global objects as received by the PHP
	 * side, and apply them into the proper classes, after which the
	 * global objects will be destroyed.
	 *
	 * This will instantiate {@link Zarafa.core.Container container}.
	 * @private
	 */
	initializeGlobals : function()
	{
		// Use native json handling of browser for performance benefit
		Ext.USE_NATIVE_JSON = true;

		//show confirm dialog before user leave the page.
		Zarafa.core.Util.enableLeaveRequester();

		// When the browser is unloading, all active requests will be aborted.
		window.onunload = function () {
			container.getRequest().paralyze(Zarafa.core.data.ParalyzeReason.BROWSER_RELOADING);
		};

		// Create global container object
		container = new Zarafa.core.Container();

		// Load all settings
		container.getSettingsModel().initialize(settings);
		delete settings;

		// Set the server object
		container.setServerConfig(serverconfig);
		delete serverconfig;

		// Set the user object
		container.setUser(user);
		delete user;
		
		// Set the version object
		container.setVersion(version);
		delete version;

		// Set the language object
		container.setLanguages(languages);
		delete languages;
	},

	/**
	 * Initialize the ExtJs/WebApp environment, register generic event listeners,
	 * This will listen to the 'contextmenu' event on the {@link Ext#getBody body element}
	 * as well as the exception events on the {@link Zarafa.core.data.IPMStoreMgr} and
	 * {@link Zarafa.core.ResponseRouter}.
	 * @private
	 */
	initializeEnvironment : function()
	{
		// Register the State provider which uses the SettingsModel.
		Ext.state.Manager.setProvider(new Zarafa.core.data.SettingsStateProvider());

		// Disable contextmenu globaly
		Ext.getBody().on('contextmenu', this.onBodyContextMenu, this);

		// Disable default file drop behavior
		Ext.EventManager.on(window, 'dragover', this.onWindowDragDrop, this);
		Ext.EventManager.on(window, 'drop', this.onWindowDragDrop, this);

		// Add main event handlers to listen for errors
		container.getRequest().on('connectionparalyzed', this.onConnectionParalyze, this);
		container.getRequest().on('connectioninterrupted', this.onConnectionLoss, this);
		container.getRequest().on('connectionrestored', this.onConnectionRestore, this);
		container.getResponseRouter().on('receiveexception', this.onReceiveException, this);
		// We listen on the Ext.data.DataProxy object to listen in on all exception events
		Ext.data.DataProxy.on('exception', this.onException, this);

		// Enable tooltips
		Ext.QuickTips.init();
	},

	/**
	 * Event handler which is fired when the {@link Ext#getBody &lt;body&gt;} elements fires
	 * the 'contextmenu' event. If the element which fired the event doesn't have the
	 * 'zarafa-contextmenu-enabled' class then the Browser contextmenu will be disabled.
	 * @param {Ext.EventObject} event The event object
	 * @param {Ext.Element} el The element on which the contextmenu was requested
	 * @private
	 */
	onBodyContextMenu : function(event, el)
	{
		// Disable contextmenu globally, only when the 'zarafa-contextmenu-enabled'
		// CSS class is applied on the element will we allow the contextmenu to be shown.
		if (!Ext.get(el).hasClass('zarafa-contextmenu-enabled')) {
			event.preventDefault();
		}
	},

	/**
	 * Event handler which is fired when the 'window' element fires the 'dragover' or
	 * the 'drop' event. This happens when the user drops a file over the webpage. On
	 * some UI fields this will provide a special action, but the browsers default is
	 * to open the file in the current page, which is not what we want.
	 * @param {Ext.EventObject} event The event object
	 * @private
	 */
	onWindowDragDrop : function(event)
	{
		event.stopPropagation();
		event.preventDefault();
		return false;
	},

	/**
	 * Event handler called when the {@link Ext.data.DataProxy} fired the
	 * {@link Ext.data.DataProxy#storeexception storeexception} event.
	 * This will check what type of exception it was ('response' or 'remote') and
	 * handle the exception accordingly.
	 *
	 * @param {Misc} misc See {@link Ext.data.DataProxy}#{@link Ext.data.DataProxy#exception exception}
	 * for description.
	 * @private
	 */
	onException : function(proxy, type, action, options, response, args)
	{
		var message;

		if (type === 'response') {
			// The error message can be in args when it is an Error object. This happens when the
			// processing of the response throws an Javascript Exception.
			if (Ext.isDefined(args) && args.error instanceof Error) {
				message = args.error.toString();
			} else {
				// When the exception has to do with the response itself we delegate this behavior to
				// onReceiveException
				this.onReceiveException(options, response);
				return;
			}
		} else if (response && response.error) {
			switch (response.error.type) {
				case Zarafa.core.ErrorType['MAPI']:
				case Zarafa.core.ErrorType['ZARAFA']:
				case Zarafa.core.ErrorType['GENERAL']:
					message = response.error.info.display_message;
					break;
				default:
					message = _('The server reported an unknown error on your request.');
					break;
			}
		} else {
			message = _('The server reported an unspecified error on your request.');
		}

		if (Ext.get('loading')) {
			this.setErrorLoadingMask(_('Error'), message);
		} else {
			container.getNotifier().notify('error.proxy', _('Error'), message);
		}
	},

	/**
	 * Called when the connection is being paralyzed and no further requests can be made
	 * to the server. Check if we should show a notification to the user about this, and
	 * ask if the user wishes to return to the logon page.
	 * @param {Zarafa.core.Request} request The request object
	 * @param {Zarafa.core.data.ParalyzeReason} reason The reason to paralyze the WebApp
	 * @private
	 */
	onConnectionParalyze : function(request, reason)
	{
		var message = '';
		var logoutFn = Ext.emptyFn;

		switch (reason) {
			case Zarafa.core.data.ParalyzeReason.BROWSER_RELOADING:
			default:
				// No message for the user needed.
				return;
			case Zarafa.core.data.ParalyzeReason.SESSION_EXPIRED:
				message = _('The session has expired, reauthentication is required.');
				// When logging out, we preserve the username for convenience.
				logoutFn = container.logout.createDelegate(container, [ true ], false);
				break;
			case Zarafa.core.data.ParalyzeReason.SESSION_INVALID:
				message = _('The session in the current browser window has been closed from another browser window or tab.');
				// When logging out, we preserve the username for convenience,
				// but we won't close the session which was created in the other tab.
				logoutFn = container.logout.createDelegate(container, [ true, true ], false);
				break;
		}

		if (Ext.get('loading')) {
			this.setErrorLoadingMask(_('Error'), message);
		} else {
			Ext.MessageBox.show({
				title: _('Zarafa WebApp'),
				msg : message + '<br>' +  _('Do you wish to be redirected to the logon page?'),
				icon : Ext.MessageBox.ERROR,
				buttons : Ext.MessageBox.YESNO,
				fn : this.onConnectionParalyzeConfirmation,
				scope: this,
				logoutFn : logoutFn
			});
		}
	},

	/**
	 * Event handler for the {@link Ext.MessageBox MessageBox} which was opened by {@link #onConnectionParalyze}.
	 * This determines what the user has pressed, and if the user wishes to {@link Zarafa.core.Container#logout}
	 * immediately or not. If not, then a {@link Zarafa.core.ui.notifier.Notifier notification} will be shown
	 * to remind the user about the session.
	 * @param {String} button The button which was pressed by the user
	 * @param {String} id The id of the button which was clicked
	 * @param {Object} opt The options which was used to create the MessageBox.
	 * @private
	 */
	onConnectionParalyzeConfirmation : function(button, value, opt)
	{
		if (button === 'yes') {
			opt.logoutFn.call(this);
		} else {
			container.getNotifier().notify('error.connection', _('Session expired'), _('Reauthentication required, click here to go to back to logon page.'), {
				persistent : true,
				listeners : {
					click : opt.logoutFn
				}
			});
		}
	},

	/**
	 * Periodic function which is used to update the {@link Zarafa.core.ui.notification.Notifier notification}
	 * on the screen with the message that there is a connection problem, and the requests will be retried
	 * after the given timeout. This timeout counter will be updated every second.
	 * @param Zarafa.core.PingService} service The ping service handling the connection loss
	 * @param {Object} object Containing the information related to the connection loss
	 * @param {Number} timeout The number of seconds until the next retry to connect to the server
	 * @private
	 */
	onConnectionTimeupdate : function(service, object, timeout)
	{
		var request = container.getRequest();

		// Since we use defers, it is possible that the
		// connection was already restored. In that case,
		// we don't need to update the Notification message.
		if (!request.isInterrupted() || request.isParalyzed()) {
			return;
		}

		// Create the notification, we store the reference in connEl,
		// if it already exists, this will be an update action, otherwise
		// a new notification will be created.
		this.connEl = container.getNotifier().notify('error.connection', _('Connection problem'),
								 String.format(_('Could not connect to server, retrying in {0} second(s)'), timeout / 1000) + '<br />' + _('Click to retry now'), {
			persistent : true,
			update : !!this.connEl,
			reference : this.connEl,
			listeners : {
				// If the user clicks on the notification,
				// immediately retry to connecto to the server.
				click : service.retry,
				scope: service
			}
		});

		// Defer the function by 1 second, so we can update the retry counter.
		if (timeout > 1000) {
			this.connElTask.delay(1000, this.onConnectionTimeupdate, this, [ service, object, timeout - 1000 ]);
		}
	},

	/**
	 * Event handler for the {@link Zarafa.core.PingService#retry} event. This will
	 * cancel the currently scheduled call to {@link #onConnectionTimeupdate} and
	 * invoke it manually with the updated information.
	 * @param Zarafa.core.PingService} service The ping service handling the connection loss
	 * @param {Object} object Containing the information related to the connection loss
	 * @param {Number} timeout The number of seconds until the next retry to connect to the server
	 * @private
	 */
	onConnectionRetry : function(service, object, timeout)
	{
		// In case there was still a pending
		// update task, we interrupt that one.
		this.connElTask.cancel();

		// No we can update the notification
		this.onConnectionTimeupdate(service, object, timeout);
	},

	/**
	 * Event handler for the {@link Zarafa.core.Request#connectionloss} event. This will register the
	 * event handlers to the provided {@link Zarafa.core.PingService} which will give us the information
	 * about the next reconnect attempt.
	 * @param {Zarafa.core.Request} request The request object
	 * @param {Zarafa.core.PingService} service The ping service which is going to handle the connection loss
	 * @private
	 */
	onConnectionLoss : function(request, service)
	{
		this.connElTask = new Ext.util.DelayedTask(this.onConnectionTimeupdate, this);
		service.on('retry', this.onConnectionRetry, this);
	},

	/**
	 * Event handler for the {@link Zarafa.core.Request#connectionrestore} event. This will remove
	 * the error.connection {@link Zarafa.core.ui.notification.Notifier notification}
	 * and will show a info.connection.restore {@link Zarafa.core.ui.notification.Notifier notification}.
	 * @param {Zarafa.core.Request} request The request object
	 * @private
	 */
	onConnectionRestore : function(request)
	{
		if (this.connElTask) {
			this.connElTask.cancel();
			delete this.connElTask;
		}

		if (this.connEl) {
			container.getNotifier().notify('error.connection', null, null, {
				destroy : true,
				reference : this.connEl
			});
			container.getNotifier().notify('info.connection.restore', _('Connection restored'), _('Connection with server has been restored'));
			delete this.connEl;
		}
	},

	/**
	 * Event handler called when the PHP server returned an error
	 * in the root of the response. This indicates that the communication
	 * with the PHP server failed and the user should login again.
	 *
	 * @param {Object} requestdata The request data which was send to the server.
	 * @param {Object} xmlHttpRequest The raw browser response objec
	 * @private
	 */
	onReceiveException : function(requestdata, xmlHttpRequest)
	{
		var loading = Ext.get('loading');
		var errorTitle;
		var errorMsg;

		if (xmlHttpRequest.status !== 200) {
			// # TRANSLATORS: Example: HTTP 404
			errorTitle = String.format(_('HTTP {0}'), xmlHttpRequest.status);
			errorMsg = xmlHttpRequest.statusText;
		} else {
			errorTitle = _('Error');
			errorMsg = _('Invalid data received from the server');
		}

		if (loading) {
			this.setErrorLoadingMask(errorTitle, errorMsg);
		} else {
			container.getNotifier().notify('error.json', errorTitle, errorMsg);
		}
	},

	/**
	 * Hide the loading mask which is shown before the {@link Ext.Viewport} is being rendered.
	 * The loadmask is marked with the element classname 'loading' and the background 'loading-mask'.
	 * @private
	 */
	hideLoadingMask : function()
	{
		var loadingMask = Ext.get('loading-mask');
		var loading = Ext.get('loading');

		if (loadingMask && loading) {
			//  Hide loading message
			loading.fadeOut({
				duration: 0.2,
				remove: true
			});

			// Hide loading mask
			loadingMask.setOpacity(0.9);
			loadingMask.shift({
				xy: loading.getXY(),
				width: loading.getWidth(),
				height: loading.getHeight(),
				remove: true,
				duration: 2,
				opacity: 0.1,
				easing: 'bounceOut'
			});
		}
	},

	/**
	 * Set an error text in the loading screen.
	 * @param {String} newError The title for the loading screen
	 * @param {String} newMessage The message for the loading screen
	 * @private
	 */
	setErrorLoadingMask : function(newTitle, newMessage)
	{
		var template = new Ext.Template('<div><b>{title}</b><br />{msg}</div>', { compiled : true, disableFormats : true });
		var message = Ext.get('loading-message');
		if (message) {
			message.dom.className = 'loading-error';
			template.overwrite(message, { title: newTitle, msg: newMessage });
		}
	},

	/**
	 * Validate the {@link Zarafa.hierarchy.data.HierarchyStore HierarchyStore} to determine
	 * if the {@link Zarafa.hierarchy.data.HierarchyStore#getDefaultStore Default Store} is present
	 * along with all the {@link Zarafa.hierarchy.data.HierarchyStore#getDefaultFolder Default Folders}.
	 * If there is a problem, this will show a {@link Zarafa.core.ui.notifier.Notifier Notification}
	 * indicating which store or folders are missing, and warning the user that not all functionality
	 * might be working as expected.
	 * @param {Zarafa.hierarchy.data.HierarchyStore} store The Hierarchy Store to validate
	 * @private
	 */
	validateHierarchy : function(store)
	{
		if (!store.getDefaultStore()) {
			container.getNotifier().notify('error.hierarchy.defaultfolder',
				_('Missing store'),
				_('The default store is missing from the hierarchy.') +
					'<br>' +
					_('Not all functionality of WebApp might be working properly because of this.'),
				{
					persistent : true,
					listeners : {
						'click' : this.onHierarchyNotifierClick,
						'scope': this
					}
				}
			);
			return;
		}

		// The following default folders are required to be present
		// to be able to properly work with the WebApp.
		var defaultFolders = [{
			type : 'inbox',
			name : pgettext('hierarchy.foldername', 'Inbox')
		},{
			type : 'outbox',
			name : pgettext('hierarchy.foldername', 'Outbox')
		},{
			type : 'sent',
			name : pgettext('hierarchy.foldername', 'Sent Items')
		},{
			type : 'wastebasket',
			name : pgettext('hierarchy.foldername', 'Deleted items')
		},{
			type : 'calendar',
			name : pgettext('hierarchy.foldername', 'Calendar')
		},{
			type : 'contact',
			name : pgettext('hierarchy.foldername', 'Contacts')
		},{
			type : 'drafts',
			name : pgettext('hierarchy.foldername', 'Drafts')
		},{
			type : 'journal',
			name : pgettext('hierarchy.foldername', 'Journal')
		},{
			type : 'note',
			name : pgettext('hierarchy.foldername', 'Notes')
		},{
			type : 'task',
			name : pgettext('hierarchy.foldername', 'Tasks')
		},{
			type : 'junk',
			name : pgettext('hierarchy.foldername', 'Junk E-mail')
		}];

		var missing = [];

		// Go over all default folders which we expect to be present,
		// if any of them is missing we update the 'missing' array
		// and will show them in a notification box.
		for (var i = 0, len = defaultFolders.length; i < len; i++) {
			var folder = defaultFolders[i];

			if (!store.getDefaultFolder(folder.type)) {
				missing.push(folder.name);
			}
		}

		// If any of the folders is missing, show a notification to the
		// user to inform him of the missing folders.
		if (!Ext.isEmpty(missing)) {
			// Construct an HTML list of the missing folders
			var list = '<ul><li>' + missing.join('</li><li>') + '</li></ul>';

			container.getNotifier().notify('error.hierarchy.defaultfolder',
				_('Missing folders'),
				String.format(
					ngettext('The following required folder is missing in the hierarchy: {0}',
						 'The following required folders are missing in the hierarchy: {0}', missing.length), list) +
					_('Not all functionality of WebApp might be working properly because of this.'),
				{
					persistent : true,
					listeners : {
						'click' : this.onHierarchyNotifierClick,
						'scope': this
					}
				}
			);
		}
	},

	/**
	 * Event handler which is fired when the user clicked on the {@link Zarafa.core.ui.notifier.Notifier notification}.
	 * This will remove the notification.
	 * @param {Ext.Element} element The notification element
	 * @param {Ext.EventObject} event The event object
	 * @private
	 */
	onHierarchyNotifierClick : function(element, event)
	{
		container.getNotifier().notify('error.hierarchy.defaultfolder', null, null, {
			reference : element,
			destroy : true
		});
	},

	/**
	 * Event handler called when load is received in hierarchy. This will {@link #hideLoadingMask hide the loadmask}.
	 * @param {Zarafa.hierarchy.data.HierarchyStore} store The store which was loaded
	 * @param {Ext.data.Record|Array} records The records which were loaded by the store
	 * @param {Object} options The options which were originally passed to {@link Ext.data.Store#load}.
	 * @private
	 */
	onHierarchyLoad : function(store, records, options)
	{
		if (!Ext.isEmpty(records)) {
			// We have the hierarchy, load the entire UI
			container.getMainPanel();

			// validate the hierarchy, if the hierarchy is not valid a warning
			// notification box will be shown to the user informing him of incompatibilities.
			this.validateHierarchy(store);

			// Remove loading mask, we might still be busy loading the data for the
			// store of the user, but the context will have its own loadmask for that.
			// The user is at least allowed to see the Hierarchy and press buttons.
			this.hideLoadingMask();

			// Register webapp to handle mailto urls
			this.registerMailto();

			// Process data that was passed as URL data
			Zarafa.core.URLActionMgr.execute(urlActionData);
			delete urlActionData;

			// Start the keepalive to make sure we stay logged into the zarafa-server,
			// the keepalive is also used to get notifications back to the client.
			store.startKeepAlive();

			// Post logon succesful to parent.
			if (window.location !== window.parent.location){ 
				window.parent.postMessage('logonSuccesful', '*');
			}
		} else {
			this.setErrorLoadingMask(_('Error'), _('Loading model from server failed'));
		}
	},

	/**
	 * Function will register webapp as default client to handle mailto urls
	 * Ideally we should only register this handler if already not registered
	 * but browsers support is not proper for that, so we are always registering it
	 * in chrome this is not a problem, in firefox it will show bar that handler is already registered
	 * but that is a bug and already fixed in trunk version https://bugzilla.mozilla.org/show_bug.cgi?id=912347
	 * Support for isProtocolHandlerRegistered is also limited https://bugzilla.mozilla.org/show_bug.cgi?id=440620
	 */
	registerMailto : function()
	{
		var navigator = window.navigator;

		if (Ext.isFunction(navigator.registerProtocolHandler)) {
			// Register webapp handler for 'mailto' protocol in browsers.
			var url = container.getBaseURL() + '?action=mailto&to=%s';

			// Check if we have already registered for this protocol
			var register = true;
			if (Ext.isFunction(navigator.isProtocolHandlerRegistered)) {
				register = !navigator.isProtocolHandlerRegistered('mailto', url);
			}

			// Register if required
			if (register) {
				navigator.registerProtocolHandler('mailto', url, 'Zarafa WebApp');
			}
		}
	},

	/**
	 * Load the default Webclient into the browser. This will initialize the environment,
	 * load the {@link Zarafa.hierarchy.data.HierarchyStore} and open the
	 * {@link Zarafa.core.ui.MainViewport MainViewport}.
	 */
	loadWebclient : function()
	{
		// Initialize globals & environment
		Zarafa.initializeGlobals();
		Zarafa.initializeEnvironment();

		// Start loading all plugins
		Zarafa.fireReady();

		// Check if user is out of office and ask them if they want to switch it off
		this.checkOof();

		// Initialize context - check if there is one selected in settings
		this.initContext();

		// Start polling the server to get reminders data back to client
		container.getReminderStore().initializeReminderInterval();

		// Setup the hierarchy, this will complete the initialization
		// and allow the Main Viewport to be opened
		var hierarchyStore = container.getHierarchyStore();

		// For the hierarchy we only need to register the event handler once,
		// as only during the first time we have to remove the load mask.
		hierarchyStore.on('load', this.onHierarchyLoad, this, { single: true });

		// Load the folder hierarchy
		hierarchyStore.load();

		// When a client timeout has been defined, we will start keeping track
		// of idle time.
		var server = container.getServerConfig();
		var clientTimeout = server.getClientTimeout();
		if (clientTimeout){
			this.startIdleTimeChecker(clientTimeout);
		}
	},
	
	/**
	 * Starts the checking of idle time.
	 * This function uses original javascript events because we cannot set 
	 * the useCapture property with ExtJS events.
	 * See https://developer.mozilla.org/en/docs/Web/API/EventTarget.addEventListener
	 * 
	 * @param {Number} clientTimeout The timout time in seconds.
	 * @private
	 */
	startIdleTimeChecker : function(clientTimeout)
	{
		if ( !document.addEventListener ) {
			// User is using a browser that does not support addEventListener.
			// Probably IE<9 which we don't support.
			// However there is no reason to create errors for IE<9
			// Client timeout will still be handled by the backend though,
			// but the message will only be shown to the user when he tries to
			// connect to the backend after the session has timed out. 
			return;
		}
		var me = this;

		document.addEventListener('click', function(){
			me.idleTime = 0;
		}, true);
		document.addEventListener('mousemove', function(){
			me.idleTime = 0;
		}, true);
		document.addEventListener('keydown', function(){
			me.idleTime = 0;
		}, true);
		
		var hierarchyStore = container.getHierarchyStore();

		// Start an interval for increasing the idle time
		Ext.TaskMgr.start.createDelegate(this, [{
			run : function(){
				// Add 5 seconds to the idle time counter
				this.idleTime += 5;
				if ( this.idleTime > clientTimeout ){
					hierarchyStore.sendDestroySession();
				}
			},
			scope : this,
			interval : 5000 //Run every 5 seconds
		}]).defer(5000); // Start after 5 seconds
		
		// Start an interval for sending keepalive requests
		// We need this keepalive to keep the connection alive if the user has made an
		// action in the WebApp without connecting to the server. (like mousemove, click, keydown)
		// Substracting 5 seconds to account for latency
		var interval = (clientTimeout-5)*1000;
		if ( interval < 5000 ){
			// This one is especially for Sean who was so smart to set a client timeout of 5 seconds
			// causing keepalive requests to be sent constantly and thereby ddos'ing his own server :-)
			// Let's never send keepalive requests with an interval lower than 5 seconds.
			// Anyone who sets a timeout this low deserves to be logged out! (and punished severly)
			interval = 5000;
		}
		
		Ext.TaskMgr.start.createDelegate(this, [{
			run : function(){
				hierarchyStore.sendKeepAlive();
			},
			scope : this,
			interval : interval
		}]).defer(interval); //Start sending keepalives after interval milliseconds.
	},

	/**
	 * init UI by lazily constructing the main panel (implicit in container.getMainPanel) and
	 * setting the default context to visible. Note that during onHierarchyLoad we will also
	 * open the default folder for that specific context to ensure the user can see all from
	 * the folder.
	 * @private
	 */
	initContext : function()
	{
		var defaultContext = container.getSettingsModel().get('zarafa/v1/main/default_context');
		var plugin = container.getContextByName(defaultContext);
		if (!plugin) {
			// If the defaultContext has an invalid name,
			// we will default to the 'today' context
			plugin = container.getContextByName('today');
		}
		container.switchContext(plugin);
	},

	/**
	 * Check if user is out of office
	 * If so, ask them if they want to switch OOF off
	 * @private
	 */
	checkOof : function()
	{
		if (container.getSettingsModel().get('zarafa/v1/contexts/mail/outofoffice/set') === true) {
			Ext.MessageBox.confirm(
				_('Zarafa WebApp'),
				_('Out of Office currently on. Would you like to turn it off?'),
				this.onOofConfirm,
				this);
			return;
		}
	},

	/**
	 * Handler for the out of office confirmation box
	 * If the user pressed Yes/Ok, then disable out of office
	 * @param {String} id of the button that was clicked
	 * @private
	 */
	onOofConfirm : function(button)
	{
		if (button === 'yes') {
			container.getSettingsModel().set('zarafa/v1/contexts/mail/outofoffice/set', false);
			container.getNotifier().notify('info.saved', _('Out of office off'), _('Out of office has been turned off'));
		}
	},

	/**
	 * Load the Welcome message for new users into the browser. This will initialize
	 * the environment and open the {@link Zarafa.core.ui.WelcomeViewport WelcomeViewport}.
	 */
	loadWelcome : function()
	{
		// Setup globals & environment
		this.initializeGlobals();
		this.initializeEnvironment();

		// Start loading all plugins
		this.fireReady();

		// Load the welcome view
		container.getWelcomePanel();
		this.hideLoadingMask();
		// Post logon succesful to parent.
		if (window.location !== window.parent.location) {
			window.parent.postMessage('firstLogon', '*');
		}
	},

	//
	// Generic Utility functions, should probably be moved elsewhere
	//

	/**
	 * Determine if the browser is a modern browser which is capable of working with FormData instances,
	 * and has support for the Files JS API. All modern browsers have support for this, unfortunately
	 * WebApp also has to support some broken browsers as Internet Explorer 9.
	 * @return {Boolean} True if Files API is supported
	 */
	supportsFilesAPI : function()
	{
		if (!container.getSettingsModel().get('zarafa/v1/main/use_files_api')) {
			return false;
		}

		return Ext.isDefined(window.File) && Ext.isDefined(window.FileList);
	},

	/**
	 * Determine if Canvas rendering is supported. Support is based on two different factors,
	 * first the canvas rendering can be disabled using a configuration option, secondly
	 * the browser might be limited in its support for canvas.
	 *
	 * @return {Boolean} True if canvas rendering is supported
	 */
	supportsCanvas : function()
	{
		if (!container.getSettingsModel().get('zarafa/v1/main/use_canvas_rendering')) {
			return false;
		}

		// Small performance tweak, we don't expect the canvas support in a browser
		// to be changed dynamically. So we only need to check the document.createElement
		// call once.
		if (!Ext.isDefined(this.browserSupportsCanvas)) {
			this.browserSupportsCanvas = !!document.createElement('canvas').getContext;
		}

		return this.browserSupportsCanvas;
	},

	/**
	 * This will resize a canvas {@link Ext.Element element}.
	 * Canvas resizing is more tricky then one would expect. For canvas 2 settings are
	 * important: the CSS and attribute dimensions.
	 *
	 * As one would expect, the CSS dimensions, given through a CSS file, or
	 * 'style' attribute (controlled by the functions {@link Ext.Element#setWidth setWidth}
	 * and {@link Ext.Element#setHeight setHeight}) controls how big the element itself
	 * is. It however does not mean that the drawing area is of the same size. The drawing
	 * area is controlled by the element attributes 'width' and 'height'.
	 *
	 * Now for the fun part, if you use CSS to size of the element to 1000x1000px,
	 * thus the element looks like:
	 *   <canvas style="width=1000px; height=1000px;">
	 * and set the attributes to 50x50px, making the complete element look like:
	 *   <canvas style="width=1000px; height=1000px;" width="50" height="50">
	 * you can then draw anything you like on the canvas, but only the first
	 * 50x50px of the canvas will be resized to the entire element size. Making
	 * your cute little drawing of a giraffe completely stretched.
	 *
	 * @param {Ext.Element} canvas The canvas element which must be resized.
	 * @param {Number} width The desired width of the canvas element
	 * @param {Number} height The desired height of the canvas element
	 */
	resizeCanvas : function(canvas, width, height)
	{
		canvas.setWidth(canvas.dom.width = width);
		canvas.setHeight(canvas.dom.height = height);
	},

	/**
	 * Generate a random string.
	 * @param {Number} len Length of the string
	 * @return {String} Random string
	 */
	generateId : function(len)
	{
		var text = "";
		var possible = "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789";

		for( var i = 0; i < len; i++ ) {
			text += possible.charAt(Math.floor(Math.random() * possible.length));
		}

		return text;
	}
});
Ext.namespace('Zarafa.core');

/**
 * @class Zarafa.core.Actions
 * List of valid action types. Action types are used for identifying request and response types for modules.
 * For instance, to request a mail from the server the client performs an 'open' action on the 'previewreadmailitemmodule'.
 * The server then responds with an 'item' action, containing the email data.
 * @singleton
 */
Zarafa.core.Actions = 
{
	/**
	 * The list action retrieves a list of items such as mails, tasks, etc. The server then responds with a list action containing
	 * a list of items. 
	 * @property
	 * @type String
	 */
	list : "list",

	/**
	 * Retrieves a list of entries in the global address book  
	 * @property
	 * @type String
	 */
	globaladdressbook : "globaladdressbook",

	/**
	 * Retrieves the hierarchy, a set of stores and folders within those stores. Used for updating the
	 * hierarchy tree.  
	 * @property
	 * @type String
	 */
	hierarchy : "hierarchy",

	/**
	 * Open an item, usually a mail item. The server responds with an item action.
	 * @property
	 * @type String
	 */
	open : 'open',

	/**
	 * Can mean different things in different contexts. Can be used to request a single item from the server, and is returned by the
	 * server to return the contents on a single item.  
	 * @property
	 * @type String
	 */
	item : 'item',

	/**
	 * Update item(s)
	 * @property
	 * @type String
	 */
	update : 'update',

	/**
	 * Save an item. 
	 * @property
	 * @type String
	 */
	save : 'save',

	/**
	 * Copy an item. 
	 * @property
	 * @type String
	 */
	copy : 'copy',

	/**
	 * Delete an item. 
	 * @property
	 * @type String
	 */
	'delete' : 'delete',

	/**
	 * Gets folder details. 
	 * @property
	 * @type String
	 */
	folder : 'folder',

	/**
	 * Used for setting properties. 
	 * @property
	 * @type String
	 */
	set : 'set',

	/**
	 * Used for getting properties. 
	 * @property
	 * @type String
	 */
	get : 'get',

	/**
	 * Used for reset properties. 
	 * @property
	 * @type String
	 */
	reset : 'reset',

	/**
	 * Used for deleting properties/items. 
	 * @property
	 * @type String
	 */
	_delete : 'delete',

	/**
	 * Used for searching on a folder.
	 * @property
	 * @type String
	 */
	search : 'search',

	/**
	 * Used for incremental search on folder.
	 * @property
	 * @type String
	 */
	updatesearch : 'updatesearch',
	
	/**
	 * Used for live scroll.
	 * @property
	 * @type String
	 */
	updatelist : 'updatelist',

	/**
	 * Used for stopping search on folder.
	 * @property
	 * @type String
	 */
	stopsearch : 'stopsearch',

	/**
	 * Used for requesting contacts from addressbook
	 * @property
	 * @type String
	 */		
	contacts : 'contacts',

	/**
	 * Used to send a keepalive to the server
	 * @property
	 * @type String
	 */
	keepalive: 'keepalive',

	/**
	 * Used to send a request to destroy the session to the server
	 * @property
	 * @type String
	 */
	destroysession: 'destroysession',

	/**
	 * Used when receiving update from server indicating there is new mail
	 * @property
	 * @type String
	 */
	newmail: 'newmail',

	/**
	 * Used for creating new folder
	 * @property
	 * @type String
	 */
	addFolder: 'add',

	/**
	 * Used for renaming folder in tree
	 * @property
	 * @type String
	 */
	modifyFolder: 'modify',

	/**
	 * Used for deleteing folder from tree
	 * @property
	 * @type String
	 */
	deleteFolder: 'delete',

	/**
	 * Used on Deleted Items to empty the folder
	 * @property
	 * @type String
	 */
	emptyFolder: 'emptyfolder',

	/**
	 * Used on folders to mark all messages as 'read'
	 * @property
	 * @type String
	 */
	readAllMsgs: 'readflags',
	/**
	 * Used in {@link Zarafa.hierarchy.dialogs.FolderPropertiesContentPanel FolderPropertiesContentPanel} show/update folder props
	 * @property
	 * @type String
	 */
	folderProps: 'folderprops',
	/**
	 * Used in {@link Zarafa.core.data.IPMRecipientStoreCheckNamesProxy IPMRecipientStoreCheckNamesProxy} for resolve requests
	 * @property
	 * @type String
	 */
	checknames: 'checknames',

	/**
	 * Used in {@link Zarafa.core.data.IPMExpandDistlistProxy IPMExpandDistlistProxy} for expand requests
	 * @property
	 * @type String
	 */
	expand: 'expand',

	/**
	 * Used in {@link Zarafa.core.data.IPMAttachmentProxy IPMAttachmentProxy} for uploading attachments.
	 * @property
	 * @type String
	 */
	upload : 'upload'
};
Ext.namespace('Zarafa.core');

/**
 * @class Zarafa.core.ColorSchemes
 * @singleton
 * 
 * An object that can be used to handle color schemes.
 * Color schemes are used by {@link Zarafa.core.Context contexts}
 * that can display different folders in one view. Currently only
 * the {@link Zarafa.calendar.CalendarContext calendar context}
 * is such a context.
 * 
 * It has methods to create color schemes based on a single color
 * and to add complete color schemes.
 */
Zarafa.core.ColorSchemes = {
	
	/**
	 * An array with the fields that represent a color available in 
	 * a color scheme. The fields have a name that can be used a key
	 * in a color scheme, and a weight (percentage) that will be used
	 * to create these color based on the base color of a color scheme.
	 * Fields can be added using the function {@link #addField}. When
	 * a field is added, all existing color schemes will also get the
	 * new field.
	 * @property
	 * @type Object[]
	 * @private
	 */
	fields : [
		{
			name : 'base',
			weight : 1
		}
	],
	
	/**
	 * The list of color schemes. Contexts can add color schemes
	 * using the function {@link #createColorScheme} or {@link #addColorScheme}.
	 * @property
	 * @type Object[]
	 * @private
	 */
	colorSchemes : [],
	
	/**
	 * Adds a field to the {@link #fields colorScheme fields}
	 * @param {Object|Array} field An object with properties name (mandatory) and 
	 * weight (the weight of the color as ratio of the base color of the color
	 * scheme) or color (an RGB hex value that will be used as color for this field
	 * (e.g. '#0067AC')), or an array with field objects.
	 */
	addField : function(field)
	{
		var i;
		
		if ( Ext.isArray(field) ){
			for ( i=0; i<field.length; i++ ){
				this.addField(field[i]);
			}
			return;
		}
		
		this.fields.push(field);
		
		// Add the color to existing color schemes
		for ( i =0; i<this.colorSchemes.length; i++ ){
			if ( !Ext.isDefined(this.colorSchemes[i][field.name]) ){
				if ( Ext.isDefined(field.color) ){
					this.colorSchemes[i][field.name] = field.color;
				}else if ( Ext.isDefined(field.weight) ){
					this.colorSchemes[i][field.name] = this.createColor(this.colorSchemes[i].base, field.weight);
				}
			}
		}
	},
	
	/**
	 * Converts a hexadecimal RGB color value into an object with
	 * red, green, and blue fields
	 * @param {String} hexColor A hexadecimal RGB color value
	 * (e.g. '#0067AC for Zarafa Blue)
	 * @return {Object} An object with decimal red, green, and blue values
	 * @private
	 */
	hexToRgb : function(hexColor)
	{
		return {
			red: parseInt(hexColor.substring(1,3), 16),
			green: parseInt(hexColor.substring(3,5), 16),
			blue: parseInt(hexColor.substring(5,7), 16)
		};
	},
	
	/**
	 * Converts an RGB object into a hexadecimal RGB color value
	 * @param {Object} rgbObj An decimal RGB color object
	 * @return {String} A hexadecimal RGB color value
	 */
	rgbToHex : function(rgbObj)
	{
		// function that will convert a number to a hexadecimal string (between 0 and 255) 
		function _toHex(n) {
			n = parseInt(n,10);
			if (isNaN(n)) return "00";
			n = Math.max(0,Math.min(n,255));
			return "0123456789ABCDEF".charAt((n-n%16)/16) + "0123456789ABCDEF".charAt(n%16);
		}
		
		return '#' + _toHex(rgbObj.red)+_toHex(rgbObj.green)+_toHex(rgbObj.blue);
	},
	
	/**
	 * Creates a color for a color scheme based on the baseColor of
	 * that scheme and a weight factor.
	 * @param {String} baseColor A hexadecimal RGB color value
	 * @param {Number} colorWeight A value that defines the new color
	 * as related to the baseColor. Should be between 0 and 1.
	 * @return {String} A hexadecimal RGB color value
	 * @private
	 */
	createDarkColor : function(baseColor, colorWeight)
	{
		var rgbBaseColor = this.hexToRgb(baseColor);
		
		var rgbColor = {
			red : rgbBaseColor.red * colorWeight,
			green : rgbBaseColor.green * colorWeight,
			blue : rgbBaseColor.blue * colorWeight
		};
		
		return this.rgbToHex(rgbColor);
	},
	
	/**
	 * Creates a color for a color scheme based on the baseColor of
	 * that scheme and a weight factor.
	 * @param {String} baseColor A hexadecimal RGB color value
	 * @param {Number} colorWeight A value that defines the new color
	 * as related to the baseColor. Should be larger than 1.
	 * @return {String} A hexadecimal RGB color value
	 * @private
	 */
	createLightColor : function(baseColor, colorWeight)
	{
		var rgbBaseColor = this.hexToRgb(baseColor);
		
		var rgbColor = {
			red : 255 - (255-rgbBaseColor.red) * (255-128*colorWeight) / 127,
			green : 255 - (255-rgbBaseColor.green) * (255-128*colorWeight) / 127,
			blue : 255 - (255-rgbBaseColor.blue) * (255-128*colorWeight) / 127
		};
		
		return this.rgbToHex(rgbColor);
	},
	
	/**
	 * Creates a color for a color scheme based on the baseColor of
	 * that scheme and a weight factor.
	 * @param {String} baseColor A hexadecimal RGB color value
	 * @param {Number} colorWeight A value that defines the new color
	 * as related to the baseColor. Between 0 and 1 for a color that is darker
	 * than the baseColor. Larger then 1 for a color that is lighter
	 * than the baseColor.
	 * @return {String} A hexadecimal RGB color value
	 * @private
	 */
	createColor : function(baseColor, colorWeight)
	{
		var rgbBaseColor = this.hexToRgb(baseColor);
		
		if ( Math.max(rgbBaseColor.red, rgbBaseColor.green, rgbBaseColor.blue) > 127 ){
			return this.createLightColor(baseColor, colorWeight);
		}else{
			return this.createDarkColor(baseColor, colorWeight);
		}
	},
	
	/**
	 * Creates a color scheme based on a single base color
	 * 
	 * @param {String} name The unique name for this color scheme
	 * (can be used to identify the color scheme)
	 * @param {String} displayName The name of the color scheme that
	 * will be used if a name for the color scheme must be shown 
	 * to the user.
	 * @param {String} baseColor an RGB hexadecimal color value 
	 * (e.g. '#0067AC for Zarafa Blue)
	 */
	createColorScheme : function(name, displayName, baseColor)
	{
		var i;
		
		if ( Ext.isArray(name) ){
			for ( i=0; i<name.length; i++ ){
				this.createColorScheme(name[i]);
			}
			
			return;
		}
		
		if ( Ext.isObject(name) ){
			displayName = name.displayName;
			baseColor = name.baseColor || name.base;
			name = name.name;
		}
		
		var colorScheme = {
			name : name,
			displayName : displayName,
			base : baseColor
		};
		
		// Loop through all the fields and create a color for it in this scheme
		for ( i=0; i<this.fields.length; i++ ){
			if ( this.fields[i].name !== 'base' ){
				if ( Ext.isDefined(this.fields[i].color) ){
					colorScheme[this.fields[i].name] = this.fields[i].color;
				}else{
					colorScheme[this.fields[i].name] = this.createColor(baseColor, this.fields[i].weight);
				}
			}
		}
		
		if ( !Ext.isDefined(this.getColorScheme(name)) ){
			this.colorSchemes.push(colorScheme);
		}
	},
	
	/**
	 * Adds a complete color scheme to the color scheme list
	 * @param {Object} colorScheme
	 */
	addColorScheme : function(colorScheme)
	{
		// Simple check
		if ( !Ext.isDefined(colorScheme.name) || !Ext.isDefined(colorScheme.displayName) || !Ext.isDefined(colorScheme.base) ){
			// Missing necessary properties for a valid color scheme
			// So don't add the color scheme.
			return;
		}
		
		// Create colors that are not available in the passed color scheme
		for ( var i=0; i<this.fields.length; i++ ){
			if ( !Ext.isDefined(colorScheme[this.fields[i].name]) ){
				if ( Ext.isDefined(this.fields[i].color) ){
					colorScheme[this.fields[i].name] = this.fields[i].color;
				}else{
					colorScheme[this.fields[i].name] = this.createColor(colorScheme.base, this.fields[i].weight);
				}
			}
		}
		
		this.colorSchemes.push(colorScheme);
	},
	
	/**
	 * Adds the color schemes and additional color schemes that are defined 
	 * in the config.php/default.php
	 */
	addColorSchemesFromConfig : function()
	{
		var i;
		var colorSchemes = container.getServerConfig().getColorSchemes();
		if ( colorSchemes && Ext.isArray(colorSchemes) ){
			for ( i=0; i<colorSchemes.length; i++ ){
				this.addColorScheme(colorSchemes[i]);
			}
		}
		var additionalColorSchemes = container.getServerConfig().getAdditionalColorSchemes();
		if ( additionalColorSchemes && Ext.isArray(additionalColorSchemes) ){
			for ( i=0; i<additionalColorSchemes.length; i++ ){
				this.addColorScheme(additionalColorSchemes[i]);
			}
		}
	},
	
	/**
	 * Returns the color scheme with the passed name if found,
	 * or undefined otherwise
	 * @param {String} colorSchemeName The name of the color scheme
	 * @return {Object|undefined} A color scheme or undefined if not found
	 */
	getColorScheme : function(colorSchemeName)
	{
		for ( var i=0; i<this.colorSchemes.length; i++ ){
			if ( this.colorSchemes[i].name === colorSchemeName ){
				return this.colorSchemes[i];
			}
		}
		
		return undefined;
	},
	
	/**
	 * Returns the array with all defined color schemes
	 * @return {Object|Array} An array with all defined color schemes
	 */
	getColorSchemes : function()
	{
		return this.colorSchemes;
	}
	
};

Zarafa.onReady(function(){
		// Load the color schemes that are defined in the config
		Zarafa.core.ColorSchemes.addColorSchemesFromConfig();
});
Ext.namespace('Zarafa.core');

/**
 * @class Zarafa.core.DateRange
 * @extends Ext.util.Observable
 * 
 * Represents a date range defined by a start and due date. The start date is inclusive, while the due date is exclusive. For example,
 * to denote an appointment that lasts all day on July 1st, 2010 one would write this as (00:00 July 1st 2010, 00:00 July 2nd 2010).
 * In sort, the range is defined as [startDate, dueDate>.
 * <p>
 * This class encapsulates such ranges because they are used often in especially the calendering components.
 */
Zarafa.core.DateRange = Ext.extend(Ext.util.Observable, {
	/**
	 * @cfg {Date} startDate start date for this {@link Zarafa.core.DateRange DateRange}.
	 * use {@link #getStartDate} and {@link #setStartDate} to modify this value.
	 */
	startDate : null,

	/**
	 * @cfg {Date} dueDate due date for this {@link Zarafa.core.DateRange DateRange}.
	 * use {@link #getDueDate} and {@link #setDueDate} to modify this value.
	 */
	dueDate : null,

	/**
	 * @cfg {Boolean} allowBlank Specifies empty dates are accepted by this {@link Zarafa.core.DateRange DateRange},
	 * if {@link #allowBlank} is true then only we can use empty start/due dates in {@link Zarafa.core.DateRange DateRange}.
	 * otherwise {@link #startDate} and {@link #dueDate} will be initialized with current dates.
	 */
	allowBlank : false,

	/**
	 * @constructor
	 * @param {Date} startDate (optional) start date
	 * @param {Date} dueDate (optional) due date
	 */
	constructor : function(config)
	{
		config = config || {};

		Ext.applyIf(config, {
			allowBlank : this.allowBlank
		});

		if(config.allowBlank) {
			// if allowBlank is true then we should initialize start / due date with undefined
			// but this should only be done when start / due dates are not provided in the configs hence used Ext.applyIf
			// there is a restriction that start date can not exist without due date, so we will be initializing due date
			// same as start date if only start date is provided
			Ext.applyIf(config, {
				startDate : null,
				dueDate : config.startDate ? config.startDate.clone() : null
			});
		} else {
			// if allowBlank is false then we should initialize start / due date with current dates
			Ext.applyIf(config, {
				startDate : new Date(),
				dueDate : config.startDate ? config.startDate.clone() : new Date()
			});
		}

		Ext.apply(this, config);

		// Precondition
		if (Ext.isDate(this.startDate) && Ext.isDate(this.dueDate) && (this.getStartTime() > this.getDueTime())) {
			throw 'Invalid date range, start date is after due date';
		}

		this.addEvents(
			/**
			 * @event update
			 * Fires when the daterange is modified.
			 * @param {Zarafa.core.DateRange} newRange The changed daterange object
			 * @param {Zarafa.core.DateRange} oldRange The orignal daterange values (clone of the daterange object,
			 * prior of the change).
			 */
			'update'
		);

		Zarafa.core.DateRange.superclass.constructor.call(this);
	},
	
	/**
	 * @return {Date} the range's start date.
	 */
	getStartDate : function()
	{
		return this.startDate;
	},
	
	/**
	 * @return {Date} the range's due date.
	 */
	getDueDate : function()
	{
		return this.dueDate;
	},
	
	/**
	 * Sets the range start date.
	 * @param {Date} startDate the range's start date.
	 * @param {Boolean} silence When set to true it will not throw an start date after due date error.
	 * @param {Boolean} ignoreUpdate When set to true it will not fire the update event.
	 * @return {Zarafa.core.DateRange} this date range
	 */
	setStartDate : function(startDate, silence, ignoreUpdate)
	{
		var original = null;

		// make sure that you are saving null when no valid date is passed
		startDate = Ext.isDate(startDate) ? startDate : null;

		// Preconditions
		if (!this.allowBlank && !Ext.isDate(startDate)) {
			throw 'Cannot set DateRange start to undefined';
		} else if (Ext.isDate(this.dueDate) && Ext.isDate(startDate) && startDate.getTime() > this.getDueTime() && !silence) {
			throw 'Cannot set DateRange start date to after its due date';
		}

		// No update event when new value equals old value
		ignoreUpdate = ignoreUpdate || (Ext.isDate(this.startDate) && Ext.isDate(startDate) && this.getStartTime() == startDate.getTime());

		if (!ignoreUpdate)
			original = this.clone();

		this.startDate = startDate;

		if(!ignoreUpdate) {
			this.fireEvent("update", this, original);
		}

		return this;
	},
	
	/**
	 * Sets the range due date.
	 * @param {Date} dueDate the range's due date.
	 * @param {Boolean} silence When set to true it will not throw an start date after due date error.
	 * @param {Boolean} ignoreUpdate When set to true it will not fire the update event.
	 * @return {Zarafa.core.DateRange} this date range
	 */
	setDueDate : function(dueDate, silence, ignoreUpdate)
	{
		var original = null;

		// make sure that you are saving null when no valid date is passed
		dueDate = Ext.isDate(dueDate) ? dueDate : null;

		// Precondition
		if (!this.allowBlank && !Ext.isDate(dueDate)) {
			throw 'Cannot set DateRange due date to undefined';
		} else if (Ext.isDate(this.startDate) && Ext.isDate(dueDate) && dueDate.getTime() < this.getStartTime() && !silence) {
			throw 'Cannot set DateRange due date to before its start date';
		}

		// No update event when new value equals old value
		ignoreUpdate = ignoreUpdate || (Ext.isDate(this.dueDate) && Ext.isDate(dueDate) && this.getDueTime() == dueDate.getTime());

		if (!ignoreUpdate)
			original = this.clone();

		this.dueDate = dueDate;

		if(!ignoreUpdate) {
			this.fireEvent("update", this, original);
		}

		return this;
	},
	
	/**
	 * Sets the start and due dates.
	 * @param {Date} startDate the range's start date.
	 * @param {Date} dueDate the range's due date.
	 * @param {Boolean} silence When set to true it will not throw an start date after due date error.
	 * @param {Boolean} ignoreUpdate When set to true it will not fire the update event.
	 * @return {Zarafa.core.DateRange} this date range
	 */
	set : function(startDate, dueDate, silence, ignoreUpdate)
	{
		var original = null;

		// make sure that you are saving null when no valid date is passed
		startDate = Ext.isDate(startDate) ? startDate : null;
		dueDate = Ext.isDate(dueDate) ? dueDate : null;

		// Precondition
		if (!this.allowBlank && !Ext.isDate(startDate)) {
			throw 'Cannot set DateRange start to undefined';
		}

		if (!this.allowBlank && !Ext.isDate(dueDate)) {
			throw 'Cannot set DateRange due date to undefined';
		}

		if (Ext.isDate(startDate) && Ext.isDate(dueDate) && startDate.getTime() > dueDate.getTime() && !silence) {
			throw 'Invalid date range, start date is after due date';
		}

		ignoreUpdate = ignoreUpdate || (Ext.isDate(startDate) && Ext.isDate(dueDate) && (startDate.getTime() == this.getStartTime()) && (dueDate.getTime() == this.getDueTime()));

		if (!ignoreUpdate)
			original = this.clone();

		// don't fire update event from these functions
		this.setStartDate(startDate, true, true);
		this.setDueDate(dueDate, true, true);

		if (!ignoreUpdate) {
			this.fireEvent("update", this, original);
		}

		return this;
	},

	/**
	 * @return {Number} the range's start time in milliseconds since epoch, GMT.
	 */
	getStartTime : function()
	{
		return (Ext.isDate(this.startDate) && this.startDate.getTime()) || this.startDate;
	},
	
	/**
	 * @return {Number} the range's due time in milliseconds since epoch, GMT.
	 */
	getDueTime : function()
	{
		return (Ext.isDate(this.dueDate) && this.dueDate.getTime()) || this.dueDate;
	},

	/**
	 * Sets the range start time.
	 * @param {Number} startTime the range's start time.
	 * @param {Boolean} silence When set to true it will not throw an start date after due date error.
	 * @param {Boolean} ignoreUpdate When set to true it will not fire the update event.
	 * @return {Zarafa.core.DateRange} this date range
	 */
	setStartTime : function(startTime, silence, ignoreUpdate)
	{
		var original = null;

		// Preconditions
		if (!this.allowBlank && !startTime) {
			throw 'Cannot set DateRange start to undefined';
		} else if (Ext.isDate(this.dueDate) && startTime > this.dueDate.getTime() && !silence) {
			throw 'Cannot set DateRange start date to after its due date';
		}

		// No update event when new value equals old value
		ignoreUpdate = ignoreUpdate || (this.getStartTime() == startTime);

		if (!ignoreUpdate)
			original = this.clone();

		if (!Ext.isEmpty(startTime)) {
			this.startDate = new Date(startTime);
		} else {
			this.startDate = null;
		}

		if(!ignoreUpdate) {
			this.fireEvent("update", this, original);
		}

		return this;
	},

	/**
	 * Sets the range due time.
	 * @param {Number} dueTime the range's due time.
	 * @param {Boolean} silence When set to true it will not throw an start date after due date error.
	 * @param {Boolean} ignoreUpdate When set to true it will not fire the update event.
	 * @return {Zarafa.core.DateRange} this date range
	 */
	setDueTime : function(dueTime, silence, ignoreUpdate)
	{
		var original = null;

		// Precondition
		if (!this.allowBlank && !dueTime) {
			throw 'Cannot set DateRange due date to undefined';
		} else if (Ext.isDate(this.startDate) && dueTime < this.startDate.getTime() && !silence) {
			throw 'Cannot set DateRange due date to before its start date';
		}

		// No update event when new value equals old value
		ignoreUpdate = ignoreUpdate || (this.getDueTime() == dueTime);

		if (!ignoreUpdate)
			original = this.clone();

		if (!Ext.isEmpty(dueTime)) {
			this.dueDate = new Date(dueTime);
		} else {
			this.dueDate = null;
		}

		if(!ignoreUpdate) {
			this.fireEvent("update", this, original);
		}

		return this;
	},

	/**
	 * Sets the start and due times.
	 * @param {Number} startDate the range's start time.
	 * @param {Number} dueDate the range's due time.
	 * @param {Boolean} silence When set to true it will not throw an start date after due date error.
	 * @param {Boolean} ignoreUpdate When set to true it will not fire the update event.
	 * @return {Zarafa.core.DateRange} this date range
	 */
	setTime : function(startTime, dueTime, silence, ignoreUpdate)
	{
		var original = null;

		// Precondition
		if(!startTime || !dueTime) {
			if (!this.allowBlank && !startTime) {
				throw 'Cannot set DateRange start to undefined';
			}

			if (!this.allowBlank && !dueTime) {
				throw 'Cannot set DateRange due date to undefined';
			}
		} else if (startTime > dueTime && !silence) {
			throw 'Invalid date range, start date is after due date';
		}

		ignoreUpdate = ignoreUpdate || ((startTime == this.getStartTime()) && (dueTime == this.getDueTime()));

		if (!ignoreUpdate)
			original = this.clone();

		this.setStartTime(startTime, true, true);
		this.setDueTime(dueTime, true, true);

		if (!ignoreUpdate) {
			this.fireEvent("update", this, original);
		}

		return this;
	},

	/**
	 * @param {String} interval (optional) A valid date interval enum value.
	 * @return {Number} the range's duration in milliseconds
	 */
	getDuration : function(interval)
	{
		if (Ext.isDate(this.dueDate) && Ext.isDate(this.startDate)) {
			return Date.diff(interval || Date.MILLI, this.dueDate, this.startDate);
		}

		return 0;
	},

	/**
	 * @param {Number} duration the new duration of the range in milliseconds.
	 * @param {Boolean} ignoreUpdate When set to true it will not fire the update event.
	 * @return {Zarafa.core.DateRange} this date range
	 */
	setDuration : function(duration, ignoreUpdate)
	{
		var original = null;

		if(!Ext.isDate(this.startDate)) {
			// if no start date is provided then we can't set duration
			throw 'Cannot set duration when start date is not specified';
		}

		// No update event when new value equals old value
		ignoreUpdate = ignoreUpdate || (this.getDuration() == duration);

		if (!ignoreUpdate)
			original = this.clone();

		this.dueDate = new Date(this.getStartTime() + duration);

		if (!ignoreUpdate) {
			this.fireEvent("update", this, original);
		}

		return this;
	},
	
	/**
	 * Calculates the number of days spanned by this appointment, rounded to whole days. The function
	 * assumes that the range is an all day range. Even so, the rounding is still required to deal with
	 * date ranges that start and end in different time zones.
	 * 
	 * @return {Number} the number of days spanned by this appointment, rounded to whole days. 
	 */
	getNumDays : function()
	{
		var duration = this.getDuration(Date.DAY);
		if (Ext.isDefined(duration)) {
			return Math.round(duration);
		} else {
			return 0;
		}
	},
	
	/**
	 * Expands the range so that both the start and due times are multiples of the 'timeSlice' parameter. The start
	 * date is moved back, the due date is moved forward. Since this method does not care about time zones the value
	 * of 'timeSlice' is assumed to be <= 60 minutes.
	 * 
	 * @param {Number} timeSlice the time slice to 'snap' to.
	 * @return {Zarafa.core.DateRange} this date range
	 */
	expand : function(timeSlice)
	{
		var original = this.clone();

		if(Ext.isDate(this.startDate)) {
			this.startDate = new Date(this.getStartTime() - this.getStartTime() % timeSlice); 
		}

		if(Ext.isDate(this.dueDate)) {
			this.dueDate = new Date(this.getDueTime() + timeSlice - this.getDueTime() % timeSlice);
		}

		// only fire event if anything has changed
		if(Ext.isDate(this.startDate) || Ext.isDate(this.dueDate)) {
			this.fireEvent("update", this, original);
		}

		return this;
	},
	
	/**
	 * Deep-clones the date range.
	 * @return {Zarafa.core.DateRange} a clone of this date range.
	 */
	clone : function()
	{
		return new Zarafa.core.DateRange({
			startDate : Ext.isDate(this.startDate) ? new Date(this.getStartTime()) : undefined,
			dueDate : Ext.isDate(this.dueDate) ? new Date(this.getDueTime()) : undefined,
			allowBlank : this.allowBlank
		});
	},

	/**
	 * Test this date range for equality against another date range.
	 * @param {Zarafa.core.DateRange} otherRange a date range to compare with.
	 * @return {Boolean} true if this range equals the given other range.
	 */
	equals : function(otherRange)
	{
		if (!otherRange) {
			return false;
		}

		if(this.getStartTime() !== otherRange.getStartTime()) {
			// start dates don't match
			return false;
		}

		if(this.getDueTime() !== otherRange.getDueTime()) {
			// due dates don't match
			return false;
		}

		// both start/due dates matches
		return true;
	},
	
	/**
	 * Compares two date ranges for order. If range A comes before B, the function returns -1. If B comes before
	 * A, the function returns 1. If both are equal, this function returns 0. This functionality is equivalent
	 * to Java's Comparable interface.
	 * 
	 * Comparison is based on the range start. If the start dates are equal, further distinction is made by
	 * due date, with earlier due dates coming first.
	 * 
	 * Undefined dates will be considered as zero and compared.
	 * 
	 * @param {Zarafa.core.DateRange} otherRange Date range to compare with.
	 * @return {Number} If this range 'comes before' otherRange, the funtion returns -1. If this range 'comes
	 * after' otherRange, return 1. Otherwise return 0.
	 *   
	 */
	compare : function(otherRange)
	{
		// Compare start times.
		var aStartTime = this.getStartTime() || 0;
		var bStartTime = otherRange.getStartTime() || 0;

		if (aStartTime !== bStartTime)
			return aStartTime > bStartTime ? 1 : -1;

		// If start times are equal, compare due times.
		var aDueTime = this.getDueTime() || 0;
		var bDueTime = otherRange.getDueTime() || 0;

		if (aDueTime !== bDueTime)
			return aDueTime > bDueTime ? 1 : -1;
		
		// If ranges are equal, return 0.
		return 0;
	},
	
	/**
	 * Tests whether the date range is a so-called 'all day' range,
	 * meaning that start and due date duration time is more then one day(24 hours).
	 * @return {Boolean} true if the range is an all day range.
	 */
	isAllDay : function()
	{
		if(Ext.isDate(this.startDate) && Ext.isDate(this.dueDate) && !this.isZeroMinuteRange()) {
			return (this.startDate.clearTime(true).getTime() === this.getStartTime())
			&& (this.dueDate.clearTime(true).getTime() === this.getDueTime());
		}

		return false;
	},

	/**
	 * Tests whether the date range is a 0 minute.
	 * @return {Boolean} true if the range duration is 0 minute.
	 */
	isZeroMinuteRange : function()
	{
		return (this.dueDate.getTime() === this.startDate.getTime());
	},

	/**
	 * @return {Boolean} true if this date range overlaps with the other date range.
	 */
	overlaps : function(other)
	{
		var start1 = this.getStartTime(); 
		var due1 = this.getDueTime(); 
		var start2 = other.getStartTime();
		var due2 = other.getDueTime();

		return (start1 >= start2 && start1 < due2) || (start2 >= start1 && start2 < due1);
	},
	
	/**
	 * @param {Zarafa.core.DateRange} range date range to check against.
	 * @return {Boolean} true if this date range is inside the given date range.
	 */
	inside : function(range)
	{
		if(this.getStartTime() && range.getStartTime() && this.getDueTime() && range.getDueTime()) {
			return this.getStartTime() >= range.getStartTime() && this.getDueTime() <= range.getDueTime();
		}

		return false;
	},
	
	/**
	 * @param {Date} date the date to test.
	 * @return {Boolean} true if the give date is inside this date range.
	 */
	containsDate : function(date)
	{
		return this.getStartTime() <= date.getTime() && this.getDueTime() > date.getTime();
	},
	
	/**
	 * Formats the current visible date range as human-readable text. The formatter looks at which components the
	 * start and due dates have in common.
	 * For instance, the first and last days of a week range might lie in the same month (i.e. '13 - 19 July 2009'), 
	 * or they might not (i.e. '28 September - 2 November 2009'). Finally a range may have the start and end dates
	 * in different years, i.e. '28 December 2009 - 1 January 2010'. 
	 *   
	 * @return {String} the current date range as text. 
	 */
	format : function()
	{
		var startDate = this.startDate;
		var dueDate = this.dueDate;

		if(!Ext.isDate(startDate) || !Ext.isDate(dueDate)) {
			return '';
		}

		// If the due date is _exactly_ midnight, we must assume the last day of the period
		// is the previous day. So decrease the duedate with a day (making sure the duedate
		// does not move before the startDate).
		if (dueDate.getTime() === dueDate.clearTime(true).getTime()) {
			// Move to the previous day, use 12:00 as starting hour,
			// to prevent problems when the DST swithes at 00:00 (e.g. in Brasil).
			// We don't need to restore to the original time, as the string
			// which we are going to ignore doesn't contain a time representation.
			dueDate = dueDate.clone();
			dueDate.setHours(12);
			dueDate = dueDate.add(Date.DAY, -1);

			if (dueDate.getTime() < startDate.getTime())
				dueDate = startDate;
		}

		// The startDate and duedate are in completely different years.
		// Format the full date strings for both dates.
		if (startDate.getYear() != dueDate.getYear()) {
			// # TRANSLATORS: See http://docs.sencha.com/ext-js/3-4/#!/api/Date for the meaning of these formatting instructions
			return String.format('{0} - {1}', startDate.format(_('jS F Y')), dueDate.format(_('jS F Y')));
		}

		// The startDate and dueDate are in different months.
		// Format the date strings with the year in common.
		if (startDate.getMonth() != dueDate.getMonth()) {
			// # TRANSLATORS: See http://docs.sencha.com/ext-js/3-4/#!/api/Date for the meaning of these formatting instructions
			return String.format('{0} - {1} {2}', startDate.format(_('jS F')), dueDate.format(_('jS F')), startDate.format(_('Y')));
		}

		// The startDate and dueDate are on different days.
		// Format the date strings with the month and year in common.
		if (startDate.getDate() != dueDate.getDate()) {
			// # TRANSLATORS: See http://docs.sencha.com/ext-js/3-4/#!/api/Date for the meaning of these formatting instructions
			return String.format('{0} - {1} {2}', startDate.format(_('jS')), dueDate.format(_('jS')), startDate.format(_('F Y')));
		}

		// The startDate and dueDate are on the same day.
		// Format the date string with everything in common.
		// # TRANSLATORS: See http://docs.sencha.com/ext-js/3-4/#!/api/Date for the meaning of these formatting instructions
		return startDate.format(_('jS F Y'));
	}
});
Ext.namespace('Zarafa.core');

/**
 * @class Zarafa.core.EntryId
 *
 * Class for decoding entryids and for comparison between two entryids
 * we have basically two types of entryids object and store entryids.
 *
 * object entryids uses structure EID for entryids created using zarafa server > 6
 * and for older entryids it uses structure EID_V0. Store entryids are generally wrapped
 * with some extra information (like guid for provider, dll name) which should be removed
 * before comparing two store entryids, after removing this wrapping the unwrapped entryid
 * uses the format same as object entryids (EID or EID_V0).
 *
 * version flag in EID and EID_V0 are zarafa specific flag and indicates which structure is used
 * to create that entryid, EID always contains version as '01000000' and EID_V0 always contains
 * '00000000' as version flag.
 *
 * server part of EID and EID_V0 indicates server name and it can be variable length, padding can be
 * upto 3 bytes so it can be anything between 0 to 3 bytes.
 *
 * in public store public root folder, ipm_subtree and favorites folder are custom folders of zarafa
 * so it has static uniqueids.
 *
 * @singleton
 */
Zarafa.core.EntryId = (function()
{
	/* Bit definitions for abFlags[3] of ENTRYID */
	var ZARAFA_FAVORITE = '01';

	/* GUID of root public folder */
	var STATIC_GUID_PUBLICFOLDER = '00000000000000000000000000000003';
	/* GUID of root favorite folder */
	var STATIC_GUID_FAVORITE = '00000000000000000000000000000002';
	/* GUID of ipm_subtree of public store*/
	var STATIC_GUID_FAVSUBTREE = '00000000000000000000000000000001';
	/* GUID of Global Addressbook */
	var MUIDECSAB = 'AC21A95040D3EE48B319FBA753304425';
	/* GUID of Contact Provider */
	var MUIDZCSAB = '727F0430E3924FDAB86AE52A7FE46571';
	/* GUID for OneOff entryid */
	var MAPI_ONE_OFF_UID = '812B1FA4BEA310199D6E00DD010F5402';

	/* Hardcoded ID used for generating entryid of addressbook container */
	var ZARAFA_UID_ADDRESS_BOOK = '00000000';
	/* Hardcoded ID used for generating entryid of global addressbook container */
	var ZARAFA_UID_GLOBAL_ADDRESS_BOOK = '01000000';
	/* Hardcoded ID used for generating entryid of global addresslists container */
	var ZARAFA_UID_GLOBAL_ADDRESS_LISTS = '02000000';

	var BASE_EID = Ext.extend(Object, {

		// The entryid which this object represents
		entryId : '',

		// The length of the entryid
		length : 0,

		// Constructor
		// param: Entryid The entryid represented by this object
		constructor : function(entryId)
		{
			if(entryId) {
				// always make entryids in uppercase so comparison will be case insensitive
				this.entryId = entryId.toUpperCase();
				this.length = entryId.length;

				this.decomposeEntryId(this.entryId);
			}
		},

		// Detect padding (max 3 bytes) from the entryId
		getPadding : function(entryId)
		{
			var padding = '';
			var offset = 0;

			for (var iterations = 4; iterations > 0; iterations--) {
				if (entryId.substring(entryId.length - (offset + 2), entryId.length - offset) == '00') {
					padding += '00';
					offset += 2;
				} else {
					// if non-null character found then break the loop
					break;
				}
			}

			return padding;
		}

	});

	// Entryid from version 6
	var EID = Ext.extend(BASE_EID, {
		abFlags : '',           // BYTE[4],   4 bytes,  8 hex characters
		guid : '',              // GUID,     16 bytes, 32 hex characters
		version : '',           // ULONG,     4 bytes,  8 hex characters
		type : '',              // ULONG,     4 bytes,  8 hex characters
		uniqueId : '',          // GUID,     16 bytes, 32 hex characters
		server : '',            // CHAR,     variable length
		padding : '',           // TCHAR[3],  4 bytes,  8 hex characters (upto 4 bytes)

		MIN_LENGTH : 88,
		name : 'EID',

		// decompose the entryid and populate all flags of entryid
		decomposeEntryId : function(entryId)
		{
			var offset = 0;

			// First determine padding, and remove if from the entryId
			this.padding = this.getPadding(entryId);
			entryId = entryId.substring(0, entryId.length - this.padding.length);

			this.abFlags = entryId.substr(offset, 8);
			offset =+ 8;

			this.guid = entryId.substr(offset, 32);
			offset += 32;

			this.version = entryId.substr(offset, 8);
			offset += 8;

			this.type = entryId.substr(offset, 8);
			offset += 8;

			this.uniqueId = entryId.substr(offset, 32);
			offset += 32;

			this.server = entryId.substr(offset);
		}
	});

	// The entryid from the begin of zarafa till 5.20
	var EID_V0 = Ext.extend(BASE_EID, {
		abFlags : '',           // BYTE[4],   4 bytes,  8 hex characters
		guid : '',              // GUID,     16 bytes, 32 hex characters
		version : '',           // ULONG,     4 bytes,  8 hex characters
		type : '',              // ULONG,     4 bytes,  8 hex characters
		id : '',                // ULONG,     4 bytes,  8 hex characters
		server : '',            // CHAR,     variable length
		padding : '',           // TCHAR[3],  4 bytes,  8 hex characters (upto 4 bytes)

		MIN_LENGTH : 64,
		name : 'EID_V0',

		// decompose the entryid and populate all flags of entryid
		decomposeEntryId : function(entryId)
		{
			var offset = 0;

			// First determine padding, and remove if from the entryId
			this.padding = this.getPadding(entryId);
			entryId = entryId.substring(0, entryId.length - this.padding.length);

			this.abFlags = entryId.substr(offset, 8);
			offset =+ 8;

			this.guid = entryId.substr(offset, 32);
			offset += 32;

			this.version = entryId.substr(offset, 8);
			offset += 8;

			this.type = entryId.substr(offset, 8);
			offset += 8;

			this.id = entryId.substr(offset, 8);
			offset += 8;

			this.server = entryId.substr(offset);
		}
	});

	// wrapped store entryid
	var WrappedSEID = Ext.extend(BASE_EID, {
		flags : '',             // BYTE[4],      4 bytes,  8 hex characters
		providerUID : '',       // GUID,        16 bytes, 32 hex characters
		version : '',           // ULONG,        1 bytes,  2 hex characters	// zero
		type : '',              // ULONG,        1 bytes,  2 hex characters	// zero
		DLLFileName : '',       // BYTE,        variable length				// zarafa6client.dll
		terminationChar : '',   // BYTE[1],      1 bytes,  2 hex characters	// zero
		unWrappedEntryId : '',  // EID/EID_V0,  variable length because it contains server name

		name : 'WrappedSEID',

		// decompose the entryid and populate all flags of entryid
		decomposeEntryId : function(storeEntryId)
		{
			var offset = 0;

			this.flags = storeEntryId.substr(offset, 8);
			offset += 8;

			this.providerUID = storeEntryId.substr(offset, 32);
			offset += 32;

			this.version = storeEntryId.substr(offset, 2);
			offset += 2;

			this.type = storeEntryId.substr(offset, 2);
			offset += 2;

			// find length of dll name, find null character which indicates end of dll name after the current offset
			var termCharIndex = storeEntryId.slice(offset).indexOf('00');
			this.DLLFileName = storeEntryId.substr(offset, termCharIndex);
			offset += termCharIndex;

			this.terminationChar = storeEntryId.substr(offset, 2);
			offset += 2;

			this.unWrappedEntryId = storeEntryId.substr(offset);

			// unwrapped entryid is actually an object entryid so decompose it
			this.unWrappedEntryId = Zarafa.core.EntryId.createEntryIdObj(this.unWrappedEntryId);
		}
	});

	// The entryid for addressbook items
	var ABEID = Ext.extend(BASE_EID, {
		abFlags : '',           // BYTE[4],   4 bytes,  8 hex characters
		guid : '',              // GUID,     16 bytes, 32 hex characters
		version : '',           // ULONG,     4 bytes,  8 hex characters
		type : '',              // ULONG,     4 bytes,  8 hex characters
		id : '',                // ULONG,     4 bytes,  8 hex characters
		extid : '',             // CHAR,      variable length
		padding : '',           // TCHAR[3],  4 bytes,  8 hex characters (upto 4 bytes)

		MIN_LENGTH : 64,
		name : 'ABEID',

		// decompose the entryid and populate all flags of entryid
		decomposeEntryId : function(entryId)
		{
			var offset = 0;

			// First determine padding, and remove if from the entryId
			this.padding = this.getPadding(entryId);
			entryId = entryId.substring(0, entryId.length - this.padding.length);

			this.abFlags = entryId.substr(offset, 8);
			offset =+ 8;

			this.guid = entryId.substr(offset, 32);
			offset += 32;

			this.version = entryId.substr(offset, 8);
			offset += 8;

			this.type = entryId.substr(offset, 8);
			offset += 8;

			this.id = entryId.substr(offset, 8);
			offset += 8;

			this.extid = entryId.substr(offset);
		}
	});

	// The entryid for local addressbook items
	var WrappedABEID = Ext.extend(BASE_EID, {
		ulVersion : '',         // ULONG,     4 bytes,  8 hex characters
		muid : '',              // MAPIUID,  16 bytes, 32 hex characters
		ulObjType : '',         // ULONG,     4 bytes,  8 hex characters
		ulOffset : '',          // ULONG,     4 bytes,  8 hex characters
		unWrappedEntryId : '',  // EID/EID_V0,  variable length because it contains server name

		name : 'WrappedABEID',

		// decompose the entryid and populate all flags of entryid
		decomposeEntryId : function(ABEntryId)
		{
			var offset = 0;

			this.ulVersion = ABEntryId.substr(offset, 8);
			offset += 8;

			this.muid = ABEntryId.substr(offset, 32);
			offset += 32;

			this.ulObjType = ABEntryId.substr(offset, 8);
			offset += 8;

			this.ulOffset = ABEntryId.substr(offset, 8);
			offset += 8;

			this.unWrappedEntryId = ABEntryId.substr(offset);

			// unwrapped entryid is actually an object entryid so decompose it
			this.unWrappedEntryId = Zarafa.core.EntryId.createEntryIdObj(this.unWrappedEntryId);
		}
	});

	// Wrap an entryid into a Contact Provider entryid
	// @static
	WrappedABEID.wrapABEID = function(entryId, objType)
	{
		objType = objType.toString(16);

		// add padding for the type, which is of 4 bytes (8 characters)
		objType = String.leftPad(objType, 2, '0');
		objType = String.rightPad(objType, 8, '0');

		return '00000000' + MUIDZCSAB + objType + '00000000' + entryId;
	};

	// Unwrap an Contact Provider entryid
	// @static
	WrappedABEID.unwrapABEID = function(entryId)
	{
		// Remove ulVersion (8 char), muid (32 char), ulObjType (8 char) and ulOffset (8 char)
		return entryId.substring(56);
	};

	return {
		/**
		 * Creates an object that has split up all the components of an AB entryID.
		 * @param {String} entryid Entryid
		 * @return {Object} EntryID object
		 */
		createABEntryIdObj : function(entryid)
		{
			return new ABEID(entryid);
		},

		/**
		 * Compares two AB entryIds. It is possible to have two different entryIds that should match as they
		 * represent the same object (in multiserver environments).
		 * @param {String} entryId1 EntryID
		 * @param {String} entryId2 EntryID
		 * @return {Boolean} Result of the comparison
		 */
		compareABEntryIds : function(entryId1, entryId2)
		{
			if(!Ext.isString(entryId1) || !Ext.isString(entryId2)) {
				return false;
			}

			if(entryId1 === entryId2) {
				// if normal comparison succeeds then we can directly say that entryids are same
				return true;
			}

			var eid1 = Zarafa.core.EntryId.createABEntryIdObj(entryId1);
			var eid2 = Zarafa.core.EntryId.createABEntryIdObj(entryId2);

			if(eid1.length != eid2.length)
				return false;

			if(eid1.abFlags != eid2.abFlags)
				return false;

			if(eid1.version != eid2.version)
				return false;

			if(eid1.type != eid2.type)
				return false;

			if(eid1.length < eid1.MIN_LENGTH)
				return false;

			if(eid1.extid != eid2.extid)
				return false;

			return true;
		},

		/**
		 * Creates an object that has split up all the components of an entryID.
		 * @param {String} entryid Entryid
		 * @return {Object} EntryID object
		 */
		createEntryIdObj : function(entryid)
		{
			// check if we are dealing with old or new object entryids
			var versionString = entryid.substr(40, 8);

			if(versionString == '00000000') {
				// use EID_V0 struct
				var eidObj = new EID_V0(entryid);
			} else {
				// use EID struct
				var eidObj = new EID(entryid);
			}

			return eidObj;
		},

		/**
		 * Compares two entryIds. It is possible to have two different entryIds that should match as they
		 * represent the same object (in multiserver environments).
		 * @param {String} entryId1 EntryID
		 * @param {String} entryId2 EntryID
		 * @return {Boolean} Result of the comparison
		 */
		compareEntryIds : function(entryId1, entryId2)
		{
			if(!Ext.isString(entryId1) || !Ext.isString(entryId2)) {
				return false;
			}

			if(entryId1 === entryId2) {
				// if normal comparison succeeds then we can directly say that entryids are same
				return true;
			}

			var eid1 = Zarafa.core.EntryId.createEntryIdObj(entryId1);
			var eid2 = Zarafa.core.EntryId.createEntryIdObj(entryId2);

			if(eid1.length != eid2.length)
				return false;

			if(eid1.abFlags != eid2.abFlags)
				return false;

			if(eid1.version != eid2.version)
				return false;

			if(eid1.type != eid2.type)
				return false;

			if(eid1.name == 'EID_V0') {
				if(eid1.length < eid1.MIN_LENGTH)
					return false;

				if(eid1.id != eid2.id)
					return false;
			} else {
				if(eid1.length < eid1.MIN_LENGTH)
					return false;

				if(eid1.uniqueId != eid2.uniqueId)
					return false;
			}

			return true;
		},

		/**
		 * Creates an object that has split up all the components of a store entryid.
		 * @param {String} storeEntryId unwrapped store entryid.
		 * @return {Object} store entryid object.
		 */
		createStoreEntryIdObj : function(storeEntryId)
		{
			return new WrappedSEID(storeEntryId);
		},

		/**
		 * Compares two entryIds. It is possible to have two different entryIds that should match as they
		 * represent the same object (in multiserver environments).
		 * @param {String} storeEntryId1 store entryid
		 * @param {String} storeEntryId2 store entryid
		 * @return {Boolean} Result of the comparison
		 */
		compareStoreEntryIds : function(storeEntryId1, storeEntryId2)
		{
			if(!Ext.isString(storeEntryId1) || !Ext.isString(storeEntryId2)) {
				return false;
			}

			if(storeEntryId1 === storeEntryId2) {
				// if normal comparison succeeds then we can directly say that entryids are same
				return true;
			}

			var seid1 = Zarafa.core.EntryId.createStoreEntryIdObj(storeEntryId1);
			var seid2 = Zarafa.core.EntryId.createStoreEntryIdObj(storeEntryId2);

			// we are only interested in unwrapped entryid part
			seid1 = seid1.unWrappedEntryId;
			seid2 = seid2.unWrappedEntryId;

			if(seid1 === seid2) {
				// if normal comparison succeeds then we can directly say that entryids are same
				return true;
			}

			if(seid1.length < seid1.MIN_LENGTH || seid2.length < seid2.MIN_LENGTH)
				return false;

			if(seid1.guid != seid2.guid)
				return false;

			if(seid1.version != seid2.version)
				return false;

			if(seid1.type != seid2.type)
				return false;

			if(seid1.name == 'EID_V0') {
				if(seid1.length < seid1.MIN_LENGTH)
					return false;

				if(seid1.id != seid2.id)
					return false;
			} else {
				if(seid1.length < seid1.MIN_LENGTH)
					return false;

				if(seid1.uniqueId != seid2.uniqueId)
					return false;
			}

			return true;
		},

		/**
		 * Unwrap an Entryid which is of the Contact Provider ({@link #hasContactProviderGUID}
		 * returned true for this entryid}.
		 * @param {String} entryId the Address Book entryid.
		 * @return {String} The unwrapped entryId
		 */
		unwrapContactProviderEntryId : function(entryId)
		{
			return WrappedABEID.unwrapABEID(entryId)
		},

		/**
		 * Wrap an EntryId which should be wrapped using the Contact Provider
		 * @param {String} entryId The entryid
		 * @return {String} The wrapped entryId
		 */
		wrapContactProviderEntryId : function(entryId, objType)
		{
			return WrappedABEID.wrapABEID(entryId, objType)
		},

		/**
		 * Create a one-off entryid from the applied parameters.
		 * @param {String} displayname displaye name as configured in record.
		 * @param {String} addrtype weather the record is of type SMTP.
		 * @param {String} emailaddress email address as configured in record.
		 * @return {String} The oneoff entryId
		 */
		createOneOffEntryId : function(displayname, addrtype, emailaddress)
		{
			return '00000000' + MAPI_ONE_OFF_UID + '00000080' + Zarafa.core.Util.encode_utf16(displayname) + '0000' + Zarafa.core.Util.encode_utf16(addrtype) + '0000' + Zarafa.core.Util.encode_utf16(emailaddress) + '0000';
		},

		/**
		 * Checks if the passed folder entryid is a folder in the favorites folder, favorites folder
		 * contains 0x01 in the abFlags[3] flag.
		 * @param {String} entryId folder entryid
		 * @return {Boolean} true of folder is a favorite folder else false
		 */
		isFavoriteFolder : function(entryId)
		{
			var entryIdObj = Zarafa.core.EntryId.createEntryIdObj(entryId);

			return (entryIdObj.abFlags.substr(6, 8) == ZARAFA_FAVORITE);
		},

		/**
		 * Checks if the given entryid is a oneoff entryid.
		 * @param {String} entryId The entryid
		 * @return {Boolean} true if the entryid is a oneoff
		 */
		isOneOffEntryId : function(entryId)
		{
			var entryIdObj = Zarafa.core.EntryId.createEntryIdObj(entryId);

			return entryIdObj.guid == MAPI_ONE_OFF_UID;
		},

		/**
		 * Checks if the passed folder entryid is root favorites folder.
		 * @param {String} entryId folder entryid
		 * @return {Boolean} true of folder is a root favorite folder else false
		 */
		isFavoriteRootFolder : function(entryId)
		{
			var entryIdObj = Zarafa.core.EntryId.createEntryIdObj(entryId);

			return entryIdObj.uniqueId == STATIC_GUID_FAVORITE;
		},

		/**
		 * Checks if the passed folder entryid is root public folder.
		 * @param {String} entryId folder entryid
		 * @return {Boolean} true of folder is a root public folder else false
		 */
		isPublicRootFolder : function(entryId)
		{
			var entryIdObj = Zarafa.core.EntryId.createEntryIdObj(entryId);

			return entryIdObj.uniqueId == STATIC_GUID_PUBLICFOLDER;
		},

		/**
		 * Checks if the passed folder entryid is public subtree folder.
		 * @param {String} entryId folder entryid
		 * @return {Boolean} true of folder is a root public folder else false
		 */
		isPublicSubtreeFolder : function(entryId)
		{
			var entryIdObj = Zarafa.core.EntryId.createEntryIdObj(entryId);

			return entryIdObj.uniqueId == STATIC_GUID_FAVSUBTREE;
		},

		/**
		 * Checks if the GUID part of the entryid is of the Contact Provider.
		 * @param {String} entryId Address Book entryid
		 * @return {Boolean} true if guid matches the Contact Provider else false
		 */
		hasContactProviderGUID : function(entryId)
		{
			var entryIdObj = Zarafa.core.EntryId.createEntryIdObj(entryId);

			return entryIdObj.guid == MUIDZCSAB;
		},

		/**
		 * Checks if the GUID part of the entryid is of the Global Addressbook.
		 * @param {String} entryId Address Book entryid
		 * @return {Boolean} true if guid matches the Global Addressbook else false
		 */
		hasAddressBookGUID : function(entryId)
		{
			var entryIdObj = Zarafa.core.EntryId.createABEntryIdObj(entryId);

			return entryIdObj.guid == MUIDECSAB;
		},

		/**
		 * Checks if the GUID part of the entryid is of the Global Addressbook Container.
		 * @param {String} entryId Address Book entryid
		 * @return {Boolean} true if guid matches the Global Addressbook Container else false
		 */
		isGlobalAddressbookContainer : function(entryId)
		{
			// check for global addressbook entryid
			if(Zarafa.core.EntryId.hasAddressBookGUID(entryId) === false) {
				return false;
			}

			var entryIdObj = Zarafa.core.EntryId.createABEntryIdObj(entryId);

			// check for object_type == MAPI_ABCONT and id == 1
			return (entryIdObj.type == '04000000' && entryIdObj.id == ZARAFA_UID_GLOBAL_ADDRESS_BOOK);
		}
	};
})();
Ext.namespace('Zarafa.core');

/**
 * @class Zarafa.core.Events
 * Utility class for handling special events.
 * @singleton
 */
Zarafa.core.Events = {
	/**
	 * The list of fields and the corresponding {@link Ext.util.Observable}
	 * objects which are used to register the event handlers. Use
	 * {@link #registerListener} and {@link #unregisterListener} to add/remove
	 * {@Link Ext.util.Observable} instances. Use {@link #getListener}
	 * to obtain a {@link Ext.util.Observable} instance.
	 * @property
	 * @type Object
	 * @private
	 */
	listeners : {},

	/**
	 * Register a {@link Ext.util.Observable} instance for the given
	 * {@link Ext.form.Field}/{@link Ext.Element} pair to the {@link #listeners}
	 * object.
	 * @param {Ext.form.Field} field The register for which the observable is added
	 * @param {Ext.Element} el The element for which the the observable is added
	 * @param {Ext.util.Observable} observable The observable to register
	 * @private
	 */
	registerListener : function(field, el, observable)
	{
		var observables = Zarafa.core.Events.listeners[field.id];
		if (observables) {
			observables.push(observable);
		} else {
			Zarafa.core.Events.listeners[field.id] = [ observable ];
		}
	},

	/**
	 * Unregister a previously {@link #registerListener registered}
	 * {@link Ext.util.Observable} from the {@link #listeners}.
	 * @param {Ext.form.Field} field The register to which the observable belongs
	 * @param {Ext.Element} el The element to which the observable belongs
	 * @private
	 */
	unregisterListener : function(field, el)
	{
		var observables = Zarafa.core.Events.listeners[field.id];
		if (observables) {
			for (var i = 0, len = observables.length; i < len; i++) {
				var observable = observables[i];
				if (observable.el === el) {
					observables.splice(i, 1);
					if (observables.length === 0) {
						delete Zarafa.core.Events.listeners[field.id];
					}
				}
			}
		}
	},

	/**
	 * Obtain a previously {@link #registerListener registered}
	 * {@link Ext.util.Observable} from the {@link #listeners}.
	 * @param {Ext.form.Field} field The register to which the observable belongs
	 * @param {Ext.Element} el The element to which the observable belongs
	 * @private
	 */
	getListener : function(field, el)
	{
		var observables = Zarafa.core.Events.listeners[field.id];
		if (!observables) {
			return undefined;
		}

		for (var i = 0, len = observables.length; i < len; i++) {
			var observable = observables[i];
			if (observable.el === el) {
				return observable;
			}
		}

		return undefined;
	},

	/**
	 * Add a special event handler to the given field to catch
	 * 'paste' events. There are multiple ways to paste text into
	 * a textfield.
	 * 1) contextmenu
	 * 2) Ctrl-V (shortcut depending on OS)
	 * 3) Drag & Drop text
	 *
	 * Support for catching these options depends severely on the browser,
	 * and thus this special event handler will serve as compatibility
	 * handler to handle the various cases correctly.
	 *
	 * @param {Ext.form.Field} field The field on which to listen for
	 * paste events.
	 * @param {Ext.Element} el The element on which the event should be registered
	 * @param {Function} fn The callback function to be called when text
	 * has been pasted. This function has no arguments (See {@link Ext.util.Observable#on}).
	 * @param {Object} scope The scope in which the functon will be called (See {@link Ext.util.Observable#on}).
	 * @param {Object} obj (optional) Additional options (See {@link Ext.util.Observable#on}).
	 */
	addPasteEventHandler : function(field, el, fn, scope, obj)
	{
		// Check if this field has already been used to register
		// an event handler for pasting. If that is not the case
		// we need to construct a new Ext.util.Observable object
		// for the field and add the event handlers.
		var observable = Zarafa.core.Events.getListener(field, el);
		if (!Ext.isDefined(observable)) {
			var noOn = !Ext.isFunction(el.on);
			observable = new Ext.util.Observable();
			observable.field = field;
			observable.el = el;
			observable.hasFocus = field.hasFocus;
			observable.originalValue = field.getValue();
			observable.addEvents('paste');

			// Register the event handler for paste events. This will ensure
			// the input element will be resized when pasting.
			// Opera doesn't support the paste event, and thus we need to listen
			// to the 'keyup' event to check if a 'Ctrl-V' is pressed.
			if (Ext.isOpera) {
				if (noOn) {
					el.addEventListener('keyup', this.onPasteKeyUp.createDelegate(observable));
				} else {
					field.mon(el, 'keyup', this.onPasteKeyUp, observable);
				}
			} else {
				if (noOn) {
					el.addEventListener('paste', this.onPaste.createDelegate(observable));
				} else {
					field.mon(el, 'paste', this.onPaste, observable);
				}
			}

			// A special kind of pasting is dragging & dropping text into
			// the boxfield. There really isn't a true event handler for that,
			// as it isn't pasting, but neither is it typing. So use the mouse
			// to detect such changes.
			//
			// For some reason the 'mouseup' event is not being triggered when
			// the user drops the text into the input field. So we must use
			// the event which is fired a bit later.
			if (noOn) {
				el.addEventListener('mouseover', this.onPasteMouseOver.createDelegate(observable));
			} else {
				field.mon(el, 'mouseover', this.onPasteMouseOver, observable);
			}
			field.on('blur', this.onPasteBlur, observable);

			Zarafa.core.Events.registerListener(field, el, observable);
		}

		observable.addListener('paste', fn, scope, obj);
	},

	/**
	 * Removes the event handler as registered by {@link #addPasteEventHandler}.
	 *
	 * @param {Ext.form.Field} field The field on which the event was registered
	 * @param {Ext.Element} el The element on which the event should be registered
	 * @param {Function} fn The function to unregister
	 * @param {Object} scope The scope for the function.
	 */
	removePasteEventHandler : function(field, el, fn, scope)
	{
		var observable = Zarafa.core.Events.getListener(field, el);
		if (Ext.isDefined(observable)) {
			observable.removeListener('paste', fn, scope);
			// If this was the last event handler, delete
			// the Observable instance.
			if (!observable.hasListener('paste')) {
				Zarafa.core.Events.unregisterListener(field, el);
			}
		}
	},

	/**
	 * As Opera doesn't have the 'paste' browser event we are mimicking the behavior
	 * here by having a global 'keyup' event handler. This event will simply check
	 * if 'Ctrl-V has been pressed and will fire the event handler.
	 *
	 * This function is called in the scope of an {@link Ext.util.Observable}.
	 *
	 * @param {Number} key The key which was pressed
	 * @param {Ext.EventObject} e The event object which fired the event
	 * @private
	 */
	onPasteKeyUp : function(key, e)
	{
		// There are references online which indicate that Opera doesn't detect
		// the Ctrl key properly and the keyCode is 0. But Opera 11 seems to
		// behave correctly...
		if (key.ctrlKey === true && key.keyCode === Ext.EventObject.V) {
			this.fireEvent('paste');
		}
	},

	/**
	 * Event handler for the 'paste' event on the {@link #el input element}.
	 * This will start the {@link #onPastePoll} function to wait for the contents
	 * to be pasted so it can fire the event handler.
	 *
	 * This function is called in the scope of an {@link Ext.util.Observable}.
	 *
	 * @private
	 */
	onPaste : function()
	{
		Zarafa.core.Events.onPastePoll.call(this, this.field, this.field.getValue(), 5);
	},

	/**
	 * The 'paste' event on an input element is fired before the text which must
	 * be pasted is added into the input field. There is no cross-browser solution
	 * to access the text before it is put into the input field, hence polling is used
	 * to wait for the input field to be updated with the new text, so it can fire
	 * the event handler.
	 *
	 * This function is called in the scope of an {@link Ext.util.Observable}.
	 *
	 * @param {Ext.form.Field} field The field to poll for the new value
	 * @param {String} oldValue The original value of the input element
	 * @param {Number} limit The number of polling tries left
	 * @private
	 */
	onPastePoll : function(field, oldValue, limit)
	{
		if (limit === 0) {
			return;
		} else if (field.getValue() === oldValue) {
			Zarafa.core.Events.onPastePoll.defer(1, this, [field, oldValue, --limit]);
			return;
		}
		this.fireEvent('paste');
		this.originalValue = this.field.getValue();
	},

	/**
	 * Event handler which is fired when the Mouse is being moved over the
	 * input field. This will check if the input field has been magically
	 * changed without the user pressing any button. If that happens, then
	 * the user has dragged text into the input element and we need to
	 * fire the event handler.
	 *
	 * This function is called in the scope of an {@link Ext.util.Observable}.
	 *
	 * @private
	 */
	onPasteMouseOver : function()
	{
		if (this.hasFocus === false) {
			if (this.originalValue !== this.field.getValue()) {
				this.field.focus();
				this.fireEvent('paste');
				this.originalValue = this.field.getValue();
			}

			this.hasFocus = true;
		}
	},

	/**
	 * Event handler which is fired when the registered field is being
	 * {@link Ext.form.Field#blur blurred}. This will store the value
	 * currently in the editor so during {@link #onPasteMouseOver} we can
	 * check if text has been dropped into the editor.
	 * @private
	 */
	onPasteBlur : function()
	{
		this.hasFocus = false;
		this.originalValue = this.field.getValue();
	}
};
Ext.namespace("Zarafa.core");

/**
 * @class Zarafa.core.HTMLParser
 *
 * Class for performing operations on HTML content.
 *
 * @singleton
 */
Zarafa.core.HTMLParser = (function() {
	// regular expression to strip all style tags with its content
	var stripStylesRe = /<style[^>]*>[\s\S]*?<\/style[^>]*>/gim;

	// regular expression to strip all style tags with its content
	var stripScriptsRe = /<script[^>]*>[\s\S]*?<\/script[^>]*>/gim;

	// regular expression to convert <br /> tags to newlines
	var br2nlRe = /<br\s*?\/*?>/gim;

	// regular expression to convert \r\n, \n or \r tags to <br />
	var nl2brRe = /\r\n|\n|\r/gim;

	// regular expression to convert outlook style inline image urls to url which can request image using download_attachment.php
	var cidToUrlRe = /(src\s*=\s*[\"\']?)cid:([^ \"\']*)([\"\']?)/igm;

	// regular expression to convert url for inline images to outlook style url
	var urlToCidRe = /(src\s*=\s*[\"\']?)\S*attachCid=([^ &\"\']*)[^ \"\']*([\"\']?)/igm;

	return {
		/**
		 * Strips all style tags, and also remove its contents
		 * @param {Mixed} value The text from which to strip style tags
		 * @return {String} The stripped text
		 */
		stripStyles : function(v)
		{
			return !v ? v : String(v).replace(stripStylesRe, '');
		},

		/**
		 * Strips all script tags, and also remove its contents
		 * @param {Mixed} value The text from which to strip script tags
		 * @return {String} The stripped text
		 */
		stripScripts : function(v)
		{
			return !v ? v : String(v).replace(stripScriptsRe, '');
		},

		/**
		 * Converts HTML tag <br /> to newline characters
		 * @param {String} The string value to format.
		 * @return {String} The string with embedded \n tags in place of <br />.
		 */
		br2nl : function(v)
		{
			return Ext.isEmpty(v) ? '' : v.replace(br2nlRe, '\n');
		},

		/**
		 * Converts newline characters to HTML tag <br />
		 * @param {String} The string value to format.
		 * @return {String} The string with embedded <br /> tags in place of \n.
		 */
		nl2br : function(v)
		{
			return Ext.isEmpty(v) ? '' : v.replace(nl2brRe, '<br>');
		},

		/**
		 * Format string with plain-text contents into a HTML formatted string.
		 *
		 * This will convert new-line characters into <br /> elements.
		 *
		 * @param {String} content The plain-text contents to be formatted
		 * @return {String} The HTML representation of the content
		 */
		convertPlainToHTML : function(content)
		{
			if(Ext.isEmpty(content)) {
				return content;
			}

			// convert all html entities to their html equivalents
			content = Zarafa.core.HTMLParser.entityEncode(content, 'ENT_NOQUOTES');

			// We should wrap the content in <pre> tag to maintain
			// text indentation/spacing when we conver it to HTML.
			content = '<div><pre wrap style=\"white-space: -moz-pre-wrap; white-space: -pre-wrap; white-space: -o-pre-wrap; white-space: pre-wrap; word-wrap: break-word;\">' + content + '</pre></div>';

			// simple text markup *bold* and _underlined_ text
			content = content.replace(XRegExp("(?<prefix>^|[^\\p{L}])(?<word>\\*+[\\p{L},\ ,\t,0-9]+\\*+)(?<postfix>[^\\p{L}]|$)", "gi"), function(match) {
				return match.prefix + '<strong>' + match.word + '</strong>' + match.postfix;
			});

			content = content.replace(XRegExp("(?<prefix>^|[^\\p{L}])(?<word>_+[\\p{L},\ ,\t,0-9]+_+)(?<postfix>[^\\p{L}]|$)", "gi"), function(match) {
				return match.prefix + '<span style="text-decoration: underline">' + match.word + '</span>' + match.postfix;
			})

			// convert all breaklines
			content = Zarafa.core.HTMLParser.nl2br(content);

			return content;
		},

		/**
		 * Format string with HTML contents into a plain-text string.
		 *
		 * This will convert <br /> characters into \n elements.
		 *
		 * @param {String} content The HTML contents to be formatted
		 * @return {String} The plain-text representation of the content
		 */
		convertHTMLToPlain : function(content)
		{
			if(Ext.isEmpty(content)) {
				return content;
			}

			//----- remove tags but preserve the content ----

			// remove all select / options tags
			content = content.replace( /<[\/]?(?:select)[^>]*>/gim, '\n');
			content = content.replace(/<[\/]?(?:option)[^>]*>/gim, '\t');

			// replace all tags with their text equivalents
			content = content.replace(/<(?:hr)[^>]*>/gim, '\n-----------\n');
			content = content.replace(/<[\/]?(?:h[123456]|div|p|pre|title)[^>]*>/gim, '\n\n');
			content = content.replace(/<[\/]?(?:ul|ol|dl|dt|textarea|img)[^>]*>/gim, '\n');
			content = content.replace(/<[\/]?(?:dd|li)[^>]*>/gim, '\t');

			// tags related to table
			content = content.replace(/<[\/]?(?:table)[^>]*>/gim, '\n\n');
			content = content.replace(/<[\/]?(?:caption|tr)[^>]*>/gim, '\n');
			content = content.replace(/<[^\/]?(?:th|td)[^>]*>/gim, '<br />');

			// remove anchor tag by preserving the links, links will be added after content of anchor tag in between <> signs
			content = content.replace(/<a[^>]* href=[\'\"]?([^\s\'\">]*)[^>]*>([\s\S]*?)<\/a[^>]*>/gim, '$2 &lt;$1&gt;');

			//------ remove tags without preserving the contents -----

			// remove style tags
			content = Zarafa.core.HTMLParser.stripStyles(content);

			// remove script tags
			content = Zarafa.core.HTMLParser.stripScripts(content);

			// remove comments
			content = content.replace(/<!--[\s\S]*?-->/gim, '');

			// we have processed tags which are usefull for plain text conversion so now remove all remaining tags
			content = Zarafa.core.HTMLParser.stripUnwantedTags(content, ['br']);

			// decode html entities
			content = Zarafa.core.HTMLParser.entityDecode(content);

			// remove unnecessary space
			content = content.replace(/^\s*$/gm, '');

			// add <br> in line which hasn't <br> at end of line
			content = content.replace(/(.*[^<>\n]$)/gm, '$1 <br />');

			// remove extra line breaks
			content = content.replace(/\n/gm, '');

			// convert all breaklines
			content = Zarafa.core.HTMLParser.br2nl(content);

			// remove remaining html entities
			content = content.replace(/[&][#0-9a-z]*[;]/gim, '');

			// remove breaklines from the end of the lines
			content = content.replace(/\n(?!\n)$/gm, '');

			return content;
		},

		/**
		 * Function which strips unwanted tags, This function is whitelisting based so you need
		 * to pass tags that are allowed in the text, all other tags will be removed. This function will
		 * only remove tags and will not remove content in between tags.
		 * @param {String} content html content
		 * @param {Array} allowedTags tags that should not be removed from the content.
		 * @return {String} content after removing unwanted tags.
		 */
		stripUnwantedTags : function (content, allowedTags)
		{
			// Match all HTML tags
			var matches = content.match(/(<\/?[^>]+>)/gi);

			var key = '';
			var html = '';
			var allowedTag = '';
			var i = -1;
			var allowed = false;

			if(!allowedTags) {
				allowedTags = [];
			}

			// Go through all HTML tags
			if(matches && matches.length > 0) {
				for(var index1 = 0, len1 = matches.length; index1 < len1; index1++) {
					// Save HTML tag
					html = matches[index1].toString();

					// flag to indicate that tag should be removed or not
					allowed = false;

					// Go through all allowed tags
					for(var index2 = 0, len2 = allowedTags.length; index2 < len2; index2++) {
						allowedTag = allowedTags[index2].toLowerCase();
						i = -1;

						if (i !== 0) {
							i = html.toLowerCase().indexOf('<'+allowedTag+'>');
						}

						if (i !== 0) {
							i = html.toLowerCase().indexOf('<'+allowedTag+' ');
						}

						if (i !== 0) {
							i = html.toLowerCase().indexOf('</'+allowedTag);
						}

						// Determine
						if (i == 0) {
							allowed = true;
							break;
						}
					}

					if (allowed !== true) {
						content = content.split(html).join('');		// Custom replace. No regexing
					}
				}
			}

			return content;
		},

		/**
		 * Function will check if data contains external contents in any html tag (img, audio, video),
		 * and will also check for external stylesheets.
		 * @param {String} data data that should be checked for external content.
		 * @return {Boolean} true if data contains external content else false.
		 */
		hasExternalContent : function(data)
		{
			if(Ext.isEmpty(data)) {
				return false;
			}

			// @TODO more work needs for these regular expressions or else a dom based html parser
			// check tags whose attributes are src or background
			if(data.search(/(src|background)\s*=\s*([\'\"])*?\s*(https*:[^ \'\"]*)([\'\"])*/igm) !== -1) {
				return true;
			}

			// check tags whose attributes contains style attribute with external url
			if(data.search(/(style)\s*=(\S*)(url)\(([\'\"]*?)\s*(https*:.*[^\'\"])([\'\"]*?)\)/igm) !== -1) {
				return true;
			}

			return false;
		},

		/**
		 * Function will replace external content links with blank strings, so external content would not be loaded.
		 * @param {String} data raw data.
		 * @return {String} filtered data.
		 */
		blockExternalContent : function(data)
		{
			if(Ext.isEmpty(data)) {
				return data;
			}

			if (!Zarafa.core.HTMLParser.hasExternalContent(data)) {
				return data;
			}

			// @TODO more work needs for these regular expressions or else a dom based html parser
			data = data.replace(/(src|background)\s*=\s*([\'\"])*?\s*(https*:[^ \'\"]*)\s*([\'\"])*/igm, "$1=$2$4");
			data = data.replace(/(style)\s*=(\S*)(url)\(([\'\"]*?)\s*(https*:.*[^\'\"])([\'\"]*?)\)/igm, "$1=$2$3($4$6)");

			return data;
		},

		/**
		 * Function will return translation table for HTML entities which can be used to
		 * encode/decode HTML entities.
		 * @param {String} tableType type of table to get, options are HTML_SPECIALCHARS or HTML_ENTITIES.
		 * @param {String} quoteStyle options are ENT_COMPAT, ENT_NOQUOTES or ENT_QUOTES.
		 * @return {Object} table for translation of HTML entities
		 */
		getHTMLTranslationTable : function(tableType, quoteStyle)
		{
			if(!tableType) {
				tableType = 'HTML_SPECIALCHARS';
			}

			if(!quoteStyle) {
				quoteStyle = 'ENT_COMPAT';
			}

			var entities = {};
			entities['38'] = '&amp;';

			if (tableType === 'HTML_ENTITIES') {
				entities['94'] = '&circ;';
				entities['126'] = '&tilde;';

				entities['130'] = '&sbquo;';	// Single Low-9 Quotation Mark
				entities['131'] = '&fnof;';		// Latin Small Letter F With Hook
				entities['132'] = '&bdquo;';	// Double Low-9 Quotation Mark
				entities['133'] = '&hellip;';	// Horizontal Ellipsis
				entities['134'] = '&dagger;';	// Dagger
				entities['135'] = '&Dagger;';	// Double Dagger
				entities['136'] = '&circ;';		// Modifier Letter Circumflex Accent
				entities['137'] = '&permil;';	// Per Mille Sign
				entities['138'] = '&Scaron;';	// Latin Capital Letter S With Caron
				entities['139'] = '&lsaquo;';	// Single Left-Pointing Angle Quotation Mark
				entities['140'] = '&OElig;';	// Latin Capital Ligature OE
				entities['145'] = '&lsquo;';	// Left Single Quotation Mark
				entities['146'] = '&rsquo;';	// Right Single Quotation Mark
				entities['147'] = '&ldquo;';	// Left Double Quotation Mark
				entities['148'] = '&rdquo;';	// Right Double Quotation Mark
				entities['149'] = '&bull;';		// Bullet
				entities['150'] = '&ndash;';	// En Dash
				entities['151'] = '&mdash;';	// Em Dash
				entities['152'] = '&tilde;';	// Small Tilde
				entities['153'] = '&trade;';	// Trade Mark Sign
				entities['154'] = '&scaron;';	// Latin Small Letter S With Caron
				entities['155'] = '&rsaquo;';	// Single Right-Pointing Angle Quotation Mark
				entities['156'] = '&oelig;';	// Latin Small Ligature OE
				entities['159'] = '&Yuml;';		// Latin Capital Letter Y With Diaeresis

				entities['160'] = '&nbsp;';		// Non-breaking space
				entities['161'] = '&iexcl;';	// Inverted exclamation mark
				entities['162'] = '&cent;';		// Cent sign
				entities['163'] = '&pound;';	// Pound sign
				entities['164'] = '&curren;';	// Currency sign
				entities['165'] = '&yen;';		// Yen sign
				entities['166'] = '&brvbar;';	// Broken vertical bar
				entities['167'] = '&sect;';		// Section sign
				entities['168'] = '&uml;';		// Diaeresis
				entities['169'] = '&copy;';		// Copyright sign
				entities['170'] = '&ordf;';		// Feminine ordinal indicator
				entities['171'] = '&laquo;';	// Left-pointing double angle quotation mark
				entities['172'] = '&not;';		// Not sign
				entities['173'] = '&shy;';		// Soft hyphen
				entities['174'] = '&reg;';		// Registered sign
				entities['175'] = '&macr;';		// Macron
				entities['176'] = '&deg;';		// Degree sign
				entities['177'] = '&plusmn;';	// Plus-minus sign
				entities['178'] = '&sup2;';		// Superscript two
				entities['179'] = '&sup3;';		// Superscript three
				entities['180'] = '&acute;';	// Acute accent
				entities['181'] = '&micro;';	// Micro sign
				entities['182'] = '&para;';		// Pilcrow sign
				entities['183'] = '&middot;';	// Middle dot
				entities['184'] = '&cedil;';	// Cedilla
				entities['185'] = '&sup1;';		// Superscript one
				entities['186'] = '&ordm;';		// Masculine ordinal indicator
				entities['187'] = '&raquo;';	// Right-pointing double angle quotation mark
				entities['188'] = '&frac14;';	// Vulgar fraction one-quarter
				entities['189'] = '&frac12;';	// Vulgar fraction one-half
				entities['190'] = '&frac34;';	// Vulgar fraction three-quarters
				entities['191'] = '&iquest;';	// Inverted question mark
				entities['192'] = '&Agrave;';	// A with grave
				entities['193'] = '&Aacute;';	// A with acute
				entities['194'] = '&Acirc;';	// A with circumflex
				entities['195'] = '&Atilde;';	// A with tilde
				entities['196'] = '&Auml;';		// A with diaeresis
				entities['197'] = '&Aring;';	// A with ring above
				entities['198'] = '&AElig;';	// AE
				entities['199'] = '&Ccedil;';	// C with cedilla
				entities['200'] = '&Egrave;';	// E with grave
				entities['201'] = '&Eacute;';	// E with acute
				entities['202'] = '&Ecirc;';	// E with circumflex
				entities['203'] = '&Euml;';		// E with diaeresis
				entities['204'] = '&Igrave;';	// I with grave
				entities['205'] = '&Iacute;';	// I with acute
				entities['206'] = '&Icirc;';	// I with circumflex
				entities['207'] = '&Iuml;';		// I with diaeresis
				entities['208'] = '&ETH;';		// Eth
				entities['209'] = '&Ntilde;';	// N with tilde
				entities['210'] = '&Ograve;';	// O with grave
				entities['211'] = '&Oacute;';	// O with acute
				entities['212'] = '&Ocirc;';	// O with circumflex
				entities['213'] = '&Otilde;';	// O with tilde
				entities['214'] = '&Ouml;';		// O with diaeresis
				entities['215'] = '&times;';	// Multiplication sign
				entities['216'] = '&Oslash;';	// O with stroke
				entities['217'] = '&Ugrave;';	// U with grave
				entities['218'] = '&Uacute;';	// U with acute
				entities['219'] = '&Ucirc;';	// U with circumflex
				entities['220'] = '&Uuml;';		// U with diaeresis
				entities['221'] = '&Yacute;';	// Y with acute
				entities['222'] = '&THORN;';	// Thorn
				entities['223'] = '&szlig;';	// Sharp s. Also known as ess-zed
				entities['224'] = '&agrave;';	// a with grave
				entities['225'] = '&aacute;';	// a with acute
				entities['226'] = '&acirc;';	// a with circumflex
				entities['227'] = '&atilde;';	// a with tilde
				entities['228'] = '&auml;';		// a with diaeresis
				entities['229'] = '&aring;';	// a with ring above
				entities['230'] = '&aelig;';	// ae. Also known as ligature ae
				entities['231'] = '&ccedil;';	// c with cedilla
				entities['232'] = '&egrave;';	// e with grave
				entities['233'] = '&eacute;';	// e with acute
				entities['234'] = '&ecirc;';	// e with circumflex
				entities['235'] = '&euml;';		// e with diaeresis
				entities['236'] = '&igrave;';	// i with grave
				entities['237'] = '&iacute;';	// i with acute
				entities['238'] = '&icirc;';	// i with circumflex
				entities['239'] = '&iuml;';		// i with diaeresis
				entities['240'] = '&eth;';		// eth
				entities['241'] = '&ntilde;';	// n with tilde
				entities['242'] = '&ograve;';	// o with grave
				entities['243'] = '&oacute;';	// o with acute
				entities['244'] = '&ocirc;';	// o with circumflex
				entities['245'] = '&otilde;';	// o with tilde
				entities['246'] = '&ouml;';		// o with diaeresis
				entities['247'] = '&divide;';	// Division sign
				entities['248'] = '&oslash;';	// o with stroke. Also known as o with slash
				entities['249'] = '&ugrave;';	// u with grave
				entities['250'] = '&uacute;';	// u with acute
				entities['251'] = '&ucirc;';	// u with circumflex
				entities['252'] = '&uuml;';		// u with diaeresis
				entities['253'] = '&yacute;';	// y with acute
				entities['254'] = '&thorn;';	// thorn
				entities['255'] = '&yuml;';		// y with diaeresis
				entities['264'] = '&#264;';		// Latin capital letter C with circumflex
				entities['265'] = '&#265;';		// Latin small letter c with circumflex
				entities['338'] = '&OElig;';	// Latin capital ligature OE
				entities['339'] = '&oelig;';	// Latin small ligature oe
				entities['352'] = '&Scaron;';	// Latin capital letter S with caron
				entities['353'] = '&scaron;';	// Latin small letter s with caron
				entities['372'] = '&#372;';		// Latin capital letter W with circumflex
				entities['373'] = '&#373;';		// Latin small letter w with circumflex
				entities['374'] = '&#374;';		// Latin capital letter Y with circumflex
				entities['375'] = '&#375;';		// Latin small letter y with circumflex
				entities['376'] = '&Yuml;';		// Latin capital letter Y with diaeresis
				entities['402'] = '&fnof;';		// Latin small f with hook, function, florin
				entities['710'] = '&circ;';		// Modifier letter circumflex accent
				entities['732'] = '&tilde;';	// Small tilde
				entities['913'] = '&Alpha;';	// Alpha
				entities['914'] = '&Beta;';		// Beta
				entities['915'] = '&Gamma;';	// Gamma
				entities['916'] = '&Delta;';	// Delta
				entities['917'] = '&Epsilon;';	// Epsilon
				entities['918'] = '&Zeta;';		// Zeta
				entities['919'] = '&Eta;';		// Eta
				entities['920'] = '&Theta;';	// Theta
				entities['921'] = '&Iota;';		// Iota
				entities['922'] = '&Kappa;';	// Kappa
				entities['923'] = '&Lambda;';	// Lambda
				entities['924'] = '&Mu;';		// Mu
				entities['925'] = '&Nu;';		// Nu
				entities['926'] = '&Xi;';		// Xi
				entities['927'] = '&Omicron;';	// Omicron
				entities['928'] = '&Pi;';		// Pi
				entities['929'] = '&Rho;';		// Rho
				entities['931'] = '&Sigma;';	// Sigma
				entities['932'] = '&Tau;';		// Tau
				entities['933'] = '&Upsilon;';	// Upsilon
				entities['934'] = '&Phi;';		// Phi
				entities['935'] = '&Chi;';		// Chi
				entities['936'] = '&Psi;';		// Psi
				entities['937'] = '&Omega;';	// Omega
				entities['945'] = '&alpha;';	// alpha
				entities['946'] = '&beta;';		// beta
				entities['947'] = '&gamma;';	// gamma
				entities['948'] = '&delta;';	// delta
				entities['949'] = '&epsilon;';	// epsilon
				entities['950'] = '&zeta;';		// zeta
				entities['951'] = '&eta;';		// eta
				entities['952'] = '&theta;';	// theta
				entities['953'] = '&iota;';		// iota
				entities['954'] = '&kappa;';	// kappa
				entities['955'] = '&lambda;';	// lambda
				entities['956'] = '&mu;';		// mu
				entities['957'] = '&nu;';		// nu
				entities['958'] = '&xi;';		// xi
				entities['959'] = '&omicron;';	// omicron
				entities['960'] = '&pi;';		// pi
				entities['961'] = '&rho;';		// rho
				entities['962'] = '&sigmaf;';	// sigmaf
				entities['963'] = '&sigma;';	// sigma
				entities['964'] = '&tau;';		// tau
				entities['965'] = '&upsilon;';	// upsilon
				entities['966'] = '&phi;';		// phi
				entities['967'] = '&chi;';		// chi
				entities['968'] = '&psi;';		// psi
				entities['969'] = '&omega;';	// omega
				entities['977'] = '&thetasym;';	// Theta symbol
				entities['978'] = '&upsih;';	// Greek upsilon with hook symbol
				entities['982'] = '&piv;';		// Pi symbol
				entities['8194'] = '&ensp;';	// En space
				entities['8195'] = '&emsp;';	// Em space
				entities['8201'] = '&thinsp;';	// Thin space
				entities['8204'] = '&zwnj;';	// Zero width non-joiner
				entities['8205'] = '&zwj;';		// Zero width joiner
				entities['8206'] = '&lrm;';		// Left-to-right mark
				entities['8207'] = '&rlm;';		// Right-to-left mark
				entities['8211'] = '&ndash;';	// En dash
				entities['8212'] = '&mdash;';	// Em dash
				entities['8216'] = '&lsquo;';	// Left single quotation mark
				entities['8217'] = '&rsquo;';	// Right single quotation mark
				entities['8218'] = '&sbquo;';	// Single low-9 quotation mark
				entities['8220'] = '&ldquo;';	// Left double quotation mark
				entities['8221'] = '&rdquo;';	// Right double quotation mark
				entities['8222'] = '&bdquo;';	// Double low-9 quotation mark
				entities['8224'] = '&dagger;';	// Dagger
				entities['8225'] = '&Dagger;';	// Double dagger
				entities['8226'] = '&bull;';	// Bullet
				entities['8230'] = '&hellip;';	// Horizontal ellipsis
				entities['8240'] = '&permil;';	// Per mille sign
				entities['8242'] = '&prime;';	// Prime
				entities['8243'] = '&Prime;';	// Double Prime
				entities['8249'] = '&lsaquo;';	// Single left-pointing angle quotation
				entities['8250'] = '&rsaquo;';	// Single right-pointing angle quotation
				entities['8254'] = '&oline;';	// Overline
				entities['8260'] = '&frasl;';	// Fraction Slash
				entities['8364'] = '&euro;';	// Euro sign
				entities['8472'] = '&weierp;';	// Script capital
				entities['8465'] = '&image;';	// Blackletter capital I
				entities['8476'] = '&real;';	// Blackletter capital R
				entities['8482'] = '&trade;';	// Trade mark sign
				entities['8501'] = '&alefsym;';	// Alef symbol
				entities['8592'] = '&larr;';	// Leftward arrow
				entities['8593'] = '&uarr;';	// Upward arrow
				entities['8594'] = '&rarr;';	// Rightward arrow
				entities['8595'] = '&darr;';	// Downward arrow
				entities['8596'] = '&harr;';	// Left right arrow
				entities['8629'] = '&crarr;';	// Downward arrow with corner leftward. Also known as carriage return
				entities['8656'] = '&lArr;';	// Leftward double arrow. ISO 10646 does not say that lArr is the same as the 'is implied by' arrow but also does not have any other character for that function. So ? lArr can be used for 'is implied by' as ISOtech suggests
				entities['8657'] = '&uArr;';	// Upward double arrow
				entities['8658'] = '&rArr;';	// Rightward double arrow. ISO 10646 does not say this is the 'implies' character but does not have another character with this function so ? rArr can be used for 'implies' as ISOtech suggests
				entities['8659'] = '&dArr;';	// Downward double arrow
				entities['8660'] = '&hArr;';	// Left-right double arrow

				// Mathematical Operators
				entities['8704'] = '&forall;';	// For all
				entities['8706'] = '&part;';	// Partial differential
				entities['8707'] = '&exist;';	// There exists
				entities['8709'] = '&empty;';	// Empty set. Also known as null set and diameter
				entities['8711'] = '&nabla;';	// Nabla. Also known as backward difference
				entities['8712'] = '&isin;';	// Element of
				entities['8713'] = '&notin;';	// Not an element of
				entities['8715'] = '&ni;';		// Contains as member
				entities['8719'] = '&prod;';	// N-ary product. Also known as product sign. Prod is not the same character as U+03A0 'greek capital letter pi' though the same glyph might be used for both
				entities['8721'] = '&sum;';		// N-ary summation. Sum is not the same character as U+03A3 'greek capital letter sigma' though the same glyph might be used for both
				entities['8722'] = '&minus;';	// Minus sign
				entities['8727'] = '&lowast;';	// Asterisk operator
				entities['8729'] = '&#8729;';	// Bullet operator
				entities['8730'] = '&radic;';	// Square root. Also known as radical sign
				entities['8733'] = '&prop;';	// Proportional to
				entities['8734'] = '&infin;';	// Infinity
				entities['8736'] = '&ang;';		// Angle
				entities['8743'] = '&and;';		// Logical and. Also known as wedge
				entities['8744'] = '&or;';		// Logical or. Also known as vee
				entities['8745'] = '&cap;';		// Intersection. Also known as cap
				entities['8746'] = '&cup;';		// Union. Also known as cup
				entities['8747'] = '&int;';		// Integral
				entities['8756'] = '&there4;';	// Therefore
				entities['8764'] = '&sim;';		// tilde operator. Also known as varies with and similar to. The tilde operator is not the same character as the tilde, U+007E, although the same glyph might be used to represent both
				entities['8773'] = '&cong;';	// Approximately equal to
				entities['8776'] = '&asymp;';	// Almost equal to. Also known as asymptotic to
				entities['8800'] = '&ne;';		// Not equal to
				entities['8801'] = '&equiv;';	// Identical to
				entities['8804'] = '&le;';		// Less-than or equal to
				entities['8805'] = '&ge;';		// Greater-than or equal to
				entities['8834'] = '&sub;';		// Subset of
				entities['8835'] = '&sup;';		// Superset of. Note that nsup, 'not a superset of, U+2283' is not covered by the Symbol font encoding and is not included.
				entities['8836'] = '&nsub;';	// Not a subset of
				entities['8838'] = '&sube;';	// Subset of or equal to
				entities['8839'] = '&supe;';	// Superset of or equal to
				entities['8853'] = '&oplus;';	// Circled plus. Also known as direct sum
				entities['8855'] = '&otimes;';	// Circled times. Also known as vector product
				entities['8869'] = '&perp;';	// Up tack. Also known as orthogonal to and perpendicular
				entities['8901'] = '&sdot;';	// Dot operator. The dot operator is not the same character as U+00B7 middle dot

				// Miscellaneous Technical
				entities['8968'] = '&lceil;';	// Left ceiling. Also known as an APL upstile
				entities['8969'] = '&rceil;';	// Right ceiling
				entities['8970'] = '&lfloor;';	// left floor. Also known as APL downstile
				entities['8971'] = '&rfloor;';	// Right floor
				entities['9001'] = '&lang;';	// Left-pointing angle bracket. Also known as bra. Lang is not the same character as U+003C 'less than'or U+2039 'single left-pointing angle quotation mark'
				entities['9002'] = '&rang;';	// Right-pointing angle bracket. Also known as ket. Rang is not the same character as U+003E 'greater than' or U+203A 'single right-pointing angle quotation mark'

				// Geometric Shapes
				entities['9642'] = '&#9642;';	// Black small square
				entities['9643'] = '&#9643;';	// White small square
				entities['9674'] = '&loz;';		// Lozenge

				// Miscellaneous Symbols
				entities['9702'] = '&#9702;';	// White bullet
				entities['9824'] = '&spades;';	// Black (filled) spade suit
				entities['9827'] = '&clubs;';	// Black (filled) club suit. Also known as shamrock
				entities['9829'] = '&hearts;';	// Black (filled) heart suit. Also known as shamrock
				entities['9830'] = '&diams;';   // Black (filled) diamond suit
			}

			if (quoteStyle !== 'ENT_NOQUOTES') {
				entities['34'] = '&quot;';
			}

			if (quoteStyle === 'ENT_QUOTES') {
				entities['39'] = '&#39;';
			}

			entities['60'] = '&lt;';
			entities['62'] = '&gt;';

			// ascii decimals to real symbols
			var decimal;
			var hashMap = {};
			for (decimal in entities) {
				hashMap[String.fromCharCode(decimal)] = entities[decimal];
			}

			return hashMap;
		},

		/**
		 * Function will decode HTML entities in the string.
		 * @param {String} content string that should be decoded.
		 * @param {String} quoteStyle options are ENT_COMPAT, ENT_NOQUOTES or ENT_QUOTES.
		 * @return {String} decoded string.
		 */
		entityDecode : function(content, quoteStyle)
		{
			var hashMap = Zarafa.core.HTMLParser.getHTMLTranslationTable('HTML_ENTITIES', quoteStyle);
			var symbol = '';
			var entity = '';

			for (symbol in hashMap) {
				entity = hashMap[symbol];
				content = content.split(entity).join(symbol);
			}
			content = content.split('&#039;').join('\'');

			return content;
		},

		/**
		 * Function will encode HTML entities in the string.
		 * @param {String} content string that should be encoded.
		 * @param {String} quoteStyle options are ENT_COMPAT, ENT_NOQUOTES or ENT_QUOTES.
		 * @return {String} decoded string.
		 */
		entityEncode : function(content, quoteStyle)
		{
			var hashMap = Zarafa.core.HTMLParser.getHTMLTranslationTable('HTML_ENTITIES', quoteStyle);
			var symbol = '';
			var entity = '';

			for (symbol in hashMap) {
				entity = hashMap[symbol];
				content = content.split(symbol).join(entity);
			}
			content = content.split('\'').join('&#039;');

			return content;
		},

		/**
		 * Function to convert from Outlook inline attachment format (src="cid:...") to img tag
		 * that will display the image (by querying download_attachment.php)
		 * Builds up the query string for asking the image, using store and record entryid, and content id
		 * @param {String} body The message body to be modified
		 * @param {String} storeEntryId The store entryid
		 * @param {String} entryId The message entryid
		 * @param {Array} attachNum Attachment number of the attachment that should be downloaded.
		 * When accessing embedded messages this array can contain multiple elements indicating
		 * attachment numbers at each level, So value [0, 1] will indicate we want to download
		 * second attachment of first embedded message.
		 * @return {String} the modified HTML body
		 */
		inlineImgOutlookToZarafa : function(body, storeEntryId, entryId, attachNum)
		{
			var cidToUrl = function(match, srcStart, imgCid, srcEnd, offset, str) {
				if(imgCid) {
					var url = container.getBaseURL();
					url = Ext.urlAppend(url, 'load=download_attachment');
					url = Ext.urlAppend(url, 'sessionid=' + container.getUser().getSessionId());
					url = Ext.urlAppend(url, 'attachCid=' + imgCid);
					url = Ext.urlAppend(url, 'store=' + storeEntryId);
					url = Ext.urlAppend(url, 'entryid=' + entryId);
					url = Ext.urlAppend(url, 'contentDispositionType=inline');
					if(!Ext.isEmpty(attachNum)) {
						for (var i = 0; i< attachNum.length; i++)
						{
							url = Ext.urlAppend(url, 'attachNum[]=' + attachNum[i]);
						}
					}

					// return our own url for getting inline attachments
					return srcStart + url + srcEnd;
				}

				// return match as it is but in a real world this is not going to happen
				return match;
			};

			// replace cid: with our own url to get inline attachment
			return body.replace(cidToUrlRe, cidToUrl);
		},

		/**
		 * Function to convert from Zarafa inline img format to Outlook format (src="cid:...")
		 * Grabs the cid from the img tag (src="...attachCid=...")
		 * @param {String} body the message body to be modified
		 * @return {String} the modified html body
		 */
		inlineImgZarafaToOutlook : function(body)
		{
			var urlToCid = function(match, srcStart, imgCid, srcEnd, offset, str) {
				if(imgCid) {
					// return img src with just cid: tag
					return srcStart + 'cid:' + imgCid +  srcEnd;
				}

				// return match as it is but in a real world this is not going to happen
				return match;
			};

			// replace our own url with cid:
			return body.replace(urlToCidRe, urlToCid);
		}
	};
})();
Ext.namespace('Zarafa.core');

/**
 * @class Zarafa.core.KeyMap
 * @extends Ext.KeyMap
 * 
 * This class extends {@link Ext.KeyMap KeyMap} to assign component with key events
 * so we can pass component as a parameter while calling callback function, to make
 * key event handling easier.
 */
Zarafa.core.KeyMap = Ext.extend(Ext.KeyMap, {
	/**
	 * @cfg {Ext.Component} component The component on which key event is registered.
	 */
	component : undefined,

	/**
	 * @constructor
	 * @param {Ext.Component} component The component which should be used to pass in handlers and additionally
	 * if element is not passed then get it from component.
	 * @param {Object} config The configuration object (see {@link #addBinding})
	 * @param {Ext.Element} element {optional} The element on which this keymap will be bound.
	 * @param {String} eventName (optional) The event to bind to (defaults to "keydown")
	 */
	constructor : function(component, config, element, eventName)
	{
		if(component) {
			// store reference of component in keymap that will be passed with call to handlers
			this.component = component;
		}

		// if no element is passed then get it from component
		if(!element) {
			element = component.getEl();
		}

		// if element is passed as dom node then find its corresponding Ext.Element object
		element = Ext.get(element);

		Zarafa.core.KeyMap.superclass.constructor.call(this, element, config, eventName);
	},

	/**
	 * Function overrides {@link Ext.KeyMap#addBinding addBinding} to <br>
	 * 1) add component in callback function parameter for which event is fired. <br>
	 * 2) to solve the problem of {@link Ext.KeyMap} that, When registering two different key bindings to
	 * same {@link Ext.KeyMap} object, we can't set 'stopEvent: false' in any of the bindings <br>
	 * for more info - http://www.sencha.com/forum/showthread.php?265199-Ext.KeyMap-problems-with-stopEvent-flag <br>
	 * 3) additionally this can accept some more config options which is used in {@link Zarafa.core.KeyMapMgr KeyMapMgr},
	 * <pre>
Property           Type        Description
----------         ---------   ----------------------------------------------------------------------
enableGlobally     Boolean     A flag to indicate the key binding should also be registered with the body of webapp so that key combination can globally be disabled
settingsCfg        Object      Object containing two keys:
                                    1) description - description of the key combination to show in keyboard settings widget
                                    2) category - name of the category in which this key combination will be added
</pre>
	 */
	addBinding : function(config)
	{
		if(Ext.isArray(config)){
			Ext.each(config, function(c){
				this.addBinding(c);
			}, this);
			return;
		}
		var keyCode = config.key,
			fn = config.fn || config.handler,
			scope = config.scope,
			stopEvent = config.stopEvent;

		if(!Ext.isDefined(stopEvent)) {
			stopEvent = this.stopEvent;
		}

		if(typeof keyCode == "string"){
			var ks = [];
			var keyString = keyCode.toUpperCase();
			for(var j = 0, len = keyString.length; j < len; j++){
				ks.push(keyString.charCodeAt(j));
			}
			keyCode = ks;
		}
		var keyArray = Ext.isArray(keyCode);
		
		var handler = function(e){
			if(this.checkModifiers(config, e)){
				var k = e.getKey();

				if(keyArray){
					for(var i = 0, len = keyCode.length; i < len; i++){
						if(keyCode[i] == k){
							if(stopEvent){
								e.stopEvent();
							}
							fn.call(scope || window, k, e, this.component);
							return;
						}
					}
				}else{
					if(k == keyCode){
						if(stopEvent){
							e.stopEvent();
						}
						fn.call(scope || window, k, e, this.component);
					}
				}
			}
		};
		this.bindings.push(handler);
	}
});Ext.namespace('Zarafa.core');

/**
 * @class Zarafa.core.MessageClass
 *
 * Special utility class for manipulation and comparisons
 * of Message Class Strings
 * @singleton
 */
Zarafa.core.MessageClass = {

	/**
	 * Check if the given messageclass property on this record matches the given 'expectedClass'.
	 * This comparison is done in a case-insensite way. This function can be used
	 * for comparing the value of 'message_class' safely.
	 *
	 * @param {String} className The className which should be checked,
	 * @param {String|Array} expectedClass The expected class against which the className
	 * must be compared
	 * @param {Boolean} baseOnly (optional) Only compare the start of the className,
	 * this can be used for partial comparison (e.g. record.isClass('message_class', 'IPM', true)
	 * will return true when the actual message_class is 'IPM.Note'). Defaults to false.
	 * @return {Boolean} True when the className matches.
	 */
	isClass : function(className, expectedClass, baseOnly)
	{
		if (Ext.isEmpty(className)) {
			return false;
		}

		// Convert the className to uppercase
		className = className.toUpperCase();

		// If the expectedClass is an array, we check if the className
		// is either one of the names in expectedClas
		if (Ext.isArray(expectedClass)) {
			for (var i = 0, len = expectedClass.length; i < len; i++) {
				if (this.isClass(className, expectedClass[i], baseOnly)) {
					return true;
				}
			}
			return false;
		}

		// If the expectedClass length is larger then the className,
		// then it can never be equal.
		if (expectedClass.length > className.length) {
			return false;
		}

		// If baseOnly is true, we only want to match expectedClass against
		// the start of the className. Although we can use className.search()
		// it is faster to create a substring to ensure className and expectedClass
		// have the same length.
		if (baseOnly === true) {
			className = className.substring(0, expectedClass.length);
		}

		// Now the entire string must match
		return className == expectedClass.toUpperCase();
	},

	/**
	 * Test if the messageClass is compatible with the give containerClass.
	 * An IPM.Note is for example compatible with IPF.Note containerClass.
	 *
	 * @param {String} messageClass The Message Class to compare
	 * @param {String} containerClass The Container Class to compare
	 * @return {Boolean} True when the messageClass is compatible with the given containerClass
	 */
	isContainerClassCompatible : function(messageClass, containerClass)
	{
		messageClass = messageClass.toUpperCase();
		containerClass = containerClass.toUpperCase();

		if (Ext.isEmpty(containerClass)) {
			return true;
		}

		switch (messageClass) {
			case 'IPM.APPOINTMENT':
			case 'IPM.OLE.CLASS.{00061055-0000-0000-C000-000000000046}':
				return Zarafa.core.ContainerClass.isClass(containerClass, 'IPF.Appointment', true);
			case 'IPM.STICKYNOTE':
				return Zarafa.core.ContainerClass.isClass(containerClass, 'IPF.StickyNote', true);
			case 'IPM.CONTACT':
			case 'IPM.DISTLIST':
				return Zarafa.core.ContainerClass.isClass(containerClass, 'IPF.Contact', true);
			case 'IPM.TASK':
				return Zarafa.core.ContainerClass.isClass(containerClass, 'IPF.Task', true);
			case 'IPM.NOTE':
			case 'REPORT.IPM':
			case 'IPM.SCHEDULE':
			case "IPM.TASKREQUEST":
				// These are all normal messages, so those should be in IPF.Note containers,
				// however the IPF.Note.OutlookHomepage is special (as it is used for RSS feeds),
				// and should thus be excluded.
				return this.isClass(containerClass, 'IPF.Note', true) && !this.isClass(containerClass, 'IPF.Note.OutlookHomepage', true);
		}

		// Our fallthrough, the switch didn't catch any message class,
		// so perhaps we should cut of everything after the last dot (.)
		// and then we just retry by calling this function recursively.
		var index = messageClass.lastIndexOf('.');
		if (index > 0) {
			messageClass = messageClass.substr(0, index);
			return this.isContainerClassCompatible(messageClass, containerClass);
		}

		return false;
	},

	/**
	 * Function will return default foldertype from the hierarchy
	 * for the supplied message_class to the function.
	 * 
	 * @param {String} messageClass The message_class of the mapi record.
	 * @return {String} The foldertype of the default folder for the supplied message_class
	 */
	getDefaultFolderTypeFromMessageClass : function(messageClass)
	{
		messageClass = messageClass.toUpperCase();

		switch (messageClass) {
			case 'IPM.APPOINTMENT':
			case 'IPM.OLE.CLASS.{00061055-0000-0000-C000-000000000046}':
				return 'calendar';
			case 'IPM.STICKYNOTE':
				return 'note';
			case 'IPM.CONTACT':
			case 'IPM.DISTLIST':
				return 'contact';
			case 'IPM.TASK':
				return 'task';
			case 'IPM.NOTE':
			case 'REPORT.IPM':
			case 'IPM.SCHEDULE':
			case "IPM.TASKREQUEST":
				return 'inbox';
		}

		// Our fallthrough, the switch didn't catch any message class,
		// so perhaps we should cut of everything after the last dot (.)
		// and then we just retry by calling this function recursively.
		var index = messageClass.lastIndexOf('.');
		if (index > 0) {
			messageClass = messageClass.substr(0, index);
			return this.getDefaultFolderTypeFromMessageClass(messageClass);
		}

		return '';
	}
};
Ext.namespace('Zarafa.core');

/**
 * @class Zarafa.core.ModuleNames
 * List of module names. Each module represents a server side component that can process actions. A list module
 * for instance will allow 'list' actions.
 * @singleton
 */
Zarafa.core.ModuleNames = 
{
	/**
	 * Module information for the Address book.  
	 * @property
	 * @type Mixed
	 */
	'ADDRESSBOOK' : {
		list : 'addressbooklistmodule',
		item : 'addressbookitemmodule'
	},
	
	/**
	 * Module information for IPM.Appointment
	 * @property
	 * @type Mixed
	 */
	'IPM.APPOINTMENT' : {
		list : 'appointmentlistmodule',
		item : 'appointmentitemmodule'
	},

	/**
	 * Module information for Appointment recurrence exceptions
	 * @property
	 * @type Mixed
	 */
	'IPM.OLE.CLASS.{00061055-0000-0000-C000-000000000046}' : {
		list : 'appointmentlistmodule',
		item : 'appointmentitemmodule'
	},

	/**
	 * Module information for IPM.Task
	 * @property
	 * @type Mixed
	 */
	'IPM.TASK' : {
		list : 'tasklistmodule',
		item : 'taskitemmodule'
	},

	/**
	 * contact list module.  
	 * @property
	 * @type Mixed
	 */
	'IPM.CONTACT' : {
		list : 'contactlistmodule',
		item : 'contactitemmodule'
	},

	/**
	 * Distribution list module.
	 * Here we use same module as contact module.
	 * @property
	 * @type Mixed
	 */
	'IPM.DISTLIST' : {
		list : 'contactlistmodule',
		item : 'contactitemmodule'
	},
	
	/**
	 * Module information for hierarchy.
	 * The hierarchy is the set of stores a user can see, and the folders within those
	 * stores.   
	 * @property
	 * @type Mixed
	 */
	'HIERARCHY' : {
		list : 'hierarchymodule',
		item : 'hierarchymodule'
	},

	/**
	 * Module information for settings. 
	 * stores.   
	 * @property
	 * @type Mixed
	 */
	'SETTINGS' : {
		list : 'settingsmodule'
	},

	/**
	 * Module information for IPM.Note
	 * @property
	 * @type Mixed
	 */
	'IPM.NOTE' : {
		list : 'maillistmodule',
		item : 'createmailitemmodule'
	},

	/**
	 * Module information for IPM.StickyNote
	 * @property
	 * @type Mixed
	 */
	'IPM.STICKYNOTE' : {
		list : 'stickynotelistmodule',
		item : 'stickynoteitemmodule'
	},

	/**
	 * Module information for freebusy
	 * @property
	 * @type Mixed
	 */
	'FREEBUSY' : {
		list : 'freebusymodule'
	},

	/**
	 * Module information for busytime
	 * @property
	 * @type Mixed
	 */
	 'BUSYTIME' : {
		 list : 'busytimelistmodule'
	 },

	/**
	 * Module information for freebusy
	 * @property
	 * @type Mixed
	 */
	'SUGGESTEMAILADDRESS' : {
		list : 'suggestemailaddressmodule',
		item : 'suggestemailaddressmodule'
	},
	
	/**
	 * Module information for reminder
	 * @property
	 * @type Mixed
	 */
	'REMINDER' : {
		list : 'reminderlistmodule',
		item : 'reminderlistmodule'
	},

	/**
	 * Module information for delegates
	 * @property
	 * @type Mixed
	 */
	'DELEGATES' : {
		list : 'delegatesmodule',
		item : 'delegatesmodule'
	},

	/**
	 * Module information for Rules
	 * @property
	 * @type Mixed
	 */
	'RULES' : {
		list : 'rulesmodule',
		item : 'rulesmodule'
	},

	/**
	 * Module information for Restore Soft Deleted Items
	 * @property
	 * @type Mixed
	 */
	'RESTOREITEMS' : {
		list : 'restoreitemslistmodule',
		item : 'restoreitemslistmodule'
	},

	/**
	 * Modul information for advance search.
	 * @property
	 * @type Mixed
	 */
	'IPM.SEARCH' : {
		list : 'advancedsearchlistmodule',
		item : 'createmailitemmodule'
	},

	/**
	 * Obtain the moduleName for an appropriate key.
	 * The key could either be a component name, or a Message Class.
	 *
	 * @param {String} key The key for which the module must be found
	 * @param {Boolean} baseOnly (optional) When the the key is not exactly
	 * matched againsts a moduleName, remove everything after the last '.',
	 * defaults to false.
	 * @return {Object} An object containing a 'list' and 'item' key,
	 * for the moduleNames for the 'list' and 'item' module respectively.
	 */
	getModule : function(key, baseOnly)
	{
		key = key.toUpperCase();

		var module = this[key];
		if (!module && baseOnly === true) {
			var index = key.lastIndexOf('.');
			if (index > 0) {
				key = key.substr(0, index);
				module = this.getModule(key, baseOnly);
			}
		}

		return module;
	},

	/**
	 * Obtain the moduleName for the Item module for the appropriate key.
	 * This uses {@link #getModule} to obtain the module, and returns the
	 * itemModulename from the result.
	 * 
	 * @param {String} key The key for which the module must be found
	 * @param {Boolean} baseOnly (optional) When the the key is not exactly
	 * matched againsts a moduleName, remove everything after the last '.',
	 * defaults to false.
	 * @return {String} The item moduleName for the requested module
	 */
	getItemName : function(key, baseOnly)
	{
		var module = this.getModule(key, baseOnly);
		if (module)
			return module.item;
	},

	/**
	 * Obtain the moduleName for the List module for the appropriate key.
	 * This uses {@link #getModule} to obtain the module, and returns the
	 * listModulename from the result.
	 * 
	 * @param {String} key The key for which the module must be found
	 * @param {Boolean} baseOnly (optional) When the the key is not exactly
	 * matched againsts a moduleName, remove everything after the last '.',
	 * defaults to false.
	 * @return {String} The list moduleName for the requested module
	 */
	getListName : function(key, baseOnly)
	{
		var module = this.getModule(key, baseOnly);
		if (module)
			return module.list;
	}
};
Ext.namespace('Zarafa.core');

/**
 * @class Zarafa.core.PingService
 * @extends Ext.util.Observable
 *
 * Ping service which will periodically ping the server to determine
 * if the HTTP and ZCP server are both available again, so the user can
 * continue working with the WebApp.
 */
Zarafa.core.PingService = Ext.extend(Ext.util.Observable, {

	/**
	 * @cfg {String} url
	 * The url used to send the requests to. defaults to zarafa.php.
	 */
	url : 'zarafa.php',

	/**
	 * @cfg {String} cmd
	 * The GET attribute to send to the server. defaults to 'ping'
	 */
	cmd : 'ping',

	/**
	 * @cfg {Object} headers
	 * The default headers to be applied to the request. defaults to
	 *      'Content-Type' => 'application/json; charset=utf-8;'
	 */
	headers : undefined,

	/**
	 * @cfg {Number} timeout The initial timeout value for the call
	 * to {@link #sendPing}. This will be incremented by {@link #getNextTimeout}.
	 */
	timeout : 1000,

	/**
	 * @cfg {Number} maxTimeout The maximum timeout value which can be used
	 * and returned by {@link #getNextTimeout}.
	 */
	maxTimeout : 300000,

	/**
	 * The DelayedTask which will be used to defer the {@link #sendPing}
	 * function call to periodically poll the server for availability.
	 * @property
	 * @type Ext.util.DelayedTask
	 * @private
	 */
	pingTask : undefined,

	/**
	 * The current timeout value for the {@link #pingTask}.
	 * This is obtained (and incremented) through {@link #getNextTimeout}.
	 * @property
	 * @type Number
	 * @private
	 */
	currentTimeout : undefined,

	/**
	 * True if the Ping has been send to the server, and the PingService
	 * is currently awaiting the response from the server.
	 * @property
	 * @type Boolean
	 * @private
	 */
	pingPending : false,

	/**
	 * @constructor
	 * @param {Object} config Configuration object
	 */
	constructor : function(config)
	{
		config = config || {};

		Ext.apply(config, {
			// Apply here instead of class prototype, to prevent
			// accidental sharing of object between all instances.
			headers : {
				'Content-Type' : 'application/json; charset=utf-8;'
			}
		});

		Ext.apply(this, config);

		this.addEvents(
			/**
			 * @event start
			 * Fired during {@link #start} to indicate that the Ping Service will start
			 * polling the server for link connectivity.
			 * @param {Zarafa.core.PingService} service This object
			 * @param {Number} timeout The timeout for the next ping which will be send
			 * to the server
			 */
			'start',
			/**
			 * @event stop
			 * Fired during {@link #stop} to indicate that the Ping Service will stop
			 * polling the server for link connectivity. This doen't imply that the
			 * Service has been stopped before or after a successful response was returned.
			 * @param {Zarafa.core.PingService} service This object
			 */
			'stop',
			/**
			 * @event send
			 * Fired during {@link #sendPing} to indicate that a new Ping request will
			 * be made to the server.
			 * @param {Zarafa.core.PingService} service This object
			 * @param {XMLHttpRequest} xhrObj The XMLHttpRequest which is send to the server
			 */
			'send',
			/**
			 * @event retry
			 * Fired when a ping request was completed, but the connectivity has failed,
			 * a new attempt will be made after a specific timeout.
			 * @param {Zarafa.core.PingService} service This object
			 * @param {Object} response The response, if any, as send by the server
			 * @param {Number} timeout The timeout after which the next ping will be send
			 */
			'retry',
			/**
			 * @event restored
			 * Fired when a ping request was completed, and the connectivity has been restored.
			 * This means connectivity is restored, but the user might no longer have an active
			 * session. No new attempts will be made after this.
			 * @param {Zarafa.core.PingService} service This object
			 * @param {Object} response The response as send by the server
			 */
			'restored'
		);

		Zarafa.core.PingService.superclass.constructor.call(this, config);

		// Instantiate the delayed task for the sendPing function
		this.pingTask = new Ext.util.DelayedTask(this.sendPing, this);
	},

	/**
	 * Start the Ping Service and schedule the first run of {@link #pingTask}.
	 */
	start : function()
	{
		// reset the current timeout
		delete this.currentTimeout;

		// Obtain a new timeout and start polling
		var timeout = this.getNextTimeout();
		this.fireEvent('start', this, timeout);
		this.pingTask.delay(timeout);
	},

	/**
	 * Stop the Ping Service and cancel the {@link #pingTask}.
	 */
	stop : function()
	{
		this.pingTask.cancel();
		this.fireEvent('stop', this);
	},

	/**
	 * Interrupt the current timeout for {@link #pingTask} and manually
	 * invoke {@link #sendPing} causing a new request to be send out right now.
	 */
	retry : function()
	{
		this.pingTask.cancel();
		this.sendPing();
	},

	/**
	 * Obtain the next timeout value for the {@link #pingTask}. If
	 * {@link #currentTimeout} is initialized this will double the value
	 * (restricted by {@link #maxTimeout}, or otherwise it will take {@link #timeout}.
	 * @return {Number} The timeout for the {@link #pingTask}
	 * @private
	 */
	getNextTimeout : function()
	{
		this.currentTimeout = this.currentTimeout ? (2 * this.currentTimeout) : this.timeout;
		this.currentTimeout = Math.min(this.maxTimeout, this.currentTimeout);
		return this.currentTimeout;
	},

	/**
	 * Send a Ping request to the configured {@link #url} with the given {@link #cmd GET action}.
	 * {@link #onStateChange} will handle the response as received from the server.
	 * @private
	 */
	sendPing : function()
	{
		// A Ping request was already send,
		// we will not send another one simultaneously.
		if (this.pingPending) {
			return;
		}

		var xmlHttpRequest = new XMLHttpRequest();

		// Open the HTTP request object
		var url = this.url;
		url = Ext.urlAppend(url, 'sessionid=' + container.getUser().getSessionId());
		url = Ext.urlAppend(url, this.cmd);
		xmlHttpRequest.open('GET', url, true);

		// Apply the headers
		for (var key in this.headers) {
			xmlHttpRequest.setRequestHeader(key, this.headers[key]);
		}

		// Register statechange callback function
		xmlHttpRequest.onreadystatechange = this.onStateChange.createDelegate(this, [ xmlHttpRequest ]);

		// Mark that the Ping request is currently pending.
		this.pingPending = true;

		// Send the request
		xmlHttpRequest.send();

		this.fireEvent('send', this, xmlHttpRequest);
	},

	/**
	 * Called by {@link XMLHttpRequest} when a response from the server has been received.
	 * This will determine if the link connection to the server has been restored or not.
	 * @param {XMLHttpRequest} xmlHttpRequest The request object
	 * @private
	 */
	onStateChange : function(xmlHttpRequest)
	{
		var response;

		// The readyState can be 4 values:
		//  0 - Object is created, but not initialized
		//  1 - Request has been opened, but send() has not been called yet
		//  2 - send() has been called, no data available yet
		//  3 - Some data has been received, responseText nor responseBody are available
		//  4 - All data has been received
		//
		// readyState 0 - 3 can be completely ignored by us, as they are only updates
		// about the current progress. Only on readyState 4, should we continue and
		// start checking for the response status.
		if (xmlHttpRequest.readyState != 4) {
			return;
		}

		// Regardless of the state, the Ping request
		// is no longer pending.
		this.pingPending = false;

		// HTTP request must have succeeded
		if (xmlHttpRequest.status !== 200) {
			this.failure();
			return;
		}

		// Depending on the response type, convert it into a data Object.
		if (xmlHttpRequest.responseText) {
			// JSON response
			response = Ext.decode(xmlHttpRequest.responseText);
		} else {
			// XML response is not supported
			this.failure();
			return;
		}

		// Determine if the response indicates that a link connection
		// exists, and call the proper handler.
		if (response.success) {
			this.restored(response);
		} else {
			this.failure(response);
		}
	},

	/**
	 * Called when the link connection has been restored. This will fire
	 * the {@link #restored} event.
	 * @param {Object} response The response as received by the server
	 * @private
	 */
	restored : function(response)
	{
		this.fireEvent('restored', this, response);
	},

	/**
	 * Called when the link connection has not been restored and will be
	 * retried later. This will fire the {@link #retry} event.
	 * @param {Object} response The response, if any, as received by the server
	 * @private
	 */
	failure : function(response)
	{
		var timeout = this.getNextTimeout();
		this.fireEvent('retry', this, response, timeout);
		this.pingTask.delay(timeout);
	}
});
Ext.namespace('Zarafa.core');

/**
 * @class Zarafa.core.Request
 * @extends Ext.util.Observable
 * 
 * Request object for asynchronous communication with the server. The request object
 * automatically serialises single or multiple action requests into corresponding
 * JSON and provides a callback system to allow asynchronous handling of the
 * response from the server.
 * <p>
 * The request object supports global events surrounding the sending of the data
 * to the server. For handling the responses it communicates directly with the
 * {@link Zarafa.core.ResponseRouter ResponseRouter} on which events are provided
 * surrounding the receiving of the Responses (as well as handling of the Notifications).
 * <p>
 * You shouldn't have to instantiate this class yourself as a global instance
 * can be obtained from the globally available {@link Zarafa.core.Container container}
 * object.
 * <p>
 * The structure of data that will be used for communication will be like
 * <pre><code>
 * zarafa : {
 *	module_name : {
 *		module_id : {
 *			action_type : {
 *				// data
 *			}
 *		}
 *	}
 * }
 * </code></pre>
 */
Zarafa.core.Request = Ext.extend(Ext.util.Observable, (function() {
	/**
	 * True if the conection has been {@link #paralyze}. This will
	 * prevent any requests from being send to the server, or any responses
	 * from being processed.
	 * @property
	 * @type Boolean
	 * @private
	 */
	var paralyzed = false;

	/**
	 * True if the Webapp has been {@link #interrupt}. This will
	 * queue all requests which will be send to the server, until the connection
	 * is restored again.
	 * @property
	 * @type Boolean
	 * @private
	 */
	var interrupted = false;

	/**
	 * The current base number for the generating new requestsIds
	 * using the {@link #getRequestId} function.
	 * @property
	 * @type Number
	 * @private
	 */
	var requestIdBase = 0;

	/**
	 * The current JSON object containing all requests which
	 * should be send to the server on the next call to
	 * {@link #send}. Will be reset on {@link #reset}.
	 * @property
	 * @type Object
	 * @private
	 */
	var zarafaTag = undefined;

	/**
	 * The flag to indicate if zarafaTag holds JSON data.
	 * This influences if the contents of zarafaTag will be encoded or not.
	 * @property
	 * @type Boolean
	 * @private
	 */
	var hasJson = false;

	/**
	 * The flag to indicate if zarafaTag holds RAW data.
	 * This influences if the contents of zarafaTag will be encoded or not.
	 * @property
	 * @type Boolean
	 * @private
	 */
	var hasData = false;

	/**
	 * The list of request ids as generated by {@link #getRequestId} of all
	 * requests which are added after a {@link #reset} and before {@link #send}.
	 * These will be merged into {@link #activeRequests} on {@link #send}.
	 * @property
	 * @type Array
	 * @private
	 */
	var queuedRequests = [];

	/**
	 * Key-value list of request ids and the corresponding {@link XMLHttpRequest} objects
	 * which was used to send the request to the server. These requests are all pending
	 * on the PHP side. These will be cleared as soon as the response has been received.
	 * As long as a request is listed here, it can be canceled using {@link #cancelActiveRequest}.
	 * @property
	 * @type Array
	 * @private
	 */
	var activeRequests = {};

	/**
	 * The list of {@link #XMLHttpRequest} objects which are queued while the
	 * {@link #interrupted connection is interrupted}. These will be dequeued
	 * later when the connection is restored.
	 * @property
	 * @type Array
	 * @private
	 */
	var queuedInterruptedHttpRequests = [];

	/**
	 * The unique id for subsystem which will be used to differentiate between two sessions of same user
	 * using same browser but in different tabs.
	 * @property
	 * @type String
	 * @private
	 */
	var subSystemId = undefined;

	return {
		// public variables
		/**
		 * @cfg {String} defaultUrl
		 * The url used to send the requests to. defaults to zarafa.php.
		 */
		defaultUrl : 'zarafa.php',

		/**
		 * @cfg {Object} defaultHeaders
		 * The default headers to be applied to the request. defaults to
		 *	'Content-Type' => 'application/json; charset=utf-8;'
		 */
		defaultHeaders : undefined,

		/**
		 * @cfg {String} subSystemPrefix string that will be used to generate session filename which
		 * will be used to store session data in php. current timestamp will be appended to this string
		 * so different tabs opened in same browser will have different session data. So every request object
		 * corresponds to one unique subSystem string id (accuracy upto miliseconds).
		 */
		subSystemPrefix : 'webapp',

		/**
		 * @cfg {Object} requestHeaders
		 * Headers that should be sent with every request.
		 */
		requestHeaders : undefined,

		/**
		 * @constructor
		 * @param {Object} config Configuration object
		 */
		constructor: function(config)
		{
			config = config || {};

			Ext.apply(config, {
				// Apply here instead of class prototype, to prevent
				// accidental sharing of object between all instances.
				defaultHeaders : {
					'Content-Type' : 'application/json; charset=utf-8;'
				},
				requestHeaders : {}
			});

			Ext.apply(this, config);

			this.addEvents(
				/**
				 * @event connectionparalyzed
				 * Fired when the window is about to be unloaded. At this moment
				 * WebApp will shutdown and start dropping all communication with the PHP
				 * @param {Zarafa.core.Request} request 
				 * @param {Zarafa.core.data.ParalyzeReason} reason The reason to paralyze the WebApp
				 */
				'connectionparalyzed',
				/**
				 * @event connectioninterrupted
				 * Fired when the connection to the server has been lost, but is likely
				 * to be restored soon. WebApp will start pausing all communication with
				 * the PHP until the connection has been restored.
				 * @param {Zarafa.core.Request} request 
				 * @param {Zarafa.core.PingService} service The ping service which will
				 * be used to ping for the connection restoration.
				 */
				'connectioninterrupted',
				/**
				 * @event connectionrestored
				 * Fired when the connection to the server has been restored. This will
				 * resend any requests to the PHP which were scheduled while the connection
				 * was lost.
				 * @param {Zarafa.core.Request} request 
				 */
				'connectionrestored',
				/**
				 * @event beforesend
				 * Fires before the XmlHttpRequest sends a request to the server.
				 * Return false to not send request to server.
				 * @param {Zarafa.core.Request} request 
				 * @param {window.XMLHttpRequest} xmlHttpRequest XmlHttpRequest object
				 */
				'beforesend',
				/**
				 * @event aftersend
				 * Fires after the XmlHttpRequest sent a request to the server.
				 * @param {Zarafa.core.Request} request 
				 * @param {window.XMLHttpRequest} xmlHttpRequest XmlHttpRequest object
				 */
				'aftersend'
			);

			Zarafa.core.Request.superclass.constructor.call(this, config);

			// Initialize all private variables
			this.initialize();
		},

		/**
		 * Initialize all private variables of the Request object
		 * @private
		 */
		initialize : function()
		{
			paralyzed = false;
			interrupted = false;
			requestIdBase = 0;
			zarafaTag = undefined;
			hasJson = false;
			hasData = false;
			queuedRequests = [];
			activeRequests = {};
			queuedInterruptedHttpRequests = [];
			subSystemId = this.subSystemPrefix + '_' + new Date().getTime();
		},

		/**
		 * Set the {@link #paralyzed} property which will prevent any requests from
		 * being send out to the server. This will fire the {@link #connectionparalyzed} event.
		 * @param {Zarafa.core.data.ParalyzeReason} reason The reason to paralyze the WebApp
		 */
		paralyze : function(reason)
		{
			if (this.isParalyzed()) {
				return;
			}

			paralyzed = true;
			this.fireEvent('connectionparalyzed', this, reason);
		},

		/**
		 * @return {Boolean} True if the connection is {@link #paralyzed}.
		 */
		isParalyzed : function()
		{
			return paralyzed;
		},

		/**
		 * Set the {@link #interrupted} property which will pause all requests to be send out
		 * to the server. These will be send when the connection is {@link #restore restored}.
		 * @private
		 */
		interrupt : function()
		{
			if (this.isInterrupted() || this.isParalyzed()) {
				return;
			}

			interrupted = true;

			// Instantiate the PingService for polling the
			// server for the recovery of the connection.
			var service = new Zarafa.core.PingService({
				url : this.defaultUrl,
				headers : this.defaultHeaders
			});

			// Add event handler which will restore interaction with
			// the PHP server when the link has been restored.
			service.on('restored', this.restore, this);

			// Add some event handlers which will stop the PingService
			// as soon as the connection is restored, or if the WebApp
			// is being paralyzed. We defer the connectionrestored by 1 ms,
			// so we force the 'stop' event from the service to be fired after
			// all 'restored' event handlers from the service have been handled.
			this.on('connectionparalyzed', service.stop, service);
			this.on('connectionrestored', service.stop, service, { delay : 1 });

			// Fire the event, allow interested parties to register
			// for events on the service before we start it.
			this.fireEvent('connectioninterrupted', this, service);

			// start the service
			service.start();
		},

		/**
		 * Clear the {@link #interrupted} property which will restore all requests which were
		 * paused while the connection was interrupted.
		 * @param {Zarafa.core.PingService} service This object
		 * @param {Object} response The response as send by the server
		 * @private
		 */
		restore : function(service, response)
		{
			if (!this.isInterrupted()) {
				return;
			}

			// If the response object indicates that our session
			// is no longer active, we must paralyze the WebApp
			// as no further requests can be send to the server.
			if (response && response.active === false) {
				this.paralyze(Zarafa.core.data.ParalyzeReason.SESSION_EXPIRED);
				return;
			} else {
				interrupted = false;
				this.fireEvent('connectionrestored', this);

				// If there was a HTTP request queued, we should
				// send the first one now. The next one will be
				// send when the first one has responded. (This prevents
				// a DDOS if the connection was lost for a long time,
				// and many clients are connected to the server).
				if (this.hasQueuedHttpRequests()) {
					this.dequeueHttpRequest();
				}
			}
		},

		/**
		 * @return {Boolean} True if the connection is {@link #interrupted}.
		 */
		isInterrupted : function()
		{
			return interrupted;
		},

		/**
		 * Sets request headers that will be sent with every request made through {@link Zarafa.core.Request} object.
		 * @param {String} key header key.
		 * @param {String} value header value.
		 */
		setRequestHeader: function(key, value)
		{
			this.requestHeaders[key] = value;
		},

		/**
		 * Generates a new unique Request ID, which can be used to match
		 * the request with a response.
		 * @param {String} prefix (optional) The prefix for the unique request id.
		 * @return {String} request ID
		 */
		getRequestId : function(prefix)
		{
			return (prefix || 'z-gen') + (++requestIdBase);
		},

		/**
		 * Function will be used to cancel a previously generated request which is waiting for the server request.
		 * When doing multiple requests we need to make sure that previous requests are cancelled otherwise data 
		 * can be overwritten by previous requests. Special requirement was that we can't do {@link window.XMLHttpRequest#abort}
		 * as it will abort the whole request and discard data that was added to response by server side, which is not
		 * good when server has added any notification data to response (like new mail), so we will need to overwrite
		 * responseHandler of this particular request with {@link Zarafa.core.data.DummyResponseHandler}, which will
		 * have empty functions for discarding response came from server and notifications will be handled normally.
		 * @param {String} requestId The request id.
		 */
		cancelActiveRequest : function(requestId)
		{
			var responseRouter = container.getResponseRouter();

			// If the request is still queued, we have an opportunity to
			// prevent the entire request. However this is only possible
			// if no other request was queued, as then we can cancel all the
			// pending data. If that is not the case, we have to register the
			// DummyResponseHandler
			if (queuedRequests.indexOf(requestId) >= 0) {
				if (queuedRequests.length === 1) {
					// Reset environment
					this.reset();
					responseRouter.removeRequestResponseHandler(requestId);
				} else {
					responseRouter.addRequestResponseHandler(requestId, new Zarafa.core.data.DummyResponseHandler());
				}
				return;
			}

			// If there is a link interruption, then the request might still be queued.
			// This means we can interrupt the request before it is being send to the
			// server.
			if (this.hasQueuedHttpRequests()) {
				// Search through the list of pending HTTP requests
				// to find the one which holds our cancelled request id.
				for (var i = 0; i < queuedInterruptedHttpRequests.length; i++) {
					var xhrObj = queuedInterruptedHttpRequests[i];
					var index = xhrObj.queuedRequests.indexOf(requestId);

					if (index >= 0) {
						// We found the request, but now comes the problem.
						// We can only cancel the entire HTTPRequest if this
						// was the only request on the object. Because we don't
						// know which part of the JSON corresponds to this request.
						if (xhrObj.queuedRequests.length === 1) {
							queuedInterruptedHttpRequests.splice(i, 1);
							responseRouter.removeRequestResponseHandler(requestId);
							return;
						}

						// If we can't cancel the request, we can break
						// the loop and just change the response handler.
						break;
					}
				}

				// We didn't find the request to which the request corresponds,
				// or we couldn't cancel the entire request. In either case,
				// we have to register the DummyResponseHandler.
				responseRouter.addRequestResponseHandler(requestId, new Zarafa.core.data.DummyResponseHandler());
				return;
			}

			// If the request has been send out to the server, we cannot cancel
			// it and thus we will have to register the DummyResponseHandler to
			// prevent the response from being processed. However we can mark
			// the corresponding xmlHttpRequest so we will not retry it if all
			// requests have been cancelled.
			var xhrObj = activeRequests[requestId];
			if (Ext.isDefined(xhrObj)) {
				// Only prevent the retry if all requests attached
				// to the xmlHttpRequest have been cancelled.
				if (xhrObj.queuedRequests.length === 1) {
					xhrObj.preventRetry = true;
				}
				responseRouter.addRequestResponseHandler(requestId, new Zarafa.core.data.DummyResponseHandler());
				return;
			}
		},

		/**
		 * Queue a request to {@link #queuedRequests}. This can be dequeued
		 * in {@link #dequeueRequests}.
		 * @param {String} The request to be queued
		 * @private
		 */
		queueRequest : function(request)
		{
			queuedRequests.push(request);
		},

		/**
		 * Obtain and dequeue any requests from {@link #queuedRequests}. Use
		 * {@link #activateRequests} to add them the the {@link #activeRequests} array
		 * once the request has been send out.
		 * @return {Array} The array of previously queued requests
		 * @private
		 */
		dequeueRequests : function()
		{
			var requests = queuedRequests;
			queuedRequests = [];
			return requests;
		},

		/**
		 * Move all requests into {@link #activeRequests}. The provided requests
		 * should previously have been obtained through {@link #dequeueRequests}.
		 * @param {Array} requests The list of requests to activate.
		 * @private
		 */
		activateRequests : function(requests, xmlHttpRequest)
		{
			for (var i = 0; i < requests.length; i++) {
				activeRequests[requests[i]] = xmlHttpRequest;
			}
		},

		/**
		 * Remove all given requests from the {@link #activeRequests} queue.
		 * @param {Array} requests The list of requests to complete
		 * @private
		 */
		completeRequests : function(requests)
		{
			for (var i = 0, len = requests.length; i < len; i++) {
				delete activeRequests[requests[i]];
			}
		},

		/**
		 * Prepare a {@link XMLHttpRequest} object which will be used to transmit data to the server for processing.
		 * it also registers a callback function for onreadystatechange event that will be called when server sends response.
		 * @param {Mixed} requestData data that will be send using this request object, for XML request it
		 * will be XML document or else it will be JSON string.
		 * @param {Boolean} encoded True of the requestData is encoded, and must be decoded to be used.
		 * @param {String} url (optional) url to post to. Defaults to {@link #defaultUrl}
		 * @param {Object} headers (optional) headers to apply to the request. Defaults to {@link #defaultHeaders}
		 * @return {XMLHttpRequest} The initialized XMLHttpRequest 
		 * @private
		 */
		prepareHttpRequest : function(requestData, encoded, url, headers)
		{
			var xmlHttpRequest = new XMLHttpRequest();

			if (!url) {
				url = this.defaultUrl;
				url = Ext.urlAppend(url, 'sessionid=' + container.getUser().getSessionId());
				url = Ext.urlAppend(url, 'subsystem=' + subSystemId);
			}

			xmlHttpRequest.open('POST', url, true);
			xmlHttpRequest.requestUrl = url;

			// Apply header from argument, or apply the default headers
			headers = Ext.apply({}, this.requestHeaders, headers || this.defaultHeaders);
			for (var key in headers) {
				xmlHttpRequest.setRequestHeader(key, headers[key]);
			}
			xmlHttpRequest.requestHeaders = headers;

			// Store requestData into the xmlHttpRequest, when the request failed,
			// we can still determine report the failure back to the requestee.
			xmlHttpRequest.requestData = requestData;
			xmlHttpRequest.requestDataEncoded = encoded;
			xmlHttpRequest.queuedRequests = this.dequeueRequests();

			return xmlHttpRequest;
		},

		/**
		 * Clone a {@link XMLHttpRequest} instance which was prepared by {@link #prepareHttpRequest} and
		 * has previously been attempted to be {@link #sendHttpRequest send out} but failed due to a broken
		 * connection. This will clone the instance, so the request can be retried when the connection
		 * returns.
		 * @param {XMLHttpRequest} xmlHttpRequest The request object to clone
		 * @return {XMLHttpRequest} The cloned XMLhttpRequest object
		 * @private
		 */
		cloneRequest : function(xmlHttpRequest)
		{
			var cloneRequest = new XMLHttpRequest();
			cloneRequest.open('POST', xmlHttpRequest.requestUrl, true);
			cloneRequest.requestUrl = xmlHttpRequest.requestUrl;

			var headers = xmlHttpRequest.requestHeaders;
			for (var key in headers) {
				cloneRequest.setRequestHeader(key, headers[key]);
			}
			cloneRequest.requestHeaders = headers;

			cloneRequest.requestData = xmlHttpRequest.requestData;
			cloneRequest.requestDataEncoded = xmlHttpRequest.requestDataEncoded;
			cloneRequest.queuedRequests = xmlHttpRequest.queuedRequests;

			return cloneRequest;
		},

		/**
		 * Send out a {@link XMLHttpRequest} which was initialized by {@link #prepareHttpRequest}. This will
		 * perform the final step for transmitting the request, by firing the {@link #beforesend} event,
		 * and {@link #activeRequests activating the queued requests}.
		 * @param {XMLHttpRequest} xmlHttpRequest The request object
		 * @private
		 */
		sendHttpRequest : function(xmlHttpRequest)
		{
			var requestData = xmlHttpRequest.requestData;
			var requests = xmlHttpRequest.queuedRequests;

			// When the connection paralyzed, we cannot handle any incoming data anymore.
			// So all outgoing requests will be dropped.
			if (this.isParalyzed()) {
				return;
			}

			if (this.fireEvent('beforesend', this, xmlHttpRequest) === false) {
				return;
			}

			// Register the onready StateChange event handler
			xmlHttpRequest.onreadystatechange = this.stateChange.createDelegate(this, [xmlHttpRequest]);

			// Move the queued requests to the active list.
			this.activateRequests(requests, xmlHttpRequest);

			// send request
			xmlHttpRequest.send(requestData);

			this.fireEvent('aftersend', this, xmlHttpRequest);
		},

		/**
		 * {@link #queuedInterruptedHttpRequests Queue} a {@link XMLHttpRequest} which was initialized by
		 * {@link #prepareHttpRequest}. This will keep the request instance until it can be send
		 * to the server at a {@link #dequeueHttpRequest later time}.
		 * @param {XMLHttpRequest} xmlHttpRequest The request object
		 * @private
		 */
		queueHttpRequest : function(xmlHttpRequest)
		{
			queuedInterruptedHttpRequests.push(xmlHttpRequest);
		},

		/**
		 * Check if there are queued {@link #queuedInterruptedHttpRequests HTTP requests}
		 * @return {Boolean} True if there are queued HTTP requests
		 * @private
		 */
		hasQueuedHttpRequests : function()
		{
			return !Ext.isEmpty(queuedInterruptedHttpRequests);
		},

		/**
		 * Dequeue a {@link #queuedInterruptedHttpRequests HTTP request} and {@link #sendHttpRequest send it} to the server
		 * @private
		 */
		dequeueHttpRequest : function()
		{
			var xmlHttpRequest = queuedInterruptedHttpRequests.shift();
			if (xmlHttpRequest) {
				this.sendHttpRequest(xmlHttpRequest);
			}
		},

		/**
		 * Called by XMLHttpRequest when a response from the server is coming in. When the entire response has been
		 * completed it calls {@link Zarafa.core.ResponseRouter#receive receive}} which handles the rest of the process.
		 * @param {Object} xmlHttpRequest The raw HTTP request object that is used for communication.
		 * @private
		 */
		stateChange : function(xmlHttpRequest)
		{
			var requestIds = xmlHttpRequest.queuedRequests;
			var responseRouter = container.getResponseRouter();
			var response;

			// When the connection is paralyzed, we cannot handle any incoming data anymore.
			// All incoming responses from the server will be dropped.
			if (this.isParalyzed()) {
				return;
			}

			// The readyState can be 4 values:
			//  0 - Object is created, but not initialized
			//  1 - Request has been opened, but send() has not been called yet
			//  2 - send() has been called, no data available yet
			//  3 - Some data has been received, responseText nor responseBody are available
			//  4 - All data has been received
			//
			// readyState 0 - 3 can be completely ignored by us, as they are only updates
			// about the current progress. Only on readyState 4, should we continue and
			// start checking for the response status.
			if (xmlHttpRequest.readyState != 4) {
				return;
			}

			// The transaction is complete, all requests must now be cleared from the list.
			// Note that when we got a HTTP error, the requests are still removed, since
			// the HTTP request is no longer pending.
			this.completeRequests(requestIds);

			// HTTP request must have succeeded
			switch (xmlHttpRequest.status) {
				case 401: /* Unauthorized */
					// Indicate that the user is no longer logged in, and
					// he must re-authenticate. This must be done through the
					// normal logon page, so here we just paralyze the Request.
					// The exact reason for the paralyzation can be found in the
					// headers.
					var reason = xmlHttpRequest.getResponseHeader('X-Zarafa-Hresult');
					if (reason === 'MAPI_E_INVALID_WORKSTATION_ACCOUNT') {
						this.paralyze(Zarafa.core.data.ParalyzeReason.SESSION_INVALID);
					} else {
						this.paralyze(Zarafa.core.data.ParalyzeReason.SESSION_EXPIRED);
					}
					return;
				case 500: /* Internal Server Error */
					// The connection is present, if there still are queued
					// HTTP requests, we can send the next one right now.
					if (this.hasQueuedHttpRequests()) {
						this.dequeueHttpRequest();
					}

					// Indicate that the request failed
					// inside the server and exit the function.
					this.receiveFailure(xmlHttpRequest);
					return;
				case 200: /* OK */
					// The connection is present, if there still are queued
					// HTTP requests, we can send the next one right now.
					if (this.hasQueuedHttpRequests()) {
						this.dequeueHttpRequest();
					}
					break;
				default: /* Connection errors */
					// Interrupt the connection
					this.interrupt();

					// Clone the XMLHttpRequest and queue it
					// for when the connection is restored.
					if (xmlHttpRequest.preventRetry !== true) {
						var clone = this.cloneRequest(xmlHttpRequest);
						this.queueHttpRequest(clone);
					}
					return;
			}

			// Depending on the response type, convert it into a data Object.
			if (xmlHttpRequest.responseText) {
				// JSON response
				response = Ext.decode(xmlHttpRequest.responseText);
			} else {
				// XML response is not supported
				this.receiveFailure(xmlHttpRequest);
				return;
			}

			// Check for empty response, sometimes the PHP server doesn't bother with
			// responding with any data.
			if (Ext.isEmpty(response) || Ext.isEmpty(response.zarafa)) {
				this.receiveFailure(xmlHttpRequest);
				return;
			}

			responseRouter.receive(response);
		},

		/**
		 * Called when the {@link XMLHttpRequest} object indicates a problem
		 * in the returned data from the server. This will call
		 * {@link Zarafa.core.ResponseRouter#receiveFailure receiveFailure} on
		 * the {@link Zarafa.core.ResponseRouter ResponseRouter}.
		 * @param {XMLHttpRequest} xmlHttpRequest The xmlHttpRequest which contains the problem
		 * @private
		 */
		receiveFailure : function(xmlHttpRequest)
		{
			var responseRouter = container.getResponseRouter();
			var requestData = xmlHttpRequest.requestData;
			if (xmlHttpRequest.requestDataEncoded) {
				requestData = Ext.decode(requestData);
			}
			responseRouter.receiveFailure(requestData, xmlHttpRequest);
		},
	
		/**
		 * Resets the {@link Zarafa.core.Request Request} object and prepares it for
		 * accepting new calls to {@link #addRequest} or {@link #singleRequest}.
		 */
		reset : function()
		{
			queuedRequests = [];

			zarafaTag = {
				'zarafa' : {}
			};
			hasJson = false;
			hasData = false;
		},

		/**
		 * Adds a single request to the Request object. This is a combination of a module name, id and action.
		 * The parameters objected will be encoded and added into the action tag.
		 * The callbacks are used when the responds for this specific request has returned.
		 * @param {String} moduleName name of the module to communicate with (i.e. 'addressbooklistmodule')
		 * @param {String} actionType action to perform (i.e. 'list' or 'globaladdressbook')
		 * @param {Object} actionData data that will included in the action tag (i.e. { restriction : { name : 'piet' } })
		 * @param {Zarafa.core.data.AbstractResponseHandler} responseHandler The response handler which must be
		 * used for handling the responds for this specific request.
		 * @return {String} The unique Request ID which was assigned to this request.
		 */
		addRequest : function(moduleName, actionType, actionData, responseHandler)
		{
			if (Ext.isEmpty(zarafaTag)) {
				throw 'Request object not initialised. Call reset() first';
			}
			if (hasData) {
				throw 'Request object initialized with RAW data';
			}

			var requestId = this.getRequestId(moduleName);
			this.queueRequest(requestId);

			if (Ext.isDefined(responseHandler)) {
				container.getResponseRouter().addRequestResponseHandler(requestId, responseHandler);
			}

			actionData = Ext.value(actionData, {});

			// create new module tag if not present
			if (!Ext.isDefined(zarafaTag.zarafa[moduleName])) {
				zarafaTag.zarafa[moduleName] = {};
			}

			// create new module id tag if not present
			if (!Ext.isDefined(zarafaTag.zarafa[moduleName][requestId])) {
				zarafaTag.zarafa[moduleName][requestId] = {};
			}

			// add action data
			zarafaTag.zarafa[moduleName][requestId][actionType] = actionData;

			// JSON data was added
			hasJson = true;

			return requestId;
		},

		/**
		 * Adds a single request to the Request object. Opposed to {@link #addRequest} this will be raw data.
		 * The paramenters object will be placed to the action tag.
		 * The callbacks are used when the responds for this specific request has returned.
		 * @param {String} moduleName name of the module to communicate with (i.e. 'addressbooklistmodule')
		 * @param {String} actionType action to perform (i.e. 'list' or 'globaladdressbook')
		 * @param {Object} actionData data that will included in the action tag
		 * @param {Zarafa.core.data.AbstractResponseHandler} responseHandler The response handler which must be
		 * used for handling the responds for this specific request.
		 * @return {String} The unique Request ID which was assigned to this request.
		 */
		addDataRequest : function(moduleName, actionType, actionData, responseHandler)
		{
			if (Ext.isEmpty(zarafaTag)) {
				throw 'Request object not initialised. Call reset() first';
			}
			if (hasJson) {
				throw 'Request object intitialized with JSON data';
			}
			if (hasData) {
				throw 'Request object already contains RAW data';
			}

			var requestId = this.getRequestId(moduleName);
			this.queueRequest(requestId);

			if (Ext.isDefined(responseHandler)) {
				container.getResponseRouter().addRequestResponseHandler(requestId, responseHandler);
			}

			// add action data
			zarafaTag = actionData;

			// Raw data was added
			hasData = true;

			return requestId;
		},

		/**
		 * Calls {@link #prepareHttpRequest} to actually create the request using passed data,
		 * and calls {@link #sendHttpRequest} to send request.
		 * @param {String} url (optional) url to post to. Defaults to {@link #defaultUrl}
		 * @param {Object} headers (optional) headers to apply to the request. Defaults to {@link #defaultHeaders}
		 */
		send : function(url, headers)
		{
			if (Ext.isEmpty(zarafaTag)) {
				throw 'Request object not initialised. Call reset() first';
			}

			if (Ext.isEmpty(queuedRequests)) {
				throw 'No requests have been added. Use addRequest()';
			}

			if (!this.isParalyzed()) {
				var xmlHttpRequest = this.prepareHttpRequest(hasJson ? Ext.encode(zarafaTag) : zarafaTag, hasJson, url, headers);
				if (this.isInterrupted() || this.hasQueuedHttpRequests()) {
					this.queueHttpRequest(xmlHttpRequest);
				} else {
					this.sendHttpRequest(xmlHttpRequest);
				}
			}

			// All data has been send, reset the zarafaTag to
			// prevent sending the same request twice.
			zarafaTag = undefined;
			hasJson = false;
			hasData = false;
		},

		/**
		 * Convenience method for performing a single JSON request to the server.
		 * It calls {@link #reset reset()}, {@link #addRequest addRequest()}, and {@link #send send()} in turn.
		 * @param {String} moduleName name of the module to communicate with (i.e. 'addressbooklistmodule')
		 * @param {String} actionType action to perform (i.e. 'list' or 'globaladdressbook')
		 * @param {Object} actionData data that will included in the action tag (i.e. { restriction : { name : 'piet' } })
		 * @param {Zarafa.core.data.AbstractResponseHandler} responseHandler The response handler which must be
		 * used for handling the responds for this specific request.
		 * @return {String} The unique request ID which was assigned to this transaction object.
		 */
		singleRequest : function(moduleName, actionType, actionData, responseHandler)
		{
			var requestId;

			this.reset();
			requestId = this.addRequest(moduleName, actionType, actionData, responseHandler);
			this.send();

			return requestId;
		}
	};
})());
Ext.namespace('Zarafa.core');

/**
 * @class Zarafa.core.ResponseRouter
 * @extends Ext.util.Observable
 *
 * The router for Responses to requests made by the {@link Zarafa.core.Request Request} object.
 * Each response is delivered to its destination {@link Ext.data.Store store} through the
 * {@link Zarafa.core.data.AbstractResponseHandler ResponseHandler}.
 * Upon recieving a response, the ResponseRouter will determine if it is a response to
 * a direct request from the {@link Zarafa.core.Request Request Object} or a notification
 * generated by the PHP-side.
 * If the response came from a request from the {@link Zarafa.core.Request Request Object} a
 * {@link Zarafa.core.data.AbstractResponseHandler ResponseHandler} will have been registered
 * by the {@link Ext.data.DataProxy Proxy} which made the request. If the
 * {@link Zarafa.core.data.AbstractResponseHandler ResponseHandler} is not available the
 * response is considered a Notification, in which case the
 * {@link Zarafa.core.data.NotificationResolver NotificationResolver} is used to generate
 * a special {@link Zarafa.core.data.AbstractNotificationResponseHandler ResponseHandler} which can
 * update the {@link Ext.data.Store stores} which contain the updated {@link Ext.data.Record records}.
 */
Zarafa.core.ResponseRouter = Ext.extend(Ext.util.Observable, {
	/**
	 * The collection of {@link Zarafa.core.data.AbstractResponseHandler ResponseHandlers}
	 * stored by using the moduleid of the outgoing request.
	 * @property
	 * @type Object
	 */
	responseHandlers : undefined,

	/**
	 * @constructor
	 * @param {Object} config Configuration object
	 */
	constructor : function(config)
	{
		config = config || {};

		Ext.applyIf(config, {
			responseHandlers : {}
		});

		this.addEvents(
			/**
			 * @event beforereceive
			 * Main event which is triggered when data has been received from
			 * the PHP server, and is about to be processed by the router.
			 * @param {Object} data The data which was received by the router.
			 */
			'beforereceive',
			/**
			 * @event afterreceive
			 * Main event which is triggered when the data which has been received from
			 * the PHP server has been processed.
			 * @param {Object} data The data which was received by the router.
			 */
			'afterreceive',
			/**
			 * @event receiveexception
			 * Main event which is triggered when a Request has failed, or the response
			 * doesn't contain sufficient data to handle it.
			 * @param {Object} requestdata The request data which was send to the server.
			 * @param {Object} xmlHttpRequest The raw browser response object.
			 */
			'receiveexception',
			/**
			 * @event response
			 * Main event which is triggered when a response has been received
			 * from the PHP server.
			 * @param {String} module The module name for this response
			 * @param {String} id The module id for this reponse
			 * @param {Object} response The response which was received.
			 * @param {Number} timestamp The {@link Date#getTime timestamp} on which the response was received
			 * @return {Boolean} False to cancel the handling of the response by ResponseHandlers
			 */
			'response'
		);

		Ext.apply(this, config);

		Zarafa.core.ResponseRouter.superclass.constructor.call(this, config);
	},

	/**
	 * Register a {@link Zarafa.core.data.AbstractResponseHandler ResponseHandler} to the
	 * Response Router. This handler will be used to handle the Response for the request
	 * with the given identifier.
	 * @param {String} id The unique request identifier on which the ResponseHandler
	 * must be registerdd.
	 * @param {Zarafa.core.data.AbstractResponseHandler} handler The ResponseHandler
	 * which must be registered for the given id.
	 */
	addRequestResponseHandler : function(id, handler)
	{
		this.responseHandlers[id] = handler;
	},

	/**
	 * Get a {@link Zarafa.core.data.AbstractResponseHandler ResponseHandler} from
	 * the {@link #responseHandlers} which has been registerd for the given module identifier.
	 * This will automatically deregister the handler to prevent it being used twice.
	 * @param {String} id The unique request identifier on which the ResponseHandler
	 * could be registered.
	 * @return {Zarafa.core.data.AbstractResponseHandler} The registered response handler.
	 * @private
	 */
	getRequestResponseHandler : function(id)
	{
		var handler = this.responseHandlers[id];
		this.removeRequestResponseHandler(id);
		return handler;
	},

	/**
	 * Removes a {@link Zarafa.core.data.AbstractResponseHandler ResponseHandler} from the
	 * {@link Zarafa.core.ResponsRouter ResponsRouter} to prevent it being called twice.
	 * @param {String} id The unique request identifier of the ResponseHandler which will be
	 * deregistered from {@link Zarafa.core.ResponsRouter ResponsRouter}.
	 */
	removeRequestResponseHandler : function(id)
	{
		delete this.responseHandlers[id];
	},

	/**
	 * Get a {@link Zarafa.core.data.AbstractResponseHandler ResponseHandler} from
	 * the {@link Zarafa.core.data.NotificationResolver NotificationResolver}. This
	 * will construct a special {@link Zarafa.core.data.AbstractResponseHandler ResponseHandler}
	 * which is dedicated to handling this specific notification.
	 * @param {String} module The module from which the notification originated.
	 * @param {Object} data The response data which was send as notification, this is used
	 * to determine which {@link Ext.data.Store stores} are affected by this notification.
	 * @private
	 */
	getNotificationResponseHandler : function(module, data)
	{
		return container.getNotificationResolver().getHandlerForResponse(module, data);
	},

	/**
	 * This will have a Response from the PHP server and will process it
	 * by delivering the Response over the configured route to the destination.
	 *
	 * @param {Object} data The data object which was received and
	 * must be processed.
	 */
	receive : function(data)
	{
		this.fireEvent('beforereceive', data);

		this.processResponse(data);

		this.fireEvent('afterreceive', data);
	},

	/**
	 * This will report a Receive failure which can be triggered when
	 * the request failed with HTTP-error (e.g. 404) or when the response data
	 * was incomplete. This will go through all
	 * {@link Zarafa.core.data.AbstractResponseHandlers ResponseHandlers} which
	 * are affected by this error to report the error using:
	 * {@link Zarafa.core.data.AbstractResponseHandlers#responseFailure responseFailure}.
	 * @param {Object} requestdata The request data which was send to the server.
	 * @param {Object} xmlHttpRequest The raw browser response object.
	 */
	receiveFailure : function(requestData, xmlHttpRequest)
	{
		this.fireEvent('receiveexception', requestData, xmlHttpRequest);

		// Without requestData we cannot report the request failure
		// back to the requestee.
		if (!Ext.isObject(requestData)) {
			return;
		}

		// Find all registered ResponseHandlers which are affected by this
		// failure and propogate the responseFailure to them, 
		Ext.iterate(requestData.zarafa, function(moduleName, modules) {
			Ext.iterate(modules, function(moduleId, moduleData) {
				var handler = this.getRequestResponseHandler(moduleId);
				if (!Ext.isEmpty(handler)) {
					handler.responseFailure(xmlHttpRequest);
				}
			}, this);
		}, this);
	},

	/**
	 * Resolve all response data into a collection of {@link Zarafa.core.data.AbstractResponseHandlers}
	 * which will be in charge of handling all responsed. The responsehandlers will be returned into
	 * an array which is sorted on priority, meaning that the response handlers should be called in
	 * the order in which they are listed in the array.
	 * @param {Object} data The data from the response
	 * @return {Array} Array of objects containing the data
	 * @private
	 */
	resolveResponseHandlers : function(data)
	{
		var responses = [];
		var notifications = [];

		// Iterate over all modules and ids, and obtain the corresponding
		// ResponseHandlers. We separate the RequestResponses from the notifications.
		Ext.iterate(data, function(moduleName, modules) {
			// iterate over module ids
			Ext.iterate(modules, function(moduleId, moduleData) {
				var handler = {
					moduleName : moduleName,
					moduleId : moduleId,
					moduleData : moduleData
				}

				// Check if a RequestResponse Handler is registered for this moduleId
				handler.handler = this.getRequestResponseHandler(moduleId);
				if (!Ext.isEmpty(handler.handler)) {
					responses.push(handler);
					return;
				}

				// No RequestResponse was available, this is a notification
				handler.handler = this.getNotificationResponseHandler(moduleName, moduleData);
				if (!Ext.isEmpty(handler.handler)) {
					notifications.push(handler);
					return;
				}
			}, this);
		}, this);

		// Return the objects as a single array, the RequestResponses have highest priority,
		// followed by the notifications.
		return responses.concat(notifications);
	},

	/**
	 * Perform a transaction through the {@link Zarafa.core.data.AbstractResponseHandler ResponseHandler}.
	 *
	 * @param {Zarafa.core.data.AbstractResponseHandler} handler The handler which should be used
	 * for handling the response.
	 * @param {String} moduleName The name of the module from which the response was received
	 * @param {String} moduleId The unique id which is used to correlate the response to a request
	 * @param {Object} moduleData The data which was provided for the given moduleName/moduleId
	 * @param {Number} timestamp The {@link Date#getTime timestamp} on which the response was received
	 * @private
	 */
	handleResponse : function(handler, moduleName, moduleId, moduleData, timestamp)
	{
		var success = true;

		// Begin the Response transaction. When the transaction cannot
		// be started, we bail out immediately.
		if (handler.start(moduleName, moduleId, moduleData, timestamp) === false) {
			return;
		}

		if (Ext.isObject(moduleData)) {
			// Iterate over each action, and start handling them with
			// the corresponding actionData. If one of the handlers indicate
			// failure, we only change the 'success' status, but continue
			// with the other handlers. The 'success' status itself will
			// be used when the transaction is being completed.
			Ext.iterate(moduleData, function(actionType, actionData) {
				if (handler.handle(actionType, actionData) === false) {
					success = false;
				}
			}, this);
		}

		// Complete the transaction.
		handler.done(success);
	},

	/**
	 * Processes a response from the server. the data is examined for
	 * any error tags and call error listeners.
	 * @param {Object} jsonData A JSON object containing server response.
	 * @private
	 */
	processResponse : function(jsonData)
	{
		// check for errors, these are global errors which can be generated from zarafa.php
		// file, module level errors will be handled by module callback functions.
		if (!Ext.isEmpty(jsonData.zarafa.error)) {
			// Fire the exception event on the DataProxy like this, as the response cannot be matched to a specific proxy.
			Ext.data.DataProxy.fireEvent('exception', Ext.data.DataProxy, 'remote', null, null, jsonData.zarafa, null);
			return;
		};

		// Create the timestamp which is used as receive date for the current response
		var timestamp = new Date().getTime();

		// when all's fine, unpack the server response and obtain the responseHandlers
		var handlers = this.resolveResponseHandlers(jsonData.zarafa);

		for (var i = 0, len = handlers.length; i < len; i++) {
			var handler = handlers[i];
			var moduleName = handler.moduleName;
			var moduleId = handler.moduleId;
			var moduleData = handler.moduleData;
			var responseHandler = handler.handler;

			if (this.fireEvent('response', moduleName, moduleId, moduleData, timestamp) !== false) {
				this.handleResponse(responseHandler, moduleName, moduleId, moduleData, timestamp);
			}
		}
	}
});
Ext.namespace('Zarafa.core');

/**
 * @class Zarafa.core.Util
 * Utility class
 * @singleton
 */
Zarafa.core.Util =
{
	/**
	 * Sort an array of objects
	 *
	 * @param {Array} list The array of objects which must be sorted
	 * @param {String} order The order in which the array must be sorted, can be either "ASC" or "DESC"
	 * @param {String/Function} sort (optional) Can either be a string, a function or nothing.
	 * A String argument will be used to sort the array on the given fieldname. A function will be used
	 * as comparison function for sorting. When not provided the default comparison function shall be
	 * used which compares items based on their value.
	 * @return {Array} The sorted array
	 * @method
	 */
	sortArray : function(list, order, sort)
	{
		var collection = new Ext.util.MixedCollection();
		var fn = undefined;
		collection.addAll(list);

		if (Ext.isFunction(sort)) {
			// Use given sort function
			fn = sort;
		} else if (Ext.isString(sort)) {
			// Sort on object attribute value, by default this is a
			// numeric sort. We need to wrap the function to
			// access the attribute value.
			fn = function(obj1, obj2) {
				return Zarafa.core.Util.numericComparison(obj1[sort], obj2[sort]);
			};
		} else {
			// Sort the object, by default this is a numeric sort
			fn = this.numericComparison;
		}

		collection.sort(order, fn);
		return collection.getRange();
	},

	/**
	 * Comparison function for comparing two numbers.
	 * This function can be used for sorting in for example the functions
	 * {@link Ext.util.MixedCollection#sort} and {@link #sortArray}.
	 *
	 * @param {Number} number1 The first number to compare
	 * @param {Number} number2 The second number to compare
	 * @return {Number} A positive value when number1 is greater then number2.
	 * A negative value when number2 is greater then number1. 0 when both objects are equal.
	 */
	numericComparison : function(number1, number2)
	{
		return number1 - number2;
	},

	/**
	 * Comparison function for comparing two strings using case sensitive comparison.
	 * This function can be used for sorting in for example the functions
	 * {@link Ext.util.MixedCollection#sort} and {@link #sortArray}.
	 *
	 * @param {String} string1 The first object to compare
	 * @param {String} string2 The second object to compare
	 * @return {Number} A positive value when string1 is greater then string2.
	 * A negative value when string2 is greater then string1. 0 when both objects are equal.
	 */
	caseSensitiveComparison : function(string1, string2)
	{
		return string1 > string2 ? 1 : (string1 < string2 ? -1 : 0);
	},

	/**
	 * Comparison function for comparing two strings using case insensitive comparison.
	 * This function can be used for sorting in for example the functions
	 * {@link Ext.util.MixedCollection#sort} and {@link #sortArray}.
	 *
	 * @param {String} string1 The first object to compare
	 * @param {String} string2 The second object to compare
	 * @return {Number} A positive value when string1 is greater then string2.
	 * A negative value when string2 is greater then string1. 0 when both objects are equal.
	 */
	caseInsensitiveComparison : function(string1, string2)
	{
		var v1 = String(string1).toUpperCase(), v2 = String(string2).toUpperCase();
		return Zarafa.core.Util.caseSensitiveComparison(v1, v2);
	},

	/**
	 * Remove all duplicate entries from an array of objects
	 *
	 * @param {Array} list The array of objects which must be filtered
	 * @param {String} attr (optional) The fieldname on which objects must be compared to detect duplicates
	 * If not provided the comparison is done on the object value.
	 * @return {Array} The array with only unique elements
	 * @method
	 */
	uniqueArray : function(list, attr)
	{
		var collection = new Ext.util.MixedCollection();

		Ext.each(list, function(item) {
			var value = attr ? item[attr] : item;

			if (!collection.containsKey(value))
				collection.add(value, item);
		}, this);

		return collection.getRange();
	},

	/**
	 * This is a utility function to trim strings in a single or multi dimensional array
	 * this function should only be used with array which has only string values, otherwise
	 * this function will give unpredicated results without any error
	 * @param {Array} arrayToTrim array whose values should be trimmed
	 * @return {Array} trimmed array
	 */
	trimStringArray : function(arrayToTrim)
	{
		var tmpArray = arrayToTrim;
		arrayToTrim = [];				// reset array

		for(var index = 0, len = tmpArray.length; index < len; index++) {
			if(Ext.isArray(tmpArray[index])) {
				// recursively call the same function
				arrayToTrim.push(Zarafa.core.Util.trimStringArray(tmpArray[index]));
			} else {
				if(Ext.isString(tmpArray[index]) && !Ext.isEmpty(tmpArray[index].trim())) {
					arrayToTrim.push(tmpArray[index].trim());
				}
			}
		}

		return arrayToTrim;
	},

	/**
	 * This is a utility function to check if any multi dimensional array contains any token
	 * @param {Array} multiDimArray single or multi-dimensional array
	 * @param {Mixed} tokenToSearch token to search in array
	 * @param {Boolean} caseInSensitive case sensitive match
	 * @param {Boolean} matchPartial comparison will also check for partial match
	 * @return {Boolean} true if token is found else false
	 */
	inArray : function(multiDimArray, tokenToSearch, caseInSensitive, matchPartial)
	{
		for(var index = 0, len = multiDimArray.length; index < len; index++) {
			if(Ext.isArray(multiDimArray[index])) {
				// recursively call the same function
				if(Zarafa.core.Util.inArray(multiDimArray[index], tokenToSearch, caseInSensitive, matchPartial)) {
					return true;
				}
			} else {
				if(tokenToSearch) {
					if(matchPartial) {
						if(caseInSensitive) {
							if(multiDimArray[index].indexOf(tokenToSearch.toLowerCase()) != -1 || tokenToSearch.indexOf(multiDimArray[index].toLowerCase()) != -1) {
								return true;
							}
						} else {
							if(multiDimArray[index].indexOf(tokenToSearch) != -1) {
								return true;
							}
						}
					} else {
						if(caseInSensitive) {
							if(multiDimArray[index].toLowerCase() === tokenToSearch.toLowerCase()) {
								return true;
							}
						} else {
							if(multiDimArray[index] === tokenToSearch) {
								return true;
							}
						}
					}
				}
			}
		}

		return false;
	},

	/**
	 * This is a utility function to copy all the properties of config to obj recursively.
	 * Ext.appply() does the same thing but this extension preserves child object's previous values
	 * instead of overwriting it with new values.
	 * @param {Object} obj The receiver of the properties
	 * @param {Object} config The source of the properties
	 * @param {Object} defaults A different object that will also be applied for default values
	 * @return {Object} returns obj
	 */
	applyRecursive : function(obj, config, defaults)
	{
		if(defaults) {
			obj = Zarafa.core.Util.applyRecursive(obj, defaults);
		}

		if(obj && config && (Ext.isObject(config) || Ext.isArray(config))) {
			for(var key in config) {
				if (Ext.isDefined(obj[key]) && Ext.isObject(obj[key])) {
					// object with child elements, so call this function recursively
					obj[key] = Zarafa.core.Util.applyRecursive(obj[key], config[key]);
				} else if (Ext.isObject(config[key])) {
					obj[key] = Zarafa.core.Util.applyRecursive({}, config[key]);
				} else {
					// normal copy
					obj[key] = config[key];
				}
			}
		}

		return obj;
	},

	/**
	 * This is a utility function to copy all the properties of config to obj recursively,
	 * if they don't already exist. Ext.appplyIf() does the same thing but this extension
	 * preserves child object's previous values instead of overwriting it with new values.
	 * @param {Object} obj The receiver of the properties
	 * @param {Object} config The source of the properties
	 * @param {Object} defaults A different object that will also be applied for default values
	 * @return {Object} returns obj
	 */
	applyIfRecursive : function(obj, config, defaults)
	{
		if(defaults) {
			obj = Zarafa.core.Util.applyIfRecursive(obj, defaults);
		}

		if(obj && config && (Ext.isObject(config) || Ext.isArray(config))) {
			for(var key in config) {
				if(Ext.isDefined(obj[key]) && Ext.isObject(obj[key])) {
					// object with child elements, so call this function recursively
					obj[key] = Zarafa.core.Util.applyIfRecursive(obj[key], config[key]);
				} else if (Ext.isObject(config[key])) {
					obj[key] = Zarafa.core.Util.applyIfRecursive({}, config[key]);
				} else if(!Ext.isDefined(obj[key])) {
					// normal copy
					obj[key] = config[key];
				}
			}
		}

		return obj;
	},

	/**
	 * Recursively flattens a JSON object hierarchy into a flat list of key/value pairs.
	 *
	 * For example: The object:
	 *	{
	 *   	'zarafa' : {
	 *   		'v1' : {
	 *   			'main' : {
	 *   				'settingA' : 'value1',
	 *					'settingB' : 'value2'
	 *   			}
	 *   		}
	 *   	}
	 *	}
	 *
	 * will be flattened to:
	 *	{
	 * 		'zarafa/v1/main/settingA' : 'value1',
	 * 		'zarafa/v1/main/settingB' : 'value2'
	 *	}
	 *
	 * @param {Object} obj The object to flatten.
	 * @param {String} sep The separator which must be applied between each path-key (e.g: '/')
	 * @param {String} path The basePath for the keys inside the object.
	 * @return {Object} The flattened object
	 */
	flattenObject : function(obj, sep, path)
	{
		var ret = {};

		var separator = '';
		if (Ext.isEmpty(path)) {
			path = '';
		} else {
			separator = sep;
		}

		if (Ext.isObject(obj)) {
			for (var key in obj) {
				Ext.apply(ret, Zarafa.core.Util.flattenObject(obj[key], sep, path + separator + key));
			}
		} else {
			ret[path] = obj;
		}

		return ret;
	},

	/**
	 * Function will return object which has all keys in lowercase
	 * 
	 * @param {Object} obj The object.
	 * @return {Object} The object with all keys as lowercase
	 */
	objectKeysToLowerCase : function(object)
	{
		var key, keys = Object.keys(object);
		var newObject={}
		for (var i=0; i<keys.length; i++) {
			key = keys[i];
			newObject[key.toLowerCase()] = object[key];
		}
		return newObject;
	},

	/** 
	 * Split a string in pieces based on whether each piece matches the passed 
	 * pattern. It returns both the pieces that match and that do not match the 
	 * pattern.
	 * @param {String} str The input string to be split up
	 * @param {RegExp} pattern The regex pattern used to be split the string
	 * @return {Array} The array of pieces
	 * @private
	 */
	splitStringByPattern : function(str, pattern)
	{
		var cutOffPoints = [0];
		var found;
		// Find the cutOffPoints in the str
		while((found = pattern.exec(str)) !== null){
			if(found.index!=0){
				cutOffPoints.push(found.index);
			}
			if(pattern.lastIndex < str.length){
				cutOffPoints.push(pattern.lastIndex);
			}
		}
		// Cut the string up into the pieces based on the cutOffPoints
		var parts = [];
		if(cutOffPoints.length > 1){
			for(var i=0;i<cutOffPoints.length;i++){
				// Use the current and the next cutOffPoint to calculate the number of character we need to extract.
				if(Ext.isDefined(cutOffPoints[i+1])){
					parts.push(str.slice(cutOffPoints[i], cutOffPoints[i+1]));
				}else{
					parts.push(str.slice(cutOffPoints[i]));
				}
			}
		}else{
			parts = [str];
		}
		return parts;
	},

	/**
	 * Explode a previously {@link #flattenObject flattened} object back into the original object hierarchy.
	 *
	 * For example:
	 *	{
	 * 		'zarafa/v1/main/settingA' : 'value1',
	 * 		'zarafa/v1/main/settingB' : 'value2'
	 *	}
	 *
	 * will be exploded into:
	 *	{
	 *   	'zarafa' : {
	 *   		'v1' : {
	 *   			'main' : {
	 *   				'settingA' : 'value1',
	 *					'settingB' : 'value2'
	 *   			}
	 *   		}
	 *   	}
	 *	}
	 *
	 * @param {Object} obj The object to explode
	 * @param {String} sep The separator which was applied between each path-key (e.g: '/')
	 * @return {Object} The exploded object
	 */
	explodeObject : function(obj, sep)
	{
		var ret = {};

		for (var key in obj) {
			var pieces = key.split(sep);
			var pos = ret;

			// Build the hierarchy (make sure, we do no set the last key yet,
			// because that will be the key into which the value is stored.
			for (var i = 0, len = pieces.length - 1; i < len; i++) {
				var piece = pieces[i];

				if (!Ext.isDefined(pos[piece])) {
					pos[piece] = {}
				}

				pos = pos[piece]
			}

			// Set the last item which is the actual key-value pair.
			pos[pieces[pieces.length - 1]] = obj[key];
		}

		return ret;
	},

	/**
	 * Convenience method to check if a given point (x,y) is inside a box (x,y,width,height)
	 * @param {Object} box a (x, y, with, height) tuple
	 * @param {Number} x point x component
	 * @param {Number} y point y component
	 * @return {Boolean} True if the given point is inside the box.
	 */
	inside : function(box, x, y)
	{
		return (x >= box.x && x < (box.x + box.width) && y >= box.y && y < (box.y + box.height));
	},

	/**
	 * Restrict a box containing 'x', 'y', 'width' and 'height' properties,
	 * to fall completely inside the given container box (which has the same properties).
	 * This ensures that the position of the box left-top corner will always be inside
	 * the container, and will attempt to move the x and y coordinates in such a way
	 * that the full width and height will fit inside the container box.
	 * @param {Object} container The container box
	 * @param {Object} box The box
	 * @return {Object} The updated box position
	 */
	restrictBox : function(container, box)
	{
		// Ensure we copy the box
		box = Ext.apply({}, box);

		// For all our calculations, we at least
		// want the top-left position to be inside
		// the container.
		box.x = Math.max(container.x, box.x);
		box.y = Math.max(container.y, box.y);

		// We can only correct the x-coordinate
		// if it doesn't equal are most-left position
		if (box.x > container.x) {
			var overflowX = Math.max(0, (box.x + box.width) - (container.x + container.width));
			box.x -= overflowX;
		}

		// We can only correct the x-coordinate
		// if it doesn't equal are most-upper position
		if (box.y > container.y) {
			var overflowY = Math.max(0, (box.y + box.height) - (container.y + container.height));
			box.y -= overflowY;
		}

		// Technically we could have moved the boxed
		// beyond our minumum top-left position. Fix
		// that here, and just accept that we will
		// overflow...
		box.x = Math.max(container.x, box.x);
		box.y = Math.max(container.y, box.y);

		return box;
	},

	/**
	 * Function will get the index of the position of the cursor in the textfield.
	 * @param {Ext.Element/HTMLElement} obj Reference to the textfield
	 * @return {Number} Index of position, -1 if object is invalid
	 */
	getCaretPosition: function(obj)
	{
		obj = obj.dom || obj;

		// MSIE has to use selection object in document, FF does not have this object
		if(document.selection) { // MSIE
			/**
			 * In MSIE the position is calculated by first creating a selection
			 * from the cursors position to the start of the string and
			 * calculating the length of this selection string. This number is
			 * the actual position of cursor.
			 */
			obj.focus();
			var range = document.selection.createRange().duplicate();
			range.moveStart("character", -obj.value.length);
			return range.text.length;
		}else if (obj.selectionStart || (obj.selectionStart == "0")) { // Mozilla/Netscape
			// Just use the kickass selectionStart property in FF.
			return obj.selectionStart;
		}
	},

	/**
	 * Function will get the index of the start and end position of the current selection.
	 * @param {Ext.Element/HTMLElement} obj Reference to the textfield
	 * @return {Object} An object containing a 'start' and 'end' field which indicate
	 * the current position of the start and end of the selection. The fields will
	 * be -1 if there is no selection.
	 */
	getSelectionRange : function(obj)
	{
		obj = obj.dom || obj;

		// MSIE has to use selection object in document, FF does not have this object
		if (document.selection) { // MSIE
			obj.focus();
			var range = document.selection.createRange().duplicate();

			// Temporarily save the currently selected text.
			var text = range.text;

			// Move the selection to the beginning, so we can determine what the start offset is.
			range.moveStart("character", -obj.value.length);

			// Now we now the start and end of the seleciton.
			return { start : range.text.length, end : range.text.length + text.length };
		} else if (obj.selectionStart || (obj.selectionStart == "0")) { // Mozilla/Netscape
			return { start : obj.selectionStart, end : obj.selectionEnd };
		}
	},

	/**
	 * Function which will set the current caret position in the given textfield.
	 * @param {Ext.Element/HTMLElement} obj Reference to the textfield or the id of the textfield
	 * @param {Number} position The desired position for the caret
	 */
	setCaretPosition: function(obj, position)
	{
		Zarafa.core.Util.setSelectionRange(obj, position, position);
	},

	/**
	 * Function makes a selection in the textfield
	 * @param {Ext.Element/HTMLElement} obj Reference to the textfield or the id of the textfield
	 * @param {Number} selectionStart Index of the starting position of the selection
	 * @param {Number} selectionEnd Index of the ending position of the selection
	 */
	setSelectionRange: function(obj, selectionStart, selectionEnd)
	{
		obj = obj.dom || obj;

		if (obj && typeof obj == "object" && obj.setSelectionRange) {	// Mozilla/Netscape???
			obj.focus();
			obj.setSelectionRange(selectionStart, selectionEnd);
		}else if (obj.createTextRange) {	// MSIE
			var range = obj.createTextRange();
			range.collapse(true);
			range.moveEnd('character', selectionEnd);
			range.moveStart('character', selectionStart);
			range.select();
		}
	},

	/**
	 * Function replaces selection in the textfield
	 * @param {HTMLElement} obj Reference to the textfield or the id of the textfield
	 * @param {String} sText Text that should replace selection
	 */
	textboxReplaceSelection: function(obj, sText)
	{
		if(obj && typeof obj == "object") {
			if(document.selection) { // MSIE
				var oRange = document.selection.createRange();
				oRange.text = sText;
				oRange.collapse(true);
				oRange.select();
			} else if (obj.selectionStart || (obj.selectionStart == "0")) { // Mozilla/Netscape
				var iStart = obj.selectionStart;
				obj.value = obj.value.substring(0, iStart) + sText + obj.value.substring(obj.selectionEnd, obj.value.length);
				obj.setSelectionRange(iStart + sText.length, iStart + sText.length);
			}
			obj.focus();
		}
	},

	/**
	 * Checks whether a string is a valid email address using a regular expression. This check is
	 * not performed to the extend as the RFC 5322 specification describes the format.
	 * @param {String} str String to be validated
	 * @return {Boolean} Returns true when the email address is valid
	 */
	validateEmailAddress: function(str)
	{
		//TODO make a better check
		var filter = new RegExp(/^([^<]*<){0,1}(([a-z0-9\.\!\#\$\%\&\'\*\+\-\/\=\?\^\_\`\{\|\}\~])+\@(([a-z0-9\-])+\.)+([a-z0-9]{2,5})+)>{0,1}$|^\[[^\]]+\]$/i);
		if(Ext.isString(str) && str.length > 0 ){
			return filter.test(str);
		}else{
			return false;
		}
	},

	/**
	 * Function will be used to convert an object to an array, this function has some limitations
	 * it truncates keys and also doesn't preserve the order of items. This function does same as Ext.toArray
	 * but that function can not handle objects
	 * @param {Object} obj object to be converted to an array
	 * @return {Array} converted array
	 */
	objToArray : function(obj)
	{
		var ret = [];

		if(Ext.isArray(obj)) {
			// we already have an array
			return obj;
		}

		if(!Ext.isObject(obj)) {
			return ret;
		}

		Ext.iterate(obj, function(key, value) {
			ret.push(value);
		}, this);

		return ret;
	},

	/**
	 * Merge 2 objects containing event hanlers into a single object,
	 * while preserving scopes. This can be used when a {@link Ext.Component}
	 * receives a {@link Ext.Component#listeners} object while it also needs
	 * to add listeners in the same way (while it cannot use {@link Ext.Component#on}).
	 * By default the source listeners will {@link Function#createInterceptor intercept}
	 * the functions from the target.
	 *
	 * @param {Object} target The object with event handlers into which the new
	 * handlers will be merged.
	 * @param {Object} sourcec The object with event handlers which will be merged
	 * into the target
	 * @param {Boolean} intercept (optional) False to use {@link Function#createSequence}
	 * rather then {@link Function#createInterceptor}.
	 * @return {Object} The merged object
	 */
	mergeListeners : function(target, source, intercept)
	{
		// Make sure we have a target
		target = Ext.value(target, {});

		// Take the scope from our source, otherwise we default to the target
		var scope = source['scope'] || target['scope'];

		// Go over all listeners
		for (var key in source) {
			if (key === 'scope') {
				continue;
			}

			// Always create a delegate, the default scope inside the
			// target might might be equal to the scope of the source.
			var handler = source[key].createDelegate(scope);

			// Add the event handler
			if (Ext.isDefined(target[key])) {
				if (intercept !== false) {
					target[key] = target[key].createInterceptor(handler)
				} else {
					target[key] = target[key].createSequence(handler);
				}
			} else {
				target[key] = handler;
			}
		}

		return target;
	},

	/**
	 * Encode string in utf-16 hex format
	 * @param {String} str String to be converted into utf-16 encoded hex string
	 * @return {String} The utf-16 encoded string in hex format
	 */
	encode_utf16 : function(str)
	{
		var num1, num2;
		var result = ''; 

		if(!Ext.isString(str)) {
		    str = String(str);
		}

		for (var i = 0, len = str.length; i < len; i++) {
		    num2 = (str.charCodeAt(i) >> 8).toString(16);
		    num1 = (str.charCodeAt(i) & 0xff).toString(16);

		    result += String.leftPad(String(num1), 2, '0');
		    result += String.leftPad(String(num2), 2, '0');
		}

		return result;
	},

	/**
	 * Function converts string in to hexadecimal string
	 * @param {String} str ASCII string to be converted into hexadecimal representation.
	 * @return {String} The hexadecimal representation as a string.
	 */
	stringToHex : function(string)
	{
		string = string.toUpperCase();
		var hexString = '';
		for(var i=0; i < string.length; i++) {
			hexString += '' + string.charCodeAt(i).toString(16);
		}

		return hexString;
	},

	/**
	 * Function converts hexadecimal string in to ASCII string
	 * @param {String} hexString The hexadecimal string
	 * @return {String} converted ASCII string
	 */
	hexToString : function(hexString)
	{
		hexString = hexString.toString();
		var string = '';
		for (var i = 0; i < hexString.length; i += 2) {
			string += String.fromCharCode(parseInt(hexString.substr(i, 2), 16));
		}
		return string.toLowerCase();
	},

	/**
	 * Function used to reload the webapp.
	 */
	reloadWebapp : function()
	{
		this.disableLeaveRequester();
		window.location.reload();
	},

	/**
	 * Function is use to register onbeforeunload event to show confirm dialog
	 * when user trying to leave the page.
	 */
	enableLeaveRequester : function()
	{
		window.onbeforeunload = function () {
			return _('Your changes will be lost if you leave this page now.');
		};
	},

	/**
	 * Function is use to deregistering onbeforeunload event.
	 */
	disableLeaveRequester : function()
	{
		window.onbeforeunload = null;
	}
};
Ext.namespace('Zarafa.core.data');

/**
 * @class Zarafa.core.data.CallbackQueue
 * @extends Ext.util.Observable
 *
 * A Special queue containing callback functions which can be used
 * to serialize a series of actions/validations where it is possible
 * that each action/validation might use a {@link Ext.MessageBox MessageBox}
 * to request the user input. While the {@link Ext.MessageBox MessageBox}
 * is opened, the queue will be paused until the user has closed the message.
 */
Zarafa.core.data.CallbackQueue = Ext.extend(Ext.util.Observable, {
	/**
	 * The queue which contains all the tasks which should be run
	 * @property
	 * @type Array
	 * @private
	 */
	queue : undefined,

	/**
	 * Indicates that {@link #run} has been called, and the various callbacks
	 * in the {@link #queue} are being executed.
	 * @property
	 * @type Boolean
	 * @private
	 */
	running : false,

	/**
	 * Internal counter to keep track at which task is currently being executed,
	 * this will be reset when the {@link #run queue starts} and will be updated
	 * after the {@link #onCompleteTask completion of each task}.
	 * @property
	 * @type Number
	 * @private
	 */
	currentTask : 0,

	/**
	 * The function which is provided to {@link #run} which should be called as
	 * soon as the last task has been called. It will be called with the scope
	 * {@link #completionScope}.
	 * @property
	 * @type Function
	 * @private
	 */
	completionFn : undefined,

	/**
	 * The scope for the {@link #completionFn} which is provided to {@link #run}.
	 * @property
	 * @type Object
	 * @private
	 */
	completionScope : undefined,

	/**
	 * @constructor
	 * @param {Object} config Configuration object
	 */
	constructor : function(config)
	{
		config = config || {};

		// Initialize the queue
		config.queue = [];

		this.addEvents(
			/**
			 * @event startqueue
			 * Event which is fired when the queue is about to start
			 * @param {Zarafa.core.data.CallbackQueue} queue The queue which is started
			 */
			'startqueue',
			/**
			 * @event completequeue
			 * Event which is fired when the last callback from the queue has been completed.
			 * @param {Zarafa.core.data.CallbackQueue} queue The queue which has completed
			 * @param {Boolean} success True if all callbacks were executed successfully
			 */
			'completequeue',
			/**
			 * @event starttask
			 * Event which is fired when a task has started
			 * @param {Zarafa.core.data.CallbackQueue} queue The queue to which the task belongs
			 * @param {Function} fn The task function which has been started
			 * @param {Object} scope The scope of the function which has started
			 */
			'starttask',
			/**
			 * @event completetask
			 * Event which is fired when a task has completed
			 * @param {Function} fn The task function which has been completed
			 * @param {Object} scope The scope of the function which has completed
			 * @param {Boolean} success True if the callback was executed successfully
			 */
			'completetask'
		);

		Ext.apply(this, config);

		Zarafa.core.data.CallbackQueue.superclass.constructor.call(this, config);
	},

	/**
	 * Add a callback function to the end of the {@link #queue}. When {@link #run} is called,
	 * this function will be executed with the provided scope.
	 * @param {Function} fn The function which will be called
	 * @param {Object} scope The scope in which the function will be called
	 */
	add : function(fn, scope)
	{
		this.queue.push({ fn : fn, scope : scope });
	},

	/**
	 * Remove a callback function which was previously registered using {@link #add}.
	 * This will search for the task in the {@link #queue} which matches the given
	 * function and scope exactly, and removed it from the {@link #queue}.
	 * @param {Function} fn The function to remove
	 * @param {Object} scope The scope of the function
	 */
	remove : function(fn, scope)
	{
		var queue = this.queue;

		for (var i = 0; i < queue.length; i++) {
			var task = queue[i];

			// Check if this is the same function and scope,
			// if so remove it from the queue.
			if (task.fn === fn && task.scope === scope) {
				this.queue.splice(i, 1);
				return;
			}
		}
	},

	/**
	 * Run all Callback functions in the {@link #queue}. This will fire the {@link #start} event,
	 * and starts all tasks {@link #currentTask starting with} 0.
	 */
	run : function(fn, scope)
	{
		this.running = true;
		this.currentTask = 0;
		this.completionFn = fn,
		this.completionScope = scope,

		this.fireEvent('startqueue', this);

		this.doTask(this.currentTask);
	},

	/**
	 * @returns {Boolean} True if the queue is currently running
	 */
	isRunning : function()
	{
		return this.running;
	},

	/**
	 * Called to execute the task at the specified location in the {@link #queue}.
	 * This will execute the callback function, and pass the {@link #onCompleteTask} function
	 * as callback function.
	 * @param {Number} index The index in the queue of the callback to execute
	 * @private
	 */
	doTask : function(index)
	{
		var task = this.queue[index];
		this.fireEvent('starttask', this, task.fn, task.scope);
		task.fn.call(task.scope, this.onCompleteTask.createDelegate(this, [ task ], 1));
	},

	/**
	 * Callback function for the task which was executed using {@link #doTask}. This
	 * checks if the task was successfully completed and if so if this was the last task.
	 * If either the task has failed, or this was the last task, the queue will be stopped,
	 * and the {@link #complete} event will be fired. Otherwise {@link #doTask} will be
	 * called to execute the {@link #currentTask next task}.
	 * @param {Boolean} success True if the task completed successfully
	 * @param {Object} task The task which was completed successfully
	 * @private
	 */
	onCompleteTask : function(success, task)
	{
		// If not provided, then assume success
		success = success !== false;

		this.fireEvent('completetask', this, task.fn, task.scope, success);

		if (success && this.currentTask < (this.queue.length - 1)) {
			this.currentTask++;
			this.doTask(this.currentTask);
		} else {
			if (this.completionFn) {
				this.completionFn.call(this.completionScope, success);
				this.completionFn = undefined;
				this.completionScope = undefined;
			}
			this.fireEvent('completequeue', this, success);
			this.running = false;
		}
	}
});
Ext.namespace('Zarafa.core.data');

/**
 * @class Zarafa.core.data.ContentPanelMgr
 * @extends Ext.util.Observable
 * @singleton
 *
 * The {@link Zarafa.core.ui.ContentPanel ContentPanel} manager. Each
 * {@link Zarafa.core.ui.ContentPanel ContentPanel} which is created
 * must register itself to this manager.
 *
 * This manager can be used by Plugins to hook into
 * {@link Zarafa.core.ui.ContentPanel ContentPanel} from the moment they
 * are being displayed.
 */
Zarafa.core.data.ContentPanelMgr = Ext.extend(Ext.util.Observable, {
	/**
	 * The collection of {@link Zarafa.core.ui.ContentPanel contentPanels}
	 * which have been registered to this manager.
	 * @property
	 * @type Ext.util.MixedCollection
	 */
	contentPanels : undefined,
	/**
	 * @constructor
	 */
	constructor : function()
	{
		this.contentPanels = new Ext.util.MixedCollection();
		this.addEvents([
			/**
			 * @event createcontentpanel
			 * Fires when a {@link Zarafa.core.ui.ContentPanel contentPanel} is being created.
			 * @param {Zarafa.core.ui.ContentPanel} contentPanels The 
			 * {@link Zarafa.core.ui.ContentPanel} which is being created.
			 */
			'createcontentpanel',
			/**
			 * @event destroycontentpanel
			 * Fires when a {@link Zarafa.core.ui.ContentPanel contentPanel} is being destroyed.
			 * @param {Zarafa.core.ui.ContentPanel} contentPanel The 
			 * {@link Zarafa.core.ui.ContentPanel} which is being destroyed.
			 */
			'destroycontentpanel'
		]);

		Zarafa.core.data.ContentPanelMgr.superclass.constructor.call(this);
	},

	/**
	 * Register a {@link Zarafa.core.ui.ContentPanel contentPanel} with the {@link Zarafa.core.data.ContentPanelMgr ContentPanelMgr}.
	 * @param {Zarafa.core.ui.ContentPanel} contentPanel the {@link Zarafa.core.ui.ContentPanel contentPanel} which must be registered.
	 */
	register : function(contentPanel)
	{
		contentPanel.on('show', this.onContentPanelShow, this);
		contentPanel.on('close', this.onContentPanelHide, this);
		contentPanel.on('hide', this.onContentPanelHide, this);

		this.contentPanels.add(contentPanel);
	},

	/**
	 * UnRegister a {@link Zarafa.core.ui.ContentPanel contentPanel} from the {@link Zarafa.core.data.ContentPanelMgr ContentPanelMgr}.
	 * @param {Zarafa.core.ui.ContentPanel} contentPanel the {@link Zarafa.core.ui.ContentPanel contentPanel} which must be unregistered.
	 */
	unregister : function(contentPanel)
	{
		contentPanel.un('show', this.onContentPanelShow, this);
		contentPanel.un('close', this.onContentPanelHide, this);
		contentPanel.un('hide', this.onContentPanelHide, this);

		this.contentPanels.remove(contentPanel);
	},

	/**
	 * Event handler which is raised before the {@link Zarafa.core.ui.ContentPanel contentPanel} is
	 * being shown. This will raise the {@link #createcontentpanel} event to allow any
	 * listeners to hook into further events coming from the given
	 * {@link Zarafa.core.ui.ContentPanel contentPanel}.
	 *
	 * @param {Ext.Container} contentPanel The contentPanel which is being rendered
	 * @private
	 */
	onContentPanelShow : function(contentPanel)
	{
		this.fireEvent('createcontentpanel', contentPanel);
	},

	/**
	 * Event handler which is raised when the {@link Zarafa.core.ui.ContentPanel contentPanel} is
	 * being hidden. This will raise the {@link #destroycontentpanel} event to inform
	 * any listeners that their {@link Zarafa.core.ui.ContentPanel contentPanel} is going to disappear.
	 *
	 * @param {Ext.Container} contentPanel The contentPanel which is being destroyed
	 * @private
	 */
	onContentPanelHide : function(contentPanel)
	{
		this.fireEvent('destroycontentpanel', contentPanel);
	},

	/**
	 * Find instances of {@link Zarafa.core.ui.ContentPanel contentPanel} from the {@link Zarafa.core.data.ContentPanelMgr ContentPanelMgr}.
	 * @param {Ext.Component} component the class name of the contentPanel for which we should perform the search.
	 * @return {Ext.util.MixedCollection} {@link Ext.util.MixedCollection MixedCollection} of contentPanels for specified
	 * component.
	 */
	getContentPanelInstances : function(component)
	{
		return this.contentPanels.filterBy(function(contentPanel) {
			return contentPanel instanceof component;
		});
	}
});

Zarafa.core.data.ContentPanelMgr = new Zarafa.core.data.ContentPanelMgr();
Ext.namespace('Zarafa.core.data');

/**
 * @class Zarafa.core.data.IPFStoreMgr
 * @extends Ext.util.Observable
 *
 * The {@link Zarafa.core.data.IPFStore IPFStore} manager. Each
 * {@link Zarafa.core.data.IPFStore IPFStore} which is created
 * must register itself to this manager.
 *
 * {@link Zarafa.core.data.IPFStoreMgr} will then handle inter-store
 * communications. Listening to {@link Zarafa.core.data.IPFStoreMgr IPFStoreMgr}
 * events allow UI components and {@link Zarafa.core.data.IPFStore stores} to
 * detect {@link Zarafa.core.data.IPFRecord record} editing in dialogs.
 * @singleton
 */
Zarafa.core.data.IPFStoreMgr = Ext.extend(Ext.util.Observable, {
		/**
		 * The collection of {@link Zarafa.core.data.IPFStore stores}
		 * which have been registered to this manager.
		 * @property
		 * @type Ext.util.MixedCollection
		 */
		IPFStores : undefined,
		/**
		 * @constructor
		 */
		constructor : function()
		{
			this.IPFStores = new Ext.util.MixedCollection();
			this.addEvents([
				/**
				 * @event storeexception
				 * Fires when the {@link Zarafa.core.data.IPFStore IPFStore} fired an {@link Zarafa.core.data.IPFStore#exception exception}.
				 * @param {Zarafa.core.data.IPFStore} store The store on which the exception occured
				 * @param {Ext.data.DataProxy} proxy The proxy from where the exception event originated
				 * @param {String} type The value of this parameter will be either 'response' or 'remote'.
				 * @param {String} action Name of the action.
				 * @param {Object} options The options for the action that were specified in the request.
				 * @param {Object} response If the 'type' argument was 'response' this will be the raw browser response object,
				 * otherwise it will be the decoded response object
				 * @param {Mixed} arg (optional) Additional arguments for the exception
				 */
				'storeexception',
				/**
				 * @event beforerecordsave
				 * Fires before a Message {@link Zarafa.core.data.IPFRecord record} is being saved.
				 * @param {Zarafa.core.data.IPFStore} store The {@link Zarafa.core.data.IPFStore store} in which the
				 * {@link Zarafa.core.data.IPFRecord record} is located while being saved.
				 * @param {Object} data An object containing the data that is to be saved.
				 * The object will contain a key for each appropriate action, with an array
				 * of updated data for each record.				 
				 */
				'beforerecordsave',
				/**
				 * @event afterrecordsave
				 * Fires when a Message {@link Zarafa.core.data.IPFRecord record} has been saved.
				 * @param {Zarafa.core.data.IPFStore} store The {@link Zarafa.core.data.IPFStore store} in which the
				 * {@link Zarafa.core.data.IPFRecord record} is located while being saved.
				 * @param {Mixed} obj The object containing {@link Zarafa.core.data.IPFRecord record} which has been saved. This
				 * {@link Zarafa.core.data.IPFRecord record} is the most recent version which came from the server.
				 */
				'afterrecordsave',
				/**
				 * @event afterrecordupdate
				 * Fires when a Message {@link Zarafa.core.data.IPFRecord record} has been updated.
				 * @param {Zarafa.core.data.IPFStore} store The {@link Zarafa.core.data.IPFStore store} to which the
				 * {@link Zarafa.core.data.IPFRecord record} belongs.
				 * @param {Zarafa.core.data.IPFrecord} record The most recent version which came from the server.
				 * @param {String} operation  The update operation being performed. 
				 * ({@link Ext.data.Record#EDIT}, {@link Ext.data.Record#REJECT}, {@link Ext.data.Record#COMMIT}).
				 */
				'afterrecordupdate',
				/**
				 * @event recordremove
				 * Fires when a Message {@link Zarafa.core.data.IPFRecord record} has been removed from the server
				 * @param {Zarafa.core.data.IPFStore} store The {@link Zarafa.core.data.IPFStore store} to which the
				 * {@link Zarafa.core.data.IPFRecord record} belongs.
				 * @param {Zarafa.core.data.IPFrecord} record The most recent version which came from the server.
				 */
				'recordremove',
				/**
				 * @event afterrecordwrite
				 * Fires when {@link Zarafa.core.IPFRecord IPFRecords} are modified (created, updated, destroyed, opened)
				 * on {@link Zarafa.core.data.IPFStore IPFStore}.
				 * @param {Zarafa.core.data.IPFStore} store The {@link Zarafa.core.data.IPFStore store} to which the
				 * {@link Zarafa.core.data.IPFRecord record} belongs.
				 * @param {String} action [Ext.data.Api.actions.create|update|destroy|open]
				 * @param {Object} result The 'data' picked-out out of the response for convenience.
				 * @param {Ext.Direct.Transaction} res
				 * @param {Zarafa.core.data.IPFrecord|Array} records The most recent version of the records
				 * which came from the server.
				 */				
				'afterrecordwrite'
			]);

			Zarafa.core.data.IPFStoreMgr.superclass.constructor.call(this);
		},

		/**
		 * Register a {@link Zarafa.core.data.IPFStore IPFStore} with the {@link Zarafa.core.data.IPFStoreMgr IPFStoreMgr}.
		 * @param {Zarafa.core.data.IPFStore} IPFStore the {@link Zarafa.core.data.IPFStore IPFStore} which must be registered.
		 * @param {Boolean} serverOnly True to register the {@link Zarafa.core.data.IPFStore IPFStore} only for
		 * events originating directly from the server.
		 */
		register : function(IPFStore, serverOnly)
		{
			if (!serverOnly) {
				IPFStore.on('beforesave', this.onBeforeSave, this);
				IPFStore.on('save', this.onSave, this);
				IPFStore.on('update', this.onUpdate, this);
				IPFStore.on('remove', this.onRemove, this);
			}
			IPFStore.on('write', this.onWrite, this);
			IPFStore.on('exception', this.onException, this);

			this.IPFStores.add(IPFStore);
		},

		/**
		 * Unregister a {@link Zarafa.core.data.IPFStore IPFStore} from the {@link Zarafa.core.data.IPFStoreMgr IPFStoreMgr}.
		 * @param {Zarafa.core.data.IPFStore} IPFStore the {@link Zarafa.core.data.IPFStore IPFStore} which must be unregistered.
		 * @param {Boolean} serverOnly True if the {@link Zarafa.core.data.IPFStore IPFStore} was {@link #register registered}
		 * with the serverOnly argument.
		 */
		unregister : function(IPFStore, serverOnly)
		{
			if (!serverOnly) {
				IPFStore.un('beforesave', this.onBeforeSave, this);
				IPFStore.un('save', this.onSave, this);
				IPFStore.un('update', this.onUpdate, this);
				IPFStore.un('remove', this.onRemove, this);
			}
			IPFStore.un('write', this.onWrite, this);
			IPFStore.un('exception', this.onException, this);

			this.IPFStores.remove(IPFStore);
		},

		/**
		 * Filter the list of {@link Zarafa.core.data.IPFRecord records} to contain
		 * only those which can be {@link Zarafa.core.data.IPFRecord#eventPropagation propagated}
		 * over a new event to other {@link Zarafa.core.data.IPFStore stores}.
		 * 
		 * @param {Zarafa.core.data.IPFRecord/Array} records The record or records to filter
		 * out the non-propagatable records.
		 * @return {Zarafa.core.data.IPFRecord/Array} All propagatable records
		 * @private
		 */
		getPropagatableRecords : function(records)
		{
			var propagateRecords = [];

			if (!Ext.isArray(records)) {
				records = [ records ]
			}
	
			for (var i = 0, len = records.length; i < len; i++) {
				var record = records[i];

				if (record instanceof Zarafa.core.data.IPFRecord && record.hasEventPropagation()) {
					propagateRecords.push(record);
				}
			}

			return propagateRecords;
		},

		/**
		 * Event handler which is triggered when a {@link Zarafa.core.data.IPFRecord record} is about to
		 * be saved in a {@link Zarafa.core.data.IPFStore store}. This function will inform the
		 * {@link Zarafa.core.data.IPFStore IPFStores} about the event through the
		 * {@link #beforerecordsave} event.
		 *
		 * @param {Zarafa.core.data.IPFStore} IPFStore
		 * @param {Object} data An object containing the data that is to be saved.
		 * The object will contain a key for each appropriate action, with an array
		 * of updated data for each record.
		 * @private
		 */
		onBeforeSave : function(IPFStore, data)
		{
			var propagateData = {}; 

			for (var key in data) {
				var propagateRecords = this.getPropagatableRecords(data[key]);
				if (!Ext.isEmpty(propagateData)) {
					propagateData[key] = propagateRecords;
				}
			}

			this.fireEvent('beforerecordsave', IPFStore, propagateData);
		},

		/**
		 * Event handler which is triggered when a {@link Zarafa.core.data.IPFRecord record} has been
		 * saved in a {@link Zarafa.core.data.IPFStore store}. This function will inform the
		 * {@link arafa.core.data.IPFStore IPFStores} about the event through the
		 * {@link #afterrecordsave} event.
		 *
		 * @param {Zarafa.core.data.IPFStore} IPFStore
		 * @param {Number} batch The identifier for the batch that was saved.
		 * @param {Object} data An object containing the data that is to be saved.
		 * The object will contain a key for each appropriate action, with an array
		 * of updated data for each record.
		 * @private
		 */
		onSave : function(IPFStore, batch, data)
		{
			this.fireEvent('afterrecordsave', IPFStore, data);
		},

		/**
		 * Event handler which is triggered when a {@link Zarafa.core.data.IPFRecord record} has been
		 * updated from the server. This function will inform the {@link Zarafa.core.data.IPFStore IPFStores}
		 * about the event through the {@link #afterrecordupdate} event.
		 *
		 * @param {Zarafa.core.data.IPFStore} IPFStore
		 * @param {Zarafa.core.data.IPFRecord} record The Record which has been updated
		 * @param {String} operation  The update operation being performed. 
		 * ({@link Ext.data.Record#EDIT}, {@link Ext.data.Record#REJECT}, {@link Ext.data.Record#COMMIT}).
		 * @private
		 */
		onUpdate : function(IPFStore, record, operation)
		{
			if (record.hasEventPropagation()) {
				this.fireEvent('afterrecordupdate', IPFStore, record, operation);
			}
		},

		/**
		 * Event handler which is triggered when a {@link Zarafa.core.data.IPFRecord record} has been
		 * removed from the server. This function will inform the {@link Zarafa.core.data.IPFStore IPFStores}
		 * about the event through the {@link #recordremove} event.
		 *
		 * @param {Zarafa.core.data.IPFStore} IPFStore
		 * @param {Zarafa.core.data.IPFRecord} record The Record which has been updated
		 * @param {String} The index at which the record was removed
		 * @private
		 */
		onRemove : function(IPFStore, record, index)
		{
			if (record.hasEventPropagation()) {
				this.fireEvent('recordremove', IPFStore, record);
			}
		},

		/**
		 * Event handler will be called on successfull completion of any CRUD operation,
		 * the difference between this event and {@link #save} event is that {@link #save}
		 * event will pass only data set that is modified not the record that is modified.
		 * so this event removes burden of finding record from the record set.
		 *
		 * @param {Zarafa.core.data.IPFStore} store The {@link Zarafa.core.data.IPFStore store} to which the
		 * {@link Zarafa.core.data.IPFRecord record} belongs.
		 * @param {String} action [Ext.data.Api.actions.create|update|destroy|open]
		 * @param {Object} data The 'data' picked-out out of the response for convenience.
		 * @param {Ext.Direct.Transaction} res
		 * @param {Zarafa.core.data.IPFrecord|Array} records The most recent version of the records
		 * which came from the server.
		 * @private
		 */
		onWrite : function(IPFStore, action, data, transaction, records)
		{
			var propagateRecords = this.getPropagatableRecords(records);
			if (!Ext.isEmpty(propagateRecords)) {
				this.fireEvent('afterrecordwrite', IPFStore, action, data, transaction, propagateRecords);
			}
		},

		/**
		 * Event handler which will be called when the store has fired the {@link Ext.data.Store#exception} event.
		 * This will look up which store has exactly fired the event, and will fire the {@link #exception} event.
		 * @param {Ext.data.DataProxy} proxy The proxy from where the exception event originated
		 * @param {String} type The value of this parameter will be either 'response' or 'remote'.
		 * @param {String} action Name of the action.
		 * @param {Object} options The options for the action that were specified in the request.
		 * @param {Object} response If the 'type' argument was 'response' this will be the raw browser response object,
		 * otherwise it will be the decoded response object
		 * @param {Mixed} arg (optional) Additional arguments for the exception
		 * @private
		 */
		onException : function(proxy, type, action, options, response, arg)
		{
			var proxies = Ext.pluck(this.IPFStores.items, 'proxy');
			if (!Ext.isEmpty(proxies)) {
				var storeIndex = proxies.indexOf(proxy);
				this.fireEvent('storeexception', this.IPFStores.get(storeIndex), proxy, type, action, options, response, arg);
			}
		},

		/**
		 * Executes the specified function once for every {@link Zarafa.core.data.IPFStore store} in the collection,
		 * passing the following arguments:
		 * <div class="mdetail-params"><ul>
		 * <li><b>item</b> : Zarafa.core.data.IPFStore<p class="sub-desc">The collection item</p></li>
		 * <li><b>index</b> : Number<p class="sub-desc">The item's index</p></li>
		 * <li><b>length</b> : Number<p class="sub-desc">The total number of items in the collection</p></li>
		 * </ul></div>
		 * The function should return a boolean value. Returning false from the function will stop the iteration.
		 * @param {Function} fn The function to execute for each item.
		 * @param {Object} scope (optional) The scope (<code>this</code> reference) in which the function is executed.
		 * Defaults to the current item in the iteration.
		 */
		each : function(fn, scope)
		{
			this.IPFStores.each.apply(this, arguments);
		},

		/**
		 * Obtain a list of {@link Zarafa.core.data.IPFStore stores} which have one or more of the
		 * requested {@link Zarafa.core.data.IPFRecord folders} currently
		 * {@link Zarafa.core.data.IPFStore#containsStoreInLastLoad loaded}.
		 * @param {Array} folders The list of {@link Zarafa.core.data.IPFRecord folders} or
		 * {@link Zarafa.hierarchy.data.MAPIFolderRecord#getId folder entryIds}
		 * for which the {@link Zarafa.core.data.IPFStore stores} are requested.
		 * @return {Array} The array of {@link Zarafa.core.data.IPFStore stores} which have the
		 * one or more of the requested stores loaded.
		 */
		getStoresForStores : function(folders)
		{
			var stores = [];

			if (!Ext.isArray(folders)) {
				folders = [ folders ];
			}

			if (folders[0] instanceof Zarafa.core.data.IPFRecord) {
				folders = Ext.pluck(folders, 'id');
			}

			this.IPFStores.each(function(store) {
				if (store.containsStoreInLastLoad(folders)) {
					stores.push(store);
				}
			}, this);

			return stores;
		}
});

Zarafa.core.data.IPFStoreMgr = new Zarafa.core.data.IPFStoreMgr();
Ext.namespace('Zarafa.core.data');

/**
 * @class Zarafa.core.data.IPMRecipientResolveRecord
 * Contains a description of what the data that will be used to resolve an user looks like. It is 
 * used by the reader in the Zarafa.core.data.CheckNamesResponseHandler.
 */
Zarafa.core.data.IPMRecipientResolveRecordFields = [
	{name: 'user_name'},
	{name: 'display_name'},
	{name: 'address_type'},
	{name: 'smtp_address'},
	{name: 'email_address'},
	{name: 'entryid'},
	{name: 'search_key'},
	{name: 'object_type', type: 'int'},
	{name: 'display_type', type: 'int'},
	{name: 'display_type_ex', type: 'int'}

];
Zarafa.core.data.IPMRecipientResolveRecord = Ext.data.Record.create(Zarafa.core.data.IPMRecipientResolveRecordFields);
Ext.namespace('Zarafa.core.data');

/**
 * @class Zarafa.core.data.IPMStoreMgr
 * @extends Ext.util.Observable
 *
 * The {@link Zarafa.core.data.IPMStore IPMStore} manager. Each
 * {@link Zarafa.core.data.IPMStore IPMStore} which is created
 * must register itself to this manager.
 *
 * {@link Zarafa.core.data.IPMStoreMgr} will then handle inter-store
 * communications. Listening to {@link Zarafa.core.data.IPMStoreMgr IPMStoreMgr}
 * events allow UI components and {@link Zarafa.core.data.IPMStore stores} to
 * detect {@link Zarafa.core.data.IPMRecord record} editing in dialogs.
 * @singleton
 */
Zarafa.core.data.IPMStoreMgr = Ext.extend(Ext.util.Observable, {
		/**
		 * The collection of {@link Zarafa.core.data.IPMStore stores}
		 * which have been registered to this manager.
		 * @property
		 * @type Ext.util.MixedCollection
		 */
		IPMStores : undefined,
		/**
		 * @constructor
		 */
		constructor : function()
		{
			this.IPMStores = new Ext.util.MixedCollection();
			this.addEvents([
				/**
				 * @event storeexception
				 * Fires when the {@link Zarafa.core.data.IPMStore IPMStore} fired an {@link Zarafa.core.data.IPMStore#exception exception}.
				 * @param {Zarafa.core.data.IPMStore} store The store on which the exception occured
				 * @param {Ext.data.DataProxy} proxy The proxy from where the exception event originated
				 * @param {String} type The value of this parameter will be either 'response' or 'remote'.
				 * @param {String} action Name of the action.
				 * @param {Object} options The options for the action that were specified in the request.
				 * @param {Object} response If the 'type' argument was 'response' this will be the raw browser response object,
				 * otherwise it will be the decoded response object
				 * @param {Mixed} arg (optional) Additional arguments for the exception
				 */
				'storeexception',
				/**
				 * @event beforerecordsave
				 * Fires before a Message {@link Zarafa.core.data.IPMRecord record} is being saved.
				 * @param {Zarafa.core.data.IPMStore} store The {@link Zarafa.core.data.IPMStore store} in which the
				 * {@link Zarafa.core.data.IPMRecord record} is located while being saved.
				 * @param {Object} data An object containing the data that is to be saved.
				 * The object will contain a key for each appropriate action, with an array
				 * of updated data for each record.				 
				 */
				'beforerecordsave',
				/**
				 * @event afterrecordsave
				 * Fires when a Message {@link Zarafa.core.data.IPMRecord record} has been saved.
				 * @param {Zarafa.core.data.IPMStore} store The {@link Zarafa.core.data.IPMStore store} in which the
				 * {@link Zarafa.core.data.IPMRecord record} is located while being saved.
				 * @param {Mixed} obj The object containing {@link Zarafa.core.data.IPMRecord record} which has been saved. This
				 * {@link Zarafa.core.data.IPMRecord record} is the most recent version which came from the server.
				 */
				'afterrecordsave',
				/**
				 * @event afterrecordupdate
				 * Fires when a Message {@link Zarafa.core.data.IPMRecord record} has been updated.
				 * @param {Zarafa.core.data.IPMStore} store The {@link Zarafa.core.data.IPMStore store} to which the
				 * {@link Zarafa.core.data.IPMRecord record} belongs.
				 * @param {Zarafa.core.data.IPMrecord} record The most recent version which came from the server.
				 * @param {String} operation  The update operation being performed. 
				 * ({@link Ext.data.Record#EDIT}, {@link Ext.data.Record#REJECT}, {@link Ext.data.Record#COMMIT}).
				 */
				'afterrecordupdate',
				/**
				 * @event recordremove
				 * Fires when a Message {@link Zarafa.core.data.IPMRecord record} has been removed from the server
				 * @param {Zarafa.core.data.IPMStore} store The {@link Zarafa.core.data.IPMStore store} to which the
				 * {@link Zarafa.core.data.IPMRecord record} belongs.
				 * @param {Zarafa.core.data.IPMrecord} record The most recent version which came from the server.
				 */
				'recordremove',
				/**
				 * @event afterrecordwrite
				 * Fires when {@link Zarafa.core.IPMRecord IPMRecords} are modified (created, updated, destroyed, opened)
				 * on {@link Zarafa.core.data.IPMStore IPMStore}.
				 * @param {Zarafa.core.data.IPMStore} store The {@link Zarafa.core.data.IPMStore store} to which the
				 * {@link Zarafa.core.data.IPMRecord record} belongs.
				 * @param {String} action [Ext.data.Api.actions.create|update|destroy|open]
				 * @param {Object} result The 'data' picked-out out of the response for convenience.
				 * @param {Ext.Direct.Transaction} res
				 * @param {Zarafa.core.data.IPMrecord|Array} records The most recent version of the records
				 * which came from the server.
				 */				
				'afterrecordwrite'
			]);

			Zarafa.core.data.IPMStoreMgr.superclass.constructor.call(this);
		},

		/**
		 * Register a {@link Zarafa.core.data.IPMStore IPMStore} with the {@link Zarafa.core.data.IPMStoreMgr IPMStoreMgr}.
		 * @param {Zarafa.core.data.IPMStore} IPMStore the {@link Zarafa.core.data.IPMStore IPMStore} which must be registered.
		 * @param {Boolean} serverOnly True to register the {@link Zarafa.core.data.IPMStore IPMStore} only for
		 * events originating directly from the server.
		 */
		register : function(IPMStore, serverOnly)
		{
			if (!serverOnly) {
				IPMStore.on('beforesave', this.onBeforeSave, this);
				IPMStore.on('save', this.onSave, this);
				IPMStore.on('update', this.onUpdate, this);
				IPMStore.on('remove', this.onRemove, this);
			}
			IPMStore.on('write', this.onWrite, this);
			IPMStore.on('exception', this.onException, this);

			this.IPMStores.add(IPMStore);
		},

		/**
		 * Unregister a {@link Zarafa.core.data.IPMStore IPMStore} from the {@link Zarafa.core.data.IPMStoreMgr IPMStoreMgr}.
		 * @param {Zarafa.core.data.IPMStore} IPMStore the {@link Zarafa.core.data.IPMStore IPMStore} which must be unregistered.
		 * @param {Boolean} serverOnly True if the {@link Zarafa.core.data.IPMStore IPMStore} was {@link #register registered}
		 * with the serverOnly argument.
		 */
		unregister : function(IPMStore, serverOnly)
		{
			if (!serverOnly) {
				IPMStore.un('beforesave', this.onBeforeSave, this);
				IPMStore.un('save', this.onSave, this);
				IPMStore.un('update', this.onUpdate, this);
				IPMStore.un('remove', this.onRemove, this);
			}
			IPMStore.un('write', this.onWrite, this);
			IPMStore.un('exception', this.onException, this);

			this.IPMStores.remove(IPMStore);
		},

		/**
		 * Filter the list of {@link Zarafa.core.data.IPMRecord records} to contain
		 * only those which can be {@link Zarafa.core.data.IPMRecord#eventPropagation propagated}
		 * over a new event to other {@link Zarafa.core.data.IPMStore stores}.
		 * 
		 * @param {Zarafa.core.data.IPMRecord/Array} records The record or records to filter
		 * out the non-propagatable records.
		 * @return {Zarafa.core.data.IPMRecord/Array} All propagatable records
		 * @private
		 */
		getPropagatableRecords : function(records)
		{
			var propagateRecords = [];

			if (!Ext.isArray(records)) {
				records = [ records ]
			}
	
			for (var i = 0, len = records.length; i < len; i++) {
				var record = records[i];

				if (record instanceof Zarafa.core.data.IPMRecord && record.hasEventPropagation()) {
					propagateRecords.push(record);
				}
			}

			return propagateRecords;
		},

		/**
		 * Event handler which is triggered when a {@link Zarafa.core.data.IPMRecord record} is about to
		 * be saved in a {@link Zarafa.core.data.IPMStore store}. This function will inform the
		 * {@link Zarafa.core.data.IPMStore IPMStores} about the event through the
		 * {@link #beforerecordsave} event.
		 *
		 * @param {Zarafa.core.data.IPMStore} IPMStore
		 * @param {Object} data An object containing the data that is to be saved.
		 * The object will contain a key for each appropriate action, with an array
		 * of updated data for each record.
		 * @private
		 */
		onBeforeSave : function(IPMStore, data)
		{
			var propagateData = {}; 

			for (var key in data) {
				var propagateRecords = this.getPropagatableRecords(data[key]);
				if (!Ext.isEmpty(propagateData)) {
					propagateData[key] = propagateRecords;
				}
			}

			this.fireEvent('beforerecordsave', IPMStore, propagateData);
		},

		/**
		 * Event handler which is triggered when a {@link Zarafa.core.data.IPMRecord record} has been
		 * saved in a {@link Zarafa.core.data.IPMStore store}. This function will inform the
		 * {@link arafa.core.data.IPMStore IPMStores} about the event through the
		 * {@link #afterrecordsave} event.
		 *
		 * @param {Zarafa.core.data.IPMStore} IPMStore
		 * @param {Number} batch The identifier for the batch that was saved.
		 * @param {Object} data An object containing the data that is to be saved.
		 * The object will contain a key for each appropriate action, with an array
		 * of updated data for each record.
		 * @private
		 */
		onSave : function(IPMStore, batch, data)
		{
			this.fireEvent('afterrecordsave', IPMStore, data);
		},

		/**
		 * Event handler which is triggered when a {@link Zarafa.core.data.IPMRecord record} has been
		 * updated from the server. This function will inform the {@link Zarafa.core.data.IPMStore IPMStores}
		 * about the event through the {@link #afterrecordupdate} event.
		 *
		 * @param {Zarafa.core.data.IPMStore} IPMStore
		 * @param {Zarafa.core.data.IPMRecord} record The Record which has been updated
		 * @param {String} operation  The update operation being performed. 
		 * ({@link Ext.data.Record#EDIT}, {@link Ext.data.Record#REJECT}, {@link Ext.data.Record#COMMIT}).
		 * @private
		 */
		onUpdate : function(IPMStore, record, operation)
		{
			if (record.hasEventPropagation()) {
				this.fireEvent('afterrecordupdate', IPMStore, record, operation);
			}
		},

		/**
		 * Event handler which is triggered when a {@link Zarafa.core.data.IPMRecord record} has been
		 * removed from the server. This function will inform the {@link Zarafa.core.data.IPMStore IPMStores}
		 * about the event through the {@link #recordremove} event.
		 *
		 * @param {Zarafa.core.data.IPMStore} IPMStore
		 * @param {Zarafa.core.data.IPMRecord} record The Record which has been updated
		 * @param {String} The index at which the record was removed
		 * @private
		 */
		onRemove : function(IPMStore, record, index)
		{
			if (record.hasEventPropagation()) {
				this.fireEvent('recordremove', IPMStore, record);
			}
		},

		/**
		 * Event handler will be called on successfull completion of any CRUD operation,
		 * the difference between this event and {@link #save} event is that {@link #save}
		 * event will pass only data set that is modified not the record that is modified.
		 * so this event removes burden of finding record from the record set.
		 *
		 * @param {Zarafa.core.data.IPMStore} store The {@link Zarafa.core.data.IPMStore store} to which the
		 * {@link Zarafa.core.data.IPMRecord record} belongs.
		 * @param {String} action [Ext.data.Api.actions.create|update|destroy|open]
		 * @param {Object} data The 'data' picked-out out of the response for convenience.
		 * @param {Ext.Direct.Transaction} res
		 * @param {Zarafa.core.data.IPMrecord|Array} records The most recent version of the records
		 * which came from the server.
		 * @private
		 */
		onWrite : function(IPMStore, action, data, transaction, records)
		{
			var propagateRecords = this.getPropagatableRecords(records);
			if (!Ext.isEmpty(propagateRecords)) {
				this.fireEvent('afterrecordwrite', IPMStore, action, data, transaction, propagateRecords);
			}
		},

		/**
		 * Event handler which will be called when the store has fired the {@link Ext.data.Store#exception} event.
		 * This will look up which store has exactly fired the event, and will fire the {@link #exception} event.
		 * @param {Ext.data.DataProxy} proxy The proxy from where the exception event originated
		 * @param {String} type The value of this parameter will be either 'response' or 'remote'.
		 * @param {String} action Name of the action.
		 * @param {Object} options The options for the action that were specified in the request.
		 * @param {Object} response If the 'type' argument was 'response' this will be the raw browser response object,
		 * otherwise it will be the decoded response object
		 * @param {Mixed} arg (optional) Additional arguments for the exception
		 * @private
		 */
		onException : function(proxy, type, action, options, response, arg)
		{
			var proxies = Ext.pluck(this.IPMStores.items, 'proxy');
			if (!Ext.isEmpty(proxies)) {
				var storeIndex = proxies.indexOf(proxy);
				this.fireEvent('storeexception', this.IPMStores.get(storeIndex), proxy, type, action, options, response, arg);
			}
		},

		/**
		 * Executes the specified function once for every {@link Zarafa.core.data.IPMStore store} in the collection,
		 * passing the following arguments:
		 * <div class="mdetail-params"><ul>
		 * <li><b>item</b> : Zarafa.core.data.IPMStore<p class="sub-desc">The collection item</p></li>
		 * <li><b>index</b> : Number<p class="sub-desc">The item's index</p></li>
		 * <li><b>length</b> : Number<p class="sub-desc">The total number of items in the collection</p></li>
		 * </ul></div>
		 * The function should return a boolean value. Returning false from the function will stop the iteration.
		 * @param {Function} fn The function to execute for each item.
		 * @param {Object} scope (optional) The scope (<code>this</code> reference) in which the function is executed.
		 * Defaults to the current item in the iteration.
		 */
		each : function(fn, scope)
		{
			this.IPMStores.each.apply(this, arguments);
		},

		/**
		 * Obtain a list of {@link Zarafa.core.data.IPMStore stores} which have one or more of the
		 * requested {@link Zarafa.core.date.IPFRecord folders} currently
		 * {@link Zarafa.core.data.IPMStore#containsFolderInLastLoad loaded}.
		 * @param {Array} folders The list of {@link Zarafa.core.data.IPFRecord folders} or
		 * {@link Zarafa.hierarchy.data.MAPIFolderRecord#getId folder entryIds}
		 * for which the {@link Zarafa.core.data.IPMStore stores} are requested.
		 * @return {Array} The array of {@link Zarafa.core.data.IPMStore stores} which have the
		 * one or more of the requested folders loaded.
		 */
		getStoresForFolders : function(folders)
		{
			var stores = [];

			if (!Ext.isArray(folders)) {
				folders = [ folders ];
			}

			if (folders[0] instanceof Zarafa.core.data.IPFRecord) {
				folders = Ext.pluck(folders, 'id');
			}

			this.IPMStores.each(function(store) {
				if (store.containsFolderInLastLoad(folders)) {
					stores.push(store);
				}
			}, this);

			return stores;
		}
});

Zarafa.core.data.IPMStoreMgr = new Zarafa.core.data.IPMStoreMgr();
Ext.namespace('Zarafa.core.data');

/**
 * @class Zarafa.core.data.JsonReader
 * @extends Ext.data.JsonReader
 *
 * This extension of the {@link Ext.data.JsonReader} supports {@link Zarafa.core.data.IPMStore stores}
 * which can hold different type of {@link Zarafa.core.data.IPMRecord records}.
 *
 * If in the constructor no recordType is provided, dynamic {@link Zarafa.core.data.IPMRecord record} type
 * support is assumed. With dynamic types, the incoming response data is used to determine which
 * {@link Zarafa.core.data.MAPIRecord MAPIRecord} must be constructed for each individual root element.
 */
Zarafa.core.data.JsonReader = Ext.extend(Ext.data.JsonReader, {
	/**
	 * In {@link #getEfMapping} we generate a mapping of all Record Fields per ObjectType,
	 * all mappings are stored in a special cache to prevent them from being regenerated
	 * each time.
	 * @property
	 * @type Object
	 * @private
	 */
	efCache : undefined,

	/**
	 * @cfg {Boolean} dynamicRecord Enable dynamic detection of the records
	 * which are read by this JsonReader. When enabled this will prefer using
	 * the {@link Zarafa.core.data.RecordFactory} to detect the recordType rather
	 * then using the {@link #recordType} directly. (defaults to true)
	 */
	dynamicRecord : true,

	/**
	 * @constructor
	 * @param {Object} meta Metadata configuration options.
	 * @param {Object} recordType (optional) Optional Record type matches the type
	 * which must be read from response. If no type is given, {@link Zarafa.core.data.JsonReader}
	 * will dynamicly detect the record type based on the response.
	 */
	constructor : function(meta, recordType)
	{
		meta = Ext.applyIf(meta || {}, {
			totalProperty : 'count',
			root : 'item',
			id : 'entryid',
			idProperty : 'entryid'
		});

		// Check if the meta object contained the successProperty.
		// Note that this must be called before the superclass constructor,
		// because the meta object will be altered there.
		var hasSuccessProperty = Ext.isDefined(meta.successProperty);

		// If no recordType is provided, enable the dynamic record handling
		// of the JsonReader.
		if (!Ext.isDefined(recordType)) {
			// FIXME: We shouldn't for IPM as base recordclass here, instead
			// this should be handled as configuration option, or something even smarter.
			recordType = Zarafa.core.data.RecordFactory.getRecordClassByMessageClass('IPM');
		}

		// Check we dynamic records are disabled. 
		if (Ext.isDefined(meta.dynamicRecord)) {
			this.dynamicRecord = meta.dynamicRecord;
		}

		if (this.dynamicRecord !== false) {
			this.efCache = {};
		}

		Zarafa.core.data.JsonReader.superclass.constructor.call(this, meta, recordType);

		// This fixes a bug in the Ext.data.JsonReader. Even when no successProperty
		// is given, the getSuccess function will still be implemented after which
		// the function will often fail due to the lack of the success property within
		// the response data.
		if (!hasSuccessProperty)
			this.getSuccess = function(o) { return true; };
	},

	/**
	 * Build the extractors which are used when reading the Json data. This initialized
	 * functions like {@link #getTotal}, {@link #getSuccess}, {@link #getId}.
	 * @private
	 */
	buildExtractors : function()
	{
		var s = this.meta;

		Zarafa.core.data.JsonReader.superclass.buildExtractors.call(this);

		// Wrap the original getId function to check if the data is the raw
		// data which has wrapped the 'props' field, or if this is the unwrapped
		// data.
		if (s.id || s.idProperty) {
			var old = this.getId;
			this.getId = function(rec) {
				if (rec.props) {
					var id = old(rec.props);
					if (!Ext.isEmpty(id)) {
						return id;
					}
				}

				return old(rec);
			};
		}
	},

	/**
	 * Obtain the mapping between response objects and {@link Zarafa.core.data.IPMRecord record} fields for
	 * the given recordType. If no mapping yet exist for this recordType one will be constructed
	 * and added to the {@link Zarafa.core.data.IPMRecord.efCache cache}.
	 *
	 * @param {String} key The unique key for this record type (used for caching purposes).
	 * @param {Array} items The array of {@link Zarafa.core.data.IPMRecord record} items.
	 * @param {Number} len The length of the items array. 
	 * @return {Array} The name/value list of response to {@link Zarafa.core.data.IPMRecord record} fields.
	 * @private
	 */
	getEfMapping : function(key, items, len)
	{
		if (Ext.isString(key)) {
			key = key.toUpperCase();
		}

		var ef = this.efCache[key];

		if (!Ext.isDefined(ef))
		{
			ef = [];
        	for(var i = 0; i < len; i++){
				var f = items[i];
				var map = (!Ext.isEmpty(f.mapping)) ? f.mapping : f.name;
				ef.push(this.createAccessor.call(this, map));
			}
			this.efCache[key] = ef;
		}

		return ef;
	},

	/**
	 * Type-casts a single row of raw-data from server
	 * @param {Object} data The data object which must be deserialized.
	 * @param {Array} items The {@link Ext.data.Field Field} used for deserialization.
	 * @param {Integer} len The length of the items array.
	 * @private
	 */
	extractValues : function(data, items, len)
	{
		var values = {};
	
		// If the data object is wrapped (it contains objects like 'props', 'attachments',
		// 'recipients', etc... Then we must call extractValues for each individual subobject.
		if (Ext.isDefined(data.props)) {
			values = Ext.apply({}, data);
			values.props = this.extractValues(data.props, items, len);
			return values;
		}

		if (this.dynamicRecord == true)
		{
			var recordType = Zarafa.core.data.RecordFactory.getRecordClassByRecordData(data);
			if (!Ext.isDefined(recordType))
				recordType = this.recordType;

			items = recordType.prototype.fields.items;
			len = recordType.prototype.fields.length;

			// Normally the caller has initialized this.ef for us, but only at this time
			// do we know the real recordType. As such we have to override the previous
			// this.ef mapping.
			this.ef = this.getEfMapping(data.message_class || data.object_type, items, len);
		}

		// Extract the values per object which we want to deserialize.
		for (var j = 0; j < len; j++) {
			var f = items[j];
			var v = this.ef[j](data);
			if (Ext.isDefined(v))
				values[f.name] = f.convert(v, data);
		}

		return values;
	},

	/**
	 * Returns extracted, type-cast rows of data.  Iterates to call #extractValues for each row
	 *
	 * This function is exactly copied from {@link Ext.data.DataReader.extractData} with the only
	 * difference is using the RecordFactory for record allocation.
	 *
	 * @param {Object|Array} data-root from server response
	 * @param {Boolean} returnRecords [false] Set true to return instances of {@link Zarafa.core.data.MAPIRecord MAPIRecord}.
	 * @private
	 */
	extractData : function(root, returnRecords)
	{
		// A bit ugly this, too bad the Record's raw data couldn't be saved in a common property named "raw" or something.
		var rawName = (this instanceof Ext.data.JsonReader) ? 'json' : 'node';

		var rs = [];

		// Had to add Check for XmlReader, #isData returns true if root is an Xml-object.  Want to check in order to re-factor
		// #extractData into DataReader base, since the implementations are almost identical for JsonReader, XmlReader
		if (this.isData(root) && !(this instanceof Ext.data.XmlReader)) {
			root = [root];
		}
		if (returnRecords === true) {
			for (var i = 0; i < root.length; i++) {
				var n = root[i];

				var record = undefined;
				var id = this.getId(n);
				var data = n.props || n;

				// Clear all data from the object which must be deserialized,
				// we only want the 'object_type' and 'message_class' properties.
				data = { message_class : data.message_class, object_type : data.object_type };

				if (this.dynamicRecord == true) {
					record = Zarafa.core.data.RecordFactory.createRecordObjectByRecordData(data, id);
				}

				if (!record) {
					record = new this.recordType({}, id);
				}

				// move primitive properties and identification properties at same level
				this.moveIdProperties(n, record.baseIdProperties);

				var f		= record.fields,
					fi		= f.items,
					fl		= f.length;
				this.update(record, this.extractValues(n, fi, fl));
				record[rawName] = n;    // <-- There's implementation of ugly bit, setting the raw record-data.
				rs.push(record);
			}
		} else {
			for (var i = 0; i < root.length; i++) {
				var n = root[i];

				var Record = undefined;
				if (this.dynamicRecord == true) {
					Record = Zarafa.core.data.RecordFactory.getRecordClassByRecordData(n.props || n);
				}
				
				// Fall back to specified record type if we can't get the type from the data
				if (!Record) {
					Record = this.recordType;
				}

				// move primitive properties and identification properties at same level
				this.moveIdProperties(n, Record.prototype.baseIdProperties);

				var f		= Record.prototype.fields,
					fi		= f.items,
					fl		= f.length;

				// here we can't do anything about complex structures so its ignored here
				var data = this.extractValues(n, fi, fl);
				data[this.meta.idProperty] = this.getId(n.props || n);
				rs.push(data);
			}
		}
		return rs;
	},

	/**
	 * Function will merge all identification properties and primitive properties
	 * to the props field and return the merged data. so {@link Zarafa.core.JsonReader JsonReader}
	 * can read the data and extract the properties.
	 * @param {Object} data data that is passed to {@link #extractData} to extract properties.
	 * @param {String|Array} idProperties The id properties which should be moved into the properties object.
	 * @return {Object} The updated data object.
	 */
	moveIdProperties : function(data, idProperties)
	{
		// If there is not data then return no data
		if (!data) {
			return data;
		}

		if (!data.props) {
			data.props = {};
		}

		// move the base identification property to <props> tag level
		var idProperty = this.meta.idProperty;
		if (idProperty) {
			var value = data[idProperty];
			if (Ext.isDefined(value)) {
				data.props[idProperty] = value;
				delete data[idProperty];
			}
		}

		// move all extra identification properties to <props> tag level
		if (Ext.isString(idProperties)) {
			var value = data[idProperties];
			if (Ext.isDefined(value)) {
				data.props[idProperties] = value;
				delete data[idProperties];
			}
		} else if (idProperties) {
			for (var i = 0, len = idProperties.length; i < len; i++) {
				var idProp = idProperties[i];
				var value = data[idProp];

				if(Ext.isDefined(value)) {
					data.props[idProp] = value;
					delete data[idProp];
				}
			}
		}

		return data;
	},

	/**
	 * Used for un-phantoming a record after a successful database insert.
	 * Sets the records pk along with new data from server.
	 * You must return at least the database pk using the idProperty defined in
	 * your DataReader configuration. The incoming data from server will be merged
	 * with the data in the local record. In addition, you must return record-data
	 * from the server in the same order received. Will perform a commit as well,
	 * un-marking dirty-fields. Store's "update" event will be suppressed.
	 *
	 * @param {Record/Record|Array} record The phantom record to be realized.
	 * @param {Object/Object|Array} data The new record data to apply. Must include the primary-key from database defined in idProperty field.
	 * @private
	 */
	realize : function(record, data)
	{
		// This function is copy & pasted from Ext.js Ext.data.JsonReader#realize.
		// Our only difference is the assignment of the record.data field.
		if (Ext.isArray(record)) {
			for (var i = record.length - 1; i >= 0; i--) {
				// recurse
				if (Ext.isArray(data)) {
					this.realize(record.splice(i,1).shift(), data.splice(i,1).shift());
				} else {
					// weird...record is an array but data isn't??  recurse but just send in the whole invalid data object.
					// the else clause below will detect !this.isData and throw exception.
					this.realize(record.splice(i,1).shift(), data);
				}
			}
		} else {
			// If records is NOT an array but data IS, see if data contains just 1 record.  If so extract it and carry on.
			if (Ext.isArray(data)) {
				data = data.shift();
			}
			if (!this.isData(data)) {
				// TODO: Let exception-handler choose to commit or not rather than blindly records.commit() here.
				// record.commit();
				throw new Ext.data.DataReader.Error('realize', record);
			}
			record.phantom = false; // <-- That's what it's all about
			record._phid = record.id;  // <-- copy phantom-id -> _phid, so we can remap in Store#onCreateRecords
			record.id = this.getId(data);

			// And now the infamous line for which we copied this entire function.
			// Extjs expects that we transfer _all_ properties back from the server to the client after
			// a new item was created. Since this has a negative impact on the performance.
			//
			// This has been solved to let the server-side only return the properties which have
			// changed, or are new (like the entryid). We can then simply use Ext.apply to apply
			// the updated data over the already available data.
			//
			// But for those who have paid attention in the data flow of Extjs, know that this
			// sounds quite a lot like the function description of Ext.data.JsonReader#update.
			// 
			// So to make everything even simpler, we don'e update the record.data object here,
			// but instead we simply continue to the Ext.data.JsonReader#update function to
			// handle the rest of the work. 
			//
			//record.data = data;

			// Since we postpone the record.data update, there is no need to commit,
			// this too is done in update().
			//record.commit();

			// Time for the real work...
			this.update(record, data);

			// During realize the record might have received a new
			// id value. We have to reMap the store to update the keys.
			if (record.store) {
				record.store.reMap(record);
			}
		}
	},

	/**
	 * Used for updating a non-phantom or "real" record's data with fresh data from
	 * server after remote-save. If returning data from multiple-records after a batch-update,
	 * you must return record-data from the server in the same order received.
	 * Will perform a commit as well, un-marking dirty-fields. Store's "update" event
	 * will be suppressed as the record receives fresh new data-hash
	 *
	 * @param {Record/Record|Array} record 
	 * @param {Object/Object|Array} data
	 * @private
	 */
	update : function(record, data)
	{
		// Recursively call into update to update each record individually.
		if (Ext.isArray(record)) {
			for (var i = 0; i < record.length; i++) {
				if(Ext.isArray(data)) {
					this.update(record[i], data[i]);
				} else {
					this.update(record[i], data);
				}
			}
			return;
		}

		// It can happen that the data is wrapped in a array of length 1.
		if (Ext.isArray(data))
			data = data.shift();

		if (this.isData(data)) {
			// move primitive properties and identification properties at same level
			data = this.moveIdProperties(data, record.baseIdProperties);

			// All preprocessing has been done. All remaining data
			// can be applied into the IPMRecord directly.
			record.data = Ext.apply(record.data, data.props || data);

			// scalar values from props are applied so remove it from data object
			delete data.props;

			// Put the action response from the server in the record
			if(data.action_response){
				record.action_response = data.action_response;
				delete data.action_response
			}

			// If the record contains substores to store complex data then we have to first 
			// serialize those data into its consecutive stores and then we can continue
			// with normal processing
			Ext.iterate(data, function(key, value) {
				if (Ext.isArray(value) || Ext.isObject(value)) {
					var store;

					if (record.supportsSubStore(key)) {
						store = record.getSubStore(key);
						if (!store) {
							store = record.createSubStore(key);
						} else {
							store.removeAll(true);
						}

						// Load data into the SubStore, and remove
						// it from the data object.
						if (!Ext.isEmpty(value)) {
							store.loadData(value);
							delete data[key];
						}
					}
				}
			}, this);

			// Discard any additional data which was set on the data object,
			// this data has probably been set by plugins, but they have sufficient
			// alternatives to fit their custom data into the IPMRecord structure.
		}

		record.commit();
	}
});
Ext.namespace('Zarafa.core.data');

/**
 * @class Zarafa.core.data.JsonWriter
 * @extends Ext.data.JsonWriter
 */
Zarafa.core.data.JsonWriter = Ext.extend(Ext.data.JsonWriter, {
	/**
	 * @constructor
	 * @param {Object} config Configuration object
	 */
	constructor : function(config)
	{
		config = config || {};

		Ext.applyIf(config, {
			writeAllFields : false,
			// FIXME: Disable automatic encoding for now,
			// the MAPIProxy needs an individuall encoded string
			// for each record in the request. We might want to
			// fix this in the future though.
			encode : false
		});

		Zarafa.core.data.JsonWriter.superclass.constructor.call(this, config);
	},

	/**
	 * Render the data in the data object which will be {@link Ext.encode encoded}
	 * and send over the protocol to the server after this function call. During
	 * rendering all {@link Date date} objects will be converted to UNIX timestamps.
	 * This will prevent ExtJs/JSON specific encoding functions to convert the
	 * date object into a "YYYY-MM-DDTHH:MM:SS" timestring.
	 * @param {Object/Array} data The object which musted be rendered
	 * @private
	 */
	renderData : function(data)
	{
		if (Ext.isArray(data)) {
			for (var i = 0, len = data.length; i < len; i++) {
				this.renderData(data[i]);
			}
			return;
		}

		Ext.iterate(data, function(key, value) {
			if (Ext.isDate(value)) {
				data[key] = Math.floor(value.getTime() / 1000);
			}
			if (Ext.isObject(value)) {
				this.renderData(value);
			}
		}, this);
	},

	/**
	 * Final action of a write event.  Apply the written data-object to params.
	 * This function is extended from {@link Ext.data.JsonWriter Extjs}, to use
	 * {@link #renderData} to add some extra data conversions before encoding
	 * the data by {@link Ext.encode Ext.encode}.
	 * @param {Object} http params-object to write-to.
	 * @param {Object} baseParams as defined by {@link Ext.data.Store#baseParams}.
	 * The baseParms must be encoded by the extending class, eg: {@link Ext.data.JsonWriter}, {@link Ext.data.XmlWriter}.
	 * @param {Object/Object|Array} data Data-object representing compiled Store-recordset.
	 */
	render : function(params, baseParams, data)
	{
		// Apply the parameters into the data object, this allows
		// optional data to be send to the server.
		Ext.apply(data, baseParams, params);

		// Apply special rendering to convert all objects
		this.renderData(data);
		Zarafa.core.data.JsonWriter.superclass.render.call(this, params, baseParams, data);
	},

	/**
	 * Adds special function for serialization needed when openening
	 * a record. We can use the default {@link Zarafa.core.data.JsonWriter.toIdHash toIdHash}
	 * function.
	 *
	 * @param {Ext.data.Record} record
	 * @return {Object}
	 * @private
	 */
	openRecord : function(record)
	{
		return this.toIdHash(record);
	},

	/**
	 * Rather then using the regular {@link Ext.data.JsonWriter#toHash toHash}
	 * function, this will use the specialized {@link Zarafa.core.data.JsonWriter#toPropHash toPropHash}
	 * function.
	 *
	 * @param {Ext.data.Record} record
	 * @return {Object}
	 * @override
	 * @private
	 */
	createRecord : function(record)
	{
		return this.toPropHash(record);
	},

	/**
	 * Rather then using the regular {@link Ext.data.JsonWriter#toHash toHash}
	 * function, this will use the specialized {@link Zarafa.core.data.JsonWriter#toPropHash toPropHash}
	 * function.
	 *
	 * @param {Ext.data.Record} record
	 * @return {Object}
	 * @override
	 * @private
	 */
	updateRecord : function(record)
	{
		return this.toPropHash(record);
	},

	/**
	 * Use the {@link Zarafa.core.data.JsonWriter#toIdHash toIdHash} function for creating the hash.
	 *
	 * @param {Ext.data.Record} record
	 * @return {Object}
	 * @override
	 * @private
	 */
	destroyRecord : function(record)
	{
		return this.toIdHash(record);
	},

	/**
	 * Similar to {@link Ext.data.JsonWriter#toHash}
	 *
	 * This will limit the serialization to only the ID properties and message
	 * action commands for the given {@link Zarafa.core.data.IPMRecord record}.
	 *
	 * @param {Ext.data.Record} record The record to hash
	 * @param {Boolean} allowEmpty True to allow empty ID elements to be send
	 * @return {Object} The hashed object
	 * @private
	 */
	toIdHash : function(record, allowEmpty)
	{
		var hash = {};

		Ext.each(record.getIdProps(), function(idProp) {
			var id = record.get(idProp);
			if (allowEmpty || Ext.isDefined(id))
				hash[idProp] = id;
		}, this);

		this.addMessageActionsHash(hash, record);

		return hash;
	},

	/**
	 * Similar to {@link Ext.data.JsonWriter#toHash}
	 *
	 * Besides serializing the data itself, it will insert
	 * the recipients, attachments and message action commands
	 * into the object data.
	 *
	 * @param {Ext.data.Record} record The record to hash
	 * @return {Object} The hashed object
	 * @private
	 */
	toPropHash : function(record)
	{
		var hash = this.toIdHash(record, false);

		// FIXME: How to pass on deleted properties?
		hash.props = this.toHash.call(this, record);

		// FIXME: remove identification entryids from props,
		// in the future Extjs will support the 'config'
		// argument to toHash which we can use the filter
		// out the ID properties...
		this.removeIdHashFromProps(hash, record);

		// Add additional information from the subStores into the hash
		for (var key in record.subStores) {
			if (record.supportsSubStore(key) === true) {
				var store = record.subStores[key];

				if (store && store.writer) {
					Ext.apply(hash, store.writer.toPropHash(record)); 
				}
			}
		}

		this.addMessageActionsHash(hash, record);

		return hash;
	},

	/**
	 * remove additional identification properties from the props using the
	 * {@link Zarafa.core.data.JsonWriter.idProperties idProperties}
	 * field.
	 *
	 * @param {Object} hash The hash into which the identification fields must be added
	 * @param {Zarafa.core.data.IPMrecord} record The record to serialize from
	 * @private
	 */
	removeIdHashFromProps : function(hash, record)
	{
		Ext.each(record.getIdProps(), function(idProp) {
			if(Ext.isDefined(hash.props) && Ext.isDefined(hash.props[idProp]))
				delete hash.props[idProp];
		}, this);
	},

	/**
	 * Add message actions into the hash. Message actions are not properties
	 * which come from the server, but are used to add an additional action
	 * instruction for the server to perform. As such the action needs to
	 * be serialized seperately into the hash object.
	 *
	 * @param {Object} hash The hash into which the message actions must be added
	 * @param {Zarafa.core.data.IPMrecord} record The record to serialize from
	 * @private
	 */
	addMessageActionsHash : function(hash, record)
	{
		var actions = record.getMessageActions();
		var message_action = {};

		// No Message actions defined
		if (!Ext.isDefined(actions))
			return;

		for (var key in actions) {
			if (Ext.isDefined(actions[key])) {
				message_action[key] = actions[key];
			}
		}

		hash.message_action = message_action;
	}
});
Ext.namespace('Zarafa.core.data');

/**
 * @class Zarafa.core.data.MAPIProxy
 * @extends Ext.data.DataProxy
 */
Zarafa.core.data.MAPIProxy = Ext.extend(Ext.data.DataProxy, {
	/**
	 * @cfg {String} listModuleName Name of the listModule on the server.
	 */
	listModuleName : undefined,
	/**
	 * @cfg {String} itemModuleName Name of the itemModule on the server.
	 */
	itemModuleName : undefined,

	/**
	 * Currently active requests for {@link Zarafa.core.data.MAPIProxy MAPIProxy} mapped by the
	 * action type used in requests, {@link Zarafa.core.data.MAPIProxy MAPIProxy} can use this
	 * active request data to abort any previous request and start a new request.
	 * @property
	 * @type Object
	 * @private
	 */
	activeRequestMapping : undefined,

	/**
	 * The {@link Date#getTime timestamps} for the last time a response was received for
	 * a given {@link Zarafa.core.Actions action}.
	 * @property
	 * @type Object
	 * @private
	 */
	lastResponseTime : undefined,

	/**
	 * @constructor
	 * @param {Object} config Configuration object
	 */
	constructor : function(config)
	{
		Ext.apply(this, config);
		Zarafa.core.data.MAPIProxy.superclass.constructor.call(this, config);

		this.activeRequestMapping = {};
		this.lastResponseTime = {};
	},

	/**
	 * Check if the given action has registered requestIds in the {@link #activeRequestMapping}.
	 * When this is the case, the action is considered to be active. When no action is passed,
	 * this function will check if there are any requestsIds pending for any action.
	 *
	 * @param {Zarafa.core.Action} action The action which is to be checked.
	 * @return {Boolean} True if the given action has registered requestIds.
	 */
	isExecuting : function(action)
	{
		if (Ext.isEmpty(action)) {
			return !Ext.isEmpty(Object.keys(this.activeRequestMapping));
		} else {
			return !Ext.isEmpty(this.activeRequestMapping[action]);
		}
	},

	/**
	 * Register a requestId to a particular action. This will update {@link activeRequestMapping}
	 * to contain the requestId for the given action. By this registration it is possible to
	 * track all current outstanding requests, and it is possible to cancel them using {@link #cancelRequests}.
	 * @param {Zarafa.core.Actions} action The action for which this request id was generated
	 * @param {String} requestId The unique id which was given to the request
	 */
	addRequestId : function(action, requestId)
	{
		if (!Ext.isDefined(this.activeRequestMapping[action])) {
			this.activeRequestMapping[action] = [ requestId ];
		} else {
			this.activeRequestMapping[action].push(requestId);
		}
	},

	/**
	 * Remove a requestId from a particular action. This will update {@link activeRequestMapping}
	 * to remove the requestId from the given action.
	 * @param {Zarafa.core.Actions} requestId The unique id which was given to the request.
	 */
	deleteRequestId : function(requestId)
	{
		for (var key in this.activeRequestMapping) {
			if (Ext.isArray(this.activeRequestMapping[key])) {
				this.activeRequestMapping[key].remove(requestId);
				if (Ext.isEmpty(this.activeRequestMapping[key])) {
					delete this.activeRequestMapping[key];
				}
			}
		}
	},

	/**
	 * Cancel all requests made by this proxy for a particular action.
	 * This will call {@link Zarafa.core.Request#cancelActiveRequest} to cancel
	 * the response handling of all requests which were send out by this proxy for
	 * the given action.
	 * @param {Zarafa.core.Actions} action The action
	 * @protected
	 */
	cancelRequests : function(action)
	{
		if (this.activeRequestMapping[action]) {
			var requests = this.activeRequestMapping[action];

			for (var i = 0, len = requests.length; i < len; i++) {
				container.getRequest().cancelActiveRequest(requests[i]);
			}

			delete this.activeRequestMapping[action];
		}
	},

	/**
	 * Update the {@link #lastResponseTime} with the {@link Date#getTime timestamp}
	 * of when the response for the given action was received.
	 * @param {Zarafa.core.Actions} action The action
	 * @param {Number} timestamp The timestamp
	 */
	updateExecutionTimestamp : function(action, timestamp)
	{
		this.lastResponseTime[action] = timestamp;
	},

	/**
	 * Obtain the {@link Date#getTime timestamp} of the last time
	 * the given {@link #updateExecutionTimestamp action was executed}.
	 * It will mark the time of when the action was completed (because the
	 * response was returned from the server).
	 * @param {Zarafa.core.Actions} action The action
	 * @return {Number} The timestamp of the last action
	 */
	lastExecutionTime : function(action)
	{
		return this.lastResponseTime[action] || 0;
	},

	/**
	 * Obtain the name of the listModule for communication with PHP.
	 * @param {Zarafa.core.data.MAPIRecord} record the record for which the listModuleName is requested
	 * @return {String} The name of the listModule.
	 * @private
	 */
	getListModuleName : function(record)
	{
		return this.listModuleName;
	},

	/**
	 * Obtain the name of the itemModule for communication with PHP.
	 * @param {Zarafa.core.data.MAPIRecord} record the record for which the itemModuleName is requested
	 * @return {String} The name of the itemModule.
	 * @private
	 */
	getItemModuleName : function(record)
	{
		return this.itemModuleName;
	},

	/**
	 * This will create a {@link Zarafa.core.data.ProxyResponseHandler ProxyResponseHandler} object
	 * which will be used by the {@link Zarafa.core.data.ResponseRouter ResponseRouter} when the
	 * response for the given request has returned.
	 *
	 * @param {String} modulename The modulename which is being accessed with this request
	 * @param {Zarafa.core.Actions} serverAction The action to perform on the server.
	 * @param {Ext.data.Api.action} action name of the action to perform.
	 * @param {Ext.data.Record|Array} records list of records to operate on.
	 * @param {Object} parameters object containing user parameters such as range (pagination) information, sorting information, etc.
	 * @param {Ext.data.DataReader} reader data reader. Converts raw JavaScript objects (in our case) to instances of {@link Ext.data.Record}
	 * @param {Function} callback call back function to call when the request has finished successfully.
	 * @param {Object} scope scope for the call back function.
	 * @param {Object} args arguments object. This will be passed to the call back function on successful read.
	 * @return {Object} An instance of the {@link Zarafa.core.data.ProxyResponseHandler ProxyResponseHandler}
	 * which should be used for this request.
	 * @private
	 */
	getResponseHandlerForRequest : Ext.emptyFn,

	/**
	 * Performs a request for a store. This single entry point carries out all CRUD requests. 
	 * @param {Ext.data.Api.action} action name of the action to perform. One of 'create', 'destroy', 'update', and 'read'.
	 * @param {Ext.data.Record|Array} records list of records to operate on. In case of 'read' this will be ignored.
	 * @param {Object} parameters object containing user parameters such as range (pagination) information, sorting information, etc.
	 * @param {Ext.data.DataReader} reader data reader. Converts raw JavaScript objects (in our case) to instances of {@link Ext.data.Record}
	 * @param {Function} callback call back function to call when the request has finished successfully.
	 * @param {Object} scope scope for the call back function.
	 * @param {Object} args arguments object. This will be passed to the call back function on successful read.
	 */
	request : function(action, records, parameters, reader, callback, scope, args)
	{
		switch (action)
		{
			case 'update':
			case 'create':
				this.createUpdateAction(action, records, parameters, reader, callback, scope, args);
				break;
			case 'destroy':
				this.destroyAction(action, records, parameters, reader, callback, scope, args);
				break;
			case 'read':
				this.readAction(action, records, parameters, reader, callback, scope, args);
				break;
			case 'open':
				this.openAction(action, records, parameters, reader, callback, scope, args);
				break;
		}
	},
	
	/**
	 * @param {Ext.data.Api.action} action name of the action to perform. Either 'create' or 'update'.
	 * @param {Ext.data.Record|Array} records list of records to operate on.
	 * @param {Object} parameters object containing user parameters such as range (pagination) information, sorting information, etc.
	 * @param {Ext.data.DataReader} reader data reader. Converts raw JavaScript objects (in our case) to instances of {@link Ext.data.Record}
	 * @param {Function} callback call back function to call when the request has finished successfully.
	 * @param {Object} scope scope for the call back function.
	 * @param {Object} args arguments object. This will be passed to the call back function on successful read.
	 * @private
	 */
	createUpdateAction : function(action, records, parameters, reader, callback, scope, args)
	{
		this.doRequests(args.actionType || Zarafa.core.Actions['save'], action, records, parameters, reader, callback, scope, args);
	},

	/**
	 * Performs a destroy action on one or more records.
	 * @param {Ext.data.Api.action} action name of the action to perform. Always 'destroy'.
	 * @param {Ext.data.Record|Array} records list of records to operate on.
	 * @param {Object} parameters object containing user parameters such as range (pagination) information, sorting information, etc.
	 * @param {Ext.data.DataReader} reader data reader. Converts raw JavaScript objects (in our case) to instances of {@link Ext.data.Record}
	 * @param {Function} callback call back function to call when the request has finished successfully.
	 * @param {Object} scope scope for the call back function.
	 * @param {Object} args arguments object. This will be passed to the call back function on successful read.
	 * @private
	 */
	destroyAction : function(action, records, parameters, reader, callback, scope, args)
	{
		this.doRequests(args.actionType || Zarafa.core.Actions['delete'], action, records, parameters, reader, callback, scope, args);
	},
	
	/**
	 * Performs a read action on one or more records.
	 * @param {Ext.data.Api.action} action name of the action to perform. Always 'open'.
	 * @param {Ext.data.Record|Array} records list of records to operate on.
	 * @param {Object} parameters object containing user parameters such as range (pagination) information, sorting information, etc.
	 * @param {Ext.data.DataReader} reader data reader. Converts raw JavaScript objects (in our case) to instances of {@link Ext.data.Record}
	 * @param {Function} callback call back function to call when the request has finished successfully.
	 * @param {Object} scope scope for the call back function.
	 * @param {Object} args arguments object. This will be passed to the call back function on successful read.
	 * @private
	 */
	openAction : function(action, records, parameters, reader, callback, scope, args)
	{
		this.doRequests(args.actionType || Zarafa.core.Actions['open'], action, records, parameters, reader, callback, scope, args);
	},
	
	/**
	 * Performs a read action on a Folder/Store to load all records.
	 * @param {Ext.data.Api.action} action name of the action to perform. Always 'read'.
	 * @param {Ext.data.Record|Array} records list of records to operate on. In case of 'read' this will be ignored.
	 * @param {Object} parameters object containing user parameters such as range (pagination) information, sorting information, etc.
	 * @param {Ext.data.DataReader} reader data reader. Converts raw JavaScript objects (in our case) to instances of{@link Ext.data.Record}
	 * @param {Function} callback call back function to call when the request has finished successfully.
	 * @param {Object} scope scope for the call back function.
	 * @param {Object} args arguments object. This will be passed to the call back function on successful read.
	 * @private
	 */
	readAction : function(action, records, parameters, reader, callback, scope, args)
	{
		this.doRequests(args.actionType || Zarafa.core.Actions['list'], action, records, parameters, reader, callback, scope, args);
	},

	/**
	 * Initialize the the {@link Zarafa.core.Request request} structure. The initial
	 * {@link Zarafa.core.Request request} will be reset and new requests will be added
	 * for each individual {@link Zarafa.core.date.MAPIRecord record} which was provided.
	 * @param {Zarafa.core.Actions} serverAction The action to perform on the server.
	 * @param {Ext.data.Api.action} action name of the action to perform.
	 * @param {Ext.data.Record|Array} records list of records to operate on.
	 * @param {Object} parameters object containing user parameters such as range (pagination) information, sorting information, etc.
	 * @param {Ext.data.DataReader} reader data reader. Converts raw JavaScript objects (in our case) to instances of {@link Ext.data.Record}
	 * @param {Function} callback call back function to call when the request has finished successfully.
	 * @param {Object} scope scope for the call back function.
	 * @param {Object} args arguments object. This will be passed to the call back function on successful read.
	 * @private
	 */
	doRequests : function(serverAction, action, records, parameters, reader, callback, scope, args)
	{
		// Check if the previous request needs to be cancelled.
		if (args.cancelPreviousRequest === true) {
			this.cancelRequests(serverAction);
		}

		// reset the request object, starts a new composite request 
		container.getRequest().reset();

		// If records are provided, we must perform a 'itemmodule' action on each of the given records
		if (records && args.listRequest !== true) {
			var items = parameters.jsonData[reader.meta.root];

			// Force the records object to be an array
			if (!Ext.isArray(records)) {
				records = [ records ];
			}

			// Force the serialized data to be an array
			if (!Ext.isArray(items)) {
				items = [ items ];
			}

			for (var i = 0; i < records.length; i++) {
				var record = records[i];
				var data = items[i];
				var module = this.getItemModuleName(record);
				var handler = this.getResponseHandlerForRequest(module, serverAction, action, record, parameters, reader, callback, scope, args);

				// Add the request
				var requestId = container.getRequest().addRequest(module, serverAction, data, handler);

				// store reference of transaction id to active request mapping
				this.addRequestId(serverAction, requestId); 
			}
		} else {
			// No records were provided, we must perform a 'listmodule' action
			var module = this.getListModuleName(records);
			var handler = this.getResponseHandlerForRequest(module, serverAction, action, records, parameters, reader, callback, scope, args);

			// Add the request
			var requestId = container.getRequest().addRequest(module, serverAction, parameters, handler);

			// store reference of transaction id to active request mapping
			this.addRequestId(serverAction, requestId); 
		}

		// send out the request
		container.getRequest().send();
	}
});
Ext.namespace('Zarafa.core.data');

Ext.data.Api.actions.open = 'open';

/**
 * @class Zarafa.core.data.MAPIStore
 * @extends Ext.data.GroupingStore
 * @xtype zarafa.mapistore
 *
 * Extension of the Extjs store which adding support for the 'open' command,
 * which is used by MAPI to request additional data for a record.
 */
Zarafa.core.data.MAPIStore = Ext.extend(Ext.data.GroupingStore, {
	/**
	 * @cfg {Boolean} persistentFilter True when the {@link #filter} which
	 * has been applied on this store should be reapplied when the store
	 * has been {@link #load loaded}
	 */
	persistentFilter : true,

	/**
	 * The currently active function which was given to {@link #filterBy}.
	 * @property
	 * @type Function
	 * @private
	 */
	filterFn : undefined,

	/**
	 * The currently active {@link #filterFn function} scope which was given to {@link #filterBy}.
	 * @property
	 * @type Object
	 * @private
	 */
	filterScope : undefined,

	/**
	 * @constructor
	 * @param config Configuration structure
	 */
	constructor : function(config)
	{
		config = config || {};

		Ext.applyIf(config, {
			// Don't automatically update changes to records to the server.
			autoSave : false,

			// When autoSave is false, indicates that CRUD operations are batched into a single request.
			batch : true
		});

		this.addEvents(
			/**
			 * @event open
			 * Fires when the {@link Zarafa.core.data.MAPIStore MAPIStore} gets extra data for a specific
			 * {@link Zarafa.core.data.MAPIRecord MAPIRecord}.
			 * @param {Zarafa.core.data.MAPIStore} store The {@link Zarafa.core.data.MAPIStore MAPIStore} which issues
			 * open request to get extra data for specific record.
			 * @param {Zarafa.core.data.MAPIRecord} record Record which is being opened to get extra information.
			 */
			'open'
		);

		Zarafa.core.data.MAPIStore.superclass.constructor.call(this, config);

		// Update the getKey function inside the MixedCollection to match
		// the one provided getRecordKey.
		this.data.getKey = this.getRecordKey;

		this.initEvents();
	},

	/**
	 * Initialize all events which Zarafa.core.data.MAPIStore MAPIStore} will listen to.
	 * @protected
	 */
	initEvents : function()
	{
		this.on('beforeload', this.onBeforeLoad, this);
		this.on('add', this.onAdd, this);
		this.on('remove', this.onRemove, this);
		this.on('write', this.onWrite, this);
	},

	/**
	 * The {@link Ext.util.MixedCollection#getKey} function which must be
	 * applied to the {@link #data}{@link Ext.util.MixedCollection#getKey #getKey}
	 * function. This is assigned by the constructor and allows subclasses to
	 * simply override this function rather then apply it manually to {@link #data}
	 * themselves.
	 * @param {Ext.data.Record} o The record for which the key is requested
	 * @return {String} The key by which the record must be saved into the {@link Ext.util.MixedCollection}.
	 * @protected
	 */
	getRecordKey : Ext.util.MixedCollection.prototype.getKey,

	/**
	 * Check if a particular {@link Zarafa.core.Action action} is being executed
	 * by the {@link #proxy} of this store. When no action is given, this function
	 * will check if the proxy is busy with any action.
	 *
	 * @param {Zarafa.core.Action} action The action which is being checked
	 * @return {Boolean} True if the given action is being executed  by the proxy
	 */
	isExecuting : function(action)
	{
		return this.proxy.isExecuting(action);
	},

	/**
	 * Determine if a {@link #isExecuting 'list'} or {@link #isExecuting 'open'}
	 * request is still pending. And if so
	 * {@link Zarafa.core.data.MAPIProxy#cancelRequest cancel} those requests.
	 */
	cancelLoadRequests : function()
	{
		// If we are loading data, we want to cancel
		// the request as we don't want the data anymore.
		if (this.isExecuting('list')) {
			this.proxy.cancelRequests('list');
		}

		// If we are opening the record, we want to cancel
		// the request as we don't want the data anymore.
		if (this.isExecuting('open')) {
			this.proxy.cancelRequests('open');
		}

		// Saving is still interesting as the user might
		// not expect that action to be still pending.
	},

	/**
	 * Get the {@link Date#getTime timestamp} of the last time a response was given
	 * for the given action.
	 * @param {Zarafa.core.Action} action The action which is being checked
	 * @return {Number} The timestamp of the last action time
	 */
	lastExecutionTime : function(action)
	{
		return this.proxy.lastExecutionTime(action);
	},	

	/**
	 * <p>Reloads the Record cache from the configured Proxy. See the superclass {@link Ext.data.Store#reload documentation}
	 * for more detaiils.
	 * During reload we add an extra option into the {@link #load} argument which marks the action as a reload
	 * action.
	 */
	reload : function(options)
	{
		options = Ext.applyIf(options || {}, { reload : true });
		Zarafa.core.data.MAPIStore.superclass.reload.call(this, options);
	},

	/**
	 * Saves all pending changes to the store. See the superclass {@link Ext.data.Store#save documentation}
	 * for more details. Where the superclass saves all {@link #removed} and {@link #modified} records,
	 * this function will only save the records which are passed as argument.
	 *
	 * @param {Zarafa.core.data.MAPIRecord/Zarafa.core.data.MAPIRecord|Array} records The records which
	 * must be saved to the server.
	 * @return {Number} batch Returns a number to uniquely identify the "batch" of saves occurring. -1 will be returned
	 * if there are no items to save or the save was cancelled.
	 */
	save : function(records) {
		// When no records are provided, fall back to the default behavior of the superclass.
		if (!Ext.isDefined(records)) {
			return Zarafa.core.data.MAPIStore.superclass.save.call(this);
		}

		if (!Ext.isArray(records)) {
			records = [ records ];
		}

		if (!this.writer) {
			throw new Ext.data.Store.Error('writer-undefined');
		}

		var destroyed = [],
			created = [],
			updated = [],
			queue = [],
			trans,
			batch,
			data = {};

		for (var i = 0, len = records.length; i < len; i++) {
			var record = records[i];

			if (this.removed.indexOf(record) >= 0) {
				// Check for removed records first, a record located in this.removed is
				// guarenteed to be a non-phantom. See store.remove().
				destroyed.push(record);
			} else if (this.modified.indexOf(record) >= 0) {
				// Only accept valid records.
				if (record.isValid()) {
					if (record.phantom) {
						created.push(record);
					} else {
						updated.push(record);
					}
				}
			}
		}

		if (destroyed.length > 0) {
			queue.push(['destroy', destroyed]);
		}
		if (created.length > 0) {
			queue.push(['create', created]);
		}
		if (updated.length > 0) {
			queue.push(['update', updated]);
		}

		var len = queue.length;
		if(len){
			batch = ++this.batchCounter;
			for(var i = 0; i < len; ++i){
				trans = queue[i];
				data[trans[0]] = trans[1];
			}
			if(this.fireEvent('beforesave', this, data) !== false){
				for(var i = 0; i < len; ++i){
					trans = queue[i];
					this.doTransaction(trans[0], trans[1], batch);
				}
				return batch;
			}
		}
		return -1;
	},

	/**
	 * Event handler which is fired when we are about to (re)load the store.
	 * When this happens we should cancel all pending {@link #open} requests,
	 * as they cannot be completed anymore (the record will have been deleted,
	 * so the opened record has become useless.
	 * @private
	 */
	onBeforeLoad : function()
	{
		if (this.isExecuting('open')) {
			this.proxy.cancelRequests('open');
		}
	},

	/**
	 * Event handler which is raised when a {@link Zarafa.core.data.MAPIRecord MAPIRecord} has been added
	 * to this {@link Zarafa.core.data.MAPIStore MAPIStore}.
	 *
	 * @param {Zarafa.core.data.MAPIStore} store The {@link Zarafa.core.data.MAPIStore MAPIStore} to which the store was added.
	 * @param {Zarafa.core.data.MAPIRecord|Array} records The array of {@link Zarafa.core.data.MAPIRecord records} which have been added.
	 * @param {Number} index The index at which the record(s) were added
	 * @private
	 */
	onAdd : function(store, records, index)
	{
		this.setRecordsStore(store, records);
	},

	/**
	 * Event handler which is raised when a {@link Zarafa.core.data.MAPIRecord MAPIRecord} has been removed
	 * from this {@link Zarafa.core.data.MAPIStore MAPIStore}.
	 *
	 * @param {Zarafa.core.data.MAPIStore} store The {@link Zarafa.core.data.MAPIStore MAPIStore} from which the records were removed.
	 * @param {Zarafa.core.data.MAPIRecord|Array} records The array of {@link Zarafa.core.data.MAPIRecord records} which have been removed.
	 * @param {Number} index The index at which the record(s) were removed.
	 * @private
	 */
	onRemove: function(store, records, index)
	{
		this.setRecordsStore(undefined, records);
	},

	/**
	 * Event handler which is raised when the {@link #write} event has been fired. This will clear
	 * all {@link Zarafa.core.data.MAPIRecord#actions Message Actions} from the given records.
	 *
	 * @param {Zarafa.core.data.MAPIStore} store The store which fired the event
	 * @param {String} action [Ext.data.Api.actions.create|update|destroy]
	 * @param {Object} result The 'data' picked-out out of the response for convenience
	 * @param {Ext.Direct.Transaction} res The transaction
	 * @param {Record/Record|Array} records The records which were written to the server
	 * @private
	 */
	onWrite : function(store, action, result, res, records)
	{
		if (!Ext.isArray(records)) {
			records = [ records ];
		}

		for (var i = 0, len = records.length; i < len; i++) {
			records[i].clearMessageActions();
			records[i].clearActionResponse();
		}
	},

	/**
	 * Iterates through all {@link Zarafa.core.data.MAPIRecord records} and sets the
	 * reference to the {@link Zarafa.core.data.MAPIStore MAPIStore} to which it belongs.
	 *
	 * @param {Zarafa.core.data.MAPIStore} store The {@link Zarafa.core.data.MAPIStore MAPIStore} to which the
	 * {@link Zarafa.core.data.MAPIRecord records} must be assigned.
	 * @param {Zarafa.core.data.MAPIRecord|Array} records The array of
	 * {@link Zarafa.core.data.MAPIRecord records} which must be updated.
	 * @private
	 */
	setRecordsStore : function(store, records)
	{
		records = Ext.isArray(records) ? records : [ records ];
		Ext.each(records, function(record) { record.join(store); }, this);
	},

	/**
	 * Function is used to get extra properties from the server, which are not received in
	 * 'list' action. function will call {@link #execute} event, which is entry point for every
	 * CRUD operation, {@link #execute} will internall call {@link #createCallback} to create a
	 * callback function based on operation type ('open' -> onOpenRecords).
	 * @param {Zarafa.core.data.MAPIRecord} record record for which we need extra properties.
	 * @param {Object} options Extra options which can be used for opening the record
	 */
	open : function(record, options)
	{
		try {
			return this.execute('open', record, options);
		} catch (e) {
			this.handleException(e);
			return false;
		}
	},

	/**
	 * Function will work as callback function for 'open' operation, and update the
	 * existing record with the new data that is received from server.
	 * @param {Boolean} success true if operation completed successfully else false.
	 * @param {Zarafa.core.data.MAPIRecord} record updated record.
	 * @param {Object} data properties of record which is received from server (in key/value pair).
	 */
	onOpenRecords : function(success, record, data)
	{
		if (success === true && this.indexOf(record) > -1) {
			try {
				// call reader to update record data
				var oldRecord = record;
				
				this.reader.update(record, data);
				record.afterOpen();
				
				this.fireEvent('open', this, record, oldRecord);
			} catch (e) {
				this.handleException(e);
				if (Ext.isArray(record)) {
					// Recurse to run back into the try {}.  DataReader#update splices-off the rs until empty.
					this.onOpenRecords(success, record, data);
				}
			}
		}
	},

	/**
	 * Get the Record with the specified id.
	 * If the {@link #reader} has the {@link Ext.data.JsonReader#idProperty} set to 'entryid',
	 * then this function will also use {@link Zarafa.core.EntryId#compareEntryIds}. For
	 * 'store_entryid' then {@link Zarafa.core.EntryId#compareStoreEntryIds} is used.
	 * @param {String} id The id of the Record to find.
	 * @return {Ext.data.Record} The Record with the passed id. Returns undefined if not found.
	 */
	getById : function(id)
	{
		// First use the original implementation
		var item = Zarafa.core.data.MAPIStore.superclass.getById.call(this, id);

		// If no item was found, and the reader uses the 'entryid' property,
		// we should retry searching using the compareEntryIds function. If that
		// fails as well, then the item is really not present.
		if (!item) {
			var index = this.findBy(function(record) { return this.idComparison(id, record.id); }, this);
			if (index >= 0) {
				item = this.getAt(index);
			}
		}

		return item;
	},

	/**
	 * Compare a {@link Ext.data.Record#id ids} to determine if they are equal.
	 * @param {String} a The first id to compare
	 * @param {String} b The second id to compare
	 * @protected
	 */
	idComparison : function(a, b)
	{
		return a === b;
	},

	/**
	 * Filter by a function. Returns a <i>new</i> collection that has been filtered.
	 * The passed function will be called with each object in the collection.
	 * If the function returns true, the value is included otherwise it is filtered.
	 * @param {Function} fn The function to be called, it will receive the args o (the object), k (the key)
	 * @param {Object} scope (optional) The scope (<code>this</code> reference) in which the function is executed. Defaults to this MixedCollection.
	 * @return {MixedCollection} The new filtered collection
	 */
	filterBy : function(fn, scope)
	{
		// Save the function for later usage.
		this.filterFn = fn;
		this.filterScope = scope;

		Zarafa.core.data.MAPIStore.superclass.filterBy.apply(this, arguments);
	},

	/**
	 * Revert to a view of the Record cache with no filtering applied.
	 * @param {Boolean} suppressEvent If <tt>true</tt> the filter is cleared silently without firing the
	 * {@link #datachanged} event.
	 */
	clearFilter : function()
	{
		// Reset the filter
		delete this.filterFn;
		delete this.filterScope;

		Zarafa.core.data.MAPIStore.superclass.clearFilter.apply(this, arguments);
	},

	/**
	 * Callback function which will be called when 'read' action is executed 
	 * and {@link Zarafa.core.data.JsonReader JsonReader} has deserialized data
	 * into {@link Zarafa.core.data.MAPIRecord MAPIRecord},
	 * so the records can be added to the {@link Zarafa.core.data.NoSyncStore NoSyncStore}.
	 * @param {Object} o response object containing array of {@link Zarafa.core.data.MAPIRecord MAPIRecord}
	 * and optionally a property indicating total number of records.
	 * @param {Object} options optionally can contain 'add' which will append {@link Zarafa.core.data.MAPIRecord MAPIRecord}
	 * to the existing set of cached {@link Zarafa.core.data.MAPIRecord MAPIRecord}.
	 * @private
	 */
	loadRecords : function(o, options, success)
	{
		Zarafa.core.data.MAPIStore.superclass.loadRecords.apply(this, arguments);

		if (this.persistentFilter === true && !this.isDestroyed && (!options || options.add !== true)) {
			if (this.filterFn) {
				this.filterBy(this.filterFn, this.filterScope);
			}
		}
	},

	/**
	 * Clear all data in the store
	 * @private
	 */
	clearData : function()
	{
		this.data.each(function(rec) {
			rec.destroy();
		});
		Zarafa.core.data.MAPIStore.superclass.clearData.apply(this, arguments);
	},

	/**
	 * Sort the data in the store using the given sort function.
	 * This will call {@link Ext.util.MixedCollection#sort sort} on the
	 * {@link #data} object.
	 * @param {String} direction (optional) 'ASC' or 'DESC'. Defaults to 'ASC'.
	 * @param {Function} fn (optional) Comparison function that defines the sort order. Defaults to sorting by numeric value.
	 */
	sortBy : function(direction, fn)
	{
		this.data.sort(direction, fn);
		if (this.snapshot && this.snapshot != this.data) {
			this.snapshot.sort(direction, fn);
		}
		this.fireEvent('datachanged', this);
	},

	/**
	 * Clears any existing grouping and refreshes the data using the default sort.
	 */
	clearGrouping : function()
	{
		// Only clear grouping when
		// grouping was previously applied
		if (this.groupField) {
			Zarafa.core.data.MAPIStore.superclass.clearGrouping.apply(this, arguments);
		}
	},

	/**
	 * Destroys the store
	 */
	destroy : function()
	{
		// Make sure we cancel all load requests
		// to the server as we are no longer
		// interested in the results.
		if (!this.isDestroyed) {
			this.cancelLoadRequests();
		}

		Zarafa.core.data.MAPIStore.superclass.destroy.apply(this, arguments);
	}
});

Ext.reg('zarafa.mapistore', Zarafa.core.data.MAPIStore);
Ext.namespace('Zarafa.core.data');

/**
 * @class Zarafa.core.data.NoSyncStore
 * @extends Ext.util.Observable
 *
 * The {@link Zarafa.core.data.NoSyncStore NoSyncStore} represents the collection of
 * {@link Ext.data.Record records}. It offers the same interface
 * as {@link Zarafa.core.data.Store Store} without any CRUD operations being
 * send to the server. This implies that the {@link Zarafa.core.data.NoSyncStore NoSyncStore}
 * will only work for working on {@link Ext.data.Record records} offline.
 */
Zarafa.core.data.NoSyncStore = Ext.extend(Ext.util.Observable, {
	/**
	 * The {@link Ext.data.Record Record} constructor as supplied to (or created by) the
	 * {@link Ext.data.DataReader Reader}. Read-only.
	 * <p>If the Reader was constructed by passing in an Array of {@link Ext.data.Field} definition objects,
	 * instead of a Record constructor, it will implicitly create a Record constructor from that Array (see
	 * {@link Ext.data.Record}.{@link Ext.data.Record#create create} for additional details).</p>
	 * <p>This property may be used to create new Records of the type held in this Store
	 * @property recordType
	 * @type Function
	 */
	recordType : undefined,

	/**
	 * A {@link Ext.util.MixedCollection MixedCollection} containing the defined {@link Ext.data.Field Field}s
	 * for the {@link Ext.data.Record Records} stored in this Store. Read-only.
	 * @property fields
	 * @type Ext.util.MixedCollection
	 */
	fields : undefined,

	/**
	 * True if this store is currently sorted by more than one field/direction combination.
	 * @property
	 * @type Boolean
	 */
	hasMultiSort: false,

	/**
	 * Object containing the current sorting information.
	 * @property
	 * @type Object
	 */
	sortToggle : undefined,

	/**
	 * @cfg {String} sortField
	 * (optional) Initial column on which to sort.
	 */
	sortField : undefined,

	/**
	 * @cfg {String} sortDir
	 * (Optional) Initial direction to sort (<code>"ASC"</code> or <code>"DESC"</code>). Defaults to
	 * <code>"ASC"</code>.
	 */
	sortDir : 'ASC',

	/**
	 * @cfg {Object} sortInfo A config object to specify the sort order in the request of a Store's load operation.
	 * Note that for local sorting, the direction property is case-sensitive.
	 */
	sortInfo : undefined,

	/**
	 * @constructor
	 * @param config Configuration structure
	 */
	constructor : function(config)
	{
		config = config || {};

		Ext.apply(this, config);

		// If the recordType is provided, we can obtain the fields.
		if (this.recordType) {
			this.fields = this.recordType.prototype.fields;
		}

		this.addEvents(
			/**
			 * @event add
			 * Fires when Records have been {@link #add}ed to the Store
			 * @param {Store} this
			 * @param {Ext.data.Record|Array} records The array of Records added
			 * @param {Number} index The index at which the record(s) were added
			 */
			'add',
			/**
			 * @event remove
			 * Fires when a Record has been {@link #remove}d from the Store
			 * @param {Store} this
			 * @param {Ext.data.Record} record The Record that was removed
			 * @param {Number} index The index at which the record was removed
			 */
			'remove',
			/**
			 * @event update
			 * Fires when a Record has been updated
			 * @param {Store} this
			 * @param {Ext.data.Record} record The Record that was updated
			 * @param {String} operation The update operation being performed. Value may be one of:
			 * <pre><code>
	Ext.data.Record.EDIT
	Ext.data.Record.REJECT
	Ext.data.Record.COMMIT
			 * </code></pre>
			 */
			'update',
			/**
			 * @event clear
			 * Fires when the data cache has been cleared.
			 * @param {Store} this
			 * @param {Record|Array} The records that were cleared.
			 */
			'clear'
		);

		this.sortToggle = {};
		if (this.sortField){
			this.setDefaultSort(this.sortField, this.sortDir);
		} else if(this.sortInfo) {
			this.setDefaultSort(this.sortInfo.field, this.sortInfo.direction);
		}

		Zarafa.core.data.NoSyncStore.superclass.constructor.call(this, config);

		this.initEvents();
		this.initData();
	},

	/**
	 * Initialize data structures in which the {@link Ext.data.Record records} are stored.
	 * @private
	 */
	initData : function()
	{
		this.data = new Ext.util.MixedCollection(false);
		this.data.getKey = function(o){
			return o.id;
		};

		this.removed = [];
		this.modified = [];
	},

	/**
	 * Initialize events which can be raised by the {@link Zarafa.core.data.NoSyncStore NoSyncStore}
	 * @private
	 */
	initEvents : function()
	{
		this.on({
			scope: this,
			add: this.createRecords,
			remove: this.destroyRecord,
			clear: this.onClear
		});
	},

	/**
	 * Destroys the store.
	 */
	destroy : function()
	{
		if (!this.isDestroyed) {
			this.clearData();
			this.data = null;
			this.purgeListeners();
			this.isDestroyed = true;
		}
	},

	/**
	 * Add Records to the Store and fires the {@link #add} event.
	 * See also <code>{@link #insert}</code>.
	 * @param {Ext.data.Record|Array} records An Array of Ext.data.Record objects
	 * to add to the cache.
	 * @param {Boolean} silent [false] Defaults to <tt>false</tt>.
	 * Set <tt>true</tt> to not fire add event.
	 */
	add : function(records, silent)
	{
		records = [].concat(records);
		if(records.length < 1)
			return;

		for (var i = 0, len = records.length; i < len; i++)
			records[i].join(this);

		var index = this.data.length;
		this.data.addAll(records);

		if(this.snapshot){
			this.snapshot.addAll(records);
		}

		if (silent !== true) {
			this.fireEvent('add', this, records, index);
		}
	},

	/**
	 * Remove Records from the Store and fires the {@link #remove} event.
	 * @param {Ext.data.Record/Ext.data.Record|Array} record The record object or array of records to remove from the cache.
	 * @param {Boolean} silent [false] Defaults to <tt>false</tt>. Set <tt>true</tt> to not fire remove event.
	 */
	remove : function(record, silent)
	{
		if (Ext.isArray(record)) {
			Ext.each(record, function(r){
				this.remove(r, silent);
			}, this);
		}

		var index = this.data.indexOf(record);
		if(this.snapshot){
			this.snapshot.remove(record);
		}
		if (index > -1) {
			record.join(null);

			this.data.removeAt(index);
			this.modified.remove(record);

			if (silent !== true)
				this.fireEvent('remove', this, record, index);
		}
	},

	/**
	 * Remove a Record from the Store at the specified index. Fires the {@link #remove} event.
	 * @param {Number} index The index of the record to remove.
	 * @param {Boolean} silent [false] Defaults to <tt>false</tt>. Set <tt>true</tt> to not fire remove event.
	 */
	removeAt : function(index, silent)
	{
		this.remove(this.getAt(index), silent);
	},

	/**
	 * Remove all Records from the Store and fires the {@link #clear} event.
	 * @param {Boolean} silent [false] Defaults to <tt>false</tt>. Set <tt>true</tt> to not fire clear event.
	 */
	removeAll : function(silent)
	{
		var items = [];
		this.each(function(rec){
			items.push(rec);
		});

		this.clearData();
		if(this.snapshot){
			this.snapshot.clear();
		}
		this.modified = [];
		this.removed = [];

		if (silent !== true)
			this.fireEvent('clear', this, items);
	},

	/**
	 * Remove all Records for which the callback returns true
	 * from the Store and fires the {@link #remove} event.
	 *
	 * @param {Function} callback The callback function which is used to determine
	 * if a record must be removed. Function must accept a {@link Ext.data.Record}
	 * as argument.
	 * @param {Object} scope The scope which must be used for the callback function
	 */
	removeIf : function(callback, scope)
	{
		this.each(function(record) {
			if (callback.call(scope || this, record))
				this.remove(record);
		}, this);
	},

	/**
	 * Inserts Records into the Store at the given index and fires the {@link #add} event.
	 * See also <code>{@link #add}</code>.
	 * @param {Number} index The start index at which to insert the passed Records.
	 * @param {Ext.data.Record|Array} records An Array of Ext.data.Record objects to add to the cache.
	 */
	insert : function(index, records)
	{
		records = [].concat(records);
		for (var i = 0, len = records.length; i < len; i++) {
			this.data.insert(index, records[i]);
			records[i].join(this);
		}

		if(this.snapshot){
			this.snapshot.addAll(records);
		}

		this.fireEvent('add', this, records, index);
	},

	/**
	 * Get the index within the cache of the passed Record.
	 * @param {Ext.data.Record} record The Ext.data.Record object to find.
	 * @return {Number} The index of the passed Record. Returns -1 if not found.
	 */
	indexOf : function(record)
	{
		return this.data.indexOf(record);
	},

	/**
	 * Get the index within the cache of the Record with the passed id.
	 * @param {String} id The id of the Record to find.
	 * @return {Number} The index of the Record. Returns -1 if not found.
	 */
	indexOfId : function(id)
	{
		return this.data.indexOfKey(id);
	},

	/**
	 * Get the Record at the specified index.
	 * @param {Number} index The index of the Record to find.
	 * @return {Ext.data.Record} The Record at the passed index. Returns undefined if not found.
	 */
	getAt : function(index)
	{
		return this.data.itemAt(index);
	},

	/**
	 * Returns a range of Records between specified indices.
	 * @param {Number} startIndex (optional) The starting index (defaults to 0)
	 * @param {Number} endIndex (optional) The ending index (defaults to the last Record in the Store)
	 * @return {Ext.data.Record|Array} An array of Records
	 */
	getRange : function(start, end)
	{
		return this.data.getRange(start, end);
	},

	/**
	 * Gets the number of cached records
	 * @return {Number} The number of records
	 */
	getCount : function()
	{
		return this.data.length;
	},

	/**
	 * Gets the total number of records in the dataset.
	 * @return {Number} The number of records
	 */
	getTotalCount : function()
	{
		return this.getCount();
	},

	/**
	 * Calls the specified function for each of the {@link Ext.data.Record Records} in the cache.
	 * @param {Function} fn The function to call. The {@link Ext.data.Record Record} is passed as the first parameter.
	 * Returning <tt>false</tt> aborts and exits the iteration.
	 * @param {Object} scope (optional) The scope (<code>this</code> reference) in which the function is executed.
	 * Defaults to the current {@link Ext.data.Record Record} in the iteration.
	 */
	each : function(fn, scope)
	{
		this.data.each(fn, scope);
	},

	/**
	 * Gets all {@link Ext.data.Record records} modified since the last commit.
	 * <b>Note</b>: deleted records are not included.
	 * See also {@link Ext.data.Record}<tt>{@link Ext.data.Record#markDirty markDirty}.</tt>.
	 * @return {Ext.data.Record|Array} An array of {@link Ext.data.Record Records} containing outstanding
	 * modifications. To obtain modified fields within a modified record see
	 *{@link Ext.data.Record}<tt>{@link Ext.data.Record#modified modified}.</tt>.
	 */
	getModifiedRecords : function()
	{
		return this.modified;
	},

	/**
	 * Gets all {@link Ext.data.Record records} removed since the last commit.
	 * @return {Ext.data.Record|Array} An array of {@link Ext.data.Record Records} which have
	 * been marked as removed.
	 */
	getRemovedRecords : function()
	{
		return this.removed;
	},

	/**
	 * Called by {@link Ext.data.Record}
	 * @param {Ext.data.Record} The record which has been edited.
	 * @private
	 */
	afterEdit : function(record)
	{
		if(this.modified.indexOf(record) == -1)
			this.modified.push(record);
		this.fireEvent('update', this, record, Ext.data.Record.EDIT);
	},

	/**
	 * Called by {@link Ext.data.Record}
	 * @param {Ext.data.Record} The record which has been rejected.
	 * @private
	 */
	afterReject : function(record)
	{
		this.modified.remove(record);
		this.fireEvent('update', this, record, Ext.data.Record.REJECT);
	},

	/**
	 * Called by {@link Ext.data.Record}
	 * @param {Ext.data.Record} The record which has been committed.
	 * @private
	 */
	afterCommit : function(record)
	{
		this.modified.remove(record);
		this.fireEvent('update', this, record, Ext.data.Record.COMMIT);
	},

	/**
	 * 'Commit' outstanding changes. Since {@link Zarafa.core.data.NoSyncStore NoSyncStore}
	 * has no commit capability, changes are not actually sent, but are only cleared.
	 */
	commitChanges : function()
	{
		var m = this.modified.slice(0);
		for(var i = 0, len = m.length; i < len; i++){
			var mi = m[i];
			// Committing means unphantoming.
			mi.phantom = false;
			mi.commit();
		}

		this.modified = [];
		this.removed = [];
	},

	/**
	 * Clear the data within this store
	 * @private
	 */
	clearData: function()
	{
		this.data.each(function(rec) {
			rec.join(null);
		});

		this.data.clear();
	},

	/**
	 * Should not be used directly. Store#add will call this automatically
	 * @param {Object} store
	 * @param {Object} rs
	 * @param {Object} index
	 * @private
	 */
	createRecords : function(store, rs, index)
	{
		for (var i = 0, len = rs.length; i < len; i++) {
			if (rs[i].phantom && rs[i].isValid()) {
				rs[i].markDirty();  // <-- Mark new records dirty
				this.modified.push(rs[i]);  // <-- add to modified
			}
		}
	},

	/**
	 * Destroys a record or records. Should not be used directly. It's called by Store#remove automatically
	 * @param {Store} store
	 * @param {Ext.data.Record/Ext.data.Record|Array} record
	 * @param {Number} index
	 * @private
	 */
	destroyRecord : function(store, record, index)
	{
		if (this.modified.indexOf(record) != -1)
			this.modified.remove(record);

		if (!record.phantom)
			this.removed.push(record);
	},

	/**
	 * Clears all records. Show not be used directly. It's called by Store#removeAll automatically
	 * @param {Store} store
	 * @param {Ext.data.Record/Ext.data/Record|Array} records
	 * @private
	 */
	onClear: function(store, records)
	{
		Ext.each(records, function(rec, index) {
			this.destroyRecord(this, rec, index);
		}, this);
	},

	/**
	 * Returns an object describing the current sort state of this Store.
	 * @return {Object} The sort state of the Store. An object with two properties:
	 * field : String The name of the field by which the Records are sorted.
	 * direction : String The sort order, 'ASC' or 'DESC' (case-sensitive).
	 *
	 * Added for grid support with store, grid's store needs sortinfo.
	 *
	 * See <tt>{@link #sortInfo}</tt> for additional details.
	 */
	getSortState : Ext.data.Store.prototype.getSortState,

	/**
	 * Invokes sortData if we have sortInfo to sort on and are not sorting remotely
	 * @private
	 */
	applySort : Ext.data.Store.prototype.applySort,

	/**
	 * Performs the actual sorting of data. This checks to see if we currently have a multi sort or not. It applies
	 * each sorter field/direction pair in turn by building an OR'ed master sorting function and running it against
	 * the full dataset
	 * @private
	 */
	sortData : Ext.data.Store.prototype.sortData,

	/**
	 * Creates and returns a function which sorts an array by the given field and direction
	 * @param {String} field The field to create the sorter for
	 * @param {String} direction The direction to sort by (defaults to "ASC")
	 * @return {Function} A function which sorts by the field/direction combination provided
	 * @private
	 */
	createSortFunction : Ext.data.Store.prototype.createSortFunction,

	/**
	 * Sets the default sort column and order to be used by the next {@link #load} operation.
	 * @param {String} fieldName The name of the field to sort by.
	 * @param {String} dir (optional) The sort order, 'ASC' or 'DESC' (case-sensitive, defaults to <tt>'ASC'</tt>)
	 */
	setDefaultSort : Ext.data.Store.prototype.setDefaultSort,

	/**
	 * Sort the Records.
	 * If remote sorting is used, the sort is performed on the server, and the cache is reloaded. If local
	 * sorting is used, the cache is sorted internally. See also {@link #remoteSort} and {@link #paramNames}.
	 * This function accepts two call signatures - pass in a field name as the first argument to sort on a single
	 * field, or pass in an array of sort configuration objects to sort by multiple fields.
	 * Single sort example:
	 * store.sort('name', 'ASC');
	 * Multi sort example:
	 * store.sort([
	 *   {
	 *     field    : 'name',
	 *     direction: 'ASC'
	 *   },
	 *   {
	 *     field    : 'salary',
	 *     direction: 'DESC'
	 *   }
	 * ], 'ASC');
	 * In this second form, the sort configs are applied in order, with later sorters sorting within earlier sorters' results.
	 * For example, if two records with the same name are present they will also be sorted by salary if given the sort configs
	 * above. Any number of sort configs can be added.
	 * @param {String/Array} fieldName The name of the field to sort by, or an array of ordered sort configs
	 * @param {String} dir (optional) The sort order, 'ASC' or 'DESC' (case-sensitive, defaults to <tt>'ASC'</tt>)
	 */
	sort : Ext.data.Store.prototype.sort,

	/**
	 * Sorts the store contents by a single field and direction. This is called internally by {@link sort} and would
	 * not usually be called manually
	 * @param {String} fieldName The name of the field to sort by.
	 * @param {String} dir (optional) The sort order, 'ASC' or 'DESC' (case-sensitive, defaults to <tt>'ASC'</tt>)
	 */
	singleSort : Ext.data.Store.prototype.singleSort,

	/**
	 * Sorts the contents of this store by multiple field/direction sorters. This is called internally by {@link sort}
	 * and would not usually be called manually.
	 * Multi sorting only currently applies to local datasets - multiple sort data is not currently sent to a proxy
	 * if remoteSort is used.
	 * @param {Array} sorters Array of sorter objects (field and direction)
	 * @param {String} direction Overall direction to sort the ordered results by (defaults to "ASC")
	 */
	multiSort : Ext.data.Store.prototype.multiSort,

	/**
	 * Sums the value of property for each record between start and end and returns the result
	 * @param {String} property A field in each record
	 * @param {Number} start (optional) The record index to start at (defaults to 0)
	 * @param {Number} end (optional) The last record index to include (defaults to length - 1)
	 * @return {Number} The sum
	 */
	sum : Ext.data.Store.prototype.sum,

	/**
	 * Returns a filter function used to test a the given property's value. Defers most of the work to
	 * Ext.util.MixedCollection's createValueMatcher function
	 * @param {String} property The property to create the filter function for
	 * @param {String/RegExp} value The string/regex to compare the property value to
	 * @param {Boolean} anyMatch True if we don't care if the filter value is not the full value (defaults to false)
	 * @param {Boolean} caseSensitive True to create a case-sensitive regex (defaults to false)
	 * @param {Boolean} exactMatch True to force exact match (^ and $ characters added to the regex). Defaults to false. Ignored if anyMatch is true.
	 * @private
	 */
	createFilterFn : Ext.data.Store.prototype.createFilterFn,

	/**
	 * Filters an array of elements to only include matches of a simple selector (e.g. div.some-class or span:first-child)
	 * @param {Array} el An array of elements to filter
	 * @param {String} selector The simple selector to test
	 * @param {Boolean} nonMatches If true, it returns the elements that DON'T match
	 * the selector instead of the ones that match
	 * @return {Array} An Array of DOM elements which match the selector. If there are
	 * no matches, and empty Array is returned.
	 */
	filter : Ext.data.Store.prototype.filter,

	/**
	 * Filter by a function. Returns a <i>new</i> collection that has been filtered.
	 * The passed function will be called with each object in the collection.
	 * If the function returns true, the value is included otherwise it is filtered.
	 * @param {Function} fn The function to be called, it will receive the args o (the object), k (the key)
	 * @param {Object} scope (optional) The scope (<code>this</code> reference) in which the function is executed. Defaults to this MixedCollection.
	 * @return {MixedCollection} The new filtered collection
	 */
	filterBy : Ext.data.Store.prototype.filterBy,

	/**
	 * Query the records by a specified property.
	 * @param {String} field A field on your records
	 * @param {String/RegExp} value Either a string that the field
	 * should begin with, or a RegExp to test against the field.
	 * @param {Boolean} anyMatch (optional) True to match any part not just the beginning
	 * @param {Boolean} caseSensitive (optional) True for case sensitive comparison
	 * @return {MixedCollection} Returns an Ext.util.MixedCollection of the matched records
	 */
	query : Ext.data.Store.prototype.query,

	/**
	 * Query the cached records in this Store using a filtering function. The specified function
	 * will be called with each record in this Store. If the function returns <tt>true</tt> the record is
	 * included in the results.
	 * @param {Function} fn The function to be called. It will be passed the following parameters:<ul>
	 * <li><b>record</b> : Ext.data.Record<p class="sub-desc">The {@link Ext.data.Record record}
	 * to test for filtering. Access field values using {@link Ext.data.Record#get}.</p></li>
	 * <li><b>id</b> : Object<p class="sub-desc">The ID of the Record passed.</p></li>
	 * </ul>
	 * @param {Object} scope (optional) The scope (<code>this</code> reference) in which the function is executed. Defaults to this Store.
	 * @return {MixedCollection} Returns an Ext.util.MixedCollection of the matched records
	 **/
	queryBy : Ext.data.Store.prototype.queryBy,

	/**
	 * Finds the index of the first matching Record in this store by a specific field value.
	 * @param {String} fieldName The name of the Record field to test.
	 * @param {String/RegExp} value Either a string that the field value
	 * should begin with, or a RegExp to test against the field.
	 * @param {Number} startIndex (optional) The index to start searching at
	 * @param {Boolean} anyMatch (optional) True to match any part of the string, not just the beginning
	 * @param {Boolean} caseSensitive (optional) True for case sensitive comparison
	 * @return {Number} The matched index or -1
	 */
	find : Ext.data.Store.prototype.find,

	/**
	 * Finds the index of the first matching Record in this store by a specific field value.
	 * @param {String} fieldName The name of the Record field to test.
	 * @param {Mixed} value The value to match the field against.
	 * @param {Number} startIndex (optional) The index to start searching at
	 * @return {Number} The matched index or -1
	 */
	findExact : Ext.data.Store.prototype.findExact,

	/**
	 * Find the index of the first matching Record in this Store by a function.
	 * If the function returns <tt>true</tt> it is considered a match.
	 * @param {Function} fn The function to be called. It will be passed the following parameters:<ul>
	 * <li><b>record</b> : Ext.data.Record<p class="sub-desc">The {@link Ext.data.Record record}
	 * to test for filtering. Access field values using {@link Ext.data.Record#get}.</p></li>
	 * <li><b>id</b> : Object<p class="sub-desc">The ID of the Record passed.</p></li>
	 * </ul>
	 * @param {Object} scope (optional) The scope (<code>this</code> reference) in which the function is executed. Defaults to this Store.
	 * @param {Number} startIndex (optional) The index to start searching at
	 * @return {Number} The matched index or -1
	 */
	findBy : Ext.data.Store.prototype.findBy,

	/**
	 * Get the Record with the specified id.
	 * @param {String} id The id of the Record to find.
	 * @return {Ext.data.Record} The Record with the passed id. Returns undefined if not found.
	 */
	getById : Ext.data.Store.prototype.getById,

	/**
	 * Revert to a view of the Record cache with no filtering applied.
	 * @param {Boolean} suppressEvent If <tt>true</tt> the filter is cleared silently without firing the
	 * {@link #datachanged} event.
	 */
	clearFilter : Ext.data.Store.prototype.clearFilter,

	/**
	 * Returns true if this store is currently filtered
	 * @return {Boolean}
	 */
	isFiltered : Ext.data.Store.prototype.isFiltered,

	/**
	 * Collects unique values for a particular dataIndex from this store.
	 * @param {String} dataIndex The property to collect
	 * @param {Boolean} allowNull (optional) Pass true to allow null, undefined or empty string values
	 * @param {Boolean} bypassFilter (optional) Pass true to collect from all records, even ones which are filtered
	 * @return {Array} An array of the unique values
	 **/
	collect : Ext.data.Store.prototype.collect,

	/**
	 * When store's reader provides new metadata (fields) this function is called.
	 * @param {Object} meta The JSON metadata
	 * @private
	 */
	onMetaChange : Ext.data.Store.prototype.onMetaChange
});
Ext.namespace('Zarafa.core.data');

/**
 * @class Zarafa.core.data.NotificationResolver
 * @extends Ext.util.Observable
 *
 * The NotificationResolver is used when the {@link Zarafa.core.ResponseRouter ResponseRouter}
 * finds Responses which were not a reponse to a direct Request. This is usually the case,
 * when the PHP-side generated a Notification to update particular UI components based on
 * a server-side change.
 *
 * This resolver will read the response to determine which
 * {@link Zarafa.core.data.AbstractNotificationResponseHandler ResponseHandler} will
 * be needed to handle the response correctly.
 */
Zarafa.core.data.NotificationResolver = Ext.extend(Ext.util.Observable, {
	/**
	 * @constructor
	 * @param {Object} config Configuration object.
	 */
	constructor: function(config)
	{
		config = config || {};

		Ext.apply(this, config);

		Zarafa.core.data.NotificationResolver.superclass.constructor.call(this, config);
	},

	/**
	 * Obtain the {@link Zarafa.core.data.AbstractNotificationResponseHandler ResponseHandler}
	 * which can be used for handling the given response in a correct way. This will look into
	 * the response data to determine what kind of Notifications are in there, and will look
	 * up the most suitable {@link Zarafa.core.data.AbstractNotificationResponseHandler ResponseHandler}.
	 *
	 * @param {String} moduleName The Module which generated the notification.
	 * @param {Object} response The Json response data for which the responseHandler is needed
	 * @return {Zarafa.core.data.AbstractResponseHandler} The response handler
	 * which is suitable for handling the given response object.
	 */
	getHandlerForResponse : function(moduleName, response)
	{
		var handlers;

		if (!Ext.isObject(response)) {
			return null;
		}

		if (moduleName == 'hierarchynotifier' || moduleName == 'newmailnotifier') {
			handlers = this.getHandlersForIPFResponse(response);
		} else {
			handlers = this.getHandlersForIPMResponse(response);
		}

		if (Ext.isArray(handlers)) {
			if (handlers.length > 1) {
				return new Zarafa.core.data.CompositeResponseHandler({
					handlers: handlers
				});
			} else {
				return handlers[0];
			}
		} else {
			return handlers;
		}
	},

	/**
	 * Helper function for {@link #getHandlerForResponse}, this will construct
	 * the {@link Zarafa.hierarchy.data.HierarchyNotificationResponseHandler Response Handlers}
	 * for the {@link Zarafa.core.data.IPFStore IPFStores}.
	 *
	 * @param {Object} response The Json response data for which the responseHandler is needed
	 * @return {Zarafa.core.data.AbstractResponseHandler|Array} The response handlers
	 * which are suitable for handling the given response object.
	 */
	getHandlersForIPFResponse : function(response)
	{
		var folderParents = [];
		var folderStores = [];

		// get handlers for folder's notifications.
		if(response['folders'] || response['newmail']) {
			var folders = response['folders'];
			if (Ext.isDefined(folders) && !Ext.isEmpty(folders['item'])) {
				folderParents = folderParents.concat(Ext.pluck(folders['item'], 'store_entryid'));
			}

			folderStores = Zarafa.core.data.IPFStoreMgr.getStoresForStores(folderParents);

			var newMail = response['newmail'];
			if (Ext.isDefined(newMail)) {
				folderStores.push(container.getHierarchyStore());
			}

			if (Ext.isArray(folderStores)) {
				var responseHandlers = [];
				for (var i = 0, len = folderStores.length; i < len; i++) {
					responseHandlers.push(new Zarafa.hierarchy.data.HierarchyNotificationResponseHandler({
						store : folderStores[i],
						reader : folderStores[i].reader,
						notifyObject : folderStores[i]
					}));
				}

				return responseHandlers;
			} else {
				return new Zarafa.hierarchy.data.HierarchyNotificationResponseHandler({
					store : folderStores,
					reader : folderStores.reader,
					notifyObject : folderStores
				});
			}
		} else if (response['stores']) {
			// get handlers for stores's notifications.
			var hierarchyStore = container.getHierarchyStore();
			return new Zarafa.hierarchy.data.HierarchyNotificationResponseHandler({
				store : hierarchyStore,
				reader : hierarchyStore.reader,
				notifyObject : hierarchyStore
			});
		}
	},

	/**
	 * Helper function for {@link #getHandlerForResponse}, this will construct
	 * the {@link Zarafa.core.data.IPMNotificationResponseHandler Response Handlers}
	 * for the {@link Zarafa.core.data.IPMStore IPMStores}.
	 *
	 * @param {Object} response The Json response data for which the responseHandler is needed
	 * @return {Zarafa.core.data.AbstractResponseHandler|Array} The response handlers
	 * which are suitable for handling the given response object.
	 */
	getHandlersForIPMResponse : function(response)
	{
		var messageParents = [];
		var messageStores;

		var creates = response['newobject'];
		if (Ext.isDefined(creates) && Ext.isArray(creates.item)) {
			messageParents = messageParents.concat(Ext.pluck(creates.item, 'entryid'));
		}

		var updates = response['update'];
		if (Ext.isDefined(updates) && Ext.isArray(updates.item)) {
			messageParents = messageParents.concat(Ext.pluck(updates.item, 'parent_entryid'));
		}

		var deletes = response['delete'];
		if (Ext.isDefined(deletes) && Ext.isArray(deletes.item)) {
			messageParents = messageParents.concat(Ext.pluck(deletes.item, 'parent_entryid'));
		}

		messageStores = Zarafa.core.data.IPMStoreMgr.getStoresForFolders(messageParents);

		if (Ext.isArray(messageStores)) {
			var responseHandlers = [];
			for (var i = 0, len = messageStores.length; i < len; i++) {
				responseHandlers.push(new Zarafa.core.data.IPMNotificationResponseHandler({
					store : messageStores[i],
					reader : messageStores[i].reader,
					notifyObject : messageStores[i]
				}));
			}

			return responseHandlers;
		} else {
			return new Zarafa.core.data.IPMNotificationResponseHandler({
				store : messageStores,
				reader : messageStores.reader,
				notifyObject : messageStores
			});
		}
	}
});
Ext.namespace('Zarafa.core.data');

/**
 * @class Zarafa.core.data.Notifications
 * List of valid notification types.
 * @singleton
 */
Zarafa.core.data.Notifications = 
{
	/**
	 * The 'New Mail' notification to indicate a new Mail has
	 * been delivered to the user.
	 * @property
	 * @type String
	 */
	newMail : 'newMail',

	/**
	 * The 'Object Created' notification to indicate that a new
	 * object has been added to the {@link Zarafa.core.MAPIStore Store}
	 * or {@link Zarafa.hierarchy.data.MAPIFolderRecord Folder}.
	 * @property
	 * @type String
	 */
	objectCreated : 'objectCreated',

	/**
	 * The 'Object Deleted' notification to indicate that an
	 * object has been deleted from the {@link Zarafa.core.MAPIStore Store}
	 * or {@link Zarafa.hierarchy.data.MAPIFolderRecord Folder}.
	 * @property
	 * @type String
	 */
	objectDeleted : 'objectDeleted',

	/**
	 * The 'Object Modified' notification to indicate that an
	 * object has been modified in the {@link Zarafa.core.MAPIStore Store}
	 * or {@link Zarafa.hierarchy.data.MAPIFolderRecord Folder}.
	 * @property
	 * @type String
	 */
	objectModified : 'objectModified'
};
Ext.namespace('Zarafa.core.data');

/**
 * @class Zarafa.core.data.Record
 * @singleton
 */
Zarafa.core.data.Record = {
	/**
	 * Create subclass of a {@link Ext.data.Record record} using the
	 * default list of fields.
	 * @param {Object} fields The fields which must be added to the subclass.
	 * @param {Type} base The base type from which the subclass must be derived.
	 * @return {Object} The type of the subclass.
	 */
	create : function(fields, base)
	{
		var subclass = Ext.extend(base || Ext.data.Record, {});
		var proto = subclass.prototype;

		proto.fields = new Ext.util.MixedCollection(false, function(field) {
			return field.name;
		});

		this.addFields(proto, fields || []);

		subclass.getField = function(name)
		{
			return proto.fields.get(name);
		};

		return subclass;
	},

	/**
	 * Add fields to an object
	 * @param {Object} proto The object to update the fields to
	 * @param {Object} fields The array of fields which must be added
	 * @private
	 */
	addFields : function(proto, fields)
	{
		for(var i = 0, len = fields.length; i < len; i++) {
			if (Ext.isArray(fields[i]))
				this.addFields(proto, fields[i]);
			else
				proto.fields.add(new Ext.data.Field(fields[i]));
		}
	}
};
Ext.namespace('Zarafa.core.data');

/**
 * @class Zarafa.core.data.RecordDefinition
 * @extends Ext.util.Observable
 * The definition for a {@link Ext.data.Record record}
 * for a particular message class. Each message class can have its
 * own set of fields and default values. Within this class, the
 * specific values are kept. By default all fields and default values
 * are inherited from the parent definition. So each definition only
 * contains the data which is new for this particular message class.
 *
 * NOTE: This class should never be used outside of the
 * {@link Zarafa.core.data.RecordFactory RecordFactory}.
 */
Zarafa.core.data.RecordDefinition = Ext.extend(Ext.util.Observable, {
	/**
	 * @cfg {Zarafa.core.data.RecordDefinition} parent The parent record definition for this definition.
	 */
	parent : undefined,

	/**
	 * The object class definition for the record.
	 * This type is always a subclass of the {@link Zarafa.core.data.RecordDefinition.base base},
	 * field within this {@link Zarafa.core.data.RecordDefinition definition}.
	 * @property
	 * @type Class
	 * @private
	 */
	type: undefined,

	/**
	 * The configuration object in where all options for the Record instances are configured.
	 *
	 * This is editable as long as no record has been created using this definition yet, after
	 * that this property can no longer be used.
	 * @property
	 * @type Object
	 * @private
	 */
	cfg : undefined,

	/**
	 * @cfg {Object} base The object class definition which must be used as base for the record.
	 * In most cases this base must be extended from {@link Ext.data.Record record}.
	 *
	 * This is applied on the {@link #cfg} object and can be changed as long as no record has been
	 * created using this definition yet, after that this property can no longer be used.
	 */
	base: undefined,

	/**
	 * @cfg {Object} subStores The key-value array where for each subStore type is provided with the type
	 * of the {@link Zarafa.core.data.SubStore SubStore} which must be used for the given subStore.
	 *
	 * This is applied on the {@link #cfg} object and can be changed as long as no record has been
	 * created using this definition yet, after that this property can no longer be used.
	 */
	subStores : undefined,

	/**
	 * @cfg {Ext.data.Field|Array} fields The array of fields which belong to this {@link Ext.data.Record record}.
	 * This is mapped on the server to MAPI properties which are stored in the zarafa-server.
	 *
	 * This is applied on the {@link #cfg} object and can be changed as long as no record has been
	 * created using this definition yet, after that this property can no longer be used.
	 */
	fields: undefined,

	/**
	 * @cfg {Object} defaults The key-value array where the default values for particular fields is provided.
	 * When a new {@link Ext.data.Record record} is created (one that does
	 * not yet exist on the server) all default values will be applied to the new
	 * {@link Ext.data.Record record}.
	 *
	 * This is applied on the {@link #cfg} object and can be changed as long as no record has been
	 * created using this definition yet, after that this property can no longer be used.
	 */
	defaults: undefined,

	/**
	 * @cfg {Object} createDefaults The key-value array for default values that should be applied
	 * when a record is created (either a phantom or non-phantom).
	 *
	 * This is applied on the {@link #cfg} object and can be changed as long as no record has been
	 * created using this definition yet, after that this property can no longer be used.
	 */
	createDefaults : undefined,

	/**
	 * @constructor
	 * @param {Object} config Configuration object
	 */
	constructor : function(config)
	{
		this.addEvents([
			/**
			 * @event createphantom
			 * Fires when the {@link Zarafa.core.data.RecordFactory RecordFactory} has
			 * created a new phantom record.
			 * @param {Ext.data.Record} record The created phantom record
			 * @param {Object} data The data used to initialize the record
			 */
			'createphantom',
			/**
			 * @event createrecord
			 * Fires when the {@link Zarafa.core.data.RecordFactory RecordFactory} has
			 * created a new record.
			 * @param {Ext.data.Record} record The created record
			 * @param {Object} data The data used to initialize the record
			 */
			'createrecord'
		]);

		// Only the parent is applied on the current
		// class. the rest can be applied to the cfg object.
		this.parent = config.parent;
		delete config.parent;

		this.cfg = Ext.apply({}, config);

		Zarafa.core.data.RecordDefinition.superclass.constructor.call(this, config);
	},

	/**
	 * Create a new {@link Ext.data.Record record} object
	 * based on this {@link Zarafa.core.data.RecordDefinition definition}.
	 *
	 * The data object used to instantiate the new Record will be created using the
	 * default values from {@link Ext.data.Field field} definitions as obtained by
	 * {@link #getFieldDefaultValues}, the default values configured through
	 * the {@link Zarafa.core.data.RecordFactory Record Factory} which are obtained
	 * by the {@link #getDefaultValues} function. And finally the data passed as
	 * argument will be applied. This ordering implies that the {@link Ext.data.Field field}
	 * default values are most likely to be overridden by the other values.
	 *
	 * @param {Object} data (Optional) An object, the properties of which provide
	 * values for the new Record's fields. If not specified the {@link Ext.data.Field.defaultValue}
	 * for each field will be assigned.
	 * @param {Object} id (Optional) The id of the Record. The id is used by the
	 * {@link Ext.data.Store Store} object which owns the {@link Ext.data.Record}
	 * to index its collection of Records (therefore this id should be unique within
	 * each store). If an id is not specified a phantom Record will be created with an
	 * automatically generated id.
	 * @return {Ext.data.Record} the new record which was created
	 */
	createRecord : function(data, id)
	{
		var RecordClass = this.getType();
		var applyData = {};
		Ext.apply(applyData, this.getFieldDefaultValues());
		if (!Ext.isDefined(id)) {
			Ext.apply(applyData, this.getDefaultValues());
		}
		if (data) {
			Ext.apply(applyData, data);
		}

		// Apply the createDefaults object to initialize the record with.
		var record = new RecordClass(applyData, id, this);

		if (record.phantom) {
			this.fireEvent('createphantom', record, data);
		} else {
			// field default values will be applied by the jsonreader
			this.fireEvent('createrecord', record, data);
		}

		// All changes which were made should be committed,
		// as only after this function returns should the changes
		// for the record be kept.
		record.commit();

		return record;
	},

	/**
	 * Obtain the {@link #type object class} definition which can be used for constructing
	 * new {@link Ext.data.Record records}. The type will be a subclass of
	 * {@link Zarafa.core.data.RecordDefinition#base base}.
	 * @return {Class} The object class definition for this record definition
	 */
	getType : function()
	{
		if (!Ext.isDefined(this.type)) {
			var baseClass = this.getBaseClass();
			var fields = this.getFields();
			this.type = Zarafa.core.data.Record.create(fields, baseClass);
		}
		return this.type;
	},

	/**
	 * Set the {@link #base} class for the {@link Zarafa.core.data.RecordDefinition definition}.
	 *
	 * NOTE: If {@link #createRecord} has been called, then changes made by this function will
	 * no longer have effect.
	 *
	 * @param {Class} baseClass The base class to set.
	 */
	setBaseClass : function(baseClass)
	{
		this.cfg.base = baseClass;
	},

	/**
	 * Obtain the {@link #base} class which must be used when creating a new record
	 * from this {@link Zarafa.core.data.RecordDefinition definition}. If
	 * no base class is yet provided, this function will call recursively
	 * into its {@link #parent parents} until a valid base class has been found.
	 * @return {Class} The base class to be used for creating a new record.
	 * @private
	 */
	getBaseClass : function()
	{
		if (!Ext.isDefined(this.base)) {
			if (this.cfg.base) {
				this.base = this.cfg.base;
			} else if (Ext.isDefined(this.parent)) {
				this.base = this.parent.getBaseClass();
			} else {
				this.base = Ext.data.Record;
			}
		}
		return this.base;
	},

	/**
	 * Set the type which must be used for a particular SubStore.
	 * When a SubStore type has been set on a RecordType, then the
	 * subStore is considered 'supported' by the Record.
	 *
	 * NOTE: If {@link #createRecord} has been called, then changes made by this function will
	 * no longer have effect.
	 *
	 * @param {String} name The name of the SubStore (This matches the name how the
	 * data for this subStore is send through Json).
	 * @param {Type} type The Object type which must be used to allocate the subStore
	 */
	setSubStore : function(name, type)
	{
		this.cfg.subStores = Ext.value(this.cfg.subStores, {});
		this.cfg.subStores[name] = type;
	},

	/**
	 * Obtain the key-value array of subStore types which are active for
	 * this {@link Zarafa.core.data.RecordDefinition definition}. This object
	 * is used to {@link Zarafa.core.data.MAPIRecord#setSubStoreTypes set} on
	 * the {@link Zarafa.core.data.MAPIRecord MAPIRecord} which is created.
	 * @return {Object} key-value array of all subStore Types
	 * @private
	 */
	getSubStores : function()
	{
		if (!this.subStores) {
			this.subStores = {};
			if (Ext.isDefined(this.parent)) {
				Ext.apply(this.subStores, this.parent.getSubStores());
			}
			Ext.apply(this.subStores, this.cfg.subStores);
		}

		return Ext.apply({}, this.subStores);
	},

	/**
	 * Add a {@link Ext.data.Field field} to the {@link Zarafa.core.data.RecordDefinition definition}.
	 *
	 * NOTE: If {@link #createRecord} has been called, then changes made by this function will
	 * no longer have effect.
	 *
	 * @param {Ext.data.Field} field The field to add.
	 */
	addField : function(field)
	{
		this.cfg.fields = Ext.value(this.cfg.fields, []);

		if (!Ext.isArray(field)) {
			this.cfg.fields.push(field);
		} else {
			this.cfg.fields = this.cfg.fields.concat(field);
		}
	},

	/**
	 * Obtain the list of {@link Ext.data.Field fields} which are active
	 * for this {@link Zarafa.core.data.RecordDefinition definition}.
	 * These fields must be used to create a record, to indicate which
	 * {@link Ext.data.Field fields} are allowed.
	 * This function will recursively call into its parents to obtain
	 * all fields from the parents.
	 *
	 * @return {Ext.data.Field|Array} All fields which are active for
	 * this definition.
	 * @private
	 */
	getFields : function()
	{
		if (!this.fields) {
			this.fields = [];

			if (Ext.isDefined(this.parent)) {
				this.fields = this.fields.concat(this.parent.getFields());
			}
			if (this.cfg.fields) {
				this.fields = this.fields.concat(this.cfg.fields);
			}
		}

		return this.fields;
	},

	/**
	 * Add a default value for a {@link Ext.data.Field field} to the
	 * {@link Zarafa.core.data.RecordDefinition definition}.
	 *
	 * NOTE: If {@link #createRecord} has been called, then changes made by this function will
	 * no longer have effect.
	 *
	 * @param {Ext.data.Field} field The field for which the default value applies
	 * @param {Mixed} value The default value for the field
	 */
	addDefaultValue : function(field, value)
	{
		this.cfg.defaults = Ext.value(this.cfg.defaults, {});
		var name = Ext.isString(field) ? field : field.name;
		this.cfg.defaults[name] = value;
	},

	/**
	 * Apply all default values for the {@link Ext.data.Field field}
	 * which have been set for the {@link Zarafa.core.data.RecordDefinition definition}.
	 *
	 * This function will recursively call into its parents to apply
	 * all default values from the parents.
	 * @private
	 */
	getDefaultValues : function()
	{
		if (!this.defaults) {
			this.defaults = {};

			if (Ext.isDefined(this.parent)) {
				Ext.apply(this.defaults, this.parent.getDefaultValues());
			}

			if (this.cfg.defaults) {
				Ext.apply(this.defaults, this.cfg.defaults);
			}
		}

		return this.defaults;
	},

	/**
	 * Apply all default values from the {@link Ext.data.Field field}
	 * definitions. These are applied by the {@link Ext.data.DataReader DataReader}
	 * when reading {@link Ext.data.Record records} from the server, but not for
	 * {@link Ext.data.Record#phantom phantom} records.
	 * @private
	 */
	getFieldDefaultValues : function()
	{
		if (!this.fieldDefaults) {
			var fields = this.getFields();

			this.fieldDefaults = {};
			for (var i = 0, len = fields.length; i < len; i++) {
				var field = fields[i];
				var value = field.defaultValue;
				if (!Ext.isDefined(value)) {
					value = Ext.data.Field.prototype.defaultValue;
				}

				this.fieldDefaults[field.name] = value;
			}
			Ext.apply(this.fieldDefaults, this.cfg.createDefaults);
		}

		return this.fieldDefaults;
	},

	/**
	 * Fires the specified event with the passed parameters (minus the event name).
	 *
	 * This function will recursively call into its parents to fire the event
	 * from the parents.
	 * @param {String} eventName The name of the event to fire.
	 * @param {Object...} args The arguments for the event
	 * @return {Boolean} returns false if any of the handlers return false otherwise it returns true.
	 */
	fireEvent : function()
	{
		if (Ext.isDefined(this.parent)) {
			this.parent.fireEvent.apply(this.parent, arguments);
		}
		Zarafa.core.data.RecordDefinition.superclass.fireEvent.apply(this, arguments);
	}
});
Ext.namespace('Zarafa.core.data');

/**
 * @class Zarafa.core.data.RecordFactory
 * The factory class for creating objects of {@link Ext.data.Record record}
 * which are specialized for certain messageclasses (IPM.Note, IPM.Contact, etc).
 * The exact field definition of the different {@link Ext.data.Record records}
 * can be configured by {@link Zarafa.core.Context contexts} and {@link Zarafa.core.Plugin plugins}
 * to add new fields, but also to control the default values for particular fields.
 *
 * For fine-grained control over all classes, the factory also supports
 * {@link Ext.data.Record record} for sub messageclasses like (IPM.Note.NDR),
 * where the sub messageclass depends on its parent (IPM.Note.NDR inherits its fields, and
 * default values from IPM.Note).
 *
 * For plugins with custom types (no official MAPI ObjectType, or Message Class definition),
 * we have the {@link Zarafa.core.data.RecordCustomObjectType Custom Type} definitions to which
 * they can register their custom type, which can then be used to work together with
 * the record factory.
 *
 * @singleton
 */
Zarafa.core.data.RecordFactory = {
	/**
	 * Key-value map of {@link Zarafa.core.data.RecordDefinition} as registered by
	 * {@link #createRecordDefinition}. 
	 *
	 * @property
	 * @type Object
	 */
	definitions : {},

	/**
	 * Obtain the parent messageclass for the given messageclass.
	 * The hierarchy of the messageclass is based on the dot ('.')
	 * character. Each dot indicates a step deeper into the hierarchy.
	 * Thus if the messageclass is IPM.Note, the parent is IPM.
	 *
	 * @param {String} messageClass The messageclass for which
	 * the parent is requested.
	 * @return {String} The parent messageclass (undefined, if
	 * the messageclass has no parent.
	 * @private
	 */
	getMessageClassParent : function(messageClass)
	{
		var lastDot = messageClass.lastIndexOf('.');
		if (lastDot > 0)
			return messageClass.substr(0, lastDot);
		return undefined;
	},

	/**
	 * Obtain the parent {@link Zarafa.core.data.RecordDefinition RecordDefinition}
	 * for this messageclass.
	 * @param {String} The messageclass for the parent is searched for
	 * @return {Zarafa.core.data.RecordDefinition} The parent record definition.
	 * Undefined if the messageclass does not have a parent.
	 */
	getMessageClassParentDefinition : function(messageClass)
	{
		var messageClassParent = this.getMessageClassParent(messageClass);

		if (Ext.isDefined(messageClassParent))
			return this.getRecordDefinitionByMessageClass(messageClassParent);
		return undefined;
	},

	/**
	 * Create a new {@link Zarafa.core.data.RecordDefinition RecordDefinition}
	 * for the given key value.
	 *
	 * @param {Mixed} key The key value on how the definition is stored
	 * in the definitions table.
	 * @param {Zarafa.core.data.RecordDefinition} parent The parent record
	 * definition.
	 * @param {Object} defaults The default values which must be applied to
	 * the record definition.
	 * @return {Zarafa.core.data.RecordDefinition} The new record definition.
	 */
	createRecordDefinition : function(key, parent, defaults)
	{
		var definition = new Zarafa.core.data.RecordDefinition({
			base : undefined,
			parent : parent,
			createDefaults : defaults
		});

		this.definitions[key] = definition;
		return definition;
	},

	/**
	 * Obtain the {@link Zarafa.core.data.RecordDefinition RecordDefinition} object
	 * which belongs to the given custom type. If the custom type is previously unknown
	 * and the {@link Zarafa.core.data.RecordDefinition definition} does not exist,
	 * a new {@link Zarafa.core.data.RecordDefinition definition} will be created and
	 * returned.
	 *
	 * @param {Zarafa.core.data.RecordCustomObjectType} customType The custom type for which the definition is requested.
	 * @return {Zarafa.core.data.RecordDefinition} The definition object for the custom type
	 * @private
	 */
	getRecordDefinitionByCustomType : function(customType)
	{
		var definition = this.definitions[customType];
		if (Ext.isDefined(definition))
			return definition;

		// Custom types don't have any hierarchy...
		return this.createRecordDefinition(customType, undefined);
	},

	/**
	 * Obtain the {@link Zarafa.core.data.RecordDefinition RecordDefinition} object
	 * which belongs to the given object class. If the object type is previously unknown
	 * and the {@link Zarafa.core.data.RecordDefinition definition} does not exist,
	 * a new {@link Zarafa.core.data.RecordDefinition definition} will be created and
	 * returned.
	 *
	 * @param {Zarafa.core.mapi.ObjectType} objectType The object type for which the definition is requested.
	 * @return {Zarafa.core.data.RecordDefinition} The definition object for the object type
	 * @private
	 */
	getRecordDefinitionByObjectType : function(objectType)
	{
		var definition = this.definitions[objectType];
		if (Ext.isDefined(definition))
			return definition;

		// Object types don't have any hierarchy...
		return this.createRecordDefinition(objectType, undefined, {'object_type' : objectType});
	},

	/**
	 * Obtain the {@link Zarafa.core.data.RecordDefinition RecordDefinition} object
	 * which belongs to the given messageclass. If the messageClass is previously unknown
	 * and the {@link Zarafa.core.data.RecordDefinition definition} does not exist,
	 * a new {@link Zarafa.core.data.RecordDefinition definition} will be created and
	 * returned.
	 *
	 * @param {String} messageClass The messageclass for which the definition is requested.
	 * @return {Zarafa.core.data.RecordDefinition} The definition object for the messageclass. 
	 * @private
	 */
	getRecordDefinitionByMessageClass : function(messageClass)
	{
		var keyName = messageClass.toUpperCase();
		var definition = this.definitions[keyName];
		var parent = undefined;

		if (Ext.isDefined(definition))
			return definition;

		parent = this.getMessageClassParentDefinition(keyName);
		return this.createRecordDefinition(keyName, parent, {'message_class' : messageClass});
	},

	/**
	 * Create a new {@link Ext.data.Record record} object based on the
	 * definitions given for the custom type.
	 *
	 * @param {Zarafa.core.data.RecordCustomObjectType} customType The custom type for which
	 * the record is created.
	 * @param {Object} data (Optional) An object, the properties of which provide
	 * values for the new Record's fields. If not specified the {@link Ext.data.Field.defaultValue}
	 * for each field will be assigned.
	 * @param {Object} id (Optional) The id of the Record. The id is used by the
	 * {@link Ext.data.Store Store} object which owns the {@link Ext.data.Record}
	 * to index its collection of Records (therefore this id should be unique within
	 * each store). If an id is not specified a phantom Record will be created with an
	 * automatically generated id.
	 * @return {Ext.data.Record} The new record
	 */
	createRecordObjectByCustomType : function(customType, data, id)
	{
		var definition = this.getRecordDefinitionByCustomType(customType);
		return definition.createRecord(data, id);
	},

	/**
	 * Create a new {@link Ext.data.Record record} object based on the
	 * definitions given for the object type.
	 *
	 * @param {Zarafa.core.mapi.ObjectType} objectType The object type for which
	 * the record is created.
	 * @param {Object} data (Optional) An object, the properties of which provide
	 * values for the new Record's fields. If not specified the {@link Ext.data.Field.defaultValue}
	 * for each field will be assigned.
	 * @param {Object} id (Optional) The id of the Record. The id is used by the
	 * {@link Ext.data.Store Store} object which owns the {@link Ext.data.Record}
	 * to index its collection of Records (therefore this id should be unique within
	 * each store). If an id is not specified a phantom Record will be created with an
	 * automatically generated id.
	 * @return {Ext.data.Record} The new record
	 */
	createRecordObjectByObjectType : function(objectType, data, id)
	{
		var definition = this.getRecordDefinitionByObjectType(objectType);
		return definition.createRecord(data, id);
	},

	/**
	 * Create a new {@link Ext.data.Record record} object based on the
	 * definitions given for the messageclass.
	 *
	 * @param {String} messageClass The messageclass for which the record is created.
	 * @param {Object} data (Optional) An object, the properties of which provide
	 * values for the new Record's fields. If not specified the {@link Ext.data.Field.defaultValue}
	 * for each field will be assigned.
	 * @param {Object} id (Optional) The id of the Record. The id is used by the
	 * {@link Ext.data.Store Store} object which owns the {@link Ext.data.Record}
	 * to index its collection of Records (therefore this id should be unique within
	 * each store). If an id is not specified a phantom Record will be created with an
	 * automatically generated id.
	 * @return {Ext.data.Record} The new record
	 */
	createRecordObjectByMessageClass : function(messageClass, data, id)
	{
		var definition = this.getRecordDefinitionByMessageClass(messageClass);
		return definition.createRecord(data, id);
	},

	/**
	 * Create a new {@link Ext.data.Record record} object based on the
	 * definitions from the record data, this will determine if
	 * the messageclass of object type has been provided. If neither has been provided
	 * this function will return undefined.
	 *
	 * @param {Object} recordData The record data from which the record class must be detected
	 * it slaos contains the properties of which provide values for the new Record's fields. 
	 * @param {Object} id (Optional) The id of the Record. The id is used by the
	 * {@link Ext.data.Store Store} object which owns the {@link Ext.data.Record}
	 * to index its collection of Records (therefore this id should be unique within
	 * each store). If an id is not specified a phantom Record will be created with an
	 * automatically generated id.
	 * @return {Class} The record class definition
	 */
	createRecordObjectByRecordData : function(recordData, id)
	{
		if (!Ext.isEmpty(recordData.message_class))
			return this.createRecordObjectByMessageClass(recordData.message_class, recordData, id);
		else if (!Ext.isEmpty(recordData.object_type))
			return this.createRecordObjectByObjectType(recordData.object_type, recordData, id);
		else
			return undefined;
	},

	/**
	 * Get the record class definition for this custom type
	 * This class definition can be used to construct new {@link Ext.data.Record record}
	 * objects which is completely tuned for the given object type.
	 *
	 * @param {Zarafa.core.data.RecordCustomObjectType} customType The object type for which the
	 * record class is requested
	 * @return {Class} The record class definition
	 */
	getRecordClassByCustomType : function(customType)
	{
		var definition = this.getRecordDefinitionByCustomType(customType);
		return definition.getType();
	},

	/**
	 * Get the record class definition for this object type
	 * This class definition can be used to construct new {@link Ext.data.Record record}
	 * objects which is completely tuned for the given object type.
	 *
	 * @param {Zarafa.core.mapi.ObjectType} objectType The object type for which the
	 * record class is requested
	 * @return {Class} The record class definition
	 */
	getRecordClassByObjectType : function(objectType)
	{
		var definition = this.getRecordDefinitionByObjectType(objectType);
		return definition.getType();
	},

	/**
	 * Get the record class definition for this messageclass
	 * This class definition can be used to construct new {@link Ext.data.Record record}
	 * objects which is completely tuned for the given messageclass.
	 *
	 * @param {String} messageClass The messageclass for which the record class is requested
	 * @return {Class} The record class definition
	 */
	getRecordClassByMessageClass : function(messageClass)
	{
		var definition = this.getRecordDefinitionByMessageClass(messageClass);
		return definition.getType();
	},

	/**
	 * Get the record class definition for this record data, this will determine if
	 * the messageclass of object type has been provided. If neither has been provided
	 * this function will return undefined.
	 * This class definition can be used to construct new {@link Ext.data.Record record}
	 * objects which is completely tuned for the given messageclass.
	 *
	 * @param {Object} recordData The record data from which the record class must be detected
	 * @return {Class} The record class definition
	 */
	getRecordClassByRecordData : function(recordData)
	{
		if (Ext.isDefined(recordData.message_class))
			return this.getRecordClassByMessageClass(recordData.message_class);
		else if (Ext.isDefined(recordData.object_type))
			return this.getRecordClassByObjectType(recordData.object_type);
		else
			return undefined;
	},

	/**
	 * This will set a new base class on the {@link Zarafa.core.data.RecordDefinition definition}.
	 * The base class is used when creating a new record, by default the ExtJS
	 * {@link Ext.data.Record record} will be used as base.
	 *
	 * @param {Zarafa.core.data.RecordCustomObjectType} customType The custom type for which
	 * the new base class must be set.
	 * @param {Class} baseClass The new base class to use
	 */
	setBaseClassToCustomType : function(customType, baseClass)
	{
		var definition = this.getRecordDefinitionByCustomType(customType);
		definition.setBaseClass(baseClass);
	},

	/**
	 * This will set a new base class on the {@link Zarafa.core.data.RecordDefinition definition}.
	 * The base class is used when creating a new record, by default the ExtJS
	 * {@link Ext.data.Record record} will be used as base.
	 *
	 * @param {Zarafa.core.mapi.ObjectType} objectType The object type for which
	 * the new base class must be set.
	 * @param {Class} baseClass The new base class to use
	 */
	setBaseClassToObjectType : function(objectType, baseClass)
	{
		var definition = this.getRecordDefinitionByObjectType(objectType);
		definition.setBaseClass(baseClass);
	},

	/**
	 * This will set a new base class on the {@link Zarafa.core.data.RecordDefinition definition}.
	 * The base class is used when creating a new record, by default the ExtJS
	 * {@link Ext.data.Record record} will be used as base.
	 *
	 * @param {String} messageClass The messageclass for which
	 * the new base class must be set.
	 * @param {Class} baseClass The new base class to use
	 */
	setBaseClassToMessageClass : function(messageClass, baseClass)
	{
		var definition = this.getRecordDefinitionByMessageClass(messageClass);
		definition.setBaseClass(baseClass);
	},

	/**
	 * This will set the SubStore type for a particular subStore name
	 * in the {@link Zarafa.core.data.RecordDefinition definition}. Inside
	 * a SubStore a Table can be loaded like Attachments, recipients, members, etc.
	 *
	 * @param {Zarafa.core.data.RecordCustomObjectType} customType The custom type for which the
	 * SubStore type must be set
	 * @param {String} name The name of the SubStore table
	 * @param {Constructor} type The SubStore type
	 */
	setSubStoreToCustomType : function(customType, name, type)
	{
		var definition = this.getRecordDefinitionByCustomType(customType);
		definition.setSubStore(name, type);
	},

	/**
	 * This will set the SubStore type for a particular subStore name
	 * in the {@link Zarafa.core.data.RecordDefinition definition}. Inside
	 * a SubStore a Table can be loaded like Attachments, recipients, members, etc.
	 *
	 * @param {Zarafa.core.mapi.ObjectType} objectType The object type for which the
	 * SubStore type must be set
	 * @param {String} name The name of the SubStore table
	 * @param {Constructor} type The SubStore type
	 */
	setSubStoreToObjectType : function(objectType, name, type)
	{
		var definition = this.getRecordDefinitionByObjectType(objectType);
		definition.setSubStore(name, type);
	},

	/**
	 * This will set the SubStore type for a particular subStore name
	 * in the {@link Zarafa.core.data.RecordDefinition definition}. Inside
	 * a SubStore a Table can be loaded like Attachments, recipients, members, etc.
	 *
	 * @param {String} messageClass The messageclass for which
	 * @param {String} name The name of the SubStore table
	 * @param {Constructor} type The SubStore type
	 */
	setSubStoreToMessageClass : function(messageClass, name, type)
	{
		var definition = this.getRecordDefinitionByMessageClass(messageClass);
		definition.setSubStore(name, type);
	},

	/**
	 * This will add a new {@link Ext.data.Field field} to the
	 * {@link Zarafa.core.data.RecordDefinition definition}.
	 *
	 * @param {Zarafa.core.data.RecordCustomObjectType} customType The custom type to which the field
	 * should be added.
	 * @param {String/Ext.data.Field} field The field which must be added.
	 */
	addFieldToCustomType : function(customType, field)
	{
		var definition = this.getRecordDefinitionByCustomType(customType);
		definition.addField(field);
	},

	/**
	 * This will add a new {@link Ext.data.Field field} to the
	 * {@link Zarafa.core.data.RecordDefinition definition}.
	 *
	 * @param {Zarafa.core.mapi.ObjectType} objectType The object type to which the field
	 * should be added.
	 * @param {String/Ext.data.Field} field The field which must be added.
	 */
	addFieldToObjectType : function(objectType, field)
	{
		var definition = this.getRecordDefinitionByObjectType(objectType);
		definition.addField(field);
	},

	/**
	 * This will add a new {@link Ext.data.Field field} to the
	 * {@link Zarafa.core.data.RecordDefinition definition}.
	 *
	 * @param {String} messageClass The messageclass to which the field
	 * should be added.
	 * @param {String/Ext.data.Field} field The field which must be added.
	 */
	addFieldToMessageClass : function(messageClass, field)
	{
		var definition = this.getRecordDefinitionByMessageClass(messageClass);
		definition.addField(field);
	},

	/**
	 * This will add a default value for a {@link Ext.data.Field field}
	 * to the {@link Zarafa.core.data.RecordDefinition definition}. When
	 * creating a new local item
	 * ({@link Zarafa.core.data.RecordFactory.createRecordObjectByCustomType createRecordObjectByCustomType}
	 * with the phantom argument to true) the default values which be applied
	 * to the newly created {@link Ext.data.Record record}.
	 *
	 * @param {Zarafa.core.data.RecordCustomObjectType} customType The custom type to which the field belongs
	 * @param {String/Ext.data.Field} field The field for which the default
	 * value should be applied
	 * @param {Mixed} defaultValue the default value for the given fieldname
	 */
	addDefaultValueToCustomType : function(customType, field, defaultValue)
	{
		var definition = this.getRecordDefinitionByCustomType(customType);
		definition.addDefaultValue(field, defaultValue);
	},

	/**
	 * This will add a default value for a {@link Ext.data.Field field}
	 * to the {@link Zarafa.core.data.RecordDefinition definition}. When
	 * creating a new local item
	 * ({@link Zarafa.core.data.RecordFactory.createRecordObjectByObjectType createRecordObjectByObjectType}
	 * with the phantom argument to true) the default values which be applied
	 * to the newly created {@link Ext.data.Record record}.
	 *
	 * @param {Zarafa.core.mapi.ObjectType} objectType The object type to which the field belongs
	 * @param {String/Ext.data.Field} field The field for which the default
	 * value should be applied
	 * @param {Mixed} defaultValue the default value for the given fieldname
	 */
	addDefaultValueToObjectType : function(objectType, field, defaultValue)
	{
		var definition = this.getRecordDefinitionByObjectType(objectType);
		definition.addDefaultValue(field, defaultValue);
	},

	/**
	 * This will add a default value for a {@link Ext.data.Field field}
	 * to the {@link Zarafa.core.data.RecordDefinition definition}. When
	 * creating a new local item
	 * ({@link Zarafa.core.data.RecordFactory.createRecordObjectByMessageClass createRecordObjectByMessageClass}
	 * with the phantom argument to true) the default values which be applied
	 * to the newly created {@link Ext.data.Record record}.
	 *
	 * @param {String} messageClass The messageclass to which the field belongs
	 * @param {String/Ext.data.Field} field The field for which the default
	 * value should be applied
	 * @param {Mixed} defaultValue the default value for the given fieldname
	 */
	addDefaultValueToMessageClass : function(messageClass, field, defaultValue)
	{
		var definition = this.getRecordDefinitionByMessageClass(messageClass);
		definition.addDefaultValue(field, defaultValue);
	},

	/**
	 * Appends an event handler to the {@link Zarafa.core.data.RecordDefinition RecordDefinition}
	 * for the given CustomType.
	 * @param {Zarafa.core.data.RecordCustomObjectType} customType The custom type to which the event must be added
	 * @param {String} eventName The name of the event to listen for
	 * @param {Function} handler The method the event invokes
	 * @param {Object} scope (optional) The scope (this reference) in which the handler function is executed.
	 * If omitted, defaults to the object which fired the event.
	 * @param {Object} options (optional) An object containing handler configuration properties.
	 */
	addListenerToCustomType : function(customType, eventName, handler, scope, options)
	{
		var definition = this.getRecordDefinitionByCustomType(customType);
		definition.on(eventName, handler, scope, options);
	},

	/**
	 * Appends an event handler to the {@link Zarafa.core.data.RecordDefinition RecordDefinition}
	 * for the given ObjectType.
	 * @param {Zarafa.core.mapi.ObjectType} objectType The object type to which the event must be added
	 * @param {String} eventName The name of the event to listen for
	 * @param {Function} handler The method the event invokes
	 * @param {Object} scope (optional) The scope (this reference) in which the handler function is executed.
	 * If omitted, defaults to the object which fired the event.
	 * @param {Object} options (optional) An object containing handler configuration properties.
	 */
	addListenerToObjectType : function(objectType, eventName, handler, scope, options)
	{
		var definition = this.getRecordDefinitionByObjectType(objectType);
		definition.on(eventName, handler, scope, options);
	},

	/**
	 * Appends an event handler to the {@link Zarafa.core.data.RecordDefinition RecordDefinition}
	 * for the given messageClass.
	 * @param {String} messageClass The messageclass to which the event must be added
	 * @param {String} eventName The name of the event to listen for
	 * @param {Function} handler The method the event invokes
	 * @param {Object} scope (optional) The scope (this reference) in which the handler function is executed.
	 * If omitted, defaults to the object which fired the event.
	 * @param {Object} options (optional) An object containing handler configuration properties.
	 */
	addListenerToMessageClass : function(messageClass, eventName, handler, scope, options)
	{
		var definition = this.getRecordDefinitionByMessageClass(messageClass);
		definition.on(eventName, handler, scope, options);
	}
};
Ext.namespace('Zarafa.core.data');

/**
 * @class Zarafa.core.data.SettingsStateProvider
 * @type Ext.state.Provider
 * 
 * A special {@link Ext.state.Provider} for the {@link Ext.state.Manager}
 * which stores the state of {@link Zarafa.core.ui.ContentPanel content panels}, {@link Ext.Panel panels}
 * and {@link Ext.grid.GridPanel grids} into the {@link Zarafa.settings.SettingsModel settings}.
 */
Zarafa.core.data.SettingsStateProvider = Ext.extend(Ext.state.Provider, {

	/**
	 * @cfg {String} basePath The base path of the {@link Zarafa.settings.SettingsModel settings}
	 * in which all State settings must be saved. The complete name for a settings path is generated
	 * by {@link #getStatePath}.
	 */
	basePath : 'zarafa/v1/state',

	/**
	 * @constructor
	 * @param {Object} config Configuration object
	 */
	constructor : function(config)
	{
		Ext.apply(this, config);

		Zarafa.core.data.SettingsStateProvider.superclass.constructor.call(this, config);
	},

	/**
	 * Create the path of the {@link Zarafa.settings.SettingsModel settings}
	 * in which the {@link Ext.Component#getState state} for the given {@link Ext.Component component}
	 * must be saved. This adds the {@link Ext.Component#getStateName State name} to the
	 * {@link #basePath}.
	 * @return {String} The Settings path in which the state for the component must be saved.
	 */
	getStatePath : function(component)
	{
		return this.basePath + '/' + component.getStateName();
	},

	/**
	 * Encodes a value including type information.  Decode with {@link #decodeValue}.
	 * @param {Mixed} value The value to encode
	 * @param {String} name The property name (if this function was called recursively) of the property being encoded
	 * @param {Ext.Component} component The component for which the settings are being encoded
	 * @return {String} The encoded value
	 */
	encodeValue : function(value, name, component)
	{
		if (Ext.isArray(value)) {
			for (var i = 0, len = value.length; i < len; i++) {
				value[i] = this.encodeValue(value[i], name, component);
			}
		} else if (Ext.isObject(value)) {
			for (var key in value) {
				value[key] = this.encodeValue(value[key], key, component);
			}
		} else if ((name === 'height' || name === 'width') && component.statefulRelativeDimensions === true) {
			value = this.scaleSizeToPercentage(component, value, name);	
		}

		return value;
	},

	/**
	 * Decodes a string previously encoded with {@link #encodeValue}.
	 * @param {Mixed} value The value to decode
	 * @param {String} name The property name (if this function was called recursively) of the property being decoded
	 * @param {Ext.Component} component The component for which the settings are being decoded
	 * @return {Mixed} The decoded value
	 */
	decodeValue : function(value, name, component)
	{
		if (Ext.isArray(value)) {
			for (var i = 0, len = value.length; i < len; i++) {
				value[i] = this.decodeValue(value[i], name, component);
			}
		} else if (Ext.isObject(value)) {
			for (var key in value) {
				value[key] = this.decodeValue(value[key], key, component);
			}
		} else if ((name === 'height' || name === 'width') && value <= 1.0) {
			value = this.scaleSizeToBody(component, value, name);
		}

		return value;
	},

	/**
	 * Convert a 'width' or 'height' value into a percentage of the current size of the body.
	 * This will ensure that when the WebApp is reloaded on a new display (with different resolution)
	 * the panel is scaled accordingly.
	 * @param {Ext.Component} component The component for which the percentages are calculated
	 * @param {Number} value The value to convert
	 * @param {String} type The type of the value (can be 'width' or 'height')
	 * @return {Number} The converted value
	 * @private
	 */
	scaleSizeToPercentage : function(component, value, type)
	{
		type = Ext.util.Format.capitalize(type);

		var body = window['inner' + type];
		value = parseFloat((value / body).toFixed(2));

		// Limit the value to 1, we don't accept it when the
		// component is resized to beyond the maximum of the
		// body element.
		return Math.min(value, 1);
	},

	/**
	 * Convert a 'width' or 'height' percentage into a real size depending on the size of the body.
	 * This will ensure that when the WebApp is reloaded on a new display (with different resolution)
	 * the panel is scaled accordingly. It will check the {@link Ext.Component#minWidth}/{@link Ext.Panel#maxWidth}
	 * and {@link Ext.Panel#minHeight}/{@link Ext.Panel#maxHeight} properties in the component to prevent
	 * it from becoming too big.
	 * @param {Ext.Component} component The component for which the real dimensions are determined.
	 * @param {Number} value The value to convert
	 * @param {String} type The type of the value (can be 'width' or 'height')
	 * @return {Number} The converted value
	 * @private
	 */
	scaleSizeToBody : function(component, value, type)
	{
		type = Ext.util.Format.capitalize(type);

		var body = window['inner' + type];
		var minSize = component['min' + type];
		var maxSize = component['max' + type];

		value = Math.round(value * body);

		// Collapsible components with a SplitBar have a minSize/maxSize properties,
		// while normal panels have the minWidth/maxWidth & minHeight/maxHeight properties.
		if (component.minSize || component.maxSize) {
			value = Math.min(Math.max(value, component.minSize || 0), component.maxSize || value);
		} else if (minSize || maxSize) {
			value = Math.min(Math.max(value, minSize || 0), maxSize || value);
		}

		return value;
	},

	/**
	 * Sets the value for a key
	 * @param {String} name The key name
	 * @param {Mixed} value The value to set
	 */
	set : function(name, value)
	{
		var component = Ext.state.Manager.getComponent(name);
		container.getSettingsModel().set(this.getStatePath(component), this.encodeValue(value, undefined, component));
		this.fireEvent('statechange', this, name, value);
	},

	/**
	 * Returns the current value for a key
	 * @param {String} name The key name
	 * @param {Mixed} defaultValue A default value to return if the key's value is not found
	 * @return {Mixed} The state data
	 */
	get : function(name, defaultValue)
	{
		var component = Ext.state.Manager.getComponent(name);
		var value = container.getSettingsModel().get(this.getStatePath(component), true);

		if (Ext.isObject(defaultValue)) {
			value = Ext.apply(defaultValue, value);
		}

		return this.decodeValue(value, undefined, component);
	},

	/**
	 * Clears a value from the state
	 * @param {String} name The key name
	 */
	clear : function(name)
	{
		var component = Ext.state.Manager.getComponent(name);
		// Call restore rather then remove, to ensure the default values
		// are applied again.
		container.getSettingsModel().restore(this.getStatePath(component));
		this.fireEvent('statechange', this, name, null);
	}
});
Ext.namespace('Zarafa.core.data');

/**
 * @class Zarafa.core.data.StatefulObservable
 * @extends Ext.util.Observable
 *
 * Extends the {@link Ext.util.Observable} and add {@link #stateful}
 * properties to the object allowing similar features as the
 * {@link Ext.Component}#{@link Ext.Component#stateful}.
 */
Zarafa.core.data.StatefulObservable = Ext.extend(Ext.util.Observable, {
	/**
	 * @cfg {Boolean} stateful A flag which causes the Component to attempt to restore the state of internal properties
	 * from a saved state on startup. The component must have a {@link #stateId}  for state to be managed.
	 */
	stateful : false,

	/**
	 * @cfg {String} stateId The unique id for this component to use for state management purposes
	 * See {@link #stateful} for an explanation of saving and restoring Component state.
	 */
	stateId : undefined,

	/**
	 * @cfg {Array} stateEvents An array of events that, when fired, should trigger this component to save its state
	 * (defaults to {@link #datamodechange}). stateEvents may be any type of event supported by this component.
	 * See {@link #stateful} for an explanation of saving and restoring Component state.
	 */
	stateEvents : undefined,

	/**
	 * @cfg {String} statefulName The unique name for this component by which the {@link #getState state}
	 * must be saved into the {@link Zarafa.settings.SettingsModel settings}.
	 * This option is only used when the {@link Zarafa.core.data.SettingsStateProvider SettingsStateProvider} is
	 * used in the {@link Ext.state.Manager}.
	 */
	statefulName : undefined,

	/**
	 * @constructor
	 * @param {Object} config Configuration object
	 */
	constructor : function(config)
	{
		Ext.apply(this, config);

		Zarafa.core.data.StatefulObservable.superclass.constructor.call(this, config);

		// If stateful, generate a stateId if not provided manually,
		// and register the object to the Ext.state.Manager.
		if (this.stateful !== false) {
			if (!this.stateId) {
				this.stateId = Ext.id(null, 'state-');
			}

			Ext.state.Manager.register(this);
			this.initStateEvents();
		}
	},

	/**
	 * Obtain the path in which the {@link #getState state} must be saved.
	 * This option is only used when the {@link Zarafa.core.data.SettingsStateProvider SettingsStateProvider} is
	 * used in the {@link Ext.state.Manager}. This returns {@link #statefulName} if provided, or else generates
	 * a custom name.
	 * @return {String} The unique name for this component by which the {@link #getState state} must be saved. 
	 */
	getStateName : function()
	{
		return this.statefulName;
	},

	/**
	 * Obtain the {@link #stateId} from the component
	 * @return {String} The state ID
	 * @private
	 */
	getStateId : function()
	{
		return this.stateId || this.id;
	},

	/**
	 * Register the {@link #stateEvents state events} to the {@link #saveState} callback function.
	 * @protected
	 */
	initStateEvents : function()
	{
		if (this.stateEvents) {
			for (var i = 0, e; e = this.stateEvents[i]; i++) {
				this.on(e, this.saveState, this, {delay:100});
			}
		}
	},

	/**
	 * Intialialize the component with {@link #applyState} using the state which has
	 * been {@link Ext.state.Manager#get obtained} from the {@link Ext.state.Manager State Manager}.
	 * @private
	 */
	initState : function()
	{
		if (Ext.state.Manager) {
			var id = this.getStateId();
			if (id) {
				var state = Ext.state.Manager.get(id);
				if (state) {
					if (this.fireEvent('beforestaterestore', this, state) !== false) {
						this.applyState(Ext.apply({}, state));
						this.fireEvent('staterestore', this, state);
					}
				}
			}
		}
	},

	/**
	 * Apply the given state to this object activating the properties which were previously
	 * saved in {@link Ext.state.Manager}.
	 * @param {Object} state The state object
	 * @protected
	 */
	applyState : function(state)
	{
		if (state) {
			Ext.apply(this, state);
		}
	},

	/**
	 * When {@link #stateful} the State object which should be saved into the
	 * {@link Ext.state.Manager}.
	 * @return {Object} The state object
	 * @protected
	 */
	getState : function()
	{
		return null;
	},

	/**
	 * Obtain the {@link #getState current state} and {@link Ext.state.Manager#set save} it
	 * to the {@link Ext.state.Manager State Manager}.
	 * @private
	 */
	saveState : function()
	{
		if (Ext.state.Manager && this.stateful !== false) {
			var id = this.getStateId();
			if (id) {
				var state = this.getState();
				if (this.fireEvent('beforestatesave', this, state) !== false) {
					Ext.state.Manager.set(id, state);
					this.fireEvent('statesave', this, state);
				}
			}
		}
	}
});
Ext.namespace('Zarafa.core.data');

/**
 * @class Zarafa.core.data.ZarafaCustomEventDispatcher
 * @extends Ext.util.Observable
 * @singleton
 *
 * This singleton can be used to register events that have to be available in the whole Webapp.
 */
Zarafa.core.data.ZarafaCustomEventDispatcher = Ext.extend(Ext.util.Observable, {
	/**
	 * @constructor
	 * @param config
	 */
	constructor: function(config) {
		// Call our superclass constructor to complete construction process.
		Zarafa.core.data.ZarafaCustomEventDispatcher.superclass.constructor.call(this, config)
	}
});

// make it a singleton
Zarafa.core.data.ZarafaCustomEventDispatcher = new Zarafa.core.data.ZarafaCustomEventDispatcher();Ext.ns('Zarafa.core.plugins');

/**
 * @class Zarafa.core.plugins.ComboAutoWidth
 * @extends Ext.util.Observable
 * @ptype zarafa.comboautowidth
 *
 * this plugin will set the width of the combo box to the width of the maximum text in list
 */
Zarafa.core.plugins.ComboAutoWidth = Ext.extend(Ext.util.Observable, {
	/**
	 * Initialize the plugin for the given {@link Ext.form.Field field}
	 * @param {Ext.form.ComboBox} comboBox on which the plugin is placed
	 */
	init : function(combo)
	{
		this.container = combo;

		combo.mon(combo.store, 'load', this.resizeToFitContent, this);
		combo.on('render', this.resizeToFitContent, this);
	},

	/**
	 * Called in response to a load event from the addressbook comboBox store.
	 * resize the combobox innerlist according to the widest list item content
	 * @param {Zarafa.core.data.IPMStore} store store that fired the event.
	 * @private
	 */
	resizeToFitContent : function()
	{
		var combo = this.container;
		var store = combo.getStore();
		var listWidth = 0;
		var textMetrics = Ext.util.TextMetrics.createInstance(combo.el);

		store.each(function(record) {
			var curWidth = textMetrics.getWidth(record.get(combo.displayField));
			if (curWidth > listWidth) {
				listWidth = curWidth;
			}
		});

		if (listWidth > 0) {
			listWidth = Math.max(combo.minListWidth, listWidth);

			listWidth += combo.getTriggerWidth(); // to accomodate combo's down arrow.
			if (combo.getWidth() < listWidth) {
				if (!combo.list) {
					combo.listWidth = listWidth;
				} else {
					combo.list.setWidth(listWidth);
				}
			}
		}
	}
});

Ext.preg('zarafa.comboautowidth', Zarafa.core.plugins.ComboAutoWidth);
Ext.namespace('Zarafa.core.plugins');

/**
 * @class Zarafa.core.plugins.TabCloseMenuPlugin
 * @extends Ext.ux.TabCloseMenu
 * @ptype zarafa.tabclosemenuplugin
 * 
 * Extended so that the context menu would call the tab's close() method instead of directly destroying it
 * This class is intended to be used with {@link Zarafa.core.ui.MainContentTabPanel}
 * TODO: handle closing multiple tabs better
 */
Zarafa.core.plugins.TabCloseMenuPlugin = Ext.extend(Ext.ux.TabCloseMenu, {
	/**
	 * Overriden to add menu items from an insertion point
	 * @override
	 * @private
	 */
	createMenu : function(){
		if(!this.menu){
			var items = [{
				itemId: 'close',
				text: this.closeTabText,
				scope: this,
				handler: this.onClose
			}];
			if(this.showCloseAll){
				items.push('-');
			}
			items.push({
				itemId: 'closeothers',
				text: this.closeOtherTabsText,
				scope: this,
				handler: this.onCloseOthers
			});
			if(this.showCloseAll){
				items.push({
					itemId: 'closeall',
					text: this.closeAllTabsText,
					scope: this,
					handler: this.onCloseAll
				});
			}
			var lazyItems = container.populateInsertionPoint('main.content.tabpanel.tabclosemenu');
			this.menu = new Ext.menu.Menu({
				items: items.concat(lazyItems)
			});
		}
		return this.menu;
	},

	/**
	 * Overriden, because the original function calls remove() directly
	 * item.close() is better because it fires an event that notifies other components that the tab wants to close
	 * @override
	 * @private
	 */
	onClose : function(){
		this.active.close();
	},

	/**
	 * Overriden, because the original function calls remove() directly
	 * item.close() is better because it fires an event that notifies other components that the tab wants to close
	 * @param {Boolean} excludeActive Flag for whether to close all tabs but preserve the currently active one
	 * @override
	 * @private
	 */
	doClose : function(excludeActive){
		this.tabs.items.each(function(item){
			if(item.closable){
				if(!excludeActive || item != this.active){
					item.close();
				}
			}
		}, this);
	}
});

Ext.preg('zarafa.tabclosemenuplugin', Zarafa.core.plugins.TabCloseMenuPlugin);
Ext.namespace('Zarafa.core.ui');

/**
 * @class Zarafa.core.ui.ButtonGroup
 * @extends Ext.ButtonGroup
 * @xtype zarafa.buttongroup
 */
Zarafa.core.ui.ButtonGroup = Ext.extend(Ext.ButtonGroup, {
	/**
	 * @constructor
	 * @param config
	 */
	constructor : function(config)
	{
		Ext.apply(this, config, {
			xtype: 'zarafa.buttongroup'
		});

		Zarafa.core.ui.ButtonGroup.superclass.constructor.call(this, config);
	},

	/**
	 * Function will check whether buttongroup has any visible buttons or not.
	 *
	 * @return {Boolean} function will return true if buttonGroups has any visible items,
	 * It will return false if buttonGroup is empty or buttonGroup has all hidden buttons.
	 */
	hasVisibleButtons : function()
	{
		var buttonsArray = this.items.getRange();
		if(!Ext.isEmpty(buttonsArray)){
			for (var i = 0; i < buttonsArray.length; i++)
			{
				//isVisible function is not working properly in current version of ExtJs.
				if(buttonsArray[i].hidden === false)
					return true;
			}
		}
		return false;
	}
});

Ext.reg('zarafa.buttongroup', Zarafa.core.ui.ButtonGroup);
Ext.namespace('Zarafa.core.ui');

/**
 * @class Zarafa.core.ui.ContentPanel
 * @extends Ext.Container
 * @xtype zarafa.contentpanel
 *
 * FIXME: this documentation needs to be completely rewritten
 * The {@link Zarafa.core.ui.ContentPanel ContentPanel} class a base class for content panels and is therefore not intended to be instantiated
 * directly. Developers must extend this class to implement a dialog window that can be shown as either an ExtJS
 * window or a separate browser window.
 * <p>
 * All dialog classes must be registered using the static register method:
 * <p>
 * <code><pre>
 * Zarafa.core.ui.ContentPanel.register(MyDialog, 'mydialog');
 * </pre></code>.
 * <p>
 * The dialog can then be instantiated using the create() method:
 * <p>
 * <code><pre>
 * MyDialog.create(config, {
 *   browser : true,
 *   width : 500,
 *   height : 300,
 * });
 * </pre></code>.
 *
 */
Zarafa.core.ui.ContentPanel = Ext.extend(Ext.Container, {
	/**
	 * @cfg {String} title The title for the ContentPanel
	 */
	title : undefined,

	/**
	 * @cfg {String} The action to take when the close header tool is clicked.
	 * Only used when using {@link Zarafa.core.data.UIFactoryWindowLayer UIFactoryWindowLayer} to
	 * display this {@link Zarafa.core.ui.ContentPanel ContentPanel}.
	 */
	closeAction : 'closeWrap',

	/**
	 * @cfg {Boolean} standalone If true, the {@link Zarafa.core.ui.ContentPanel contentpanel}
	 * will not be hooked into the {@link Zarafa.core.data.ContentPanelMgr}. This will prevent
	 * listening to events coming from the {@link Zarafa.core.data.ContentPanelMgr ContentPanelMgr}.
	 * Defaults to false.
	 */
	standalone : false,

	/**
	 * @cfg {Number} width The width for the ContentPanel
	 */
	width : 800,

	/**
	 * @cfg {Number} height The height for the ContentPanel
	 */
	height : 550,

	/**
	 * @cfg {String} iconCls Icon class for the tab
	 */
	iconCls : undefined,

	/**
	 * @constructor
	 * @param {Object} config configuration object.
	 */
	constructor : function(config)
	{
		config = config || {};

		config.plugins = Ext.value(config.plugins, []);
		config.plugins.push({
			ptype : 'zarafa.inputautofocusplugin'
		});

		Ext.applyIf(config, {
			xtype: 'zarafa.contentpanel',
			stateful : true
		});

		Zarafa.core.ui.ContentPanel.superclass.constructor.call(this, config);

		/**
		 * @event beforeclose
		 * Fires before the {@link Zarafa.core.ui.ContentPanel ContentPanel} is closed.
		 * A handler can return false to cancel the close.
		 * @param {Ext.Panel} panel The Panel being closed.
		 */
		/**
		 * @event close
		 * Fires after the Panel is closed.
		 * @param {Ext.Panel} panel The Panel that has been closed.
		 */
		this.initializeChildComponent(this);

		if (Ext.isDefined(this.title)) {
			this.setTitle(this.title);
		}

		// register with the ContentPanelMgr unless specifically instructed not to
		if (!this.standalone) {
			Zarafa.core.data.ContentPanelMgr.register(this);

		}
	},

	/**
	 * Called when the ContentPanel has been rendered.
	 * This activate the keymap on the element of this component after the normal operations of
	 * afterRender have been completed. It will activate by getting the xtype hierarchy from
	 * {@link #getXTypes} and format it into a string usable by the
	 * {@link Zarafa.core.KeyMapMgr KeyMapMgr}.
	 * @private
	 */
	afterRender: function()
	{
		Zarafa.core.ui.ContentPanel.superclass.afterRender.apply(this, arguments);
		var xtypes = this.getXTypes();

		// The first part leading up to zarafa.contentpanel will be stripped
		xtypes = xtypes.replace('component/box/container/zarafa.contentpanel','');

		// Then the "zarafa." will be stripped off from all the xtypes like "zarafa.somecontentpanel".
		xtypes = xtypes.replace(/\/zarafa\./g,'.');

		// Finally we strip string the "contentpanel" from all the xtypes. Otherwise each level will have
		// that "contentpanel" mentioned in them. Also we add contentpanel to the start as that sets
		// it apart from other components in the key mapping.
		xtypes = 'contentpanel' + xtypes.replace(/contentpanel/g, '');

		// This will activate keymaps with 'contentpanel.some.somemore' so the component
		// will have all key events register with 'contentpanel', 'some', 'somemore'
		Zarafa.core.KeyMapMgr.activate(this, xtypes);
	},

	/**
	 * This will initialize the {@link Ext.Container container} which is
	 * placed within the {@link Zarafa.core.ui.ContentPanel ContentPanel}. The
	 * {@link Ext.Container container} will recieve a reference to the
	 * {@link Zarafa.core.ui.ContentPanel ContentPanel} into which it has been embedded.
	 * @param {Ext.Component} component The component which must be initialized
	 * @private
	 */
	initializeChildComponent : function(component)
	{
		// Empty objects (undefined, null, []) cannot be initialized.
		if (Ext.isEmpty(component))
			return;

		// If it is an array, just recursively call
		// this function again for each individual item.
		var recursive = function(c) {
			this.initializeChildComponent(c);
		};

		if (Ext.isArray(component)) {
			Ext.each(component, recursive, this);
			return;
		} else if (component instanceof Ext.util.MixedCollection) {
			component.each(recursive, this);
			return;
		}

		// We only initialize containers (and their subclasses).
		if (!component.isXType('container'))
			return;

		component.dialog = this;

		// Toolbars include the top- and bottomtoolbars
		if (!Ext.isEmpty(component.toolbars)) {
			this.initializeChildComponent(component.toolbars);
		}

		// Initialize all child items
		if (!Ext.isEmpty(component.items)) {
			this.initializeChildComponent(component.items);
		}
	},

	/**
	 * Closes the panel. Destroys all child components, rendering the panel unusable.
	 */
	close : function()
	{
		if (this.fireEvent('beforeclose', this) !== false) {
			this.doClose();
		}
	},

	/**
	 * Close handler for the {@link #close} function.
	 * @private
	 */
	doClose : function()
	{
		this.fireEvent('close', this);
		Zarafa.core.data.ContentPanelMgr.unregister(this);
	},

	/**
	 * Function will be called when {@link Ext.Window#closeAction} config will be used to
	 * close the window. This is hardcoded to close the dialog instead of hiding it to make sure that
	 * we fire proper events to notify dialog is closed.
	 * @protected
	 */
	closeWrap : function()
	{
		this.close();
	},

	/**
	 * Sets the title of the panel.
	 * @param {String} title the new window title.
	 */
	setTitle : function(title)
	{
		this.title = title;
		this.fireEvent('titlechange', this, title);
	},

	/**
	 * @param {String} iconCls Icon class of the panel
	 */
	setIcon : function(iconCls)
	{
		var oldIcon = this.iconCls;
		this.iconCls = iconCls;
		this.fireEvent('iconchange', this, iconCls, oldIcon);
	},

	/**
	 * @return {Boolean} true iff the content panel is a modal dialog
	 */
	isModal : function()
	{
		return this.modal;
	},

	/**
	 * Obtain the path in which the {@link #getState state} must be saved.
	 * This option is only used when the {@link Zarafa.core.data.SettingsStateProvider SettingsStateProvider} is
	 * used in the {@link Ext.state.Manager}. This returns {@link #statefulName} if provided, or else generates
	 * a custom name.
	 * @return {String} The unique name for this component by which the {@link #getState state} must be saved.
	 */
	getStateName : function()
	{
		return 'dialogs/' + Zarafa.core.ui.ContentPanel.superclass.getStateName.call(this);
	},

	/**
	 * When {@link #stateful} the State object which should be saved into the
	 * {@link Ext.state.Manager}.
	 * @return {Object} The state object
	 * @protected
	 */
	getState : function()
	{
		var state = Zarafa.core.ui.ContentPanel.superclass.getState.call(this) || {};
		return Ext.apply(state, this.getSize());
	}
});

Ext.reg('zarafa.contentpanel', Zarafa.core.ui.ContentPanel);
Ext.namespace('Zarafa.core.ui');
/**
 * @class Zarafa.core.ui.ContextNavigationPanel
 * @extends Ext.Panel
 * @xtype zarafa.contextnavigation
 * 
 * ContextNavigationPanel provides custom navigation options to context through {@link Zarafa.hierarchy.ui.HierarchyTreePanel}.
 */
Zarafa.core.ui.ContextNavigationPanel = Ext.extend(Ext.Panel, {
	/**
	 * For this Context this panel will be visible in the {@link Zarafa.core.ui.NavigationPanel NavigationPanel}.
	 * @cfg {Zarafa.core.Context} Related Context
	 */
	context: null,

	/**
	 * @constructor
	 * @param {Object} config configuration object
	 */
	constructor : function (config) {
		config = config || {};

		// Config options for component itself.
		Ext.applyIf(config, {
			border : false,
			autoScroll : true,
			layout : {
				type : 'table',
				columns : 1,
				tableAttrs: {
					style: {
						width: '100%'
					}
				}
			},
			defaults : {
				border : false,
				autoScroll : false,
				headerCfg : { cls : 'zarafa-hierarchy-header x-panel-header' },
				defaults : { cls : 'zarafa-context-navigation-item-body' }
			}
		});

		Zarafa.core.ui.ContextNavigationPanel.superclass.constructor.call(this, config);
	},

	/**
	 * @return {Zarafa.core.Context}
	 */
	getContext : function() {
		return this.context || false;
	}
});

Ext.reg('zarafa.contextnavigation', Zarafa.core.ui.ContextNavigationPanel);
Ext.namespace('Zarafa.core.ui');

/**
 * @class Zarafa.core.ui.MainContentTabPanel
 * @extends Ext.TabPanel
 * This subclass is used in the main content area, and contains the ContextContainer and dialogs that the user opens
 * Initialized in MainViewport.CreateContentContainer
 */
Zarafa.core.ui.MainContentTabPanel = Ext.extend(Ext.TabPanel, {

	/**
	 * Overriden to modify the tab depending on whether the record has been edited or not
	 * This method is called when a contained component fires the 'titlechange' event
	 * @param {Component} item
	 * @param {String} title
	 * @override
	 * @private
	 */
	onItemTitleChanged : function(item, title, oldTitle)
	{
		var el = this.getTabEl(item);
		if (el) {
			// now elide title if it exceeds 20 character length
			var tab = Ext.get(el).child('.x-tab-strip-text', true);

			// Change tooltip, and give full title in tab tool tip
			tab.qtip = title;

			tab.innerHTML = Ext.util.Format.htmlEncodeElide(title, 20, 0);
		}
	},

	/**
	 * This method is called when a contained component fires the 'iconchange' event
	 * @param {Component} item
	 * @param {String} iconCls icon class to apply
	 * @param {String} oldCls The previous icon class
	 * @private
	 */
	onItemIconChanged : function(item, iconCls, oldCls)
	{
		var tabEl = this.getTabEl(item);
		if (!Ext.isEmpty(tabEl)) {
			// removeClass only works when the CSS classes have been split
			// into an array, as it will not do it manually. For addClass,
			// we do not have this restriction.
			if (oldCls) {
				oldCls = oldCls.split(' ');
			}

			var tabText = Ext.get(tabEl).child('.x-tab-strip-text');
			if (iconCls) {
				tabText.addClass('x-tab-strip-icon');
				tabText.replaceClass(oldCls, iconCls);
			} else {
				tabText.removeClass('x-tab-strip-icon');
				tabText.removeClass(oldCls);
			}
		}
	},

	/**
	 * This method is called when contained component fires the 'userupdaterecord' event
	 * @param {Component} item The item which fired the event
	 * @param {Ext.data.Record} record Which was updated by the user
	 * @param {Boolean} changed True if the item has been changed by the user
	 * @private
	 */
	onItemUserUpdateRecord : function(item, record, changed)
	{
		var el = this.getTabEl(item);
		if (el) {
			var tab = Ext.get(el).child('.x-tab-strip-text');
			if (record.phantom || changed) {
				tab.addClass('zarafa-tab-edited');
			} else {
				tab.removeClass('zarafa-tab-edited');
			}
		}
	},

	/**
	 * Overriden in order to listen to close event of child component
	 * @param {Component} item
	 * @param {Number} index
	 * @override
	 * @private
	 */
	initTab : function(item, index)
	{
		var title = item.title;
		if(!Ext.isEmpty(title)) {
			// provide a tooltip for tab titles
			item.tabTip = title;
			// now we can shorten the length of title if its exceeding 20 characters
			item.title = Ext.util.Format.htmlEncodeElide(title, 20, 0);
		}

		Zarafa.core.ui.MainContentTabPanel.superclass.initTab.call(this, item, index);

		item.on({
			scope : this,
			render : this.applyTooltip,
			iconchange : this.onItemIconChanged,
			userupdaterecord : this.onItemUserUpdateRecord,
			close : this.onTabClose
		});
	},

	/**
	 * This will apply tooltip on close button ('X') of {@link Ext.Panel Panel}.
	 * @param {Component} item.
	 */
	applyTooltip : function(item)
	{
		var el = item.tabEl;
		var closeTab = Ext.get(el).child('.x-tab-strip-close', true);
		if(closeTab) {
			closeTab.qtip = _('Close') + ' (Ctrl + Alt + W)';
		}
	},

	/**
	 * Handler for closing a tab when a child component has fired its close event
	 * For instance when a mail is sent, the MailCreateContentPanel needs to be closed
	 * @param {Component} tab
	 */
	onTabClose : function(tab)
	{
		if (this.fireEvent('beforeclose', tab)!==false) {
			this.remove(tab);
			this.fireEvent('close', tab);
		}
	},

	/**
	 * handler for the '+' button in the tab strip
	 * adds a new item of type depending on current context
	 * @param {Ext.EventObjectImpl} e Event object
	 * @param {Element} t Event target
	 * @param {Object} o Configuration object
	 */
	onTabAddClick : function(e, t, o)
	{

		var model = container.getCurrentContext().getModel();
		if(model){
			var record = model.createRecord();
			if (!record) {
				//if unable to create record from the current context model, invoke the handler of the first item in the 'new' menu
				var button = container.getMainPanel().mainToolbar.newButton;
				button.handler.call(button.scope);
				return;
			} else {
				Zarafa.core.data.UIFactory.openCreateRecord(record);
			}
		}
	},

	/**
	 * Overriden in order to add the '+' button to the edge of the tabstrip
	 * @param {Ext.Element} ct Container in which the panel is created
	 * @param {Element} position Element at which the panel is created (relative to its position)
	 * @override
	 */
	onRender : function(ct, position)
	{
		Zarafa.core.ui.MainContentTabPanel.superclass.onRender.call(this, ct, position);

		// insert add button into the edge element
		var edge = this.edge.update('<span id="zarafa-mainpanel-addtabbutton" class=\'x-tab-add\'></span>');
		this.mon(edge, 'click', this.onTabAddClick, this);

		// set tooltip on add button
		var addBtn = edge.child('.x-tab-add', true);
		addBtn.qtip = _('New item') + ' (Ctrl + Alt + N)';
	},

	/**
	 * Overriden in order to call close() on the item, instead of removing it immediately
	 * This allows the contained panel to fire a confirmation dialog
	 * @param {Ext.EventObjectImpl} e Event
	 * @private
	 * @override
	 */
	onStripMouseDown : function(e)
	{
		if (e.button !== 0) {
			return;
		}


		var target = this.findTargets(e);
		if (target.close) {
			target.item.close();
			return;
		}
		if (target.item && target.item != this.activeTab) {
			this.setActiveTab(target.item);
		}
	},

	/**
	 * Overriden so that '+' sign for adding tabs remains visible when there are scroll buttons
	 * @private
	 * @override
	 */
	getScrollWidth : function()
	{
		return this.edge.getOffsetsTo(this.stripWrap)[0] + this.getScrollPos() + this.edge.getWidth();
	}
});
Ext.namespace('Zarafa.core.ui');

/**
 * @class Zarafa.core.ui.MainTab
 * @extends Ext.Toolbar.Item
 * @xtype zarafa.maintab
 *
 * Is used to render the tabs in the {@link Zarafa.core.ui.MainTabBar MainTabBar}. It will relate to
 * a context and based on what is the active context according to the
 * {@link Zarafa.core.Container Container} it will mark itself as active.
 */
Zarafa.core.ui.MainTab = Ext.extend( Ext.Button, {
	/**
	 * @cfg {String} context Holds the name of the {@link Zarafa.core.Context Context} that this tab is related to.
	 */
	context: null,

	/**
	 * @constructor
	 * @param config Configuration structure
	 */
	constructor : function(config)
	{
		config = config || {};

		Ext.applyIf(config, {
			// Override from Ext.Component
			xtype: 'zarafa.maintab',
			cls : 'zarafa-maintab',
			buttonActiveCls: 'zarafa-maintabbar-maintab-active',
			handler: this.selectContext
		});

		Zarafa.core.ui.MainTab.superclass.constructor.call(this, config);

		this.mon(container, 'contextswitch', this.onContextSwitch, this);
		this.on('render', this.onRenderButton, this);
	},

	/**
	 * When switching to another context the state of the tab is re-evaluated.
	 * @private
	 */
	onContextSwitch: function(parameters, oldContext, newContext)
	{
		this.setContextActivityState(newContext);
	},

	/**
	 * When the tab is rendered the correct state is set.
	 * @private
	 */
	onRenderButton: function()
	{
		this.setContextActivityState(container.getCurrentContext());
	},

	/**
	 * Set the state of the tab based on the what the currently active context is. It will add the
	 * {@link #buttonActiveCls} as CSS class for the tab that is related to the active context. It
	 * will be removed if the related context is not active.
	 * @param {Zarafa.core.Context} currentContext The current context
	 * @private
	 */
	setContextActivityState: function(currentContext)
	{
		if(this.context == currentContext.getName()){
			this.addClass(this.buttonActiveCls);
		}else{
			this.removeClass(this.buttonActiveCls);
		}
	},


	/**
	 * Selects the context that this tab is related to set by the {@link #context} property. It will
	 * grab the {@link Zarafa.core.ContextModel ContextModel} and retrieve the default folder if the
	 * ContextModel exists. The default folder is opened for that context. The navigation panel is
	 * also just showing the folders related to that context.
	 * @private
	 */
	selectContext: function()
	{
		var context = container.getContextByName(this.context);

		if(context) {
			if(context === container.getCurrentContext()) {
				// if we are loading same context then don't do anything
				return;
			}

			var contextModel = context.getModel();
			var contextFolder;
			if (contextModel) {
				// Try to determine the folders, if previously
				// no folders were selected, we should select
				// the default folder.
				contextFolder = contextModel.getFolders();
				if (Ext.isEmpty(contextFolder)) {
					contextFolder = contextModel.getDefaultFolder();
				}
			}

			container.switchContext(context, contextFolder);
		}
	}
});

Ext.reg('zarafa.maintab', Zarafa.core.ui.MainTab);
Ext.namespace('Zarafa.core.ui');

/**
 * @class Zarafa.core.ui.MainTabBar
 * @extends Ext.Toolbar
 * @xtype zarafa.maintabbar
 * 
 * The MainTabBar shows the tabs at the top of the application. It can be filled by two insertion 
 * points that populate the left and the right side of the bar. It will use instances of the 
 * {@link Zarafa.core.ui.MainTab MainTab} to represent the tabs.
 */
Zarafa.core.ui.MainTabBar = Ext.extend(Ext.Toolbar, {
	// Insertion points for this class
	/**
	 * @insert main.maintabbar.left
	 * Insertion point for populating main tabbar to the left. The tabOrderIndex is used to 
	 * determine the order of the tabs. The lower the number the more to the left the button is 
	 * added.
	 * @param {Zarafa.core.ui.MainTabBar} toolbar This toolbar
	 */
	/**
	 * @insert main.maintabbar.right
	 * Insertion point for populating main tabbar to the right. The tabOrderIndex is used to 
	 * determine the order of the tabs. The lower the number the more to the right the button is 
	 * added.
	 * @param {Zarafa.core.ui.MainTabBar} toolbar This toolbar
	 */

	/**
	 * @constructor
	 * @param config Configuration structure
	 */
	constructor : function(config)
	{
		config = config || {};

		Ext.applyIf(config, {
			// Override from Ext.Component
			xtype: 'zarafa.maintabbar',
			id: 'zarafa-mainmenu',
			cls : 'zarafa-maintabbar',
			defaultType: 'zarafa.maintab',
			height: 28,
			boxMaxHeight: 28,
			boxMinHeight: 28,
			defaults: {
				height:20
			}
		});

		Zarafa.core.ui.MainTabBar.superclass.constructor.call(this, config);

		this.initBar();
	},

	/**
	 * Add items to this toolbar by using the main.maintabbar.left and main.maintabbar.right 
	 * insertion points. Also the text who is logged in and the log out button is added. The buttons
	 * added through the insertion points are sorted using the tabOrderIndex set on the objects in 
	 * the list returned by the insertion point.
	 * @private
	 */
	initBar : function()
	{
		
		var leftItems = container.populateInsertionPoint('main.maintabbar.left', this) || [];
		var rightItems = container.populateInsertionPoint('main.maintabbar.right', this) || [];

		// Make sure the items are properly sorted by priority.
		leftItems = Zarafa.core.Util.sortArray(leftItems, 'ASC', 'tabOrderIndex');
		// The right items are sorted so that the first item appears to the most right
		rightItems = Zarafa.core.Util.sortArray(rightItems, 'DESC', 'tabOrderIndex');
	
		this.addTooltip(leftItems, rightItems);

		var loginText = {
				xtype: 'tbtext',
				width: 'auto',
				cls: 'zarafa-maintabbar-logintext',
				text: String.format(_('You are logged on as {0}'), container.getUser().getDisplayName()),
				id: 'mainmenu-logintext'
		};

		var logoutButton = {
			text: _('Logout'),
			handler : this.onLogoutButton,
			id: 'mainmenu-button-logout'
		};

		this.add(leftItems, {xtype: 'tbfill'}, loginText, rightItems, logoutButton);
	},
	
	/**
	 * Used to apply key shortcut and context name in tooltip.
	 * @param {Array} leftItems The leftItems contains the left context items which are properly sorted by priority.
	 * @param {Array} rightItems The rightItems contains the right context items.
	 */
	addTooltip : function(leftItems, rightItems)
	{
		var contextItems = [];
		contextItems = contextItems.concat(leftItems, rightItems);
		Ext.each(contextItems, function(context, index) {
			context.tooltip = context.text + ' (Ctrl + '+index+')';
		});
	},

	/**
	 * Event handler which is called when the user presses the 'logout' button
	 * @private
	 */
	onLogoutButton : function()
	{
		container.logout();
	}
});

Ext.reg('zarafa.maintabbar', Zarafa.core.ui.MainTabBar);
Ext.namespace('Zarafa.core.ui');

/**
 * @class Zarafa.core.ui.MainViewSidebar
 * @extends Ext.Panel
 * @xtype mainviewsidebar
 *
 * The sidebars for the {@link Zarafa.core.ui.MainViewport Main Viewport}
 * which support {@link #collapse collapsing} and {@link #stateful}.
 */
Zarafa.core.ui.MainViewSidebar = Ext.extend(Ext.Panel, {

	/**
	 * @constructor
	 * @param {Object} config Configuration object
	 */
	constructor : function(config)
	{
		config = config || {};

		config.plugins = Ext.value(config.plugins, []);

		Ext.applyIf(config, {
			cls : 'zarafa-context-mainpanel',
			border : false,
			collapsible : true,
			collapsed : false,
			split : true,
			width : 250,
			minSize : 150,
			maxSize : 400
		});

		Zarafa.core.ui.MainViewSidebar.superclass.constructor.call(this, config);
	},

	/**
	 * Obtain the path in which the {@link #getState state} must be saved.
	 * This option is only used when the {@link Zarafa.core.data.SettingsStateProvider SettingsStateProvider} is
	 * used in the {@link Ext.state.Manager}. This returns {@link #statefulName} if provided, or else generates
	 * a custom name.
	 * @return {String} The unique name for this component by which the {@link #getState state} must be saved. 
	 */
	getStateName : function()
	{
		return 'sidebars/' + Zarafa.core.ui.MainViewSidebar.superclass.getStateName.call(this);
	}
});

Ext.reg('mainviewsidebar', Zarafa.core.ui.MainViewSidebar);
Ext.namespace('Zarafa.core.ui');

/**
 * @class Zarafa.core.ui.MainViewport
 * @extends Ext.Viewport
 * The main viewport for the application. It defines the basic switching context
 * containers (toolbar, main content panel)
 */
Zarafa.core.ui.MainViewport = Ext.extend(Ext.Viewport, {
	// Insertion points for this class
	/**
	 * @insert main.content
	 * Insertion point for the main panel content area.
	 */

	/**
	 * The {@link Zarafa.core.ui.NavigationPanel NavigationPanel}.
	 * @property
	 * @type Zarafa.core.ui.NavigationPanel
	 */
	navigationPanel : undefined,

	/**
	 * The {@link Zarafa.core.ui.widget.WidgetPanel WidgetPanel}
	 * @property
	 * @type Zarafa.core.ui.widget.WidgetPanel
	 */
	widgetPanel : undefined,

	/**
	 * The {@link Zarafa.core.ui.ContextContainer ContextContainer}
	 * @property
	 * @type Zarafa.core.ui.ContextContainer
	 */
	contentPanel : undefined,

	/**
	 * @constructor
	 * @param {Object} config Configuration object
	 */
	constructor : function(config)
	{
		config = config || {};

		config = Ext.applyIf(config, {
			layout : 'fit',
			items : [{
				xtype: 'container',
				id: 'zarafa-mainview',
				layout : 'border',
				cls : 'zarafa-panel-body',
				border : false,
				items : [
					this.createTopbarContainer(),
					this.createNavigationPanel(),
					this.createTodayPanel(),
					this.createContentContainer(),
					this.createBottomBar()
				]
			}]
		});
		
		// initialise the viewport with some pre-defined settings
		Zarafa.core.ui.MainViewport.superclass.constructor.call(this, config);

		// Activate global key events.
		Zarafa.core.KeyMapMgr.activate(null, 'global', Ext.getBody());
		// Don't allow propagation of all other key events which are registerd in KeyMapMgr.
		Zarafa.core.KeyMapMgr.activate(null, 'globaldisable', Ext.getBody());
	},

	/**
	 * Create the {@link Zarafa.core.ui.NavigationPanel NavigationPanel} for
	 * the west region of the client in which the treePanel can be shown.
	 * @return {Zarafa.core.ui.NavigationPanel} NavigationPanel.
	 * @private
	 */	
	createNavigationPanel : function()
	{
		this.navigationPanel = new Zarafa.core.ui.NavigationPanel({
			region : 'west',
			stateful : true,
			statefulName : 'hierarchybar',
			statefulRelativeDimensions : false
		});
		return this.navigationPanel;
	},

	/**
	 * Returns the {@link Zarafa.core.ui.NavigationPanel NavigationPanel} for
	 * the west region of the client in which the treePanel can be shown.
	 * @return {Zarafa.core.ui.NavigationPanel} NavigationPanel.
	 */	
	getNavigationPanel: function()
	{
		return this.navigationPanel;
	},

	/**
	 * Create the {@link Zarafa.core.ui.widget.WidgetPanel WidgetPanel} for
	 * the east region of the client in which the Today view can be shown.
	 * @return {Object} Configuration object for the WidgetPanel.
	 * @private
	 */
	createTodayPanel : function()
	{
		this.widgetPanel = new Zarafa.core.ui.widget.WidgetPanel({
			region : 'east',
			title : _('Zarafa'),
			numColumns : 1,

			stateful : true,
			statefulName : 'todaybar',
			statefulRelativeDimensions : false,
			settingsPath : 'zarafa/v1/contexts/today/sidebar',

			collapsed : true
		});
		return this.widgetPanel;
	},

	/**
	 * Returns the {@link Zarafa.core.ui.widget.WidgetPanel WidgetPanel} for
	 * the east region of the client in which the widgets can be shown.
	 * @return {Zarafa.core.ui.widget.WidgetPanel} widgetPanel
	 */
	getWidgetPanel : function()
	{
		return this.widgetPanel;
	},

	/**
	 * Create the {@link Zarafa.core.ui.ContextContainer ContextContainer} for
	 * the center region of the client in which the main contents can be shown.
	 * @return {Object} Configuration object for the ContextContainer
	 * @private
	 */
	createContentContainer : function()
	{
		var cc = new Zarafa.core.ui.ContextContainer({
			name : 'main.content',
			id: 'zarafa-mainpanel-content'
		});
		//get items from insertion point
		//TODO: create separate insertion points for front and back of tab strip?
		var lazyItems = container.populateInsertionPoint('main.content.tabpanel', this);
		
		this.contentPanel = new Zarafa.core.ui.MainContentTabPanel({
			id: 'zarafa-mainpanel',
			activeTab : 0,
			region : 'center',
			enableTabScroll : true,
			layoutOnTabChange : true,
			items : [ cc ].concat(lazyItems),
			plugins : [ 'zarafa.tabclosemenuplugin' ],
			cls : 'zarafa-body-tabbar'
		});
		return this.contentPanel;
	},

	/**
	 * Returns the {@link Zarafa.core.ui.ContextContainer ContentPanel} for
	 * the center region of the client in which the context will display its contents
	 * @return {Zarafa.core.ui.ContextContainer} contentPanel
	 */
	getContentPanel : function()
	{
		return this.contentPanel;
	},

	/**
	 * Create a {@link Ext.Container Container} that contains the 
	 * {@link Zarafa.core.ui.MainTabBar MainTabBar} and the {@link Zarafa.core.ui.MainTab MainTab} 
	 * at the top of the client.
	 * @return {Ext.Container} Container holding the MainTabBar and MainTab
	 * @private
	 */
	createTopbarContainer : function()
	{
		return new Ext.Container({
			name : 'main.topbar',
			region: 'north',
			layout: 'border',
			height: 84,
			items: [
				new Zarafa.core.ui.MainTabBar({
					name : 'main.maintabbar',
					region: 'center',
					height: 28,
					ref : '../../mainTabBar'
				}),
				new Zarafa.core.ui.MainToolbar({
					name : 'main.toolbar',
					region: 'south',
					height: 56,
					ref : '../../mainToolbar'
				})
			]
		});
	},

	/**
	 * Create a {@link Ext.Container Container} that will add a small bar at the bottom to give the
	 * user an visual clue of where the application ends.
	 * @return {Ext.Container} Container
	 * @private
	 */
	createBottomBar : function()
	{
		return new Ext.Container({
			name : 'main.bottombar',
			region: 'south',
			ref: 'bottomBar',
			height: 8,
			cls: 'zarafa-bottombar'
		});
	}
});
Ext.namespace('Zarafa.core.ui');

/**
 * @class Zarafa.core.ui.PreviewPanel
 * @extends Ext.Panel
 * @xtype zarafa.previewpanel
 *
 * Panel that shows a preview of a {@link Zarafa.core.data.IPMRecord message}.
 */
Zarafa.core.ui.PreviewPanel = Ext.extend(Ext.Panel, {
	// Insertion points for this class
	/**
	 * @insert previewpanel.toolbar.left
	 * Insertions point for inserting buttons in previewpane's toolbar on left side.
	 * @param {Zarafa.core.ui.PreviewPanel} panel This panel
	 * @param {Zarafa.core.ContextModel} model The contextmodel ({@link #model}).
	 */

	/**
	 * @insert previewpanel.toolbar.right
	 * Insertions point for inserting buttons in previewpane's toolbar on right side.
	 * @param {Zarafa.core.ui.PreviewPanel} panel This panel
	 * @param {Zarafa.core.ContextModel} model The contextmodel ({@link #model}).
	 */

	/**
	 * @cfg {Zarafa.core.ContextModel} model The model for which this
	 * PreviewPanel is being loaded.
	 */
	model : undefined,

	/**
	 * Reference to the {@link Zarafa.core.plugins.RecordComponentPlugin RecordComponent} plugin
	 * which is used to send update events to all child {@link Ext.Container containers}
	 * in this container. This field is initialized by the
	 * {@link Zarafa.core.plugins.RecordComponentPlugin RecordComponent} itself.
	 *
	 * @property
	 * @type Zarafa.core.plugins.RecordComponentPlugin
	 */
	recordComponentPlugin : undefined,

	/**
	 * @cfg {Object} Configuration object which will be used
	 * to instantiate the {@link Zarafa.core.plugins.RecordComponentPlugin RecordComponent}.
	 * See the plugin for the available configuration options.
	 */
	recordComponentPluginConfig : undefined,

	/**
	 * @cfg {Zarafa.core.data.IPMRecord} record (See {@link Zarafa.core.plugins.RecordComponentPlugin#record}).
	 */
	record : undefined,

	/**
	 * @cfg {Boolean} showLoadMask true if load mask should be shown else false.
	 */
	showLoadMask : true,

	/**
	 * The LoadMask object which will be shown when the {@link #record} is being opened, and
	 * the dialog is waiting for the server to respond with the desired data. This will only
	 * be set if {@link #showLoadMask} is true.
	 * @property
	 * @type Zarafa.common.ui.LoadMask
	 */
	loadMask : undefined,

	/**
	 * @constructor
	 * @param {Object} config configuration object.
	 */
	constructor : function(config)
	{
		config = config || {};

		config = Ext.applyIf(config, {
			layout : 'fit',
			stateful : true,
			minWidth : 200,
			minHeight : 200,
			xtype : 'zarafa.previewpanel'
		});

		config.tbar = Ext.applyIf(config.tbar || {}, {
			cls: 'zarafa-previewpanel-toolbar',
			xtype : 'zarafa.toolbar',
			hidden : true,
			items : []
		});

		var tbarItems = [
			container.populateInsertionPoint('previewpanel.toolbar.left', this, config.model),
			{xtype: 'tbfill'},
			config.tbar.items, // Default items in toolbar should be right aligned.
			container.populateInsertionPoint('previewpanel.toolbar.right', {scope : this, model : config.model})
		]

		config.tbar.items = Ext.flatten(tbarItems);
		config.plugins = Ext.value(config.plugins, []);
		config.plugins.push(Ext.applyIf(config.recordComponentPluginConfig || {}, {
			ptype: 'zarafa.recordcomponentplugin',
			useShadowStore : true,
			allowWrite : false
		}));

		config.plugins.push({
			ptype: 'zarafa.recordcomponentupdaterplugin'
		}, {
			ptype : 'zarafa.enablefocusplugin'
		});

		if (container.getSettingsModel().get('zarafa/v1/contexts/mail/readflag_time_enable') === true) {
			config.plugins.push({
				ptype: 'zarafa.markasreadplugin'
			});
		}

		Zarafa.core.ui.PreviewPanel.superclass.constructor.call(this, config);

		if (this.model) {
			this.mon(this.model, 'previewrecordchange', this.onPreviewRecordChange, this);
		}

		this.on({
			'beforeloadrecord' : this.onBeforeLoadRecord,
			'loadrecord' : this.onLoadRecord,
			'exceptionrecord' : this.onExceptionRecord,
			'show' : this.onPreviewPanelShow,
			'expand' : this.onPreviewPanelShow,
			'scope' : this
		});
	},

	/**
	 * Called when the ContentPanel has been rendered.
	 * This activate the keymap on the element of this component after the normal operations of
	 * afterRender have been completed. It will activate by getting the xtype hierarchy from
	 * {@link #getXTypes} and format it into a string usable by the
	 * {@link Zarafa.core.KeyMapMgr KeyMapMgr}.
	 * @private
	 */
	afterRender: function()
	{
		Zarafa.core.ui.PreviewPanel.superclass.afterRender.apply(this, arguments);
		var xtypes = this.getXTypes();

		// The first part leading up to zarafa.previewpanel will be stripped
		xtypes = xtypes.replace('component/box/container/panel/zarafa.previewpanel','');

		// Then the "zarafa." will be stripped off from start, the xtype like "zarafa.previewpanel".
		xtypes = xtypes.replace(/\/zarafa\./g,'.');

		// Finally we string the "previewpanel" from all the xtypes. Otherwise each level will have
		// that "previewpanel" mentioned in them. Also we add previewpanel to the start as that sets
		// it apart from other components in the key mapping.
		xtypes = 'previewpanel' + xtypes.replace(/previewpanel/g, '');

		// This will activate keymaps with 'previewpanel.mail' so the component
		// will have all key events register with 'previewpanel', 'mail'
		Zarafa.core.KeyMapMgr.activate(this, xtypes);
	},

	/**
	 * See {@link Zarafa.core.plugins.RecordComponentPlugin#setRecord}.
	 *
	 * @param {Zarafa.core.data.MAPIRecord} record The record to set
	 */
	setRecord : function(record)
	{
		for (var i = 0; i < this.toolbars.length; i++) {
			this.toolbars[i].setVisible(!!record);
		}

		if (this.recordComponentPlugin) {
			this.recordComponentPlugin.setRecord(record);
		}
	},

	/**
	 * Update the components with the given record.
	 *
	 * @param {Zarafa.core.data.MAPIRecord} record The record to update in this component
	 * @param {Boolean} contentReset force the component to perform a full update of the data.
	 */
	update : function(record, contentReset)
	{
		this.record = record;
	},

	/**
	 * Function for 'previewrecordchange' and 'show' events before setting record into component
	 * @param {Zarafa.core.data.MAPIRecord} record
	 * @private
	 */
	showRecordInPanel : function(record)
	{
		var panelConstructor;

		// Record is already loaded...
		if (this.record === record) {
			return;
		}

		// Check if same record is loaded again
		if (this.record && record && this.record.equals(record)) {
			return;
		}

		if (Ext.isDefined(record)) {
			panelConstructor = container.getSharedComponent(Zarafa.core.data.SharedComponentType['common.preview'], record);
			if (panelConstructor && this.get(0) instanceof panelConstructor) {
				this.setRecord(record);
				return;
			}
		}

		// Do a complete refresh, clearing all UI components
		// which might be active inside the panel.
		this.removeAll();

		if (panelConstructor) {
			this.add(new panelConstructor());
		}

		this.setRecord(record);
		this.doLayout();
		this.hideLoadMask();
	},

	/**
	 * Removes all components from this container. Additionally this function will also remove all the items from
	 * {@link Ext.Toolbar toolbar} also.
	 * @param {Boolean} autoDestroy (optional) True to automatically invoke the removed Component's {@link Ext.Component#destroy} function.
	 * Defaults to the value of this Container's {@link #autoDestroy} config.
	 * @return {Array} Array of the destroyed components
	 */
	removeAll : function(autoDestroy)
	{
		// remove toolbar items first
		var destroyedItems = [];
		if(this.getTopToolbar()) {
			destroyedItems.concat(this.getTopToolbar().removeAll.apply(this, arguments));
		}

		if(this.getBottomToolbar()) {
			destroyedItems.concat(this.getBottomToolbar().removeAll.apply(this, arguments));
		}

		destroyedItems.concat(Zarafa.core.ui.PreviewPanel.superclass.removeAll.apply(this, arguments));

		return destroyedItems;
	},

	/**
	 * Event handler which is trigggerd when the {@link Zarafa.core.ContextModel ContextModel} receives a new
	 * record for previewing (it fired the {@link Zarafa.core.ContextModel#previewrecordchange previewrecordchange} event).
	 * This updates the {@link Ext.Container previewpanel} with the selected record to be previewed.
	 *
	 * @param {Zarafa.core.ContextModel} model The context model.
	 * @param {Ext.data.Record} record The record which is selected for preview.
	 * @private
	 */
	onPreviewRecordChange : function(contextModel, record)
	{
		// Load record in preview panel
		if(this.isVisible()) {
			this.showRecordInPanel(record);
		}
	},

	/**
	 * If {@link #showLoadMask} is enabled, this function will display
	 * the {@link #loadMask}.
	 * @param {Boolean} errorMask True to show an error mask instead of the loading mask.
	 * @protected
	 */
	showLoadMask : function(errorMask)
	{
		if (this.showLoadMask === false) {
			return;
		}

		if (!this.loadMask) {
			this.loadMask = new Zarafa.common.ui.LoadMask(this.el);
		}

		if (errorMask) {
			this.loadMask.showError();
		} else {
			this.loadMask.show();
		}
	},

	/**
	 * If {@link #showLoadMask} is enabled, and {@link #showLoadMask} has been
	 * called to display the {@link #loadMask} this function will disable the
	 * loadMask.
	 * @protected
	 */
	hideLoadMask : function()
	{
		if (this.showLoadMask === false) {
			return;
		}

		if (this.loadMask) {
			this.loadMask.hide();
		}
	},

	/**
	 * Event handler for the {@link Zarafa.core.plugins.RecordComponentPlugin#beforeloadrecord beforeloadrecord} event.
	 * This will call the {@link #showLoadMask} function to show the loadmask.
	 *
	 * @param {Ext.Container} panel The panel to which the record belongs
	 * @param {Zarafa.core.data.IPMRecord} record The record which was updated
	 * @private
	 */
	onBeforeLoadRecord : function(panel, record)
	{
		this.showLoadMask();
	},

	/**
	 * Event handler for the {@link Zarafa.core.plugins.RecordComponentPlugin#loadrecord loadrecord} event.
	 * This will call the {@link #hideLoadMask} function to hide the loadmask.
	 *
	 * @param {Ext.Container} panel The panel to which the record belongs
	 * @param {Zarafa.core.data.IPMRecord} record The record which was updated
	 * @private
	 */
	onLoadRecord : function(panel, record)
	{
		this.hideLoadMask();
	},

	/**
	 * Event handler for the {@link Zarafa.core.plugins.RecordComponentPlugin#exceptionrecord} event.
	 * This will call {@link #showLoadMask} to update it with a new error message.
	 *
	 * @param {String} type See {@link Ext.data.DataProxy}.{@link Ext.data.DataProxy#exception exception}
	 * for description.
	 * @param {String} action See {@link Ext.data.DataProxy}.{@link Ext.data.DataProxy#exception exception}
	 * for description.
	 * @param {Object} options See {@link Ext.data.DataProxy}.{@link Ext.data.DataProxy#exception exception}
	 * for description.
	 * @param {Object} response See {@link Ext.data.DataProxy}.{@link Ext.data.DataProxy#exception exception}
	 * for description.
	 * @param {Zarafa.core.data.MAPIRecord} record The record which was subject of the request
	 * that encountered an exception.
	 * @param {String} error (Optional) Passed when a thrown JS Exception or JS Error is
	 */
	onExceptionRecord : function(type, action, options, response, record)
	{
		this.showLoadMask(true);
	},

	/**
	 * Event handler which is fired when this panel has been set to visible. Fetches previewRecord from
	 * the {@link Zarafa.core.ContextModel ContextModel} and displays into panel.
	 */
	onPreviewPanelShow : function()
	{
		if (Ext.isDefined(this.model)) {
			var record = this.model.getPreviewRecord();
			if (Ext.isDefined(record)) {
				this.showRecordInPanel(record);
			}
		}
	},

	/**
	 * Obtain the path in which the {@link #getState state} must be saved.
	 * This option is only used when the {@link Zarafa.core.data.SettingsStateProvider SettingsStateProvider} is
	 * used in the {@link Ext.state.Manager}. This returns {@link #statefulName} if provided, or else generates
	 * a custom name.
	 * @return {String} The unique name for this component by which the {@link #getState state} must be saved.
	 */
	getStateName : function()
	{
		return 'preview/' + Zarafa.core.ui.PreviewPanel.superclass.getStateName.call(this);
	}
});
Ext.reg('zarafa.previewpanel', Zarafa.core.ui.PreviewPanel);
Ext.namespace('Zarafa.core.ui');

/**
 * @class Zarafa.core.ui.SwitchViewContentContainer
 * @extends Ext.Container
 * @xtype zarafa.switchviewcontentcontainer
 *
 * This class represents an {@link Ext.Panel panel} which contains multiple views which
 * can be enabled/disabled at any time by the user. Using lazy loading each view will only
 * be allocated when the user switches to the given view for the first time.
 */
Zarafa.core.ui.SwitchViewContentContainer = Ext.extend(Ext.Container, {
	/**
	 * @cfg {Object|Array} lazyItems List of {@link Ext.Containers containers} which
	 * act as view for this {@link Ext.Container container}. This array consists of configuration
	 * objects which will be allocated when the user switches to the given view for the
	 * first time.
	 * All items added to this list must always contain the 'id' property,
	 * used in switchView. Without this property it is not possible to switch
	 * to this view.
	 */
	lazyItems : undefined,

	/**
	 * @cfg {Boolean} autoClean If true this container will automatically remove
	 * and delete the previously selected {@link Ext.Container container} when switching
	 * to a new active {@link Ext.Container container}.
	 */
	autoClean : true,

	/**
	 * @constructor
	 * @param {Object} config Configuration object
	 */
	constructor : function(config)
	{
		config = config || {};

		// Always ensure that at least 1 item is non-lazy
		if (Ext.isEmpty(config.items) && !Ext.isEmpty(config.lazyItems)) {
			config.items = [ config.lazyItems[0] ];
		}
		// Ensure that the non-lazy item is marked as active
		if (Ext.isEmpty(config.activeItem)) {
			config.activeItem = config.items[0].id;
		}

		Ext.applyIf(config, {
			autoDestroy: true
		});

		this.addEvents(
			/**
			 * @event switchview
			 * Fires when the active view is being changed
			 * @param {Ext.Container} this The {@link Zarafa.core.ui.SwitchViewContentContainer switchcontainer}.
			 * @param {Ext.Container} newView The new {@link Ext.Container view} which is shown
			 * @param {Ext.Container} oldView The old {@link Ext.Container view} which was shown
			 */
			'switchview'			
		);

		Zarafa.core.ui.SwitchViewContentContainer.superclass.constructor.call(this, config);
	},

	/**
	 * Called by Extjs when the container is being {@link #doLayout layed out}. This will obtain
	 * the {@link Ext.layout.CardLayout#activeItem} and {@link Ext.Panel#doLayout update the layout}
	 * on that component as well.
	 * @private
	 */
	onLayout : function()
	{
		Zarafa.core.ui.SwitchViewContentContainer.superclass.onLayout.apply(this, arguments);

		// If the activeItem contains a layout, it should be layed out as well
		var item = this.getActiveItem();
		if (Ext.isFunction(item.doLayout)) {
			item.doLayout();
		}
	},

	/**
	 * This function will be used to switch between different views.
	 * It will attempt to find the view within this panel, if found,
	 * then the layout manager will be updated with the new view.
	 * @param {String} viewId id of the view that should be shown
	 */
	switchView : function(viewId)
	{
		var oldView = this.getActiveItem();
		var newView = this.findById(viewId);

		if (!Ext.isDefined(newView) || oldView == newView) {
			return;
		}

		// Check if the layout has been created yet, if not
		// then we store the activeItem inside the current
		// panel so it can be applied to the layout when it
		// is being created. 
		var layout = this.getLayout();
		if (!Ext.isFunction(layout.setActiveItem)) {
			this.activeItem = viewId;
		} else {
			layout.setActiveItem(viewId);
		}

		this.fireEvent('switchview', this, newView, oldView);

		// TODO: We should enable some timeout mechanism which
		// removes and deletes the oldView after a particular timeout.
		// This should increase performance when switching between 2
		// views often.
		if (this.autoClean === true && oldView && oldView != this.getActiveItem()) {
			this.remove(oldView);
			delete oldView;
		}
	},

	/**
	 * This function returns the currently active item
	 * @return {Ext.Component} The active item
	 */
	getActiveItem : function()
	{
		var layout = this.getLayout();
		if (!Ext.isFunction(layout.setActiveItem)) {
			return this.activeItem;
		} else {
			return layout.activeItem;
		}
	},

	/**
	 * Find a component under this container at any level by id.
	 * This extension will read all lazy items as well, if the required id, is
	 * one of the lazy items, then the item will be created and added to the panel.
	 * @param {String} id
	 * @return Ext.Component
	 */
	findById : function(id)
	{
		var retval = Zarafa.core.ui.SwitchViewContentContainer.superclass.findById.call(this, id);
		if (!retval) {
			retval = this.findBy(function(item) { return item.id === id; });
			if (!Ext.isEmpty(retval)) {
				retval = retval[0];
			}
		}

		return retval;
	},

	/**
	 * Find a component under this container at any level by a custom function.
	 * If the passed function returns true, the component will be included in the results.
	 * The passed function is called with the arguments (component, this container).
	 *
	 * This function will not only search through {@link #items} but also through {@link #lazyItems}.
	 *
	 * @param {Function} fn The function to call
	 * @param {Object} scope (optional)
	 * @return {Array} Array of Ext.Components
	 */
	findBy : function(fn, scope)
	{
		var retval = Zarafa.core.ui.SwitchViewContentContainer.superclass.findBy.apply(this, arguments);

		if (Ext.isDefined(this.lazyItems)) {
			for (var i = 0; i < this.lazyItems.length; i++) {
				var item = this.lazyItems[i];

				if (fn.call(scope || item, item, this) === true) {
					/*
					 * Make an exact clone of the lazyItems entry. When we pass the config
					 * object to the Component constructor, we must consider the fact that
					 * this would be a reference. What we pass here, might be modified by
					 * the constructor. However we do not want those changes to be saved
					 * back into our lazyItems object, since that might cause problems
					 * when we instantiate the lazyItem for the second time.
					 */
					item = Ext.create(Ext.apply({}, item));
					this.add(item);
					retval.push(item);
				}
			}
		}

		return retval;
	}
});

Ext.reg('zarafa.switchviewcontentcontainer', Zarafa.core.ui.SwitchViewContentContainer);
Ext.namespace('Zarafa.core.ui');

/**
 * @class Zarafa.core.ui.Toolbar
 * @extends Ext.Toolbar
 * @xtype zarafa.toolbar
 */
Zarafa.core.ui.Toolbar = Ext.extend(Ext.Toolbar, {
	// Private
	// Used to keep track of whether the toolbar buttongroups have been corrected in size yet.
	initialSizeCorrectionDone: false,

	/**
	 * @constructor
	 * @param config
	 */
	constructor : function(config)
	{
		Ext.apply(this, config, {
			// Override from Ext.Component
			xtype: 'zarafa.toolbar',
			// Override from Ext.Toolbar
			enableOverflow: true
		});

		Zarafa.core.ui.Toolbar.superclass.constructor.call(this, config);

		this.on('afterlayout', this.onAfterLayout, this);
	},

	/**
	 * Add a new {@link Ext.ButtonGroup} to the {@link Ext.Toolbar} using an insertionpoint to support
	 * the addition of extra {@link Ext.Button} elements to the {@link Ext.ButtonGroup}. If the insertion
	 * point contains a {@link Ext.ButtonGroup} it will be added outside the given {@link Ext.ButtonGroup}
	 * to prevent nesting.
	 *
	 * @param {Object/String} group The {@link Ext.ButtonGroup} to which individual {@link Ext.Button} elements must be added.
	 * If this argument is a {@link String} it will be used as title for the {@link Ext.ButtonGroup}.
	 * @param {String} insertion (optional) The name of the insertion point used to add additional {@link Ext.Button} and
	 * {@link Ext.ButtonGroup} elements.
	 * @protected
	 */
	addItems : function(buttons, insertion)
	{
		var items = [];

		if (Ext.isDefined(buttons)) {
			if (!Ext.isEmpty(buttons)) {
				items = items.concat(buttons);
			}
		}

		if (Ext.isString(insertion)) {
			var insert = container.populateInsertionPoint(insertion);
			if (!Ext.isEmpty(insert)) {
				items = items.concat(insert);
			}
		}

		// Make sure we get a single array containing all buttons
		items = Ext.flatten(items);

		this.add(items);
	},

	/**
	 * When the toolbar is layed out the sizes for the different buttongroups may differ. When the
	 * afterlayout event is fired the heights are corrected to have all the buttongroups match in
	 * height. This is only done the first time the toolbar is shown.
	 * @private
	 */
	onAfterLayout: function()
	{
		// Only do it the first time the toolbar is shown.
		if(!this.initialSizeCorrectionDone && this.el.getHeight() > 0){

			var elements = [];
			var maxHeight = 0;
			this.items.each(function(item){
				if (item.isXType('buttongroup')) {
					var elem = item.body;
					maxHeight = Math.max(maxHeight, elem.getHeight());
					elements.push(elem);
				}
			});

			if(maxHeight > 0){
				Ext.each(elements, function(elem){
					elem.setHeight(maxHeight);
				}, this);
				this.initialSizeCorrectionDone = true;
			}
		}
	}
});

Ext.reg('zarafa.toolbar', Zarafa.core.ui.Toolbar);
Ext.namespace('Zarafa.core.ui');

/**
 * @class Zarafa.core.ui.ToolbarButton
 * @extends Ext.Button
 * @xtype zarafa.toolbarbutton
 *
 * Special implementation for the {@link Ext.Button button}
 * which connects to the Mail {@link Ext.Panel panel} for the
 * currently active {@link Zarafa.core.Context context}.
 * Using this link, the {@link Ext.Button button} can be disabled
 * or enabled depending on the number of currently selected
 * {@link Zarafa.core.data.IPMRecord records}.
 *
 * Also this class makes it easier for the {@link Ext.Button button}
 * to request the currently selected {@link Zarafa.core.data.IPMRecord records}
 * on which it can perform an action.
 *
 * FIXME: This should somehow be merged with Zarafa.core.ui.menu.ConditionalItem.
 */
Zarafa.core.ui.ToolbarButton = Ext.extend(Ext.Button, {
	/**
	 * @cfg {Boolean} emptySelectOnly This button must only be enabled
	 * if no record is selected
	 */
	emptySelectOnly : false,
	/**
	 * @cfg {Boolean} nonEmptySelectOnly This button must only be enabled
	 * if one or more records are selected.
	 */
	nonEmptySelectOnly : false,
	/**
	 * @cfg {Boolean} singleSelectOnly This button must only be enabled
	 * if a single record is selected
	 */
	singleSelectOnly : false,
	/**
	 * @cfg {Boolean} multiSelectOnly This button must only be enabled
	 * if multiple records are selected.
	 */
	multiSelectOnly : false,
	/**
	 * @constructor
	 * @param {Object} config Configuration object
	 */
	constructor : function(config)
	{
		config = config || {};

		Ext.applyIf(config, {
			xtype: 'zarafa.toolbarbutton'
		});

		Ext.apply(this, config);

		Zarafa.core.ui.ToolbarButton.superclass.constructor.call(this, config);

		var model = this.model || container.getCurrentContext().getModel();
		if (Ext.isDefined(model)) {
			this.onRecordSelectionChange(model, model.getSelectedRecords());
			this.mon(model, 'recordselectionchange', this.onRecordSelectionChange, this);
		}
	},

	/**
	 * Event handler which is triggered when the current {@link Zarafa.core.data.IPMRecord record}
	 * selection changes. This will then evaluate if the {@link Ext.Button button} must be
	 * enabled or disabled.
	 *
	 * @param {Zarafa.core.ContextModel} model this model.
	 * @param {Zarafa.core.data.Record|Array} records The selected records
	 */
	onRecordSelectionChange : function(model, records)
	{
		if (this.emptySelectOnly) {
			if (Ext.isDefined(records) && (!Ext.isArray(records) || records.length > 0))
				this.setDisabled(true);
			else
				this.setDisabled(false);
		} else if (this.nonEmptySelectOnly) {
			if (!Ext.isDefined(records) || (Ext.isArray(records) && records.length == 0))
				this.setDisabled(true);
			else
				this.setDisabled(false);
		} else if (this.singleSelectOnly) {
			if (!Ext.isDefined(records) || (Ext.isArray(records) && records.length != 1))
				this.setDisabled(true);
			else
				this.setDisabled(false);
		} else if (this.multiSelectOnly) {
			if (!Ext.isDefined(records) || (!Ext.isArray(records) || records.length == 1))
				this.setDisabled(true);
			else
				this.setDisabled(false);
		}
	}
});

Ext.reg('zarafa.toolbarbutton', Zarafa.core.ui.ToolbarButton);
Ext.namespace("Zarafa.core.ui");

/**
 * @class Zarafa.core.ui.View
 * @extends Ext.util.Observable
 * The View class is a component for building views. Views abstract away the details
 * of manipulating HTML and the DOM from underlying controls such as grid panels
 * and {@link Zarafa.calendar.ui.CalendarPanel CalendarPanel}. 
 * 
 * The View class supports convenience methods for creating elements, managing
 * several child views, etc.
 * <p>
 * The createDiv() method can be used to create div elements. These elements are then 
 * automatically destroyed by the destroy() method. 
 * Child views can be added using addChildView(). These are then also destroyed 
 * automatically by the default destroy() implementation.
 */
Zarafa.core.ui.View = Ext.extend(Ext.util.Observable, {
	/**
	 * @cfg {String} baseCls The base CSS class to apply to this panel's element.
	 */
	baseCls : undefined,

	/**
	 * @cfg {String} itemCls The item CSS class to apply to this panel's element.
	 */
	itemCls : undefined,

	/**
	 * @cfg {String} themeCls The CSS theme which must be applied to this view. The name will
	 * be added to the various CSS classes which are generated using {@link #getClassName}.
	 * will be used.
	 */
	themeCls : undefined,

	/**
	 * The parent element for this view
	 * @property
	 * @type Object
	 */
	parentView : undefined,

	/**
	 * The {@link Ext.Element} in which this view has been rendered.
	 * This is passed as argument to {@link #render}. 
	 * @property
	 * @type Ext.Element
	 */
	container : undefined,

	/**
	 * The list of child objects which were created for this view. When
	 * this object is destroyed all registered children will be automatically
	 * cleaned up as well.
	 * @property
	 * @type Array
	 */
	children : undefined,

	/**
	 * The list of child elements which were created by this view using
	 * the {@link #create} and {@link #createDiv} functions. When this
	 * object is destroyed all registered elements will be automatically
	 * cleaned up as well.
	 * @property
	 * @type Array
	 */
	elements : undefined,

	/**
	 * Indicates if this view has been {@link #render rendered}.
	 * @property
	 * @type Boolean
	 */
	rendered : false,

	/**
	 * True if the view has been destroyed already. Read only
	 * @property isDestroyed
	 * @type Boolean
	 */
	isDestroyed: false,

	/**
	 * @constructor
	 * @param {Object} config Configuration object
	 */
	constructor : function(config)
	{
		Ext.apply(this, config);
		
		this.addEvents(
			/**
			 * @event beforedestroy
			 * Fires before the component is {@link #destroy}ed. Return false from an event handler to stop the {@link #destroy}.
			 * @param {Ext.Component} this
			 */
			'beforedestroy',
			/**
			 * @event destroy
			 * Fires after the component is {@link #destroy}ed.
			 * @param {Ext.Component} this
			 */
			'destroy'
		);

		Zarafa.core.ui.View.superclass.constructor.call(this, config);

		this.init();
	},

	/**
	 * Initialises the view. When a {@link #parentView} is defined,
	 * this will automatically register this view to the given parentView.
	 * @protected 
	 */
	init : function()
	{
		this.children = [];
		this.elements = [];

		if (Ext.isDefined(this.parentView)) {
			this.parentView.addChildView(this);
		}
	},

	/**
	 * Returns the base className which must be applied to all {@link Ext.Element elements} within this
	 * view. This will also be the prefix of any additional CSS classes which will be applied to those elements.
	 * The exact base classname depends on the {@link #baseCls} and the availablity of the {@link #itemCls}.
	 * @return {String} The base class. 
	 * @private
	 */
	getBaseClassName : function()
	{
		return this.itemCls ? (this.baseCls + '-' + this.itemCls) : this.baseCls;
	},

	/**
	 * Helper function for generating a string with CSS class names. Each {@link Ext.Element Element} must
	 * at least contain the {@link #baseCls} CSS class and CSS classes containing the {@link #itemCls} and
	 * {@link #themeCls} but additional names will be generated for specific elements. Example:
	 * <pre>
	 *   baseCls =  'zarafa-calendar';
	 *   getClassName('body', 'image');
	 * </pre>
	 * The outcome will be:
	 * <pre>
	 *   'zarafa-calendar zarafa-calendar-body zarafa-calendar-body-image zarafa-calendar-blue zarafa-calendar-blue-body zarafa-calendar-blue-body-image'
	 * </pre>
	 *
	 * @param {String} name The main name for the CSS element, this will be appended to the {@link #baseCls}.
	 * @param {String/Array} postfix (optional) If provided extra CSS classNames will be generated with the name and postfix. If the postfix
	 * is an array, all elements from the array will be used as postfix.
	 * @return {String} the CSS class names
	 * @param {String/Array} extraCls (optional) If provided extra CSS classNames will be generated with this extraCls. If the postfix
	 * is an array, all elements from the array will be used as extraCls.
	 * @private
	 */
	getClassName : function(name, postfix, extraCls)
	{
		var baseCls = this.getBaseClassName();
		var className = this.baseCls;

		if (!Ext.isEmpty(this.themeCls)) {
			className += ' ' + this.baseCls + '-' + this.themeCls;
		}

		className += ' ' + baseCls + '-' + name;

		if (Ext.isDefined(postfix) && !Ext.isArray(postfix)) {
			postfix = [ postfix ];
		}

		if (!Ext.isEmpty(postfix)) {
			for (var i = 0, len = postfix.length; i < len; i++) {
				className += ' ' + baseCls + '-' + name + '-' + postfix[i];
			}
		}

		if (Ext.isDefined(extraCls) && !Ext.isArray(extraCls)) {
			extraCls = [ extraCls ];
		}

		if (!Ext.isEmpty(extraCls)) {
			for (var i = 0, len = extraCls.length; i < len; i++) {
				className += ' ' + baseCls + '-' + extraCls[i];
			}
		}

		return className;
	},

	/**
	 * A convenience method for creating DIV elements. The new DIV element will be
	 * wrapped in an {@link Ext.Element} and added to the view as a property called 'name'.
	 * Equivalent to <code>create('div', parent, name, className)</code>.
	 * @param {Mixed} parent element
	 * @param {String} name the name of the property to create
	 * @param {String} className (optional) class name for the new DIV 
	 * @return {Ext.Element} The created div element
	 * @private
	 */
	createDiv : function(parent, name, className)
	{
		return this.create('div', parent, name, className);
	},
	
	/**
	 * A convenience method for creating DOM elements. The new element will be
	 * wrapped in an {@link Ext.Element} and added to the view as a property called 'name'.
	 * Calling <pre>this.create('div', 'test')</pre> for instance will create a new
	 * property <pre>this.test</pre>, and the DIV can be manipulated with code such
	 * as <pre>this.test.setSize(100, 100);</pre>. If a property already exists
	 * with the given name, the property will be converted into an array and
	 * the new element will the appended to that array.
	 * @param {String|Object} tagName type of HTML tag to generate (i.e. 'div', 'img', etc.) 
	 * @param {Mixed} parent element
	 * @param {String} name the name of the property to create
	 * @param {String} className (optional) class name for the new DIV
	 * @return {Ext.Element} The created element
	 * @private
	 */
	create : function(tagName, parent, name, className)
	{
		var config = Ext.isObject(tagName) ? tagName : { tag : tagName };

		Ext.applyIf(config, { cls : className || this.baseCls });

		// create a new HTML element and add it to the parent element
		var element = Ext.get(parent).createChild(config);
		
		// add the element as a property to this
		if (!this[name])
			this[name] = element;
		else if (Ext.isArray(this[name]))
			this[name].push(element);
		else
			this[name] = [ this[name], element ];
		
		// add the element to the elements list so that we may destroy it easily later on
		this.elements.push(element);
		
		return element;
	},
	
	/**
	 * Removes a child element from the view and destroys it.
	 * @param {Ext.Element} element DOM element 
	 */
	remove : function(element)
	{
		element = Ext.get(element);
		this.elements.remove(element);
		element.remove();
	},
	
	/**
	 * @return {Ext.Element} The main element of the view. This is the element the
	 * view is rendered <i>into</i>. The container of a view may be shared with other 
	 * views, so manipulating the container of a given view might affect other views
	 * as well. 
	 */
	getContainer : function()
	{
		return this.container;
	},

	/**
	 * Calls 'render' on each of the child views
	 * @param {Ext.Element} container The container into which the children must be rendered. Defaults to {@link #el}.
	 * @private
	 */
	renderChildren : function(container)
	{
		for (var i = 0, len = this.children.length; i < len; i++) {
			this.children[i].render(container || this.container);
		}
	},
	
	/**
	 * Renders the view. 
	 * Rendering a view is done once after construction, and creates all the DOM elements needed for the visual 
	 * representation of the view. 
	 * @param {Ext.Element} container The Ext.Element into which the view must be rendered.
	 */
	render : function(container)
	{
		this.container = container;
		// TODO Can't we call renderChildren here?
		this.rendered = true;
	},
	
	/**
	 * Calls 'layout' on each of the child views.
	 * @private
	 */
	layoutChildren : function()
	{
		for (var i = 0, len = this.children.length; i < len; i++) {
			this.children[i].layout();
		}
	},

	/**
	 * Called just before this View will be {@link #layout layed out}.
	 * @protected
	 */
	onBeforeLayout : Ext.emptyFn,

	/**
	 * Called when this view is being {@link #layout layed out}.
	 * @protected
	 */
	onLayout : Ext.emptyFn,

	/**
	 * Called just after this View has been {@link #layout layed out}.
	 * @protected
	 */
	onAfterLayout : Ext.emptyFn,

	/**
	 * Lays out the view, setting the position and size of the individual DOM elements.
	 */
	layout : function()
	{
		this.onBeforeLayout();
		this.onLayout();
		this.onAfterLayout();
	},

	/**
	 * Calls 'destroy' on each of the child views.
	 * @private
	 */
	destroyChildren : function()
	{
		for (var i = 0, len = this.children.length; i < len; i++) {
			this.children[i].destroy();
		}
	},
	
	/**
	 * Destroys the view, removing all DOM elements generated by render() from the DOM tree and unhooks
	 * any events.
	 */
	destroy : function()
	{
		if (!this.isDestroyed) {
			if (this.fireEvent('beforedestroy', this) !== false) {
				// remove all elements generated with createDiv() from the DOM
				for (var i = 0, len = this.elements.length; i < len; i++) {
					this.elements[i].remove();
				}

				// destroy child views
				this.destroyChildren();

				this.onDestroy();

				this.fireEvent('destroy', this);
				this.purgeListeners();
			}

			this.isDestroyed = true;
		}
	},

	/**
	 * Set the {@link #parentView} field. This will be called by {@link #addChildView}
	 * to update the parent to this view. 
	 * @param {Zarafa.core.ui.View} parentView The parent view
	 * @private
	 */
	setParentView : function(parentView)
	{
		this.parentView = parentView;
	},

	/**
	 * Adds a child view to this view.
	 * @param {Zarafa.core.ui.View} child child view to be added 
	 */
	addChildView : function(child)
	{
		child.setParentView(this);
		this.children.push(child);
		return child;
	},
	
	/**
	 * Removes a child view from this view. 
	 * @param {Zarafa.core.ui.View} child child view to be added
	 * @param {Boolean} destroy if true, destroy the child view 
	 */
	removeChildView : function(child, destroy)
	{
		this.children.remove(child);
		child.setParentView(undefined);
		if (destroy)
			child.destroy();
	},

	/**
	 * Purge all event listeners.
	 * @private
	 */
	purgeListeners : Ext.Container.prototype.purgeListeners,

	/**
	 * Clear all monitored event listeners. This will call
	 * {@link Ext.util.Observable.un un()} on each previously
	 * registered event handler which was registered with
	 * {@link #mon}.
	 * @private
	 */
	clearMons : Ext.Container.prototype.clearMons,

	/**
	 * Create the tracking object in which we store
	 * each event handler which was registered with {@link #mon}.
	 * @private
	 */
	createMons : Ext.Container.prototype.createMons,

	/**
	 * Adds listeners to any Observable object (or Elements) which are automatically removed when this View
	 * is destroyed.
	 * @param {Observable|Element} item The item to which to add a listener/listeners.
	 * @param {Object|String} ename The event name, or an object containing event name properties.
	 * @param {Function} fn Optional. If the <code>ename</code> parameter was an event name, this
	 * is the handler function.
	 * @param {Object} scope Optional. If the <code>ename</code> parameter was an event name, this
	 * is the scope (<code>this</code> reference) in which the handler function is executed.
	 * @param {Object} opt Optional. If the <code>ename</code> parameter was an event name, this
	 * is the {@link Ext.util.Observable#addListener addListener} options.
	 */
	mon : Ext.Container.prototype.mon,

	/**
	 * Removes listeners that were added by the {@link #mon} method.
	 * @param {Observable|Element} item The item from which to remove a listener/listeners.
	 * @param {Object|String} ename The event name, or an object containing event name properties.
	 * @param {Function} fn Optional. If the <code>ename</code> parameter was an event name, this
	 * is the handler function.
	 * @param {Object} scope Optional. If the <code>ename</code> parameter was an event name, this
	 * is the scope (<code>this</code> reference) in which the handler function is executed.
	 */
	mun : Ext.Container.prototype.mun,

	/**
	 * Called when the View is being destroyed.
	 * @protected
	 */
	onDestroy : Ext.emptyFn
});
Ext.namespace('Zarafa.core.ui');

/**
 * @class Zarafa.core.ui.WelcomeViewport
 * @extends Ext.Viewport
 * The viewport to be used as welcome page for first time users, this will show
 * a welcome message, and allow the user to configure to initial settings
 * before continuing to the {@link Zarafa.core.ui.MainViewport Main viewport}.
 */
Zarafa.core.ui.WelcomeViewport = Ext.extend(Ext.Viewport, {

	/**
	 * The reference as returned by {@link Zarafa.core.ui.notifier.Notifier#notify} to reference the
	 * message in order to remove the message as soon as the save was completed.
	 * @property
	 * @type Ext.Element
	 * @private
	 */
	savingEl : undefined,

	/**
	 * @constructor
	 * @param {Object} config Configuration object
	 */
	constructor : function(config)
	{
		config = config || {};

		config = Ext.applyIf(config, {
			layout : {
				type : 'vbox',
				align : 'stretch'
			},
			items : [{
				xtype : 'container',
				flex : 0.5
			},{
				layout : {
					type : 'hbox'
				},
				xtype : 'container',
				items : [{
					xtype : 'container',
					flex : 0.5
				},{
					xtype : 'panel',
					cls : 'zarafa-welcome-body',
					border : false,
					style : { width : '45em' },
					items : [{
						xtype : 'displayfield',
						cls : 'zarafa-welcome-title',
						value : _('Welcome to Zarafa WebApp')
					},{
						xtype : 'displayfield',
						cls : 'zarafa-welcome-message',
						value : _('This is the first time you are using WebApp. <br />Please check the following settings before continuing.')
					},{
						xtype : 'zarafa.settingswelcomecategory',
						ref : '../../settingsCategory'
					}],
					buttonAlign : 'center',
					buttons : [{
						text: _('Continue'),
						handler : this.onContinueButton,
						scope : this
					}]
				},{
					xtype : 'container',
					flex : 0.5
				}]
			},{
				xtype : 'container',
				flex : 0.5
			}]
		});

		Zarafa.core.ui.WelcomeViewport.superclass.constructor.call(this, config);

		this.settingsCategory.update(container.getSettingsModel());

		// Disable autoSave, we want to call the save function manually,
		// so we can supply a callback function.
		container.getSettingsModel().autoSave = false;
	},

	/**
	 * Event handler which is fired when the user clicks the 'Continue' button
	 * This will save all settings, and reload the page to continue to the
	 * {@link Zarafa.core.ui.MainViewport}.
	 * @private
	 */
	onContinueButton : function()
	{
		var model = container.getSettingsModel();

		model.beginEdit();
		// Load settings from UI
		this.settingsCategory.updateSettings(model);
		// Disable the welcome message for next logon
		model.set('zarafa/v1/main/show_welcome', false);
		model.endEdit();

		// Register event listener, so we can redirect the user
		// once the save has completed.
		this.mon(model, 'save', this.onSettingsSave, this, { single : true });
		this.mon(model, 'exception', this.onSettingsException, this, { single : true });

		// Show an information box indicating that the settings are being saved.
		this.savingEl = container.getNotifier().notify('info.saving', '', _('Saving') + '...', {
			container : this.getEl(),
			persistent : true
		});

		// Save settings
		model.save();
	},

	/**
	 * Called when the {@link Zarafa.settings.SettingsModel} fires the {@link Zarafa.settings.SettingsModel#save save}
	 * event to indicate the settings were successfully saved.
	 * @param {Zarafa.settings.SettingsModel} model The model which fired the event.
	 * @param {Object} parameters The key-value object containing the action and the corresponding
	 * settings which were saved to the server.
	 * @private
	 */
	onSettingsSave : function(model, parameters)
	{
		container.getNotifier().notify('info.saving', null, null, {
			container : this.getEl(),
			destroy : true,
			reference : this.savingEl
		});

		Zarafa.core.Util.disableLeaveRequester();
		window.location.reload();
	},

	/**
	 * Called when the {@link Zarafa.settings.SettingsModel} fires the {@link Zarafa.settings.SettingsModel#exception exception}
	 * event to indicate the settings were not successfully saved.
	 * @private
	 */
	onSettingsException : function()
	{
		container.getNotifier().notify('info.saving', null, null, {
			container : this.getEl(),
			destroy : true,
			reference : this.savingEl
		});
	}
});
Ext.namespace('Zarafa.core.ui.menu');

/**
 * @class Zarafa.core.ui.menu.ConditionalItem
 * @extends Ext.menu.Item
 * @xtype zarafa.conditionalitem
 *
 * Extends the {@link Ext.menu.Item} class and will check if the
 * item should be enabled or disabled each time before the
 * context menu is being shown.
 */
Zarafa.core.ui.menu.ConditionalItem = Ext.extend(Ext.menu.Item, {
	/**
	 * @cfg {Boolean} hideOnDisabled This item must be hidden rather
	 * then be marked disabled.
	 */
	hideOnDisabled : true,
	/**
	 * @cfg {Boolean} emptySelectOnly This item must only be enabled
	 * if no record is selected
	 */
	emptySelectOnly : false,
	/**
	 * @cfg {Boolean} nonEmptySelectOnly This item must only be enabled
	 * if one or more records are selected
	 */
	nonEmptySelectOnly : false,
	/**
	 * @cfg {Boolean} singleSelectOnly This item must only be enabled
	 * if a single record is selected
	 */
	singleSelectOnly : false,
	/**
	 * @cfg {Boolean} multiSelectOnly This item must only be enabled
	 * if multiple records are selected.
	 */
	multiSelectOnly : false,
	/**
	 * Override of {@link Ext.menu.Item#itemTpl} to add the possibility of
	 * styling the icon.
	 * @property
	 * @Type Ext.XTemplate
	 */
    itemTpl : new Ext.XTemplate(
        '<a id="{id}" class="{cls} x-unselectable" hidefocus="true" unselectable="on" href="{href}"',
            '<tpl if="hrefTarget">',
                ' target="{hrefTarget}"',
            '</tpl>',
         '>',
             '<img alt="{altText}" src="{icon}" class="x-menu-item-icon {iconCls}" {iconStyle}/>',
             '<span class="x-menu-item-text">{text}</span>',
         '</a>'
    ),
    /**
     * Override of {@link Ext.menu.Item#getTemplateArgs} to add the possibility of
     *  styling the icon.
     * @return {Object}
     */
    getTemplateArgs: function() {
    	// Get the original template arguments from the original function
    	var templateArgs = Zarafa.core.ui.menu.ConditionalItem.superclass.getTemplateArgs.call(this);
    	// Add the argument for the icon style
    	templateArgs.iconStyle = this.iconBG ? 'style="background-color:'+this.iconBG+';"' : '';
		return templateArgs;
	},
	/**
	 * @constructor
	 * @param {Object} config Configuration object
	 */
	constructor : function(config)
	{
		Ext.applyIf(config, {
			xtype : 'zarafa.conditionalitem'
		});

		Ext.applyIf(this, config);

		Zarafa.core.ui.menu.ConditionalItem.superclass.constructor.call(this, config);
	},

	/**
	 * Initialize the component
	 * @private
	 */
	initComponent : function()
	{
		Zarafa.core.ui.menu.ConditionalItem.superclass.initComponent.apply(this, arguments);

		// Reset the enable/disable functions to their
		// show/hide counterparts if we should not display the item as disabled.
		if (this.hideOnDisabled) {
			this.enable = this.show;
			this.disable = this.hide;
		}
	},

	/**
	 * Apply the {@link #emptySelectOnly}, {@link #nonEmptySelectOnly}, {@link #singleSelectOnly}
	 * and {@link #multiSelectOnly} filters to determine if the item must be {@link #setDisabled disabled}
	 * or not.
	 * @private
	 */
	applySelectionFilter : function()
	{
		var records = this.getRecords();

		if (this.emptySelectOnly) {
			if (Ext.isDefined(records) && (!Ext.isArray(records) || records.length > 0))
				this.setDisabled(true);
			else
				this.setDisabled(false);
		}

		if (this.nonEmptySelectOnly) {
			if (Ext.isDefined(records) && Ext.isArray(records) && records.length == 0)
				this.setDisabled(true);
			else
				this.setDisabled(false);
		}

		if (this.singleSelectOnly) {
			if (!Ext.isDefined(records) || (Ext.isArray(records) && records.length != 1))
				this.setDisabled(true);
			else
				this.setDisabled(false);
		}

		if (this.multiSelectOnly) {
			if (!Ext.isDefined(records) || !Ext.isArray(records) || records.length == 1)
				this.setDisabled(true);
			else
				this.setDisabled(false);
		}
	},

	/**
	 * Called by the {@link #parentMenu} when it is about to be shown, this can be
	 * overridden by subclasses to add extra filters on the visibility of this item.
	 * @param {Zarafa.core.ui.menu.ConditionalItem} item The item which is about to be shown.
	 * @param {Zarafa.core.data.MAPIRecord} record The record which is being shown for this menu
	 */
	beforeShow : function(item, record)
	{
		item.applySelectionFilter();
	},

	/**
	 * Obtain the record/records which are attached to the {@link Zarafa.core.ui.menu.ConditionalMenu menu}
	 * to which this item belongs.
	 * @return {Zarafa.core.data.MAPIRecord|Array} records
	 */
	getRecords : function()
	{
		return this.getRootMenu().records;
	},

	/**
	 * Obtain the reference to the root menu.
	 * This will go through all parents of the {@link Zarafa.core.ui.ConditionalItem item}
	 * until no {@link Ext.menu.menu} is found as parent. The last parent is considered the
	 * root and will be returned to the caller.
	 *
	 * @return {Ext.menu.menu} The Root menu object
	 */
	getRootMenu : function()
	{
		var menu = this.parentMenu;
		if (!menu && this.ownerCt instanceof Ext.menu.Menu) {
			menu = this.ownerCt;
		}

		while (menu && (Ext.isDefined(menu.parentMenu) || menu.ownerCt instanceof Ext.menu.Menu)) {
			menu = menu.parentMenu || menu.ownerCt;
		}

		return menu;
	}
});

Ext.reg('zarafa.conditionalitem', Zarafa.core.ui.menu.ConditionalItem);
Ext.namespace('Zarafa.core.ui.menu');

/**
 * @class Zarafa.core.ui.menu.ConditionalMenu
 * @extends Ext.menu.Menu
 * @xtype zarafa.conditionalmenu
 *
 * Extends the {@link Ext.menu.Menu} class and allows menu options to determine whether to display themselfs.
 */
Zarafa.core.ui.menu.ConditionalMenu = Ext.extend(Ext.menu.Menu, {
	/**
	 * @cfg {Zarafa.core.data.IPMRecord|Array} records The records on which this contextmenu was requested
	 */
	records : undefined,

	/**
	 * @constructor
	 * @param {Object} config Configuration object
	 */
	constructor : function(config)
	{
		config = config || {};

		Ext.applyIf(config, {
			xtype : 'zarafa.conditionalmenu'
		});

		Zarafa.core.ui.menu.ConditionalMenu.superclass.constructor.call(this, config);

		this.on('beforeshow', this.onMenuBeforeShow, this);
		this.on('hide', this.onMenuHide, this);
	},

	/**
	 * Event handler for the {@link #beforeshow} event. This will go through all
	 * {@link Zarafa.core.ui.menu.ConditionalItem items} in the menu and call the
	 * {@link Zarafa.core.ui.menu.ConditionalItem#beforeShow} function.
	 *
	 * Optionally all duplicate menuseparators will be removed. This can happen when an
	 * item which was surrounded by separators has been hidden.
	 *
	 * @param {Zarafa.core.ui.menu.ConditionalMenu} menu The menu which is being opened.
	 * @private
	 */
	onMenuBeforeShow : function(menu)
	{
		var records = this.records;
		var allowSeparator = false;
		var lastItemIndex = -1;

		// move over the items list and call 'beforeOpen' on each item if that function exists
		menu.items.each(function(item, index) {
			if (item.isXType('zarafa.conditionalitem')) {
				if (Ext.isFunction(item.beforeShow)) {
					item.beforeShow.call(item.scope || item, item, records);
				}

				if (Ext.isDefined(item.menu)) {
					this.onMenuBeforeShow(item.menu);
				}
			}

			// A separator is being added, check if we actually want to display it.
			if (item.isXType('menuseparator')) {
				item.setVisible(allowSeparator === true);
				allowSeparator = false;
			} else {
				// If the non-seperator item is visible,
				// we are allowed to display a separator when requested.
				if (item.hidden === false) {
					allowSeparator = true;
					lastItemIndex = index;
				}
			}
		}, this);

		// The menu is empty, or we have hidden everything.
		if (lastItemIndex === -1) {
			return false;
		}

		// Remove all separators which are visible as last items in the menu.
		for (var i = lastItemIndex, len = menu.items.getCount(); i < len; i++) {
			var item = menu.items.items[i];
			if (item.isXType('menuseparator')) {
				item.setVisible(false);
			}
		}

		// But what if we just removed everything?
		if (lastItemIndex === -1) {
			return false;
		}
	},
	
	/**
	 * Event handler for the {@link #hide} event. It will destroy the menu when it does not have an owner component.
	 * Since the UIFactory creates a new menu on each contextclick, we must make sure the menu's are also destroyed
	 * again when they are hidden, or else we will pollute the dom with abandoned menus.
	 * Child menus will be destroyed when their parent is destroyed.
	 *
	 * @param {Zarafa.core.ui.menu.ConditionalMenu} menu The menu which is being hidden.
	 * @private
	 */
	onMenuHide : function(menu)
	{
		// Check if the menu has an owner component (only context menu's don't have it)
		if ( !Ext.isDefined(menu.ownerCt) && !Ext.isDefined(menu.parentMenu)){
			menu.destroy();
		}
	}
});

Ext.reg('zarafa.conditionalmenu', Zarafa.core.ui.menu.ConditionalMenu);
Ext.namespace('Zarafa.core.ui.widget');

/**
 * @class Zarafa.core.ui.widget.Widget
 * @extends Ext.ux.Portlet
 * @xtype zarafa.widget
 *
 * A 'widget' is a plug-in that users can instantiate and put on their today view or tool bar.
 * Examples of widgets are the clock, the weather widget, public notes, etc. 
 * 
 * Users can add new instances of a widget freely, and create multiple instances if desired. Each
 * widget instance has a unique GUID, and settings for that instance are stored in the widgets/[GUID]
 * settings folder. When a widget is destroyed, that folder is deleted.
 */
Zarafa.core.ui.widget.Widget = Ext.extend(Ext.ux.Portlet, {
	/**
	 * @cfg {Zarafa.core.WidgetInfo} info The Widget Meta Data object
	 * which was used to {@link Zarafa.core.Container#registerWidget register}
	 * this Widget to the {@link Zarafa.core.Container container}.
	 */
	info : undefined,

	/**
	 * @cfg {String} guid The unique identifier used for reference to this particular
	 * widget in side a {@link Zarafa.core.ui.widget.WidgetPanel}. This is used for
	 * {@link #set setting} and {@link #get getting} {@link Zarafa.settings.SettingsModel settings}.
	 */
	guid : undefined,

	/**
	 * @cfg {Boolean} hasConfig True if the {@link #config} function
	 * has been implemented and a 'gear' icon should be shown besides
	 * the close button.
	 */
	hasConfig : false,

	/**
	 * @cfg {String} about The about text. If provided, {@link #registerAboutText}
	 * will be automatically called during {@link #initWidget initialization}.
	 */
	about : undefined,

	/**
	 * The widget panel on which this widget is located
	 * @property
	 * @type Zarafa.core.ui.widget.WidgetPanel
	 */
	widgetPanel : undefined,

	/**
	 * @constructor
	 * @param {Object} config Configuration object
	 */
	constructor : function(config)
	{
		config = config || {};

		var tools = config.tools || [];

		// Check if the configuration function has been
		// implemented. If it is the case, create the config tool.
		if (config.hasConfig === true) {
			tools.push({
				id : 'gear',
				handler: this.config,
				scope : this
			});
		}

		// Always add the close tool.
		tools.push({
			id : 'close',
			scope : this,
			handler: this.close
		});

		Ext.applyIf(config, {
			title : config.info.getDisplayName(),
			anchor : '100%',
			frame : true,
			collapsible : true,
			draggable : {
				ddGroup : 'dd.widget'
			},
			tools : tools
		});

		Zarafa.core.ui.widget.Widget.superclass.constructor.call(this, config);

		this.initWidget();
	},

	/**
	 * Function to be implemented by Widget subclasses to initialize the widget.
	 * @protected
	 */
	initWidget : function()
	{
		// If the about text is provided, automatically register it
		if (!Ext.isEmpty(this.about)) {
			this.registerAboutText(this.title, this.about);
		}
	},

	/**
	 * Add a About/Copyright notice to the Widget.
	 * @param {String} title The title for the About widget
	 * @param {String} text The text which should be shown in the About text (may contain HTML)
	 * @protected
	 */
	registerAboutText : function(title, text)
	{
		this.tools.splice(this.tools.length - 1, 0, {
			id : 'help',
			handler : this.showAbout,
			scope : this,
			title : title,
			text : text
		});
	},

	/**
	 * Checks if this widget is currently visible. This returns true when
	 * the widget itself is {@link #isVisible visible} and the {@link #widgetPanel}
	 * is currently not {@link Ext.Panel#collapsed}.
	 */
	isWidgetVisible : function()
	{
		return this.isVisible() && !this.widgetPanel.collapsed;
	},

	/**
	 * Called when the widget has been rendered.
	 * This will initialize the {@link #widgetPanel}. 
	 */
	onRender : function()
	{
		Zarafa.core.ui.widget.Widget.superclass.onRender.apply(this, arguments);
		this.widgetPanel = this.findParentByType('zarafa.widgetpanel');
	},

	/**
	 * Get a settings property..
	 * @param {String} key settings path. This is path relative to where the widget's settings are stored.
	 * @return {String} value. 
	 */
	get : function(key)
	{
		return container.getSettingsModel().get(Zarafa.core.ui.widget.Widget.settingsPath(this.guid, key));
	},

	/**
	 * Set a settings property.
	 * @param {String} key settings path. This is path relative to where the widget's settings are stored.
	 * @param {String} value value. 
	 */
	set : function(key, value)
	{
		container.getSettingsModel().set(Zarafa.core.ui.widget.Widget.settingsPath(this.guid, key), value);
	},

	/**
	 * Closes and destroys the widget, removing its settings from the settings tree.
	 */
	close : function(e, target, panel)
	{
		this.widgetPanel.destroyWidget(this);
	},

	/**
	 * Called when a user clicks the config button on the widget panel.
	 * Should be overridden by child classes.
	 * @protected 
	 */
	config : Ext.emptyFn,

	/**
	 * Called when a user clicks the about button on the widget panel.
	 * This will show a {@link Ext.Window} containing the {@link #about} text.
	 * @protected
	 */
	showAbout : function(event, toolEl, panel, tc)
	{
		var win = new Ext.Window({
			title: tc.title,
			width: 320,
			height: 200,
			padding : 5,
			autoScroll: true,
			items : [{
				xtype : 'panel',
				layout : 'form',
				border : false,
				items :  [{
					xtype : 'displayfield',
					value : tc.text,
					hideLabel : true,
					htmlEncode : false
				}]
			}]
		});

		win.show(this);
	}
});

Ext.reg('zarafa.widget', Zarafa.core.ui.widget.Widget);

/**
 * @param {String} guid The unique identifier for a {@link Zarafa.core.ui.widget.Widget widget}.
 * @param {String} key The setting key which should be updated
 * @return {String} The settings path for the guid & key combination.
 * @static
 */
Zarafa.core.ui.widget.Widget.settingsPath = function(guid, key)
{
	return ('zarafa/v1/widgets/' + guid + '/' + key).replace(/(\/)+/,'/');
};
Ext.namespace('Zarafa.common');

/**
 * @class Zarafa.common.Actions
 * Common actions which can be used within {@link Ext.Button buttons}
 * or other {@link Ext.Component components} with action handlers.
 * @singleton
 */
Zarafa.common.Actions = {
	/**
	 * The internal 'iframe' which is hidden from the user, which is used for downloading
	 * attachments. See {@link #doOpen}.
	 * @property
	 * @type Ext.Element
	 */
	downloadFrame : undefined,	
	/**
	 * Open a {@link Zarafa.common.dialogs.CopyMoveContentPanel CopyMoveContentPanel} for
	 * copying or moving {@link Zarafa.core.data.IPMRecord records} to the
	 * preferred destination folder.
	 *
	 * @param {Zarafa.core.data.IPMRecord} records The record which must be copied or moved.
	 * @param {Object} config (optional) Configuration object to create the ContentPanel
	 */
	openCopyMoveContent : function(records, config)
	{
		config = Ext.applyIf(config || {}, {
			modal : true
		});
		var componentType = Zarafa.core.data.SharedComponentType['common.dialog.copymoverecords'];
		Zarafa.core.data.UIFactory.openLayerComponent(componentType, records, config);
	},

	/**
	 * Opens a {@link Zarafa.common.recurrence.dialogs.RecurrenceContentPanel RecurrenceContentPanel} for configuring
	 * the recurrence of the given {@link Zarafa.core.data.IPMRecord record}.
	 *
	 * @param {Zarafa.core.data.IPMRecord} records The record for which the recurrence must be configured.
	 * @param {Object} config Configuration object 
	 */
	openRecurrenceContent : function(records, config)
	{
		if (Ext.isArray(records) && !Ext.isEmpty(records)) {
			records = records[0];
		}

		config = Ext.applyIf(config || {}, {
			autoSave : true,
			modal : true
		});

		var componentType = Zarafa.core.data.SharedComponentType['common.dialog.recurrence'];
		Zarafa.core.data.UIFactory.openLayerComponent(componentType, records, config);
	},

	/**
	 * Opens a {@link Zarafa.common.categories.dialogs.CategoriesContentPanel CategoriesContentPanel} for configuring
	 * the categories of the given {@link Zarafa.core.data.IPMRecord records}.
	 *
	 * @param {Zarafa.core.data.IPMRecord} records The record, or records for which the categories
	 * must be configured
	 * @param {Object} config (optional) Configuration object for creating the ContentPanel
	 */
	openCategoriesContent : function(records, config)
	{
		if (!Ext.isArray(records)) {
			records = [ records ];
		}

		config = Ext.applyIf(config || {}, {
			autoSave : true,
			modal : true
		});
		var componentType = Zarafa.core.data.SharedComponentType['common.dialog.categories'];
		Zarafa.core.data.UIFactory.openLayerComponent(componentType, records, config);
	},

	/**
	 * Opens a {@link Zarafa.common.attachment.dialogs.LegacyFileSelectionContentPanel File Selection Content Panel}
	 * which is needed for Browsers which do not support the {@link Zarafa#supportsFilesAPI Files API} and need
	 * a valid {@link Ext.form.FormPanel form} with a {@link Ext.ux.form.FileUploadField Upload field} in order
	 * to be able to upload the files to the server.
	 *
	 * @param {Object} config (optional) Configuration object for creating the ContentPanel
	 */
	openLegacyFileSelectionContent : function(config)
	{
		config = Ext.applyIf(config || {}, {
			modal : true
		});
		
		var componentType = Zarafa.core.data.SharedComponentType['common.attachment.dialog.legacyfileselection'];
		Zarafa.core.data.UIFactory.openLayerComponent(componentType, undefined, config);
	},

	/**
	 * Opens a {@link Zarafa.common.attachment.dialogs.AttachItemContentPanel Attach Item Content Panel} which is used
	 * to attach an item as embedded attachment to a message.
	 *
	 * @param {Zarafa.core.data.MAPIRecord} record record that will be used to add embedded attachment
	 * @param {Object} config (optional) Configuration object for creating the ContentPanel
	 */
	openAttachItemSelectionContent : function(record, config)
	{
		config = Ext.applyIf(config || {}, {
			modal : true
		});
		
		var componentType = Zarafa.core.data.SharedComponentType['common.attachment.dialog.attachitem'];
		Zarafa.core.data.UIFactory.openLayerComponent(componentType, record, config);
	},

	/**
	 * Opens a {@link Zarafa.core.ui.widget.WidgetContentPanel}
	 * for inserting widgets into the {@link Zarafa.core.ui.widget.WidgetPanel}
	 * @param {Object} config (optional) Configuration object for creating the ContentPanel
	 */
	openWidgetsContent : function(config)
	{
		var componentType = Zarafa.core.data.SharedComponentType['common.dialog.widgets'];
		Zarafa.core.data.UIFactory.openLayerComponent(componentType, undefined, config);
	},
	
	/**
	 * Will open the View ContentPanel for a recipient, before opening the recipient it will
	 * first check the exact type of the recipient to see if it is an AddressBook item
	 * or personal contact. If either of those two the record is converted to assure the
	 * correct panel is opened.
	 * @param {Zarafa.core.data.IPMRecipientRecord} recipient The recipient which must be opened
	 * @param {Object} config configuration object.
	 */
	openViewRecipientContent : function(recipient, config)
	{
		if (recipient.isResolved()) {
			if (recipient.isPersonalContact()) {
				// A personal contact needs to be converted to a contact so the correct panel can be shown.
				recipient = recipient.convertToContactRecord();
				// FIXME: We put the abRecord into the ShadowStore to be able
				// to open it, and obtain all details. However, we also need to
				// find a point where we can remove it again.
				container.getShadowStore().add(recipient);
			} else if (recipient.isPersonalDistList()) {
				// A personal distlist needs to be converted to a distlist so the correct dialog can be shown.
				recipient = recipient.convertToDistListRecord();
				// FIXME: We put the abRecord into the ShadowStore to be able
				// to open it, and obtain all details. However, we also need to
				// find a point where we can remove it again.
				container.getShadowStore().add(recipient);
			} else if (!recipient.isOneOff()) {
				// A addressbook item needs to be converted to a AddressBook record so the correct dialog is shown.
				recipient = recipient.convertToABRecord();
				// FIXME: We put the abRecord into the ShadowStore to be able
				// to open it, and obtain all details. However, we also need to
				// find a point where we can remove it again.
				container.getShadowStore().add(recipient);
			}
			
			config = Ext.applyIf(config || {}, { manager : Ext.WindowMgr });
			Zarafa.core.data.UIFactory.openViewRecord(recipient, config);
		}
	},

	/**
	 * Opens a {@link Zarafa.common.delegates.dialogs.DelegatePermissionContentPanel DelegatePermissionContentPanel} for editing
	 * delegate permissions of a single delegate
	 * @param {Zarafa.common.delegates.data.DelegateRecord} delegateRecord record that should be opened in
	 * {@link Zarafa.common.delegates.dialogs.DelegatePermissionContentPanel DelegatePermissionContentPanel}.
	 * @param {Object} config configuration object that should be passed to {@link Zarafa.common.delegates.dialogs.DelegatePermissionContentPanel DelegatePermissionContentPanel}.
	 */
	openDelegatePermissionContent : function(record, config)
	{
		if(!record) {
			// can not continue without a record
			return;
		}

		config = config || {};
		Ext.apply(config, {
			modal : true
		});

		Zarafa.core.data.UIFactory.openCreateRecord(record, config);
	},

 	/**
	 * Opens a {@link @link Zarafa.common.sendas.dialogs.SendAsEditContentPanel SendAsEditContentPanel} for editing
	 * user name and email address of a sendAs.
	 * @param {Ext.data.Record} record record that should be opened in
	 * {@link Zarafa.common.sendas.dialogs.SendAsEditContentPanel SendAsEditContentPanel}.
	 * @param {Object} config configuration object that should be passed to {@link Zarafa.common.sendas.dialogs.SendAsEditContentPanel SendAsEditContentPanel}.
	 */
	openSendAsRecipientContent : function(record, config)
	{
		if(!record) {
			// can not continue without a record
			return;
		}
		var componentType = Zarafa.core.data.SharedComponentType['common.sendas.dialog.sendaseditcontentpanel'];

		config = config || {};
		Ext.apply(config, {
			modal : true
		});

		Zarafa.core.data.UIFactory.openLayerComponent(componentType, record, config);
	},

	/**
	 * Opens a {@link Zarafa.common.rules.dialogs.RulesEditContentPanel RulesEditContentPanel} for editing
	 * a single {@link Zarafa.common.rules.data.RulesRecord RulesRecord}.
	 * @param {Zarafa.common.rules.data.RulesRecord} record record to edit in
	 * {@link Zarafa.common.rules.dialogs.RulesEditContentPanel RulesEditContentPanel}.
	 * @param {Object} config config object that will be passed to {@link Zarafa.core.data.UIFactoryLayer UIFactoryLayer}.
	 */
	openRulesEditContent : function(record, config)
	{
		if(!record) {
			// can not continue without a record
			return;
		}

		config = Ext.apply(config || {}, {
			modal : true
		})

		Zarafa.core.data.UIFactory.openCreateRecord(record, config);
	},

	/**
	 * Will create an object of {@link Zarafa.common.attachment.ui.AttachmentDownloader AttachmentDownloader}
	 * and call {@link Zarafa.common.attachment.ui.AttachmentDownloader#downloadMessage} method to start download the message as file
	 * by setting the dialogFrame's location to the download URL of the given {@link Zarafa.core.data.IPMRecord records}.
	 * @param {Zarafa.core.data.IPMRecord} records The record, or records which user want to save as file.
	 * @param {Boolean} allAsZip (optional) True to downloading all the attachments as ZIP
	 */
	openSaveEmlDialog : function(records, allAsZip)
	{
		records = [].concat(records);

		if(!allAsZip){
			for (var i = 0; i < records.length; i++) {
				var record = records[i];
				// Create separate iframe for each url to handle requests individually
				var downloadComponent = new Zarafa.common.attachment.ui.AttachmentDownloader();
				downloadComponent.downloadItem(record.getDownloadMessageUrl(false));
			}
		} else {
			var downloadComponent = new Zarafa.common.attachment.ui.AttachmentDownloader();
			downloadComponent.downloadMessageAsZip(records);
		}
	},

	/**
	 * Opens a PrintDialog for printing the contents of the given {@link Zarafa.core.data.IPMRecord records}.
	 *
	 * @param {Zarafa.core.data.IPMRecord} records The record, or records for which the print will be displayed.
	 * @param {Object} config (optional) Configuration object
	 */
	openPrintDialog: function(records, config)
	{
		if (Ext.isEmpty(records)) {
			return;
		} else if (Ext.isArray(records)) {
			if (records.length > 1) {
				Ext.MessageBox.alert(_('Print'), _('Printing of multiple items has not been implemented.'));
				return;
			} else {
				// We only need the first record
				records = records[0];
			}
		}

		var openHandler = function (store, record) {
			if (store) {
				if (this !== record) {
					return;
				}
				store.un('open', openHandler, record);
			}

			var componentType = Zarafa.core.data.SharedComponentType['common.printer.renderer'];
			var component = container.getSharedComponent(componentType, record);
			if (component) {
				var renderer = new component();
				renderer.print(record);
			} else  {
				if (record instanceof Zarafa.core.data.MAPIRecord) {
					Ext.MessageBox.alert(_('Print'), _('Printing of this item is not yet available') + '\n' + _('Item type: ') + record.get('message_class'));
				} else {
					Ext.MessageBox.alert(_('Print'), _('Printing of this view is not yet available'));
				}
			}
		};

		if (records instanceof Zarafa.core.data.MAPIRecord && !records.isOpened()) {
			records.getStore().on('open', openHandler, records);
			records.open();
		} else {
			openHandler(undefined, records);
		}
	},

	/**
	 * Opens a {@link Zarafa.common.checknames.dialogs.CheckNamesContentPanel CheckNamesContentPanel}
	 *
	 * @param {Array} array of checkNames
	 * @param {Zarafa.core.data.IPMRecipientRecord} recipientrecord
	 * @param {Object} config (optional) Configuration object for creating the content panel
	 */
	openCheckNamesContent : function(checkNamesData, recipientRecord, config)
	{
		var componentType = Zarafa.core.data.SharedComponentType['common.dialog.checknames'];
		config = Ext.applyIf(config || {}, {
			checkNamesData : checkNamesData,
			modal: true
		});
		Zarafa.core.data.UIFactory.openLayerComponent(componentType, recipientRecord, config);
	},

	/**
	 * Opens a {@link Zarafa.common.reminder.dialogs.ReminderContentPanel remindercontentpanel}
	 * @param {Zarafa.common.reminder.ReminderRecord} records Records for which the reminder content panel will be displayed.
	 * @param {Object} config (optional) Configuration object
	 */
	openReminderContent : function(records, config)
	{
		var componentType = Zarafa.core.data.SharedComponentType['common.dialog.reminder'];
		var component = container.getSharedComponent(componentType, records);

		config = Ext.applyIf(config || {}, {
			modal : false,
			manager : Ext.WindowMgr
		});
		
		// check if panel is already open
		var contentPanelInstances = Zarafa.core.data.ContentPanelMgr.getContentPanelInstances(component);

		// there can be no reminder dialog or only one reminder dialog
		// multiple reminder dialogs are not allowed
		if(contentPanelInstances.getCount() === 0) {
			// create a new reminder dialog, if there are any reminders to show
			if(records.length > 0) {
				Zarafa.core.data.UIFactory.openLayerComponent(componentType, records, config);
			}
		} else if (contentPanelInstances.getCount() === 1) {
			// we already have a reminder dialog open, use it
			var reminderDialog = contentPanelInstances.first();

			if(records.length > 0) {
				// there are reminders to show, so give focus to existing reminder dialog
				reminderDialog.focus();
			} else {
				// no reminders to show, close existing dialog
				reminderDialog.close();
			}
		}
	},

	/**
	 * Function will first convert the {@link Zarafa.common.reminder.ReminderRecord ReminderRecord} to an
	 * {@link Zarafa.core.data.IPMRecord IPMRecord} based on its message_class property and then pass that
	 * {@link Zarafa.core.data.IPMRecord IPMRecord} to {@link Zarafa.core.ui.ContentPanel ContentPanel} to open it.
	 * @param {Zarafa.common.reminder.data.ReminderRecord|Zarafa.common.reminder.data.ReminderRecord|Array} record
	 * The reminder record/records which should be opened.
	 * @param {Object} config configuration object.
	 */
	openReminderRecord: function(record, config)
	{
		config = config || {};

		if(Ext.isArray(record)){
			Ext.each(record, this.openReminderRecord, this);
			return;
		}

		// convert reminder record to proper ipmrecord
		record = record.convertToIPMRecord();

		// we will always open the record into a dialog because reminders are also displayed in a dialog
		Ext.applyIf(config, {
			manager : Ext.WindowMgr
		});

		if(record) {
			Zarafa.core.data.UIFactory.openViewRecord(record, config);
		}
	},

	/**
	 * Opens a {@link Zarafa.common.dialogs.MessageBox.select MessageBox} for
	 * selecting if either a recurrence or the entire series must be opened for the Recurring
	 * Message.
	 *
	 * @param {Function} handler The handler which is invoked with the selected value
	 * from the dialog. This function only takes one argument and is either 'recurrence_occurence'
	 * when the single-occurence was selected or 'recurrence_series' when the series was selected.
	 * @param {Object} scope (optional) The scope on which the handler must be invoked.
	 */
	// TODO: Merge with deleteRecurringSelectionContentPanel
	openRecurringSelectionContent : function(record, handler, scope)
	{
		var title = _('Recurring Message');
		var text =  _('This is a recurring message. Do you want to open only this occurrence or the series?');

		if (record.isMessageClass('IPM.Appointment', true)) {
			if (record.get('meeting') == Zarafa.core.mapi.MeetingStatus.NONMEETING) {
				title = _('Recurring Appointment');
				text =  _('This is a recurring appointment. Do you want to open only this occurrence or the series?');
			} else {
				title = _('Recurring Meeting Request');
				text =  _('This is a recurring meeting request. Do you want to open only this occurrence or the series?');
			}
		} else if (record.isMessageClass('IPM.TaskRequest', true)) {
			title = _('Recurring Task Request');
			text =  _('This is a recurring task request. Do you want to open only this occurrence or the series?');
		}

		Zarafa.common.dialogs.MessageBox.select(
			title, text, handler, scope, [{
				boxLabel: _('Open this occurrence'),
				id : 'recurrence_occurence',
				name: 'select',
				checked: true
			},{
				boxLabel: _('Open the series'),
				id : 'recurrence_series',
				name: 'select'
			}]
		);
	},

	/**
	 * Opens a {@link Zarafa.common.dialogs.MessageBox.select MessageBox} for
	 * selecting if either a recurrence or the entire series must be deleted.
	 *
	 * @param {Function} handler The handler which is invoked with the selected value
	 * from the dialog. This function only takes one argument and is either 'recurrence_occurence'
	 * when the single-occurence was selected or 'recurrence_series' when the series was selected.
	 * @param {Object} scope (optional) The scope on which the handler must be invoked.
	 */
	// TODO: Merge with openRecurringSelectionContentPanel
	deleteRecurringSelectionContent : function(record, handler, scope)
	{
		var title = _('Recurring Message');
		var text =  _('This is a recurring message. Do you want to delete only this occurrence or the series?');

		if (record.isMessageClass('IPM.Appointment', true)) {
			if (record.get('meeting') == Zarafa.core.mapi.MeetingStatus.NONMEETING) {
				title = _('Recurring Appointment');
				text =  _('This is a recurring appointment. Do you want to delete only this occurrence or the series?');
			} else {
				title = _('Recurring Meeting Request');
				text =  _('This is a recurring meeting request. Do you want to delete only this occurrence or the series?');
			}
		} else if (record.isMessageClass('IPM.TaskRequest', true)) {
			title = _('Recurring Task Request');
			text =  _('This is a recurring task request. Do you want to delete only this occurrence or the series?');
		}

		Zarafa.common.dialogs.MessageBox.select(
			title, text, handler, scope, [{
				boxLabel: _('Delete this occurrence'),
				id : 'recurrence_occurence',
				name: 'select',
				checked: true
			},{
				boxLabel: _('Delete the series'),
				id : 'recurrence_series',
				name: 'select'
			}]
		);
	},

	/**
	 * Opens a {@link Zarafa.common.dialogs.MessageBox.select MessageBox} for
	 * selecting if either a update need to be send to meeting Organizer or silently deleted items.
	 *
	 * @param {Function} handler The handler which is invoked with the selected value
	 * from the dialog. This function only takes one argument and is either 'sendResponseOnDelete'
	 * when the delete and response was selected or 'onResponseOnDelete' when the delete without response was selected.
	 * @param {Object} scope (optional) The scope on which the handler must be invoked.
	 */
	// TODO: may be Merge with deleteRecurringSelectionContentPanel
	deleteMeetingRequestConfirmationContent : function(record, handler, scope)
	{
		var title = _('Confirm Delete');
		var acceptedText = _('This "{0}" meeting was already accepted.');
		var noResponsedText = _('You have not responded to the meeting request "{0}".');
		
		if(record.get('responsestatus') == Zarafa.core.mapi.ResponseStatus.RESPONSE_NOT_RESPONDED){
			var text = String.format(noResponsedText, record.get('subject'));
		}else{
			var text = String.format(acceptedText, record.get('subject'));
		}

		Zarafa.common.dialogs.MessageBox.select(
			title, text, handler, scope, [{
				boxLabel: _('Delete and send a response to the meeting organizer'),
				id : 'sendResponseOnDelete',
				name: 'select',
				checked: true
			},{
				boxLabel: _('Delete without sending'),
				id : 'noResponseOnDelete',
				name: 'select'
			}]
		);
	},

	/**
	 * Deletes all {@link Zarafa.core.data.IPMRecord records} from the {@link Zarafa.core.data.IPMStore store}.
	 * If any of the given {@link Zarafa.core.data.IPMRecord records} is an recurring item, then
	 * an {@link Zarafa.common.dialogs.MessageBox.select MessageBox} will be prompted which lets the user
	 * select between the series or the single occurence.
	 * All given {@link Zarafa.core.data.IPMRecord records} must be located in the same
	 * {@link Zarafa.core.data.IPMStore store}.
	 *
	 * @param {Array} records The array of records which must be deleted.
	 * @param {Boolean} askOcc (private) False to prevent a dialog to appear to ask if the occurence or series must
	 * be deleted
	 *
	 * FIXME: This introduces Calendar-specific actions into the Common Context, but there is no clean solution
	 * for this at this time. But we need to split this up into context-specific actions while maintaining this
	 * single-entrypoint for deleting records.
	 */
	deleteRecords : function(records, askOcc)
	{
		var store = undefined;
		var saveRecords = [];

		if (Ext.isEmpty(records)) {
			return;
		}

		if (!Ext.isArray(records)) {
			records = [ records ];
		}

		for (var i = 0, len = records.length; i < len; i++) {
			var record = records[i];
			store = record.getStore();

			// Check if the item is recurring, and if we need to ask the user
			// if the occurence or series must be deleted
			var deleteRecurring = Ext.isFunction(record.isRecurringOccurence) && record.isRecurringOccurence() && askOcc !== false;

			// Meeting and task requests are always deleted as normal, 
			// we don't care for the recurring state of the record.
			var messageClass = record.get('message_class');
			if (Zarafa.core.MessageClass.isClass(messageClass, 'IPM.Schedule.Meeting', true) ||
				Zarafa.core.MessageClass.isClass(messageClass, 'IPM.TaskRequest', true)) {
					deleteRecurring = false;
			}

			if (deleteRecurring) {
				// Deleting an recurring series requires a confirmation dialog.
				this.deleteRecurringItem(record);
			} else if (Ext.isFunction(record.isMeeting) && record.isMeeting() && !record.isAppointmentInPast() && !record.isMeetingCanceled()) {
				// delete action on future meeting items
				if (record.isMeetingSent()) {
					// We are the organizer of the meeting, so lets ask if the recipients should be notified.
					Ext.MessageBox.show({
						title: _('Zarafa WebApp'),
						msg : _('A cancellation message will be sent to all recipients, do you wish to continue?'),
						icon: Ext.MessageBox.WARNING,
						fn: this.cancelInvitation,
						scope: record,
						buttons: Ext.MessageBox.YESNO
					});
				} else if (record.isMeetingResponseRequired()) {
					// We are the attendee of the meeting, lets ask if we should inform the organizer
					Zarafa.common.Actions.deleteMeetingRequestConfirmationContent(record, this.declineInvitation, record);
				} else {
					// We are neither, we don't care, just delete the thing
					store.remove(record);
					saveRecords.push(record);
				}
			} else {
				// normal delete action
				store.remove(record);
				saveRecords.push(record);
			}
		}

		if(!Ext.isEmpty(saveRecords)) {
			store.save(saveRecords);
		}

		// If number of delete records equal to total loaded records then show load mask until server send success response.
		if(store.totalLoadedRecord) {
			if (records.length === store.totalLoadedRecord) {
				store.showLoadMask();
			}
		}
	},

	/**
	 * Function which prompt user with deleting for recurring Meeting or normal recurring 
	 * appointment and also manages sending response to meeting organizer.
	 * 
	 * @param {Ext.data.Record} record that must be deleted
	 * @private
	 */
	deleteRecurringItem : function(record){
		Zarafa.common.Actions.deleteRecurringSelectionContent(record, function(button, radio) {
			if (button != 'ok') {
				return;
			}

			if (radio.id != 'recurrence_series') {
				record = record.convertToOccurenceRecord();
			} else {
				record = record.convertToSeriesRecord();
			}
			container.getShadowStore().add(record);

			Zarafa.common.Actions.deleteRecords(record, false);
		}, this);
	},

	/**
	 * Function cancels Meeting invitation and sends Meeting Cancellation message.
	 * 
	 * @param {String} buttonClicked The ID of the button pressed,
	 * @param {String} text Value of the input field, not useful here
	 * @private
	 */
	cancelInvitation : function(buttonClicked, text)
	{
		if (buttonClicked == 'yes') {
			// Here scope is record so this refers to Appointment Record.
			this.cancelInvitation();
		}
	},

	/**
	 * Function declines a Meeting invitation and sends Meeting Decline message.
	 * 
	 * @param {String} buttonClicked The ID of the button pressed,
	 * here, one of: ok cancel.
	 * @param {Ext.form.Radio} radio The Radio which was selected by the user.
	 * @private
	 */
	declineInvitation : function(buttonClicked, radio)
	{
		if (buttonClicked == 'ok') {
			// Here scope is record so this refers to Appointment Record.
			var sendUpdateFlag = (radio.id == 'sendResponseOnDelete') ? true: false;
			this.declineMeeting(sendUpdateFlag);
		}
	},

	/**
	 * Opens a {@link Zarafa.common.restore.ui.RestoreContentPanel restoreContentPanel}
	 *
	 * @param {Zarafa.hierarchy.data.MAPIFolderRecord} folder folder that is loaded for the new context
	 * @param {Object} config (optional) Configuration object for creating the content panel
	 */
	openRestoreContent : function(folder, config)
	{
		var componentType = Zarafa.core.data.SharedComponentType['common.dialog.restoreitems'];
		config = Ext.applyIf(config || {}, {
			folder : folder
		});
		Zarafa.core.data.UIFactory.openLayerComponent(componentType, undefined, config);
	},

	/**
	 * Opens a {@link Zarafa.addressbook.dialogs.ABUserSelectionContentPanel ABUserSelectionContentPanel}
	 *
	 * @param {Object} config Configuration object. For AB this normally includes:
	 * 	callback - Callback function to be called with the user selected in the ContentPanel
	 * 	hierarchyRestriction - Restriction that has to be applied on the hierarchy of the Addressbook
	 * 	listRestriction - Restriction that has to be applied on the contents of the Addressbook
	 */
	openABUserSelectionContent : function(config)
	{
		var componentType = Zarafa.core.data.SharedComponentType['addressbook.dialog.abuserselection'];
		config = Ext.applyIf(config || {}, {
			modal : true
		});

		Zarafa.core.data.UIFactory.openLayerComponent(componentType, undefined, config);
	},

	/**
	 * Opens a {@link Zarafa.addressbook.dialog.ABMultiUserSelectionContentPanel ABMultiUserSelectionContentPanel}
	 *
	 * @param {Object} config Configuration object for the dialog
	 */
	openABUserMultiSelectionContent : function(config)
	{
		config = config || {};

		Ext.applyIf(config, {
			modal : true,
			convert : function(user) { return user; }
		});

		var componentType = Zarafa.core.data.SharedComponentType['addressbook.dialog.abmultiuserselection'];
		Zarafa.core.data.UIFactory.openLayerComponent(componentType, undefined, config);
	},

	/**
	 * Mark the given messages as read or unread. When a read receipt was requested
	 * for this message, the setttings are consulted to see if we must automatically
	 * send the receipt or not, or if we should ask the user.
	 *
	 * @param {Zarafa.core.data.IPMRecord/Array} records The record or records which must
	 * be marked as read.
	 * @param {Boolean} read (optional) False to mark the messages as unread, otherwise
	 * the message will be marked as read.
	 */
	markAsRead : function(records, read)
	{
		records = !Ext.isArray(records) ? [ records ] : records;
		read = !Ext.isDefined(read) ? true : read;

		var saveRecords = [];

		for (var i = 0, len = records.length; i < len; i++) {
			var record = records[i];

			// If the read status already matches the desired state,
			// we don't need to do anything.
			if (read !== record.isRead()) {
				if (read === true && record.needsReadReceipt()) {
					switch (container.getSettingsModel().get('zarafa/v1/contexts/mail/readreceipt_handling')) {
						case 'never':
							record.setReadFlags(read);
							// Never send a read receipt.
							record.addMessageAction('send_read_receipt', false);

							saveRecords.push(record);
							break;
						case 'always':
							record.setReadFlags(read);
							// Always send a read receipt.
							record.addMessageAction('send_read_receipt', true);

							saveRecords.push(record);
							break;
						case 'ask':
						default:
							// Ask if a read receipt must be send.
							Ext.MessageBox.confirm(_('Zarafa WebApp'), _('The sender of this message has asked to be notified when you read this message. Do you wish to notify the sender?'),
								// This function will execute when user provide some inputs,
								// So other external changes should not affect the record.
								function(buttonClicked) {
									record.setReadFlags(read);
									record.addMessageAction('send_read_receipt', buttonClicked !== 'no');
									record.save();
								}, this);
							break;
						
					}
				} else {
					record.setReadFlags(read);
					saveRecords.push(record);
				}
			}
		}

		if (!Ext.isEmpty(saveRecords)) {
			saveRecords[0].store.save(saveRecords);
		}
	},

	/**
	 * Will start the download by setting the dialogFrame's location to the download URL of the file.
	 * 
	 * @param {Zarafa.core.data.IPMAttachmentRecord} records The record of the file to be downloaded
	 * @param {Boolean} allAsZip (optional) True to downloading all the attachments as ZIP
	 */
	downloadAttachment : function(record, allAsZip)
	{
		if(!this.downloadFrame) {
			this.downloadFrame = new Zarafa.common.attachment.ui.AttachmentDownloader();
		}

		this.downloadFrame.checkForEmbeddedAttachments(record, allAsZip);
	},

	/**
	 * Opens a {@link Zarafa.common.rules.dialogs.RulesWordsEditContentPanel}
	 *
	 * @param {Object} config Configuration object for the dialog
	 */
	openRulesWordsEditContent : function(config)
	{
		var componentType = Zarafa.core.data.SharedComponentType['common.rules.dialog.ruleswordsedit'];
		Zarafa.core.data.UIFactory.openLayerComponent(componentType, undefined, config);
	},

	/**
	 * Function is used to download attachments, for embedded message attachments additionally it will
	 * convert the {@link Zarafa.core.data.IPMAttachmentRecord IPMAttachmentRecord} to {@link Zarafa.core.data.IPMRecord IPMRecord}
	 * and then will pass it to {@link Zarafa.core.ui.ContentPanel ContentPanel} to open it.
	 * @param {Zarafa.core.data.IPMAttachmentRecord} record The attachment record which should be opened.
	 * @param {Object} config configuration object.
	 */
	openAttachmentRecord: function(record, config)
	{
		if(record.isEmbeddedMessage()) {
			// if we are going to open embedded message then we need to first convert it into mail record
			record = record.convertToIPMRecord();
		}

		if(record) {
			Zarafa.core.data.UIFactory.openViewRecord(record, config);
		}
	},

	/**
	 * Open a Panel in which the {@link Zarafa.core.data.IPMRecord record}
	 * can be viewed, or further edited.
	 *
	 * @param {Zarafa.core.data.IPMRecord} records The records to open
	 * @param {Object} config (optional) Configuration object used to create
	 * the Content Panel.
	 */
	openMessageContent : function(records, config)
	{
		Ext.each(records, function(record) {
			if (record.isUnsent() && !record.isFaultyMessage()) {
				Zarafa.core.data.UIFactory.openCreateRecord(record, config);
			} else {
				Zarafa.core.data.UIFactory.openViewRecord(record, config);
			}
		});
	}
};
Ext.namespace('Zarafa.common.attachment.dialogs');

/**
 * @class Zarafa.common.attachment.dialogs.AttachItemPanel
 * @extends Ext.Panel
 * @xtype zarafa.attachitempanel
 */
Zarafa.common.attachment.dialogs.AttachItemPanel = Ext.extend(Ext.Panel, {
	/**
	 * @cfg {Zarafa.core.data.MAPIRecord} record record in which we will add embedded attachment if user is adding it as attachment.
	 */
	record : undefined,

	/**
	 * @cfg {Zarafa.common.ui.EditorField} editor editor in which we will be adding message as text if user has selected to add text in body.
	 */
	editor : undefined,

	/**
	 * @constructor
	 * @param {Object} config Configuration object
	 */
	constructor : function(config)
	{
		config = config || {};

		Ext.applyIf(config, {
			layout: {
				type: 'vbox',
				align : 'stretch',
				pack  : 'start',
				padding : 10
			},
			items : [{
				xtype : 'displayfield',
				value : _('Look in:'),
				hideLabel : true,
				height : 20
			}, {
				xtype : 'container',
				flex : 1,
				layout : {
					type : 'hbox',
					align : 'stretch',
					pack : 'start'
				},
				items : [{
					xtype: 'zarafa.hierarchytree',
					flex : 1,
					border : true,
					enableDD : false,
					treeSorter : true,
					ref : '../hierarchyTree',
					listeners : {
						// Register event that will initially select a default folder
						'load' : this.onTreeNodeLoad,
						scope : this
					}
				}, {
					xtype : 'container',
					width : 150,
					defaults : {
						style : {
							margin : '0px 0px 10px 5px'
						}
					},
					style : {
						padding : '0px 0px 0px 10px'
					},
					items : [{
						xtype : 'button',
						text : _('Ok'),
						disabled : true,
						ref : '../../okButton',
						handler : this.onOK,
						scope : this
					}, {
						xtype : 'button',
						text : _('Cancel'),
						ref : '../../cancelButton',
						handler : this.onCancel,
						scope : this
					}, {
						xtype : 'fieldset',
						title : _('Insert as'),
						padding : '0px 5px 5px 5px',
						layout : {
							type : 'fit'
						},
						items : [{
							xtype : 'radiogroup',
							vertical : true,
							columns : 1,
							ref : '../../../attachTypeRadioGroup',
							defaults : {
								style : {
									margin : '5px 0px 0px 5px'
								}
							},
							items : [{
								xtype : 'radio',
								boxLabel : _('Text only'),
								inputValue : 'text_only',
								name : 'attach_item'
							}, {
								xtype : 'radio',
								boxLabel : _('Attachment'),
								inputValue : 'attachment',
								name : 'attach_item'
							}],
							listeners : {
								// Register event that will select default radio group item
								'afterrender' : this.onRadioGroupAfterRender,
								// Register event that will save current selected radio item in state settings
								'change' : this.onRadioSelectionChange,
								scope : this
							}
						}]
					}]
				}]
			}, {
				xtype: 'zarafa.attachitemgrid',
				flex : 1,
				ref : 'attachItemGrid'
			}]
		});

		Zarafa.common.attachment.dialogs.AttachItemPanel.superclass.constructor.call(this, config);
	},

	/**
	 * Initialize the event handlers
	 * @protected
	 */
	initEvents : function()
	{
		Zarafa.common.attachment.dialogs.AttachItemPanel.superclass.initEvents.apply(this, arguments);

		// Register event that will be fired when selection in hierarchy is changed
		var sm = this.hierarchyTree.getSelectionModel();
		this.mon(sm, 'selectionchange', this.onTreeSelectionChange, this);

		// Register event that will enable/disable ok button based on current selection
		this.mon(this.attachItemGrid.getSelectionModel(), 'selectionchange', this.onGridSelectionChange, this);
	},

	/**
	 * Function will call {@link Zarafa.common.attachment.dialogs.AttachItemGrid#attachItem}.
	 * @private
	 */
	onOK : function()
	{
		this.attachItem();
	},

	/**
	 * Handler will called when user presses the cancel button, this will simply close the {@link Zarafa.common.attachment.AttachItemContentPanel AttachItemContentPanel}
	 * @private
	 */
	onCancel : function()
	{
		this.dialog.close();
	},

	/**
	 * Fired when the {@link Zarafa.hierarchy.ui.Tree Tree} fires the {@link Zarafa.hierarchy.ui.Tree#load load}
	 * event. This function will try to select the {@link Ext.tree.TreeNode TreeNode} in
	 * {@link Zarafa.hierarchy.ui.Tree Tree} intially. When the given node is not loaded yet, it will try again
	 * later when the event is fired again.
	 * @param {Ext.tree.TreeNode} node node that has been loaded
	 * @private
	 */
	onTreeNodeLoad : function(node)
	{
		var folder = this.dialog.getSelectedFolder();

		// If the folder could be selected, then unregister the event handler.
		if (this.hierarchyTree.selectFolderInTree(folder)) {
			this.mun(this.hierarchyTree, 'load', this.onTreeNodeLoad, this);
		}
	},

	/**
	 * Invoked when {@link Ext.form.RadioGroup RadioGroup} is rendered fully and {@link Ext.form.RadioGroup#afterrender} event is fired.
	 * This function will try to select the {@link Ext.form.Radio Radio} in {@link Ext.form.RadioGroup RadioGroup} intially.
	 * @param {Ext.form.RadioGroup} radioGroup radio group that has been rendered.
	 * @private
	 */
	onRadioGroupAfterRender : function(radioGroup)
	{
		var option = this.dialog.getSelectedRadioItem();

		radioGroup.setValue(option);
	},

	/**
	 * Fired when a radio item is selected in {@link Ext.form.RadioGroup RadioGroup}.
	 * It will store inputValue of selected radio item into state settings.
	 * @param {Ext.form.RadioGroup} radioGroup radio group that has fired change event.
	 * @param {Ext.form.Radio} checkedRadio radio item which is selected.
	 * @private
	 */
	onRadioSelectionChange : function(radioGroup, checkedRadio)
	{
		// save new selection in state settings
		this.dialog.setRadioItemInState(checkedRadio);
	},

	/**
	 * Fired when a node is selected in {@link Zarafa.hierarchy.ui.Tree}. It loads items of the folder in grid.
	 * @param {Ext.tree.DefaultSelectionModel} selectionModel selection model that is used in {@link Zarafa.hierarchy.ui.Tree HierarchyTree}
	 * @param {Ext.tree.TreeNode} node node which is selected
	 * @private
	 */
	onTreeSelectionChange : function(selectionModel, node)
	{
		var folder = node.getFolder();

		var store = this.dialog.getStoreByFolder(folder);
		var colModel = this.dialog.getColumnModelByFolder(folder);

		this.attachItemGrid.reconfigure(store, colModel);

		// load the contents of the store
		store.load();

		// update state settings, and store newly selected folder entryid as last selected folder
		this.dialog.setFolderInState(folder);
	},

	/**
	 * Event handler which is triggered when the {@link Zarafa.calendar.ui.CounterProposalGrid grid}
	 * {@link Zarafa.core.data.IPMRecord record} selection is changed.
	 *
	 * @param {Ext.grid.RowSelectionModel} selectionModel The selection model used by the grid.
	 * @private
	 */
	onGridSelectionChange : function(selectionModel)
	{
		var count = selectionModel.getCount();

		this.okButton.setDisabled(count === 0);
	},

	/**
	 * Wrapper function to call {@link #attachItemAsAttachment} or {@link #attachItemAsText} based on selected radio item.
	 * @param {String} radioValue String that indicates which radio item is selected, based on this value different function will be called.
	 * @private
	 */
	attachItem : function(radioValue)
	{
		var selected = this.attachItemGrid.getSelectionModel().getSelected();
		var selectedRadio = this.attachTypeRadioGroup.getValue().inputValue;

		if (Ext.isEmpty(selected)) {
			Ext.MessageBox.alert(_('Attach Item'), _('No message selected'));
			return;
		}

		if(selectedRadio === 'text_only') {
			// add selected message data to parent message's body
			this.attachItemAsText(selected);
		} else {
			// add selected message as an attachment to parent message
			this.attachItemAsAttachment(selected);
		}
	},

	/**
	 * Function will be used to add selected message to another message as an embedded attachment.
	 * @param {Zarafa.core.data.IPMRecord} record record that is selected in grid and which should be added as attachment.
	 * @private
	 */
	attachItemAsAttachment : function(record)
	{
		// add new attachment into parent record
		this.record.getAttachmentStore().addEmbeddedAttachment(record);

		// now close the dialog
		this.dialog.close();
	},

	/**
	 * Function will be used to add selected message to another message as body contents.
	 * @param {Zarafa.core.data.IPMRecord} record record that is selected in grid and which should be added as text.
	 * @private
	 */
	attachItemAsText : function(record)
	{
		// if record is not opened then open it and get all contents of the record
		if(!record.isOpened()) {
			this.openRecord(record, this.attachItemAsText);
			return;
		}

		var renderer = this.dialog.getRendererByMessage(record);

		// Get the editor in which we need to add text and add text generated by renderer
		this.dialog.editor.insertAtCursor(renderer.generateText(record, this.record.get('isHTML')));

		// now close the dialog
		this.dialog.close();
	},

	/**
	 * Function will send an 'open' request to server to get all data of the passed {@link Zarafa.core.data.IPMRecord IPMRecord}.
	 * After record is opened function will call function specified in callback parameter.
	 * @param {Zarafa.core.data.IPMRecord} record record which needs to be opened to get all data.
	 * @param {Ext.Function} callback callback function that will be called after successfully opening the record.
	 */
	openRecord : function(record, callback)
	{
		// show load mask till we fetch full data from server
		this.dialog.showLoadMask();

		// store a reference of record's store which can be used to deregister events in exception handler
		var store = record.getStore();

		var fnOpen = function(recStore, rec) {
			if (record === rec) {
				// remove registered handlers
				store.un('open', fnOpen, this);
				store.un('exception', fnException, this);

				// hide load mask as data has arrived
				this.dialog.hideLoadMask();

				// call the callback function
				callback.call(this, record);
			}
		};

		var fnException = function(proxy, type, action, options, response, args) {
			if(action === Ext.data.Api.actions.open) {
				// remove registered handlers, this exception has been fired by Proxy so we will not get store in parameters
				store.un('open', fnOpen, this);
				store.un('exception', fnException, this);

				// actual error message will be shown by common exception handler
				// we will just hide the load mask
				this.dialog.hideLoadMask();
			}
		};

		store.on('open', fnOpen, this);
		store.on('exception', fnException, this);

		record.open();
	}
});

Ext.reg('zarafa.attachitempanel', Zarafa.common.attachment.dialogs.AttachItemPanel);Ext.namespace('Zarafa.common.attachment.dialogs');

/**
 * @class Zarafa.common.attachment.dialogs.MixAttachItemPanel
 * @extends Ext.Panel
 * @xtype zarafa.mixattachitempanel
 *
 * Panel for users to list out the unsupported attachments while downloading all the attachments as ZIP.
 * It also allows user to provide his/her wish to show this warning again or not.
 */
Zarafa.common.attachment.dialogs.MixAttachItemPanel = Ext.extend(Ext.Panel, {
	/**
	 * @cfg {Array} records The {@link Zarafa.core.data.IPMRecord record(s)} which are being
	 * used in this panel
	 */
	records : undefined,

	/**
	 * @constructor
	 * @param config Configuration structure
	 */
	constructor : function(config)
	{
		config = config || {};

		Ext.applyIf(config, {
			xtype : 'zarafa.mixattachitempanel',
			border: false,
			items: [{
				xtype : 'fieldset',
				layout : 'form',
				anchor: '100% 30%',
				header : true,
				iconCls : 'mixattach-message-box-icon',
				border : false,
				items: this.getMixAttachComponents()
			}]
		});

		Zarafa.common.categories.dialogs.CategoriesPanel.superclass.constructor.call(this, config);
	},

	/**
	 * Prepare all the components of warning dialog.
	 * @return {Array} Array containing all the required components
	 */
	getMixAttachComponents : function()
	{
		return [{
			xtype: 'displayfield',
			value: _('These attachments have an unsupported file type and can not be added to the ZIP file:'),
			hideLabel : true,
			htmlEncode : true
		}, {
			xtype: 'textarea',
			hideLabel: true,
			anchor: '100%',
			style: 'padding-left: 5px;',
			readOnly : true,
			listeners: {
				afterrender : this.onAfterRenderTextArea,
				scope: this
			}
		}, {
			xtype: 'displayfield',
			value: _("Would you like to continue downloading without adding these files?"),
			hideLabel : true,
			htmlEncode : true
		}, {
			xtype : 'checkbox',
			ref : '../../dontShowCheckBox',
			boxLabel : _('Always continue download of ZIP files without adding unsupported file types.'),
			hideLabel : true,
			scope : this
		}]
	},

	/**
	 * Handler which is called when 'textarea' is rendered. it prepares a list of all the unsupported attachments,
	 * which is to be shown in the text area, line by line.
	 * @param {Ext.form.TextArea} leftoutTextArea The textarea
	 * @return {String} list of unsupported attachments, separated by escaping sequence
	 */
	onAfterRenderTextArea : function(leftoutTextArea)
	{
		var leftOutList = "";

		Ext.each(this.records, function(record) {
			if(record.isEmbeddedMessage()){
				leftOutList += record.get('name') + "\n" ;
			}
		});

		leftoutTextArea.setValue(leftOutList);
	}
});

Ext.reg('zarafa.mixattachitempanel', Zarafa.common.attachment.dialogs.MixAttachItemPanel);
Ext.namespace('Zarafa.common.attachment.ui');

/**
 * @class Zarafa.common.attachment.ui.AttachmentButton
 * @extends Ext.SplitButton
 * @xtype zarafa.attachmentbutton
 *
 * Special button which can be used for attaching items to a {@link Zarafa.core.data.IPMRecord IPMRecord}.
 * This utilizes the {@link #main.attachment.method} insertion point to allow plugins to register
 * alternative methods for attaching items to the record. These options will be shown inside the dropdown
 * list, while the default button action will be opening the Browsers File Selection dialog
 *
 * If the {@link Zarafa.core.plugins.RecordComponentUpdaterPlugin} is installed
 * in the {@link #plugins} array of this component, this component will automatically
 * load the {@link Zarafa.core.data.MAPIRecord record} into the component.
 * Otherwise the user of this component needs to call {@link #bindRecord}.
 */
Zarafa.common.attachment.ui.AttachmentButton = Ext.extend(Ext.SplitButton, {
	/**
	 * @insert main.attachment.method
	 * Provide a new method for attaching files to a {@link Zarafa.core.data.IPMRecord IPMRecord}.
	 * This can be used by 3rd party plugins to insert a new MenuItem into the dropdown
	 * box for the {@link Zarafa.common.attachment.ui.AttachmentButton AttachmentButton}.
	 * This insertion point should return a {@link Ext.menu.Item item} instance of configuration
	 * @param {Zarafa.common.attachment.ui.AttachmentButton} button This button
	 */

	/**
	 * @constructor
	 * @param {Object} config Configuration object
	 */
	constructor : function(config)
	{
		config = config || {};

		Ext.applyIf(config, {
			menu : {
				items : [{
					text : _('File upload'),
					handler : this.onFileUpload,
					scope: this,
					iconCls : 'icon_attachment'
				}, {
					text : _('Attach item'),
					handler : this.onFileAttach,
					scope: this,
					iconCls : 'icon_embedded_attachment'
				},
				container.populateInsertionPoint('main.attachment.method', this)
				]
			},
			handler : this.onFileUpload,
			scope : this
		});

		Zarafa.common.attachment.ui.AttachmentButton.superclass.constructor.call(this, config);
	},

	/**
	 * Event handler for opening the Browser's file selection dialog.
	 * See {@link #onFileInputChange} for the handling of the selected files.
	 * @param {Ext.Button} button the button on which click event is performed.
	 * @param {Ext.EventObject} event The event object
	 * @private
	 */
	onFileUpload : function(button, event)
	{
		var attachComponent = new Zarafa.common.attachment.ui.UploadAttachmentComponent({
			callback : this.uploadAttachmentCallback,
			multiple : true,
			scope : this
		});

		attachComponent.openAttachmentDialog();
	},

	/**
	 * The callback function of {@link Zarafa.common.attachment.ui.UploadAttachmentComponent}
	 * which used to upload the attachment file on server.
	 * 
	 * @param {Object/Array} files The files contains file information.
	 * @param {Object} form the form is contains {@link Ext.form.BasicForm bacisform} info.
	 */
	uploadAttachmentCallback : function(files, form)
	{
		var store = this.record.getSubStore('attachments');
		store.uploadFiles(files, form);
	},

	/**
	 * Event handler for opening the {@link Zarafa.common.attachment.AttachItemContentPanel AttachItemContentPanel}.
	 * @private
	 */
	onFileAttach : function(field, event)
	{
		// get the parent from which we can find the editorfield
		var panel = this.findParentByType('zarafa.recordcontentpanel');
		var editor = panel.findByType('zarafa.editorfield');
		if(!Ext.isEmpty(editor)) {
			editor = editor[0];
		}

		Zarafa.common.Actions.openAttachItemSelectionContent(this.record, {editor : editor});
	},

	/**
	 * Apply the record to the button. The given record will be used by the attachment handlers
	 * for adding the attachments to the record.
	 * @param {Zarafa.core.data.IPMRecord} record
	 */
	bindRecord : function(record)
	{
		this.record = record;
	},

	/**
	 * Update the components with the given record.
	 *
	 * @param {Zarafa.core.data.MAPIRecord} record The record to update in this component
	 * @param {Boolean} contentReset force the component to perform a full update of the data.
	 */
	update : function(record, contentReset)
	{
		if (record && record instanceof Zarafa.core.data.MAPIRecord) {
			// In case the recordcomponentupdaterplugin is installed
			// we have a special action to update the component.
			if (contentReset) {
				this.bindRecord(record);
			}
		} else {
			// The recordcomponentupdaterplugin is not installed and the
			// caller really wants to perform the update() function. Probably
			// a bad move, but lets not disappoint the caller.
			Zarafa.common.attachment.ui.AttachmentButton.superclass.update.apply(this, arguments);
		}
	}
});

Ext.reg('zarafa.attachmentbutton', Zarafa.common.attachment.ui.AttachmentButton);
Ext.namespace('Zarafa.common.attachment.ui');
/**
 * @class Zarafa.common.attachment.ui.AttachmentDownloader
 * @extends Ext.Component
 * @xtype zarafa.attachmentdownloader
 *
 * Independent component to encapsulate process of downloading attachments,
 * this is achieved by use of hidden iframe, in which we will set url of the
 * attachment that needs to be downloaded. The file pointed by url should
 * return attachment data with content disposition type as attachment, so
 * browser will show a dialog to open/save attachment, but if the server side
 * needs to send an error then make sure it returns it with content disposition
 * type as inline, so that iframe's onload event will be fired and proper error
 * message will be displayed to user.
 */
Zarafa.common.attachment.ui.AttachmentDownloader = Ext.extend(Ext.Component, {
	/**
	 * @constructor
	 * @param {Object} config Configuration structure
	 */
	constructor : function(config)
	{
		config = config || {};

		Ext.applyIf(config, {
			xtype : 'zarafa.attachmentdownloader',
			renderTo : Ext.getBody(),
			hidden : true,
			autoEl: {
				tag: 'iframe',
				src: Ext.SSL_SECURE_URL
			}
		});

		Zarafa.common.attachment.ui.AttachmentDownloader.superclass.constructor.call(this, config);

		//Register a listener to onload event of iframe, but please take a note that
		//the 'load' event is fired only if content disposition type is configured as 'inline'.
		Ext.EventManager.on(this.getEl(), 'load', this.onIframeLoad, this);
	},

	/**
	 * Will get iframe element and set its src property to the supplied url.
	 * After successfull response, iframe will pops up and ask user to start/cancel
	 * downloading of that particular message/attachment as file.
	 * @param {String} url The url to download message/attachment, containing necessary parameters.
	 */
	downloadItem : function(url)
	{
		var iframeElement = Ext.getDom(this.getEl());

		//setting iframe's location to the download url
		iframeElement.src = url;
	},

	/**
	 * Function prepares necessary HTML structure to send post request containing
	 * entryids of all the selected messages needs to be downloaded as eml in ZIP.
	 * Will get iframe element and create a form element dynamically.
	 * Create input elements for every entryid we need to send into post request,
	 * and append those dynamically created input elements into form.
	 * 
	 * @param {Zarafa.core.data.IPMRecord} records The records which user want to save in eml format as ZIP.
	 */
	downloadMessageAsZip : function(records)
	{
		// Get the total size of all the messages which is requested to be included in ZIP
		var totalSize = 0;
		Ext.each(records, function(record) {
			totalSize += record.get('message_size');
		}, this);

		// Check if total size of eml is more than 30 MB, avoid ZIP creation as the request will
		// be timed out by the time while preparing eml for next message and there is no
		// technical provision to detect request-time-out exception.
		// 31457280 byte is equivalent to 30MB.
		if (totalSize < 31457280) {
			var url = records[0].getDownloadMessageUrl(true);

			var iframeBody = Ext.getDom(this.getEl()).contentDocument.body;
			var form = Ext.DomHelper.append( iframeBody || Ext.getBody(), {tag : 'form', action : url, method : 'POST'}, true);

			// Create and append input elements containing entryid as its value
			for (var i = 0; i < records.length; i++) {
				Ext.DomHelper.append(form, {tag : 'input', type : 'hidden', name : 'entryids[]', value : records[i].get('entryid')});
			}

			// In IE10 and IE9 beforeunload will be called while submitting form.
			// So prevent this situation by disable leave requester which prevent the requester dialog to appear.
			if(Ext.isIE10 || Ext.isIE9) {
				Zarafa.core.Util.disableLeaveRequester();
			}

			// Submit the form to send a POST request
			form.dom.submit();

			// Enable leave requester back if user browser is IE10 or IE9.
			if(Ext.isIE10 || Ext.isIE9) {
				Zarafa.core.Util.enableLeaveRequester();
			}

		} else {
			container.getNotifier().notify(
				'error.attachment',
				_('Attachment error'),
				_('Cannot create ZIP, The allowed maximum size is 30 MB.')
			);
			return;
		}
	},

	/**
	 * Handler for the 'load' event of iframe, fired immediately after iframe is loaded.
	 * The exception response is received in json format, so we are using {@link Ext.util.JSON#decode}
	 * to decode (parse) a JSON string to an object.
	 * @private
	 */
	onIframeLoad : function()
	{
		var contentDocument;
		var responseText;

		contentDocument = this.getEl().dom.contentDocument;

		if (!Ext.isEmpty(contentDocument)) {
			responseText = contentDocument.body.textContent;
		}

		if (!Ext.isEmpty(responseText)) {
			var responseObject = Ext.util.JSON.decode(responseText);
			this.displaySaveEmailException(responseObject);
		}
	},

	/**
	 * Raise a warning dialog to inform user that there is some embedded attachments which can not be included in ZIP.
	 * otherwise continue with creating ZIP for normal attachments.
	 * 
	 * @param {Zarafa.core.data.IPMAttachmentRecord} records The record of the file to be downloaded
	 * @param {Boolean} allAsZip (optional) True to downloading all the attachments as ZIP
	 */
	checkForEmbeddedAttachments : function(record, allAsZip)
	{
		var containsEmbedded = false;

		// We need all this additional checking only when we are going to download attachments as ZIP, skip otherwise.
		if(allAsZip) {
			// If user already made his/her decision to not show this warning again then it is there is state setting, just act accordingly.
			var dontShowWarning = container.getSettingsModel().get('zarafa/v1/state/dialogs/mixattachitemcontentpanel/dontshowagain');
			var attachmentStore = record.store;

			// Check if there is any embedded attachments only when user want this warning to be raised
			// and there is more than one attachments.
			if(!dontShowWarning && attachmentStore.getCount() > 1) {
				attachmentStore.each(function(record){
					if(record.isEmbeddedMessage()){
						containsEmbedded = true;
						return false;
					}
				});
			}
		}

		// Embedded attachment(s) found, raise warning dialog
		if(containsEmbedded) {
			this.openMixAttachmentsDialog(attachmentStore.getRange(), {'allAsZip' : allAsZip});
		} else {
			this.downloadItem(record.getAttachmentUrl(allAsZip));
		}
	},

	/**
	 * Opens a {@link Zarafa.common.attachment.dialogs.MixAttachItemContentPanel MixAttachItemContentPanel} for informing
	 * user that there is some embedded attachments are requested to be included in ZIP, which is not possible.
	 *
	 * @param {Zarafa.core.data.IPMRecord} records The record, or records for which the attachments are 
	 * requested to be downloaded as ZIP
	 * @param {Object} config (optional) Configuration object for creating the ContentPanel
	 */
	openMixAttachmentsDialog : function(records, config)
	{
		if (!Ext.isArray(records)) {
			records = [ records ];
		}

		config = Ext.applyIf(config || {}, {
			modal : true,
			downloadItem : this.downloadItem.createDelegate(this, [ records[0].getAttachmentUrl(config.allAsZip) ], 1)
		});

		var componentType = Zarafa.core.data.SharedComponentType['common.attachment.dialog.mixattachitem'];
		Zarafa.core.data.UIFactory.openLayerComponent(componentType, records, config);
	},

	/**
	 * It displays proper message to user, as per the exception received
	 * while unsuccessful download.
	 * @param {Object} responseObject Object contained the exception details.
	 * @private
	 */
	displaySaveEmailException : function(responseObject)
	{
		Ext.MessageBox.show({
			title : _('Zarafa WebApp'),
			msg : responseObject.zarafa.error.info.display_message,
			icon: Ext.MessageBox.ERROR,
			buttons : Ext.MessageBox.OK
		});
	}
});
Ext.reg('zarafa.attachmentdownloader', Zarafa.common.attachment.ui.AttachmentDownloader);
Ext.namespace('Zarafa.common.attachment.ui');
/**
 * @class Zarafa.common.attachment.ui.UploadAttachmentComponent
 * @extends Ext.Component
 * @xtype zarafa.uploadattachmentcomponent
 */
Zarafa.common.attachment.ui.UploadAttachmentComponent = Ext.extend(Ext.Component, {
	/**
	 * @cfg {Function} callback The callback function which must be called when the
	 * file has be selected from Browser's file selection dialog.
	 */
	callback : Ext.emptyFn,

	/**
	 * @cfg {Object} scope The scope for the {@link #callback} function
	 */
	scope : undefined,

	/**
	 * @cfg {Boolean} multiple The multiple true to allow upload multiple files
	 * else allow single file only. by default it is false.
	 */
	multiple : false,

	/**
	 * @cfg {String} accept the accept define which type of files allow to
	 * show in Browser's file selection dialog. i.e image/* to allow all type of images.
	 */
	accept : undefined,

	/**
	 * @constructor
	 * @param {Object} config Configuration structure
	 */
	constructor : function(config)
	{
		config = config || {};

		Ext.applyIf(config,{
			xtype : 'zarafa.uploadattachmentcomponent'
		});

		Zarafa.common.attachment.ui.UploadAttachmentComponent.superclass.constructor.call(this, config);
	},

	/**
	 * Event handler for opening the Browser's file selection dialog.
	 * See {@link #onFileInputChange} for the handling of the selected files.
	 * @private
	 */
	openAttachmentDialog : function()
	{
		if (Zarafa.supportsFilesAPI()) {
			var attachEl = this.getAttachmentEl();

			// Register the change event handler
			// so we detect when the user selects a file.
			attachEl.on('change', this.onFileInputChange, this);

			// Mimick clicking on the <input> field
			// to open the File Selection dialog.
			attachEl.dom.click();
		} else {
			// With legacy upload we need a valid form, and we cannot
			// mimick clicking on the <input> field. Hence a secondary
			// dialog which should handle that.
			Zarafa.common.Actions.openLegacyFileSelectionContent({
				callback : this.onLegacyFileInputChange,
				scope : this
			});
		}
	},

	/**
	 * Obtain or instantiate the {@link Ext.Element attachment} &lt;input&gt; element used
	 * for opening the File selection dialog.
	 * @return {Ext.Element} The file input element
	 * @private
	 */
	getAttachmentEl : function()
	{
		var attachEl = Ext.DomHelper.append(Ext.getBody(), {
			cls : 'x-hidden',
			tag : 'input',
			type : 'file'
		});

		if(Ext.isDefined(this.multiple) && this.multiple) {
			attachEl.multiple = this.multiple;
		}

		if(Ext.isDefined(this.accept)) {
			attachEl.accept = this.accept;
		}

		attachEl = Ext.get(attachEl);
		return attachEl;
	},

	/**
	 * Event handler which is fired when the {@link Zarafa#supportsFilesAPI Files API} implementation for
	 * selecting files could not be used.
	 * @param {Ext.form.BasicForm} form The form
	 * @private
	 */
	onLegacyFileInputChange : function(form)
	{
		var files = Ext.pluck(form.items.items, 'value');

		if(this.accept === 'image/*') {
			var fileName = Ext.util.Format.basename(files[0]);
			var extension = fileName.split('.')[1];
			var fileType = 'image/'+extension;
			if (this.isSupportedImage(fileType)) {
				this.callback.call(this.scope, files, form);
			} else {
				this.showAttachmentError();
			}
		} else {
			this.callback.call(this.scope, files, form);
		}
	},

	/**
	 * Event handler which is fired when the {@link #attachEl} has been changed.
	 * @param {Ext.EventObject} event The event
	 * @private
	 */
	onFileInputChange : function(event)
	{
		var browserEvent = event.browserEvent;
		var attachEl = Ext.get(browserEvent.target);
		var transfer = browserEvent.dataTransfer;
		var transferFile = transfer ? transfer.files : undefined;
		var files = attachEl.dom.files || transferFile;
		var record = this.scope.record;
		var store = record.getAttachmentStore();

		if(store.canUploadFiles(files)) {
			if(this.accept === 'image/*') {
				var fileType = files[0].type;
				if (this.isSupportedImage(fileType)) {
					this.callback.call(this.scope, files);
				} else {
					this.showAttachmentError();
				}
			} else {
				this.callback.call(this.scope, files);
			}
		}
		// remove attachment element.
		attachEl.remove();
	},

	/**
	 * Function shows the attachment error message when picture formate is not supported by webapp 
	 */
	showAttachmentError : function()
	{
		var message = _('Picture format is not supported. ');
		message += _('Supported format for contact pictures are');
		message += '<br/>';
		message += 'JPEG, GIF, PNG, BMP';
		Ext.MessageBox.show({
			title: _('Attachment Error'),
			msg : message,
			icon: Ext.MessageBox.ERROR,
			buttons: Ext.MessageBox.OK
		});
	},

	/**
	 * Function check that selected file is supported image or not.
	 * @param {String} fileType the fileType is defined the mime type of selected file.
	 * @return {Boolean} return true if selected picture is supported by webapp else false.
	 */
	isSupportedImage : function(fileType)
	{
		var mimeType = ['image/bmp','image/jpg','image/jpeg','image/gif','image/png'];
		return mimeType.indexOf(fileType.toLowerCase()) >= 0;
	}
});

Ext.reg('zarafa.uploadattachmentcomponent', Zarafa.common.attachment.ui.AttachmentButton);Ext.namespace('Zarafa.common.categories.data');
/**
 * @class Zarafa.common.categories.data.CategoriesStore
 * @extends Ext.data.ArrayStore
 * @xtype zarafa.categoriesstore
 * 
 * Store which will get the records from setting and insertion 
 * points which has user-defined categories.
 */
Zarafa.common.categories.data.CategoriesStore = Ext.extend(Ext.data.ArrayStore, {
	// Insertion points for this class
	/**
	 * @insert main.categories
	 * can be used to add extra user-defined categories by 3rd party plugins
	 */
	
	/**
	 * @constructor
	 * @param {Object} config Configuration structure
	 */
	constructor: function(config)
	{
		config = config || {};	
		var categories = [];

		categories = categories.concat(container.getSettingsModel().get('zarafa/v1/main/categories'));
		categories = categories.concat(container.populateInsertionPoint('main.categories'));
		
		for (var i = 0; i < categories.length; i++) {
			if (!Ext.isArray(categories[i]))
				categories[i] = [ categories[i] ];
		}
		
		Ext.applyIf(config, {
			fields : ['category'],
			data: categories
		});
		
		Ext.apply(this, config);
		
		Zarafa.common.categories.data.CategoriesStore.superclass.constructor.call(this, config);
	}
});

Ext.reg('zarafa.categoriesstore', Zarafa.common.categories.data.CategoriesStore);
Ext.namespace('Zarafa.common.categories.dialogs');

/**
 * @class Zarafa.common.categories.dialogs.CategoriesPanel
 * @extends Ext.Panel
 * @xtype zarafa.categoriespanel
 *
 * Panel for users to edit the categories on a given {@link Zarafa.core.data.IPMRecord record}
 */
Zarafa.common.categories.dialogs.CategoriesPanel = Ext.extend(Ext.Panel, {
	/**
	 * @cfg {Zarafa.core.data.IPMRecord} record The record(s) which are being
	 * edited through this panel
	 */
	record : undefined,
	/**
	 * @cfg {String} categorySeparator The string which must be used to separate the
	 * various categories.
	 */
	categorySeparator : ';',
	/**
	 * @constructor
	 * @param config Configuration structure
	 */
	constructor : function(config)
	{
		config = config || {};

		Ext.applyIf(config, {
			// Override from Ext.Component
			xtype : 'zarafa.categoriespanel',
			layout: {
				type: 'vbox',
				align: 'stretch'
			},
			border: false,
			bodyStyle: 'padding: 5px; background-color: inherit;',
			defaults: {
				border: false,
				bodyStyle: 'padding-bottom: 5px; background-color: inherit;'
			},
			items: [
				this.createFormCategoriesField(),
				this.createFormSelectionArea()
			]
		});

		Zarafa.common.categories.dialogs.CategoriesPanel.superclass.constructor.call(this, config);
	},

	/**
	 * Create the main categories input panel. This contains the main
	 * {@link Ext.form.TextArea textarea} in which the user can type the categories.
	 * @return {Object} The configuration object for the categories panel.
	 * @private
	 */
	createFormCategoriesField : function()
	{
		return {
			xtype: 'panel',
			layout: 'fit',
			height: 100,
			title: _('Item belongs to these categories'),
			items: [{
				xtype: 'textarea',
				hideLabel: true,
				ref: '../categoriesTextArea'
			}]
		};
	},

	/**
	 * Create the main selection panel which contains the {@link Ext.grid.GridPanel GridPanel}
	 * which can be used by the user to select the category from a predefined list.
	 * @return {Object} The configuration object for the selection panel.
	 *
 	 * @private
	 */
	createFormSelectionArea : function()
	{

		return [{
			xtype: 'panel',
			layout: {
				type: 'hbox',
				align: 'stretch'
			},
			flex: 1,
			title: _('Available categories'),
			defaults: {
				border: false,
				bodyStyle: 'background-color: inherit;'
			},
			items: [{
				xtype: 'panel',
				layout: 'fit',
				flex: 1,
				items: this.createCategoriesGrid()
			}]
		}];
	},

	/**
	 * create gird panel for categories which contains {@link Ext.grid.CheckboxSelectionModel CheckboxSelectionModel}
	 * which can be used by the user to select/change the category from a predefined list.
	 * @return {Object} The configuration object for the grid panel.
	 * @private
	 */
	createCategoriesGrid : function()
	{
		var model = new Ext.grid.CheckboxSelectionModel({
			singleSelect: false,
			checkOnly : true,
			listeners: {
				selectionchange : this.onSelectionChange,
				scope: this
			}
		});

		return {
			xtype: 'grid',
			border: true,
			enableHdMenu: false,
			deferRowRender:false,
			autoExpandColumn: 'category',
			ref: '../../categoriesGrid',
			viewConfig: {
				forceFit: true,
				autoExpandColumn: true,
				scrollOffset: 0
			},
			store: new Zarafa.common.categories.data.CategoriesStore(),
			colModel: new Ext.grid.ColumnModel({
				columns: [
					model,
				{
					id		: 'category',
					header	: _('Category'),
					dataIndex: 'category',
					sortable: false,
					renderer : Ext.util.Format.htmlEncode
				}]
			}),
			sm: model,
			listeners: {
				scope: this,
				afterrender: this.onAfterRender
			}
		};
	},

	/**
	 * Obtain a reference to the {@link Ext.form.TextArea textarea} in which
	 * the user can type his categories manually.
	 *
	 * @return {Ext.form.TextArea} The textarea
	 * @private
	 */
	getCategoriesTextArea : function()
	{
		return this.categoriesTextArea;
	},

	/**
	 * Obtain a reference to the {@link Ext.grid.GridPanel} used to select
	 * from a list of existing Categories.
	 *
	 * @return {Ext.grid.GridPanel} The grid panel
	 * @private
	 */
	getCategoriesGrid : function()
	{
		return this.categoriesGrid;
	},

	/**
	 * handler for event afterrender
	 * @pirvate 
	 */
	onAfterRender : function()
	{
		this.getCategoriesGrid().getSelectionModel().suspendEvents(false);
		var categories = [];
		var records = this.getAvailableCategories();
		if(!Ext.isEmpty(records)) {
			this.getCategoriesGrid().getSelectionModel().selectRecords(records);
		}
		
		this.updateCategories(this.getActiveCategories());
		this.getCategoriesGrid().getSelectionModel().resumeEvents();
	},
	
	/**
	 * return all categories which are active/set for record
	 * @return {Array} The array of categories
	 * @private
	 */
	getActiveCategories : function()
	{
		var activeCategories  = [];
		var category = [];
		var categoryString = '';
		
		if(Ext.isDefined(this.record)){
			Ext.each(this.record, function(record) {
				categoryString = record.get('categories');
								
				if(!Ext.isEmpty(categoryString)){
					category = categoryString.split(this.categorySeparator);
					for(var i=0;i<category.length;i++){
						activeCategories.push(category[i]);
					}
				}
				
			}, this);
		}
		
		activeCategories = Zarafa.core.Util.trimStringArray(activeCategories);
		
		return Zarafa.core.Util.uniqueArray(activeCategories);
	},
	
	/**
	 * return all categories records from the predefined categories
	 * @return {Array} of {@link Ext.data.Record Records}
	 * @private
	 */
	getAvailableCategories : function()
	{
		var activeAvailableCategories = [];
		var categoriesGrid = this.getCategoriesGrid();
		var categories = this.getActiveCategories();		
		
		for (var i = 0; i < categories.length; i++) {
			for(var j = 0; j < categoriesGrid.getStore().getCount(); j++) {
				if(categories[i] == categoriesGrid.getStore().getAt(j).get('category')) {
					activeAvailableCategories.push(categoriesGrid.getStore().getAt(j));
					break;
				}
			}
		}
		return activeAvailableCategories;
	},
	
	/**
	 * return all categories records which are not in the predefined categories
	 * @return {Array} The array of categories
	 * @private
	 */
	getActiveExtraCategories : function()
	{
		var activeExtraCategories = [];
		var categoriesGrid = this.getCategoriesGrid();
		var categories = this.getActiveCategories();		
		
		var isActiveCategory = false;
		for (var i = 0; i < categories.length; i++) {
			isActiveCategory = false;
			for(var j = 0; j < categoriesGrid.getStore().getCount(); j++) {
				if(categories[i] == categoriesGrid.getStore().getAt(j).get('category')) {
					isActiveCategory = true;
					break;
				}
			}
			if(!isActiveCategory) {
				//@TODO create and send a record for future use
				activeExtraCategories.push(categories[i]);
			}
		}
		return Zarafa.core.Util.trimStringArray(activeExtraCategories);
	},

	/**
	 * Event handler which is raised when the current selection of the {@link Ext.grid.GridPanel}
	 * changes. This is used to enable/disable the "Add" button.
	 *
	 * @param {RowSelectionModel} selectionModel The selection model which raised the event
	 * @private
	 */
	onSelectionChange : function(selectionModel)
	{
		this.getCategoriesTextArea().setValue('');
		var records = this.getCategoriesGrid().getSelectionModel().getSelections();
		var categories = [];
		
		Ext.each(records, function(record) {
			categories.push(record.get('category'));
		}, this);
		
		this.updateCategories(categories);
	},

	/**
	 * Update the {@link Ext.form.TextArea textarea} with new categories which were selected
	 * from the {@link Ext.grid.GridPanel grid} or from the old values of the
	 * {@link Zarafa.core.data.IPMRecord records}. All categories will be trimmed to remove
	 * extra spaces. Also the categories will be sorted and duplicates will be removed.
	 *
	 * @param {String/String|Array} newCategories The categories which must be added
	 * to the {@link Ext.form.TextArea textarea}
	 * @private
	 */
	updateCategories : function(newCategories)
	{
			var categoryInput = this.getCategoriesTextArea();
			var categoriesGrid = this.getCategoriesGrid();
			var categoryString = categoryInput.getValue();
			var categories = [];

			if(!Ext.isEmpty(newCategories)){
				if(!Ext.isEmpty(categoryString, false)){
					categories = categoryString.split(this.categorySeparator);
					categories = categories.concat(newCategories);
				} else if (Ext.isArray(newCategories)) {
					categories = newCategories;
				} else {
					categories = [ newCategories ];
				}
		
			}
		
			var extraCategories = this.getActiveExtraCategories();
			if(extraCategories.length){
				for(var i=0;i<extraCategories.length;i++)
					categories.push(extraCategories[i]);
			}
		
			categories = Zarafa.core.Util.trimStringArray(categories);
			categories = Zarafa.core.Util.sortArray(categories, 'ASC', Zarafa.core.Util.caseInsensitiveComparison);
			categories = Zarafa.core.Util.uniqueArray(categories);

			categoryString = categories.join(this.categorySeparator + ' ');
			if (categoryString.length > 0)
				categoryString += this.categorySeparator;

			categoryInput.setValue('');
			categoryInput.setValue(categoryString);
	}
});

Ext.reg('zarafa.categoriespanel', Zarafa.common.categories.dialogs.CategoriesPanel);
Ext.namespace('Zarafa.common.checknames.dialogs');

/**
 * @class Zarafa.common.checknames.dialogs.CheckNamesPanel
 * @extends Ext.Panel
 * @xtype zarafa.checknamespanel
 */
Zarafa.common.checknames.dialogs.CheckNamesPanel = Ext.extend(Ext.Panel, {
	/**
	 * The listView which is used to display all possible names.
	 * @property
	 * @type Ext.list.ListView
	 */
	checkNamesList : undefined,

	/**
	 * @cfg {Ext.data.JsonStore} store The store containing all suggestions
	 */
	store : undefined,

	/**
	 * @constructor
	 * @param config Configuration structure
	 */
	constructor : function(config)
	{
		config = config || {};

		// Prepare the suggestion store
		if (!Ext.isDefined(config.store)) {
			config.store = new Ext.data.JsonStore({
				autoDestroy : true,
				idProperty : 'smtp_address',
				fields : Zarafa.core.data.IPMRecipientResolveRecord,
				data : []
			});
		}

		Ext.applyIf(config, {
			xtype : 'zarafa.checknamespanel',
			layout: {
				type: 'vbox',
				align: 'stretch'
			},
			border: false,
			bodyStyle: 'padding: 5px; background-color: inherit;',
			header: true,
			items: [{
				xtype: 'displayfield',
				value: _('Select an address to use') + ':',
				hideLabel : true
			},{
				xtype : 'grid',
				ref : 'checkNamesList',
				flex: 1,
				store : config.store,
				viewConfig : {
					forceFit : true
				},
				columns: [{
					dataIndex: 'display_name',
					header: _('Name'),
					renderer : Ext.util.Format.htmlEncode
				},{
					dataIndex: 'smtp_address',
					header: _('Email'),
					renderer : Ext.util.Format.htmlEncode
				}],
				listeners : {
					scope : this,
					viewready : this.onViewReady,
					dblclick : this.onDblClick
				},
				selModel : new Ext.grid.RowSelectionModel({
					singleSelect : true
				}),
				cls: 'zarafa-checknames-dialog-filelist'
			}]
		});

		Zarafa.common.checknames.dialogs.CheckNamesPanel.superclass.constructor.call(this, config);
	},

	/**
	 * Load an {@link Zarafa.core.data.IPMRecipientRecord} and
	 * {@link Zarafa.core.data.IPMRecipientResolveRecord ChecknamesRecord} data into this panel.
	 *
	 * @param {Zarafa.core.data.IPMRecipientRecord} record The record for which the checknames dialog is shown
	 * @param {Array} data The {@link Zarafa.core.data.IPMRecipientResolveRecord ChecknamesRecord suggestions}
	 * which exist for the given record.
	 */
	update : function(record, data)
	{
		this.setTitle(String.format(_('More than one \'{0}\' found.'), Ext.util.Format.htmlEncode(record.get('display_name'))));
		this.store.add(data || []);
	},

	/**
	 * Event handler which is fired when the gridPanel is ready. This will automatically
	 * select the first row in the grid.
	 * @private
	 */
	onViewReady: function()
	{
		this.checkNamesList.getSelectionModel().selectFirstRow();
	},

	/**
	 * Update an {@link Zarafa.core.data.IPMRecipientRecord} with the selected
	 * {@link Zarafa.core.data.IPMRecipientResolveRecord record} from the {@link #checkNamesList}.
	 *
	 * @param Zarafa.core.data.IPMRecipientRecord} record The record which must be updated
	 * @return {Boolean} False if the record could not be updated.
	 */
	updateRecord : function(record)
	{
		var selection = this.checkNamesList.getSelectionModel().getSelected();

		if (Ext.isEmpty(selection)) {
			Ext.Msg.alert(_('Alert'), _('Please select a recipient'));
			return false;
		} else {	
			record.applyResolveRecord(selection);
		}
	},

	/**
	 * Event handler which is fired when a row inside the {@link #checkNamesList}
	 * has been double-clicked.
	 * @private
	 */
	onDblClick : function()
	{
		this.dialog.onOk();
	}
});

Ext.reg('zarafa.checknamespanel', Zarafa.common.checknames.dialogs.CheckNamesPanel);
Ext.namespace('Zarafa.common.delegates.dialogs');

/**
 * @class Zarafa.common.delegates.dialogs.DelegatePermissionPanel
 * @extends Ext.form.FormPanel
 * @xtype zarafa.delegatepermissionpanel
 *
 * Will generate UI for {@link Zarafa.common.delegates.dialogs.DelegatePermissionContentPanel DelegatePermissionContentPanel}.
 */
Zarafa.common.delegates.dialogs.DelegatePermissionPanel = Ext.extend(Ext.form.FormPanel, {

	/**
	 * @cfg {Array} folderTypes array of folder type that will be used
	 * generate permissions combo box filed
	 */
	folderTypes : ['calendar','tasks', 'inbox', 'contacts', 'notes','journal'],

	/**
	 * @constructor
	 * @param config Configuration structure
	 */
	constructor : function(config)
	{
		config = config || {};

		config.plugins = Ext.value(config.plugins, []);
		config.plugins.push('zarafa.recordcomponentupdaterplugin');

		Ext.applyIf(config, {
			// Override from Ext.Component
			xtype : 'zarafa.delegatepermissionpanel',
			labelAlign : 'left',
			items : this.createPanelItems()
		});

		Zarafa.common.delegates.dialogs.DelegatePermissionPanel.superclass.constructor.call(this, config);
	},

	/**
	 * Function will create panel items for {@link Zarafa.common.delegates.dialogs.DelegatePermissionPanel DelegatePermissionPanel}
	 * @return {Array} array of items that should be added to panel.
	 * @private
	 */
	createPanelItems : function()
	{
		return [{
			xtype : 'fieldset',
			style : {
				margin : '10px',
				padding : '10px'
			},
			title : _('This delegate has the following permissions'),
			autoHeight : true,
			autoWidth : true,
			cls : 'zarafa-fieldset',
			labelAlign : 'left',
			items : this.createFieldItems(this.folderTypes)
		}, {
			xtype : 'checkbox',
			boxLabel : _('Delegate can see my private items.'),
			style: 'margin-left : 10px;',
			ref : 'delegatePrivateCheck',
			name : 'can_see_private',
			hideLabel : true,
			checked : false,
			listeners : {
				check : this.onPrivateCheck,
				scope : this
			}
		}];
	},

	/**
	 * Generic function to create check box for delegate meeting rules check box.
	 * @return {Obect} config object to create {@link Ext.form.CheckBox CheckBox}.
	 */
	createDelegateMeetingRuleCheck : function()
	{
		return{
			xtype : 'checkbox',
			boxLabel : _('Delegate receives copies of meeting-related messages sent to me.'),
			ref : '../delegateMeetingRuleCheck',
			name : 'has_meeting_rule',
			hideLabel : true,
			checked : false,
			listeners : {
				check : this.onDelegateRuleCheck,
				scope : this
			}
		}
	},

	/**
	 * Generic function to create comboboxes for different permission levels for default folders
	 * of the hierarchy.
	 * @param {Array} type type of the default folder (calendar, inbox, notes etc.)
	 * @return {Array} items array to create a {@link Ext.form.ComboBox ComboBox}.
	 * @private
	 */
	createFieldItems : function(folderTypes)
	{
		var items = [];
		for(var i =0; i < folderTypes.length; i++) {

			var profileStore = {
				xtype : 'jsonstore',
				fields : ['name', 'value'],
				data : Zarafa.common.delegates.data.DelegatePermissionProfiles
			};

			var item = {
				xtype : 'combo',
				name : 'rights_' + folderTypes[i],
				boxMinWidth : 200,
				anchor : '100%',
				fieldLabel : Ext.util.Format.capitalize(folderTypes[i]),
				store : profileStore,
				mode : 'local',
				triggerAction : 'all',
				displayField : 'name',
				valueField : 'value',
				lazyInit : false,
				// "Full Control", "No Rights" etc. folder permissions are not supported
				// by the delegate so we just show the "Other" as text in combo box.
				valueNotFoundText : _('Other'),
				forceSelection : true,
				editable : false,
				value : Zarafa.core.mapi.Rights.RIGHTS_NONE,
				listeners : {
					select : this.onProfileSelect,
					scope : this
				}
			}
			items.push(item);

			if(folderTypes[i] === 'calendar') {
				items.push(this.createDelegateMeetingRuleCheck());
			}
			
		}
		return items
	},

	/**
	 * Updates the panel by loading data from the record into the form panel.
	 * @param {Zarafa.common.delegates.data.DelegateRecord} record The record update the panel with.
	 * @param {Boolean} contentReset force the component to perform a full update of the data.
	 */
	update : function(record, contentReset)
	{
		this.record = record;

		this.getForm().loadRecord(record);

		this.updateUI(record, contentReset);
	},

	/**
	 * Updates the UI of the panel.
	 * @param {Zarafa.common.delegates.data.DelegateRecord} record The record update the panel with.
	 * @param {Boolean} contentReset force the component to perform a full update of the data.
	 */
	updateUI : function(record, contentReset)
	{
		if(contentReset || record.isModifiedSinceLastUpdate('rights_calendar')) {
			var calendarRights = record.get('rights_calendar');
			if(!calendarRights || calendarRights === Zarafa.core.mapi.Rights.RIGHTS_NONE || calendarRights === Zarafa.core.mapi.Rights.RIGHTS_READONLY) {
				this.delegateMeetingRuleCheck.setDisabled(true);
			} else {
				this.delegateMeetingRuleCheck.setDisabled(false);
			}
		}
	},

	/**
	 * Update the given {@link Zarafa.core.data.IPMRecord record} with
	 * the values from this {@link Ext.Panel panel}.
	 * @param {Zarafa.core.data.IPMRecord} record The record to update
	 */
	updateRecord : function(record)
	{
		this.getForm().updateRecord(record);
	},

	/**
	 * Handler function that will be called when user selects permission level for
	 * any default folder type. This is a common function for every combobox.
	 * @param {Ext.form.Combobox} comboBox permission level combobox
	 * @param {Ext.data.Record} record currently selected record in the combobox
	 * @param {Number} index index of the currently selected record in combobox
	 * @private
	 */
	onProfileSelect : function(comboBox, record, index)
	{
		var type = comboBox.name;

		this.record.beginEdit();

		// update the record data so we can use it in next function
		this.record.set(type, record.get(comboBox.valueField));

		if(type === 'rights_calendar') {
			var calendarRights = this.record.get(type);

			if (!calendarRights || calendarRights === Zarafa.core.mapi.Rights.RIGHTS_NONE || calendarRights === Zarafa.core.mapi.Rights.RIGHTS_READONLY) {
				this.record.set('has_meeting_rule', false);
			}

			this.updateUI(this.record, false);
		}

		this.record.endEdit();
	},

	/**
	 * Handler function that will be called when user checks/unchecks checkbox of delegate meeting rule,
	 * it will save the checked value to record.
	 * @param {Ext.form.Checkbox} checkBox checkbox for delegate meeting rule.
	 * @param {Boolean} checked current state of the checkbox.
	 * @private
	 */
	onDelegateRuleCheck : function(checkBox, checked)
	{
		this.record.set('has_meeting_rule', checked);
	},

	/**
	 * Handler function that will be called when user checks/unchecks checkbox of delegate private flag,
	 * it will save the checked value to record.
	 * @param {Ext.form.Checkbox} checkBox checkbox for delegate meeting rule.
	 * @param {Boolean} checked current state of the checkbox.
	 * @private
	 */
	onPrivateCheck : function(checkBox, checked)
	{
		this.record.set('can_see_private', checked);
	}
});

Ext.reg('zarafa.delegatepermissionpanel', Zarafa.common.delegates.dialogs.DelegatePermissionPanel);
Ext.namespace('Zarafa.common.delegates.ui');

/**
 * @class Zarafa.common.delegates.ui.DelegatesGrid
 * @extends Ext.grid.GridPanel
 * @xtype zarafa.delegatesgrid
 *
 * {@link Zarafa.common.delegates.ui.DelegatesGrid DelegatesGrid} will be used to display
 * delegates of the current user.
 */
Zarafa.common.delegates.ui.DelegatesGrid = Ext.extend(Ext.grid.GridPanel, {
	/**
	 * @constructor
	 * @param {Object} config Configuration structure
	 */
	constructor : function(config)
	{
		config = config || {};

		if(!config.store) {
			config.store = new Zarafa.common.delegates.data.DelegateStore();
		}
	
		Ext.applyIf(config, {
			xtype : 'zarafa.delegatesgrid',
			border : true,
			store : config.store,
			viewConfig : {
				forceFit : true,
				emptyText : '<div class=\'emptytext\'>' + _('No delegate exists') + '</div>'
			},
			loadMask : this.initLoadMask(),
			columns : this.initColumnModel(),
			selModel : this.initSelectionModel(),
			listeners : {
				viewready : this.onViewReady,
				rowdblclick : this.onRowDblClick,
				scope : this
			}
		});

		Zarafa.common.delegates.ui.DelegatesGrid.superclass.constructor.call(this, config);
	},

	/**
	 * initialize events for the grid panel.
	 * @private
	 */
	initEvents : function()
	{
		Zarafa.common.delegates.ui.DelegatesGrid.superclass.initEvents.call(this);

		// select first delegate when store has finished loading
		this.mon(this.store, 'load', this.onViewReady, this, {single : true});
	},

	/**
	 * Creates a column model object, used in {@link #colModel} config
	 * @return {Ext.grid.ColumnModel} column model object
	 * @private
	 */
	initColumnModel : function()
	{
		return [{
			dataIndex : 'display_name',
			header : _('Name'),
			renderer : Zarafa.common.ui.grid.Renderers.text
		}]
	},

	/**
	 * Creates a selection model object, used in {@link #selModel} config
	 * @return {Ext.grid.RowSelectionModel} selection model object
	 * @private
	 */
	initSelectionModel : function()
	{
		return new Ext.grid.RowSelectionModel({
			singleSelect : true
		});
	},

	/**
	 * Initialize the {@link Ext.grid.GridPanel.loadMask} field
	 *
	 * @return {Ext.LoadMask} The configuration object for {@link Ext.LoadMask}
	 * @private
	 */
	initLoadMask : function()
	{
		return {
			msg : _('Loading delegates') + '...'
		};
	},

	/**
	 * Event handler which is fired when the gridPanel is ready. This will automatically
	 * select the first row in the grid.
	 * @private
	 */
	onViewReady : function()
	{
		this.getSelectionModel().selectFirstRow();
	},

	/**
	 * Event handler which is fired when the {@link Zarafa.common.delegates.ui.DelegatesGrid DelegatesGrid} is double clicked.
	 * it will call generic function to handle the functionality.
	 * @private
	 */
	onRowDblClick : function(grid, rowIndex)
	{
		this.openDelegatePermissions(grid.getStore().getAt(rowIndex));
	},

	/**
	 * Generic function to open {@link Zarafa.common.delegates.dialogs.DelegatePermissionContentPanel DelegatePermissionContentPanel} for
	 * the user which has been selected.
	 * @param {Zarafa.common.delegates.data.DelegateRecord} delegateRecord record that should be opened in {@link Zarafa.common.delegates.dialogs.DelegatePermissionContentPanel DelegatePermissionContentPanel}.
	 * if not passed then currently selected record will be used.
	 * @param {Object} config configuration object that should be passed to {@link Zarafa.common.delegates.dialogs.DelegatePermissionContentPanel DelegatePermissionContentPanel}.
	 */
	openDelegatePermissions : function(delegateRecord, config)
	{
		config = config || {};

		delegateRecord = delegateRecord || this.getSelectionModel().getSelected();
		if(!delegateRecord) {
			Ext.Msg.alert(_('Alert'), _('Please select a delegate.'));
			return;
		}

		Ext.apply(config, {
			recordComponentPluginConfig : {
				// we will open records ourself
				enableOpenLoadTask : false
			}
		});

		// open detailed permissions content panel
		Zarafa.common.Actions.openDelegatePermissionContent(delegateRecord, config);
	},

	/**
	 * Function will be called to remove a delegate.
	 */
	removeDelegate : function()
	{
		var selectionModel = this.getSelectionModel();
		var delegateRecord = selectionModel.getSelected();

		if(!delegateRecord) {
			Ext.Msg.alert(_('Alert'), _('Please select a delegate.'));
			return;
		}

		// before removing delegate we should select next available delegate,
		// because deleting delegate will remove selection
		if (selectionModel.hasNext()) {
			selectionModel.selectNext();
		} else if (selectionModel.hasPrevious()) {
			selectionModel.selectPrevious();
		}

		this.store.remove(delegateRecord);
	}
});

Ext.reg('zarafa.delegatesgrid', Zarafa.common.delegates.ui.DelegatesGrid);
Ext.namespace('Zarafa.common.delegates.ui');

/**
 * @class Zarafa.common.delegates.ui.DelegatesPanel
 * @extends Ext.Panel
 * @xtype zarafa.delegatespanel
 * Will generate UI for the {@link Zarafa.common.settings.SettingsDelegateWidget SettingsDelegateWidget}.
 */
Zarafa.common.delegates.ui.DelegatesPanel = Ext.extend(Ext.Panel, {
	/**
	 * @cfg {Zarafa.common.delegates.data.DelegateStore} store Delegate store that will be used to load delegates data
	 */
	store : undefined,

	/**
	 * The LoadMask object which will be shown when the {@link #record} is being opened, and
	 * the dialog is waiting for the server to respond with the desired data.
	 * @property
	 * @type Zarafa.common.ui.LoadMask
	 */
	loadMask : undefined,

	/**
	 * @constructor
	 * @param config Configuration structure
	 */
	constructor : function(config)
	{
		config = config || {};

		if(!config.store) {
			config.store = new Zarafa.common.delegates.data.DelegateStore();
		}

		Ext.applyIf(config, {
			// Override from Ext.Component
			xtype : 'zarafa.delegatespanel',
			border : false,
			layout : {
				type : 'vbox',
				align : 'stretch',
				pack  : 'start'
			},
			items : this.createPanelItems(config.store)
		});

		Zarafa.common.delegates.ui.DelegatesPanel.superclass.constructor.call(this, config);
	},

	/**
	 * Function will create panel items for {@link Zarafa.common.delegates.ui.DelegatesPanel DelegatesPanel}
	 * @return {Array} array of items that should be added to panel.
	 * @param {Zarafa.common.delegates.data.DelegateStore} store store that will be used to load delegates data.
	 * @private
	 */
	createPanelItems : function(store)
	{
		return [{
			xtype : 'displayfield',
			value : _('Delegates can send items on your behalf. To grant permission to others to access your folders without also giving them send-on-behalf-of privileges, go to properties for each folder and change the options on the Permissions Tab.'),
			fieldClass : 'x-form-display-field zarafa-settings-widget-extrainfo'
		}, {
			xtype : 'container',
			flex : 1,
			layout : {
				type : 'hbox',
				align : 'stretch',
				pack  : 'start'
			},
			items : [{
				xtype : 'zarafa.delegatesgrid',
				ref : '../delegatesGrid',
				store : store,
				flex : 1
			}, {
				xtype : 'container',
				width : 160,
				defaults : {
					width : 140
				},
				layout : {
					type : 'vbox',
					align : 'center',
					pack  : 'start'
				},
				items : [{
					xtype : 'button',
					text : _('Add') + '...',
					handler : this.onDelegateAdd,
					ref : '../../addButton',
					scope : this
				}, {
					xtype : 'spacer',
					height : 20
				}, {
					xtype : 'button',
					text : _('Remove') + '...',
					disabled : true,
					ref : '../../removeButton',
					handler : this.onDelegateRemove,
					scope : this
				}, {
					xtype : 'spacer',
					height : 20
				}, {
					xtype : 'button',
					text : _('Permission') + '...',
					disabled : true,
					ref : '../../permissionButton',
					handler : this.onDelegatePermission,
					scope : this
				}]
			}]
		}];
	},

	/**
	 * initialize events for the panel.
	 * @private
	 */
	initEvents : function()
	{
		Zarafa.common.delegates.ui.DelegatesPanel.superclass.initEvents.call(this);

		// register event to open permissions dialog after adding new delegate
		this.mon(this.store, 'add', this.afterDelegateAdd, this);

		// register event to enable/disable buttons
		this.mon(this.delegatesGrid.getSelectionModel(), 'selectionchange', this.onGridSelectionChange, this);
	},

	/**
	 * If {@link #showLoadMask} is enabled, this function will display the {@link #loadMask}.
	 * @protected
	 */
	showLoadMask : function()
	{
		if (!this.loadMask) {
			this.loadMask = new Zarafa.common.ui.LoadMask(this.el);
		}

		this.loadMask.show();
	},

	/**
	 * If {@link #showLoadMask} is enabled, and {@link #showLoadMask} has been
	 * called to display the {@link #loadMask} this function will disable the loadMask.
	 * @protected
	 */
	hideLoadMask : function()
	{
		if (this.loadMask) {
			this.loadMask.hide();
		}
	},

	/**
	 * Handler function will be called when user clicks on 'Add' button,
	 * this will show addressbook dialog to select delegate user.
	 * @private
	 */
	onDelegateAdd : function()
	{
		Zarafa.common.Actions.openABUserSelectionContent({
			callback : this.abCallBack,
			scope : this,
			hierarchyRestriction : {
				hide_contacts : true
			},
			listRestriction : {
				hide_users : ['contact', 'system', 'non_security', 'room', 'equipment', 'non_active'],
				hide_groups : ['non_security'],
				hide_companies : true
			}
		});
	},

	/**
	 * Callback function for {@link Zarafa.addressbook.dialogs.ABUserSelectionContentPanel AddressBook}
	 * @param {Ext.data.Record} record user selected from AddressBook
	 * @private
	 */
	abCallBack : function(record)
	{
		record = record.convertToDelegate();

		if(this.store.getById(record.get('entryid'))) {
			Ext.Msg.alert(_('Alert'), _('Delegate already exists in the list.'));
			return;
		}

		this.store.add(record);
	},

	/**
	 * Function will be called when a delegate record is added to the store and we want to
	 * open {@link Zarafa.common.delegates.dialogs.DelegatePermissionContentPanel DelegatePermissionContentPanel}
	 * for the currently added delegate.
	 * @param {Zarafa.common.delegates.data.DelegateStore} store store that was used to add delegate user.
	 * @param {Zarafa.common.delegates.data.DelegateRecord|Array} record array of records that are added in store.
	 * @param {Number} index index of the store at where records are added in store.
	 */
	afterDelegateAdd : function(store, record, index)
	{
		if(Ext.isArray(record)) {
			for(var i = 0, j = record.length; i < j; i++) {
				this.afterDelegateAdd(store, record[i], i);
			}

			return;
		}

		// only open permissions dialog for phantom records
		if(record.phantom) {
			this.openDelegatePermissions(record, true);
		}
	},

	/**
	 * Helper function that will be used when user adds a new {@link Zarafa.common.delegates.data.DelegateRecord DelegateRecord}
	 * so this will check if we need to get folder permissions that is already saved for the current delegate and fill it in delegates record
	 * and then {@link Zarafa.common.delegates.dialogs.DelegatePermissionContentPanel DelegatePermissionContentPanel} can be opened
	 * using the record.
	 * @param {Zarafa.common.delegates.data.DelegateRecord} record delegate record which is added in grid
	 */
	openDelegatePermissions : function(record)
	{
		if(!record.isOpened()) {
			this.openDelegateRecord(record, this.openDelegatePermissions);
			return;
		}

		this.delegatesGrid.openDelegatePermissions(record, { removeRecordOnCancel : true });
	},

	/**
	 * Helper function will get folder permissions already assigned to particular delegate when adding it to the delegates store.
	 * This will also show / hide load mask when requesting more data
	 * @param {Zarafa.common.delegates.data.DelegateRecord} record delegate record for which we need to get existing folder permissions
	 * @param {Ext.Function} callback callback function that will be called after successfully opening the delegate record.
	 */
	openDelegateRecord : function(record, callback)
	{
		// show load mask till we fetch full data from server
		this.showLoadMask();

		// store a reference of record's store which can be used to deregister events in exception handler
		var store = record.getStore();

		var fnOpen = function(recStore, rec) {
			if (record === rec) {
				// remove registered handlers
				store.un('open', fnOpen, this);
				store.un('exception', fnException, this);

				// hide load mask as data has arrived
				this.hideLoadMask();

				// call the callback function
				callback.call(this, record);
			}
		};

		var fnException = function(proxy, type, action, options, response, args) {
			if(action === Ext.data.Api.actions.open) {
				// remove registered handlers, this exception has been fired by Proxy so we will not get store in parameters
				store.un('open', fnOpen, this);
				store.un('exception', fnException, this);

				// actual error message will be shown by common exception handler
				// we will just hide the load mask
				this.hideLoadMask();
			}
		};

		store.on('open', fnOpen, this);
		store.on('exception', fnException, this);

		record.open();
	},

	/**
	 * Event handler will be called when selection in {@link Zarafa.common.delegates.ui.DelegatesGrid DelegatesGrid}
	 * has been changed
	 * @param {Ext.grid.RowSelectionModel} selectionModel selection model that fired the event
	 */
	onGridSelectionChange : function(selectionModel)
	{
		var noSelection = (selectionModel.hasSelection() === false);

		this.removeButton.setDisabled(noSelection);
		this.permissionButton.setDisabled(noSelection);
	},

	/**
	 * Handler function will be called when user clicks on 'Remove' button,
	 * this will remove currently selected delegate from delegates list.
	 * @private
	 */
	onDelegateRemove : function()
	{
		this.delegatesGrid.removeDelegate();
	},

	/**
	 * Handler function will be called when user clicks on 'Permission' button,
	 * this will open {@link Zarafa.common.delegates.dialogs.DelegatePermissionContentPanel DelegatePermissionContentPanel}
	 * to show detailed permissions of delegate user.
	 * @private
	 */
	onDelegatePermission : function()
	{
		this.delegatesGrid.openDelegatePermissions();
	},

	/**
	 * Function will be used to reload data in the {@link Zarafa.common.delegates.data.DelegateStore DelegateStore}.
	 */
	discardChanges : function()
	{
		this.store.load();
	},

	/**
	 * Function will be used to save changes in the {@link Zarafa.common.delegates.data.DelegateStore DelegateStore}.
	 */
	saveChanges : function()
	{
		this.store.save();
	}
});

Ext.reg('zarafa.delegatespanel', Zarafa.common.delegates.ui.DelegatesPanel);
Ext.namespace('Zarafa.common.dialogs');

/**
 * @class Zarafa.common.dialogs.CopyMovePanel
 * @extends Ext.Panel
 * @xtype zarafa.copymovepanel
 *
 * Panel for users to copy or move the given {@link Zarafa.core.data.IPMRecord[] records}
 * to a differnt {@link Zarafa.hierarchy.data.MAPIFolderRecord folder}.
 */
Zarafa.common.dialogs.CopyMovePanel = Ext.extend(Ext.Panel, {
	/**
	 * @cfg {Zarafa.core.data.IPMRecord} record The record(s) which are being
	 * copied or moved through this panel
	 */
	record : undefined,
	/**
	 * @cfg {Zarafa.core.mapi.ObjectType} objectType The Objecttype of the
	 * {@link #record} which have been set on this panel. This is needed
	 * to determine if we are copy/moving folders or messages.
	 */
	objectType : undefined,
	/**
	 * @constructor
	 * @param {Object} config Configuration structure
	 */
	
	/**
	 * {Zarafa.mail.MailStore} store or {Zarafa.hierarchy.data.IPFSubStore} store 
	 * depending on the objectType.
	 * This store is cached in the panel, because the store of a record is removed 
	 * when a user receives new email.
	 */
	store : undefined,

	constructor : function(config)
	{
		config = config || {};

		if (config.record) {
			if (!Ext.isArray(config.record)) {
				config.record = [ config.record ];
				this.store = config.record.getStore();
			} else {
				this.store = config.record[0].getStore();
			}

			if (!config.objectType) {
				config.objectType = config.record[0].get('object_type')
			}
		}

		Ext.applyIf(config, {
			// Override from Ext.Component
			xtype : 'zarafa.copymovepanel',
			layout: {
				type: 'vbox',
				align: 'stretch'
			},
			border: false,
			bodyStyle: 'padding: 5px; background-color: inherit;',
			header: true,
			items: [
				this.createTreePanel()
			],
			buttonAlign: 'left',
			buttons: [{
				text: _('New folder'),
				handler: this.onCreateFolder,
				scope: this,
				ref: '../createFolderButton',
				disabled: true
			},
			'->',
			{
				text: _('Copy'),
				handler: this.onCopy,
				scope: this,
				ref: '../copyButton',
				disabled: true
			},{
				text: _('Move'),
				handler: this.onMove,
				scope: this,
				ref: '../moveButton',
				disabled: true
			},{
				text: _('Cancel'),
				handler: this.onCancel,
				scope: this
			}]
		});

		Zarafa.common.dialogs.CopyMovePanel.superclass.constructor.call(this, config);
	},

	/**
	 * Creates a {@link Zarafa.hierarchy.ui.Tree treepanel}
	 * which contains all the {@link Zarafa.hierarchy.data.MAPIFolderRecord folders}
	 * to which the {@link Zarafa.core.data.IPMRecord records} can be
	 * copied or moved to.
	 * @return {Object} Configuration object for the tree panel.
	 * @private
	 */
	createTreePanel : function()
	{
		return {
			xtype: 'panel',
			layout: 'form',
			border: false,
			flex: 1,
			bodyStyle: 'background-color: inherit;',
			items: [{
				xtype: 'displayfield',
				value: _('Destination folder') + ':',
				hideLabel : true
			},{
				xtype: 'zarafa.hierarchytree',
				border: true,
				enableDD : false,
				treeSorter : true,
				anchor: '100% 90%',
				ref: '../hierarchyTree',
				treeSorter : true
			}]
		};
	},

	/**
	 * Function which is called automatically by ExtJs when the {@link Ext.Panel panel}
	 * is being rendered. This will add the event handler for selection changes, and
	 * will load the hierarchy model.
	 * @param {Ext.Container} ct The parent container for this panel
	 * @param {Number} position The position of this panel inside its parent
	 * @private
	 */
	onRender : function(ct, position)
	{
		Zarafa.common.dialogs.CopyMovePanel.superclass.onRender.call(this, ct, position);

		if (this.objectType == Zarafa.core.mapi.ObjectType.MAPI_MESSAGE) {
			this.setTitle(String.format(ngettext('There is {0} message selected.', 'There are {0} messages selected.', this.record.length), this.record.length));
		} else if (this.objectType == Zarafa.core.mapi.ObjectType.MAPI_FOLDER) {
			this.setTitle(String.format(_('Folder \'{0}\' selected.'), Ext.util.Format.htmlEncode(this.record[0].getDisplayName())));
		}
	},

	/**
	 * Initialize the event handlers
	 * @protected
	 */
	initEvents : function()
	{
		Zarafa.common.dialogs.CopyMovePanel.superclass.initEvents.apply(this, arguments);

		// If there is a folder we should select, the enable the 'load' event handler
		// as we will have to wait until the correct node has been loaded.
		var folder = this.dialog.getSelectedFolder();
		if (folder) {
			this.mon(this.hierarchyTree, 'load', this.onTreeNodeLoad, this);
		}
		this.mon(this.hierarchyTree.getSelectionModel(), 'selectionchange', this.onSelectionChange, this);
	},

	/**
	 * Fired when the {@link Zarafa.hierarchy.ui.Tree Tree} fires the {@link Zarafa.hierarchy.ui.Tree#load load}
	 * event. This function will try to select the {@link Ext.tree.TreeNode TreeNode} in
	 * {@link Zarafa.hierarchy.ui.Tree Tree} intially. When the given node is not loaded yet, it will try again
	 * later when the event is fired again.
	 *
	 * @private
	 */
	onTreeNodeLoad : function() {
		// Select folder in hierarchy tree.
		var folder = this.dialog.getSelectedFolder();

		// If the folder could be selected, then unregister the event handler.
		if (this.hierarchyTree.selectFolderInTree(folder)) {
			this.mun(this.hierarchyTree, 'load', this.onTreeNodeLoad, this);
		}
	},

	/**
	 * Event handler which is trigggered when the user select a {@link Zarafa.hierarchy.data.MAPIFolderRecord folder}
	 * from the {@link Zarafa.hierarchy.ui.Tree tree}. This will determine if a valid
	 * {@link Zarafa.hierarchy.data.MAPIFolderRecord folder} is selected to which the {@link Zarafa.core.data.IPMRecord records}
	 * can indeed be copied or moved to.
	 * @param {DefaultSelectionModel} selectionModel The selectionModel for the treepanel
	 * @param {TreeNode} node The selected tree node
	 * @private
	 */
	onSelectionChange : function(selectionModel, node)
	{
		if (!Ext.isDefined(node) || (node.getFolder().isIPMSubTree() && this.objectType == Zarafa.core.mapi.ObjectType.MAPI_MESSAGE)) {
			this.copyButton.disable();
			this.moveButton.disable();
			this.createFolderButton.disable();
		} else {
			this.copyButton.enable();
			this.moveButton.enable();
			this.createFolderButton.enable();
		}
	},

	/**
	 * Event handler which is triggered when the user presses the Create new folder {@link Ext.Button button}. 
	 * This will call {@link Zarafa.hierachy.actions.openCreateFolderContent} with the selected {@link Zarafa.hierarchy.data.MAPIFolderRecord folder}.
	 * @private
	 */
	onCreateFolder : function()
	{
		var folder = this.hierarchyTree.getSelectionModel().getSelectedNode().getFolder();
		Zarafa.hierarchy.Actions.openCreateFolderContent(folder);
		this.mon(this.hierarchyTree, 'append', this.onTreeAppend, this, {delay: 10});
	},

	/**
	 * Event handler which is triggered when a new folder is appended to the tree and selects the newly created folder
	 * @param {Zarafa.hierarchy.ui.Tree} tree the folder tree
	 * @param {Ext.data.Node} parent the parent of the newly created node
	 * @param {Ext.data.Node} node the appended node
	 * @private
	 */
	onTreeAppend: function(tree, parent, node)
	{
		// Sometimes the 'append' is fired but the node is not rendered yet,so add a delay of 10 ms.
		if (!node.parentNode) {
			// The node is probably removed and appended again, so let's find the
			// correct node in the tree again 
			node = tree.getNodeById(node.id); 
		} 
		tree.selectPath(node.getPath()); 
		this.mun(this.hierarchyTree, 'append', this.onTreeAppend, this);
	},

	/**
	 * Event handler which is triggered when the user presses the Copy
	 * {@link Ext.Button button}. This will copy all {@link Zarafa.core.data.IPMRecord records}
	 * and will close the {@link Zarafa.common.dialogs.CopyMovePanel dialog} when it is done.
	 * @private
	 */
	onCopy : function()
	{
		var folder = this.hierarchyTree.getSelectionModel().getSelectedNode().getFolder();
		var records = this.record;

		if (!Ext.isDefined(folder)) {
			return;
		}

		if (Ext.isEmpty(this.record)) {
			return;
		}


		Ext.each(records, function(record, index) {
			// When we have this panel open and we receive a new email, the records store is
			// not accessible anymore, so we need to get a new record by the entryid of the old record.
			if(this.objectType === Zarafa.core.mapi.ObjectType.MAPI_MESSAGE && !record.getStore()) {
				record = records[index] = this.store.getById(record.id);
			}
			record.copyTo(folder);
		}, this);

		this.dialog.selectFolder(folder);

		this.store.save(records);

		this.dialog.close();
	},

	/**
	 * Event handler which is triggered when the user presses the Move
	 * {@link Ext.Button button}. This will move all {@link Zarafa.core.data.IPMRecord records}
	 * and will close the {@link Zarafa.common.dialogs.CopyMovePanel dialog} when it is done.
	 * @private
	 */
	onMove : function()
	{
		var folder = this.hierarchyTree.getSelectionModel().getSelectedNode().getFolder();
		var records = this.record;

		if (!Ext.isDefined(folder)) {
			return;
		}

		if (Ext.isEmpty(this.record)) {
			return;
		}


		Ext.each(records, function(record, index) {
			// When we have this panel open and we receive a new email, the records store is
			// not accessible anymore, so we need to get a new record by the entryid of the old record.
			if(this.objectType === Zarafa.core.mapi.ObjectType.MAPI_MESSAGE && !record.getStore()) {
				record = records[index] = this.store.getById(record.id);
			}
			record.moveTo(folder);
		}, this);

		this.dialog.selectFolder(folder);

		this.store.save(records);

		this.dialog.close();
	},

	/**
	 * Event handler which is triggered when the user presses the cancel
	 * {@link Ext.Button button}. This will close the {@link Zarafa.common.dialogs.CopyMovePanel dialog}
	 * without copying or moving any {@link Zarafa.core.data.IPMRecord records}.
	 * @private
	 */
	onCancel : function()
	{
		this.dialog.close();
	}
});

Ext.reg('zarafa.copymovepanel', Zarafa.common.dialogs.CopyMovePanel);
Ext.namespace('Zarafa.common.dialogs');

/**
 * @class Zarafa.common.dialogs.MessageBox
 * @extends Ext.MessageBox
 *
 * Extension to the default {@link Ext.MessageBox MessageBox} which
 * offers some special features like displaying a selection list.
 */
// Use Ext.apply instead of Ext.extend because Ext.MessageBox is not
// a class but an instantiated object. The Ext.apply({}, a, b) trick
// is used to create a full copy of Ext.MessageBox instead of changing
// the existing object.
Zarafa.common.dialogs.MessageBox = Ext.apply({}, Ext.MessageBox, {
	/**
	 * The custom buttons which are added in {@link Ext.MessageBox messagebox}.
	 * it will be removed when {@link Ext.MessageBox messagebox} gets hide.
	 * 
	 * @property
	 * @type Array
	 * @private
	 */
	customButton : undefined,

	/**
	 * Initialize the {@link Ext.MessageBox.dlg Dialog}.
	 * Because the {@link Ext.MessageBox MessageBox} hides the usable
	 * interface from use, we must apply a hack to access the Dialog
	 * before it is displayed to the user.
	 *
	 * This function will add a list of {@link Ext.Component Components}
	 * to the dialog, which can be used to fine-tune the look&feel.
	 *
	 * @param {Array} items The array of items which must be added to the
	 * MessageBox.
	 * @private
	 */
	initDialog : function(items)
	{
		var dlg = Ext.MessageBox.getDialog();

		// Automatically remove all items which we had added.
		// This makes sure we can use the same Dialog multiple times
		dlg.on('hide', function(dlg) {
			dlg.removeAll();
		}, dlg, {single: true});

		// In case the 'hide' event was not fired,
		// we also listen to the destroy event as fallback.
		dlg.on('destroy', function(dlg) {
			dlg.removeAll();
		}, dlg, {single: true});

		// Before showing the dialog, we must first
		// add all items to the dialog.
		dlg.on('beforeshow', function(dlg) {
			dlg.add(items);
		}, dlg, {single: true});
	},

	/**
	 * Displays a message box with OK and Cancel buttons prompting the user to make a selection
	 * from a list of {@link Ext.form.Radio radio} buttons.
	 * If a callback function is passed it will be called after the user
	 * clicks either button, and the id of the button that was clicked (could also be the top-right
	 * close button) and the selected {@link Ext.form.Radio radio} as the two parameters to the callback.
	 * @param {String} title The title bar text
	 * @param {String} msg The message box body text
	 * @param {Function} fn (optional) The callback function invoked after the message box is closed
	 * @param {Object} scope (optional) The scope (<code>this</code> reference) in which the callback is executed. Defaults to the browser wnidow.
	 * @param {Array} selections (optional) Array of configuration objects for the {@link Ext.form.Radio radios} from which the user can select.
	 * @param {String} value (optional) Default value for the {@link Ext.form.RadioGroup radiogroup}.
	 * @return {Zarafa.common.dialogs.RadioMessageBox} this
	 */
	select : function(title, msg, fn, scope, selections, value)
	{
		var label = Ext.create({
			xtype: 'displayfield',
			autoHeight: true,
			autoWidth: true,
			value: msg,
			hideLabel : true,
			htmlEncode : true
		});
		var radioGroup = Ext.create({
			xtype: 'radiogroup',
			hideLabel: true,
			style: 'padding-left: 50px;',
			columns: 1,
			items: selections,
			value: value
		});

		this.initDialog([{
			xtype: 'container',
			anchor: '100% 100%',
			items: [ label, radioGroup ]
		}]);

		Ext.MessageBox.show({
			title : title,
			buttons: Ext.MessageBox.OKCANCEL,
			fn: function(button) {
				Ext.callback(fn, scope || window, [button, radioGroup.getValue()], 1);
			},
			minWidth: Ext.MessageBox.minPromptWidth,
			scope : scope,
			prompt: false,
			value: value
		});

		return this;
	},

	/**
	 * Display {@link Ext.MessageBox messagebox} with custom buttons.
	 * @param {Object} config The config contains the configuration
	 * options of message box as wall as custom buttons.
	 */
	addCustomButtons : function(config)
	{
		var dlg = Ext.MessageBox.getDialog();
		this.customButton = dlg.getFooterToolbar().add(config.customButton);
		Ext.MessageBox.show(config);

		if(!Ext.isEmpty(this.customButton)) {
			Ext.each(this.customButton, function(button) {
				dlg.mon(button, 'click', Ext.createDelegate(this.onButtonClick, config.scope, [config.fn], true), this);
			}, this);
		}

		dlg.on('hide', this.onDestroy, this, {single : true});
		dlg.on('destroy', this.onDestroy, this, {single : true});
	},

	/**
	 * Event handler triggered when custom button is clicked. 
	 * it will hide the {@link Ext.MessageBox messagebox}.
	 * @param {Ext.Button}  button The button which user pressed.
	 * @param {Ext.EventObject} event the event object
	 * @parma {Function} callback The callback function to call when button is pressed.
	 */
	onButtonClick : function(button, event, callback)
	{
		var buttonName = button.name || 'cancel';
		Ext.MessageBox.hide();
		callback.call(this, buttonName);
	},

	/**
	 * Event handler which is triggered when {@link Ext.MessageBox messagebox} gets hide.
	 * also it will remove all custom buttons from message box.
	 * @param {Ext.Window} dlg The window
	 */
	onDestroy : function(dlg)
	{
		if(!Ext.isEmpty(this.customButton)) {
			for(var i = 0; i < this.customButton.length; i++) {
				dlg.getFooterToolbar().remove(this.customButton[i]);
			}
			this.customButton = [];
		}
	}
});
Ext.namespace('Zarafa.common.form');

/**
 * @class Zarafa.common.form.TextArea
 * @extends Ext.form.TextArea
 * @xtype zarafa.textarea
 *
 * The Textarea which extends{@link Ext.form.TextArea} and adds extra functionality
 * like inserting text at cursor position.
 */
Zarafa.common.form.TextArea = Ext.extend(Ext.form.TextArea, {

	/**
	 * @cfg {Boolean} enableSystemContextMenu Enable the browser's default contextmenu
	 * to be opened on this {@link #el element}.
	 */
	enableSystemContextMenu : false,

	/**
	 * Called during {@link #render}. If {@link #enableSystemContextMenu} is eanbled, this
	 * will apply the 'zarafa-contextmenu-enabled' CSS class on the {@link #el element}.
	 * @param {Ext.Container} ct The container in which the component is being rendered
	 * @param {Number} position The position inside the container where the component is being rendered
	 * @private
	 */
	onRender : function(ct, position)
	{
		Zarafa.common.form.TextArea.superclass.onRender.apply(this, arguments);

		if (this.enableSystemContextMenu) {
			this.el.addClass('zarafa-contextmenu-enabled');
		}
	},

	/**
	 * Function adds passed text in textarea at cursor's position.
	 * @param {String} text The text value which you want to add in text area
	 */
	insertAtCursor : function(text) {
		if (Ext.isIE) {
			this.el.focus();
			var sel = document.selection.createRange();
			sel.text = text;
			sel.moveEnd('character',text.length);
			sel.moveStart('character',text.length);
		}else{
			var startPos = this.el.dom.selectionStart;
			var endPos = this.el.dom.selectionEnd;
			this.el.dom.value = this.el.dom.value.substring(0, startPos)
				+ text
				+ this.el.dom.value.substring(endPos, this.el.dom.value.length);

			this.el.focus();
			this.el.dom.setSelectionRange(endPos + text.length, endPos + text.length);
		}
	},

	/**
	 * Function sets the cursor position to the start of the text
	 */
	setCursorLocation : function()
	{
		var textAreaDom = this.getEl().dom;

		// When a user replies to an email and the textarea contains text then by default 
		// the cursor is set to the end of the text. The following code sets the cursor position to
		// the start of the text.
		var textLen = Ext.isIE ? textAreaDom.innerText.length : textAreaDom.textLength;
		var startPos = textAreaDom.selectionStart;
		var endPos = textAreaDom.selectionEnd;

		if(startPos === textLen && endPos === textLen) {
			textAreaDom.setSelectionRange(0,0);
		}
	}
});

Ext.reg('zarafa.textarea', Zarafa.common.form.TextArea);
Ext.namespace('Zarafa.common.freebusy.data');

/**
 * @class Zarafa.common.freebusy.data.FreebusyModel
 * @extends Ext.util.Observable
 */
Zarafa.common.freebusy.data.FreebusyModel = Ext.extend(Ext.util.Observable,
{
	/**
	 * @cfg {Boolean} nonWorkingHoursHidden
	 * The {@link Zarafa.common.freebusy.ui.TimelineView TimelineView} must only
	 * display the working hours for the user.
	 */
	nonWorkingHoursHidden : true,
	/**
	 * @cfg {Zarafa.core.DateRange} daterange
	 * The {@link Zarafa.core.DateRange} object that determines what time period will be shown
	 * (defaults to 7 days before today till 3 weeks after).
	 */
	daterange: null,
	/**
	 * @cfg {Zarafa.core.DateRange} selectorRange
	 * The {@link Zarafa.core.DateRange} object that determines what time period will be selected
	 * (defaults to next whole or half hour and will last by default for 30 minutes).
	 */
	selectorRange: null,
	/**
	 * The {@link Zarafa.core.DateRange} object that indicates the daterange for which the
	 * suggestions for the {@link #suggestionBlockStore} must be calculated.
	 * @property
	 * @type Zarafa.common.freebusy.data.FreebusyBlockStore
	 */
	suggestionRange: null,
	/**
	 * @cfg {Zarafa.core.data.IPMRecipientStore} userStore
	 * The {@link Zarafa.core.data.IPMRecipientStore store} object that handles the freebusy blocks displayed on the timeline.
	 */
	userStore: null,
	/**
	 * @cfg {Zarafa.common.freebusy.data.FreebusyBlockStore} blockStore
	 * The {@link Zarafa.common.freebusy.data.FreebusyBlockStore store} object that handles the freebusy blocks displayed on the timeline.
	 */
	blockStore: null,
	/**
	 * @cfg {Zarafa.common.freebusy.data.FreebusyBlockStore} sumBlockStore
	 * The {@link Zarafa.common.freebusy.data.FreebusyBlockStore} object that handles the freebusy sum blocks displayed in the header of the timeline.
	 */
	sumBlockStore: null,
	/**
	 * freeBlockStore Internal store which keeps track of all blocks which mark
	 * the periods in which no user is occupied. This is used to generate the {@link #suggestionBlockStore}.
	 * @property
	 * @type Zarafa.common.freebusy.data.FreebusyBlockStore
	 */
	freeBlockStore: null,
	/**
	 * @cfg {Zarafa.common.freebusy.data.FreebusyBlockStore} suggestionBlockStore
	 * The {@link Zarafa.common.freebusy.data.FreebusyBlockStore} object that handles the freebusy sum blocks which indicates which blocks are
	 * available for new meetings.
	 */
	suggestionBlockStore: null,
	/**
	 * This property is used to prevent the firing of the selectorrangeupdate event. When an outside
	 * source changes the selector range's values this model does not fire that event.
	 * @property
	 * @type Boolean
	 */
	updatingSelectorRangeExternally: false,
	/**
	 * @constructor
	 * @param {Object} config Configuration object
	 */
	constructor: function(config)
	{
		config = config || {};

		// Use 12:00 as base time, this prevents problems for DST switches
		// at 00:00 like in Brasil.
		var now = new Date();
		now.setHours(12);

		var server = container.getServerConfig();
		var startOffset = server.getFreebusyLoadStartOffset();
		var startDate = now.add(Date.DAY, -startOffset).clearTime();
		var endOffset = server.getFreebusyLoadEndOffset();
		var endDate = now.add(Date.DAY, endOffset).clearTime();

		var selectStart, selectEnd;
		if(config.selectorRange){
			// If selectorRange has been supplied copy the startdate and duedate values.
			selectStart = config.selectorRange.getStartDate();
			selectEnd = config.selectorRange.getDueDate();
		}else{
			// Defaults to next whole or half hour and will last by default for 30 minutes
			selectStart = new Date().ceil(Date.MINUTE, 30);
			selectEnd = selectStart.add(Date.MINUTE, 30);
		}

		Ext.applyIf(config, {
			userStore: config.userStore || new Zarafa.core.data.IPMRecipientStore(),
			daterange: config.daterange || new Zarafa.core.DateRange({ startDate : startDate, dueDate : endDate}),
			selectorRange: new Zarafa.core.DateRange({ startDate : selectStart, dueDate : selectEnd}),
			blockStore: config.blockStore || new Zarafa.common.freebusy.data.FreebusyBlockStore(),
			sumBlockStore: config.sumBlockStore || new Zarafa.common.freebusy.data.FreebusyBlockStore({ remoteSort: false}),
			suggestionBlockStore: config.suggestionBlockStore || new Zarafa.common.freebusy.data.FreebusyBlockStore({ remoteSort: false})
		});

		Ext.apply(this, config);

		this.addEvents(
			/**
			 * @event showworkinghourschange
			 * Fires when the {@link #nonWorkingHoursHidden} has been changed
			 * @param {Boolean} hideNonWorkingHours True if only working hours
			 * should be shown.
			 */
			'showworkinghourschange',
			/**
			 * @event userstorechange
			 * Fires when the user store is changed.
			 * @param {Zarafa.core.data.IPMRecipientStore} newStore The new store
			 * @param {Zarafa.core.data.IPMRecipientStore} oldStore The old store
			 */
			'userstorechange',
			/**
			 * @event blockstorechange
			 * Fires when the block store is changed.
			 * @param {Zarafa.common.freebusy.data.FreebusyBlockStore} newStore The new store
			 * @param {Zarafa.common.freebusy.data.FreebusyBlockStore} oldStore The old store
			 */
			'blockstorechange',
			/**
			 * @event sumblockstorechange
			 * Fires when the sumblock store is changed.
			 * @param {Zarafa.common.freebusy.data.FreebusyBlockStore} newStore The new store
			 * @param {Zarafa.common.freebusy.data.FreebusyBlockStore} oldStore The old store
			 */
			'sumblockstorechange',
			/**
			 * @event suggestionblockstorechange
			 * Fires when the suggestionblock store is changed.
			 * @param {Zarafa.common.freebusy.data.FreebusyBlockStore} newStore The new store
			 * @param {Zarafa.common.freebusy.data.FreebusyBlockStore} oldStore The old store
			 */
			'suggestionblockstorechange',
			/**
			 * @event daterangechange
			 * Fires when the {@link Zarafa.core.DateRange DateRange} has been changed.
			 * @param {Zarafa.core.DateRange} newRange The new DateRange object.
			 * @param {Zarafa.core.DateRange} oldRange The old DateRange object.
			 */
			'daterangechange',
			/**
			 * @event selectorrangeupdate
			 * Fires when the selector {@link Zarafa.core.DateRange DateRange}
			 * has been changed.
			 * @param {Zarafa.core.DateRange} newRange The new DateRange object.
			 * @param {Zarafa.core.DateRange} oldRange The old DateRange object.
			 */
			'selectorrangeupdate'
		);

		Zarafa.common.freebusy.data.FreebusyModel.superclass.constructor.call(this, config);

		this.initEvents();

		this.setDateRange(config.daterange, true);
		this.setUserStore(config.userStore, true);
		this.setSelectorRange(config.selectorRange, true);
		this.setBlockStore(config.blockStore, true);
		this.setSumBlockStore(config.sumBlockStore, true);
		this.setSuggestionBlockStore(config.suggestionBlockStore, true);

		this.createFreeBlockStore();
		this.createSuggestionRange();
	},

	/**
	 * Initialize all {@link Zarafa.common.freebusy.data.FreebusyModel FreebusyModel} related events.
	 * @private
	 */
	initEvents: function()
	{
		this.on('showworkinghourschange', this.onShowWorkingHoursChange, this);
		this.on('userstorechange', this.onUserStoreChange, this);
	},

	/**
	 * Create a {@link Zarafa.common.freebusy.data.FreebusyBlockStore FreebusyBlockStore} which
	 * is used internally to keep track of the TimeBlocks in which all users are available.
	 * @return {Zarafa.common.freebusy.data.FreebusyBlockStore} The created store
	 * @private
	 */
	createFreeBlockStore : function()
	{
		if (!this.freeBlockStore)
			this.freeBlockStore = new Zarafa.common.freebusy.data.FreebusyBlockStore({ remoteSort: false});
		return this.freeBlockStore;
	},

	/**
	 * Create a {@link Zarafa.core.DateRange DateRange} object which is used internally to keep
	 * track of the range for which the suggestions must be calculated.
	 * @return {Zarafa.core.DateRange} The created daterange
	 * @private
	 */
	createSuggestionRange : function()
	{
		if (!this.suggestionRange) {
			this.suggestionRange = new Zarafa.core.DateRange();
			this.setSuggestionDate(this.selectorRange.getStartDate());
			this.suggestionRange.on('update', this.onSuggestionRangeUpdate, this, { buffer: 5 });
		}
		return this.suggestionRange;
	},

	/**
	 * Get the visibility of the non-working hours.
	 * @return {Boolean} True if the non-working hours are hidden
	 */
	showOnlyWorkingHours : function()
	{
		return this.nonWorkingHoursHidden;
	},

	/**
	 * Set the visibility of the non-working hours.
	 * @param {Boolean} hide True to hide the non-working hours
	 */
	hideNonWorkingHours : function(hide)
	{
		var oldNonWorkingHoursHidden = this.nonWorkingHoursHidden;
		this.nonWorkingHoursHidden = hide;
		if (oldNonWorkingHoursHidden != this.nonWorkingHoursHidden)
			this.fireEvent('showworkinghourschange', this.nonWorkingHoursHidden);
		return this.nonWorkingHoursHidden;
	},

	/**
	 * Returns the User store.
	 * @return {Zarafa.core.data.IPMRecipientStore} store
	 */
	getUserStore: function()
	{
		return this.userStore;
	},

	/**
	 * Sets the store.
	 * @param {Zarafa.core.data.IPMRecipientStore} store store
	 * @param {Boolean} initial (optional) True if this function is called
	 * during initialization.
	 * @return {Zarafa.core.data.IPMRecipientStore} store
	 */
	setUserStore: function(store, initial)
	{
		if (initial !== true && this.userStore === store) {
			return;
		}

		var oldUserStore = this.userStore;
		if (this.userStore) {
			this.userStore.un('resolved', this.onUserResolved, this);
			this.userStore.un('load', this.onUserLoad, this);
			this.userStore.un('add', this.onUserAdd, this);
			this.userStore.un('remove', this.onUserRemove, this);
			this.userStore.un('clear', this.onUserClear, this);
		}

		this.userStore = Ext.StoreMgr.lookup(store);

		if (this.userStore) {
			this.userStore.on({
				scope: this,
				resolved: this.onUserResolved,
				load: this.onUserLoad,
				add: this.onUserAdd,
				remove: this.onUserRemove,
				clear: this.onUserClear
			});
		}

		this.fireEvent('userstorechange', this.userStore, oldUserStore);
		if (this.userStore !== oldUserStore && oldUserStore && oldUserStore.autoDestroy)
			oldUserStore.destroy();

		return this.userStore;
	},

	/**
	 * Returns the block store.
	 * @return {Zarafa.common.freebusy.data.FreebusyBlockStore} store
	 */
	getBlockStore: function()
	{
		return this.blockStore;
	},

	/**
	 * Sets the store.
	 * @param {Zarafa.common.freebusy.data.FreebusyBlockStore} store
	 * @param {Boolean} initial (optional) True if this function is called
	 * during initialization.
	 * @return {Zarafa.common.freebusy.data.FreebusyBlockStore} store
	 */
	setBlockStore: function(store, initial)
	{
		if (initial !== true && this.blockStore === store) {
			return;
		}

		var oldBlockStore = this.blockStore;
		if (this.blockStore) {
			this.blockStore.un('load', this.onBlockLoad, this);
			this.blockStore.un('remove', this.onBlockRemove, this, { buffer : 100 });
			this.blockStore.un('clear', this.onBlockRemove, this);
		}

		this.blockStore = Ext.StoreMgr.lookup(store);

		if (this.blockStore) {
			this.blockStore.on('load', this.onBlockLoad, this);
			this.blockStore.on('remove', this.onBlockRemove, this, { buffer : 100 });
			this.blockStore.on('clear', this.onBlockRemove, this);
		}

		this.fireEvent('blockstorechange', this.blockStore, oldBlockStore);
		if (this.blockStore !== oldBlockStore && oldBlockStore && oldBlockStore.autoDestroy)
			oldBlockStore.destroy();

		return this.blockStore;
	},

	/**
	 * Returns the block store.
	 * @return {Zarafa.common.freebusy.data.FreebusyBlockStore} store
	 */
	getSumBlockStore : function()
	{
		return this.sumBlockStore;
	},

	/**
	 * Sets the block store.
	 * @param {Zarafa.common.freebusy.data.FreebusyBlockStore} store The sum block store
	 * @param {Boolean} initial (optional) True if this function is called
	 * during initialization.
	 * @return {Zarafa.common.freebusy.data.FreebusyBlockStore} The sum block store
	 * @private
	 */
	setSumBlockStore : function(store, initial)
	{
		if (initial !== true && this.sumBlockStore === store) {
			return;
		}

		var oldSumBlockStore = this.sumBlockStore;

		if (this.sumBlockStore) {
			this.sumBlockStore.un('load', this.onSumBlockLoad, this, { buffer: 5 });
		}

		this.sumBlockStore = Ext.StoreMgr.lookup(store);

		if (this.sumBlockStore) {
			this.sumBlockStore.on('load', this.onSumBlockLoad, this, { buffer: 5 });
		}

		this.fireEvent('sumblockstorechange', this.sumBlockStore, oldSumBlockStore);
		if (this.sumBlockStore !== oldSumBlockStore && oldSumBlockStore && oldSumBlockStore.autoDestroy)
			oldSumBlockStore.destroy();

		return this.sumBlockStore;
	},

	/**
	 * Returns the block store.
	 * @return {Zarafa.common.freebusy.data.FreebusyBlockStore} store
	 */
	getSuggestionBlockStore : function()
	{
		return this.suggestionBlockStore;
	},

	/**
	 * Sets the block store.
	 * @param {Zarafa.common.freebusy.data.FreebusyBlockStore} store The free block store
	 * @param {Boolean} initial (optional) True if this function is called
	 * during initialization.
	 * @return {Zarafa.common.freebusy.data.FreebusyBlockStore} The free block store
	 * @private
	 */
	setSuggestionBlockStore : function(store, initial)
	{
		if (initial !== true && this.suggestionBlockStore === store) {
			return;
		}

		var oldSuggestionBlockStore = this.suggestionBlockStore;

		this.suggestionBlockStore = Ext.StoreMgr.lookup(store);

		this.fireEvent('suggestionblockstorechange', this.suggestionBlockStore, oldSuggestionBlockStore);
		if (this.suggestionBlockStore !== oldSuggestionBlockStore && oldSuggestionBlockStore && oldSuggestionBlockStore.autoDestroy)
			oldSuggestionBlockStore.destroy();

		return this.suggestionBlockStore;
	},

	/**
	 * This returns the number of users inside the {@link #userStore}.
	 * @return {Number} The number of users inside the userStore.
	 */
	getUserCount : function()
	{
		return this.userStore.getCount();
	},

	/**
	 * Adds user to the user list. The freebusy information is automatically requested when the user is resolved.
	 * @param {String} name name of the user that will be resolved and added to the list
	 * @param {Zarafa.core.mapi.RecipientType} type (optional) The recipientType for the user (defaults to MAPI_TO)
	 * @return {Zarafa.core.data.IPMRecipientRecord} the record representing the newly added user.
	 */
	addUser: function(name, type)
	{
		type = Ext.isNumber(type) ? type : Zarafa.core.mapi.RecipientType.MAPI_TO;

		var record = Zarafa.core.data.RecordFactory.createRecordObjectByCustomType(Zarafa.core.data.RecordCustomObjectType.ZARAFA_RECIPIENT, {
			display_name : name,
			recipient_type : type
		});

		// Adding record to userlist store to update the userlist
		this.userStore.add(record);

		return record;
	},

	/**
	 * Returns the daterange
	 * @return {Zarafa.util.DateRange} Daterange set in this model
	 */
	getDateRange: function()
	{
		return this.daterange;
	},

	/**
	 * Sets the date range.
	 * @param {Zarafa.util.DateRange} dateRange Daterange set for this model
	 * @param {Boolean} initial (optional) True if this function is called
	 * during initialization.
	 * @return {Zarafa.util.DateRange} Daterange set for this model
	 */
	setDateRange: function(dateRange, initial)
	{
		if (initial !== true && this.daterange === dateRange) {
			return;
		}

		var oldDateRange = this.daterange
		this.daterange = dateRange;
		this.fireEvent('daterangechange', this.daterange, oldDateRange);
		return this.daterange;
	},

	/**
	 * Returns the selector range
	 * @return {Zarafa.util.DateRange} Daterange set for the selector in this model
	 */
	getSelectorRange: function()
	{
		return this.selectorRange;
	},

	/**
	 * Sets the selector range.
	 * @param {Zarafa.util.DateRange} selectorRange Daterange set for the selector in this model
	 * @param {Boolean} initial (optional) True if this function is called
	 * during initialization.
	 * @return {Zarafa.util.DateRange} Daterange set for the selector in this model
	 * @private
	 */
	setSelectorRange: function(selectorRange, initial)
	{
		if (initial !== true && this.selectorRange === selectorRange) {
			return;
		}

		if (this.selectorRange) {
			this.selectorRange.un('update', this.onSelectorRangeUpdate, this, { buffer: 5 });
		}

		this.selectorRange = selectorRange;

		if (this.selectorRange) {
			this.selectorRange.on('update', this.onSelectorRangeUpdate, this, { buffer: 5 });
		}

		return this.selectorRange;
	},

	/**
	 * Modifies the selected range in the timeline by changing the selectorRange.
	 * @param {Date} startDate Start date
	 * @param {Date} dueDate Due date
	 */
	selectRange: function(startDate, dueDate){
		/* Set a temporary state to disable the firing of the selectorrangeupdate event. When the
		 * start and due date are equal to those currently set in the selectorRange the DateRange
		 * will not fire an update. This temporary state is only reset to false when this update is
		 * fired. To prevent problems with this we check whether the selectorRange will fire an
		 * update.
		 */
		if(!this.selectorRange.equals( new Zarafa.core.DateRange({ startDate : startDate, dueDate : dueDate}) )){
			this.updatingSelectorRangeExternally = true;
		}
		this.selectorRange.set(startDate, dueDate);
	},

	/**
	 * Returns the suggestion range
	 * @return {Zarafa.util.DateRange} Daterange set for the suggestions in this model
	 */
	getSuggestionRange: function()
	{
		return this.suggestionRange;
	},

	/**
	 * Set the date for which the suggestionlist must be calculated.
	 * This will set the {@link #suggestionRange} object to the given
	 * date, and will reset the timerange according to the working/non-working
	 * hour settings.
	 * @param {Ext.Date} startDate The start Date for the suggestions
	 * @param {Number} duration (optional) The period for the suggestions
	 */
	setSuggestionDate : function(startDate, duration)
	{
		var start = startDate.clone().clearTime();
		var due = startDate.clone().clearTime();

		if (this.nonWorkingHoursHidden) {
			start = start.add(Date.MINUTE, container.getSettingsModel().get('zarafa/v1/main/start_working_hour'));
			due = due.add(Date.MINUTE, container.getSettingsModel().get('zarafa/v1/main/end_working_hour'));
		} else {
			// Use 12:00 as time, this prevents problems
			// for DST switches at 00:00 like in Brasil.
			due.setHours(12);
			due = due.add(Date.DAY, 1).clearTime();
		}

		// If the current suggestionTime equals the new time, the
		// DateRange will not trigger the update event. However, when
		// the duration has been set, we must assume a forced update
		// must be performed. So just fire the update event manually.
		//
		if (this.suggestionRange.getStartTime() !== start.getTime())
			this.suggestionRange.set(start, due);
		else if (Ext.isDefined(duration))
			this.suggestionRange.fireEvent('update', this.suggestionRange);
	},

	/**
	 * Load freebusy data for the {@link #blockStore} for the provided users.
	 * @param {Zarafa.core.data.IPMRecipient|Array} userRecords The users for which the
	 * freebusy data is requested
	 * @private
	 */
	loadFreebusyData : function(userRecords)
	{
		var dateRange = this.getDateRange();
		var loadData = {
			add: true, // All blocks will be appended to the existing list.
			actionType : Zarafa.core.Actions['list'],
			params: {
				users: [],
				start: dateRange.getStartTime() / 1000,
				end: dateRange.getDueTime() / 1000
			}
		}

		if (!Ext.isArray(userRecords)) {
			userRecords = [ userRecords ];
		}

		// Collect all unique identifiers of the resolved users, we sent
		// both the record ID as the entryid. The entryid is used by the PHP
		// to collect the data for the given recipient, the userid is used
		// in the response to correlate the response to the correct recipientRecord.
		Ext.each(userRecords, function(userRecord) {
			if (userRecord.isResolved()) {
				loadData.params.users.push({
					userid : userRecord.id,
					entryid : userRecord.get('entryid'),
					organizer : userRecord.isMeetingOrganizer()
				});
			}
		});

		// Perhaps none of the so-called "Resolved" users were
		// resolved. Thats lame, but it does mean less work for us.
		if (!Ext.isEmpty(loadData.params.users)) {
			this.blockStore.load(loadData);
		}
	},

	/**
	 * When the {@link #userStore} has resolved users, we can request all freebusy data
	 * for the resolved users.
	 * @param {Zarafa.core.data.IPMRecipientStore} userStore
	 * @param {Zarafa.core.data.IPMRecipient} userRecords
	 * @private
	 */
	onUserResolved : function(userStore, userRecords)
	{
		this.loadFreebusyData(userRecords);
	},

	/**
	 * When the {@link #userStore} has been loaded, make sure that all users will be
	 * resolved.
	 * @param {Zarafa.core.data.IPMRecipientStore} userStore
	 * @param {Zarafa.core.data.IPMRecipient} userRecords
	 * @private
	 */
	onUserLoad : function(userStore, userRecords)
	{
		this.loadFreebusyData(userRecords);
	},

	/**
	 * When an user is added to the userStore make sure that it is being resolved.
	 * @param {Zarafa.core.data.IPMRecipientStore} userStore Store
	 * @param {Zarafa.core.data.IPMRecipientRecord|Array} userRecords Records of the added users
	 * @param {Number} index Index
	 * @private
	 */
	onUserAdd: function(userStore, userRecords, index)
	{
		this.loadFreebusyData(userRecords);
	},

	/**
	 * When an user is removed from the userStore removing the related data in the blockStore.
	 * @param {Zarafa.core.data.IPMRecipientStore} userStore Store
	 * @param {Zarafa.core.data.IPMRecipientRecord|Array} userRecords
	 * @param {Number} index Index
	 * @private
	 */
	onUserRemove: function(userStore, userRecords, index)
	{
		Ext.each(userRecords, function(userRecord) {
			var records = this.blockStore.getRange();
			Ext.each(records, function(record) {
				if(record.get('userid') == userRecord.id) {
					this.blockStore.remove(record);
				}
			}, this);
		}, this);
	},

	/**
	 * When all users are removed from the userStore, all blocks can be removed from the blockStore.
	 * @param {Zarafa.core.data.IPMRecipientStore} userStore Store
	 * @param {Zarafa.core.data.IPMRecipientStore|Array} userRecords
	 * @private
	 */
	onUserClear : function(userStore, records)
	{
		this.blockStore.removeAll();
	},

	/**
	 * Construct a {@link Ext.data.Record record} for the {@link #sumBlockStore}
	 * based on the start and end information from the given {@link Ext.data.Record record}.
	 * @param {Ext.data.Record} record The original record from which the sum block is based
	 * @return {Ext.data.Record} the record for the sumBlockStore
	 */
	createSumBlock : function(record)
	{
		return new Zarafa.common.freebusy.data.FreebusyBlockRecord({
			start: record.get('start'),
			end: record.get('end'),
			status: record.get('status')
		});
	},

	/**
	 * Construct a number of {@link Ext.data.Record records} for the {@link #suggestionStore}
	 * based on the start and end information. Calculating the number of possible suggestions
	 * is based on the start and end time. Within this range, (starting with 'start') we check
	 * if an suggestion with the given duration is possible. If it is, then a new block is created,
	 * and we move our 'start' value with the 'interval' value. If at this point we still have
	 * sufficient space until the end time, we can create another suggestion. This continues
	 * until we have reached the end of the interval.
	 */
	createSuggestionBlocks : function(start, end, duration, interval)
	{
		var blocks = [];

		// Reduce the end of the period with the duration. This way we can
		// schedule a suggestion for each interval between start and end.
		end -= duration;

		for (var i = start; i <= end; i += interval) {
			blocks.push(new Zarafa.common.freebusy.data.FreebusyBlockRecord({
				start: i,
				end: i + duration
			}));
		}

		return blocks;
	},

	/**
	 * Merge a {@link Ext.data.Record record} from the {@link #blockStore} into
	 * the {@link Ext.data.Record sumRecord} if the two overlap.
	 * @param {Ext.data.Record} record The record to merge into sumRecord
	 * @param {Ext.data.Record} sumRecord The record containing the summed data
	 * @return {Boolean} True if the record has been merged into sumRecord.
	 * @private
	 */
	mergeRecordIntoSumBlock : function(record, sumRecord)
	{
		// Status is different, we can't merge the record into sumRecord
		if (record.get('status') !== sumRecord.get('status'))
			return false;

		// Obtain the start & end timestamps for each record
		var start = record.get('start');
		var end = record.get('end');
		var sumStart = sumRecord.get('start');
		var sumEnd = sumRecord.get('end');

		// If the block is completely outside the sumBlock,
		// we can't do merge the record.
		if (end < sumStart || sumEnd < start)
			return false;

		// If the sumblock completely encapsulates the block,
		// we don't need to do anything, except marking the record as merged.
		if (sumStart <= start && end <= sumEnd)
			return true;

		// If the block completely encapsulates the sumblock,
		// we need to replace the sumblock with the new block.
		if (start < sumStart && sumEnd < end) {
			sumRecord.set('start', start);
			sumRecord.set('end', end);
			return true;
		}

		// If the block starts earlier then the sumblock,
		// we need to expand the sumblock to start earlier.
		if (start < sumStart) {
			sumRecord.set('start', start);
			return true;
		}

		// If the block starts later then the sumblock,
		// we need to expand the sumblock the end later.
		if (sumEnd < end) {
			sumRecord.set('end', end);
			return true;
		}

		// We checked earlier if the record is within sumRecord,
		// outside sumRecord or partially overlaps. Why did we
		// arrive here then? Perhaps the record exists in a
		// parallel universe?
		return false;
	},

	/**
	 * Merge the given {@link Ext.data.Record records} with normal FB block information
	 * into the {@link #sumBlockStore}. Each {@link Ext.data.Record record}
	 * is compared to the existing {@link Ext.data.Record records} from the {@link #sumBlockStore}.
	 * If needed the existing {@link Ext.data.Record record} is expanded to include the updated time.
	 * If the original {@link Ext.data.Record record} does not match any sumBlock then a new
	 * sumBlock will be created.
	 * @param {Ext.data.Record|Array} records The records which must be merged into the sum block store
	 * @param {Zarafa.common.freebusy.data.FreebusyBlockStore} sumBlockStore The store containing the sum blocks
	 * @param {Boolean} splitBusyStatus (optional) If true then sumBlocks will be generated per
	 * BusyStatus group, otherwise a single sumBlock will be generated.
	 */
	mergeBlocksToSumBlockStore : function(records, sumBlockStore, splitBusyStatus)
	{
		var lastBlock = {};

		// Sort all records by start date, this ensures that we can apply the merging safely.
		records.sort(function(a, b) {
			return a.get('start') - b.get('start');
		});

		Ext.each(records, function(record) {
			//  We are not generating sumBlocks for the free status.
			var busyStatus = record.get('status');
			if (busyStatus === Zarafa.core.mapi.BusyStatus.UNKNOWN || busyStatus === Zarafa.core.mapi.BusyStatus.FREE)
				return true;

			if (splitBusyStatus === false)
				busyStatus = Zarafa.core.mapi.BusyStatus.BUSY;

			// The first item can be inserted directly
			if (sumBlockStore.getCount() == 0) {
				sumBlockStore.add(this.createSumBlock(record));
				return true;
			}

			var sumRecord = lastBlock[busyStatus];

			// Check if the new record can be merged into the sumRecord,
			// if this is not the case, we should add it as a new block.
			if (!Ext.isDefined(sumRecord) || this.mergeRecordIntoSumBlock(record, sumRecord) !== true) {
				var newSumBlock = this.createSumBlock(record);
				sumBlockStore.add(newSumBlock);
				lastBlock[busyStatus] = newSumBlock;
			}
		}, this);
	},

	/**
	 * Walk through the {@link Ext.data.Record records} from the {@link #sumBlockStore}
	 * to determine if any {@link Ext.data.Record record} overlapse. If this is the
	 * case the two {@link Ext.data.Record records} will be merged into a single block.
	 * @param {Zarafa.common.freebusy.data.FreebusyBlockStore} sumBlockStore The store containing the sumblocks
	 */
	mergeSumBlocks : function(sumBlockStore)
	{
		var lastBlock = {};

		// Start sorting the sumblocks based on their start date, they
		// could not have been added sorted into the store, since we
		// have been modifying the start/end values of the blocks after adding.
		sumBlockStore.sort('start', 'ASC');

		sumBlockStore.each(function(record) {
			var prevRecord = lastBlock[record.get('status')];

			// Check if the previous record and the current record overlap,
			// if this is the case the records can be merged. The comparison
			// is very simple since we only need to check if the end-time for
			// the previous item is after the start-time of the current item.
			if (Ext.isDefined(prevRecord)) {
				if (prevRecord.get('end') >= record.get('start')) {
					// Update the previous block
					prevRecord.set('end', record.get('end'));
					// And we don't need the new block anymore
					sumBlockStore.remove(record);
					return true;
				}
			}

			lastBlock[record.get('status')] = record;
		}, this);
	},

	/**
	 * Fires after a new set of Records has been loaded.
	 * @param {Zarafa.common.freebusy.data.FreebusyBlockStore} store
	 * @param {Ext.data.Record|Array} records The Records that were loaded
	 * @param {Object} options The loading options that were specified (see {@link #load} for details)
	 * @private
	 */
	onBlockLoad : function(store, records, options)
	{
		// Always start with a clean store
		this.sumBlockStore.removeAll();

		if (this.getUserStore().getCount() == 0) {
			// No users, easy, we're done!
		} else if (this.getUserStore().getCount() == 1) {
			// One user, still easy, sumBlockStore is the same as blockStore
			this.blockStore.each(function(record) {
				//  We are not generating sumBlocks for the free status.
				var busyStatus = record.get('status');
				if (busyStatus !== Zarafa.core.mapi.BusyStatus.UNKNOWN && busyStatus !== Zarafa.core.mapi.BusyStatus.FREE) {
					this.sumBlockStore.add(this.createSumBlock(record));
				}
			}, this);
		} else {
			// Multiple users, tricky, merge the blockstore into the sumBlockStore
			this.mergeBlocksToSumBlockStore(this.blockStore.getRange(), this.sumBlockStore);
		}

		// Sort all sumblocks based on the status. This will force the
		// TENTATIVE records to be rendered before the BUSY which in turn is before
		// the OUTOFOFFICE. This in turn forces the browser to position the OUTOFOFFICE
		// divs on top of the BUSY blocks (which in turn are on top of TENATIVE) when
		// the blocks overlap.
		this.sumBlockStore.sort('status', 'ASC');

		this.sumBlockStore.fireEvent('load', this.sumBlockStore, this.sumBlockStore.getRange(), {});
	},

	/**
	 * Fires when a Record has been removed from the Store
	 *
	 * NOTE: Because of buffering which is applied for the invocation event handler,
	 * this handler will only be called for the first remove event
	 * in a batch. This implies that we cannot rely on the record and
	 * index arguments.
	 *
	 * @param {Zarafa.common.freebusy.data.FreebusyBlockStore} store
	 * @param {Ext.data.Record} record The Record that was removed
	 * @param {Number} index The index at which the record was removed
	 * @private
	 */
	onBlockRemove : function(store, record, index)
	{
		this.onBlockLoad(store, store.getRange(), {});
	},

	/**
	 * Fires after a new set of Records has been loaded.
	 * @param {Zarafa.common.freebusy.data.FreebusyBlockStore} store
	 * @param {Ext.data.Record|Array} records The Records that were loaded
	 * @param {Object} options The loading options that were specified (see {@link #load} for details)
	 * @private
	 */
	onSumBlockLoad : function(store, records, options)
	{
		// Always start with a clean store
		this.freeBlockStore.removeAll();

		if (store.getCount() > 0) {
			// FIXME: We should actually build this store while building
			// the busy blocks. This is most likely faster then doing it
			// separately.
			this.mergeBlocksToSumBlockStore(records, this.freeBlockStore, false);
		}

		this.loadSuggestionBlocks();
	},

	/**
	 * Fires when the {@link #nonWorkingHoursHidden} has been updated.
	 * This will updated the {@link #suggestionRange} accordingly.
	 * @param {Boolean} hideNonWorkingHours True to only show working hours.
	 * @private
	 */
	onShowWorkingHoursChange : function(hideNonWorkingHours)
	{
		this.setSuggestionDate(this.suggestionRange.getStartDate());
	},

	/**
	 * Fires when the {@link #userStore} has been updated. This will force
	 * a load on the store to automatically update the {@link #blockStore}.
	 * @param {Zarafa.core.data.IPMRecipientStore} newStore
	 * @param {Zarafa.core.data.IPMRecipientStore} oldStore
	 * @private
	 */
	onUserStoreChange : function(newStore, oldStore)
	{
		newStore.fireEvent('load', newStore, newStore.getRange());
	},

	/**
	 * Fires when the {@link #selectorRange} has been updated. This
	 * will update the {@link #suggestionRange} accordingly.
	 * @param {Zarafa.core.DateRange} newRange The new selected range.
	 * @param {Zarafa.core.DateRange} oldRange The old selected range.
	 * @private
	 */
	onSelectorRangeUpdate : function(newRange, oldRange)
	{
		// Check to see if the update was triggered by outside sources
		if(!this.updatingSelectorRangeExternally){
			this.fireEvent('selectorrangeupdate', newRange, oldRange);
		}
		// Reset the state
		this.updatingSelectorRangeExternally = false;
		this.setSuggestionDate(newRange.getStartDate(), newRange.getDuration());
	},

	/**
	 * Fires when the {@link #suggestionRange} has been updated. This
	 * will recalculate the {@link #suggestionBlockStore} accordingly.
	 * @param {Zarafa.core.DateRange} newRange The new selected range.
	 * @param {Zarafa.core.DateRange} oldRange The old selected range.
	 * @private
	 */
	onSuggestionRangeUpdate : function(newRange, oldRange)
	{
		this.loadSuggestionBlocks();
	},

	/**
	 * This will recalculate the suggestion blocks in the {@link #suggestionBlockStore}
	 * based on the information from the {@link #freeBlockStore} and the {@link #selectorRange}.
	 * @private
	 */
	loadSuggestionBlocks : function()
	{
		// Always start with a clean store
		this.suggestionBlockStore.removeAll();

		if (this.freeBlockStore.getCount() > 0) {
			var start = this.suggestionRange.getStartTime() / 1000;
			var end = this.suggestionRange.getDueTime() / 1000;
			var duration = this.selectorRange.getDuration(Date.SECOND);
			var interval = Ext.min([duration, 30 * 60]); // FIXME: make configurable

			// But what if the appointment takes 0 minutes..
			// That would be dumb, but we won't be fooled!
			if (interval <= 0) {
				interval = 30 * 60;
			}

			this.freeBlockStore.each(function(sumBlock) {
				var sumStart = sumBlock.get('start');
				var sumEnd = sumBlock.get('end');

				if (sumEnd < start || sumStart > end) {
					// The entire block falls before the requested range,
					// just keep looping until we find the desired range.
				} else if (sumStart > end) {
					// The entire block falls after the requested range
					// we don't need to do anything anymore.
					return false;
				} else if (sumStart <= start) {
					// The block overlap our range, our new start
					// time is the end time of this block.
					start = sumEnd;
				} else {
					// The entire block falls after our start range,
					// simply add a suggestionblock from start to the sumBlock start.
					this.suggestionBlockStore.add(this.createSuggestionBlocks(start, Ext.min([sumStart, end]), duration, interval));
					start = sumEnd;
				}
			}, this);

			// Check if we still have a leftover...
			if (start < end) {
				this.suggestionBlockStore.add(this.createSuggestionBlocks(start, end, duration, interval));
			}
		}

		this.suggestionBlockStore.fireEvent('load', this.suggestionBlockStore, this.suggestionBlockStore.getRange(), {});
	},

	/**
	 * Function can be used to check if all added attendees are free on particular timeslot.
	 * @param {Date} periodStartTime object of start time
	 * @param {Date} periodEndTime object of end time
	 * @param {Boolean} modifiedOnly True if only the modified attendees should be checked
	 * @return {Boolean} return true if all attendees are free else false.
	 */
	checkAttendeesBusyStatus : function(periodStartTime, periodEndTime, modifiedOnly)
	{
		var userStore = this.getUserStore();
		if (!userStore) {
			return false;
		}

		var modified = userStore.getRange();
		for (var i = 0, len = modified.length; i < len; i++) {
			var user = modified[i];
			if ((modifiedOnly !== true || user.phantom === true) && !user.isMeetingOrganizer()) {
				if (this.checkAttendeeBusyStatus(user.id, periodStartTime, periodEndTime)) {
					return true;
				}
			}
		}

		return false;
	},

	/**
	 * Check if the given user is free on the given timeslot
	 * @param {String} userid The userid of the user to check for availability
	 * @param {Date} periodStartTime object of start time
	 * @param {Date} periodEndTime object of end time
	 * @return {Boolean} return true if the attendee is free.
	 */
	checkAttendeeBusyStatus : function(userid, periodStartTime, periodEndTime)
	{
		var blockStore = this.getBlockStore();
		if(!blockStore) {
			return false;
		}

		// Ensure that only the current userid is shown
		blockStore.filter('userid', userid, false, true, true);

		// Sort on start date
		blockStore.sort('start', 'ASC');

		// We need timestamps rather then Date objects
		periodStartTime = periodStartTime.getTime() / 1000;
		periodEndTime = periodEndTime.getTime() / 1000;

		// Lets search the block for Blocks that overlap with the requested period.
		var busy = false;

		for (var index = 0, len = blockStore.getCount(); index < len; index++) {
			var record = blockStore.getAt(index);

			/*
			 * First we need to remove appointments which are occuring extermely before/after our
			 * selected time, because then we have only set of appointments which are overlapping/inside
			 * our time slot.
			 * For that to achieve we first need sort the records based on start time and then to find
			 * out sum block record whose end time is greater then our selected start time
			 * and start time is less then our selected end time then we can say that
			 * the selected time is not proper for all the attendees.
			 */
			// remove appointments occuring extremely before our selected time
			if (record.get('end') > periodStartTime) {
				// check if we are really interested in this block
				if (record.get('status') === Zarafa.core.mapi.BusyStatus.FREE || record.get('status') === Zarafa.core.mapi.BusyStatus.UNKNOWN)
					continue;

				// before we have sorted the records based on start time so we will be having a record which either
				// overlaps current selected time or doesn't overlap it
				// below condition will check if the record overlaps current selected time
				if (record.get('start') < periodEndTime) {
					busy = true;
				}

				// if the above condition is not satisfied then we can say that
				// record is not overlapping current selected time and therefore
				// break the loop
				break;
			}
		}

		blockStore.clearFilter();

		return busy;
	}
});
Ext.namespace('Zarafa.common.freebusy.data');

/**
 * @class Zarafa.common.freebusy.data.TimelineSelector
 * @extends Ext.util.Observable
 */
Zarafa.common.freebusy.data.TimelineSelector = Ext.extend(Ext.util.Observable, 
{
	// Private
	// Properties used to identify the mouse position.
	MOUSE_ON_EDGE_LEFT: 1,
	MOUSE_ON_CENTER: 2,
	MOUSE_ON_EDGE_RIGHT: 3,

	// Private
	// When set to true the user is in the middle of a selection.
	selecting: false,
	// Private
	// Timestamp of the start of the selection (not necessarily the start date).
	selectionStart: null,
	// Private
	// Timestamp of the end of the selection (necessarily the end date).
	selectionEnd: null,
	// Private
	// DateRange used by the TimelineSelector
	selectorRange: null,

	/**
	 * @cfg {Number} dragSelectionEdgeArea
	 * The number of pixels from the edge the cursor snaps to the edge of the selection and drags 
	 * either the start date or end date. If the cursor is further away from the either egdge than 
	 * the supplied number than the a click will create a whole new selection (defaults to 10).
	 */
	dragSelectionEdgeArea: 10,

	/**
	 * @constructor
	 * @param {Object} config The configuration options.
	 */
	constructor: function(config)
	{
		Ext.apply(this, config || {});
		Zarafa.common.freebusy.data.TimelineSelector.superclass.constructor.call(this, config);
	},

	/**
	 * Initializes the selector
	 * @param {Zarafa.common.freebusy.ui.TimelineView} parent Object of the parent TimelineView
	 */
	init: function(parent){
		this.parent = parent;

		this.masterTpl = new Ext.XTemplate(
			'<div class="x-freebusy-selector">',
			'</div>',
			{
				// Format functions like capitalize in the Ext.util.Format are not 
				// used in this template anyways. Speeds up the apply time.
				disableFormats: true
			}
		);

		/**
		 * We can only render the selector when the TimelineView has been rendered. The TimelineView
		 * can also rerender the timeline to show/hide the non-working hours. In that case the 
		 * render event is not called, but the rendertimeline event is always called when rendering
		 * the timeline HTML. 
		 */
		this.parent.on("rendertimeline", this.onParentRenderTimeline, this);
		this.parent.on("timelinemousedown", this.onParentTimelineMouseDown, this);
		this.parent.on("timelinemousemove", this.onParentTimelineMouseMove, this);
		this.parent.on("timelinemouseup", this.onParentTimelineMouseUp, this);

		this.bindSelectorRange(this.parent.model.getSelectorRange(), true);
	},

	/**
	 * Registers update event to trigger visualizing changes in TimelineSelector
	 * @param {Zarafa.core.DateRange} selectorRange selectorRange
	 * @param {Boolean} initial Internally used to indicate that this is the first call after render.
	 */
	 bindSelectorRange: function(selectorRange, initial)
	{
		if(this.selectorRange && !initial){
			this.selectorRange.un('update', this.onSelectorRangeUpdate, this);
		}
		this.selectorRange = selectorRange;
		if(this.selectorRange){
			this.selectorRange.on({
				scope: this,
				// Listening in to updates in the selector daterange.
				update: this.onSelectorRangeUpdate
			});
		}
	},

	/**
	 * Renders the HTML needed for the selector and positions it. The parent is accessed to get the 
	 * HTML Element to render the selector in.
	 */
	render: function(){
		// The parent contains a container to render the selector in
		var parentSelectorContainer = this.parent.getSelectorContainer();

		// Render the HTML Elements
		this.masterTpl.overwrite(parentSelectorContainer);
		this.selectorElem = Ext.get(parentSelectorContainer.dom.firstChild);
		this.selectorElem.setVisibilityMode(Ext.Element.DISPLAY);

		this.selectorElem.on('mousemove', this.onMouseMoveSelector, this);

		// Position the selector by using the selector daterange
		this.positionSelector(this.selectorRange);
	},

	/**
	 * @todo Perhaps this function does not need that argument any more?
	 * Position the selector on the start and end dates. When the selector indicates a daterange 
	 * that is not visible on the timeline, the selector will be hidden.
	 * @param {Zarafa.core.DateRange} selectorRange Daterange for the new position of the selector.
	 */
	positionSelector: function(selectorRange){
		// Check whether the selector has to be shown inside the daterange of the timeline
		if(this.parent.model.getDateRange().overlaps(selectorRange)){
			// Transform start/end date into timestamps
			var start = selectorRange.getStartDate().getTime()/1000;
			var end = selectorRange.getDueDate().getTime()/1000;

			// Get the leftoffset of the start of the selector
			var pixelOffsetLeft = this.parent.findBlockPixelOffset(start, true);
			this.selectorElem.setLeft(pixelOffsetLeft);

			// Get the leftoffset of the end of the selector
			var pixelOffset = this.parent.findBlockPixelOffset(end, false);
			this.selectorElem.setWidth( pixelOffset - pixelOffsetLeft );
			this.selectorElem.setVisible(true);
		}else{
			// Hide the element when it is outside the visible period
			this.selectorElem.setVisible(false);
		}
	},

	/**
	 * Scrolls the timelineView to the date that is selected in the selectorRange.
	 */
	scrollTimelineToSelection: function(){
		this.parent.scrollDateIntoView( this.selectorRange );
	},

	/**
	 * Fired when the selector daterange is modified. When the daterange is changed the selector 
	 * needs to update UI component to visualize the change.
	 * @param {Zarafa.core.DateRange} selectorRange Changed daterange.
	 */
	onSelectorRangeUpdate: function(daterange){
		this.positionSelector(daterange);

		if(!this.selecting){
			this.scrollTimelineToSelection();
		}
	},

	/**
	 * Fired when the parent TimelineView renders the timeline. When the TimelineView renders the UI
	 * the selector needs to render the UI part.
	 * @param {Zarafa.common.freebusy.ui.TimelineView} timelineView TimelineView
	 */
	onParentRenderTimeline: function(timelineView){
		this.render();
	},

	/**
	 * Fired when the parent TimelineView detects a mousedown event.
	 * @param {Ext.EventObject} evt The {@link Ext.EventObject} encapsulating the DOM event.
	 * @param {HtmlElement} target The target of the event.
	 * @param {Object} cfg The options configuration passed to the {@link #addListener} call.
	 */
	onParentTimelineMouseDown: function(evt, target, cfg){
		// Check to see if the mouse event is not done on the scrollbar
		if(this.isMouseEventOnScrollbar(evt, target)) return true;

		this.selecting = true;

		//Normalize the coordinate to the correct coordinate for the timeline and not the page
		var timelineElem = this.parent.bodyElem;
		var timestampCoordX = evt.getPageX() - timelineElem.getLeft() + timelineElem.getScroll().left;

		// Get the position of the mouse to determine wether it is on the outer edges or in the center.
		var mousePos = this.getMousePosition(evt.getPageX());

		// Transform X coordinate into a timestamp by using the TimelineView's methods.
		var selectionClick = this.parent.findTimestampByTimelineXCoord(timestampCoordX);

		switch(mousePos){
			case this.MOUSE_ON_EDGE_LEFT:
				// Leave the end date as the start point for the selection
				var endDate = this.selectorRange.getDueDate();
				this.selectionStart = endDate.getTime()/1000;
				break;
			case this.MOUSE_ON_EDGE_RIGHT:
				// Leave the start date as the start point for the selection
				var startDate = this.selectorRange.getStartDate();
				this.selectionStart = startDate.getTime()/1000;
				break;
			default:
				// Snap the timestamp for the start to a half hour slot
				var startDate = new Date(selectionClick*1000);
				startDate.round(Date.MINUTE, 30);
				this.selectionStart = startDate.getTime() / 1000;
				// Make the appointment duration 30 minutes by default
				var endDate = startDate.add(Date.MINUTE, 30);
				this.selectorRange.set(startDate, endDate);
		}
		evt.preventDefault();
	},

	/**
	 * Fired when the parent TimelineView detects a mousemove event.
	 * @param {Ext.EventObject} evt The {@link Ext.EventObject} encapsulating the DOM event.
	 * @param {HtmlElement} target The target of the event.
	 * @param {Object} cfg The options configuration passed to the {@link #addListener} call.
	 */
	onParentTimelineMouseMove: function(evt, target, cfg){
		if(this.selecting){
			//Normalize the coordinate to the correct coordinate for the timeline and not the page
			var timelineElem = this.parent.bodyElem;
			var timestampCoordX = evt.getPageX() - timelineElem.getLeft() + timelineElem.getScroll().left;

			this.selectionEnd = this.parent.findTimestampByTimelineXCoord(timestampCoordX);

			// Snap the timestamp to a half hour slot
			var endDate = new Date(this.selectionEnd*1000);
			endDate.round(Date.MINUTE, 30);
			this.selectionEnd = endDate.getTime() / 1000;

			// Check if the range does not have a duration of zero 
			if(this.selectionStart != this.selectionEnd){

				// Check if the start date is before the end date and swap if needed
				if(this.selectionStart <= this.selectionEnd){
					var selectorStartDate = new Date(this.selectionStart*1000);
					var selectorEndDate = endDate;
				}else{
					var selectorStartDate = endDate;
					var selectorEndDate = new Date(this.selectionStart*1000);
				}

				// Check if the start date or end date have been changed
				var startDateChanged = (this.selectorRange.getStartDate().getTime() != selectorStartDate.getTime());
				var dueDateChanged = (this.selectorRange.getDueDate().getTime() != selectorEndDate.getTime());
				if(startDateChanged || dueDateChanged){
					this.selectorRange.set(selectorStartDate, selectorEndDate);
				}
			}

			evt.preventDefault();
		}
	},

	/**
	 * Fired when the parent TimelineView detects a mouseup event.
	 * @param {Ext.EventObject} evt The {@link Ext.EventObject} encapsulating the DOM event.
	 * @param {HtmlElement} target The target of the event.
	 * @param {Object} cfg The options configuration passed to the {@link #addListener} call.
	 */
	onParentTimelineMouseUp: function(evt, target, cfg){
		if(this.selecting){
			this.selecting = false;
			evt.preventDefault();
		}
	},

	onMouseMoveSelector: function(evt, target, cfg)
	{
		if(!this.selecting){
			var mousePos = this.getMousePosition(evt.getPageX());
			switch(mousePos){
				case this.MOUSE_ON_EDGE_LEFT:
				case this.MOUSE_ON_EDGE_RIGHT:
					this.selectorElem.setStyle('cursor', 'w-resize');
					break;
				default:
					this.selectorElem.setStyle('cursor', '');
			}
		}
	},

	getMousePosition: function(clickX)
	{
		var selectorX = this.selectorElem.getXY()[0];
		var rightSelectorY = selectorX + this.selectorElem.getWidth();

		if(Math.abs(clickX - selectorX) < this.dragSelectionEdgeArea){
			return this.MOUSE_ON_EDGE_LEFT;
		}else if(Math.abs(rightSelectorY - clickX) < this.dragSelectionEdgeArea){
			return this.MOUSE_ON_EDGE_RIGHT;
		}else{
			return this.MOUSE_ON_EDGE_CENTER;
		}
	},

	/**
	 * Checks whether the mouse event takes place on a scrollbar or whether it takes place inside 
	 * the element. Returns true if it takes plae on the scrollbar.
	 * @param {Ext.EventObject} evt The {@link Ext.EventObject} encapsulating the DOM event.
	 * @param {HtmlElement} target The target of the event.
	 * @return {Boolean} When mouse clicked on scrollbar returns true, otherwise false.
	 * @private
	 */
	isMouseEventOnScrollbar: function(evt, target){
		// Get the bodyElem of the TimelineView because this is the element that contains the scrollbars
		var scrollContainer = this.parent.bodyElem.dom;
		// Prevent the selector from making a selection when you are only dragging the scrollbar.
		if(evt && evt.browserEvent && scrollContainer){ 
			var topleft = Ext.get(scrollContainer).getXY(); 
			if(evt.browserEvent.clientX-topleft[0] > scrollContainer.clientWidth || evt.browserEvent.clientY-topleft[1] > scrollContainer.clientHeight) { 
				// Clicking outside viewable area -> must be a click in the scrollbar, allow default action. 
				return true; 
			} 
		} 
		return false;
	}
});
Ext.namespace('Zarafa.common.freebusy.ui');

/**
 * @class Zarafa.common.freebusy.ui.FreebusyPanel
 * @extends Ext.Panel
 * @xtype zarafa.freebusypanel
 */
Zarafa.common.freebusy.ui.FreebusyPanel = Ext.extend(Ext.Panel, {
	/**
	 * @cfg {Zarafa.common.freebusy.data.FreebusyBlockStore} blockStore
	 * The {@link Ext.data.Store} object that handles the freebusy blocks displayed on the timeline.
	 */
	blockStore: null,
	/**
	 * @cfg {Ext.data.Store} userStore
	 * The {@link Ext.data.Store} object that handles the list of users loaded. It can be a normal
	 * store, but it is preferably an {@link  Zarafa.core.data.IPMRecipientStore}.
	 */
	userStore: null,
	/**
	 * @cfg {Zarafa.common.freebusy.data.FreebusyModel} model
	 * The model that keeps track of the userStore, dates, etc.
	 */
	model: null,
	/**
	 * @cfg {Object} userlistConfig The configuration object which will be applied
	 * to the {@link Zarafa.common.freebusy.ui.UserListView UserListView} object.
	 */
	userlistConfig: null,
	/**
	 * @cfg {Object} timelineConfig The configuration object which will be applied
	 * to the {@link Zarafa.common.freebusy.ui.TimelineView TimelineView} object.
	 */
	timelineConfig: null,
	/**
	 * @cfg {Object} suggestionConfig The configuration object which will be applied
	 * to the {@link Zarafa.common.freebusy.ui.SuggestionListView SuggestionListView} object.
	 */
	suggestionConfig: null,
	/**
	 * @cfg {Object} legendaConfig The configuration object which will be applied
	 * to the {@link Ext.Panel legendaView} object.
	 */
	legendaConfig: null,
	/**
	 * @cfg {Boolean} showUserList
	 * When set to true it will show the userlist to the left. When set to false it will hide it
	 * (defaults to true).
	 */
	showUserList: true,
	/**
	 * @cfg {Boolean} showSuggestionList
	 * When set to true it will show the Suggestionlist to the left. When set to false it will hide it
	 * (defaults to true).
	 */
	showSuggestionList : true,
	/**
	 * @cfg {Boolean} editable
	 * When set to false it will make the userlist non-editable (defaults to true).
	 */
	editable: true,
	/**
	 * @cfg {Boolean} showLegenda
	 * When set to true it will show the legenda at the bottom. When set to false it will hide it
	 * (defaults to true).
	 */
	showLegenda: true,
	/**
	 * @cfg {Number} headerHeight
	 * Height of the header that will be set for the {@link Zarafa.common.freebusy.ui.UserlistView} and the
	 * {@link Zarafa.common.freebusy.ui.TimelineView} (defaults to 50).
	 */
	headerHeight: 50,
	// Private
	// Height of inputfield that will be shown in the {@link Zarafa.common.freebusy.ui.UserlistView}.
	inputfieldHeight: 20,
	/**
	 * @cfg {Boolean} initialScrollToCurrentDay
	 * When set to true it will scroll to the current day or the selected period. This is done after
	 * the initial laying out (Defaults to true).
	 */
	initialScrollToCurrentDay: true,
	// Private
	// The afterlayout event triggers the scrolling of the timeline to the current day or selection.
	// This flag indicates whether we should react the afterlayout event or whether this has already
	// happened.
	initialScrollDone: false,

	/**
	 * @constructor
	 * @param {Object} config The configuration options.
	 */
	constructor: function(config)
	{
		config = config || {};

		var modelConfig = config.modelConfig || {};
		Ext.applyIf(modelConfig, {
			userStore: config.userStore,
			blockStore: config.blockStore
		});
		var model = new Zarafa.common.freebusy.data.FreebusyModel(modelConfig);

		Ext.applyIf(config, {
			layout: 'border',
			border: true,

			blockStore: model.getBlockStore(),
			userStore: model.getUserStore(),
			model: model,

			userlistConfig: {},
			timelineConfig: {},
			suggestionConfig: {},
			legendaConfig: {}
		});

		Zarafa.common.freebusy.ui.FreebusyPanel.superclass.constructor.call(this, config);

		this.add([
			this.createUserListView(),
			this.createSuggestionView(),
			this.createTimelineView(),
			this.createLegendaView()
		]);

		this.initFreebusyEvents();
	},

	/**
	 * This will create the listview panel containing all the users for
	 * which the freebusy should be shown.
	 * @return {Object} Configuration object for the user listview
	 * @private
	 */
	createUserListView : function()
	{
		return {
			xtype: 'panel',
			ref: 'userListView',
			region: 'west',
			layout: 'fit',
			title: _('Select attendees'),
			hidden: (!this.showUserList),
			cls: 'x-freebusy-userlist',
			footer: true,
			footerCfg: {
				tag: 'div',
				cls: 'x-panel-footer'
			},
			border: false,
			width: 200,
			collapsible: true,
			split: true,
			collapseMode: 'mini',
			items: [Ext.applyIf(this.userlistConfig, {
				xtype: 'zarafa.freebusyuserlistview',

				model: this.model,

				inputFieldHeight: (this.editable) ? this.inputfieldHeight : 0,
				editable: this.editable
			})]
		};
	},

	/**
	 * This will create the suggestion listview which contains all possible
	 * times for an appointment for the selected users on a given day.
	 * @return {Object} Configuration object for the suggestion listview
	 * @private
	 */
	createSuggestionView : function()
	{
		return {
			xtype: 'panel',
			ref: 'suggestionListView',
			region: 'east',
			layout: 'fit',
			title: _('Suggested Times'),
			hidden: (!this.showSuggestionList),
			footer: true,
			footerCfg: {
				tag: 'div',
				cls: 'x-panel-footer'
			},
			border: false,
			width: 200,
			collapsible: true,
			split: true,
			collapseMode: 'mini',
			items: [Ext.applyIf(this.suggestionConfig, {
				xtype: 'zarafa.freebusysuggestionlistpanel',

				model: this.model,

				listeners: {
					select: this.onSelectSuggestion,
					dateselect: this.onSuggestionDateSelect,
					scope: this
				}
			})]
		};
	},

	/**
	 * This will create the timeline panel which contains all the freebusy blocks
	 * for each user which has been selected.
	 * @return {Object} Configuration object for the timelineview
	 * @private
	 */
	createTimelineView : function()
	{
		return Ext.applyIf(this.timelineConfig, {
			xtype: 'zarafa.freebusytimelineview',
			ref: 'timelineView',
			region: 'center',
			blockStore: this.blockStore,
			model: this.model,
			selector: new Zarafa.common.freebusy.data.TimelineSelector(),
			extraBodyHeight: (this.editable) ? this.inputfieldHeight : 0,
			headerHeight: this.headerHeight,
			// TODO: Make this configurable and toggable
			hideNonWorkingHours: true,
			listeners: {
				bodyscroll : this.onTimelineScroll,
				scope: this
			}
		});
	},

	/**
	 * This will create the legenda panel which contains all the possible
	 * block colors which will be used in the timelineview.
	 * @return {Object} configuration object for the legenda
	 * @private
	 */
	createLegendaView : function()
	{
		return Ext.applyIf(this.legendaConfig, {
			xtype: 'container',
			region: 'south',
			border: true,
			style: 'margin: 3px 0px 0px 0px; padding: 3px 0px 3px 0px;',
			layout: 'hbox',
			hidden: (!this.showLegenda),
			autoHeight: true,
			cls: 'x-freebusy-timeline-container x-freebusy-legenda',
			items: [{
				xtype: 'container',
				cls: 'x-freebusy-timeline-block-busy',
				width: 25,
				height: 25
			},{
				xtype: 'displayfield',
				value: _('Busy'),
				hideLabel : true,
				width: 100
			},{
				xtype: 'container',
				cls: 'x-freebusy-timeline-block-tentative',
				width: 25,
				height: 25
			},{
				xtype: 'displayfield',
				value: _('Tentative'),
				hideLabel : true,
				width: 100
			},{
				xtype: 'container',
				cls: 'x-freebusy-timeline-block-outofoffice',
				width: 25,
				height: 25
			},{
				xtype: 'displayfield',
				value: _('Out of Office'),
				hideLabel : true,
				width: 100
			},{
				xtype: 'container',
				cls: 'x-freebusy-timeline-block-blur',
				width: 25,
				height: 25
			},{
				xtype: 'displayfield',
				value: _('No Information'),
				hideLabel : true,
				width: 100
			}]
		});
	},

	/**
	 * Initialize freebusy events
	 * @private
	 */
	initFreebusyEvents: function()
	{
		// Register to afterlayout to support the timeline in scrolling to the current day
		this.on('afterlayout', this.onAfterLayout, this);
	},

	/**
	 * Returns the model.
	 * @return {Zarafa.common.freebusy.data.FreebusyModel} Model
	 */
	getModel: function()
	{
		return this.model;
	},

	/**
	 * Change the {@link #editable} field on this panel
	 * @param {Boolean} value The new editable status
	 */
	setEditable : function(value)
	{
		if (this.editable !== value) {
			this.editable = value;
			this.userListView.get(0).setEditable(value);
		}
	},

	/**
	 * Is called when the layouting is done and the sizes are calculated. When the timeline needs to
	 * be scrolled to the current day or selection, this listener triggers the scrolling. It looks
	 * for the first usable afterlayout event and the ones after that will not trigger any scrolling
	 * anymore. The first usable event is the one where the container has a width larger than zero.
	 * @param {Ext.Container} container Container
	 * @private
	 */
	onAfterLayout: function(container)
	{
		// Calculate the desired header size, we have the configured header size, but we need
		// to substract any padding/borders and margins which are applied to the header.
		if (this.userListView) {
			var height = this.headerHeight;

			height -= this.userListView.header.getMargins('tb');
			height -= this.userListView.header.getBorderWidth('tb');
			height -= this.userListView.header.getPadding('tb');

			this.userListView.header.setStyle('height', height + 'px');

			// Ext.getScrollBarWidth() is always off by 2 pixels, this is a hardcoded
			// thing, so we can fix it hardcoded...
			this.userListView.footer.setStyle('height', (Ext.getScrollBarWidth() - 2) + 'px');
		}
		if (this.suggestionListView) {
			var height = this.headerHeight;

			height -= this.userListView.header.getMargins('tb');
			height -= this.userListView.header.getBorderWidth('tb');
			height -= this.userListView.header.getPadding('tb');

			this.suggestionListView.header.setStyle('height', height + 'px');
			// Ext.getScrollBarWidth() is always off by 2 pixels, this is a hardcoded
			// thing, so we can fix it hardcoded...
			this.suggestionListView.footer.setStyle('height', (Ext.getScrollBarWidth() - 2) + 'px');
		}

		if (this.initialScrollToCurrentDay) {
			/**
			 * We only need to scroll to the current day or selection once at startup. We have to
			 * look for the first afterlayout event with a container that has a width larger than 0.
			 */
			 // @TODO every time on layout we must need to scroll timeline to current selection as "scroll" event is not working properly in IE.
			if (!Ext.isIE && (this.initialScrollDone || this.timelineView.getWidth() == 0)) {
				return;
			} else {
				this.initialScrollDone = true;
			}

			this.scrollTimelineToSelection();
		}
	},

	/**
	 * Scroll the TimelineView to the current Selector date, or to
	 * the current date if no selector is available.
	 */
	scrollTimelineToSelection : function()
	{
		if (this.timelineView.selector) {
			this.timelineView.selector.scrollTimelineToSelection();
		} else {
			this.timelineView.scrollDateIntoView(new Date());
		}
	},

	/**
	 * Scroll the TimelineView to the provided date argument.
	 * @param {Date} date Date object
	 */
	scrollTimelineToDate : function(date)
	{
		this.timelineView.scrollDateIntoView(date);
	},

	/**
	 * Used to syncronize the scrolling height of the timeline view with the user list
	 * @param {Object} scrollPos Scroll position
	 * @private
	 */
	onTimelineScroll: function(scrollPos)
	{
		this.userListView.body.scrollTo('top', scrollPos.top);
	},

	/**
	 * Fired when the user clicks on a suggestion from the SuggestionPanel.
	 * This will update the Selector DateRange to match the suggestion.
	 * @param {Zarafa.common.freebusy.ui.SuggestionListPanel} panel The panel which raised the event
	 * @param {Zarafa.common.freebusy.data.FreebusyBlockRecord} record The selected suggestion
	 * @private
	 */
	onSelectSuggestion : function(panel, record)
	{
		var start = record.get('start') * 1000;
		var end = record.get('end') * 1000;

		this.getModel().getSelectorRange().setTime(start, end);
	},

	/**
	 * Called when the {@link Zarafa.common.freebusy.ui.SuggestionListPanel SuggestionListPanel} fires
	 * the {@link Zarafa.common.freebusy.ui.SuggestionListPanel#dateselect dateselect} event.
	 * @param {Zarafa.common.freebusy.ui.SuggestionListPanel} this This panel
	 * @param {Date} date The selected date
	 * @private
	 */
	onSuggestionDateSelect: function(suggestionListPanel, date)
	{
		this.scrollTimelineToDate(date);
	}

});

Ext.reg('zarafa.freebusypanel',Zarafa.common.freebusy.ui.FreebusyPanel);
Ext.namespace('Zarafa.common.freebusy.ui');

/**
 * @class Zarafa.common.freebusy.ui.SuggestionListPanel
 * @extends Ext.Panel
 * @xtype zarafa.freebusysuggestionlistpanel
 *
 * The SuggestionList shows the list of suggested times
 * at which an appointment can be held.
 */
Zarafa.common.freebusy.ui.SuggestionListPanel = Ext.extend(Ext.Panel, {
	/**
	 * @constructor
	 * @param {Object} config Configuration object
	 */
	constructor : function(config)
	{
		config = config || {};

		config = Ext.applyIf(config, {
			layout: {
				type: 'vbox',
				align: 'stretch'
			},
			border: false,
			bodyStyle: 'background-color: inherit;',
			items: [{
				xtype: 'panel',
				border: true,
				autoHeight: true,
				bodyStyle: 'background-color: inherit; padding: 3px; border-style: none none solid none;',
				items: [{
					xtype: 'container',
					border : false,
					height : 168,
					width: 175,
					layout : 'fit',
					items : [{
						xtype: 'datepicker',
						ref: '../../suggestionDate',
						showToday : false,
						minDate : config.model ? config.model.getDateRange().getStartDate() : null,
						minText: _('No freebusy information available for this date'),
						maxDate : config.model ? config.model.getDateRange().getDueDate() : null,
						maxText: _('No freebusy information available for this date'),
						style : 'border : none',
						listeners: {
							select: this.onDateSelect,
							scope: this
						}
					}]
				}]
			},{
				xtype: 'zarafa.freebusysuggestionlistview',
				bodyStyle: 'background-color: inherit;',
				flex: 1,
				autoScroll: true,
				model: config.model,
				listeners: {
					click: this.onSuggestionClick,
					scope: this
				}
			}]
		});
		
		this.addEvents([
			/**
			 * @event dateselect
			 * This event is fired when a date is selected from the datepicker.
			 * @param {Zarafa.common.freebusy.ui.SuggestionListPanel} this This panel
			 * @param {Date} date The selected date
			 */
			'dateselect',
			/**
			 * @event select
			 * This event is fired when a suggestion has been selected from the
			 * main {@link Zarafa.common.freebusy.ui.FreebusySuggestionListView FreebusySuggestionListView}
			 * @param {Zarafa.common.freebusy.ui.SuggestionListPanel} this This panel
			 * @param {Zarafa.common.freebusy.data.FreebusyBlockRecord} record The record containing the suggestion which was selected
			 */
			'select'
		]);

		Zarafa.common.freebusy.ui.SuggestionListPanel.superclass.constructor.call(this, config);
	},

	/**
	 * Initialize events
	 * @private
	 */
	initEvents: function()
	{
		Zarafa.common.freebusy.ui.SuggestionListPanel.superclass.initEvents.call(this);

		this.mon(this.model.getDateRange(), 'update', this.onDateRangeUpdate, this);
		this.mon(this.model.getSuggestionRange(), 'update', this.onSuggestionRangeUpdate, this);
	},

	/**
	 * Event handler which is raised when a suggestion from the {@link Ext.DataView DataView} has been
	 * clicked. This will search for the corresponding record and raise the {@link #selected} event.
	 * @param {Ext.DataView} view The DataView which raised the event
	 * @param {Number} index The index of the item which was clicked
	 * @param {HTMLElement} el The HTML element which was clicked
	 * @param {Object} event The event object
	 * @private
	 */
	onSuggestionClick : function(view, index, el, event)
	{
		var record = view.getStore().getAt(index);
		this.fireEvent('select', this, record);
	},

	/**
	 * Event handler which is raised when a date has been selected from the {@link Ext.DatePicker DatePicker}.
	 * This will update the suggestionlist to show the suggestions for the given {@link Ext.Date date}. It 
	 * will also fire the {@link #dateselect} event.
	 * @param {Ext.DatePicker} datepicker The datepicker object which raised the event
	 * @param {Ext.Date} date The selected date
	 * @private
	 */
	onDateSelect : function(datepicker, date)
	{
		this.model.setSuggestionDate(date);

		this.fireEvent('dateselect', this, date);
	},

	/**
	 * Event handler which is raised when the daterange of the freebusy has been updated,
	 * this will update the minValue/maxValue of the {@link Ext.DatePicker DatePicker}
	 * accordingly.
	 * @param {Zarafa.core.DateRange} newRange The new DateRange
	 * @param {Zarafa.core.DateRange} oldRange The old DateRange
	 * @private
	 */
	onDateRangeUpdate : function(newRange, oldRange)
	{
		this.suggestionDate.setMinDate(newRange.getStartDate());
		this.suggestionDate.setmaxDate(newRange.getDueDate());
	},

	/**
	 * Event handler which is raised when the suggestion daterange has been updated, this will
	 * update the {@link Ext.DatePicker DatePicker} to display the new value.
	 * @param {Zarafa.core.DateRange} newRange The new DateRange
	 * @param {Zarafa.core.DateRange} oldRange The old DateRange
	 * @private
	 */
	onSuggestionRangeUpdate : function(newRange, oldRange)
	{
		this.suggestionDate.setValue(newRange.getStartDate());
	}
});

Ext.reg('zarafa.freebusysuggestionlistpanel', Zarafa.common.freebusy.ui.SuggestionListPanel);
Ext.namespace('Zarafa.common.freebusy.ui');

/**
 * @class Zarafa.common.freebusy.ui.SuggestionListView
 * @extends Ext.DataView
 * @xtype zarafa.freebusysuggestionlistview
 *
 * The SuggestionList shows the list of suggested times
 * at which an appointment can be held, this view will only
 * display the possible times for a particular date. Loading
 * of data (and date selection) occurs outside of this view.
 */
Zarafa.common.freebusy.ui.SuggestionListView = Ext.extend(Ext.DataView, {

	/**
	 * @constructor
	 * @param {Object} config Configuration object
	 */
	constructor : function(config)
	{
		config = config || {};

		var suggestionBlockStore = config.model ? config.model.getSuggestionBlockStore() : undefined;

		Ext.applyIf(config, {
			deferEmptyText : false,
			emptyText: '<div class="zarafa-freebusy-suggestionrow zarafa-freebusy-suggestionrow-empty">' + _('No suggestions available') + '</div>',
			tpl: new Ext.XTemplate(
				'<tpl for=".">',
				'<div class="x-btn zarafa-freebusy-suggestionrow" id="zarafa-freebusy-suggestionrow-{[xindex]}">',
					'<div class="x-btn-small x-btn-icon-small-left">',
						'<em class="" class="x-unselectable" unselectable="on">',
							'<button type="button" class=" x-btn-text">',
								'<img class="x-freebusy-suggestionrow-icon" src="' + Ext.BLANK_IMAGE_URL + '" />',
								'{[this.formatTime(values.start)]} - {[this.formatTime(values.end)]}',
							'</button>',
						'</em>',
					'</div>',
				'</div>',
				'</tpl>',
			{
				formatTime : function(time)
				{
					// # TRANSLATORS: See http://docs.sencha.com/ext-js/3-4/#!/api/Date for the meaning of these formatting instructions
					return new Date(time * 1000).format(_('G:i'));
				}

			}),
			store: suggestionBlockStore,
			itemSelector: 'div.zarafa-freebusy-suggestionrow'
		});

		Zarafa.common.freebusy.ui.SuggestionListView.superclass.constructor.call(this, config);

		this.mon(this.model, 'suggestionblockstorechange', this.onSuggestionBlockStoreChange, this);
	},

	/**
	 * Event listener which is triggered when a new SuggestionBlockStore has been
	 * loaded on the {@link Zarafa.common.freebusy.data.FreebusyModel FreebusyModel} class.
	 *
	 * @param {Ext.data.Store} store The new store which has been set
	 */
	onSuggestionBlockStoreChange : function(store)
	{
		this.bindStore(store);
	}
});

Ext.reg('zarafa.freebusysuggestionlistview', Zarafa.common.freebusy.ui.SuggestionListView);
Ext.namespace('Zarafa.common.freebusy.ui');

/**
 * @class Zarafa.common.freebusy.ui.TimelineView
 * @extends Ext.BoxComponent
 * @xtype zarafa.freebusytimelineview
 */
Zarafa.common.freebusy.ui.TimelineView = Ext.extend(Ext.BoxComponent,
{
	/**
	 * @cfg {Zarafa.common.freebusy.data.FreebusyModel} model
	 * <b>This is a required setting</b>. The {@link Zarafa.common.freebusy.data.FreebusyModel} holds the data for the
	 * Freebusy component.
	 */
	model: null,
	/**
	 * @cfg {Zarafa.common.freebusy.data.TimelineSelector} selector
	 * The used in the timeline.
	 */
	selector: null,
	/**
	 * @cfg {Number} headerHeight
	 * The height of the header (defaults to 50).
	 */
	headerHeight: 50,
	/**
	 * @cfg {Number} defaultHourCellWidth
	 * The width of the cells displaying an hour on the timeline at 100% zoomlevel (defaults to 60).
	 */
	defaultHourCellWidth: 60,
	/**
	 * @cfg {Number} bufferTimesViewportWidth
	 * To determine what range should be loaded this property is used to to see how many times the
	 * width of the viewport should be loaded in days. If the viewport is 100px wide and the
	 * bufferTimesViewportWidth is set to 5, the range that will be loaded is 500px.(defaults to 5).
	 */
	bufferTimesViewportWidth: 5,	// Times the width of viewport, not days
	/**
	 * The daysMap is an array mapping of all the days that the freebusy timeline will show. For
	 * each day there is an object defined with the following information.
	 * {
	 *   label: Label of the day "Tuesday 12 February 2010"(String)
	 *   currentDay: Indication whether this day is the current day (Boolean)
	 *   timestamp: Timestamp of the start of the day (Number)
	 *   displayNodeHeader: HTML node of the element that shows the day in the header, if not displayed set to false (HTML Element|boolean}
	 *   displayNodeBody: HTML node of the element that shows the day in the body, if not displayed set to false (HTML Element|boolean}
	 *   leftPos: The pixel offset for this day
	 * }
	 * This property is set by the buildDayMapping method.
	 */
	daysMap: null,
	/**
	 * The hoursEachDayMap is an array mapping of the hours that have to be shown for each day. Each
	 * day has the an object defined with the following information.
	 * {
	 *   label: Label of the hour "12:00"(String)
	 *   startDayOffset: Number of seconds since start of the day (Number)
	 *   workinghour: Indication whether this hour is a working hour(Boolean)
	 * }
	 * This property is set by the buildDayMapping method.
	 */
	hoursEachDayMap: null,
	/**
	 * @cfg {Number} daySpacing
	 * The spacing between two days in pixels (defaults to 10).
	 */
	daySpacing: 10,
	/**
	 * @cfg {Number} daySpacing
	 * The width of the borders. The cellspacing in the tables is used to create the borders (defaults to 1).
	 * Changing this property to another value alone is not witout issues.
	 */
	borderSpacing: 1,
	/**
	 * @cfg {Number} blockRowHeight
	 * The height of the block rows (defaults to 22).
	 */
	blockRowHeight: 22,
	/**
	 * @cfg {Number} sumBlockRowHeight
	 * The height of the sumblock rows (defaults to 10).
	 */
	sumBlockRowHeight : 10,
	/**
	 * @cfg {Number} blockRowHeight
	 * Height that is added to the body container to match the height of the userlist body (defaults to 0).
	 */
	extraBodyHeight: 0,
	/**
	 * @cfg {Number} workingHoursStart
	 * Defines the first working hour. This is defined as the number of minutes
	 * since the start of the day. (defaults to the default settings value for
	 * 'zarafa/v1/main/start_working_hour').
	 */
	workingHoursStart: 0,
	/**
	 * @cfg {Number} workingHoursEnd
	 * Defines the hour the working hours range ends. This is defined as the number of minutes
	 * since the start of the day. (defaults to the default settings value for
	 * 'zarafa/v1/main/end_working_hour').
	 */
	workingHoursEnd: 24,
	/**
	 * @cfg {Array} workDays
	 * Defines the array of day numbers containing all working days (default to
	 * the default settings value for 'zarafa/v1/main/working_days').
	 */
	workDays: undefined,
	/**
	 * @cfg {String} blockSelector
	 * <b>This is a required setting</b>. A simple CSS selector (e.g. <tt>div.some-class</tt> or
	 * <tt>span:first-child</tt>) that will be used retrieve the block nodes this for TimelineView.
	 */
	blockSelector: 'div.x-freebusy-timeline-block',

	/**
	 * The height of the top row of the timeline header. In this row the date of the day is placed.
	 * This value is calculated based on the headerHoursHeight and the headerSumRowHeight which are
	 * subtracted from the total available headerHeight.
	 * @property
	 * @type Number
	 * @private
	 */
	headerDayHeight: 0,

	/**
	 * The height of the hours row in the timeline header.
	 * @property
	 * @type Number
	 * @private
	 */
	headerHoursHeight: 15,

	/**
	 * The height of the sum row (all attendees bar) in the timeline header.
	 * @property
	 * @type Number
	 * @private
	 */
	headerSumRowHeight: 10,

	/**
	 * Number of seconds per hour slot. Zoom of 100% will have 60 minutes per slot (in seconds).
	 * @property
	 * @type Number
	 * @private
	 */
	slotDuration: null,

	/**
	 * The width of a day, set by the buildDayMapping method.
	 * @property
	 * @type Number
	 * @private
	 */
	dayWidth: null,

	/**
	 * The width of an hour, set by the buildDayMapping method.
	 * @property
	 * @type Number
	 * @private
	 */
	hourWidth: null,

	/**
	 * The width of the entire timeline, set by the buildDayMapping method
	 * @property
	 * @type Number
	 * @private
	 */
	timelineWidth: null,

	/**
	 * List of block elements in a {@link Ext.CompositeElementLite}.
	 * @property
	 * @type Array
	 * @private
	 */
	all: null,

	/**
	 * The user store which contains all the users for which the freebusy data will be shown.
	 * This store is obtained from the {@link #model}.
	 * @property
	 * @type Ext.data.Store
	 * @private
	 */
	userStore: null,

	/**
	 * The block store which contains all the blocks which must be rendered on the row for the user to which the block belongs to
	 * This store is obtained from the {@link #model}.
	 * @property
	 * @type Ext.data.Store
	 * @private
	 */
	blockStore : null,

	/**
	 * The sumblock store which contains all the sumblocks which must be rendered at the top of the timeline.
	 * This store is obtained from the {@link #model}.
	 * @property
	 * @type Ext.data.Store
	 * @private
	 */
	sumBlockStore : null,

	/**
	 * Id prefix to make freebusy blocks on the timeline unique in combination with the record ID's
	 * from the block store records.
	 * @property
	 * @type String
	 * @private
	 */
	uniqueBlockId: null,

	/**
	 * @constructor
	 * @param {Object} config The configuration options.
	 */
	constructor: function(config)
	{
		config = config || {};

		config = Ext.applyIf(config, {
			workingHoursStart: container.getSettingsModel().get('zarafa/v1/main/start_working_hour'),
			workingHoursEnd: container.getSettingsModel().get('zarafa/v1/main/end_working_hour'),
			workDays: container.getSettingsModel().get('zarafa/v1/main/working_days')
		});

		// No working days, default back to the entire week.
		// FIXME: Not sure if this is the right location for this check,
		// we should consider a more appropriate location for validating
		// the settings.
		if (Ext.isEmpty(config.workDays)) {
			config.workDays = [ 0, 1, 2, 3, 4, 5, 6 ];
		}

		this.addEvents(
			/**
			 * @event bodyscroll
			 * Fires when the body of the timeline is scrolled.
			 * @param {Object} An object containing the scroll position in the format {left: (scrollLeft), top: (scrollTop)}
			 */
			'bodyscroll',
			/**
			 * @event mousedown
			 * Fires when a mousedown is detected within the timeline.
			 * @param {Ext.EventObject} e The {@link Ext.EventObject} encapsulating the DOM event.
			 * @param {HtmlElement} t The target of the event.
			 * @param {Object} o The options configuration passed to the {@link #addListener} call.
			 */
			'timelinemousedown',
			/**
			 * @event timelinemouseup
			 * Fires when a mouseup is detected within the timeline.
			 * @param {Ext.EventObject} e The {@link Ext.EventObject} encapsulating the DOM event.
			 * @param {HtmlElement} t The target of the event.
			 * @param {Object} o The options configuration passed to the {@link #addListener} call.
			 */
			'timelinemouseup',
			/**
			 * @event timelinemousemove
			 * Fires when a mousemove is detected with the timeline.
			 * @param {Ext.EventObject} e The {@link Ext.EventObject} encapsulating the DOM event.
			 * @param {HtmlElement} t The target of the event.
			 * @param {Object} o The options configuration passed to the {@link #addListener} call.
			 */
			'timelinemousemove',
			/**
			 * @event rendertimeline
			 * Fires when the timeline elements are rendered.
			 * @param {Zarafa.common.freebusy.ui.TimelineView} Reference to the TimelineView
			 */
			'rendertimeline',
			/**
			 * @event beforerefreshtimeline
			 * Fires before the timeline is refreshed.
			 * @param {Zarafa.common.freebusy.ui.TimelineView} Reference to the TimelineView
			 */
			'beforerefreshtimeline',
			/**
			 * @event afterrefreshtimeline
			 * Fires after the timeline is refreshed and the sizes of the timeline have been
			 * recalculated.
			 * @param {Zarafa.common.freebusy.ui.TimelineView} Reference to the TimelineView
			 */
			'afterrefreshtimeline'
		);

		Zarafa.common.freebusy.ui.TimelineView.superclass.constructor.call(this, config);

		this.daterange = this.model.getDateRange();
		this.mon(this.model, {
			'userstorechange': this.onUserStoreChange,
			'blockstorechange': this.onBlockStoreChange,
			'sumblockstorechange': this.onSumBlockStoreChange,
			'showworkinghourschange': this.onShowWorkingHoursChange,
			scope: this
		});
	},

	/**
	 * Sets up the required templates, prepares the list of freebusy blocks and seeks for the store.
	 * @private
	 */
	initComponent: function(){
		this.masterTpl = new Ext.XTemplate(
			'<div class="x-freebusy-header">',
				'<div class="x-freebusy-header-body"></div>',
				'<div class="x-freebusy-sumblockcontainer"></div>',
			'</div>',
			'<div class="x-freebusy-body">',
				'<div class="x-freebusy-background"></div>',
				'<div class="x-freebusy-blockcontainer"></div>',
				'<div class="x-freebusy-selectorcontainer"></div>',
			'</div>',
			{
				// Format functions like capitalize in the Ext.util.Format are not
				// used in this template anyways. Speeds up the apply time.
				disableFormats: true
			}
		);

		this.headerTemplate = new Ext.XTemplate(
			'<table class="x-freebusy-timeline-day-header" cellpadding="0" cellspacing="{borderSpacing}" style="width: {dayWidth}px">',
				'<tr class="x-freebusy-timeline-day">',
					'<td colspan="{numHours}" style="height:{headerDayHeight}px;">',
					'{dayLabel}',
					'</td>',
				'</tr>',
				'<tr class="x-freebusy-timeline-hour">',
					'<tpl for="hours">',
						'<td style="width: {parent.hourWidth}px; height:{parent.headerHoursHeight}px;">{label}</td>',
					'</tpl>',
				'</tr>',
			'</table>',
			{ disableFormats: true }
		);
		this.headerSumTemplate = new Ext.XTemplate(
			'<table class="x-freebusy-timeline-day-sum" cellpadding="0" cellspacing="{borderSpacing}" style="width: {dayWidth}px">',
				'<tr class="x-freebusy-timeline-hour x-freebusy-timeline-sum">',
					'<tpl for="hours">',
						'<td style="width: {parent.hourWidth}px; height:{parent.headerSumRowHeight}px;"></td>',
					'</tpl>',
				'</tr>',
			'</table>',
			{ disableFormats: true }
		);
		this.bodyBGTemplate = new Ext.XTemplate(
			'<table class="x-freebusy-timeline-day-body" cellpadding="0" cellspacing="{borderSpacing}" style="width: {dayWidth}px">',
				'<tr class="x-freebusy-timeline-hour">',
					'<tpl for="hours">',
						'<td style="width: {parent.hourWidth}px">&nbsp;</td>',
					'</tpl>',
				'</tr>',
			'</table>',
			{ disableFormats: true }
		);
		this.blockTemplate = new Ext.XTemplate(
			'<tpl for=".">',
				'<div class="x-freebusy-timeline-block x-freebusy-timeline-block-{busyStatusName}" id="{blockId}" style="width: {blockWidth}px; left: {blockLeft}px; top:{blockTop}px; height: {blockHeight}px;"></div>',
			'</tpl>',
			{ disableFormats: true }
		);
		this.sumBlockTemplate = new Ext.XTemplate(
			'<tpl for=".">',
				'<div class="x-freebusy-timeline-sumblock x-freebusy-timeline-block-{busyStatusName}" style="width: {blockWidth}px; left: {blockLeft}px; height: {blockHeight}px;"></div>',
			'</tpl>',
			{ disableFormats: true }
		);

		this.uniqueBlockId = Ext.id();
		this.all = new Ext.CompositeElementLite();

		Zarafa.common.freebusy.ui.TimelineView.superclass.initComponent.call(this);

		if(this.selector){
			this.selector.init(this);
		}
	},

	/**
	 * Renders the background container and sets up the day and hour mappings.
	 * @private
	 */
	onRender: function(){
		this.autoEl = {
			cls: 'x-freebusy-timeline-container'
		};
		Zarafa.common.freebusy.ui.TimelineView.superclass.onRender.apply(this, arguments);

//TODO: The initial render should also contain a refresh call to directly put in all the blocks of the already loaded users.
		this.buildDayMapping();
		this.renderTimeline();
	},

	/**
	 * Called to re-render the entire TimelineView.
	 * @private
	 */
	refreshTimeline: function(){
		this.fireEvent("beforerefreshtimeline", this);

		// Capture the snapshot of the viewable area.
		var viewportSnapshot = this.captureViewportSnapshot();
		// Clearing the old HTML node references to the viewed days in the timeline background as
		// well as regenerating the list of hours and days shown on the timeline.
		this.buildDayMapping();
		// Rendering the HTML elements for the timeline (and discarding the old ones).
		this.renderTimeline();
		// Triggering the resizing of the timeline. Will also trigger the repainting of the timeline days.
		this.syncSize();
		// Restore the viewable area inside the viewport
		this.restoreViewportSnapshot(viewportSnapshot);
		// Triggers the repainting of the blocks on the timeline.
		this.refresh();

		this.fireEvent("afterrefreshtimeline", this);
	},

	/**
	 * Creates a snapshot object that will contain information to get the viewport of the timeline
	 * focussed on the same area.
	 * The object contains the following properties.
	 * <ul>
	 * <li>selectionInView {@link Boolean} indication whether the selection of the {@link #selectorRange}
	 * is visible inside the viewport.</li>
	 * <li>focusDateRange {@link Zarafa.core.DateRange} The period of time that is centered on in the
	 * viewport.</li>
	 * <li>diffOffset {@link Number} The number of pixels between the start of the selection of the
	 * {@link #selectorRange} and the center of the viewport. A negative values means the selection
	 * is before the center.</li>
	 * </ul>
	 * @return {Object} An object containing selectionInView, focussedDateRange and diffOffset properties.
	 */
	captureViewportSnapshot: function(){
		// Setup the viewportSnapshot
		var snapshot = {
			selectionInView: false,
			focusDateRange: null,
			diffOffset: null
		};

		// Check to see if the selection is inside the view
		var selectorRange = this.model.getSelectorRange();
		// FIXME check if we have a selectorRange (Freebusy without selection is possible)
		snapshot.selectionInView = this.getViewedDateRange().overlaps(selectorRange);

		// If selection is inside view calculate the offset between the start of the selection and
		// the center of the viewport in pixels
		if(snapshot.selectionInView){
			var selectionStart = selectorRange.getStartDate();
			var startOffset = this.findBlockPixelOffset(selectionStart.getTime()/1000);

			// Get the left side of the viewport
			var leftOffset = this.bodyElem.getScroll().left;
			// Use the half of the width of the viewport plus the leftOffset to get the center offset.
			var viewportSize = Ext.get(this.bodyElem).getViewSize();
			var centerOffset = leftOffset + (viewportSize.width/2);

			// Get difference between centerOffset(viewport) and the startOffset(start of selection)
			snapshot.diffOffset = startOffset - centerOffset;

		}else{
			// Retrieve the focus to be used after the refreshing has been done
			snapshot.focusDateRange = this.getFocusDateRange();
		}

		return snapshot;
	},

	/**
	 * Will restore the viewport based on the snapshot it will get passed. If the user selection was
	 * visible inside the viewport when the snapshot was created, than this method will keep the
	 * selection at the same place inside the viewport.
	 * If the selection is outside the viewable area than it will center on the time it was centered
	 * on when the snapshot was created.
	 * The difference between these two approaches is that when you have the selection in view it
	 * will not get scrolled away when it is visible near the edge of the viewport. Which would be
	 * the case if you were to center on the position that was in the middle of the viewport. If the
	 * selection was not visible then we do not need to go through all that trouble.
	 * @param {Object} snapshot Needs a snapshot that is returned by the
	 * {@link @createViewportSnapshot} method.
	 */
	restoreViewportSnapshot: function(snapshot){
		var focusDateRange;

		if(snapshot.selectionInView){
			// Get the start date of the selection
			var selectorRange = this.model.getSelectorRange();
			var selectionStart = selectorRange.getStartDate();

			// Use the start of the selection to get the offset in pixels from the start of the timeline.
			var startOffset = this.findBlockPixelOffset(selectionStart.getTime()/1000);
			// Based on the startOffset we now calculate the position the viewport should be centered on.
			var centerOffset = startOffset - snapshot.diffOffset;
			// Transform this center offset in pixels to a timestamp
			var centerTimestamp = this.findTimestampByTimelineXCoord(centerOffset);

			// Create a daterange to be used to focus the viewport on
			focusDateRange = new Zarafa.core.DateRange({ startDate : new Date(centerTimestamp*1000), dueDate : new Date(centerTimestamp*1000) });
		}else{
			// Just focus on the center when the selection was not visible
			focusDateRange = snapshot.focusDateRange;
		}

		// Scroll to the DateRange of the original focus
		this.scrollDateIntoView(focusDateRange);
	},

	/**
	 * Returns the {@link Zarafa.core.data.DateRange DateRange} that the focus is centered on. It
	 * looks for the hour slot that is in the center of the viewport. That slot will be returned in
	 * a DateRange.
	 * @return {Zarafa.core.DateRange} Range of the focus
	 */
	getFocusDateRange: function(){
		// Get the dimensions and scroll offset
		var leftOffset = this.bodyElem.getScroll().left;
		var viewportSize = Ext.get(this.bodyElem).getViewSize();

		// Get the offset of the center (focus) of the viewport
		var focusOffset = leftOffset + (viewportSize.width/2);
		// The dayWidth is excluding the spaces between the days so we have to include those too
		var dayIndex = focusOffset / (this.dayWidth + this.daySpacing);

		// Get the decimal part from the dayIndex, to be used for calculating what part of the day to focus on
		var dayRatioIndex = dayIndex % 1;
		dayIndex = Math.floor(dayIndex);

		var dayMap = this.daysMap[ dayIndex ];
		var numDisplayedHours = this.hoursEachDayMap.length;
		// Get the index of the hour the focus is centered on using the decimal part of the dayIndex
		var hourIndex = Math.floor(numDisplayedHours * dayRatioIndex);

		// Get the start and the end of the hour slot that is focus is centered on
		var focusStartTimestamp = dayMap.timestamp + this.hoursEachDayMap[ hourIndex ].startDayOffset
		var focusEndTimestamp = focusStartTimestamp + this.slotDuration

		return new Zarafa.core.DateRange({ startDate : new Date(focusStartTimestamp*1000), dueDate : new Date(focusEndTimestamp*1000) });
	},

	/**
	 * Returns the {@link Zarafa.core.data.DateRange DateRange} that the viewport is showing. The
	 * start date is set to the date that the left edge of the viewport is on and the due date is
	 * set to the date the right edge of the viewport is on.
	 * @return {Zarafa.core.DateRange} Range of the viewed area
	 */
	getViewedDateRange: function(){
		// Get scroll offset of the left and right edges of the viewport
		var leftOffset = this.bodyElem.getScroll().left;
		var viewportSize = Ext.get(this.bodyElem).getViewSize();
		var rightOffset = leftOffset + viewportSize.width;

		// Transform the scroll offsets to timestamps
		var startTimestamp = this.findTimestampByTimelineXCoord(leftOffset);
		var endTimestamp = this.findTimestampByTimelineXCoord(rightOffset);

		return new Zarafa.core.DateRange({ startDate : new Date(startTimestamp*1000), dueDate : new Date(endTimestamp*1000) });
	},

	/**
	 * Rendering the HTML elements needed for the TimelineView. Will also create references of to
	 * important HTML elements and will header and body elements.
	 * @private
	 */
	renderTimeline: function(){
		// Remove the old scroll event from the body and header
		if(this.bodyElem){
			this.mun(this.bodyElem, {
				scope: this,
				"scroll": this.onBodyScroll,
				"mousedown": this.onBodyMouseDown,
				"mousemove": this.onBodyMouseMove,
				"mouseup": this.onBodyMouseUp,
				"contextmenu": this.onBodyContextMenu
			});
		}
		if(this.headerElem) {
			this.mun(this.headerElem, {
				scope: this,
				"contextmenu": this.onHeaderContextMenu
			});
		}

		// Setup the HTML structure for the whole TimelineView
		this.masterTpl.overwrite(this.el);

		// Set references to the different elements in the timeline
		this.containerElem = Ext.get(this.el.dom);
		this.headerElem = Ext.get(this.containerElem.dom.firstChild);
		this.headerBodyElem = Ext.get(this.headerElem.dom.firstChild);
		this.headerSumContainer = Ext.get(this.headerElem.dom.childNodes[1]);
		this.bodyElem = Ext.get(this.containerElem.dom.childNodes[1]);
		this.bodyBackgroundElem = Ext.get(this.bodyElem.dom.firstChild);
		this.bodyBlockContainer = Ext.get(this.bodyElem.dom.childNodes[1]);
		this.bodySelectorContainer = Ext.get(this.bodyElem.dom.childNodes[2]);

		// Set scroll event on body to fire scroll event in this TimelineView
		this.mon(this.bodyElem, {
			scope: this,
			"scroll": this.onBodyScroll,
			"mousedown": this.onBodyMouseDown,
			"mousemove": this.onBodyMouseMove,
			"mouseup": this.onBodyMouseUp,
			"contextmenu": this.onBodyContextMenu
		});
		this.mon(this.headerElem, {
			scope: this,
			"contextmenu": this.onHeaderContextMenu
		});

		// Position and set width and height for header and body elements in timeline
		this.layoutTimelineElements();

		this.fireEvent("rendertimeline", this);
	},

	/**
	 * Set the correct width for the header, body background timeline and body block container
	 * elements. It will also calculate the correct height for the header of the timeline.
	 * @private
	 */
	layoutTimelineElements: function(){
		this.headerBodyElem.setWidth(this.timelineWidth);
		this.bodyBackgroundElem.setWidth(this.timelineWidth);
		this.bodyBlockContainer.setWidth(this.timelineWidth);

		// Calculate the height settings of the header elements.
		// Header consists of three rows. The day row and the hours row have no horizontal border-spacing.
		// The header sum row is a separate table and has two border-spacings (top & bottom).
		var heightSumRowBordersIncluded = this.headerSumRowHeight + (this.borderSpacing*2);
		this.headerDayHeight = (this.headerHeight - heightSumRowBordersIncluded - this.headerHoursHeight);

		// Calculating the top style for the sumblocks container in the header that contains the
		// cumulative freebusy blocks. For this we take the full height of the header and extract
		// the height of the sum row and the border that is at the bottom.
		var sumBlockContainerTopSpace = this.headerHeight - (this.headerSumRowHeight + this.borderSpacing);
		this.headerSumContainer.setTop(sumBlockContainerTopSpace);

		 // IE6 & IE7 fixes: The check for Ext.isIE7 does not detect the IE7 compatibility mode in IE8.
		if(Ext.isIE6 || Ext.isIE7){
			/**
			 * IE6 and IE7 do not support the border-spacing property. Therefore we have to use the
			 * cellspacing attribute in the <table>-tag. This attribute does not allow us to specify a
			 * separate spacing for the top/bottom borders and the left/right borders. Therefor we need
			 * to add three more pixels to the height of the header to compensate.
			 */
			this.headerDayHeight = this.headerDayHeight - 3;
			/**
			 * The background timeline has the cellspacing set to one pixel as well. This means that
			 * the table also has a border on top. This causes the background to start one pixel
			 * below and is not in sync with rows in the userlist. Therefor the <div> that contains
			 * the background timeline table is moved up one pixel.
			 */
			this.bodyBackgroundElem.setStyle('top', -1);
			this.bodySelectorContainer.setStyle('top', -1);
		}
	},

	/**
	 * Binding the blockStore and the userStore
	 * @private
	 */
	afterRender : function(){
		Zarafa.common.freebusy.ui.TimelineView.superclass.afterRender.apply(this, arguments);

		if (this.model.getBlockStore())
			this.bindBlockStore(this.model.getBlockStore(), true);
		if (this.model.getSumBlockStore())
			this.bindSumBlockStore(this.model.getSumBlockStore(), true);
		if (this.model.getUserStore())
			this.bindUserStore(this.model.getUserStore(), true);
	},

	/**
	 * Get the template target that is used to add the blocks to.
	 * @private
	 */
	getBlockTemplateTarget: function(){
		// We return the container that is used for the freebusy blocks.

		return this.bodyBlockContainer;
	},

	/**
	 * Get the template target for the background days.
	 * @private
	 */
	getBGTemplateTarget: function(){
		// In this container the background days are rendered.
		return this.bodyBackgroundElem;
	},

	/**
	 * Get the HTML Element that functions as the container for the selector.
	 * @return {HTMLElement} Selector container
	 * @private
	 */
	getSelectorContainer: function(){
		// In this container the selector can be rendered.
		return this.bodySelectorContainer;
	},

	/**
	 * Called after the component is resized, this method is empty by default but can be implemented by any
	 * subclass that needs to perform custom logic after a resize occurs.
	 * @param {Number} adjWidth The box-adjusted width that was set
	 * @param {Number} adjHeight The box-adjusted height that was set
	 * @param {Number} rawWidth The width that was originally specified
	 * @param {Number} rawHeight The height that was originally specified
	 * @protected
	 */
	onResize : function(adjWidth, adjHeight, rawWidth, rawHeight){
		this.headerElem.setHeight(this.headerHeight);

		this.bodyElem.setTop(this.headerHeight);
		var bodyElemHeight = adjHeight - this.headerHeight;
		// IE6 returns NaN for the adjHeight when initializing the UI
		this.bodyElem.setHeight( bodyElemHeight || 0 );

		this.sizeTimelineBackground();

		// In IE the bodyElem has the correct only after calling the sizeTimelineBackground().
		this.headerElem.setWidth(this.bodyElem.dom.clientWidth);

		this.repaintTimeline();
	},

	/**
	 * Handles the actions that should take place when the user scrolls the body
	 * @private
	 */
	onBodyScroll: function(){
		this.repaintTimeline();

		this.fireEvent("bodyscroll", this.bodyElem.getScroll());
	},

	/**
	 * Called when the mousedown event is fired on the body.
	 * @param {Ext.EventObject} evt The {@link Ext.EventObject} encapsulating the DOM event.
	 * @param {HtmlElement} target The target of the event.
	 * @param {Object} cfg The options configuration passed to the {@link #addListener} call.
	 * @private
	 */
	onBodyMouseDown: function(evt, target, cfg){
		this.fireEvent("timelinemousedown", evt, target, cfg);
	},

	/**
	 * Called when the mousemove event is fired on the body.
	 * @param {Ext.EventObject} evt The {@link Ext.EventObject} encapsulating the DOM event.
	 * @param {HtmlElement} target The target of the event.
	 * @param {Object} cfg The options configuration passed to the {@link #addListener} call.
	 * @private
	 */
	onBodyMouseMove: function(evt, target, cfg){
		this.fireEvent("timelinemousemove", evt, target, cfg);
	},

	/**
	 * Called when the mouseup event is fired on the body.
	 * @param {Ext.EventObject} evt The {@link Ext.EventObject} encapsulating the DOM event.
	 * @param {HtmlElement} target The target of the event.
	 * @param {Object} cfg The options configuration passed to the {@link #addListener} call.
	 * @private
	 */
	onBodyMouseUp: function(evt, target, cfg){
		this.fireEvent("timelinemouseup", evt, target, cfg);
	},

	/**
	 * Called when the contextmenu event is fired on the body.
	 * @param {Ext.EventObject} evt The {@link Ext.EventObject} encapsulating the DOM event.
	 * @param {HtmlElement} target The target of the event.
	 * @param {Object} cfg The options configuration passed to the {@link #addListener} call.
	 * @private
	 */
	onBodyContextMenu : function(evt, target, cfg)
	{
		Zarafa.core.data.UIFactory.openContextMenu(Zarafa.core.data.SharedComponentType['common.contextmenu.freebusy.timelinebody'], undefined, { position : evt.getXY(), model : this.model });
	},

	/**
	 * Called when the contextmenu event is fired on the header.
	 * @param {Ext.EventObject} evt The {@link Ext.EventObject} encapsulating the DOM event.
	 * @param {HtmlElement} target The target of the event.
	 * @param {Object} cfg The options configuration passed to the {@link #addListener} call.
	 * @private
	 */
	onHeaderContextMenu: function(evt, target, cfg)
	{
		Zarafa.core.data.UIFactory.openContextMenu(Zarafa.core.data.SharedComponentType['common.contextmenu.freebusy.timelineheader'], undefined, { position : evt.getXY(), model : this.model });
	},

	/**
	 * Resizes the vertical sizes based on the number of users that have been added. This needs to
	 * be recalculated because if the number of users exceeds the amount that can be shown in the
	 * viewport at one time the height of the timeline needs to be resized beyond the height of the
	 * viewport. Everytime a new user is added or one is removed this function can be called to
	 * resized the timeline to the correct height.
	 * @private
	 */
	sizeTimelineBackground: function(){
		if(this.bodyElem.dom.clientHeight <= 0) {
			// if height is zero that means timelineView is hidden, so we shouldn't do resizing
			return;
		}

		// Set the height of the timeline background based on the number of users in the user store.
		var userStore = this.userStore;
		var numRows = userStore.getCount();
		var bodyHeight = numRows * this.blockRowHeight + this.extraBodyHeight;

		/**
		 * The columns of the background timeline need to be sized to fill the body viewport. If
		 * there is an active scrollbar then they need to be sized even bigger to also show behind
		 * the blocks that are in the rows outside the viewport.
		 * this.bodyElem.clientHeight => height of body element without scrollbars
		 */
		this.bodyBackgroundElem.setHeight( Math.max( this.bodyElem.dom.clientHeight, bodyHeight ) );
		// Resize the bodybackground
		this.bodySelectorContainer.setHeight( Math.max( this.bodyElem.dom.clientHeight, bodyHeight ) );

		// Always scroll to the bottom of the page...
		this.bodyElem.scrollTo("top", this.bodyElem.dom.scrollHeight);
	},

	/**
	 * Everytime the user scrolls or when the component is resized the background needs to be
	 * redrawn. The background of the timeline only loads the days that are visible and their
	 * surrounding days. To determine what range should be loaded it looks for the
	 * bufferTimesViewportWidth to see how many times the width of the viewport should be loaded in
	 * days. If the viewport is 100px wide and the bufferTimesViewportWidth is set to 5, the range
	 * that will be loaded is 500px.
	 * It will map the range to the days that will need to be loaded. Next it will figure out what
	 * days need to be rendered and what days have to be cleaned up because they do no longer fall
	 * within the range.
	 * @private
	 */
	repaintTimeline: function(){
		var el = this.getBGTemplateTarget();

		// Get some basic values of the width of one day, the size of the
		// viewport and the position of the scrollbar.
		var viewportSize = Ext.get(this.bodyElem).getViewSize();
		var scrollPxls = this.bodyElem.getScroll();

		/**
		 * Next we need to decide what range needs to be buffered. We use the value of
		 * this.bufferTimesViewportWidth to calculate the number of pixels that need to be loaded
		 * outside of the visual area in the viewport. If you want to buffer five times the width of
		 * the viewport (100px) then there are 400px that are shown outside of the viewport.
		 */
		var visiblePxlsOutsideViewport = (this.bufferTimesViewportWidth-1)*viewportSize.width;
		// Calculate the offset in pixels for the start and end point of the visual range.
		var startOffsetPixels = scrollPxls.left - (visiblePxlsOutsideViewport / 2);
		var endOffsetPixels = scrollPxls.left + viewportSize.width + (visiblePxlsOutsideViewport / 2);

		// Calculate what the first and last day is that must be buffered.
		var startDayIndex = Math.floor(startOffsetPixels / this.dayWidth);
		var endDayIndex = Math.ceil(endOffsetPixels / this.dayWidth);

		// Make sure the indexes do not exceed the number of days available.
		startDayIndex = (startDayIndex < 0) ? 0 : startDayIndex;
		endDayIndex = (endDayIndex < this.daysMap.length) ? endDayIndex : (this.daysMap.length - 1);

		// Determine what days need to be loaded extra and what days can be removed.
		var loadDays = [];
		var cleanupDays = [];
		for(var i=0;i<this.daysMap.length;i++){
			// If day is between the range to visualize.
			if(i >= startDayIndex && i <= endDayIndex){
				// Only add to loadDays when it is not displayed yet
				if(!this.daysMap[ i ].displayNodeBody){
					loadDays[ loadDays.length ] = i;
				}
			// If day falls outside visual range
			}else{
				// If day is marked as displayed it needs to be cleaned up
				if(this.daysMap[ i ].displayNodeBody){
					cleanupDays[ cleanupDays.length ] = i;
				}
			}
		}

		// First render days, then scroll, then cleanup. This way no empty header should be shown.
		this.renderTimelineDays( loadDays );
		this.headerElem.scrollTo("left", this.bodyElem.getScroll().left );	// Sync the header
		this.cleanUpTimelineDays( cleanupDays );
	},

	/**
	 * Renders the background days that have been supplied in renderDays.
	 * @param renderDays {Array} List of indexes of this.daysMap that will be rendered.
	 * @private
	 */
	renderTimelineDays: function(renderDays){
		var headerElem = Ext.get(this.headerBodyElem);
		var bodyElem = Ext.get(this.bodyBackgroundElem);

		for(var i=0;i<renderDays.length;i++){
			var dayIndex = renderDays[i];
			var currDayCls = '';
			if(this.daysMap[ dayIndex ].currentDay){
				currDayCls = ' x-freebusy-timeline-day-current';
			}

			var bodyDayElem = bodyElem.createChild({
				cls: 'x-freebusy-timeline-day' + currDayCls
			});
			bodyDayElem.dom.style.left = this.daysMap[ dayIndex ].leftPos+"px";

			this.bodyBGTemplate.overwrite(bodyDayElem, {
				dayLabel: this.daysMap[ dayIndex ].label,
				hours: this.hoursEachDayMap,
				hourWidth: this.hourWidth,
				numHours: this.hoursEachDayMap.length,
				dayWidth: this.dayWidth
			});

			var headerDayElem = headerElem.createChild({
				cls: 'x-freebusy-timeline-day' + currDayCls
			});
			headerDayElem.dom.style.left = this.daysMap[ dayIndex ].leftPos+"px";

			this.headerTemplate.overwrite(headerDayElem, {
				dayLabel: this.daysMap[ dayIndex ].label,
				hours: this.hoursEachDayMap,
				hourWidth: this.hourWidth,
				numHours: this.hoursEachDayMap.length,
				dayWidth: this.dayWidth,
				headerDayHeight: this.headerDayHeight,
				headerHoursHeight: this.headerHoursHeight,
				borderSpacing: this.borderSpacing
			});
			this.headerSumTemplate.append(headerDayElem, {
				hours: this.hoursEachDayMap,
				hourWidth: this.hourWidth,
				numHours: this.hoursEachDayMap.length,
				dayWidth: this.dayWidth,
				headerSumRowHeight: this.headerSumRowHeight,
				borderSpacing: this.borderSpacing
			});

			this.daysMap[ dayIndex ].displayNodeHeader = headerDayElem;
			this.daysMap[ dayIndex ].displayNodeBody = bodyDayElem;
		}
	},

	/**
	 * Cleans up the background days that have been supplied in cleanupDays.
	 * @param cleanupDays {Array} List of indexes of this.daysMap that will be cleaned up.
	 * @private
	 */
	cleanUpTimelineDays: function(cleanupDays){
		var bodyElem = Ext.get(this.bodyBackgroundElem);
		var headerElem = Ext.get(this.bodyBackgroundElem);
		var elem;

		for(var i=0;i<cleanupDays.length;i++){
			var dayIndex = cleanupDays[i];

			elem = this.daysMap[ dayIndex ].displayNodeHeader;
			Ext.removeNode( elem.dom );
			elem = this.daysMap[ dayIndex ].displayNodeBody;
			Ext.removeNode( elem.dom );

			this.daysMap[ dayIndex ].displayNodeHeader = false;
			this.daysMap[ dayIndex ].displayNodeBody = false;
		}
	},

	/**
	 * Scrolls the supplied date into view. It will be centered into the view.
	 * @param date {Date|Number|Zarafa.core.DateRange.js} Date to be scrolled into view.
	 * @private
	 */
	scrollDateIntoView: function(date){
		// Make a timestamp out of a date
		if(date instanceof Date){
			date = Math.ceil(date.getTime()/1000);
		// Make a timestamp out of a Daterange
		}else if(date instanceof Zarafa.core.DateRange){
			var start = date.getStartDate().getTime()/1000;
			var end = date.getDueDate().getTime()/1000;
			var duration = end - start;
			date = start + (duration/2);
		}

		var viewport = Ext.get(this.bodyElem)
		var viewportSize = Ext.get(this.bodyElem).getViewSize();

		var pixelOffsetLeft = this.findBlockPixelOffset(date, true);

		// Make sure the date will be centered in the viewport
		pixelOffsetLeft = pixelOffsetLeft - (viewportSize.width / 2);

		viewport.scrollTo('left', pixelOffsetLeft);
	},

	/**
	 * Creates a mapping for each day and a mapping for the hours that are displayed.
	 * It also calculates and sets the widths for the days and hour cells.
	 * @private
	 */
	buildDayMapping: function()
	{
		var currDay = new Date().clearTime();

		this.daysMap = [];
		this.hoursEachDayMap = [];
		this.slotDuration = 60*60;
		var availableSlotsPerDay = (24*60*60) / this.slotDuration;

		// Dirty way of assigning the working hours
		for(var i=0;i<availableSlotsPerDay;i++) {
			// FIXME: We should support minutes as well, so we shouldn't
			// round to entire hours but to 15 minutes for example.
			var startHour = Math.floor(this.workingHoursStart / 60);
			var endHour = Math.ceil(this.workingHoursEnd / 60);
			if (!this.model.showOnlyWorkingHours() || (i >= startHour && i < endHour)) {
				this.hoursEachDayMap[ this.hoursEachDayMap.length ] = {
					// Run the hour through a formatter
					label: Date.parseDate(i, 'G').format(_('G:i')),
					startDayOffset: i * this.slotDuration,
					workingHour: (i >= startHour && i < endHour) ? true : false
				};
			}
		}

		var numHourBlocks = this.hoursEachDayMap.length;
		this.hourWidth = this.defaultHourCellWidth;
		// Each hour has two borders (left and right) which it shares with it's neighbours.
		this.dayWidth = ( (this.hourWidth + this.borderSpacing) * numHourBlocks ) + this.borderSpacing;

		// Set clone flag in clearTime to prevent changing the time in the daterange
		// Use 12:00 as basetime, to prevent problems when the DST switch is at 00:00
		// like in Brasil.
		var startdate = this.daterange.getStartDate().clearTime(true);
		startdate.setHours(12);

		for (var i = 0; i < this.daterange.getNumDays(); i++) {
			var date = startdate.add(Date.DAY, i).clearTime();

			// Check whether this iterated day is the current day
			var currDayCheck = currDay.getTime() == date.getTime();
			var workDay = false;

			for (var j = 0, len = this.workDays.length; j < len; j++) {
				if (this.workDays[j] == date.getDay()) {
					workDay = true;
					break;
				}
			}

			if (!this.model.showOnlyWorkingHours() || workDay) {
				this.daysMap.push({
					// Run date through formatter
					// # TRANSLATORS: See http://docs.sencha.com/ext-js/3-4/#!/api/Date for the meaning of these formatting instructions
					label: date.format(_('l jS F Y')),
					currentDay: currDayCheck,
					timestamp: date.getTime() / 1000,
					displayNodeHeaderBody: false,
					displayNodeBody: false,
					// Left boundary are used for determining whether to show the day or not
					leftPos: (this.daysMap.length * (this.dayWidth + this.daySpacing))
				});
			}
		}

		var numDays = this.daysMap.length;
		this.timelineWidth = (this.dayWidth + this.daySpacing) * numDays - this.daySpacing;
	},

	/**
	 * Changes the data store bound to this view and refreshes it.
	 * @param {Store} store The store to bind to this view
	 * @private
	 */
	bindBlockStore : function(store, initial)
	{
		if(this.blockStore) {
			this.mun(this.blockStore, 'load', this.onBlockLoad, this);
		}
		this.blockStore = Ext.StoreMgr.lookup(store);
		if(this.blockStore){
			this.mon(this.blockStore, {
				scope: this,
				load: this.onBlockLoad
			});

			this.refresh();
		}
	},

	/**
	 * Changes the data store bound to this view and refreshes it.
	 * @param {Store} store The store to bind to this view
	 * @private
	 */
	bindSumBlockStore : function(store, initial)
	{
		if (this.sumBlockStore) {
			this.mun(this.sumBlockStore, 'load', this.onSumBlocksLoad, this);
		}
		this.sumBlockStore = Ext.StoreMgr.lookup(store);
		if(this.sumBlockStore){
			this.mon(this.sumBlockStore, {
				scope: this,
				load: this.onSumBlocksLoad
			});

			this.refresh();
		}
	},
	/**
	 * load block(s) to the UI.
	 * @param {Ext.data.Store} ds Store of the record
	 * @param {Ext.data.Record|Array} records List of records that have to be loaded
	 * @param {Object} options The options used the load the blocks from the server
	 * @private
	 */
	onBlockLoad : function(ds, records, options){
		if(this.all.getCount() === 0){
			this.refresh();
			return;
		}
		var filteredRecords = this.filterRecords(records);

		var nodes = this.bufferBlockRender(filteredRecords);
		this.all.last().insertSibling(nodes, 'after', true);
		this.all.elements.push.apply(this.all.elements, nodes);
	},

	/**
	 * Called when the {@link #sumBlockStore} has loaded new data and we
	 * can update the top freebusy bar with summed blocks
	 *
	 * Renders the sum blocks that show the cumulative freebusy information for all the recipients.
	 * The data from the blocks is combined into cumulative blocks for all recipients. When this
	 * calculation is done it does it seperately for the different busy statuses. So you will get a
	 * different track for tentative, busy and outofoffice. By displaying them on top of eachother
	 * the outofoffice is more important than busy and busy more important than tentative.
	 *
	 * @param {Ext.data.Store} store The store which raised the event
	 * @oaram {Ext.data.Record|Array} records The records which have been loaded
	 * @param {Object} options The options from the load event
	 * @private
	 */
	onSumBlocksLoad : function(store, records, options)
	{
		if (store.getCount() == 0) {
			// No sum records are available, simply empty the sumBlockTemplate
			this.sumBlockTemplate.overwrite(this.headerSumContainer.dom, []);
			return;
		}

		this.sumBlockTemplate.overwrite(this.headerSumContainer.dom, this.collectData(records, 0, true));
	},

	/**
	 * Remove block from the UI.
	 * @param {Ext.data.Store} ds Store of the record
	 * @param {Ext.data.Record} record Record begin removed
	 * @param {Number} index Index of the record that has to be removed
	 * @private
	 */
	onRemove : function(ds, record, index){
		var blockId = this.uniqueBlockId+"-"+record.id;
		this.all.removeElement(blockId, true);
		if (this.blockStore.getCount() === 0){
			this.refresh();
		}
	},

	/**
	 * Refreshes the view by reloading the data from the store and re-rendering the template.
	 * @private
	 */
	refresh : function(){
		// Empty the container of blocks
		this.bodyBlockContainer.update("");
		this.all.clear();

		if (this.blockStore) {
			var records = this.blockStore.getRange();

			if (records.length > 0) {
				var filteredRecords = this.filterRecords(records);
				this.blockTemplate.overwrite(this.bodyBlockContainer.dom, this.collectData(filteredRecords));
				var nodes = Ext.query(this.blockSelector, this.bodyBlockContainer.dom);
				this.all.fill(nodes);
			}
		}

		if (this.sumBlockStore) {
			var sumRecords = this.sumBlockStore.getRange();

			if (sumRecords.length > 0) {
				var filteredSumRecords = this.filterRecords(sumRecords);
				this.sumBlockTemplate.overwrite(this.headerSumContainer.dom, this.collectData(filteredSumRecords));
			}
		}
	},

	/**
	 * Render the HTML for the records and return the nodes.
	 * @param {Ext.data.Record|Array} records Records
	 * @return {Array} List of HTML nodes
	 * @private
	 */
	bufferBlockRender : function(records){
		var div = document.createElement('div');
		this.blockTemplate.overwrite(div, this.collectData(records));
		var nodes = Ext.query(this.blockSelector, div);
		return nodes;
	},

	/**
	 * Call prepareData for all the records in the list and return the list of data for all records.
	 * @param {Ext.data.Record|Array} records List of records
	 * @param {Number} startIndex Start index
	 * @param {Boolean} sumHeader True if the blocks will be positioned into the SumBlock header.
	 * @return {Array} List of data for all records
	 * @private
	 */
	collectData : function(records, startIndex, sumHeader){
		var r = [];
		for(var i = 0, len = records.length; i < len; i++){
			r[r.length] = this.prepareData(records[i].data, startIndex+i, records[i], sumHeader);
		}
		return r;
	},

	/**
	 * Function that provides the formatting needed for the freebusy blocks to be displayed
	 * properly. It determines what the left pixel offset should be and what the width should be.
	 * This function is used to provide custom formatting for each Record that is used by this
	 * {@link Ext.DataView}'s {@link #tpl template} to render each node.
	 * @param {Array/Object} data The raw data object that was used to create the Record.
	 * @param {Number} recordIndex the index number of the Record being prepared for rendering.
	 * @param {Record} record The Record being prepared for rendering.
	 * @param {Boolean} sumHeader True if the block will be positioned into the SumBlock header.
	 * @return {Array/Object} The formatted data in a format expected by the internal {@link #tpl template}'s overwrite() method.
	 * (either an array if your params are numeric (i.e. {0}) or an object (i.e. {foo: 'bar'}))
	 * @private
	 */
	prepareData: function(data, recordIndex, record, sumHeader){
		// Copy the data object to prevent adding properties to the Record object
		data = Ext.apply({}, data);

		data.blockId = this.uniqueBlockId + "-" + record.id;
		if(data.status != Zarafa.core.mapi.BusyStatus.UNKNOWN){
			data.busyStatusName = Zarafa.core.mapi.BusyStatus.getName(data.status) || 'busy';
			data.busyStatusName = data.busyStatusName.toLowerCase();
		}else{
			data.busyStatusName = 'blur';

			if(this.model.showOnlyWorkingHours()) {
				var day = new Date(data.start*1000).getDay();
				// if day is Saturday then add two day in current day and get time stamp of Monday
				if(day === 6) {
					data.start = new Date(data.start*1000).add(Date.DAY, 2).getTime()/1000;
				} else if(day === 0) {
					// if day is Sunday then add one day in current day and get time stamp of Monday
					data.start = new Date(data.start*1000).add(Date.DAY, 1).getTime()/1000;
				}
			}
			var dayIndex = this.findDayIndexByTimestamp(data.start,true);
			var timestamp = this.daysMap[dayIndex].timestamp;
			data.start = timestamp;
		}

		var startRowTopOffset = 1;
		var rowHeight;

		if (sumHeader === true) {
			rowHeight = this.sumBlockRowHeight;
		} else {
			rowHeight = this.blockRowHeight;

			// Find the index of the user record
			var userIndex = this.userStore.indexOfId(record.get('userid'));

			if (userIndex >= 0) {
				startRowTopOffset += userIndex * rowHeight;
			}
		}

		data.blockTop = startRowTopOffset;

		var periodStart = this.daterange.getStartDate().getTime()/1000;
		var periodEnd = this.daterange.getDueDate().getTime()/1000;
		// The start/end date from the record have to be parsed since they are in String format and
		// we will have to do some calculations with them.
		var blockStart = parseInt(data.start, 10);
		var blockEnd = parseInt(data.end, 10);

		// Filter out any records that for some reason are beyond the period range.
		if(blockStart < periodEnd && blockEnd > periodStart){

			data.blockHeight = rowHeight - 2;

			if(blockStart < periodStart){
				data.blockLeft = 0;
			}else{
				// Get the leftoffset of the start of the block
				var pixelOffset = this.findBlockPixelOffset(blockStart, true);
				data.blockLeft = pixelOffset;
			}

			if(blockEnd > periodEnd){
				// Run the block until the end of the timeline
				data.blockWidth = this.bodyBackgroundElem.getWidth() - data.blockLeft;
			}else{
				// Get the leftoffset of the end of the block
				var pixelOffset = this.findBlockPixelOffset(blockEnd, false);
				// Offset has to be transformed into a width
				data.blockWidth = pixelOffset - data.blockLeft;
			}
		}
		return data;
	},

	/**
	 * Filters the list of records by checking whether they are going to be visible on the timeline.
	 * @param {Ext.data.Record|Array} records List of records
	 * @return {Ext.data.Record|Array} Filtered list of records
	 * @private
	 */
	filterRecords : function(records){
		var filteredRecords = new Array();
		var periodStart = this.daterange.getStartDate().getTime()/1000;
		var periodEnd = this.daterange.getDueDate().getTime()/1000;

		for(var i=0;i<records.length;i++){
			var blockStart = parseInt( records[i].get("start"), 10);
			var blockEnd = parseInt( records[i].get("end"), 10);

			if(this.filterBlockData(blockStart, blockEnd, periodStart, periodEnd)){
				filteredRecords.push(records[i]);
			}
		}

		return filteredRecords;
	},

	/**
	 * Determines whether the block should be rendered or not. When based on the supplied data it
	 * should not be rendered it will return false. Otherwise it will return true.
	 * @param {Number} blockStart Timestamp of start of block (in seconds).
	 * @param {Number} blockEnd Timestamp of end of block (in seconds).
	 * @param {Number} periodStart Timestamp of start of period (in seconds).
	 * @param {Number} periodEnd Timestamp of end of period (in seconds).
	 * @return {Boolean} Returns false when block should be hidden, otherwise returns true
	 * @private
	 */
	filterBlockData: function(blockStart, blockEnd, periodStart, periodEnd){
		// Filter out any records that for some reason are beyond the period range.
		if(blockStart < periodEnd && blockEnd > periodStart){

			if (this.model.showOnlyWorkingHours()) {
				var blockDuration = blockEnd - blockStart;
				var hiddenHoursDuration = (this.workingHoursStart + ((24 * 60) - this.workingHoursEnd)) * 60;

				// Block duration is less than the hours that are hidden
				if(blockDuration <= hiddenHoursDuration){
					// Convert start and end minutes to seconds since start day
					var workStart= this.workingHoursStart * 60;
					var workEnd = this.workingHoursEnd * 60;

					// Convert the blockStart and blockEnd to seconds since start day
					var blockStartDate = new Date(blockStart*1000);
					var blockEndDate   = new Date(blockEnd*1000);
					var blockStartSecs = (blockStartDate.getHours()*60*60) +
					        (blockStartDate.getMinutes()*60) + (blockStartDate.getSeconds());
					var blockEndSecs = (blockEndDate.getHours()*60*60) +
					        (blockEndDate.getMinutes()*60) + (blockEndDate.getSeconds());

					// Check to see whether the start or end part of the block will show itself
					var startBlockShown = (blockStartSecs >= workStart && blockStartSecs < workEnd);
					var endBlockShown = (blockEndSecs > workStart && blockEndSecs <= workEnd);

					if(startBlockShown || endBlockShown){
						return true;
					}

				// Block duration is higher than the hours that are hidden
				}else{
					return true;
				}
			// No hours are hidden, so all blocks within the period duration are shown
			}else{
				return true;
			}
		}
		return false;
	},

	/**
	 * Get the pixel offset from the start of the day of the timestamp.
	 * The argument inclusive is used to determine when the supplied timestamp matches the start of
	 * the day whether the timestamp belongs to the that day or to the previous day. When dealing
	 * with a start date you have to set inclusive to true and when dealing with an end date you
	 * have to set it to false. By default inclusive is set to true.
	 * @param {Number} timestamp Timestamp
	 * @param {Boolean} inclusive Is used to determine if timestamp which is set to 0:00, is set to
	 * the start of the next day (true) or at the end of the day before (false).
	 * day or the day before.
	 * @return {Number} Pixel Offset since start of the timeline
	 * @private
	 */
	findBlockPixelOffset: function(timestamp, inclusive){
		// Get the index of the day the timestamp falls on
		var dayIndex = this.findDayIndexByTimestamp(timestamp, inclusive);

		/*
		 * If there is a difference of time between timestamp and day pointed by dayIndex then probably
		 * timestamp is lying on next day, this thing occurs for 'showing working hours only' setup.
		 * 
		 * if we are showing working hours only then appointments on saturday will have dayindex of
		 * friday so we need to set it's offset to end of the day so directlly settings it's offset
		 * to most end of the day.
		 */
		if(timestamp - this.daysMap[dayIndex].timestamp >= 86400) {
			return this.daysMap[dayIndex].leftPos + this.dayWidth;
		}

		/* Extract the hour and minutes from the timestamp. We need this to prevent DST issues when
		 * there is a DST change during a day. If that happens the number of seconds since the start
		 * of a day is no longer correct.
		 */
		var timestampDate = new Date(timestamp*1000);
		var DSTSafeHours = timestampDate.getHours();
		var DSTSafeMinutes = timestampDate.getMinutes();

		var startDayPixelOffset;
		/* If we have a timestamp that starts at 0:00 we need to determine whether this is at the start of
		 * a day or at the end of a day. We can determine this based on the inclusive argument. If the callee
		 * wants to have an inclusive pixel offset we should give the pixel offset of the start of the day.
		 * If it is not inclusive we have to give the pixel offset of the end of the day. This is not the
		 * same as there is a space between two days.
		 */
		if(DSTSafeHours == 0 && DSTSafeMinutes == 0 && !inclusive){
			// Set the pixel offset at the end of the current day
			startDayPixelOffset = this.dayWidth;
		}else{
			var secondsSinceStartOfDay = ( ( DSTSafeHours * 60 ) + DSTSafeMinutes ) * 60;
			startDayPixelOffset = this.findPixelOffsetStartOfDayBySecs(secondsSinceStartOfDay);
		}

		// Timestamp ends after last visible hour.
		if(startDayPixelOffset == -1){
			/* When the timestamp is after the last visible hour of that day then we should add the
			 * daySpacing to the pixel offset to make it start on the next day. When this is the
			 * last day on the timeline the daySpacing should not be added. Otherwise the timeline
			 * width would be stretched and the scroll width will get out-of-sync with the header.
			 */
			var lastDayOnTimeline = (dayIndex < this.daysMap.length-1);
			startDayPixelOffset = this.dayWidth + (lastDayOnTimeline ? this.daySpacing : 0);
		}

		// Adding the number of pixels from the start of the timeline till the start of the day to
		// the pixels since the start of the day till the timestamp.
		var pixelOffset = this.daysMap[dayIndex].leftPos + startDayPixelOffset;

		return pixelOffset;
	},

	/**
	 * Returns the number of pixels since the start of the day based on the number of seconds since
	 * the start of the day. It returns -1 if the number of seconds go beyond the last displayed
	 * hour. This can happen when only working hours are shown.
	 * @param {Number} secondsSinceStartDay Seconds since the start of the day
	 * @return {Number} Pixel Offset since start of the day
	 * @private
	 */
	findPixelOffsetStartOfDayBySecs: function(secondsSinceStartDay){
		var firstHourStartDayOffsetSecs = this.hoursEachDayMap[0].startDayOffset;
		var lastHourStartDayOffsetSecs = this.hoursEachDayMap[ this.hoursEachDayMap.length-1 ].startDayOffset;

		// Timestamp takes place after last visible hour
		if(lastHourStartDayOffsetSecs + this.slotDuration < secondsSinceStartDay){
			return -1;
		}

		// Timestamp takes place before or at the start of the first hour
		if(firstHourStartDayOffsetSecs >= secondsSinceStartDay){
			return 0;
		}

		var numVisibleSeconds = this.slotDuration * this.hoursEachDayMap.length;
		var secondsSinceFirstHour = secondsSinceStartDay - firstHourStartDayOffsetSecs;
		/**
		 * Use the secondsSinceFirstHour:numVisibleSeconds ratio to turn the number of seconds since
		 * the start of the first hour into the ammount of pixels since the start of the first hour.
		 */
		var pixelOffset = (secondsSinceFirstHour / numVisibleSeconds) * this.dayWidth;
		pixelOffset = Math.round(pixelOffset);
		return pixelOffset;

	},

	/**
	 * Get the dayIndex of the day in the daysMap table that the timestamp belongs to.
	 * The argument inclusive is used to determine when the supplied timestamp matches the start of
	 * the day whether the timestamp belongs to the that day or to the previous day. When dealing
	 * with a start date you have to set inclusive to true and when dealing with an end date you
	 * have to set it to false. By default inclusive is set to true.
	 * @param {Number} timestamp Timestamp
	 * @param {Boolean} inclusive Is used to determine if timestamp which is set to 0:00, is set to
	 * the start of the next day (true) or at the end of the day before (false).
	 * @return {Number} Index of the day as used in the this.daysMap.
	 * @private
	 */
	findDayIndexByTimestamp: function(timestamp, inclusive){
		if(!Ext.isDefined(inclusive)) inclusive = true;

		/**
		 * If date takes place before the start of te first day in the daysMap the selector will
		 * select from the first day in the daysMap.
		 */
		var dayIndex = 0;
		for(var i=0;i<this.daysMap.length;i++){
			if(inclusive && this.daysMap[i].timestamp <= timestamp){
				dayIndex = i;
			}else if(!inclusive && this.daysMap[i].timestamp < timestamp){
				dayIndex = i;
			}else{
				break;
			}
		}

		return dayIndex;
	},

	/**
	 * Find the timestamp based on the supplied coordinate.
	 * @param {Number} coordX X coordinate
	 * @return {Number} Timestamp
	 * @private
	 */
	findTimestampByTimelineXCoord : function(coordX){
		var dayIndex = coordX / (this.dayWidth + this.daySpacing);
		var pixelsPastDayStart = (dayIndex % 1) * (this.dayWidth + this.daySpacing);

		// Calculate how far along the day the coordinate is (ratioDay)
		var ratioDay = pixelsPastDayStart / this.dayWidth;
		var timeSinceStartOfDay;
		if(ratioDay < 1){
			// Calculate the duration of the visible hours
			var durationVisHours = this.slotDuration * this.hoursEachDayMap.length;
			/**
			 * Calculate how many seconds past the start of the day is using the rationDay and the
			 * startDayOffset of the first hour.
			 */
			timeSinceStartOfDay = (ratioDay * durationVisHours) + this.hoursEachDayMap[0].startDayOffset;
		}else{
			// One slot duration past last shown hour slot
			timeSinceStartOfDay = this.hoursEachDayMap[ this.hoursEachDayMap.length-1 ].startDayOffset + this.slotDuration;
		}
		/* Use timestamp of the start of the day to make an date object that we can add the hour and
		 * minutes to. We devide the number of seconds since the start of the day to get the hour
		 * and minutes. This we will add to the Date object by using the setHours and setMinutes
		 * methods. This will prevent problems after Daylight Saving Time.
		 */
		var date = new Date( this.daysMap[ Math.floor(dayIndex) ].timestamp * 1000 );
		var hoursSinceStartOfDay = Math.floor(timeSinceStartOfDay/(60*60));
		var minutesSinceStartOfDay = Math.floor( (timeSinceStartOfDay%(60*60)) / 60 );
		/* NOTE: Using setHours/setMinutes will cause an issue right on the DST change. If you look
		 * at the Dutch DST change in March, the clock will move forward an hour at 02:00. Using
		 * these function it will switch 02:00 to 01:00 and 02:30 to 01:30. That might not be what
		 * the user expects, but it only happens during the DST change.
		 */
		date.setHours( hoursSinceStartOfDay );
		date.setMinutes( minutesSinceStartOfDay );
		return Math.floor( date.getTime()/1000 );
	},

	/**
	 * Binds the user store to the timeline.
	 * @param {Ext.data.Store} store Store
	 * @param {Boolean} initial Internally used to indicate that this is the first call after render.
	 * @private
	 */
	bindUserStore: function(store, initial){
		if(this.userStore){
			this.mun(this.userStore, {
				'datachanged': this.onUserRefresh,
				'add': this.onUserAdd,
				'remove': this.onUserRemove,
				'clear': this.onUserRefresh,
				scope: this
			});
		}

		this.userStore = Ext.StoreMgr.lookup(store);

		if(this.userStore){
			this.mon(this.userStore, {
				'datachanged': this.onUserRefresh,
				'add': this.onUserAdd,
				'remove': this.onUserRemove,
				'clear': this.onUserRefresh,
				scope: this
			});

			if(!initial){
				this.sizeTimelineBackground();
			}
		}
	},

	/**
	 * Called when the {@link Zarafa.common.freebusy.data.FreebusyModel model} fires the userstorechange event to indicate
	 * that another user store is set.
	 * @param {Ext.data.Store} store The new store
	 * @private
	 */
	onUserStoreChange: function(store)
	{
		this.bindUserStore(store);
		this.refresh();
		this.sizeTimelineBackground();
	},

	/**
	 * Called when the {@link Zarafa.common.freebusy.data.FreebusyModel model} fires the blockstorechange event to indicate
	 * that another block store is set.
	 * @param {Ext.data.Store} store The new store
	 * @private
	 */
	onBlockStoreChange : function(store)
	{
		this.bindBlockStore(store);
	},

	/**
	 * Called when the {@link Zarafa.common.freebusy.data.FreebusyModel model} fires the sumblockstorechange event to indicate
	 * that another sum block store is set.
	 * @private
	 */
	onSumBlockStoreChange : function(store)
	{
		this.bindSumBlockStore(store);
	},

	/**
	 * Fires when the visibility of the non-working hours has been changed
	 * @param {Boolean} hideNonWorkingHours True to hide the non-working hours
	 * @private
	 */
	onShowWorkingHoursChange : function(hideNonWorkingHours)
	{
		this.refreshTimeline();
	},

	/**
	 * Called when the userStore fires the datachanged or clear event. This TimelineView will sync
	 * the timeline background scrolling height with the user list based on the new amount of users
	 * in the userStore.
	 * @private
	 */
	onUserRefresh: function(){
		this.refresh();
		this.sizeTimelineBackground();
	},

	/**
	 * Called when the userStore fires the add event. This TimelineView will sync the timeline
	 * background scrolling height with the user list based on the new amount of users in the
	 * userStore.
	 * @private
	 */
	onUserAdd: function(){
		this.sizeTimelineBackground();
	},

	/**
	 * Called when the userStore fires the remove event. This TimelineView will sync the timeline
	 * background scrolling height with the user list based on the new amount of users in the
	 * userStore.
	 * @param {Ext.data.Store} store Store
	 * @param {Ext.data.Record} userRecord Record of the user that is removed
	 * @param {Number} index Index of the user record in the store
	 * @private
	 */
	onUserRemove: function(store, userRecord, index){
		this.refresh();
		this.sizeTimelineBackground();
	}
});

Ext.reg('zarafa.freebusytimelineview', Zarafa.common.freebusy.ui.TimelineView);
Ext.ns('Zarafa.common.plugins');

/**
 * @class Zarafa.common.plugins.FieldLabeler
 * @extends Ext.util.Observable
 * @ptype zarafa.fieldlabeler
 *
 * <p>A plugin for Field Components which renders standard Ext form wrapping and labels
 * round the Field at render time regardless of the layout of the Container.</p>
 * <p>Usage:</p>
 * <pre><code>
    {
        xtype: 'combo',
        plugins: [{
            ptype : 'zarafa.fieldlabeler'
        }],
        triggerAction: 'all',
        fieldLabel: 'Select type',
        store: typeStore
    }
 * </code></pre>
 */
Zarafa.common.plugins.FieldLabeler = Ext.extend(Ext.util.Observable, {
	/**
	 * Initialize the plugin for the given {@link Ext.form.Field field}
	 * @param {Ext.form.Field} field The field on which the plugin is placed
	 */
	init : function(field)
	{
		field.onRender = field.onRender.createSequence(this.onRender);
		field.onResize = this.onResize;
		field.adjustPosition = this.adjustPosition;
		field.onDestroy = field.onDestroy.createSequence(this.onDestroy);
	},

	/**
	 * Renders the component
	 * @private
	 */
	onRender : function()
	{
		// Do nothing if being rendered by a form layout
		if (this.ownerCt) {
			if (this.ownerCt.layout instanceof Ext.layout.FormLayout) {
				return;
			}
		}

		//  Pulls a named property down from the first ancestor Container it's found in
		function getParentProperty(propName) {
			for (var p = this.ownerCt; p; p = p.ownerCt) {
				if (p[propName]) {
					return p[propName];
				}
			}
		}

		this.resizeEl = (this.wrap || this.el).wrap({
			cls: 'x-form-element',
			style: (Ext.isIE9m && !Ext.isIE9 || Ext.isOpera) ? 'position:absolute;top:0;left:0;overflow:visible' : ''
		});
		this.positionEl = this.itemCt = this.resizeEl.wrap({
			cls: 'x-form-item '
		});
		if (this.nextSibling()) {
			this.margins = {
				top: 0,
				right: 0,
				bottom: this.positionEl.getMargins('b'),
				left: 0
			};
		}
		this.actionMode = 'itemCt';

		// If our Container is hiding labels, then we're done!
		if (!Ext.isDefined(this.hideLabels)) {
			this.hideLabels = getParentProperty.call(this, 'hideLabels');
		}
		if (this.hideLabels) {
			this.resizeEl.setStyle('padding-left', '0px');
			return;
		}

		// Collect the info we need to render the label from our Container.
		if (!Ext.isDefined(this.labelSeparator)) {
			this.labelSeparator = getParentProperty.call(this, 'labelSeparator');
		}
		if (!Ext.isDefined(this.labelPad)) {
			this.labelPad = getParentProperty.call(this, 'labelPad');
		}
		if (!Ext.isDefined(this.labelAlign)) {
			this.labelAlign = getParentProperty.call(this, 'labelAlign') || 'left';
		}
		this.itemCt.addClass('x-form-label-' + this.labelAlign);

		if(this.labelAlign == 'top'){
			if (!this.labelWidth) {
				this.labelWidth = 'auto';
			}
			this.resizeEl.setStyle('padding-left', '0px');
		} else {
			if (!Ext.isDefined(this.labelWidth)) {
				this.labelWidth = getParentProperty.call(this, 'labelWidth') || 100;
			}
			this.resizeEl.setStyle('padding-left', (this.labelWidth + (this.labelPad || 5)) + 'px');
			this.labelWidth += 'px';
		}

		this.label = this.itemCt.insertFirst({
			tag: 'label',
			cls: 'x-form-item-label',
			style: {
				width: this.labelWidth
			},
			html: this.fieldLabel + (this.labelSeparator || ':')
		});
	},

	/**
	 * Ensure the input field is sized to fit in the content area of the resizeEl (to the right of its padding-left)
	 * We perform all necessary sizing here. We do NOT call the current class's onResize because we need this control
	 * we skip that and go up the hierarchy to {@link Ext.form.Field Field}
	 *
	 * @param {Number} adjWidth The box-adjusted width that was set
	 * @param {Number} adjHeight The box-adjusted height that was set
	 * @param {Number} rawWidth The width that was originally specified
	 * @param {Number} rawHeight The height that was originally specified
	 * @private
	 */
	onResize : function(adjWidth, adjHeight, rawWidth, rawHeight)
	{
		var width = adjWidth;
		var height = adjHeight;
		var widthCt = adjWidth;
		var heightCt = adjHeight;

		// If this field is not placed on a FormLayout, and this
		// field has the 'flex' property set, then the padding
		// (better known as the labelWidth + padding) must be
		// removed from the width.
		if (!(this.layout instanceof Ext.layout.FormLayout)) {
			if (!Ext.isEmpty(this.flex) || !Ext.isEmpty(this.columnWidth)) {
				width -= this.resizeEl.getPadding('l');
			} else {
				widthCt += this.resizeEl.getPadding('l');
			}
		}

		// If there is an inner Container (in other words, was this a CompositeField)
		// then we must resize this accordingly. Note that the fieldLabel comes before
		// the innerContainer, so the width excluding the label must be used.
		if (this.innerCt) {
			this.innerCt.setSize(width, heightCt);
		}

		Ext.form.Field.prototype.onResize.apply(this, arguments);

		if (this.resizeEl)
			this.resizeEl.setWidth(heightCt);
		if (this.itemCt)
			this.itemCt.setWidth(widthCt);

		if (this.getTriggerWidth) {
			this.wrap.setWidth(width);
			this.el.setWidth(width - this.getTriggerWidth());
		} else {
			this.el.setWidth(width);
		}

		if (this.el.dom.tagName.toLowerCase() == 'textarea') {
			height = this.resizeEl.getHeight(true);
			if (!this.hideLabels && (this.labelAlign == 'top')) {
				height -= this.label.getHeight();
			}
			this.el.setHeight(height);
		}

		if (this instanceof Ext.form.ComboBox) {
			if (!isNaN(width) && this.isVisible() && this.list) {
				this.doResize(width);
			} else {
				this.bufferSize = width;
			}
		}
	},

	/**
	 * If the owner is a HBox, we must compensate for the
	 * resizing of the container element of the _previous_
	 * item. This if the current item is not the first item,
	 * we must change the offset to the left-offset + size of the
	 * previous item.
	 * @param {Number} x The suggested X offset for the component
	 * @param {Number} y The suggested Y offset for the component
	 * @private
	 */
	adjustPosition : function(x, y)
	{
		if (this.ownerCt.layout instanceof Ext.layout.HBoxLayout) {
			x = 0;
			this.ownerCt.items.each(function(item) {
				if (item === this)
					return false;

				x += item.getActionEl().getWidth();
			}, this);
		}

		return { x: x, y: y};
	},

	/**
	 * Remove elements created by this plugin
	 * @private
	 */
	onDestroy: function()
	{
		if(this.ownerCt) {
			Ext.destroy(this.ownerCt);
			this.ownerCt.remove();
		}
	}
});

Ext.preg('zarafa.fieldlabeler', Zarafa.common.plugins.FieldLabeler);
Ext.ns('Zarafa.common.plugins');

/**
 * @class Zarafa.common.plugins.SpinnerPlugin
 * @extends Ext.ux.Spinner
 *
 * Extension to the {@link Ext.ux.Spinner Spinner} object,
 * to fix any bugs which are inside there. This plugin
 * should not be used directly, instead it should be used
 * as baseclass for any Spinner plugins.
 */
Zarafa.common.plugins.SpinnerPlugin = Ext.extend(Ext.ux.Spinner, {
	/**
	 * Automatically called when the parent {@link Zarafa.common.ui.SpinnerField SpinnerField}
	 * is being rendered. This will apply the default value to the field.
	 * @param {Container} ct The parent container into which the SpinnerField is being rendered.
	 * @param {Number} position The position where the SpinnerField is being rendered
	 * @private
	 */
	doRender: function(ct, position)
	{
		Zarafa.common.plugins.SpinnerPlugin.superclass.doRender.call(this, ct, position);
		this.field.trigger = this.trigger;

		if (!Ext.isDefined(this.field.getTriggerWidth))
			this.field.getTriggerWidth = Ext.form.TriggerField.prototype.getTriggerWidth;
	},

	/**
	 * Handle the click event on {@link Ext.form.TriggerField TriggerField}.
	 * This function is overriden to fix problem when trigger is clicked directly without gaining
	 * focus on the {@link Zarafa.common.ui.SpinnerField SpinnerField} then we should shift the
	 * focus to {@link Zarafa.common.ui.SpinnerField SpinnerField} and then execute this event.
	 * @param {Event} event The event object for this event
	 */
	onTriggerClick: function(event)
	{	   
		if (this.disabled || this.el.dom.readOnly) {
			return;
		}

		var middle = this.getMiddle();
		var ud = (event.getPageY() < middle) ? 'Up' : 'Down';

		// Grab focus on the element, this will ensure that
		// the field which we are editing has the browser focus.
		this.el.focus();

		this['onSpin' + ud]();
	},

	/**
	 * checks if control is allowed to spin
	 * @return {Boolean} True if the spin action is allowed
	 * @private
	 */
	isSpinnable: function()
	{
		if (this.disabled || (this.rendered && this.el.dom.readOnly) || !Ext.isDefined(this.field.getValue())) {
			Ext.EventObject.preventDefault(); //prevent scrolling when disabled/readonly
			return false;
		}
		return true;
	}
});
Ext.ns('Zarafa.common.plugins');

/**
 * @class Zarafa.common.plugins.splitFieldLabeler
 * @extends Ext.util.Observable
 * @ptype zarafa.splitfieldlabeler
 *
 * This plugin functions similarly as the {@link Zarafa.common.plugins.FieldLabeler FieldLabeler}.
 * This plugin should be attached to an {@link Ext.Container container} or anything which
 * has an 'items' field containing components. And is therefor primarily intended for
 * splitting a label for more then one component.
 * On the {@link Ext.Container container} the fieldLabel must be set consisting
 * of a string which can be split into multiple substrings.
 * Each component should have the {@link Zarafa.common.plugins.FieldLabeler.labelSplitter labelSplitter}
 * field set to indicate which variable it represents in the fieldLabel string of the container.
 *
 * Before rendering this plugin will breakup the fieldLabel on the {@link Ext.Container container}
 * and adds {@link Ext.form.Label Label} components and the original {@link Ext.form.Field Fields}
 * into the items array of the container.
 */
Zarafa.common.plugins.SplitFieldLabeler = Ext.extend(Ext.util.Observable, {
	/**
	 * @cfg {Object} labelCfg The labelConfiguration which must be
	 * applied to all generated labels.
	 */
	labelCfg : {
		style: 'text-align: left; padding: 3px 3px 3px 3px'
	},
	/**
	 * @cfg {Object} firstLabelCfg The labelConfiguration which must be
	 * applied to the first generated label (if this is also the first
	 * displayed item inside the field!). This configuration applied after
	 * the {@link #labelCfg}.
	 */
	firstLabelCfg : {
		style: 'text-align: left; padding: 3px 3px 3px 0px'
	},
	/**
	 * @cfg {Object} lastLabelCfg The labelConfiguration which must be
	 * applied to the last generated label (if this is also the last
	 * displayed item inside the field!). This configuration applied after
	 * the {@link #labelCfg}.
	 */
	lastLabelCfg : {
		style: 'text-align: left; padding: 3px 3px 3px 3px'
	},
	/**
	 * Plugin initializer for the SplitFieldLabeler.
	 * @param {Ext.Component} field The field for which the fieldlabeler is initialized
	 */
	init : function(field)
	{
		// This is not a container, or no label has been assigned.
		if (!Ext.isDefined(field.items) || !Ext.isDefined(field.fieldLabel) || field.hideLabel === true)
			return;

		var labelPieces = this.createLabelDistribution(field.items, field.fieldLabel);
		var labeledItems = this.applyLabelDistribution(field.items, labelPieces);

		this.applyLabelWidths(field.labelWidth, labeledItems);

		// Overwrite the items, be careful not to change the type
		// of the items object (it could be an array or MixedCollection).
		if (Ext.isArray(field.items)) {
			field.items = labeledItems;
		} else {
			field.items.clear();
			field.items.addAll(labeledItems);
			// The innerCt has already been created at this point so the items 
			// need to be added to that Component as well
			field.innerCt.items.clear();
			field.innerCt.items.addAll(labeledItems);
		}

		// The main field must not display any labels
		field.hideLabel = true,
		delete field.fieldLabel;
	},

	/**
	 * Create a list of all sub-labels which can be constructed from the given label.
	 * This will return a list of all sub-labels, including the seperator string which
	 * indicates the correct position of an item.
	 *
	 * @param {Array/MixedCollection} items The items for which the label distribution
	 * is required.
	 * @param {String} label The label which must be distributed over the items.
	 * @return {Array} The array of labelstrings which must be applied to the components.
	 */
	createLabelDistribution : function(items, label)
	{
		var splitters = [];
		var generateSplitters = function(item) {   
			if (Ext.isDefined(item.labelSplitter)) {
				splitters.push(item.labelSplitter);
			}
		};

		// Items can be an Array or MixedCollection, depending
		// on the current state of the field for which we are generating
		// the labels.
		if (Ext.isArray(items)) {
			Ext.each(items, generateSplitters, this);
		} else {
			items.each(generateSplitters, this);
		}

		return Zarafa.util.Translations.MultiSplitTranslation(label, splitters);
	},

	/**
	 * Apply the label distribution which was generated with the
	 * {@link #createLabelDistribution createLabelDistribution}
	 * function. This will construct an array of {@link Ext.Component Components} with the
	 * labels and components in the order of which they should be displayed.
	 *
	 * @param {Array/MixedCollection} items The items for which the labels must be applied.
	 * @param {Array} labels The array of labels which must be applied to the items.
	 * @return {Array} items The array of items with the labels applied, and in the
	 * order in which they must be shown.
	 */
	applyLabelDistribution : function(items, labels)
	{
		var labeledItems = [];

		for (var i = 0; i < labels.length; i++) {
			var foundItem = undefined;
			var label = labels[i];

			var findItem = function(item) {
				if (item.labelSplitter === labels[i]) {
					foundItem = item;
					return false;
				}
			}

			// Items can be an Array or MixedCollection, depending
			// on the current state of the field for which we are generating
			// the labels.
			if (Ext.isArray(items)) {
				Ext.each(items, findItem, this);
			} else {
				items.each(findItem, this);
			}

			// If the label was recognized as labelSplitter,
			// we must insert the component itself, otherwise
			// generate the label.
			if (Ext.isDefined(foundItem)) {
				labeledItems.push(foundItem);
			} else {
				labeledItems.push(Ext.apply({}, this.labelCfg, {
					xtype: 'displayfield',
					value: label
				}));
			}
		}

		// Apply special configurations to the first and last items.
		if (labeledItems[0].xtype == 'displayfield') {
			Ext.apply(labeledItems[0], this.firstLabelCfg);
		}
		if (labeledItems[labeledItems.length - 1].xtype == 'displayfield') {
			Ext.apply(labeledItems[labeledItems.length - 1], this.lastLabelCfg);
		}

		for (var i = 0; i < labeledItems.length; i++) {
			labeledItems[i] = Ext.create(labeledItems[i]);
		}

		return labeledItems;
	},

	/**
	 * Calculate the {@link Ext.form.Label.width labelWidths} which
	 * must be applied to each individual component which doesn't have the
	 * {@link Ext.form.Label.width labelWidth} set explicitely.
	 *
	 * @param {Number} totalWidth The total width which can be used to
	 * for all the labels combined.
	 * @param {Ext.Component} labeledItems The components which contain the
	 * fieldLabels for which the lengths must be calculated.
	 */
	applyLabelWidths : function(totalWidth, labeledItems)
	{
		// No width configured, all labels have autoWidth
		if (!Ext.isNumber(totalWidth)) {
			return;
		}

		// The combined length of all labels
		var labelLength = 0;

		// Loop over all items to see what labels have been
		// applied, and if a width has already been specified. 
		for (var i = 0; i < labeledItems.length; i++) {
			var item = labeledItems[i];

			if (item.xtype == 'displayfield') {
				// If no width is configured, but a label is provided,
				// increase the labelLength to have this label be part
				// of the labelWidth calculations.
				// If a width was provided, we reduce the totalWidth
				// because we must only calculate the labelWidth for
				// the non-fixed labels.
				if ((!Ext.isDefined(item.width) || item.width == 0) && !Ext.isEmpty(item.text)) {
					labelLength += item.text.length;
				} else {
					totalWidth -= item.width;
				}
			}
		}

		// Calculate the labelWidth 
		for (var i = 0; i < labeledItems.length; i++) {
			var item = labeledItems[i];

			if (item.xtype == 'displayfield') {
				if ((!Ext.isDefined(item.width) || item.width == 0) && !Ext.isEmpty(item.text)) {
					item.width = Math.round(totalWidth * (item.text.length / labelLength));
				}
			}
		}
	}
});

Ext.preg('zarafa.splitfieldlabeler', Zarafa.common.plugins.SplitFieldLabeler);
Ext.namespace('Zarafa.common.recipientfield.data');

/**
 * @class Zarafa.common.recipientfield.data.SuggestionListJsonWriter
 * @extends Ext.data.JsonWriter
 */
Zarafa.common.recipientfield.data.SuggestionListJsonWriter = Ext.extend(Ext.data.JsonWriter, {
	/**
	 * @constructor
	 * @param {Object} config Configuration object
	 */
	constructor : function(config)
	{
		config = config || {};

		Ext.applyIf(config, {
			writeAllFields : true,
			encode : false
		});

		Zarafa.core.data.JsonWriter.superclass.constructor.call(this, config);
	},

	/**
	 * Use the {@link Ext.data.JsonWriter#toHash toHash} function for creating the hash.
	 *
	 * @param {Ext.data.Record} record
	 * @return {Object}
	 * @override
	 * @private
	 */
	destroyRecord : function(record){
		return this.toHash(record);
	}
});
Ext.namespace('Zarafa.common.recipientfield.data');

/**
 * @class Zarafa.common.recipientfield.data.SuggestionListStore
 * @class Ext.data.Store
 * @xtype zarafa.suggestionliststore
 *
 * The main store which holds the suggestions as shown inside the
 * {@link Zarafa.common.recipientfield.ui.RecipientField RecipientField}.
 */
Zarafa.common.recipientfield.data.SuggestionListStore = Ext.extend(Ext.data.Store, {
	/**
	 * @cfg {String} actionType type of action that should be used to send request to server,
	 * valid action types are defined in {@link Zarafa.core.Actions Actions}, default value is 'list'.
	 */
	actionType : undefined,

	/**
	 * @constructor
	 * @param {Object} config Configuration object
	 */
	constructor : function(config)
	{
		config = config || {};

		Ext.applyIf(config, {
			batch: true,
			autoSave: true,
			remoteSort: false,
			actionType : Zarafa.core.Actions['list'],
			proxy: new Zarafa.common.recipientfield.data.SuggestionListProxy(),
			writer: new Zarafa.common.recipientfield.data.SuggestionListJsonWriter(),
			reader: new Ext.data.JsonReader({
				root: 'result',
				id : 'id'
			}, Zarafa.common.recipientfield.data.SuggestionListRecord)
		});

		Zarafa.common.recipientfield.data.SuggestionListStore.superclass.constructor.call(this, config);

		// Use multi-sorting on the suggestions,
		// we can't apply this in the configuration object
		// so we have to do it here.
		this.sort([{
			field: 'display_name',
			direction : 'ASC'
		},{
			field : 'smtp_address',
			direction : 'ASC'
		},{
			field : 'email_address',
			direction : 'ASC'
		}]);
	},

	/**
	 * Load all data from the store
	 * @param {Object} options Additional options
	 */
	load : function(options)
	{
		if (!Ext.isObject(options)) {
			options = {};
		}

		if (!Ext.isObject(options.params)) {
			options.params = {};
		}

		// By default 'load' must cancel the previous request.
		if (!Ext.isDefined(options.cancelPreviousRequest)) {
			options.cancelPreviousRequest = true;
		}

		if (!Ext.isDefined(options.actionType)) {
			options.actionType = this.actionType;
		}

		Zarafa.common.recipientfield.data.SuggestionListStore.superclass.load.call(this, options);
	}
});

Ext.reg('zarafa.suggestionliststore', Zarafa.common.recipientfield.data.SuggestionListStore);
Ext.namespace('Zarafa.common.recurrence.data');

/**
 * @class Zarafa.common.recurrence.data.DayData
 * @singleton
 */
Zarafa.common.recurrence.data.DayData = [
	{ name: _('Day'),			value: 127 },
	{ name: _('Weekday'),		value: 62 },
	{ name: _('Weekend Day'),	value: 65 },
	// The following are initialized empty,
	// because the order of the days is
	// depending on the 'zarafa/v1/main/week_start'
	// configuration option, which we cannot use
	// until the Document has been loaded.
	{ name: null,				value: 0 },
	{ name: null,				value: 0 },
	{ name: null,				value: 0 },
	{ name: null,				value: 0 },
	{ name: null,				value: 0 },
	{ name: null,				value: 0 },
	{ name: null,				value: 0 }
];

// With the document loaded, we can now access the 'zarafa/v1/main/week_start'
// configuration option, which we need to build the last 7 items from the
// Zarafa.common.recurrence.data.DayData structure.
Zarafa.onReady(function() {
	var weekStart = container.getSettingsModel().get('zarafa/v1/main/week_start');      

	for (var i = 3; i < Zarafa.common.recurrence.data.DayData.length; i++) {
		var index = (weekStart + (i - 3)) % 7;
		Zarafa.common.recurrence.data.DayData[i].name = Date.dayNames[index];
		Zarafa.common.recurrence.data.DayData[i].value = Math.pow(2, index);
	}
}, undefined, { single: true });
Ext.namespace('Zarafa.common.recurrence.data');

/**
 * @class Zarafa.common.recurrence.data.DayRankData
 * @singleton
 */
Zarafa.common.recurrence.data.DayRankData = [
	{ name: _('1st'),	value: 1 },
	{ name: _('2nd'),	value: 2 },
	{ name: _('3rd'),	value: 3 },
	{ name: _('4th'),	value: 4 },
	{ name: _('last'),	value: 5 }
];
Ext.namespace('Zarafa.common.recurrence.data');

/**
 * @class Zarafa.common.recurrence.data.MonthData
 * Month value is calculated on the bases of total minutes before that month.
 * As for exaple, February starts after 44640 minutes obtained by 31(days of January month) * 1440(minutes of a day).
 * @singleton
 */
Zarafa.common.recurrence.data.MonthData = [
	{ name: Date.monthNames[0],		value: 0 },
	{ name: Date.monthNames[1],		value: 44640 },
	{ name: Date.monthNames[2],		value: 84960 },
	{ name: Date.monthNames[3],		value: 129600 },
	{ name: Date.monthNames[4],		value: 172800 },
	{ name: Date.monthNames[5],		value: 217440 },
	{ name: Date.monthNames[6],		value: 260640 },
	{ name: Date.monthNames[7],		value: 305280 },
	{ name: Date.monthNames[8],		value: 349920 },
	{ name: Date.monthNames[9],		value: 393120 },
	{ name: Date.monthNames[10],	value: 437760 },
	{ name: Date.monthNames[11],	value: 480960 }
];
Ext.namespace('Zarafa.common.recurrence.dialogs');

/**
 * @class Zarafa.common.recurrence.dialogs.RecurrencePanel
 * @extends Ext.Panel
 * @xtype zarafa.recurrencepanel
 * 
 * Panel that is used to create Recurrences
 */
Zarafa.common.recurrence.dialogs.RecurrencePanel = Ext.extend(Ext.Panel, {
	/**
	 * @constructor
	 * @param {Object} config configuration object.
	 */
	constructor : function(config)
	{
		config = config || {};

		config.plugins = Ext.value(config.plugins, []);
		config.plugins.push('zarafa.recordcomponentupdaterplugin');

		config = Ext.applyIf(config, {
			xtype: 'zarafa.recurrencepanel',
			layout: 'form',
			border: false,
			defaults: {
				border: false,
				bodyStyle: 'padding: 10px;'
			},
			items : [
				this.createTimePanel(),
				this.createRecurrencePanel(),
				this.createRangePanel()
			]
		});

		Zarafa.common.recurrence.dialogs.RecurrencePanel.superclass.constructor.call(this, config);
	},

	/**
	 * Creates the Time panel in which the user can select the message time.
	 * @return {Object} The configuration object for the Time
	 * @private
	 */
	createTimePanel : function()
	{
		return {
			xtype: 'panel',
			title: _('Time'),
			layout: {
				type: 'vbox',
				align: 'stretch'
			},
			anchor: '100%',
			height: 90,
			items: [{
				xtype: 'displayfield',
				ref: '../timeperiodLabel',
				htmlEncode : true,
				hideLabel : true,
				height: 20
			},{
				xtype: 'panel',	
				layout: 'hbox',
				anchor: '100% 100%',
				border: false,
				bodyStyle: 'background-color: inherit;',
				items: [{
					xtype: 'zarafa.timeperiodfield',
					ref: '../../timeperiodField',
					layout: 'hbox',
					defaultPeriod : 30,
					defaultPeriodType : Date.MINUTE,
					flex: 0.6,
					spacerConfig: {
						width: 5
					},
					listeners: {
						change : this.onDurationChange,
						scope: this
					}
				},{
					xtype: 'spacer',
					width: 5
				},{
					xtype: 'container',
					flex: 0.4,
					border: false,
					style: 'background-color: inherit;',
					items: [{
						xtype: 'checkbox',
						ref: '../../../alldayCheckbox',
						name: 'alldayevent',
						boxLabel: _('All Day Event'),
						handler: this.onToggleAllDay,
						scope: this
					}]
				}]
			}]
		};
	},

	/**
	 * Creates the recurrence recurrence pattern panel, in which the user can configure
	 * the recurrence pattern (daily/weekly/monhtly/yearly) for this message.
	 * @return {Object} The configuration object for the Recurrence Pattern
	 * @private
	 */
	createRecurrencePanel : function()
	{
		return {
			xtype: 'panel',
			layout: {
				type: 'hbox',
				align: 'stretch'
			},
			title: _('Recurrence pattern'),
			anchor: '100%',
			height: 125,
			items: [{
				xtype: 'radiogroup',
				ref: '../recurrencePatternSelect',
				width: 100,
				border: false,
				columns: 1,
				items: [{
					boxLabel: _('Daily'),
					name: 'pattern',
					targetId: 'card-daily',
					patternValue : Zarafa.common.recurrence.data.RecurrenceType.DAILY,
					handler: this.onSwitchRecurrenceView,
					scope: this
				},{
					boxLabel: _('Weekly'),
					name: 'pattern',
					targetId: 'card-weekly',
					patternValue : Zarafa.common.recurrence.data.RecurrenceType.WEEKLY,
					handler: this.onSwitchRecurrenceView,
					scope: this
				},{
					boxLabel: _('Monthly'),
					name: 'pattern',
					targetId: 'card-monthly',
					patternValue : Zarafa.common.recurrence.data.RecurrenceType.MONTHLY,
					handler: this.onSwitchRecurrenceView,
					scope: this
				},{
					boxLabel: _('Yearly'),
					name: 'pattern',
					targetId: 'card-yearly',
					patternValue : Zarafa.common.recurrence.data.RecurrenceType.YEARLY,
					handler: this.onSwitchRecurrenceView,
					scope: this
				}]
			},{
				xtype: 'panel',
				ref: '../recurrencePattern',
				layout: 'card',
				flex: 1,
				border: true,
				bodyStyle: 'background-color: inherit; border-style: none none none solid; padding: 0px 10px 0px 10px',
				defaults: {
					border: false,
					bodyStyle: 'background-color: inherit;',
					autoHeight: true
				},
				items: [{
					xtype: 'zarafa.recurrencedailypanel',
					id: 'card-daily'
				},{
					xtype: 'zarafa.recurrenceweeklypanel',
					id: 'card-weekly'
				},{
					xtype: 'zarafa.recurrencemonthlypanel',
					id: 'card-monthly'
				},{
					xtype: 'zarafa.recurrenceyearlypanel',
					id: 'card-yearly'
				}]
			}]
		};
	},

	/**
	 * Creates the recurrence recurrence pattern panel, in which the user can configure
	 * the recurrence range for this message.
	 * @return {Object} The configuration object for the Recurrence Range
	 * @private
	 */
	createRangePanel : function()
	{
		return {
			xtype: 'panel',
			layout: 'column',
			title: _('Range of recurrence'),
			anchor: '100%',
			items: [{
				xtype: 'panel',
				layout: 'form',
				columnWidth: 0.4,
				border: false,
				items: [{
					xtype: 'datefield',
					ref: '../../startDateField',
					name: 'recurrence_start',
					fieldLabel: _('Start'),
					// # TRANSLATORS: See http://docs.sencha.com/ext-js/3-4/#!/api/Date for the meaning of these formatting instructions
					format : _('d/m/Y'),
					listeners: {
						select: this.onUTCFieldChange,
						change: this.onUTCFieldChange,
						scope: this
					}
				}]
			},{
				xtype: 'panel',
				layout: 'form',
				ref: '../endPatternPanel',
				columnWidth: 0.6,
				border: false,
				bodyStyle: 'background-color: inherit;',
				defaults: {
					anchor :'100%',
					border: false,
					bodyStyle: 'background-color: inherit;',
					defaults: {
						height: 25,
						labelWidth: 75
					}
				},
				items: [{
					xtype: 'panel',
					layout: 'hbox',
					items: [{
						xtype: 'radio',
						name: 'recurrence_term',
						endTerm: Zarafa.common.recurrence.data.RecurrenceEnd.NEVER,
						hideLabel: true,
						width: 25,
						listeners : {
							check : this.onSwitchRecurrenceTerm,
							scope : this
						}
					},{
						xtype: 'displayfield',
						value: _('No end date'),
						hideLabel : true
					}]
				},{
					xtype: 'panel',
					layout: 'column',
					items: [{
						xtype: 'radio',
						name: 'recurrence_term',
						endTerm: Zarafa.common.recurrence.data.RecurrenceEnd.N_OCCURENCES,
						hideLabel: true,
						width: 25,
						listeners : {
							check : this.onSwitchRecurrenceTerm,
							scope : this
						}
					},{
						xtype: 'zarafa.compositefield',
						plugins: [ 'zarafa.splitfieldlabeler' ],
						fieldLabel: _('End after {A} occurrences'),
						labelWidth: 160,
						items: [{
							xtype: 'zarafa.spinnerfield',
							plugins: [ 'zarafa.numberspinner' ],
							ref: '../../../../endOccurencesSpinner',
							name : 'recurrence_numoccur',
							labelSplitter: '{A}',
							allowNegative: false,
							minValue: 1,
							width: 50,
							listeners: {
								change: this.onFieldChange,
								scope: this
							}
						}]
					}]
				},{
					xtype: 'panel',
					layout: 'column',
					items: [{
						xtype: 'radio',
						name: 'recurrence_term',
						endTerm: Zarafa.common.recurrence.data.RecurrenceEnd.ON_DATE,
						hideLabel: true,
						width: 25,
						listeners : {
							check : this.onSwitchRecurrenceTerm,
							scope : this
						}
					},{
						xtype: 'zarafa.compositefield',
						plugins: [ 'zarafa.splitfieldlabeler' ],
						fieldLabel: _('End by {A}'),
						labelWidth: 80,
						combineErrors: false,
						items: [{
							xtype: 'datefield',
							ref: '../../../../endOnDateField',
							name: 'recurrence_end',
							width: 120,
							labelSplitter: '{A}',
							// # TRANSLATORS: See http://docs.sencha.com/ext-js/3-4/#!/api/Date for the meaning of these formatting instructions
							format : _('d/m/Y'),
							listeners: {
								select: this.onUTCFieldChange,
								change: this.onUTCFieldChange,
								scope: this
							}
						}]
					}]
				}]
			}]
		};
	},

	/**
	 * Event handler which is fired when either the All Day checkbox has been
	 * checked, or when the timeDuration field has been updated. In noth situations
	 * the label belonging to the field will be updated to reflect the new duration.
	 * @private
	 */
	updateDurationLabel : function()
	{
		if (!this.record) {
			return;
		}

		var duration = Math.floor(this.timeperiodField.getValue().getDuration(Date.MINUTE));
		var days = Math.floor(duration / (24 * 60));
		duration %= (24 * 60);
		var hours = Math.floor(duration / 60);
		duration %= 60;
		var minutes = Math.floor(duration);	

		// # TRANSLATORS: This informs the user what the exact duration of the appointment is. Where {D} represents the days, {H} the hours and {M} the minutes.
		// # For example: 'Occurence duration: 1 day 2 hours 45 minutes', or when the appointment is shorter then 1 day: 'Occurence duration: 1 hour'
		var label = _('Occurrence duration: {D} {H} {M}');
		if (days > 0) {
			label = label.replace('{D}', String.format(ngettext('{0} day', '{0} days', days), days));
		} else {
			label = label.replace('{D} ', '');
		}
		if (hours > 0) {
			label = label.replace('{H}', String.format(ngettext('{0} hour', '{0} hours', hours), hours));
		} else {
			label = label.replace('{H} ', '');
		}
		if (minutes > 0) {
			label = label.replace('{M}', String.format(ngettext('{0} minute', '{0} minutes', minutes), minutes));
		} else {
			label = label.replace('{M}', '');
		}

		this.timeperiodLabel.setValue(label);
	},

	/**
	 * Event handler which is fired when a field has been changed.
	 * This will update the corresponding field inside the {@link Zarafa.core.data.IPMRecord record}.
	 * @param {Ext.form.Field} field The field which has changed
	 * @param {Mixed} newValue The new value for the field
	 * @param {Mixed} oldValue The original value for the field
	 * @private
	 */
	onFieldChange : function(field, newValue, oldValue)
	{
		this.record.set(field.getName(), newValue);
	},

	/**
	 * Event handler which is fired when the recurrence start or end date has
	 * been changed. This will convert the value to the UTC start of the day,
	 * and update the record.
	 * @param {Ext.form.Field} field The field which has changed
	 * @param {Mixed} newValue The new value for the field
	 * @param {Mixed} oldValue The original value for the field
	 * @private
	 */
	onUTCFieldChange : function(field, newValue, oldValue)
	{
		// The field is represented in UTC time,
		// so convert it to local to get the time for the property
		this.record.set(field.getName(), newValue.fromUTC());
	},

	/**
	 * Event handler which is fired when the duration field has been updated.
	 * This will calculate the correct startocc and endocc values, and updates the record.
	 * @param {Ext.form.Field} field The field which has changed
	 * @param {Mixed} newValue The new value for the field
	 * @param {Mixed} oldValue The original value for the field
	 * @private
	 */
	onDurationChange : function(field, newValue, oldValue)
	{
		var startOcc = (newValue.getStartDate().getHours() * 60) + newValue.getStartDate().getMinutes();
		var endOcc = startOcc + newValue.getDuration(Date.MINUTE);

		this.record.beginEdit();
		this.record.set('recurrence_startocc', startOcc);
		this.record.set('recurrence_endocc', endOcc);
		this.record.endEdit();
	},

	/**
	 * Event handler which is fired when the Recurrence End radio button
	 * has been toggled. This will change the recurrence term status.
	 *
	 * @param {Ext.form.Radio} radio The radio which has changed
	 * @param {Boolean} checked True if the radio was checked
	 * @private
	 */
	onSwitchRecurrenceTerm : function(radio, checked)
	{
		if (checked) {
			this.record.set('recurrence_term', radio.endTerm);
		}
	},

	/**
	 * Event handler which is fired when a checkbox has been toggled
	 * to switch the recurrence pattern. This will change the recurrence
	 * pattern panel to the selected type.
	 *
	 * @param {Ext.form.CheckBox} checkbox The checkbox which was changed
	 * @param {Boolean} Checked True if the checkbox was checked
	 * @private
	 */
	onSwitchRecurrenceView : function(checkbox, checked)
	{
		if (checked) {
			this.record.set('recurrence_type', checkbox.patternValue);
			this.recurrencePattern.getLayout().setActiveItem(checkbox.targetId);
			this.recurrencePattern.doLayout();
		}
	},

	/**
	 * A function called when the checked value changes for the
	 * all day event checkbox.
	 * @param {Ext.form.Checkbox} checkbox The Checkbox being toggled.
	 * @param {Boolean} checked The new checked state of the checkbox.
	 * @private
	 */
	onToggleAllDay : function(checkbox, checked)
	{
		// When the user already has an appointment that last a more then a day, we should
		// round the end of the occurences up to a whole number of days.
		var days = Math.ceil(this.timeperiodField.getValue().getDuration(Date.MINUTE) / 1440);

		this.record.beginEdit();
		this.record.set('alldayevent', checked);
		this.record.set('recurrence_startocc', 0);
		this.record.set('recurrence_endocc', days * 1440);
		this.record.endEdit();
	},

	/**
	 * Enable/disable/hide/unhide all {@link Ext.Component Components} within the {@link Ext.Panel Panel}
	 * using the given {@link Zarafa.core.data.IPMRecord IPMRecord}.
	 * @param {Zarafa.core.data.IPMRecord} record The record to update the panel with
	 * @param {Boolean} contentReset force the component to perform a full update of the data.
	 * @private
	 */
	updateUI : function(record, contentReset)
	{
		var layout = false;

		if (contentReset === true || record.isModifiedSinceLastUpdate('alldayevent')) {
			if (record.get('alldayevent')) {
				this.timeperiodField.disable();
			} else {
				this.timeperiodField.enable();
			}
		}

		if (contentReset === true || record.isModifiedSinceLastUpdate('message_class')) {
			if (record.isMessageClass('IPM.TaskRequest', true)) {
				this.alldayCheckbox.hide();
			} else {
				this.alldayCheckbox.show();
			}

			layout = true;
		}

		if (layout)
			this.doLayout();
	},

	/**
	 * Panel updater. This will initialize all UI components inside the panel with
	 * the data from the {@link Zarafa.core.data.IPMRecord record}.
	 *
	 * @param {Zarafa.core.data.IPMRecord} record The record used to update the panel
	 * @param {Boolean} contentReset force the component to perform a full update of the data.
	 */
	update : function(record, contentReset)
	{
		this.record = record;
		this.updateUI(record, contentReset);

		var type = record.get('recurrence_type');
		this.recurrencePatternSelect.items.each(function(radio) {
			radio.setValue(radio.patternValue == type);
		});

		var startDate = record.get('recurrence_start');
		if (startDate) {
			// The start date is an UTC representation
			startDate = startDate.toUTC();
		} else {
			startDate = new Date().clearTime();
		}
		var endDate = record.get('recurrence_end');
		if (endDate) {
			// The end date is an UTC representation
			endDate = endDate.toUTC();
		} else {
			endDate = startDate.clone();
		}
		var startOcc = record.get('recurrence_startocc');
		var endOcc = record.get('recurrence_endocc');

		// We construct a startTime based on the first day of the year,
		// this guarentees that we are absolutely DST safe, and any time
		// can be selected.
		var startTime = new Date().clearTime();
		startTime.setDate(1);
		startTime.setMonth(0);
		startTime.setHours(startOcc / 60);
		startTime.setMinutes(startOcc % 60);

		// We construct a endTime based on the first day of the year,
		// this guarentees that we are absolutely DST safe, and any time
		// can be selected.
		var endTime = new Date().clearTime();
		endTime.setDate(1);
		endTime.setMonth(0);
		endTime.setHours(endOcc / 60);
		endTime.setMinutes(endOcc % 60);

		this.startDateField.setValue(startDate);
		this.timeperiodField.getValue().set(startTime, endTime);

		if(contentReset === true || record.isModifiedSinceLastUpdate('recurrence_startocc') || record.isModifiedSinceLastUpdate('recurrence_endocc')) {
			this.updateDurationLabel();
		}

		var endTerm = record.get('recurrence_term');
		Ext.each(this.endPatternPanel.findByType('radio'), function(radio) {
			radio.setValue(radio.endTerm == endTerm);
		});

		switch (endTerm) {
			case Zarafa.common.recurrence.data.RecurrenceEnd.NEVER:
				// Only apply default values at the first update
				if (contentReset === true) {
					this.endOccurencesSpinner.setValue(10);
				}
				if (contentReset === true || (record.isModifiedSinceLastUpdate('recurrence_start') && startDate > this.endOnDateField.getValue())) {
					this.endOnDateField.setValue(startDate.clearTime(true));
				}
				break;
			case Zarafa.common.recurrence.data.RecurrenceEnd.N_OCCURENCES:
				// Only apply default values at the first update
				if (contentReset === true || record.isModifiedSinceLastUpdate('recurrence_numoccur')) {
					this.endOccurencesSpinner.setValue(record.get('recurrence_numoccur'));
				}
				if (contentReset === true) {
					this.endOnDateField.setValue(startDate.clearTime(true));
				}
				break;
			case Zarafa.common.recurrence.data.RecurrenceEnd.ON_DATE:
				// Only apply default values at the first update
				if (contentReset === true) {
					this.endOccurencesSpinner.setValue(10);
				}
				if (contentReset === true || record.isModifiedSinceLastUpdate('recurrence_end')) {
					this.endOnDateField.setValue(endDate.clearTime(true));
				}
				break;
		}

		this.alldayCheckbox.setValue(record.get('alldayevent'));
	},

	/**
	 * Record updater. This will update the record with all data from the UI components
	 * inside the Panel.
	 *
	 * @param {Zarafa.core.data.IPMRecord} record The record to update
	 * @private
	 */
	updateRecord : function(record)
	{
		record.beginEdit();

		var pattern = this.recurrencePatternSelect.getValue();
		if (pattern) {
			record.set('recurrence_type', pattern.patternValue);
		} else {
			record.set('recurrence_type', Zarafa.common.recurrence.data.RecurrenceType.NONE);
		}

		// The start is represented in UTC time,
		// so convert it to local to get the time for the property
		record.set('recurrence_start', this.startDateField.getValue().fromUTC());

		Ext.each(this.endPatternPanel.findByType('radio'), function(radio) {
			if (radio.getValue()) {
				record.set('recurrence_term', radio.endTerm);
				return false;
			}
		})

		switch (record.get('recurrence_term')) {
			case Zarafa.common.recurrence.data.RecurrenceEnd.NEVER:
				record.set('recurrence_numoccur', undefined);
				// The end is represented in UTC time,
				// so convert it to local to get the time for the property
				record.set('recurrence_end', new Date('Jan 01 4501').fromUTC());
				break;
			case Zarafa.common.recurrence.data.RecurrenceEnd.N_OCCURENCES:
				record.set('recurrence_numoccur', this.endOccurencesSpinner.getValue());
				// The end is represented in UTC time,
				// so convert it to local to get the time for the property
				record.set('recurrence_end', new Date('Jan 01 4501').fromUTC());
				break;
			case Zarafa.common.recurrence.data.RecurrenceEnd.ON_DATE:
				record.set('recurrence_numoccur', undefined);
				// The end is represented in UTC time,
				// so convert it to local to get the time for the property
				record.set('recurrence_end', this.endOnDateField.getValue().fromUTC());
				break;
		}

		// The recurrence_start and the time values might have changed, time
		// to update the appointment data now as well.
		record.set('alldayevent', this.alldayCheckbox.getValue());

		var timeRange = this.timeperiodField.getValue();
		var startOcc = (timeRange.getStartDate().getHours() * 60) + timeRange.getStartDate().getMinutes();
		var endOcc = startOcc + timeRange.getDuration(Date.MINUTE);

		record.set('recurrence_startocc', startOcc);
		record.set('recurrence_endocc', endOcc);

		var startDate = this.startDateField.getValue();
		var dueDate = startDate.clone();

		// Small workaround for allday appointments, in Brasil the DST occurs at 00:00,
		// which means for allday events, that we might think the appointment should start
		// at 01:00, but that might be because that is the start of the actual day. Hence
		// we need to correct the startOcc here for that.
		if (record.get('alldayevent') && startDate.clearTime(true).getTime() === startDate.getTime()) {
			var offset = startDate.getHours() * 60;
			startOcc -= offset;
			endOcc -= offset;
		}

		startDate = startDate.add(Date.MINUTE, startOcc);
		record.set('startdate', startDate);
		record.set('commonstart', startDate);

		dueDate = dueDate.add(Date.MINUTE, endOcc);
		record.set('duedate', dueDate);
		record.set('commonend', dueDate);

		record.endEdit();
	}
});

Ext.reg('zarafa.recurrencepanel', Zarafa.common.recurrence.dialogs.RecurrencePanel);
Ext.namespace('Zarafa.common.recurrence.dialogs');

/**
 * @class Zarafa.common.recurrence.dialogs.RecurrenceSubPanel
 * @extends Ext.Panel
 * @xtype zarafa.recurrencesubpanel
 *
 * The Panel used for configuring the {@link Zarafa.common.recurrence.data.RecurrenceSubtype subtypes}.
 * This class should be inherited by the different {@link Zarafa.common.recurrence.data.RecurrenceType types}.
 */
Zarafa.common.recurrence.dialogs.RecurrenceSubPanel = Ext.extend(Ext.Panel, {
	/**
	 * The record on which this panel is operating. This s provided in {@link #update} by
	 * the {@link Zarafa.core.plugins.RecordComponentUpdaterPlugin Record ComponentUpdater plugin}.
	 * @property
	 * @type Zarafa.core.data.MAPIRecord
	 */
	record : undefined,

	/**
	 * @cfg {Zarafa.common.recurrence.data.RecurrenceType} recurrenceType The recurrence type on
	 * which is panel is operating.
	 */
	recurrenceType : undefined,

	/**
	 * @cfg {Zarafa.common.recurrence.data.RecurrenceSubtype|Array} recurrenceSubtypes The possible
	 * subtypes which are available for the configured {@link #recurrenceType}.
	 */
	recurrenceSubtypes : undefined,

	/**
	 * @constructor
	 * @param {Object} config Configuration object
	 */
	constructor : function(config)
	{
		config = config || {};

		config.plugins = Ext.value(config.plugins, []);
		config.plugins.push('zarafa.recordcomponentupdaterplugin');

		Zarafa.common.recurrence.dialogs.RecurrenceSubPanel.superclass.constructor.call(this, config);
	},

	/**
	 * Compare 2 {@link Zarafa.common.recurrence.data.RecurrenceSubtype} instances to see if they are equal
	 * @param {Zarafa.common.recurrence.data.RecurrenceSubtype} left The first object to compare to
	 * @param {Zarafa.common.recurrence.data.RecurrenceSubtype} right The second object to compare to
	 * @return {Boolean} True when the two {@link Zarafa.common.recurrence.data.RecurrenceSubtype subtype}
	 * objects are equal
	 * @protected
	 */
	isSubtype : function(left, right)
	{
		return (left.type === right.type && left.regen === right.regen);
	},

	/**
	 * Compare if the given {@link Zarafa.core.data.MAPIRecord} has a subtype configured which matches the given
	 * {@link Zarafa.common.recurrence.data.RecurrenceSubtype subtype}.
	 * @param {Zarafa.core.data.MAPIRecord} record The record to compare to
	 * @param {Zarafa.common.recurrence.data.RecurrenceSubtype} pattern the Subtype pattern to compare with
	 * @return {Boolean} True when the given record has the same subtype configured as the given pattern
	 * @protected
	 */
	isRecordSubtype : function(record, pattern)
	{   
		return (record.get('recurrence_subtype') === pattern.type && record.get('recurrence_regen') === pattern.regen);
	},

	/**
	 * Event handler which is called when the Subtype has been changed. This must be configured by the
	 * subclass as event handler on the {@link Ext.form.Radio#change change} event. This will {@link #updateRecord update}
	 * the {@link #record}.
	 * @param {Ext.form.Radio} field The radio which was changed
	 * @param {Boolean} value True when the radio was enabled
	 * @protected
	 */
	onSubtypeChange : function(field, value)
	{
		if (value) {
			var pattern = field.patternValue;

			// The recurrence properties use 'forceProtocol' on the field definition, this means
			// that they will always be markked as changed when we set them. So to be safe,
			// we manually check if we are about to set a different value or not.
			//
			// We also apply a full update of all properties, because when the subtype changes,
			// the meaning of the recurrence properties like 'everyn', 'nday', 'weekdays', etc. also changes.
			if (!this.isRecordSubtype(this.record, pattern)) {
				this.updateRecord(this.record);
			}
		}
	},

	/**
	 * Event handler which is called when a property for a specific subtype has been changed.
	 * This must be called by the subclass with the subtype object to indicate for which subtype
	 * the given field must be changed.
	 * @param {Zarafa.common.recurrence.data.RecurrenceSubtype} pattern The subtype pattern to which
	 * the given field belongs. The property will only be updated when this subtype is currently enabled.
	 * @param {Ext.form.Field} field The field which was changed by the user
	 * @param {Mixed} value The value which was entered into the field
	 * @protected
	 */
	onSubtypePropertyChange : function(pattern, field, value)
	{
		if (this.isRecordSubtype(this.record, pattern)) {
			this.record.set(field.name, value);
		}
	},

	/**
	 * Enable/disable/hide/unhide all {@link Ext.Component Components} within the {@link Ext.Panel Panel}
	 * using the given {@link Zarafa.core.data.MAPIRecord record}.
	 * @param {Zarafa.core.data.MAPIRecord} record The record to update the panel with
	 * @param {Boolean} contentReset force the component to perform a full update of the data.
	 * @protected
	 */
	updateUI : Ext.emptyFn,

	/**
	 * Update the {@link Ext.Component components} which belong to the given
	 * {@link Zarafa.common.recurrence.data.RecurrenceSubtype subtype}.
	 * @param {Zarafa.core.data.MAPIRecord} record The record from where the values must be read
	 * @param {Zarafa.common.recurrence.data.RecurrenceSubtype} subtype The subtype for which the UI
	 * components must be updated
	 * @param {Boolean} useDefault True if default values should be used rather then the data from
	 * the given record.
	 * @protected
	 */
	updateSubtype : Ext.emptyFn,

	/**
	 * Panel updater. This will initialize all UI components inside the panel with
	 * the data from the {@link Zarafa.core.data.MAPIRecord record}.
	 *
	 * @param {Zarafa.core.data.MAPIRecord} record The record used to update the panel
	 * @param {Boolean} contentReset force the component to perform a full update of the data.
	 */
	update : function(record, contentReset)
	{
		this.record = record;
		this.updateUI(record, contentReset);

		// The complexity of this panel lies in the fact that we have a recurrence type,
		// and subtype. The properties inside the record have different meanings based on
		// the selected subtype. So when the recurrence type doesn't match this panel's type,
		// we must load default values for everything.
		//
		// However when this panel's type does match the recurrence type, then we must load
		// the correct values for the UI components belonging to the subtype, and default
		// values for all other components.
		//
		// Only do this during intializing as the user will probably be switching around
		// types and subtypes a couple of times.
		if (contentReset) {
			var subTypeRadios = this.findByType('radio');

			if (record.get('recurrence_type') !== this.recurrenceType) {

				// Mark the first radio as the default subtype.
				subTypeRadios[0].setValue(true);

				// Load the default information for all subtypes
				for (var key in this.recurrenceSubtypes) {
					var pattern = this.recurrenceSubtypes[key];

					this.updateSubtype(record, pattern, true);
				}
			} else {
				// Search which radio represents the desired subtype
				Ext.each(subTypeRadios, function(radio) {
					radio.setValue(this.isRecordSubtype(record, radio.patternValue));
				}, this);

				// Load the correct settings for the enabled subtype, and the
				// default values for all others.
				for (var key in this.recurrenceSubtypes) {
					var pattern = this.recurrenceSubtypes[key];

					this.updateSubtype(record, pattern, !this.isRecordSubtype(record, pattern));
				}
			}
		}
	},

	/**
	 * Called by {@link #updateRecordSubType} to indicate that the record must be updated for the
	 * given {@link Zarafa.common.recurrence.data.RecurrenceSubtype recurrence subtype}. The record
	 * itself is already in {@link Zarafa.core.data.MAPIRecord#editing editing} mode.
	 * @param {Zarafa.core.data.MAPIRecord} record The record which must be updated from the UI
	 * @param {Zarafa.common.recurrence.data.RecurrenceSubtype} pattern The Subtype which is
	 * currently enabled. Only the components for this subtype must be used to update the record.
	 * @protected
	 */
	updateRecordSubType : Ext.emptyFn,

	/**
	 * Record updater. This will update the record with all data from the UI components
	 * inside the Panel.
	 *
	 * @param {Zarafa.core.data.MAPIRecord} record The record to update
	 * @private
	 */
	updateRecord : function(record)
	{
		// We are not the active Panel, we have nothing to configure on the record.
		if (record.get('recurrence_type') !== this.recurrenceType) {
			return;
		}

		// Search which subtype has been enabled, we didn't use
		// a RadioGroup, which would remove the requirements for
		// findByType and the looping.
		Ext.each(this.findByType('radio'), function(radio) {
			if (radio.getValue()) {
				var pattern = radio.patternValue;
				
				record.beginEdit();

				// Update the subtype which was selected
				record.set('recurrence_subtype', pattern.type);
				record.set('recurrence_regen', pattern.regen);

				// Load the corresponding properties
				this.updateRecordSubType(record, pattern);

				record.endEdit();

				return false;
			}
		}, this);

	}
});

Ext.reg('zarafa.recurrencesubpanel', Zarafa.common.recurrence.dialogs.RecurrenceSubPanel);
Ext.namespace('Zarafa.common.reminder.dialogs');

/**
 * @class Zarafa.common.reminder.dialogs.ReminderPanel
 * @extends Ext.Panel
 * @xtype zarafa.reminderpanel
 */
Zarafa.common.reminder.dialogs.ReminderPanel = Ext.extend(Ext.Panel, {
	/**
	 * @cfg {Zarafa.common.reminder.data.ReminderStore} store store that will be used to get reminder information.
	 */
	store : undefined,

	/**
	 * @cfg {Ext.Template/String} activeReminderTemplate The template or template string which
	 * must be applied to the {@link Zarafa.common.reminder.dialogs.ReminderPanel #activeReminder} to build selected reminder info when the 
	 * {@link Zarafa.common.reminder.data.ReminderRecord record} has been {@link #update updated}. 
	 * The arguments of this template will be the {@link Zarafa.common.reminder.data.ReminderRecord#data record.data} field.
	 */
	activeReminderTemplate : 
		'<tpl>' +
			'<div>'+
				'<span class="zarafa-reminder-dialog-active-reminder-icon {[Zarafa.common.ui.IconClass.getIconClassFromMessageClass(false, values.message_class)]}"></span>'+
				'<span class="zarafa-reminder-dialog-active-reminder-subject">' +
					'<tpl if="!Ext.isEmpty(values.subject)">' + 
						// @FIXME this should be changed to truncate strings based on dialog width
						'{values.subject:htmlEncodeEllipsis(60)}'+
					'</tpl>' +
					'<tpl if="Ext.isEmpty(values.subject)">' + 
						'&nbsp;'+
					'</tpl>' +
				'</span>'+
			'</div>'+
			'<div>'+
				'<span>' +
					// Use &quot; instead of " or \" as neither will be accepted by the XTemplate for creating a string
					'<tpl if="Zarafa.core.ContainerClass.isClass(values.message_class, &quot;IPM.Task&quot;, true)">' +
						'<tpl if="Ext.isDefined(values.task_duedate)">' +
							_('Due') + ': {values.task_duedate:date("' +
							// # TRANSLATORS: See http://docs.sencha.com/ext-js/3-4/#!/api/Date for the meaning of these formatting instructions
							_('l jS F Y') + '")}' +
						'</tpl>' +
						'<tpl if="Ext.isDefined(values.reminder_time) && !Ext.isDefined(values.task_duedate)">' +
							_('Reminder time') + ': {values.reminder_time:date("' +
							// # TRANSLATORS: See http://docs.sencha.com/ext-js/3-4/#!/api/Date for the meaning of these formatting instructions
							_('l jS F Y G:i') + '")}'+
						'</tpl>' +
					'</tpl>' +
					// Use &quot; instead of " or \" as neither will be accepted by the XTemplate for creating a string
					'<tpl if="!Zarafa.core.ContainerClass.isClass(values.message_class, &quot;IPM.Task&quot;, true)">' +
						'<tpl if="Ext.isDefined(values.reminder_time)">' +
							_('Start time') + ': {values.reminder_time:date("' +
							// # TRANSLATORS: See http://docs.sencha.com/ext-js/3-4/#!/api/Date for the meaning of these formatting instructions
							_('l jS F Y G:i') + '")}'+
						'</tpl>' +
					'</tpl>' +
				'</span>'+
			'</div>'+
			// @TODO show location data
		'</tpl>',

	/**
	 * @constructor
	 * @param config Configuration structure
	 */
	constructor : function(config)
	{
		config = config || {};

		Ext.applyIf(config, {
			xtype : 'zarafa.reminderpanel',
			layout: {
				type : 'vbox',
				align : 'stretch',
				pack  : 'start'
			},
			header: false,
			border: false,
			defaults: {
				border: false
			},
			items: [
				this.activeReminderPanel(),
				this.reminderGrid(config.store),
				this.reminderAction(),
				this.reminderSnooze()
			]
		});

		Zarafa.common.reminder.dialogs.ReminderPanel.superclass.constructor.call(this, config);

		if (Ext.isString(this.activeReminderTemplate)) {
			this.activeReminderTemplate = new Ext.XTemplate(this.activeReminderTemplate, {
				compiled: true
			});
		}
	},
	
	/**
	 * Create the {@link Ext.Panel panel} containing the information about active/selected reminder
	 * and some extra information about it.
	 * @return {Object} Configuration object for the panel containing the fields
	 * @private
	 */
	activeReminderPanel : function()
	{
		return{
			xtype: 'panel',
			cls:'zarafa-reminder-dialog-active-reminder',
			layout: 'fit',
			ref: 'activeReminder',
			height : 40
		}
	},
	
	/**
	 * Create the {@link Zarafa.common.reminder.dialogs.ReminderGrid ReminderGrid} object.
	 * @param {Zarafa.common.reminder.ReminderStore} store reminder store.
	 * @return {Object} The configuration object for the reminder grid
	 * @private
	 */
	reminderGrid : function(store)
	{
		return{
			xtype: 'zarafa.remindergrid',
			ref : 'reminderGridView',
			store : store,
			border : true,
			flex : 2
		};
	},
	
	/**
	 * Create the {@link Ext.Panel Panel} containing the form button fields
	 * which performs different action on selected reminders
	 * @return {Object} Configuration object for the panel with reminder action fields
	 * @private
	 */
	reminderAction : function()
	{
		return{
			xtype : 'panel',
			layout : 'fit',
			height : 40,
			ref : 'reminderAction',
			minButtonWidth : 90,
			buttons : [{
				text	: _('Dismiss All'),
				handler	: this.onDismissAll,
				ref: '../../dismissAllButton',
				scope	: this
			},{
				text	: _('Open Item'),
				handler	: this.onOpenItem,
				ref: '../../openButton',
				scope	: this
			},{
				text	: _('Dismiss'),
				handler	: this.onDismiss,
				ref: '../../dismissButton',
				scope	: this
			}]
		};
	},
	
	/**
	 * Create the {@link Ext.Panel Panel} containing the form snooze button fields
	 * which will set the snoozing time for selected reminders
	 * @return {Object} Configuration object for the panel with snooze field
	 * @private
	 */
	reminderSnooze : function()
	{
		var reminderStore = {
			xtype: 'jsonstore',
			fields: ['name', 'value'],
			data : Zarafa.calendar.data.ReminderPeriods
		};

		return{
			xtype: 'panel',
			cls: 'zarafa-reminder-dialog-snooze',
			layout: {
				type : 'vbox',
				align : 'stretch',
				pack  : 'start'
			},
			height : 80,
			ref: 'reminderAction',
			items:[{
				xtype: 'displayfield',
				value:  _('Click Snooze to be reminded again in') + ': ',
				hideLabel : true,
				height : 25,
				ref: '../snoozeInfoText'
			},{
				xtype: 'panel',
				layout: 'hbox',
				border: false,
				flex : 1,
				items: [{
					xtype: 'combo',
					ref: '../../snoozeTimeCombo',
					name: 'snooze',
					flex : 1,
					store: reminderStore,
					mode: 'local',
					triggerAction: 'all',
					displayField: 'name',
					valueField: 'value',
					value : container.getSettingsModel().get('zarafa/v1/main/reminder/default_snooze_time'),
					lazyInit: false,
					forceSelection: true,
					editable: false
				},{
					xtype: 'spacer',
					width: 10
				},{
					xtype	: 'button',
					width : 90,
					text	: _('Snooze'),
					handler	: this.onSnooze,
					ref: '../../snoozeButton',
					scope	: this
				}]
			}]
		};
	},

	/**
	 * initialize events for the {@link Zarafa.common.reminder.dialogs.ReminderPanel ReminderPanel}
	 * @private
	 */
	initEvents : function()
	{
		Zarafa.common.reminder.dialogs.ReminderPanel.superclass.initEvents.call(this);
	
		this.mon(this.reminderGridView, 'viewready', this.onViewReady, this);
		this.mon(this.reminderGridView.getSelectionModel(), 'selectionchange', this.onSelectionChange, this);
		this.mon(this.reminderGridView.getSelectionModel(), 'selectionchange', this.toggleFields, this);
	},

	/**
	 * Event handler which is fired when the {@link Zarafa.common.reminder.dialogs.ReminderGrid ReminderGrid}
	 * is ready. This will automatically select the first row in the grid.
	 * @private
	 */
	onViewReady: function()
	{
		this.reminderGridView.getSelectionModel().selectFirstRow();
	},

	/**
	 * Event handler which is triggered when the {@link Zarafa.common.reminder.dialogs.ReminderGrid ReminderGrid}
	 * {@link Zarafa.core.data.IPMRecord record} selection is changed. This will inform
	 * the {@link Zarafa.common.reminder.dialogs.ReminderPanel} about the change.
	 *
	 * @param {Ext.grid.RowSelectionModel} selectionModel The selection model used by the grid.
	 * @private
	 */
	onSelectionChange : function()
	{
		var selections = this.reminderGridView.getSelectionModel().getSelections();

		// clear the previous contents
		var el = this.activeReminder.getEl();
		if(Ext.isDefined(el.dom)) {
			el.dom.innerHTML = '';
		}

		if (selections.length == 1) {
			this.activeReminderTemplate.overwrite(el.dom , selections[0].data);
		} else {
			el.createChild({
				tag: 'div',
				html: String.format(
					npgettext('reminder.dialog', '{0} reminder is selected.', '{0} reminders are selected.', selections.length),
					selections.length
				)
			});
		}

		this.doLayout();
	},

	/**
	 * Event handler which is triggered when the {@link Zarafa.common.reminder.dialogs.ReminderGrid grid}
	 * {@link Zarafa.core.data.IPMRecord record} selection is changed.
	 * 
	 * Function will disable buttons if there isn't any reminder selected.
	 * @private
	 */
	toggleFields : function()
	{
		var reminderCount = this.reminderGridView.getStore().getCount();

		var hasSelection = false;
		if(reminderCount > 0) {
			hasSelection = this.reminderGridView.getSelectionModel().hasSelection();
		}

		// If any reminder is not selected then disable buttons and fields.
		this.dismissButton.setDisabled(!hasSelection);
		this.openButton.setDisabled(!hasSelection);
		this.snoozeTimeCombo.setDisabled(!hasSelection);
		this.snoozeButton.setDisabled(!hasSelection);
		this.snoozeInfoText.setDisabled(!hasSelection);

		// if any reminder is present then show dismiss all button
		// @FIXME this should be done on store events instead of here
		this.dismissAllButton.setDisabled(reminderCount <= 0);
	},
	
	/**
	 * Function will send request to dismiss all reminder present at the given time.
	 * @param {Ext.Button} button button object.
	 * @param {EventObject} eventObject The click event object.
	 * @private
	 */
	onDismissAll : function()
	{
		this.store.dismissReminders(this.store.getRange());

		// close the dialog, if error occurs in dismiss all then reminder dialog will popup again after reminder interval
		this.dialog.close();
	},

	/**
	 * Function will open appointment/task/mail item for the selected reminder item.
	 * if multiple reminders are selected then it will open only the first item.
	 * @param {Ext.Button} button button object.
	 * @param {EventObject} eventObject The click event object.
	 * @private
	 */
	onOpenItem : function()
	{
		var record = this.reminderGridView.getSelectionModel().getSelected();

		if(record) {
			Zarafa.common.Actions.openReminderRecord(record);
		}
	},

	/**
	 * Function will send request to dismiss selected reminder
	 * @param {Ext.Button} button button object.
	 * @param {EventObject} eventObject The click event object.
	 * @private
	 */
	onDismiss : function()
	{
		var selectedRecords = this.reminderGridView.getSelectionModel().getSelections();

		this.store.dismissReminders(selectedRecords);

		// we have selected all reminders to dismiss, so a virtual dismiss all
		if(this.store.getCount() === 0) {
			// close the dialog, if error occurs in dismiss then reminder dialog will popup again after reminder interval
			this.dialog.close();
		}
	},
	
	/**
	 * Function will set the snooze timing for selected reminders
	 * @param {Ext.Button} button button object.
	 * @param {EventObject} eventObject The click event object.
	 * @private
	 */
	onSnooze : function()
	{
		var selectedRecords = this.reminderGridView.getSelectionModel().getSelections();
		this.store.snoozeReminders(selectedRecords, this.snoozeTimeCombo.getValue());
	}
});

Ext.reg('zarafa.reminderpanel', Zarafa.common.reminder.dialogs.ReminderPanel);
Ext.namespace('Zarafa.common.restoreitem.dialogs');

/**
 * @class Zarafa.common.restoreitem.dialogs.RestoreItemPanel
 * @extends Ext.grid.GridPanel
 * @xtype zarafa.restoreitempanel
 *
 * A gridPanel which provides UI for RestoreItem dialog to display the list of soft deleted items.
 * It also contains a {@link Zarafa.core.ui.ContentPanelToolbar ContentPanelToolbar}.
 */
Zarafa.common.restoreitem.dialogs.RestoreItemPanel = Ext.extend(Ext.grid.GridPanel, {

	/**
	 * @cfg {Zarafa.hierarchy.data.MAPIFolderRecord} folder default folder for the contextModel.
	 */
	folder : undefined,

	/**
	 * The current type of soft deleted items list.
	 * Default value is 'message' as By default Grid is loaded with soft deleted message list.
	 * This String is changed by {@link #onRadioChange}.
	 * @property
	 * @type String
	 * @private
	 */
	itemType : 'message',

	/**
	 * @constructor
	 * @param {Object} config Configuration structure
	 */
	constructor: function(config)
	{
		config = config || {};

		Ext.applyIf(config,{
			xtype: 'zarafa.restoreitempanel',
			border: false,
			store: new Zarafa.common.restoreitem.data.RestoreItemStore({
				autoLoad: {
					folder: config.folder
				}
			}),
			loadMask: true,
			viewConfig: {
				forceFit: true,
				emptyText : '<div class="emptytext">' + _('There are no items to show in this view') + '</div>',
				getRowClass : this.viewConfigGetRowClass
			},
			tbar: {
				xtype: 'zarafa.contentpaneltoolbar',
				insertionPointBase: 'common.restoreitemcontentpanel',
				actionItems: this.createActionButtons()
			},
			bbar: [{
					xtype: 'tbfill'
				},{
					xtype : 'displayfield',
					height : 15,
					ref : '../statusMessage'
			}],
			colModel: this.initMessageColumnModel(),
			sm : new Ext.grid.RowSelectionModel()
		});

		Zarafa.common.restoreitem.dialogs.RestoreItemPanel.superclass.constructor.call(this,config);

		this.mon(this.store, 'load', this.onStoreLoad, this);
		this.mon(this.store, 'load', this.disableToolbarButton, this);
		this.mon(this.store, 'load', this.setStatusMessage, this);
		this.mon(this.store, 'remove', this.setStatusMessage, this);
		this.mon(this.store, 'remove', this.disableToolbarButton, this);
		this.mon(this.getSelectionModel(), 'selectionchange', this.onSelectionChange, this);

	},

	/**
	 * Apply custom style and content for the row body. This will always
	 * apply the Read/UnRead style to the entire row.
	 *
	 * @param {Ext.data.Record} record The {@link Ext.data.Record Record} corresponding to the current row.
	 * @param {Number} rowIndex The row index
	 * @param {Object} rowParams A config object that is passed to the row template during
	 * rendering that allows customization of various aspects of a grid row.
	 * @param {Ext.data.Store} store The Ext.data.Store this grid is bound to
	 * @return {String} a CSS class name to add to the row
	 * @private
	 */
	viewConfigGetRowClass :function(record, rowIndex, rowParams, store)
	{
		var cssClass = !record.isRead() ? 'mail_unread' : 'mail_read';
		return cssClass;
	},

	/**
	 * Function will create buttons for top toolbar and it will be
	 * attached to zarafa.contentpaneltoolbar
	 * @return {Array} array consist of buttons for restore content panel
	 * @private
	 */
	createActionButtons : function()
	{
		var radioGroupName = Ext.id(null, 'restoreitem-');
		return [{
			xtype: 'button',
			overflowText: _('Restore'),
			text: _('Restore'),
			iconCls: 'icon_restore',
			handler : this.onRestore,
			disabled : true,
			ref : '../restoreButton',
			scope: this
		}, {
			xtype: 'button',
			overflowText: _('Restore All'),
			text: _('Restore All'),
			iconCls: 'icon_restore',
			handler : this.onRestoreAll,
			disabled : true,
			ref : '../restoreAllButton',
			scope: this
		},{
			xtype: 'button',
			overflowText: _('Permanent Delete'),
			text: _('Permanent Delete'),
			iconCls: 'icon_delete',
			handler : this.onPermanentDelete,
			disabled : true,
			ref : '../permanentDeleteButton',
			scope: this
		},{
			xtype: 'button',
			overflowText: _('Delete All'),
			text: _('Delete All'),
			iconCls: 'icon_delete',
			handler : this.onDeleteAll,
			disabled : true,
			ref : '../deleteAllButton',
			scope: this
		},{
			xtype: 'tbfill'
		},{
			xtype : 'radio',
			width: 70,
			boxLabel: _('Messages'),
			name: radioGroupName,
			inputValue: 'message',
			checked: true,
			listeners : {
				check : this.onRadioChecked,
				scope : this
			}
		},{
			xtype : 'radio',
			width: 70,
			boxLabel: _('Folders'),
			name: radioGroupName,
			inputValue: 'folder',
			listeners : {
				check : this.onRadioChecked,
				scope : this
			}
		}]
	},

	/**
	 * Event handler which is fired when a {@link Ext.form.Radio}
	 * has been checked. This will send request to load data in store.
	 * @param {Ext.form.Radio} radio The radio which was checked
	 * @param {Boolean} checked The new checked value
	 * @private
	 */
	onRadioChecked : function(radio, checked)
	{
		// load data about deleted message or folder in store based on the selected radio
		if (checked === true) {
			this.itemType = radio.inputValue;

			this.store.load({
				params: {
					itemType : this.itemType
				}
			});
		}
	},

	/**
	 * Event handler when the "Permanent Delete" button has been pressed.
	 * This will obtain the selected record(s) to be hard-deleted by
	 * {@link Ext.grid.RowSelectionModel#getSelections getSelection} method.
	 * @param {Ext.Button} button button component
	 * @param {Ext.EventObject} eventObj event object for the click event.
	 * @private
	 */
	onPermanentDelete : function(button, eventObj)
	{
		var records = this.getSelectionModel().getSelections();
		this.doPermanentDelete(records);
	},

	/**
	 * Event handler when the "Delete All" button has been pressed.
	 * This will raise a Conformation which ask user that he/she want to go ahead and
	 * Permanently delete all the items or not.
	 * @param {Ext.Button} button button component
	 * @param {Ext.EventObject} eventObj event object for the click event.
	 * @private
	 */
	onDeleteAll : function(button, eventObj)
	{
		var records = this.store.getRange();

		Ext.MessageBox.confirm(
			_('Zarafa WebApp'),
			_('Are you sure you want to Permanently Delete all items?'),
			function (buttonClicked) {
				if (buttonClicked == 'yes') {
					this.doPermanentDelete(records);
				}
			},
		this);
	},

	/**
	 * This will Permanently delete the selected record(s) which is passed as argument.
	 * @param {Zarafa.common.restoreitem.data.RestoreItemRecord|Array} records The records that should be hard-deleted
	 * @private
	 */
	doPermanentDelete : function(records)
	{
		var saveRecords = [];

		if (!Ext.isEmpty(records)) {
			for (var i = 0, len = records.length; i < len; i++) {
				var record = records[i];

				record.addMessageAction('action_type', 'delete' + this.itemType);

				this.store.remove(record);
				saveRecords.push(record);
			}
			this.store.save(saveRecords);
		}
	},

	/**
	 * Event handler when the "Restore" button has been pressed.
	 * This will obtain the selected record(s) to be restored by
	 * {@link Ext.grid.RowSelectionModel#getSelections getSelection} method.
	 * @param {Ext.Button} button button component
	 * @param {Ext.EventObject} eventObj event object for the click event.
	 * @private
	 */
	onRestore : function(button, eventObj)
	{
		var records = this.getSelectionModel().getSelections();
		this.doRestore(records);
	},

	/**
	 * Event handler when the "Restore All" button has been pressed.
	 * This will raise a Conformation which ask user that he/she want to go ahead and
	 * restore all the items or not.
	 * @param {Ext.Button} button button component
	 * @param {Ext.EventObject} eventObj event object for the click event.
	 * @private
	 */
	onRestoreAll : function(button, eventObj)
	{
		var records = this.store.getRange();
		Ext.MessageBox.confirm(
			_('Zarafa WebApp'),
			_('Are you sure you want to Restore all items?'),
			function (buttonClicked) {
				if (buttonClicked == 'yes') {
					this.doRestore(records);
				}
			},
		this);
	},

	/**
	 * This will retore the selected record(s) which is passed as argument.
	 * @param {Zarafa.common.restoreitem.data.RestoreItemRecord|Array} records The records that should be restored
	 * @private
	 */
	doRestore : function(records)
	{
		var saveRecords = [];

		if (!Ext.isEmpty(records)) {
			for (var i = 0, len = records.length; i < len; i++) {
				var record = records[i];
				record.addMessageAction('action_type', 'restore' + this.itemType);
				this.store.remove(record);
				saveRecords.push(record);
			}

			this.store.save(saveRecords);
		}
	},

	/**
	 * Event handler which is called when the {@link #store} has been loaded.
	 * The Appropriate Column Model will be configured accordingly.
	 * @param {Ext.data.Store} store The store which was loaded
	 * @param {Ext.data.Record|Array} records The records which were loaded from the store
	 * @param {Object} options The options which were used to load the data
	 * @private
	 */
	onStoreLoad : function( store, records, options )
	{
		if (Ext.isDefined(options.params.itemType) && options.params.itemType == 'folder') {
			this.reconfigure(store, this.initFolderColumnModel());
		} else {
			this.reconfigure(store, this.initMessageColumnModel());
		}

		this.getSelectionModel().selectFirstRow();
	},

	/**
	 * Event handler which is called when the selection has been changed.
	 * The tool bar buttons for Permanent-Delete/Restore will be activate/deactivate accordingly.
	 * @param {Ext.grid.RowSelectionModel} selectionModel The selection model used by the grid.
	 * @private
	 */
	onSelectionChange : function(selectionModel)
	{
		var noSelection = (selectionModel.hasSelection() === false);

		this.restoreButton.setDisabled(noSelection);
		this.permanentDeleteButton.setDisabled(noSelection);
	},

	/**
	 * Event handler for the {@link Ext.data.Store#remove} event which is fired
	 * by the {@link Ext.data.Store}.
	 * This will update the value of {@link #statusMessage}.
	 * @param {Ext.data.Store} store The store which fired the event
	 * @param {Ext.data.Record} record The record which was updated
	 * @param {Number} index The index from where the records were deleted
	 * @private
	 */
	setStatusMessage : function(store, record, index)
	{
		this.statusMessage.setValue(String.format(ngettext('Total {0} recoverable item', 'Total {0} recoverable items', store.getCount()), store.getCount()));
	},

	/**
	 * Event handler for the {@link Ext.data.Store#remove} event which is fired
	 * by the {@link Ext.data.Store}.
	 * This will activate/deactivate tool bar buttons for DeleteAll/RestoreAll accordingly.
	 * @param {Ext.data.Store} store The store which fired the event
	 * @param {Ext.data.Record} record The record which was updated
	 * @param {Number} index The index from where the records were deleted
	 * @private
	 */
	disableToolbarButton : function(store, record, index)
	{
		var noItem = (store.getCount() === 0);
		this.deleteAllButton.setDisabled(noItem);
		this.restoreAllButton.setDisabled(noItem);
	},

	/**
	 * Creates and returns a column model object to display Folder related columns,
	 * used in {@link Ext.grid.GridPanel.colModel colModel} config
	 * @return {Ext.grid.ColumnModel} column model object
	 * @private
	 */
	initFolderColumnModel : function()
	{
		return new Ext.grid.ColumnModel([{
			dataIndex : 'icon_index',
			header : '<p class="icon_index">&nbsp;</p>',
			tooltip : _('Sort by: Icon'),
			width : 25,
			sortable: true,
			fixed : true,
			renderer : Zarafa.common.ui.grid.Renderers.icon
		},{
			dataIndex:'display_name',
			header: _('Name'),
			tooltip : _('Sort by: Name'),
			sortable: true,
			renderer: Ext.util.Format.htmlEncode
		},{
			dataIndex:'deleted_on',
			header: _('Deleted On'),
			tooltip : _('Sort by: Deleted On'),
			sortable: true,
			renderer: Zarafa.common.ui.grid.Renderers.datetime
		},{
			dataIndex: 'content_count',
			header: _('Item Count'),
			tooltip : _('Sort by: Item Count'),
			sortable: true,
			renderer: Ext.util.Format.htmlEncode
		}]);
	},

	/**
	 * Creates and returns a column model object to display Messages related columns,
	 * used in {@link Ext.grid.GridPanel.colModel colModel} config
	 * @return {Ext.grid.ColumnModel} column model object
	 * @private
	 */
	initMessageColumnModel : function()
	{
		return new Ext.grid.ColumnModel([{
			dataIndex : 'icon_index',
			header : '<p class="icon_index">&nbsp;</p>',
			tooltip : _('Sort by: Icon'),
			width : 25,
			sortable: true,
			fixed : true,
			renderer : Zarafa.common.ui.grid.Renderers.icon
		},{
			dataIndex : 'hasattach',
			header : '<p class=\'icon_attachment\'>&nbsp;</p>',
			tooltip : _('Sort by: Attachment'),
			width : 24,
			sortable: true,
			fixed : true,
			renderer : Zarafa.common.ui.grid.Renderers.attachment
		},{
			dataIndex:'sender_name',
			header: _('From'),
			tooltip : _('Sort by: From'),
			sortable: true,
			renderer: Ext.util.Format.htmlEncode
		},{
			dataIndex: 'subject',
			header: _('Subject'),
			tooltip : _('Sort by: Subject'),
			sortable: true,
			renderer: Ext.util.Format.htmlEncode
		},{
			dataIndex:'deleted_on',
			header: _('Deleted On'),
			tooltip : _('Sort by: Deleted On'),
			sortable: true,
			renderer: Zarafa.common.ui.grid.Renderers.datetime
		},{
			dataIndex : 'message_delivery_time',
			header : _('Received'),
			tooltip : _('Sort by: Received'),
			sortable: true,
			renderer : Zarafa.common.ui.grid.Renderers.datetime
		},{
			dataIndex:'message_size',
			header: _('Size'),
			tooltip : _('Sort by: Size'),
			sortable: true,
			renderer: Zarafa.common.ui.grid.Renderers.size
		}]);
	}
});

Ext.reg('zarafa.restoreitempanel',Zarafa.common.restoreitem.dialogs.RestoreItemPanel);Ext.namespace('Zarafa.common.rules.dialogs');

/**
 * @class Zarafa.common.rules.dialogs.FolderSelectionLink
 * @extends Ext.BoxComponent
 * @xtype zarafa.folderselectionlink
 *
 * Action component for the {@link Zarafa.common.rules.data.ActionFlags#MOVE MOVE}
 * and {@link Zarafa.common.rules.data.ActionFlags#COPY COPY} Action Flags. This
 * will show the currently selected {@link #folder} to the user, and allows
 * the user to select the desired destination folder.
 */
Zarafa.common.rules.dialogs.FolderSelectionLink = Ext.extend(Ext.BoxComponent, {
	/**
	 * @cfg {String} fieldLabel The label which must be applied to template
	 * as a prefix to the list of attachments.
	 */
	emptyText :_('Select one...'),

	/**
	 * The folder which was selected by the user
	 * @property
	 * @type Zarafa.hierarchy.data.MAPIFolderRecord
	 */
	folder : undefined,

	/**
	 * The Action type which is handled by this view
	 * This is set during {@link #setAction}.
	 * @property
	 * @type Zarafa.common.rules.data.ActionFlags
	 */
	actionFlag : undefined,

	/**
	 * The action property which was configured during
	 * {@link #setAction}.
	 * @property
	 * @type Object
	 */
	action : undefined,

	/**
	 * True if the action was modified by the user, if this is false,
	 * then {@link #getAction} will return {@link #action} instead
	 * of returning a new object.
	 * @property
	 * @type Boolean
	 */
	isModified : false,

	/**
	 * True if the action/condition is complete and valid,
	 * False will denote that action/condition is invalid or incomplete
	 * if this is true, then {@link #getCondition} will return {@link #condition} instead
	 * of returning a new object and {@link #getAction} will return {@link #action}
	 * instead of returning a new Object.
	 * @property
	 * @type Boolean
	 */
	isValid : true,

	/**
	 * @constructor
	 * @param {Object} config configuration object.
	 */
	constructor : function(config)
	{
		config = config || {};

		Ext.applyIf(config,{
			xtype: 'zarafa.folderselectionlink',
			border : false,
			autoScroll :true,
			anchor : '100%',
			tpl : new Ext.XTemplate(
				'<div class="zarafa-folder-link">' +
					'<tpl if="!Ext.isEmpty(values.display_name)">' +
						'&quot;{display_name:htmlEncode}&quot;' +
					'</tpl>' +
					'<tpl if="Ext.isEmpty(values.display_name)">' +
						'&quot;' + _('Unnamed folder') + '&quot;' + 
					'</tpl>' +
				'</div>',
				{
					compiled : true
				}
			)
		});

		Zarafa.common.rules.dialogs.FolderSelectionLink.superclass.constructor.call(this, config);
	},

	/**
	 * This function is called after the component has been rendered.
	 * This will register the {@link #onActivate} and {@link #onClick} event handlers.
	 * @private
	 */
	afterRender : function()
	{
		Zarafa.common.rules.dialogs.FolderSelectionLink.superclass.afterRender.apply(this, arguments);

		this.mon(this.getActionEl(), 'click', this.onClick, this);
	},

	/**
	 * Called when user clicks on a {@link Zarafa.common.rules.dialogs.FolderSelectionLink}
	 * It opens hierarchy folder selection dialog
	 * @param {Ext.DataView} dataView Reference to this object
	 * @param {Number} index The index of the target node
	 * @param {HTMLElement} node The target node
	 * @param {Ext.EventObject} evt The mouse event
 	 * @protected
	 */
	onClick : function(dataView, index, node, evt)
	{
		Zarafa.hierarchy.Actions.openFolderSelectionContent({
			folder : this.folder,
			callback : function(folder) {
				this.folder = folder;
				this.isModified = true;
				this.update(folder);
			},
			scope : this,
			modal : true
		})
	},

	/**
	 * Apply an action onto the DataView, this will parse the action and show
	 * the contents in a user-friendly way to the user.
	 * @param {Zarafa.common.rules.data.ActionFlags} actionFlag The action type
	 * which identifies the exact type of the action.
	 * @param {Object} action The action to apply
	 */
	setAction : function(actionFlag, action)
	{
		this.folder = undefined;
		this.isValid = false;

		if (action) {
			var store = container.getHierarchyStore().getById(action.storeentryid);
			if (store) {
				this.folder = store.getSubStore('folders').getById(action.folderentryid);
				if(Ext.isDefined(this.folder)) {
					this.isValid = true;
				}
			}
		}

		this.actionFlag = actionFlag;
		this.action = action;
		this.isModified = !Ext.isDefined(action);
		this.update(this.folder);
	},

	/**
	 * Obtain the action as configured by the user
	 * @return {Object} The action
	 */
	getAction : function()
	{
		if (this.isModified !== true && this.isValid === true) {
			return this.action;
		}

		var action = {};

		// No folder selected, this means
		// we can't create an action.
		if (!this.folder) {
			return false;
		}

		// Set the folder properties.
		action.folderentryid = this.folder.get('entryid');
		action.storeentryid = this.folder.get('store_entryid');

		// Fill in the additional properties required for the action.
		switch (this.actionFlag) {
			case Zarafa.common.rules.data.ActionFlags.MOVE:
				action.action = Zarafa.core.mapi.RuleActions.OP_MOVE;
				action.flags = 0;
				action.flavor = 0;
				break;
			case Zarafa.common.rules.data.ActionFlags.COPY:
				action.action = Zarafa.core.mapi.RuleActions.OP_COPY;
				action.flags = 0;
				action.flavor = 0;
				break;
			default:
				// invalid actionFlag
				return false;
		}

		return action;
	},

	/**
	 * Update the contents of this dataview, this will apply the {@link #tpl} for
	 * the {@link #folder}.
	 * @param {Zarafa.hierarchy.data.MAPIFolderRecord} folder The folder to show
	 */
	update : function(folder)
	{
 		var data = folder ? folder.data : { display_name : this.emptyText };
		Zarafa.common.rules.dialogs.FolderSelectionLink.superclass.update.call(this, this.tpl.apply(data));
	}
});

Ext.reg('zarafa.folderselectionlink', Zarafa.common.rules.dialogs.FolderSelectionLink);
Ext.namespace('Zarafa.common.rules.dialogs');

/**
 * @class Zarafa.common.rules.dialogs.ImportanceLink
 * @extends Ext.Container
 * @xtype zarafa.importancelink
 *
 * Condition component for the {@link Zarafa.common.rules.data.ConditionFlags#IMPORTANCE IMPORTANCE}
 * condition. This will allow the user to select the preferred importance and can generate a proper
 * condition for it.
 */
Zarafa.common.rules.dialogs.ImportanceLink = Ext.extend(Ext.Container, {

	/**
	 * The Condition type which is handled by this view
	 * This is set during {@link #setCondition}.
	 * @property
	 * @type Zarafa.common.rules.data.ConditionFlags
	 */
	conditionFlag : undefined,

	/**
	 * The condition property which was configured during
	 * {@link #setCondition}.
	 * @property
	 * @type Object
	 */
	condition : undefined,

	/**
	 * True if the condition was modified by the user, if this is false,
	 * then {@link #getCondition} will return {@link #condition} instead
	 * of returning a new object.
	 * @property
	 * @type Boolean
	 */
	isModified : false,

	/**
	 * @constructor
	 * @param {Object} config Configuration object
	 */
	constructor : function(config)
	{
		config = config || {};

		Ext.applyIf(config, {
			items : [{
				xtype : 'combo',
				ref : 'importanceCombo',
				width : 100,
				store : {
					xtype : 'jsonstore',
					fields : [ 'name', 'value' ],
					data : Zarafa.common.data.ImportanceFlags.flags
				},
				mode : 'local',
				triggerAction : 'all',
				displayField : 'name',
				valueField : 'value',
				lazyInit : false,
				forceSelection : true,
				editable : false,
				listeners : {
					select : function() { this.isModified = true; },
					scope : this
				}
			}]
		});

		Zarafa.common.rules.dialogs.ImportanceLink.superclass.constructor.call(this, config);
	},

	/**
	 * Apply an action onto the DataView, this will parse the condition and show
	 * the contents in a user-friendly way to the user.
	 * @param {Zarafa.common.rules.data.ConditionFlags} conditionFlag The condition type
	 * which identifies the exact type of the condition.
	 * @param {Object} condition The condition to apply
	 */
	setCondition : function(conditionFlag, condition)
	{
		var importance = Zarafa.core.mapi.Importance['NORMAL'];

		if (condition) {
			importance = condition[1][Zarafa.core.mapi.Restrictions.VALUE]['PR_IMPORTANCE'];
		}

		this.conditionFlag = conditionFlag;
		this.condition = condition;
		this.isModified = !Ext.isDefined(condition);

		this.importanceCombo.setValue(importance);
	},

	/**
	 * Obtain the condition as configured by the user
	 * @return {Object} The condition
	 */
	getCondition : function()
	{
		if (this.isModified !== true) {
			return this.condition;
		}

		var RestrictionFactory = Zarafa.core.data.RestrictionFactory;
		var Restrictions = Zarafa.core.mapi.Restrictions;
		var value = this.importanceCombo.getValue();

		return RestrictionFactory.dataResProperty('PR_IMPORTANCE', Restrictions.RELOP_EQ, value);
	}
});

Ext.reg('zarafa.importancelink', Zarafa.common.rules.dialogs.ImportanceLink);
Ext.namespace('Zarafa.common.rules.dialogs');

/**
 * @class Zarafa.common.rules.dialogs.RulesActionsContainer
 * @extends Ext.Container
 * @xtype zarafa.rulesactionscontainer
 *
 * The container in which all actions can be edited. This container
 * can be expanded to include multiple actions, and is able to parse
 * the rules_actions property of a {@link Zarafa.common.rules.data.RulesRecord rule}.
 */
Zarafa.common.rules.dialogs.RulesActionsContainer = Ext.extend(Ext.Container, {
	/**
	 * The current number of action boxes which are present in the container.
	 * This number is changed by {@link #addActionBox} and {@link #removeActionBox}.
	 * @property
	 * @type Number
	 * @private
	 */
	actionCount : 0,

	/**
	 * @constructor
	 * @param {Object} config Configuration object
	 */
	constructor : function(config)
	{
		config = config || {};

		config.plugins = Ext.value(config.plugins, []);
		config.plugins.push('zarafa.recordcomponentupdaterplugin');

		Ext.applyIf(config, {
			layout : 'form',
			autoHeight: true,
			items : [{
				xtype : 'zarafa.compositefield',
				hideLabel : true,
				items : [{
					xtype : 'button',
					ref : '../addActionBtn',
					text : _('Add action'),
					handler : this.addActionBox,
					scope : this
				},{
					xtype : 'button',
					ref : '../removeActionBtn',
					text : _('Remove action'),
					handler : this.removeActionBox,
					scope : this
				}]
			}]
		});

		Zarafa.common.rules.dialogs.RulesActionsContainer.superclass.constructor.call(this, config);
	},

	/**
	 * Generic function to create containers in which an action is represented. This consists of
	 * 2 components, the first one is the combobox in which the action type is selected, and the
	 * second in which special option for the given action can be configured.
	 * @param {Number} The index of the action which is created
	 * @return {Object} config object to create a {@link Ext.Container}.
	 * @private
	 */
	createActionBox : function(index)
	{
		var id =  'rule-action-' + String(index);
		var profileStore = {
			xtype : 'jsonstore',
			fields : [
				{ name : 'name' },
				{ name : 'value', type : 'int' }
			],
			data : Zarafa.common.rules.data.ActionProfiles
		};

		return {
			xtype : 'container',
			id : id,
			flex : 1,
			height : 25,
			layout : {
				type : 'hbox',
				align : 'stretch',
				defaultMargins : '0 5 0 0'
			},
			items : [{
				xtype : 'combo',
				width : 300,
				store : profileStore,
				mode : 'local',
				triggerAction : 'all',
				displayField : 'name',
				valueField : 'value',
				lazyInit : false,
				forceSelection : true,
				editable : false,
				value : _('Select one...'),
				listeners : {
					'select' : this.onActionComboSelect,
					'scope' : this
				}
			}, {
				xtype : 'container',
				flex : 1,
				layout : 'card',
				activeItem : 0,
				items : this.createActionContentPanels(id)
			}]
		};
	},

	/**
	 * Create a set of ContentPanels which are used to configure the various action type.
	 * The array which is returned contains should be applied on a {@link Ext.Container} with
	 * a {@link Ext.layout.CardLayout CardLayout} to ensure only one container is visible
	 * at a time.
	 * In each container the user is able to set various configuration options for the
	 * action type as selected in the combobox.
	 * @param {String} baseId The baseId which is used to create the id for the individual containers.
	 * @return {Array} Array of config objects to create a {@link Ext.Container}.
	 * @private
	 */
	createActionContentPanels : function(baseId)
	{
		return [{
			xtype : 'container',
			id : baseId + '-empty'
		},{
			xtype : 'zarafa.folderselectionlink',
			id : baseId + '-folder'
		},{
			xtype : 'zarafa.deletelink',
			id : baseId + '-delete'
		},{
			xtype : 'zarafa.userselectionlink',
			id : baseId + '-to'
		}];
	},

	/**
	 * Function that can be used to add more actions in a rule.
	 * @return {Ext.Container} The Action Box which was inserted
	 * @private
	 */
	addActionBox : function()
	{
		this.actionCount++;

		var container = this.createActionBox(this.actionCount);
		container = this.insert(this.items.getCount() - 1, container);

		// Toggle the removeActionBtn
		this.removeActionBtn.setDisabled(this.actionCount <= 1);

		this.doLayout();

		return container;
	},

	/**
	 * Function that can be used to remove an action from a rule.
	 * This will always remove the last action.
	 * @private
	 */
	removeActionBox : function()
	{
		if (this.actionCount > 1) {
			// Don't remove the last item, as that is the container
			// to add and remove actions.
			this.remove(this.get(this.items.getCount() - 2));
			this.actionCount--;

			// Toggle the removeActionBtn
			this.removeActionBtn.setDisabled(this.actionCount <= 1);

			this.doLayout();
		}
	},

	/**
	 * {@link #addActionBox add} or {@link #removeActionBox remove}
	 * Action Boxes for a rule, until the {@link #actionCount} reaches
	 * the given count.
	 * @param {Number} count The desired number of action boxes
	 * @private
	 */
	setActionBoxCount : function(count)
	{
		while (count < this.actionCount) {
			this.removeActionBox();
		}
		while (count > this.actionCount) {
			this.addActionBox();
		}
	},

	/**
	 * Updates the panel by loading data from the record into the form panel.
	 * @param {Zarafa.common.rules.data.RulesRecord} record The record update the panel with.
	 * @param {Boolean} contentReset force the component to perform a full update of the data.
	 */
	update : function(record, contentReset)
	{
		this.record = record;

		if (contentReset || record.isModifiedSinceLastUpdate('rule_actions')) {
			var actions = record.get('rule_actions');
			if (Ext.isEmpty(actions)) {
				this.setActionBoxCount(1);
				return;
			}

			// Force actions to be an array
			actions = [].concat(actions);

			// We have to ensure that there are sufficient action fields
			// present in the container. When the rule doesn't have any
			// actions specified, we will create an empty action
			var count = Math.max(1, actions.length);
			this.setActionBoxCount(count);

			for (var i = 0, len = actions.length; i < len; i++) {
				// Apply the action to the corresponding container
				if (actions[i]) {
					this.applyAction(this.get(i), actions[i]);
				}
			}
		}
	},

	/**
	 * Update the given {@link Zarafa.core.data.IPMRecord record} with
	 * the values from this {@link Ext.Panel panel}.
	 * @param {Zarafa.core.data.IPMRecord} record The record to update
	 */
	updateRecord : function(record)
	{
		var actions = [];
		var actionsValid = true;

		for (var i = 0; i < this.actionCount; i++) {
			var panel = this.get(i);
			var combo = panel.get(0);
			var activeItem = panel.get(1).layout.activeItem;
			var action = null;

			if (Ext.isFunction(activeItem.getAction)) {
				action = activeItem.getAction();
			}

			// If no valid action was found, then
			// we have a problem and we can't save
			// the action. Break out of the loop
			// and invalidate the rule_actions property.
			if (!action) {
				panel.get(0).markInvalid();
				actionsValid = false;
			}

			actions.push(action);
		}

		record.set('rule_actions', actions);
		record.setActionsValid(actionsValid);
	},

	/**
	 * Load an Action from a {@Link Zarafa.common.rules.data.RulesRecord} and apply it
	 * onto the {@link Ext.Container} which was created by {@link #addActionBox}. 
	 * @param {Ext.Container} panel The container on which the action will be loaded
	 * @param {Object} action The action which should be loaded
	 * @private
	 */
	applyAction : function(panel, action)
	{
		var actionFlag = this.getActionFlagFromAction(action);
		var combo = panel.get(0);
		var content = panel.get(1);
		var store = combo.store;

		// Apply the correct value to the combobox.
		var index = store.findExact(combo.valueField, actionFlag);
		if (index >= 0) {
			var record = store.getAt(index);
			combo.setValue(actionFlag);
			this.onActionComboSelect(combo, record, index);
		} else {
			actionFlag = Zarafa.common.rules.data.ActionFlags.UNKNOWN;
			combo.setValue(_('Unknown action'));
			combo.markInvalid(_('This action for the current rule is unknown'));
		}

		// Fill the content with the data from the action
		var layout = content.getLayout();
		switch (actionFlag) {
			case Zarafa.common.rules.data.ActionFlags.UNKNOWN:
			default:
				break;
			case Zarafa.common.rules.data.ActionFlags.MOVE:
			case Zarafa.common.rules.data.ActionFlags.COPY:
			case Zarafa.common.rules.data.ActionFlags.DELETE:
			case Zarafa.common.rules.data.ActionFlags.REDIRECT:
			case Zarafa.common.rules.data.ActionFlags.FORWARD:
			case Zarafa.common.rules.data.ActionFlags.FORWARD_ATTACH:
				layout.activeItem.setAction(actionFlag, action);
				break;
		}
	},

	/**
	 * Read a Action object as located in the {@link Zarafa.common.rules.data.RulesRecord Rule}
	 * and convert it to the corresponding ActionFlag which properly represents the action
	 * to be taken.
	 * @param {Object} action The action which should be converted to a Action Flag
	 * @return {Zarafa.common.rules.data.ActionFlags} The Action Flag
	 * @private
	 */
	getActionFlagFromAction : function(action)
	{
		switch (action.action) {
			case Zarafa.core.mapi.RuleActions.OP_MOVE:
				// The MOVE action can be used for either
				// DELETE or MOVE action. The decision for this
				// depends on if the selected folder is set the
				// the "Deleted Items" folder, and if the rule_state
				// property has the ST_EXIT_LEVEL flag.
				var RulesStates = Zarafa.core.mapi.RuleStates;
				if (this.record.get('rule_state') & RulesStates.ST_EXIT_LEVEL === RulesStates.ST_EXIT_LEVEL) {
					var deletedItems = container.getHierarchyStore().getDefaultFolder('wastebasket');
					if (deletedItems && Zarafa.core.EntryId.compareEntryIds(deletedItems.get('entryid'), action.folderentryid)) {
						return Zarafa.common.rules.data.ActionFlags.DELETE;
					}
				}
				return Zarafa.common.rules.data.ActionFlags.MOVE;
			case Zarafa.core.mapi.RuleActions.OP_COPY:
				// Normal copy, nothing fancy.
				return Zarafa.common.rules.data.ActionFlags.COPY;
			case Zarafa.core.mapi.RuleActions.OP_FORWARD:
				// The exact forward action depends on the 'flavor' property.
				var FlavorFlags = Zarafa.core.mapi.FlavorFlags;
				switch (action.flavor) {
					case 0: // Forward
						return Zarafa.common.rules.data.ActionFlags.FORWARD;
					case FlavorFlags.FWD_PRESERVE_SENDER | FlavorFlags.FWD_DO_NOT_MUNGE_MSG:
						return Zarafa.common.rules.data.ActionFlags.REDIRECT;
					case FlavorFlags.FWD_AS_ATTACHMENT:
						return Zarafa.common.rules.data.ActionFlags.FORWARD_ATTACH;
					default:
						return Zarafa.common.rules.data.ActionFlags.UNKNOWN;
				}
			default:
				// Any other RuleAction is not supported
				return Zarafa.common.rules.data.ActionFlags.UNKNOWN;
		}

	},

	/**
	 * The event handler for the {@link Ext.form.ComboBox#select} event for the combobox for
	 * a particular action. This will update the corresponding content panel to show the correct
	 * content type.
	 * @param {Ext.form.ComboBox} combo The combobox which fired the event
	 * @param {Ext.data.Record} record The record which was selected from the combobox
	 * @param {Number} index The selected index from the combobox list
	 * @private
	 */
	onActionComboSelect : function(combo, record, index)
	{
		var panel = combo.ownerCt;
		var content = panel.get(1);

		var layout = content.getLayout();
		var value = record.get(combo.valueField);

		switch (value) {
			case Zarafa.common.rules.data.ActionFlags.UNKNOWN:
			default:
				layout.setActiveItem(panel.id + '-empty');
				break;
			case Zarafa.common.rules.data.ActionFlags.MOVE:
			case Zarafa.common.rules.data.ActionFlags.COPY:
				layout.setActiveItem(panel.id + '-folder');
				layout.activeItem.setAction(value);
				break;
			case Zarafa.common.rules.data.ActionFlags.DELETE:
				layout.setActiveItem(panel.id + '-delete');
				layout.activeItem.setAction(value);
				// For the DELETE action, the rule_state
				// must contain the ST_EXIT_LEVEL flag.
				this.record.set('rule_state', this.record.get('rule_state') | Zarafa.core.mapi.RuleStates.ST_EXIT_LEVEL);
				break;
			case Zarafa.common.rules.data.ActionFlags.REDIRECT:
			case Zarafa.common.rules.data.ActionFlags.FORWARD:
			case Zarafa.common.rules.data.ActionFlags.FORWARD_ATTACH:
				layout.setActiveItem(panel.id + '-to');
				layout.activeItem.setAction(value);
				break;
		}
	}
});

Ext.reg('zarafa.rulesactionscontainer', Zarafa.common.rules.dialogs.RulesActionsContainer);
Ext.namespace('Zarafa.common.rules.dialogs');

/**
 * @class Zarafa.common.rules.dialogs.RulesConditionContainer
 * @extends Ext.Container
 * @xtype zarafa.rulesconditioncontainer
 *
 * The container in which all conditions can be edited. This container
 * can be expanded to include multiple conditions, and is able to parse
 * the rules_condition property of a {@link Zarafa.common.rules.data.RulesRecord rule}.
 */
Zarafa.common.rules.dialogs.RulesConditionContainer = Ext.extend(Ext.Container, {
	/**
	 * The current number of condition boxes which are present in the container.
	 * This number is changed by {@link #addConditionBox} and {@link #removeConditionBox}.
	 * @property
	 * @type Number
	 * @private
	 */
	conditionCount : 0,

	/**
	 * @constructor
	 * @param {Object} config Configuration object
	 */
	constructor : function(config)
	{
		config = config || {};

		config.plugins = Ext.value(config.plugins, []);
		config.plugins.push('zarafa.recordcomponentupdaterplugin');

		Ext.applyIf(config, {
			layout : 'form',
			autoHeight: true,
			items : [{
				xtype : 'zarafa.compositefield',
				hideLabel : true,
				items : [{
					xtype : 'button',
					ref : '../addConditionBtn',
					text : _('Add condition'),
					handler : this.addConditionBox,
					scope : this
				},{
					xtype : 'button',
					ref : '../removeConditionBtn',
					text : _('Remove condition'),
					handler : this.removeConditionBox,
					scope : this
				}]
			}]
		});

		Zarafa.common.rules.dialogs.RulesConditionContainer.superclass.constructor.call(this, config);
	},

	/**
	 * Generic function to create containers in which a condition is represented. This consists of
	 * 2 components, the first one is the combobox in which the condition type is selected, and the
	 * second in which special option for the given condition can be configured.
	 * @param {Number} The index of the condition which is created
	 * @return {Object} config object to create a {@link Ext.Container}.
	 * @private
	 */
	createConditionBox : function(index)
	{
		var id =  'rule-condition-' + String(index);
		var profileStore = {
			xtype : 'jsonstore',
			fields : [
				{ name : 'name' },
				{ name : 'value', type : 'int' }
			],
			data : Zarafa.common.rules.data.ConditionProfiles
		};

		return {
			xtype : 'container',
			id : id,
			flex : 1,
			height : 25,
			layout : {
				type : 'hbox',
				align : 'stretch',
				defaultMargins : '0 5 0 0'
			},
			items : [{
				xtype : 'combo',
				width : 300,
				store : profileStore,
				mode : 'local',
				triggerAction : 'all',
				displayField : 'name',
				valueField : 'value',
				lazyInit : false,
				forceSelection : true,
				editable : false,
				value : _('Select one...'),
				listeners : {
					'select' : this.onConditionComboSelect,
					'scope' : this
				}
			}, {
				xtype : 'container',
				flex : 1,
				layout : 'card',
				activeItem : 0,
				items : this.createConditionContentPanels(id)
			}]
		};
	},

	/**
	 * Create a set of ContentPanels which are used to configure the various condition type.
	 * The array which is returned contains should be applied on a {@link Ext.Container} with
	 * a {@link Ext.layout.CardLayout CardLayout} to ensure only one container is visible
	 * at a time.
	 * In each container the user is able to set various configuration options for the
	 * condition type as selected in the combobox.
	 * @param {String} baseId The baseId which is used to create the id for the individual containers.
	 * @return {Array} Array of config objects to create a {@link Ext.Container}.
	 * @private
	 */
	createConditionContentPanels : function(baseId)
	{
		return [{
			xtype : 'container',
			id : baseId + '-empty'
		},{
			xtype : 'zarafa.userselectionlink',
			id : baseId + '-from'
		},{
			xtype : 'zarafa.wordselectionlink',
			id : baseId + '-senderwords'
		},{
			xtype : 'zarafa.wordselectionlink',
			id : baseId + '-words'
		},{
			xtype : 'zarafa.wordselectionlink',
			id : baseId + '-bodywords'
		},{
			xtype : 'zarafa.importancelink',
			id : baseId + '-importance'
		},{
			xtype : 'zarafa.userselectionlink',
			id : baseId + '-to'
		},{
			xtype : 'zarafa.senttomelink',
			id : baseId + '-to-me'
		}];
	},

	/**
	 * Function that can be used to add more conditions in a rule.
	 * @return {Ext.Container} The Condition Box which was inserted
	 * @private
	 */
	addConditionBox : function()
	{
		this.conditionCount++;

		var container = this.createConditionBox(this.conditionCount);
		container = this.insert(this.items.getCount() - 1, container);

		// Toggle the removeConditionBtn
		this.removeConditionBtn.setDisabled(this.conditionCount <= 1);

		this.doLayout();

		return container;
	},

	/**
	 * Function that can be used to remove a condition from a rule.
	 * This will always remove the last condition.
	 * @private
	 */
	removeConditionBox : function()
	{
		if (this.conditionCount > 1) {
			// Don't remove the last item, as that is the container
			// to add and remove conditions.
			this.remove(this.get(this.items.getCount() - 2));
			this.conditionCount--;

			// Toggle the removeConditionBtn
			this.removeConditionBtn.setDisabled(this.conditionCount <= 1);

			this.doLayout();
		}
	},

	/**
	 * {@link #addConditionBox add} or {@link #removeConditionBox remove}
	 * Condition Boxes for a rule, until the {@link #conditionCount} reaches
	 * the given count.
	 * @param {Number} count The desired number of condition boxes
	 * @private
	 */
	setConditionBoxCount : function(count)
	{
		while (count < this.conditionCount) {
			this.removeConditionBox();
		}
		while (count > this.conditionCount) {
			this.addConditionBox();
		}
	},

	/**
	 * Updates the panel by loading data from the record into the form panel.
	 * @param {Zarafa.common.rules.data.RulesRecord} record The record update the panel with.
	 * @param {Boolean} contentReset force the component to perform a full update of the data.
	 */
	update : function(record, contentReset)
	{
		this.record = record;

		if (contentReset || record.isModifiedSinceLastUpdate('rule_condition')) {
			var conditions = record.get('rule_condition');
			if (Ext.isEmpty(conditions)) {
				this.setConditionBoxCount(1);
				return;
			}

			conditions = this.getConditionsArray(conditions);

			// We have to ensure that there are sufficient condition fields
			// present in the container. When the rule doesn't have any
			// condition specified, we will create an empty condition
			var count = Math.max(1, conditions.length);
			this.setConditionBoxCount(count);

			for (var i = 0, len = conditions.length; i < len; i++) {
				// Apply the action to the corresponding container
				if (conditions[i]) {
					this.applyCondition(this.get(i), conditions[i]);
				}
			}
		}
	},

	/**
	 * Update the given {@link Zarafa.core.data.IPMRecord record} with
	 * the values from this {@link Ext.Panel panel}.
	 * @param {Zarafa.core.data.IPMRecord} record The record to update
	 */
	updateRecord : function(record)
	{
		var conditions = [];
		var conditionsValid = true;

		for (var i = 0; i < this.conditionCount; i++) {
			var panel = this.get(i);
			var combo = panel.get(0);
			var activeItem = panel.get(1).layout.activeItem;
			var condition = null;

			if (Ext.isFunction(activeItem.getCondition)) {
				condition = activeItem.getCondition();  
			}

			// If no valid condition was found, then
			// we have a problem and we can't save
			// the action. Break out of the loop
			// and invalidate the rule_condition property.
			if (!condition) {
				combo.markInvalid();
				conditionsValid = false;
			}

			conditions.push(condition);
		}

		// Check if we need to create a AND restriction
		if (conditions) {
			if (conditions.length > 1) {
				conditions = Zarafa.core.data.RestrictionFactory.createResAnd(conditions);
			} else {
				conditions = conditions[0];
			}
		}

		record.set('rule_condition', conditions);
		record.setConditionsValid(conditionsValid);
	},

	/**
	 * Convert the conditions object as stored in a {@link Zarafa.common.rules.data.RulesRecord rule}
	 * and convert it to an array of individual conditions. Each element in the returned array represents
	 * a single condition.
	 * @param {Object} conditions The condition which must be converted into an array
	 * @return {Array} The array of conditions
	 * @private
	 */
	getConditionsArray : function(conditions)
	{
		// Check the conditions, if the RES property indicates a AND
		// restriction we have to check the contents, as we need to determine
		// if this represents a single condition or a list of conditions.
		if (conditions[0] === Zarafa.core.mapi.Restrictions.RES_AND) {
			var single = false;

			// Check if this AND/OR restriction represents a single
			// condition or not.
			for (var i = 0, len = conditions[1].length; i < len; i++) {
				var sub = conditions[1][i];
				if (sub && sub[0] === Zarafa.core.mapi.Restrictions.RES_PROPERTY &&
					sub[1][Zarafa.core.mapi.Restrictions.ULPROPTAG] === 'PR_MESSAGE_TO_ME') {
					single = true;
					break;
				}
			}

			// Now return the conditions
			if (single) {
				conditions = [ conditions ];
			} else {
				conditions = conditions[1];
			}
		} else {
			// Single condition, just convert it to an array
			conditions = [ conditions ];
		}

		return conditions;
	},

	/**
	 * Load a Condition from a {@Link Zarafa.common.rules.data.RulesRecord} and apply it
	 * onto the {@link Ext.Container} which was created by {@link #addConditionBox}. 
	 * @param {Ext.Container} panel The container on which the condition will be loaded
	 * @param {Object} condition The condition which should be loaded
	 * @private
	 */
	applyCondition : function(panel, condition)
	{
		var conditionFlag = this.getConditionFlagFromCondition(condition);
		var combo = panel.get(0);
		var content = panel.get(1);
		var store = combo.store;

		// Apply the correct value to the combobox.
		var index = store.findExact(combo.valueField, conditionFlag);
		if (index >= 0) {
			var record = store.getAt(index);
			combo.setValue(conditionFlag);
			this.onConditionComboSelect(combo, record, index);
		} else {
			conditionFlag = Zarafa.common.rules.data.ConditionFlags.UNKNOWN;
			combo.setValue(_('Unknown condition'));
			combo.markInvalid(_('This condition for the current rule is unknown'));
		}

		// Fill the content with the data from the condition
		var layout = content.getLayout();
		switch (conditionFlag) {
			case Zarafa.common.rules.data.ConditionFlags.UNKNOWN:
			default:
				break;
			case Zarafa.common.rules.data.ConditionFlags.SUBJECT_WORDS:
			case Zarafa.common.rules.data.ConditionFlags.BODY_WORDS:
			case Zarafa.common.rules.data.ConditionFlags.IMPORTANCE:
			case Zarafa.common.rules.data.ConditionFlags.RECEIVED_FROM:
			case Zarafa.common.rules.data.ConditionFlags.SENDER_WORDS:
			case Zarafa.common.rules.data.ConditionFlags.SENT_TO:
			case Zarafa.common.rules.data.ConditionFlags.SENT_TO_ME_ONLY:
				layout.activeItem.setCondition(conditionFlag, condition);
				break;
		}
	},

	/**
	 * Read a Condition object as located in the {@link Zarafa.common.rules.data.RulesRecord Rule}
	 * and convert it to the corresponding ConditionFlag which properly represents the condition.
	 * @param {Object} condition The condition which should be converted to a Condition Flag
	 * @return {Zarafa.common.rules.data.ConditionFlags} The Condition Flag
	 * @private
	 */
	getConditionFlagFromCondition : function(condition)
	{
		var Restrictions = Zarafa.core.mapi.Restrictions;

		switch (condition[0]) {
			case Restrictions.RES_COMMENT:
				switch (condition[1][Restrictions.RESTRICTION][1][Restrictions.ULPROPTAG]) {
					case 'PR_SENDER_SEARCH_KEY':
						return Zarafa.common.rules.data.ConditionFlags.RECEIVED_FROM;
					default:
						return Zarafa.common.rules.data.ConditionFlags.UNKNOWN;
				}
				break;
			case Restrictions.RES_CONTENT:
			case Restrictions.RES_PROPERTY:
			case Restrictions.RES_SUBRESTRICTION:
				switch (condition[1][Restrictions.ULPROPTAG]) {
					case 'PR_BODY':
						return Zarafa.common.rules.data.ConditionFlags.BODY_WORDS;
					case 'PR_SUBJECT':
						return Zarafa.common.rules.data.ConditionFlags.SUBJECT_WORDS;
					case 'PR_IMPORTANCE':
						return Zarafa.common.rules.data.ConditionFlags.IMPORTANCE;
					case 'PR_MESSAGE_RECIPIENTS':
						return Zarafa.common.rules.data.ConditionFlags.SENT_TO;
					case 'PR_SENDER_SEARCH_KEY':
						return Zarafa.common.rules.data.ConditionFlags.SENDER_WORDS;
					default:
						return Zarafa.common.rules.data.ConditionFlags.UNKNOWN;
				}
			case Restrictions.RES_AND:
				for (var i = 0, len = condition[1].length; i < len; i++) {
					var sub = condition[1][i];

					// Check if the RES_AND contains the restriction for PR_MESSAGE_TO_ME,
					// this indicates that this restriction is the SENT_TO_ME_ONLY condition
					if (sub[0] === Restrictions.RES_PROPERTY &&
					    sub[1][Restrictions.ULPROPTAG] === 'PR_MESSAGE_TO_ME') {
						return Zarafa.common.rules.data.ConditionFlags.SENT_TO_ME_ONLY;
					}
				}
				return Zarafa.common.rules.data.ConditionFlags.UNKNOWN;
			case Restrictions.RES_OR:
				for (var i = 0, len = condition[1].length; i < len; i++) {
					var sub = condition[1][i];
					var type = this.getConditionFlagFromCondition(sub);
					if (type !== Zarafa.common.rules.data.ConditionFlags.UNKNOWN) {
						return type;
					}
				}
				return Zarafa.common.rules.data.ConditionFlags.UNKNOWN;
			default:
				return Zarafa.common.rules.data.ConditionFlags.UNKNOWN;
		}
	},

	/**
	 * The event handler for the {@link Ext.form.ComboBox#select} event for the combobox for
	 * a particular action. This will update the corresponding content panel to show the correct
	 * content type.
	 * @param {Ext.form.ComboBox} combo The combobox which fired the event
	 * @param {Ext.data.Record} record The record which was selected from the combobox
	 * @param {Number} index The selected index from the combobox list
	 * @private
	 */
	onConditionComboSelect : function(combo, record, index)
	{
		var panel = combo.ownerCt;
		var content = panel.get(1);

		var layout = content.getLayout();
		var value = record.get(combo.valueField);

		switch (value) {
			case Zarafa.common.rules.data.ConditionFlags.UNKNOWN:
			default:
				layout.setActiveItem(panel.id + '-empty');
				break;
			case Zarafa.common.rules.data.ConditionFlags.RECEIVED_FROM:
				layout.setActiveItem(panel.id + '-from');
				layout.activeItem.setCondition(value);
				break;
			case Zarafa.common.rules.data.ConditionFlags.SENDER_WORDS:
				layout.setActiveItem(panel.id + '-senderwords');
				layout.activeItem.setCondition(value);
				break;
			case Zarafa.common.rules.data.ConditionFlags.SUBJECT_WORDS:
				layout.setActiveItem(panel.id + '-words');
				layout.activeItem.setCondition(value);
				break;
			case Zarafa.common.rules.data.ConditionFlags.BODY_WORDS:
				layout.setActiveItem(panel.id + '-bodywords');
				layout.activeItem.setCondition(value);
				break;
			case Zarafa.common.rules.data.ConditionFlags.IMPORTANCE:
				layout.setActiveItem(panel.id + '-importance');
				layout.activeItem.setCondition(value);
				break;
			case Zarafa.common.rules.data.ConditionFlags.SENT_TO:
				layout.setActiveItem(panel.id + '-to');
				layout.activeItem.setCondition(value);
				break;
			case Zarafa.common.rules.data.ConditionFlags.SENT_TO_ME_ONLY:
				layout.setActiveItem(panel.id + '-to-me');
				layout.activeItem.setCondition(value);
				break;
		}
	}
});

Ext.reg('zarafa.rulesconditioncontainer', Zarafa.common.rules.dialogs.RulesConditionContainer);
Ext.namespace('Zarafa.common.rules.dialogs');

/**
 * @class Zarafa.common.rules.dialogs.RulesEditPanel
 * @extends Ext.form.FormPanel
 * @xtype zarafa.ruleeditpanel
 *
 * Will generate UI for {@link Zarafa.common.rules.dialogs.RulesEditContentPanel RulesEditContentPanel}.
 */
Zarafa.common.rules.dialogs.RulesEditPanel = Ext.extend(Ext.form.FormPanel, {
	/**
	 * @constructor
	 * @param config Configuration structure
	 */
	constructor : function(config)
	{
		config = config || {};

		config.plugins = Ext.value(config.plugins, []);
		config.plugins.push('zarafa.recordcomponentupdaterplugin');

		Ext.applyIf(config, {
			// Override from Ext.Component
			xtype : 'zarafa.ruleseditpanel',
			layout : 'anchor',
			autoScroll : true,
			bodyStyle : 'padding: 5px;',
			items : this.createPanelItems()
		});

		Zarafa.common.rules.dialogs.RulesEditPanel.superclass.constructor.call(this, config);
	},

	/**
	 * Function will create panel items for {@link Zarafa.common.rules.dialogs.RulesEditPanel RulesEditPanel}
	 * @return {Array} array of items that should be added to panel.
	 * @private
	 */
	createPanelItems : function()
	{
		return [{
			xtype : 'displayfield',
			value : _('Rule name'),
			height : 20
		}, {
			xtype : 'textfield',
			name : 'rule_name',
			width : 250,
			listeners : {
				change : this.onChange,
				scope : this
			}
		}, {
			xtype : 'spacer',
			height : 15
		}, {
			xtype : 'displayfield',
			value : _('When the message') + '...',
			height : 20
		}, {
			xtype : 'zarafa.rulesconditioncontainer',
			anchor : '100%'
		}, {
			xtype : 'spacer',
			height : 15
		}, {
			xtype : 'displayfield',
			value : _('Do the following') + '...',
			height : 20
		}, {
			xtype : 'zarafa.rulesactionscontainer',
			anchor : '100%'
		}, {
			xtype : 'spacer',
			height : 15
		}, {
			xtype : 'checkbox',
			ref : 'stopProcessingCheckbox',
			boxLabel : _('Stop processing more rules'),
			handler : this.onToggleStopProcessing,
			scope : this
		}];
	},

	/**
	 * Save the changes in record when change event is fired on fields.
	 * @param {Object} field The field updated field
	 * @param {Object} value The value of the field updated
	 * @private
	 */
	onChange : function(field, value)
	{
		this.record.set(field.name, value);
	},

	/**
	 * Function will be called when user toggles state of checkbox to indicate that
	 * this will be last rule to be executed after that no rules should be executed.
	 * @param {Ext.form.Checkbox} checkbox The Checkbox being toggled.
	 * @param {Boolean} checked The new checked state of the checkbox.
	 * @private
	 */
	onToggleStopProcessing : function(checkbox, checked)
	{
		var state = this.record.get('rule_state');

		if (checked) {
			state |= Zarafa.core.mapi.RuleStates.ST_EXIT_LEVEL;
		} else {
			state &= ~Zarafa.core.mapi.RuleStates.ST_EXIT_LEVEL;
		}

		this.record.set('rule_state', state);
	},

	/**
	 * Updates the panel by loading data from the record into the form panel.
	 * @param {Zarafa.rules.delegates.data.RulesRecord} record The record update the panel with.
	 * @param {Boolean} contentReset force the component to perform a full update of the data.
	 */
	update : function(record, contentReset)
	{
		this.record = record;

		this.getForm().loadRecord(record);

		var RuleStates = Zarafa.core.mapi.RuleStates;
		var state = record.get('rule_state');

		this.stopProcessingCheckbox.setValue((state & RuleStates.ST_EXIT_LEVEL) === RuleStates.ST_EXIT_LEVEL);
	},

	/**
	 * Update the given {@link Zarafa.core.data.IPMRecord record} with
	 * the values from this {@link Ext.Panel panel}.
	 * @param {Zarafa.core.data.IPMRecord} record The record to update
	 */
	updateRecord : function(record)
	{
		record.beginEdit();

		this.getForm().updateRecord(record);

		var state = this.record.get('rule_state');
		if (this.stopProcessingCheckbox.getValue()) {
			state |= Zarafa.core.mapi.RuleStates.ST_EXIT_LEVEL;
		} else {
			state &= ~Zarafa.core.mapi.RuleStates.ST_EXIT_LEVEL;
		}
		this.record.set('rule_state', state);

		record.endEdit();
	}
});

Ext.reg('zarafa.ruleseditpanel', Zarafa.common.rules.dialogs.RulesEditPanel);
Ext.namespace('Zarafa.common.rules.dialogs');

/**
 * @class Zarafa.common.rules.dialogs.RulesWordsEditPanel
 * @extends Ext.form.FormPanel
 * @xtype zarafa.ruleswordseditpanel
 *
 * Panel to edit the {@link Zarafa.common.rules.data.ConditionFlags#SUBJECT_WORDS} condition
 * for a {@link Zarafa.common.rules.data.RulesRecord Rule}.
 */
Zarafa.common.rules.dialogs.RulesWordsEditPanel = Ext.extend(Ext.form.FormPanel, {
	/**
	 * @cfg {Ext.data.Store} store The store which contains the words on which
	 * is being filtered.
	 */
	store : undefined,

	/**
	 * @constructor
	 * @param config Configuration structure
	 */
	constructor : function(config)
	{
		config = config || {};

		Ext.applyIf(config, {
			// Override from Ext.Component
			xtype : 'zarafa.ruleswordseditpanel',
			layout : {
				type : 'vbox',
				align : 'stretch'
			},
			padding : 5,
			items : [{
				xtype : 'zarafa.compositefield',
				items : [{
					xtype : 'button',
					ref : '../editWordBtn',
					text : _('Edit'),
					width : 70,
					disabled : true,
					handler : this.onWordEdit,
					scope : this
				},{
					xtype : 'spacer',
					width : 5
				},{
					xtype : 'button',
					ref : '../deleteWordBtn',
					text : _('Delete'),
					width : 70,
					disabled : true,
					handler : this.onWordDelete,
					scope : this
				}]
			},{
				xtype : 'spacer',
				height : 5
			},{
				xtype : 'zarafa.compositefield',
				items : [{
					xtype : 'textfield',
					ref : '../wordField',
					hideLabel : true,
					flex : 1,
					listeners : {
						'specialkey' : this.onInputSpecialKey,
						'scope' : this
					}
				},{
					xtype : 'button',
					autoWidth : true,
					iconCls : 'zarafa-rules-add',
					handler : this.onWordAdd,
					scope : this
				}]
			},{
				xtype : 'spacer',
				height : 5
			},{
				xtype : 'grid',
				ref : 'gridPanel',
				flex : 1,
				border : true,
				enableHdMenu : false,
				enableColumnMove : false,
				store : config.store,
				viewConfig : {
					headersDisabled : true,
					forceFit: true,
					autoExpandColumn: true,
					markDirty : false,
					scrollOffset: 0
				},
				colModel : new Ext.grid.ColumnModel({
					columns: [{
						dataIndex: 'words',
						sortable: false,
						renderer : Ext.util.Format.htmlEncode
					}]
				}),
				sm : new Ext.grid.RowSelectionModel({
					singleSelect : true,
					listeners : {
						'selectionchange' : this.onSelectionChange,
						'scope' : this
					}			
				})
			}]
		});

		Zarafa.common.rules.dialogs.RulesWordsEditPanel.superclass.constructor.call(this, config);
	},

	/**
	 * Event handler which is fired when the user pressed the {@link Ext.EventObject#ENTER ENTER} key.
	 * This will call {@link #onWordAdd}.
	 * @param {Ext.form.Field} field The field which fired the event
	 * @param {Ext.EventObject} e The event object
	 * @private
	 */
	onInputSpecialKey : function(field, e)
	{
		if (e.getKey() == e.ENTER) {
			this.onWordAdd();
		}
	},

	/**
	 * Event handler which is fired when the 'Add' button has been clicked. This
	 * will read the content of the input field, and add it to the {@link #store}.
	 * After that the wordField is cleared, and the focus is reapplied to the
	 * input field.
	 * @return {Boolean} False when user tries to add another word.
	 * @private
	 */
	onWordAdd : function()
	{
		var value = this.wordField.getValue().trim();
		if (!Ext.isEmpty(value)) {
			this.store.add(new Ext.data.Record({ words : value }));
			this.wordField.reset();
		}
		this.wordField.focus();
	},

	/**
	 * Event handler which is fired when the 'Edit' button has been clicked. This
	 * will obtain the currently selected record, remove it from the {@link #store}
	 * and place the contents into the input field for further editing.
	 * @private
	 */
	onWordEdit : function()
	{
		var record = this.gridPanel.getSelectionModel().getSelected();
		if (record) {
			this.store.remove(record);
			this.wordField.setValue(record.get('words'));
			this.wordField.focus();
		}
	},

	/**
	 * Event handler which is fired when the 'Delete' button has been clicked. This
	 * will remove the selected record from the {@link #store}.
	 * @private
	 */
	onWordDelete : function()
	{
		var record = this.gridPanel.getSelectionModel().getSelected();
		if (record) {
			this.store.remove(record);
		}
	},

	/**
	 * Event handler which is fired when the {@link Ext.grid.RowSelectionModel} fires the
	 * {@link Ext.grid.RowSelectionModel#selectionchange 'selectionchange'} event. This will
	 * update the "Edit" and "Delete" buttons.
	 * @param {Ext.grid.RowSelectionModel} selModel The selection model which fired the event
	 * @private
	 */
	onSelectionChange : function(selModel)
	{
		var hasSelection = selModel.hasSelection();
		this.editWordBtn.setDisabled(!hasSelection);
		this.deleteWordBtn.setDisabled(!hasSelection);
	}
});

Ext.reg('zarafa.ruleswordseditpanel', Zarafa.common.rules.dialogs.RulesWordsEditPanel);
Ext.namespace('Zarafa.common.rules.dialogs');

/**
 * @class Zarafa.common.rules.dialogs.SentToMeLink
 * @extends Ext.Container
 * @xtype zarafa.senttomelink
 *
 * Condition component for the {@link Zarafa.common.rules.data.ConditionFlags#SENT_TO_ME_ONLY SENT_TO_ME_ONLY}
 * condition Flag. This will not show anything to the user, but does generate a proper
 * restriction in {@link #getCondition}.
 */
Zarafa.common.rules.dialogs.SentToMeLink = Ext.extend(Ext.Container, {

	/**
	 * The Condition type which is handled by this view
	 * This is set during {@link #setCondition}.
	 * @property
	 * @type Zarafa.common.rules.data.ConditionFlags
	 */
	conditionFlag : undefined,

	/**
	 * The condition property which was configured during
	 * {@link #setCondition}.
	 * @property
	 * @type Object
	 */
	 condition : undefined,

	 /**
	  * True if the condition was modified by the user, if this is false,
	  * then {@link #getCondition} will return {@link #condition} instead
	  * of returning a new object.
	  * @property
	  * @type Boolean
	  */
	isModified : false,

	/**
	 * Apply an action onto the DataView, this will parse the condition and show
	 * the contents in a user-friendly way to the user.
	 * @param {Zarafa.common.rules.data.ConditionFlags} conditionFlag The condition type
	 * which identifies the exact type of the condition.
	 * @param {Object} condition The condition to apply
	 */
	setCondition : function(conditionFlag, condition)
	{
		this.conditionFlag = conditionFlag;
		this.condition = condition;
		this.isModified = !Ext.isDefined(condition);
	},

	/**
	 * Obtain the condition as configured by the user
	 * @return {Object} The condition
	 */
	getCondition : function()
	{
		if (this.isModified !== true) {
			return this.condition;
		}

		var RestrictionFactory = Zarafa.core.data.RestrictionFactory;
		var Restrictions = Zarafa.core.mapi.Restrictions;

		// Invalid conditionFlag
		if (this.conditionFlag !== Zarafa.common.rules.data.ConditionFlags.SENT_TO_ME_ONLY) {
			return false;
		}

		// Create a restriction
		return RestrictionFactory.createResAnd([
			// The PR_MESSAGE_TO_ME property should be set to 'true'
			RestrictionFactory.dataResProperty('PR_MESSAGE_TO_ME', Restrictions.RELOP_EQ, true, '0x0057000B'),
			// The PR_DISPLAY_TO property should not contain the ';' character (this implies a single recipient).
			RestrictionFactory.createResNot(
				RestrictionFactory.dataResContent('PR_DISPLAY_TO', Restrictions.FL_SUBSTRING, ';', '0x0E04001E')
			),
			// The PR_DISPLAY_CC property should be empty
			RestrictionFactory.dataResProperty('PR_DISPLAY_CC', Restrictions.RELOP_EQ, '', '0x00E03001E')
		]);
	}
});

Ext.reg('zarafa.senttomelink', Zarafa.common.rules.dialogs.SentToMeLink);
Ext.namespace('Zarafa.common.rules.dialogs');

/**
 * @class Zarafa.common.rules.dialogs.UserSelectionLink
 * @extends Ext.BoxComponent
 * @xtype zarafa.userselectionlink
 */
Zarafa.common.rules.dialogs.UserSelectionLink = Ext.extend(Ext.BoxComponent, {
	/**
	 * @cfg {String} fieldLabel The label which must be applied to template
	 * as a prefix to the list of attachments.
	 */
	emptyText :_('Select one...'),

	/**
	 * @cfg {String} userStringSeparator The separator which is used to separate list
	 * of user while displaying them as string.
	 */
	userStringSeparator : _('and'),

	/**
	 * @cfg {Zarafa.core.data.IPMRecipientStore} store The store in which
	 * the recipients are stored
	 */
	store : undefined,

	/**
	 * The Action type which is handled by this view
	 * This is set during {@link #setAction}.
	 * @property
	 * @type Zarafa.common.rules.data.ActionFlags
	 */
	actionFlag : undefined,

	/**
	 * The action property which was configured during
	 * {@link #setAction}.
	 * @property
	 * @type Object
	 */
	action : undefined,

	/**
	 * The Condition type which is handled by this view
	 * This is set during {@link #setCondition}.
	 * @property
	 * @type Zarafa.common.rules.data.ConditionFlags
	 */
	conditionFlag : undefined,

	/**
	 * The condition property which was configured during
	 * {@link #setCondition}.
	 * @property
	 * @type Object
	 */
	condition : undefined,

	/**
	 * True if the action/condition was modified by the user, if this is false,
	 * then {@link #getCondition} will return {@link #condition} instead
	 * of returning a new object and {@link #getAction} will return {@link #action}
	 * instead of returning a new Object.
	 * @property
	 * @type Boolean
	 */
	isModified : false,

	/**
	 * True if the action/condition is complete and valid,
	 * False will denote that action/condition is invalid or incomplete
	 * if this is true, then {@link #getCondition} will return {@link #condition} instead
	 * of returning a new object and {@link #getAction} will return {@link #action}
	 * instead of returning a new Object.
	 * @property
	 * @type Boolean
	 */
	isValid : true,

	/**
	 * @constructor
	 * @param {Object} config configuration object.
	 */
	constructor : function(config)
	{
		config = config || {};

		Ext.applyIf(config,{
			xtype: 'zarafa.userselectionlink',
			border : false,
			autoScroll:true,
			anchor : '100%',
			multiSelect : false,
			store : new Zarafa.core.data.IPMRecipientStore(),
			tpl : new Ext.XTemplate(
				'<div class="zarafa-user-link">' +
					'<tpl for="list">' + 
						'<tpl if="!Ext.isEmpty(values.display_name)">' +
							'&quot;{display_name:htmlEncode}&quot;' +
						'</tpl>' +
						'<tpl if="Ext.isEmpty(values.display_name) && !Ext.isEmpty(values.smtp_address)">' +
							'&quot;{smtp_address:htmlEncode}&quot;' +
						'</tpl>' +
						'<tpl if="xcount &gt; 0 && xindex != xcount">' +
							'<span>&nbsp;{parent.seperator}&nbsp;</span>' +
						'</tpl>' +
					'</tpl>' +
				'</div>',
				{
					compiled : true
				}
			)
		});

		Zarafa.common.rules.dialogs.UserSelectionLink.superclass.constructor.call(this, config);
	},

	/**
	 * This function is called after the component has been rendered.
	 * This will register the {@link #onActivate} and {@link #onClick} event handlers.
	 * @private
	 */
	afterRender : function()
	{
		Zarafa.common.rules.dialogs.UserSelectionLink.superclass.initComponent.apply(this, arguments);

		this.mon(this.getActionEl(), 'click', this.onClick, this);
		this.mon(this.store, 'update', this.onRecipientUpdate, this);
		this.mon(this.store, 'add', this.onRecipientAdd, this);
		this.mon(this.store, 'resolved', this.onRecipientAdd, this);
	},

	/**
	 * Handler will be called when recipient in {@link Zarafa.core.data.IPMRecipientStore IPMRecipientStore}.
	 * It could happen that recipient is updated after we have closed addressbook dialog (mainly in case of resolving)
	 * so we need to update the ui also
	 * @param {Zarafa.core.data.IPMRecipientStore} store The store which fired the event
	 * @param {Zarafa.core.data.IPMRecipientRecord} record The record which was updated
	 * @param {String} operation The update operation being performed.
	 * @private
	 */
	onRecipientUpdate : function(store, records, operation)
	{
		if (operation !== Ext.data.Record.COMMIT) {
			// update ui, after store is updated
			this.update(this.store);
		}
	},

	/**
	 * Handler will be called when recipient is added to {@link Zarafa.core.data.IPMRecipientStore IPMRecipientStore}.
	 * @param {Zarafa.core.data.IPMRecipientStore} store The store which fired the event
	 * @param {Zarafa.core.data.IPMRecipientRecord} records The records which have been added
	 * @private
	 */
	onRecipientAdd : function(store, records)
	{
		var record = records[0];
		if (record.get('object_type') === Zarafa.core.mapi.ObjectType.MAPI_DISTLIST) {
			Ext.MessageBox.show({
				title: _('Zarafa WebApp'), 
				msg: _('Distribution lists are not supported in rules, would you like to replace the distribution list with its members?'), 
				buttons: Ext.MessageBox.YESNO, 
				fn: this.onExpandDistList.createDelegate(this, [record], 1),
				scope: this
			});
		}
	},

	/**
	 * Handler for messagebox which asks the user to expand the distribution list of remove it.
	 * @param {String} btn string containing either 'yes' or 'no
	 * @param {Zarafa.core.data.IPMRecipientRecord} recip The records which should been expanded
	 */
	onExpandDistList: function(btn, recip)
	{
		if(btn === 'yes') {
			this.store.expand(recip, true);
		}
		this.store.remove(recip);
	},

	/**
	 * Called when user clicks on a {@link Zarafa.common.rules.dialogs.UserSelectionLink}
	 * It opens user selection dialog dialog
	 * @param {Ext.DataView} dataView Reference to this object
	 * @param {Number} index The index of the target node
	 * @param {HTMLElement} node The target node
	 * @param {Ext.EventObject} evt The mouse event
	 * @protected
	 */
	onClick : function(dataView, index, node, evt)
	{
		var hideGroups = [];
		if(this.conditionFlag === Zarafa.common.rules.data.ConditionFlags.RECEIVED_FROM) {
			hideGroups.push('distribution_list');
		}

		var label;
		if (this.conditionFlag === Zarafa.common.rules.data.ConditionFlags.RECEIVED_FROM) {
			label = _('From');
		} else {
			label = _('To');
		}

		// Open addressbook dialog for selection of users
		Zarafa.common.Actions.openABUserMultiSelectionContent({
			listRestriction : {
				hide_groups : hideGroups
			},
			callback : function(record) {
				this.isModified = true;
				this.update(this.store);
			},
			convert : function(r) {
				return r.convertToRecipient();
			},
			scope : this,
			store : this.store,
			modal : true,
			selectionCfg : [{
				xtype : 'zarafa.recipientfield',
				fieldLabel : label + ':',
				height : 50,
				boxStore : this.store,
				flex : 1
			}]
		});
	},

	/**
	 * Apply an action onto the DataView, this will parse the condition and show
	 * the contents in a user-friendly way to the user.
	 * @param {Zarafa.common.rules.data.ConditionFlags} conditionFlag The condition type
	 * which identifies the exact type of the condition.
	 * @param {Object} condition The condition to apply
	 */
	setCondition : function(conditionFlag, condition)
	{
		this.store.removeAll();
		this.isValid = false;
		if (condition) {
			var Restrictions = Zarafa.core.mapi.Restrictions;

			switch (conditionFlag) {
				case Zarafa.common.rules.data.ConditionFlags.SENT_TO:
					var subs;

					// Check if a RES_OR restriction was provided, if
					// so we need to loop over all recipients from the list.
					if (condition[0] == Restrictions.RES_OR) {
						subs = condition[1];
					} else {
						subs = [ condition ];
					}

					for (var i = 0, len = subs.length; i < len; i++) {
						var value = subs[i][1][Restrictions.RESTRICTION];

						// Do not add empty recipient, it is possible that restriction might not have recipient.
						if(value[1] && value[1][Restrictions.PROPS] && value[1][Restrictions.PROPS]['0x0001001E']) {
							var recipient = this.store.parseRecipient(value[1][Restrictions.PROPS]['0x0001001E']);
							recipient.set('display_type', value[1][Restrictions.PROPS]['PR_DISPLAY_TYPE']);
							recipient.set('search_key', value[1][Restrictions.RESTRICTION][1][Restrictions.VALUE]['0x00010102']);

							this.store.add(recipient);
							this.isValid = true;
						}
					}
					break;
				case Zarafa.common.rules.data.ConditionFlags.RECEIVED_FROM:
					var RecordFactory = Zarafa.core.data.RecordFactory;
					var RecordCustomObjectType = Zarafa.core.data.RecordCustomObjectType;
					var subs;

					// Check if a RES_OR restriction was provided, if
					// so we need to loop over all recipients from the list.
					if (condition[0] == Restrictions.RES_OR) {
						subs = condition[1];
					} else {
						subs = [ condition ];
					}

					for (var i = 0, len = subs.length; i < len; i++) {
						var value = subs[i][1];

						// Do not add empty recipient, it is possible that restriction might not have recipient.
						if(value[Restrictions.PROPS] && value[Restrictions.PROPS]['0x0001001E']) {
							var recipient = this.store.parseRecipient(value[Restrictions.PROPS]['0x0001001E']);
							recipient.set('display_type', value[Restrictions.PROPS]['PR_DISPLAY_TYPE']);
							recipient.set('search_key', value[Restrictions.RESTRICTION][1][Restrictions.VALUE]['0x00010102']);

							this.store.add(recipient);
							this.isValid = true;
						}
					}
					break;
				default:
					break;
			}
		}

		this.conditionFlag = conditionFlag;
		this.condition = condition;
		this.userStringSeparator = _('or');
		this.isModified = !Ext.isDefined(condition);
		this.update(this.store);
	},

	/**
	 * Obtain the condition as configured by the user
	 * @return {Object} The condition
	 */
	getCondition : function()
	{
		if (this.isModified !== true && this.isValid === true) {
			return this.condition;
		}

		var conditions = [];

		// No recipients selected, this means
		// we can't create a condition.
		if (this.store.getCount() === 0) {
			return false;
		}

		switch (this.conditionFlag) {
			case Zarafa.common.rules.data.ConditionFlags.SENT_TO:
				this.store.each(function(recipient) {
					if (recipient.isResolved()) {
						conditions.push(this.createToRestriction(recipient));
					}
				}, this);
				break;
			case Zarafa.common.rules.data.ConditionFlags.RECEIVED_FROM:
				this.store.each(function(recipient) {
					if (recipient.isResolved()) {
						conditions.push(this.createFromRestriction(recipient));
					}
				}, this);
				break;
			default:
				// invalid actionFlag
				return false;
		}

		// If there was only 1 recipient, we don't need to convert
		// it to a OR subrestriction. If we have more then 1 recipient,
		// then we should create the OR restriction.
		if (conditions.length === 1) {
			return conditions[0];
		} else if (conditions.length > 1) {
			return Zarafa.core.data.RestrictionFactory.createResOr(conditions);
		} else {
			return false;
		}
	},

	/**
	 * Convert a {@link Zarafa.core.data.IPMRecipientRecord Recipient} into
	 * a Restriction used for the {@link #getCondition}.
	 * @param {Zarafa.core.data.IPMRecipientRecord} recipient The recipient for which
	 * the restriction should be created
	 * @return {Object} The restriction
	 * @private
	 */
	createToRestriction : function(recipient)
	{
		var RestrictionFactory = Zarafa.core.data.RestrictionFactory;
		var Restrictions = Zarafa.core.mapi.Restrictions;

		return RestrictionFactory.createResSubRestriction('PR_MESSAGE_RECIPIENTS',
			RestrictionFactory.dataResComment(
				RestrictionFactory.dataResProperty('PR_SEARCH_KEY', Restrictions.RELOP_EQ, recipient.get('search_key'), '0x00010102'),
				{
					'0x60000003' : Zarafa.core.mapi.RecipientType.MAPI_TO,
					'0x00010102' : recipient.get('search_key'),
					'0x0001001E' : recipient.get('display_name') + ' <' + recipient.get('smtp_address') + '>',
					'PR_DISPLAY_TYPE' : recipient.get('display_type')
				}
			)
		);
	},

	/**
	 * Convert a {@link Zarafa.core.data.IPMRecipientRecord Recipient} into
	 * a Restriction used for the {@link #getCondition}.
	 * @param {Zarafa.core.data.IPMRecipientRecord} recipient The recipient for which
	 * the restriction should be created
	 * @return {Object} The restriction
	 * @private
	 */
	createFromRestriction : function(recipient)
	{
		var RestrictionFactory = Zarafa.core.data.RestrictionFactory;
		var Restrictions = Zarafa.core.mapi.Restrictions;

		return RestrictionFactory.dataResComment(
			RestrictionFactory.dataResProperty('PR_SENDER_SEARCH_KEY', Restrictions.RELOP_EQ, recipient.get('search_key'), '0x00010102'),
			{
				'0x60000003' : Zarafa.core.mapi.RecipientType.MAPI_TO,
				'0x00010102' : recipient.get('search_key'),
				'0x0001001E' : recipient.get('display_name') + ' <' + recipient.get('smtp_address') + '>',
				'PR_DISPLAY_TYPE' : recipient.get('display_type')
			}
		);
	},

	/**
	 * Apply an action onto the DataView, this will parse the action and show
	 * the contents in a user-friendly way to the user.
	 * @param {Zarafa.common.rules.data.ActionFlags} actionFlag The action type
	 * which identifies the exact type of the action.
	 * @param {Object} action The action to apply
	 */
	setAction : function(actionFlag, action)
	{
		this.store.removeAll();
		this.isValid = false;
		if (action) {
			var RecordFactory = Zarafa.core.data.RecordFactory;
			var CustomObjectType = Zarafa.core.data.RecordCustomObjectType;

			for (var i = 0, len = action.adrlist.length; i < len; i++) {
				var address = action.adrlist[i];
				this.isValid = true;

				var recipient = RecordFactory.createRecordObjectByCustomType(CustomObjectType.ZARAFA_RECIPIENT, {
					'entryid' : address['PR_ENTRYID'],
					'object_type' : address['PR_OBJECT_TYPE'],
					'display_name' : address['PR_DISPLAY_NAME'],
					'display_type' : address['PR_DISPLAY_TYPE'],
					'email_address' : address['PR_EMAIL_ADDRESS'],
					'smtp_address' : address['PR_SMTP_ADDRESS'],
					'address_type' : address['PR_ADDRTYPE'],
					'recipient_type' : address['PR_RECIPIENT_TYPE'],
					'search_key' : address['PR_SEARCH_KEY']
				});
				this.store.add(recipient);
			}
		}

		this.actionFlag = actionFlag;
		this.action = action;
		this.userStringSeparator = _('and');
		this.isModified = !Ext.isDefined(action);
		this.update(this.store);
	},

	/**
	 * Obtain the action as configured by the user
	 * @return {Object} The action
	 */
	getAction : function()
	{
		if (this.isModified !== true && this.isValid === true) {
			return this.action;
		}

		var action = {};

		// No recipients selected, this means
		// we can't create an action.
		if (this.store.getCount() === 0) {
			return false;
		}

		// Set the Address list in the action
		action.adrlist = [];
		this.store.each(function(recipient) {
			action.adrlist.push({
				PR_ENTRYID : recipient.get('entryid'),
				PR_OBJECT_TYPE : recipient.get('object_type'),
				PR_DISPLAY_NAME : recipient.get('display_name'),
				PR_DISPLAY_TYPE : recipient.get('display_type'),
				PR_EMAIL_ADDRESS : recipient.get('email_address') || recipient.get('smtp_address'),
				PR_SMTP_ADDRESS : recipient.get('smtp_address'),
				PR_ADDRTYPE : recipient.get('address_type'),
				PR_RECIPIENT_TYPE : recipient.get('recipient_type'),
				PR_SEARCH_KEY : recipient.get('search_key')
			});
		}, this);

		// Fill in the additional properties required for the action.
		var ActionFlags = Zarafa.common.rules.data.ActionFlags;
		var RuleActions = Zarafa.core.mapi.RuleActions;
		var FlavorFlags = Zarafa.core.mapi.FlavorFlags;
		switch (this.actionFlag) {
			case ActionFlags.REDIRECT:
				action.action = RuleActions.OP_FORWARD;
				action.flags = 0;
				action.flavor = FlavorFlags.FWD_PRESERVE_SENDER | FlavorFlags.FWD_DO_NOT_MUNGE_MSG;
				break;
			case ActionFlags.FORWARD:
				action.action = RuleActions.OP_FORWARD;
				action.flags = 0;
				action.flavor = 0;
				break;
			case ActionFlags.FORWARD_ATTACH:
				action.action = RuleActions.OP_FORWARD;
				action.flags = 0;
				action.flavor = FlavorFlags.FWD_AS_ATTACHMENT;
				break;
			default:
				// invalid actionFlag
				return false;
		}

		return action;
	},

	/**
	 * Update the contents of this dataview, this will apply the {@link #tpl} for
	 * the {@link #store}.
	 * @param {Zarafa.core.data.IPMRecipientStore} store The store to show
	 */
	update : function(store)
	{
		var data = {
			seperator : this.userStringSeparator,
			list : Ext.pluck(store.getRange(), 'data')
		};

		if (Ext.isEmpty(data.list)) {
			data.list = [{
				display_name : this.emptyText
			}];
		}

		Zarafa.common.rules.dialogs.UserSelectionLink.superclass.update.call(this, this.tpl.apply(data));
	}
});

Ext.reg('zarafa.userselectionlink', Zarafa.common.rules.dialogs.UserSelectionLink);
Ext.namespace('Zarafa.common.rules.dialogs');

/**
 * @class Zarafa.common.rules.dialogs.WordSelectionLink
 * @extends Ext.BoxComponent
 * @xtype zarafa.wordselectionlink
 */
Zarafa.common.rules.dialogs.WordSelectionLink = Ext.extend(Ext.BoxComponent, {
	/**
	 * @cfg {String} fieldLabel The label which must be applied to template
	 * as a prefix to the list of attachments.
	 */
	emptyText :_('Select one...'),

	/**
	 * @cfg {String} wordStringSeparator The separator which is used to separate list
	 * of words while displaying them as string.
	 */
	wordStringSeparator :_('or'),

	/**
	 * @cfg {Ext.data.Store} store The store in which
	 * the words are stored
	 */
	store : undefined,

	/**
	 * The Condition type which is handled by this view
	 * This is set during {@link #setCondition}.
	 * @property
	 * @type Zarafa.common.rules.data.ConditionFlags
	 */
	conditionFlag : undefined,

	/**
	 * The condition property which was configured during
	 * {@link #setCondition}.
	 * @property
	 * @type Object
	 */
	condition : undefined,

	/**
	 * True if the action/condition was modified by the user, if this is false,
	 * then {@link #getCondition} will return {@link #condition} instead
	 * of returning a new object and {@link #getAction} will return {@link #action}
	 * instead of returning a new Object.
	 * @property
	 * @type Boolean
	 */
	isModified : false,

	/**
	 * True if the action/condition is complete and valid,
	 * False will denote that action/condition is invalid or incomplete
	 * if this is true, then {@link #getCondition} will return {@link #condition} instead
	 * of returning a new object and {@link #getAction} will return {@link #action}
	 * instead of returning a new Object.
	 * @property
	 * @type Boolean
	 */
	isValid : true,

	/**
	 * @constructor
	 * @param {Object} config configuration object.
	 */
	constructor : function(config)
	{
		config = config || {};

		Ext.applyIf(config,{
			xtype: 'zarafa.wordselectionlink',
			border : false,
			autoScroll:true,
			anchor : '100%',
			multiSelect : false,
			store : new Ext.data.Store({ fields : [ 'words' ] }),
			tpl : new Ext.XTemplate(
				'<div class="zarafa-word-link">' +
					'<tpl for=".">' + 
						'<tpl if="!Ext.isEmpty(values.words)">' +
							'&quot;{words:htmlEncode}&quot;'+
						'</tpl>' +
						'<tpl if="xcount &gt; 0 && xindex != xcount">' +
							'<span>&nbsp;' + this.wordStringSeparator  + '&nbsp;</span>' +
						'</tpl>' +
					'</tpl>' +
				'</div>',
				{
					compiled : true,
					wordStringSeparator : config.wordStringSeparator || this.wordStringSeparator
				}
			)
		});

		Zarafa.common.rules.dialogs.WordSelectionLink.superclass.constructor.call(this, config);
	},

	/**
	 * This function is called after the component has been rendered.
	 * This will register the {@link #onActivate} and {@link #onClick} event handlers.
	 * @private
	 */
	afterRender : function()
	{
		Zarafa.common.rules.dialogs.WordSelectionLink.superclass.initComponent.apply(this, arguments);

		this.mon(this.getActionEl(), 'click', this.onClick, this);
	},

	/**
	 * Called when user clicks on a {@link Zarafa.common.rules.dialogs.WordSelectionLink}
	 * It opens words edit dialog.
	 * @param {Ext.DataView} dataView Reference to this object
	 * @param {Number} index The index of the target node
	 * @param {HTMLElement} node The target node
	 * @param {Ext.EventObject} evt The mouse event
 	 * @protected
	 */
	onClick : function(dataView, index, node, evt)
	{
		var tmpStore = new Ext.data.Store({ fields : [ 'words' ] });
		tmpStore.add(this.store.getRange());

		// Open RulesWordsEditDialog
		Zarafa.common.Actions.openRulesWordsEditContent({
			store : tmpStore,
			callback : this.rulesWordsEditDialogCallback,
			scope : this,
			modal : true
		});
	},

	/**
	 * Callback function for {@link Zarafa.common.rules.dialogs.RulesWordsEditDialog}
	 * @param {Ext.data.Store} store the store which contains list of words
	 * @private
	 */
	rulesWordsEditDialogCallback : function(store)
	{
		// Remove old words and add words which we have got
		// from callback function of RulesWordsEditDialog.
		this.store.removeAll();
		this.store.add(store.getRange());
		this.isModified = true;
		this.update(this.store);
	},

	/**
	 * Apply an action onto the DataView, this will parse the condition and show
	 * the contents in a user-friendly way to the user.
	 * @param {Zarafa.common.rules.data.ConditionFlags} conditionFlag The condition type
	 * which identifies the exact type of the condition.
	 * @param {Object} condition The condition to apply
	 */
	setCondition : function(conditionFlag, condition)
	{
		this.store.removeAll();
		this.isValid = false;

		// Parse the condition and add words in the store.
		if (condition) {
			var Restrictions = Zarafa.core.mapi.Restrictions;
			switch (conditionFlag) {
				case Zarafa.common.rules.data.ConditionFlags.SENDER_WORDS:
					var subs;

					// Check if a RES_OR restriction was provided, if
					// so we need to loop over all words from the list.
					if (condition[0] == Restrictions.RES_OR) {
						subs = condition[1];
					} else {
						subs = [ condition ];
					}

					for (var i = 0, len = subs.length; i < len; i++) {
						var value = subs[i][1][Restrictions.VALUE];
						Ext.iterate(value, function(key, value) {
							if(!Ext.isEmpty(value)) {
								this.isValid = true;
								if(conditionFlag === Zarafa.common.rules.data.ConditionFlags.SENDER_WORDS) {
									value = Zarafa.core.Util.hexToString(value);
								}
								this.store.add(new Ext.data.Record({ words : value }));
							}
						}, this);
					}
					break;

				case Zarafa.common.rules.data.ConditionFlags.SUBJECT_WORDS:
				case Zarafa.common.rules.data.ConditionFlags.BODY_WORDS:
				default:
					var subs;

					// Check if a RES_OR restriction was provided, if
					// so we need to loop over all words from the list.
					if (condition[0] == Restrictions.RES_OR) {
						subs = condition[1];
					} else {
						subs = [ condition ];
					}

					for (var i = 0, len = subs.length; i < len; i++) {
						var value = subs[i][1][Restrictions.VALUE];
						Ext.iterate(value, function(key, value) {
							if(!Ext.isEmpty(value)) {
								this.isValid = true;
								this.store.add(new Ext.data.Record({ words : value }));
							}
						}, this);
					}
					break;
			}
		}

		this.conditionFlag = conditionFlag;
		this.condition = condition;
		this.isModified = !Ext.isDefined(condition);
		this.update(this.store);
		
	},

	/**
	 * Obtain the condition as configured by the user
	 * @return {Object} The condition
	 */
	getCondition : function()
	{
		if (this.isModified !== true && this.isValid === true) {
			return this.condition;
		}

		// No words are added, so we can't create a condition.
		if (this.store.getCount() === 0) {
			return false;
		}

		var conditions = [];
		var RestrictionFactory = Zarafa.core.data.RestrictionFactory;
		var Restrictions = Zarafa.core.mapi.Restrictions;

		switch (this.conditionFlag) {
			case Zarafa.common.rules.data.ConditionFlags.SENDER_WORDS:
				this.store.each(function(word) {
						var value = Zarafa.core.Util.stringToHex(word.get('words'));
						conditions.push(RestrictionFactory.dataResContent('PR_SENDER_SEARCH_KEY', Restrictions.FL_SUBSTRING, value));
					}, this);
				break;

			case Zarafa.common.rules.data.ConditionFlags.SUBJECT_WORDS:
				this.store.each(function(word) {
					conditions.push(RestrictionFactory.dataResContent('PR_SUBJECT', Restrictions.FL_SUBSTRING | Restrictions.FL_IGNORECASE, word.get('words')));
				}, this);
				break;
			case Zarafa.common.rules.data.ConditionFlags.BODY_WORDS:
				this.store.each(function(word) {
					conditions.push(RestrictionFactory.dataResContent('PR_BODY', Restrictions.FL_SUBSTRING | Restrictions.FL_IGNORECASE, word.get('words')));
				}, this);
				break;

			default:
				// Invalid conditionFlag
				return false;
		}

		// If there was only 1 word condtion, we don't need to convert
		// it to a OR subrestriction. If we have more then 1 word condtion,
		// then we should create the OR restriction.
		if (conditions.length === 1) {
			return conditions[0];
		} else {
			return Zarafa.core.data.RestrictionFactory.createResOr(conditions);
		}
	},

	/**
	 * Update the contents of this dataview, this will apply the {@link #tpl} for
	 * the {@link #store}.
	 * @param {Zarafa.core.data.IPMRecipientStore} store The store to show
	 */
	update : function(store)
	{
		var data = Ext.pluck(store.getRange(), 'data');
		if (Ext.isEmpty(data)) {
			data = [{
				words : this.emptyText
			}]
		}

		Zarafa.common.rules.dialogs.WordSelectionLink.superclass.update.call(this, this.tpl.apply(data));
	}
});

Ext.reg('zarafa.wordselectionlink', Zarafa.common.rules.dialogs.WordSelectionLink);
Ext.namespace('Zarafa.common.rules.ui');

/**
 * @class Zarafa.common.rules.ui.RulesPanel
 * @extends Ext.Panel
 * @xtype zarafa.rulespanel
 * Will generate UI for the {@link Zarafa.common.settings.SettingsRuleWidget SettingsRuleWidget}.
 */
Zarafa.common.rules.ui.RulesPanel = Ext.extend(Ext.Panel, {
	/**
	 * @cfg {Zarafa.common.rules.data.RulesStore} store store to use for loading rules
	 */
	store : undefined,

	/**
	 * @constructor
	 * @param config Configuration structure
	 */
	constructor : function(config)
	{
		config = config || {};

		if(Ext.isEmpty(config.store)) {
			config.store = new Zarafa.common.rules.data.RulesStore();
		}

		Ext.applyIf(config, {
			// Override from Ext.Component
			xtype : 'zarafa.rulespanel',
			border : false,
			layout : 'fit',
			items : this.createPanelItems(config)
		});

		Zarafa.common.rules.ui.RulesPanel.superclass.constructor.call(this, config);
	},

	/**
	 * Function will create panel items for {@link Zarafa.common.rules.ui.RulesPanel RulesPanel}
	 * @param {Array} config config passed to the constructor
	 * @return {Array} array of items that should be added to panel.
	 * @private
	 */
	createPanelItems : function(config)
	{
		return [{
			xtype : 'container',
			layout : {
				type : 'vbox',
				align : 'stretch',
				pack  : 'start'
			},
			items : [{
				xtype : 'zarafa.rulesgrid',
				ref : '../rulesGrid',
				flex : 1,
				store : config.store
			}]
		}];
	},

	/**
	 * Function will be used to reload data in the {@link Zarafa.common.rules.data.RulesStore RulesStore}.
	 */
	discardChanges : function()
	{
		this.store.load();
	},

	/**
	 * Function will be used to save changes in the {@link Zarafa.common.rules.data.RulesStore RulesStore}.
	 */
	saveChanges : function()
	{
		this.store.save();
	}
});

Ext.reg('zarafa.rulespanel', Zarafa.common.rules.ui.RulesPanel);
Ext.namespace('Zarafa.common.sendas.ui');

/**
 * @class Zarafa.common.sendas.ui.SendAsGrid
 * @extends Ext.grid.GridPanel
 * @xtype zarafa.sendasgrid
 *
 * {@link Zarafa.common.sendas.ui.SendAsGrid SendAsGrid} will be used to display
 * sendas of the current user.
 */
Zarafa.common.sendas.ui.SendAsGrid = Ext.extend(Ext.grid.GridPanel, {
	/**
	 * @constructor
	 * @param {Object} config Configuration structure
	 */
	constructor : function(config)
	{
		config = config || {};
	
		Ext.applyIf(config, {
			xtype : 'zarafa.sendasgrid',
			border : true,
			viewConfig : {
				forceFit : true,
				deferEmptyText : false,
				emptyText : '<div class=\'emptytext\'>' + _('No send as address configured') + '</div>'
			},
			loadMask : {
				msg : _('Loading send as addresses') + '...'
			},
			columns : this.initColumnModel(),
			selModel : new Zarafa.common.ui.grid.RowSelectionModel({
				singleSelect : true
			}),
			listeners : {
				viewready : this.onViewReady,
				rowdblclick : this.onRowDblClick,
				scope : this
			}
		});

		Zarafa.common.sendas.ui.SendAsGrid.superclass.constructor.call(this, config);
	},

	/**
	 * Creates a column model object, used in {@link #colModel} config
	 * @return {Ext.grid.ColumnModel} column model object
	 * @private
	 */
	initColumnModel : function()
	{
		return [{
			dataIndex : 'icon_index',
			header : '<p class="icon_index">&nbsp;</p>',
			width : 25,
			fixed : true,
			renderer : Zarafa.common.ui.grid.Renderers.icon
		},{
			dataIndex : 'display_name',
			header : _('Name'),
			renderer : Zarafa.common.ui.grid.Renderers.text
		},{
			dataIndex : 'smtp_address',
			header : _('Email Address'),
			renderer : Zarafa.common.ui.grid.Renderers.text
		}]
	},

	/**
	 * Event handler which is fired when the gridPanel is ready. This will automatically
	 * select the first row in the grid.
	 * @private
	 */
	onViewReady : function()
	{
		this.getSelectionModel().selectFirstRow();
	},

	/**
	 * Function will be called to remove a sendas address.
	 * It will raise an alert if there is no selection made by user,
	 * but user is trying to remove record from the grid.
	 */
	removeSendAs : function()
	{
		var selectionModel = this.getSelectionModel();
		var sendasRecord = this.getSelectionModel().getSelected();

		if(!sendasRecord) {
			Ext.Msg.alert(_('Alert'), _('Please select a send as record.'));
			return;
		}

		// before removing send as we should select next available send as,
		// because deleting send as will remove selection
		if (selectionModel.hasNext()) {
			selectionModel.selectNext();
		} else if (selectionModel.hasPrevious()) {
			selectionModel.selectPrevious();
		}

		this.store.remove(sendasRecord);
	},
	
	/**
	 * Event handler which is fired when the 
	 * {@link Zarafa.common.sendas.ui.SendAsGrid SendAsGrid} is double clicked.
	 * it will call generic function to handle the functionality.
	 * @param {Ext.grid.GridPanel} grid the grid of which the row double clicked.	 
	 * @param {Number} rowIndex number of the row double clicked.
	 * @private
	 */
	onRowDblClick : function(grid, rowIndex)
	{
		var record = grid.getStore().getAt(rowIndex);

		if(record.isOneOff()) {
			this.editSendAsRecipient(record, false);
		} else {
			this.viewSendAsRecipient(record);
		}
	},

	/**
	 * It will call {@link Zarafa.common.Actions#openSendAsRecipientContent} to add or edit external contact.
	 * @param {Ext.data.Record} record record to be opened
	 * @param {Boolean} removeOnCancel true to remove the record 
	 * from store while pressing cancel button, false otherwise
	 */
	editSendAsRecipient : function(record, removeOnCancel)
	{
		Zarafa.common.Actions.openSendAsRecipientContent(record, { removeOnCancel : removeOnCancel});
	},
	
	/**
	 * It will call {@link Zarafa.common.Actions#openViewRecipientContent} to open the address book contact.
	 * @param {Ext.data.Record} record record to be opened
	 */
	viewSendAsRecipient : function(record)
	{
		Zarafa.common.Actions.openViewRecipientContent(record);
	}
});

Ext.reg('zarafa.sendasgrid', Zarafa.common.sendas.ui.SendAsGrid);
Ext.namespace('Zarafa.common.sendas.ui');

/**
 * @class Zarafa.common.sendas.ui.SendAsPanel
 * @extends Ext.Panel
 * @xtype zarafa.sendaspanel
 * Will generate UI for the {@link Zarafa.common.settings.SettingsSendAsWidget SettingsSendAsWidget}.
 */
Zarafa.common.sendas.ui.SendAsPanel = Ext.extend(Ext.Panel, {
	/**
	 * @constructor
	 * @param config Configuration structure
	 */
	constructor : function(config)
	{
		config = config || {};

		Ext.applyIf(config, {
			// Override from Ext.Component
			xtype : 'zarafa.sendaspanel',
			border : false,
			layout : {
				type : 'vbox',
				align : 'stretch',
				pack  : 'start'
			},
			items : this.createPanelItems(config.store)
		});

		Zarafa.common.sendas.ui.SendAsPanel.superclass.constructor.call(this, config);
	},
	
	/**
	 * Function will create panel items for {@link Zarafa.common.sendas.ui.SendAsPanel SendAsPanel}
	 * @param {Zarafa.core.data.IPMRecipientStore} store store which configured in the grid
	 * @return {Array} array of items that should be added to panel.
	 * @private
	 */
	createPanelItems : function(store)
	{
		return [{
			xtype : 'displayfield',
			value : _('Here you can setup your alias email addresses.'),
			fieldClass : 'x-form-display-field zarafa-settings-widget-extrainfo'
		}, {
			xtype : 'container',
			flex : 1,
			layout : {
				type : 'hbox',
				align : 'stretch',
				pack  : 'start'
			},
			items : [{
				xtype : 'zarafa.sendasgrid',
				ref : '../sendasGrid',
				store : store,
				flex : 1
			}, {
				xtype : 'container',
				width : 160,
				defaults : {
					width : 140
				},
				layout : {
					type : 'vbox',
					align : 'center',
					pack  : 'start'
				},
				items : [{
					xtype : 'button',
					text : _('Address Book') + '...',
					handler : this.onSendAsAddressBook,
					scope : this
				},{
					xtype : 'spacer',
					height : 20
				},{
					xtype : 'button',
					text : _('Add') + '...',
					handler : this.onSendAsAdd,
					scope : this
				},{
					xtype : 'spacer',
					height : 20
				},{
					xtype : 'button',
					text : _('Edit') + '...',
					disabled : true,
					ref : '../../editButton',
					handler : this.onSendAsEdit,
					scope : this
				},{
					xtype : 'spacer',
					height : 20
				},{
					xtype : 'button',
					text : _('View') + '...',
					disabled : true,
					ref : '../../viewButton',
					handler : this.onSendAsView,
					scope : this
				},{
					xtype : 'spacer',
					height : 20
				},{
					xtype : 'button',
					text : _('Remove') + '...',
					disabled : true,
					ref : '../../removeButton',
					handler : this.onSendAsRemove,
					scope : this
				}]
			}]
		}];
	},

	/**
	 * initialize events for the panel.
	 * @private
	 */
	initEvents : function()
	{
		Zarafa.common.sendas.ui.SendAsPanel.superclass.initEvents.call(this);

		// register event to enable/disable buttons
		this.mon(this.sendasGrid.getSelectionModel(), 'selectionchange', this.onGridSelectionChange, this);
	},

	/**
	 * Handler function will be called when user clicks on 'Add' button,
	 * this will show addressbook dialog to select sendas user.
	 * @private
	 */
	onSendAsAdd : function()
	{
		// find rowid value
		var data = Ext.pluck(this.store.getRange(), 'data');
		var rowId = Ext.max(Ext.pluck(data, 'rowid')) || 0;

		var record = Zarafa.core.data.RecordFactory.createRecordObjectByCustomType(Zarafa.core.data.RecordCustomObjectType.ZARAFA_RECIPIENT, {
			// rowid is the {@link Ext.data.JsonReader#idProperty} in {@link Zarafa.core.data.IPMRecipientStore IPMRecipientStore}
			// so we must have to configure the rowid properly.
			rowid : rowId + 1,
			display_type : Zarafa.core.mapi.DisplayType.DT_REMOTE_MAILUSER
		});
		
		this.store.add(record);

		this.sendasGrid.editSendAsRecipient(record, true);
	},

	/**
	 * Event handler will be called when selection in {@link Zarafa.common.ui.SendAsGrid SendAsGrid}
	 * has been changed.
	 * @param {Ext.grid.RowSelectionModel} selectionModel selection model that fired the event
	 */
	onGridSelectionChange : function(selectionModel)
	{
		var record = selectionModel.getSelected();

		var isOneOff = record && record.isOneOff();
		var hasSelection = selectionModel.hasSelection();

		// +----------------------------------------------------+
		// |             | Enable / Disable (Edit, View, Remove)|
		// +----------------------------------------------------+
		// |             | Selection | Operation | isOneOff     |
		// +----------------------------------------------------+
		// |Edit Button  | !(true)   | OR        | !(true)      |
		// |View Button  | !(true)   | OR        | true         |
		// |Remove Button| !(true)   | Null      | Null         |
		// +----------------------------------------------------+
		// Here is toggle the view and edit button as per the contact type
		// and if there is no selection in sendAsGrid, all three (Edit, View, Remove) buttons are disable.
		this.editButton.setDisabled(!hasSelection || !isOneOff);
		this.viewButton.setDisabled(!hasSelection || isOneOff);
		this.removeButton.setDisabled(!hasSelection);
	},

	/**
	 * Handler function will be called when user clicks on 'Remove' button,
	 * this will remove currently selected sendas from sendass list.
	 * @private
	 */
	onSendAsRemove : function()
	{
		this.sendasGrid.removeSendAs();
	},

	/**
	 * Fuction will be call to get the selected contact from {@link Zarafa.common.ui.SendAsGrid SendAsGrid}
	 * @return {Ext.data.Record} The record which is selected from {@link Zarafa.common.ui.SendAsGrid SendAsGrid}.
	 * @private
	 */
	getSendAsRecord : function()
	{
		var sendasRecord = this.sendasGrid.getSelectionModel().getSelected();

		if(!sendasRecord) {
			Ext.Msg.alert(_('Alert'), _('Please select a sendas record.'));
			return;
		}

		return sendasRecord;
	},

	/**
	 * Handler function will be called when user click on 'Edit' button,
	 * this will edit currently selected contact from sendas list.
	 * @private
	 */
	onSendAsEdit : function()
	{
		var sendasRecord = this.getSendAsRecord();

		this.sendasGrid.editSendAsRecipient(sendasRecord, false);
	},

	/**
	 * Handler function will be called when user click on 'View' button,
	 * this will open currently selected address book contact from sendas list.
	 * @private
	 */
	onSendAsView : function()
	{
		var sendasRecord = this.getSendAsRecord();

		this.sendasGrid.viewSendAsRecipient(sendasRecord);
	},

	/**
	 * Handler function will be called when user clicks on 'Address Book' button,
	 * this will open an Address Book and one can select sendas from any of the address book contact list.
	 * @private
	 */
	onSendAsAddressBook : function()
	{
		Zarafa.common.Actions.openABUserSelectionContent({
			callback : this.abCallBack,
			scope : this,
			singleSelect : false,
			listRestriction : {
				hide_users : ['system', 'everyone'],
				hide_companies : true
			}
		});
	},

	/**
	 * Callback function for {@link Zarafa.addressbook.dialogs.ABUserSelectionContent AddressBook}
	 * @param {Ext.data.Record} record user selected from AddressBook
	 * @private
	 */
	abCallBack : function(records)
	{
		// find rowid value
		var data = Ext.pluck(this.store.getRange(), 'data');
		var rowId = Ext.max(Ext.pluck(data, 'rowid')) || 0;

		for (var i = 0; i < records.length; i++) {
			var record = records[i];

			record = record.convertToRecipient();
			record.set('rowid', ++rowId);

			this.store.add(record);
		}
	}
});

Ext.reg('zarafa.sendaspanel', Zarafa.common.sendas.ui.SendAsPanel);
Ext.namespace('Zarafa.common.ui');

/**
 * @class Zarafa.common.ui.Box
 * @extends Ext.BoxComponent
 * @xtype zarafa.box
 *
 * The Box which is used by {@link Zarafa.common.ui.BoxField} components
 * to display the boxes inside itself. This box is dediciated to the BoxField
 * and cannot be used seperately.
 */
Zarafa.common.ui.Box = Ext.extend(Ext.BoxComponent, {
	/**
	 * @cfg {Ext.data.Record} record The record which is bound to this box.
	 */
	record : undefined,

	/**
	 * @cfg {Zarafa.common.ui.BoxField} parent The boxfield
	 * in which this box is being displayed.
	 */
	parent: undefined,

	/**
	 * The {@link Ext.Element} which contains the icon for this box,
	 * this will is filled during the {@link #updateIcons} function.
	 * @property
	 * @type Ext.Element
	 */
	iconEl : undefined,

	/**
	 * The {@link Ext.Element} which contains the text contents for this box,
	 * this is filled with the {@link #textTpl} template.
	 * @property
	 * @type Ext.Element
	 */
	textEl : undefined,

	/**
	 * The {@link Ext.Element} which contains the delete button for this box.
	 * @property
	 * @type Ext.Element
	 */
	delBtnEl : undefined,

	/**
	 * @cfg {Boolean} editable False when none of the buttons or user actions is
	 * allowed to change the record inside this box.
	 */
	editable : true,

	/**
	 * @cfg {Boolean} enableButtons True if {@link #renderButtons} should be called
	 * which will create the buttons inside the box.
	 */
	enableButtons : true,

	/**
	 * @cfg {Boolean} enableIcons True if {@link #renderIcons} should be called
	 * which will create the icon inside the box.
	 */
	enableIcons : true,

	/**
	 * @cfg {String} hoverCls The CSS class which will be applied to {@link #el} when
	 * the cursor is hovering over this component.
	 */
	hoverCls : 'x-zarafa-boxfield-item-hover',

	/**
	 * @cfg {String} focusCls The CSS class which will be applied to {@link #el} when
	 * the component has been selected.
	 */
	focusCls : 'x-zarafa-boxfield-item-focus',

	/**
	 * @cfg {String} textCls The CSS class which will be applied to the {@link #textEl} when
	 * the component is rendered.
	 */
	textCls : 'x-zarafa-boxfield-item-text',

	/**
	 * @cfg {String} iconCls The CSS class which will be applied to the {@link #iconEl} when
	 * the component is rendered.
	 */
	iconCls : 'x-zarafa-boxfield-item-icon',

	/**
	 * @cfg {String} btnCls The CSS class which will be applied to the {@link #btnEl} when
	 * the component is rendered.
	 */
	btnCls : 'x-zarafa-boxfield-item-close',

	/**
	 * @cfg {String} btnHoverCls The CSS class which will be applied to the {@link #btnEl} when
	 * the cursor is hovering over the button.
	 */
	btnHoverCls : 'x-zarafa-boxfield-item-close-hover',

	/**
	 * @cfg {String/Ext.XTemplate} textTpl <b>Required.</b> The template or template string
	 * which must be applied to the {@link #textEl inner span} of the box.
	 * The template arguments are have been returned by {@link #collectData}.
	 */
	textTpl : undefined,

	/**
	 * Similar to {@link #isDestroying} only this is only set when
	 * {@link #doDestroy} has been called with animation enabled.
	 * Because of the animation, the actual {@link #isDestroying}
	 * will be set a bit later. During this period we should still block
	 * all events.
	 * @property
	 * @type Boolean
	 */
	isAnimDestroying : false,

	/**
	 * @cfg {Boolean} enableTextSelection Enable text selection in the {@link #el},
	 * this will prevent {@link Ext.Element#unselectable} to be called on {@link #el}.
	 */
	enableTextSelection : false,

	/**
	 * @cfg {Number} maximum length of text allowed before truncations,
	 * truncation will be replaced with ellipsis ('...').
	 */
	ellipsisStringStartLength : 30,

	/**
	 * @cfg {Number} maximum length of text allowed after truncations,
	 * truncation will be replaced with ellipsis ('...').
	 */
	ellipsisStringEndLength : 30,

	/**
	 * @constructor
	 * @param config Configuration object
	 */
	constructor : function(config)
	{
		config = config || {};

		Ext.applyIf(config, {
			autoEl : {
				tag: 'li',
				cls: 'x-zarafa-boxfield-item'
			}
		});

		Zarafa.common.ui.Box.superclass.constructor.call(this, config);

		if (Ext.isString(this.textTpl)) {
			this.textTpl = new Ext.XTemplate(this.textTpl, {
				compiled : true,
				ellipsisStringStartLength : this.ellipsisStringStartLength,
				ellipsisStringEndLength : this.ellipsisStringEndLength
			});
		}
	},

	/**
	 * Function called during the {@link #render rendering} of this component. This
	 * will call {@link #renderBox} and {@link #renderButtons} to generate the two
	 * basic elements of which the box consists of.
	 * @param {Ext.Container} ct The container in which the component is being rendered.
	 * @param {NUmber} position The position within the container where the component will be rendered.
	 * @private
	 */
	onRender : function(ct, position)
	{
		Zarafa.common.ui.Box.superclass.onRender.call(this, ct, position);

		if (this.enableTextSelection !== true) {
			this.el.unselectable();
		}

		if (this.enableIcons === true) {
			this.renderIcons();
		}

		this.renderBox();

		if (this.enableButtons === true) {
			this.renderButtons();
		}

		if (Ext.isDefined(this.record)) {
			this.update(this.record);
		}

		this.el.addClassOnOver(this.hoverCls);

		this.mon(this.el, 'click', this.onClick, this);
		this.mon(this.el, 'contextmenu', this.onContextMenu, this);
		this.mon(this.el, 'dblclick', this.onDblClick, this);
	},

	/**
	 * Function called after the {@link #render rendering} of this component.
	 * This will hide the {@link #delBtnEl} when the {@link #editable} flag is false
	 * @param {Ext.Container} ct The container in which the component is being rendered.
	 * @private.
	 */
	afterRender : function(ct)
	{
		Zarafa.common.ui.Box.superclass.afterRender.call(this, ct);
		this.delBtnEl.setVisible(this.editable);
	},
	
	/**
	 * Set the {@link #editable} flag, making the box non-editable.
	 * @param {Boolean} value The new editable status.
	 */
	setEditable : function(value)
	{
		if (this.editable !== value) {
			this.editable = value;
			this.delBtnEl.setVisible(this.editable);
		}
	},

	/**
	 * Renders the {@link #iconEl} into the main {@link #el} element.
	 * This element will be used to render the icon into the box. This function
	 * will only be called when {@link #enableIcons} is true.
	 * @private
	 */
	renderIcons : function()
	{
		this.iconEl = this.el.createChild({
			tag : 'span',
			cls : this.iconCls
		});
	},

	/**
	 * Renders the {@link #textEl} into the main {@link #el} element.
	 * This element will contain the actual contents which is being updated
	 * during {@link #update}.
	 * @Private
	 */
	renderBox : function()
	{
		this.textEl = this.el.createChild({
			tag : 'span',
			cls : this.textCls
		});
	},

	/**
	 * Setup the buttons of the box. This will be done after the template is done rendering. At that
	 * point we can start adding event listeners to the DOM elements. This function will only be
	 * called when {@link #enableButtons} is true.
	 * @private
	 */
	renderButtons: function()
	{
		this.renderRemoveButton();
	},

	/** 
	 * Sets event listeners on the remove button of the box. This is done in a separate function 
	 * than in {@link #initButtons initButtons} so it can easily be overwritten if needed.
	 * @private
	 */
	renderRemoveButton: function()
	{
		this.delBtnEl = this.el.createChild({
			tag : 'span',
			cls : this.btnCls
		});

		this.delBtnEl.addClassOnOver(this.btnHoverCls);
		this.mon(this.delBtnEl, 'click', this.onClickRemove, this);
	},

	/**
	 * Enables the focus on this component.
	 */
	focus: function()
	{
		Zarafa.common.ui.Box.superclass.focus.call(this);
		this.el.addClass(this.focusCls);
		this.parent.doBoxFocus(this);
	},

	/**
	 * Removes the focus on this component.
	 */
	blur: function()
	{
		Zarafa.common.ui.Box.superclass.blur.call(this);
		this.el.removeClass(this.focusCls);
		this.parent.doBoxBlur(this);
	},

	/**
	 * @return {Boolean} Returns true if this box currently has focus.
	 */
	hasFocus : function()
	{
		return this.el.hasClass(this.focusCls);
	},

	/**
	 * Called when the user has clicked on the remove button.
	 * @private
	 */
	onClickRemove: function()
	{
		if (!this.isAnimDestroying && !this.isDestroying) {
			// Set focus on parent element.
			this.parent.focus();
			this.parent.doBoxRemove(this);
		}
	},

	/**
	 * Called when the user has clicked on the element.
	 * @param {Ext.EventObject} e The event object
	 * @private
	 */
	onClick: function(e)
	{
		e.stopEvent();
		if (!this.isAnimDestroying && !this.isDestroying) {
			this.parent.doBoxClick(this);
		}
	},

	/**
	 * Called when the user requested the contextmenu of the element.
	 * @param {Ext.EventObject} e The event object
	 * @private
	 */
	onContextMenu : function(e)
	{
		e.stopEvent();
		if (!this.isAnimDestroying && !this.isDestroying) {
			this.parent.doBoxContextMenu(this);
		}
	},

	/**
	 * Called when the user has doubleclicked on the element.
	 * @param {Ext.EventObject} e The event object
	 * @private
	 */
	onDblClick : function(e)
	{
		e.stopEvent();
		if (!this.isAnimDestroying && !this.isDestroying) {
			this.parent.doBoxDblClick(this);
		}
	},

	/**
	 * Function which can be overriden to provide custom formatting for the given {@link Ext.data.Record}
	 * to the {@link #update} function. The data object returned here is used by the {@link #textTpl template}
	 * to render the contents of the box.
	 * @param {Ext.data.Record} record The record which is going to be rendered
	 * @return {Object} The data object which can be passed to {@link #textTpl}.
	 * @private
	 */
	prepareData: function(record)
	{
		return record.data;
	},

	/**
	 * Function which can be overriden to provide custom icon rendering for the given {@link Ext.data.Record}
	 * to the {@link #iconEl} element. The string returned here is the CSS class which will be set on the
	 * {@link #iconEl}.
	 * @param {Ext.data.Record} record The record which is going to be rendered
	 * @return {String} The CSS class which must be applied to the {@link #iconEl}.
	 * @private
	 */
	prepareIcon : function(record)
	{
		return Zarafa.common.ui.IconClass.getIconClass(record);
	},

	/**
	 * Update the {@link #textEl inner HTML} of this component using the {@link #textTpl template}.
	 * @param {Ext.data.Record} record The Ext.data.Record which data must be applied to the template
	 */
	update: function(record)
	{
		this.textTpl.overwrite(this.textEl, this.prepareData(record));

		if (this.enableIcons === true) {
			var icon_class = this.prepareIcon(record);
			if (!Ext.isEmpty(icon_class)) {
				this.iconEl.addClass(icon_class);
			}
		}
	},

	/**
	 * Event handler which is fired when the box is being resized. This will determine
	 * what the desired with of the text should be, to ensure that the buttons (if available)
	 * will be positioned completely at the right.
	 * @param {Number} adjWidth The box-adjusted width that was set
	 * @param {Number} adjHeight The box-adjusted height that was set
	 * @param {Number} rawWidth The width that was originally specified
	 * @param {Number} rawHeight The height that was originally specified
	 * @private
	 */
	onResize : function(adjWidth, adjHeight, rawWidth, rawHeight)
	{
		if (Ext.isNumber(rawWidth)) {
			if (!Ext.isNumber(adjWidth)) {
				adjWidth = rawWidth;
			}
			adjWidth -= (this.el.getMargins('lr') +
						 this.iconEl.getWidth() + this.iconEl.getFrameWidth('lr') + this.iconEl.getMargins('lr') +
						 this.delBtnEl.getWidth() + this.delBtnEl.getFrameWidth('lr') + this.delBtnEl.getMargins('lr') +
						 this.getResizeEl().getFrameWidth('lr') + this.getResizeEl().getMargins('lr'));

			// Adjust the text width according to expand button as well, if it is defined.
			if(Ext.isDefined(this.expandBtnEl)){
				adjWidth -= (this.expandBtnEl.getWidth() + this.expandBtnEl.getFrameWidth('lr') + this.expandBtnEl.getMargins('lr'));
			}

			this.textEl.setWidth(adjWidth);
		}
	},

	/**
	 * Wrapper around the {@link #destroy} function which supports
	 * the animating of the event.
	 *
	 * @param {Boolean} animate True to enable the animation of the
	 * box removal.
	 */
	doDestroy : function(animate)
	{
		if (animate === true) {
			this.isAnimDestroying = true;
			this.getEl().hide({
				duration: 0.2,
				callback: function() { this.destroy(); },
				scope: this
			});
		} else {
			this.destroy();
		}
	}
});

Ext.reg('zarafa.box', Zarafa.common.ui.Box);
Ext.namespace('Zarafa.common.ui');

/**
 * @class Zarafa.common.ui.BoxField
 * @extends Ext.form.ComboBox
 * @xtype zarafa.boxfield
 *
 * A special input field which displays all contents inside special {@link Zarafa.common.ui.Box}
 * components which are rendered inside this {@link #el}. This makes the user interface nicer,
 * and allows the user to perform extra actions on each box.
 *
 * The contents of the field is based on the contents of the {@link #boxStore} which contains
 * the {@link Ext.data.Record records} which must be rendered as {@link Zarafa.common.ui.Box box}.
 */
Zarafa.common.ui.BoxField = Ext.extend(Ext.form.ComboBox, {
	/**
	 * @cfg {String} boxType The xtype of the component which is used as box.
	 * This defaults to {@link Zarafa.common.ui.Box 'zarafa.box'}.
	 */
	boxType: 'zarafa.box',
	/**
	 * @cfg {Ext.data.Store} boxStore The store which contains all {@link Ext.data.Record records}
	 * which must be rendered as {@link Zarafa.common.ui.Box}. Before rendering, the contents
	 * will be filtered by {@link #filterRecords}.
	 */
	boxStore: undefined,
	/**
	 * @cfg {Object} boxConfig A configuration object which must be applied to all
	 * {@link Zarafa.common.ui.Box box} components which are rendered into this component
	 * by default.
	 */
	boxConfig: undefined,
	/**
	 * The collection of all {@link Zarafa.common.ui.Box Box} components which have been
	 * rendered inside the component.
	 * @property
	 * @type Ext.util.MixedCollection
	 */
	items: undefined,
	/**
	 * @cfg {Number|Char} handleInputKey The input key which must be pressed before the
	 * {@link #handleInput} function is being called. This is normally used to indicate
	 * that the current input is ready to be converted into {@link #items boxes}.
	 * This can be given as a charCode directly, or as a character,
	 * in which case it will be converted to the corresponding code using charCodeAt().
	 */
	handleInputKey: ';',
	/***
	 * @cfg {Boolean} enableComboBox True to enable all combobox functionality on this field.
	 * When this is true, all required fields from the {@link Ext.from.ComboBox Ext.form.ComboBox}
	 * must be configured.
	 */
	enableComboBox : true,
	/**
	 * The {@link Zarafa.common.ui.Box box} which currently has the userfocus.
	 * @property
	 * @type Zarafa.common.ui.Box
	 */
	currentFocus: false,
	/**
	 * A dummy 'a' element which is created as child of the {@link #wrap}. This element is
	 * not hidden, but instead has the dimensions 0x0, otherwise the browser is not able to
	 * put the focus on the element. By using this focus element, we can capture {@link #keyMap} events
	 * and control the different {@link #items boxes}.
	 * @property
	 * @type Ext.Element
	 */
	boxFocusEl : undefined,
	/**
	 * The 'ul' element which contains all rendered {@link #items} and the {@link #inputEl}.
	 * @property
	 * @type Ext.Element
	 */
	wrapBoxesEl : undefined,
	/**
	 * The 'li' element which has been wrapped around {@link #el}, and is to make sure the
	 * input element is positioned correctly within the list. This should always be the
	 * last element inside the {@link #wrapBoxesEl}.
	 * @property
	 * @type Ext.Element
	 */
	inputEl : undefined,
	/**
	 * The {@link Ext.KeyMap} which is used on the {@link #boxFocusEl} to add keyboard
	 * control over the current {@link #items box} selection. This will only be enabled
	 * when this field is {@link #editable editable}.
	 *
	 * @property
	 * @type Ext.KeyMap
	 */
	boxKeyMap : undefined,

	/**
	 * The {@link Ext.KeyMap} which is used on the {@link #el} to add keyboard
	 * control over the input field. This will only be enabled
	 * when this field is {@link #editable editable}.
	 * This KeyMap is currently only used for the handleInputKey,
	 * which causes the BoxField to evaluate its input and convert it to boxes.
	 * This is separated from {@link #specialInputKeyMap} because it needs reliable
	 * character code detection, which can only happen on a 'keypress' event.
	 *
	 * @property
	 * @type Ext.KeyMap
	 */
	inputKeyMap : undefined,

	/**
	 * The {@link Ext.KeyMap} which is used on the {@link #el} to add keyboard
	 * control over the input field. This will only be enabled
	 * when this field is {@link #editable editable}.
	 * This KeyMap is used only for special keys
	 * that do not insert characters (e.g. arrow keys, page up/down, enter, etc.).
	 * These keys are separated from {@link #inputKeyMap} because they have to be caught on
	 * a 'keydown' event.
	 *
	 * @property
	 * @type Ext.KeyMap
	 */
	specialInputKeyMap : undefined,

	/**
	 * The {@link Ext.KeyMap} which is used on the {@link #el} to add keyboard control
	 * for DELETE key. This will only be enabled when {@link #enableComboBox} is configured true.
	 *
	 * @property
	 * @type Ext.KeyMap
	 */
	listKeyMap : undefined,

	/**
	 * @cfg {Number} minInputFieldWidth The minimum number of pixels which must be available
	 * for the {@link #el input field}, to type in the text. The number of pixels should be
	 * sufficient to at least contain 1 character (in any characterset).
	 */
	minInputFieldWidth : 25,

	/**
	 * @cfg {Number} inputFieldHeight The default height for the {@link #el input field}.
	 */
	inputFieldHeight: 20,

	/**
	 * @cfg {Boolean} enableAnim Enable special {@link Ext.Fx FX} effects for
	 * this this field and all container {@link #items boxes}.
	 */
	enableAnim : true,

	/**
	 * @cfg {String} wrapCls The CSS classes which must be applied to the {@link #wrap} element.
	 */
	wrapCls : 'x-form-text x-zarafa-boxfield',

	/**
	 * Instance of the {@link Ext.util.TextMetrics TextMetrics} which is bound to the {@link #innerList}.
	 * This is used to calculate the desired width of text inside the {@link #innerList}. This field
	 * should only be obtained by the {@link #getListTextMetric} function.
	 * @property
	 * @type Ext.util.TextMetrics
	 */
	listTextMetric : undefined,

	/**
	 * Instance of the {@link Ext.util.TextMetrics TextMetrics} which is bound to the {@link #el}.
	 * This is used to calculate the desired width of text inside the {@link #el}. This field
	 * should only be obtained by the {@link #getInputTextMetric} function.
	 * @property
	 * @type Ext.util.TextMetrics
	 */
	inputTextMetric : undefined,

	/**
	 * @cfg {Boolean} listMode True to show all boxes in a list rather then side-by-side.
	 * When enabled, all boxes (and input field) will get the full width of the component.
	 */
	listMode : false,

	/**
	 * @cfg {Number} boxLimit If set, this number will limit the number of boxes which will
	 * be rendered into the field. It will also disable the {@link #inputEl} when the limit
	 * has been reached.
	 */
	boxLimit : undefined,

	/**
	 * @constructor
	 * @param config Configuration object
	 */
	constructor : function(config)
	{
		config = config || {};

		if (config.readOnly === true) {
			config.editable = false;
		}

		Ext.applyIf(config, {
			// Override from Ext.Component
			xtype : 'zarafa.boxfield',
			cls : 'x-zarafa-boxfield-input',
			hideTrigger : true,
			boxMinHeight: 22,
			boxMaxHeight: 50,

			autoHeight: false,
			autoScroll: true,
			
			/*
			 * Override value of Ext.form.TriggerField
			 * Because we don't want TAB-key to blur the element
			 */
			monitorTab: false
		});

		this.addEvents(
			/**
			 * @event boxfocus
			 * Fired when a box receives focus
			 * @param {Zarafa.common.ui.BoxField} boxField Parent of the box
			 * @param {Zarafa.common.ui.Box} box The box that has been focussed
			 * @param {Ext.data.Record} record The record that belongs to the box
			 */
			'boxfocus',
			/**
			 * @event boxblur
			 * Fired when a box looses focus
			 * @param {Zarafa.common.ui.BoxField} boxField Parent of the box
			 * @param {Zarafa.common.ui.Box} box The box that has been blurred
			 * @param {Ext.data.Record} record The record that belongs to the box
			 */
			'boxblur',
			/**
			 * @event boxclick
			 * Fires when the user clicked on a box
			 * @param {Zarafa.common.ui.BoxField} boxField Parent of the box
			 * @param {Zarafa.common.ui.Box} box The box that has been clicked
			 * @param {Ext.data.Record} record The record that belongs to the box
			 */
			'boxclick',
			/**
			 * @event boxdblclick
			 * Fires when the user doubleclicked on a box
			 * @param {Zarafa.common.ui.BoxField} boxField Parent of the box
			 * @param {Zarafa.common.ui.Box} box The box that has been doubleclicked
			 * @param {Ext.data.Record} record The record that belongs to the box
			 */
			'boxdblclick',
			/**
			 * @event boxcontextmenu
			 * Fires when the user requested the contextmenu for a box
			 * @param {Zarafa.common.ui.BoxField} boxField Parent of the box
			 * @param {Zarafa.common.ui.Box} box The box for which the contextmenu was requested
			 * @param {Ext.data.Record} record The record that belongs to the box
			 */
			'boxcontextmenu',
			/**
			 * @event boxadd
			 * Fires when a box has been added
			 * @param {Zarafa.common.ui.BoxField} boxField Parent of the to be added box
			 * @param {Zarafa.common.ui.Box} box The box that has been added
			 * @param {Ext.data.Record} record The record that belongs to the box
			 */
			'boxadd',
			/**
			 * @event boxremove
			 * Fires when a box will be removed.
			 * @param {Zarafa.common.ui.BoxField} boxField Parent of the to be removed box
			 * @param {Zarafa.common.ui.Box} box The box that will be removed
			 * @param {Ext.data.Record} record The record that belongs to the box
			 */
			'boxremove'
		);

		Zarafa.common.ui.BoxField.superclass.constructor.call(this, config);

		this.on('boxremove', this.onBoxRemove, this);
		this.on('boxadd', this.onBoxAdd, this);
	},

	/**
	 * Called by the superclass to initialize the component
	 * @private
	 */
	initComponent : function()
	{
		Zarafa.common.ui.BoxField.superclass.initComponent.call(this);
		this.items = new Ext.util.MixedCollection();

		// Convert input key to character code if necessary
		if (Ext.isString(this.handleInputKey)) {
			this.handleInputKey = this.handleInputKey.charCodeAt(0);
		}
	},

	/**
	 * Called by the superclass to initialize all events for this component.
	 * This is called on {@link #afterrender}.
	 * @private
	 */
	initEvents : function()
	{
		this.mon(this.el, {
			keydown : this.onKeyDownHandler,
			scope   : this
		});

		this.mon(this.getContentTarget(), 'click', this.onContainerClick, this);

		// Initialize keyboard control
		if (this.editable !== false) {
			this.boxKeyMap = this.createBoxKeyMap(this.boxFocusEl);
			this.specialInputKeyMap = this.createSpecialInputKeyMap(this.el);
			this.inputKeyMap = this.createInputKeyMap(this.el);

			// Initialize keyboard control for suggestion list, only
			// if {@link #enableComboBox} configured to true.
			if (this.enableComboBox === true) {
				this.listKeyMap = this.createListKeyMap(this.el);
			}

			/*
			 * Register the event handler for paste events. This will ensure
			 * the input element will be resized when pasting.
			 * Use the Zarafa.core.Event function to ensure compatibility
			 * with all browsers and support for drag & dropping text.
			 */
			Zarafa.core.Events.addPasteEventHandler(this, this.el, this.onPaste, this);
		}

		// Register event handler when listMode is enabled,
		// this will force the boxes to be resized when the parent
		// has been layed out.
		if (this.listMode === true) {
			this.mon(this.ownerCt, 'afterlayout', this.onParentLayout, this);
		}

		// Listen to select events in order to convert them to boxes.
		this.on('select', this.onSelect, this);

		Zarafa.common.ui.BoxField.superclass.initEvents.call(this);
	},

	/**
	 * Sets a new {@link boxStore boxStore}. It will first unregister all events from the old one.
	 * Then the new store will be set, the recipient field will be refreshed and after that the
	 * event listeners will be registered again.
	 * @param {Ext.data.Store} boxStore The store that will be applied to this field.
	 * @param {Boolean} initial True if this function is called from the constructor.
	 */
	setBoxStore: function(boxStore, initial)
	{
		if (this.boxStore === boxStore && initial !== true) {
			return;
		}

		if (this.boxStore) {
			this.mun(this.boxStore, {
				'datachanged': this.onBoxStoreDataChanged,
				'add': this.onBoxStoreAdd,
				'remove': this.onBoxStoreRemove,
				'update': this.onBoxStoreUpdate,
				'clear': this.onBoxStoreClear,
				scope: this
			});
			this.clearBoxes();
		}

		this.boxStore = Ext.StoreMgr.lookup(boxStore);

		if (this.boxStore) {
			this.loadBoxes(this.boxStore);
			this.mon(this.boxStore, {
				'datachanged': this.onBoxStoreDataChanged,
				'add': this.onBoxStoreAdd,
				'remove': this.onBoxStoreRemove,
				'update': this.onBoxStoreUpdate,
				'clear': this.onBoxStoreClear,
				scope: this
			});
		}
	},

	/**
	 * Returns the {@link boxStore boxStore}.
	 * @return {Ext.data.Store} The set {@link boxStore boxStore}.
	 */
	getBoxStore: function()
	{
		return this.boxStore;
	},

	/**
	 * Obtain (and create if required) the {@link #listTextMetric} instance.
	 * @return {Ext.util.TextMetrics} The textmetrics for {@link #innerList}.
	 * @private
	 */
	getListTextMetric : function()
	{
		if (!this.listTextMetric) {
			this.listTextMetric = Ext.util.TextMetrics.createInstance(this.innerList);
		}

		return this.listTextMetric
	},

	/**
	 * Obtain (and create if required) the {@link #listTextMetric} instance.
	 * @return {Ext.util.TextMetrics} The textmetrics for {@link #el}.
	 * @private
	 */
	getInputTextMetric : function()
	{
		if (!this.inputTextMetric) {
			this.inputTextMetric = Ext.util.TextMetrics.createInstance(this.el);
		}

		return this.inputTextMetric;
	},

	/**
	 * The function that should handle the trigger's click event.
	 *
	 * This method is disabled if {@link #enableComboBox} is set to false.
	 *
	 * @param {EventObject} e
	 * @private
	 */
	onTriggerClick : function()
	{
		if (this.enableComboBox === true) {
			Zarafa.common.ui.BoxField.superclass.onTriggerClick.apply(this, arguments);
		}
	},

	/**
	 * Execute a query to filter the dropdown list.  Fires the {@link #beforequery} event prior to performing the
	 * query allowing the query action to be canceled if needed.
	 *
	 * This method is disabled if {@link #enableComboBox} is set to false.
	 *
	 * @param {String} query The SQL query to execute
	 * @param {Boolean} forceAll <tt>true</tt> to force the query to execute even if there are currently fewer
	 * characters in the field than the minimum specified by the <tt>{@link #minChars}</tt> config option.  It
	 * also clears any filter previously saved in the current store (defaults to <tt>false</tt>)
	 * @private
	 */
	doQuery : function()
	{
		if (this.enableComboBox === true) {
			Zarafa.common.ui.BoxField.superclass.doQuery.apply(this, arguments);
		}
	},

	/**
	 * Check if the value typed into the field matches a record from the store.
	 *
	 * This method is disabled if {@link #enableComboBox} is set to false.
	 *
	 * @private
	 */
	assertValue : function()
	{
		if (this.enableComboBox === true) {
			Zarafa.common.ui.BoxField.superclass.assertValue.apply(this, arguments);
		}
	},

	/**
	 * Event handler which is called when any key related to navigation (arrows, tab, enter, esc,
	 * etc.) is pressed.
	 * @param {Ext.data.Record} record
	 * @private
	 */
	onSelect: function(record)
	{
		this.collapse();
		this.el.dom.value = '';
		this.sizeInputfield();
		this.store.removeAll(true);
		this.handleSelection(record);
		this.lastQuery = '';
	},

	/**
	 * Calculate the desired width of the dropdown {@link #list}.
	 * This should only be called when {@link #list} has been filled with the
	 * data which is going to be shown, as we calculate the desired width
	 * based on the current contents of the {@link #list}.
	 *
	 * This means that if a subclass overrides {@link #tpl} and adds all sort
	 * of extra text, this function should still be relatively accurate in the
	 * required width.
	 *
	 * @private
	 */
	getDesiredListWidth : function()
	{
		var metric = this.getListTextMetric();
		var desiredWidth = 0;

		if (this.innerList.dom.children.length > 0) {
			for (var i = 0, len = this.innerList.dom.children.length; i < len; i++) {
				var child = Ext.fly(this.innerList.dom.children[i]);
				var width = child.getPadding('lr') + metric.getWidth(child.dom.innerHTML);

				if (width > desiredWidth) {
					desiredWidth = width;
				}
			};
		} else {
			desiredWidth = metric.getWidth(this.innerList.dom.innerHTML);
		}

		return desiredWidth;
	},

	/**
	 * Called just before the {@link #store} is going to call 'list' to the server.
	 * This will show a {@link #loadingText} in the {@link #list.
	 *
	 * We make sure that we resize our {@link #list} here using {@link #getDesiredListWidth}
	 * to guarentee that our {@link #loadingText} and loading image receive sufficient
	 * space.
	 *
	 * @private
	 */
	onBeforeLoad : function()
	{
		Zarafa.common.ui.BoxField.superclass.onBeforeLoad.apply(this, arguments);

		var width = this.getDesiredListWidth();
		this.restrictWidth(width);
	},

	/**
	 * Called when the {@link #store} has loaded and we can fill the dropdown list
	 * with the suggestionlist. Based on the records in the store we will also
	 * determine the {@link #getDesiredListWidth} desired width of the list.
	 *
	 * Note that this has been called after the dropdown box has been filled with
	 * the contents. Hence it is safe to call {@link #getDesiredListWidth} here.
	 *
	 * @param {Ext.data.Store} store The store which fired the event
	 * @param {Ext.data.Record|Array} records The records which were loaded from the server
	 * @param {Object} options The options object which were provided during load
	 * @private
	 */
	onLoad : function(store, records, options)
	{
		if (this.hasFocus && (this.store.getCount() > 0 || this.listEmptyText)) {
			var desiredWidth = this.getDesiredListWidth();

			// getDesiredListWidth obtains the desired width for
			// the 'innerList'. The 'list' however is slightly larger.
			desiredWidth += this.list.getFrameWidth('lr');

			// When there are sufficient results, a scrollbar is shown.
			// Add the required space.
			desiredWidth += Ext.getScrollBarWidth();

			// Resize the dropdownbox.
			this.restrictWidth(desiredWidth);
		}

		Zarafa.common.ui.BoxField.superclass.onLoad.apply(this, arguments);
	},

	/**
	 * Change the width of the {@link #list} and {@link #innerList}. We never
	 * fill out the dropdown list to the complete width of the field, and the
	 * list itself is aligned to the {@link #el} element (which can be positioned
	 * on various locations in the field depending on the number of {@link #items boxes}
	 * which have been rendered.
	 *
	 * The width must fall within the the {@link #minListWidth} and the {@link #container}
	 * width range. Depending on the width, and the position of the input element, we
	 * also have to force the list a few pixels to the left of the {@link #el} if by
	 * that we can prevent the list to exceed the limits of the field itself.
	 *
	 * @param {Number} width The desired width of the list
	 * @private
	 */
	restrictWidth : function(width)
	{
		var target = this.getEl();
		var container = this.getResizeEl();

		var offset = target.getOffsetsTo(container);
		var fieldWidth = container.getWidth();

		var desiredOffsetLeft = 0;
		var desiredOffsetBottom = target.getFrameWidth('b');
		var desiredWidth = width;

		// Establish the actual width we can apply for this list.
		if (this.minListWidth > desiredWidth) {
			desiredWidth = this.minListWidth;
		} else if (fieldWidth < desiredWidth) {
			desiredWidth = fieldWidth;
		}

		// We now have the width we need. Since we are going to align
		// the list to the input field, we must check the available width,
		// from the start of the input field to the end of the box field.
		var availableWidth = container.getWidth() - offset[0];
		if (availableWidth < desiredWidth) {
			desiredOffsetLeft = availableWidth - desiredWidth;
		}

		// But what if so much size was needed, that the offset would cause
		// the dropdown box to appear to the left of the boxfield...
		// Lets limit that case as well, and make sure we decrease the
		// width.
		if (desiredOffsetLeft > offset[0]) {
			desiredWidth -= desiredOffsetLeft - offset[0];
			desiredOffsetLeft = -offset[0];
		}

		this.list.setWidth(desiredWidth);
		this.innerList.setWidth(desiredWidth - this.list.getFrameWidth('lr'));
		this.listAlign = [ 'tl-bl', [ desiredOffsetLeft, desiredOffsetBottom ] ];
	},

	/**
	 * Function called during the {@link #render rendering} of this component. This will
	 * wrap {@link #el} into a 'li' element which is saved as {@link #inputEl}. This element
	 * will then be wrapped into a 'ul' element which is saved as {@link #wrapBoxesEl}, and
	 * will later contain all {@link #items} as well.
	 * @param {Ext.Container} ct The container in which the component is being rendered.
	 * @param {NUmber} position The position within the container where the component will be rendered.
	 * @private
	 */
	onRender : function(ct, position)
	{
		Zarafa.common.ui.BoxField.superclass.onRender.call(this, ct, position);

		// Apply some extra CSS classes to the wrap element.
		if (!Ext.isEmpty(this.wrapCls)) {
			this.wrap.addClass(this.wrapCls);
		}

		// If autoHeight is set to false we need to set our own height. Otherwise the CSS
		// class x-form-text will set it to a default height.
		if(this.autoHeight === false){
			if(Ext.isDefined(this.height))
				this.wrap.setHeight(this.height);
		}else{
			 this.wrap.applyStyles('height: auto;');
		}

		// Create a focus element which is used for focussing a box.
		this.boxFocusEl = this.wrap.createChild({
			tag : 'a',
			href : '#',
			// Disable tab-index, and position it somewhere where it cannot be seen
			// by the user. This will make the element completely invisible for the
			// user while we can benefit from the focus capabilities.
			tabindex: -1,
			style: 'position: absolute; left:-10000px; top:-10000px;'
		});

		// Wraps the items (boxes) in an UL-tag
		this.wrapBoxesEl = this.el.wrap({
			tag: 'ul'
		});

		this.inputEl = this.el.wrap({
			tag: 'li',
			cls: 'x-zarafa-boxfield-input-item'
		});

		if (this.border === false) {
			this.el.addClass('x-zarafa-boxfield-input-noborder');
		}

		// If a boxStore was previously configured, we
		// can now set it (this will cause the boxes to
		// be rendered).
		if (Ext.isDefined(this.boxStore)) {
			this.setBoxStore(this.boxStore, true);
		}
	},

	/**
	 * Callback function from {@link #setEditable} when the {@link #editable} state
	 * has been changed. This will go over all {@link #items} and change the
	 * {@link Zarafa.common.ui.Box#editable} state on there.
	 * @private
	 */
	updateEditState : function()
	{
		Zarafa.common.ui.BoxField.superclass.updateEditState.apply(this, arguments);

		this.inputEl.setVisible(this.editable);
		this.items.each(function(box) {
			box.setEditable(this.editable);
		});
	},

	/**
	 * Return the {@link Ext.Element} which acts as as the content element. Normally
	 * this is {@link #el}, but as this combobox is more a container, we return {@link #wrap}.
	 *
	 * @return {Ext.Element} The content element
	 * @private
	 */
	getContentTarget : function()
	{
		return this.wrap;
	},

	/**
	 * Called after the component is resized. This will automatically update
	 * the sizing of the {@link #sizeInputfield input} and {@link #sizeContainer container}.
	 *
	 * @param {Number} adjWidth The box-adjusted width that was set
	 * @param {Number} adjHeight The box-adjusted height that was set
	 * @param {Number} rawWidth The width that was originally specified
	 * @param {Number} rawHeight The height that was originally specified
	 * @private
	 */
	onResize: function(w, h, rw, rh)
	{
		Zarafa.common.ui.BoxField.superclass.onResize.call(this, w, h, rw, rh);

		this.sizeInputfield();
		this.sizeContainer();
	},

	/**
	 * Apply auto-sizing to the component. When {@link #autoHeight} is true,
	 * we apply automatic height calculations to make sure the component
	 * is always correctly sized. When {@link #boxMaxHeight} is also provided,
	 * we check if the limit has been exceeded and display the
	 * {@link #setAutoScroll scrollbar} if needed.
	 *
	 * @private
	 */
	sizeContainer : function()
	{
		if (!this.rendered || this.autoHeight === false) {
			return false;
		}

		var target = this.getResizeEl();
		var outerHeight = target.getHeight();
		var innerHeight = target.dom.scrollHeight;
		var doLayout = false;

		if (outerHeight > this.boxMaxHeight) {
			// The height of the box exceeds the maximim height,
			// enable the scrollbar and scroll to the bottom.
			target.setHeight(this.boxMaxHeight);
			if (this.initialConfig.autoScroll !== false) {
				this.setAutoScroll(true);
				target.scrollTo('top', outerHeight);
			}
			doLayout = true;
		} else if (innerHeight <= outerHeight) {
			// The scroll height is smaller then the height of the
			// box, this means we do not need the scrollbar anymore.
			target.setHeight('auto');
			if (this.initialConfig.autoScroll !== false) {
				this.setAutoScroll(false);
			}
			doLayout = true;
		}

		// FIXME: This is ugly, we shouldn't need to force a
		// relayout action on our grandparent...
		if (doLayout === true && this.ownerCt.ownerCt) {
			this.ownerCt.ownerCt.doLayout();
		}
	},

	/**
	 * Called when the {@link #ownerCt owner} has been layed out, this
	 * will obtain the new desired {@link #getDesiredBoxWidth boxWidth}
	 * and applies them to all available {@link #items boxes}.
	 * @private
	 */
	onParentLayout : function()
	{
		var width = this.getDesiredBoxWidth();
		if (Ext.isDefined(width)) {
			this.items.each(function(box) { box.setWidth(width); });
		}
	},

	/**
	 * Returns the desired width for the {@link #items boxes} which should be applied
	 * whenever the container resizes, or a new box is created. When {@link #listMode}
	 * is disabled, this function returns undefined, to ensure dynamic sizing of the box.
	 * @return {Number} The desired width for all boxes
	 * @private
	 */
	getDesiredBoxWidth : function()
	{
		if (this.listMode === true) {
			var target = this.getResizeEl();
			return target.getWidth() - target.getFrameWidth('lr') - this.el.getFrameWidth('lr');
		} else {
			return undefined;
		}
	},

	/**
	 * Apply auto-sizing to the {@link #el input element}.
	 * Whenever the contents of the input field has changed, or when
	 * the width of the entire component has changed, we must recalculate
	 * the desired width of the input element. Normally it should be only
	 * a few pixels wider then the typed text requires, with a maximum
	 * of the current width of the entire component (to prevent a
	 * horizontal scrollbar to appear while typing).
	 *
	 * @private
	 */
	sizeInputfield: function()
	{
		if (!this.rendered) {
			return false;
		}

		var target = this.getResizeEl();
		var width;

		if (this.listMode === true) {
			width = target.getWidth() - target.getFrameWidth('lr') - this.el.getFrameWidth('lr');
		} else {
			var metric = this.getInputTextMetric();

			// Calculate the desired width of the component based on the
			// required width for the input text.
			var value = Ext.util.Format.htmlEncode(this.el.dom.value);
			width = metric.getWidth(value);

			// Add spacing to the input field. This will ensure the
			// the input box will always have the correct minimum size.
			width += this.minInputFieldWidth;

			// Ensure the width does not exceed the component width.
			width = Math.min(width, target.getWidth());
		}

		// Set the inputfield to the calculated width
		this.el.setSize(width, this.inputFieldHeight - this.el.getBorderWidth('tb'));

		// Store the current height, this way we can detect if the input
		// field has been wrapped to the next line after the resize.
		var oldHeight = target.getHeight();

		// Check if linewrapping has occurred, and update the container accordingly.
		var newHeight = target.getHeight();
		if (this.listMode === true || oldHeight !== newHeight) {
			this.sizeContainer();
		}

		// When scrolling is enabled, we must always scroll to the
		// bottom of the container to ensure the input field is
		// fully visible.
		target.scrollTo('top', newHeight);
	},

	/**
	 * Called to handle the input when the user presses the handleInputKey or another trigger makes
	 * this component need to handle the input. Has to be overwritten to implement the desired
	 * behavior and the creation of the correct type of record.
	 * @param {String} value The value from the input field
	 * @protected
	 */
	handleInput : Ext.emptyFn,

	/**
	 * Called to handle a selection from the dropdown list. This function needs to
	 * convert the selected record into a record for the {@link #boxStore}.
	 * Has to be overwritten to implement the desired behavior of creation of the correct type
	 * of record.
	 * @param {Ext.data.Record} record The record which was selected from {@link #store}
	 * @protected
	 */
	handleSelection : Ext.emptyFn,

	/**
	 * Check {@link #el} for input and call {@link #handleInput} to convert
	 * the input into a new {@link Zarafa.common.ui.Box Box}.
	 * @private
	 */
	convertInputToBox : function()
	{
		var value = this.el.getValue();

		if (!Ext.isEmpty(value)) {
			this.el.dom.value = '';
			this.sizeInputfield();
			this.store.removeAll(true);
			this.handleInput(value);
		}
	},

	/**
	 * Called to filter out records before they are added to this field.
	 * This will loop over all records and call {@link #filterRecord} to
	 * check if the record should be visible or not.
	 * @param {Ext.data.Store} store The store
	 * @param {Ext.data.Record|Array} records The records to filter
	 * @return {Ext.data.Record|Array} The records which are visible
	 * @protected
	 */
	filterRecords: function(store, records)
	{
		var ret = [];

		for (var i = 0, len = records.length; i < len; i++) {
			var record = records[i];
			if (this.filterRecord(record)) {
				ret.push(record);
			}
		}

		return ret;
	},

	/**
	 * Called by {@link #filterRecords} to check if the given record
	 * must be visible in the field or not.
	 * @param {Ext.data.Record} record The record to filter
	 * @return {Boolean} True if the record should be visible, false otherwise
	 * @protected
	 */
	filterRecord : function(record)
	{
		return true;
	},

	/**
	 * Find the {@link Zarafa.common.ui.Box} which belongs to the given record.
	 * @param {Ext.data.Record} record The record for which the box is searched
	 * @return {Zarafa.common.ui.Box} The box which belongs to the given record
	 */
	getBoxForRecord : function(record)
	{
		return this.items.find(function(box) { return box.record === record;  });
	},

	/**
	 * Call before the field is being blurred. When we still have an
	 * {@link #currentFocus boxfocus} it must be removed now.
	 *
	 * @private
	 */
	beforeBlur : function()
	{
		Zarafa.common.ui.BoxField.superclass.beforeBlur.apply(this, arguments);
		this.convertInputToBox();
		this.boxBlur();
	},

	/**
	 * Put the focus on a particular box. This will call {@link Zarafa.common.ui.Box#blur}
	 * on the {@link #currentFocus currently focussed} element. And then call
	 * {@link Zarafa.common.ui.Box#focus} on the new box which will be set in {@link #currentFocus}.
	 *
	 * @param {Zarafa.common.ui.Box} box The box to put the focus on
	 * @private
	 */
	boxFocus : function(box)
	{
		if (this.currentFocus) {
			this.currentFocus.blur();
		}

		this.currentFocus = box;

		if (this.currentFocus) {
			this.currentFocus.focus();
		}
	},

	/**
	 * Remove the focus from the {@link #currentFocus currently focussed} box.
	 * This will call {@link Zarafa.common.ui.Box#blur} on the box, and then
	 * reset {@link #currentFocus}.
	 * @private
	 */
	boxBlur : function()
	{
		if (this.currentFocus) {
			this.currentFocus.blur();
		}

		this.currentFocus = false;
	},

	/**
	 * Try to focus this component.
	 * @param {Boolean} selectText (optional) If applicable, true to also select the text in this component
	 * @param {Boolean/Number} delay (optional) Delay the focus this number of milliseconds (true for 10 milliseconds)
	 * @return {Ext.Component} this
	 */
	focus : function(selectText, delay)
	{
		if (delay) {
			this.focusTask = new Ext.util.DelayedTask(this.focus, this, [selectText, false]);
			this.focusTask.delay(Ext.isNumber(delay) ? delay : 10);
			return;
		}
		if (this.rendered && !this.isDestroyed) {
			this.inputFocus(undefined, selectText);
		}
		return this;
	},

	/**
	 * Put the focus on the {@link #el Input element}, when a box is already
	 * {@link #currentFocus focussed} it will be {@link #boxBlur blurred} after
	 * which the focus is put on the {@link #el Input element}.
	 *
	 * When provided, the new caret position for the input field can also be provided.
	 * @param {Number} caretPos (optional) The desired caret position
	 * @private
	 */
	inputFocus : function(caretPos, selectText)
	{
		this.boxBlur();
		this.el.focus();
		if (!Ext.isEmpty(caretPos)) {
			Zarafa.core.Util.setCaretPosition(this.el, caretPos);
		}
		if (selectText === true) {
			this.el.dom.select();
		}
	},

	/**
	 * Remove the focus from the {@link #el} element.
	 * @private
	 */
	inputBlur : function()
	{
		this.el.blur();
	},

	/**
	 * Event handler which is fired when the user clicked anywhere
	 * on the {@link #container}, but not on a specific {@link #items box}.
	 * In this case, we redirect our focus to the input element.
	 * @private
	 */
	onContainerClick : function(e, element)
	{
		this.collapse();
		if (this.editable) {
			if (element === this.el.dom) {
				this.inputFocus();
			} else {
				// If element is visible then and only we can set focus on the element.
				if(this.el.isVisible()) {
					// Stop event propagation.
					e.stopEvent();
					this.inputFocus(this.el.dom.value.length);
				}
			}
		} else if (!this.currentFocus) {
			this.boxFocus(this.items.last());
		}
	},

	/**
	 * Create the {@link Ext.KeyMap} for an {@link Ext.Element} which
	 * serves as the focus element for the {@link #items boxes}.
	 * @param {Ext.Element} focusEl The element which acts as the focusElement for the boxes.
	 * @return {Ext.KeyMap} The keymap for the focus element.
	 * @private
	 */
	createBoxKeyMap : function(focusEl)
	{
		var bindings = [{
			key: [
				Ext.EventObject.ENTER
			],
			handler: this.onBoxKeyEnter,
			scope: this
		},{
			key: [
				Ext.EventObject.UP,
				Ext.EventObject.PAGE_UP
			],
			handler: this.onBoxKeyUp,
			scope: this
		},{
			key: [
				Ext.EventObject.DOWN,
				Ext.EventObject.PAGE_DOWN
			],
			handler: this.onBoxKeyDown,
			scope: this
		},{
			key: [
				Ext.EventObject.LEFT
			],
			handler: this.onBoxKeyLeft,
			scope: this
		},{
			key: [
				Ext.EventObject.RIGHT
			],
			handler: this.onBoxKeyRight,
			scope: this
		},{
			key: [
				Ext.EventObject.HOME
			],
			handler: this.onBoxKeyHome,
			scope: this
		},{
			key: [
				Ext.EventObject.END
			],
			handler: this.onBoxKeyEnd,
			scope: this
		},{
			key: [
				Ext.EventObject.BACKSPACE
			],
			handler: this.onBoxKeyBackspace,
			scope: this
		},{
			key: [
				Ext.EventObject.DELETE
			],
			handler: this.onBoxKeyDelete,
			scope: this
		}];

		// disable modifier keys in the bindings
		for(var i = 0, len = bindings.length; i < len; i++) {
			var binding = bindings[i];

			Ext.apply(binding, {
				shift: false,
				alt: false,
				ctrl: false
			});
		}

		return new Ext.KeyMap(focusEl, bindings);
	},

	/**
	 * Key handler for the {@link Ext.EventObject.ENTER ENTER} event
	 * for the {@link #boxFocusEl Box focus element}.
	 *
	 * This will perform the same action as {@link #doBoxDblClick}.
	 *
	 * @param {Number} key The key which was pressed
	 * @param {Ext.EventObject} e The event object which fired the event
	 * @private
	 */
	onBoxKeyEnter : function(key, e)
	{
		e.stopEvent();

		this.doBoxDblClick(this.currentFocus);
	},

	/**
	 * Key handler for the {@link Ext.EventObject.UP} event
	 * for the {@link #boxFocusEl Box focus element}.
	 *
	 * This will select the box left to the currently selected box.
	 * If this is the first box of the {@link #items} array, the focus
	 * on the items is lost, and the {@link #el input element} is focussed.
	 *
	 * @param {Number} key The key which was pressed
	 * @param {Ext.EventObject} e The event object which fired the event
	 * @private
	 */
	onBoxKeyUp : function(key, e)
	{
		this.onBoxKeyLeft(key, e);
	},

	/**
	 * Key handler for the {@link Ext.EventObject.DOWN} event
	 * for the {@link #boxFocusEl Box focus element}.
	 *
	 * This will select the box right to the currently selected box.
	 * If this is the last box of the {@link #items} array, the focus
	 * on the items is lost, and the {@link #el input element} is focussed.
	 *
	 * @param {Number} key The key which was pressed
	 * @param {Ext.EventObject} e The event object which fired the event
	 * @private
	 */
	onBoxKeyDown : function(key, e)
	{
		this.onBoxKeyRight(key, e);
	},

	/**
	 * Key handler for the {@link Ext.EventObject.LEFT} event
	 * for the {@link #boxFocusEl Box focus element}.
	 *
	 * This will select the box left to the currently selected box.
	 * If this is the first box of the {@link #items} array, the focus
	 * remains on the current box.
	 *
	 * @param {Number} key The key which was pressed
	 * @param {Ext.EventObject} e The event object which fired the event
	 * @private
	 */
	onBoxKeyLeft : function(key, e)
	{
		e.stopEvent();

		var index = this.items.indexOf(this.currentFocus);
		if (index !== 0) {
			this.boxFocus(this.items.itemAt(index - 1));
		}
	},

	/**
	 * Key handler for the {@link Ext.EventObject.RIGHT} event
	 * for the {@link #boxFocusEl Box focus element}.
	 *
	 * This will select the box right to the currently selected box.
	 * If this is the last box of the {@link #items} array, the focus
	 * on the items is lost, and the {@link #el input element} is focussed.
	 *
	 * @param {Number} key The key which was pressed
	 * @param {Ext.EventObject} e The event object which fired the event
	 * @private
	 */
	onBoxKeyRight : function(key, e)
	{
		e.stopEvent();

		var index = this.items.indexOf(this.currentFocus);
		if (index !== this.items.getCount() - 1) {
			this.boxFocus(this.items.itemAt(index + 1));
		} else if (this.editable) {
			this.inputFocus(0);
		}
	},

	/**
	 * Key handler for the {@link Ext.EventObject.HOME} event
	 * for the {@link #boxFocusEl Box focus element}.
	 *
	 * This will select the first item in the {@link #items} array.
	 *
	 * @param {Number} key The key which was pressed
	 * @param {Ext.EventObject} e The event object which fired the event
	 * @private
	 */
	onBoxKeyHome : function(key, e)
	{
		e.stopEvent();

		var item = this.items.first();
		if (!item.hasFocus()) {
			this.boxFocus(item);
		}
	},

	/**
	 * Key handler for the {@link Ext.EventObject.END} event
	 * for the {@link #boxFocusEl Box focus element}.
	 *
	 * This will select the last item in the {@link #items} array.
	 *
	 * @param {Number} key The key which was pressed
	 * @param {Ext.EventObject} e The event object which fired the event
	 * @private
	 */
	onBoxKeyEnd : function(key, e)
	{
		e.stopEvent();

		var item = this.items.last();
		if (!item.hasFocus()) {
			this.boxFocus(item);
		} else if (this.editable) {
			this.inputFocus(this.el.dom.value.length);
		}
	},

	/**
	 * Key handler for the {@link Ext.EventObject.BACKSPACE} event
	 * for the {@link #boxFocusEl Box focus element}.
	 *
	 * This will delete the currently selected item from the field,
	 * and will select the item to the left of the deleted item.
	 *
	 * @param {Number} key The key which was pressed
	 * @param {Ext.EventObject} e The event object which fired the event
	 * @private
	 */
	onBoxKeyBackspace : function(key, e)
	{
		e.stopEvent();

		if (this.editable !== false) {
			var index = this.items.indexOf(this.currentFocus);

			this.doBoxRemove(this.currentFocus);

			if (index > 0) {
				this.boxFocus(this.items.itemAt(index - 1));
			} else if (this.items.getCount() > 0) {
				this.boxFocus(this.items.itemAt(0));
			} else {
				this.inputFocus();
			}
		}
	},

	/**
	 * Key handler for the {@link Ext.EventObject.DELETE} event
	 * for the {@link #boxFocusEl Box focus element}.
	 *
	 * This.will delete the currently selected item from the field,
	 * and will select the item to the right of the deleted item.
	 *
	 * @param {Number} key The key which was pressed
	 * @param {Ext.EventObject} e The event object which fired the event
	 * @private
	 */
	onBoxKeyDelete : function(key, e)
	{
		e.stopEvent();

		if (this.editable !== false) {
			var index = this.items.indexOf(this.currentFocus);

			this.doBoxRemove(this.currentFocus);

			if (index < this.items.getCount()) {
				this.boxFocus(this.items.itemAt(index));
			} else {
				this.inputFocus(0);
			}
		}
	},

	/**
	 * Create the {@link Ext.KeyMap} for the {@link #el}.
	 * This KeyMap is only used for the {@link handleInputKey}, which needs its character code detected.
	 * That is why the 'keypress' event is used instead of the default 'keydown',
	 * which does not report character codes.
	 *
	 * @param {Ext.Element} focusEl The element which acts as the focusElement for the input element.
	 * @return {Ext.KeyMap} The keymap for the focus element.
	 * @private
	 */
	createInputKeyMap : function(focusEl)
	{
		return new Ext.KeyMap(focusEl, [{
			key: [
				this.handleInputKey
			],
			shift: false,
			alt: false,
			ctrl: false,
			handler: this.onInputKey,
			scope: this
		}], 'keypress');
	},

	/**
	 * Create the {@link Ext.KeyMap} for the {@link #el} to handle
	 * various key board controls for suggestion {@link #list}.
	 *
	 * @param {Ext.Element} element The {@link Zarafa.common.ui.BoxField#el combo box} element.
	 * @return {Ext.KeyMap} The keymap for the {@link Zarafa.common.ui.BoxField#el combo box} element.
	 * @private
	 */
	createListKeyMap : function(element)
	{
		// Listen to DELETE key event which remove record from suggestion list.
		return new Ext.KeyMap(element, {
			key: [
				Ext.EventObject.DELETE
			],
			handler: this.onListKeyDelete,
			scope: this,
			shift: false,
			alt: false,
			ctrl: false
		});
	},

	/**
	 * Create the {@link Ext.KeyMap} for the {@link #el}.
	 * This KeyMap is used for special keys, such as enter, page up/down, arrows, etc.
	 *
	 * @param {Ext.Element} focusEl The element which acts as the focusElement for the input element.
	 * @return {Ext.KeyMap} The keymap for the focus element.
	 * @private
	 */
	createSpecialInputKeyMap : function(focusEl)
	{
		var bindings = [{
			key: [
				Ext.EventObject.ENTER
			],
			handler: this.onInputKeyEnter,
			scope: this
		},{
			key: [
				Ext.EventObject.TAB
			],
			handler: this.onInputKeyTab,
			scope: this
		},{
			key: [
				Ext.EventObject.UP,
				Ext.EventObject.PAGE_UP
			],
			handler: this.onInputKeyUp,
			scope: this
		},{
			key: [
				Ext.EventObject.LEFT
			],
			handler: this.onInputKeyLeft,
			scope: this
		},{
			key: [
				Ext.EventObject.HOME
			],
			handler: this.onInputKeyHome,
			scope: this
		},{
			key: [
				Ext.EventObject.BACKSPACE
			],
			handler: this.onInputKeyBackspace,
			scope: this
		}];

		// disable modifier keys in the bindings
		for(var i = 0, len = bindings.length; i < len; i++) {
			var binding = bindings[i];

			Ext.apply(binding, {
				shift: false,
				alt: false,
				ctrl: false
			});
		}

		return new Ext.KeyMap(focusEl, bindings);
	},

	/**
	 * Generic {@link Ext.Element#keydown} event handler for the {@link #el input element}.
	 * This handler is run for all keys which were not handled by the {@link #inputKeyMap}.
	 * This will simply {@link #sizeInputfield resize} the {@link #el input element}.
	 *
	 * @param {Number} key The key which was pressed
	 * @param {Ext.EventObject} e The event object which fired the event
	 * @private
	 */
	onKeyDownHandler: function(e)
	{
		// Don't resize the inputfield on special keys (arrow-keys, backspace, etc)
		// for some weird reason these keys would trigger the input field to become
		// larger, while no input is being added...
		if (!e.isSpecialKey()) {
			this.collapse();
			this.sizeInputfield();
		}
	},

	/**
	 * Key handler for the {@link #handleInputKey} event
	 * for the {@link #el input element}.
	 *
	 * This will clear  the {@link #el} element, and send the
	 * old value to {@link #handleInput} to create a new {@link #items box}.
	 *
	 * @param {Number} key The key which was pressed
	 * @param {Ext.EventObject} e The event object which fired the event
	 * @private
	 */
	onInputKey : function(key, e)
	{
		e.stopEvent();
		this.convertInputToBox();
	},

	/**
	 * Key handler for the {@link Ext.EventObject.ENTER} event
	 * for the {@link #el input element}.
	 *
	 * This will clear  the {@link #el} element, and send the
	 * old value to {@link #handleInput} to create a new {@link #items box}.
	 *
	 * @param {Number} key The key which was pressed
	 * @param {Ext.EventObject} e The event object which fired the event
	 * @private
	 */
	onInputKeyEnter : function(key, e)
	{
		// Make sure we are not expanded, because in that
		// case the ENTER command is reserved for selecting
		// an item from the dropdown list.
		if (!(this.isExpanded() && this.store.getCount() > 0)) {
			e.stopEvent();
			this.convertInputToBox();
		}
	},

	/**
	 * Key handler for the {@link Ext.EventObject.TAB} event
	 * for the {@link #el input element}.
	 *
	 * @param {Number} key The key which was pressed
	 * @param {Ext.EventObject} e The event object which fired the event
	 * @private
	 */
	onInputKeyTab : function(key, e)
	{
		//If the user entered something, we should prevent default tab behavior
		if (!Ext.isEmpty(this.getValue())) {
			e.stopEvent();

			if (!(this.isExpanded() && this.store.getCount() > 0)){
			    this.convertInputToBox();
			}
		}else{
			//Trigger a blur to remove the focus class from the element
			this.triggerBlur();
		}
	},

	/**
	 * Key handler for the {@link Ext.EventObject.UP} event
	 * for the {@link #el input element}.
	 *
	 * When the cursor is at the most-left position of the {@link #el input element}
	 * then the box to the left of the input element is selected.
	 *
	 * @param {Number} key The key which was pressed
	 * @param {Ext.EventObject} e The event object which fired the event
	 * @private
	 */
	onInputKeyUp : function(key, e)
	{
		this.onInputKeyLeft(key, e);
	},

	/**
	 * Key handler for the {@link Ext.EventObject.LEFT} event
	 * for the {@link #el input element}.
	 *
	 * When the cursor is at the most-left position of the {@link #el input element}
	 * then the box to the left of the input element is selected.
	 *
	 * @param {Number} key The key which was pressed
	 * @param {Ext.EventObject} e The event object which fired the event
	 * @private
	 */
	onInputKeyLeft : function(key, e)
	{
		var caret = Zarafa.core.Util.getSelectionRange(this.el);
		if (caret.start === 0 && caret.end === 0 && this.items.getCount() > 0) {
			e.stopEvent();
			this.boxFocus(this.items.last());
		}
	},

	/**
	 * Key handler for the {@link Ext.EventObject.HOME} event
	 * for the {@link #el input element}.
	 *
	 * When the cursor is at the most-left position of the {@link #el input element}
	 * then the first box of {@link #items} is selected.
	 *
	 * @param {Number} key The key which was pressed
	 * @param {Ext.EventObject} e The event object which fired the event
	 * @private
	 */
	onInputKeyHome : function(key, e)
	{
		var caret = Zarafa.core.Util.getSelectionRange(this.el);
		if (caret.start === 0 && caret.end === 0 && this.items.getCount() > 0) {
			e.stopEvent();
			this.boxFocus(this.items.first());
		}
	},

	/**
	 * Key handler for the {@link Ext.EventObject.BACKSPACE event
	 * for the {@link #el input element}.
	 *
	 * When the cursor is at the most-left position of the {@link #el input element}
	 * then the box to the left of the input element is selected (so that pressing
	 * 'backspace' again will delete it.
	 *
	 * @param {Number} key The key which was pressed
	 * @param {Ext.EventObject} e The event object which fired the event
	 * @private
	 */
	onInputKeyBackspace : function(key, e)
	{
		var caret = Zarafa.core.Util.getSelectionRange(this.el);
		if (caret.start === 0 && caret.end === 0 && this.items.getCount() > 0) {
			e.stopEvent();
			this.boxFocus(this.items.last());
		}
	},

	/**
	 * Key handler for the {@link Ext.EventObject.DELETE} event
	 * for the {@link #el element} defined by {@link Ext.KeyMap}.
	 *
	 * This will remove current record identified by {@link #selectedIndex} from
	 * {@link Zarafa.common.recipientfield.data.SuggestionListStore SuggestionListStore}.
	 *
	 * @param {Number} key The key which was pressed
	 * @param {Ext.EventObject} e The event object which fired the event
	 * @private
	 */

	onListKeyDelete : function(key, e)
	{
		var store = this.getStore();

		if (store.getCount() !== 0) {
			// Prevent default browser behaviour of deleting characters from input text
			// as we are going to use this event to delete records from list.
			e.stopEvent();
			store.removeAt(this.selectedIndex);

			// If deleted item is the last one in the list than previous item of the deleted item will be selected.
			if (store.getCount() ==  this.selectedIndex) {
				this.selectPrev();
			} else {
				this.select(this.selectedIndex);
			}
			this.restrictHeight();
		}
	},

	/**
	 * Event handler for the 'paste' event on the {@link #el input element}.
	 * This is fired by the special {@link Zarafa.core.Event#addPasteEventHandler}
	 * event handler. This will resize the input field, to fit the field to the
	 * new contents.
	 * @private
	 */
	onPaste : function()
	{
		this.sizeInputfield();
	},

	/**
	 * Event handler for the {@link Ext.data.Store#datachanged} event from the {@link #boxStore}.
	 * This will {@link #clearBoxes clear all boxes} and then {@link #loadBoxes rebuild them}.
	 * @param {Ext.data.Store} store The store which fired the event
	 * @private
	 */
	onBoxStoreDataChanged : function(store)
	{
		this.clearBoxes();
		this.loadBoxes(store);
	},

	/**
	 * Event handler for the {@link Ext.data.Store#clear} event from the {@link #boxStore}.
	 * This will {@link #clearBoxes clear all boxes}.
	 * @private
	 */
	onBoxStoreClear : function()
	{
		this.clearBoxes();
	},

	/**
	 * Event handler for the {@link Ext.data.Store#add} event from the {@link #boxStore}.
	 * This will check if the provided records should be {@link #filterRecords shown} inside
	 * this field, and those that are not filtered will be {@link #addBox added}.
	 * @param {Ext.data.Store} store The store which fired the event
	 * @param {Ext.data.Record|Array} records The records which were added
	 * @param {Number} index The index position at which the records where added
	 * @private
	 */
	onBoxStoreAdd: function(store, records, index)
	{
		if (!Ext.isArray(records)) {
			records = [ records ];
		}

		records = this.filterRecords(store, records);

		for (var i = 0; i < records.length; i++) {
			this.addBox(records[i]);
		}
	},

	/**
	 * Event handler for the {@link Ext.data.Store#remove} event from the {@link #boxStore}.
	 * This will {@link #removeBox remove} the box which is attached to the given record.
	 * @param {Ext.data.Store} store The store which fired the event
	 * @param {Ext.data.Record} record The record which was removed
	 * @param {Number} index The position from where the record was removed
	 * @private
	 */
	onBoxStoreRemove: function(store, record, index)
	{
		this.removeBox(record);
	},

	/**
	 * Event handler for the {@link Ext.data.Store#update} event from the {@link #boxStore}.
	 * This will {@link Zarafa.common.ui.Box#update update} the box with the new
	 * {@link Ext.data.Record#data record data}.
	 * When the field is updated, the modified record may have changed whether it passes the field's filter or not.
	 * Therefore, the function adds or removes this record's box.
	 * @param {Ext.data.Store} store The store which fired the event
	 * @param {Ext.data.Record} record THhe record which was updated
	 * @private
	 */
	onBoxStoreUpdate: function(store, record)
	{
		if (this.filterRecord(record)) {
			var box = this.getBoxForRecord(record);
			if (box) {
				// record passes filter and has a box - just update
				box.update(record);
			} else {
				// record passes the filter and does not have a box in this field yet - add it
				this.addBox(record);
			}
		} else {
			// if this record does not pass the filter, remove its box
			this.removeBox(record);
		}
	},

	/**
	 * Callback function from {@link Zarafa.common.ui.Box} which indicates
	 * that the box has been clicked. This will {@link #boxFocus focus the
	 * box}, and fire the {@link #boxclick} event.
	 * @param {Zarafa.common.ui.Box} box The box which called this function
	 */
	doBoxClick: function(box)
	{
		this.boxFocus(box);

		this.fireEvent('boxclick', this, box, box.record);
	},

	/**
	 * Callback function from {@link Zarafa.common.ui.Box} which indicates
	 * that the box has been doubleclicked. This will fire the
	 * {@link #boxdblclick} event.
	 * @param {Zarafa.common.ui.Box} box The box which called this function
	 */
	doBoxDblClick : function(box)
	{
		this.fireEvent('boxdblclick', this, box, box.record);
	},

	/**
	 * Callback function from {@link Zarafa.common.ui.Box} which indicates
	 * that the contextmenu has been requested for the given box. This will
	 * {@link #boxFocus focus the box}, and fire the {@link #boxcontextmenu} event.
	 * @param {Zarafa.common.ui.Box} box The box which called this function
	 */
	doBoxContextMenu : function(box)
	{
		this.boxFocus(box);

		this.fireEvent('boxcontextmenu', this, box, box.record);
	},

	/**
	 * Callback function from {@link Zarafa.common.ui.Box} which indicates that
	 * the box has received the focus. This will focus the {@link #boxFocusEl}
	 * and {@link Ext.Element#scrollIntoView scroll the box into view}. Finally
	 * the {@link #boxfocus} event.
	 * @param {Zarafa.common.ui.Box} box The box which called this function
	 */
	doBoxFocus : function(box)
	{
		// It could happen a box is being focussed without
		// the field being focussed.
		if (this.hasFocus !== true) {
			this.onFocus();
		}

		this.boxFocusEl.focus();
		box.getPositionEl().scrollIntoView(this.getContentTarget());

		this.fireEvent('boxfocus', this, box, box.record);
	},

	/**
	 * Callback function from {@link Zarafa.common.ui.Box} which indicates
	 * that the box has been blurred. This will blur the {@link #boxFocuEl}
	 * and will fire the {@link #boxblur} event.
	 * @param {Zarafa.common.ui.Box} box The box which called this function
	 */
	doBoxBlur : function(box)
	{
		this.boxFocusEl.blur();

		this.fireEvent('boxblur', this, box, box.record);
	},

	/**
	 * Callback function from {@link Zarafa.common.ui.Box} which indicates that
	 * the box is being removed by the user. This will fire the {@link #boxremove}
	 * event.
	 * @param {Zarafa.common.ui.Box} box The box which called this function
	 */
	doBoxRemove: function(box)
	{
		this.fireEvent('boxremove', this, box, box.record);
	},

	/**
	 * Event handler for the {@link #boxremove} event. This will remove
	 * the record belonging to the box from the {@link #boxStore}. Then
	 * {@link #sizeContainer} is called. This is needed because the
	 * removal of the box might have triggered a resize of the {@link #wrap}
	 * and thus we might no longer need the scrollbars.
	 * @param {Zarafa.common.ui.BoxField} field The field which has fired the event
	 * @param {Zarafa.common.ui.Box} box The box which has been removed
	 * @param {Ext.data.Record} record The record which belongs to the given box
	 * @private
	 */
	onBoxRemove: function(field, box, record)
	{
		this.boxStore.remove(record);
		this.sizeContainer();
		if (this.boxStore.getCount() < this.boxLimit && this.initialConfig.readOnly !== true) {
			this.setReadOnly(false);
			this.inputFocus();
		}
	},

	/**
	 * Event handler for the {@link #boxadd} event. This will
	 * {@link #sizeContainer resize the container}. This is needed because
	 * the new box, might have triggered a resize of the {@link #wrap}
	 * and thus we might need scrollbars.
	 * @param {Zarafa.common.ui.BoxField} field The field which has fired the event
	 * @param {Zarafa.common.ui.Box} box The box which has been added
	 * @param {Ext.data.Record} record The record which belongs to the given box
	 * @private
	 */
	onBoxAdd : function(field, box, record)
	{
		this.sizeContainer();
		if (this.boxStore.getCount() >= this.boxLimit) {
			this.setReadOnly(true);
		}
	},

	/**
	 * Adds a box to the field. It will instantiate the {@link Zarafa.common.ui.Box Box} component that will
	 * render the box according to the data in the supplied record. It will render the box inside
	 * the wrapBoxesEl. After rendering the box it will add the box to the {@link #items items} and
	 * fire the {@link #boxadd} event.
	 * @param {Ext.data.Record} record The record to add as a box into the field
	 * @private
	 */
	addBox: function(record)
	{
		/*
		 * Create the configuration object for the Box. The this.boxConfig is used as default and we
		 * apply the record and renderTo properties onto that object.
		 */
		var configObj = {}
		Ext.apply(configObj, {
			xtype : this.boxType,
			parent : this,
			record : record,
			editable : this.editable,
			width : this.getDesiredBoxWidth()
		}, this.boxConfig);

		var box = Ext.create(configObj);
		box.render(this.wrapBoxesEl, this.items.length);

		this.items.add(box);

		this.fireEvent('boxadd', this, box, record);
	},

	/**
	 * Removes a {@link Zarafa.common.ui.Box Box} from the field.
	 * @param {Ext.data.Record} record The record which belongs to the box which must be removed.
	 * @private
	 */
	removeBox : function(record)
	{
		v