/*
 *  SPDX-FileCopyrightText: 2008 Boudewijn Rempt <boud@valdyas.org>
 *
 *  SPDX-License-Identifier: GPL-2.0-or-later
 */
#ifndef KIS_GENERATOR_LAYER_H_
#define KIS_GENERATOR_LAYER_H_

#include "kis_selection_based_layer.h"
#include <kritaimage_export.h>
#include <KisDelayedUpdateNodeInterface.h>

#include <QScopedPointer>

class KisFilterConfiguration;

/**
 * A generator layer is a special kind of layer that can be prefilled
 * with some pixel pattern generated by a KisGenerator plugin.
 * A KisGenerator is similar to a filter, but doesn't take
 * input pixel data and creates new pixel data.
 *
 * It is not possible to destructively paint on a generator layer.
 *
 * XXX: what about threadedness?
 */
class KRITAIMAGE_EXPORT KisGeneratorLayer
        : public KisSelectionBasedLayer,
          public KisDelayedUpdateNodeInterface
{
    Q_OBJECT

public:
    /**
     * Create a new Generator layer with the given configuration
     * and selection. Note that the selection will be _copied_
     * (using COW, though).
     */
    KisGeneratorLayer(KisImageWSP image, const QString &name, KisFilterConfigurationSP  kfc, KisSelectionSP selection);
    KisGeneratorLayer(const KisGeneratorLayer& rhs);
    ~KisGeneratorLayer() override;

    KisNodeSP clone() const override {
        return KisNodeSP(new KisGeneratorLayer(*this));
    }

    void setFilter(KisFilterConfigurationSP filterConfig) override;
    /**
     * Changes the filter configuration without triggering an update.
     */
    void setFilterWithoutUpdate(KisFilterConfigurationSP filterConfig);

    bool accept(KisNodeVisitor &) override;
    void accept(KisProcessingVisitor &visitor, KisUndoAdapter *undoAdapter) override;

    QIcon icon() const override;
    KisBaseNode::PropertyList sectionModelProperties() const override;

    /**
     * re-run the generator. This happens over the bounds
     * of the associated selection.
     */
    void update();
    /**
     * re-runs the generator with the specified configuration.
     * Used for previewing the layer inside the stroke.
     */
    void previewWithStroke(const KisStrokeId stroke);

    using KisSelectionBasedLayer::setDirty;
    void setDirty(const QVector<QRect> &rects) override;
    /**
     * Updates the selected tiles without triggering the update job.
     */
    void setDirtyWithoutUpdate(const QVector<QRect> &rects);
    void setX(qint32 x) override;
    void setY(qint32 y) override;

    void resetCache() override;

    void forceUpdateTimedNode() override;
    bool hasPendingTimedUpdates() const override;

private Q_SLOTS:
    void slotDelayedStaticUpdate();

private:
    /**
     * Injects render jobs into the given stroke.
     */
    void requestUpdateJobsWithStroke(const KisStrokeId stroke, const KisFilterConfigurationSP configuration);
    /**
     * Resets the projection cache without triggering the update job.
     */
    void resetCacheWithoutUpdate();

public:
    // KisIndirectPaintingSupport
    KisLayer* layer() {
        return this;
    }

private:
    struct Private;
    const QScopedPointer<Private> m_d;
};

#endif

