/*
 *   Copyright (C) 2012 Aleix Pol Gonzalez <aleixpol@blue-systems.com>
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU Library/Lesser General Public License
 *   version 2, or (at your option) any later version, as published by the
 *   Free Software Foundation
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details
 *
 *   You should have received a copy of the GNU Library/Lesser General Public
 *   License along with this program; if not, write to the
 *   Free Software Foundation, Inc.,
 *   51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */

#include <QApplication>
#include <KAboutData>
#include <KDBusService>
#include <KLocalizedString>
#include <QCommandLineParser>
#include <qwindow.h>
#include "MuonDiscoverMainWindow.h"
#include <MuonBackendsFactory.h>
#include "MuonVersion.h"

int main(int argc, char** argv)
{
    QApplication app(argc, argv);
    app.setWindowIcon(QIcon::fromTheme("muondiscover"));
    app.setAttribute(Qt::AA_DontCreateNativeWidgetSiblings);
    KLocalizedString::setApplicationDomain("muon-discover");
    KAboutData about("muondiscover", i18n("Muon Discover"), version, i18n("An application explorer"),
                     KAboutLicense::GPL, i18n("© 2010-2014 Muon Development Team"));
    about.addAuthor(i18n("Aleix Pol Gonzalez"), QString(), "aleixpol@blue-systems.com");
    about.addAuthor(i18n("Jonathan Thomas"), QString(), "echidnaman@kubuntu.org");
    about.setProductName("muon/discover");
    KAboutData::setApplicationData(about);

    KDBusService service(KDBusService::Unique);
    MuonDiscoverMainWindow *mainWindow = nullptr;
    {
        QCommandLineParser parser;
        parser.addOption(QCommandLineOption("application", i18n("Directly open the specified application by its package name."), "name"));
        parser.addOption(QCommandLineOption("mime", i18n("Open with a program that can deal with the given mimetype."), "name"));
        parser.addOption(QCommandLineOption("category", i18n("Display a list of entries with a category."), "name"));
        parser.addOption(QCommandLineOption("mode", i18n("Open Muon Discover in a said mode. Modes correspond to the toolbar buttons."), "name"));
        parser.addOption(QCommandLineOption("listmodes", i18n("List all the available modes.")));
        MuonBackendsFactory::setupCommandLine(&parser);
        about.setupCommandLine(&parser);parser.addHelpOption();
        parser.addVersionOption();
        parser.process(app);
        about.processCommandLine(&parser);
        MuonBackendsFactory::processCommandLine(&parser);

        mainWindow = new MuonDiscoverMainWindow;
        QObject::connect(&app, SIGNAL(aboutToQuit()), mainWindow, SLOT(deleteLater()));

        if(parser.isSet("application"))
            mainWindow->openApplication(parser.value("application"));
        else if(parser.isSet("mime"))
            mainWindow->openMimeType(parser.value("mime"));
        else if(parser.isSet("category"))
            mainWindow->openCategory(parser.value("category"));
        else if(parser.isSet("mode"))
            mainWindow->openMode(parser.value("mode").toLocal8Bit());
        else if(parser.isSet("listmodes")) {
            fprintf(stdout, "%s", qPrintable(i18n("Available modes:\n")));
            foreach(const QString& mode, mainWindow->modes())
                fprintf(stdout, " * %s\n", qPrintable(mode));
            return 0;
        }
    }

    mainWindow->show();
    QObject::connect(mainWindow->windowHandle(), &QWindow::visibleChanged, [](bool b){
        if(!b)
            QCoreApplication::instance()->quit();
    });

    return app.exec();
}
