/*
 * vartuple.h
 *
 * Copyright (c) 1998 Michael Kropfberger <michael.kropfberger@gmx.net>
 *
 * Requires the Qt widget libraries, available at no cost at
 * http://www.troll.no/
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */



#ifndef __VARTUPLE_H__
#define __VARTUPLE_H__

#include <qobject.h>
#include <qstring.h>

#define SEPARATOR ':'


/** 
  * @short VarTuple is the class-representation of one of the @ref Serialize objects' attributes (variables). 
  * It provides you the basic functions for manipulating/retrieving the
  * VarTag (a descriptive name of the variable), the variables value and type.
  * @author Michael Kropfberger <michael.kropfberger@gmx.net>
 **/
class VarTuple : public QObject
{ Q_OBJECT

 public:
/**
  * different Constructors for all VarTypes, VarTag only or
  * totally empty @ref #VarTuple
 **/
    VarTuple(QObject *parent=0, const char *name=0);

    VarTuple(QString varTag, QObject *parent=0, const char *name=0);

    VarTuple(QString varTag, int var, QObject *parent=0
                                          , const char *name=0);

    VarTuple(QString varTag, long var, QObject *parent=0
                                          , const char *name=0);

    VarTuple(QString varTag, QString var, QObject *parent=0
                                          , const char *name=0);

    VarTuple(QString varTag, float var, QObject *parent=0
                                          , const char *name=0);


    VarTuple(QString varTag, double var, QObject *parent=0
                                          , const char *name=0);


    /*******************************************************************
    *  extracts the VarTag-Field
    *  This is useful for building a hashtable (qdict) on all variables
    *  @return the VarTag-Field
    **/
    QString tag() const ;

    /*******************************************************************
    *  these are the known types of variables.
    *
    *  If you think, some type is missing, try to build your type with
    *  multipe base-types from @ref #VarType or, if you think a very
    *  common type is missing, just send me an email!  :)
    **/
    enum VarType {UNKNOWN=0, INT=1, LONG=2, STRING=3, FLOAT=4, DOUBLE=5};

    /*******************************************************************
    *  extracts the VarType-Field
    *  @return the VarType-Field as a VarTuple::VarType enum
    **/
    VarType type() const;

    /*******************************************************************
    *  extracts the Varibles value and returns it (ignoring its type)
    *  as a QString.
    *  @return the Value-Field as a QString
    **/
    QString valueToString() const; 

    /*******************************************************************
    *  sets the internal Value and Type of this VarTuple-object
    *  according to the given var
    *  @param var is the variable to set. The type is detected automatically.
    **/
    void setValue(int var);
    void setValue(long var);
    void setValue(QString var);
    void setValue(float var);
    void setValue(double var);

    /*******************************************************************
    *  sets the internal Value and Type of this VarTuple-object
    *  according to the given var
    *  @param var is the variable to write in. The type is detected
    *              automatically.
    *             If not, you have to store it into a known type and then
    *             convert it by normal ways. Don't use
    *             casts, as @ref #storeValueTo can't store the value back to
    *             a casted value!
    **/
    void storeValueTo(int &var);
    void storeValueTo(long &var);
    void storeValueTo(QString &var);
    void storeValueTo(float &var);
    void storeValueTo(double &var);
    
   /*******************************************************************
    *  generates a QString looking like VarTag:VarType:Value.
    *
    *  With @ref #data you can easily build all needed variable-Tuples
    *  for @ref #objState
    *
    *  @return the VarTag:VarType:Value QString
    **/
    QString data() const; 

   /*******************************************************************
    *  sets the Contents according to the given QString looking like 
    *  VarTag:VarType:Value
    *
    *  With @ref #setData you can easily retrieve variable-Tuples from
    *  serialized objects
    *  for @ref Serialize setObjState
    *
    *  @param varTuple the VarTag:VarType:Value QString
    **/
    void setData(QString varTuple);

 protected:
  
 private:

   void init();
   
   VarTuple::VarType  _type;
   QString            _value;
   QString            _tag;
   bool               OK;   

};

#endif
