/////////////////////////////////////////////////////////////////////////////
//
// KPlugin - dynamicly loaded object
//
// Time-stamp: <97/05/06 03:52:07 vels>
// Copyright (c) VDOnet Corp. 1996
//
/////////////////////////////////////////////////////////////////////////////
  
#ifndef _KPLUGIN_H_
#define _KPLUGIN_H_

#include <qobject.h>
#include <qstrlist.h>

class KPluginManager;

/// KPlugin: Pluggable dynamicly loadable objects.
/** 
  This is a base class for all KDE plugins.
  To create your plugin, you must derive your class from KPlugin and
  implement all or some of the virtual methods listed below.
  */
class KPlugin : public QObject
{
  Q_OBJECT

public:
  /// Constructor.
  /**
     Initialize plugin.
  */
  KPlugin();

  /// Desstructor.
  /**
     Destroy plugin.
  */
  ~KPlugin();


  /// Get/Set plugin type.
  /**
     Return/Change plugin type. Type can be one of the following -
     KPlugin::Input, KPlugin::Output or KPlugin::Filter.
     Used by KPluginManager.
    */
  enum { Input, Output, Filter };
  void setType(int type) { m_type = type; }
  int  type()            { return m_type; }

  /// Get/Set plugin name.
  /**
     Return/Change human readable plugin name.
    */
  const char *name()             { return m_name; }
  void setName(const char *name) { m_name = name; }

  /// Set plugin manager.
  /**
     Remember manager that owns this plugin.
     Used by KPluginManager.
    */
  void setManager(KPluginManager *mng) { m_mng = mng; }

  /// Start working.
  /**
     This method should only be implemented by input (or output ?) plugins.
     Plugin should start working. I really encourage everyone to use
     asynchnonous approach (timers). At least until thread safe Qt is
     ready :)
    */
  virtual int start();

  /// Stop working.
  /**
     This method should only be implemented by input (or output ?) plugins.
     It is only meaningfull when plugin is working asynchoniously.
     Plugin should stop all data processing and prepare to die :).
    */
  virtual void stop();

  /// Input or Output plugin has finished it's job.
  /**
     Input and Output plugin should call this function when they have
     finished working.
  */
  void finished();

  /// Input plugin has finished it's job.
  /**
     This method should only be implemented by output plugin, working in
     asynchronous mode. It should process the remaining data and call
     finished().
     If this function is not implemented it's assumed that output
     finishes working as soon as input finishes working (blocking mode).
  */
  virtual void inputFinished();


  /// Set plugin option.
  /**
     First arument is option name, second is value to set this option to.
     Returns 0 if set was ok, -1 on error (no such option)
     The list of plugin options may be obtained with options() method.
    */
  virtual int set(const char *option, void *value);

  /// Get plugin option.
  /**
     First arument is option name, second is a placeholder for returned value.
     Returns 0 if get was ok, -1 on error (no such option)
    */
  virtual int get(const char *option, void *value);

  /// Get list of options this plugin has.
  /**
     Returns the list of registered options. May be useful for applications
     like Plugin Manager (kplugmng).
    */
  virtual QStrList *options(void);

  /// Get list of input MIME types.
  /**
     Returns the list of MIME types accepted by the plugin.
     For example, HTML output plugin may accept the following MIME types:
     text/plain, text/html, image/gif, image/jpeg ...
    */
  virtual QStrList *inputMIMETypes(void);

  /// Get list of output MIME types.
  /**
     Returns the list of MIME types produced by the plugin.
     For example, URL input plugin may produce the following mimetypes.
     text/plain, text/html, image/gif, image/jpeg ...
    */
  virtual QStrList *outputMIMETypes(void);

protected:  
  //
  // Helper functions for KDE Plugins. I encourage you to use them, as this
  // is more convenient and should be more bug free :)
  //
  
  /// Register input MIME type
  /**
     Add new MIME type to the list of input MIME types.
     KPluginManager uses this info to create chains.
    */
  void registerInputMIMEType(const char *type);
  
  /// Register output MIME type
  /**
     Add new MIME type to the list of output MIME types.
     KPluginManager uses this info to create chains.
    */
  void registerOutputMIMEType(const char *type);

  /// Pass data to appropriate input of the plugin.
  /**
     First argument is an index. 'Specifies' one of the registered MIME types.
     E.g. first registered MIME type has index 0 and so on.
     See StdoutPlugin for example of usage.
    */
  virtual int callInputMemberFunction(int index, void *data);

  /// Pass data forward in chain.
  /**
     Called when the previous plugin in chain has data to pass forward.
     First argument is a (MIME) type of passed data. The following plugin
     will receive this data (callInputMemberFunction()).
     See DirPlugin for example of usage.
    */
  virtual int output(const char *mime, void *data);

  /// Register plugin option.
  /**
     Add option to the list of plugin options. Each option has two actions
     associated with it. First one is a 'set' action and second is a 'get'
     action.
    */
  void registerOption(const char *option);

  /// Call action associated with registered option.
  /**
     Called when user wants to access some plugin option (get/set).
     First argument is an index of the action. In general every even index 
     specifies the 'set' action and every odd index specifies the 'get' action.
     For example: the index 0 refers to 'set' action of first registered
     option and naturally the index 1 - to 'get' action of this option.
     See DirPlugin for example of usage.
    */
  virtual int callOptionMemberFunction(int index, void *arg);

protected:
  QStrList  *m_inputMIMETypes;
  QStrList  *m_outputMIMETypes;
  QStrIList *m_options;

private:
  KPluginManager *m_mng;
  int             m_type;
  QString         m_name;
};

#endif // _KPLUGIN_H_
