/*  This file is part of the KDE libraries
    Copyright (C) 1998 Jrgen Hochwald (juergen.hochwald@privat.kkf.net

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this library; see the file COPYING.  If not, write to
    the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
    Boston, MA 02111-1307, USA.*/



#ifndef KWR_H
#define KWR_H
#include <stdlib.h>
#include <qframe.h>
#include <qpainter.h>
#include <qpushbutton.h>
#include <qapp.h>  // only for TRUE and FALSE declaration.....
#include <kaccel.h>

#define KWR_VALID 0     // gltige Daten
#define KWR_NODIR 1     // keine Richtung / umlaufend
#define KWR_INVALID -1  // ngultige/keine Daten

/**
* @short KWindRose
* This is KWindrose v0.15 a compass like widget for KDE.
* KWindRose is a widget to enter an angle from 0 to 360 degrees interactivly.
* It displays a compass like windrose with an arrow which can be rotated with 
* the mouse or by keypresses.
*
* <h4>Mouse</h4>
* Click on the widget, the arrow rotates to this direction. Dragging with the
* mouse rotates the arrow continously. If a <i>resolution</i> is defiened, the
* arrow rotates in steps by multiples of the resolution value. The direction
* changes in steps by clicking the buttons in the lower corners. The minus
* button rotates left, the plus button rotates right. Each step has
* <i>resolution</i> width or one, if no stepping is active.<br>
* Clicking in the round center of the windrose sets the <i>nodir</i> state.
* This means than it is windless or no data is avaible. The windrose displays
* no arrow.
*
* <h4>Keyboard</h4>
* The following keys are assigned:
* <table border=1 cellspacing="0" cellpadding="2">
* <tr><th>Key</th><th>Angle</th><th>Function</th></tr>
* <tr><td>+,PgDn,<br>Downarrow,<br>Rightarrow</td><td>+Step</td><td>Steps forward by resolution</td>></tr>
* <tr><td>-,PgUp,<br>Leftarrow,<br>Uparrow</td><td>-Step</td><td>Steps backward by resolution</td>></tr>
* <tr><td>8</td><td>0</td><td>Jumps to N</td>></tr>
* <tr><td>9</td><td>45</td><td>Jumps to NO</td>></tr>
* <tr><td>6</td><td>90</td><td>Jumps to O</td>></tr>
* <tr><td>3</td><td>135</td><td>Jumps to SO</td>></tr>
* <tr><td>2</td><td>180</td><td>Jumps to S</td>></tr>
* <tr><td>1</td><td>225</td><td>Jumps to SW</td>></tr>
* <tr><td>4</td><td>270</td><td>Jumps to W</td>></tr>
* <tr><td>7</td><td>315</td><td>Jumps to NW</td>></tr>
* <tr><td>5</td><td> - </td><td>Sets KWR_NODIR state</td>></tr>
* <tr><td>Space</td><td> X </td><td>Sets KWR_INVALID state</td>></tr>
* </table><br>
* @author Jrgen Hochwald
* @version 0.15 (22.4.2000)
*/
class KWindRose:public QFrame {
   Q_OBJECT
public:
      /**
      * Constructs a widget with the following defaults:
      * <ul>
      * <li> Direction = 0 (North)
      * <li> Width = 0.2
      * <li> Details = 2 (eight heads)
      * <li> Arrow type 1 (one color with tail)
      * <li> Editable = TRUE
      * <li> Resolution = 1 (1 per step)
      * <li> colors c1=(128,128,255)  c2=(192,255,255)  c3=(16,16,16)  c4=(64,64,64)
      * </ul>
      */
   // Pfeiltypen
   //#define ARROW1 0
   //#define ARROW2 1
   enum {ARROW1, ARROW2};

   KWindRose( QWidget* parent=0, const char* name=0,WFlags f=0 ,bool allowLines=TRUE);
   ~KWindRose(void);
      /**
      * Set the colors for the widget<br>
      * <ul>
      * <li> c1 = dark part of the windrose
      * <li> c2 = light part of the windrose
      * <li> c3 = dark part of the arrow
      * <li> c4 = light part of the arrow (only for ARROW2)
      * </ul>
      */
   void setColors(QColor c1, QColor c2, QColor c3, QColor c4);
      /**
      * Return the current color settings. <i>Num</i> determinines the the color
      * to return (1=c1, 2=c2...)
      */
   QColor getColor(int Num);
      /**
      * Sets a single color. <i>Num</i> specifies the color to set (1=c1, 2=c2...)
      */
   void setColor(int Num, QColor Col);
      /**
      * Set the width of the rose heads. Lower value make thinner heads.
      * The range is limited from 0.03 to 0.4.
      */
   void setRoseWidth(float w);
      /**
      * Sets the number of heads of the windrose.
      * <table border=1 cellspacing="0" cellpadding="2">
      * <tr><th>Detail</th><th>Angle</th><th># Heads</th><th></th></tr>
      * <tr><td>1</td><td>90   </td><td> 4</td><td>only main directions</td></tr>
      * <tr><td>2</td><td>45   </td><td> 8</td><td></td></tr>
      * <tr><td>3</td><td>22.5 </td><td>16</td><td></td></tr>
      * <tr><td>4</td><td>11.25</td><td>32</td><td>most details</td></tr>
      * </table><br>
      */
   void setDetails(int d);
      /**
      * Returns the current detaillevel (see above)
      */
   int  getDetails(void);
      /**
      * Sets the wind direction. The value is truncated from 0 to 360 degrees.
      */
   void setDirection(float Angle);
      /**
      * Returns the wind direction, range 0 <= dir < 360.
      */
   float  getDirection(void);
      /**
      * Sets the shape for the arrow. Two arrows are avaible:
      * <table border=1 cellspacing="0" cellpadding="2">
      * <tr><td>ARROW1</td><td>one color, head with tail</td></tr>
      * <tr><td>ARROW2</td><td>two colors, only head</td></tr>
      * </table><br>
      */
   void setArrowType(int T);
      /**
      * Returns the current arrow shape number.
      */
   int  getArrowType(void) {return ArrowType;};
      /**
      * Sets the editable flag. If <i>E</i> if TRUE, the direction is interactive
      * editable.
      */
   void setEditable(int E);
      /**
      * Returns the editable flag (TRUE or FALSE).
      */
   int getEditable(void) { return Editable;};
      /**
      * Sets the resolution of the wind direction in interactive mode. Only
      * directions that are multiple of the resolution are adjustable. If the
      * value is zero or less, no stepping is done.
      */
   void setResolution(float Res);
      /**
      * Returns the current resolution value.
      */
   float getResolution(void) { return Resolution;};

   static const char* getDirText(int Item);
      /**
      * <a name="getdirtext"></a>
      * Returns a text containing the wind direction for filling a listbox.
      * There are 33 Items defiended with the following contents:
      * <table border=1 cellspacing="0" cellpadding="2">
      * <tr><th>Index</th><th>Dir(Text)</th><th>Dir ()</th><th>Description</th></tr>
      * <tr><td> 0</td><td>N</td>   <td>0</td>     <td>North</td></tr>
      * <tr><td> 1</td><td>N_E</td> <td>11.25</td> <td>North to East</td></tr>
      * <tr><td> 2</td><td>NNE</td> <td>22.5</td>  <td>North North East</td></tr>
      * <tr><td> 3</td><td>NE_N</td><td>33.75</td> <td>North East to North</td></tr>
      * <tr><td> 4</td><td>NE</td>  <td>45</td>    <td>North East</td></tr>
      * <tr><td> 5</td><td>NE_E</td><td>56.25</td> <td>North East to East</td></tr>
      * <tr><td> 6</td><td>ENE</td> <td>67.5</td>  <td>East North East</td></tr>
      * <tr><td> 7</td><td>E_N</td> <td>78.75</td> <td>East to North</td></tr>
      * <tr><td> 8</td><td>E</td>   <td>90</td>    <td>East</td></tr>
      * <tr><td> 9</td><td>E_S</td> <td>101.25</td><td>East to South</td></tr>
      * <tr><td>10</td><td>ESE</td> <td>112.5</td> <td>East South East</td></tr>
      * <tr><td>11</td><td>SE_E</td><td>123.75</td><td>South East to East</td></tr>
      * <tr><td>12</td><td>SE</td>  <td>135</td>   <td>South East</td></tr>
      * <tr><td>13</td><td>SE_S</td><td>146.25</td><td>South East to South</td></tr>
      * <tr><td>14</td><td>SSE</td> <td>157.5</td> <td>South South East</td></tr>
      * <tr><td>15</td><td>S_E</td> <td>168.75</td><td>South to East</td></tr>
      * <tr><td>16</td><td>S</td>   <td>180</td>   <td>South</td></tr>
      * <tr><td>17</td><td>S_W</td> <td>191.25</td><td>South to West</td></tr>
      * <tr><td>18</td><td>SSW</td> <td>202.5</td> <td>South South West</td></tr>
      * <tr><td>19</td><td>SW_S</td><td>213.75</td><td>South West to South</td></tr>
      * <tr><td>20</td><td>SW</td>  <td>225</td>   <td>South West</td></tr>
      * <tr><td>21</td><td>SW_W</td><td>236.25</td><td>South West to West</td></tr>
      * <tr><td>22</td><td>WSW</td> <td>247.5</td> <td>West South West</td></tr>
      * <tr><td>23</td><td>W_S</td> <td>258.75</td><td>West to South</td></tr>
      * <tr><td>24</td><td>W</td>   <td>270</td>   <td>West</td></tr>
      * <tr><td>25</td><td>W_N</td> <td>281.25</td><td>West to North</td></tr>
      * <tr><td>26</td><td>WNW</td> <td>292.5</td> <td>West North West</td></tr>
      * <tr><td>27</td><td>NW_W</td><td>303.75</td><td>North West to West</td></tr>
      * <tr><td>28</td><td>NW</td>  <td>315</td>   <td>North West</td></tr>
      * <tr><td>29</td><td>NW_N</td><td>326.25</td><td>North West to North</td></tr>
      * <tr><td>30</td><td>NNW</td> <td>337.5</td> <td>North North West</td></tr>
      * <tr><td>31</td><td>N_W</td> <td>348.75</td><td>North to West</td></tr>
      * <tr><td>32</td><td> "" </td><td>-</td>     <td>No Direction/Windless</td></tr>
      * </table><br>
      */
   static int directionToIndex(float Dir,int Res);
      /**
      * Returns the index in the stringlist for the direction and resolotion.
      * <i>Dir</i> is a direction (0-360). <i>Res</i> is the resolution (0-3).
      * If <i>Res</i> is out of range, it is assumed to 2 (16 directions).
      * The returned index has the range 0-31, depending on the resolution,
      * the index is mapped to avaible directions; for example : res=0 allows
      * only the four main directions (N,S,W,O), a Dir of 130 id mapped to
      * 90 (O), index 8. The returnvalue in this case is 8.
      */
   static const char* directionToText(float Dir,int Res);
      /**
      * Return a string containing the direction from the stringlist.
      * Uses <i>directionToIndex</i> to calculate the list index.
      */
   const char* directionToText(int Res);
      /**
      * Returns s string containing the current direction of the widget.
      */
   const char* directionToText(void);
      /**
      * Returns a string containing the current direction of the widget 
      * rounded by the current resolution.
      */
   int indexToDirection(int Index);
      /**
      * Sets the direction by the index, see table by 
      * <a href="#getdirtext">getDirText</a> for the index and 
      * the associated directions.
      */
   int textToDirection(const char* Text);
      /**
      * Parses the text parameter for a valid direction and setis this 
      * direction. The returnvalue id TRUE, if <i>Text</i> contains a valid
      * direction else FALSE is returned.
      */
   int getValid(void);
      /**
      * Returns the valid statis of the current direction. Three states
      * are possible:
      * <table border=1 cellspacing="0" cellpadding="2">
      * <tr><td>KWR_VALID</td> <td>a valid direction</td></tr>
      * <tr><td>KWR_NODIR</td> <td>no direction (windless, no data avail.)</td></tr>
      * <tr><td>KWR_INVALID</td><td>invalid direction (eg. invalid data in <i>textToDirection</i></td></tr>
      * </table><br>
      * <ul>
      * <li>In the state KWR_INVALID, the center of the windrose is displayed in red.
      * <li>Only in the state KWR_VALID, the arrow is visible.
      * <li>Functions (like <i>getDirection</i> returns an angle in the 
      * state <> KWR_VALID, this must be checked seperatly.
      * </ul>
      */
   void setValid(int Vld);
      /**
      * Sets the valid status to <i>Vld</i>
      */
   
signals:
      /**
      * This signal is emitted every time, when the wind direction is changed
      * by a direct user interaction (clicking with the mouse, keypresses) but
      * not when the value is changed by a call to setDirection(); The parameter
      * <i>val</i> contains the current direction.
      */
   void valueChanged(float val);
   
protected:
   void paintEvent(QPaintEvent * e);
   void resizeEvent ( QResizeEvent * rev);
   void mouseMoveEvent(QMouseEvent * qme);
   void mousePressEvent(QMouseEvent * qme);
   virtual void focusInEvent ( QFocusEvent * );
   virtual void focusOutEvent ( QFocusEvent * );
   void keyPressEvent (QKeyEvent*); 
    
private slots:
   void PlusBtnClick();
   void MinusBtnClick();
      
private:
   void CalcDir(QMouseEvent * qme);   // neue Richtung berechnen und setzen

   QColor ColL;         // Linienfarbe
   QColor Col1,Col2;    // Fllfarben
   QColor Col3,Col4;    // Fllfarbe Pfeil
   QPainter *Paint;
   QPointArray *Pnts3;  // fr die Windrose
   QPointArray *Pnts8;  // fr den Pfeil
   QPushButton *PlusBtn,*MinusBtn;  // Buttons fr manuelle Einstellung
   float Direction;     // Windrichtung
   float Width;         // Multiplikator fr die Breite der Spitzen (def=0.1)
   int Details;         // Anzahl der Spitzen auf der Windrose
                        // 1->4, 2->8, 3->16, 4->32
   int ArrowType;       // Pfeintyp
   int Editable;        // Bool: Editierbar=TRUE, nicht editierbar=FALSE
   float Resolution;    // Feinheit der Winkeleinstellung
   int Valid;           // Valid-Status (VALID, NODIR, INVALID)
};
#endif
