/* ----------------------------------------------------------------------- *
 * 
 * spawn.c - run programs synchronously
 *   
 *   Copyright 1997 Transmeta Corporation - All Rights Reserved
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, Inc., 675 Mass Ave, Cambridge MA 02139,
 *   USA; either version 2 of the License, or (at your option) any later
 *   version; incorporated herein by reference.
 *
 * ----------------------------------------------------------------------- */

#include <stdlib.h>
#include <stdarg.h>
#include <unistd.h>
#include <signal.h>
#include <sys/wait.h>

#include "automount.h"

/* Used by subprocesses to avoid carrying over the main daemon's rather
   weird signalling environment */
void reset_signals(void)
{
  struct sigaction sa;
  sigset_t allsignals;
  int i;
  
  sigfillset(&allsignals);
  sigprocmask(SIG_BLOCK, &allsignals, NULL);
  
  sa.sa_handler = SIG_DFL;
  sigemptyset(&sa.sa_mask);
  sa.sa_flags = 0;
  for ( i = 1 ; i < NSIG ; i++ )
    sigaction(i, &sa, NULL);
  
  sigprocmask(SIG_UNBLOCK, &allsignals, NULL);
}

int spawnv(char *prog, char **argv) {
  pid_t f;
  int status;
  sigset_t allsignals, oldsig;
  
  sigfillset(&allsignals);
  sigprocmask(SIG_BLOCK, &allsignals, &oldsig);
  
  f = fork();
  if ( f == 0 ) {
    reset_signals();
    execv(prog, argv);
    _exit(255);	/* execv() failed */
  } else {
    sigprocmask(SIG_SETMASK, &oldsig, NULL);
    if ( f < 0 || waitpid(f, &status, 0) != f )
      return -1; /* waitpid() failed */
    else
      return status;
  }
}

int spawnl(char *prog, ...)
{
  va_list arg;
  int argc;
  char **argv, **p;
  
  va_start(arg,prog);
  for ( argc = 1 ; va_arg(arg,char *) ; argc++ ); 
  va_end(arg);
  
  if ( !(argv = alloca(sizeof(char *) * argc)) )
    return -1;
  
  va_start(arg,prog);
  p = argv;
  while ((*p++ = va_arg(arg,char *)));
  va_end(arg);
  
  return spawnv(prog, argv);
}

