/*
 * lookup_yp.c
 *
 * Module for Linux automountd to access a YP (NIS) automount map
 */

#include <stdio.h>
#include <malloc.h>
#include <errno.h>
#include <netdb.h>
#include <fcntl.h>
#include <unistd.h>
#include <syslog.h>
#include <sys/param.h>
#include <sys/socket.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <rpc/rpc.h>
#include <rpc/xdr.h>
#include <rpcsvc/yp_prot.h>
#include <rpcsvc/ypclnt.h>

#include "automount.h"

static char *domainname;
static char *mapname;
static int udpproto;
static short port_discard;

int lookup_version = AUTOFS_LOOKUP_VERSION; /* Required by protocol */

int lookup_init(int argc, char **argv)
{
  int err;
  struct protoent *udp;
  struct servent *port_dis;

  if ( argc < 1 ) {
    syslog(LOG_CRIT, "lookup_init(yp): No map name");
    return 1;
  }
  mapname = argv[0];

  udp = getprotobyname("udp");
  udpproto = udp ? udp->p_proto : 0;
  port_dis = getservbyname("discard","udp");
  if ( port_dis )
    port_discard = port_dis->s_port;
  else
    port_discard = htons(9);	/* 9 is the standard discard port */

  err = yp_get_default_domain(&domainname);
  if ( err ) {
    syslog(LOG_CRIT, "lookup_init(yp): map %s: %s\n", mapname, yperr_string(err));
    return 1;
  }
  return 0;
}

int lookup_mount(char *root, char *name, int name_len)
{
  int err;
  char *mapent, *pmapent, *colon, *p, *q, **haddr;
  int mapent_len;
  struct hostent *he;
  struct sockaddr_in saddr, laddr;
  int sock, len;
  int amp, local;

  syslog(LOG_INFO, "lookup_mount(yp): looking up %s", name);

  err = yp_match(domainname, mapname, name, name_len, &mapent, &mapent_len);
  if ( err ) {
    syslog(LOG_NOTICE, "lookup for %s failed: %s", name, yperr_string(err));
    return 1;
  }

  mapent[mapent_len] = '\0';

  syslog(LOG_INFO, "lookup_mount(yp): %s -> %s", name, mapent);

  if ( mapent[0] == '/' ) {
    /* Uh-oh, a multientry; don't support now */
    free(mapent);
    syslog(LOG_NOTICE, "entry %s is a multipath entry", name);
    return 1;
  }

  amp = 0;
  for ( p = mapent ; *p ; p++ )
    if ( *p == '&' )
      amp++;
  if ( amp ) {
    pmapent = malloc(mapent_len + (strlen(name)-1)*amp + 1);
    if (!pmapent) {
      free(mapent);
      syslog(LOG_ERR, "lookup_mount(yp): malloc: %m");
      return 1;
    }
    q = pmapent;
    for ( p = mapent ; *p ; p++ ) {
      if ( *p == '&' ) {
	strcpy(q,name);
	q = strchr(q,'\0');
      } else
	*(q++) = *p;
    }
    free(mapent);
    mapent = pmapent;
  }

  syslog(LOG_INFO, "lookup_mount(yp): parsed %s -> %s", name, mapent);

  colon = strchr(mapent, ':');
  if ( !colon ) {
    /* No colon, take this as a symlink (local) entry */
    syslog(LOG_INFO, "lookup_mount(yp): entry %s -> %s: no colon, assume local", name, mapent);
    chdir(root);
    err = symlink(mapent, name);
    if ( err && errno == EEXIST )
      err = 0;
    if ( err )
      syslog(LOG_NOTICE, "lookup_mount(yp): symlink %s: %m", name);
    chdir("/");
    free(mapent);
    return err ? 1 : 0;
  }

  *colon = '\0';
  if ( !(he = gethostbyname(mapent)) ) {
    free(mapent);
    syslog(LOG_NOTICE, "lookup_mount(yp): entry %s: host %s: lookup failure",
	   name, mapent);
    return 1;			/* No such host */
  }

  /* Probe to see if we are the local host.  Open a UDP socket and see
     if the local address is the same as the remote one */

  local = 0;
  for ( haddr = he->h_addr_list ; *haddr ; haddr++ ) {
    sock = socket(AF_INET, SOCK_DGRAM, udpproto);
    if ( sock < 0 ) {
      syslog(LOG_ERR, "lookup_mount(yp): socket: %m");
      free(mapent);
      return 1;
    }
    saddr.sin_family = AF_INET;
    bcopy(*haddr, &saddr.sin_addr, he->h_length);
    saddr.sin_port = port_discard;

    len = sizeof(laddr);
    if ( connect(sock, (struct sockaddr *) &saddr, sizeof(saddr)) < 0 
	 || getsockname(sock, (struct sockaddr *) &laddr, &len) < 0 ) {
      syslog(LOG_ERR, "lookup_mount(yp): connect+getsockname failed for %s", name);
      close(sock);
      free(mapent);
      return 1;
    }
    close(sock);
    
    if ( !memcmp(&saddr.sin_addr,&laddr.sin_addr,he->h_length) ) {
      local = 1;
      break;
    }
  }

  if ( local ) {
    /* Local host -- do a symlink */
    
    syslog(LOG_INFO, "lookup_mount(yp): %s is local, symlinking", name);
    chdir(root);
    err = symlink(colon+1, name);
    if ( err && errno == EEXIST )
      err = 0;
    if ( err )
      syslog(LOG_NOTICE, "lookup_mount(yp): symlink %s: %m", name);
    chdir("/");
    free(mapent);
    return err ? 1 : 0;
  } else {
    /* Not a local host - do a mount */
    
    chdir(root);
    *colon = ':';
    syslog(LOG_INFO, "lookup_mount(yp): calling mkdir %s", name);
    if ( mkdir(name, 0555) && errno != EEXIST ) {
      syslog(LOG_NOTICE, "lookup_mount(yp): mkdir %s failed: %m", name);
      return 1;
    }
    syslog(LOG_INFO, "lookup_mount(yp): calling mount -t nfs %s %s", mapent, name);
    err = spawnl(_PATH_MOUNT, _PATH_MOUNT, "-t", "nfs", mapent, name, NULL);
    free(mapent);
    if ( err )
      syslog(LOG_NOTICE, "lookup_mount(yp): nfs entry %s: mount failure",
	     name);
    else
      syslog(LOG_INFO, "lookup_mount(yp): mounted %s", name);
    return ( err != 0 );
  }
}

int lookup_done(void)
{
  free(domainname);
  return 0;
}
