/*
 * Copyright 2008 Sony Corporation
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *   * Redistributions of source code must retain the above copyright notice,
 *     this list of conditions and the following disclaimer.
 *   * Redistributions in binary form must reproduce the above copyright
 *     notice, this list of conditions and the following disclaimer in the
 *     documentation and/or other materials provided with the distribution.
 *   * Neither the names of the copyright holders nor the names of their
 *     contributors may be used to endorse or promote products derived from this
 *     software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#ifndef LIBSPE2_RUNTIME_H
#define LIBSPE2_RUNTIME_H

/*
 * Choose how to wait for queues.
 *
 * None: Poll memory by busy loop to check queue status.
 * ENABLE_SYNC_FUTEX: Call futex_wait to sleep until the queue status is changed.
 * ENABLE_SYNC_YIELD: Poll memory by busy loop with calling sched_yield.
 * ENABLE_SYNC_RELAX: Poll memory by busy loop with sleeping during a short time.
 */
#define ENABLE_SYNC_FUTEX 1
//#define ENABLE_SYNC_YIELD 1
//#define ENABLE_SYNC_RELAX 1

typedef struct cell_queue_st
	{
/*
 * status:
 * [0]     write lock flag
 * [1:15]  write next pointer
 * [16]    read lock flag
 * [17:31] read next pointer
 */
	uint32_t status;
	
	uint32_t elm_size;  /* size of element */
	uint32_t aligned_elm_size;
	uint32_t depth;  /* depth of queue */
	uint64_t buffer; /* buffer's EA */
	uint8_t pad3[SHARED_ATOMIC_DATA_ALIGN - 24];
	} cell_queue_t;

#define CELL_QUEUE_DEPTH_MAX 0x3fff

#define CELL_QUEUE_PACK_STATUS(r, w) ((r) | ((w) << 16))

#define CELL_QUEUE_UNPACK_R_STATUS(p)  ((p) & 0xffff)
#define CELL_QUEUE_UNPACK_W_STATUS(p)  ((p) >> 16)

#define CELL_QUEUE_LOCKED_FLAG(p) ((p) & 0x8000)
#define CELL_QUEUE_IS_LOCKED(p) CELL_QUEUE_LOCKED_FLAG(p)
#define CELL_QUEUE_SET_LOCKED(p) ((p) | 0x8000)
#define CELL_QUEUE_CLEAR_LOCKED(p) ((p) & ~0x8000)
#define CELL_QUEUE_POINTER(p) CELL_QUEUE_CLEAR_LOCKED(p)

#ifdef __SPU__

#ifdef ENABLE_SYNC_FUTEX

static inline void sync_notify(uint64_t sync_ea)
	{
	spu_write_out_mbox(sync_ea >> 32);
	spu_write_out_intr_mbox(sync_ea & 0xffffffff);
	}
#else /* !ENABLE_SYNC_FUTEX */

static inline void sync_notify(uint64_t sync_ea)
	{
	/* do nothing */
	}

#endif /* !ENABLE_SYNC_FUTEX */

#else /* !__SPU__ */

struct spe_task_st
	{
	spe_context_ptr_t spe;
	pthread_t thread;
	spe_program_handle_t *prog;
	uint64_t arg1;
	uint64_t arg2;
#ifdef ENABLE_SYNC_FUTEX
        pthread_t handler_thread;
#endif /* ENABLE_SYNC_FUTEX */
	};

#include <unistd.h>

#define sync_relax() usleep(1)

#ifdef ENABLE_SYNC_FUTEX
#include <sys/syscall.h>
#include <linux/futex.h>

static inline int futex(void *uaddr, int op, int val,
	const struct timespec *timeout, int *uaddr2, int val3)
	{
	return syscall(__NR_futex, uaddr, op, val, timeout, uaddr2, val3);
	}

static inline int futex_wait(void *uaddr, int val,
	const struct timespec *timeout)
	{
	return futex(uaddr, FUTEX_WAIT, val, timeout, NULL, 0);
	}

static inline int futex_wake(void *uaddr, int val)
	{
	return futex(uaddr, FUTEX_WAKE, val, NULL, NULL, 0);
	}

static inline void sync_notify(void *sync)
	{
	int ret;
	ret = futex_wake(sync, INT_MAX);
	ASSERT(ret != -1);
	}

static inline void sync_wait(void *sync, unsigned int from_val)
	{
	int ret = futex_wait(sync, from_val, NULL);
	ASSERT (ret != -1 || errno == EWOULDBLOCK);
	}

#else /* !ENABLE_SYNC_FUTEX */

static inline void sync_notify(void *sync)
	{
	/* do nothing */
	}

static inline void sync_wait(void *sync, unsigned int from_val)
	{
#ifdef ENABLE_SYNC_YIELD
	sched_yield();
#elif defined(ENABLE_SYNC_RELAX)
	sync_relax();
#else
	/* do nothing */
#endif
	}

#endif /* !ENABLE_SYNC_FUTEX */

#endif /* !__SPU__ */

#endif /* !LIBSPE2_RUNTIME_H */
