/*
 * Copyright 2008 Sony Corporation
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *   * Redistributions of source code must retain the above copyright notice,
 *     this list of conditions and the following disclaimer.
 *   * Redistributions in binary form must reproduce the above copyright
 *     notice, this list of conditions and the following disclaimer in the
 *     documentation and/or other materials provided with the distribution.
 *   * Neither the names of the copyright holders nor the names of their
 *     contributors may be used to endorse or promote products derived from this
 *     software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include "e_cell.h"

#define CHUNK_MAX 4096

static int do_update(spe_digest_request_t *dgst, uint64_t ea, unsigned int size)
	{
	unsigned int cur;
	const int tag[2] = { 0, 1, };
	unsigned int chunk_size, next_chunk_size;
	ALIGNED(static unsigned char buffer[2][CHUNK_MAX], SHARED_DATA_ALIGN);

	ALIGN_HINT(dgst, SHARED_DATA_ALIGN, 0);

	if (!size) return 1;

	/* perform 'double buffering' */
	cur = 0;

	/* request the first data */
	next_chunk_size = (size <= CHUNK_MAX) ? size : CHUNK_MAX;
	mfc_get(buffer[cur], ea,
		ALIGN_CEIL(next_chunk_size, SHARED_DATA_ALIGN), tag[cur], 0, 0);
	size -= next_chunk_size;
	ea += next_chunk_size;

	chunk_size = next_chunk_size;
	while ((next_chunk_size = (size <= CHUNK_MAX) ? size : CHUNK_MAX) > 0)
		{
		/* prepare the next data */
		mfc_get(buffer[cur ^ 1], ea,
			ALIGN_CEIL(next_chunk_size, SHARED_DATA_ALIGN), tag[cur ^ 1], 0, 0);
		size -= next_chunk_size;
		ea += next_chunk_size;

		/* wait for the current data */
		mfc_write_tag_mask(1 << tag[cur]);
		mfc_read_tag_status_all();

		/* calculate digest */
		spe_digest_update(dgst, buffer[cur], chunk_size);

		cur ^= 1;
		chunk_size = next_chunk_size;
		}

	/* process the last chunk */

	mfc_write_tag_mask(1 << tag[cur]);
	mfc_read_tag_status_all();

	spe_digest_update(dgst, buffer[cur], chunk_size);

	return 1;
	}

int spe_main(uint64_t q_in_ea, uint64_t q_out_ea)
	{
	for ( ; ; )
		{
		ALIGNED(static spe_digest_request_t dgst, SHARED_DATA_ALIGN);
		ALIGNED(static spe_digest_result_t result, SHARED_DATA_ALIGN);

		DPRINTF("\n");
		cell_queue_pop(q_in_ea, &dgst);

		/* execute commands */
		switch (dgst.request)
			{
			case SPE_DIGEST_COMMAND_INIT:
				DPRINTF("INIT\n");
				spe_digest_init(&dgst);
				break;
			case SPE_DIGEST_COMMAND_UPDATE:
				DPRINTF("UPDATE\n");
				do_update(&dgst, dgst.in_ea, dgst.size);
				break;
			case SPE_DIGEST_COMMAND_FINAL:
				DPRINTF("FINAL\n");
				spe_digest_final(&dgst, &result);
				cell_queue_push(q_out_ea, &result);
				break;
			case SPE_DIGEST_COMMAND_QUIT:
				DPRINTF("QUIT\n");
				return 0;
			default:
				/* FIXME: error handling */
				DPRINTF("Unknown request type!!!\n");
				break;
			}
		}

	return 0;
	}
