/*
 * Copyright (c) 1998,1999 Michael Elizabeth Chastain.
 * Copyright (c) 2000,2001 Christoph Hellwig.
 * All rights resered.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>

#include "mconfig.h"
#include "parser.tab.h"

#define MAXLINE 80

static int      is_mode_old = 0;/* make oldconfig */

static int      check_yes_no_mod(const statement_type * statement);
static int      check_ask(const statement_type * statement);
static int      check_nchoice(const statement_type * statement);

static void 
scb_line(const statement_type * statement, void *UNUSED)
{
	char            str[MAXLINE];
	int             first_comment = 1;

	switch (statement->verb) {
	default:
		error_enum_bogus();
		break;

	case verb_MENU:{
			snprintf(str, statement->sc_title->value.len + 1, "%s",
				 statement->sc_title->value.ptr);

			printf("\n---> %s\n", str);
			break;
		}

	case verb_mainmenu_name:
		/* ignore this statement */
		break;

	case verb_comment:{
			piece_type      prompt = statement->sb_prompt->value;

			if (first_comment) {
				first_comment = 0;
				break;
			}
			snprintf(str, prompt.len + 1, "%s", prompt.ptr);
			printf("--- %s\n", str);

			break;
		}

	case verb_text:{
			piece_type      prompt = statement->sb_prompt->value;

			snprintf(str, prompt.len + 1, "%s", prompt.ptr);
			printf("%s\n", str);

			break;
		}

	case verb_unset:{
			symbol_type    *symbol;

			for (symbol = statement->su_unset_list->symbol_first;
			     symbol != NULL; symbol = symbol->next) {
				symbol->value = piece_empty;
				symbol->origin = origin_unset;
			}

			break;
		}

	case verb_def_bool:
	case verb_def_hex:
	case verb_def_int:
	case verb_def_string:
	case verb_def_tristate:
		statement->sb_symbol->value =
			word_eval(statement->sb_def_value, 1);
		statement->sb_symbol->origin = origin_statement;
		break;

	case verb_ask_bool:
	case verb_dep_bool:
	case verb_dep_mbool:{
			piece_type prompt = statement->sb_prompt->value;
			piece_type value = statement->sb_symbol->value;

			if ((statement_bools_allowed(statement) & ~0x01) == 0)
				break;

			if (is_mode_old &&
			    (statement->sb_symbol->origin != origin_unset)) {
				if (value.len != 1 || value.ptr[0] != 'y')
					statement->sb_symbol->value = piece_n;
				break;
			}
			if (value.len == 1 && value.ptr[0] == 'y')
				printf(" [*] ");
			else {
				printf(" [ ] ");
				/*
				 * force value to be 'n', whatever it was
				 * before
				 */
				statement->sb_symbol->value = piece_n;
			}

			snprintf(str, prompt.len + 1, "%s", prompt.ptr);

			printf("%s", str);

			if (statement->sb_symbol->origin == origin_unset)
				printf(" (NEW)");

			check_yes_no_mod(statement);

			break;
		}
	case verb_ask_tristate:
	case verb_dep_tristate:{
			piece_type prompt = statement->sb_prompt->value;
			piece_type value = statement->sb_symbol->value;

			if ((statement_bools_allowed(statement) & ~0x01) == 0)
				break;

			if (is_mode_old &&
			    (statement->sb_symbol->origin != origin_unset)) {
				if (value.len != 1 ||
				    (value.ptr[0] != 'y' &&
				     value.ptr[0] != 'm'))
					statement->sb_symbol->value = piece_n;
				break;
			}
			if (value.len == 1 && value.ptr[0] == 'y')
				printf(" <*> ");
			else if (value.len == 1 && value.ptr[0] == 'm')
				printf(" <m> ");
			else {
				printf(" < > ");
				/*
				 * force value to be 'n', whatever it was
				 * before
				 */
				statement->sb_symbol->value = piece_n;
			}

			snprintf(str, prompt.len + 1, "%s", prompt.ptr);

			printf("%s", str);

			if (statement->sb_symbol->origin == origin_unset)
				printf(" (NEW)");

			check_yes_no_mod(statement);

			break;
		}

	case verb_ask_hex:
	case verb_dep_hex:
	case verb_ask_int:
	case verb_dep_int:
	case verb_ask_string:
	case verb_dep_string:{
			piece_type      prompt = statement->sb_prompt->value;
			piece_type      value = statement->sb_symbol->value;

			if ((statement_bools_allowed(statement) & ~0x01) == 0)
				break;

			if (is_mode_old &&
			    (statement->sb_symbol->origin != origin_unset))
				break;

			snprintf(str, prompt.len + 1, "%s", prompt.ptr);
			printf(" ... %s", str);

			if (statement->sb_symbol->origin == origin_unset) {
				printf(" (NEW)");
				/* this should be a new kind of origin */
				value = word_eval(statement->sb_def_value, 0);
				statement->sb_symbol->value = value;
			}
			printf(" [");

			if (statement->verb == verb_ask_hex ||
			    statement->verb == verb_dep_hex)
				printf("0x");

			if (statement->verb == verb_ask_string ||
			    statement->verb == verb_dep_string)
				printf("%c", '"');

			snprintf(str, value.len + 1, "%s", value.ptr);
			printf("%s", str);

			if (statement->verb == verb_ask_string ||
			    statement->verb == verb_dep_string)
				printf("%c", '"');

			printf("] ");

			check_ask(statement);

			break;
		}

	case verb_nchoice:{
			const prompt_type *prompt;
			symbol_type *symbol;
			int i = 0;

			if (is_mode_old) {
				int is_new = 0;
				for (prompt = statement->sn_choice_list->prompt_first,
				     symbol = statement->sn_choice_list->symbol_first;
				     prompt != NULL && symbol != NULL;
				     prompt = prompt->next, symbol = symbol->next) {
					if (symbol->origin == origin_unset)
						is_new++;
				}

				if (!is_new)
					break;
			}
			snprintf(str, statement->sn_prompt->value.len + 1, "%s",
				 statement->sn_prompt->value.ptr);

			printf(" ### %s\n", str);

			/*
                         * Render subordinate lines.
                         */

			for (prompt = statement->sn_choice_list->prompt_first,
			   symbol = statement->sn_choice_list->symbol_first;
			     prompt != NULL && symbol != NULL;
			     prompt = prompt->next, symbol = symbol->next) {
				if (symbol->value.len == 1 && symbol->value.ptr[0] == 'y')
					printf("*(%d)  ", i++);
				else {
					printf(" (%d)  ", i++);
					/*
					 * force value to be 'n', whatever it
					 * was before
					 */
					symbol->value = piece_n;
				}

				printf("%s", prompt->value.ptr);

				if (symbol->origin == origin_unset)
					printf(" (NEW)\n");
				else
					printf("\n");
			}

			check_nchoice(statement);

			break;
		}

	}
}

static int 
check_yes_no_mod(const statement_type * statement)
{
	const int       allowed = statement_bools_allowed(statement);
	char            iput[MAXLINE];

	printf(" [");
	if (allowed & 0x04)
		printf("y|");
	if (allowed & 0x02)
		printf("m|");
	if (allowed & 0x01)
		printf("n|");
	printf("?] ");

	fgets(iput, MAXLINE, stdin);

	switch (iput[0]) {
	default:
		break;
	case ' ':{
			int             itry;
			piece_type      value = statement->sb_symbol->value;

			for (itry = 0; itry < 3; itry++) {
				int             ivalue = 0;

				if (value.len == 1) {
					if (value.ptr[0] == 'n')
						ivalue = 1;
					if (value.ptr[0] == 'm')
						ivalue = 2;
				}
				value.ptr = "nmy" + ivalue;
				value.len = 1;

				if ((allowed & (1 << ivalue)) != 0) {
					statement->sb_symbol->value = value;
					statement->sb_symbol->origin = origin_statement;
					return 0;
				}
			}

			fprintf(stderr, "I can't find any legal value!");
			return 1;
		}
		break;

	case 'N':
	case 'n':
		if ((allowed & 0x01) != 0) {
			statement->sb_symbol->value = piece_n;
			statement->sb_symbol->origin = origin_statement;
			return 0;
		} else {
			fprintf(stderr, "Value 'n' not allowed!");
			return 1;
		}
		break;

	case 'M':
	case 'm':
		if ((allowed & 0x02) != 0) {
			statement->sb_symbol->value = piece_m;
			statement->sb_symbol->origin = origin_statement;
			return 0;
		} else {
			fprintf(stderr, "Value 'm' not allowed.");
			return 1;
		}
		break;

	case '*':
	case 'Y':
	case 'y':
		if ((allowed & 0x04) != 0) {
			statement->sb_symbol->value = piece_y;
			statement->sb_symbol->origin = origin_statement;
			return 0;
		} else {
			fprintf(stderr, "Value 'y' not allowed.");
			return 1;
		}
		break;
	}
	return 1;
}

static int 
check_ask(const statement_type * statement)
{
	piece_type      value = statement->sb_symbol->value;
	char            iput[MAXLINE];

	fgets(iput, MAXLINE, stdin);

	if (isprint(iput[0])) {
		if (!statement_allow_char(statement, iput[0])) {
			fprintf(stderr,
				"Illegal character for this data type.");
			return 1;
		} else {
			int             len;
			char           *tmp;
			for (tmp = iput, len = 0; *tmp != '\n'; tmp++, len++)
				;
			*tmp = '\0';

			strncpy(value.ptr, iput, len);
			value.len = len;
			statement->sb_symbol->value = value;
			statement->sb_symbol->origin = origin_statement;
			return 1;
		}
	}
	return 0;
}

static int 
check_nchoice(const statement_type * statement)
{
	symbol_type    *symbol;
	char            str[MAXLINE];
	char            iput[MAXLINE];
	int             i;

	snprintf(str, statement->sn_prompt->value.len + 1, "%s",
		 statement->sn_prompt->value.ptr);

	printf(" ### %s [0...%d] ", str, statement->sn_choice_count - 1);

	fgets(iput, MAXLINE, stdin);

	switch (iput[0]) {
	default:
		break;

	case '0':
	case '1':
	case '2':
	case '3':
	case '4':
	case '5':
	case '6':
	case '7':
	case '8':
	case '9':
		statement->sn_choice_index = atoi(iput);
		/* set the symbols appropriately */
		for (symbol = statement->sn_choice_list->symbol_first, i = 0;
		     symbol != NULL; symbol = symbol->next, i++) {
			if (i == statement->sn_choice_index)
				symbol->value = piece_y;
			else
				symbol->value = piece_n;
			symbol->origin = origin_statement;
		}

		break;
	}

	return 1;
}

void 
do_mode_line(const block_type * block)
{
	const char     *str_error;
	block_walk(block, 1, scb_line, NULL);

	if (block_output(block, &str_error) < 0) {
		fprintf(stderr, "Write failed: %s\n", str_error);
		exit(1);
	}
}

void 
do_mode_old(const block_type * block)
{
	const char     *str_error;

	is_mode_old = 1;

	block_walk(block, 1, scb_line, NULL);

	if (block_output(block, &str_error) < 0) {
		fprintf(stderr, "Write failed: %s\n", str_error);
		exit(1);
	}
}
