#ifdef MODVERSIONS
#include <linux/modversions.h>
#endif

#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/ctype.h>
#include <linux/string.h>
#include <asm/uaccess.h>
#include <asm/fcntl.h>
#include <linux/file.h>
#include <linux/smp_lock.h>
#include <linux/fs.h>
#include <linux/mm.h>
#include <linux/uio.h>
#include <net/sock.h>

#define SHFS_DEBUG_HERE

#include "shfs.h"
#include "shfs_proc.h"
#include "shfs_proto.h"

void
shfs_init_root_dirent(struct shfs_sb_info *server, struct shfs_fattr *fattr)
{
	memset(fattr, 0, sizeof(*fattr));
	fattr->f_nlink = 1;
	fattr->f_uid = server->mnt.uid;
	fattr->f_gid = server->mnt.gid;
	fattr->f_blksize = 512;

	fattr->f_ino = 2;
	fattr->f_mtime = CURRENT_TIME;
	fattr->f_mode = server->mnt.root_mode;
	fattr->f_size = 512;
	fattr->f_blocks = 0;
}

static int
shfs_parse_mode(char *mode)
{
	return ((mode[0]-'0')<<6) + ((mode[1]-'0')<<3) + (mode[2]-'0');
}

int
shfs_parse_options(struct shfs_sb_info *info, char *opts)
{
	char *p, *q;
	int i;

	strcpy(info->mnt.root, "$HOME");
	info->mnt.pin = info->mnt.pout = NULL;
	info->mnt.fmask = 00177777;

	if (!opts) 
		return -1;

	while ((p = strsep(&opts, ","))) {
		if (strncmp(p, "root=", 5) == 0) {
			if (*(p+5) == '/') {
				if (strlen(p+5) + 1 > SHFS_PATH_MAX)
					goto ugly_opts;
				if (*(p+6) == '\0') {
					/* root => mnt.root = "", cygwin have 
					 * toubles resolving //something
					 */
					strcpy(info->mnt.root, "");
				} else {
					strcpy(info->mnt.root, p+5);
				}
			} else {
				if (strlen(p+5) + strlen(info->mnt.root) + 2 > SHFS_PATH_MAX)
					goto ugly_opts;
				strcat(info->mnt.root, "/");
				strcat(info->mnt.root, p+5);
			}
		} else if (strncmp(p, "mnt=", 4) == 0) {
			if (strlen(p+4) + 1 > SHFS_PATH_MAX)
				goto ugly_opts;
			strcpy(info->mnt.mount_point, p+4);
		} else if (strncmp(p, "ro", 2) == 0) {
			info->mnt.readonly = 1;
		} else if (strncmp(p, "rw", 2) == 0) {
			info->mnt.readonly = 0;
		} else if (strncmp(p, "suid", 4) == 0) {
			info->mnt.fmask |= S_ISUID|S_ISGID;
		} else if (strncmp(p, "nosuid", 6) == 0) {
			info->mnt.fmask &= ~(S_ISUID|S_ISGID);
		} else if (strncmp(p, "dev", 3) == 0) {
			info->mnt.fmask |= S_IFCHR|S_IFBLK;
		} else if (strncmp(p, "nodev", 5) == 0) {
			info->mnt.fmask &= ~(S_IFCHR|S_IFBLK);
		} else if (strncmp(p, "exec", 4) == 0) {
			info->mnt.fmask |= S_IXUSR|S_IXGRP|S_IXOTH;
		} else if (strncmp(p, "noexec", 6) == 0) {
			info->mnt.fmask &= ~(S_IXUSR|S_IXGRP|S_IXOTH);
		} else if (strncmp(p, "nocache", 7) == 0) {
			info->mnt.disable_fcache = 1;
		} else if (strncmp(p, "generic", 7) == 0) {
			info->mnt.generic_host = 1;
		} else if (strncmp(p, "preserve", 8) == 0) {
			info->mnt.preserve_own = 1;
		} else if (strncmp(p, "stable", 6) == 0) {
			info->mnt.stable_symlinks = 1;
		} else if (strncmp(p, "ttl=", 4) == 0) {
			if (strlen(p+4) > 10)
				goto ugly_opts;
			q = p+4;
			i = simple_strtoul(q, &q, 10);
			info->mnt.ttl = i * 1000;
		} else if (strncmp(p, "uid=", 4) == 0) {
			if (strlen(p+4) > 10)
				goto ugly_opts;
			q = p+4;
			i = simple_strtoul(q, &q, 10);
			info->mnt.uid = i; 
		} else if (strncmp(p, "gid=", 4) == 0) {
			if (strlen(p+4) > 10)
				goto ugly_opts;
			q = p+4;
			i = simple_strtoul(q, &q, 10);
			info->mnt.gid = i; 
		} else if (strncmp(p, "rmode=", 6) == 0) {
			if (strlen(p+6) > 3)
				goto ugly_opts;
			info->mnt.root_mode = S_IFDIR | (shfs_parse_mode(p+6) & (S_IRWXU | S_IRWXG | S_IRWXO));
		} else if (strncmp(p, "fdi=", 4) == 0) {
			if (strlen(p+4) > 5)
				goto ugly_opts;
			q = p+4;
			i = simple_strtoul(q, &q, 10);
			DEBUG("fdi=%d\n", i);
			info->mnt.pin = fget(i);
		} else if (strncmp(p, "fdo=", 4) == 0) {
			if (strlen(p+4) > 5)
				goto ugly_opts;
			q = p+4;
			i = simple_strtoul(q, &q, 10);
			DEBUG("fdo=%d\n", i);
			info->mnt.pout = fget(i);
		} else if (strncmp(p, "version=", 8) == 0) {
			if (strlen(p+8) > 5)
				goto ugly_opts;
			q = p+8;
			i = simple_strtoul(q, &q, 10);
			DEBUG("version: %u\n", i);
			info->mnt.version = i;
		}

	}
	if (!info->mnt.pin || !info->mnt.pout)
		goto ugly_opts;
	if (info->mnt.preserve_own)
		info->mnt.readonly = 1;
	return 0;
	
ugly_opts:
	VERBOSE("Your options sucks!\n");
	return -1;
}

static int clear_garbage(struct shfs_sb_info *);

static int
pipe_write(struct shfs_sb_info *info, const void *buf, int len)
{
	int bytes, wr = 0;
	int res;
	struct file *f = info->mnt.pout;
	mm_segment_t fs;

	if (info->mnt.garbage)
		if ((res = clear_garbage(info)) < 0)
			return res;
	
	fs = get_fs();
	set_fs(get_ds());
	bytes = len;
retry:
	while (bytes && ((wr = f->f_op->write(f, buf, bytes, &f->f_pos)) > 0)) {
		DEBUG("Wrote: %d\n", wr);
		buf += wr;
		bytes -= wr;
	}
	if (wr == -EAGAIN)
		goto retry;
	set_fs(fs);
	
	if (wr < 0) {
		info->mnt.garbage = 1;
		info->mnt.garbage_write = bytes;
		return wr;
	}
	return len;
}

int 
shfs_pipe_printf(struct shfs_sb_info *info, const char *fmt, ...)
{
	va_list ap;
	int res;

	if (info->mnt.garbage)
		if ((res = clear_garbage(info)) < 0)
			return res;

	va_start(ap, fmt);
	/* shfs is locked, we can write to printf_buffer safely */
	vsnprintf(info->mnt.printf_buffer, PRINTF_BUFFER_SIZE, fmt, ap);
	va_end(ap);

	DEBUG("Write line: %s\n", info->mnt.printf_buffer);
	return pipe_write(info, (void *)info->mnt.printf_buffer, strlen(info->mnt.printf_buffer));
}

#define BUFFER info->mnt.readln_buffer
#define LEN    info->mnt.readln_buffer_len

static int
pipe_read(struct shfs_sb_info *info, void *buf, int len)
{
	int bytes, c, res, rd = 0;
	int sigint, sigstop, counter;
	struct file *f = info->mnt.pin;
	mm_segment_t fs;

	if (info->mnt.garbage)
		if ((res = clear_garbage(info)) < 0)
			return res;

	if (LEN > 0) {
		c = len > LEN ? LEN : len;
		memcpy(buf, BUFFER, c);
		buf+= c;
		LEN -= c;
		len -= c;
		if (LEN > 0)
			memmove(BUFFER, BUFFER+c, LEN);
	}

	fs = get_fs();
	set_fs(get_ds());
	bytes = len;
	sigint = 0;
	sigstop = 0;
	counter = 0;
restart:
	while (bytes && ((rd = f->f_op->read(f, buf, bytes, &f->f_pos)) > 0)) {
		buf += rd;
		bytes -= rd;
	}
	if (rd == -ERESTARTSYS && counter < 10) {
		counter++;
		if (sigismember(&current->pending.signal, SIGINT)) {
			sigdelset(&current->pending.signal, SIGINT);
			current->sigpending--;
			sigint = 1;
			DEBUG("SIGINT ignored...\n");
			goto restart;
		}
		if (sigismember(&current->pending.signal, SIGSTOP)) {
			sigdelset(&current->pending.signal, SIGSTOP);
			current->sigpending--;
			sigstop = 1;
			DEBUG("SIGSTOP ignored...\n");
			goto restart;
		}
	}
	if (sigint && !sigismember(&current->pending.signal, SIGINT)) {
		sigaddset(&current->pending.signal, SIGINT);
		current->sigpending++;
		DEBUG("SIGINT restored...\n");
	}
	if (sigstop && !sigismember(&current->pending.signal, SIGSTOP)) {
		sigaddset(&current->pending.signal, SIGSTOP);
		current->sigpending++;
		DEBUG("SIGSTOP restored...\n");
	}
	set_fs(fs);
	
	if (rd < 0) {
		info->mnt.garbage = 1;
		info->mnt.garbage_read = bytes;
		return rd;
	}
	return len;
}

int 
shfs_pipe_readln(struct shfs_sb_info *info, char *buf, int len)
{
	struct file *f = info->mnt.pin;
	int c, res, l = 0, rd = 0;
	int sigint, sigstop, counter;
	char *str;
	mm_segment_t fs;

	if (info->mnt.garbage)
		if ((res = clear_garbage(info)) < 0)
			return res;

	do {
		str = memchr(BUFFER, '\n', LEN);
		if (str) {
			DEBUG("hit!\n");
			*str = '\0';
			strncpy(buf, BUFFER, len-1);
			buf[len-1] = '\0';
			c = LEN-(str-BUFFER+1);
			memmove(BUFFER, str+1, c);
			LEN = c;
		} else {
			DEBUG("miss, size: %d\n", LEN);
			c = READLN_BUFFER_SIZE - LEN;
			if (c > 0) {
				fs = get_fs();
				set_fs(get_ds());
				sigint = 0;
				sigstop = 0;
				counter = 0;
restart:
				rd = f->f_op->read(f, BUFFER+LEN, c, &f->f_pos);
				if (rd == -ERESTARTSYS && counter < 10) {
					counter++;
					if (sigismember(&current->pending.signal, SIGINT)) {
						sigdelset(&current->pending.signal, SIGINT);
						current->sigpending--;
						sigint = 1;
						DEBUG("SIGINT ignored...\n");
						goto restart;
					}
					if (sigismember(&current->pending.signal, SIGSTOP)) {
						sigdelset(&current->pending.signal, SIGSTOP);
						current->sigpending--;
						sigstop = 1;
						DEBUG("SIGSTOP ignored...\n");
						goto restart;
					}
				}
				if (sigint && !sigismember(&current->pending.signal, SIGINT)) {
					sigaddset(&current->pending.signal, SIGINT);
					current->sigpending++;
					DEBUG("SIGINT restored...\n");
				}
				if (sigstop && !sigismember(&current->pending.signal, SIGSTOP)) {
					sigaddset(&current->pending.signal, SIGSTOP);
					current->sigpending++;
					DEBUG("SIGSTOP restored...\n");
				}
				DEBUG("rd: %d\n", rd);
				set_fs(fs);
				if (rd == -EAGAIN)
					continue;
				if (rd < 0) {
					info->mnt.garbage = 1;
					info->mnt.garbage_read = 0;
					return rd;
				}
				LEN += rd;
			} else {
				BUFFER[READLN_BUFFER_SIZE-1] = '\n';
			}
		}
		/* do not lock while reading 0 bytes */
		if (l++ > READLN_BUFFER_SIZE && LEN == 0)
			return -1;
	} while (!str);
	
	DEBUG("size: %d\n", LEN);
	DEBUG("Read line(%d): %s\n", strlen(buf), buf);
	return strlen(buf);
}

int
reply(char *s)
{
	if (strncmp(s, "### ", 4))
		return 0;
	return simple_strtoul(s+4, NULL, 10);
}

static int
clear_garbage(struct shfs_sb_info *info)
{
	char line[SHFS_LINE_MAX];
	static unsigned long seq = 12345;
	int i, state, c, garbage;
	int res;

	DEBUG("start\n");
	garbage = info->mnt.garbage_write;
	if (garbage)
		memset(line, ' ', SHFS_LINE_MAX);
	DEBUG("write %d\n", garbage);
	while (garbage > 0) {
		c = garbage < SHFS_LINE_MAX ? garbage : SHFS_LINE_MAX;
		info->mnt.garbage = 0;
		res = pipe_write(info, line, c);
		if (res < 0) {
			info->mnt.garbage_write = garbage;
			goto error;
		}
		garbage -= res;
	}
	info->mnt.garbage_write = 0;
	garbage = info->mnt.garbage_read;
	DEBUG("read %d\n", garbage);
	while (garbage > 0) {
		c = garbage < SHFS_LINE_MAX ? garbage : SHFS_LINE_MAX;
		info->mnt.garbage = 0;
		res = pipe_read(info, line, c);
		if (res < 0) {
			info->mnt.garbage_read = garbage;
			goto error;
		}
		garbage -= res;
	}
	info->mnt.garbage_read = 0;
		
	info->mnt.garbage = 0;
	if ((res = do_ping(info, seq++)) < 0)
		goto error;
	DEBUG("reading..\n");
	state = 0;
	for (i = 0; i < 100000; i++) {
		info->mnt.garbage = 0;
		if ((res = shfs_pipe_readln(info, line, SHFS_LINE_MAX)) < 0)
			goto error;
		if (((state == 0) || (state == 1)) && reply(line) == REP_PRELIM) {
			state = 1;
		} else if (state == 1 && simple_strtoul(line, NULL, 10) == (seq-1)) {
			state = 2;
		} else if (state == 2 && reply(line) == REP_NOP) {
			DEBUG(" cleared.\n");
			return 0;
		} else {
			state = 0;
		}
	}
error:
	info->mnt.garbage = 1;
	DEBUG(" failure!\n");
	return res;
}

/* aaa'aaa -> aaa'\''aaa */
static int
replace_quote(char *name)
{
	char *s;
	char *r = name;
	int q = 0;
	
	while (*r) {
		if (*r == '\'')
			q++;
		r++;
	}
	s = r+(3*q);
	if ((s-name+1) > SHFS_PATH_MAX)
		return -1;
	
	*(s--) = '\0';
	r--;
	while (q) {
		if (*r == '\'') {
			s -= 3;
			s[0] = '\'';
			s[1] = '\\';
			s[2] = '\'';
			s[3] = '\'';
			q--;
		} else {
			*s = *r;
		}
		s--;
		r--;
	}
	return 0;
}

int
shfs_get_name(struct dentry *d, char *name)
{
	int len = 0;
	struct dentry *p;

	for (p = d; !IS_ROOT(p); p = p->d_parent)
		len += p->d_name.len + 1;
		
	if (len + 1 > SHFS_PATH_MAX) 
		return -1;
	if (len == 0) {
		name[0] = '/';
		name[1] = '\0';
		return 0;
	}
	
	name[len] = 0;
	for (p = d; !IS_ROOT(p); p = p->d_parent) {
		len -= p->d_name.len;
		strncpy(&(name[len]), p->d_name.name, p->d_name.len);
		len--;
		name[len] = '/';
	}

	return replace_quote(name);
}

inline int
shfs_lock(struct shfs_sb_info *info)
{
	int res;
	DEBUG("Locking...\n");
	res = down_interruptible(&(info->sem));
	DEBUG("Locked.\n");
	return (res != -EINTR);
}

inline void
shfs_locku(struct shfs_sb_info *info)
{
	DEBUG("Locking...\n");
	down(&(info->sem));
	DEBUG("Locked.\n");
}

inline void
shfs_unlock(struct shfs_sb_info *info)
{
	DEBUG("Unlocking...\n");
	up(&(info->sem));
	DEBUG("Unlocked.\n");
}

inline int
shfs_remove_sigpipe(int result)
{
	switch (result) {
//	    case -ERESTARTSYS:
//	    	result = -EINTR;
//		goto cont;
	    case -EPIPE:
		result = -EIO;
//	    cont:
		if (sigismember(&current->pending.signal, SIGPIPE)) {
			sigdelset(&current->pending.signal, SIGPIPE);
			current->sigpending--;
		}
		break;
	}
	return result;
}

int
shfs_get_attr(struct dentry *dentry, struct shfs_fattr *fattr)
{
	struct shfs_sb_info *info = (struct shfs_sb_info*)dentry->d_sb->u.generic_sbp;
	struct shfs_dir_entry *entry;
	int error;
	
	error = shfs_dcache_get_entry(dentry, &entry);
	if (error < 0) {
		VERBOSE(" shfs_dcache_get_entry failed! (%d)\n", error);
		return error;
	}
	if (!entry) {
		DEBUG(" file not found in parent dir cache!\n");
		return -ENOENT;
	}

	fattr->f_mode = entry->mode;
	fattr->f_ino = entry->ino;
	fattr->f_size = entry->size;
	fattr->f_rdev = entry->rdev;
	fattr->f_blksize = entry->blocksize;
	fattr->f_blocks = entry->blocks;
	fattr->f_nlink = entry->nlink;
	fattr->f_atime = entry->atime;

	/* Any way to get the real mtime, ctime? */
	fattr->f_mtime = entry->atime;
	fattr->f_ctime = entry->atime;
	if (info->mnt.preserve_own) {
		fattr->f_uid = entry->uid;
		fattr->f_gid = entry->gid;
	} else {
		fattr->f_uid = info->mnt.uid;
		fattr->f_gid = info->mnt.gid;
	}
	return 0;
}

/* returns 1 if file is "non-empty" but has zero size */

int
shfs_proc_open(struct shfs_sb_info *info, char *name, int mode)
{
	char buf[SHFS_LINE_MAX];
	char bufmode[3];
	int res;
	
	if (strlen(name) + 1 > SHFS_PATH_MAX) {
		VERBOSE("Path too long!\n");
		return -1;
	}
	bufmode[0] = '\0';
	if (mode == O_RDONLY)
		strcpy(bufmode, "R");
	else if (mode == O_WRONLY)
		strcpy(bufmode, "W");
	else if (mode == O_RDWR)
		strcpy(bufmode, "RW");

	DEBUG("File open: %s (%s)\n", name, bufmode);
	
	if (!shfs_lock(info))
		return -EINTR;
	if ((res = do_open(info, name, bufmode)) < 0) {
		VERBOSE("Couldn't create directory!\n");
		shfs_unlock(info);
		return res;
	}
	shfs_pipe_readln(info, buf, SHFS_LINE_MAX);
	shfs_unlock(info);
	
	switch (reply(buf)) {
	case REP_COMPLETE:
		break;
	case REP_NOTEMPTY:
		return 1;	/* not empty file */
	case REP_EPERM:
		return -EPERM;
	case REP_ENOENT:
		return -ENOENT;
	default:
		return -EIO;
	}
	return 0;
}

/* data should be aligned (offset % count == 0) */
int
shfs_proc_read(struct dentry *dentry, unsigned offset, unsigned count, char *buffer, int slow)
{
	struct inode *inode = dentry->d_inode;
	struct super_block *sb = inode->i_sb;
	struct shfs_sb_info *info = (struct shfs_sb_info*)sb->u.generic_sbp;
	char file[SHFS_PATH_MAX + 6];
	char line[SHFS_LINE_MAX];
	int res;

	if (shfs_get_name(dentry, file) < 0)
		return -ENAMETOOLONG;
	DEBUG(" %s, off %u, %u bytes\n", file, (unsigned)offset, (unsigned)count);
	if (!slow && count && offset % count)
		VERBOSE("Reading unaligned data, could hang!\n");

	if (!slow)
		res = do_read(info, file, offset, count);
	else
		res = do_slow_read(info, file, offset, count, inode->i_ino);

	if (res < 0) {
		VERBOSE("Can't send read command!\n");
		return res;
	}
	if ((res = shfs_pipe_readln(info, line, SHFS_LINE_MAX)) < 0) {
		VERBOSE("Can't see read response!\n");
		info->mnt.garbage = 1;
		info->mnt.garbage_read = count;
		return res;
	}

	switch (reply(line)) {
	case REP_PRELIM:
		break;
	case REP_EPERM:
		return -EPERM;
	case REP_ENOENT:
		return -ENOENT;
	default:
		info->mnt.garbage = 1;
		info->mnt.garbage_read = count;
		return -EIO;
	}
	if (slow) {
		if ((res = shfs_pipe_readln(info, line, SHFS_LINE_MAX)) < 0) {
			VERBOSE("Can't see read response!\n");
			info->mnt.garbage = 1;
			info->mnt.garbage_read = count;
			return res;
		}
		count = simple_strtoul(line, NULL, 10);
	}

	if ((res = pipe_read(info, buffer, count)) < 0) {
		VERBOSE("Read: no data (%d)!\n", res);
		info->mnt.garbage = 1;
		info->mnt.garbage_read = count;
		return res;
	}
	if ((res = shfs_pipe_readln(info, line, SHFS_LINE_MAX)) < 0) {
		VERBOSE("Can't see read response!\n");
		return res;
	}
	
	switch (reply(line)) {
	case REP_COMPLETE:
		break;
	case REP_EPERM:
		return -EPERM;
	case REP_ENOENT:
		return -ENOENT;
	default:
		info->mnt.garbage = 1;
		return -EIO;
	}

	DEBUG(" received %u bytes\n", count);
	return count;
}

int
shfs_proc_write(struct dentry *dentry, unsigned offset, unsigned count, char *buffer){
	struct inode *inode = dentry->d_inode;
	struct super_block *sb = inode->i_sb;
	struct shfs_sb_info *info = (struct shfs_sb_info*)sb->u.generic_sbp;
	char file[SHFS_PATH_MAX];
	char line[SHFS_LINE_MAX];
	int res;
	
//	if (!count)
//		return 0;
	
	if (shfs_get_name(dentry, file) < 0)
		return -ENAMETOOLONG;
	DEBUG(" %s, off %u, %u bytes\n", file, offset, count);
		
	if ((res = do_write(info, file, offset, count, inode->i_ino)) < 0) {
		VERBOSE("Couldn't send write command!\n");
		goto out;
	}
	if ((res = shfs_pipe_readln(info, line, SHFS_LINE_MAX)) < 0) {
		VERBOSE("Cannot read response!\n");
		info->mnt.garbage = 1;
		info->mnt.garbage_write = count;
		goto out;
	}
	switch (reply(line)) {
	case REP_PRELIM:
		break;
	case REP_EPERM:
		res = -EPERM;
		goto out;
	case REP_ENOENT:
		res = -ENOENT;
		goto out;
	default:
		info->mnt.garbage = 1;
		res = -EIO;
		goto out;
	}

	if ((res = pipe_write(info, buffer, count)) < 0) {
		VERBOSE("Couldn't write file!\n");
		goto out;
	}
	if ((res = shfs_pipe_readln(info, line, SHFS_LINE_MAX)) < 0) {
		VERBOSE("Cannot read response!\n");
		goto out;
	}

	switch (reply(line)) {
	case REP_COMPLETE:
		break;
	case REP_EPERM:
		res = -EPERM;
		goto out;
	case REP_ENOENT:
		res = -ENOENT;
		goto out;
	case REP_ENOSPC:
		res = -ENOSPC;
		info->mnt.garbage = 1;
		info->mnt.garbage_write = count;
		goto out;
	default:
		info->mnt.garbage = 1;
		res = -EIO;
		goto out;
	}

	DEBUG(" sent %u bytes\n", count);
	return count;
out:
	return res;
}

int
shfs_proc_mkdir(struct shfs_sb_info *info, char *dir)
{
	int res;
	char buf[SHFS_LINE_MAX];
	
	if (strlen(dir) + 1 > SHFS_PATH_MAX) {
		VERBOSE("Path too long!\n");
		return -1;
	}

	DEBUG("Creating directory: %s\n", dir);
	
	if (!shfs_lock(info))
		return -EINTR;
	if ((res = do_mkdir(info, dir)) < 0) {
		VERBOSE("Could'n create directory!\n");
		shfs_unlock(info);
		return res;
	}

	shfs_pipe_readln(info, buf, SHFS_LINE_MAX);
	shfs_unlock(info);
	
	switch (reply(buf)) {
	case REP_COMPLETE:
		break;
	case REP_EPERM:
		return -EPERM;
	default:
		return -EIO;
	}
	return 0;
}

int
shfs_proc_rmdir(struct shfs_sb_info *info, char *dir)
{
	char buf[SHFS_LINE_MAX];
	int res;
	
	if (strlen(dir) + 1 > SHFS_PATH_MAX) {
		VERBOSE("Path too long!\n");
	        return -1;
	}

	DEBUG("Removing directory: %s\n", buf);
	
	if (!shfs_lock(info))
		return -EINTR;
	if ((res = do_rmdir(info, dir)) < 0) {
		VERBOSE("Couldn't remove directory!\n");
		shfs_unlock(info);
		return res;
	}

	shfs_pipe_readln(info, buf, SHFS_LINE_MAX);
	shfs_unlock(info);
	
	switch (reply(buf)) {
	case REP_COMPLETE:
		break;
	case REP_EPERM:
		return -EPERM;
	case REP_ENOENT:
		return -ENOENT;
	default:
		return -EIO;
	}
	return 0;
}

int
shfs_proc_rename(struct shfs_sb_info *info, char *old, char *new)
{
	char buf[SHFS_LINE_MAX];
	int res;
	
	if ((strlen(new) + 1 > SHFS_PATH_MAX) || (strlen(old) + 1 > SHFS_PATH_MAX)) {
		VERBOSE("Path too long!\n");
	        return -1;
	}

	DEBUG("Rename %s -> %s\n", old, new);
	
	if (!shfs_lock(info))
		return -EINTR;
	if ((res = do_mv(info, old, new)) < 0) {
		VERBOSE("Couldn't rename!\n");
		shfs_unlock(info);
		return res;
	}

	shfs_pipe_readln(info, buf, SHFS_LINE_MAX);
	shfs_unlock(info);
	
	switch (reply(buf)) {
	case REP_COMPLETE:
		break;
	case REP_EPERM:
		return -EPERM;
	case REP_ENOENT:
		return -ENOENT;
	default:
		return -EIO;
	}
	return 0;
}

int
shfs_proc_unlink(struct shfs_sb_info *info, char *file)
{
	char buf[SHFS_LINE_MAX];
	int res;
	
	if (strlen(file) + 1 > SHFS_PATH_MAX) {
		VERBOSE("Path too long!\n");
	        return -1;
	}

	DEBUG("Removing file: %s\n", file);
	
	if (!shfs_lock(info))
		return -EINTR;
	if ((res = do_rm(info, file)) < 0) {
		VERBOSE("Couldn't remove file!\n");
		shfs_unlock(info);
		return res;
	}

	shfs_pipe_readln(info, buf, SHFS_LINE_MAX);
	shfs_unlock(info);
	
	switch (reply(buf)) {
	case REP_COMPLETE:
		break;
	case REP_EPERM:
		return -EPERM;
	case REP_ENOENT:
		return -ENOENT;
	default:
		return -EIO;
	}
	return 0;
}

int
shfs_proc_create(struct shfs_sb_info *info, char *file, int mode)
{
	char buf[SHFS_LINE_MAX];
	int res;
	
	if (strlen(file) + 1 > SHFS_PATH_MAX) {
		VERBOSE("Path too long!\n");
	        return -1;
	}
	
	DEBUG("Creating file: %s\n", file);
	
	if (!shfs_lock(info))
		return -EINTR;
	if ((res = do_creat(info, file, mode & S_IALLUGO)) < 0) {
		VERBOSE("Couldn't remove file!\n");
		shfs_unlock(info);
		return res;
	}

	shfs_pipe_readln(info, buf, SHFS_LINE_MAX);
	shfs_unlock(info);
	
	switch (reply(buf)) {
	case REP_COMPLETE:
		break;
	case REP_EPERM:
		return -EPERM;
	default:
		return -EIO;
	}
	return 0;
}

int
shfs_proc_link(struct shfs_sb_info *info, char *old, char *new)
{
	char buf[SHFS_LINE_MAX];
	int res;
	
	if ((strlen(new) + 1 > SHFS_PATH_MAX) || (strlen(old) + 1 > SHFS_PATH_MAX)) {
		VERBOSE("Path too long!\n");
	        return -1;
	}

	DEBUG("Link %s -> %s\n", old, new);
	
	if (!shfs_lock(info))
		return -EINTR;
	if ((res = do_ln(info, old, new)) < 0) {
		VERBOSE("Couldn't make link!\n");
		shfs_unlock(info);
		return res;
	}

	shfs_pipe_readln(info, buf, SHFS_LINE_MAX);
	shfs_unlock(info);
	
	switch (reply(buf)) {
	case REP_COMPLETE:
		break;
	case REP_EPERM:
		return -EPERM;
	default:
		return -EIO;
	}
	return 0;
}

int
shfs_proc_symlink(struct shfs_sb_info *info, const char *old, const char *new)
{
	char buf[SHFS_LINE_MAX];
	int res;
	
	if ((strlen(new) + 1 > SHFS_PATH_MAX) || (strlen(old) + 1 > SHFS_PATH_MAX)) {
		VERBOSE("Path too long!\n");
	        return -1;
	}

	DEBUG("Symlink %s -> %s\n", old, new);
	
	if (!shfs_lock(info))
		return -EINTR;
	if ((res = do_sln(info, old, new)) < 0) {
		VERBOSE("Couldn't make symlink!\n");
		shfs_unlock(info);
		return res;
	}

	shfs_pipe_readln(info, buf, SHFS_LINE_MAX);
	shfs_unlock(info);
	
	switch (reply(buf)) {
	case REP_COMPLETE:
		break;
	case REP_EPERM:
		return -EPERM;
	default:
		return -EIO;
	}
	return 0;
}

int
shfs_proc_chmod(struct shfs_sb_info *info, char *file, umode_t mode)
{
	char buf[SHFS_LINE_MAX];
	int res;
	
	if (strlen(file) + 1 > SHFS_PATH_MAX) {
		VERBOSE("Path too long!\n");
	        return -1;
	}

	DEBUG("Chmod %o %s\n", mode, file);
	
	if (!shfs_lock(info))
		return -EINTR;
	if ((res = do_chmod(info, file, mode & S_IALLUGO)) < 0) {
		VERBOSE("Couldn't do chmod!\n");
		shfs_unlock(info);
		return res;
	}

	shfs_pipe_readln(info, buf, SHFS_LINE_MAX);
	shfs_unlock(info);
	
	switch (reply(buf)) {
	case REP_COMPLETE:
		break;
	case REP_EPERM:
		return -EPERM;
	case REP_ENOENT:
		return -ENOENT;
	default:
		return -EIO;
	}
	return 0;
}

int
shfs_proc_chown(struct shfs_sb_info *info, char *file, uid_t user)
{
	char buf[SHFS_LINE_MAX];
	int res;
	
	if (strlen(file) + 1 > SHFS_PATH_MAX) {
		VERBOSE("Path too long!\n");
	        return -1;
	}

	DEBUG("Chown %u %s\n", user, file);
	
	if (!shfs_lock(info))
		return -EINTR;
	if ((res = do_chown(info, file, user)) < 0) {
		VERBOSE("Couldn't do chown!\n");
		shfs_unlock(info);
		return res;
	}

	shfs_pipe_readln(info, buf, SHFS_LINE_MAX);
	shfs_unlock(info);
	
	switch (reply(buf)) {
	case REP_COMPLETE:
		break;
	case REP_EPERM:
		return -EPERM;
	case REP_ENOENT:
		return -ENOENT;
	default:
		return -EIO;
	}
	return 0;
}

int
shfs_proc_chgrp(struct shfs_sb_info *info, char *file, gid_t group)
{
	char buf[SHFS_LINE_MAX];
	int res;
	
	if (strlen(file) + 1 > SHFS_PATH_MAX) {
		VERBOSE("Path too long!\n");
	        return -1;
	}

	DEBUG("Chown %u %s\n", group, file);
	
	if (!shfs_lock(info))
		return -EINTR;
	if ((res = do_chgrp(info, file, group)) < 0) {
		VERBOSE("Couldn't do chgrp!\n");
		shfs_unlock(info);
		return res;
	}

	shfs_pipe_readln(info, buf, SHFS_LINE_MAX);
	shfs_unlock(info);
	
	switch (reply(buf)) {
	case REP_COMPLETE:
		break;
	case REP_EPERM:
		return -EPERM;
	case REP_ENOENT:
		return -ENOENT;
	default:
		return -EIO;
	}
	return 0;
}

int
shfs_proc_trunc(struct shfs_sb_info *info, char *file, loff_t size)
{
	char buf[SHFS_LINE_MAX];
	int res;
	
	if (strlen(file) + 1 > SHFS_PATH_MAX) {
		VERBOSE("Path too long!\n");
	        return -1;
	}

	DEBUG("Truncate %s %u\n", file, (unsigned)size);
	
	if (!shfs_lock(info))
		return -EINTR;
	if ((res = do_trunc(info, file, size)) < 0) {
		VERBOSE("Couldn't do truncate!\n");
		shfs_unlock(info);
		return res;
	}

	shfs_pipe_readln(info, buf, SHFS_LINE_MAX);
	shfs_unlock(info);
	
	switch (reply(buf)) {
	case REP_COMPLETE:
		break;
	case REP_EPERM:
		return -EPERM;
	case REP_ENOENT:
		return -ENOENT;
	default:
		return -EIO;
	}
	return 0;
}

/* this code is borrowed from dietlibc */

static int
__isleap(int year) {
  /* every fourth year is a leap year except for century years that are
   * not divisible by 400. */
/*  return (year % 4 == 0 && (year % 100 != 0 || year % 400 == 0)); */
  return (!(year%4) && ((year%100) || !(year%400)));
}

/* days per month -- nonleap! */
const short __spm[12] =
  { 0,
    (31),
    (31+28),
    (31+28+31),
    (31+28+31+30),
    (31+28+31+30+31),
    (31+28+31+30+31+30),
    (31+28+31+30+31+30+31),
    (31+28+31+30+31+30+31+31),
    (31+28+31+30+31+30+31+31+30),
    (31+28+31+30+31+30+31+31+30+31),
    (31+28+31+30+31+30+31+31+30+31+30),
  };

/* seconds per day */
#define SPD 24*60*60

static char *
proc_ctime(const time_t *timep, char *s)
{
	time_t i;
	int sec, min, hour;
	int day, month, year;
	time_t work = *timep % (SPD);
	
	sec = work % 60; work /= 60;
	min = work % 60; hour = work / 60;
	work = *timep / (SPD);

	for (i = 1970; ; i++) {
		time_t k = __isleap(i) ? 366 : 365;
		if (work >= k)
			work -= k;
		else
			break;
	}
	year = i;

	day=1;
	if (__isleap(i) && (work > 58)) {
		if (work == 59)
			day = 2; /* 29.2. */
		work -= 1;
	}

	for (i = 11; i && (__spm[i] > work); i--)
		;
	month = i;
 	day += work - __spm[i];

	sprintf(s, "%.4d%.2d%.2d%.2d%.2d.%.2d", year, month+1, day, hour, min, sec);

	return s;
}

int
shfs_proc_settime(struct shfs_sb_info *info, char *file, int atime, time_t *time)
{
	char buf[SHFS_LINE_MAX];
	char str[20];
	int res;
	
	if (strlen(file) + 1 > SHFS_PATH_MAX) {
		VERBOSE("Path too long!\n");
	        return -1;
	}

	proc_ctime(time, str);

	DEBUG("Settime %s (%s) %s\n", str, atime ? "atime" : "mtime", file);
	
	if (!shfs_lock(info))
		return -EINTR;
	if ((res = do_settime(info, file, atime, str)) < 0) {
		VERBOSE("Couldn't do settime!\n");
		shfs_unlock(info);
		return res;
	}

	shfs_pipe_readln(info, buf, SHFS_LINE_MAX);
	shfs_unlock(info);
	
	switch (reply(buf)) {
	case REP_COMPLETE:
		break;
	case REP_EPERM:
		return -EPERM;
	case REP_ENOENT:
		return -ENOENT;
	default:
		return -EIO;
	}
	return 0;
}

int 
shfs_init(struct shfs_sb_info *info)
{	
	char buf[SHFS_LINE_MAX];
	int i, known;
	int res;
	
	DEBUG("Identify remote host\n");
	
	if (!shfs_lock(info))
		return -EINTR;
	
	known = 0;
	for (i = 0; *shfs_system_identify[i]; i++) {
		if (info->mnt.generic_host)
			continue;
		if ((res = do_identify(info, i)) < 0) {
			VERBOSE("Couldn't get uname!\n");
			goto error;
		}
		if ((res = shfs_pipe_readln(info, buf, SHFS_LINE_MAX)) < 0) {
			VERBOSE("Error reading response!\n");
			goto error;
		}
		if (strcmp(buf, "unknown")) {
			printk(KERN_INFO "shfs: remote peer: %s\n", buf);
			known = 1;
			break;
		}
	}
	if (!known)
		printk(KERN_INFO "shfs: remote peer: generic\n");

	if ((res = do_init(info, i)) < 0) {
		VERBOSE("Cannot send init!\n");
		goto error;
	}

	if (shfs_pipe_readln(info, buf, SHFS_LINE_MAX) < 0) {
		VERBOSE("Error reading response!\n");
		goto error;
	}
	shfs_unlock(info);
	switch (reply(buf)) {
	case REP_COMPLETE:
		break;
	case REP_EPERM:
		return -EPERM;
	default:
		return -EIO;
	}
	return 0;
error:	
	shfs_unlock(info);
	return shfs_remove_sigpipe(res);
}

int
shfs_finish(struct shfs_sb_info *info)
{
	char buf[SHFS_LINE_MAX];
	int res;

	if (!shfs_lock(info))
		return -EINTR;
	if ((res = do_finish(info)) < 0) {
		VERBOSE("Cannot send finish!\n");
		goto error;
	}
	if (shfs_pipe_readln(info, buf, SHFS_LINE_MAX) < 0) {
		VERBOSE("Error reading response!\n");
		goto error;
	}
	shfs_unlock(info);
	switch (reply(buf)) {
	case REP_COMPLETE:
		break;
	case REP_EPERM:
		return -EPERM;
	default:
		return -EIO;
	}
	return 0;
error:	
	shfs_unlock(info);
	return res;
}

int
shfs_notify_change(struct dentry *dentry, struct iattr *attr)
{
	struct inode *inode = dentry->d_inode;
	struct shfs_sb_info *info = (struct shfs_sb_info*)inode->i_sb->u.generic_sbp;
	char file[SHFS_PATH_MAX];
	int error = 0;

	DEBUG(" \n");
	if (shfs_get_name(dentry, file) < 0)
		return -ENAMETOOLONG;
	error = inode_change_ok(inode, attr);
	if (error < 0)
		return error;

	if (attr->ia_valid & ATTR_MODE) {
		if ((error = shfs_proc_chmod(info, file, attr->ia_mode)) < 0)
			goto error;
	}
	if (attr->ia_valid & ATTR_UID) {
		if ((error = shfs_proc_chown(info, file, attr->ia_uid)) < 0)
			goto error;
	}
	if (attr->ia_valid & ATTR_GID) {
		if ((error = shfs_proc_chgrp(info, file, attr->ia_gid)) < 0)
			goto error;
	}
	if (attr->ia_valid & ATTR_SIZE) {
                filemap_fdatasync(inode->i_mapping);
                filemap_fdatawait(inode->i_mapping);
		if ((error = shfs_proc_trunc(info, file, attr->ia_size)) < 0)
			goto error;
		if ((error = vmtruncate(inode, attr->ia_size) != 0))
			goto error;
		if (!shfs_lock(info))
			return -EINTR;
		inode->i_size = attr->ia_size;
		mark_inode_dirty(inode);
		shfs_dcache_update_fattr(dentry, attr);
		shfs_unlock(info);
		return error;
	}
	if (attr->ia_valid & ATTR_ATIME) {
		if ((error = shfs_proc_settime(info, file, 1, &attr->ia_atime)) < 0)
			goto error;
	}
	if (attr->ia_valid & ATTR_MTIME) {
		if ((error = shfs_proc_settime(info, file, 0, &attr->ia_mtime)) < 0)
			goto error;
	}
error:
	if (!shfs_lock(info))
		return -EINTR;
	shfs_dcache_update_fattr(dentry, attr);
	shfs_unlock(info);
	return error;
}

