/*
 * systrace.h
 *
 * Copyright (c) 2002 Marius Aamodt Eriksen <marius@umich.edu>
 *
 * These definitions are muchly replicated from Niels Provos' OpenBSD
 * implementation.
 */

#ifndef _SYSTRACE_H
#define _SYSTRACE_H

/*
 * XXX this is kind of nasty -- should add manually to everything that
 * needs it
 */

#define SYSTR_EMULEN    8       /* sync with sys proc */

#ifdef __KERNEL__
/* XXX ugly.  argh... linux... */
typedef u32 register_t;
#endif __KERNEL__

struct str_msg_emul {
        char emul[SYSTR_EMULEN];
};

#define SYSTR_MAX_POLICIES      64
#define SYSTR_MAXARGS           64

/* XXX change register_t (args, rval) to something portable. */
struct str_msg_ask {
        int code;
        int argsize;
        u32 args[SYSTR_MAXARGS];
        u32 rval[2];
        int result;
};

/* Queued on fork or exit of a process */

struct str_msg_child {
        pid_t new_pid;
};
struct str_msg_ugid {
        uid_t uid;
        gid_t gid;
};

struct str_msg_execve {
	register_t patharg;
};

#define SYSTR_MSG_ASK     1
#define SYSTR_MSG_RES     2
#define SYSTR_MSG_EMUL    3
#define SYSTR_MSG_CHILD   4
#define SYSTR_MSG_UGID    5
#define SYSTR_MSG_EXECVE  6

#define SYSTR_MSG_NOPROCESS(x) \
        ((x)->msg.msg_type == SYSTR_MSG_CHILD)

#define MAXPATHLEN PATH_MAX

struct str_message {
	/* XXX - should be u_int16_t */
	int   msg_seqnr;
        int   msg_type;
        pid_t msg_pid;
        short msg_policy;
        short reserved;
        union {
                struct str_msg_emul    msg_emul;
                struct str_msg_ask     msg_ask;
                struct str_msg_child   msg_child;
                struct str_msg_ugid    msg_ugid;
                struct str_msg_execve  msg_execve;
        }     msg_data;
};

struct systrace_answer {
	/* XXX - should be u_int16_t */
	int   stra_seqnr;
        pid_t stra_pid;
        int   stra_policy;
        int   stra_error;
        int   stra_flags;
	uid_t stra_seteuid;     /* elevated privileges for system call */
        gid_t stra_setegid;
};

#define SYSTR_READ              1
#define SYSTR_WRITE             2

struct systrace_io {
        pid_t   strio_pid;
        int     strio_op;
        void   *strio_offs;
        void   *strio_addr;
        size_t  strio_len;
};

#define SYSTR_POLICY_NEW        1
#define SYSTR_POLICY_ASSIGN     2
#define SYSTR_POLICY_MODIFY     3

struct systrace_policy {
        int strp_op;
        int strp_num;
        union {
                struct {
                        short code;
                        short policy;
                } assign;
                pid_t pid;
                int maxents;
        } strp_data;
};


struct systrace_replace {
	pid_t strr_pid;
	int strr_nrepl;
	void *strr_base;
	size_t strr_len;
	int strr_argind[SYSTR_MAXARGS];
	size_t strr_off[SYSTR_MAXARGS];
	size_t strr_offlen[SYSTR_MAXARGS];
};

#define strp_pid        strp_data.pid
#define strp_maxents    strp_data.maxents
#define strp_code       strp_data.assign.code
#define strp_policy     strp_data.assign.policy

/* ioctl definitions */
#define STR_MAGIC 's'

#define STRIOCATTACH  _IOW(STR_MAGIC, 101, pid_t)
#define STRIOCDETACH  _IOW(STR_MAGIC, 102, pid_t)
#define STRIOCANSWER  _IOW(STR_MAGIC, 103, struct systrace_answer)
#define STRIOCIO      _IOWR(STR_MAGIC, 104, struct systrace_io)
#define STRIOCPOLICY  _IOWR(STR_MAGIC, 105, struct systrace_policy)
#define STRIOCGETCWD  _IOW(STR_MAGIC, 106, pid_t)
#define STRIOCRESCWD  _IO(STR_MAGIC, 107)
#define STRIOWAKE     _IO(STR_MAGIC, 108)
#define STRIOCLONE    _IOW(STR_MAGIC, 109, int *);
#define STRIOCREPLACE _IOW(STR_MAGIC, 110, struct systrace_replace)

#define SYSTR_POLICY_ASK        0
#define SYSTR_POLICY_PERMIT     1
#define SYSTR_POLICY_NEVER      2

#define SYSTR_FLAGS_RESULT      0x001
#define SYSTR_FLAGS_SETEUID     0x002
#define SYSTR_FLAGS_SETEGID     0x004

#ifdef __KERNEL__

struct str_process;
struct fsystrace {
	struct semaphore                      lock;
	wait_queue_head_t                     wqh;
        TAILQ_HEAD(strprocessq, str_process)  processes;
        TAILQ_HEAD(strpolicyq, str_policy)    policies;
        int                                   nprocesses;
        struct strprocessq                    messages;
        int                                   npolicynr;
        int                                   npolicies;

        int                                   issuser;
	uid_t                                 euid;
	gid_t                                 egid;

	pid_t                                 pid;
        /* cwd magic */
        pid_t                                 pwd_pid;
	struct vfsmount                      *pwd_mnt;
        struct dentry                        *pwd_dentry;
        struct vfsmount                      *root_mnt;
        struct dentry                        *root_dentry;
};

/* Internal prototypes */

/*
  int systrace_redirect(int, struct proc *, void *, register_t *);
  void systrace_exit(struct proc *);
  void systrace_fork(struct proc *, struct proc *);
*/

int  init_systrace(void);
void systrace_fork(struct task_struct *, struct task_struct *);
void systrace_exit(struct task_struct *);

/* crud needed to make systrace happy */
struct sysent {             /* system call table */
	short   sy_narg;    /* number of args */
	short   sy_argsize; /* total size of arguments */
};

/* Macros to set/clear/test flags. */
#define SET(t, f)       ((t) |= (f))
#define CLR(t, f)       ((t) &= ~(f))
#define ISSET(t, f)     ((t) & (f))


#endif /* __KERNEL__ */

#ifndef __KERNEL__
//typedef u_int32_t register_t;
#endif /* !__KERNEL__ */

#endif /* _SYSTRACE_H */
