/*
 *	Deadman	0.02:	A Software Deadman Device
 *
 *	(c) Copyright 2000 Eric Soderberg <seric@cup.hp.com>, All Rights Reserved.
 *				http://www.hp.com
 *
 *	This program is free software; you can redistribute it and/or
 *	modify it under the terms of the GNU General Public License
 *	as published by the Free Software Foundation; either version
 *	2 of the License, or (at your option) any later version.
 *	
 *	Neither Eric Soderberg or Hewlett-Packard admit liability nor provide 
 *	warranty for any of this software. This material is provided 
 *	"AS-IS" and at no charge.	
 *
 *      Fixes
 *      Release 0.02.
 *              Chad N. Tindel :        Port to 2.4.0-test7 kernel,
 *                                      added DEADMAN_MINOR as 185
 *      Release 0.01
 *              Eric Soderberg :        Initial Version for 2.2.16 kernel
 */
 
#include <linux/module.h>
#include <linux/config.h>
#include <linux/types.h>
#include <linux/kernel.h>
#include <linux/fs.h>
#include <linux/mm.h>
#include <linux/miscdevice.h>
#include <linux/reboot.h>
#include <linux/init.h>
#include <asm/uaccess.h>
#include <linux/deadman.h>

/* #define DEADMAN_TESTING 1 */

/*
 * deadman_timer is THE kernel timer.
 */

static struct timer_list deadman_timer;

/*
 * deadman_enabled is 1 when enabled and 0 when disabled.
 */
static int deadman_enabled = 0;

/*
 * If the timer expires..
 */

static void deadman_timer_fire(unsigned long data)
{
#ifdef DEADMAN_TESTING
	printk(KERN_CRIT "DEADMAN: Time expired, would restart machine.\n");
	(void)del_timer(&deadman_timer);
	deadman_enabled = 0;
	MOD_DEC_USE_COUNT;
#else
	printk(KERN_CRIT "DEADMAN: Time expired, initiating system restart.\n");
	machine_restart(NULL);
	printk(KERN_CRIT "DEADMAN: Restart failed!!!\n");
#endif
}

static int deadman_open(struct inode *inode, struct file *file)
{
	MOD_INC_USE_COUNT;
	return 0;
}

static int deadman_release(struct inode *inode, struct file *file)
{
	MOD_DEC_USE_COUNT;
	return 0;
}

static void deadman_set(struct timespec *value)
{
	/*
	 *	Refresh the timer.
	 */
	mod_timer(&deadman_timer, timespec_to_jiffies(value));
#ifdef DEADMAN_TESTING
	/*printk(KERN_DEBUG "DEADMAN: Set to %lu (%lus %lun)\n",*/
	printk(KERN_CRIT "DEADMAN: Set to %lu (%lus %lun)\n",
	 timespec_to_jiffies(value), value->tv_sec, value->tv_nsec);
#endif
	return;
}

static int deadman_ioctl(struct inode *inode, struct file *file,
 unsigned int cmd, unsigned long arg)
{
	static struct timespec tmp;

	switch(cmd) {

		case DEADMAN_QUERY_CURRENT:
			jiffies_to_timespec(jiffies, &tmp);
			if (copy_to_user((struct timespec *)arg, &tmp, sizeof(tmp))) {
				return -EFAULT;
			}
			return 0;

		case DEADMAN_ENABLE:
			if (current->uid != 0 && current->euid != 0) {
				return -EPERM;
			}
			if (deadman_enabled != 0) {
				printk(KERN_NOTICE "DEADMAN: Attempt to enable while "
				 "already enabled\n"); 
				return -EBUSY;
			}
			if (copy_from_user(&tmp, (struct timespec *)arg, sizeof(tmp))) {
				return -EFAULT;
			}
			printk("DEADMAN: Enabled\n");
			deadman_set(&tmp);
			deadman_enabled = 1;
			/*
			 * Add an additional module reference count to prevent
			 * unloading this module while the timer is running even if
			 * the caller closes the device
			 */
			MOD_INC_USE_COUNT;
			printk("DEADMAN: Enabled by process %d\n", current->pid);
			return 0;

		case DEADMAN_DISABLE:
			if (current->uid != 0 && current->euid != 0) {
				return -EPERM;
			}
			if (deadman_enabled == 0) {
				printk(KERN_NOTICE "DEADMAN: Attempt to disable while "
				 "already disabled\n"); 
				return -EINVAL;
			}
			del_timer(&deadman_timer);
			deadman_enabled = 0;

			MOD_DEC_USE_COUNT;
			printk("DEADMAN: Disabled\n");
			return 0;

		case DEADMAN_UPDATE:
			if (current->uid != 0 && current->euid != 0) {
				return -EPERM;
			}
			if (deadman_enabled == 0) {
				printk(KERN_NOTICE "DEADMAN: Attempt to update while "
	   			 "disabled\n"); 
				return -EINVAL;
			}
			if (copy_from_user(&tmp, (struct timespec *)arg, sizeof(tmp))) {
				return -EFAULT;
			}
			deadman_set(&tmp);
			return 0;

		default:
			return -ENOIOCTLCMD;
	} /* switch on command */
} /* deadman_ioctl() */

static struct file_operations deadman_fops =
{
	owner:		THIS_MODULE,
	ioctl:		deadman_ioctl,
	open:		deadman_open,
	release:	deadman_release
};

static struct miscdevice deadman_miscdev =
{
	DEADMAN_MINOR,
	"deadman",
	&deadman_fops
};

int __init deadman_init(void)
{
	misc_register(&deadman_miscdev);
	init_timer(&deadman_timer);
	deadman_timer.function = deadman_timer_fire;
	printk("Deadman: %s\n", DEADMAN_VERSION);
	return 0;
}

void deadman_cleanup(void)
{
	misc_deregister(&deadman_miscdev);
}

module_init(deadman_init);
module_exit(deadman_cleanup);

MODULE_LICENSE("GPL");
