/*
 *  inode.c
 *
 *  Copyright (C) 1995, 1996 by Paal-Kr. Engstad and Volker Lendecke
 *  Copyright (C) 1997 by Volker Lendecke
 *  Copyright (C) 2001 by SungHun Kim, edit and change for dav support
 *
 *  Please add a note about your changes to davfs in the ChangeLog file.
 */

#include <linux/config.h>
#include <linux/module.h>
#include <linux/sched.h>
#include <linux/kernel.h>
#include <linux/mm.h>
#include <linux/string.h>
#include <linux/stat.h>
#include <linux/errno.h>
#include <linux/smp_lock.h>
#include <linux/locks.h>
#include <linux/slab.h>
#include <linux/init.h>
#include <linux/file.h>
#include <linux/dcache.h>

#include <linux/ncp_no.h>
/*
#include <linux/dav_fs.h>
#include <linux/davno.h>
#include <linux/dav_mount.h>

*/
#include <asm/system.h>
#include <asm/uaccess.h>

#include "davfs.h"
#include "dav_debug.h"

static void dav_delete_inode(struct inode *);
static void dav_put_super(struct super_block *);
static int  dav_statfs(struct super_block *, struct statfs *);

static struct super_operations dav_sops =
{
	put_inode:	force_delete,
	delete_inode:	dav_delete_inode,
	put_super:	dav_put_super,
	statfs:		dav_statfs,
};


/* We are always generating a new inode here */
struct inode *
dav_iget(struct super_block *sb, struct dav_fattr *fattr)
{
    struct inode *result;
    struct dav_i_info *dav_i_info;
    
    TRACE();

    result = new_inode(sb);
    if (!result)
	return result;
    
    result->i_ino = fattr->f_ino;
    
    /* malloc for dav_i_info */
    dav_i_info = (struct dav_i_info *) 
      kmalloc(sizeof(struct dav_i_info), GFP_KERNEL);
    
    if(!dav_i_info)
	return NULL;
    
    memset(dav_i_info, 0x00, sizeof(struct dav_i_info));
    
    /* we need to use generic pointer */
    result->u.generic_ip = dav_i_info;
    
    dav_set_inode_attr(result, fattr);
    
    if (S_ISREG(result->i_mode)) {
	result->i_op = &dav_file_inode_operations;
	result->i_fop = &dav_file_operations;
	result->i_data.a_ops = &dav_file_aops;
    } else if (S_ISDIR(result->i_mode)) {
	result->i_op = &dav_dir_inode_operations;
	result->i_fop = &dav_dir_operations;
    }
    insert_inode_hash(result);
    
    DEBUG1("dav_iget: %p\n", fattr);
    
    return result;
}

/*
 * Copy the inode data to a dav_fattr structure.
 */
void
dav_get_inode_attr(struct inode *inode, struct dav_fattr *fattr)
{
    struct dav_i_info *dii = inode->u.generic_ip;

    DEBUG1("here\n");

    memset(fattr, 0, sizeof(struct dav_fattr));
    fattr->f_mode	= inode->i_mode;
    fattr->f_nlink	= inode->i_nlink;
    fattr->f_ino	= inode->i_ino;
    fattr->f_uid	= inode->i_uid;
    fattr->f_gid	= inode->i_gid;
    fattr->f_rdev	= inode->i_rdev;
    fattr->f_size	= inode->i_size;
    fattr->f_mtime	= inode->i_mtime;
    fattr->f_ctime	= inode->i_ctime;
    fattr->f_atime	= inode->i_atime;
    fattr->f_blksize= inode->i_blksize;
    fattr->f_blocks	= inode->i_blocks;
    
    fattr->attr	= dii->attr;
    /*
     * Keep the attributes in sync with the inode permissions.
     */
    if (fattr->f_mode & S_IWUSR)
	fattr->attr &= ~aRONLY;
    else
	fattr->attr |= aRONLY;
}

void
dav_set_inode_attr(struct inode *inode, struct dav_fattr *fattr)
{
    struct dav_i_info *dii = inode->u.generic_ip;

    TRACE();
    
    inode->i_mode	= fattr->f_mode;
    inode->i_nlink	= fattr->f_nlink;
    inode->i_uid	= fattr->f_uid;
    inode->i_gid	= fattr->f_gid;
    inode->i_rdev	= fattr->f_rdev;
    inode->i_ctime	= fattr->f_ctime;
    inode->i_blksize    = fattr->f_blksize;
    inode->i_blocks     = fattr->f_blocks;
    /*
     * Don't change the size and mtime/atime fields
     * if we're writing to the file.
     */
    /*
    if (!(dii->flags & DAV_F_LOCALWRITE)) 
    */
    {
	inode->i_size  = fattr->f_size;
	inode->i_mtime = fattr->f_mtime;
	inode->i_atime = fattr->f_atime;
    }

    if(dii) {
    
       dii->attr = fattr->attr;
      /*
       * Update the "last time refreshed" field for revalidation.
       */
      dii->oldmtime = jiffies;
    }
}

/*
 * This is called if the connection has gone bad ...
 * try to kill off all the current inodes.
 */
void
dav_invalidate_inodes(struct dav_sb_info *server)
{
	VERBOSE("\n");
	shrink_dcache_sb(dav_SB_of(server));
	invalidate_inodes(dav_SB_of(server));
}

/*
 * This is called to update the inode attributes after
 * we've made changes to a file or directory.
 */
static int
dav_refresh_inode(struct dentry *dentry)
{
    struct inode *inode = dentry->d_inode;
    int error;
    struct dav_fattr fattr;
    
    DEBUG1("here\n");
    error = dav_proc_getattr(dentry, &fattr);
    
    if (!error) {
	dav_renew_times(dentry);
	/*
	 * Check whether the type part of the mode changed,
	 * and don't update the attributes if it did.
	 */
	if ((inode->i_mode & S_IFMT) == (fattr.f_mode & S_IFMT)) {
	    dav_set_inode_attr(inode, &fattr);
	} else {
	    /*
	     * Big trouble! The inode has become a new object,
	     * so any operations attempted on it are invalid.
	     *
	     * To limit damage, mark the inode as bad so that
	     * subsequent lookup validations will fail.
	     */
	    PARANOIA("%s/%s changed mode, %07o to %07o\n",
		     DENTRY_PATH(dentry),
		     inode->i_mode, fattr.f_mode);
	    
	    fattr.f_mode = inode->i_mode; /* save mode */
	    make_bad_inode(inode);
	    inode->i_mode = fattr.f_mode; /* restore mode */
	    /*
	     * No need to worry about unhashing the dentry: the
	     * lookup validation will see that the inode is bad.
	     * But we do want to invalidate the caches ...
	     */
	    if (!S_ISDIR(inode->i_mode))
		invalidate_inode_pages(inode);
	    else
		dav_invalid_dir_cache(inode);
	    error = -EIO;
	}
    }
    return error;
}

/*
 * This is called when we want to check whether the inode
 * has changed on the server.  If it has changed, we must
 * invalidate our local caches.
 */
int
dav_revalidate_inode(struct dentry *dentry)
{
	struct dav_sb_info *s = dav_server_from_dentry(dentry);
       	struct inode *inode = dentry->d_inode;
	struct dav_i_info *dii = inode->u.generic_ip;
	time_t last_time;
	loff_t last_sz;
	int error = 0;

	TRACE();

	VERBOSE(" for dentry %s/%s\n",
               DENTRY_PATH(dentry));

	lock_kernel();

	/*
	 * Check whether we've recently refreshed the inode.
	 */
	if (time_before(jiffies, dii->oldmtime + DAV_MAX_AGE(s))) {
		VERBOSE("up-to-date, ino=%ld, jiffies=%lu, oldtime=%lu\n",
			inode->i_ino, jiffies, dii->oldmtime);
		goto out;
	}

	/*
	 * Save the last modified time, then refresh the inode.
	 * (Note: a size change should have a different mtime,
	 *  or same mtime but different size.)
	 */
	last_time = inode->i_mtime;
	last_sz   = inode->i_size;
	error = dav_refresh_inode(dentry);
	
	if (error || inode->i_mtime != last_time || inode->i_size != last_sz) {
		VERBOSE("%s/%s changed, old=%ld, new=%ld\n",
			DENTRY_PATH(dentry),
			(long) last_time, (long) inode->i_mtime);

		if (!S_ISDIR(inode->i_mode))
			invalidate_inode_pages(inode);
	}
out:
	unlock_kernel();
	return error;
}

/*
 * This routine is called when i_nlink == 0 and i_count goes to 0.
 * All blocking cleanup operations need to go here to avoid races.
 */
static void
dav_delete_inode(struct inode *ino)
{
	int ret;

	DEBUG1("ino=%ld\n", ino->i_ino);
	lock_kernel();
	
	
	if(ino->u.generic_ip) 
	    kfree(ino->u.generic_ip);
	
	if ((ret=dav_close(ino)))
		PARANOIA("could not close inode %ld(%d)\n", ino->i_ino, ret);

	unlock_kernel();

	clear_inode(ino);
}


static void
dav_put_super(struct super_block *sb)
{
    struct dav_sb_info *server = sb->u.generic_sbp;
    
    TRACE();
	
    /* disconnect with davfsd */
    dav_proc_disconnect(server);

    /* free general buf */
    kfree(server->buf);
    kfree(server->req);

    /*
      if (server->sock_file) {
      dav_proc_disconnect(server);
      dav_dont_catch_keepalive(server);
      fput(server->sock_file);
      }
      
      if (server->conn_pid)
      kill_proc(server->conn_pid, SIGTERM, 1);
    */
    
    /*
      kfree(sb->u.davfs_sb.temp_buf);
      
      if (server->packet)
      dav_vfree(server->packet);
    */	
}

char * dav_strdup(char *str) {
    char *ret =  kmalloc(strlen(str)+1 ,GFP_KERNEL);
    if(!ret) 
	return ret;

    memset(ret, 0, sizeof(strlen(str)+1));
    strcpy(ret, str);

    return ret;
}

struct super_block *
dav_read_super(struct super_block *sb, void *raw_data, int silent)
{
    struct dav_mount_data *davmnt;
    struct inode *root_inode;
    struct dav_fattr root;
    struct dav_sb_info *server;

    TRACE();

    if (!raw_data)
	goto out_no_data;
    
    davmnt = (struct dav_mount_data *) raw_data;

    /* malloc for sb */
    server = (struct dav_sb_info *) 
      kmalloc(sizeof(struct dav_sb_info) ,GFP_KERNEL);
    if(!server)
	goto out_no_mem;
    
    memset(server, 0, sizeof(*server));
    
    /* copy data from mount information to server data */
    server->flags = davmnt->flags;
    server->ttl = 1000;
    server->uid = davmnt->uid;
    server->gid = davmnt->gid;
   
    server->dir_mode  = S_IRWXU | S_IRWXG | S_IRWXO | S_IFDIR;
    server->file_mode = S_IRWXU | S_IRWXG | S_IRWXO | S_IFREG;
    server->sb = sb;

    server->path = dav_strdup(davmnt->path);
    if(!server->path)
	goto out_no_mem;

    /* socket handler to communicate davfsd */
    server->sock = dav_get_sock(server, davmnt->dav_fd);
    if(!server->sock) 
	goto out_bad_sock;

    /* init_ MUTEX and waitques */
    init_MUTEX(&(server->sem));
    init_waitqueue_head(&server->wait);

    /* memory alloc for general buf */
    server->buf = kmalloc(DAV_MAXPATHLEN ,GFP_KERNEL);
    server->req = kmalloc(DAV_MAXPATHLEN ,GFP_KERNEL);
    if(!server->buf || !server->req)
	goto out_no_mem;

 
#ifdef SERVER_TEST
    /* test server */
    dav_server_test(server);
#endif

    /* assign generic sbp */
    sb->u.generic_sbp = server;
	
    sb->s_blocksize = 1024;	/* Eh...  Is this correct? */
    sb->s_blocksize_bits = 12;
    sb->s_magic = DAV_SUPER_MAGIC;
    sb->s_flags = 0;
    sb->s_op = &dav_sops;
    sb->s_maxbytes = MAX_NON_LFS;	/* client support missing */
	
    dav_init_root_dirent(server, &root);

    DEBUG1("root_mode : %d\n", root.f_mode);
    root_inode = dav_iget(sb, &root);
    if (!root_inode)
	goto out_no_root;
    
    sb->s_root = d_alloc_root(root_inode);
    if (!sb->s_root)
	goto out_no_root;
    dav_new_dentry(sb->s_root);
	
    PRINT_INODE(root_inode);
    PRINT_DENTRY(sb->s_root);

    DEBUG1("org_mode %d\n", server->dir_mode);

    return sb;
    
out_no_root:
	iput(root_inode);
out_bad_sock:
	printk(KERN_ERR "dav_get_socket: no socket\n");
out_no_mem:
	/*
	if (!sb->u.davfs_sb.mnt)
		printk(KERN_ERR "dav_read_super: allocation failure\n");
	*/
	goto out_fail;
out_no_data:
	printk(KERN_ERR "dav_read_super: missing data argument\n");
out_fail:
	return NULL;
}

static int
dav_statfs(struct super_block *sb, struct statfs *buf)
{
    /*NetWare Server, because free space is distributed over
      volumes, and the current user might have disk quotas. So
      free space is not that simple to determine. Our decision
      here is to err conservatively. */
    
    DEBUG1("here\n");

    /* get statfs information from davfsd */
    dav_proc_statfs(sb->s_root, buf);

    return 0;
}

int
dav_notify_change(struct dentry *dentry, struct iattr *attr)
{
	struct inode *inode = dentry->d_inode;
	struct dav_sb_info *server = dav_server_from_dentry(dentry);
	struct dav_i_info *dii = DAV_INOP(inode);
	unsigned int mask = (S_IFREG | S_IFDIR | S_IRWXU | S_IRWXG | S_IRWXO);
	int error, changed, refresh = 0;
	struct dav_fattr fattr;

	TRACE();

	error = dav_revalidate_inode(dentry);
	if (error)
		goto out;

	if ((error = inode_change_ok(inode, attr)) < 0)
		goto out;

	error = -EPERM;
	if ((attr->ia_valid & ATTR_UID) && (attr->ia_uid != server->uid))
		goto out;

	if ((attr->ia_valid & ATTR_GID) && (attr->ia_uid != server->gid))
		goto out;

	if ((attr->ia_valid & ATTR_MODE) && (attr->ia_mode & ~mask))
		goto out;

	if ((attr->ia_valid & ATTR_SIZE) != 0)
	{
		VERBOSE("changing %s/%s, old size=%ld, new size=%ld\n",
			DENTRY_PATH(dentry),
			(long) inode->i_size, (long) attr->ia_size);
		error = dav_open(dentry, O_WRONLY);
		if (error)
			goto out;
		error = dav_proc_trunc(server, dii->fileid,
					 attr->ia_size);
		if (error)
			goto out;
		vmtruncate(inode, attr->ia_size);
		refresh = 1;
	}

	/*
	 * Initialize the fattr and check for changed fields.
	 * Note: CTIME under DAV is creation time rather than
	 * change time, so we don't attempt to change it.
	 */
	dav_get_inode_attr(inode, &fattr);

	changed = 0;
	if ((attr->ia_valid & ATTR_MTIME) != 0)
	{
		fattr.f_mtime = attr->ia_mtime;
		changed = 1;
	}
	if ((attr->ia_valid & ATTR_ATIME) != 0)
	{
		fattr.f_atime = attr->ia_atime;

	}
	if (changed)
	{
		error = dav_proc_setattr(dentry, &fattr);
		if (error)
			goto out;
		refresh = 1;
	}

	/*
	 * Check for mode changes ... we're extremely limited in
	 * what can be set for DAV servers: just the read-only bit.
	 */
	if ((attr->ia_valid & ATTR_MODE) != 0)
	{
		VERBOSE("%s/%s mode change, old=%x, new=%x\n",
			DENTRY_PATH(dentry), fattr.f_mode, attr->ia_mode);
		changed = 0;
		if (attr->ia_mode & S_IWUSR)
		{
			if (fattr.attr & aRONLY)
			{
				fattr.attr &= ~aRONLY;
				changed = 1;
			}
		} else {
			if (!(fattr.attr & aRONLY))
			{
				fattr.attr |= aRONLY;
				changed = 1;
			}
		}
		if (changed)
		{
			error = dav_proc_setattr(dentry, &fattr);
			if (error)
				goto out;
			refresh = 1;
		}
	}
	error = 0;

out:
	if (refresh)
		dav_refresh_inode(dentry);
	return error;
}

#ifdef DEBUG_DAV_MALLOC
int dav_malloced;
int dav_current_kmalloced;
int dav_current_vmalloced;
#endif

static DECLARE_FSTYPE( dav_fs_type, "davfs", dav_read_super, 0);

static int __init init_dav_fs(void)
{
	DEBUG1("registering ...\n");

#ifdef DEBUG_DAV_MALLOC
	dav_malloced = 0;
	dav_current_kmalloced = 0;
	dav_current_vmalloced = 0;
#endif

	return register_filesystem(&dav_fs_type);
}

static void __exit exit_dav_fs(void)
{
	DEBUG1("unregistering ...\n");
	unregister_filesystem(&dav_fs_type);
#ifdef DEBUG_DAV_MALLOC
	printk(KERN_DEBUG "dav_malloced: %d\n", dav_malloced);
	printk(KERN_DEBUG "dav_current_kmalloced: %d\n",dav_current_kmalloced);
	printk(KERN_DEBUG "dav_current_vmalloced: %d\n",dav_current_vmalloced);
#endif
}

EXPORT_NO_SYMBOLS;

/* 
** Added by Philipp Hahn (pmhahn)
*/
MODULE_AUTHOR("Sung Kim <hunkim@cs.ucsc.edu>");
MODULE_DESCRIPTION("Web Distributed Authoring and Versioning Filesystem");
MODULE_LICENSE("GPL");
 
module_init(init_dav_fs)
module_exit(exit_dav_fs)
