/*
 *  linux/fs/open.c
 *
 *  Copyright (C) 1991, 1992  Linus Torvalds
 *  Linux Trustees support added (c) 1999 Vyacheslav Zavadsky
 */

#include <linux/string.h>
#include <linux/mm.h>
#include <linux/utime.h>
#include <linux/file.h>
#include <linux/smp_lock.h>
#include <linux/quotaops.h>
#include <linux/dnotify.h>
#include <linux/module.h>
#include <linux/slab.h>
#include <linux/tty.h>
#include <linux/iobuf.h>
#include <linux/grsecurity.h>

#include <asm/uaccess.h>

#define special_file(m) (S_ISCHR(m)||S_ISBLK(m)||S_ISFIFO(m)||S_ISSOCK(m))

/* RSBAC */
#ifdef CONFIG_RSBAC
#include <rsbac/adf.h>
#endif

int vfs_statfs(struct super_block *sb, struct statfs *buf)
{
	int retval = -ENODEV;

	if (sb) {
		retval = -ENOSYS;
		if (sb->s_op && sb->s_op->statfs) {
			memset(buf, 0, sizeof(struct statfs));
			lock_kernel();
			retval = sb->s_op->statfs(sb, buf);
			unlock_kernel();
		}
	}
	return retval;
}


asmlinkage long sys_statfs(const char * path, struct statfs * buf)
{
	struct nameidata nd;
	int error;

        /* RSBAC */
        #ifdef CONFIG_RSBAC
        enum  rsbac_target_t          rsbac_target;
        union rsbac_target_id_t       rsbac_target_id;
        union rsbac_attribute_value_t rsbac_attribute_value;
        #endif

	error = user_path_walk(path, &nd);
	if (!error) {
		struct statfs tmp;

		/* RSBAC */
		#ifdef CONFIG_RSBAC
#ifdef CONFIG_RSBAC_DEBUG
		if (rsbac_debug_aef)
		  printk(KERN_DEBUG "sys_statfs(): calling ADF\n");
#endif
		rsbac_target = T_FILE;
		if (S_ISDIR(nd.dentry->d_inode->i_mode))
		  rsbac_target = T_DIR;
		else if (S_ISFIFO(nd.dentry->d_inode->i_mode))
		  rsbac_target = T_FIFO;
		else if (S_ISLNK(nd.dentry->d_inode->i_mode))
		  rsbac_target = T_SYMLINK;
       	        rsbac_target_id.file.device = nd.dentry->d_inode->i_dev;
		rsbac_target_id.file.inode  = nd.dentry->d_inode->i_ino;
		rsbac_target_id.file.dentry_p = nd.dentry;
		rsbac_attribute_value.dummy = 0;
		if (!rsbac_adf_request(R_GET_STATUS_DATA,
				       current->pid,
				       rsbac_target,
				       rsbac_target_id,
				       A_none,
				       rsbac_attribute_value))
		  {
		    path_release(&nd);
		    return -EPERM;
		  }
		#endif

		error = vfs_statfs(nd.dentry->d_inode->i_sb, &tmp);
		if (!error && copy_to_user(buf, &tmp, sizeof(struct statfs)))
			error = -EFAULT;
		path_release(&nd);
	}
	return error;
}

asmlinkage long sys_fstatfs(unsigned int fd, struct statfs * buf)
{
	struct file * file;
	struct statfs tmp;
	int error;

        /* RSBAC */
        #ifdef CONFIG_RSBAC
        enum  rsbac_target_t          rsbac_target;
        union rsbac_target_id_t       rsbac_target_id;
        union rsbac_attribute_value_t rsbac_attribute_value;
        #endif

	error = -EBADF;
	file = fget(fd);
	if (!file)
		goto out;

	/* RSBAC */
	#ifdef CONFIG_RSBAC
#ifdef CONFIG_RSBAC_DEBUG
	if (rsbac_debug_aef)
	  printk(KERN_DEBUG "sys_fstatfs(): calling ADF\n");
#endif
	rsbac_target = T_FILE;
	if (S_ISDIR(file->f_dentry->d_inode->i_mode))
	  rsbac_target = T_DIR;
	else if (S_ISFIFO(file->f_dentry->d_inode->i_mode))
	  rsbac_target = T_FIFO;
	else if (S_ISLNK(file->f_dentry->d_inode->i_mode))
	  rsbac_target = T_SYMLINK;
	rsbac_target_id.file.device = file->f_dentry->d_inode->i_dev;
	rsbac_target_id.file.inode  = file->f_dentry->d_inode->i_ino;
	rsbac_target_id.file.dentry_p = file->f_dentry;
	rsbac_attribute_value.dummy = 0;
	if (!rsbac_adf_request(R_GET_STATUS_DATA,
			       current->pid,
			       rsbac_target,
			       rsbac_target_id,
			       A_none,
			       rsbac_attribute_value))
	  {
	    fput(file);
	    return -EPERM;
	  }
	#endif

	error = vfs_statfs(file->f_dentry->d_inode->i_sb, &tmp);
	if (!error && copy_to_user(buf, &tmp, sizeof(struct statfs)))
		error = -EFAULT;
	fput(file);
out:
	return error;
}

/*
 * Install a file pointer in the fd array.  
 *
 * The VFS is full of places where we drop the files lock between
 * setting the open_fds bitmap and installing the file in the file
 * array.  At any such point, we are vulnerable to a dup2() race
 * installing a file in the array before us.  We need to detect this and
 * fput() the struct file we are about to overwrite in this case.
 *
 * It should never happen - if we allow dup2() do it, _really_ bad things
 * will follow.
 */

void fd_install(unsigned int fd, struct file * file)
{
	struct files_struct *files = current->files;
	
	write_lock(&files->file_lock);
	if (files->fd[fd])
		BUG();
	files->fd[fd] = file;
	write_unlock(&files->file_lock);
}

int do_truncate(struct dentry *dentry, loff_t length, struct vfsmount *mnt)
{
	struct inode *inode = dentry->d_inode;
	int error;
	struct iattr newattrs;

        /* RSBAC */
        #ifdef CONFIG_RSBAC
        union rsbac_target_id_t       rsbac_target_id;
        union rsbac_target_id_t       rsbac_new_target_id;
        union rsbac_attribute_value_t rsbac_attribute_value;
        #endif
        #ifdef CONFIG_RSBAC_SECDEL
        loff_t old_len = inode->i_size;
        #endif


	/* Not pretty: "inode->i_size" shouldn't really be signed. But it is. */
	if (length < 0)
		return -EINVAL;

        /* RSBAC */
        #ifdef CONFIG_RSBAC
#ifdef CONFIG_RSBAC_DEBUG
        if (rsbac_debug_aef)
          printk(KERN_DEBUG "do_truncate() [open_namei(), do_sys_truncate() [sys_truncate()]]: calling ADF\n");
#endif
        rsbac_target_id.file.device = inode->i_dev;
        rsbac_target_id.file.inode  = inode->i_ino;
        rsbac_target_id.file.dentry_p = dentry;
        rsbac_attribute_value.dummy = 0;
        if (!rsbac_adf_request(R_TRUNCATE,
                               current->pid,
                               T_FILE,
                               rsbac_target_id,
                               A_none,
                               rsbac_attribute_value))
          {
            return -EPERM;
          }
        #endif

	if (!gr_acl_handle_truncate(dentry, mnt))
		return -EACCES;

	down_write(&inode->i_alloc_sem);
	down(&inode->i_sem);
	newattrs.ia_size = length;
	newattrs.ia_valid = ATTR_SIZE | ATTR_CTIME;
	error = notify_change(dentry, &newattrs);
	up(&inode->i_sem);
	up_write(&inode->i_alloc_sem);
	return error;
}

static inline long do_sys_truncate(const char * path, loff_t length)
{
	struct nameidata nd;
	struct inode * inode;
	int error;

	error = -EINVAL;
	if (length < 0)	/* sorry, but loff_t says... */
		goto out;

	error = user_path_walk(path, &nd);
	if (error)
		goto out;
	inode = nd.dentry->d_inode;

	/* For directories it's -EISDIR, for other non-regulars - -EINVAL */
	error = -EISDIR;
	if (S_ISDIR(inode->i_mode))
		goto dput_and_out;

	error = -EINVAL;
	if (!S_ISREG(inode->i_mode))
		goto dput_and_out;

	error = permission_dentry(nd.dentry,MAY_WRITE);
	if (error)
		goto dput_and_out;

	error = -EROFS;
	if (IS_RDONLY(inode))
		goto dput_and_out;

	error = -EPERM;
	if (IS_IMMUTABLE(inode) || IS_APPEND(inode))
		goto dput_and_out;

	/*
	 * Make sure that there are no leases.
	 */
	error = get_lease(inode, FMODE_WRITE);
	if (error)
		goto dput_and_out;

	error = get_write_access(inode);
	if (error)
		goto dput_and_out;

	error = locks_verify_truncate(inode, NULL, length);
	if (!error) {
		DQUOT_INIT(inode);
		error = do_truncate(nd.dentry, length, nd.mnt);
	}
	put_write_access(inode);

dput_and_out:
	path_release(&nd);
out:
	return error;
}

asmlinkage long sys_truncate(const char * path, unsigned long length)
{
	/* on 32-bit boxen it will cut the range 2^31--2^32-1 off */
	return do_sys_truncate(path, (long)length);
}

static inline long do_sys_ftruncate(unsigned int fd, loff_t length, int small)
{
	struct inode * inode;
	struct dentry *dentry;
	struct file * file;
	int error;

	error = -EINVAL;
	if (length < 0)
		goto out;
	error = -EBADF;
	file = fget(fd);
	if (!file)
		goto out;

	/* explicitly opened as large or we are on 64-bit box */
	if (file->f_flags & O_LARGEFILE)
		small = 0;

	dentry = file->f_dentry;
	inode = dentry->d_inode;
	error = -EINVAL;
	if (!S_ISREG(inode->i_mode) || !(file->f_mode & FMODE_WRITE))
		goto out_putf;

	error = -EINVAL;
	/* Cannot ftruncate over 2^31 bytes without large file support */
	if (small && length > MAX_NON_LFS)
		goto out_putf;

	error = -EPERM;
	if (IS_APPEND(inode))
		goto out_putf;

	error = locks_verify_truncate(inode, file, length);
	if (!error)
		error = do_truncate(dentry, length, file->f_vfsmnt);
out_putf:
	fput(file);
out:
	return error;
}

asmlinkage long sys_ftruncate(unsigned int fd, unsigned long length)
{
	return do_sys_ftruncate(fd, length, 1);
}

/* LFS versions of truncate are only needed on 32 bit machines */
#if BITS_PER_LONG == 32
asmlinkage long sys_truncate64(const char * path, loff_t length)
{
	return do_sys_truncate(path, length);
}

asmlinkage long sys_ftruncate64(unsigned int fd, loff_t length)
{
	return do_sys_ftruncate(fd, length, 0);
}
#endif

#if !(defined(__alpha__) || defined(__ia64__))

/*
 * sys_utime() can be implemented in user-level using sys_utimes().
 * Is this for backwards compatibility?  If so, why not move it
 * into the appropriate arch directory (for those architectures that
 * need it).
 */

/* If times==NULL, set access and modification to current time,
 * must be owner or have write permission.
 * Else, update from *times, must be owner or super user.
 */
asmlinkage long sys_utime(char * filename, struct utimbuf * times)
{
	int error;
	struct nameidata nd;
	struct inode * inode;
	struct iattr newattrs;

        /* RSBAC */
        #ifdef CONFIG_RSBAC
        enum  rsbac_target_t          rsbac_target;
        union rsbac_target_id_t       rsbac_target_id;
        union rsbac_attribute_value_t rsbac_attribute_value;
        #endif

	error = user_path_walk(filename, &nd);
	if (error)
		goto out;
	inode = nd.dentry->d_inode;

	error = -EROFS;
	if (IS_RDONLY(inode))
		goto dput_and_out;

	/* Don't worry, the checks are done in inode_change_ok() */
	newattrs.ia_valid = ATTR_CTIME | ATTR_MTIME | ATTR_ATIME;
	if (times) {
		error = -EPERM;
		if (IS_APPEND(inode) || IS_IMMUTABLE(inode))
			goto dput_and_out;
		error = get_user(newattrs.ia_atime, &times->actime);
		if (!error) 
			error = get_user(newattrs.ia_mtime, &times->modtime);
		if (error)
			goto dput_and_out;

		newattrs.ia_valid |= ATTR_ATIME_SET | ATTR_MTIME_SET;
	} else {
		error = -EACCES;
		if (IS_IMMUTABLE(inode))
			goto dput_and_out;
		if (current->fsuid != inode->i_uid &&
		    (error = permission_dentry(nd.dentry,MAY_WRITE)) != 0)
			goto dput_and_out;
	}

	if (!gr_acl_handle_utime(nd.dentry, nd.mnt)) {
		error = -EACCES;
		goto dput_and_out;
	}

	error = notify_change(nd.dentry, &newattrs);
dput_and_out:
	path_release(&nd);
out:
	return error;
}

#endif

/* If times==NULL, set access and modification to current time,
 * must be owner or have write permission.
 * Else, update from *times, must be owner or super user.
 */
asmlinkage long sys_utimes(char * filename, struct timeval * utimes)
{
	int error;
	struct nameidata nd;
	struct inode * inode;
	struct iattr newattrs;

        /* RSBAC */
        #ifdef CONFIG_RSBAC
        enum  rsbac_target_t          rsbac_target;
        union rsbac_target_id_t       rsbac_target_id;
        union rsbac_attribute_value_t rsbac_attribute_value;
        #endif

	error = user_path_walk(filename, &nd);

	if (error)
		goto out;
	inode = nd.dentry->d_inode;

	error = -EROFS;
	if (IS_RDONLY(inode))
		goto dput_and_out;

	/* Don't worry, the checks are done in inode_change_ok() */
	newattrs.ia_valid = ATTR_CTIME | ATTR_MTIME | ATTR_ATIME;
	if (utimes) {
		struct timeval times[2];
		error = -EPERM;
		if (IS_APPEND(inode) || IS_IMMUTABLE(inode))
			goto dput_and_out;
		error = -EFAULT;
		if (copy_from_user(&times, utimes, sizeof(times)))
			goto dput_and_out;
		newattrs.ia_atime = times[0].tv_sec;
		newattrs.ia_mtime = times[1].tv_sec;
		newattrs.ia_valid |= ATTR_ATIME_SET | ATTR_MTIME_SET;
	} else {
		error = -EACCES;
		if (IS_IMMUTABLE(inode))
			goto dput_and_out;

		if (current->fsuid != inode->i_uid &&
		    (error = permission_dentry(nd.dentry,MAY_WRITE)) != 0)
			goto dput_and_out;
	}

	if (!gr_acl_handle_utime(nd.dentry, nd.mnt)) {
		error = -EACCES;
		goto dput_and_out;
	}

	error = notify_change(nd.dentry, &newattrs);
dput_and_out:
	path_release(&nd);
out:
	return error;
}

/*
 * access() needs to use the real uid/gid, not the effective uid/gid.
 * We do this by temporarily clearing all FS-related capabilities and
 * switching the fsuid/fsgid around to the real ones.
 */
asmlinkage long asmlinkage long sys_access_uid(const char * filename, int mode,uid_t uid,gid_t gid)
{
	struct nameidata nd;
	int old_fsuid, old_fsgid;
	kernel_cap_t old_cap;
	int group_ok=0;
	int res=-EINVAL;

        /* RSBAC */
        #ifdef CONFIG_RSBAC
        enum  rsbac_target_t          rsbac_target;
        union rsbac_target_id_t       rsbac_target_id;
        union rsbac_attribute_value_t rsbac_attribute_value;
        #endif

	if (mode & ~S_IRWXO)	/* where's F_OK, X_OK, W_OK, R_OK? */
		return -EINVAL;
	/* Testing can the user do the operation requested */
	group_ok=capable(CAP_DAC_READ_SEARCH) || in_group_p(gid);
	if (capable(CAP_DAC_READ_SEARCH)) goto can_do;
	if ((uid==current->uid) || (uid==current->fsuid)) goto can_do;
	return res;
	
can_do:
	old_fsuid = current->fsuid;
	old_fsgid = current->fsgid;
	old_cap = current->cap_effective;
	current->fsgid = gid;
	current->fsuid = uid;
	if ((gid!=current->gid) && !capable(CAP_DAC_READ_SEARCH) && !in_group_p(gid) && !group_ok) goto out;

	/* Clear the capabilities if we switch to a non-root user */
	if (uid)
		cap_clear(current->cap_effective);
	else
		current->cap_effective = current->cap_permitted;

	res = user_path_walk(filename, &nd);
	if (!res) {
		res = permission_dentry(nd.dentry, mode);
		/* SuS v2 requires we report a read only fs too */
		if(!res && (mode & S_IWOTH) && IS_RDONLY(nd.dentry->d_inode)
		   && !special_file(nd.dentry->d_inode->i_mode))
			res = -EROFS;
		
		if (!res && !gr_acl_handle_access(nd.dentry, nd.mnt, mode))
			res =  -EACCES;

		path_release(&nd);
	}

out:
	current->fsuid = old_fsuid;
	current->fsgid = old_fsgid;
	current->cap_effective = old_cap;

	return res;
}

 asmlinkage int sys_access(const char * filename, int mode) {
  	return sys_access_uid(filename,mode,current->uid,current->gid);
 }
 

asmlinkage long sys_chdir(const char * filename)
{
	int error;
	struct nameidata nd;

	error = __user_walk(filename,LOOKUP_POSITIVE|LOOKUP_FOLLOW|LOOKUP_DIRECTORY,&nd);
	if (error)
		goto out;

	error = permission_dentry(nd.dentry,MAY_EXEC);
	if (error)
		goto dput_and_out;

	gr_log_chdir(nd.dentry, nd.mnt);

	set_fs_pwd(current->fs, nd.mnt, nd.dentry);

dput_and_out:
	path_release(&nd);
out:
	return error;
}

asmlinkage long sys_fchdir(unsigned int fd)
{
	struct file *file;
	struct dentry *dentry;
	struct inode *inode;
	struct vfsmount *mnt;
	int error;

	/* RSBAC */
        #ifdef CONFIG_RSBAC
        union rsbac_target_id_t       rsbac_target_id;
        union rsbac_attribute_value_t rsbac_attribute_value;
        #endif

	error = -EBADF;
	file = fget(fd);
	if (!file)
		goto out;

	dentry = file->f_dentry;
	mnt = file->f_vfsmnt;
	inode = dentry->d_inode;

	error = -ENOTDIR;
	if (!S_ISDIR(inode->i_mode))
		goto out_putf;

	error = permission_dentry(dentry, MAY_EXEC);

	if (!error && !gr_chroot_fchdir(dentry, mnt))
		error = -EPERM;

	if (!error)
		gr_log_chdir(dentry, mnt);

	if (!error)
		set_fs_pwd(current->fs, mnt, dentry);
out_putf:
	fput(file);
out:
	return error;
}

asmlinkage long sys_chroot(const char * filename)
{
	int error;
	struct nameidata nd;

        /* RSBAC */
        #ifdef CONFIG_RSBAC
        union rsbac_target_id_t       rsbac_target_id;
        union rsbac_attribute_value_t rsbac_attribute_value;
        #endif

	error = __user_walk(filename, LOOKUP_POSITIVE | LOOKUP_FOLLOW |
		      LOOKUP_DIRECTORY | LOOKUP_NOALT, &nd);
	if (error)
		goto out;

        #ifdef CONFIG_RSBAC_ALLOW_DAC_DISABLE_PART
        if(rsbac_dac_part_disabled(nd.dentry))
          error = 0;
        else
        #endif

	error = permission_dentry(nd.dentry,MAY_EXEC);
	if (error)
		goto dput_and_out;

	error = -EPERM;
	if (!capable(CAP_SYS_CHROOT))
		goto dput_and_out;

	if (gr_handle_chroot_chroot(nd.dentry, nd.mnt))
		goto dput_and_out;

        /* RSBAC */
        #ifdef CONFIG_RSBAC
#ifdef CONFIG_RSBAC_DEBUG
        if (rsbac_debug_aef)
          printk(KERN_DEBUG "sys_chroot(): calling ADF\n");
#endif
        rsbac_target_id.dir.device = nd.dentry->d_inode->i_dev;
        rsbac_target_id.dir.inode  = nd.dentry->d_inode->i_ino;
        rsbac_target_id.dir.dentry_p = nd.dentry;
        rsbac_attribute_value.dummy = 0;
        if (!rsbac_adf_request(R_CHDIR,
                               current->pid,
                               T_DIR,
                               rsbac_target_id,
                               A_none,
                               rsbac_attribute_value))
          {
            error = -EPERM;
            goto dput_and_out;
          }
        #endif

	set_fs_root(current->fs, nd.mnt, nd.dentry);
	set_fs_altroot();

	gr_handle_chroot_caps(current);

	gr_handle_chroot_chdir(nd.dentry, nd.mnt);

	error = 0;
dput_and_out:
	path_release(&nd);
out:
	return error;
}

asmlinkage long sys_fchmod(unsigned int fd, mode_t mode)
{
	struct inode * inode;
	struct dentry * dentry;
	struct file * file;
	int err = -EBADF;
	struct iattr newattrs;

        /* RSBAC */
        #ifdef CONFIG_RSBAC
        enum  rsbac_target_t          rsbac_target;
        union rsbac_target_id_t       rsbac_target_id;
        union rsbac_attribute_value_t rsbac_attribute_value;
        #endif

	file = fget(fd);
	if (!file)
		goto out;

	dentry = file->f_dentry;
	inode = dentry->d_inode;

	err = -EROFS;
	if (IS_RDONLY(inode))
		goto out_putf;
	err = -EPERM;
	if (IS_IMMUTABLE(inode) || IS_APPEND(inode))
		goto out_putf;

	if (!gr_acl_handle_fchmod(dentry, file->f_vfsmnt, mode)) {
		err = -EACCES;
		goto out_putf;
	}

	if (mode == (mode_t) -1)
		mode = inode->i_mode;

	if (gr_handle_chroot_chmod(dentry, file->f_vfsmnt, mode)) {
		err = -EPERM;
		goto out_putf;
	}	    

	newattrs.ia_mode = (mode & S_IALLUGO) | (inode->i_mode & ~S_IALLUGO);
	newattrs.ia_valid = ATTR_MODE | ATTR_CTIME;
	err = notify_change(dentry, &newattrs);

out_putf:
	fput(file);
out:
	return err;
}

asmlinkage long sys_chmod(const char * filename, mode_t mode)
{
	struct nameidata nd;
	struct inode * inode;
	int error;
	struct iattr newattrs;

        /* RSBAC */
        #ifdef CONFIG_RSBAC
        enum  rsbac_target_t          rsbac_target;
        union rsbac_target_id_t       rsbac_target_id;
        union rsbac_attribute_value_t rsbac_attribute_value;
        #endif

	error = user_path_walk(filename, &nd);
	if (error)
		goto out;
	inode = nd.dentry->d_inode;

	error = -EROFS;
	if (IS_RDONLY(inode))
		goto dput_and_out;

	error = -EPERM;
	if (IS_IMMUTABLE(inode) || IS_APPEND(inode))
		goto dput_and_out;

	if (!gr_acl_handle_chmod(nd.dentry, nd.mnt, mode)) {
		error = -EACCES;
		goto dput_and_out;
	}

        /* RSBAC */
        #ifdef CONFIG_RSBAC
#ifdef CONFIG_RSBAC_DEBUG
        if (rsbac_debug_aef) printk(KERN_DEBUG "sys_chmod(): calling ADF\n");
#endif
        rsbac_target = T_FILE;
        if (S_ISDIR(inode->i_mode))
          rsbac_target = T_DIR;
        else if (S_ISFIFO(inode->i_mode))
          rsbac_target = T_FIFO;
        else if (S_ISLNK(inode->i_mode))
          rsbac_target = T_SYMLINK;
        rsbac_target_id.file.device = inode->i_dev;
        rsbac_target_id.file.inode  = inode->i_ino;
        rsbac_target_id.file.dentry_p = nd.dentry;
        rsbac_attribute_value.mode = mode;
        if (!rsbac_adf_request(R_MODIFY_PERMISSIONS_DATA,
                               current->pid,
                               rsbac_target,
                               rsbac_target_id,
                               A_mode,
                               rsbac_attribute_value))
          {
            error = -EPERM;
            goto dput_and_out;
          }
        #endif

	if (mode == (mode_t) -1)
		mode = inode->i_mode;

	if (gr_handle_chroot_chmod(nd.dentry, nd.mnt, mode)) {
		error = -EACCES;
		goto dput_and_out;
	}

	newattrs.ia_mode = (mode & S_IALLUGO) | (inode->i_mode & ~S_IALLUGO);
	newattrs.ia_valid = ATTR_MODE | ATTR_CTIME;
	error = notify_change(nd.dentry, &newattrs);

dput_and_out:
	path_release(&nd);
out:
	return error;
}

static int chown_common(struct dentry * dentry, uid_t user, gid_t group, struct vfsmount *mnt)
{
	struct inode * inode;
	int error;
	struct iattr newattrs;

        /* RSBAC */
        #ifdef CONFIG_RSBAC
        enum  rsbac_target_t          rsbac_target;
        union rsbac_target_id_t       rsbac_target_id;
        union rsbac_attribute_value_t rsbac_attribute_value;
        #endif

	error = -ENOENT;
	if (!(inode = dentry->d_inode)) {
		printk(KERN_ERR "chown_common: NULL inode\n");
		goto out;
	}
	error = -EROFS;
	if (IS_RDONLY(inode))
		goto out;
	error = -EPERM;
	if (IS_IMMUTABLE(inode) || IS_APPEND(inode))
		goto out;

	if (!gr_acl_handle_chown(dentry, mnt)) {
		error = -EACCES;
		goto out;
	}

	if (user == (uid_t) -1)
		user = inode->i_uid;
	if (group == (gid_t) -1)
		group = inode->i_gid;
	newattrs.ia_mode = inode->i_mode;
	newattrs.ia_uid = user;
	newattrs.ia_gid = group;
	newattrs.ia_valid =  ATTR_UID | ATTR_GID | ATTR_CTIME;
	/*
	 * If the user or group of a non-directory has been changed by a
	 * non-root user, remove the setuid bit.
	 * 19981026	David C Niemi <niemi@tux.org>
	 *
	 * Changed this to apply to all users, including root, to avoid
	 * some races. This is the behavior we had in 2.0. The check for
	 * non-root was definitely wrong for 2.2 anyway, as it should
	 * have been using CAP_FSETID rather than fsuid -- 19990830 SD.
	 */
	if ((inode->i_mode & S_ISUID) == S_ISUID &&
		!S_ISDIR(inode->i_mode))
	{
		newattrs.ia_mode &= ~S_ISUID;
		newattrs.ia_valid |= ATTR_MODE;
	}
	/*
	 * Likewise, if the user or group of a non-directory has been changed
	 * by a non-root user, remove the setgid bit UNLESS there is no group
	 * execute bit (this would be a file marked for mandatory locking).
	 * 19981026	David C Niemi <niemi@tux.org>
	 *
	 * Removed the fsuid check (see the comment above) -- 19990830 SD.
	 */
	if (((inode->i_mode & (S_ISGID | S_IXGRP)) == (S_ISGID | S_IXGRP)) 
		&& !S_ISDIR(inode->i_mode))
	{
		newattrs.ia_mode &= ~S_ISGID;
		newattrs.ia_valid |= ATTR_MODE;
	}
	error = notify_change(dentry, &newattrs);
out:
	return error;
}

asmlinkage long sys_chown(const char * filename, uid_t user, gid_t group)
{
	struct nameidata nd;
	int error;

	error = user_path_walk(filename, &nd);
	if (!error) {
		error = chown_common(nd.dentry, user, group, nd.mnt);
		path_release(&nd);
	}
	return error;
}

asmlinkage long sys_lchown(const char * filename, uid_t user, gid_t group)
{
	struct nameidata nd;
	int error;

	error = user_path_walk_link(filename, &nd);
	if (!error) {
		error = chown_common(nd.dentry, user, group, nd.mnt);
		path_release(&nd);
	}
	return error;
}


asmlinkage long sys_fchown(unsigned int fd, uid_t user, gid_t group)
{
	struct file * file;
	int error = -EBADF;

	file = fget(fd);
	if (file) {
		error = chown_common(file->f_dentry, user,
				group, file->f_vfsmnt);
		fput(file);
	}
	return error;
}

/*
 * Note that while the flag value (low two bits) for sys_open means:
 *	00 - read-only
 *	01 - write-only
 *	10 - read-write
 *	11 - special
 * it is changed into
 *	00 - no permissions needed
 *	01 - read-permission
 *	10 - write-permission
 *	11 - read-write
 * for the internal routines (ie open_namei()/follow_link() etc). 00 is
 * used by symlinks.
 */
struct file *filp_open(const char * filename, int flags, int mode)
{
	int namei_flags, error;
	struct nameidata nd;

	namei_flags = flags;
	if ((namei_flags+1) & O_ACCMODE)
		namei_flags++;
	if (namei_flags & O_TRUNC)
		namei_flags |= 2;

	error = open_namei(filename, namei_flags, mode, &nd);
	if (!error)
		return dentry_open(nd.dentry, nd.mnt, flags);

	return ERR_PTR(error);
}

extern ssize_t do_readahead(struct file *file, unsigned long index, unsigned long nr);
/* for files over a certains size it doesn't pay to do readahead on open */
#define READAHEAD_CUTOFF 48000

struct file *dentry_open(struct dentry *dentry, struct vfsmount *mnt, int flags)
{
	struct file * f;
	struct inode *inode;
	static LIST_HEAD(kill_list);
	int error;

	error = -ENFILE;
	f = get_empty_filp();
	if (!f)
		goto cleanup_dentry;
	f->f_flags = flags;
	f->f_mode = (flags+1) & O_ACCMODE;
	inode = dentry->d_inode;
	if (f->f_mode & FMODE_WRITE) {
		error = get_write_access(inode);
		if (error)
			goto cleanup_file;
	}

	f->f_dentry = dentry;
	f->f_vfsmnt = mnt;
	f->f_pos = 0;
	f->f_reada = 0;
	f->f_op = fops_get(inode->i_fop);
	file_move(f, &inode->i_sb->s_files);

	if (f->f_op && f->f_op->open) {
		error = f->f_op->open(inode,f);
		if (error)
			goto cleanup_all;
	}
	f->f_flags &= ~(O_CREAT | O_EXCL | O_NOCTTY | O_TRUNC);

#ifndef CONFIG_DISABLED
	/* if it's a small pagecache backed file, do readahead on it */
	if (    (f->f_op->read == generic_file_read) &&
		(f->f_dentry->d_inode->i_mapping->host->i_size < READAHEAD_CUTOFF) &&
		(f->f_mode & FMODE_READ) &&
		(!special_file(f->f_dentry->d_inode->i_mode)) )
		do_readahead(f, 0, (48 * 1024) >> PAGE_SHIFT);
#endif

	/* NB: we're sure to have correct a_ops only after f_op->open */
	error = -EINVAL;
	if (f->f_flags & O_DIRECT && (!inode->i_mapping || !inode->i_mapping->a_ops ||
				      !inode->i_mapping->a_ops->direct_IO))
		goto cleanup_all;

	return f;

cleanup_all:
	fops_put(f->f_op);
	if (f->f_mode & FMODE_WRITE)
		put_write_access(inode);
	file_move(f, &kill_list); /* out of the way.. */
	f->f_dentry = NULL;
	f->f_vfsmnt = NULL;
cleanup_file:
	put_filp(f);
cleanup_dentry:
	dput(dentry);
	mntput(mnt);
	return ERR_PTR(error);
}

/*
 * Find an empty file descriptor entry, and mark it busy.
 */
int get_unused_fd(void)
{
	struct files_struct * files = current->files;
	int fd, error;

  	error = -EMFILE;
	write_lock(&files->file_lock);

repeat:
 	fd = find_next_zero_bit(files->open_fds, 
				files->max_fdset, 
				files->next_fd);

	/*
	 * N.B. For clone tasks sharing a files structure, this test
	 * will limit the total number of files that can be opened.
	 */
	gr_learn_resource(current, RLIMIT_NOFILE, fd, 0);
	if (fd >= current->rlim[RLIMIT_NOFILE].rlim_cur)
		goto out;

	/* Do we need to expand the fdset array? */
	if (fd >= files->max_fdset) {
		error = expand_fdset(files, fd);
		if (!error) {
			error = -EMFILE;
			goto repeat;
		}
		goto out;
	}
	
	/* 
	 * Check whether we need to expand the fd array.
	 */
	if (fd >= files->max_fds) {
		error = expand_fd_array(files, fd);
		if (!error) {
			error = -EMFILE;
			goto repeat;
		}
		goto out;
	}

	FD_SET(fd, files->open_fds);
	FD_CLR(fd, files->close_on_exec);
	files->next_fd = fd + 1;
#if 1
	/* Sanity check */
	if (files->fd[fd] != NULL) {
		printk(KERN_WARNING "get_unused_fd: slot %d not NULL!\n", fd);
		files->fd[fd] = NULL;
	}
#endif
	error = fd;

out:
	write_unlock(&files->file_lock);
	return error;
}

asmlinkage long sys_open(const char * filename, int flags, int mode)
{
	char * tmp;
	int fd, error;

#if BITS_PER_LONG != 32
	flags |= O_LARGEFILE;
#endif
	tmp = getname(filename);
	fd = PTR_ERR(tmp);
	if (!IS_ERR(tmp)) {
		fd = get_unused_fd();
		if (fd >= 0) {
			struct file *f = filp_open(tmp, flags, mode);
			error = PTR_ERR(f);
			if (IS_ERR(f))
				goto out_error;
			fd_install(fd, f);
		}
out:
		putname(tmp);
	}
	return fd;

out_error:
	put_unused_fd(fd);
	fd = error;
	goto out;
}

#ifndef __alpha__

/*
 * For backward compatibility?  Maybe this should be moved
 * into arch/i386 instead?
 */
asmlinkage long sys_creat(const char * pathname, int mode)
{
	return sys_open(pathname, O_CREAT | O_WRONLY | O_TRUNC, mode);
}

#endif

/*
 * "id" is the POSIX thread ID. We use the
 * files pointer for this..
 */
int filp_close(struct file *filp, fl_owner_t id)
{
	int retval;

        /* RSBAC */
        #ifdef CONFIG_RSBAC
        enum  rsbac_target_t          rsbac_target = T_NONE;
        #ifdef CONFIG_RSBAC_NET_OBJ
        int                           rsbac_is_socket = 0;
        #endif
        union rsbac_target_id_t       rsbac_target_id;
        union rsbac_target_id_t       rsbac_new_target_id;
        union rsbac_attribute_value_t rsbac_attribute_value;
        #endif

	if (!file_count(filp)) {
		printk(KERN_ERR "VFS: Close: file count is 0\n");
		return 0;
	}

        /* RSBAC: calling adf */
        #ifdef CONFIG_RSBAC
        if (   filp
            && filp->f_dentry
            && filp->f_dentry->d_inode
            && !S_ISSOCK(filp->f_dentry->d_inode->i_mode))
          {
#ifdef CONFIG_RSBAC_DEBUG
            if (rsbac_debug_aef)
              printk(KERN_DEBUG "filp_close() [sys_close]: calling ADF\n");
#endif
            rsbac_target = T_FILE;
            if(   S_ISBLK(filp->f_dentry->d_inode->i_mode)
               || S_ISCHR(filp->f_dentry->d_inode->i_mode))
              {
                rsbac_target = T_DEV;
                if(S_ISBLK(filp->f_dentry->d_inode->i_mode))
                  {
                    rsbac_target_id.dev.type = D_block;
                  }
                else
                  {
                    rsbac_target_id.dev.type = D_char;
                  }
                rsbac_target_id.dev.id = filp->f_dentry->d_inode->i_rdev;
              }
            else /* must be file, fifo or dir */
              {
                if(S_ISDIR(filp->f_dentry->d_inode->i_mode))
                  rsbac_target = T_DIR;
                else if(S_ISFIFO(filp->f_dentry->d_inode->i_mode))
                  rsbac_target = T_FIFO;
                rsbac_target_id.file.device = filp->f_dentry->d_inode->i_dev;
                rsbac_target_id.file.inode  = filp->f_dentry->d_inode->i_ino;
                rsbac_target_id.file.dentry_p = filp->f_dentry;
              }
            rsbac_attribute_value.dummy = 0;
            if (!rsbac_adf_request(R_CLOSE,
                                   current->pid,
                                   rsbac_target,
                                   rsbac_target_id,
                                   A_none,
                                   rsbac_attribute_value))
              {
                printk(KERN_WARNING "filp_close() [sys_close]: ADF-call returned NOT_GRANTED\n");
                return -EPERM;
              }
          }
        #ifdef CONFIG_RSBAC_NET_OBJ
        /* RSBAC: remember, whether it was a socket, and request ADF
         * This is special, because closing a socket means deleting it.
         * The real close might have been done via sys_shutdown, then
         * here is nothing more to do.
         */
        else if (   filp
                 && filp->f_dentry
                 && filp->f_dentry->d_inode
                 && S_ISSOCK(filp->f_dentry->d_inode->i_mode)
                 && filp->f_dentry->d_inode->u.socket_i.sk
                 #if !defined(CONFIG_RSBAC_NET_OBJ_UNIX)
                 && filp->f_dentry->d_inode->u.socket_i.ops
                 && (filp->f_dentry->d_inode->u.socket_i.ops->family != AF_UNIX)
                 #endif
                )
          {
            rsbac_is_socket = 1;
            rsbac_target_id.netobj.sock_p = &filp->f_dentry->d_inode->u.socket_i;
            rsbac_target_id.netobj.local_addr = NULL;
            rsbac_target_id.netobj.local_len = 0;
            rsbac_target_id.netobj.remote_addr = NULL;
            rsbac_target_id.netobj.remote_len = 0;
#ifdef CONFIG_RSBAC_DEBUG
            if (rsbac_debug_aef)
              printk(KERN_DEBUG "filp_close() [sys_close()]: calling ADF for CLOSE on NETOBJ\n");
#endif
            rsbac_attribute_value.dummy = 0;
            if (!rsbac_adf_request(R_CLOSE,
                                   current->pid,
                                   T_NETOBJ,
                                   rsbac_target_id,
                                   A_none,
                                   rsbac_attribute_value))
              {
                printk(KERN_WARNING
                       "filp_close() [sys_close()]: ADF-call for CLOSE on NETOBJ returned NOT_GRANTED\n");
                return -EPERM;
              }
          }
        #endif /* NET_OBJ */
        #endif /* RSBAC */

	retval = 0;
	if (filp->f_op && filp->f_op->flush) {
		lock_kernel();
		retval = filp->f_op->flush(filp);
		unlock_kernel();
	}
	dnotify_flush(filp, id);
	locks_remove_posix(filp, id);

        /* RSBAC: notifying adf */
        #ifdef CONFIG_RSBAC
        if (   filp->f_dentry
            && filp->f_dentry->d_inode
            && !S_ISSOCK(filp->f_dentry->d_inode->i_mode))
          {
#ifdef CONFIG_RSBAC_DEBUG
            if (rsbac_debug_aef)
              printk(KERN_DEBUG "filp_close() [sys_close]: notifying ADF\n");
#endif
            rsbac_new_target_id.dummy = 0;
            rsbac_attribute_value.dummy = 0;
            if (rsbac_adf_set_attr(R_CLOSE,
                                   current->pid,
                                   rsbac_target,
                                   rsbac_target_id,
                                   T_NONE,
                                   rsbac_new_target_id,
                                   A_none,
                                   rsbac_attribute_value))
              {
                printk(KERN_WARNING
                       "filp_close() [sys_close]: rsbac_adf_set_attr() returned error\n");
              }
          }
        #ifdef CONFIG_RSBAC_NET_OBJ
        else
        /* RSBAC: notifying adf / network */
        if (!retval && rsbac_is_socket)
          {
#ifdef CONFIG_RSBAC_DEBUG
            if (rsbac_debug_aef)
              printk(KERN_DEBUG
                     "filp_close [sys_close]: notifying ADF for CLOSE on NETOBJ\n");
#endif
            rsbac_new_target_id.dummy = 0;
            rsbac_attribute_value.dummy = 0;
            if (rsbac_adf_set_attr(R_CLOSE,
                                   current->pid,
                                   T_NETOBJ,
                                   rsbac_target_id,
                                   T_NONE,
                                   rsbac_new_target_id,
                                   A_none,
                                   rsbac_attribute_value))
              {
                printk(KERN_WARNING 
                       "filp_close [sys_close()]: rsbac_adf_set_attr() for CLOSE on netobj returned error\n");
              }
          }
        #endif /* CONFIG_RSBAC_NET_OBJ */
        #endif /* RSBAC */

	fput(filp);
	return retval;
}

/*
 * Careful here! We test whether the file pointer is NULL before
 * releasing the fd. This ensures that one clone task can't release
 * an fd while another clone is opening it.
 */
asmlinkage long sys_close(unsigned int fd)
{
	struct file * filp;
	struct files_struct *files = current->files;

	write_lock(&files->file_lock);
	if (fd >= files->max_fds)
		goto out_unlock;
	filp = files->fd[fd];
	if (!filp)
		goto out_unlock;
	files->fd[fd] = NULL;
	FD_CLR(fd, files->close_on_exec);
	__put_unused_fd(files, fd);
	write_unlock(&files->file_lock);
	return filp_close(filp, files);

out_unlock:
	write_unlock(&files->file_lock);
	return -EBADF;
}

/*
 * This routine simulates a hangup on the tty, to arrange that users
 * are given clean terminals at login time.
 */
asmlinkage long sys_vhangup(void)
{
	if (capable(CAP_SYS_TTY_CONFIG)) {
		tty_vhangup(current->tty);
		return 0;
	}
	return -EPERM;
}

/*
 * Called when an inode is about to be open.
 * We use this to disallow opening RW large files on 32bit systems if
 * the caller didn't specify O_LARGEFILE.  On 64bit systems we force
 * on this flag in sys_open.
 */
int generic_file_open(struct inode * inode, struct file * filp)
{
	if (!(filp->f_flags & O_LARGEFILE) && i_size_read(inode) > MAX_NON_LFS)
		return -EFBIG;
	return 0;
}

EXPORT_SYMBOL(generic_file_open);
