/*
 * High performance packet classification - dimension tree
 *
 * Authors: Michael Bellion and Thomas Heinz
 * (c) 2002-2003 by the hipac core team <nf@hipac.org>:
 *      +-----------------------+----------------------+
 *      |   Michael Bellion     |     Thomas Heinz     |
 *      | <mbellion@hipac.org>  |  <creatix@hipac.org> |
 *      +-----------------------+----------------------+
 * Licenced under the GNU General Public Licence, version >= 2.
 */


#ifndef _DIMTREE_H
#define _DIMTREE_H

#include "global.h"
#include "btree.h"

/* upper bound for matches of the given bit type */
#define MAXKEY(bittype) \
((bittype) == BIT_U8 ? 0xff : ((bittype) == BIT_U16 ? 0xffff : 0xffffffff))

/* used to distinguish a rule from an elementary interval */
#define RT_RULE 0
#define RT_ELEM 1


/* header for dimtree rules and elementary intervals */
struct dt_rule_elem_spec
{
	unsigned bittype : 2; // must be BIT_RULE	
	unsigned rtype   : 1; // {RT_RULE, RT_ELEM}
};

/* header for dimtree rules */
struct dt_rule_spec
{
	unsigned bittype :  2; // must be BIT_RULE	
	unsigned rtype   :  1; // must be RT_RULE
	unsigned action  :  4; // packet action
	unsigned pos     : 25; // position of the rule in the chain
};

/* dt_match represents the native interval match [left, right] associated
   with dimension dimid whereby [left, right] may not be a wildcard match */
struct dt_match
{
	__u8 dimid;
	__u32 left, right;
	char next_match[0];
};

/* lock is used to implement a very fine grained locking mechanism which
   synchronizes the race between hipac_match() and dimtree_finish_snapshot() */
struct counter_t
{
	__u64 packet_ct[2], byte_ct[2];
	spinlock_t lock;
} __cacheline_aligned;

/* dt_rule is an entry in the dt_chain; at the end of the struct we have
   dt_match_len >= 0 dt_matches and smp_num_cpus structs counter_t;
   if the rule has a function based target then exec_target points to the
   target's data which is handled by target_fn;
   the rule's exec_match pointer block references >= 0 blocks each of >= 1
   function based matches, called fblocks;
   the (2 * i)-th pointer of exec_match points to the beginning of the i-th
   fblock;
   the (2 * i + 1)-th pointer of exec_match points to the end of the i-th
   fblock;
   the start and end pointers are handed to match_fn */
struct dt_rule
{
	struct dt_rule_spec spec;
	struct list_head head;
	struct ptrblock *exec_match;
	void *exec_target;
	__u32 exec_target_size;
	__u8 deleted;
	__u8 dt_match_len;
	struct dt_match first_dt_match[0];
};

#define IS_RULE(r) (((struct dt_rule_elem_spec *) (r))->bittype == BIT_RULE \
		    && ((struct dt_rule_elem_spec *) (r))->rtype == RT_RULE)
#define COUNTER(rule, cpu) ((struct counter_t *)                             \
			    ((char *) rule + SMP_CACHE_ALIGN(                \
				    sizeof(*(rule)) + (rule)->dt_match_len * \
				    sizeof(*(rule)->first_dt_match)) +       \
			     (cpu) * sizeof(struct counter_t)))
#define HAS_EXEC_MATCH(r)  ((r)->exec_match != NULL)
#define IS_TARGET_DUMMY(r) ((r)->spec.action == TARGET_DUMMY)
#define IS_TARGET_NONE(r)  ((r)->spec.action == TARGET_NONE)
#define IS_TARGET_EXEC(r)  ((r)->spec.action == TARGET_EXEC)
#define IS_TARGET_TERM(r)  ((r)->spec.action == TARGET_ACCEPT || \
			    (r)->spec.action == TARGET_DROP)
#define IS_RULE_TERM(r)    (IS_TARGET_TERM(r) && !HAS_EXEC_MATCH(r))

/* return the size of a dt_rule with dt_match_len dt_matches */
static inline __u32
dt_rule_size(__u8 dt_match_len)
{
	return SMP_CACHE_ALIGN(sizeof(struct dt_rule) + dt_match_len *
			       sizeof(struct dt_match)) +
		smp_num_cpus * sizeof(struct counter_t);
}

/* initialize the counter block of rule with packet_ct and byte_ct */
static inline void
dt_rule_init_counters(struct dt_rule *rule, __u64 packet_ct, __u64 byte_ct)
{
	struct counter_t *ct;
	int i;
	
	if (rule == NULL) {
		return;
	}
	ct = COUNTER(rule, 0);
	memset(ct, 0, smp_num_cpus * sizeof(struct counter_t));
	ct->packet_ct[0] = packet_ct;
	ct->byte_ct[0] = byte_ct;
	for (i = 0; i < smp_num_cpus; i++, ct++) {
		spin_lock_init(&ct->lock);
	}
}

/* head of the list of rules */
struct dt_chain
{
	struct list_head head;
	char name[HIPAC_CHAIN_NAME_MAX_LEN];
	struct dt_rule *first; // optimization of dimtree_chain_fix
	__u32 len;
};



/* header for elementary intervals */
struct dt_elem_spec
{
	unsigned bittype : 2; // must be BIT_RULE
	unsigned rtype   : 1; // must be RT_ELEM
	unsigned newspec : 1; // indicates whether the elementary interval is
	                      // contained in newspec
};

/* elementary interval */
struct dt_elem
{
	struct dt_elem_spec spec;
	/* terminating target (TARGET_ACCEPT, TARGET_DROP) without function
	   based matches */
	struct dt_rule *term_rule;
	/* block of non-terminating rules (function based matches or no
	   terminal target) whose position is < term_rule->spec.pos */
	struct ptrblock ntm_rules;
};

#define IS_ELEM(e) (((struct dt_rule_elem_spec *) (e))->bittype == BIT_RULE \
		    && ((struct dt_rule_elem_spec *) (e))->rtype == RT_ELEM)



/* main dimtree struct: ctindex and top are protected by BR_HIPAC_LOCK */
struct dimtree
{
	__u32 origin;
	unsigned ctindex : 1;
	struct gen_spec *top;
	struct gen_spec *top_new;    // new not yet active top level structure
	int need_commit;             // 1 if top_new is valid
	struct dt_chain *chain;
};



/* create new dimtree and store it in *newdt; chain_name is copied to
   dt->chain->name; memory for newdt is allocated within dimtree_new;
   origin is a bit vector where exactly one bit is set; it is used to
   uniquely define the "origin property" of newdt; dummy and policy
   define the base ruleset; dummy must have TARGET_DUMMY as target,
   policy must be a terminal rule without any dt_matches;
   possible errors: HE_LOW_MEMORY, HE_IMPOSSIBLE_CONDITION */
hipac_error
dimtree_new(struct dimtree **newdt, __u32 origin, const char *chain_name,
	    struct dt_rule *dummy, struct dt_rule *policy);

/* free memory for dt and all embedded structures; make sure that no packet
   matching occurs on dt any more */
void
dimtree_free(struct dimtree *dt);

/* remove all rules except the first and the last one from dt->chain and
   free them; set dt->top to the last rule in the chain */
void
dimtree_flush(struct dimtree *dt);

const char *
dimtree_get_chain_name(const struct dimtree *dt);

/* insert rule into the dt_chain and the btree; inc indicates whether all
   rule positions >= rule->spec.pos should be incremented by 1;
   if commit is not 0 then the top level structure in dt is replaced by the
   new one and the old btrees and elementary intervals are freed;
   in case of a fault all newly created btrees and elementary intervals
   are freed; origin is a bit vector describing the allowed dimtrees
   into which rule may be inserted; if rule must not be inserted into dt
   it is anyway inserted into dt->chain (so take care to remove it from
   there);
   NOTICE: if commit is not 0 it is assumed that this operation is the
           first one (at all or directly after a previously committed
           operation or series of operations (-> dimtree_commit))
   possible errors: HE_LOW_MEMORY, HE_IMPOSSIBLE_CONDITION,
                    HE_RULE_ORIGIN_MISMATCH */
hipac_error
dimtree_insert(struct dimtree *dt, struct dt_rule *rule, __u32 origin,
	       int inc, int commit);

/* delete rule from btree, _NOT_ from the dt_chain; 'rule' must point to a
   rule in dt->chain; if commit is not 0 then the top level structure in dt
   is replaced by the new one and the old btrees and elementary intervals
   are freed; in case of a fault all newly created btrees and elementary
   intervals are freed;
   NOTICE: if commit is not 0 it is assumed that this operation is the
           first one (at all or directly after a previously committed
           operation or series of operations (-> dimtree_commit))
   possible errors: HE_LOW_MEMORY, HE_IMPOSSIBLE_CONDITION */
hipac_error
dimtree_delete(struct dimtree *dt, struct dt_rule *rule, int commit);

/* for each rule r contained in the chains of the dimtrees in dt_block the
   following operation is performed:
   if r has a match (l, r) in dimid[k] (k \in {0, 1}): for all 0 <= i <= len:
   - if l == r == org[i] then l = r = new[i]
   - if l == org[i] + 1  then l = new[i] + 1 (assuming that
                                              r == MAXKEY(dimid[k]))
   - if r == org[i] - 1  then r = new[i] - 1 (assuming that l == 0)
   the original rule r is replaced by the modified r;
   org[i] and new[i] must be > 0 and < MAXKEY(dimid[k]) and MAXKEY(dimid[0]) must
   be == MAXKEY(dimid[1]);
   possible errors: HE_LOW_MEMORY, HE_IMPOSSIBLE_CONDITION */
hipac_error
dimtree_substitute(struct ptrblock *dt_block, const __u8 dimid[2],
		   const __u32 org[], const __u32 new[], const __u32 len);

/* called at the end of a successful series of dimtree_insert and/or
   dimtree_delete operation(s) to make the result visible, i.e. set dt->top
   to dt->top_new for each dimtree dt in dt_block and free the old btrees
   and elementary intervals */
void
dimtree_commit(struct ptrblock *dt_block);

/* called at the end of an unsuccessful series of dimtree_insert and/or
   dimtree_delete operation(s) to undo the changes, i.e. set dt->top_new
   to NULL and need_commit to 0 for each dimtree dt in dt_block and free the
   new btrees and elementary intervals */
void
dimtree_failed(struct ptrblock *dt_block);

#ifdef DEBUG
int
rule_occur(struct gen_spec *g, struct dt_rule *rule, int print);
#endif

/* remove all rules between start and the rule(s) r with position end_pos inc.
   start and r themselves; the positions of the rules behind r are not
   changed */
static inline void
dimtree_chain_delete(struct dimtree *dt, struct dt_rule *start, __u32 end_pos)
{
	struct dt_rule *rule;
		struct list_head *lh;

	if (unlikely(dt == NULL || start == NULL ||
		     start->spec.pos > end_pos)) {
		ARG_MSG;
		return;
	}

	assert(dt->need_commit == 0);
	if (start->head.prev == &dt->chain->head) {
		/* start is the first element => dt->chain->first stays
		   NULL until dimtree_chain_fix has been called */
		dt->chain->first = NULL;
	} else if (dt->chain->first != NULL &&
		   dt->chain->first->spec.pos >= start->spec.pos) {
		dt->chain->first = list_entry(start->head.prev,
					      struct dt_rule, head);
	}
	for (lh = &start->head, rule = start; lh != &dt->chain->head &&
		     rule->spec.pos <= end_pos;) {
		lh = lh->next;
		list_del(lh->prev);
#ifdef DEBUG
		if (rule_occur(dt->top, rule, 1)) {
			ERR("rule present in original structure");
			return;
		}
#endif
		if (rule->exec_match != NULL) {
			ptrblock_free(rule->exec_match);
		}
		hp_free(rule);
		dt->chain->len--;
		rule = list_entry(lh, struct dt_rule, head);
	}
}

/* iterate over the dt_chain in dt and tighten the position numbers */
void
dimtree_chain_fix(struct ptrblock *dt_block);

/* required for calculating a counter snapshot; packet_ct[0] and byte_ct[0]
   remain unchanged after this function until dimtree_finish_snapshot
   is called */
static inline void
dimtree_request_snapshot(struct dimtree *dt)
{
	if (unlikely(dt == NULL)) {
		ARG_MSG;
		return;
	}
	br_write_lock_bh(BR_HIPAC_LOCK);
	dt->ctindex = 1;
	br_write_unlock_bh(BR_HIPAC_LOCK);
}

void
dimtree_finish_snapshot(struct dimtree *dt);

/* add the counters of all rules between start and the rule(s) r with position
   end_pos inc. start and r themselves to packet_ct respectively byte_ct;
   NOTICE: the function may only be called during a counter snapshot */
static inline void
dimtree_get_counters(struct dimtree *dt, struct dt_rule *start, __u32 end_pos,
		     __u64 *packet_ct, __u64 *byte_ct)
{
	struct list_head *lh;
	struct dt_rule *rule;
	struct counter_t *ct;
	__u32 i;

	if (unlikely(dt == NULL || start == NULL || packet_ct == NULL ||
		     byte_ct == NULL || start->spec.pos > end_pos)) {
		ARG_MSG;
		return;
	}

	for (lh = &start->head, rule = start; lh != &dt->chain->head &&
		     rule->spec.pos <= end_pos;) {
		ct = COUNTER(rule, 0);
		for (i = 0; i < smp_num_cpus; i++, ct++) {
			*packet_ct += ct->packet_ct[0];
			*byte_ct += ct->byte_ct[0];
		}
		lh = lh->next;
		rule = list_entry(lh, struct dt_rule, head);
	}
}

/* clear the counters of all rules between start and the rule(s) r with
   position end_pos inc. start and r themselves;
   NOTICE: the function may only be called during a counter snapshot */
static inline void
dimtree_zero_counters(struct dimtree *dt, struct dt_rule *start, __u32 end_pos)
{
	struct list_head *lh;
	struct dt_rule *rule;
	struct counter_t *ct;
	__u32 i;

	if (unlikely(dt == NULL || start == NULL ||
		     start->spec.pos > end_pos)) {
		ARG_MSG;
		return;
	}

	for (lh = &start->head, rule = start; lh != &dt->chain->head &&
		     rule->spec.pos <= end_pos;) {
		ct = COUNTER(rule, 0);
		for (i = 0; i < smp_num_cpus; i++, ct++) {
			ct->packet_ct[0] = ct->byte_ct[0] = 0;
		}
		lh = lh->next;
		rule = list_entry(lh, struct dt_rule, head);
	}
}

#ifdef DEBUG
/* matching algorithm used for correctness checks; the returned ptrblock
   contains the rules matching the packet ordered after their positions;
   the last rule should always have TARGET_ACCEPT or TARGET_DROP as action
   and may not contain exec_matches */
struct ptrblock *
hipac_match_debug(struct dimtree *dt, const void *packet, __u32 pkt_size);
#endif

#endif
