/* pam_mail module */

/*
 * $Id: pam_mail.c,v 1.1 1997/01/04 20:33:02 morgan Exp $
 *
 * Written by Andrew Morgan <morgan@parc.power.net> 1996/3/11
 *
 * $Log: pam_mail.c,v $
 * Revision 1.1  1997/01/04 20:33:02  morgan
 * Initial revision
 *
 *
 */

#define DEFAULT_MAIL_DIRECTORY    "/var/spool/mail"
#define MAIL_FILE_FORMAT          "%s/%s"
#define MAIL_ENV_FORMAT           "MAIL=%s"
#define YOUR_MAIL_FORMAT          "You have %s mail in %s"

#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <syslog.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <unistd.h>

/*
 * here, we make a definition for the externally accessible function
 * in this file (this definition is required for static a module
 * but strongly encouraged generally) it is used to instruct the
 * modules include file to define the function prototypes.
 */

#define PAM_SM_SESSION

#include <security/pam_modules.h>
#include <security/_pam_macros.h>

/* some syslogging */

static char *xstrdup(const char *x)
{
    char *y;

    if (x) {
	y = (char *) malloc(strlen(x)+1);
	if (y)
	    strcpy(y,x);
	return y;
    } else
	return NULL;
}

static void _log_err(int err, const char *format, ...)
{
    char buf[1024];
    va_list args;

    va_start(args, format);
    vsprintf(buf, format, args);

    openlog("pam_mail", LOG_CONS|LOG_PID, LOG_AUTH);
    syslog(err, buf);
    closelog();
}

/* argument parsing */

#define PAM_DEBUG_ARG       01
#define PAM_NO_LOGIN        02
#define PAM_LOGOUT_TOO      04
#define PAM_NEW_MAIL_DIR   010
#define PAM_MAIL_SILENT    020
#define PAM_NO_ENV         040

static int _pam_parse(int flags, int argc, const char **argv, char **maildir)
{
    int ctrl=0;

    if (flags & PAM_SILENT) {
	ctrl |= PAM_MAIL_SILENT;
    }

    /* step through arguments */
    for (; argc-- > 0; ++argv) {

	/* generic options */

	if (!strcmp(*argv,"debug"))
	    ctrl |= PAM_DEBUG_ARG;
	else if (!strncmp(*argv,"dir=",6)) {
	    *maildir = xstrdup(4+*argv);
	    if (*maildir != NULL) {
		D(("new mail directory: %s", maildir));
		ctrl |= PAM_NEW_MAIL_DIR;
	    } else {
		_log_err(LOG_CRIT,
			 "failed to duplicate mail directory - ignored");
	    }
	} else if (!strcmp(*argv,"close")) {
	    ctrl |= PAM_LOGOUT_TOO;
	} else if (!strcmp(*argv,"nopen")) {
	    ctrl |= PAM_NO_LOGIN;
	} else if (!strcmp(*argv,"noenv")) {
	    ctrl |= PAM_NO_ENV;
	} else {
	    _log_err(LOG_ERR,"pam_parse: unknown option; %s",*argv);
	}
    }

    return ctrl;
}

/* a front end for conversations */

static int converse(pam_handle_t *pamh, int ctrl, int nargs
		    , struct pam_message **message
		    , struct pam_response **response)
{
    int retval;
    struct pam_conv *conv;

    D(("begin to converse"));

    retval = pam_get_item( pamh, PAM_CONV, (const void **) &conv ) ; 
    if ( retval == PAM_SUCCESS ) {

	retval = conv->conv(nargs, ( const struct pam_message ** ) message
			    , response, conv->appdata_ptr);

	D(("returned from application's conversation function"));

	if (retval != PAM_SUCCESS && (PAM_DEBUG_ARG & ctrl) ) {
	    _log_err(LOG_DEBUG, "conversation failure [%s]"
		     , pam_strerror(retval));
	}

    } else {
	_log_err(LOG_ERR, "couldn't obtain coversation function [%s]"
		 , pam_strerror(retval));
    }

    D(("ready to return from module conversation"));

    return retval;                  /* propagate error status */
}

static int get_folder(pam_handle_t *pamh, int ctrl
		      , char **path_mail, char **folder_p)
{
    int retval;
    const char *user, *path;
    char *folder;

    retval = pam_get_user(pamh, &user, NULL);
    if (retval != PAM_SUCCESS || user == NULL) {
	_log_err(LOG_ERR, "no user specified");
	return PAM_USER_UNKNOWN;
    }

    if (ctrl & PAM_NEW_MAIL_DIR) {
	path = *path_mail;
    } else {
	path = DEFAULT_MAIL_DIRECTORY;
    }

    /* put folder together */

    folder = malloc(sizeof(MAIL_FILE_FORMAT)
		    +strlen(path)+strlen(user));
    if (folder == NULL) {
	_log_err(LOG_CRIT, "out of memory for mail folder");
	_pam_overwrite(*path_mail);
	_pam_drop(*path_mail);
	return PAM_BUF_ERR;
    }
    sprintf(folder, MAIL_FILE_FORMAT, path, user);

    _pam_overwrite(*path_mail);
    _pam_drop(*path_mail);                                   /* tidy up */
    user = NULL;

    *folder_p = folder;

    return PAM_SUCCESS;
}

static const char *get_mail_status(const char *folder)
{
    const char *type;
    struct stat mail_st;

    if (stat(folder, &mail_st) == 0 && mail_st.st_size > 0) {
	type = (mail_st.st_atime < mail_st.st_mtime) ? "new":"old" ;
    } else {
	type = "no";
    }

    memset(&mail_st, 0, sizeof(mail_st));
    D(("user has %s mail in %s folder", type, folder));
    return type;
}

static int report_mail(pam_handle_t *pamh, int ctrl
		       , const char *type, const char *folder)
{
    int retval;

    if (!(ctrl & PAM_MAIL_SILENT)) {
	char *remark;

	remark = malloc(sizeof(YOUR_MAIL_FORMAT)+strlen(type)+strlen(folder));
	if (remark == NULL) {
	    retval = PAM_BUF_ERR;
	} else {
	    struct pam_message msg[1], *mesg[1];
	    struct pam_response *resp=NULL;

	    sprintf(remark, YOUR_MAIL_FORMAT, type, folder);

	    mesg[0] = &msg[0];
	    msg[0].msg_style = PAM_TEXT_INFO;
	    msg[0].msg = remark;

	    retval = converse(pamh, ctrl, 1, mesg, &resp);

	    _pam_overwrite(remark);
	    _pam_drop(remark);
	}
    } else {
	D(("keeping quiet"));
	retval = PAM_SUCCESS;
    }

    D(("returning %s", pam_strerror(retval)));
    return retval;
}

/* --- authentication management functions (only) --- */

PAM_EXTERN
int pam_sm_open_session(pam_handle_t *pamh,int flags,int argc
			,const char **argv)
{
    int retval, ctrl;
    char *path_mail=NULL, *folder;
    const char *type;

    /*
     * this module sets the MAIL environment variable, and checks if
     * the user has any new mail.
     */

    ctrl = _pam_parse(flags, argc, argv, &path_mail);

    /* which folder? */

    retval = get_folder(pamh, ctrl, &path_mail, &folder);
    if (retval != PAM_SUCCESS) {
	D(("failed to find folder"));
	return retval;
    }

    /* set the MAIL variable? */

    if (!(ctrl & PAM_NO_ENV)) {
	char *tmp;

	tmp = malloc(strlen(folder)+sizeof(MAIL_ENV_FORMAT));
	if (tmp != NULL) {
	    sprintf(tmp, MAIL_ENV_FORMAT, folder);
	    D(("setting env: %s", tmp));
	    retval = pam_putenv(pamh, tmp);
	    _pam_overwrite(tmp);
	    _pam_drop(tmp);
	    if (retval != PAM_SUCCESS) {
		_pam_overwrite(folder);
		_pam_drop(folder);
		_log_err(LOG_CRIT, "unable to set MAIL variable");
		return retval;
	    }
	} else {
	    _log_err(LOG_CRIT, "no memory for MAIL variable");
	    _pam_overwrite(folder);
	    _pam_drop(folder);
	    return retval;
	}
    } else {
	D(("not setting MAIL variable"));
    }

    /*
     * OK. we've got the mail folder... what about its status?
     */

    if (!(ctrl & PAM_NO_LOGIN)) {
	type = get_mail_status(folder);
	retval = report_mail(pamh, ctrl, type, folder);
	type = NULL;
    }

    _pam_overwrite(folder);                             /* clean up */
    _pam_drop(folder);

    /* indicate success or failure */

    return retval;
}

PAM_EXTERN
int pam_sm_close_session(pam_handle_t *pamh,int flags,int argc
			 ,const char **argv)
{
    int retval, ctrl;
    char *path_mail=NULL, *folder;
    const char *type;

    /*
     * this module sets the MAIL environment variable, and checks if
     * the user has any new mail.
     */

    ctrl = _pam_parse(flags, argc, argv, &path_mail);

    /* which folder? */

    retval = get_folder(pamh, ctrl, &path_mail, &folder);
    if (retval != PAM_SUCCESS) {
	D(("failed to find folder"));
	return retval;
    }

    /*
     * OK. we've got the mail folder... what about its status?
     */

    if (ctrl & PAM_LOGOUT_TOO) {
	type = get_mail_status(folder);
	retval = report_mail(pamh, ctrl, type, folder);
	type = NULL;
    }

    _pam_overwrite(folder);                             /* clean up */
    _pam_drop(folder);

    /* indicate success or failure */

    return retval;
}

#ifdef PAM_STATIC

/* static module data */

struct pam_module _pam_mail_modstruct = {
     "pam_mail",
     NULL,
     NULL,
     NULL,
     pam_sm_open_session,
     pam_sm_close_session,
     NULL,
};

#endif

/* end of module definition */
